/**
 * A "zones" object manages a list of "zone" objects (see) and their state.
 * 
 * Basically, the zones objects reads the list of the available zones from a
 * file (normally objects/zones.txt) and initializes a corresponding list of
 * "zone" objects. Each zone is actually read and loaded or unloaded as needed
 * based on the location of each player (normally only one).
 * 
 * The zones file is a text file containing one line for each available zone;
 * each line must contains 5 fields separated by at least one white space:
 * 
 * 1. Minimum covered latitude.
 * 2. Maximum covered latitude.
 * 3. Minimum covered longitude.
 * 4. Maximum covered longitude.
 * 5. File name of the ACM scenery; may contain a path relative to the
 *    zones file itself.
 * 
 * Latitudes and longitudes may have the usual syntax, examples:
 * 
 *   10N   10-20-30.400S   30.5S
 *   10E   10-20-30.400W   30.5W
 * 
 * The ordering is important, so 90S &lt; 90N, 180W &lt; 180E.
 * 
 * For each zone (that is, each ACM scenery file) a specific zone object is
 * created; runways, craft types, features and NAVAIDS belonging to a specific
 * zone, have that zone as one of their "struct" field. That field allows to
 * recognize to which zone each object belongs.
 * 
 * Each player's aircraft (normally only one per program) should then feed its
 * current position by calling the zones_load() function, which in turn triggers
 * the loading of the appropriate scenery; sceneries that are not needed anymore
 * are removed from memory along with all their own objects.
 * 
 * @file
 * @author Umberto Salsi <salsi@icosaedro.it>
 * @version $Date: 2020/01/08 06:11:32 $
 */

#ifndef ZONES_H
#define ZONES_H

#include "../dis/dis/earth.h"
#include "zone.h"

#ifdef zones_IMPORT
	#define EXTERN
#else
	#define EXTERN extern
#endif

/**
 * Available zones and their state.
 */
typedef struct zones_Type zones_Type;

/**
 * Creates a new zones object. The zones file is read, ranges of latitude and
 * longitude for each zone is parsed, sceneries files are checked for
 * readability but not read yet. Non-fatal errors while parsing the zones file
 * are sent to standard error; fatal errors causes abort.
 * @param path Path of the zones file, normally objects/zones.txt.
 * @return New zones object. Must be released with memory_dispose(), which in
 * turn triggers the release of any loaded zone and all the items therein
 * defined.
 */
EXTERN zones_Type * zones_new(char *path);

/**
 * Feed the zones object with the current position of a player. In turn, this
 * may update the zones cache by triggering the loading or the un-loading of some
 * zone. For performances reason, this request is only evaluated from time to
 * time and ignored most of the times.
 * @param this Zones object.
 * @param p Position of the player; the altitude does not matter.
 * @param last_known Last known zone the point (that is, the craft) was flying
 * over. Set to NULL if still unknown.
 * @param forced If true, forces the evaluation of the position and guarantees
 * the appropriate zone be loaded immediately if available.
 * @return Current zone below the specified point. If forced update has been
 * requested, this is the real zone below the point, possibly NULL if no zone
 * available. If not forced, the last know zone is returned most of the times.
 */
EXTERN zone_Type * zones_load(zones_Type *this, earth_LatLonAlt *p, zone_Type *last_known, int forced);

#undef EXTERN
#endif
