/*
 *   DIS/x : An implementation of the IEEE 1278.1 protocol
 *
 *   Copyright (C) 1996, Riley Rainey (rainey@netcom.com)
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of either:
 *
 *   a) the GNU Library General Public License as published by the Free
 *   Software Foundation; either version 2 of the License, or (at your
 *   option) any later version.  A description of the terms and conditions
 *   of the GLPL may be found in the "COPYING.LIB" file.
 *
 *   b) the "Artistic License" which comes with this Kit.  Information
 *   about this license may be found in the "Artistic" file.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Library General Public License or the Artistic License for more details.
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Information describing how to contact the author can be found in the
 *   README file.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "../../util/memory.h"

#define disx_IMPORT
#include "disx.h"

/*
 *  This table defines the correct protocol family based on the pdu type
 */

static unsigned char pdu_family[256] =
{
	0, 1, 2, 2, 1, 3, 3, 3, 3, 3, 3, 5, 5, 5, 5, 5,  /*   0..15 */
	5, 5, 5, 5, 5, 5, 5, 6, 6, 4, 4, 0, 0, 0, 0, 0,  /*  16..31 */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  /*  32..47 */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  /*  48..63 */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  /*  64..79 */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  /*  80..95 */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  /*  96..111 */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  /* 112..127 */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  /* 128..143 */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 5, 5, 0, 0,  /* 144..159 */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  /* 160..175 */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  /* 176..191 */
};

static int protocol_version = DISProtocolVersionIEEE1278_95;

int
disx_setProtocolVersion(int version)
{
	int result = protocol_version;

	protocol_version = version;
	return result;
}


disx_ApplicationInfo *
disx_initializeApplication(dis_Transceiver *xcvr,
	int exercise_id,
	int site_id,
	int application_id)
{
	disx_ApplicationInfo *p = (disx_ApplicationInfo *) memory_allocate(
		sizeof(disx_ApplicationInfo), NULL);
	memset(p, 0, sizeof(*p));

	p->hdr.protocol_version = protocol_version;
	p->hdr.exercise_id = exercise_id;
	p->hdr.padding = 0;

	p->last_event = 0;
	p->last_entity = 0;
	p->last_request = 0;

	p->xcvr = xcvr;
	if (dis_setNBIOState(p->xcvr, 1) != 0) {
		fprintf(stderr, "Socket rejected non-blocking mode.\n");
		dis_closeTransceiver(p->xcvr);
		memory_dispose(p);
		return NULL;
	}
	p->id.site_id = site_id;
	p->id.application_id = application_id;
	return p;
}

void
disx_getSimulationAddress(disx_ApplicationInfo * info,
						 dis_simulation_addr * p)
{
	*p = info->id;
}

int
disx_writePDU(disx_ApplicationInfo * info, dis_pdu * p)
{
	p->hdr.protocol_version = info->hdr.protocol_version;
	p->hdr.exercise_id = info->hdr.exercise_id;
	p->hdr.protocol_family = pdu_family[p->hdr.pdu_type];
	/* don't set time here until there is a function to set
	   the value (time) and type (relative/absolute) of the time
	   in the DISx library
	   dis_getTimestamp(&p->hdr.time_stamp); */
	return dis_writePDU(info->xcvr, p);
}

int
disx_readPDU(disx_ApplicationInfo * info, dis_pdu * p)
{
	return dis_readPDU(info->xcvr, p);
}

void
disx_closeApplication(disx_ApplicationInfo * info)
{
	if( info == NULL )
		return;
	dis_closeTransceiver(info->xcvr);
	memory_dispose(info);
}

dis_entity_id *
disx_issueEntityID(disx_ApplicationInfo * info, dis_entity_id * e)
{
	e->sim_id = info->id;
	// Valid range for entity ID is [1,0xfffd];
	// zero, 0xfffe and 0xffff are not allowed;
	// cyclic.
	info->last_entity++;
	if (info->last_entity >= 0xfffe) {
		info->last_entity = 1;
	}
	e->entity_id = info->last_entity;
	return e;
}

dis_request_id
disx_issueRequestID( disx_ApplicationInfo * info )
{
	// Valid range for request ID: any 32-bits value;
	// cyclic.
    return ++info->last_request;
}

dis_event_id *
disx_issueEventID(disx_ApplicationInfo * info, dis_event_id * event)
{
	event->sim_id = info->id;
	// Valid range for event ID is [1,0xffff];
	// zero not allowed;
	// cyclic.
	info->last_event++;
	if (info->last_event == 0 ) {
		info->last_event = 1;
	}
	event->event_id = info->last_event;
	return event;
}


void
disx_initializeDatumInfo ( dis_datum_spec_record *pd )
{
	memset (pd, 0, sizeof(dis_datum_spec_record));
}
