/*
 *   DIS/x : An implementation of the IEEE 1278.1 protocol
 *
 *   Copyright (C) 1997, Riley Rainey (rrainey@ix.netcom.com)
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of either:
 *
 *   a) the GNU Library General Public License as published by the Free
 *   Software Foundation; either version 2 of the License, or (at your
 *   option) any later version.  A description of the terms and conditions
 *   of the GLPL may be found in the "COPYING.LIB" file.
 *
 *   b) the "Artistic License" which comes with this Kit.  Information
 *   about this license may be found in the "Artistic" file.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Library General Public License or the Artistic License for more details.
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Information describing how to contact the author can be found in the
 *   README file.
 */

#include <assert.h>
#include <string.h>

#include "../../util/error.h"
#include "datum.h"

#define xdr_dis_IMPORT
#include "xdr_dis.h"

/*
 *  This file was manually generated.  It was NOT created by RPCGEN.
 *  It contains xdr definitions for structures that do not fit the
 *  conventional RPCGEN structure model for variable length vectors or
 *  union definitions.
 */

/*
 *  These xdr_byte_*() routines act just as their xdr_ equivalents except that
 *  input data is not required to be aligned on a 4-byte boundary.  This
 *  implies that data alignment is the responsibility of the protocol
 *  designer.
 */

static int
xdr_byte_opaque(xdr_Type * xdrs, void * cp, u_int cnt)
{

	/*
	 * if no data we are done
	 */
	if (cnt == 0)
		return 1;

	if (xdr_getOperation(xdrs) == xdr_DECODE) {
		if (!xdr_getBytes(xdrs, cp, cnt)) {
			return 0;
		}
		return 1;
	}

	if (xdr_getOperation(xdrs) == xdr_ENCODE) {
		if (!xdr_putBytes(xdrs, cp, cnt)) {
			return 0;
		}
		return 1;
	}

	if (xdr_getOperation(xdrs) == xdr_FREE) {
		return 1;
	}

	return 0;
}

static int
xdr_byte_uint32(xdr_Type * xdrs, uint32_t *cp)
{
	uint32_t   u;

	u = htonl(*cp);
	if (!xdr_byte_opaque(xdrs, &u, 4)) {
		return 0;
	}
	if (xdr_getOperation(xdrs) == xdr_DECODE )
		*cp = ntohl(u);
	return 1;
}

static int
xdr_byte_uint16(xdr_Type * xdrs, uint16_t *cp)
{
	u_short   u;

	u = htons(*cp);
	if (!xdr_byte_opaque(xdrs, &u, 2)) {
		return 0;
	}
	if (xdr_getOperation(xdrs) == xdr_DECODE )
		*cp = ntohs(u);
	return 1;
}

static int
xdr_byte_uchar(xdr_Type * xdrs, u_char * cp)
{
	return xdr_byte_opaque(xdrs, cp, 1);
}


static int
xdr_dis_variable_datum(xdr_Type * xdrs, dis_variable_datum * objp)
{
	static char pad[8];			/* force padding to be zeroes */
	int       padbytes;
	uint32_t  len;

	if (!xdr_uint32(xdrs, &objp->datum_id)) {
		return 0;
	}
	if (!xdr_uint32(xdrs, &objp->value_length)) {
		return 0;
	}

	switch (objp->datum_id) {

	/*
	 * Double variable data items
	 */

	case datum_GeocentricCoordinatesX:
	case datum_GeocentricCoordinatesY:
	case datum_GeocentricCoordinatesZ:
		if (!xdr_double(xdrs, &objp->value.double_value)) {
			return 0;
		}
		break;

	/* 
	 * BEWARE.
	 * ======
	 * Datum type defaults to String, which are dynamically allocated and then
	 * must be released. Currently anything other than the types above is
	 * assumed to be string here and in the dis_disposeVariableDatum() function,
	 * then both MUST be kept synchronized.
	 */

	default:
		/*
		 * Assumes string of chars. Note that:
		 * - Length is given in bits (!), so the actual number of bytes is len/8.
		 *   Here we ignore any fractional byte, that is we ignore the last
		 *   3 bits of the length.
		 * - Strings are padded to 8 bytes fields.
		 * - Decoding, strings are dynamically allocated and a trailing NUL byte
		 *   is added for safety.
		 */
		len = objp->value_length / 8;
		if( xdr_getOperation(xdrs) == xdr_ENCODE ){
			if( ! xdr_putBytes(xdrs, objp->value.ptr_value, len) )
				return 0;
			padbytes = len % 8;
			if (padbytes != 0) {
				memset(pad, 0, 8 - padbytes);
				if( ! xdr_putBytes(xdrs, pad, 8 - padbytes) )
					return 0;
			}
			
		} else if( xdr_getOperation(xdrs) == xdr_DECODE ){
			if( ! xdr_getBytesAllocated(xdrs, (void **) &objp->value.ptr_value, len, 1) )
				return 0;
			
			padbytes = len % 8;
			if (padbytes != 0) {
				if( ! xdr_getBytes(xdrs, pad, 8 - padbytes) )
					return 0;
			}
			
		}
		break;
	}
	return 1;
}

static int
xdr_dis_articulation_parm(xdr_Type * xdrs, dis_articulation_parm * objp)
{
	if (!xdr_byte_uchar(xdrs, &objp->type)) {
		return 0;
	}
	if (!xdr_byte_uchar(xdrs, &objp->change)) {
		return 0;
	}
	if (!xdr_byte_uint16(xdrs, &objp->attachment_id)) {
		return 0;
	}
	if (!xdr_byte_uint32(xdrs, &objp->attached_part)) {
		return 0;
	}
	if (!xdr_byte_uint32(xdrs, &objp->articulated_part)) {
		return 0;
	}
	// FIXME: missing support for articulation parameter value.
	// Here we should add a specific encoding for each type of parameter.
	// Assume double just to pass incoming packets, but ignored by ACM anyway.
	if (!xdr_double(xdrs, &objp->value.d)) {
		return 0;
	}
	return 1;
}

static int
xdr_dis_timestamp(xdr_Type * xdrs, dis_timestamp * objp)
{
	if (!xdr_uint32(xdrs, (u_int *) objp)) {
		return 0;
	}
	return 1;
}


/*
static int
xdr_dis_float_vector(XDR *xdrs, dis_float_vector *objp)
{
	if (!xdr_float(xdrs, &objp->x)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->y)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->z)) {
		return 0;
	}
	return 1;
}
*/

static int
xdr_dis_angular_vel_vector(xdrs, objp)
	xdr_Type *xdrs;
	dis_angular_vel_vector *objp;
{
	if (!xdr_float(xdrs, &objp->x)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->y)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->z)) {
		return 0;
	}
	return 1;
}
	

static int
xdr_dis_linear_acc_vector(xdrs, objp)
	xdr_Type *xdrs;
	dis_linear_acc_vector *objp;
{
	if (!xdr_float(xdrs, &objp->x)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->y)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->z)) {
		return 0;
	}
	return 1;
}


static int
xdr_dis_linear_vel_vector(xdrs, objp)
	xdr_Type *xdrs;
	dis_linear_vel_vector *objp;
{
	if (!xdr_float(xdrs, &objp->x)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->y)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->z)) {
		return 0;
	}
	return 1;
}


static int
xdr_dis_entity_coord_vector(xdrs, objp)
	xdr_Type *xdrs;
	dis_entity_coord_vector *objp;
{
	if (!xdr_float(xdrs, &objp->x)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->y)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->z)) {
		return 0;
	}
	return 1;
}


static int
xdr_dis_entity_type(xdrs, objp)
	xdr_Type *xdrs;
	dis_entity_type *objp;
{
	if (!xdr_byte_uchar(xdrs, &objp->kind)) {
		return 0;
	}
	if (!xdr_byte_uchar(xdrs, &objp->domain)) {
		return 0;
	}
	if (!xdr_byte_uint16(xdrs, &objp->country)) {
		return 0;
	}
	if (!xdr_byte_uchar(xdrs, &objp->category)) {
		return 0;
	}
	if (!xdr_byte_uchar(xdrs, &objp->subcategory)) {
		return 0;
	}
	if (!xdr_byte_uchar(xdrs, &objp->specific)) {
		return 0;
	}
	if (!xdr_byte_uchar(xdrs, &objp->extra)) {
		return 0;
	}
	return 1;
}


static int
xdr_dis_entity_marking(xdrs, objp)
	xdr_Type *xdrs;
	dis_entity_marking *objp;
{
	if (!xdr_byte_uchar(xdrs, &objp->charset)) {
		return 0;
	}
	if (!xdr_vector(xdrs, (char *)objp->marking, sizeof(objp->marking), sizeof(u_char), (xdr_Callback) xdr_byte_uchar)) {
		return 0;
	}
	return 1;
}


static int
xdr_dis_fixed_datum(xdrs, objp)
	xdr_Type *xdrs;
	dis_fixed_datum *objp;
{
	if (!xdr_uint32(xdrs, &objp->datum_id)) {
		return 0;
	}
	if (!xdr_uint32(xdrs, &objp->value)) {
		return 0;
	}
	return 1;
}


static int
xdr_dis_simulation_addr(xdrs, objp)
	xdr_Type *xdrs;
	dis_simulation_addr *objp;
{
	if (!xdr_byte_uint16(xdrs, &objp->site_id)) {
		return 0;
	}
	if (!xdr_byte_uint16(xdrs, &objp->application_id)) {
		return 0;
	}
	return 1;
}


static int
xdr_dis_emitter_system(xdrs, objp)
	xdr_Type *xdrs;
	dis_emitter_system *objp;
{
	if (!xdr_byte_uint16(xdrs, &objp->name)) {
		return 0;
	}
	if (!xdr_byte_uchar(xdrs, &objp->function)) {
		return 0;
	}
	if (!xdr_byte_uchar(xdrs, &objp->id)) {
		return 0;
	}
	return 1;
}


static int
xdr_dis_entity_id(xdrs, objp)
	xdr_Type *xdrs;
	dis_entity_id *objp;
{
	if (!xdr_dis_simulation_addr(xdrs, &objp->sim_id)) {
		return 0;
	}
	if (!xdr_byte_uint16(xdrs, &objp->entity_id)) {
		return 0;
	}
	return 1;
}


static int
xdr_dis_euler_angles(xdrs, objp)
	xdr_Type *xdrs;
	dis_euler_angles *objp;
{
	if (!xdr_float(xdrs, &objp->psi)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->theta)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->phi)) {
		return 0;
	}
	return 1;
}


static int
xdr_dis_event_id(xdrs, objp)
	xdr_Type *xdrs;
	dis_event_id *objp;
{
	if (!xdr_dis_simulation_addr(xdrs, &objp->sim_id)) {
		return 0;
	}
	if (!xdr_byte_uint16(xdrs, &objp->event_id)) {
		return 0;
	}
	return 1;
}


static int
xdr_dis_fundamental_parameters(xdrs, objp)
	xdr_Type *xdrs;
	dis_fundamental_parameters *objp;
{
	if (!xdr_float(xdrs, &objp->freq)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->freq_range)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->erp)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->prf)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->pulse_width)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->beam_azimuth_center)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->beam_azimuth_sweep)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->beam_elev_center)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->beam_elev_sweep)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->beam_sweep_sync)) {
		return 0;
	}
	return 1;
}


/*
static int
xdr_dis_modulation_type(xdrs, objp)
	XDR *xdrs;
	dis_modulation_type *objp;
{
	if (!xdr_byte_u_short(xdrs, &objp->spread_spectrum)) {
		return 0;
	}
	if (!xdr_byte_u_short(xdrs, &objp->major_type)) {
		return 0;
	}
	if (!xdr_byte_u_short(xdrs, &objp->detail)) {
		return 0;
	}
	if (!xdr_byte_u_short(xdrs, &objp->system)) {
		return 0;
	}
	return 1;
}
*/


static int
xdr_dis_pdu_header(xdrs, objp)
	xdr_Type *xdrs;
	dis_pdu_header *objp;
{
	if (!xdr_byte_uchar(xdrs, &objp->protocol_version)) {
		return 0;
	}
	if (!xdr_byte_uchar(xdrs, &objp->exercise_id)) {
		return 0;
	}
	if (!xdr_byte_uchar(xdrs, &objp->pdu_type)) {
		return 0;
	}
	if (!xdr_byte_uchar(xdrs, &objp->protocol_family)) {
		return 0;
	}
	if (!xdr_dis_timestamp(xdrs, &objp->time_stamp)) {
		return 0;
	}
	if (!xdr_byte_uint16(xdrs, &objp->length)) {
		return 0;
	}
	if (!xdr_byte_uint16(xdrs, &objp->padding)) {
		return 0;
	}
	return 1;
}


/*
static int
xdr_dis_double_vector(xdrs, objp)
	XDR *xdrs;
	dis_double_vector *objp;
{
	if (!xdr_byte_double(xdrs, &objp->x)) {
		return 0;
	}
	if (!xdr_byte_double(xdrs, &objp->y)) {
		return 0;
	}
	if (!xdr_byte_double(xdrs, &objp->z)) {
		return 0;
	}
	return 1;
}
*/


static int
xdr_earth_XYZ(xdrs, objp)
	xdr_Type *xdrs;
	VPoint *objp;
{
	if (!xdr_double(xdrs, &objp->x)) {
		return 0;
	}
	if (!xdr_double(xdrs, &objp->y)) {
		return 0;
	}
	if (!xdr_double(xdrs, &objp->z)) {
		return 0;
	}
	return 1;
}


/*
static int
xdr_dis_relative_coordinates(xdrs, objp)
	XDR *xdrs;
	dis_relative_coordinates *objp;
{
	if (!xdr_float(xdrs, &objp->x)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->y)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->z)) {
		return 0;
	}
	return 1;
}
*/


/*
static int
xdr_dis_antenna_location(xdrs, objp)
	XDR *xdrs;
	dis_antenna_location *objp;
{
	if (!xdr_earth_XYZ(xdrs, &objp->ant_location)) {
		return 0;
	}
	if (!xdr_dis_relative_coordinates(xdrs, &objp->relative)) {
		return 0;
	}
	return 1;
}
*/


/*
static int
xdr_dis_beam_antenna_pattern(xdrs, objp)
	XDR *xdrs;
	dis_beam_antenna_pattern *objp;
{
	if (!xdr_dis_euler_angles(xdrs, &objp->direction)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->azimuth_width)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->elev_width)) {
		return 0;
	}
	if (!xdr_byte_u_char(xdrs, &objp->reference_system)) {
		return 0;
	}
	if (!xdr_vector(xdrs, (char *)objp->pad, 3, sizeof(byte_u_char), (xdrproc_t) xdr_byte_u_char)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->Ez)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->Ex)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->phase)) {
		return 0;
	}
	return 1;
}
*/


/*
static int
xdr_dis_spherical_harmonic_antenna_pattern(xdrs, objp)
	XDR *xdrs;
	dis_spherical_harmonic_antenna_pattern *objp;
{
	if (!xdr_byte_u_char(xdrs, &objp->pattern)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->coefficients)) {
		return 0;
	}
	if (!xdr_byte_u_char(xdrs, &objp->ref_system)) {
		return 0;
	}
	return 1;
}
*/


static int
xdr_dis_time(xdrs, objp)
	xdr_Type *xdrs;
	dis_time *objp;
{
	if (!xdr_uint32(xdrs, &objp->hour)) {
		return 0;
	}
	if (!xdr_dis_timestamp(xdrs, &objp->rel)) {
		return 0;
	}
	return 1;
}


static int
xdr_dis_burst_descriptor(xdrs, objp)
	xdr_Type *xdrs;
	dis_burst_descriptor *objp;
{
	if (!xdr_dis_entity_type(xdrs, &objp->munition)) {
		return 0;
	}
	if (!xdr_byte_uint16(xdrs, &objp->warhead)) {
		return 0;
	}
	if (!xdr_byte_uint16(xdrs, &objp->fuze)) {
		return 0;
	}
	if (!xdr_byte_uint16(xdrs, &objp->quantity)) {
		return 0;
	}
	if (!xdr_byte_uint16(xdrs, &objp->rate)) {
		return 0;
	}
	return 1;
}


static int
xdr_dis_dead_reckoning(xdrs, objp)
	xdr_Type *xdrs;
	dis_dead_reckoning *objp;
{
	if (!xdr_byte_uchar(xdrs, &objp->algorithm)) {
		return 0;
	}
	if (!xdr_vector(xdrs, (char *)objp->other, sizeof(objp->other), sizeof(u_char), (xdr_Callback) xdr_byte_uchar)) {
		return 0;
	}
	if (!xdr_dis_linear_acc_vector(xdrs, &objp->linear_acc)) {
		return 0;
	}
	if (!xdr_dis_angular_vel_vector(xdrs, &objp->angular_vel)) {
		return 0;
	}
	return 1;
}


static int
xdr_dis_capabilities(xdrs, objp)
	xdr_Type *xdrs;
	dis_capabilities *objp;
{
	if (!xdr_uint32(xdrs, objp)) {
		return 0;
	}
	return 1;
}


/*
static int
xdr_dis_supply_quantity(xdrs, objp)
	XDR *xdrs;
	dis_supply_quantity *objp;
{
	if (!xdr_dis_entity_type(xdrs, &objp->entity)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->quantity)) {
		return 0;
	}
	return 1;
}
*/


static int
xdr_dis_entity_appearance(xdrs, objp)
	xdr_Type *xdrs;
	dis_entity_appearance *objp;
{
	if (!xdr_uint32(xdrs, objp)) {
		return 0;
	}
	return 1;
}


static int
xdr_dis_entity_state_pdu(xdrs, objp)
	xdr_Type *xdrs;
	dis_entity_state_pdu *objp;
{
	if (!xdr_dis_pdu_header(xdrs, &objp->hdr)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->id)) {
		return 0;
	}
	if (!xdr_byte_uchar(xdrs, &objp->force_id)) {
		return 0;
	}
	if (!xdr_byte_uchar(xdrs, &objp->art_parm_count)) {
		return 0;
	}
	if (!xdr_dis_entity_type(xdrs, &objp->type)) {
		return 0;
	}
	if (!xdr_dis_entity_type(xdrs, &objp->alt_type)) {
		return 0;
	}
	if (!xdr_dis_linear_vel_vector(xdrs, &objp->vel)) {
		return 0;
	}
	if (!xdr_earth_XYZ(xdrs, &objp->pos)) {
		return 0;
	}
	if (!xdr_dis_euler_angles(xdrs, &objp->orientation)) {
		return 0;
	}
	if (!xdr_dis_entity_appearance(xdrs, &objp->appearance)) {
		return 0;
	}
	if (!xdr_dis_dead_reckoning(xdrs, &objp->dr_parm)) {
		return 0;
	}
	if (!xdr_dis_entity_marking(xdrs, &objp->marking)) {
		return 0;
	}
	if (!xdr_dis_capabilities(xdrs, &objp->capabilities)) {
		return 0;
	}
	if (!xdr_var_array(xdrs, (void **)&objp->art_parm,
		objp->art_parm_count, sizeof(dis_articulation_parm), (xdr_Callback) xdr_dis_articulation_parm)) {
		return 0;
	}
	return 1;
}


static int
xdr_dis_collision_pdu(xdrs, objp)
	xdr_Type *xdrs;
	dis_collision_pdu *objp;
{
	if (!xdr_dis_pdu_header(xdrs, &objp->hdr)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->id)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->collision_id)) {
		return 0;
	}
	if (!xdr_dis_event_id(xdrs, &objp->event)) {
		return 0;
	}
	if (!xdr_dis_linear_vel_vector(xdrs, &objp->vel)) {
		return 0;
	}
	if (!xdr_uint32(xdrs, &objp->mass)) {
		return 0;
	}
	if (!xdr_dis_entity_coord_vector(xdrs, &objp->loc)) {
		return 0;
	}
	return 1;
}


static int
xdr_dis_fire_pdu(xdrs, objp)
	xdr_Type *xdrs;
	dis_fire_pdu *objp;
{
	if (!xdr_dis_pdu_header(xdrs, &objp->hdr)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->firing_id)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->target_id)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->munition_id)) {
		return 0;
	}
	if (!xdr_dis_event_id(xdrs, &objp->event)) {
		return 0;
	}
	if (!xdr_uint32(xdrs, &objp->fire_mission_index)) {
		return 0;
	}
	if (!xdr_earth_XYZ(xdrs, &objp->pos)) {
		return 0;
	}
	if (!xdr_dis_burst_descriptor(xdrs, &objp->burst)) {
		return 0;
	}
	if (!xdr_dis_linear_vel_vector(xdrs, &objp->vel)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->range)) {
		return 0;
	}
	return 1;
}


static int
xdr_dis_detonation_pdu(xdrs, objp)
	xdr_Type *xdrs;
	dis_detonation_pdu *objp;
{
	if (!xdr_dis_pdu_header(xdrs, &objp->hdr)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->firing_id)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->target_id)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->munition_id)) {
		return 0;
	}
	if (!xdr_dis_event_id(xdrs, &objp->event)) {
		return 0;
	}
	if (!xdr_dis_linear_vel_vector(xdrs, &objp->vel)) {
		return 0;
	}
	if (!xdr_earth_XYZ(xdrs, &objp->pos)) {
		return 0;
	}
	if (!xdr_dis_burst_descriptor(xdrs, &objp->burst)) {
		return 0;
	}
	if (!xdr_dis_entity_coord_vector(xdrs, &objp->loc)) {
		return 0;
	}
	if (!xdr_byte_uchar(xdrs, &objp->result)) {
		return 0;
	}
	if (!xdr_byte_uchar(xdrs, &objp->num_art_parms)) {
		return 0;
	}
	if (!xdr_byte_uint16(xdrs, &objp->pad)) {
		return 0;
	}
	if (!xdr_var_array(xdrs, (void **)&objp->art_parm,
		objp->num_art_parms, sizeof(dis_articulation_parm), (xdr_Callback) xdr_dis_articulation_parm)) {
		return 0;
	}
	return 1;
}


/*
static int
xdr_dis_service_type(xdrs, objp)
	XDR *xdrs;
	dis_service_type *objp;
{
	if (!xdr_byte_u_char(xdrs, objp)) {
		return 0;
	}
	return 1;
}
*/


/*
static int
xdr_dis_repair_type(xdrs, objp)
	XDR *xdrs;
	dis_repair_type *objp;
{
	if (!xdr_byte_u_short(xdrs, objp)) {
		return 0;
	}
	return 1;
}
*/


/*
static int
xdr_dis_repair_result(xdrs, objp)
	XDR *xdrs;
	dis_repair_result *objp;
{
	if (!xdr_byte_u_char(xdrs, objp)) {
		return 0;
	}
	return 1;
}
*/


/*
static int
xdr_dis_service_request_pdu(xdrs, objp)
	XDR *xdrs;
	dis_service_request_pdu *objp;
{
	if (!xdr_dis_pdu_header(xdrs, &objp->hdr)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->requestor_id)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->server_id)) {
		return 0;
	}
	if (!xdr_dis_service_type(xdrs, &objp->requested_service)) {
		return 0;
	}
	if (!xdr_byte_u_char(xdrs, &objp->num_supply_types)) {
		return 0;
	}
	if (!xdr_byte_u_short(xdrs, &objp->pad)) {
		return 0;
	}
	if (!xdr_var_array(xdrs, (char **)&objp->supplies,
		objp->num_supply_types, objp->num_supply_types, sizeof(dis_supply_quantity), (xdrproc_t) xdr_dis_supply_quantity)) {
		return 0;
	}
	return 1;
}
*/


/*
static int
xdr_dis_resupply_offer_pdu(xdrs, objp)
	XDR *xdrs;
	dis_resupply_offer_pdu *objp;
{
	if (!xdr_dis_pdu_header(xdrs, &objp->hdr)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->receiver_id)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->supplier_id)) {
		return 0;
	}
	if (!xdr_byte_u_char(xdrs, &objp->num_supply_types)) {
		return 0;
	}
	if (!xdr_vector(xdrs, (char *)objp->pad, 3, sizeof(byte_u_char), (xdrproc_t) xdr_byte_u_char)) {
		return 0;
	}
	if (!xdr_var_array(xdrs, (char **)&objp->supplies,
		objp->num_supply_types, objp->num_supply_types, sizeof(dis_supply_quantity), (xdrproc_t) xdr_dis_supply_quantity)) {
		return 0;
	}
	return 1;
}
*/


/*
static int
xdr_dis_resupply_received_pdu(xdrs, objp)
	XDR *xdrs;
	dis_resupply_received_pdu *objp;
{
	if (!xdr_dis_pdu_header(xdrs, &objp->hdr)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->receiver_id)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->supplier_id)) {
		return 0;
	}
	if (!xdr_byte_u_char(xdrs, &objp->num_supply_types)) {
		return 0;
	}
	if (!xdr_vector(xdrs, (char *)objp->pad, 3, sizeof(byte_u_char), (xdrproc_t) xdr_byte_u_char)) {
		return 0;
	}
	if (!xdr_var_array(xdrs, (char **)&objp->supplies,
		objp->num_supply_types, objp->num_supply_types, sizeof(dis_supply_quantity), (xdrproc_t) xdr_dis_supply_quantity)) {
		return 0;
	}
	return 1;
}
*/


/*
static int
xdr_dis_resupply_cancel_pdu(xdrs, objp)
	XDR *xdrs;
	dis_resupply_cancel_pdu *objp;
{
	if (!xdr_dis_pdu_header(xdrs, &objp->hdr)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->receiver_id)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->supplier_id)) {
		return 0;
	}
	return 1;
}
*/


/*
static int
xdr_dis_repair_complete_pdu(xdrs, objp)
	XDR *xdrs;
	dis_repair_complete_pdu *objp;
{
	if (!xdr_dis_pdu_header(xdrs, &objp->hdr)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->receiver_id)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->supplier_id)) {
		return 0;
	}
	if (!xdr_dis_repair_type(xdrs, &objp->repair)) {
		return 0;
	}
	if (!xdr_byte_u_short(xdrs, &objp->pad)) {
		return 0;
	}
	return 1;
}
*/


/*
static int
xdr_dis_repair_response_pdu(xdrs, objp)
	XDR *xdrs;
	dis_repair_response_pdu *objp;
{
	if (!xdr_dis_pdu_header(xdrs, &objp->hdr)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->receiver_id)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->supplier_id)) {
		return 0;
	}
	if (!xdr_dis_repair_result(xdrs, &objp->result)) {
		return 0;
	}
	if (!xdr_vector(xdrs, (char *)objp->pad, 3, sizeof(byte_u_char), (xdrproc_t) xdr_byte_u_char)) {
		return 0;
	}
	return 1;
}
*/


static int
xdr_dis_request_id(xdrs, objp)
	xdr_Type *xdrs;
	dis_request_id *objp;
{
	if (!xdr_uint32(xdrs, objp)) {
		return 0;
	}
	return 1;
}


static int
xdr_dis_create_entity_pdu(xdrs, objp)
	xdr_Type *xdrs;
	dis_create_entity_pdu *objp;
{
	if (!xdr_dis_pdu_header(xdrs, &objp->hdr)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->orig_id)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->recv_id)) {
		return 0;
	}
	if (!xdr_dis_request_id(xdrs, &objp->request_id)) {
		return 0;
	}
	return 1;
}


static int
xdr_dis_remove_entity_pdu(xdrs, objp)
	xdr_Type *xdrs;
	dis_remove_entity_pdu *objp;
{
	if (!xdr_dis_pdu_header(xdrs, &objp->hdr)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->orig_id)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->recv_id)) {
		return 0;
	}
	if (!xdr_dis_request_id(xdrs, &objp->request_id)) {
		return 0;
	}
	return 1;
}


static int
xdr_dis_start_pdu(xdrs, objp)
	xdr_Type *xdrs;
	dis_start_pdu *objp;
{
	if (!xdr_dis_pdu_header(xdrs, &objp->hdr)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->orig_id)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->recv_id)) {
		return 0;
	}
	if (!xdr_dis_time(xdrs, &objp->real_time)) {
		return 0;
	}
	if (!xdr_dis_time(xdrs, &objp->sim_time)) {
		return 0;
	}
	if (!xdr_dis_request_id(xdrs, &objp->request_id)) {
		return 0;
	}
	return 1;
}


static int
xdr_dis_stop_pdu(xdrs, objp)
	xdr_Type *xdrs;
	dis_stop_pdu *objp;
{
	if (!xdr_dis_pdu_header(xdrs, &objp->hdr)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->orig_id)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->recv_id)) {
		return 0;
	}
	if (!xdr_dis_time(xdrs, &objp->real_time)) {
		return 0;
	}
	if (!xdr_byte_uchar(xdrs, &objp->reason)) {
		return 0;
	}
	if (!xdr_byte_uchar(xdrs, &objp->behavior)) {
		return 0;
	}
	if (!xdr_vector(xdrs, (char *)objp->pad, 2, sizeof(u_char), (xdr_Callback) xdr_byte_uchar)) {
		return 0;
	}
	if (!xdr_dis_request_id(xdrs, &objp->request_id)) {
		return 0;
	}
	return 1;
}


static int
xdr_dis_acknowledge_pdu(xdrs, objp)
	xdr_Type *xdrs;
	dis_acknowledge_pdu *objp;
{
	if (!xdr_dis_pdu_header(xdrs, &objp->hdr)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->orig_id)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->recv_id)) {
		return 0;
	}
	if (!xdr_byte_uint16(xdrs, &objp->acknowledge_flag)) {
		return 0;
	}
	if (!xdr_byte_uint16(xdrs, &objp->resp_flag)) {
		return 0;
	}
	if (!xdr_dis_request_id(xdrs, &objp->request_id)) {
		return 0;
	}
	return 1;
}


/*
static int
xdr_dis_data_query_pdu(xdrs, objp)
	XDR *xdrs;
	dis_data_query_pdu *objp;
{
	if (!xdr_dis_pdu_header(xdrs, &objp->hdr)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->orig_id)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->recv_id)) {
		return 0;
	}
	if (!xdr_dis_time(xdrs, &objp->interval)) {
		return 0;
	}
	if (!xdr_dis_request_id(xdrs, &objp->request_id)) {
		return 0;
	}
	if (!xdr_byte_u_long(xdrs, &objp->num_fixed_data)) {
		return 0;
	}
	if (!xdr_byte_u_long(xdrs, &objp->num_variable_data)) {
		return 0;
	}
	if (!xdr_var_array(xdrs, (char **)&objp->fixed_datum_id,
		objp->num_fixed_data, objp->num_fixed_data, sizeof(byte_u_long), (xdrproc_t) xdr_byte_u_long)) {
		return 0;
	}
	if (!xdr_var_array(xdrs, (char **)&objp->variable_datum_id,
		objp->num_variable_data, objp->num_variable_data, sizeof(byte_u_long), (xdrproc_t) xdr_byte_u_long)) {
		return 0;
	}
	return 1;
}
*/


static int
xdr_dis_datum_spec_record(xdrs, objp)
	xdr_Type *xdrs;
	dis_datum_spec_record *objp;
{
	if (!xdr_uint32(xdrs, &objp->num_fixed_data)) {
		return 0;
	}
	if (!xdr_uint32(xdrs, &objp->num_variable_data)) {
		return 0;
	}
	if (!xdr_var_array(xdrs, (void **)&objp->fixed_datum,
		objp->num_fixed_data, sizeof(dis_fixed_datum), (xdr_Callback) xdr_dis_fixed_datum)) {
		return 0;
	}
	if (!xdr_var_array(xdrs, (void **)&objp->variable_datum,
		objp->num_variable_data, sizeof(dis_variable_datum), (xdr_Callback) xdr_dis_variable_datum)) {
		return 0;
	}
	return 1;
}


static int
xdr_dis_set_data_pdu(xdrs, objp)
	xdr_Type *xdrs;
	dis_set_data_pdu *objp;
{
	if (!xdr_dis_pdu_header(xdrs, &objp->hdr)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->orig_id)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->recv_id)) {
		return 0;
	}
	if (!xdr_dis_request_id(xdrs, &objp->request_id)) {
		return 0;
	}
	if (!xdr_dis_datum_spec_record(xdrs, &objp->datum_info)) {
		return 0;
	}
	return 1;
}


/*
static int
xdr_dis_data_pdu(xdrs, objp)
	XDR *xdrs;
	dis_data_pdu *objp;
{
	if (!xdr_dis_pdu_header(xdrs, &objp->hdr)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->orig_id)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->recv_id)) {
		return 0;
	}
	if (!xdr_dis_request_id(xdrs, &objp->request_id)) {
		return 0;
	}
	if (!xdr_dis_datum_spec_record(xdrs, &objp->datum_info)) {
		return 0;
	}
	return 1;
}
*/


/*
static int
xdr_dis_event_report_pdu(xdrs, objp)
	XDR *xdrs;
	dis_event_report_pdu *objp;
{
	if (!xdr_dis_pdu_header(xdrs, &objp->hdr)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->orig_id)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->recv_id)) {
		return 0;
	}
	if (!xdr_dis_event_id(xdrs, &objp->event_type)) {
		return 0;
	}
	if (!xdr_dis_datum_spec_record(xdrs, &objp->datum_info)) {
		return 0;
	}
	return 1;
}
*/


static int
xdr_dis_comment_pdu(xdr_Type *xdrs, dis_comment_pdu *objp)
{
	if (!xdr_dis_pdu_header(xdrs, &objp->hdr)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->orig_id)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->recv_id)) {
		return 0;
	}
	if (!xdr_uint32(xdrs, &objp->num_fixed_data)) {
		return 0;
	}
	if (!xdr_uint32(xdrs, &objp->num_variable_data)) {
		return 0;
	}
	if (!xdr_var_array(xdrs, (void **)&objp->fixed_datum,
		objp->num_fixed_data, sizeof(dis_fixed_datum), (xdr_Callback) xdr_dis_fixed_datum)) {
		return 0;
	}
	if (!xdr_var_array(xdrs, (void **)&objp->variable_datum,
		objp->num_variable_data, sizeof(dis_variable_datum), (xdr_Callback) xdr_dis_variable_datum)) {
		return 0;
	}
	return 1;
}


static int
xdr_dis_track_info(xdrs, objp)
	xdr_Type *xdrs;
	dis_track_info *objp;
{
	if (!xdr_dis_entity_id(xdrs, &objp->target)) {
		return 0;
	}
	if (!xdr_byte_uchar(xdrs, &objp->emitter_id)) {
		return 0;
	}
	if (!xdr_byte_uchar(xdrs, &objp->beam_id)) {
		return 0;
	}
	return 1;
}


static int
xdr_dis_beam_info(xdrs, objp)
	xdr_Type *xdrs;
	dis_beam_info *objp;
{
	if (!xdr_byte_uchar(xdrs, &objp->beam_data_length)) {
		return 0;
	}
	if (!xdr_byte_uchar(xdrs, &objp->beam_id)) {
		return 0;
	}
	if (!xdr_byte_uint16(xdrs, &objp->beam_parm_index)) {
		return 0;
	}
	if (!xdr_dis_fundamental_parameters(xdrs, &objp->fundamental)) {
		return 0;
	}
	if (!xdr_byte_uchar(xdrs, &objp->beam_function)) {
		return 0;
	}
	if (!xdr_byte_uchar(xdrs, &objp->num_targets)) {
		return 0;
	}
	if (!xdr_byte_uchar(xdrs, &objp->high_density_track_jam)) {
		return 0;
	}
	if (!xdr_byte_uchar(xdrs, &objp->pad)) {
		return 0;
	}
	if (!xdr_uint32(xdrs, &objp->jamming_mode)) {
		return 0;
	}
	if (!xdr_var_array(xdrs, (void **)&objp->tracked_target,
		objp->num_targets, sizeof(dis_track_info), (xdr_Callback) xdr_dis_track_info)) {
		return 0;
	}
	return 1;
}


static int
xdr_dis_em_system_info(xdrs, objp)
	xdr_Type *xdrs;
	dis_em_system_info *objp;
{
	if (!xdr_byte_uchar(xdrs, &objp->sys_data_length)) {
		return 0;
	}
	if (!xdr_byte_uchar(xdrs, &objp->num_beams)) {
		return 0;
	}
	if (!xdr_byte_uint16(xdrs, &objp->pad)) {
		return 0;
	}
	if (!xdr_dis_emitter_system(xdrs, &objp->emitter_system)) {
		return 0;
	}
	if (!xdr_dis_entity_coord_vector(xdrs, &objp->location)) {
		return 0;
	}
	if (!xdr_var_array(xdrs, (void **)&objp->beam,
		objp->num_beams, sizeof(dis_beam_info), (xdr_Callback) xdr_dis_beam_info)) {
		return 0;
	}
	return 1;
}


static int
xdr_dis_em_emission_pdu(xdrs, objp)
	xdr_Type *xdrs;
	dis_em_emission_pdu *objp;
{
	if (!xdr_dis_pdu_header(xdrs, &objp->hdr)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->emitter_id)) {
		return 0;
	}
	if (!xdr_dis_event_id(xdrs, &objp->event)) {
		return 0;
	}
	if (!xdr_byte_uchar(xdrs, &objp->state_update)) {
		return 0;
	}
	if (!xdr_byte_uchar(xdrs, &objp->num_systems)) {
		return 0;
	}
	if (!xdr_byte_uint16(xdrs, &objp->pad)) {
		return 0;
	}
	if (!xdr_var_array(xdrs, (void **)&objp->system,
		objp->num_systems, sizeof(dis_em_system_info), (xdr_Callback) xdr_dis_em_system_info)) {
		return 0;
	}
	return 1;
}


/*
static int
xdr_dis_designator_pdu(xdrs, objp)
	XDR *xdrs;
	dis_designator_pdu *objp;
{
	if (!xdr_dis_pdu_header(xdrs, &objp->hdr)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->designating_id)) {
		return 0;
	}
	if (!xdr_byte_u_short(xdrs, &objp->code_name)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->designated_id)) {
		return 0;
	}
	if (!xdr_byte_u_char(xdrs, &objp->pad)) {
		return 0;
	}
	if (!xdr_byte_u_char(xdrs, &objp->code)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->power)) {
		return 0;
	}
	if (!xdr_float(xdrs, &objp->wavelength)) {
		return 0;
	}
	if (!xdr_dis_entity_coord_vector(xdrs, &objp->spot_rel)) {
		return 0;
	}
	if (!xdr_earth_XYZ(xdrs, &objp->spot_pos)) {
		return 0;
	}
	return 1;
}
*/


/*
static int
xdr_dis_experimental_request_control_pdu(xdrs, objp)
	XDR *xdrs;
	dis_experimental_request_control_pdu *objp;
{
	if (!xdr_dis_pdu_header(xdrs, &objp->hdr)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->requesting_id)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->control_target_id)) {
		return 0;
	}
	return 1;
}
*/


/*
static int
xdr_dis_experimental_grant_control_pdu(xdrs, objp)
	XDR *xdrs;
	dis_experimental_grant_control_pdu *objp;
{
	if (!xdr_dis_pdu_header(xdrs, &objp->hdr)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->granting_id)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->control_target_id)) {
		return 0;
	}
	return 1;
}
*/


static int
xdr_dis_transfer_control_pdu(xdrs, objp)
	xdr_Type *xdrs;
	dis_transfer_control_pdu *objp;
{
	if (!xdr_dis_pdu_header(xdrs, &objp->hdr)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->orig_id)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->recv_id)) {
		return 0;
	}
	if (!xdr_uint32(xdrs, &objp->request_id)) {
		return 0;
	}
	if (!xdr_byte_uchar(xdrs, &objp->reliability_service)) {
		return 0;
	}
	if (!xdr_byte_uchar(xdrs, &objp->transfer_type)) {
		return 0;
	}
	if (!xdr_dis_entity_id(xdrs, &objp->target_id)) {
		return 0;
	}
	if (!xdr_uint32(xdrs, &objp->num_record_sets)) {
		return 0;
	}
	return 1;
}

int
xdr_dis_pdu(xdr_Type * xdrs, dis_pdu * objp)
{
	uint32_t    pos;

	/*
	 * On decoding, retrieve the header first just to get the PDU type and
	 * then apply the specific decoding function next. Each specific function
	 * will then decode the header back again, which is a bit redundant, but
	 * necessary because each specific function must serve also for encoding.
	 */
	if (xdr_getOperation(xdrs) == xdr_DECODE) {

		pos = xdr_getpos(xdrs);

		if (!xdr_dis_pdu_header(xdrs, (dis_pdu_header *) objp)) {
			return 0;
		}

		if (!xdr_setpos(xdrs, pos)) {
			return 0;
		}
	}

	switch (objp->hdr.pdu_type) {
	case PDUTypeOther:
		if (!xdr_dis_pdu_header(xdrs, &objp->hdr)) {
			return 0;
		}
		break;
	case PDUTypeEntityState:
		if (!xdr_dis_entity_state_pdu(xdrs, (dis_entity_state_pdu *) objp)) {
			return 0;
		}
		break;
	case PDUTypeCollision:
		if (!xdr_dis_collision_pdu(xdrs, (dis_collision_pdu *) objp)) {
			return 0;
		}
		break;
	case PDUTypeFire:
		if (!xdr_dis_fire_pdu(xdrs, (dis_fire_pdu *) objp)) {
			return 0;
		}
		break;
	case PDUTypeDetonation:
		if (!xdr_dis_detonation_pdu(xdrs, (dis_detonation_pdu *) objp)) {
			return 0;
		}
		break;
	case PDUTypeCreateEntity:
		if (!xdr_dis_create_entity_pdu(xdrs,
									   (dis_create_entity_pdu *) objp)) {
			return 0;
		}
		break;
	case PDUTypeRemoveEntity:
		if (!xdr_dis_remove_entity_pdu(xdrs,
									   (dis_remove_entity_pdu *) objp)) {
			return 0;
		}
		break;
	case PDUTypeStopFreeze:
		if (!xdr_dis_stop_pdu(xdrs, (dis_stop_pdu *) objp)) {
			return 0;
		}
		break;
	case PDUTypeStartResume:
		if (!xdr_dis_start_pdu(xdrs, (dis_start_pdu *) objp)) {
			return 0;
		}
		break;
	case PDUTypeSetData:
		if (!xdr_dis_set_data_pdu(xdrs, (dis_start_pdu *) objp)) {
			return 0;
		}
		break;
	case PDUTypeComment:
		if (!xdr_dis_comment_pdu(xdrs, (dis_comment_pdu *) objp)) {
			return 0;
		}
		break;
	case PDUTypeEmission:
		if (!xdr_dis_em_emission_pdu(xdrs,
									 (dis_em_emission_pdu *) objp)) {
			return 0;
		}
		break;
	case PDUTypeTransferControl:
		if (!xdr_dis_transfer_control_pdu(xdrs, (dis_transfer_control_pdu *) objp)) {
			return 0;
		}
		break;
	case PDUTypeAcknowledge:
		if (!xdr_dis_acknowledge_pdu(xdrs, (dis_acknowledge_pdu *) objp)) {
			return 0;
		}
		break;
	default:
		return xdr_setError(xdrs, "unknown/unsupported DIS PDU type");
	}
	return 1;
}
