/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

     Splityear  splityear       Split in years
     Splityear  splityearmon    Split in years and month
*/

#include <climits>
#include <cdi.h>

#include "cdo_options.h"
#include "process_int.h"
#include "util_files.h"

#define MAX_YEARS 99999

void *
Splityear(void *process)
{
  CdoStreamID streamID2 = CDO_STREAM_UNDEF;
  int varID;
  int nrecs;
  int levelID;
  int day;
  int ic = 0;
  size_t nmiss;

  cdoInitialize(process);

  if (processSelf().m_ID != 0) cdoAbort("This operator can't be combined with other operators!");

  const bool lcopy = unchangedRecord();

  // clang-format off
  const auto SPLITYEAR    = cdoOperatorAdd("splityear",     0, 10000, nullptr);
  const auto SPLITYEARMON = cdoOperatorAdd("splityearmon",  0,   100, nullptr);
  // clang-format on

  const auto operatorID = cdoOperatorID();
  const auto operintval = cdoOperatorF2(operatorID);

  operatorCheckArgc(0);

  Varray<int> cyear(MAX_YEARS, 0);

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  char filename[8192];
  strcpy(filename, cdoGetObase());
  const int nchars = strlen(filename);

  auto refname = cdoGetStreamName(0);
  char filesuffix[32] = { 0 };
  cdoGenFileSuffix(filesuffix, sizeof(filesuffix), cdoInqFiletype(streamID1), vlistID1, refname);

  Varray<double> array;
  // if ( ! lcopy )
  {
    auto gridsizemax = vlistGridsizeMax(vlistID1);
    if (vlistNumber(vlistID1) != CDI_REAL) gridsizemax *= 2;
    array.resize(gridsizemax);
  }

  VarList varList1;
  varListInit(varList1, vlistID1);

  const auto nvars = vlistNvars(vlistID1);
  int nconst = 0;
  for (varID = 0; varID < nvars; varID++)
    if (varList1[varID].timetype == TIME_CONSTANT) nconst++;

  FieldVector2D vars;
  if (nconst)
    {
      vars.resize(nvars);

      for (varID = 0; varID < nvars; varID++)
        {
          if (vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT)
            {
              const auto gridID = varList1[varID].gridID;
              const auto gridsize = varList1[varID].gridsize;
              const auto nlevels = varList1[varID].nlevels;

              vars[varID].resize(nlevels);

              for (levelID = 0; levelID < nlevels; levelID++)
                {
                  vars[varID][levelID].grid = gridID;
                  vars[varID][levelID].resize(gridsize);
                }
            }
        }
    }

  int index1 = -INT_MAX;
  int index2;
  int year1 = -1, year2;
  int mon1 = -1, mon2;
  int tsID = 0;
  int tsID2 = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      taxisCopyTimestep(taxisID2, taxisID1);

      const auto vdate = taxisInqVdate(taxisID1);
      cdiDecodeDate(vdate, &year2, &mon2, &day);

      if (operatorID == SPLITYEAR)
        {
          if (tsID == 0 || year1 != year2 || (year1 == year2 && mon1 > mon2))
            {
              tsID2 = 0;

              ic = (year1 != year2) ? 0 : ic + 1;
              if (year2 >= 0 && year2 < MAX_YEARS)
                {
                  ic = cyear[year2];
                  cyear[year2]++;
                }

              year1 = year2;

              if (streamID2 != CDO_STREAM_UNDEF) cdoStreamClose(streamID2);

              sprintf(filename + nchars, "%04d", year1);
              if (ic > 0) sprintf(filename + strlen(filename), "_%d", ic + 1);
              if (filesuffix[0]) sprintf(filename + strlen(filename), "%s", filesuffix);

              if (Options::cdoVerbose) cdoPrint("create file %s", filename);

              streamID2 = cdoOpenWrite(filename);
              cdoDefVlist(streamID2, vlistID2);
            }
          mon1 = mon2;
        }
      else if (operatorID == SPLITYEARMON)
        {
          index2 = (vdate / operintval);

          if (tsID == 0 || index1 != index2)
            {
              tsID2 = 0;

              index1 = index2;

              if (streamID2 != CDO_STREAM_UNDEF) cdoStreamClose(streamID2);

              sprintf(filename + nchars, "%04d", index1);
              // if ( ic > 0 ) sprintf(filename+strlen(filename), "_%d", ic+1);
              if (filesuffix[0]) sprintf(filename + strlen(filename), "%s", filesuffix);

              if (Options::cdoVerbose) cdoPrint("create file %s", filename);

              streamID2 = cdoOpenWrite(filename);
              cdoDefVlist(streamID2, vlistID2);
            }
        }

      cdoDefTimestep(streamID2, tsID2);

      if (tsID > 0 && tsID2 == 0 && nconst)
        {
          for (varID = 0; varID < nvars; varID++)
            {
              if (vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT)
                {
                  const auto nlevels = varList1[varID].nlevels;
                  for (levelID = 0; levelID < nlevels; levelID++)
                    {
                      nmiss = vars[varID][levelID].nmiss;
                      cdoDefRecord(streamID2, varID, levelID);
                      cdoWriteRecord(streamID2, vars[varID][levelID].vec_d.data(), nmiss);
                    }
                }
            }
        }

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);
          cdoDefRecord(streamID2, varID, levelID);

          if (lcopy && !(tsID == 0 && nconst))
            {
              cdoCopyRecord(streamID2, streamID1);
            }
          else
            {
              cdoReadRecord(streamID1, array.data(), &nmiss);
              cdoWriteRecord(streamID2, array.data(), nmiss);

              if (tsID == 0 && nconst)
                {
                  if (varList1[varID].timetype == TIME_CONSTANT)
                    {
                      varrayCopy(varList1[varID].gridsize, array, vars[varID][levelID].vec_d);
                      vars[varID][levelID].nmiss = nmiss;
                    }
                }
            }
        }

      tsID2++;
      tsID++;
    }

  cdoStreamClose(streamID1);
  cdoStreamClose(streamID2);

  cdoFinish();

  return nullptr;
}
