/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#include "field_vinterp.h"

void
genind(std::vector<int> &vert_index, Varray<double> &plev, Field3D &full_level, size_t gridsize, bool lreverse)
{
  const auto nplev = plev.size();
  const auto nhlevf = full_level.nlevels;
  if (full_level.memType == MemType::Float)
    genind(vert_index.data(), plev.data(), full_level.vec_f.data(), gridsize, nplev, nhlevf, lreverse);
  else
    genind(vert_index.data(), plev.data(), full_level.vec_d.data(), gridsize, nplev, nhlevf, lreverse);
}

void
genindmiss(std::vector<int> &vert_index, Varray<double> &plev, size_t gridsize, Field &level0, std::vector<size_t> &pnmiss, bool lreverse)
{
  const auto nplev = plev.size();
  if (level0.memType == MemType::Float)
    genindmiss(vert_index.data(), plev.data(), gridsize, nplev, level0.vec_f.data(), pnmiss.data(), lreverse);
  else
    genindmiss(vert_index.data(), plev.data(), gridsize, nplev, level0.vec_d.data(), pnmiss.data(), lreverse);
}

void
vertical_interp_T(size_t nlevels, Field3D &full_level, Field3D &half_level, Field3D &field1, Field3D &field2, Field &sgeopot, std::vector<int> &vert_index, Varray<double> &plev, size_t gridsize)
{
  const auto nplev = plev.size();
  const auto missval = field1.missval;
  if (field1.memType == MemType::Float)
    {
      vertical_interp_T(sgeopot.vec_f.data(), field1.vec_f.data(), field2.vec_f.data(), full_level.vec_f.data(), half_level.vec_f.data(),
                        &vert_index[0], plev.data(), nplev, gridsize, nlevels, missval);
    }
  else
    {
      vertical_interp_T(sgeopot.vec_d.data(), field1.vec_d.data(), field2.vec_d.data(), full_level.vec_d.data(), half_level.vec_d.data(),
                        &vert_index[0], plev.data(), nplev, gridsize, nlevels, missval);
    }
}

void
vertical_interp_Z(size_t nlevels, Field3D &full_level, Field3D &half_level, Field3D &field1, Field3D &field2, Field3D &temp, Field &sgeopot, std::vector<int> &vert_index, Varray<double> &plev, size_t gridsize)
{
  const auto nplev = plev.size();
  const auto missval = field1.missval;
  if (field1.memType == MemType::Float)
    {
      vertical_interp_Z(sgeopot.vec_f.data(), field1.vec_f.data(), field2.vec_f.data(), full_level.vec_f.data(), half_level.vec_f.data(),
                        &vert_index[0], temp.vec_f.data(), plev.data(), nplev, gridsize, nlevels, missval);
    }
  else
    {
      vertical_interp_Z(sgeopot.vec_d.data(), field1.vec_d.data(), field2.vec_d.data(), full_level.vec_d.data(), half_level.vec_d.data(),
                        &vert_index[0], temp.vec_d.data(), plev.data(), nplev, gridsize, nlevels, missval);
    }
}

void
vertical_interp_X(size_t nlevels, Field3D &full_level, Field3D &half_level, Field3D &field1, Field3D &field2, std::vector<int> &vert_index, Varray<double> &plev, size_t gridsize)
{
  const auto nplev = plev.size();
  const auto missval = field1.missval;
  if (field1.memType == MemType::Float)
    {
      const auto &hyb_level = (nlevels == full_level.nlevels) ? full_level.vec_f : half_level.vec_f;
      vertical_interp_X(field1.vec_f.data(), field2.vec_f.data(), hyb_level.data(), vert_index.data(), plev.data(), nplev,
                        gridsize, nlevels, missval);
    }
  else
    {
      const auto &hyb_level = (nlevels == full_level.nlevels) ? full_level.vec_d : half_level.vec_d;
      vertical_interp_X(field1.vec_d.data(), field2.vec_d.data(), hyb_level.data(), vert_index.data(), plev.data(), nplev,
                        gridsize, nlevels, missval);
    }
}

void
vertical_interp_X(size_t nlevels, Field3D &levels, Field3D &field1, Field3D &field2, std::vector<int> &vert_index, Varray<double> &plev, size_t gridsize)
{
  const auto nplev = plev.size();
  const auto missval = field1.missval;
  if (field1.memType == MemType::Float)
    {
      vertical_interp_X(field1.vec_f.data(), field2.vec_f.data(), levels.vec_f.data(), vert_index.data(), plev.data(), nplev,
                        gridsize, nlevels, missval);
    }
  else
    {
      vertical_interp_X(field1.vec_d.data(), field2.vec_d.data(), levels.vec_d.data(), vert_index.data(), plev.data(), nplev,
                        gridsize, nlevels, missval);
    }
}
