/*
 * cowbell
 * Copyright (c) 2005 Brad Taylor
 *
 * This file is part of cowbell.
 *
 * cowbell is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * cowbell is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with cowbell; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using Gtk;
using System;
using System.IO;
using Cowbell.Base;
using System.Collections;

namespace Cowbell.Gui
{
	public class PropertiesDialog : GladeWindow
	{
		/* public methods */
		public PropertiesDialog (ArrayList resultset, int location) : base ("PropertiesDialog.glade", "window")
		{
			this.resultset = resultset;
			this.location = location;

			// XXX: Workaround glade being a dumbass and Gtk# 1.5
			// not having a Button.Image property
			((Image)((HBox)((Alignment)ForwardButton.Children[0]).Children[0]).Children[0]).Stock = Gtk.Stock.GoForward;
			Image.Pixbuf = new Gdk.Pixbuf (null, "gnome-mime-audio.png");

			BackButton.Clicked += new EventHandler (OnBackButtonClicked);
			ForwardButton.Clicked += new EventHandler (OnForwardButtonClicked);
			CloseButton.Clicked += new EventHandler (OnCloseButtonClicked);

			LoadMetadata ();
		}

		/* private fields */
		[Glade.Widget] Dialog window;

		[Glade.Widget] Label ArtistLabel;
		[Glade.Widget] Label AlbumLabel;
		[Glade.Widget] Label TitleLabel;
		[Glade.Widget] Image Image;
		[Glade.Widget] Label GenreLabel;
		[Glade.Widget] Label TrackNumberLabel;
		[Glade.Widget] Label DurationLabel;
		[Glade.Widget] Label BitrateLabel;
		[Glade.Widget] Label FileTypeLabel;
		[Glade.Widget] Label FilenameLabel;
		[Glade.Widget] Label LocationLabel;

		[Glade.Widget] TextView CommentsTextView;
		
		[Glade.Widget] Button BackButton;
		[Glade.Widget] Button ForwardButton;
		[Glade.Widget] Button CloseButton;

		ArrayList resultset;
		int location;

		/* private methods */
		private void LoadMetadata ()
		{
			Song s = (Song)resultset[location];
			
			TitleLabel.Markup = String.Format ("<span weight='bold' size='x-large'>{0}</span>",
			                                   GLib.Markup.EscapeText (s.Title));
			ArtistLabel.Markup = String.Format (Catalog.GetString ("by {0}"),
			                                    GLib.Markup.EscapeText (s.Artist));
			AlbumLabel.Markup = String.Format (Catalog.GetString ("<small>from <i>{0}</i></small>"),
			                                   GLib.Markup.EscapeText (s.Album));
			GenreLabel.Text = s.Genre;
			TrackNumberLabel.Text = s.TrackNumber.ToString ();
			DurationLabel.Text = Utils.HumanReadableTimeSpan (s.Length);
			BitrateLabel.Text = String.Format (Catalog.GetString ("{0} kbps"), s.Bitrate);
			
			string ext = Path.GetExtension (s.Filename);
			if (ext == ".ogg") {
				FileTypeLabel.Text = Catalog.GetString ("Ogg Vorbis Audio");
			} else if (ext == ".mp3") {
				FileTypeLabel.Text = Catalog.GetString ("MP3 Audio");
			} else if (ext == ".flac") {
				FileTypeLabel.Text = Catalog.GetString ("FLAC Audio");
			} else if (ext == ".mpc") {
				FileTypeLabel.Text = Catalog.GetString ("Musepack Audio");
			} else {
				FileTypeLabel.Text = Catalog.GetString ("Audio File");
			}

			FilenameLabel.Text = Path.GetFileName (s.Filename);
			LocationLabel.Text = Path.GetDirectoryName (s.Filename);
			
			CommentsTextView.Buffer.Text = s.Comment;

			ForwardButton.Sensitive = location < resultset.Count - 1;
			BackButton.Sensitive = location > 0;
		}

		private void OnBackButtonClicked (object o, EventArgs args)
		{
			location--;
			LoadMetadata ();
		}

		private void OnForwardButtonClicked (object o, EventArgs args)
		{
			location++;
			LoadMetadata ();
		}
		
		private void OnCloseButtonClicked (object o, EventArgs args)
		{
			window.Destroy ();
		}
	}
}
