/*************************************************************************
* Copyright (C) 2014 Intel Corporation
*
* Licensed under the Apache License,  Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* 	http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law  or agreed  to  in  writing,  software
* distributed under  the License  is  distributed  on  an  "AS IS"  BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the  specific  language  governing  permissions  and
* limitations under the License.
*************************************************************************/

/* 
// 
//  Purpose:
//     Cryptography Primitive.
//     EC over Prime Finite Field (P256r1 precomputed)
// 
// 
*/
#include "owncp.h"
#include "pcpgfpecstuff.h"


#define OPERAND_BITSIZE (256)
#define LEN_P256        (BITS_BNU_CHUNK(OPERAND_BITSIZE))

/* P256 affine point */
typedef struct{
   BNU_CHUNK_T X[LEN_P256];
   BNU_CHUNK_T Y[LEN_P256];
} P256_POINT_AFFINE;

extern const __ALIGN64 P256_POINT_AFFINE ec_p256r1_precomputed[37][64];


#if defined( _IPP_DATA )

#if !defined(_DISABLE_ECP_256R1_HARDCODED_BP_TBL_)
/* see ippcp_baseptbl.cpp test for generation details */

const __ALIGN64 P256_POINT_AFFINE ec_p256r1_precomputed[37][64] = {
/* digit=0 base_pwr=2^0 */
{
   LL(0x18a9143c,0x79e730d4),LL(0x5fedb601,0x75ba95fc),LL(0x77622510,0x79fb732b),LL(0xa53755c6,0x18905f76), LL(0xce95560a,0xddf25357),LL(0xba19e45c,0x8b4ab8e4),LL(0xdd21f325,0xd2e88688),LL(0x25885d85,0x8571ff18),
   LL(0x10ddd64d,0x850046d4),LL(0xa433827d,0xaa6ae3c1),LL(0x8d1490d9,0x73220503),LL(0x3dcf3a3b,0xf6bb32e4), LL(0x61bee1a5,0x2f3648d3),LL(0xeb236ff8,0x152cd7cb),LL(0x92042dbe,0x19a8fb0e),LL(0x0a5b8a3b,0x78c57751),
   LL(0x4eebc127,0xffac3f90),LL(0x087d81fb,0xb027f84a),LL(0x87cbbc98,0x66ad77dd),LL(0xb6ff747e,0x26936a3f), LL(0xc983a7eb,0xb04c5c1f),LL(0x0861fe1a,0x583e47ad),LL(0x1a2ee98e,0x78820831),LL(0xe587cc07,0xd5f06a29),
   LL(0x46918dcc,0x74b0b50d),LL(0xc623c173,0x4650a6ed),LL(0xe8100af2,0x0cdaacac),LL(0x41b0176b,0x577362f5), LL(0xe4cbaba6,0x2d96f24c),LL(0xfad6f447,0x17628471),LL(0xe5ddd22e,0x6b6c36de),LL(0x4c5ab863,0x84b14c39),
   LL(0xc45c61f5,0xbe1b8aae),LL(0x94b9537d,0x90ec649a),LL(0xd076c20c,0x941cb5aa),LL(0x890523c8,0xc9079605), LL(0xe7ba4f10,0xeb309b4a),LL(0xe5eb882b,0x73c568ef),LL(0x7e7a1f68,0x3540a987),LL(0x2dd1e916,0x73a076bb),
   LL(0x3e77664a,0x40394737),LL(0x346cee3e,0x55ae744f),LL(0x5b17a3ad,0xd50a961a),LL(0x54213673,0x13074b59), LL(0xd377e44b,0x93d36220),LL(0xadff14b5,0x299c2b53),LL(0xef639f11,0xf424d44c),LL(0x4a07f75f,0xa4c9916d),
   LL(0xa0173b4f,0x0746354e),LL(0xd23c00f7,0x2bd20213),LL(0x0c23bb08,0xf43eaab5),LL(0xc3123e03,0x13ba5119), LL(0x3f5b9d4d,0x2847d030),LL(0x5da67bdd,0x6742f2f2),LL(0x77c94195,0xef933bdc),LL(0x6e240867,0xeaedd915),
   LL(0x9499a78f,0x27f14cd1),LL(0x6f9b3455,0x462ab5c5),LL(0xf02cfc6b,0x8f90f02a),LL(0xb265230d,0xb763891e), LL(0x532d4977,0xf59da3a9),LL(0xcf9eba15,0x21e3327d),LL(0xbe60bbf0,0x123c7b84),LL(0x7706df76,0x56ec12f2),
   LL(0x264e20e8,0x75c96e8f),LL(0x59a7a841,0xabe6bfed),LL(0x44c8eb00,0x2cc09c04),LL(0xf0c4e16b,0xe05b3080), LL(0xa45f3314,0x1eb7777a),LL(0xce5d45e3,0x56af7bed),LL(0x88b12f1a,0x2b6e019a),LL(0xfd835f9b,0x086659cd),
   LL(0x9dc21ec8,0x2c18dbd1),LL(0x0fcf8139,0x98f9868a),LL(0x48250b49,0x737d2cd6),LL(0x24b3428f,0xcc61c947), LL(0x80dd9e76,0x0c2b4078),LL(0x383fbe08,0xc43a8991),LL(0x779be5d2,0x5f7d2d65),LL(0xeb3b4ab5,0x78719a54),
   LL(0x6245e404,0xea7d260a),LL(0x6e7fdfe0,0x9de40795),LL(0x8dac1ab5,0x1ff3a415),LL(0x649c9073,0x3e7090f1), LL(0x2b944e88,0x1a768561),LL(0xe57f61c8,0x250f939e),LL(0x1ead643d,0x0c0daa89),LL(0xe125b88e,0x68930023),
   LL(0xd2697768,0x04b71aa7),LL(0xca345a33,0xabdedef5),LL(0xee37385e,0x2409d29d),LL(0xcb83e156,0x4ee1df77), LL(0x1cbb5b43,0x0cac12d9),LL(0xca895637,0x170ed2f6),LL(0x8ade6d66,0x28228cfa),LL(0x53238aca,0x7ff57c95),
   LL(0x4b2ed709,0xccc42563),LL(0x856fd30d,0x0e356769),LL(0x559e9811,0xbcbcd43f),LL(0x5395b759,0x738477ac), LL(0xc00ee17f,0x35752b90),LL(0x742ed2e3,0x68748390),LL(0xbd1f5bc1,0x7cd06422),LL(0xc9e7b797,0xfbc08769),
   LL(0xb0cf664a,0xa242a35b),LL(0x7f9707e3,0x126e48f7),LL(0xc6832660,0x1717bf54),LL(0xfd12c72e,0xfaae7332), LL(0x995d586b,0x27b52db7),LL(0x832237c2,0xbe29569e),LL(0x2a65e7db,0xe8e4193e),LL(0x2eaa1bbb,0x152706dc),
   LL(0xbc60055b,0x72bcd8b7),LL(0x56e27e4b,0x03cc23ee),LL(0xe4819370,0xee337424),LL(0x0ad3da09,0xe2aa0e43), LL(0x6383c45d,0x40b8524f),LL(0x42a41b25,0xd7663554),LL(0x778a4797,0x64efa6de),LL(0x7079adf4,0x2042170a),
   LL(0x0bc6fb80,0x808b0b65),LL(0x3ffe2e6b,0x5882e075),LL(0x2c83f549,0xd5ef2f7c),LL(0x9103b723,0x54d63c80), LL(0x52a23f9b,0xf2f11bd6),LL(0x4b0b6587,0x3670c319),LL(0xb1580e9e,0x55c4623b),LL(0x01efe220,0x64edf7b2),
   LL(0xd53c5c9d,0x97091dcb),LL(0xac0a177b,0xf17624b6),LL(0x2cfe2dff,0xb0f13975),LL(0x6c7a574e,0xc1a35c0a), LL(0x93e79987,0x227d3146),LL(0xe89cb80e,0x0575bf30),LL(0x0d1883bb,0x2f4e247f),LL(0x3274c3d0,0xebd51226),
   LL(0x56ada97a,0x5f3e51c8),LL(0x8f8b403e,0x4afc964d),LL(0x412e2979,0xa6f247ab),LL(0x6f80ebda,0x675abd1b), LL(0x5e485a1d,0x66a2bd72),LL(0x8f4f0b3c,0x4b2a5caf),LL(0x1b847bba,0x2626927f),LL(0x0502394d,0x6c6fc7d9),
   LL(0xa5659ae8,0xfea912ba),LL(0x25e1a16e,0x68363aba),LL(0x752c41ac,0xb8842277),LL(0x2897c3fc,0xfe545c28), LL(0xdc4c696b,0x2d36e9e7),LL(0xfba977c5,0x5806244a),LL(0xe39508c1,0x85665e9b),LL(0x6d12597b,0xf720ee25),
   LL(0xd2337a31,0x8a979129),LL(0x0f862bdc,0x5916868f),LL(0x5dd283ba,0x048099d9),LL(0xfe5bfb4e,0xe2d1eeb6), LL(0x7884005d,0x82ef1c41),LL(0xffffcbae,0xa2d4ec17),LL(0x8aa95e66,0x9161c53f),LL(0xc5fee0d0,0x5ee104e1),
   LL(0xc135b208,0x562e4cec),LL(0x4783f47d,0x74e1b265),LL(0x5a3f3b30,0x6d2a506c),LL(0xc16762fc,0xecead9f4), LL(0xe286e5b9,0xf29dd4b2),LL(0x83bb3c61,0x1b0fadc0),LL(0x7fac29a4,0x7a75023e),LL(0xc9477fa3,0xc086d5f1),
   LL(0x2f6f3076,0x0fc61135),LL(0xe3912a9a,0xc99ffa23),LL(0xd2f8ba3d,0x6a0b0685),LL(0xe93358a4,0xfdc777e8), LL(0x35415f04,0x94a787bb),LL(0x4d23fea4,0x640c2d6a),LL(0x153a35b5,0x9de917da),LL(0x5d5cd074,0x793e8d07),
   LL(0x2de45068,0xf4f87653),LL(0x9e2e1f6e,0x37c7a7e8),LL(0xa3584069,0xd0825fa2),LL(0x1727bf42,0xaf2cea7c), LL(0x9e4785a9,0x0360a4fb),LL(0x27299f4a,0xe5fda49c),LL(0x71ac2f71,0x48068e13),LL(0x9077666f,0x83d0687b),
   LL(0x15d02819,0x6d3883b2),LL(0x40dd9a35,0x6d0d7550),LL(0x1d2b469f,0x61d7cbf9),LL(0x2efc3115,0xf97b232f), LL(0xb24bcbc7,0xa551d750),LL(0x88a1e356,0x11ea4949),LL(0x93cb7501,0x7669f031),LL(0xca737b8a,0x595dc55e),
   LL(0xd837879f,0xa4a319ac),LL(0xed6b67b0,0x6fc1b49e),LL(0x32f1f3af,0xe3959933),LL(0x65432a2e,0x966742eb), LL(0xb4966228,0x4b8dc9fe),LL(0x43f43950,0x96cc6312),LL(0xc9b731ee,0x12068859),LL(0x56f79968,0x7b948dc3),
   LL(0xed1f8008,0x61e4ad32),LL(0xd8b17538,0xe6c9267a),LL(0x857ff6fb,0x1ac7c5eb),LL(0x55f2fb10,0x994baaa8), LL(0x1d248018,0x84cf14e1),LL(0x628ac508,0x5a39898b),LL(0x5fa944f5,0x14fde97b),LL(0xd12e5ac7,0xed178030),
   LL(0x97e2feb4,0x042c2af4),LL(0xaebf7313,0xd36a42d7),LL(0x084ffdd7,0x49d2c9eb),LL(0x2ef7c76a,0x9f8aa54b), LL(0x09895e70,0x9200b7ba),LL(0xddb7fb58,0x3bd0c66f),LL(0x78eb4cbb,0x2d97d108),LL(0xd84bde31,0x2d431068),
   LL(0x172ccd1f,0x4b523eb7),LL(0x30a6a892,0x7323cb28),LL(0xcfe153eb,0x97082ec0),LL(0xf2aadb97,0xe97f6b6a), LL(0xd1a83da1,0x1d3d393e),LL(0x804b2a68,0xa6a7f9c7),LL(0x2d0cb71e,0x4a688b48),LL(0x40585278,0xa9b4cc5f),
   LL(0xcb66e132,0x5e5db46a),LL(0x0d925880,0xf1be963a),LL(0x0317b9e2,0x944a7027),LL(0x48603d48,0xe266f959), LL(0x5c208899,0x98db6673),LL(0xa2fb18a3,0x90472447),LL(0x777c619f,0x8a966939),LL(0x2a3be21b,0x3798142a),
   LL(0x3298b343,0xb4241cb1),LL(0xb44f65a1,0xa3a14e49),LL(0x3ac77acd,0xc5f4d6cd),LL(0x52b6fc3c,0xd0288cb5), LL(0x1c040abc,0xd5cc8c2f),LL(0x06bf9b4a,0xb675511e),LL(0x9b3aa441,0xd667da37),LL(0x51601f72,0x460d45ce),
   LL(0x6755ff89,0xe2f73c69),LL(0x473017e6,0xdd3cf7e7),LL(0x3cf7600d,0x8ef5689d),LL(0xb1fc87b4,0x948dc4f8), LL(0x4ea53299,0xd9e9fe81),LL(0x98eb6028,0x2d921ca2),LL(0x0c9803fc,0xfaecedfd),LL(0x4d7b4745,0xf38ae891),
   LL(0xc5e3a3d8,0xd8c5fccf),LL(0x4079dfbf,0xbefd904c),LL(0xfead0197,0xbc6d6a58),LL(0x695532a4,0x39227077), LL(0xdbef42f5,0x09e23e6d),LL(0x480a9908,0x7e449b64),LL(0xad9a2e40,0x7b969c1a),LL(0x9591c2a4,0x6231d792),
   LL(0x0f664534,0x87151456),LL(0x4b68f103,0x85ceae7c),LL(0x65578ab9,0xac09c4ae),LL(0xf044b10c,0x33ec6868), LL(0x3a8ec1f1,0x6ac4832b),LL(0x5847d5ef,0x5509d128),LL(0x763f1574,0xf909604f),LL(0xc32f63c4,0xb16c4303),
   LL(0x7ca23cd3,0xb6ab2014),LL(0xa391849d,0xcaa7a5c6),LL(0x75678d94,0x5b0673a3),LL(0xdd303e64,0xc982ddd4), LL(0x5db6f971,0xfd7b000b),LL(0x6f876f92,0xbba2cb1f),LL(0x3c569426,0xc77332a3),LL(0x570d74f8,0xa159100c),
   LL(0xdec67ef5,0xfd16847f),LL(0x233e76b7,0x742ee464),LL(0xefc2b4c8,0x0b8e4134),LL(0x42a3e521,0xca640b86), LL(0x8ceb6aa9,0x653a0190),LL(0x547852d5,0x313c300c),LL(0x6b237af7,0x24e4ab12),LL(0x8bb47af8,0x2ba90162),
   LL(0xa8219bb7,0x3d5e58d6),LL(0x1b06c57f,0xc691d0bd),LL(0xd257576e,0x0ae4cb10),LL(0xd54a3dc3,0x3569656c), LL(0x94cda03a,0xe5ebaebd),LL(0x162bfe13,0x934e82d3),LL(0xe251a0c6,0x450ac0ba),LL(0xdd6da526,0x480b9e11),
   LL(0x8cce08b5,0x00467bc5),LL(0x7f178d55,0xb636458c),LL(0xa677d806,0xc5748bae),LL(0xdfa394eb,0x2763a387), LL(0x7d3cebb6,0xa12b448a),LL(0x6f20d850,0xe7adda3e),LL(0x1558462c,0xf63ebce5),LL(0x620088a8,0x58b36143),
   LL(0x4d63c0ee,0x8a2cc3ca),LL(0x0fe948ce,0x51233117),LL(0x222ef33b,0x7463fd85),LL(0x7c603d6c,0xadf0c7dc), LL(0xfe7765e5,0x0ec32d3b),LL(0xbf380409,0xccaab359),LL(0x8e59319c,0xbdaa84d6),LL(0x9c80c34d,0xd9a4c280),
   LL(0xa059c142,0xa9d89488),LL(0xff0b9346,0x6f5ae714),LL(0x16fb3664,0x068f237d),LL(0x363186ac,0x5853e4c4), LL(0x63c52f98,0xe2d87d23),LL(0x81828876,0x2ec4a766),LL(0xe14e7b1c,0x47b864fa),LL(0x69192408,0x0c0bc0e5),
   LL(0xb82e9f3e,0xe4d7681d),LL(0xdf25e13c,0x83200f0b),LL(0x66f27280,0x8909984c),LL(0x75f73227,0x462d7b00), LL(0xf2651798,0xd90ba188),LL(0x36ab1c34,0x74c6e18c),LL(0x5ef54359,0xab256ea3),LL(0xd1aa702f,0x03466612),
   LL(0x2ed22e91,0x624d6049),LL(0x6f072822,0x6fdfe0b5),LL(0x39ce2271,0xeeca1115),LL(0xdb01614f,0x98100a4f), LL(0xa35c628f,0xb6b0daa2),LL(0xc87e9a47,0xb6f94d2e),LL(0x1d57d9ce,0xc6773259),LL(0x03884a7b,0xf70bfeec),
   LL(0xed2bad01,0x5fb35ccf),LL(0x1da6a5c7,0xa155cbe3),LL(0x30a92f8f,0xc2e2594c),LL(0x5bfafe43,0x649c89ce), LL(0xe9ff257a,0xd158667d),LL(0xf32c50ae,0x9b359611),LL(0x906014cf,0x4b00b20b),LL(0x89bc7d3d,0xf3a8cfe3),
   LL(0x248a7d06,0x4ff23ffd),LL(0x878873fa,0x80c5bfb4),LL(0x05745981,0xb7d9ad90),LL(0x3db01994,0x179c85db), LL(0x61a6966c,0xba41b062),LL(0xeadce5a8,0x4d82d052),LL(0xa5e6a318,0x9e91cd3b),LL(0x95b2dda0,0x47795f4f),
   LL(0xd55a897c,0xecfd7c1f),LL(0xb29110fb,0x009194ab),LL(0xe381d3b0,0x5f0e2046),LL(0xa98dd291,0x5f3425f6), LL(0x730d50da,0xbfa06687),LL(0x4b083b7f,0x0423446c),LL(0xd69d3417,0x397a247d),LL(0x387ba42a,0xeb629f90),
   LL(0xd5cd79bf,0x1ee426cc),LL(0x946c6e18,0x0032940b),LL(0x57477f58,0x1b1e8ae0),LL(0x6d823278,0xe94f7d34), LL(0x782ba21a,0xc747cb96),LL(0xf72b33a5,0xc5254469),LL(0xc7f80c81,0x772ef6de),LL(0x2cd9e6b5,0xd73acbfe),
   LL(0x49ee90d9,0x4075b5b1),LL(0xa06e9eba,0x785c339a),LL(0xabf825e0,0xa1030d5b),LL(0xa42931dc,0xcec684c3), LL(0xc1586e63,0x42ab62c9),LL(0x5ab43f2b,0x45431d66),LL(0x55f7835d,0x57c8b2c0),LL(0xc1b7f865,0x033da338),
   LL(0xcaa76097,0x283c7513),LL(0x36c83906,0x0a624fa9),LL(0x715af2c7,0x6b20afec),LL(0xeba78bfd,0x4b969974), LL(0xd921d60e,0x220755cc),LL(0x7baeca13,0x9b944e10),LL(0x5ded93d4,0x04819d51),LL(0x6dddfd27,0x9bbff86e),
   LL(0x77adc612,0x6b344130),LL(0xbbd803a0,0xa7496529),LL(0x6d8805bd,0x1a1baaa7),LL(0x470343ad,0xc8403902), LL(0x175adff1,0x39f59f66),LL(0xb7d8c5b7,0x0b26d7fb),LL(0x529d75e3,0xa875f5ce),LL(0x41325cc2,0x85efc7e9),
   LL(0x1ff6acd3,0x21950b42),LL(0x53dc6909,0xffe70484),LL(0x28766127,0xff4cd0b2),LL(0x4fb7db2b,0xabdbe608), LL(0x5e1109e8,0x837c9228),LL(0xf4645b5a,0x26147d27),LL(0xf7818ed8,0x4d78f592),LL(0xf247fa36,0xd394077e),
   LL(0x488c171a,0x0fb9c2d0),LL(0x13685278,0xa78bfbaa),LL(0xd5b1fa6a,0xedfbe268),LL(0x2b7eaba7,0x0dceb8db), LL(0x9ae2b710,0xbf9e8089),LL(0xa4449c96,0xefde7ae6),LL(0xcc143a46,0x43b7716b),LL(0xc3628c13,0xd7d34194),
   LL(0x3b3f64c9,0x508cec1c),LL(0x1e5edf3f,0xe20bc0ba),LL(0x2f4318d4,0xda1deb85),LL(0x5c3fa443,0xd20ebe0d), LL(0x73241ea3,0x370b4ea7),LL(0x5e1a5f65,0x61f1511c),LL(0x82681c62,0x99a5e23d),LL(0xa2f54c2d,0xd731e383),
   LL(0x83445904,0x2692f36e),LL(0xaf45f9c0,0x2e0ec469),LL(0xc67528b7,0x905a3201),LL(0xd0e5e542,0x88f77f34), LL(0x5864687c,0xf67a8d29),LL(0x22df3562,0x23b92eae),LL(0x9bbec39e,0x5c27014b),LL(0x9c0f0f8d,0x7ef2f226),
   LL(0x546c4d8d,0x97359638),LL(0x92f24679,0x5f9c3fc4),LL(0xa8c8acd9,0x912e8bed),LL(0x306634b0,0xec3a318d), LL(0xc31cb264,0x80167f41),LL(0x522113f2,0x3db82f6f),LL(0xdcafe197,0xb155bcd2),LL(0x43465283,0xfba1da59),
   LL(0xb212cf53,0xa0425b8e),LL(0xf8557c5f,0x4f2e512e),LL(0x25c4d56c,0xc1286ff9),LL(0xee26c851,0xbb8a0fea), LL(0xe7d6107e,0xc28f70d2),LL(0xe76265aa,0x7ee0c444),LL(0x1d1936b1,0x3df277a4),LL(0xea9595eb,0x1a556e3f),
   LL(0xe7305683,0x258bbbf9),LL(0x07ef5be6,0x31eea5bf),LL(0x46c814c1,0x0deb0e4a),LL(0xa7b730dd,0x5cee8449), LL(0xa0182bde,0xeab495c5),LL(0x9e27a6b4,0xee759f87),LL(0x80e518ca,0xc2cf6a68),LL(0xf14cf3f4,0x25e8013f),
   LL(0x7e8d7a14,0x8fc44140),LL(0x9556f36a,0xbb1ff3ca),LL(0x14600044,0x6a844385),LL(0x7451ae63,0xba3f0c4a), LL(0x1f9af32a,0xdfcac25b),LL(0xb1f2214b,0x01e0db86),LL(0xa4b596ac,0x4e9a5bc2),LL(0x026c2c08,0x83927681),
   LL(0x7acaca28,0x3ec832e7),LL(0xc7385b29,0x1bfeea57),LL(0xfd1eaf38,0x068212e3),LL(0x6acf8ccc,0xc1329830), LL(0x2aac9e59,0xb909f2db),LL(0xb661782a,0x5748060d),LL(0xc79b7a01,0xc5ab2632),LL(0x00017626,0xda44c6c6),
   LL(0xa7ea82f0,0xf26c00e8),LL(0xe4299aaf,0x99cac80d),LL(0x7ed78be1,0xd66fe3b6),LL(0x648d02cd,0x305f725f), LL(0x623fb21b,0x33ed1bc4),LL(0x7a6319ad,0xfa70533e),LL(0xbe5ffb3e,0x17ab562d),LL(0x56674741,0x06374994),
   LL(0x5c46aa8e,0x69d44ed6),LL(0xa8d063d1,0x2100d5d3),LL(0xa2d17c36,0xcb9727ea),LL(0x8add53b7,0x4c2bab1b), LL(0x15426704,0xa084e90c),LL(0xa837ebea,0x778afcd3),LL(0x7ce477f8,0x6651f701),LL(0x46fb7a8b,0xa0624998),
   LL(0xed8a6e19,0xdc1e6828),LL(0x4189d9c7,0x33fc2336),LL(0x671c39bc,0x026f8fe2),LL(0xbc6f9915,0xd40c4ccd), LL(0xf80e75ca,0xafa135bb),LL(0x22adff2c,0x12c651a0),LL(0x4f51ad96,0xc40a04bd),LL(0xbbe4e832,0x04820109),
   LL(0x7f4c04cc,0x3667eb1a),LL(0xa9404f84,0x59556621),LL(0x7eceb50a,0x71cdf653),LL(0x9b8335fa,0x994a44a6), LL(0xdbeb9b69,0xd7faf819),LL(0xeed4350d,0x473c5680),LL(0xda44bba2,0xb6658466),LL(0x872bdbf3,0x0d1bc780),
   LL(0xa1962f91,0xe535f175),LL(0xed58f5a7,0x6ed7e061),LL(0x2089a233,0x177aa4c0),LL(0xe539b413,0x0dbcb03a), LL(0xbb32e38e,0xe3dc424e),LL(0x6806701e,0x6472e5ef),LL(0x814be9ee,0xdd47ff98),LL(0x35ace009,0x6b60cfff),
   LL(0x9ff91fe5,0xb8d3d931),LL(0xf0518eed,0x039c4800),LL(0x9182cb26,0x95c37632),LL(0x82fc568d,0x0763a434), LL(0x383e76ba,0x707c04d5),LL(0x824e8197,0xac98b930),LL(0x91230de0,0x92bf7c8f),LL(0x40959b70,0x90876a01),
   LL(0x05968b80,0xdb6d96f3),LL(0x089f73b9,0x380a0913),LL(0xc2c61e01,0x7da70b83),LL(0x569b38c7,0x95fb8394), LL(0x80edfe2f,0x9a3c6512),LL(0x8faeaf82,0x8f726bb9),LL(0x78424bf8,0x8010a4a0),LL(0x0e844970,0x29672044),
},
/* digit=1 base_pwr=2^7 */
{
   LL(0x7a2ad62a,0x63c5cb81),LL(0xac62ff54,0x7ef2b6b9),LL(0xb3ad9db5,0x3749bba4),LL(0x46d5a617,0xad311f2c), LL(0xc2ff3b6d,0xb77a8087),LL(0x367834ff,0xb46feaf3),LL(0x75d6b138,0xf8aa266d),LL(0xec008188,0xfa38d320),
   LL(0x696946fc,0x486d8ffa),LL(0xb9cba56d,0x50fbc6d8),LL(0x90f35a15,0x7e3d423e),LL(0xc0dd962c,0x7c3da195), LL(0x3cfd5d8b,0xe673fdb0),LL(0x889dfca5,0x0704b7c2),LL(0xf52305aa,0xf6ce581f),LL(0x914d5e53,0x399d49eb),
   LL(0x6ec293cd,0x380a496d),LL(0x8e7051f5,0x733dbda7),LL(0xb849140a,0x037e388d),LL(0x5946dbf6,0xee4b32b0), LL(0xcae368d1,0xb1c4fda9),LL(0xfdb0b2f3,0x5001a7b0),LL(0x2e3ac46e,0x6df59374),LL(0x39b3e656,0x4af675f2),
   LL(0x39949296,0x44e38110),LL(0x361db1b5,0x5b63827b),LL(0x206eaff5,0x3e5323ed),LL(0xc21f4290,0x942370d2), LL(0xe0d985a1,0xf2caaf2e),LL(0x7239846d,0x192cc64b),LL(0xae6312f8,0x7c0b8f47),LL(0x96620108,0x7dc61f91),
   LL(0xc2da7de9,0xb830fb5b),LL(0x0ff8d3be,0xd0e643df),LL(0x188a9641,0x31ee77ba),LL(0xbcf6d502,0x4e8aa3aa), LL(0x9a49110f,0xf9fb6532),LL(0x2dd6b220,0xd18317f6),LL(0x52c3ea5a,0x7e3ced41),LL(0x7d579c4a,0x0d296a14),
   LL(0xed4c3717,0x35d6a53e),LL(0x3d0ed2a3,0x9f8240cf),LL(0xe5543aa5,0x8c0d4d05),LL(0xdd33b4b4,0x45d5bbfb), LL(0x137fd28e,0xfa04cc73),LL(0xc73b3ffd,0x862ac6ef),LL(0x31f51ef2,0x403ff9f5),LL(0xbc73f5a2,0x34d5e0fc),
   LL(0x08913f4f,0xf2526820),LL(0xeac93d95,0xea20ed61),LL(0x6ca6b26c,0x51ed38b4),LL(0xea4327b0,0x8662dcbc), LL(0x725d2aaa,0x6daf295c),LL(0x8e52dcda,0xbad2752f),LL(0x0b17dacc,0x2210e721),LL(0xd51e8232,0xa37f7912),
   LL(0x44cc3add,0x4f7081e1),LL(0x87be82cf,0xd5ffa1d6),LL(0x0edd6472,0x89890b6c),LL(0x3ed17863,0xada26e1a), LL(0x63483caa,0x276f2715),LL(0x2f6077fd,0xe6924cd9),LL(0x0a466e3c,0x05a7fe98),LL(0xb1902d1f,0xf1c794b0),
   LL(0x82a8042c,0xe5213688),LL(0xcd278298,0xd931cfaf),LL(0xf597a740,0x069a0ae0),LL(0xeb59107c,0x0adbb3f3), LL(0x5eaa8eb8,0x983e951e),LL(0x11b48e78,0xe663a8b5),LL(0x8a03f2c5,0x1631cc0d),LL(0x11e271e2,0x7577c11e),
   LL(0x08369a90,0x33b2385c),LL(0x190eb4f8,0x2990c59b),LL(0xc68eac80,0x819a6145),LL(0x2ec4a014,0x7a786d62), LL(0x20ac3a8d,0x33faadbe),LL(0x5aba2d30,0x31a21781),LL(0xdba4f565,0x209d2742),LL(0x55aa0fbb,0xdb2ce9e3),
   LL(0x168984df,0x8cef334b),LL(0x33879638,0xe81dce17),LL(0x263720f0,0xf6e6949c),LL(0xf593cbec,0x5c56feaf), LL(0xfde58c84,0x8bff5601),LL(0x2eccb314,0x74e24117),LL(0x4c9a8a78,0xbcf01b61),LL(0x544c9868,0xa233e35e),
   LL(0x8bd7aff1,0xb3156bf3),LL(0x1d81b146,0x1b5ee4cb),LL(0xd628a915,0x7ba1ac41),LL(0xfd89699e,0x8f3a8f9c), LL(0xa0748be7,0x7329b9c9),LL(0xa92e621f,0x1d391c95),LL(0x4d10a837,0xe51e6b21),LL(0x4947b435,0xd255f53a),
   LL(0xf1788ee3,0x07669e04),LL(0xa86938a2,0xc14f27af),LL(0xe93a01c0,0x8b47a334),LL(0xd9366808,0xff627438), LL(0xca2a5965,0x7a0985d8),LL(0xd6e9b9b3,0x3d9a5542),LL(0x4cf972e8,0xc23eb80b),LL(0x4fdf72fd,0x5c1c33bb),
   LL(0x74a86108,0x0c4a58d4),LL(0xee4c5d90,0xf8048a8f),LL(0xe86d4c80,0xe3c7c924),LL(0x056a1e60,0x28c889de), LL(0xb214a040,0x57e2662e),LL(0x37e10347,0xe8c48e98),LL(0x80ac748a,0x87742862),LL(0x186b06f2,0xf1c24022),
   LL(0x5f74040a,0xac2dd4c3),LL(0xfceac957,0x409aeb71),LL(0x55c4ec23,0x4fbad782),LL(0x8a7b76ec,0xb359ed61), LL(0xed6f4a60,0x12744926),LL(0x4b912de3,0xe21e8d7f),LL(0xfc705a59,0xe2575a59),LL(0xed2dbc0e,0x72f1d4de),
   LL(0xeb7926b8,0x3d2b24b9),LL(0xcdbe5509,0xbff88cb3),LL(0xe4dd640b,0xd0f399af),LL(0x2f76ed45,0x3c5fe130), LL(0x3764fb3d,0x6f3562f4),LL(0x3151b62d,0x7b5af318),LL(0xd79ce5f3,0xd5bd0bc7),LL(0xec66890f,0xfdaf6b20),
   LL(0x6063540c,0x735c67ec),LL(0xe5f9cb8f,0x50b259c2),LL(0x3f99c6ab,0xb8734f9a),LL(0xa3a7bc85,0xf8cc13d5), LL(0xc5217659,0x80c1b305),LL(0x4ec12a54,0xfe5364d4),LL(0x681345fe,0xbd87045e),LL(0x582f897f,0x7f8efeb1),
   LL(0xd5923359,0xe8cbf1e5),LL(0x539b9fb0,0xdb0cea9d),LL(0x49859b98,0x0c5b34cf),LL(0xa4403cc6,0x5e583c56), LL(0xd48185b7,0x11fc1a2d),LL(0x6e521787,0xc93fbc7e),LL(0x05105b8b,0x47e7a058),LL(0xdb8260c8,0x7b4d4d58),
   LL(0x46eb842a,0xe33930b0),LL(0x7bdae56d,0x8e844a9a),LL(0x13f7fdfc,0x34ef3a9e),LL(0x636ca176,0xb3768f82), LL(0x4e09e61c,0x2821f4e0),LL(0xa0c7cddc,0x414dc3a1),LL(0x54945fcd,0xd5379437),LL(0xb3555ff1,0x151b6eef),
   LL(0x6339c083,0xb31bd613),LL(0xdfb64701,0x39ff8155),LL(0xe29604ab,0x7c3388d2),LL(0xa6b10442,0x1e19084b), LL(0xeccd47ef,0x17cf54c0),LL(0x4a5dfb30,0x89693385),LL(0x47daf9f6,0x69d023fb),LL(0x7d91d959,0x9222840b),
   LL(0x803bac62,0x439108f5),LL(0x379bd45f,0x0b7dd91d),LL(0xca63c581,0xd651e827),LL(0x509c104f,0x5c5d75f6), LL(0x1f2dc308,0x7d5fc738),LL(0xd98454be,0x20faa7bf),LL(0xa517b031,0x95374bee),LL(0x642692ac,0xf036b9b1),
   LL(0x39842194,0xc5106109),LL(0x49d05295,0xb7e2353e),LL(0xefb42ee0,0xfc8c1d5c),LL(0x08ce811c,0xe04884eb), LL(0x7419f40e,0xf1f75d81),LL(0xa995c241,0x5b0ac162),LL(0xc4c55646,0x120921bb),LL(0x8d33cf97,0x713520c2),
   LL(0xe98c5100,0xb4a65a5c),LL(0x2ddd0f5a,0x6cec871d),LL(0x9ba2e78b,0x251f0b7f),LL(0xce3a2a5f,0x224a8434), LL(0x25f5c46f,0x26827f61),LL(0x48545ec0,0x6a22bedc),LL(0xb1bb5cdc,0x25ae5fa0),LL(0xfcb9b98f,0xd693682f),
   LL(0x91e5d7d3,0x32027fe8),LL(0x73a07678,0xf14b7d17),LL(0xc0dfdd61,0xf88497b3),LL(0x2a8c4f48,0xf7c2eec0), LL(0x3756e621,0xaa5573f4),LL(0x1825b948,0xc013a240),LL(0x63878572,0x1c03b345),LL(0x653a4184,0xa0472bea),
   LL(0x0ac69a80,0xf4222e27),LL(0xf51e54f6,0x34096d25),LL(0x8fffa591,0x00a648cb),LL(0x69b6527f,0x4e87acdc), LL(0xe285ccb4,0x0575e037),LL(0x50ddcf52,0x188089e4),LL(0x870ff719,0xaa96c9a8),LL(0x1fc7e369,0x74a56cd8),
   LL(0x1726931a,0x41d04ee2),LL(0x3660ecfd,0x0bbbb2c8),LL(0x24818e18,0xa6ef6de5),LL(0xe7d57887,0xe421cc51), LL(0xbea87be6,0xf127d208),LL(0xb1cdd682,0x16a475d3),LL(0x439b63f7,0x9db1b684),LL(0xf0f113b6,0x5359b3db),
   LL(0x8bf06e31,0xdfccf1de),LL(0xdd383901,0x1fdf8f44),LL(0x5017e7d2,0x10775cad),LL(0x58d11eef,0xdfc3a597), LL(0xb1ecff10,0x6ec9c8a0),LL(0x28400549,0xee6ed6cc),LL(0x1b4f8d73,0xb5ad7bae),LL(0xe00aaab9,0x61b4f11d),
   LL(0xd4eff2d7,0x7b32d69b),LL(0x4288b60f,0x88ae6771),LL(0x37a1e723,0x159461b4),LL(0x570aae8c,0x1f3d4789), LL(0x7f9871da,0x869118c0),LL(0xf635e278,0x35fbda78),LL(0xe1541dac,0x738f3641),LL(0xc0dae45f,0x6794b13a),
   LL(0x09cc0917,0x065064ac),LL(0xc68540fd,0x27c53729),LL(0xef227671,0x0d2d4c8e),LL(0xa1785a04,0xd23a9f80), LL(0x52650359,0x98c59528),LL(0x74a1acad,0xfa09ad01),LL(0x0b55bf5c,0x082d5a29),LL(0x419b8084,0xa40f1c67),
   LL(0xdcc18770,0x3a5c752e),LL(0x8825c3a5,0x4baf1f2f),LL(0x21b153ed,0xebd63f74),LL(0xb2f64723,0xa2383e47), LL(0x2646d19a,0xe7bf620a),LL(0x03c83ffd,0x56cb44ec),LL(0x4f6be9f1,0xaf7267c9),LL(0xc06bb5e9,0x8b2dfd7b),
   LL(0xa672c5c7,0xb87072f2),LL(0x0d53c5e2,0xeacb11c8),LL(0xff435932,0x22dac29d),LL(0x4408693c,0x37bdb99d), LL(0x2899c20f,0xf6e62fb6),LL(0x447ece24,0x3535d512),LL(0xff577ce3,0xfbdc6b88),LL(0x190575f2,0x726693bd),
   LL(0xab4b35a2,0x6772b0e5),LL(0xf5eeaacf,0x1d8b6001),LL(0x795b9580,0x728f7ce4),LL(0x41fb81da,0x4a20ed2a), LL(0x4fec01e6,0x9f685cd4),LL(0xa7ff50ad,0x3ed7ddcc),LL(0x0c2d97fd,0x460fd264),LL(0xeb82f4f9,0x3a241426),
   LL(0x6a8ea820,0x17d1df2c),LL(0xf22cc254,0xb2b50d3b),LL(0xb7291426,0x03856cba),LL(0x04f5ee39,0x87fd26ae), LL(0x02bee4ba,0x9cb696cc),LL(0x06820fd6,0x53121804),LL(0x0212e985,0xa5dfc269),LL(0x160f9a09,0x666f7ffa),
   LL(0xbccd9617,0xc503cd33),LL(0xba7730a3,0x365dede4),LL(0x5ddb0786,0x798c6355),LL(0xfc9cd3bc,0xa6c3200e), LL(0xe5e35efd,0x060ffb2c),LL(0x5555a1c1,0x99a4e25b),LL(0xf70b3751,0x11d95375),LL(0x160e1bf6,0x0a57354a),
   LL(0xf8e4b065,0xecb3ae4b),LL(0x2e53022b,0x07a834c4),LL(0x8692ed96,0x1cd300b3),LL(0x61ee14ec,0x16a6f792), LL(0x6a8649ed,0x8f1063c6),LL(0x869f3e14,0xfbcdfcfe),LL(0x00a7b3ec,0x2cfb97c1),LL(0x7130c2f1,0xcea49b3c),
   LL(0xe9d96488,0x462d044f),LL(0x8182a0c1,0x4b53d52e),LL(0x0391e9e9,0x84b6ddd3),LL(0xb1741a09,0x80ab7b48), LL(0x27d3317f,0xec0e15d4),LL(0x1a64671e,0x8dfc1ddb),LL(0xd49c5b92,0x93cc5d5f),LL(0x3674a331,0xc995d53d),
   LL(0x090090ae,0x302e41ec),LL(0xedb06830,0x2278a0cc),LL(0xfbc99690,0x1d025932),LL(0xb80d68da,0x0c32fbd2), LL(0xf341a6c1,0xd79146da),LL(0x1bef68a0,0xae0ba139),LL(0x8d774b3a,0xc6b8a563),LL(0x880ba4d7,0x1cf307bd),
   LL(0x19803511,0xc033bdc7),LL(0x8888c3be,0xa9f97b3b),LL(0x85c6d05e,0x3d68aebc),LL(0x193919eb,0xc3b88a9d), LL(0xc48b0ee3,0x2d300748),LL(0x07a746c1,0x7506bc7c),LL(0x6e6d57f3,0xfc48437c),LL(0xcfeaa91a,0x5bd71587),
   LL(0xc1bc5225,0xa4ed0408),LL(0x2719226d,0xd0b946db),LL(0x758d2d43,0x109ecd62),LL(0x2751759b,0x75c8485a), LL(0x9ce4177a,0xb0b75f49),LL(0x79c10c3d,0x4fa61a1e),LL(0xa167fcd7,0xc062d300),LL(0x750f0fa8,0x4df3874c),
   LL(0x83dfedc9,0x29ae2cf9),LL(0x8d87631a,0xf8437134),LL(0x7429c8d2,0xaf571711),LL(0x146d9272,0x18d15867), LL(0x69769bb7,0x83053ecf),LL(0xc479ab82,0xc55eb856),LL(0x21b0f4b2,0x5ef7791c),LL(0x3d491525,0xaa5956ba),
   LL(0x9fe20eba,0x407a96c2),LL(0xe52a5ad3,0xf27168bb),LL(0xbf1d9d89,0x43b60ab3),LL(0x710e727a,0xe45c51ef), LL(0x099b4221,0xdfca5276),LL(0x2557a159,0x8dc6407c),LL(0x91035895,0x0ead8335),LL(0x9c55dc32,0x0a9db957),
   LL(0xdf61bc76,0xe40736d3),LL(0x3f778cdb,0x13a619c0),LL(0xc56ea28f,0x6dd921a4),LL(0x2fa647b4,0x76a52433), LL(0xac5bdc5d,0x23591891),LL(0xbac7dc01,0xff4a1a72),LL(0x62df8453,0x9905e261),LL(0xe63b265f,0x3ac045df),
   LL(0xad53dba7,0x8a3f341b),LL(0x837b625a,0x8ec269cc),LL(0x3ae31189,0xd71a2782),LL(0x55e96120,0x8fb4f9a3), LL(0xff9875cf,0x804af823),LL(0x5d442a9b,0x23224f57),LL(0xecc62679,0x1c4d3b9e),LL(0xa0e7ddb1,0x91da22fb),
   LL(0x6c04a661,0xa370324d),LL(0x5e376d17,0x9710d3b6),LL(0x3044e357,0xed8c98f0),LL(0x6422701c,0xc364ebbe), LL(0x7733d61c,0x347f5d51),LL(0xcea826c3,0xd55644b9),LL(0x55a25548,0x80c6e0ad),LL(0x844220a7,0x0aa7641d),
   LL(0x31810660,0x1438ec81),LL(0xde4b4043,0x9dfa6507),LL(0xcc3e0273,0x10b515d8),LL(0x28d8cfb2,0x1b6066dd), LL(0x9c9efebd,0xd3b04591),LL(0xa21c1ff4,0x425d4bdf),LL(0xd57607d3,0x5fe5af19),LL(0x54481084,0xbbf773f7),
   LL(0x94b03ed1,0x8435bd69),LL(0x634cc546,0xd9ad1de3),LL(0x00e420ca,0x2cf423fc),LL(0xa03096dd,0xeed26d80), LL(0xa4db09d2,0xd7f60be7),LL(0x960622f7,0xf47f569d),LL(0x7296c729,0xe5925fd7),LL(0x26ca2715,0xeff2db26),
   LL(0xb913e759,0xa6fcd014),LL(0x8ff4de93,0x53da4786),LL(0xc32068e1,0x14616d79),LL(0xccdf352e,0xb187d664), LL(0x1dc90b59,0xf7afb650),LL(0x7daa1b26,0x8170e943),LL(0x700c0a84,0xc8e3bdd8),LL(0x6482bdfa,0x6e8d345f),
   LL(0xc5c5ea50,0x84cfbfa1),LL(0x67960681,0xd3baf14c),LL(0x0dd50942,0x26398403),LL(0x4716a663,0xe4b7839c), LL(0xe7de6dc0,0xd5f1f794),LL(0x622aa7ce,0x5cd0f4d4),LL(0x59acfeec,0x5295f3f1),LL(0x953e0607,0x8d933552),
   LL(0x776c5722,0xc7db8ec5),LL(0x2b5f290c,0xdc467e62),LL(0x4ff425a9,0xd4297e70),LL(0x0cf7bb72,0x4be924c1), LL(0xa1892131,0x0d5dc5ae),LL(0xa705c992,0x8bf8a8e3),LL(0x7a305ac5,0x73a0b064),LL(0x9a8c77a8,0x00c9ca4e),
   LL(0x83774bdd,0x5dfee80f),LL(0x85734485,0x63131602),LL(0x914a69a9,0xa1b524ae),LL(0xd4e300d7,0xebc2ffaf), LL(0x7cfa46a5,0x52c93db7),LL(0x21653b50,0x71e6161f),LL(0xa4bc580a,0x3574fc57),LL(0xe1bc1253,0xc09015dd),
   LL(0xd174d7aa,0x4b7b47b2),LL(0xf3a15d04,0x4072d8e8),LL(0xd6fa07ed,0xeeb7d47f),LL(0xedbdafb1,0x6f2b9ff9), LL(0x3760fe8a,0x18c51615),LL(0xf06c6c13,0x7a96e6bf),LL(0x0ea2d071,0x4d7a0410),LL(0x0be2a5ce,0xa1914e9b),
   LL(0xd8a3c5cf,0x5726e357),LL(0x2abb2b13,0x1197ecc3),LL(0x31ae88dd,0x6c0d7f7f),LL(0xfdbb3efe,0x15b20d1a), LL(0x70584039,0xcd06aa26),LL(0xa7dc9747,0x2277c969),LL(0x7855d815,0xbca69587),LL(0x5188b32a,0x899ea238),
   LL(0x760c1c9d,0x37d9228b),LL(0x9b5c18da,0xc7efbb11),LL(0x19f6dbc5,0x7f0d1bc8),LL(0x07e6905b,0x4875384b), LL(0x3ba8cd86,0xc7c50baa),LL(0xc2905de0,0xb0ce40fb),LL(0x7a231952,0x70840673),LL(0xcf43de26,0xa912a262),
   LL(0xeb5b76c1,0x9c38ddcc),LL(0x26fc0ab4,0x746f5285),LL(0xd62c269f,0x52a63a50),LL(0x99458621,0x60049c55), LL(0x3c2f7c9e,0xe7f48f82),LL(0x917d5cf3,0x6bd99043),LL(0x8701f469,0xeb1317a8),LL(0x9a449fe0,0xbd3fe2ed),
   LL(0x12ef3d36,0x421e79ca),LL(0x3e7ea5de,0x9ee3c36c),LL(0xcdff36f7,0xe48198b5),LL(0xc6b82228,0xaff4f967), LL(0xc47adb7e,0x15e19dd0),LL(0x032e7dfa,0x45699b23),LL(0x1fae026a,0x40680c8b),LL(0x550dbf4d,0x5a347a48),
   LL(0x3cef0d7d,0xe652533b),LL(0x2bbb4381,0xd94f7b18),LL(0x0e80f500,0x838752be),LL(0x9e9c9bfb,0x8e6e2488), LL(0x16caca6a,0xc9751697),LL(0x38531ad9,0x866c49d8),LL(0x7151ade1,0xc917e239),LL(0x6037c407,0x2d016ec1),
   LL(0x00eac3f9,0xa407ccc9),LL(0xe2ed4748,0x835f6280),LL(0x1cc98e0d,0xcc54c347),LL(0xdcb572eb,0x0e969937), LL(0x8f30c9cb,0x1b16c8e8),LL(0x373c4661,0xa606ae75),LL(0x35502cab,0x47aa689b),LL(0x4d9bb64f,0xf89014ae),
   LL(0x31c71f7b,0x202f6a9c),LL(0x296ffe5c,0x01f95aa3),LL(0x53cec3a3,0x5fc06014),LL(0x5f498a45,0xeb991237), LL(0x5d91ba87,0xae9a935e),LL(0x0b564a19,0xc6ac6281),LL(0x3bd44e69,0x8a8fe81c),LL(0x9dd11d45,0x7c8b467f),
   LL(0xea5b8e69,0xf772251f),LL(0xc5b75fbc,0xaeecb3bd),LL(0x887ff0e5,0x1aca3331),LL(0x19f0a131,0xbe5d49ff), LL(0xe5c8646f,0x582c13aa),LL(0x20e19980,0xdbaa12e8),LL(0xf7abbd94,0x8f40f31a),LL(0x1dfc7663,0x1f13f5a8),
   LL(0xaceb4fc0,0x5d81f1ee),LL(0x5e6f0f42,0x36256002),LL(0x751370c8,0x4b67d6d7),LL(0x03e80589,0x2608b698), LL(0x05268301,0xcfc0d2fc),LL(0x40309212,0xa6943d39),LL(0x1fd0e1c2,0x192a90c2),LL(0x37f1dc76,0xb209f113),
   LL(0x97bf1298,0xefcc5e06),LL(0x219d639e,0xcbdb6730),LL(0xb81e8c6f,0xd009c116),LL(0x1a7ce2e5,0xa3ffdde3), LL(0xa914d3ba,0xc53fbaaa),LL(0x88df85ee,0x836d500f),LL(0x66ee0751,0xd98dc71b),LL(0x714516fd,0x5a3d7005),
   LL(0x39eedbba,0x21d3634d),LL(0x0455a46d,0x35cd2e68),LL(0xf9d7eb0c,0xc8cafe65),LL(0x00cefb3e,0xbda3ce9e), LL(0x2c9cf7a4,0xddc17a60),LL(0x7bcb8773,0x01572ee4),LL(0x8c7548df,0xa92b2b01),LL(0xa84600e3,0x732fd309),
   LL(0x16543a40,0xe22109c7),LL(0xfede3c6c,0x9acafd36),LL(0x6824e614,0xfb206852),LL(0xda25dca0,0x2a4544a9), LL(0x91d60b06,0x25985262),LL(0x28753545,0x281b7be9),LL(0x90f13b27,0xec667b1a),LL(0x940e2eb4,0x33a83aff),
   LL(0xd5d721d5,0x80009862),LL(0x5bd3a182,0x0c3357a3),LL(0x7aa2cda4,0x27f3a83b),LL(0xf6f83085,0xb58ae74e), LL(0x2e6dad6b,0x2a911a81),LL(0xf43d6c5b,0xde286051),LL(0xf996c4d8,0x4bdccc41),LL(0x0ae1e24e,0xe7312ec0),
},
/* digit=2 base_pwr=2^14 */
{
   LL(0x6e6485b3,0xf8d112e7),LL(0x771c52f8,0x4d3e24db),LL(0x684a2f6d,0x48e3ee41),LL(0x21d95551,0x7161957d), LL(0xcdb12a6c,0x19631283),LL(0x2e50e164,0xbf3fa882),LL(0x3166cc73,0xf6254b63),LL(0xaee8cc38,0x3aefa7ae),
   LL(0x3b36f9fd,0x79b0fe62),LL(0xfde19fc0,0x26543b23),LL(0x958482ef,0x136e64a0),LL(0x9b095825,0x23f63771), LL(0xb6a1142e,0x14cfd596),LL(0x335aac0b,0x5ea6aac6),LL(0xf3081dd5,0x86a0e8bd),LL(0x003dc12a,0x5fb89d79),
   LL(0xf72e34d4,0xf615c33a),LL(0x110eec35,0x0bd9ea40),LL(0xc1dea34e,0x1c12bc5b),LL(0x49ae4699,0x686584c9), LL(0x8c97b942,0x13ad95d3),LL(0x4e5c7562,0x4609561a),LL(0xf2737f89,0x9e94a4ae),LL(0x371c78b6,0xf57594c6),
   LL(0xe3779ee3,0x0f0165fc),LL(0xbd495d9e,0xe00e7f9d),LL(0x20284e7a,0x1fa4efa2),LL(0x47ac6219,0x4564bade), LL(0xc4708e8e,0x90e6312a),LL(0xa71e9adf,0x4f5725fb),LL(0x3d684b9f,0xe95f55ae),LL(0x1e94b415,0x47f7ccb1),
   LL(0x8d946581,0x7322851b),LL(0xbdf4a012,0xf0d13133),LL(0x6584dae0,0xa3510f69),LL(0x3c9f6c6d,0x03a7c171), LL(0xe475381a,0x5be97f38),LL(0x85823334,0xca1ba422),LL(0x0be17dda,0xf83cc5c7),LL(0x0b918c0f,0x158b1494),
   LL(0x522e6b69,0xda3a77e5),LL(0xbbcd6c18,0x69c908c3),LL(0xd924fd56,0x1f1b9e48),LL(0xaa4bb3f7,0x37c64e36), LL(0xee478d7d,0x5a4fdbdf),LL(0x0193f7a0,0xba75c8bc),LL(0x56cd16df,0x84bc1e84),LL(0x46fad151,0x1fb08f08),
   LL(0x842e9f30,0x8a7cabf9),LL(0x5eab83af,0xa331d4bf),LL(0x017f2a6a,0xd272cfba),LL(0x83aba0e3,0x27560abc), LL(0x0e3a6b75,0x94b83387),LL(0x6b9f50f5,0x25c6aea2),LL(0xb5fdf6d0,0x803d691d),LL(0xe6333514,0x03b77509),
   LL(0x61a341c1,0x36178903),LL(0x0cfd6142,0x3604dc60),LL(0x8533316c,0x022295eb),LL(0x44af2922,0x3dbde4ac), LL(0x1c7eef69,0x898afc5d),LL(0xd14f4fa1,0x58896805),LL(0x203c21ca,0x05002160),LL(0x40ef730b,0x6f0d1f30),
   LL(0x196224f8,0x8e8c44d4),LL(0x374d079d,0x75a4ab95),LL(0x7d48f123,0x79085ecc),LL(0x1bf65ad8,0x56f04d31), LL(0xbda602b2,0xe220bf1c),LL(0xf9612c69,0x73ee1742),LL(0x084fd06b,0x76008fc8),LL(0xf11380d1,0x4000ef9f),
   LL(0x12cfe297,0x48201b4b),LL(0x292f74e5,0x3eee129c),LL(0xc9e874e8,0xe1fe114e),LL(0x92c5fc41,0x899b055c), LL(0x3a39c8cf,0x4e477a64),LL(0x78963cc9,0x82f09efe),LL(0xd333f863,0x6fd3fd8f),LL(0xdc949c63,0x85132b2a),
   LL(0x516eb17b,0x7e06a3ab),LL(0xd2c7372b,0x73bec06f),LL(0xba896da6,0xe4f74f55),LL(0x8e9eb40f,0xbb4afef8), LL(0xe61d66b0,0x2d75bec8),LL(0xef29300b,0x02bda4b4),LL(0x026baa5a,0x8bbaa8de),LL(0xa07f4440,0xff54befd),
   LL(0xbe7a2af3,0xbd9b8b1d),LL(0x4fb74a72,0xec51caa9),LL(0x63879697,0xb9937a4b),LL(0xec2687d5,0x7c9a9d20), LL(0x6ef5f014,0x1773e44f),LL(0xe90c6900,0x8abcf412),LL(0x8142161e,0x387bd022),LL(0xfcb6ff2a,0x50393755),
   LL(0xed6def63,0x9813fd56),LL(0x7d53106c,0x53cf6482),LL(0x431f7ac1,0x991a35bd),LL(0x63e65faf,0xf1e274dd), LL(0x44cc7880,0xf63ffa3c),LL(0x7c256981,0x411a426b),LL(0x93a420e0,0xb698b9fd),LL(0xae53f8fe,0x89fdddc0),
   LL(0x32398baa,0x766e0722),LL(0x5cfca031,0x205fee42),LL(0x7a029cf2,0xa49f5341),LL(0x4023890d,0xa88c68b8), LL(0x7337aaa8,0xbc275041),LL(0x0eb384f4,0x9ed364ad),LL(0x29aba92f,0xe0816f85),LL(0x04e38a88,0x2e9e1941),
   LL(0x3dafd2d5,0x57eef44a),LL(0x97ed98d8,0x35d1fae5),LL(0x2307f9b1,0x50628c09),LL(0xd6cba5c6,0x09d84aae), LL(0x88aaa691,0x67071bc7),LL(0xafe6cb03,0x2dea57a9),LL(0x3d78ac01,0xdfe11bb4),LL(0x7fd7aa51,0x7286418c),
   LL(0x77f7195a,0xfabf7709),LL(0xadeb838f,0x8ec86167),LL(0xbb4f012d,0xea1285a8),LL(0x9a3eab3f,0xd6883503), LL(0x309004c2,0xee5d24f8),LL(0x13ffe95e,0xa96e4b76),LL(0xbd223ea4,0x0cdffe12),LL(0xb6739a53,0x8f5c2ee5),
   LL(0xdd968198,0x5cb4aaa5),LL(0x72413a6c,0xfa131c52),LL(0x9536d903,0x53d46a90),LL(0x48606d8e,0xb270f0d3), LL(0xa053a3bc,0x518c7564),LL(0x1a86caef,0x088254b7),LL(0x0ab5efd0,0xb3ba8cb4),LL(0x4605945d,0x5c59900e),
   LL(0xa1887395,0xecace1dd),LL(0x932a65de,0x40960f36),LL(0x3aa95529,0x9611ff5c),LL(0x7c1e5a36,0xc58215b0), LL(0xf0e1a524,0xd48c9b58),LL(0xf590dfb8,0xb406856b),LL(0x9cd95662,0xc7605e04),LL(0xa33ecf82,0x0dd036ee),
   LL(0xc33156b3,0xa50171ac),LL(0x4a80172e,0xf09d24ea),LL(0x76dc8eef,0x4e1f72c6),LL(0x5e3d44ee,0xe60caadc), LL(0x979b1d8f,0x006ef8a6),LL(0x97788d26,0x60908a1c),LL(0x266feec0,0x6e08f95b),LL(0x22e8c94e,0x618427c2),
   LL(0x59145a65,0x3d613339),LL(0xfa406337,0xcd9bc368),LL(0x2d8a52a0,0x82d11be3),LL(0x97a1c590,0xf6877b27), LL(0xf5cbdb25,0x837a819b),LL(0xde090249,0x2a4fd1d8),LL(0x74990e5f,0x622a7de7),LL(0x7945511b,0x840fa5a0),
   LL(0x6558842d,0x30b974be),LL(0x17f3d0a6,0x70df8c64),LL(0x7542e46d,0x7c803520),LL(0xe4ecc823,0x7251fe7f), LL(0x5e9aac9a,0xe59134cb),LL(0xf0045d71,0x11bb0934),LL(0xdbcb1d4e,0x53e5d9b5),LL(0x92defc91,0x8d97a905),
   LL(0x7946d3f9,0xfe289327),LL(0x07472273,0xe132bd24),LL(0x1eb6ae86,0xeeeb510c),LL(0xf0595067,0x777708c5), LL(0x1297029e,0x18e2c8cd),LL(0xbbf9305e,0x2c61095c),LL(0x6b85d6d9,0xe466c258),LL(0xda1ea530,0x8ac06c36),
   LL(0xa1304668,0xa365dc39),LL(0x07f89606,0xe4a9c885),LL(0xacc7228d,0x65a4898f),LL(0x84ca8303,0x3e2347ff), LL(0xea7d23a3,0xa5f6fb77),LL(0x672a71cd,0x2fac257d),LL(0x7e6a44d3,0x6908bef8),LL(0x891d3d7a,0x8ff87566),
   LL(0x6b0cf82e,0xe58e90b3),LL(0x2615b5e7,0x6438d246),LL(0x669c145a,0x07b1f8fc),LL(0x36f1e1cb,0xb0d8b2da), LL(0xd9184c4d,0x54d5dadb),LL(0xf93d9976,0x3dbb18d5),LL(0xd1147d47,0x0a3e0f56),LL(0xa0a48609,0x2afa8c8d),
   LL(0xbc36742c,0x275353e8),LL(0xeea0ed90,0x898f427e),LL(0x3e477b00,0x26f4947e),LL(0x308741e3,0x8ad8848a), LL(0xd74a2a46,0x6c703c38),LL(0x9ba17ba2,0x5e3e05a9),LL(0x4ab9a9e4,0xc1fa6f66),LL(0x3841d6ec,0x474a2d9a),
   LL(0x653ae326,0x871239ad),LL(0xa74cbb43,0x14bcf72a),LL(0x20d4c083,0x8737650e),LL(0x110ed4af,0x3df86536), LL(0xb53ca555,0xd2d86fe7),LL(0xabd5d538,0x688cb00d),LL(0x1ad38468,0xcf81bda3),LL(0xf01167b6,0x7ccfe3cc),
   LL(0x6c4c1fe6,0xcf4f47e0),LL(0x298bbb79,0x557e1f1a),LL(0x30d45a14,0xf93b974f),LL(0x0baf97c4,0x174a1d2d), LL(0xc51fbf53,0x7a003b30),LL(0xee68b225,0xd8940991),LL(0x1c0f4173,0x5b0aa7b7),LL(0xa20a7153,0x975797c9),
   LL(0xe3533d77,0x26e08c07),LL(0x2e341c99,0xd7222e6a),LL(0x8d2dc4ed,0x9d60ec3d),LL(0x7c476cf8,0xbdfe0d8f), LL(0x1d056605,0x1fe59ab6),LL(0x86a8551f,0xa9ea9df6),LL(0x47fb8d8c,0x8489941e),LL(0x4a7f1b10,0xfeb874eb),
   LL(0x7ee0d98f,0xfe5fea86),LL(0xdbf61864,0x201ad34b),LL(0x37c031d4,0x45d8fe47),LL(0x795f0822,0xd5f49fae), LL(0xc7f4a40c,0xdb0fb291),LL(0x730ddd92,0x2e69d9c1),LL(0x49d76987,0x754e1054),LL(0x7662db87,0x8a24911d),
   LL(0x60a71676,0x61fc1810),LL(0xf66a8ad1,0xe852d1a8),LL(0x6417231e,0x172bbd65),LL(0x3babb11f,0x0d6de7bd), LL(0xc8e347f8,0x6fde6f88),LL(0x9bd99cc3,0x1c587547),LL(0x34076950,0x78e54ed0),LL(0x796e83ba,0x97f0f334),
   LL(0x4924867a,0xe4dbe1ce),LL(0x60b84917,0xbd5f51b0),LL(0x3cb09a79,0x37530040),LL(0xff1743d8,0xdb3fe0f8), LL(0x556fa9db,0xed7894d8),LL(0x23412fbf,0xfa262169),LL(0xba7b9291,0x563be0db),LL(0x0c9fb234,0x6ca8b8c0),
   LL(0xbd763802,0xed406aa9),LL(0x65303da1,0xc21486a0),LL(0xc7e62ec4,0x61ae291e),LL(0xdf99333e,0x622a0492), LL(0xbb7a8ee0,0x7fd80c9d),LL(0x6c01aedb,0xdc2ed3bc),LL(0x08be74ec,0x35c35a12),LL(0x469f671f,0xd540cb1a),
   LL(0xcf84f6c7,0xd16ced4e),LL(0x2d090f43,0x8561fb9c),LL(0x6f239db4,0x7e693d79),LL(0x77bd0d94,0xa736f928), LL(0x2c1950ee,0x07b4d929),LL(0x56dc11b3,0xda177543),LL(0x7a6a878e,0xa5dfbbaa),LL(0x4decb08a,0x1c70cb29),
   LL(0x6f0f7c50,0xfba28c8b),LL(0x854dcc6d,0xa8eba2b8),LL(0x36b78642,0x5ff8e89a),LL(0xf6873adf,0x070c1c8e), LL(0x6484d2e4,0xbbd3c371),LL(0x0d414129,0xfb78318f),LL(0x6ad93b0b,0x2621a39c),LL(0xa9e917f7,0x979d74c2),
   LL(0x61fb0428,0xfc195647),LL(0xbee624d4,0x4d78954a),LL(0xb8ae86fd,0xb94896e0),LL(0xc91c8b13,0x6667ac0c), LL(0x43bcf832,0x9f180512),LL(0xa0010137,0xfbadf8b7),LL(0xb3ba8aa7,0xc69b4089),LL(0xe687ce85,0xfac4bacd),
   LL(0x977eab40,0x9164088d),LL(0x2760b390,0x51f4c5b6),LL(0x340dd553,0xd238238f),LL(0xdb1d31c9,0x358566c3), LL(0x5068f5ff,0x3a5ad69e),LL(0xdaff6b06,0xf31435fc),LL(0xd6debff0,0xae549a5b),LL(0x75e01331,0x59e5f0b7),
   LL(0x98559acf,0x5d492fb8),LL(0x4db79b50,0x96018c2e),LL(0x609f66aa,0x55f4a48f),LL(0x4900a14f,0x1943b3af), LL(0x15a40d39,0xc22496df),LL(0x4c20f7c5,0xb2a44684),LL(0x3b98404c,0x76a35afa),LL(0xff5d1b77,0xbec75725),
   LL(0xbea06444,0xb67aa163),LL(0xf724b6f2,0x27e95bb2),LL(0xd238c8ab,0x3c20e3e9),LL(0xddd6ae17,0x1213754e), LL(0x716e0f74,0x8c431020),LL(0xffc095c2,0x6679c82e),LL(0xd0ac2932,0x2eb3adf4),LL(0x01bb7a76,0x2cc970d3),
   LL(0x740f0e66,0x70c71f2f),LL(0x2b6b23cc,0x545c616b),LL(0xb40a8bd7,0x4528cfcb),LL(0x2ab27722,0xff839633), LL(0x025ac99a,0x049127d9),LL(0x2b63e33b,0xd314d4a0),LL(0x28d84519,0xc8c310e7),LL(0xb3bc84ba,0x0fcb8983),
   LL(0x38634818,0x2cc52261),LL(0xb44c2e0b,0x501814f4),LL(0x54dfdba3,0xf7e181aa),LL(0xe759718c,0xcfd58ff0), LL(0xd3b507a8,0xf90cdb14),LL(0xc50bdad8,0x57bd478e),LL(0x50e5f9aa,0x29c197e2),LL(0xe40bc855,0x4db6eef8),
   LL(0xd1fc0654,0x2cc8f21a),LL(0x81269d73,0xc71cc963),LL(0x077f49f9,0xecfbb204),LL(0xca56b793,0xdde92571), LL(0xf97ad8f7,0x9abed6a3),LL(0x924de3bd,0xe6c19d3f),LL(0xa140a800,0x8dce92f4),LL(0x1337af07,0x85f44d1e),
   LL(0x09d64c52,0x5953c08b),LL(0xf5df9749,0xa1b5e49f),LL(0x52735f7d,0x336a8fb8),LL(0x9add676b,0xb332b6db), LL(0xb4511aa4,0x558b88a0),LL(0xdbd5cc55,0x09788752),LL(0xd8cd52bd,0x16b43b9c),LL(0xc2a2696b,0x7f0bc5a0),
   LL(0xc11f61ef,0x146e12d4),LL(0x3a83e79e,0x9ce10754),LL(0x6cbfca15,0x08ec73d9),LL(0x5b49653f,0x09ff29ad), LL(0xe7da946e,0xe31b72bd),LL(0xee80a4f2,0xebf9eb3b),LL(0x17598ce4,0xd1aabd08),LL(0x53f37e80,0x18b5fef4),
   LL(0x5958cd79,0xd5d5cdd3),LL(0x1d373114,0x3580a1b5),LL(0xfa935726,0xa36e4c91),LL(0xef20d760,0xa38c534d), LL(0x2ff5845b,0x7088e40a),LL(0xbd78177f,0xe5bb40bd),LL(0x857f9920,0x4f06a7a8),LL(0xe968f05d,0xe3cc3e50),
   LL(0xe5682d26,0x1d68b7fe),LL(0xaec7f87c,0x5206f76f),LL(0x041951ab,0x41110530),LL(0xd4b5a71a,0x58ec52c1), LL(0x0f75cf9a,0xf3488f99),LL(0xba82d0d5,0xf411951f),LL(0x618895ab,0x27ee75be),LL(0x6d8aab14,0xeae060d4),
   LL(0x7fb54dc2,0x9ae1df73),LL(0x25963649,0x1f3e391b),LL(0xfe055081,0x242ec32a),LL(0x8491c9bd,0x5bd450ef), LL(0x981eb389,0x367efc67),LL(0x3a0550d5,0xed7e1928),LL(0xab3ce75c,0x362e776b),LL(0x1f24c523,0xe890e308),
   LL(0xfeccef76,0xb961b682),LL(0x8bba6d92,0x8b8e11f5),LL(0x2b2375c4,0x8f2ccc4c),LL(0xe2f86cfa,0x0d7f7a52), LL(0x9efe5633,0xfd94d30a),LL(0x5451f934,0x2d8d246b),LL(0x244e6a00,0x2234c6e3),LL(0xddec8c50,0xde2b5b0d),
   LL(0xbf776f5b,0x2ce53c5a),LL(0x60357b05,0x6f724071),LL(0x71bf3f7a,0xb2593717),LL(0x440c4a9f,0x87d2501c), LL(0x87b05340,0x440552e1),LL(0x21624c32,0xb7bf7cc8),LL(0x22facddb,0x4155a6ce),LL(0x889837ef,0x5a4228cb),
   LL(0xfd4fd671,0xef87d6d6),LL(0xc2daa10e,0xa233687e),LL(0x03c0eb96,0x75622244),LL(0x8bf19be6,0x7632d184), LL(0x40735ff4,0x05d0f8e9),LL(0xc00931f1,0x3a3e6e13),LL(0xdafe3f18,0x31ccde6a),LL(0xcfe51207,0xf381366a),
   LL(0x60167d92,0x24c222a9),LL(0x7529f18c,0x62f9d6f8),LL(0x0353b114,0x412397c0),LL(0xef808043,0x334d89dc), LL(0x2a4383ce,0xd9ec63ba),LL(0x5cf92ba0,0xcec8e937),LL(0xc8be74c0,0xfb8b4288),LL(0x105d4391,0x67d6912f),
   LL(0x1b913149,0x7b996c46),LL(0x3a4e02da,0x36aae2ef),LL(0x972de594,0xb68aa003),LL(0x4ec6d545,0x284ec70d), LL(0x61391d54,0xf3d2b2d0),LL(0xfe114e92,0x69c5d5d6),LL(0xb4482dff,0xbe0f00b5),LL(0xf5bf33c5,0xe1596fa5),
   LL(0x96a71cba,0x10595b56),LL(0xfdcadeb7,0x944938b2),LL(0xfccd8471,0xa282da4c),LL(0x0d37bfe1,0x98ec05f3), LL(0x0698304a,0xe171ce1b),LL(0x21bdf79b,0x2d691444),LL(0x1b21dec1,0xd0cd3b74),LL(0x16a15f71,0x712ecd8b),
   LL(0x00fd56e1,0x8d4c00a7),LL(0xf9527c18,0x02ec9692),LL(0x4a3e42e1,0x21c44937),LL(0x1392ae0a,0x9176fbab), LL(0x44b7b618,0x8726f1ba),LL(0xf1de491c,0xb4d7aae9),LL(0x07b582c0,0xf91df7b9),LL(0xef60aa3a,0x7e116c30),
   LL(0x466265d7,0x99270f81),LL(0x4df7adf0,0xb15b6fe2),LL(0xf9738f7f,0xfe33b2d3),LL(0xd6d70f95,0x48553ab9), LL(0xc21e94db,0x2cc72ac8),LL(0xbdc0bbee,0x795ac38d),LL(0x2e40478f,0x0a1be449),LL(0x052bde55,0x81bd3394),
   LL(0x56b3c4f2,0x63c8dbe9),LL(0x904177cc,0x017a99cf),LL(0x4d010fc1,0x947bbddb),LL(0xbb2c9b21,0xacf9b00b), LL(0x47173611,0x2970bc8d),LL(0xac7d756f,0x1a4cbe08),LL(0x67d541a2,0x06d9f4aa),LL(0x59c2cf44,0xa3e8b689),
   LL(0x4d88f1dd,0xaad066da),LL(0x7ad35dea,0xc604f165),LL(0x4478ca67,0x7edc0720),LL(0xba02ce06,0xa10dfae0), LL(0xaf36f4e4,0xeceb1c76),LL(0xaf3f8f48,0x994b2292),LL(0x77c8a68c,0xbf9ed77b),LL(0x51744c9d,0x74f544ea),
   LL(0x8113a757,0x82d05bb9),LL(0x8a9885e4,0x4ef2d2b4),LL(0x1aa7865f,0x1e332be5),LL(0x290d1a52,0x22b76b18), LL(0x44351683,0x308a2310),LL(0xa3f22840,0x9d861896),LL(0x841ed947,0x5959ddcd),LL(0x154b73bf,0x0def0c94),
   LL(0x4c7c15e0,0xf0105417),LL(0x3a277c32,0x539bfb02),LL(0xf9dccf5f,0xe699268e),LL(0x0247a3bd,0x9f5796a5), LL(0x4f157269,0x8b839de8),LL(0x7a30196b,0xc825c1e5),LL(0xdc8a5a91,0x6ef0aabc),LL(0x498b7fe6,0xf4a8ce6c),
   LL(0x70cbac78,0x1cce35a7),LL(0xf6b23958,0x83488e9b),LL(0xd76cb011,0x0341a070),LL(0xae1b2658,0xda6c9d06), LL(0xdd648c52,0xb701fb30),LL(0x52fb9fd1,0x994ca02c),LL(0x6f563086,0x06933117),LL(0x17856bab,0x3d2b8100),
   LL(0x5963a46e,0xe89f48c8),LL(0xa99e61c7,0x658ab875),LL(0x4b8517b4,0x6e296f87),LL(0xfc1bc656,0x36c4fcdc), LL(0xa3906def,0xde5227a1),LL(0x62418945,0x9fe95f57),LL(0xfdd96cde,0x20c91e81),LL(0xda4480de,0x5adbe47e),
   LL(0x396de2b6,0xa009370f),LL(0xf0ecc7bd,0x98583d4b),LL(0xe51d0672,0xf44f6b57),LL(0x556b1984,0x03d6b078), LL(0xb0b64912,0x27dbdd93),LL(0x15687b09,0x9b3a3434),LL(0x51ec20a9,0x0dba6461),LL(0xff28187c,0xec93db7f),
   LL(0x66e48bdd,0x00ff8c24),LL(0x11ccd78e,0x2514f2f9),LL(0xe1250603,0xeba11f4f),LL(0x243fa156,0x8a22cd41), LL(0xb283e4c6,0xa4e58df4),LL(0x8b39783f,0x78c29859),LL(0xa5259809,0x5235aee2),LL(0x0e0227dd,0xc16284b5),
   LL(0x1338830d,0xa5f57916),LL(0xd2123fca,0x6d4b8a6b),LL(0xf9c546f8,0x236ea68a),LL(0xfa608d36,0xc1d36873), LL(0x8d436d13,0xcd76e495),LL(0x8fb080af,0xd4d9c221),LL(0xe8ad3fb5,0x665c1728),LL(0xb3d572e0,0xcf1ebe4d),
   LL(0x584c5e20,0xa7a8746a),LL(0xb9dc7035,0x267e4ea1),LL(0xb9548c9b,0x593a15cf),LL(0x4bd012f3,0x5e6e2135), LL(0x8c8f936e,0xdf31cc6a),LL(0xb5c241dc,0x8af84d04),LL(0x345efb86,0x63990a6f),LL(0xb9b962cb,0x6fef4e61),
},
/* digit=3 base_pwr=2^21 */
{
   LL(0x25722608,0xf6368f09),LL(0x131cf5c6,0x131260db),LL(0xfab4f7ac,0x40eb353b),LL(0x37eee829,0x85c78880), LL(0xc3bdf24e,0x4c1581ff),LL(0xf5c3c5a8,0x5bff75cb),LL(0xa14e6f40,0x35e8c83f),LL(0x0295e0ca,0xb81d1c0f),
   LL(0xf43a730f,0xfcde7cc8),LL(0x33ab590e,0xe89b6f3c),LL(0xad03240b,0xc823f529),LL(0x98bea5db,0x82b79afe), LL(0x962fe5de,0x568f2856),LL(0x60c591f3,0x0c590adb),LL(0x4a28a858,0x1fc74a14),LL(0xb3203f4c,0x3b662498),
   LL(0x6c39765a,0x91e3cf0d),LL(0xac3cca0b,0xa2db3acd),LL(0xcb953b50,0x288f2f08),LL(0xcf43cf1a,0x2414582c), LL(0x60eee9a8,0x8dec8bbc),LL(0x729aa042,0x54c79f02),LL(0x6532f5d5,0xd81cd5ec),LL(0xcf82e15f,0xa672303a),
   LL(0x719c0563,0x376aafa8),LL(0xbc5fc79f,0xcd8ad2dc),LL(0xcb750cd3,0x303fdb9f),LL(0x4418b08e,0x14ff052f), LL(0x3e2d6520,0xf75084cf),LL(0x144ed509,0x7ebdf0f8),LL(0xd3f25b98,0xf43bf0f2),LL(0xa354d837,0x86ad71cf),
   LL(0x26f43572,0xb827fe92),LL(0x5d824758,0xdfd3ab5b),LL(0x539094c1,0x315dd23a),LL(0x66623d68,0x85c0e37a), LL(0x7be19ae0,0x575c7972),LL(0xdf0d36b5,0x616a3396),LL(0x26b1ff7e,0xa1ebb3c8),LL(0x140ad453,0x635b9485),
   LL(0xda430c0b,0x92bf3cda),LL(0x3a96dac6,0x4702850e),LL(0x15ac326a,0xc91cf0a5),LL(0xab8c25e4,0x95de4f49), LL(0xe265c17c,0xb01bad09),LL(0x087b3881,0x24e45464),LL(0xe1fac5ca,0xd43e583c),LL(0x6ead97a6,0xe17cb318),
   LL(0x74dcec46,0x6cc39243),LL(0x54c2b73f,0x33cfc02d),LL(0xf26cd99c,0x82917844),LL(0xd1773f89,0x8819dd95), LL(0x0871f427,0x09572aa6),LL(0xf6f01c34,0x8e0cf365),LL(0xbff1f5af,0x7fa52988),LL(0xe75e8e50,0x4eb357ea),
   LL(0x868af75d,0xd9d0c8c4),LL(0x45c8c7ea,0xd7325cff),LL(0xcc81ecb0,0xab471996),LL(0x611824ed,0xff5d55f3), LL(0x1977a0ee,0xbe314541),LL(0x722038c6,0x5085c4c5),LL(0xf94bb495,0x2d5335bf),LL(0xc8e2a082,0x894ad8a6),
   LL(0xada35438,0x5c3e2341),LL(0x049b8c4e,0xf4a9fc89),LL(0x9f17cf34,0xbeeb355a),LL(0x6c91fe10,0x3f311e0e), LL(0x92ab9891,0xc2d20038),LL(0x3e8ce9a9,0x257bdcc1),LL(0x88c53bee,0x1b2d9789),LL(0xcdba143a,0x927ce89a),
   LL(0x523db280,0xb0a32cca),LL(0x50d43783,0x5c889f8a),LL(0x4897d16f,0x503e04b3),LL(0x08f5f2e8,0x8cdb6e78), LL(0x179c8e74,0x6ab91cf0),LL(0x48211d60,0xd8874e52),LL(0xea851200,0xf948d4d5),LL(0xe6f9840a,0x4076d41e),
   LL(0x47b517ea,0xc20e263c),LL(0x30685e5e,0x79a448fd),LL(0xf90631a0,0xe55f6f78),LL(0xa79e6346,0x88a790b1), LL(0x80969fe8,0x62160c7d),LL(0x41491bb9,0x54f92fd4),LL(0x5c957526,0xa6645c23),LL(0xbea3ce7b,0xf44cc5ae),
   LL(0x8b1e68b7,0xf7628327),LL(0x303f29d3,0xc731ad7a),LL(0x57d03ecb,0xfe5a9ca9),LL(0x41bc97a7,0x96c0d50c), LL(0x9b4f7f24,0xc4669fe7),LL(0x3d9967ef,0xfdd781d8),LL(0x5d2c208d,0x7892c7c3),LL(0xae545cb3,0x8bf64f7c),
   LL(0x467be912,0xc01f862c),LL(0xc73d30cc,0xf4c85ee9),LL(0x6ab83ec7,0x1fa6f4be),LL(0x4e3e3cf9,0xa07a3c1c), LL(0x0c00beb3,0x87f8ef45),LL(0x000d4c3e,0x30e2c2b3),LL(0xfe08bf5b,0x1aa00b94),LL(0x9224ef52,0x32c133aa),
   LL(0x32e5685d,0x38df16bb),LL(0x58e6f544,0x68a9e069),LL(0xcdc5ebc6,0x495aaff7),LL(0x378b135f,0xf894a645), LL(0x09e27ecf,0xf316350a),LL(0x58f7179d,0xeced201e),LL(0xe97861ba,0x2eec273c),LL(0xd693be2e,0x47ec2cae),
   LL(0xf68367ce,0xfa4c97c4),LL(0xbe5a5755,0xe4f47d0b),LL(0xb298a979,0x17de815d),LL(0xc177dc7d,0xd7eca659), LL(0x49ded0a3,0x20fdbb71),LL(0xfb34d3c5,0x4cb2aad4),LL(0x60858a33,0x2cf31d28),LL(0xa24aa40f,0x3b6873ef),
   LL(0x2c11bb37,0x540234b2),LL(0xed4c74a3,0x2d0366dd),LL(0xeec5f25d,0xf9a968da),LL(0x67b63142,0x36601068), LL(0x68d7b6d4,0x07cd6d2c),LL(0x0c842942,0xa8f74f09),LL(0x7768b1ee,0xe2751404),LL(0xfe62aee4,0x4b5f7e89),
   LL(0x89070d26,0xc6a77177),LL(0xdd1c8bc7,0xa1f28e4e),LL(0x469e1f17,0xea5f4f06),LL(0xfbdb78e0,0x78fc242a), LL(0x8b0588f1,0xc9c7c592),LL(0x1535921e,0xb6b7a0fd),LL(0xbde5ae35,0xcc5bdb91),LL(0x12ff1864,0xb42c485e),
   LL(0xdbab98aa,0xa1113e13),LL(0xa17b1024,0xde9d469b),LL(0xc0462d3a,0x23f48b37),LL(0x7c5c078d,0x3752e537), LL(0x15544eb9,0xe3a86add),LL(0x80fba279,0xf013aea7),LL(0xf22001b5,0x8b5bb76c),LL(0xf02891ab,0xe617ba14),
   LL(0x936219d3,0xd39182a6),LL(0xae51cb19,0x5ce1f194),LL(0xbf07a74c,0xc78f8598),LL(0x22cbf1bc,0x6d7158f2), LL(0xe300ce18,0x3b846b21),LL(0x2d11275d,0x35fba630),LL(0xa0239b9b,0x5fe25c36),LL(0xdf05d940,0xd8beb35d),
   LL(0x1f7e320d,0x4db02bb0),LL(0x6da320ea,0x0641c364),LL(0x821389a3,0x6d95fa5d),LL(0x8fcd8e3d,0x92699748), LL(0xceb6c143,0x316fef17),LL(0xd933762b,0x67fcb841),LL(0x118b17f8,0xbb837e35),LL(0x9fd24821,0x4b92552f),
   LL(0x46aca793,0xae6bc70e),LL(0xe579311b,0x1cf0b0e4),LL(0x5802f716,0x8dc631be),LL(0xbddbee4d,0x099bdc6f), LL(0x0caf8b05,0xcc352bb2),LL(0x72d63df2,0xf74d505a),LL(0x91c4f408,0xb9876d4b),LL(0x9e229b2d,0x1ce18473),
   LL(0x83abdb4a,0x49507597),LL(0xdee84b18,0x850fbcb6),LL(0x609e67dc,0x6325236e),LL(0x9336c6d8,0x04d831d9), LL(0xfa12d45d,0x8deaae3b),LL(0x4746e246,0xe425f8ce),LL(0x24f5f31e,0x8004c175),LL(0xad62c3b7,0xaca16d8f),
   LL(0x9152f934,0x0dc15a6a),LL(0xed0e12c1,0xf1235e5d),LL(0xda477dac,0xc33c06ec),LL(0xb2ea0006,0x76be8732), LL(0x0c0cd313,0xcf3f7831),LL(0xa614260d,0x3c524553),LL(0xcab22d15,0x31a756f8),LL(0x77827a20,0x03ee10d1),
   LL(0x1994ef20,0xd1e059b2),LL(0x638ae318,0x2a653b69),LL(0x2f699010,0x70d5eb58),LL(0x09f5f84a,0x279739f7), LL(0x8b799336,0x5da4663c),LL(0x203c37eb,0xfdfdf14d),LL(0xa1dbfb2d,0x32d8a9dc),LL(0x77d48f9b,0xab40cff0),
   LL(0xd20b42d5,0xc018b383),LL(0x9f78845f,0xf9a810ef),LL(0xbdba9df0,0x40af3753),LL(0x131dfdf9,0xb90bdcfc), LL(0xf01ab782,0x18720591),LL(0x6af12a88,0xc823f211),LL(0x0dc14401,0xa51b80f3),LL(0xfb2dfbe3,0xde248f77),
   LL(0x0cafe751,0xef5a44e5),LL(0xd4dcd221,0x73997c9c),LL(0xde854024,0x32fd86d1),LL(0xa09b84bb,0xd5b53adc), LL(0xdcedd8d1,0x008d7a11),LL(0x74b32c84,0x406bd1c8),LL(0x05dde8b1,0x5d4472ff),LL(0xfce2b32f,0x2e25f2cd),
   LL(0x29dfc254,0xbec0dd5e),LL(0x2b98b267,0x4455fcf6),LL(0xc72df2ad,0x0b4d43a5),LL(0x48a75397,0xea70e6be), LL(0x5820f3bf,0x2aad6169),LL(0x9e37f68f,0xf410d2dd),LL(0x7be5ac83,0x70fb7dba),LL(0x36ec3eec,0x636bb645),
   LL(0x9754e21c,0x27104ea3),LL(0x8d63c373,0xbc87a3e6),LL(0x4109db9a,0x483351d7),LL(0x60134da7,0x0fa724e3), LL(0xb0720b16,0x9ff44c29),LL(0x06aceead,0x2dd0cf13),LL(0xe26929a6,0x5942758c),LL(0xb766a92b,0x96c5db92),
   LL(0x5f18395e,0xcec7d4c0),LL(0x1f80d032,0xd3f22744),LL(0xcb86075b,0x7a68b37a),LL(0xafef92db,0x074764dd), LL(0x7bc7f389,0xded1e950),LL(0xb9756460,0xc580c850),LL(0x7da48157,0xaeeec2a4),LL(0x82c587b3,0x3f0b4e7f),
   LL(0xa9f19c53,0x231c6de8),LL(0x6974e34e,0x5717bd73),LL(0xf1508fa9,0xd9e1d216),LL(0xdadaa124,0x9f112361), LL(0x823b7348,0x80145e31),LL(0xac634069,0x4dd8f0d5),LL(0x2297c258,0xe3d82fc7),LL(0x9cee7431,0x276fcfee),
   LL(0x2bc0aea9,0x8eb61b5e),LL(0xde329431,0x4f668fd5),LL(0x38e4b87e,0x03a32ab1),LL(0x73d0ef0b,0xe1374517), LL(0x853ac983,0x1a46f7e6),LL(0x68e78a57,0xc3bdf42e),LL(0x2ea96dd1,0xacf20785),LL(0xf1638460,0xa10649b9),
   LL(0x879fbbed,0xf2369f0b),LL(0xda9d1869,0x0ff0ae86),LL(0x56766f45,0x5251d759),LL(0x2be8d0fc,0x4984d8c0), LL(0xd21008f0,0x7ecc95a6),LL(0x3a1a1c49,0x29bd54a0),LL(0xd26c50f3,0xab9828c5),LL(0x51d0d251,0x32c0087c),
   LL(0x0c1cdb26,0x9bac3ce6),LL(0x557ca205,0xcd94d947),LL(0x9db1fdcd,0x1b1bd598),LL(0xa3d8b149,0x0eda0108), LL(0x56152fcc,0x95066610),LL(0xe7192b33,0xc2f037e6),LL(0xc92e05a4,0xdeffb41a),LL(0xc2f6c62e,0x1105f6c2),
   LL(0x8733913c,0x68e73500),LL(0x3f3adc40,0xcce86163),LL(0x38a278e9,0xf407a942),LL(0x2ab21292,0xd13c1b9d), LL(0x1c74cf5c,0x93ed7ec7),LL(0xf1a4c1b4,0x8887dc48),LL(0x4b3a11f1,0x3830ff30),LL(0x58937cb6,0x358c5a3c),
   LL(0x89022829,0x027dc404),LL(0x3b798f79,0x40e93977),LL(0x38be6ead,0x90ad3337),LL(0xf34c0a5d,0x9c23f6bc), LL(0xfbffd8bb,0xd1711a35),LL(0x1949d3dd,0x60fcfb49),LL(0x7825d93a,0x09c8ef4b),LL(0xa0a8c968,0x24233cff),
   LL(0xe6d982af,0x67ade46c),LL(0xe7544d7c,0xebb6bf3e),LL(0x3d8bd087,0xd6b9ba76),LL(0x4dc61280,0x46fe382d), LL(0xb5bdbd75,0xbd39a7e8),LL(0xb8f228fe,0xab381331),LL(0xce1c4300,0x0709a77c),LL(0xf337ceac,0x6a247e56),
   LL(0x636288be,0x8f34f21b),LL(0xc8a7c305,0x9dfdca74),LL(0xea919e04,0x6decfd1b),LL(0x8e1991f8,0xcdf2688d), LL(0xd0f8a67e,0xe607df44),LL(0x0b58d010,0xd985df4b),LL(0x0c24f8f4,0x57f834c5),LL(0xa0bf01ae,0xe976ef56),
   LL(0xa1c32373,0x536395ac),LL(0x734c0a13,0x351027aa),LL(0x5e6bd5bc,0xd2f1b5d6),LL(0x223debed,0x2b539e24), LL(0x0eaa1d71,0xd4994cec),LL(0x661dcf65,0x2a83381d),LL(0x7b54c740,0x5f1aed2f),LL(0xd6dda5ee,0x0bea3fa5),
   LL(0x36cc6134,0x9d4fb684),LL(0xc0a443dd,0x8eb9bbf3),LL(0x383b7d2a,0xfc500e2e),LL(0x5b775257,0x7aad621c), LL(0x0a8f7cc0,0x69284d74),LL(0x07562d65,0xe820c2ce),LL(0x499758ee,0xbf9531b9),LL(0x6ee0cc2d,0x73e95ca5),
   LL(0xfbaf50a5,0xf61790ab),LL(0x684e0750,0xdf55e76b),LL(0xf176b005,0xec516da7),LL(0x7a2dddc7,0x575553bb), LL(0x553afa73,0x37c87ca3),LL(0x4d55c251,0x315f3ffc),LL(0xaf3e5d35,0xe846442a),LL(0x6495ff28,0x61b91149),
   LL(0xfa326dc3,0x23cc95d3),LL(0x18fc2cea,0x1df4da1f),LL(0xd0a37d59,0x24bf9adc),LL(0x320d6e1e,0xb6710053), LL(0x618344d1,0x96f9667e),LL(0xa06445af,0xcc7ce042),LL(0xd68dbc3a,0xa02d8514),LL(0x280b5a5b,0x4ea109e4),
   LL(0xb40961bf,0x5741a7ac),LL(0x6aa56bfa,0x4ada5937),LL(0x02b765d1,0x7feb9145),LL(0xe6ad1582,0x561e97be), LL(0xda3982f5,0xbbc4a5b6),LL(0xb546f468,0x0c2659ed),LL(0x59612d20,0xb8e7e6aa),LL(0xac19e8e0,0xd83dfe20),
   LL(0xb835398c,0x8530c45f),LL(0xb38a41c2,0x6106a8bf),LL(0x35f5dcdb,0x21e8f9a6),LL(0xcae498ed,0x39707137), LL(0xd8249f00,0x70c23834),LL(0xab2537a0,0x9f14b58f),LL(0x5f61c0c2,0xd043c365),LL(0x09a194a7,0xdc5926d6),
   LL(0x8e77738a,0xddec0339),LL(0xfba46426,0xd07a63ef),LL(0xee7f6e86,0x2e58e79c),LL(0xff32d241,0xe59b0459), LL(0x20fa0338,0xc5ec84e5),LL(0xeaff5ace,0x97939ac8),LL(0xb4a38313,0x0310a4e3),LL(0x8f9d9885,0x9115fba2),
   LL(0x5fadf8c3,0x8dd710c2),LL(0xce19c0e2,0x66be38a2),LL(0x4cfe5022,0xd42a279c),LL(0x0e24e1b8,0x597bb530), LL(0xc153ca7f,0x3cde86b7),LL(0x707d63bd,0xa8d30fb3),LL(0xbd60d21e,0xac905f92),LL(0x7b9a54ab,0x98e7ffb6),
   LL(0xe9726a30,0xd7147df8),LL(0xafce3533,0xb5e216ff),LL(0x2ff1ec40,0xb550b799),LL(0xa1e953fd,0x6b613b87), LL(0x792d5610,0x87b88dba),LL(0xa190fbe1,0x2ee1270a),LL(0x2ef581da,0x02f4e2dc),LL(0xeff82a95,0x016530e4),
   LL(0x8fd6ee89,0xcbb93dfd),LL(0x46848fff,0x16d3d986),LL(0x1da47adf,0x600eff24),LL(0x0ad47a71,0x1b9754a0), LL(0x70c33b98,0x8f9266df),LL(0xdf34186e,0xaadc87ae),LL(0x4ad24132,0x0d2ce8e1),LL(0x19946eba,0x8a47cbfc),
   LL(0x62b5f3af,0x47feeb66),LL(0x0abb3734,0xcefab561),LL(0x19f35cb1,0x449de60e),LL(0x157f0eb9,0x39f8db14), LL(0x3c61bfd6,0xffaecc5b),LL(0x41216703,0xa5a4d41d),LL(0x224e1cc2,0x7f8fabed),LL(0x871ad953,0x0d5a8186),
   LL(0xd22da9a9,0xf10774f7),LL(0xcc8a9b0d,0x45b8a678),LL(0xbdc32cff,0xd9c2e722),LL(0x337202a5,0xbf71b5f5), LL(0x69fc4db9,0x95c57f2f),LL(0x765d01e1,0xb6dad34c),LL(0xcb904635,0x7e0bd13f),LL(0x763a588c,0x61751253),
   LL(0x81af2c2d,0xd85c2997),LL(0x81b9d7da,0xc0f7d9c4),LL(0x08533e8d,0x838a34ae),LL(0x311d8311,0x15c4cb08), LL(0x8e121e14,0x97f83285),LL(0x85000a5f,0xeea7dc1e),LL(0x5d256274,0x0c6059b6),LL(0xb95075c0,0xec9beace),
   LL(0x1df97828,0x173daad7),LL(0xa8937877,0xbf851cb5),LL(0x01646f3c,0xb083c594),LL(0x50c6d352,0x3bad30cf), LL(0x496bbcea,0xfeb2b202),LL(0x18a1e8ba,0x3cf9fd4f),LL(0x1c066029,0xd26de7ff),LL(0x4e9ed4f8,0x39c81e9e),
   LL(0x7b390d35,0xd8be0cb9),LL(0x964aab27,0x01df2bbd),LL(0xc3ef64f8,0x3e8c1a65),LL(0x716ed1dd,0x567291d1), LL(0x5f5406d3,0x95499c6c),LL(0x5ba8e23f,0x71fdda39),LL(0xd5096ece,0xcfeb320e),LL(0xca66dd16,0xbe7ba92b),
   LL(0xc6fb5a7d,0x4608d36b),LL(0x6d2dd0e0,0xe3eea15a),LL(0x8f97a36a,0x75b0a3eb),LL(0x1c83de1e,0xf59814cc), LL(0x1c33c23f,0x56c9c5b0),LL(0x6faa4136,0xa96c1da4),LL(0xde316551,0x46bf2074),LL(0x1f756c8f,0x3b866e7b),
   LL(0x1495ed6b,0x727727d8),LL(0xb682dce7,0xb2394243),LL(0x758610f3,0x8ab8454e),LL(0x857d72a4,0xc243ce84), LL(0xdbbf370f,0x7b320d71),LL(0x78e0f7ca,0xff9afa37),LL(0xea7b523f,0x0119d1e0),LL(0x058c7d42,0xb997f8cb),
   LL(0x37bbb184,0x285bcd2a),LL(0xa45d1fa6,0x51dcec49),LL(0xe29634cb,0x6ade3b64),LL(0x26b86ef1,0x080c94a7), LL(0x2283fbe3,0xba583db1),LL(0x5a9315ed,0x902bddc8),LL(0x86964bec,0x07c1ccb3),LL(0xb6258301,0x78f4eacf),
   LL(0x56f90823,0x4bdf3a49),LL(0x741d777b,0xba0f5080),LL(0xf38bf760,0x091d71c3),LL(0x9b625b02,0x9633d50f), LL(0xb8c9de61,0x03ecb743),LL(0x5de74720,0xb4751254),LL(0x74ce1cb2,0x9f9defc9),LL(0x00bd32ef,0x774a4f6a),
   LL(0x73848f22,0xaca385f7),LL(0xf3f8558e,0x53dad716),LL(0x93c471f9,0xab7b34b0),LL(0x19644bc7,0xf530e069), LL(0xdd59d31a,0x3d9fb1ff),LL(0x08daa795,0x4382e0df),LL(0xd5cc88d7,0x165c6f4b),LL(0x4a18c900,0xeaa392d5),
   LL(0x648024ee,0x94203c67),LL(0x8c2fabcd,0x188763f2),LL(0xbbaec835,0xa80f87ac),LL(0xf29d8d54,0x632c96e0), LL(0x4c00a95e,0x29b0a60e),LL(0xe011e9fa,0x2ef17f40),LL(0x15b77223,0xf6c0e1d1),LL(0x14b04e32,0xaaec2c62),
   LL(0x3d84e58c,0xd35688d8),LL(0x958571db,0x2af5094c),LL(0x760682a6,0x4fff7e19),LL(0xe39a407c,0x4cb27077), LL(0x4ff0e321,0x0f59c547),LL(0x1b34c8ff,0x169f34a6),LL(0x52bc1ba7,0x2bff1096),LL(0x83583544,0xa25423b7),
   LL(0x0ac8b782,0x5d55d5d5),LL(0x2db3c892,0xff6622ec),LL(0x6b8bb642,0x48fce741),LL(0x69d7e3dc,0x31d6998c), LL(0xcadcaed0,0xdbaf8004),LL(0xd81d053c,0x801b0142),LL(0x59630ec6,0x94b189fc),LL(0xaf762c8e,0x120e9934),
   LL(0xfdc6a404,0x53a29aa4),LL(0xa1909948,0x19d8e01e),LL(0xd7e89681,0x3cfcabf1),LL(0x4e132d37,0x3321a50d), LL(0xe9a86111,0xd0496863),LL(0x06a3bc65,0x8c0cde61),LL(0xfc9f8eef,0xaf866c49),LL(0xff7f5141,0x2066350e),
   LL(0xe56ddfbd,0x4f8a4689),LL(0xfe32983a,0xea1b0c07),LL(0x873cb8cb,0x2b317462),LL(0x2d93229f,0x658deddc), LL(0x0f64ef58,0x65efaf4d),LL(0x730cc7a8,0xfe43287d),LL(0x3d047d70,0xaebc0c72),LL(0xd92d26c9,0x92efa539),
   LL(0x94b56526,0x06e78457),LL(0x0961002d,0x415cb80f),LL(0x76dcb10f,0x89e5c565),LL(0xff9259fe,0x8bbb6982), LL(0x9abc2668,0x4fe8795b),LL(0x1e678fb1,0xb5d4f534),LL(0x7b7da2b9,0x6601f3be),LL(0xa13d6805,0x98da59e2),
   LL(0x01799a52,0x190d8ea6),LL(0xb86d2952,0xa20cec41),LL(0x7fff2a7c,0x3062ffb2),LL(0x79f19d37,0x741b32e5), LL(0x4eb57d47,0xf80d8181),LL(0x16aef06b,0x7a2d0ed4),LL(0x1cecb588,0x09735fb0),LL(0xc6061f5b,0x1641caaa),
},
/* digit=4 base_pwr=2^28 */
{
   LL(0x20151427,0x7f99824f),LL(0x92430206,0x206828b6),LL(0xe1112357,0xaa9097d7),LL(0x09e414ec,0xacf9a2f2), LL(0x27915356,0xdbdac9da),LL(0x001efee3,0x7e0734b7),LL(0xd2b288e2,0x54fab5bb),LL(0xf62dd09c,0x4c630fc4),
   LL(0x1ac2703b,0x8537107a),LL(0x6bc857b5,0xb49258d8),LL(0xbcdaccd1,0x57df14de),LL(0xc4ae8529,0x24ab68d7), LL(0x734e59d0,0x7ed8b5d4),LL(0xc495cc80,0x5f8740c8),LL(0x291db9b3,0x84aedd5a),LL(0x4fb995be,0x80b360f8),
   LL(0x5fa067d1,0xae915f5d),LL(0x9668960c,0x4134b57f),LL(0xa48edaac,0xbd3656d6),LL(0xfc1d7436,0xdac1e3e4), LL(0xd81fbb26,0x674ff869),LL(0xb26c33d4,0x449ed3ec),LL(0xd94203e8,0x85138705),LL(0xbeeb6f4a,0xccde538b),
   LL(0xa61a76fa,0x55d5c68d),LL(0xca1554dc,0x598b441d),LL(0x773b279c,0xd39923b9),LL(0x36bf9efc,0x33331d3c), LL(0x298de399,0x2d4c848e),LL(0xa1a27f56,0xcfdb8e77),LL(0x57b8ab70,0x94c855ea),LL(0x6f7879ba,0xdcdb9dae),
   LL(0x019f2a59,0x7bdff8c2),LL(0xcb4fbc74,0xb3ce5bb3),LL(0x8a9173dd,0xea907f68),LL(0x95a75439,0x6cd3d0d3), LL(0xefed021c,0x92ecc4d6),LL(0x6a77339a,0x09a9f9b0),LL(0x7188c64a,0x87ca6b15),LL(0x44899158,0x10c29968),
   LL(0xed6e82ef,0x5859a229),LL(0x65ebaf4e,0x16f338e3),LL(0x5ead67ae,0x0cd31387),LL(0x54ef0bb4,0x1c73d228), LL(0x74a5c8c7,0x4cb55131),LL(0x7f69ad6a,0x01cd2970),LL(0xe966f87e,0xa04d00dd),LL(0x0b7b0321,0xd96fe447),
   LL(0x88fbd381,0x342ac06e),LL(0x5c35a493,0x02cd4a84),LL(0x54f1bbcd,0xe8fa89de),LL(0x2575ed4c,0x341d6367), LL(0xd238202b,0xebe357fb),LL(0xa984ead9,0x600b4d1a),LL(0x52436ea0,0xc35c9f44),LL(0xa370751b,0x96fe0a39),
   LL(0x7f636a38,0x4c4f0736),LL(0x0e76d5cb,0x9f943fb7),LL(0xa8b68b8b,0xb03510ba),LL(0x9ed07a1f,0xc246780a), LL(0x6d549fc2,0x3c051415),LL(0x607781ca,0xc2953f31),LL(0xd8d95413,0x955e2c69),LL(0x7bd282e3,0xb300fadc),
   LL(0x87e9189f,0x81fe7b50),LL(0xf42dda27,0xdb17375c),LL(0xcf0a5904,0x22f7d896),LL(0xebe348e6,0xa0e57c5a), LL(0xf40e3c80,0xa61011d3),LL(0x8db705c5,0xb1189321),LL(0x50fedec3,0x4ed9309e),LL(0x4d6d5c1d,0xdcf14a10),
   LL(0x55691342,0x056c265b),LL(0x91049dc7,0xe8e08504),LL(0xc9bae20a,0x131329f5),LL(0xd9dccdb4,0x96c8b3e8), LL(0xfb4ee6b4,0x8c5ff838),LL(0x41e8ccf0,0xfc5a9aeb),LL(0xfae050c6,0x7417b764),LL(0x00452080,0x0953c3d7),
   LL(0x38dfe7e8,0x21372682),LL(0x2bb79d4b,0xea417e15),LL(0x76e7cf2d,0x59641f1c),LL(0xea0bcfcc,0x271e3059), LL(0x7253ecbd,0x624c7dfd),LL(0x4fca6186,0x2f552e25),LL(0x4d866e9c,0xcbf84ecd),LL(0xf68d4610,0x73967709),
   LL(0xc27901b4,0xa14b1163),LL(0x899b8bf3,0xfd9236e0),LL(0xcbc6da0a,0x42b091ec),LL(0x5ad1d297,0xbb1dac6f), LL(0xa91cf76e,0x80e61d53),LL(0xd31f1ee7,0x4110a412),LL(0x13efcf77,0x2d87c3ba),LL(0xdf450d76,0x1f374bb4),
   LL(0x0d188dab,0x5e78e2f2),LL(0xf4b885ef,0xe3968ed0),LL(0x7314570f,0x46c0568e),LL(0x01170521,0x31616338), LL(0x4f0c8afe,0x18e1e7e2),LL(0xdeea78da,0x4caa75ff),LL(0x7c5d8a51,0x82db67f2),LL(0x6f505370,0x36a44d86),
   LL(0x0333974f,0xd72c5bda),LL(0x27a70146,0x5db516ae),LL(0x210ef921,0x34705281),LL(0x0c9c38e5,0xbff17a8f), LL(0x12476da1,0x78f4814e),LL(0x33c16980,0xc1e16613),LL(0x424d4bca,0x9e5b386f),LL(0xc85740de,0x4c274e87),
   LL(0x6c2f5226,0xb6a9b88d),LL(0x550d7ca8,0x14d1b944),LL(0x1fc41709,0x580c85fc),LL(0x54c6d519,0xc1da368b), LL(0xd5113cf7,0x2b0785ce),LL(0x5a34708f,0x0670f633),LL(0x15cc3f88,0x46e23767),LL(0x50c72c8f,0x1b480cfa),
   LL(0x4147519a,0x20288602),LL(0x26b372f0,0xd0981eac),LL(0xa785ebc8,0xa9d4a7ca),LL(0xdbdf58e9,0xd953c50d), LL(0xfd590f8f,0x9d6361cc),LL(0x44e6c917,0x72e9626b),LL(0x22eb64cf,0x7fd96110),LL(0x9eb288f3,0x863ebb7e),
   LL(0x6aca8ee7,0x6e6ab761),LL(0xd7b40358,0x97d10b39),LL(0x1e5feb0d,0x1687d377),LL(0x8265a27a,0xc83e50e4), LL(0xc954b313,0x8f75a9fe),LL(0x310d1f61,0xcc2e8f47),LL(0x6557d0e0,0xf5ba81c5),LL(0x3eaf6207,0x25f9680c),
   LL(0x4354080b,0xf95c6609),LL(0x7bf2fe1c,0x5225bfa5),LL(0x5c7d98fa,0xc5c004e2),LL(0x019aaf60,0x3561bf1c), LL(0xba151474,0x5e6f9f17),LL(0xb04f6eca,0xdec2f934),LL(0x269acb1e,0x64e368a1),LL(0x0cdda493,0x1332d9e4),
   LL(0xdf23de05,0x60d6cf69),LL(0x009339a0,0x66d17da2),LL(0x0a693923,0x9fcac985),LL(0xed7c6a6d,0xbcf057fc), LL(0xf0b5662c,0xc3c5c8c5),LL(0xdcba4f24,0x25318dd8),LL(0x082b69ff,0x60e8cb75),LL(0x1e728c01,0x7c23b3ee),
   LL(0x097e4403,0x15e10a0a),LL(0x19854665,0xcb3d0a86),LL(0xd67d4826,0x88d8e211),LL(0x0b9d2839,0xb39af66e), LL(0xbd475ca8,0xa5f94588),LL(0xc077b80b,0xe06b7966),LL(0xda27c26c,0xfedb1485),LL(0xfe0fd5e0,0xd290d33a),
   LL(0xf34fb0fa,0xa40bcc47),LL(0x1fb1ab09,0xb4760cc8),LL(0xa273bfe3,0x8fca0993),LL(0xf70b213c,0x13e4fe07), LL(0xfdb05163,0x3bcdb992),LL(0x0c2b19b6,0x8c484b11),LL(0xaaf2e3e2,0x1acb815f),LL(0xb89ff1b4,0xc6905935),
   LL(0x586e74e1,0xb2ad6f9d),LL(0x67b80484,0x488883ad),LL(0x369c3ddb,0x758aa2c7),LL(0x9f9afd31,0x8ab74e69), LL(0x5e21beb1,0x10fc2d28),LL(0x318c42f9,0x3484518a),LL(0x53cf40c3,0x377427dc),LL(0x391bc1d9,0x9de0781a),
   LL(0x693807e1,0x8faee858),LL(0x4e81ccc7,0xa3865327),LL(0x6f835b84,0x02c30ff2),LL(0x0d3d38d4,0xb604437b), LL(0x5ca1823d,0xb3fc8a98),LL(0x03be0324,0xb82f7ec9),LL(0xcf684a33,0xee36d761),LL(0x9f29bf7d,0x5a01df0e),
   LL(0x1306583d,0x686202f3),LL(0x437c622e,0x05b10da0),LL(0x076a7bc8,0xbf9aaa0f),LL(0x8f8f4e43,0x25e94efb), LL(0xfa3dc26d,0x8a35c9b7),LL(0x96ff03c5,0xe0e5fb93),LL(0xebc394ce,0xa77e3843),LL(0x8361de60,0xcede6595),
   LL(0xa1993545,0xd27c22f6),LL(0x24d671ba,0xab01cc36),LL(0xa169c28e,0x63fa2877),LL(0x2eb08376,0x925ef904), LL(0x53aa0b32,0x3b2fa3cf),LL(0x71c49d7a,0xb27beb5b),LL(0xd105e27f,0xb60e1834),LL(0x4f68570d,0xd6089788),
   LL(0xd6fbc2ac,0x23094ce0),LL(0x815ff551,0x738037a1),LL(0x6bef119c,0xda73b1bb),LL(0xeef506ba,0xdcf6c430), LL(0xe3ef104a,0x00e4fe7b),LL(0x0a065628,0xebdd9a2c),LL(0x8792043e,0x853a81c3),LL(0xb3b59108,0x22ad6ece),
   LL(0x39cd297d,0x9fb813c0),LL(0x05bda5d9,0x8ec7e16e),LL(0x0d104b96,0x2834797c),LL(0x7c511510,0xcc11a2e7), LL(0x96ee6380,0x96ca5a53),LL(0xcea38742,0x054c8655),LL(0xd54dfa7d,0xb5946852),LL(0x1f4ab207,0x97c422e7),
   LL(0x0c22b540,0xbf907509),LL(0xb7c267d4,0x2cde42aa),LL(0x5ab0d693,0xba18f9ed),LL(0x6e4660d9,0x3ba62aa6), LL(0xab9ea96a,0xb24bf97b),LL(0xe3b60e32,0x5d039642),LL(0x7c4d9bd5,0x4e6a4506),LL(0x7ed4a6a4,0x666c5b9e),
   LL(0x8edbd7cc,0xfa3fdcd9),LL(0xc6ccd753,0x4660bb87),LL(0x21e6b64f,0x9ae90820),LL(0xb36bfb3f,0x8a56a713), LL(0x5726d47f,0xabfce096),LL(0x0b1a9a7f,0x9eed01b2),LL(0x4eb74a37,0x30e9cad4),LL(0x53e9666d,0x7b2524cc),
   LL(0x8f4b002f,0x6a29683b),LL(0x41f4fc20,0xc2200d7a),LL(0x3a338acc,0xcf3af47a),LL(0xe7128975,0x6539a4fb), LL(0xc33c7fcf,0xcec31c14),LL(0xc7be322b,0x7eb6799b),LL(0x6646f623,0x119ef4e9),LL(0x54d7299b,0x7b7a26a5),
   LL(0x403f46f2,0xcb37f08d),LL(0x1a0ec0c7,0x94b8fc43),LL(0xc332142f,0xbb8514e3),LL(0xe80d2a7a,0xf3ed2c33), LL(0xb639126c,0x8d2080af),LL(0xe3553ade,0xf7b6be60),LL(0x1c7e2b09,0x3950aa9f),LL(0x6410f02b,0x847ff958),
   LL(0x678a31b0,0x877b7cf5),LL(0x3998b620,0xd50301ae),LL(0xc00fb396,0x734257c5),LL(0x04e672a6,0xf9fb18a0), LL(0xe8758851,0xff8bd8eb),LL(0x5d99ba44,0x1e64e4c6),LL(0x7dfd93b7,0x4b8eaedf),LL(0x04e76b8c,0xba2f2a98),
   LL(0xe8053433,0x7d790cba),LL(0x3d2c9585,0xc8e725a0),LL(0xcdd8f5ed,0x58c5c476),LL(0xefa9fe1d,0xd106b952), LL(0x0eff13a9,0x3c5c775b),LL(0xe057b930,0x242442ba),LL(0xc9b70cbd,0xe9f458d4),LL(0xa3cdb89a,0x69b71448),
   LL(0x0e2ed742,0x41ee46f6),LL(0x40067493,0x573f1045),LL(0x9d54c304,0xb1e154ff),LL(0x8d3a7502,0x2ad0436a), LL(0x431a8121,0xee4aaa2d),LL(0x886f11ed,0xcd38b3ab),LL(0x034a0eb7,0x57d49ea6),LL(0xf7e85e58,0xd2b773bd),
   LL(0x9b5c1f14,0x4a559ac4),LL(0x3e54df2b,0xc444be1a),LL(0xeda41891,0x13aad704),LL(0x5eb5c788,0xcd927bec), LL(0xe48c8a34,0xeb3c8516),LL(0x4b546669,0x1b7ac812),LL(0x594df8ec,0x1815f896),LL(0x79227865,0x87c6a79c),
   LL(0x9b56ddbd,0xae02a2f0),LL(0x8a2f1cf3,0x1339b5ac),LL(0x839dff0d,0xf2b569c7),LL(0xfee9a43d,0xb0b9e864), LL(0x77bb064e,0x4ff8ca41),LL(0xfd249f63,0x145a2812),LL(0xf86f689a,0x3ab7beac),LL(0x01d35f5e,0x9bafec27),
   LL(0x4265aa91,0x28054c65),LL(0x035efe42,0xa4b18304),LL(0x9639dec7,0x6887b0e6),LL(0x3d52aea5,0xf4b8f6ad), LL(0x971a8a13,0xfb9293cc),LL(0x4c934d07,0x3f159e5d),LL(0x09acbc29,0x2c50e9b1),LL(0x7154d129,0x08eb65e6),
   LL(0x30b75c3e,0x4feff589),LL(0x94491c93,0x0bb82fe2),LL(0x89af62bb,0xd8ac377a),LL(0x9685e49f,0xd7b51490), LL(0x04497f19,0xabca9a7b),LL(0x1a7ad13f,0x1b35ed0a),LL(0x3ec86ed6,0x6b601e21),LL(0xce0c76f1,0xda91fcb9),
   LL(0xd7ab27e1,0x9e28507b),LL(0x63945b7b,0x7c19a555),LL(0xaafc9827,0x6b43f0a1),LL(0x3aa55b91,0x443b4fbd), LL(0x6962c88f,0x962b2e65),LL(0xce0db0ca,0x139da8d4),LL(0x1b8d6c4f,0xb93f05dd),LL(0x180b9824,0x779cdff7),
   LL(0xae57c7b7,0xbba23fdd),LL(0x1b932522,0x345342f2),LL(0x556d4aa3,0xfd9c80fe),LL(0x6525bb61,0xa03907ba), LL(0xff218933,0x38b010e1),LL(0xaa52117b,0xc066b654),LL(0x94f2e6ea,0x8e141920),LL(0x0d32f2b2,0x66a27dca),
   LL(0x048b3717,0x69c7f993),LL(0xb178ae1c,0xbf5a989a),LL(0x564f1d6b,0x49fa9058),LL(0xd31fde4e,0x27ec6e15), LL(0x7276e7fc,0x4cce0373),LL(0x89d6bf02,0x64086d79),LL(0x4ccdd979,0x5a72f046),LL(0x47775631,0x909c3566),
   LL(0x75dd7125,0x1c07bc6b),LL(0x87a0428d,0xb4c6bc97),LL(0xfdeb6b9d,0x507ece52),LL(0xb2c95432,0xfca56512), LL(0xd0e8bd06,0x15d97181),LL(0xc6bb46ea,0x384dd317),LL(0x3952b624,0x5441ea20),LL(0x4e7dc2fb,0xbcf70dee),
   LL(0x6628e8c3,0x372b016e),LL(0xb60a7522,0x07a0d667),LL(0x0a344ee2,0xcf05751b),LL(0x118bdeec,0x0ec09a48), LL(0xd83dce46,0x6e4b3d4e),LL(0x99d2fc6e,0x43a6316d),LL(0x56cf044c,0xa99d8989),LL(0xae3e5fb7,0x7c7f4454),
   LL(0xfbabbe92,0xb2e6b121),LL(0xe1330076,0x281850fb),LL(0x97890015,0x093581ec),LL(0x75ff77f5,0x69b1dded), LL(0xab105105,0x7cf0b18f),LL(0xa89ccfef,0x953ced31),LL(0xeb914009,0x3151f85f),LL(0x88ed48ad,0x3c9f1b87),
   LL(0x4a7eadcb,0xc9aba1a1),LL(0x522e71cf,0x928e7501),LL(0x3a2e4f83,0xeaede727),LL(0x1ce3bbd3,0x467e10d1), LL(0xb955dcf0,0xf3442ac3),LL(0xd3d5e527,0xba96307d),LL(0xfd77f474,0xf763a10e),LL(0x6a6e1ff0,0x5d744bd0),
   LL(0xa777899e,0xd287282a),LL(0xd03f3cde,0xe20eda8f),LL(0x50b07d31,0x6a7e75bb),LL(0x6f379de4,0x0b7e2a94), LL(0x19f593cf,0x31cb64ad),LL(0x1e76ef1d,0x7b1a9e4f),LL(0xb62d609c,0xe18c9c9d),LL(0xe779a650,0x439bad6d),
   LL(0xe032f144,0x219d9066),LL(0xe8b2ec6a,0x1db632b8),LL(0xfda12f78,0xff0d0fd4),LL(0x2a25d265,0x56fb4c2d), LL(0x255a03f1,0x5f4e2ee1),LL(0xe96af176,0x61cd6af2),LL(0xd068bc97,0xe0317ba8),LL(0x264b988e,0x927d6bab),
   LL(0xe90fb21e,0xa18f07e0),LL(0xbba7fca1,0x00fd2b80),LL(0x95cd67b5,0x20387f27),LL(0xd39707f7,0x5b89a4e7), LL(0x894407ce,0x8f83ad3f),LL(0x6c226132,0xa0025b94),LL(0xf906c13b,0xc79563c7),LL(0x4e7bb025,0x5f548f31),
   LL(0xeac6d113,0x2b4c6b8f),LL(0x0e813c76,0xa67e3f9c),LL(0x3fe1f4b9,0x3982717c),LL(0x26d8050e,0x58865819), LL(0xf7f06f20,0x99f3640c),LL(0x2a66ebc2,0xdc610216),LL(0x767a1e08,0x52f2c175),LL(0x5999871b,0x05660e1a),
   LL(0x6d3c4693,0x6b0f1762),LL(0x37ed7bea,0xf0e7d627),LL(0xb75b226d,0xc51758c7),LL(0x1f91613b,0x40a88628), LL(0xbbb38ce0,0x889dbaa7),LL(0xbddcad81,0xe0404b65),LL(0x8bc9671f,0xfebccd3a),LL(0xee1f5375,0xfbf9a357),
   LL(0x28f33398,0x5dc169b0),LL(0x72e90f65,0xb07ec11d),LL(0xfaab1eb1,0xae7f3b4a),LL(0x5f17538a,0xd970195e), LL(0x0181e640,0x52b05cbe),LL(0x2643313d,0xf5debd62),LL(0x5df31f82,0x76148154),LL(0x3a9e13c5,0x23e03b33),
   LL(0x4fde0c1f,0xff758949),LL(0xe5b6ec20,0xbf8a1abe),LL(0x87e1db6c,0x702278fb),LL(0x35ed658f,0xc447ad7a), LL(0x03d0ccf2,0x48d4aa38),LL(0x819a7c03,0x80acb338),LL(0x6e17cecc,0x9bc7c89e),LL(0x03be1d82,0x46736b8b),
   LL(0xc0432f96,0xd65d7b60),LL(0xdeb5442f,0xddebe7a3),LL(0x7dff69a2,0x79a25307),LL(0x02cf3122,0x37a56d94), LL(0xf2350d0a,0x8bab8aed),LL(0x037b0d9a,0x13c3f276),LL(0x44c65cae,0xc664957c),LL(0xc2e71a88,0x88b44089),
   LL(0x5cb02664,0xdb88e5a3),LL(0x8686c72e,0x5d4c0bf1),LL(0xa682d53e,0xea3d9b62),LL(0x0b2ad431,0x9b605ef4), LL(0xc69645d0,0x71bac202),LL(0x6a1b66e7,0xa115f03a),LL(0x158f4dc4,0xfe2c563a),LL(0x4d12a78c,0xf715b3a0),
   LL(0xd413213a,0x8f7f0a48),LL(0xc04becdb,0x2035806d),LL(0x5d8587f5,0xecd34a99),LL(0x9f6d3a71,0x4d8c3079), LL(0x8d95a8f6,0x1b2a2a67),LL(0xf2110d0d,0xc58c9d7d),LL(0xcf8fba3f,0xdeee81d5),LL(0x0c7cdf68,0xa42be3c0),
   LL(0xd43b5eaa,0x2126f742),LL(0xdfa59b85,0x054a0766),LL(0x126bfd45,0x9d0d5e36),LL(0x384f8a8f,0xa1f8fbd7), LL(0xd563fccc,0x317680f5),LL(0xf280a928,0x48ca5055),LL(0x27b578cf,0xe00b81b2),LL(0x2994a514,0x10aad918),
   LL(0xb7bdc953,0xd9e07b62),LL(0x5bc086dd,0x9f0f6ff2),LL(0x655eee77,0x09d1ccff),LL(0x5bef7df1,0x45475f79), LL(0x86f702cc,0x3faa28fa),LL(0x0f021f07,0x92e60905),LL(0x7f8fa8c6,0xe9e62968),LL(0xf036ea2c,0xbd71419a),
   LL(0x6028da9a,0x171ee1cc),LL(0xc251f573,0x5352fe1a),LL(0x3fa997f4,0xf8ff236e),LL(0xa5749d5f,0xd831b6c9), LL(0xe350e2c2,0x7c872e1d),LL(0x1e0ce403,0xc56240d9),LL(0x6974f5cb,0xf9deb077),LL(0x961c3728,0x7d50ba87),
   LL(0x5a3a2518,0xd6f89426),LL(0xc6303d43,0xcf817799),LL(0x619e5696,0x510a0471),LL(0x3a5e307b,0xab049ff6), LL(0xfeb13ec7,0xe4cdf9b0),LL(0x9d8ff90c,0xd5e97117),LL(0x9afa96af,0xf6f64d06),LL(0x9d2012a2,0x00d0bf5e),
   LL(0x358bcdc0,0xe63f301f),LL(0x0a9d47f8,0x07689e99),LL(0x4f43d43a,0x1f689e2f),LL(0x90920904,0x4d542a16), LL(0x9ca0a707,0xaea293d5),LL(0x8ac68065,0xd061fe45),LL(0x0090008c,0x1033bf1b),LL(0xc08a6db6,0x29749558),
   LL(0xc1d5d034,0x74b5fc59),LL(0x67e215e0,0xf712e9f6),LL(0x860200e6,0xfd520cbd),LL(0x3ea22588,0x0229acb4), LL(0xfff0c82e,0x9cd1e14c),LL(0x59c69e73,0x87684b62),LL(0x96ccb989,0xda85e61c),LL(0xa3d06493,0x2d5dbb02),
   LL(0xe86b173c,0xf22ad33a),LL(0xa79ff0e3,0xe8e41ea5),LL(0xdd0d0c10,0x01d2d725),LL(0x032d28f9,0x31f39088), LL(0x7829839e,0x7b3f71e1),LL(0x4502ae58,0x0cf691b4),LL(0xbefc6115,0xef658dbd),LL(0xb3ab5314,0xa5cd6ee5),
   LL(0x5f1d2347,0x206c8d7b),LL(0x4cc2253a,0x794645ba),LL(0x58389e08,0xd517d8ff),LL(0x9f847288,0x4fa20dee), LL(0xd797770a,0xeba072d8),LL(0xbf429e26,0x7360c91d),LL(0x80af8279,0x7200a3b3),LL(0x82dadce3,0x6a1c9150),
   LL(0xc35d8794,0x0ee6d3a7),LL(0x0356bae5,0x042e6558),LL(0x643322fd,0x9f59698d),LL(0x50a61967,0x9379ae15), LL(0xfcc9981e,0x64b9ae62),LL(0x6d2934c6,0xaed3d631),LL(0x5e4e65eb,0x2454b302),LL(0xf9950428,0xab09f647),
},
/* digit=5 base_pwr=2^35 */
{
   LL(0x22248acc,0xb2083a12),LL(0x3264e366,0x1f6ec0ef),LL(0x5afdee28,0x5659b704),LL(0xe6430bb5,0x7a823a40), LL(0xe1900a79,0x24592a04),LL(0xc9ee6576,0xcde09d4a),LL(0x4b5ea54a,0x52b6463f),LL(0xd3ca65a7,0x1efe9ed3),
   LL(0x305406dd,0xe27a6dbe),LL(0xdd5d1957,0x8eb7dc7f),LL(0x387d4d8f,0xf54a6876),LL(0xc7762de4,0x9c479409), LL(0x99b30778,0xbe4d5b5d),LL(0x6e793682,0x25380c56),LL(0xdac740e3,0x602d37f3),LL(0x1566e4ae,0x140deabe),
   LL(0xafd32acf,0x4481d067),LL(0xe1f71ccf,0xd8f0fcca),LL(0xb596f2da,0xd208dd0c),LL(0x9aad93f9,0xd049d730), LL(0x42ab580e,0xc79f263d),LL(0x23f707b4,0x09411bb1),LL(0x835e0eda,0x8cfde1ff),LL(0x90f03402,0x72707490),
   LL(0xc49a861e,0xeaee6126),LL(0xe14f0d06,0x024f3b65),LL(0xc69bfc17,0x51a3f1e8),LL(0xa7686381,0xc3c3a8e9), LL(0xb103d4c8,0x3400752c),LL(0x9218b36b,0x02bc4613),LL(0x7651504a,0xc67f75eb),LL(0xd02aebfa,0xd6848b56),
   LL(0xc30fa92b,0xbd9802e6),LL(0x9a552784,0x5a70d96d),LL(0x3f83169b,0x9085c4ea),LL(0x06908228,0xfa9423bb), LL(0xfe97a5b9,0x2ffebe12),LL(0x71b99118,0x85da6049),LL(0x63178846,0x9cbc2f7f),LL(0x9153218e,0xfd96bc70),
   LL(0x1782269b,0x958381db),LL(0x2597e550,0xae34bf79),LL(0x5f385153,0xbb5c6064),LL(0xe3088048,0x6f0e96af), LL(0x77884456,0xbf6a0215),LL(0x69310ea7,0xb3b5688c),LL(0x04fad2de,0x17c94295),LL(0x17896d4d,0xe020f0e5),
   LL(0x0976505f,0x730ba0ab),LL(0x095e2ec5,0x567f6813),LL(0x6331ab71,0x47062010),LL(0x41d22b9f,0x72cfa977), LL(0x8a2373da,0x33e55ead),LL(0x7ba45a68,0xa8d0d5f4),LL(0x03029d15,0xba1d8f9c),LL(0xfc55b9f3,0x8f34f1cc),
   LL(0xbbe5a1a9,0xcca4428d),LL(0x3126bd67,0x8187fd5f),LL(0x48105826,0x0036973a),LL(0xb8bd61a0,0xa39b6663), LL(0x2d65a808,0x6d42deef),LL(0x94636b19,0x4969044f),LL(0xdd5d564c,0xf611ee47),LL(0xd2873077,0x7b2f3a49),
   LL(0x300eb294,0x94157d45),LL(0x169c1494,0x2b2a656e),LL(0xd3a47aa9,0xc000dd76),LL(0xa6243ea4,0xa2864e4f), LL(0xdb89842e,0x82716c47),LL(0x61479fb7,0x12dfd7d7),LL(0xe0b2f6dc,0x3b9a2c56),LL(0xd7f85d67,0x46be862a),
   LL(0x0f82b214,0x03b0d8dd),LL(0xf103cbc6,0x460c34f9),LL(0x18d79e19,0xf32e5c03),LL(0xa84117f8,0x8b8888ba), LL(0xc0722677,0x8f3c37dc),LL(0x1c1c0f27,0x10d21be9),LL(0xe0f7a0c6,0xd47c8468),LL(0xadecc0e0,0x9bf02213),
   LL(0x42b48b99,0x0baa7d12),LL(0x48424096,0x1bcb665d),LL(0xebfb5cfb,0x8b847cd6),LL(0x9ad4d10d,0x87c2ae56), LL(0x0de36726,0xf1cbb122),LL(0x3fdfbd21,0xe7043c68),LL(0x4e79d460,0x4bd0826a),LL(0x4bd1a2cb,0x11f5e598),
   LL(0xb7fe7b6e,0x97554160),LL(0x400a3fb2,0x7d16189a),LL(0xe328ca1e,0xd73e9bea),LL(0xe793d8cc,0x0dd04b97), LL(0x506db8cc,0xa9c83c9b),LL(0xcf38814c,0x5cd47aae),LL(0xb64b45e6,0x26fc430d),LL(0xd818ea84,0x079b5499),
   LL(0xc1c24a3b,0xebb01102),LL(0x1c161c1a,0xca24e568),LL(0x36f00a4a,0x103eea69),LL(0x76176c7b,0x9ad76ee8), LL(0x538e0ff7,0x97451fc2),LL(0x6604b3b0,0x94f89809),LL(0x3249cfd7,0x6311436e),LL(0x41224f69,0x27b4a7bd),
   LL(0xe0ac2941,0x03b5d21a),LL(0xc2d31937,0x279b0254),LL(0xcac992d0,0x3307c052),LL(0xefa8b1f3,0x6aa7cb92), LL(0x0d37c7a5,0x5a182580),LL(0x342d5422,0x13380c37),LL(0xd5d2ef92,0x92ac2d66),LL(0x030c63c6,0x035a70c9),
   LL(0x4ce4f152,0xc16025dd),LL(0xf9df7c06,0x1f419a71),LL(0x91e4bb14,0x6d5b2214),LL(0x839fb4ce,0xfc43c6cc), LL(0x925d6b2d,0x49f06591),LL(0x62186598,0x4b37d9d3),LL(0xd01b1629,0x8c54a971),LL(0x51d50e05,0xe1a9c29f),
   LL(0x71ba1861,0x5109b785),LL(0xd0c8f93d,0x48b22d5c),LL(0x8633bb93,0xe8fa84a7),LL(0x5aebbd08,0x53fba6ba), LL(0xe5eea7d8,0x7ff27df3),LL(0x68ca7158,0x521c8796),LL(0xce6f1a05,0xb9d5133b),LL(0xfd0ebee4,0x2d50cd53),
   LL(0xc5a3ef16,0xc82115d6),LL(0xba079221,0x993eff9d),LL(0x4b5da81c,0xe4da2c5e),LL(0x8033fd85,0x9a89dbdb), LL(0x2b892891,0x60819ebf),LL(0x5d14a4d5,0x53902b21),LL(0xd7fda421,0x6ac35051),LL(0x61c83284,0xcc6ab885),
   LL(0xf74cff17,0x14eba133),LL(0xecb813f2,0x240aaa03),LL(0x6f665bee,0xcfbb6540),LL(0xa425ad73,0x084b1fe4), LL(0xd081f6a6,0x009d5d16),LL(0xeef82c90,0x35304fe8),LL(0xaa9eaa22,0xf20346d5),LL(0xac1c91e3,0x0ada9f07),
   LL(0x968a6144,0xa6e21678),LL(0x07b31a1e,0x54c1f77c),LL(0x5781fbe1,0xd6bb787e),LL(0xe31f1c4a,0x61bd2ee0), LL(0x781105fc,0xf25aa1e9),LL(0x7b2f8e80,0x9cf2971f),LL(0xcdff919b,0x26d15412),LL(0x34bc896e,0x01db4ebe),
   LL(0xb40df1cf,0x7d9b3e23),LL(0x94e971b4,0x59337373),LL(0x669cf921,0xbf57bd14),LL(0x0c1a1064,0x865daedf), LL(0x83279125,0x3eb70bd3),LL(0x34ecdaab,0xbc3d5b9f),LL(0x5f755caf,0x91e3ed7e),LL(0xd41e6f02,0x49699f54),
   LL(0xd4a7a15b,0x185770e1),LL(0xeaac87e7,0x08f3587a),LL(0x473133ea,0x352018db),LL(0x04fd30fc,0x674ce719), LL(0x088b3e0e,0x7b8d9835),LL(0x5d0d47a1,0x7a0356a9),LL(0x6474a3c4,0x9d9e7659),LL(0xff66966c,0x61ea48a7),
   LL(0x0f3e4834,0x30417758),LL(0x17a9afcb,0xfdbb21c2),LL(0x2f9a67b3,0x756fa17f),LL(0xa245c1a8,0x2a6b2421), LL(0x4af02291,0x64be2794),LL(0x2a5804fe,0xade465c6),LL(0xa6f08fd7,0x8dffbd39),LL(0xaa14403b,0xc4efa84c),
   LL(0x442b0f5c,0xa1b91b2a),LL(0xcf997736,0xb748e317),LL(0xcee90e16,0x8d1b62bf),LL(0x0b2078c0,0x907ae271), LL(0x0c9bcddd,0xdf31534b),LL(0x39adce83,0x043fb054),LL(0xd826846a,0x99031043),LL(0xb144f393,0x61a9c0d6),
   LL(0x47718427,0xdab48046),LL(0x6e830f8b,0xdf17ff9b),LL(0xe49a1347,0x408d7ee8),LL(0x91c1d4ae,0x6ac71e23), LL(0x1defd73c,0xc8cbb9fd),LL(0xbbbbfec5,0x19840657),LL(0x9e7ef8ea,0x39db1cb5),LL(0x64105f30,0x78aa8296),
   LL(0xa3738c29,0xa3d9b7f0),LL(0xbc3250a3,0x0a2f235a),LL(0x445e4caf,0x55e506f6),LL(0x33475f7a,0x0974f73d), LL(0x5ba2f5a8,0xd37dbba3),LL(0x6af40066,0x542c6e63),LL(0xc5d73e2c,0x26d99b53),LL(0x6c3ca33e,0x06060d7d),
   LL(0x065fef4a,0xcdbef1c2),LL(0xfd5b92e3,0x77e60f7d),LL(0x26708350,0xd7c549f0),LL(0x34f121bf,0x201b3ad0), LL(0x0334fc14,0x5fcac2a1),LL(0x344552f6,0x8a9a9e09),LL(0x97653082,0x7dd8a1d3),LL(0x79d4f289,0x5fc0738f),
   LL(0x17d2d8c3,0x787d244d),LL(0x70830684,0xeffc6345),LL(0xe4f73ae5,0x5ddb96dd),LL(0x172549a5,0x8efb14b1), LL(0x2245ae7a,0x6eb73eee),LL(0xea11f13e,0xbca4061e),LL(0x30b01f5d,0xb577421d),LL(0x782e152c,0xaa688b24),
   LL(0xbd3502ba,0x67608e71),LL(0xb4de75a0,0x4ef41f24),LL(0xfd6125e5,0xb08dde5e),LL(0xa409543f,0xde484825), LL(0x65cc2295,0x1f198d98),LL(0x6e0edfa2,0x428a3771),LL(0xadf35fc7,0x4f9697a2),LL(0xf7cac3c7,0x01a43c79),
   LL(0x0fd3659a,0xb05d7059),LL(0xbb7f2d9a,0x8927f30c),LL(0x8cf984d3,0x4023d1ac),LL(0x02897a45,0x32125ed3), LL(0x3d414205,0xfb572dad),LL(0xe3fa82a9,0x73000ef2),LL(0xf10a5581,0x4c0868e9),LL(0x6b0b3ca5,0x5b61fc67),
   LL(0x7cae440c,0xc1258d5b),LL(0x402b7531,0x21c08b41),LL(0xde932321,0xf61a8955),LL(0x2d1408af,0x3568faf8), LL(0x9ecf965b,0x71b15e99),LL(0xe917276f,0xf14ed248),LL(0x820cf9e2,0xc6f4caa1),LL(0x18d83c7e,0x681b20b2),
   LL(0xc6c01120,0x6cde738d),LL(0xae70e0db,0x71db0813),LL(0x74afe18c,0x95fc0644),LL(0x129e2be7,0x34619053), LL(0xdb2a3b15,0x80615cea),LL(0xdb4c7073,0x0a49a19e),LL(0x8fd2d367,0x0e1b84c8),LL(0x033fb8aa,0xd74bf462),
   LL(0x533ef217,0x889f6d65),LL(0xc3ca2e87,0x7158c7e4),LL(0xdc2b4167,0xfb670dfb),LL(0x844c257f,0x75910a01), LL(0xcf88577d,0xf336bf07),LL(0xe45e2ace,0x22245250),LL(0x7ca23d85,0x2ed92e8d),LL(0x2b812f58,0x29f8be4c),
   LL(0x076fe12b,0xdd9ebaa7),LL(0xae1537f9,0x3f2400cb),LL(0x17bdfb46,0x1aa93528),LL(0x67883b41,0xc0f98430), LL(0x0170911d,0x5590ede1),LL(0x34d4b17f,0x7562f5bb),LL(0x1826b8d2,0xe1fa1df2),LL(0x6bd80d59,0xb40b796a),
   LL(0x3467ba92,0xd65bf197),LL(0xf70954b0,0x8c9b46db),LL(0x0e78f15d,0x97c8a0f3),LL(0x85a4c961,0xa8f3a69a), LL(0x61e4ce9b,0x4242660f),LL(0x6ea6790c,0xbf06aab3),LL(0xec986416,0xc6706f8e),LL(0x9a9fc225,0x9e56dec1),
   LL(0x9a9898d9,0x527c46f4),LL(0x5633cdef,0xd799e77b),LL(0x7d9e4297,0x24eacc16),LL(0x6b1cb734,0xabb61cea), LL(0xf778443c,0xbee2e8a7),LL(0x29de2fe6,0x3bb42bf1),LL(0x3003bb6f,0xcbed86a1),LL(0xd781cdf6,0xd3918e6c),
   LL(0x9a5103f1,0x4bee3271),LL(0xf50eac06,0x5243efc6),LL(0x6adcc119,0xb8e122cb),LL(0xc0b80a08,0x1b7faa84), LL(0x6dfcd08c,0x32c3d1bd),LL(0x0be427de,0x129dec4e),LL(0x1d263c83,0x98ab679c),LL(0xcef64eff,0xafc83cb7),
   LL(0x2fa6be76,0x85eb6088),LL(0x1328cbfe,0x892585fb),LL(0xcf618dda,0xc154d3ed),LL(0x3abaf26e,0xc44f601b), LL(0x2be1fdfd,0x7bf57d0b),LL(0x21137fee,0xa833bd2d),LL(0x2db591a8,0x9353af36),LL(0x5562a056,0xc76f26dc),
   LL(0x3fdf5a51,0x1d87e47d),LL(0x55c9cab0,0x7afb5f93),LL(0x89e0586e,0x91bbf58f),LL(0x0d843709,0x7c72c018), LL(0x99b5c3dc,0xa9a5aafb),LL(0x3844aeb0,0xa48a0f1d),LL(0xb667e482,0x7178b7dd),LL(0x6e23a59a,0x453985e9),
   LL(0x01b25dd8,0x4a54c860),LL(0xfb897c8a,0x0dd37f48),LL(0x0ea90cd9,0x5f8aa610),LL(0x16d5830d,0xc8892c68), LL(0xef514ca5,0xeb4befc0),LL(0xe72c9ee6,0x478eb679),LL(0xdbc40d5f,0x9bca20da),LL(0xdde4f64a,0xf015de21),
   LL(0xeaf4b8a5,0xaa6a4de0),LL(0x4bc60e32,0x68cfd9ca),LL(0x7fd15e70,0x668a4b01),LL(0xf27dc09d,0xd9f0694a), LL(0xba708bcd,0xf6c3cad5),LL(0x5bb95c2a,0x5cd2ba69),LL(0x33c0a58f,0xaa28c1d3),LL(0xabc77870,0x23e274e3),
   LL(0xdfd20a4a,0x44c3692d),LL(0x81a66653,0x091c5fd3),LL(0x09a0757d,0x6c0bb691),LL(0x667343ea,0x9072e8b9), LL(0x80848bec,0x31d40eb0),LL(0x79fd36cc,0x95bd480a),LL(0x65ed43f5,0x01a77c61),LL(0x2e0d40bf,0xafccd127),
   LL(0x1cc1884b,0xeccfc82d),LL(0x5d4753b4,0xc85ac201),LL(0x658e099f,0xc7a6caac),LL(0x04b27390,0xcf46369e), LL(0x506467ea,0xe2e7d049),LL(0x37cdeccc,0x481b63a2),LL(0xed80143a,0x4029abd8),LL(0xbcb00b88,0x28bfe3c7),
   LL(0x0643d84a,0x3bec1009),LL(0xabd11041,0x885f3668),LL(0xf83a34d6,0xdb02432c),LL(0x719ceebe,0x32f7b360), LL(0xdad1fe7a,0xf06c7837),LL(0x5441a0b0,0x60a157a9),LL(0xe2d47550,0x704970e9),LL(0x271b9020,0xcd2bd553),
   LL(0x33e24a0b,0xff57f82f),LL(0xf2565079,0x9cbee23f),LL(0xeb5f5825,0x16353427),LL(0xe948d662,0x276feec4), LL(0xda10032b,0xd1b62bc6),LL(0xf0e72a53,0x718351dd),LL(0x2420e7ba,0x93452076),LL(0x3a00118d,0x96368fff),
   LL(0x150a49e4,0x00ce2d26),LL(0x3f04706b,0x0c28b636),LL(0x58b196d0,0xbad65a46),LL(0xec9f8b7c,0x6c8455fc), LL(0x2d71867e,0xe90c895f),LL(0xedf9f38c,0x5c0be31b),LL(0xd8f6ec04,0x2a37a15e),LL(0x8cd85251,0x239639e7),
   LL(0x9c7c4c6b,0xd8975315),LL(0xd7409af7,0x603aa3c0),LL(0x007132fb,0xb8d53d0c),LL(0xa6849238,0x68d12af7), LL(0xbf5d9279,0xbe0607e7),LL(0xaada74ce,0x9aa50055),LL(0xba7e8ccb,0xe81079cb),LL(0xa5f4ff5e,0x610c71d1),
   LL(0x5aa07093,0x9e2ee1a7),LL(0xa75da47c,0xca84004b),LL(0x3de75401,0x074d3951),LL(0xbb311592,0xf938f756), LL(0x00a43421,0x96197618),LL(0x07bc78c8,0x39a25362),LL(0x0a171276,0x278f710a),LL(0x8d1a8f08,0xb28446ea),
   LL(0xe3b6a661,0x184781bf),LL(0xe6d279f7,0x7751cb1d),LL(0xc59eb662,0xf8ff95d6),LL(0x58d3dea7,0x186d90b7), LL(0xdfb4f754,0x0e4bb6c1),LL(0x2b2801dc,0x5c5cf56b),LL(0x1f54564d,0xc561e452),LL(0xf0dd7f13,0xb4fb8c60),
   LL(0x33ff98c7,0xf8849630),LL(0xcf17769c,0x9619fffa),LL(0x1bfdd80a,0xf8090bf6),LL(0x422cfe63,0x14d9a149), LL(0x6f6df9ea,0xb354c360),LL(0x218f17ea,0xdbcf770d),LL(0x79eb3480,0x207db7c8),LL(0x559b6a26,0x213dbda8),
   LL(0x29fc81b3,0xac4c200b),LL(0x171d87c1,0xebc3e09f),LL(0x1481aa9e,0x91799530),LL(0x92e114fa,0x051b92e1), LL(0xecb5537f,0xdf8f92e9),LL(0x290c7483,0x44b1b2cc),LL(0x2adeb016,0xa711455a),LL(0x81a10c2c,0x964b6856),
   LL(0xcec03623,0x4f159d99),LL(0xef3271ea,0x05532225),LL(0xc5ee4849,0xb231bea3),LL(0x7094f103,0x57a54f50), LL(0x9598b352,0x3e2d421d),LL(0x67412ab4,0xe865a49c),LL(0x1cc3a912,0xd2998a25),LL(0x0c74d65d,0x5d092808),
   LL(0x4088567a,0x73f45908),LL(0x1f214a61,0xeb6b280e),LL(0xcaf0c13d,0x8c9adc34),LL(0xf561fb80,0x39d12938), LL(0xbc6edfb4,0xb2dc3a5e),LL(0xfe4d210e,0x7485b1b1),LL(0xe186ae72,0x062e0400),LL(0x6eeb3b88,0x91e32d5c),
   LL(0x4be59224,0x6df574d7),LL(0x716d55f3,0xebc88ccc),LL(0xcad6ed33,0x26c2e6d0),LL(0x0d3e8b10,0xc6e21e7d), LL(0x5bcc36bb,0x2cc5840e),LL(0x7da74f69,0x9292445e),LL(0x4e5193a8,0x8be8d321),LL(0x8df06413,0x3ec23629),
   LL(0xb134defa,0xc7e9ae85),LL(0x1bb2d475,0x6073b1d0),LL(0x2863c00d,0xb9ad615e),LL(0x525f4ac4,0x9e29493d), LL(0x4e9acf4f,0xc32b1dea),LL(0xa50db88d,0x3e1f01c8),LL(0x04da916c,0xb05d70ea),LL(0xd865803e,0x714b0d0a),
   LL(0x9920cb5e,0x4bd493fc),LL(0x92c7a3ac,0x5b44b1f7),LL(0xbcec9235,0xa2a77293),LL(0xcd378553,0x5ee06e87), LL(0xda621607,0xceff8173),LL(0x99f5d290,0x2bb03e4c),LL(0xa6f734ac,0x2945106a),LL(0xd25c4732,0xb5056604),
   LL(0xe079afee,0x5945920c),LL(0x6789831f,0x686e17a0),LL(0xb74a5ae5,0x5966bee8),LL(0x1e258d46,0x38a673a2), LL(0x83141c95,0xbd1cc1f2),LL(0x0e96e486,0x3b2ecf4f),LL(0x74e5fc78,0xcd3aa896),LL(0x2482fa7a,0x415ec10c),
   LL(0x80503380,0x15234419),LL(0xd314b392,0x513d917a),LL(0x63caecae,0xb0b52f4e),LL(0x2dc7780b,0x07bf22ad), LL(0xe4306839,0xe761e8a1),LL(0x5dd7feaa,0x1b3be962),LL(0x74c778f1,0x4fe728de),LL(0x5e0070f6,0xf1fa0bda),
   LL(0x6ec3f510,0x85205a31),LL(0xd2980475,0x2c7e4a14),LL(0x6f30ebfd,0xde3c19c0),LL(0xd4b7e644,0xdb1c1f38), LL(0x5dce364a,0xfe291a75),LL(0x058f5be3,0xb7b22a3c),LL(0x37fea38c,0x2cd2c302),LL(0x2e17be17,0x2930967a),
   LL(0x0c061c65,0x87f009de),LL(0xedc6ed44,0xcb014aac),LL(0x3bafb1eb,0x49bd1cb4),LL(0x282d3688,0x81bd8b5c), LL(0xf01a17af,0x1cdab87e),LL(0xe710063b,0x21f37ac4),LL(0x42fc8193,0x5a6c5676),LL(0x56a6015c,0xf4753e70),
   LL(0xa15b0a44,0x020f795e),LL(0x8958a958,0x8f37c8d7),LL(0xa4b675b5,0x63b7e89b),LL(0x0fc31aea,0xb4fb0c0c), LL(0xa7ff1f2e,0xed95e639),LL(0x619614fb,0x9880f5a3),LL(0x947151ab,0xdeb6ff02),LL(0xa868dcdb,0x5bc5118c),
   LL(0x4c20cea5,0xd8da2055),LL(0x14c4d69a,0xcac2776e),LL(0x622d599b,0xcccb22c1),LL(0x68a9bb50,0xa4ddb653), LL(0x1b4941b4,0x2c4ff151),LL(0x6efba588,0xe1ff19b4),LL(0xc48345e0,0x35034363),LL(0x1e29dfc4,0x45542e3d),
   LL(0x349f7aed,0xf197cb91),LL(0x8fca8420,0x3b2b5a00),LL(0x23aaf6d8,0x7c175ee8),LL(0x35af32b6,0x54dcf421), LL(0x27d6561e,0x0ba14307),LL(0xd175b1e2,0x879d5ee4),LL(0x99807db5,0xc7c43673),LL(0x9cd55bcd,0x77a54455),
   LL(0x0105c072,0xe6c2ff13),LL(0x8dda7da4,0x18f7a99f),LL(0x0e2d35c1,0x4c301820),LL(0xd9cc6c82,0x06a53ca0), LL(0xf1aa1d9e,0xaa21cc1e),LL(0x4a75b1e8,0x32414334),LL(0x0ebe9fdc,0x2a6d1328),LL(0x98a4755a,0x16bd173f),
   LL(0x2133ffd9,0xfbb9b245),LL(0x830f1a20,0x39a8b2f1),LL(0xd5a1f52a,0x484bc97d),LL(0xa40eddf8,0xd6aebf56), LL(0x76ccdac6,0x32257acb),LL(0x1586ff27,0xaf4d36ec),LL(0xf8de7dd1,0x8eaa8863),LL(0x88647c16,0x0045d5cf),
},
/* digit=6 base_pwr=2^42 */
{
   LL(0xc005979d,0xa6f3d574),LL(0x6a40e350,0xc2072b42),LL(0x8de2ecf9,0xfca5c156),LL(0xa515344e,0xa8c8bf5b), LL(0x114df14a,0x97aee555),LL(0xfdc5ec6b,0xd4374a4d),LL(0x2ca85418,0x754cc28f),LL(0xd3c41f78,0x71cb9e27),
   LL(0x03605c39,0x89105079),LL(0xa142c96c,0xf0843d9e),LL(0x16923684,0xf3744934),LL(0xfa0a2893,0x732caa2f), LL(0x61160170,0xb2e8c270),LL(0x437fbaa3,0xc32788cc),LL(0xa6eda3ac,0x39cd818e),LL(0x9e2b2e07,0xe2e94239),
   LL(0x0260e52a,0x6967d39b),LL(0x90653325,0xd42585cc),LL(0x21ca7954,0x0d9bd605),LL(0x81ed57b3,0x4fa20877), LL(0xe34a0bbe,0x60c1eff8),LL(0x84f6ef64,0x56b0040c),LL(0xb1af8483,0x28be2b24),LL(0xf5531614,0xb2278163),
   LL(0x5922ac1c,0x8df27545),LL(0xa52b3f63,0xa7b3ef5c),LL(0x71de57c4,0x8e77b214),LL(0x834c008b,0x31682c10), LL(0x4bd55d31,0xc76824f0),LL(0x17b61c71,0xb6d1c086),LL(0xc2a5089d,0x31db0903),LL(0x184e5d3f,0x9c092172),
   LL(0xc00cc638,0xdd7ced5b),LL(0x61278fc2,0x1a2015eb),LL(0x6a37f8d6,0x2e8e5288),LL(0xe79933ad,0xc457786f), LL(0x2c51211a,0xb3fe4cce),LL(0x24c20498,0xad9b10b2),LL(0xd28db5e5,0x90d87a4f),LL(0x3aca2fc3,0x698cd105),
   LL(0xe91b536d,0x4f112d07),LL(0x9eba09d6,0xceb982f2),LL(0x197c396f,0x3c157b2c),LL(0x7b66eb24,0xe23c2d41), LL(0x3f330d37,0x480c57d9),LL(0x79108deb,0xb3a4c8a1),LL(0xcb199ce5,0x702388de),LL(0xb944a8d4,0x0b019211),
   LL(0x840bb336,0x24f2a692),LL(0xa669fa7b,0x7c353bdc),LL(0xdec9c300,0xda20d6fc),LL(0xa13a4f17,0x625fbe2f), LL(0xdbc17328,0xa2b1b61a),LL(0xa9515621,0x008965bf),LL(0xc620ff46,0x49690939),LL(0x8717e91c,0x182dd27d),
   LL(0xea6c3997,0x5ace5035),LL(0xc2610bef,0x54259aaa),LL(0x3c80dd39,0xef18bb3f),LL(0x5fc3fa39,0x6910b95b), LL(0x43e09aee,0xfce2f510),LL(0xa7675665,0xced56c9f),LL(0xd872db61,0x10e265ac),LL(0xae9fce69,0x6982812e),
   LL(0xce800998,0x29be11c6),LL(0xb90360d9,0x72bb1752),LL(0x5a4ad590,0x2c193197),LL(0x9fc1dbc0,0x2ba2f548), LL(0xe490ebe0,0x7fe4eebb),LL(0x7fae11c0,0x12a0a4cd),LL(0xe903ba37,0x7197cf81),LL(0xde1c6dd8,0xcf7d4aa8),
   LL(0x3fd5684c,0x92af6bf4),LL(0x80360aa1,0x2b26eecf),LL(0x00546a82,0xbd960f30),LL(0xf59ad8fe,0x407b3c43), LL(0x249c82ba,0x86cae5fe),LL(0x2463744c,0x9e0faec7),LL(0x94916272,0x87f551e8),LL(0x6ceb0615,0x033f9344),
   LL(0x8be82e84,0x1e5eb0d1),LL(0x7a582fef,0x89967f0e),LL(0xa6e921fa,0xbcf687d5),LL(0xd37a09ba,0xdfee4cf3), LL(0xb493c465,0x94f06965),LL(0x7635c030,0x638b9a1c),LL(0x66f05e9f,0x76667864),LL(0xc04da725,0xccaf6808),
   LL(0x768fccfc,0xca2eb690),LL(0xb835b362,0xf402d37d),LL(0xe2fdfcce,0x0efac0d0),LL(0xb638d990,0xefc9cdef), LL(0xd1669a8b,0x2af12b72),LL(0x5774ccbd,0x33c536bc),LL(0xfb34870e,0x30b21909),LL(0x7df25aca,0xc38fa2f7),
   LL(0xbf81f3f5,0x74c5f02b),LL(0xaf7e4581,0x0525a5ae),LL(0x433c54ae,0x88d2aaba),LL(0x806a56c5,0xed9775db), LL(0xc0edb37d,0xd320738a),LL(0x66cc1f51,0x25fdb6ee),LL(0x10600d76,0xac661d17),LL(0xbdd1ed76,0x931ec1f3),
   LL(0x19ee43f1,0x65c11d62),LL(0x60829d97,0x5cd57c3e),LL(0x984be6e8,0xd26c91a3),LL(0x8b0c53bd,0xf08d9309), LL(0xc016e4ea,0x94bc9e5b),LL(0x11d43d2b,0xd3916839),LL(0x73701155,0x886c5ad7),LL(0x20b00715,0xe0377626),
   LL(0xaa80ba59,0x7f01c9ec),LL(0x68538e51,0x3083411a),LL(0xe88128af,0x970370f1),LL(0x91dec14b,0x625cc3db), LL(0x01ac3107,0xfef9666c),LL(0xd5057ac3,0xb2a8d577),LL(0x92be5df7,0xb0f26299),LL(0x00353924,0xf579c8e5),
   LL(0x1341ed7a,0xb8fa3d93),LL(0xa7b59d49,0x4223272c),LL(0x83b8c4a4,0x3dcb1947),LL(0xed1302e4,0x4e413c01), LL(0xe17e44ce,0x6d999127),LL(0x33b3adfb,0xee86bf75),LL(0x25aa96ca,0xf6902fe6),LL(0xe5aae47d,0xb73540e4),
   LL(0x1b4a158c,0x32801d7b),LL(0x27e2a369,0xe571c99e),LL(0x10d9f197,0x40cb76c0),LL(0x3167c0ae,0xc308c289), LL(0xeb7958f2,0xa6ef9dd3),LL(0x300879b1,0xa7226dfc),LL(0x7edf0636,0x6cd0b362),LL(0x7bc37eed,0x4efbce6c),
   LL(0x8d699021,0x75f92a05),LL(0x772566e3,0x586d4c79),LL(0x761ad23a,0x378ca5f1),LL(0x1465a8ac,0x650d86fc), LL(0x842ba251,0x7a4ed457),LL(0x42234933,0x6b65e3e6),LL(0x31aad657,0xaf1543b7),LL(0xcbfec369,0xa4cefe98),
   LL(0x9f47befb,0xb587da90),LL(0x41312d13,0x6562e9fb),LL(0xeff1cefe,0xa691ea59),LL(0x05fc4cf6,0xcc30477a), LL(0x0b0ffd3d,0xa1632461),LL(0x5b355956,0xa1f16f3b),LL(0x4224ec24,0x5b148d53),LL(0xf977012a,0xdc834e7b),
   LL(0xb2c69dbc,0x7bfc5e75),LL(0x03c3da6c,0x3aa77a29),LL(0xca910271,0xde0df03c),LL(0x7806dc55,0xcbd5ca4a), LL(0x6db476cb,0xe1ca5807),LL(0x5f37a31e,0xfde15d62),LL(0xf41af416,0xf49af520),LL(0x7d342db5,0x96c5c5b1),
   LL(0xeb4ceb9b,0x155c43b7),LL(0x4e77371a,0x2e993010),LL(0x675d43af,0x1d2987da),LL(0x8599fd72,0xef2bc1c0), LL(0x9342f6b2,0x96894b7b),LL(0x7c8e71f0,0x201eadf2),LL(0x4a1f3efc,0xf3479d9f),LL(0x702a9704,0xe0f8a742),
   LL(0xb3eba40c,0xeafd44b6),LL(0xc1c1e0d0,0xf9739f29),LL(0x619d505e,0x0091471a),LL(0x9d7c263e,0xc15f9c96), LL(0x83afbe33,0x5be47285),LL(0x04f1e092,0xa3b6d6af),LL(0x751a9d11,0xe76526b9),LL(0x9a4ae4d2,0x2ec5b26d),
   LL(0x02f6fb8d,0xeb66f4d9),LL(0x96912164,0x4063c561),LL(0x80ef3000,0xeb7050c1),LL(0xeaa5b3f0,0x288d1c33), LL(0x07806fd8,0xe87c68d6),LL(0x4bbbf50f,0xb2f7f9d5),LL(0xac8d6627,0x25972f3a),LL(0x10e8c13b,0xf8547774),
   LL(0x872b4a60,0xcc50ef6c),LL(0x4613521b,0xab2a34a4),LL(0x983e15d1,0x39c5c190),LL(0x59905512,0x61dde5df), LL(0x9f2275f3,0xe417f621),LL(0x451d894b,0x0750c8b6),LL(0x78b0bdaa,0x75b04ab9),LL(0x458589bd,0x3bfd9fd4),
   LL(0xee9120b6,0xf1013e30),LL(0x23a4743e,0x2b51af93),LL(0x48d14d9e,0xea96ffae),LL(0x698a1d32,0x71dc0dbe), LL(0x0180cca4,0x914962d2),LL(0xc3568963,0x1ae60677),LL(0x437bc444,0x8cf227b1),LL(0xc9962c7a,0xc650c83b),
   LL(0xfe7ccfc4,0x23c2c7dd),LL(0x1b929d48,0xf925c89d),LL(0x06783c33,0x4460f74b),LL(0xa590475a,0xac2c8d49), LL(0xb807bba0,0xfb40b407),LL(0x69ff8f3a,0x9d1e362d),LL(0xcbef64a4,0xa33e9681),LL(0x332fb4b2,0x67ece5fa),
   LL(0x739f10e3,0x6900a99b),LL(0xff525925,0xc3341ca9),LL(0xa9e2d041,0xee18a626),LL(0x29580ddd,0xa5a83685), LL(0x9d7de3cd,0xf3470c81),LL(0x2062cf9c,0xedf02586),LL(0xc010edb0,0xf43522fa),LL(0x13a4b1ae,0x30314135),
   LL(0xdb22b94b,0xc792e02a),LL(0xa1eaa45b,0x993d8ae9),LL(0xcd1e1c63,0x8aad6cd3),LL(0xc5ce688a,0x89529ca7), LL(0xe572a253,0x2ccee3aa),LL(0x02a21efb,0xe02b6438),LL(0xc9430358,0xa7091b6e),LL(0x9d7db504,0x06d1b1fa),
   LL(0xc4744733,0x58846d32),LL(0x379f9e34,0x40517c71),LL(0x130ef6ca,0x2f65655f),LL(0xf1f3503f,0x526e4488), LL(0x7ee4a976,0x8467bd17),LL(0x921363d1,0x1d9dc913),LL(0xb069e041,0xd8d24c33),LL(0x2cdf7f51,0x5eb5da0a),
   LL(0x197b994f,0x1c0f3cb1),LL(0x2843eae9,0x3c95a6c5),LL(0xa6097ea5,0x7766ffc9),LL(0xd723b867,0x7bea4093), LL(0x4db378f9,0xb48e1f73),LL(0xe37b77ac,0x70025b00),LL(0xaf24ad46,0x943dc8e7),LL(0x16d00a85,0xb98a15ac),
   LL(0x2743b004,0x3adc38ba),LL(0x334415ee,0xb1c7f4f7),LL(0x1e62d05a,0xea43df8f),LL(0x9d76a3b6,0x32618905), LL(0xa23a0f46,0x2fbd0bb5),LL(0x6a01918c,0x5bc971db),LL(0xb4743f94,0x7801d94a),LL(0x676ae22b,0xb94df65e),
   LL(0xaf95894c,0xaafcbfab),LL(0x276b2241,0x7b9bdc07),LL(0x5bdda48b,0xeaf98362),LL(0xa3fcb4df,0x5977faf2), LL(0x052c4b5b,0xbed042ef),LL(0x067591f0,0x9fe87f71),LL(0x22f24ec7,0xc89c73ca),LL(0xe64a9f1b,0x7d37fa9e),
   LL(0x15562627,0x2710841a),LL(0xc243b034,0x2c01a613),LL(0x2bc68609,0x1d135c56),LL(0x8b03f1f6,0xc2ca1715), LL(0x3eb81d82,0xc9966c2d),LL(0x8f6df13e,0xc02abf4a),LL(0x8f72b43b,0x77b34bd7),LL(0x360c82b0,0xaff6218f),
   LL(0x8d55b9d2,0x0aa5726c),LL(0x99e9bffb,0xdc0adbe9),LL(0xefb9e72a,0x9097549c),LL(0x9dfb3111,0x16755712), LL(0xf26847f9,0xdd8bf984),LL(0xdfb30cb7,0xbcb8e387),LL(0x5171ef9c,0xc1fd32a7),LL(0x389b363f,0x977f3fc7),
   LL(0xf4babda0,0x116eaf2b),LL(0xf7113c8e,0xfeab68bd),LL(0xb7def526,0xd1e3f064),LL(0xe0b3fa02,0x1ac30885), LL(0x40142d9d,0x1c5a6e7b),LL(0x30921c0b,0x839b5603),LL(0x36a116a3,0x48f301fa),LL(0xcfd9ee6d,0x380e1107),
   LL(0x58854be1,0x7945ead8),LL(0xcbd4d49d,0x4111c12e),LL(0x3a29c2ef,0xece3b1ec),LL(0x8d3616f5,0x6356d404), LL(0x594d320e,0x9f0d6a8f),LL(0xf651ccd2,0x0989316d),LL(0x0f8fdde4,0x6c32117a),LL(0xa26a9bbc,0x9abe5cc5),
   LL(0x9723f671,0xcff560fb),LL(0x7f3d593c,0x21b2a12d),LL(0x24ba0696,0xe4cb18da),LL(0xc3543384,0x186e2220), LL(0x88312c29,0x722f64e0),LL(0x17dc7752,0x94282a99),LL(0x5a85ee89,0x62467bbf),LL(0xf10076a0,0xf435c650),
   LL(0x43b3a50b,0xc9ff1539),LL(0x1a53efbc,0x7132130c),LL(0xf7b0c5b7,0x31bfe063),LL(0x4ea994cc,0xb0179a7d), LL(0xc85f455b,0x12d064b3),LL(0x8f6e0062,0x47259328),LL(0xb875d6d9,0xf64e590b),LL(0xad92bcc7,0x22dd6225),
   LL(0xb9c3bd6d,0xb658038e),LL(0xfbba27c8,0x00cdb0d6),LL(0x1062c45d,0x0c681337),LL(0x2d33407d,0xd8515b8c), LL(0x8cbb5ecf,0xcb8f699e),LL(0xc608d7d8,0x8c4347f8),LL(0xbb3e00db,0x2c11850a),LL(0xecb49d19,0x20a8dafd),
   LL(0x45ee2f40,0xbd781480),LL(0x416b60cf,0x75e354af),LL(0x8d49a8c4,0xde0b58a1),LL(0xfa359536,0xe40e94e2), LL(0x62accd76,0xbd4fa59f),LL(0x8c762837,0x05cf466a),LL(0x448c277b,0xb5abda99),LL(0x48b13740,0x5a9e01bf),
   LL(0x326aad8d,0x9d457798),LL(0xc396f7e7,0xbdef4954),LL(0xc253e292,0x6fb274a2),LL(0x1cfe53e7,0x2800bf0a), LL(0x44438fd4,0x22426d31),LL(0x5e259f9a,0xef233923),LL(0x03f66264,0x4188503c),LL(0x7f9fdfab,0x9e5e7f13),
   LL(0x5fcc1aba,0x565eb76c),LL(0x59b5bff8,0xea632548),LL(0xaab6d3fa,0x5587c087),LL(0x6ce39c1b,0x92b639ea), LL(0x953b135c,0x0706e782),LL(0x425268ef,0x7308912e),LL(0x090e7469,0x599e92c7),LL(0x9bc35e75,0x83b90f52),
   LL(0x244975b3,0x4750b3d0),LL(0x11965d72,0xf3a44358),LL(0x9c8dc751,0x179c6774),LL(0xd23d9ff0,0xff18cdfe), LL(0x2028e247,0xc4013833),LL(0xf3bfbc79,0x96e280e2),LL(0xd0880a84,0xf60417bd),LL(0x2a568151,0x263c9f3d),
   LL(0x2d2ce811,0x36be15b3),LL(0xf8291d21,0x846dc0c2),LL(0x789fcfdb,0x5cfa0ecb),LL(0xd7535b9a,0x45a0beed), LL(0x96d69af1,0xec8e9f07),LL(0x599ab6dc,0x31a7c5b8),LL(0xf9e2e09f,0xd36d45ef),LL(0xdcee954b,0x3cf49ef1),
   LL(0x086cff9b,0x6be34cf3),LL(0x39a3360f,0x88dbd491),LL(0x0dbfbd1d,0x1e96b8cc),LL(0xcb7e2552,0xc1e5f7bf), LL(0x28819d98,0x0547b214),LL(0x7aea9dcb,0xc770dd9c),LL(0x041d68c8,0xaef0d4c7),LL(0x13cb9ba8,0xcc2b9818),
   LL(0xfe86c607,0x7fc7bc76),LL(0x502a9a95,0x6b7b9337),LL(0xd14dab63,0x1948dc27),LL(0xdae047be,0x249dd198), LL(0xa981a202,0xe8356584),LL(0x3a893387,0x3531dd18),LL(0xc85c7209,0x1be11f90),LL(0xe2a52b5a,0x93d2fe1e),
   LL(0xec6d6b97,0x8225bfe2),LL(0xbd0aa5de,0x9cf6d6f4),LL(0x54779f5f,0x911459cb),LL(0x86aeb1f3,0x5649cddb), LL(0x3f26ce5a,0x32133579),LL(0x550f431e,0xc289a102),LL(0x73b84c6f,0x559dcfda),LL(0xee3ac4d7,0x84973819),
   LL(0xf2606a82,0xb51e55e6),LL(0x90f2fb57,0xe25f7061),LL(0xb1a4e37c,0xacef6c2a),LL(0x5dcf2706,0x864e359d), LL(0x7ce57316,0x479e6b18),LL(0x3a96b23d,0x2cab2500),LL(0x8ef16df7,0xed489862),LL(0xef3758b5,0x2056538c),
   LL(0xf15d3101,0xa7df865e),LL(0x61b553d7,0x80c5533a),LL(0x4ed14294,0x366e1997),LL(0xb3c0bcd6,0x6620741f), LL(0xedc45418,0x21d1d9c4),LL(0xc1cc4a9d,0x005b859e),LL(0xa1c462f0,0xdf01f630),LL(0xf26820c7,0x15d06cf3),
   LL(0x3484be47,0x9f7f24ee),LL(0x4a0c902f,0x2ff33e96),LL(0x5a0bc453,0x00bdf457),LL(0x1aa238db,0x2378dfaf), LL(0x856720f2,0x272420ec),LL(0x96797291,0x2ad9d95b),LL(0x768a1558,0xd1242cc6),LL(0x5cc86aa8,0x2e287f8b),
   LL(0x990cecaa,0x796873d0),LL(0x675d4080,0xade55f81),LL(0x21f0cd84,0x2645eea3),LL(0xb4e17d02,0x7a1efa0f), LL(0x037cc061,0xf6858420),LL(0xd5d43e12,0x682e05f0),LL(0x27218710,0x59c36994),LL(0x3f7cd2fc,0x85cbba4d),
   LL(0x7a3cd22a,0x726f9729),LL(0x4a628397,0x9f8cd5dc),LL(0xc23165ed,0x17b93ab9),LL(0x122823d4,0xff5f5dbf), LL(0x654a446d,0xc1e4e4b5),LL(0x677257ba,0xd1a9496f),LL(0xde766a56,0x6387ba94),LL(0x521ec74a,0x23608bc8),
   LL(0x6688c4d4,0x16a522d7),LL(0x07373abd,0x9d6b4282),LL(0xb42efaa3,0xa62f07ac),LL(0xe3b90180,0xf73e00f7), LL(0x49421c3e,0x36175fec),LL(0x3dcf2678,0xc4e44f9b),LL(0x7220f09f,0x76df436b),LL(0x3aa8b6cf,0x172755fb),
   LL(0x446139cc,0xbab89d57),LL(0x5fe0208f,0x0a0a6e02),LL(0x11e5d399,0xcdbb63e2),LL(0xa8977f0b,0x33ecaa12), LL(0xf7c42664,0x59598b21),LL(0xab65d08a,0xb3e91b32),LL(0xf4502526,0x035822ee),LL(0x720a82a9,0x1dcf0176),
   LL(0x3d589e02,0x50f8598f),LL(0xb1d63d2c,0xdf0478ff),LL(0x1571cd07,0x8b8068bd),LL(0xd79670cd,0x30c3aa4f), LL(0x941ade7f,0x25e8fd4b),LL(0x32790011,0x3d1debdc),LL(0x3a3f9ff0,0x65b6dcbd),LL(0x793de69c,0x282736a4),
   LL(0xd41d3bd3,0xef69a0c3),LL(0x07a26bde,0xb533b8c9),LL(0xdb2edf9f,0xe2801d97),LL(0xe1877af0,0xdc4a8269), LL(0x3d590dbe,0x6c1c5851),LL(0xee4e9357,0x84632f6b),LL(0x79b33374,0xd36d36b7),LL(0x9bbca2e6,0xb46833e3),
   LL(0xf7fc0586,0x37893913),LL(0x66bf4719,0x385315f7),LL(0xb31855dc,0x72c56293),LL(0x849061fe,0xd1416d4e), LL(0x51047213,0xbeb3ab78),LL(0xf040c996,0x447f6e61),LL(0x638b1d0c,0xd06d310d),LL(0xbad1522e,0xe28a413f),
   LL(0x82003f86,0x685a76cb),LL(0x0bcdbca3,0x610d07f7),LL(0x9ca4c455,0x6ff66021),LL(0xcea10eec,0x7df39b87), LL(0xe22db218,0xb9255f96),LL(0x08a34c44,0x8cc6d9eb),LL(0x859f9276,0xcd4ffb86),LL(0x50d07335,0x8fa15eb2),
   LL(0xcf2c24b5,0xdf553845),LL(0x52f9c3ba,0x89f66a9f),LL(0xe4a7ceb3,0x8f22b5b9),LL(0x0e134686,0xaffef809), LL(0x8eb8fac2,0x3e53e1c6),LL(0x28aec98e,0x93c1e4eb),LL(0x32a43bcb,0xb6b91ec5),LL(0xb2d74a51,0x2dbfa947),
   LL(0xca84bad7,0xe065d190),LL(0xad58e65c,0xfb13919f),LL(0xf1cb6e31,0x3c41718b),LL(0x06d05c3f,0x688969f0), LL(0x21264d45,0xd4f94ce7),LL(0x7367532b,0xfdfb65e9),LL(0x0945a39d,0x5b1be8b1),LL(0x2b8baf3b,0x229f789c),
   LL(0x6f49f15d,0xd8f41f3e),LL(0x907f0792,0x678ce828),LL(0xfca6e867,0xc69ace82),LL(0xd01dcc89,0x106451ae), LL(0x19fc32d2,0x1bb4f7f0),LL(0xb00c52d2,0x64633dfc),LL(0xad9ea445,0x8f13549a),LL(0xfb323705,0x99a3bf50),
   LL(0x534d4dbc,0x0c9625a2),LL(0xc2a2fea3,0x45b8f1d1),LL(0xa530fc1a,0x76ec21a1),LL(0x9e5bd734,0x4bac9c2a), LL(0x7b4e3587,0x5996d76a),LL(0x1182d9e3,0x0045cdee),LL(0x1207f13d,0x1aee24b9),LL(0x97345a41,0x66452e97),
   LL(0x9f950cd0,0x16e5b054),LL(0xd7fdd075,0x9cc72fb1),LL(0x66249663,0x6edd61e7),LL(0xf043cccb,0xde4caa4d), LL(0x55c7ac17,0x11b1f57a),LL(0x1a85e24d,0x779cbd44),LL(0xe46081e7,0x78030f86),LL(0x8e20f643,0xfd4a6032),
   LL(0x0a750c0f,0xcc7a6488),LL(0x4e548e83,0x39bacfe3),LL(0x0c110f05,0x3d418c76),LL(0xb1f11588,0x3e4daa4c), LL(0x5ffc69ff,0x2733e7b5),LL(0x92053127,0x46f147bc),LL(0xd722df94,0x885b2434),LL(0xe6fc6b7c,0x6a444f65),
},
/* digit=7 base_pwr=2^49 */
{
   LL(0xc3f16ea8,0x7a1a465a),LL(0xb2f1d11c,0x115a461d),LL(0x6c68a172,0x4767dd95),LL(0xd13a4698,0x3392f2eb), LL(0xe526cdc7,0xc7a99ccd),LL(0x22292b81,0x8e537fdc),LL(0xa6d39198,0x76d8cf69),LL(0x2446852d,0xffc5ff43),
   LL(0xa90567e6,0x97b14f7e),LL(0xb6ae5cb7,0x513257b7),LL(0x9f10903d,0x85454a3c),LL(0x69bc3724,0xd8d2c9ad), LL(0x6b29cb44,0x38da9324),LL(0x77c8cbac,0xb540a21d),LL(0x01918e42,0x9bbfe435),LL(0x56c3614e,0xfffa707a),
   LL(0xd4e353b7,0x0ce4e3f1),LL(0xef46b0a0,0x062d8a14),LL(0x574b73fd,0x6408d5ab),LL(0xd3273ffd,0xbc41d1c9), LL(0x6be77800,0x3538e1e7),LL(0xc5655031,0x71fe8b37),LL(0x6b9b331a,0x1cd91621),LL(0xbb388f73,0xad825d0b),
   LL(0x1cb76219,0x56c2e05b),LL(0x71567e7e,0x0ec0bf91),LL(0x61c4c910,0xe7076f86),LL(0xbabc04d9,0xd67b085b), LL(0x5e93a96a,0x9fb90459),LL(0xfbdc249a,0x7526c1ea),LL(0xecdd0bb7,0x0d44d367),LL(0x9dc0d695,0x95399917),
   LL(0x9e240d18,0x61360ee9),LL(0xb4b94466,0x057cdcac),LL(0x2fe5325c,0xe7667cd1),LL(0x21974e3b,0x1fa297b5), LL(0xdb083d76,0xfa4081e7),LL(0xf206bd15,0x31993be6),LL(0x14c19f8c,0x8949269b),LL(0xa9d92357,0x21468d72),
   LL(0xa4c506ec,0x2ccbc583),LL(0xd1acfe97,0x957ed188),LL(0x12f1aea2,0x8baed833),LL(0x8325362d,0xef2a6cb4), LL(0x8e195c43,0x130dde42),LL(0x0e6050c6,0xc842025a),LL(0x08686a5d,0x2da972a7),LL(0xe508b4a8,0xb52999a1),
   LL(0x10a5a8bd,0xd9f090b9),LL(0x096864da,0xca91d249),LL(0x3f67dbc1,0x8e6a93be),LL(0xf5f4764c,0xacae6fba), LL(0xd21411a0,0x1563c6e0),LL(0xda0a4ad8,0x28fa787f),LL(0x908c8030,0xd524491c),LL(0x4c795f07,0x1257ba0e),
   LL(0xceca9754,0x83f49167),LL(0x4b7939a0,0x426d2cf6),LL(0x723fd0bf,0x2555e355),LL(0xc4f144e2,0xa96e6d06), LL(0x87880e61,0x4768a8dd),LL(0xe508e4d5,0x15543815),LL(0xb1b65e15,0x09d7e772),LL(0xac302fa0,0x63439dd6),
   LL(0xc14e35c2,0xb93f802f),LL(0x4341333c,0x71735b7c),LL(0x16d4f362,0x03a25104),LL(0xbf433c8e,0x3f4d069b), LL(0xf78f5a7c,0x0d83ae01),LL(0x7c4eed07,0x50a8ffbe),LL(0x76e10f83,0xc74f8906),LL(0x9ddaf8e1,0x7d080966),
   LL(0x698e04cc,0xb11df8e1),LL(0x169005c8,0x877be203),LL(0x4f3c6179,0x32749e8c),LL(0x7853fc05,0x2dbc9d0a), LL(0x9454d937,0x187d4f93),LL(0xb4800e1b,0xe682ce9d),LL(0x165e68e8,0xa9129ad8),LL(0xbe7f785b,0x0fe29735),
   LL(0x5b9e02b7,0x5303f40c),LL(0x35ee04e8,0xa37c9692),LL(0x34d6632b,0x5f46cc20),LL(0x96ac545b,0x55ef72b2), LL(0x7b91b062,0xabec5c1f),LL(0xbb33e821,0x0a79e1c7),LL(0x3a9f4117,0xbb04b428),LL(0xfd2a475a,0x0de1f28f),
   LL(0x3a4434b4,0x31019ccf),LL(0x1a7954dc,0xa3458111),LL(0xe34972a7,0xa9dac80d),LL(0x74f6b8dd,0xb043d054), LL(0x11137b1a,0x021c319e),LL(0xed5cc03f,0x00a754ce),LL(0xcbea5ad4,0x0aa2c794),LL(0x70c015b6,0x093e67f4),
   LL(0xc97e3f6b,0x72cdfee9),LL(0xb6da7461,0xc10bcab4),LL(0xb59806b9,0x3b02d2fc),LL(0xa1de6f47,0x85185e89), LL(0x0eb6c4d4,0x39e6931f),LL(0xd4fa5b04,0x4d4440bd),LL(0x34be7eb8,0x5418786e),LL(0x9d7259bc,0x6380e521),
   LL(0xd598d710,0x20ac0351),LL(0xcb3a4da4,0x272c4166),LL(0xca71de1f,0xdb82fe1a),LL(0xd8f54b0f,0x746e79f2), LL(0x4b573e9b,0x6e7fc736),LL(0xfd4b5040,0x75d03f46),LL(0x0b98d87b,0x5c1cc36d),LL(0x1f472da1,0x513ba3f1),
   LL(0xabb177dd,0x79d0af26),LL(0x7891d564,0xf82ab568),LL(0x72232173,0x2b6768a9),LL(0x8c1f6619,0xefbb3bb0), LL(0xa6d18358,0xb29c11db),LL(0xb0916d3a,0x519e2797),LL(0x9188e290,0xd4dc18f0),LL(0x98b0ca7f,0x648e86e3),
   LL(0x983c38b5,0x859d3145),LL(0x637abc8b,0xb14f176c),LL(0xcaff7be6,0x2793fb9d),LL(0x35a66a5a,0xebe5a55f), LL(0x9f87dc59,0x7cec1dcd),LL(0xfbdbf560,0x7c595cd3),LL(0x26eb3257,0x5b543b22),LL(0xc4c935fd,0x69080646),
   LL(0x81e9ede3,0x7f2e4403),LL(0xcaf6df0a,0x243c3894),LL(0x1c073b11,0x7c605bb1),LL(0xba6a4a62,0xcd06a541), LL(0x49d4e2e5,0x29168949),LL(0x4af66880,0x33649d07),LL(0xe9a85035,0xbfc0c885),LL(0xfc410f4b,0xb4e52113),
   LL(0x78a6513b,0xdca3b706),LL(0x9edb1943,0x92ea4a2a),LL(0xdb6e2dd8,0x02642216),LL(0x9fd57894,0x9b45d0b4), LL(0xc69d11ae,0x114e70db),LL(0x4c57595f,0x1477dd19),LL(0xec77c272,0xbc2208b4),LL(0xdb68f59c,0x95c5b4d7),
   LL(0x42e532b7,0xb8c4fc63),LL(0x9ae35290,0x386ba422),LL(0xd201ecbc,0xfb5dda42),LL(0xa0e38fd6,0x2353dc8b), LL(0x68f7e978,0x9a0b85ea),LL(0x2ad6d11f,0x96ec5682),LL(0xe5f6886d,0x5e279d6c),LL(0x3cb1914d,0xd3fe03cd),
   LL(0x7ea67c77,0xfe541fa4),LL(0xe3ea810c,0x952bd2af),LL(0x8d01d374,0x791fef56),LL(0x0f11336e,0xa3a1c621), LL(0xc7ec6d79,0x5ad0d5a9),LL(0x3225c342,0xff7038af),LL(0xbc69601b,0x003c6689),LL(0x45e8747d,0x25059bc7),
   LL(0xf2086fbf,0xfa4965b2),LL(0x86916078,0xf6840ea6),LL(0x70081d6c,0xd7ac7620),LL(0xb5328645,0xe600da31), LL(0x529b8a80,0x01916f63),LL(0x2d7d6f3e,0xe80e4858),LL(0xd664ca7c,0x29eb0fe8),LL(0xe7b43b0c,0xf017637b),
   LL(0x76cb2566,0x9a75c806),LL(0xb24892d9,0x8f76acb1),LL(0x1f08fe45,0x7ae7b9cc),LL(0x6a4907d8,0x19ef7329), LL(0x5f228bf0,0x2db4ab71),LL(0x817032d7,0xf3cdea39),LL(0xdcabe3c0,0x0b1f482e),LL(0xbb86325c,0x3baf76b4),
   LL(0x10089465,0xd49065e0),LL(0x8e77c596,0x3bab5d29),LL(0x193dbd95,0x7636c3a6),LL(0xb246e499,0xdef5d294), LL(0x286b2475,0xb22c58b9),LL(0xcd80862b,0xa0b93939),LL(0xf0992388,0x3002c83a),LL(0xeacbe14c,0x6de01f9b),
   LL(0xadd70482,0x6aac688e),LL(0x7b4a4e8a,0x708de92a),LL(0x758a6eef,0x75b6dd73),LL(0x725b3c43,0xea4bf352), LL(0x87912868,0x10041f2c),LL(0xef09297a,0xb1b1be95),LL(0xa9f3860a,0x19ae23c5),LL(0x515dcf4b,0xc4f0f839),
   LL(0x97f6306a,0x3c7ecca3),LL(0x68a3a4b0,0x744c44ae),LL(0xb3a1d8a2,0x69cd13a0),LL(0x5256b578,0x7cad0a1e), LL(0x33791d9e,0xea653fcd),LL(0x74b2e05f,0x9cc2a05d),LL(0xfd7affa2,0x73b391dc),LL(0xb6b05442,0xddb7091e),
   LL(0x8538a5c6,0xc71e27bf),LL(0x89abff17,0x195c63dd),LL(0x1b71e3da,0xfd315285),LL(0xfa680fa0,0x9cbdfda7), LL(0x849d7eab,0x9db876ca),LL(0x3c273271,0xebe2764b),LL(0xf208dcea,0x663357e3),LL(0x565b1b70,0x8c5bd833),
   LL(0x9837fc0d,0xccc3b4f5),LL(0xa79cf00f,0x9b641ba8),LL(0xdfdf3990,0x7428243d),LL(0x020786b1,0x83a594c4), LL(0x526c4502,0xb712451a),LL(0x6adb3f93,0x9d39438e),LL(0xe9ff0ccd,0xfdb261e3),LL(0xe07af4c3,0x80344e3c),
   LL(0x2fa4f126,0x75900d7c),LL(0x5c99a232,0x08a3b865),LL(0xdb25e0c3,0x2478b6bf),LL(0x71db2edf,0x482cc2c2), LL(0x5f321bb8,0x37df7e64),LL(0x9a8005b4,0x8a93821b),LL(0xcc8c1958,0x3fa2f10c),LL(0x2c269d0a,0x0d332218),
   LL(0xe246b0e6,0x20ab8119),LL(0xd349fd17,0xb39781e4),LL(0xb31aa100,0xd293231e),LL(0xbb032168,0x4b779c97), LL(0xc8470500,0x4b3f19e1),LL(0x0c4c869d,0x45b7efe9),LL(0xa1a6bbcc,0xdb84f38a),LL(0xb2fddbc1,0x3b59cb15),
   LL(0x3fd165e8,0xba5514df),LL(0x061f8811,0x499fd6a9),LL(0xbfef9f00,0x72cd1fe0),LL(0x79ad7e8a,0x120a4bb9), LL(0x5f4a5ac5,0xf2ffd095),LL(0x95a7a2f0,0xcfd174f1),LL(0x9d17baf1,0xd42301ba),LL(0x77f22089,0xd2fa487a),
   LL(0xb1dc77e1,0x9cb09efe),LL(0x21c99682,0xe9566939),LL(0x6c6067bb,0x8c546901),LL(0x61c24456,0xfd378574), LL(0x81796b33,0x2b6a6cbe),LL(0x58e87f8b,0x62d550f6),LL(0x7f1b01b4,0x1b763e1c),LL(0x1b1b5e12,0x4b93cfea),
   LL(0x1d531696,0xb9345238),LL(0x88cdde69,0x57201c00),LL(0x9a86afc7,0xdde92251),LL(0xbd35cea8,0xe3043895), LL(0x8555970d,0x7608c1e1),LL(0x2535935e,0x8267dfa9),LL(0x322ea38b,0xd4c60a57),LL(0x804ef8b5,0xe0bf7977),
   LL(0xc06fece4,0x1a0dab28),LL(0x94e7b49d,0xd405991e),LL(0x706dab28,0xc542b6d2),LL(0xa91618fb,0xcb228da3), LL(0x107d1cea,0x224e4164),LL(0xd0f5d8f1,0xeb9fdab3),LL(0x0d6e41cd,0xc02ba386),LL(0x9b1f7146,0x676a72c5),
   LL(0x4d6cb00b,0xffd6dd98),LL(0xde2e8d7c,0xcef9c5ca),LL(0x641c7936,0xa1bbf5d7),LL(0xee8f772e,0x1b95b230), LL(0xe8ac25b1,0xf765a92e),LL(0x3a18b7c6,0xceb04cfc),LL(0x0acc8966,0x27944cef),LL(0x434c1004,0xcbb3c957),
   LL(0xa43ff93c,0x9c9971a1),LL(0xa1e358a9,0x5bc2db17),LL(0xa8d9bc82,0x45b4862e),LL(0x2201e052,0x70ebfbfb), LL(0x92871591,0xafdf64c7),LL(0xb42d0219,0xea5bcae6),LL(0x2ad8f03c,0xde536c55),LL(0xa76aa33c,0xcd6c3f4d),
   LL(0x0bca6de3,0xbeb5f623),LL(0xb1e706fd,0xdd20dd99),LL(0xac9059d4,0x90b3ff9d),LL(0x7ccccc4e,0x2d7b2902), LL(0xce98840f,0x8a090a59),LL(0x8410680a,0xa5d947e0),LL(0x923379a5,0x49ae346a),LL(0xb28a3156,0x7dbc84f9),
   LL(0x54a1aff2,0xfd40d916),LL(0x3a78fb9b,0xabf318ba),LL(0x3029f95e,0x50152ed8),LL(0xc58ad7fa,0x9fc1dd77), LL(0x13595c17,0x5fa57915),LL(0x8f62b3a9,0xb9504668),LL(0xff3055b0,0x907b5b24),LL(0x9a84f125,0x2e995e35),
   LL(0x7e9bbcfb,0x87dacf69),LL(0xe86d96e3,0x95d0c1d6),LL(0x2d95a75c,0x65726e3c),LL(0xacd27f21,0x2c3c9001), LL(0x6c973f57,0x1deab561),LL(0xa5221643,0x108b7e2c),LL(0xc4ef79d4,0x5fee9859),LL(0x40d4b8c6,0xbd62b88a),
   LL(0x197c75d6,0xb4dd29c4),LL(0xb7076feb,0x266a6df2),LL(0x4bf2df11,0x9512d0ea),LL(0x6b0cc9ec,0x1320c24f), LL(0x01a59596,0x6bb1e0e1),LL(0xeff9aaac,0x8317c5bb),LL(0x385aa6c9,0x65bb405e),LL(0x8f07988f,0x613439c1),
   LL(0x16a66e91,0xd730049f),LL(0xfa1b0e0d,0xe97f2820),LL(0x304c28ea,0x4131e003),LL(0x526bac62,0x820ab732), LL(0x28714423,0xb2ac9ef9),LL(0xadb10cb2,0x54ecfffa),LL(0xf886a4cc,0x8781476e),LL(0xdb2f8d49,0x4b2c87b5),
   LL(0x0a44295d,0xe857cd20),LL(0x58c6b044,0x707d7d21),LL(0xf596757c,0xae8521f9),LL(0x67b2b714,0x87448f03), LL(0x5ebcd58d,0x13a9bc45),LL(0x9122d3c1,0x79bcced9),LL(0x9e076642,0x3c644247),LL(0x2df4767d,0x0cf22778),
   LL(0x71d444b6,0x5e61aee4),LL(0xc5084a1d,0x211236bf),LL(0x4fd3eaf6,0x7e15bc9a),LL(0xab622bf5,0x68df2c34), LL(0x59bf4f36,0x9e674f0f),LL(0xd7f34d73,0xf883669b),LL(0x31497b1d,0xc48ac1b8),LL(0x5106703b,0x323b925d),
   LL(0x74082008,0x22156f42),LL(0xc8482bcb,0xeffc521a),LL(0x12173479,0x5c6831bf),LL(0xc4739490,0xcaa2528f), LL(0x8f1b3c4d,0x84d2102a),LL(0x2d9bec0d,0xcf64dfc1),LL(0x78a546ef,0x433febad),LL(0x7b73cef1,0x1f621ec3),
   LL(0x37338615,0x6aecd627),LL(0x01d8edf6,0x162082ab),LL(0x19e86b66,0x833a8119),LL(0xd299b5db,0x6023a251), LL(0xbbf04b89,0xf5bb0c3a),LL(0xae749a44,0x6735eb69),LL(0x4713de3b,0xd0e058c5),LL(0x2c3d4ccd,0xfdf2593e),
   LL(0xfdd23667,0x1b8f414e),LL(0xfa2015ee,0xdd52aaca),LL(0xbd9625ff,0x3e31b517),LL(0x8db5918c,0x5ec9322d), LL(0xa96f5294,0xbc73ac85),LL(0x61a0666a,0x82aa5bf3),LL(0xbf08ac42,0x49755810),LL(0x891cedfc,0xd21cdfd5),
   LL(0x67f8be10,0x918cb57b),LL(0x56ffa726,0x365d1a7c),LL(0x6532de93,0x2435c504),LL(0x2674cd02,0xc0fc5e10), LL(0x9cbbb142,0x6e51fcf8),LL(0xafc50692,0x1d436e5a),LL(0x3fbcae22,0x766bffff),LL(0xfd55d3b8,0x3148c2fd),
   LL(0x233222fa,0x52c7fdc9),LL(0xe419fb6b,0x89ff1092),LL(0x25254977,0x3cd6db99),LL(0x1cf12ca7,0x2e85a161), LL(0xdc810bc9,0xadd2547c),LL(0x9d257c22,0xea3f458f),LL(0x27d6b19b,0x642c1fbe),LL(0x140481a6,0xed07e6b5),
   LL(0x86d2e0f8,0x6ada1d42),LL(0x0e8a9fd5,0xe5920122),LL(0x708c1b49,0x02c936af),LL(0x2b4bfaff,0x60f30fee), LL(0x858e6a61,0x6637ad06),LL(0x3fd374d0,0xce4c7767),LL(0x7188defb,0x39d54b2d),LL(0xf56a6b66,0xa8c9d250),
   LL(0xb24fe1dc,0x58fc0f5e),LL(0x6b73f24c,0x9eaf9dee),LL(0x33650705,0xa90d588b),LL(0xaf2ec729,0xde5b62c5), LL(0xd3c2b36e,0x5c72cfae),LL(0x034435da,0x868c19d5),LL(0xe17ee145,0x88605f93),LL(0x77a5d5b1,0xaa60c4ee),
   LL(0x3b60c472,0xbcf5bfd2),LL(0xeb1d3049,0xaf4ef13c),LL(0xe13895c9,0x373f44fc),LL(0x0cbc9822,0xf29b382f), LL(0x73efaef6,0x1bfcb853),LL(0xa8c96f40,0xcf56ac9c),LL(0x7a191e24,0xd7adf109),LL(0xbf8a8dc2,0x98035f44),
   LL(0x1e750c84,0xf40a71b9),LL(0x5dc6c469,0xc57f7b0c),LL(0x6fbc19c1,0x49a0e79c),LL(0xa48ebdb8,0x6b0f5889), LL(0xa07c4e9f,0x5d3fd084),LL(0xab27de14,0xc3830111),LL(0x33e08dcc,0x0e4929fe),LL(0x40bb73a3,0xf4a5ad24),
   LL(0x490f97ca,0xde86c2bf),LL(0x67a1ce18,0x288f09c6),LL(0x1844478d,0x364bb886),LL(0xceedb040,0x7840fa42), LL(0x5a631b37,0x1269fdd2),LL(0xa47c8b7d,0x94761f1e),LL(0x481c6266,0xfc0c2e17),LL(0x3daa5fa7,0x85e16ea2),
   LL(0x92491048,0xccd86033),LL(0xf4d402d7,0x0c2f6963),LL(0xdf6a865c,0x6336f7df),LL(0xb5c02a87,0x0a2a463c), LL(0xbf2f12ee,0xb0e29be7),LL(0x66bad988,0xf0a22002),LL(0x9123c1d7,0x27f87e03),LL(0x328a8c98,0x21669c55),
   LL(0x92f14529,0x186b9803),LL(0x63954df3,0xd3d056cc),LL(0x175a46f6,0x2f03fd58),LL(0x11558558,0x63e34ebe), LL(0x5b80cfa5,0xe13fedee),LL(0xd401dbd1,0xe872a120),LL(0xe8a9d667,0x52657616),LL(0xe08d6693,0xbc8da4b6),
   LL(0x1b703e75,0x370fb9bb),LL(0xd4338363,0x6773b186),LL(0xecef7bff,0x18dad378),LL(0x995677da,0xaac787ed), LL(0x0437164b,0x4801ea8b),LL(0x73fe795e,0xf430ad20),LL(0x8ee5eb73,0xb164154d),LL(0x108f7c0e,0x0884ecd8),
   LL(0x5f520698,0x0e6ec096),LL(0x44f7b8d9,0x640631fe),LL(0xa35a68b9,0x92fd34fc),LL(0x4d40cf4e,0x9c5a4b66), LL(0x80b6783d,0x949454bf),LL(0x3a320a10,0x80e701fe),LL(0x1a0a39b2,0x8d1a564a),LL(0x320587db,0x1436d53d),
   LL(0x6556c362,0xf5096e6d),LL(0xe2455d7e,0xbc23a3c0),LL(0x807230f9,0x3a7aee54),LL(0x22ae82fd,0x9ba1cfa6), LL(0x99c5d706,0x833a057a),LL(0x842315c9,0x8be85f4b),LL(0x66a72f12,0xd083179a),LL(0xcdcc73cd,0x2fc77d5d),
   LL(0x5616ee30,0x22b88a80),LL(0xe7ab1083,0xfb09548f),LL(0x511270cd,0x8ad6ab0d),LL(0x6924d9ab,0x61f6c57a), LL(0x90aecb08,0xa0f7bf72),LL(0x0df784a4,0x849f87c9),LL(0xcfaf1d03,0x27c79c15),LL(0xc463face,0xbbf9f675),
   LL(0x765ba543,0x91502c65),LL(0x42ea60dd,0x18ce3cac),LL(0x6e43ecb3,0xe5cee6ac),LL(0x68f2aeeb,0x63e4e910), LL(0xc85932ee,0x26234fa3),LL(0x4c90c44d,0x96883e8b),LL(0xa18a50f6,0x29b9e738),LL(0x3f0420df,0xbfc62b2a),
   LL(0x6d3e1fa9,0xd22a7d90),LL(0xfe05b8a3,0x17115618),LL(0xbb2b9c01,0x2a0c9926),LL(0xe07e76a2,0xc739fcc6), LL(0x165e439a,0x540e9157),LL(0x6a9063d8,0x06353a62),LL(0x61e927a3,0x84d95594),LL(0xe2e0be7f,0x013b9b26),
   LL(0x973497f1,0x4feaec3b),LL(0x093ebc2d,0x15c0f94e),LL(0x33af0583,0x6af5f227),LL(0xc61f3340,0x0c2af206), LL(0x4457397c,0xd25dbdf1),LL(0xcabcbae0,0x2e8ed017),LL(0xc2815306,0xe3010938),LL(0xe8c6cd68,0xbaa99337),
   LL(0x3b0ec7de,0x08513182),LL(0x58df05df,0x1e1b822b),LL(0xa5c3b683,0x5c14842f),LL(0x3eba34ce,0x98fe977e), LL(0x0d5e8873,0xfd2316c2),LL(0xbd0d427d,0xe48d839a),LL(0x623fc961,0x495b2218),LL(0xb46fba5e,0x24ee56e7),
   LL(0x91e4de58,0x9184a55b),LL(0xdfdea288,0xa7488ca5),LL(0xa8dcc943,0xa723862e),LL(0x849dc0fc,0x92d762b2), LL(0x091ff4a9,0x3c444a12),LL(0x0cada274,0x581113fa),LL(0x30d8eae2,0xb9de0a45),LL(0xdf6b41ea,0x5e0fcd85),
   LL(0xc094dbb5,0x6233ea68),LL(0xd968d410,0xb77d062e),LL(0x58b3002d,0x3e719bbc),LL(0x3dc49d58,0x68e7dd3d), LL(0x013a5e58,0x8d825740),LL(0x3c9e3c1b,0x21311747),LL(0x7c99b6ab,0x0cb0a2a7),LL(0xc2f888f2,0x5c48a3b3),
},
/* digit=8 base_pwr=2^56 */
{
   LL(0x991724f3,0xc7913e91),LL(0x39cbd686,0x5eda799c),LL(0x63d4fc1e,0xddb595c7),LL(0xac4fed54,0x6b63b80b), LL(0x7e5fb516,0x6ea0fc69),LL(0xd0f1c964,0x737708ba),LL(0x11a92ca5,0x9628745f),LL(0x9a86967a,0x61f37958),
   LL(0xaa665072,0x9af39b2c),LL(0xefd324ef,0x78322fa4),LL(0xc327bd31,0x3d153394),LL(0x3129dab0,0x81d5f271), LL(0xf48027f5,0xc72e0c42),LL(0x8536e717,0xaa40cdbc),LL(0x2d369d0f,0xf45a657a),LL(0xea7f74e6,0xb03bbfc4),
   LL(0x0d738ded,0x46a8c418),LL(0xe0de5729,0x6f1a5bb0),LL(0x8ba81675,0xf10230b9),LL(0x112b33d4,0x32c6f30c), LL(0xd8fffb62,0x7559129d),LL(0xb459bf05,0x6a281b47),LL(0xfa3b6776,0x77c1bd3a),LL(0x7829973a,0x0709b380),
   LL(0xa3326505,0x8c26b232),LL(0xee1d41bf,0x38d69272),LL(0xffe32afa,0x0459453e),LL(0x7cb3ea87,0xce8143ad), LL(0x7e6ab666,0x932ec1fa),LL(0x22286264,0x6cd2d230),LL(0x6736f8ed,0x459a46fe),LL(0x9eca85bb,0x50bf0d00),
   LL(0x877a21ec,0x0b825852),LL(0x0f537a94,0x300414a7),LL(0x21a9a6a2,0x3f1cba40),LL(0x76943c00,0x50824eee), LL(0xf83cba5d,0xa0dbfcec),LL(0x93b4f3c0,0xf9538148),LL(0x48f24dd7,0x61744162),LL(0xe4fb09dd,0x5322d64d),
   LL(0x3d9325f3,0x57447384),LL(0xf371cb84,0xa9bef2d0),LL(0xa61e36c5,0x77d2188b),LL(0xc602df72,0xbbd6a7d7), LL(0x8f61bc0b,0xba3aa902),LL(0x6ed0b6a1,0xf49085ed),LL(0xae6e8298,0x8bc625d6),LL(0xa2e9c01d,0x832b0b1d),
   LL(0xf1f0ced1,0xa337c447),LL(0x9492dd2b,0x800cc793),LL(0xbea08efa,0x4b93151d),LL(0xde0a741e,0x820cf3f8), LL(0x1c0f7d13,0xff1982dc),LL(0x84dde6ca,0xef921960),LL(0x45f96ee3,0x1ad7d972),LL(0x29dea0c7,0x319c8dbe),
   LL(0x7b82b99b,0xd3ea3871),LL(0x470eb624,0x75922d4d),LL(0x3b95d466,0x8f66ec54),LL(0xbee1e346,0x66e673cc), LL(0xb5f2b89a,0x6afe67c4),LL(0x290e5cd3,0x3de9c1e6),LL(0x310a2ada,0x8c278bb6),LL(0x0bdb323b,0x420fa384),
   LL(0x0eb919b0,0x0ae1d63b),LL(0xa74b9620,0xd74ee51d),LL(0xa674290c,0x395458d0),LL(0x4620a510,0x324c930f), LL(0xfbac27d4,0x2d1f4d19),LL(0x9bedeeac,0x4086e8ca),LL(0x9b679ab8,0x0cdd211b),LL(0x7090fec4,0x5970167d),
   LL(0xfaf1fc63,0x3420f2c9),LL(0x328c8bb4,0x616d333a),LL(0x57f1fe4a,0x7d65364c),LL(0x55e5c73a,0x9343e877), LL(0xe970e78c,0x5795176b),LL(0x60533627,0xa36ccebf),LL(0x09cdfc1b,0xfc7c7380),LL(0xb3fec326,0xb39a2afe),
   LL(0x6224408a,0xb7ff1ba1),LL(0x247cfc5e,0xcc856e92),LL(0xc18bc493,0x01f102e7),LL(0x2091c727,0x4613ab74), LL(0xc420bf2b,0xaa25e89c),LL(0x90337ec2,0x00a53176),LL(0x7d025fc7,0xd2be9f43),LL(0x6e6fe3dc,0x3316fb85),
   LL(0x9ac50814,0x27520af5),LL(0x9a8e4223,0xfdf95e78),LL(0x56bec5a0,0xb7e7df2a),LL(0xdf159e5d,0xf7022f7d), LL(0xcac1fe8f,0x93eeeab1),LL(0x37451168,0x8040188c),LL(0xd967dce6,0x7ee8aa8a),LL(0x3abc9299,0xfa0e79e7),
   LL(0x2064cfd1,0x67332cfc),LL(0xb0651934,0x339c31de),LL(0x2a3bcbea,0x719b28d5),LL(0x9d6ae5c6,0xee74c82b), LL(0xbaf28ee6,0x0927d05e),LL(0x9d719028,0x82cecf2c),LL(0xddb30289,0x0b0d353e),LL(0xfddb2e29,0xfe4bb977),
   LL(0x640bfd9e,0xbb5bb990),LL(0x82f62108,0xd226e277),LL(0x02ffdd56,0x4bf00985),LL(0x2ca1b1b5,0x7756758a), LL(0x5285fe91,0xc32b62a3),LL(0x8c9cd140,0xedbc546a),LL(0xaf5cb008,0x1e47a013),LL(0x073ce8f2,0xbca7e720),
   LL(0x17a91cae,0xe10b2ab8),LL(0x08e27f63,0xb89aab65),LL(0xdba3ddf9,0x7b3074a7),LL(0x330c2972,0x1c20ce09), LL(0x5fcf7e33,0x6b9917b4),LL(0x945ceb42,0xe6793743),LL(0x5c633d19,0x18fc2215),LL(0xc7485474,0xad1adb3c),
   LL(0x6424c49b,0x646f9679),LL(0x67c241c9,0xf888dfe8),LL(0x24f68b49,0xe12d4b93),LL(0xa571df20,0x9a6b62d8), LL(0x179483cb,0x81b4b26d),LL(0x9511fae2,0x666f9632),LL(0xd53aa51f,0xd281b3e4),LL(0x7f3dbd16,0x7f96a765),
   LL(0x074a30ce,0xa7f8b5bf),LL(0x005a32e6,0xd7f52107),LL(0x50237ed4,0x6f9e0907),LL(0x8096fa2b,0x2f21da47), LL(0xeec863a0,0xf3e19cb4),LL(0x9527620a,0xd18f77fd),LL(0x407c1cf8,0x9505c81c),LL(0x1b6ec284,0x9998db4e),
   LL(0xc247d44d,0x7e3389e5),LL(0x3f4f3d80,0x12507141),LL(0x4a78a6c7,0xd4ba0110),LL(0x767720be,0x312874a0), LL(0x75944370,0xded059a6),LL(0x3b2c0bdd,0xd6123d90),LL(0x51c108e3,0xa56b717b),LL(0x070623e9,0x9bb7940e),
   LL(0x84ac066c,0x794e2d59),LL(0xe68c69a0,0xf5954a92),LL(0x4fd99dcc,0x28c52458),LL(0xb1012517,0x60e639fc), LL(0x7de79248,0xc2e60125),LL(0xf12fc6d7,0xe9ef6404),LL(0x2a3b5d32,0x4c4f2808),LL(0xc768eb8a,0x865ad32e),
   LL(0x13fb70b6,0xac02331b),LL(0x95599b27,0x037b44c1),LL(0x60bd082c,0x1a860fc4),LL(0xc980cd01,0xa2e25745), LL(0x1da0263e,0xee3387a8),LL(0x2d10f3d6,0x931bfb95),LL(0xa1f24a32,0x5b687270),LL(0xca494b86,0xf140e65d),
   LL(0xb2f1ac7a,0x4f4ddf91),LL(0x760fee27,0xf99eaabb),LL(0x49c228e5,0x57f4008a),LL(0x1cf713bb,0x090be440), LL(0x5004f022,0xac91fbe4),LL(0x569e1af6,0xd838c2c2),LL(0x0f1daaa5,0xd6c7d20b),LL(0x1bbb02c0,0xaa063ac1),
   LL(0x59558a78,0x0938a422),LL(0x8435da2f,0x5343c669),LL(0x034410dc,0x96f67b18),LL(0x84510804,0x7cc1e424), LL(0x16dfbb7d,0x86a1543f),LL(0x5b5bd592,0x921fa942),LL(0xb33dd03c,0x9dcccb6e),LL(0xb843f51e,0x8581ddd9),
   LL(0x81d73c9e,0x54935fcb),LL(0x0a5e97ab,0x6d07e979),LL(0xcf3a6bab,0x4dc7b30a),LL(0x170bee11,0x147ab1f3), LL(0x9fafdee4,0x0aaf8e3d),LL(0x538a8b95,0xfab3dbcb),LL(0x6ef13871,0x405df4b3),LL(0x088d5a49,0xf1f4e9cb),
   LL(0x66b33f1d,0x9bcd24d3),LL(0x5ce445c0,0x3b97b820),LL(0xba93ff61,0xe2926549),LL(0x4dafe616,0xd9c341ce), LL(0x16efb6f3,0xfb30a76e),LL(0x605b953c,0xdf24b8ca),LL(0xc2fffb9f,0x8bd52afe),LL(0xe19d0b96,0xbbac5ff7),
   LL(0x459afccd,0x43c01b87),LL(0xb7432652,0x6bd45143),LL(0x55b5d78e,0x84734530),LL(0x1554ba7d,0x81088fdb), LL(0x1e269375,0xada0a52c),LL(0x2dc5ec10,0xf9f037c4),LL(0x94bfbc11,0xc0660607),LL(0xc9c40d2f,0xc0a630bb),
   LL(0xab64c31e,0x5efc797e),LL(0x74507144,0xffdb1dab),LL(0x1ca6790c,0xf6124287),LL(0xe69bf1bf,0xe9609d81), LL(0x00d24fc9,0xdb898595),LL(0xe51fb417,0x9c750333),LL(0xfef7bbde,0x51830a91),LL(0x945f585c,0x0ce67dc8),
   LL(0x4763eb50,0x9a730ed4),LL(0xc1ab0d66,0x24a0e221),LL(0x648748f3,0x643b6393),LL(0x6d3c6291,0x1982daa1), LL(0x8bbc5549,0x6f00a9f7),LL(0x7f36384e,0x7a1783e1),LL(0xde977f50,0xe8346323),LL(0xb245502a,0x91ab688d),
   LL(0x6d0bdd66,0x331ab6b5),LL(0x64b71229,0x0a6ef32e),LL(0xfe7c352f,0x1028150e),LL(0xce7b39d3,0x27e04350), LL(0xc1070c82,0x2a3c8acd),LL(0x80c9feef,0xfb2034d3),LL(0x709f3729,0x2d729621),LL(0x62cb4549,0x8df290bf),
   LL(0xfc2e4326,0x02f99f33),LL(0x5eddf032,0x3b30076d),LL(0x0c652fb5,0xbb21f8cf),LL(0xed91cf7b,0x314fb49e), LL(0x2f700750,0xa013eca5),LL(0x712a4575,0x2b9e3c23),LL(0xaf30fbb0,0xe5355557),LL(0x7c77e771,0x1ada3516),
   LL(0x7b135670,0x45f6ecb2),LL(0x7cfc202e,0xe85d19df),LL(0x58d1be9f,0x0f1b50c7),LL(0xead2e344,0x5ebf2c0a), LL(0xabc199c9,0x1531fe4e),LL(0x56bab0ae,0xc7032592),LL(0x6c1fec54,0x16ab2e48),LL(0x04280188,0x0f87fda8),
   LL(0x609e4a74,0xdc9f46fc),LL(0xba667f91,0x2a44a143),LL(0xb4d83436,0xbc3d8b95),LL(0xc7bd2958,0xa01e4bd0), LL(0x73483c90,0x7b182932),LL(0xa7c7b598,0xa79c6aa1),LL(0xeaaac07e,0xbf3983c6),LL(0x96e0d4e6,0x8f18181e),
   LL(0x051af62b,0x8553d37c),LL(0x0bf94496,0xe9a998eb),LL(0xb0d59aa1,0xe0844f9f),LL(0xe6afb813,0x983fd558), LL(0x65d69804,0x9670c0ca),LL(0x6ea5ff2d,0x732b22de),LL(0x5fd8623b,0xd7640ba9),LL(0xa6351782,0x9f619163),
   LL(0xacee5043,0x0bfc27ee),LL(0x2eb10f02,0xae419e73),LL(0x8943fb05,0x19c028d1),LL(0xff13aa2a,0x71f01cf7), LL(0x8887a132,0x7790737e),LL(0x66318410,0x67513309),LL(0x7ddb795e,0x9819e8a3),LL(0xdad100b2,0xfecb8ef5),
   LL(0x3021926a,0x59f74a22),LL(0x6f9b4c1c,0xb7c28a49),LL(0x912ad0ab,0xed1a733f),LL(0x01a5659c,0x42a910af), LL(0x7bd68cab,0x3842c6e0),LL(0x76d70ac8,0x2b57fa38),LL(0x3c53aaeb,0x8a6707a8),LL(0x65b4db18,0x62c1c510),
   LL(0xb2d09dc7,0x8de2c1fb),LL(0x266bd23b,0xc3dfed12),LL(0xd5b27db6,0x927d039b),LL(0x103243da,0x2fb2f0f1), LL(0x80be7399,0xf855a07b),LL(0x1f9f27a8,0xed9327ce),LL(0x729bdef7,0xa0bd99c7),LL(0x28250d88,0x2b67125e),
   LL(0x8670ced7,0x784b26e8),LL(0xc31bd3b4,0xe3dfe41f),LL(0xbcc85cbc,0x9e353a06),LL(0x60178a9d,0x302e2909), LL(0xa6eac16e,0x860abf11),LL(0xaa2b3aac,0x76447000),LL(0x850afdab,0x46ff9d19),LL(0xfdb2d4c1,0x35bdd6a5),
   LL(0x7e5c9ce9,0xe82594b0),LL(0x20af346e,0x0f379e53),LL(0xbc65ad4a,0x608b31e3),LL(0x267c4826,0x710c6b12), LL(0x71954cf1,0x51c966f9),LL(0x0d0aa215,0xb1cec793),LL(0x86bd23a8,0x1f155989),LL(0xf9452e86,0xae2ff99c),
   LL(0x340ceaa2,0xd8dd953c),LL(0x2e2e9333,0x26355275),LL(0x8586f06d,0x15d4e5f9),LL(0xf7cab546,0xd6bf94a8), LL(0xb76a9af0,0x33c59a0a),LL(0xba095af7,0x52740ab3),LL(0x24389ca0,0xc444de8a),LL(0x706da0cb,0xcc6f9863),
   LL(0x6b2515cf,0xb5a741a7),LL(0x9585c749,0x71c41601),LL(0xe683de97,0x78350d4f),LL(0x63d0b5f5,0x31d61524), LL(0xfbce090b,0x7a0cc5e1),LL(0xfbcb2a5b,0xaac927ed),LL(0x20d84c35,0xe920de49),LL(0x22b4de26,0x8c06a0b6),
   LL(0xafe7ddf3,0xd34dd58b),LL(0xc1e6e55b,0x55851fed),LL(0x960696e7,0xd1395616),LL(0x5f22705f,0x940304b2), LL(0xb0a2a860,0x6f43f861),LL(0x0e7cc981,0xcf121282),LL(0x0ab64a96,0x12186212),LL(0xb789383c,0x09215b9a),
   LL(0x37387c09,0x311eb305),LL(0xf03ee760,0xc5832fce),LL(0x32f7ea19,0x30358f58),LL(0x91d53551,0xe01d3c34), LL(0xda48ea80,0x1ca5ee41),LL(0xcf4fa4c1,0x34e71e8e),LL(0x7af1e1c7,0x312abd25),LL(0x2153f4a5,0xe3afcdeb),
   LL(0x00235e9a,0x9d5c84d7),LL(0x8c4c836f,0x0308d3f4),LL(0x89332de5,0xc0a66b04),LL(0x89e566ef,0x610dd399), LL(0xd1ac1635,0xf8eea460),LL(0x20a2c0df,0x84cbb3fb),LL(0xe74a48c5,0x40afb488),LL(0xd326b150,0x29738198),
   LL(0xa6d74081,0x2a17747f),LL(0x55a26214,0x60ea4c05),LL(0x1f88c5fe,0x53514bb4),LL(0x7e83426c,0xedd64567), LL(0x96460b25,0xd5d6cbec),LL(0x68dc115e,0xa12fd0ce),LL(0x697840ea,0xc5bc3ed2),LL(0xa6331e31,0x969876a8),
   LL(0x472ff580,0x60c36217),LL(0x4ad41393,0xf4229705),LL(0xa03b8b92,0x4bd99ef0),LL(0xc144f4f6,0x501c7317), LL(0x18464945,0x159009b3),LL(0x74c5c6be,0x6d5e594c),LL(0x321a3660,0x2d587011),LL(0x3898d022,0xd1e184b1),
   LL(0x4c6a7e04,0x5ba04752),LL(0x45550b65,0x47fa1e2b),LL(0x48c0a9a5,0x9419daf0),LL(0x7c243236,0x66362953), LL(0x5cb12a88,0xcd0744b1),LL(0x2b646188,0x561b6f9a),LL(0x66c2c0c0,0x599415a5),LL(0x0f83f09a,0xbe3f0859),
   LL(0xb92041b8,0x9141c5be),LL(0x26477d0d,0x01ae38c7),LL(0xd12c7a94,0xca8b71f3),LL(0x765c70db,0xfab5b31f), LL(0x487443e9,0x76ae7492),LL(0x990d1349,0x8595a310),LL(0x7d460a37,0xf8dbeda8),LL(0x1e45a38f,0x7f7ad082),
   LL(0x1059705a,0xed1d4db6),LL(0xe6b9c697,0xa3dd492a),LL(0x6eb38bd5,0x4b92ee3a),LL(0x67cc0bb7,0xbab2609d), LL(0x6e70ee82,0x7fc4fe89),LL(0x13e6b7e3,0xeff2c56e),LL(0x34d26fca,0x9b18959e),LL(0x889d6b45,0x2517ab66),
   LL(0xbdefdd4f,0xf167b4e0),LL(0xf366e401,0x69958465),LL(0xa73bbec0,0x5aa368ab),LL(0x7b240c21,0x12148709), LL(0x18969006,0x378c3233),LL(0xe1fe53d1,0xcb4d73ce),LL(0x130c4361,0x5f50a80e),LL(0x7ef5212b,0xd67f5951),
   LL(0x9e70c72e,0xf145e21e),LL(0x5566d2fb,0xb2e52e29),LL(0x032397f5,0x44eaba4a),LL(0x7e31a7de,0x5e56937b), LL(0x456c61e1,0x68dcf517),LL(0xa8b0a388,0xbc2e954a),LL(0x60a8b755,0xe3552fa7),LL(0x73ad0cde,0x03442dae),
   LL(0xceb26210,0x37ffe747),LL(0x787baef9,0x983545e8),LL(0x86a3de31,0x8b8c8535),LL(0xfacd46db,0xc621dbcb), LL(0x59266fbb,0x82e442e9),LL(0x339d471c,0xa3514c37),LL(0x62cdad96,0x3a11b771),LL(0xecf9bdf0,0xf0cb3b3c),
   LL(0x478e2135,0x3fcbdbce),LL(0xbda35342,0x7547b5cf),LL(0x8a677af6,0xa97e81f1),LL(0x28817987,0xc8c2bf83), LL(0x45580985,0xdf07eaaf),LL(0xc93b45cb,0xc68d1f05),LL(0xc77b4cac,0x106aa2fe),LL(0x04a7ae86,0x4c1d8afc),
   LL(0x9eb45ab2,0xdb41c3fd),LL(0xd4b22e74,0x5b234b5b),LL(0xf215958a,0xda253dec),LL(0xa04edfa0,0x67e0606e), LL(0xef751b11,0xabbbf070),LL(0xf6f06dce,0xf352f175),LL(0x6839f6b4,0xdfc4b6af),LL(0x9959848e,0x53ddf9a8),
   LL(0xc21520b0,0xda49c379),LL(0xdbd5d1b6,0x90864ff0),LL(0x5f49c7f7,0x2f055d23),LL(0xa796b2d8,0xe51e4e6a), LL(0x5c9dc340,0xc361a67f),LL(0xbca7c620,0x5ad53c37),LL(0x32c756d0,0xda1d6588),LL(0x8bb67e13,0xad60d911),
   LL(0x0eeec8c6,0xd6c47bdf),LL(0x078a1821,0x4a27fec1),LL(0xc3099524,0x081f7415),LL(0x82cd8060,0x8effdf0b), LL(0x65842df8,0xdb70ec1c),LL(0xd319a901,0x8821b358),LL(0xde42b529,0x72ee56ee),LL(0x236e4286,0x5bb39592),
   LL(0xfd6f7140,0xd1183316),LL(0xbd8e81f7,0xf9fadb5b),LL(0x5a02d962,0x701d5e0c),LL(0x1b601324,0xfdee4dbf), LL(0x35d7620e,0xbed17407),LL(0xf48c0012,0x04e3c2c3),LL(0x3455449a,0x9ee29da7),LL(0x91a836c4,0x562cdef4),
   LL(0x47701097,0x8f682a5f),LL(0xff88d0c2,0x617125d8),LL(0x57bb86dd,0x948fda24),LL(0x289f7286,0x348abb8f), LL(0x99d94bbd,0xeb10eab5),LL(0x4684d160,0xd51ba28e),LL(0x30c8f41a,0xabe0e51c),LL(0x13254f4a,0x66588b45),
   LL(0xfad097a5,0x147ebf01),LL(0x610e815d,0x49883ea8),LL(0x8a11de56,0xe44d60ba),LL(0x827a7a6d,0xa970de6e), LL(0x5e17fc19,0x2be41424),LL(0x01214057,0xd833c657),LL(0x363e723f,0x1375813b),LL(0xe6a52e9b,0x6820bb88),
   LL(0xd875d56a,0x7e7f6970),LL(0x51fbf6bf,0xd6a0a9ac),LL(0xa3083c12,0x54ba8790),LL(0x6ae7eb64,0xebaeb23d), LL(0xb99a907a,0xa8685c3a),LL(0x026bf40b,0xf1e74550),LL(0xc802cd9e,0x7b73a027),LL(0x4fef4635,0x9a8a927c),
   LL(0x08191224,0xe1b6f60c),LL(0xde4ec091,0xc4126ebb),LL(0x4ae38d84,0xe1dff4dc),LL(0x4f2ef985,0xde3f57db), LL(0xd446a1dd,0x34964337),LL(0x859e77f6,0x7bf217a0),LL(0x8e1d13f5,0x8ff10527),LL(0x74eeae27,0xa304ef03),
   LL(0xd19dfa5a,0xfc6f5e47),LL(0x7fad982b,0xdb007de3),LL(0x613715f5,0x28205ad1),LL(0x7889529e,0x251e6729), LL(0x1ae98e78,0x72705184),LL(0x271cac32,0xf818537d),LL(0xb7f410f5,0xc8a15b7e),LL(0x81f62393,0xc474356f),
   LL(0xc242316b,0x92dbdc5a),LL(0xdbf4aff5,0xabe060ac),LL(0x909a8ec6,0x6e8c38fe),LL(0x6116cb94,0x43e514e5), LL(0x07d784f9,0x2078fa38),LL(0xf4b5b357,0x1161a880),LL(0x13adea3d,0x5283ce79),LL(0xcc6a910b,0x0756c3e6),
   LL(0xaaa79697,0x60bcfe01),LL(0x56391db1,0x04a73b29),LL(0x189b45a0,0xdd8dad47),LL(0x48d5b8d9,0xbfac0dd0), LL(0x7d3d2ec2,0x34ab3af5),LL(0x207bd3af,0x6fa2fc2d),LL(0x66550ded,0x9ff40092),LL(0x1fd5b913,0x719b3e87),
   LL(0x6d17fbc7,0xa573a496),LL(0x73d2b24e,0x0cd1a70a),LL(0xb2676937,0x34e2c5ca),LL(0xbf669f21,0xe7050b06), LL(0x1ede9046,0xfbe948b6),LL(0x97662659,0xa0530051),LL(0xf10124c5,0x58cbd4ed),LL(0xdd6c06c8,0xde2646e4),
   LL(0x8cad38c0,0x332f8108),LL(0x6bd68ae2,0x471b7e90),LL(0x0d8e27a3,0x56ac3fb2),LL(0x136b4b0d,0xb54660db), LL(0xa6fd8de4,0x123a1e11),LL(0xa37799ef,0x44dbffea),LL(0xce6ac17c,0x4540b977),LL(0xaf60acef,0x495173a8),
},
/* digit=9 base_pwr=2^63 */
{
   LL(0x391c2a82,0x9ebb284d),LL(0x158308e8,0xbcdd4863),LL(0x83f1edca,0x006f16ec),LL(0x695dc6c8,0xa13e2c37), LL(0x4a057a87,0x2ab756f0),LL(0xa6b48f98,0xa8765500),LL(0x68651c44,0x4252face),LL(0xe1765e02,0xa52b540b),
   LL(0x16a0d2bb,0x4f922fc5),LL(0x1a623499,0x0d5cc16c),LL(0x57c62c8b,0x9241cf3a),LL(0xfd1b667f,0x2f5e6961), LL(0xf5a01797,0x5c15c70b),LL(0x60956192,0x3d20b44d),LL(0x071fdb52,0x04911b37),LL(0x8d6f0f7b,0xf648f916),
   LL(0xe60b7cf7,0x6dc1acaf),LL(0x84a9d869,0x25860a50),LL(0xe7ba8ac4,0x56fc6f09),LL(0x6148d29e,0x828c5bd0), LL(0xdc55ae5f,0xac6b435e),LL(0xc0117411,0xa527f56c),LL(0xfd24342c,0x94d5045e),LL(0x70b67c0d,0x2c4c0a35),
   LL(0xfac61d9a,0x027cc8b8),LL(0xe3c6fe8a,0x7d25e062),LL(0xe5bff503,0xe08805bf),LL(0x6ff632f7,0x13271e6c), LL(0x232f76a5,0x55dca6c0),LL(0x701ef426,0x8957c32d),LL(0xa10a5178,0xee728bcb),LL(0xb62c5173,0x5ea60411),
   LL(0xd0b8892b,0xfc4e964e),LL(0x9301bb74,0x9ea17683),LL(0xfcc48626,0x6265c5ae),LL(0xbb3e9102,0xe60cf82e), LL(0xd4df5531,0x57adf797),LL(0x8deeefe2,0x235b59a1),LL(0x3f306eb1,0x60adcf58),LL(0x3d09492d,0x105c2753),
   LL(0xb5def996,0x4090914b),LL(0x233dd1e7,0x1cb69c83),LL(0x9b3d5e76,0xc1e9c1d3),LL(0xfccf6012,0x1f3338ed), LL(0x2f5378a8,0xb1e95d0d),LL(0x2f00cd21,0xacf4c2c7),LL(0xeb5fe290,0x6e984240),LL(0x248088ae,0xd66c038d),
   LL(0xf94d70cf,0x804d264a),LL(0x7314bf7e,0xbdb802ef),LL(0x4333ed02,0x8fb54de2),LL(0x285635d9,0x740461e0), LL(0x365e9383,0x4113b2c8),LL(0x3fdef652,0xea762c83),LL(0x47b956c1,0x4eec6e2e),LL(0x65620fa4,0xa3d814be),
   LL(0xb4d8bc50,0x9ad5462b),LL(0xa9195770,0x181c0b16),LL(0x78412a68,0xebd4fe1c),LL(0xc0dff48c,0xae0341bc), LL(0x7003e866,0xb6bc45cf),LL(0x8a24a41b,0xf11a6dea),LL(0xd04c24c2,0x5407151a),LL(0xda5b7b68,0x62c9d27d),
   LL(0x88cceff6,0x2e964235),LL(0x8b07ed69,0x8594c54f),LL(0xc84d0d0d,0x1578e73c),LL(0xff532868,0x7b4e1055), LL(0xb5ec995a,0xa348c0d5),LL(0x14289a54,0xbf4b9d55),LL(0x58fbd777,0x9ba155a6),LL(0x1a84491d,0x186ed7a8),
   LL(0x614c0900,0xd4992b30),LL(0xbd00c24b,0xda98d121),LL(0x7ec4bfa1,0x7f534dc8),LL(0x37dc34bc,0x4a5ff674), LL(0x1d7ea1d7,0x68c196b8),LL(0x80a6d208,0x38cf2893),LL(0xe3cbbd6e,0xfd56cd09),LL(0x4205a5b6,0xec72e27e),
   LL(0xa44f77f7,0x15ea68f5),LL(0xb43c52bc,0x7aa5f9fd),LL(0x94f0e609,0x86ff676f),LL(0x2e2d432b,0xa4cde963), LL(0xeee470af,0x8cafa0c0),LL(0x8a3f5ec8,0x84137d0e),LL(0xfaa31231,0xebb40411),LL(0x6f7f7ccf,0xa239c13f),
   LL(0xa8afd30b,0x32865719),LL(0x8a826dce,0x86798328),LL(0xc4a8fbe0,0xdf04e891),LL(0xebf56ad3,0xbb6b6e1b), LL(0x471f1ff0,0x0a695b11),LL(0xbe15baf0,0xd76c3389),LL(0xbe96c43e,0x018edb95),LL(0x90794158,0xf2beaaf4),
   LL(0xc3076a27,0x152db09e),LL(0xe416545d,0x5e82908e),LL(0x356d6f2e,0xa2c41272),LL(0x31fd74e1,0xdc9c9642), LL(0x519bf615,0x66ceb88d),LL(0x05a2274e,0xe29ecd76),LL(0xbf5e2fa0,0x3a0473c4),LL(0x64284e67,0x6b6eb671),
   LL(0xb88756dd,0xe8b97932),LL(0xf17e3e61,0xed4e8652),LL(0x3ee1c4a4,0xc2dd1499),LL(0x597f8c0e,0xc0aaee17), LL(0x6c168af3,0x15c4edb9),LL(0xb39ae875,0x6563c7bf),LL(0x20adb436,0xadfadb6f),LL(0x9a042ac0,0xad55e8c9),
   LL(0xb76da1f5,0x975a1ed8),LL(0xa58acb94,0x10dfa466),LL(0xac060282,0x8dd7f7e3),LL(0x572a051e,0x6813e66a), LL(0x350cb901,0xb4ccae1e),LL(0x50cb7822,0xb653d656),LL(0xdfab3b87,0x42484710),LL(0x9b670fd0,0xcd7ee537),
   LL(0x523b8bf6,0x0a50b12e),LL(0x8f910c1b,0x8009eb5b),LL(0x4a167588,0xf535af82),LL(0xfb2a2abd,0x0f835f9c), LL(0x2afceb62,0xf59b2931),LL(0x169d383f,0xc797df2a),LL(0x66ac02b0,0xeb3f5fb0),LL(0xdaa2d0ca,0x029d4c6f),
   LL(0xafab4bc5,0xd4059bc1),LL(0x56783247,0x833f5c6f),LL(0x8d2d3605,0xb5346630),LL(0xd34d8433,0x83387891), LL(0xadd9419a,0xd973b30f),LL(0xafe3fce8,0xbcca1099),LL(0x0809aac6,0x08178315),LL(0x540f0f11,0x01b7f21a),
   LL(0x909523c8,0x65c29219),LL(0xa3a1c741,0xa62f648f),LL(0x60c9e55a,0x88598d4f),LL(0x0e4f347a,0xbce9141b), LL(0x35f9b988,0x9af97d84),LL(0x320475b6,0x0210da62),LL(0x9191476c,0x3c076e22),LL(0x44fc7834,0x7520dbd9),
   LL(0xc1ab1bbd,0x6a6b2cfe),LL(0xdc650938,0xef8a65be),LL(0x805d7bc4,0x72855540),LL(0xed11fdfd,0xda389396), LL(0x74660876,0xa9d5bd36),LL(0xb45dff35,0x11d67c54),LL(0xa4f5da94,0x6af7d148),LL(0xc0bbeb31,0xbb8d4c3f),
   LL(0xe0a1b12a,0x87a7ebd1),LL(0x770ba95f,0x1e4ef88d),LL(0xdc2ae9cb,0x8c33345c),LL(0x01cc8403,0xcecf1276), LL(0x1b39b80f,0x687c012e),LL(0x35c33ba4,0xfd90d0ad),LL(0x5c9661c2,0xa3ef5a67),LL(0xe017429e,0x368fc88e),
   LL(0x196a2fa2,0xd30c6761),LL(0xbd5b312e,0x931b9817),LL(0x72f54a31,0xba01000c),LL(0x66eaa541,0xa203d2c8), LL(0x98939db3,0xf2abdee0),LL(0x3e606c02,0xe37d6c2c),LL(0x521ff643,0xf2921574),LL(0xd7e2fca3,0x2781b3c4),
   LL(0x7850ec06,0x664300b0),LL(0x7d3a10cf,0xac5a38b9),LL(0xe34ab39d,0x9233188d),LL(0x5072cbb9,0xe77057e4), LL(0xb59e78df,0xbcf0c042),LL(0x1d97de52,0x4cfc91e8),LL(0x3ee0ca4a,0x4661a26c),LL(0xfb8507bc,0x5620a4c1),
   LL(0x049f842c,0x4b44d4aa),LL(0x1540e82b,0xceabc5d5),LL(0x15c6f156,0x306710fd),LL(0x63db1d72,0xbe5ae52b), LL(0x334957f1,0x06f1e7e6),LL(0x31144a70,0x57e388f0),LL(0xdf96447b,0xfb69bb2f),LL(0x73e38a12,0x0f78ebd3),
   LL(0x2b7ce542,0xb8222605),LL(0x7472bde1,0xe6d4ce99),LL(0x09d2f4da,0x53e16ebe),LL(0x53b92b2e,0x180ff42e), LL(0x2c34a1c6,0xc59bcc02),LL(0x422c46c2,0x3803d6f9),LL(0x5c14a8a2,0x18aff74f),LL(0x10a08b28,0x55aebf80),
   LL(0x7135593f,0x66097d58),LL(0x2be570cd,0x32e6eff7),LL(0x2a8c860d,0x584e6a10),LL(0xa2eb4163,0xcd185890), LL(0x6d97e134,0x7ceae99d),LL(0xdd8447ce,0xd42c6b70),LL(0xb8c50273,0x59ddbb4a),LL(0x3cf34e1e,0x03c612df),
   LL(0x04b6c5a0,0x84b9ca15),LL(0x18f0e3a3,0x35216f39),LL(0xbd986c00,0x3ec2d2bc),LL(0xd19228fe,0x8bf546d9), LL(0x4cd623c3,0xd1c655a4),LL(0x502b8e5a,0x366ce718),LL(0xeea0bfe7,0x2cfc84b4),LL(0xcf443e8e,0xe01d5cee),
   LL(0x036520f8,0x8ec045d9),LL(0x92d40e98,0xdfb3c3d1),LL(0xcc559a04,0x0bac4cce),LL(0x240ea6b1,0x35eccae5), LL(0xf8a5a0ac,0x180b32db),LL(0xeb699700,0x547972a5),LL(0xca26bca0,0xa3765801),LL(0xa647f25a,0x57e09d0e),
   LL(0x2fdd23cc,0xb956970e),LL(0x5682e971,0xb80288bc),LL(0x9ae86ebc,0xe6e6d91e),LL(0x8c9f1939,0x0564c83f), LL(0x39560368,0x551932a2),LL(0x049c28e2,0xe893752b),LL(0xa6a158c3,0x0b03cee5),LL(0x04964263,0xe12d656b),
   LL(0x63e3bc1d,0x4b47554e),LL(0x45044ff7,0xc719b6a2),LL(0xe48daa07,0x4f24d30a),LL(0xc8c1edc3,0xa3f37556), LL(0x0700d360,0x9a47bf76),LL(0x822ae4e2,0xbb1a1824),LL(0x89f1fb4c,0x22e275a3),LL(0x9968c5f5,0x72b1aa23),
   LL(0xbe063f64,0xa75feaca),LL(0xbce47a09,0x9b392f43),LL(0x1ad07aca,0xd4241509),LL(0x8d26cd0f,0x4b0c591b), LL(0x92f1169a,0x2d42ddfd),LL(0x4cbf2392,0x63aeb1ac),LL(0x0691a2af,0x1de9e877),LL(0xd98021da,0xebe79af7),
   LL(0x40e50acf,0xcfdf2a4e),LL(0xaf01d665,0xf0a98ad7),LL(0x1831be1f,0xefb640bf),LL(0x80e9ada0,0x6fe8bd2f), LL(0x6cafbc91,0x94c103a1),LL(0x8308e08c,0x170f8759),LL(0x9780ff4f,0x5de2d2ab),LL(0x45b201f2,0x666466bc),
   LL(0xf5b343bc,0x58af2010),LL(0xf2f142fe,0x0f2e400a),LL(0xa85f4bdf,0x3483bfde),LL(0x03bfeaa9,0xf0b1d093), LL(0xc7081603,0x2ea01b95),LL(0x3dba1097,0xe943e4c9),LL(0xb438f3a6,0x47be92ad),LL(0xe5bf6636,0x00bb7742),
   LL(0x824297b4,0x136b7083),LL(0x5584455f,0x9d0e5580),LL(0xf1c7d69e,0xab48cedc),LL(0x2a256e76,0x53a9e481), LL(0x65eb2413,0x0402b0e0),LL(0x8fc407a7,0xdadbbb84),LL(0x8d7f5492,0xa65cd5a4),LL(0x74bae294,0x21d44293),
   LL(0x3b5f1cc4,0x66917ce6),LL(0xce872e62,0x37ae52ea),LL(0x2905f244,0xbb087b72),LL(0x1e6af74f,0x12077086), LL(0x1058edea,0x4b644e49),LL(0xb638ca1d,0x827510e3),LL(0x6038591c,0x8cf2b704),LL(0xfe635063,0xffc8b47a),
   LL(0x1b4d5e63,0x3ae220e6),LL(0x9d961b4b,0xbd864742),LL(0x9bd16bed,0x610c107e),LL(0x1127147b,0x4270352a), LL(0x64cfc50e,0x7d17ffe6),LL(0x1e36cb42,0x50dee01a),LL(0x35dc5f9a,0x068a7622),LL(0xdf53f62c,0x9a08d536),
   LL(0x6be5f7de,0x4ed71457),LL(0xc2263c9e,0xd93006f8),LL(0xcacacb36,0xe073694c),LL(0x3ae118ab,0x2ff7a5b4), LL(0xcd871236,0x3cce53f1),LL(0xc2aa6d52,0xf156a39d),LL(0xb198d76d,0x9cc5f271),LL(0x81383d39,0xbc615b6f),
   LL(0xde3eee6b,0xa54538e8),LL(0xab910d91,0x58c77538),LL(0x58d278bd,0x31e5bdbc),LL(0xb963acae,0x3cde4adf), LL(0x5302169c,0xb1881fd2),LL(0xa989ed8b,0x8ca60fa0),LL(0xff96a0ee,0xa1999458),LL(0xac6c283d,0xc1141f03),
   LL(0x6dfafed3,0x7677408d),LL(0x39661588,0x33a01653),LL(0x0b726fa0,0x3c9c15ec),LL(0x6c9b56da,0x090cfd93), LL(0xa3c40af5,0xe34f4bae),LL(0xd21129f1,0x3469eadb),LL(0x1e207ce8,0xcc51674a),LL(0xc83b1ef9,0x1e293b24),
   LL(0x1e6c0bb4,0x17173d13),LL(0x90776d35,0x19004695),LL(0x6de6f922,0xe7980e34),LL(0xf4dd9a22,0x873554cb), LL(0xcbf18a51,0x0316c627),LL(0x3032c081,0x4d93651b),LL(0x3946834d,0x207f2771),LL(0x30cdbf80,0x2c08d7b4),
   LL(0x86df2a61,0x137a4fb4),LL(0xecf7b4a2,0xa1ed9c07),LL(0x7bd042ff,0xb2e460e2),LL(0x5f62f5ec,0xb7f5e2fa), LL(0xcc2423b7,0x7aa6ec6b),LL(0xba63eea7,0x75ce0a7f),LL(0xf250a6e1,0x67a45fb1),LL(0xe53cdc9f,0x93bc919c),
   LL(0x871942df,0x9271f56f),LL(0x7859ad66,0x2372ff6f),LL(0x33cb1a78,0x5f4c2b96),LL(0x5838aa83,0xe3e29101), LL(0xe4e8110c,0xa7ed1611),LL(0x330198ce,0x2a2d70d5),LL(0x6720efe0,0xbdf132e8),LL(0x66a471bf,0xe61a8962),
   LL(0x825808bd,0x796d3a85),LL(0x3fd6e902,0x51dc3cb7),LL(0x916219d1,0x643c768a),LL(0xa2ad7d32,0x36cd7685), LL(0xb22922a4,0xe3db9d05),LL(0xdba29660,0x6494c87e),LL(0xbcd2ebc7,0xf0ac91df),LL(0x45107f8d,0x4deb57a0),
   LL(0xc3d12a73,0x42271f59),LL(0xa5c2c51d,0x5f71687c),LL(0x05797bcb,0xcb1f50c6),LL(0xd6d34eb0,0x29ed0ed9), LL(0x4683c2eb,0xe5fe5b47),LL(0x97447c46,0x4956eeb5),LL(0x71207167,0x5b163a43),LL(0x0248c5ef,0x93fa2fed),
   LL(0x31f63950,0x67930af2),LL(0x14caa2c9,0xa77797c1),LL(0x27ac7e62,0x526e80ee),LL(0x58b28aec,0xe1e6e626), LL(0xb3c9fef0,0x636178b0),LL(0x6d5f90be,0xaf7752e0),LL(0xeece51cf,0x94ecaf18),LL(0xca806e1f,0x2864d0ed),
   LL(0x97c69134,0x6de2e383),LL(0xeb291293,0x5a42c316),LL(0x6a60bae0,0xc7779219),LL(0x6b7599d1,0xa24de346), LL(0xb75d4941,0x49d374aa),LL(0x2d501ff0,0x98900586),LL(0xeb7974cf,0x9f16d40e),LL(0xcdd8c115,0x1033860b),
   LL(0x2094cec3,0xb6c69ac8),LL(0x403b770c,0x9976fb88),LL(0x4859590d,0x1dea026c),LL(0x8562d1fd,0xb6acbb46), LL(0x44569d85,0x7cd6c461),LL(0x97f0891d,0xc3190a36),LL(0x48d5a17d,0xc6f53195),LL(0xd749abc8,0x7d919966),
   LL(0xdd1c8a20,0x65104837),LL(0x2f683419,0x7e5410c8),LL(0xbe94022e,0x958c3ca8),LL(0x6145dac2,0x605c3197), LL(0x01683d54,0x3fc07501),LL(0x595b1234,0x1d7127c5),LL(0x9481277f,0x10b8f87c),LL(0xe65a1adb,0x677db2a8),
   LL(0xddce3345,0xec2fccaa),LL(0x012a4350,0x2a6811b7),LL(0xac598bdc,0x96760ff1),LL(0xd1bf4128,0x054d652a), LL(0x92a21005,0x0a1151d4),LL(0x33110fdf,0xad7f3971),LL(0x1960100f,0x8c95928c),LL(0x7bf03362,0x6c91c825),
   LL(0xce309f06,0xc8c8b2a2),LL(0xca27204b,0xfdb27b59),LL(0x0848e32e,0xd223eaa5),LL(0xe7bfaf1e,0xb93e4b2e), LL(0x44aa3ded,0xc5308ae6),LL(0xc015d573,0x317a666a),LL(0x1a979707,0xc888ce23),LL(0x0d5c4958,0xf141c1e6),
   LL(0x61906373,0xb53b7de5),LL(0xeb999595,0x858dbade),LL(0xa59e5c36,0x8cbb47b2),LL(0xdcf4e842,0x660318b3), LL(0x12ba4b7a,0xbd161ccd),LL(0xf8c8282a,0xf399daab),LL(0xeeb2130d,0x1587633a),LL(0xda38dd7d,0xa465311a),
   LL(0x64d3779b,0x5f75eec8),LL(0xad64c171,0x3c5d0476),LL(0x2a914428,0x87410371),LL(0x90e2fc29,0x8096a891), LL(0x23b3ebc2,0xd3d2ae9d),LL(0xa580cfd6,0x90bdd6db),LL(0xc5b01f6c,0x52dbb7f3),LL(0xe102a2dc,0xe68eded4),
   LL(0x99eb6df0,0x17785b77),LL(0x7386b779,0x26c3cc51),LL(0x6417a48e,0x345ed988),LL(0x07d6ef31,0xe990b4e4), LL(0x2586abba,0x0f456b7e),LL(0x59c96e9a,0x239ca6a5),LL(0xe2eb4206,0xe327459c),LL(0xa002b90a,0x3a4c3313),
   LL(0xf6a3f6fb,0x2a114806),LL(0x85c251dd,0xad5cad2f),LL(0xf5a784d3,0x92c1f613),LL(0x349766d5,0xec7bfacf), LL(0x3e23cb3b,0x04b3cd33),LL(0xc5a64b2d,0x3979fe84),LL(0x7e589106,0x192e2720),LL(0xa15b527f,0xa60c43d1),
   LL(0xbe7cf3a6,0x2dae9082),LL(0xbc967274,0xcc86ba92),LL(0xaea0a8a9,0xf28a2ce8),LL(0x6ee988b3,0x404ca6d9), LL(0x005921b8,0xfd7e9c5d),LL(0x44e79bf9,0xf56297f1),LL(0x0d75ddc2,0xa163b460),LL(0xa1f2be87,0x30b23616),
   LL(0xbfe50e2b,0x4b070d21),LL(0xe1bfede1,0x7ef8cfd0),LL(0x2aac4ae0,0xadba0011),LL(0xb9ebd033,0x2a3e7d01), LL(0xe38d9d1c,0x995277ec),LL(0x9c5d2de3,0xb500249e),LL(0xf13ca8c9,0x8912b820),LL(0x877793af,0xc8798114),
   LL(0xec3f1dec,0x19e6125d),LL(0x911178da,0x07b1f040),LL(0x904a6738,0xd93ededa),LL(0x0bebedcd,0x55187a5a), LL(0xeb329d41,0xf7d04722),LL(0xf170b391,0xf449099e),LL(0xca99f828,0xfd317a69),LL(0x34a4976d,0x50c3db2b),
   LL(0x3757b392,0xe9ba7784),LL(0xaa3ca05a,0x326caefd),LL(0xf1e593d4,0x78e5293b),LL(0x0d98fd13,0x7842a937), LL(0x5f96b10d,0xe694bf96),LL(0x06a8cd05,0x373a9df6),LL(0xe8f0c7fc,0x997d1e51),LL(0x63fd972e,0x1d019790),
   LL(0x5499fb32,0x0064d858),LL(0x77a8aeb7,0x7b67bad9),LL(0x2d08eec5,0x1d3eb977),LL(0xcbabae1d,0x5fc047a6), LL(0xe54a64bb,0x0577d159),LL(0xc43497e4,0x8862201b),LL(0x2ce0608d,0xad6b4e28),LL(0x0b167aac,0x8b687b7d),
   LL(0x8b2ecfa9,0x6ed4d367),LL(0xa90c3c38,0x24dfe62d),LL(0x3fe5c42b,0xa1862e10),LL(0xd5732a9f,0x1ca73dca), LL(0x76bb87ad,0x35f038b7),LL(0xf242b81f,0x674976ab),LL(0xb0fd90cd,0x4f2bde7e),LL(0xa7fdf092,0x6efc172e),
   LL(0x92222f1f,0x3806b69b),LL(0x6cf7ae70,0x5a2459ca),LL(0xa85217ee,0x6789f69c),LL(0xe3dc85ac,0x5f232b5e), LL(0x48e9e516,0x660e3ec5),LL(0x3197eb31,0x124b4e47),LL(0xaafcca23,0x10a0cb13),LL(0x8213224f,0x7bd63ba4),
   LL(0x290a7f4f,0xaffad7cc),LL(0x0286b461,0x6b409c9e),LL(0xffa407af,0x58ab809f),LL(0xc68ac073,0xc3122eed), LL(0x4ef24d7e,0x17bf9e50),LL(0x3e2a5811,0x5d929794),LL(0x02902e01,0x519bc867),LL(0x39c8a851,0x76bba5da),
   LL(0xda94951e,0xe9f9669c),LL(0x66b8d418,0x4b6af58d),LL(0x17d426a4,0xfa321074),LL(0x9dde6027,0xc78e66a9), LL(0x4a53b964,0x0516c083),LL(0xff602330,0xfc659d38),LL(0x58c5c897,0x0ab55e5c),LL(0x838bc5df,0x985099b2),
   LL(0xc52fc238,0x061d9efc),LL(0x6ac1da3f,0x712b2728),LL(0x9283fe08,0xfb658149),LL(0xb8aaa2f7,0x4954ac94), LL(0x7fb2e74f,0x85c0ada4),LL(0xb89926b0,0xee8ba98e),LL(0x23d1af5b,0xe4f9d37d),LL(0xba9b015e,0x14ccdbf9),
   LL(0x7bfe7178,0xb674481b),LL(0x65405868,0x4e1debae),LL(0xc48c867d,0x061b2821),LL(0x513b30ea,0x69c15b35), LL(0x36871088,0x3b4a1666),LL(0x1220b1ff,0xe5e29f5d),LL(0x233d9f4d,0x4b82bb35),LL(0x18cdc675,0x4e076333),
},
/* digit=10 base_pwr=2^70 */
{
   LL(0xa3e6fced,0x0d53f5c7),LL(0xf45fbdeb,0xe8cbbdd5),LL(0x13339a70,0xf85c01df),LL(0x142ceb81,0x0ff71880), LL(0xbd70437a,0x4c4e8774),LL(0xba0bda6a,0x5fb32891),LL(0xf18bd26e,0x1cdbebd2),LL(0x03a9d522,0x2f9526f1),
   LL(0x92c4d684,0x40ce3051),LL(0x7612efcd,0x8b04d725),LL(0x6f9cae20,0xb9dcda36),LL(0xf058856c,0x0edc4d24), LL(0x85427900,0x64f2e6bf),LL(0xdc09dfea,0x3de81295),LL(0x379bf26c,0xd41b4487),LL(0x6df135a9,0x50b62c6d),
   LL(0xc72dfe67,0xd4f8e3b4),LL(0x90e19fdf,0xc416b0f6),LL(0x4c13bd35,0x18b9098d),LL(0x15b8cb9e,0xac11118a), LL(0xf0062841,0xf598a318),LL(0x89f356f4,0xbfe0602f),LL(0x30177a0c,0x7ae3637e),LL(0x61136537,0x34097747),
   LL(0xd005832a,0x0db2fb5e),LL(0x91042e4f,0x5f5efd3b),LL(0xed70f8ca,0x8c4ffdc6),LL(0xb52da9cc,0xe4645d0b), LL(0xc9001d1f,0x9596f58b),LL(0x4e117205,0x52c8f0bc),LL(0xe398a084,0xfd4aa0d2),LL(0x104f49de,0x815bfe3a),
   LL(0x23885e5f,0x97e5443f),LL(0xe8433aab,0xf72f8f99),LL(0xe4d4e604,0xbd00b154),LL(0xe5e173ff,0xd0b35e6a), LL(0x9164722d,0x57b2a048),LL(0x88761ec8,0x3e3c665b),LL(0x3da83832,0x6bdd1397),LL(0x73dafe3b,0x3c8b1a1e),
   LL(0x54317cac,0x4497ace6),LL(0x521771b3,0xbe600ab9),LL(0xb0dfe8b8,0xb42e409e),LL(0x3942310f,0x386a67d7), LL(0x4431cc28,0x25548d8d),LL(0x985dc524,0xa7cff142),LL(0x93c4be32,0x4d60f5a1),LL(0xd071c6e1,0x83ebd5c8),
   LL(0xb1fd2b0b,0xba3a80a7),LL(0x5bec33e8,0x9b3ad396),LL(0x79743fb3,0xb3868d61),LL(0xfdb462fa,0xcfd169fc), LL(0x9ce0a6af,0xd3b499d7),LL(0xe42d3ff8,0x55dc1cf1),LL(0xc6c3e1b2,0x04fb9e6c),LL(0x6f69a474,0x47e6961d),
   LL(0xe548b37b,0x54eb3acc),LL(0x84d40549,0xb38e7542),LL(0x7b341b4f,0x8c3daa51),LL(0x690bf7fa,0x2f6928ec), LL(0x86ce6c41,0x0496b323),LL(0x10adadcd,0x01be1c55),LL(0x4bb5faf9,0xc04e67e7),LL(0xe15c9985,0x3cbaf678),
   LL(0x50ca4247,0x8cd12145),LL(0xe7dd30aa,0xba1aa47a),LL(0xe58fee24,0x2f81ddf1),LL(0xeec9b0e8,0x03452936), LL(0x243aea96,0x8bdc3b81),LL(0x15c3d0e5,0x9a2919af),LL(0x10948361,0x9ea640ec),LL(0x6e0bcccf,0x5ac86d5b),
   LL(0xc36cf440,0xf892d918),LL(0xc939719c,0xaed3e837),LL(0xc0218b64,0xb07b08d2),LL(0xce9790dd,0x6f1bcbba), LL(0x60919b8e,0x4a84d6ed),LL(0x8ac1f9eb,0xd8900791),LL(0x0dd5daef,0xf84941aa),LL(0x67fd62c5,0xb22fe40a),
   LL(0x157f2db3,0x97e15ba2),LL(0x8e28ca9c,0xbda2fc8f),LL(0x37b9f454,0x5d050da4),LL(0x2379d72e,0x3d57eb57), LL(0xfb5ee997,0xe9b5eba2),LL(0xe11538ca,0x01648ca2),LL(0xf6327974,0x32bb76f6),LL(0xff3f4bb7,0x338f14b8),
   LL(0xd7ab9a2d,0x524d226a),LL(0x7dfae958,0x9c00090d),LL(0x8751d8c2,0x0ba5f539),LL(0x3ab8262d,0x8afcbcdd), LL(0xe99d043b,0x57392729),LL(0xaebc943a,0xef51263b),LL(0x20862935,0x9feace93),LL(0xb06c817b,0x639efc03),
   LL(0x66b4be7a,0x1fe054b3),LL(0x84a37a1e,0x3f25a9de),LL(0x78d75cd9,0xf39ef1ad),LL(0x5062c1b5,0xd7b58f49), LL(0xff563436,0x6f74f9a9),LL(0xe8af51e7,0xf718ff29),LL(0x15e97fec,0x5234d313),LL(0x292f1c0a,0xb6a8e2b1),
   LL(0x327720c1,0xa7f53aa8),LL(0xba092cc8,0x956ca322),LL(0x28746c4d,0x8f03d64a),LL(0x66d0d392,0x51fe1782), LL(0x3c832c80,0xd19b34db),LL(0x6da2e3b4,0x60dccc5c),LL(0x0a104ccc,0x245dd62e),LL(0x620b21fd,0xa7ab1de1),
   LL(0x3893d123,0xb293ae0b),LL(0xb15ee71c,0xf7b75783),LL(0x42a9468b,0x5aa3c614),LL(0xdb15d744,0xd686123c), LL(0xa7ab4116,0x8c616891),LL(0xa4e6a459,0x6fcd72c8),LL(0x77e5fad7,0xac219110),LL(0x704fa46b,0xfb6a20e7),
   LL(0x341d81dc,0xe839be7d),LL(0x32148379,0xcddb6889),LL(0xf7026ead,0xda6211a1),LL(0xf4d1cc5e,0xf3b2575f), LL(0xa7a73ae6,0x40cfc8f6),LL(0x61d5b483,0x83879a5e),LL(0x41a50ebc,0xc5acb1ed),LL(0x3c07d8fa,0x59a60cc8),
   LL(0xb1876262,0x1b73bdce),LL(0x12af4ee9,0x2b0d79f0),LL(0xd46e1d07,0x8bcf3b0b),LL(0xe45d152f,0x17d6af9d), LL(0x6d736451,0x73520461),LL(0x56b0bf5a,0x43cbbd97),LL(0xd5999b9d,0xb0833a5b),LL(0xeb72e398,0x702614f0),
   LL(0x59c3e9f8,0x0aadf01a),LL(0xce6b3d16,0x40200e77),LL(0xdeddafad,0xda22bdd3),LL(0x310d72e1,0x76dedaf4), LL(0x4bc2e88f,0x49ef807c),LL(0x146dd5a5,0x6ba81291),LL(0x7d8d59e9,0xa1a4077a),LL(0x802db349,0x87b6a2e7),
   LL(0x1b4e598e,0xd5679997),LL(0x06fe4b1d,0xf499ef1f),LL(0xfcb267c5,0x3978d3ae),LL(0x235786d0,0xb582b557), LL(0x1715cb07,0x32b3b2ca),LL(0x8480241d,0x4c3de6a2),LL(0xcb571ecd,0x63b5ffed),LL(0xed2fe9a9,0xeaf53900),
   LL(0xc3b81990,0xdec98d4a),LL(0x9e0cc8fe,0x1cb83722),LL(0xd2b427b9,0xfe0b0491),LL(0xe983a66c,0x0f2386ac), LL(0xb3291213,0x930c4d1e),LL(0x59a62ae4,0xa2f82b2e),LL(0xf93e89e3,0x77233853),LL(0x11777c7f,0x7f8063ac),
   LL(0x59ad2877,0xff0eb567),LL(0x9865c754,0x6f454642),LL(0x236e9a84,0xe6fe701a),LL(0x06e40fc3,0xc586ef16), LL(0x24bafad9,0x3f62b6e0),LL(0x64da906a,0xc8b42bd2),LL(0xda3276a0,0xc98e1eb4),LL(0x06cbf852,0x30d0e5fc),
   LL(0xe8b4dfd4,0x1b6b2ae1),LL(0x8301cbac,0xd754d5c7),LL(0x112a39ac,0x66097629),LL(0x93ba4ab9,0xf86b5999), LL(0x99f9d581,0x26c9dea7),LL(0xc2fafeaa,0x0473b1a8),LL(0x3b2505a5,0x1469af55),LL(0xd6a43323,0x227d16d7),
   LL(0xad3d97f9,0x3316f73c),LL(0x1f137455,0x52bf3bb5),LL(0x09954e7c,0x953eafeb),LL(0xdd732411,0xa721dfed), LL(0x141d4579,0xb4929821),LL(0xaa3bd435,0x3411321c),LL(0x17fa6015,0xafb355aa),LL(0x18e42f0e,0xb4e7ef4a),
   LL(0x59371000,0x604ac97c),LL(0x7f759c18,0xe1c48c70),LL(0xa5db6b65,0x3f62ecc5),LL(0x38a21495,0x0a78b173), LL(0xbcc8ad94,0x6be1819d),LL(0xd89c3400,0x70dc04f6),LL(0xa6b4840a,0x462557b4),LL(0x60bd21c0,0x544c6ade),
   LL(0x907a544b,0x6a00f24e),LL(0x313da210,0xa7520dcb),LL(0x11e4994b,0xfe939b75),LL(0xbc275d70,0x918b6ba6), LL(0x644be892,0xd3e5e0fc),LL(0xfdaf6c42,0x707a9816),LL(0xf15c13fe,0x60145567),LL(0xe130a54a,0x4818ebaa),
   LL(0x58d2f767,0x28aad3ad),LL(0xd7e7c773,0xdc5267fd),LL(0xc3afcc98,0x4919cc88),LL(0x2db8cd4b,0xaa2e6ab0), LL(0xd0c63eaa,0xd46fec04),LL(0x19ffa832,0xa1cb92c5),LL(0xe43a631f,0x678dd178),LL(0x3dc788b3,0xfb5ae1cd),
   LL(0x6e77de04,0x68b4fb90),LL(0xf06dbb97,0x7992bcf0),LL(0xc417c01d,0x896e6a13),LL(0xb956be01,0x8d96332c), LL(0x413aa2b9,0x902fc93a),LL(0xfc98c8a5,0x99a4d915),LL(0x565f1137,0x52c29407),LL(0x21e4f281,0x4072690f),
   LL(0x02ff6072,0x36e607cf),LL(0x8ad98cdc,0xa47d2ca9),LL(0xf5f56609,0xbf471d1e),LL(0xf264ada0,0xbcf86623), LL(0xaa9e5cb6,0xb70c0687),LL(0x17401c6c,0xc98124f2),LL(0xd4a61435,0x8189635f),LL(0xa9d98ea6,0xd28fb8af),
   LL(0x40c251f8,0xb9a67c2a),LL(0xa2da44be,0x88cd5d87),LL(0xe09b5423,0x437deb96),LL(0x64287dc1,0x150467db), LL(0xcdabb839,0xe161debb),LL(0xf1839a3e,0xa79e9742),LL(0x652d202b,0xbb8dd3c2),LL(0xe9f97d96,0x7b3e67f7),
   LL(0xb1cb6ac9,0x5aa5d78f),LL(0xca1d0d45,0xffa13e8e),LL(0x2ba5bf95,0x369295dd),LL(0x39aff05e,0xd68bd1f8), LL(0x26d783f2,0xaf0d86f9),LL(0xfc3aafc1,0x543a59b3),LL(0x7b7da97c,0x3fcf81d2),LL(0xd25dee46,0xc990a056),
   LL(0x519cce2c,0x3e6775b8),LL(0xae13d863,0xfc9af71f),LL(0x47c1605c,0x774a4a6f),LL(0x2fd205e8,0x46ba4245), LL(0xd3fd524d,0xa06feea4),LL(0x6de1acc2,0x1e724641),LL(0x334e2b42,0xf53816f1),LL(0x922f0024,0x49e5918e),
   LL(0x65c7322d,0x439530b6),LL(0xb3c1b3fb,0xcf12cc01),LL(0x0172f685,0xc70b0186),LL(0x1b58391d,0xb915ee22), LL(0xa317db24,0x9afdf03b),LL(0x17b8ffc4,0x87dec659),LL(0xe4d3d050,0x7f46597b),LL(0x006500e7,0x80a1c1ed),
   LL(0x78bf030e,0x84902a96),LL(0x50560148,0xfb5e9c9a),LL(0x63362426,0x6dae0a92),LL(0xa9e30c40,0xdcaeecf4), LL(0x518d0c6b,0xc0d887bb),LL(0xcb985b9d,0x99181152),LL(0xef7bc381,0xad186898),LL(0x9ee46201,0x18168ffb),
   LL(0x2502753c,0x9a04cdaa),LL(0x51407c41,0xbb279e26),LL(0xf23564e5,0xeacb03aa),LL(0x71e61016,0x18336582), LL(0xeb809877,0x8684b8c4),LL(0xea0e672e,0xb336e18d),LL(0x34ee5867,0xefb601f0),LL(0x1341cfd1,0x2733edbe),
   LL(0x26025c3c,0xb15e809a),LL(0x9350df88,0xe6e981a6),LL(0x8502fd8e,0x92376237),LL(0x0c12be9b,0x4791f216), LL(0x25f02425,0xb7256789),LL(0x7a974443,0xec863194),LL(0xfb41cc52,0x7c0ce882),LL(0xf25c07f2,0xc266ff7e),
   LL(0x017025f3,0x3d4da8c3),LL(0xfb9579b4,0xefcf628c),LL(0x1f3716ec,0x5c4d0016),LL(0x6801116e,0x9c27ebc4), LL(0x1da1767e,0x5eba0ea1),LL(0x47004c57,0xfe151452),LL(0x8c2373b7,0x3ace6df6),LL(0x5dbc37ac,0x75c3dffe),
   LL(0xddc925fc,0x3dc32a73),LL(0x2f65ee0b,0xb679c841),LL(0x451cbfeb,0x715a3295),LL(0xf76e9a29,0xd9889768), LL(0xb28ad247,0xec20ce7f),LL(0x00894d79,0xe99146c4),LL(0x9f5e3ea7,0x71457d7c),LL(0x38030031,0x097b2662),
   LL(0xcf9f82a8,0xdb7f6ae6),LL(0x438f473a,0x319decb9),LL(0x283856c3,0xa63ab386),LL(0xb06a361b,0x13e3172f), LL(0x7d5a006c,0x2959f8dc),LL(0x75fba752,0x2dbc27c6),LL(0x87c22c9e,0xc1227ab2),LL(0x71a268b2,0x06f61f75),
   LL(0x04779ce2,0x1b6bb971),LL(0x0aadcb1d,0xaca83812),LL(0xaeaab2d5,0x297ae0bc),LL(0x5bfb9f13,0xa5c14ee7), LL(0xf17a62c7,0xaa00c583),LL(0x173759f6,0x39eb962c),LL(0x86c9a88f,0x1eeba1d4),LL(0xdf016c5e,0x0ab6c37a),
   LL(0xa28a0749,0xa2a147db),LL(0xee519165,0x246c20d6),LL(0xd3810715,0x5068d1b1),LL(0x748160b9,0xb1e7018c), LL(0xf380ff62,0x03f5b1fa),LL(0xf3cb2c1e,0xef7fb1dd),LL(0xfc91a7da,0xeab539a8),LL(0xf3f9b561,0x83ddb707),
   LL(0xfe7df7a4,0xc550e211),LL(0x063f6f40,0xa7cd07f2),LL(0x2976879c,0xb0de3635),LL(0xe55741da,0xb5f83f85), LL(0xf3d8ac3d,0x4ea9d25e),LL(0x62819f02,0x6fe2066f),LL(0xcef4a564,0x4ab2b9c2),LL(0x5ffa2de3,0x1e155d96),
   LL(0xc3a72d00,0x0eb0a19b),LL(0x8513c31b,0x4037665b),LL(0x04c64637,0x2fb2b6bf),LL(0x08cdc639,0x45c34d6e), LL(0xf01fd796,0x56f1e10f),LL(0xfe3667b8,0x4dfb8101),LL(0x9021d0c0,0xe0eda253),LL(0x8a06c6ab,0x7a94e9ff),
   LL(0xbb9aa882,0x2d3bb0d9),LL(0xec05fd10,0xea20e4e5),LL(0x1a1ca64e,0xed7eeb5f),LL(0xc6327cbd,0x2fa6b43c), LL(0x3aa91121,0xb577e3cf),LL(0x3a34079b,0x8c6bd5ea),LL(0x60e02fc0,0xd7e5ba39),LL(0x90141bf8,0xf16dd2c3),
   LL(0x80101b98,0xb57276d9),LL(0xb82f0f66,0x760883fd),LL(0x4bc3eff3,0x89d7de75),LL(0x5dc2ab40,0x03b60643), LL(0xe05beeac,0xcd6e53df),LL(0xbc3325cd,0xf2f1e862),LL(0x774f03c3,0xdd0f7921),LL(0x4552cc1b,0x97ca7221),
   LL(0x1cd19f72,0x5a0d6afe),LL(0xf183fbeb,0xa20915dc),LL(0x832c403c,0x9fda4b40),LL(0xbe425442,0x32738edd), LL(0xb5eccf1a,0x469a1df6),LL(0x28bbe1f0,0x4b5aff42),LL(0x570dfc93,0x31359d7f),LL(0xf0088628,0xa18be235),
   LL(0xb00ed3a9,0xa5b30fba),LL(0x73cdf8be,0x34c61374),LL(0xabc56797,0x2c5c5f46),LL(0xb82a8ae2,0x5cecf93d), LL(0xa968fbf0,0x7d3dbe41),LL(0x1a5c7f3d,0xd23d4583),LL(0xc087a9c7,0xf28f69a0),LL(0x474471ca,0xc2d75471),
   LL(0x4eb732ec,0x36ec9f4a),LL(0xb1ca6bed,0x6c943bbd),LL(0xf2457892,0xd64535e1),LL(0xf7e2ac06,0x8b84a8ea), LL(0x2499dd5f,0xe0936cd3),LL(0x0ed04e57,0x12053d7e),LL(0xe4305d9d,0x4bdd0076),LL(0x1f67f0a2,0x34a527b9),
   LL(0x9cec46ea,0xe79a4af0),LL(0x658b9bc7,0xb15347a1),LL(0x35af2f75,0x6bd2796f),LL(0x4051c435,0xac957990), LL(0xc33a655d,0x2669dda3),LL(0x88514aa3,0x5d503c2e),LL(0x3753dd41,0xdfa11337),LL(0x0b754f78,0x3f054673),
   LL(0x496125bd,0xbf185677),LL(0x3775006c,0xfb0023c8),LL(0x3a037899,0xfa0f072f),LL(0x0e4aea57,0x4222b6eb), LL(0x7866d25a,0x3dde5e76),LL(0x4837aa6f,0xb6eb04f8),LL(0x2cf1cdb8,0x5315591a),LL(0x2d4e683c,0x6dfb4f41),
   LL(0x48ee1f3a,0x7e923ea4),LL(0x05a2afd5,0x9604d9f7),LL(0x40ea4948,0xbe1d4a33),LL(0xb44cbd2f,0x5b45f1f4), LL(0x4acc757e,0x5faf8376),LL(0x63d68ff7,0xa7cf9ab8),LL(0xdf0e404b,0x8ad62f69),LL(0x12bdafdf,0xd65f33c2),
   LL(0xa377b14e,0xc365de15),LL(0x8e39f60c,0x6bf5463b),LL(0x2ce68148,0x62030d2d),LL(0xe6f843a8,0xd95867ef), LL(0xef5ab017,0xd39a0244),LL(0x4ab55d12,0x0bd2d8c1),LL(0x41639169,0xc9503db3),LL(0xf7660c8a,0x2d4e25b0),
   LL(0xe224c5d7,0x760cb3b5),LL(0x68616919,0xfa3baf8c),LL(0x8d142552,0x9fbca113),LL(0x7669ebf5,0x1ab18bf1), LL(0x9bdf25dd,0x55e6f53e),LL(0xcb6cd154,0x04cc0bf3),LL(0x95e89080,0x595bef49),LL(0x104a9ac1,0xfe9459a8),
   LL(0xcce9bb32,0xad2d89ca),LL(0xf7de8285,0xddea65e1),LL(0xb351bd4b,0x62ed8c35),LL(0x0c0e19a7,0x4150ff36), LL(0x345f4e47,0x86e3c801),LL(0x203a266c,0x3bf21f71),LL(0x855b1f13,0x7ae110d4),LL(0x07262517,0x5d6aaf6a),
   LL(0x813d28f1,0x1e0f12e1),LL(0x7ad7a523,0x6000e11d),LL(0xc744a17b,0xc7d8deef),LL(0x14c05a00,0x1e990b48), LL(0x93e976d5,0x68fddaee),LL(0x46610d63,0x696241d1),LL(0x893dda88,0xb204e7c3),LL(0x6a3a6946,0x8bccfa65),
   LL(0xc5cd1411,0xb59425b4),LL(0xff3658b1,0x701b4042),LL(0x4784cf93,0xe3e56bca),LL(0x8fe68d60,0x27de5f15), LL(0xf8d53f19,0x4ab9cfce),LL(0xa40a730d,0xddb10311),LL(0x4eee0a8a,0x6fa73cd1),LL(0x5249719d,0xfd548748),
   LL(0xa8123ef0,0x49d66316),LL(0xe7f95438,0x73c32db4),LL(0x0d9e7854,0x2e2ed209),LL(0x9d9f0507,0xf98a9329), LL(0x0c6aa20a,0xc5d33cf6),LL(0x75279bb2,0x9a32ba14),LL(0x774a7307,0x7e3202cb),LL(0xe8c42dbd,0x64ed4bc4),
   LL(0xd4caed0d,0xc20f1a06),LL(0x171d22b3,0xb8021407),LL(0xd13268d7,0xd426ca04),LL(0x25f4d126,0x92377007), LL(0x71f21a85,0x4204cbc3),LL(0xf82369ba,0x18461b7a),LL(0x3fc858f9,0xc0c07d31),LL(0xe2bab569,0x5deb5a50),
   LL(0xd5eea89e,0xd5959d46),LL(0x08437f4b,0xfdff8424),LL(0x3cfe254f,0xf21071e4),LL(0x95468321,0x72417696), LL(0x102cae3e,0x5d8288b9),LL(0xf1965dff,0x2d143e3d),LL(0xa078d847,0x00c9a376),LL(0x26028731,0x6fc0da31),
   LL(0xe45083a2,0xa2baeadf),LL(0x5e5b4bcd,0x66bc7218),LL(0xd04b8e7f,0x2c826442),LL(0x6c4b586b,0xc19f5451), LL(0x5b7eeed5,0x60182c49),LL(0x7aa9dfa1,0xd9954ecd),LL(0xc73884ad,0xa403a8ec),LL(0x9bb39041,0x7fb17de2),
   LL(0xabb020e8,0x694b64c5),LL(0x19c4eec7,0x3d18c184),LL(0x1c4793e5,0x9c4673ef),LL(0x056092e6,0xc7b8aeb5), LL(0xf0f8c16b,0x3aa1ca43),LL(0xd679b2f6,0x224ed5ec),LL(0x55a205c9,0x0d56eeaf),LL(0x4b8e028b,0xbfe115ba),
   LL(0x3927f4fe,0x97e60849),LL(0x759aa7c5,0xf91fbf94),LL(0x6be90a51,0x985af769),LL(0x78ccb823,0xc1277b78), LL(0xe7a75952,0x395b656e),LL(0x928da5f5,0x00df7de0),LL(0x4ca4454f,0x09c23175),LL(0x7aa2d3c1,0x4ec971f4),
   LL(0xe75d9ccc,0x45c3c507),LL(0x3dc90306,0x63b7be8a),LL(0x5db44bdc,0x37e09c66),LL(0x6841c6a2,0x50d60da1), LL(0x08df1b12,0x6f9b65ee),LL(0x7ff089df,0x38734879),LL(0x3fe8013d,0x9c331a66),LL(0x5f42fcc8,0x017f5de9),
   LL(0xe8e57567,0x43077866),LL(0xf9fcdb18,0xc9f781ce),LL(0x9b12e174,0x38131dda),LL(0x8a03752a,0x25d84aa3), LL(0x4d0c0ce2,0x45e09e09),LL(0x92bebba5,0x1564008b),LL(0xa87284c7,0xf7e8ad31),LL(0x97e7bbaa,0xb7c4b46c),
   LL(0x97acf4ec,0x3e22a7b3),LL(0x5ea8b640,0x0426c400),LL(0x4e969285,0x5e3295a6),LL(0xa6a45670,0x22aabc59), LL(0x5f5942bc,0xb929714c),LL(0xfa3182ed,0x9a6168bd),LL(0x104152ba,0x2216a665),LL(0xb6926368,0x46908d03),
},
/* digit=11 base_pwr=2^77 */
{
   LL(0x5a1251fb,0xa9f5d874),LL(0xc72725c7,0x967747a8),LL(0x31ffe89e,0x195c33e5),LL(0xe964935e,0x609d210f), LL(0x2fe12227,0xcafd6ca8),LL(0x0426469d,0xaf9b5b96),LL(0x5693183c,0x2e9ee04c),LL(0xc8146fef,0x1084a333),
   LL(0xaed1d1f7,0x96649933),LL(0x50563090,0x566eaff3),LL(0xad2e39cf,0x345057f0),LL(0x1f832124,0x148ff65b), LL(0xcf94cf0d,0x042e89d4),LL(0x520c58b3,0x319bec84),LL(0x5361aa0d,0x2a267626),LL(0x8fbc87ad,0xc86fa302),
   LL(0x5c8b06d5,0xfc83d2ab),LL(0xfe4eac46,0xb1a785a2),LL(0x846f7779,0xb99315bc),LL(0xef9ea505,0xcf31d816), LL(0x15d7dc85,0x2391fe6a),LL(0xb4016b33,0x2f132b04),LL(0x181cb4c7,0x29547fe3),LL(0x650155a1,0xdb66d8a6),
   LL(0xadc1696f,0x6b66d7e1),LL(0x0acd72d0,0x98ebe593),LL(0xcc1b7435,0x65f24550),LL(0xb4b9a5ec,0xce231393), LL(0xdb067df9,0x234a22d4),LL(0xcaff9b00,0x98dda095),LL(0x6100c9c1,0x1bbc75a0),LL(0x939cf695,0x1560a9c8),
   LL(0x99e0925f,0xcf006d3e),LL(0x6322375a,0x2dd74a96),LL(0xb56af5ba,0xc58b446a),LL(0xe0b9b4f1,0x50292683), LL(0x1aeaffa3,0xe2c34cb4),LL(0x9b9587c1,0x8b17203f),LL(0xead1350c,0x6d559207),LL(0xfb7f9604,0x2b66a215),
   LL(0xfe51bf74,0x0850325e),LL(0x5e460094,0x9c4f579e),LL(0x76da2f25,0x5c87b92a),LL(0x6febef33,0x889de4e0), LL(0x646083ce,0x6900ec06),LL(0xbfe12773,0xbe2a0335),LL(0xc5344110,0xadd1da35),LL(0xb802cd20,0x757568b7),
   LL(0x00f7e6c8,0x75559779),LL(0x0facd2f0,0x38e8b94f),LL(0x03fde375,0xfea1f3af),LL(0x75881dfc,0x5e11a1d8), LL(0xc1e2f2ef,0xb3a6b02e),LL(0xc605a6c5,0x193d2bbb),LL(0x339a0b2d,0x325ffeee),LL(0x9e0c8846,0x27b6a724),
   LL(0xf1c367ca,0xe4050f1c),LL(0xc90fbc7d,0x9bc85a9b),LL(0xe1a11032,0xa373c4a2),LL(0xad0393a9,0xb64232b7), LL(0x167dad29,0xf5577eb0),LL(0x94b78ab2,0x1604f301),LL(0xe829348b,0x0baa94af),LL(0x41654342,0x77fbd8dd),
   LL(0xb964e39a,0xdab50ea5),LL(0xd0d3c76e,0xd4c29e3c),LL(0x56d11964,0x80dae67c),LL(0xe5ffcc2f,0x7307a8bf), LL(0x91708c3b,0x65bbc1aa),LL(0x28bf0eeb,0xa151e62c),LL(0x6fa34db7,0x6cb53381),LL(0xa29403a8,0x5139e05c),
   LL(0x94a7cd2e,0x6ff651b4),LL(0x0699336c,0x5671ffd1),LL(0x979a896a,0x6f5fd2cc),LL(0xd8148cef,0x11e893a8), LL(0x65cf7b10,0x988906a1),LL(0xc50d8485,0x81b67178),LL(0x8a35b3de,0x7c0deb35),LL(0xc1d29799,0x423ac855),
   LL(0xdac50b74,0xaf580d87),LL(0x5869734c,0x28b2b89f),LL(0x874e28fb,0x99a3b936),LL(0x25f3f73a,0xbb2c9190), LL(0x84a9d5b7,0x199f6918),LL(0x7e770374,0x7ebe2325),LL(0x0738efe2,0xf442e107),LL(0xcf9082d2,0xcf9f3f56),
   LL(0x09618708,0x719f69e1),LL(0xc183f9b1,0xcc9e8364),LL(0x366a21af,0xec203a95),LL(0x068b141f,0x6aec5d6d), LL(0x994f04e9,0xee2df78a),LL(0x271245b0,0xb39ccae8),LL(0x97e43f4f,0xb875a4a9),LL(0xdb2cea98,0x507dfe11),
   LL(0x489b03e9,0x4fbf81cb),LL(0x6ec414fa,0xdb86ec5b),LL(0xf51b3ae5,0xfad444f9),LL(0x1914e3fe,0xca7d33d6), LL(0x0ae6c4d0,0xa9c32f5c),LL(0x73969568,0xa9ca1d1e),LL(0x1aa7467e,0x98043c31),LL(0xe21b5ac6,0xe832e75c),
   LL(0x5232123d,0x314b7aea),LL(0x65ae86db,0x08307c8c),LL(0xaa4668ed,0x06e7165c),LL(0xb4d3ec39,0xb170458b), LL(0xc19bb986,0x4d2e3ec6),LL(0xae0304ed,0xc5f34846),LL(0x6c9f9722,0x917695a0),LL(0x4cab1c0a,0x6c7f7317),
   LL(0x9d6d2e8b,0x6295940e),LL(0x549f7c97,0xd318b8c1),LL(0x97713885,0x22453204),LL(0xa8a440fe,0x468d834b), LL(0xbfba796e,0xd81fe5b2),LL(0x6d71f116,0x152364db),LL(0xb5b66e53,0xbb8c7c59),LL(0x2641a192,0x0b12c61b),
   LL(0xfcf0a7fd,0x31f14802),LL(0x5488b01e,0x42fd0789),LL(0x9952b498,0x71d78d6d),LL(0x07ac5201,0x8eb572d9), LL(0x4d194a88,0xe0a2a44c),LL(0xba017e66,0xd2b63fd9),LL(0xf888aefc,0x78efc6c8),LL(0x4a881a11,0xb76f6bda),
   LL(0xb46c2397,0x187f314b),LL(0x5ded2819,0x004cf566),LL(0x38764d34,0xa9ea5704),LL(0x78084709,0xbba45217), LL(0x1171121e,0x06474571),LL(0xe7c9b671,0xad7b7eb1),LL(0x730f7507,0xdacfbc40),LL(0xc7ad7bd1,0x178cd8c6),
   LL(0xb2a67238,0xbf0be101),LL(0xaf9c14f2,0x3556d367),LL(0xa5662075,0x104b7831),LL(0x79d9e60a,0x58ca59bb), LL(0xa569a73b,0x4bc45392),LL(0x5698f6c9,0x517a52e8),LL(0xaeadd755,0x85643da5),LL(0x2a581b84,0x1aed0cd5),
   LL(0x80af1372,0xb9b4ff84),LL(0xf1ba5d1f,0x244c3113),LL(0xf5f98d31,0x2a5dacbe),LL(0x4375bc2a,0x2c3323e8), LL(0x5594b1dd,0x17a3ab4a),LL(0xceb4797e,0xa1928bfb),LL(0xe4886a19,0xe83af245),LL(0x72b5a74a,0x8979d546),
   LL(0x19f9e967,0xa0f726bc),LL(0xe8fbbf4e,0xd9d03152),LL(0xb7707d40,0xcfd6f51d),LL(0x63f6e6e0,0x633084d9), LL(0x55667eaf,0xedcd9cdc),LL(0x2e44d56f,0x73b7f92b),LL(0x4e962b14,0xfb2e39b6),LL(0xf671fcbf,0x7d408f6e),
   LL(0x164a89bb,0xcc634ddc),LL(0x3ef3bd05,0x74a42bb2),LL(0x428decbb,0x1280dbb2),LL(0x402c8596,0x6103f6bb), LL(0x355a5752,0xfa2bf581),LL(0x00946674,0x562f96a8),LL(0x6da0223b,0x4e4ca16d),LL(0x28d3aa25,0xfe47819f),
   LL(0xf8dfcf8a,0x9eea3075),LL(0x95669825,0xa284f0aa),LL(0x867d3fd8,0xb3fca250),LL(0x269d691e,0x20757b5f), LL(0x93b8a5de,0xf2c24020),LL(0xebc06da6,0xd3f93359),LL(0xb2739c33,0x1178293e),LL(0xbcd686e5,0xd2a3e770),
   LL(0xcd941534,0xa76f49f4),LL(0xe3c71c0e,0x0d37406b),LL(0x3b97f7e3,0x172d9397),LL(0xbd7fd0de,0xec17e239), LL(0x6f496ba2,0xe3290551),LL(0x36ad50e7,0x6a693172),LL(0x83e7eff5,0xc4e539a2),LL(0x18e1b4cf,0x752737e7),
   LL(0x68af43ee,0xa2f7932c),LL(0x703d00bd,0x5502468e),LL(0x2fb061f5,0xe5dc978f),LL(0x28c815ad,0xc9a1904a), LL(0x470c56a4,0xd3af538d),LL(0x193d8ced,0x159abc5f),LL(0x20108ef3,0x2a37245f),LL(0x223f7178,0xfa17081e),
   LL(0x10c8c0f5,0x27b0fb2b),LL(0x40650547,0x2102c3ea),LL(0x8ac3bfa7,0x594564df),LL(0x509dad96,0x98102033), LL(0xf1d18a13,0x6989643f),LL(0xd7fc5af0,0x35eebd91),LL(0xfaeaafd8,0x078d096a),LL(0xdef3de98,0xb7a89341),
   LL(0xecf2a73a,0x2a206e8d),LL(0x8e551994,0x066a6397),LL(0xb98d53a2,0x3a6a088a),LL(0x2d1124aa,0x0ce7c67c), LL(0x759a113c,0x48cec671),LL(0x4f6f67fa,0xe3b373d3),LL(0xfd36727b,0x5455d479),LL(0xa13c0d81,0xe5a428ee),
   LL(0x1c86682b,0xb853dbc8),LL(0xb8d02b2a,0xb78d2727),LL(0x8ebc329a,0xaaf69bed),LL(0x293b2148,0xdb6b40b3), LL(0xb8c4961f,0xe42ea77d),LL(0x20e5e0ab,0xb1a12f7c),LL(0x79e8b05e,0xa0ec5274),LL(0xfab60a80,0x68027391),
   LL(0x16b1bd5e,0x6bfeea5f),LL(0x4de30ad3,0xf957e420),LL(0x6a353b9e,0xcbaf664e),LL(0x26d14feb,0x5c873312), LL(0xb65f57cb,0x4e87f98c),LL(0x5e0cdd41,0xdb60a621),LL(0xa6881440,0x67c16865),LL(0x46ab52aa,0x1093ef1a),
   LL(0x3f4ece64,0xc095afb5),LL(0x7604551a,0x6a6bb02e),LL(0x0b26b8cd,0x55d44b4e),LL(0xf971268a,0xe5f9a999), LL(0x11a7de84,0xc08ec425),LL(0xfda469dd,0x83568095),LL(0x6c6c90a2,0x737bfba1),LL(0xbe229831,0x1cb9c4a0),
   LL(0xbb2eec64,0x93bccbba),LL(0xda03adbe,0xa0c23b64),LL(0xe0e86ac4,0x5f7aa00a),LL(0xfc1401e6,0x470b941e), LL(0x9df43574,0x5ad8d679),LL(0x0f65d810,0x4ccfb8a9),LL(0xaa7fbd81,0x1bce80e3),LL(0x9508d20a,0x273291ad),
   LL(0x42a92806,0xf5c4b46b),LL(0xa86ab44a,0x810684ec),LL(0xca0bc9f8,0x4591640b),LL(0x5c4b6054,0xb5efcdfc), LL(0x6e9edd12,0x16fc8907),LL(0xd4d792f9,0xe29d0b50),LL(0x9b03116d,0xa45fd01c),LL(0xc81765a4,0x85035235),
   LL(0xb4b4b67c,0x1fe2a9b2),LL(0xe8020604,0xc1d10df0),LL(0xbc8058d8,0x9d64abfc),LL(0x712a0fbb,0x8943b9b2), LL(0x3b3def04,0x90eed914),LL(0x4ce775ff,0x85ab3aa2),LL(0x7bbc9040,0x605fd4ca),LL(0xe2c75dfb,0x8b34a564),
   LL(0x10358560,0x41ffc94a),LL(0x9e5c28aa,0x2d8a5072),LL(0x4cc7eb15,0xe915a0fc),LL(0x8f6d0f5d,0xe9efab05), LL(0xd19e9b91,0xdbab47a9),LL(0x0276154c,0x8cfed745),LL(0x2cfede0d,0x154357ae),LL(0x19f5a4ef,0x520630df),
   LL(0xe382360f,0x25759f7c),LL(0x88bf5857,0xb6db05c9),LL(0x6c58d46c,0x2917d61d),LL(0xfd20cb7a,0x14f8e491), LL(0x11c20340,0xb68a727a),LL(0xaf7ccbb6,0x0386f86f),LL(0xfee09a20,0x5c8bc6cc),LL(0xbb7eea35,0x7d76ff4a),
   LL(0xdb15be7a,0xa7bdebe7),LL(0xd89f0302,0x67a08054),LL(0xc1193364,0x56bf0ea9),LL(0x62837ebe,0xc8244467), LL(0x20d841b8,0x32bd8e8b),LL(0xdbb8a54f,0x127a0548),LL(0x63b20236,0x83dd4ca6),LL(0x203491fa,0x87714718),
   LL(0xaa8a5288,0x4dabcaaa),LL(0xaf23a1c9,0x91cc0c8a),LL(0x3f220e0c,0x34c72c6a),LL(0x1232144a,0xbcc20bdf), LL(0xa20ede1b,0x6e2f42da),LL(0x74a00515,0xc441f00c),LL(0x734b8c4b,0xbf46a5b6),LL(0x7b56c9a4,0x57409503),
   LL(0xe4585d45,0x9f735261),LL(0x6734e642,0x9231faed),LL(0xbe70ee6c,0x1158a176),LL(0x7c3501bf,0x35f1068d), LL(0xa2d26115,0x6beef900),LL(0xef0afee3,0x649406f2),LL(0xbc2420a1,0x3f43a60a),LL(0xd5aee4ac,0x509002a7),
   LL(0x3ff3571b,0xb46836a5),LL(0x837927c1,0x24f98b78),LL(0x4533c716,0x6254256a),LL(0xd07ee196,0xf27abb0b), LL(0x5c6d5bfd,0xd7cf64fc),LL(0xf0cd7a77,0x6915c751),LL(0x8798f534,0xd9f59012),LL(0xf81d8b5f,0x772b0da8),
   LL(0x2e03fa69,0x1244260c),LL(0x3be1a374,0x36cf0e3a),LL(0xef06b960,0x6e7c1633),LL(0x671f90f6,0xa71a4c55), LL(0x33c673db,0x7a941251),LL(0x73e8c131,0xc0bea510),LL(0xd4f6c734,0x61a8a699),LL(0x341ed001,0x25e78c88),
   LL(0x8e2f7d90,0x5c18acf8),LL(0x77be32cd,0xfdbf33d7),LL(0xd2eb5ee9,0x0a085cd7),LL(0xb3201115,0x2d702cfb), LL(0x85c88ce8,0xb6e0ebdb),LL(0x1e01d617,0x23a3ce3c),LL(0x567333ac,0x3041618e),LL(0x157edb6b,0x9dd0fd8f),
   LL(0xb57872b8,0x27f74702),LL(0x657d5fe1,0x2ef26b4f),LL(0x57cf3d40,0x95426f0a),LL(0x65a6067a,0x847e2ad1), LL(0x09996a74,0xd474d9a0),LL(0x2a26115c,0x16a56acd),LL(0xd16f4d43,0x02a615c3),LL(0xaadb85b7,0xcc3fc965),
   LL(0xce07d1b0,0x386bda73),LL(0x58ad4178,0xd82910c2),LL(0xcd2617f4,0x124f82cf),LL(0xef691770,0xcc2f5e8d), LL(0xb8c30ccc,0x82702550),LL(0x1a8e575a,0x7b856aea),LL(0xb1ab9459,0xbb822fef),LL(0xec24e38e,0x085928bc),
   LL(0xba8f4b4d,0x5d0402ec),LL(0x00b4d58b,0xc07cd4ba),LL(0x29227e7a,0x5d8dffd5),LL(0x31bf386f,0x61d44d0c), LL(0x135e6f4d,0xe486dc2b),LL(0xe79410ef,0x680962eb),LL(0xf10088b5,0xa61bd343),LL(0xe2e28686,0x6aa76076),
   LL(0x8fb98871,0x80463d11),LL(0xbbc76aff,0xcb26f5c3),LL(0xfbe03614,0xd4ab8edd),LL(0xc0cf2dee,0xc8eb579b), LL(0xc93bae41,0xcc004c15),LL(0x3aeca3b2,0x46fbae5d),LL(0x0f1e9ab1,0x671235cf),LL(0x9ec285c1,0xadfba934),
   LL(0xf216c980,0x88ded013),LL(0xf79e0bc1,0xc8ac4fb8),LL(0xfb97a237,0xa29b89c6),LL(0x9922d8e7,0xb697b780), LL(0xddb945b5,0x3142c639),LL(0xe094c3a9,0x447b06c7),LL(0x72266c90,0xcdcb3642),LL(0xa9385046,0x633aad08),
   LL(0xb57c6477,0xa36c936b),LL(0xe94dbcc6,0x871f8b64),LL(0xa591a67b,0x28d0fb62),LL(0xc1d926f5,0x9d40e081), LL(0xf2d84b5a,0x3111eaf6),LL(0xa565b644,0x228993f9),LL(0x2c83188b,0x0ccbf592),LL(0x3df3e197,0xf87b30ab),
   LL(0x7642bca8,0xb8658b31),LL(0x52800f17,0x1a032d7f),LL(0x79bf9445,0x051dcae5),LL(0x54a2e253,0xeba6b8ee), LL(0xd4485692,0x5c8b9cad),LL(0x8986e9be,0x84bda40e),LL(0x2f0db448,0xd16d16a4),LL(0xa14d4188,0x8ec80050),
   LL(0x98fa7aaa,0xb2b26107),LL(0xf073aa4e,0x41209ee4),LL(0xf2d6b19b,0xf1570359),LL(0xfc577caf,0xcbe6868c), LL(0x32c04dd3,0x186c4bdc),LL(0xcfeee397,0xa6c35fae),LL(0xf086c0cf,0xb4a1b312),LL(0xd9461fe2,0xe0a5ccc6),
   LL(0x1536189f,0xc32278aa),LL(0xba6df571,0x1126c55f),LL(0xb194560e,0x0f71a602),LL(0x324bd6e1,0x8b2d7405), LL(0x3738be71,0x8481939e),LL(0x1a4d97a9,0xb5090b1a),LL(0xf05ba915,0x116c65a3),LL(0xaae448aa,0x21863ad3),
   LL(0xa7aae5d3,0xd24e2679),LL(0x0de5c1c4,0x7076013d),LL(0xbb05b629,0x2d50f8ba),LL(0x6e66efbb,0x73c1abe2), LL(0xf2488af7,0xefd4b422),LL(0x663ba575,0xe4105d02),LL(0x53a69457,0x7eb60a8b),LL(0xc945973b,0x62210008),
   LL(0x77a50ec6,0xfb255478),LL(0x0a37a72c,0xbf0392f7),LL(0x4be18e7a,0xa0a7a19c),LL(0x25b1e0af,0x90d8ea16), LL(0xef953f57,0x7582a293),LL(0xbdc5465a,0x90a64d05),LL(0xe2510717,0xca79c497),LL(0x18cb641f,0x560dbb7c),
   LL(0x4b66abfb,0x1d8e3286),LL(0x59030900,0xd26f52e5),LL(0x5584941a,0x1ee3f643),LL(0x569f5958,0x6d3b3730), LL(0x4789dba5,0x9ff2a62f),LL(0x72b5c9b7,0x91fcb815),LL(0x6c8f9a0e,0xf446cb7d),LL(0x39b7ecb5,0x48f625c1),
   LL(0x1c6219b8,0xbabae801),LL(0x28ac2f23,0xe7a562d9),LL(0x26e20588,0xe1b48732),LL(0x775af051,0x06ee1cad), LL(0xfaff79f7,0xda29ae43),LL(0x652ee9e0,0xc141a412),LL(0x195f4bd0,0x1e127f6f),LL(0x072f34f8,0x29c6ab4f),
   LL(0x30448112,0x7b7c1477),LL(0xe4a38656,0x82b51af1),LL(0x2f315010,0x2bf2028a),LL(0x6ea88cd4,0xc9a4a01f), LL(0x257e5818,0xf63e95d8),LL(0xb4519b16,0xdd8efa10),LL(0x0da910bf,0xed8973e0),LL(0x5c0fe4a9,0xed49d077),
   LL(0xb7caee1e,0xac3aac5e),LL(0xa7f4da57,0x1033898d),LL(0x5c6669b9,0x42145c0e),LL(0xc1aa2aa0,0x42daa688), LL(0x1a1d885a,0x629cc15c),LL(0xf4b76817,0x25572ec0),LL(0x9c8f8f28,0x8312e435),LL(0x81965490,0x8107f8cd),
   LL(0x6fa6110c,0x516ff3a3),LL(0xfb93561f,0x74fb1eb1),LL(0x8457522b,0x6c0c9047),LL(0x6bb8bdc6,0xcfd32104), LL(0xcc80ad57,0x2d6884a2),LL(0x86a9b637,0x7c27fc35),LL(0xadf4e8cd,0x3461baed),LL(0x617242f0,0x1d56251a),
   LL(0xc955bef4,0x0b80d209),LL(0x06adb047,0xdf02cad2),LL(0x5ec74fee,0xf0d7cb91),LL(0x1111ba44,0xd2503375), LL(0xdf53cb36,0x9671755e),LL(0x3368551b,0x54dcb612),LL(0xc8a025a4,0x66d69aac),LL(0xe77ef445,0x6be946c6),
   LL(0xa995e094,0x719946d1),LL(0xe51e04d8,0x65e848f6),LL(0x6a1e3113,0xe62f3300),LL(0x501de503,0x1541c7c1), LL(0xf4acfade,0x4daac9fa),LL(0x44cd0b71,0x0e585897),LL(0x0a51cd77,0x544fd869),LL(0x0031016d,0x60fc20ed),
   LL(0xa4276867,0x58b404ec),LL(0x34f34993,0x46f6c3cc),LL(0xc636e5bd,0x477ca007),LL(0x7c458b47,0x8018f5e5), LL(0xe47b668f,0xa1202270),LL(0xee14f203,0xcef48ccd),LL(0x62ff9b4d,0x23f98bae),LL(0xc589eddd,0x55acc035),
   LL(0x64db4444,0x3fe712af),LL(0xbecdd480,0x19e9d634),LL(0xa930978a,0xe08bc047),LL(0xa1280733,0x2dbf24ec), LL(0x2cd706b2,0x3c0ae38c),LL(0x359017b9,0x5b012a5b),LL(0x72e0f5ae,0x3943c38c),LL(0x57176fa3,0x786167ea),
   LL(0x594881dc,0xe5f9897d),LL(0xcfb820c1,0x6b5efad8),LL(0xd55018de,0xb2179093),LL(0x0bac56ce,0x39ad7d32), LL(0x2cfc0e81,0xb55122e0),LL(0xf6d89daa,0x117c4661),LL(0xcb64fa09,0x362d01e1),LL(0x3e9c4ddd,0x6a309b4e),
   LL(0xabea49b1,0xfa979fb7),LL(0x10e2c6c5,0xb4b1d27d),LL(0x23afde7a,0xbd61c2c4),LL(0x9786d358,0xeb6614f8), LL(0x7f6f7459,0x4a5d816b),LL(0x09360e7b,0xe431a44f),LL(0xc309914c,0x8c27a032),LL(0xcaede3d8,0xcea5d68a),
   LL(0x3a0a3f95,0x3668f665),LL(0x7ceba27b,0x89369416),LL(0xe4728fe9,0x89981fad),LL(0x8a093562,0x7102c8a0), LL(0x235d21c8,0xbb80310e),LL(0xbefb7f7b,0x505e55d1),LL(0x12958a67,0xa0a90811),LL(0x4d851fef,0xd67e106a),
   LL(0x431dd80e,0xb84011a9),LL(0x73306cd9,0xeb7c7cca),LL(0xd1b3b730,0x20fadd29),LL(0xfe37b3d3,0x83858b5b), LL(0xb6251d5c,0xbf4cd193),LL(0x1352d952,0x1cca1fd3),LL(0x90fbc051,0xc66157a4),LL(0x89b98636,0x7990a638),
},
/* digit=12 base_pwr=2^84 */
{
   LL(0x87dec0e1,0xe5aa692a),LL(0xf7b39d00,0x010ded8d),LL(0x54cfa0b5,0x7b1b80c8),LL(0xa0f8ea28,0x66beb876), LL(0x3476cd0e,0x50d7f531),LL(0xb08d3949,0xa63d0e65),LL(0x53479fc6,0x1a09eea9),LL(0xf499e742,0x82ae9891),
   LL(0x5ca7d866,0xab58b910),LL(0x3adb3b34,0x582967e2),LL(0xcceac0bc,0x89ae4447),LL(0x7bf56af5,0x919c667c), LL(0x60f5dcd7,0x9aec17b1),LL(0xddcaadbc,0xec697b9f),LL(0x463467f5,0x0b98f341),LL(0xa967132f,0xb187f1f7),
   LL(0x214aeb18,0x90fe7a1d),LL(0x741432f7,0x1506af3c),LL(0xe591a0c4,0xbb5565f9),LL(0xb44f1bc3,0x10d41a77), LL(0xa84bde96,0xa09d65e4),LL(0xf20a6a1c,0x42f060d8),LL(0xf27f9ce7,0x652a3bfd),LL(0x3b3d739f,0xb6bdb65c),
   LL(0xec7fae9f,0xeb5ddcb6),LL(0xefb66e5a,0x995f2714),LL(0x69445d52,0xdee95d8e),LL(0x09e27620,0x1b6c2d46), LL(0x8129d716,0x32621c31),LL(0x0958c1aa,0xb03909f1),LL(0x1af4af63,0x8c468ef9),LL(0xfba5cdf6,0x162c429f),
   LL(0x753b9371,0x2f682343),LL(0x5f1f9cd7,0x29cab45a),LL(0xb245db96,0x571623ab),LL(0x3fd79999,0xc507db09), LL(0xaf036c32,0x4e2ef652),LL(0x05018e5c,0x86f0cc78),LL(0xab8be350,0xc10a73d4),LL(0x7e826327,0x6519b397),
   LL(0x9c053df7,0xe8cb5eef),LL(0xb300ea6f,0x8de25b37),LL(0xc849cffb,0xdb03fa92),LL(0xe84169bb,0x242e43a7), LL(0xdd6f958e,0xe4fa51f4),LL(0xf4445a8d,0x6925a77f),LL(0xe90d8949,0xe6e72a50),LL(0x2b1f6390,0xc66648e3),
   LL(0x173e460c,0xb2ab1957),LL(0x30704590,0x1bbbce75),LL(0xdb1c7162,0xc0a90dbd),LL(0x15cdd65d,0x505e399e), LL(0x57797ab7,0x68434dcb),LL(0x6a2ca8e8,0x60ad35ba),LL(0xde3336c1,0x4bfdb1e0),LL(0xd8b39015,0xbbef99eb),
   LL(0x1711ebec,0x6c3b96f3),LL(0xce98fdc4,0x2da40f1f),LL(0x57b4411f,0xb99774d3),LL(0x15b65bb6,0x87c8bdf4), LL(0xc2eef12d,0xda3a89e3),LL(0x3c7471f3,0xde95bb9b),LL(0xd812c594,0x600f225b),LL(0x2b75a56b,0x54907c5d),
   LL(0x8db60e35,0xa93cc5f0),LL(0xfa833319,0x743e3cd6),LL(0xf81683c9,0x7dad5c41),LL(0x9c34107e,0x70c1e7d9), LL(0xa6be0907,0x0edc4a39),LL(0x86d0b7d3,0x36d47035),LL(0x272bfa60,0x8c76da03),LL(0x0f08a414,0x0b4a07ea),
   LL(0x45c1dd53,0x699e4d29),LL(0x231debb5,0xcadc5898),LL(0xa77f00e0,0xdf49fcc7),LL(0xa73e5a0e,0x93057bbf), LL(0x027a4cd1,0x2f8b7ecd),LL(0xc614011a,0x114734b3),LL(0x67677c68,0xe7a01db7),LL(0x7e273f4f,0x89d9be5e),
   LL(0x089808ef,0xd225cb2e),LL(0xd59e4107,0xf1f7a27d),LL(0x8211b9c9,0x53afc761),LL(0xe6819159,0x0361bc67), LL(0x7f071426,0x2a865d0b),LL(0xe7072567,0x6a3c1810),LL(0x0d6bcabd,0x3e3bca1e),LL(0x408591bc,0xa1b02bc1),
   LL(0x31fba239,0xe0deee59),LL(0x98bd91d1,0xf47424d3),LL(0x071a3c1d,0x0f8886f4),LL(0xa819233b,0x3f7d41e8), LL(0xcf6eb998,0x708623c2),LL(0x609a287f,0x86bb49af),LL(0x63c90762,0x942bb249),LL(0x55a9654b,0x0ef6eea5),
   LL(0x36f5defe,0x5f6d2d72),LL(0x56f99176,0xfa9922dc),LL(0xf78ce0c7,0x6c8c5ece),LL(0xbe09b55e,0x7b44589d), LL(0x9ea83770,0xe11b3bca),LL(0x2ab71547,0xd7fa2c7f),LL(0x2a1ddcc0,0x2a3dd6fa),LL(0x5a7b7707,0x09acb430),
   LL(0x649d4e57,0x4add4a2e),LL(0x1917526e,0xcd53a2b0),LL(0x20b44ac4,0xc5262330),LL(0xbaa2c31d,0x4028746a), LL(0x64291d4c,0x51318390),LL(0xee5ad909,0xbf48f151),LL(0x7b185681,0xcce57f59),LL(0x4854d442,0x7c3ac1b0),
   LL(0xc093c171,0x65587dc3),LL(0x24f42b65,0xae7acb24),LL(0x955996cb,0x5a338adb),LL(0x6051f91b,0xc8e65675), LL(0x28b8d0b1,0x66711fba),LL(0xb6c10a90,0x15d74137),LL(0x3a232a80,0x70cdd7eb),LL(0x6191ed24,0xc9e2f07f),
   LL(0xf79588c0,0xa80d1db6),LL(0xb55768cc,0xfa52fc69),LL(0x7f54438a,0x0b4df1ae),LL(0xf9b46a4f,0x0cadd1a7), LL(0x1803dd6f,0xb40ea6b3),LL(0x55eaae35,0x488e4fa5),LL(0x382e4e16,0x9f047d55),LL(0x2f6e0c98,0xc9b5b7e0),
   LL(0x95762649,0x6b1bd2d3),LL(0xc7aea3f6,0xa9604ee7),LL(0x6dc6f896,0x3646ff27),LL(0x2860bad1,0x9bf0e7f5), LL(0x7cb44b92,0x2d92c821),LL(0xaea9c182,0xa2f5ce63),LL(0x9154a5fd,0xd0a2afb1),LL(0x95801da6,0x482e474c),
   LL(0xb611c24b,0xc19972d0),LL(0x60a8f351,0x1d468e65),LL(0x7bcf6421,0xeb758069),LL(0x88fbc491,0xec9dd0ee), LL(0x956c2e32,0x5b59d2bf),LL(0xdcddf94e,0x73dc6864),LL(0xbcee7665,0xfd5e2321),LL(0x5e9a06c4,0xa7b4f8ef),
   LL(0x7280f855,0xfba918dd),LL(0x8baec688,0xbbaac260),LL(0x33400f42,0xa3b3f00f),LL(0x66f2e6e4,0x3d2dba29), LL(0x98509375,0xb6f71a94),LL(0xcea423cc,0x8f33031f),LL(0x4807e6fb,0x009b8dd0),LL(0x5cdb954c,0x5163cfe5),
   LL(0xcf41c6e8,0x03cc8f17),LL(0x037b925c,0xf1f03c2a),LL(0x66d2427c,0xc39c19cc),LL(0x7b6c18e4,0x823d24ba), LL(0x901f0b4f,0x32ef9013),LL(0xf8941c2e,0x684360f1),LL(0x2c28092e,0x0ebaff52),LL(0x256c932f,0x7891e4e3),
   LL(0xac445e3d,0x51264319),LL(0x8ea74381,0x553432e7),LL(0x67e9c50a,0xe6eeaa69),LL(0x62e628c7,0x27ced284), LL(0x7a4afa57,0x3f96d375),LL(0xe484c150,0xde0a14c3),LL(0x38bd9923,0x364a24eb),LL(0xe5177422,0x1df18da0),
   LL(0xd8d38a9b,0x174e8f82),LL(0xe7de1391,0x2e97c600),LL(0xa1c175dd,0xc5709850),LL(0x32ae5035,0x969041a0), LL(0x76a2086b,0xcbfd533b),LL(0xd7c2e8fe,0xd6bba71b),LL(0x099dfb67,0xb2d58ee6),LL(0x064a85d9,0x3a8b342d),
   LL(0x522f9be3,0x3bc07649),LL(0xdf1f49a8,0x690c075b),LL(0x3854ec42,0x80e1aee8),LL(0x17689dc7,0x2a7dbf44), LL(0x3faf4078,0xc004fc0e),LL(0xdf11862c,0xb2f02e9e),LL(0xa0a1b7b3,0xf10a5e0f),LL(0x8936ec80,0x30aca623),
   LL(0x02f40d9a,0xf83cbf05),LL(0x2c318a4d,0x4681c468),LL(0x0e9c2674,0x98575618),LL(0x1847092e,0xbe79d046), LL(0x78bd01e0,0xaf1e480a),LL(0x72a51db9,0x6dd359e4),LL(0xe3afbab6,0x62ce3821),LL(0x17733199,0xc5cee5b6),
   LL(0x6ffd9fbb,0xe08b30d4),LL(0x36c610b7,0x6e5bc699),LL(0x9ce262cf,0xf343cff2),LL(0x68b914c1,0xca2e4e35), LL(0x16de36c5,0x011d64c0),LL(0x42e2b829,0xe0b10fdd),LL(0x6685aaf8,0x78942981),LL(0x230ede97,0xe7511708),
   LL(0x3b922bf8,0x671ed8fc),LL(0x4c29b133,0xe4d8c0a0),LL(0x3b6e99c4,0x87eb1239),LL(0x8793beba,0xaff3974c), LL(0x2c18df9b,0x03749405),LL(0x91007139,0xc5c3a293),LL(0xe37a0b95,0x6a77234f),LL(0xb661c96b,0x02c29a21),
   LL(0x141ecf61,0xc3aaf1d6),LL(0x3bb22f53,0x9195509e),LL(0x22d51357,0x29597404),LL(0x537bed60,0x1b083822), LL(0xe07289f0,0xcd7d6e35),LL(0x6dd86eff,0x1f94c48c),LL(0xeb0f9cfa,0xc8bb1f82),LL(0x1b2eb97d,0x9ee0b7e6),
   LL(0x34d74e31,0x5a52fe2e),LL(0x3bf79ab6,0xa352c310),LL(0xabfeeb8f,0x97ff6c5a),LL(0xf5c97305,0xbfbe8fef), LL(0xa7904608,0xd6081ce6),LL(0xc4fca249,0x1f812f3a),LL(0xb9e5e200,0x9b24bc9a),LL(0x38012ee8,0x91022c67),
   LL(0x30a713a1,0xe83d9c5d),LL(0x84ef0f93,0x4876e3f0),LL(0xc1fbf928,0xc9777029),LL(0xbce7d2a4,0xef7a6bb3), LL(0xdfa2a659,0xb8067228),LL(0xd877a48f,0xd5cd3398),LL(0x025d0f3f,0xbea4fd8f),LL(0x2eae7c2b,0xd67d2e35),
   LL(0xcc5f4394,0x184de7d7),LL(0x4536e142,0xb5551b5c),LL(0xd34aa60a,0x2e89b212),LL(0xf50051d5,0x14a96fea), LL(0x0d12bb0b,0x4e21ef74),LL(0x60b9677e,0xc522f020),LL(0x2df7731d,0x8b12e467),LL(0x7b326d31,0x39f80382),
   LL(0x39024a94,0xdfb8630c),LL(0x97319452,0xaacb96a8),LL(0xeda3867c,0xd68a3961),LL(0x77c4ffca,0x0c58e2b0), LL(0x4da919fa,0x3d545d63),LL(0xf15e2289,0xef79b69a),LL(0x808bab10,0x54bc3d3d),LL(0x45f82c37,0xc8ab3007),
   LL(0x7c4a658a,0xc12738b6),LL(0x40e72182,0xb3c47639),LL(0x8798e44f,0x3b77be46),LL(0x17a7f85f,0xdc047df2), LL(0x5e59d92d,0x2439d4c5),LL(0xe8e64d8d,0xcedca475),LL(0x87ca9b16,0xa724cd0d),LL(0xa5540dfe,0x35e4fd59),
   LL(0xe4bcf6b1,0xf8c1ff18),LL(0x295018fa,0x856d6285),LL(0x3263c949,0x433f665c),LL(0xa1f21409,0xa6a76dd6), LL(0xcc7b4f79,0x17d32334),LL(0x06720e4a,0xa1d03122),LL(0x81d9bed5,0xadb6661d),LL(0x11db15d1,0xf0d6fb02),
   LL(0x1fb747d2,0x7fd11ad5),LL(0x3033762b,0xab50f959),LL(0xfbefaf5a,0x2a7e711b),LL(0x3fef2bbf,0xc7393278), LL(0x0df6f9be,0xe29fa244),LL(0x71efd215,0x9092757b),LL(0x4f3d6fd9,0xee60e311),LL(0x0acfb78b,0x338542d4),
   LL(0x38961a0f,0x44a23f08),LL(0x986987ca,0x1426eade),LL(0x4a863cc6,0x36e6ee2e),LL(0x628b8b79,0x48059420), LL(0x7396e1de,0x30303ad8),LL(0x38c5aad1,0x5c8bdc48),LL(0x5c8f5066,0x3e40e11f),LL(0x8d246bbd,0xabd6e768),
   LL(0x23330a01,0x68aa40bb),LL(0xc34eafa0,0xd23f5ee4),LL(0x5de02c21,0x3bbee315),LL(0xd1d8dd06,0x18dd4397), LL(0x122d7b44,0x3ba1939a),LL(0xa33870d6,0xe6d3b40a),LL(0x1c4fe3f8,0x8e620f70),LL(0xd3a50cbf,0xf6bba1a5),
   LL(0xcfc0aee0,0x4a78bde5),LL(0xc08c50bd,0x847edc46),LL(0xad63c9b2,0xbaa2439c),LL(0x10fc2acb,0xceb4a728), LL(0x26da033d,0xa419e40e),LL(0x03e02683,0x6cc3889d),LL(0xfdccf725,0x1cd28559),LL(0x8d13d208,0x0fd7e0f1),
   LL(0x1f0df9d4,0x01b9733b),LL(0xa2b5e4f3,0x8cc2c5f3),LL(0x3a304fd4,0x43053bfa),LL(0x0a9f1aa7,0x8e87665c), LL(0xd73dc965,0x087f29ec),LL(0x3e9023db,0x15ace455),LL(0x2bce28b4,0x2370e309),LL(0xb6b1e84a,0xf9723442),
   LL(0xb72d9f26,0xbeee662e),LL(0xf0e47109,0xb19396de),LL(0xe13289d0,0x85b1fa73),LL(0x54e58e32,0x436cf77e), LL(0xe990ef77,0x0ec833b3),LL(0x1b11fc25,0x7373e3ed),LL(0x0fc332ce,0xbe0eda87),LL(0x8d7ea856,0xced04970),
   LL(0x7e977ca0,0xf85ff785),LL(0xdfdd5d2b,0xb66ee8da),LL(0x905af461,0xf5e37950),LL(0x966d487c,0x587b9090), LL(0x32ba0127,0x6a198a1b),LL(0x141615ac,0xa7720e07),LL(0x996ef2f2,0xa23f3499),LL(0x470bcb3d,0xef5f64b4),
   LL(0x92b8c559,0xa526a962),LL(0x69740a0f,0x0c14aac0),LL(0xa6bdc0a5,0x0d41a9e3),LL(0x9c48aef4,0x97d52106), LL(0x3e7c253b,0xcf16bd30),LL(0x47fdedc1,0xcc834b1a),LL(0x373aab2e,0x7362c6e5),LL(0xc5f590ff,0x264ed85e),
   LL(0x66d41870,0x7a46d9c0),LL(0x4787ba09,0xa50c20b1),LL(0xe3d44635,0x185e7e51),LL(0x31e2d8dc,0xb3b3e080), LL(0xa179e9d9,0xbed1e558),LL(0x74a76781,0x2daa3f79),LL(0x3a40864f,0x4372baf2),LL(0x4fe75cb5,0x46900c54),
   LL(0xf76765d0,0xb95f171e),LL(0x95c87502,0x4ad726d2),LL(0x4d7c99bd,0x2ec769da),LL(0xc36cdfa8,0x5e2ddd19), LL(0xa93e6dea,0xc22117fc),LL(0x93771123,0xe8a2583b),LL(0xfa08a3a2,0xbe2f6089),LL(0x8f0e1112,0x4809d5ed),
   LL(0xda7a095e,0x3b414aa3),LL(0x26f5aadd,0x9049acf1),LL(0x6be8b84a,0x78d46a4d),LL(0xb732b9b3,0xd66b1963), LL(0xde6e9555,0x5c2ac2a0),LL(0xb5bd8770,0xcf52d098),LL(0x0fd28921,0x15a15fa6),LL(0x8b27536d,0x56ccb81e),
   LL(0x9f4ccbb8,0x0f0d8ab8),LL(0xdb221729,0xed5f44d2),LL(0x00bed10c,0x43141988),LL(0x1d735b8b,0xc94348a4), LL(0x29ef8479,0x79f3e9c4),LL(0x614c693f,0x4c13a4e3),LL(0x8e143a14,0x32c9af56),LL(0xe29ac5c4,0xbc517799),
   LL(0x2774856f,0x05e17992),LL(0x6c1bf55f,0x6e52fb05),LL(0xe4f19e16,0xaeda4225),LL(0xaf5ccb26,0x70f4728a), LL(0xb2947f22,0x5d2118d1),LL(0x281d6fb9,0xc827ea16),LL(0x8cf0eabd,0x8412328d),LL(0x03ef9dcf,0x45ee9fb2),
   LL(0xbb937d63,0x8e700421),LL(0xcc4b37a6,0xdf8ff2d5),LL(0x5ced7b68,0xa4c0d5b2),LL(0xc7308f59,0x6537c1ef), LL(0x3b37f8e8,0x25ce6a26),LL(0xdeebc6ce,0x170e9a9b),LL(0x8728d72c,0xdd037952),LL(0x850154bc,0x445b0e55),
   LL(0x83a7337b,0x4b7d0e06),LL(0xffecf249,0x1e3416d4),LL(0x66a2b71f,0x24840eff),LL(0xb37cc26d,0xd0d9a50a), LL(0x6fe28ef7,0xe2198150),LL(0x23324c7f,0x3cc5ef16),LL(0x769b5263,0x220f3455),LL(0xa10bf475,0xe2ade2f1),
   LL(0x458d3671,0x28cd20fa),LL(0x2dc4847b,0x1549722c),LL(0x591941e3,0x6dd01e55),LL(0x27128ccb,0x0e6fbcea), LL(0x3bef0262,0xae1a1e6b),LL(0x8f54e103,0xfa8c472c),LL(0x72c052ec,0x7539c0a8),LL(0x5a3490e9,0xd7b27369),
   LL(0x71684349,0x143fe1f1),LL(0x32e19b97,0x36b4722e),LL(0x90980aff,0xdc059227),LL(0x9e13d674,0x175c9c88), LL(0x6e6bfdb1,0xa7de5b22),LL(0xbedb4b46,0x5ea5b7b2),LL(0xd34a6e44,0xd5570191),LL(0xa24ff7e6,0xfcf60d2e),
   LL(0x677819e1,0x614a392d),LL(0xaa5a29e8,0x7be74c7e),LL(0x63c85f3f,0xab50fece),LL(0x46cab337,0xaca2e2a9), LL(0x122a6fe3,0x7f700388),LL(0x882a04a8,0xdb69f703),LL(0xcf7aed57,0x9a77935d),LL(0x8d91c86f,0xdf16207c),
   LL(0x63ed9998,0x2fca49ab),LL(0xa77ddf96,0xa3125c44),LL(0x24344072,0x05dd8a86),LL(0xfec3fb56,0xa023dda2), LL(0x0c743032,0x421b41fc),LL(0x5e438639,0x4f2120c1),LL(0xc83c1b07,0xfb7cae51),LL(0xcac2171a,0xb2370caa),
   LL(0x6cc820fb,0x2eb2d962),LL(0xb85a44bf,0x59feee5c),LL(0x5b6598f0,0x94620fca),LL(0x7e314051,0x6b922cae), LL(0x106bed4e,0xff8745ad),LL(0xdfa1e9ab,0x546e71f5),LL(0x1ec29487,0x935c1e48),LL(0x4d936530,0x9509216c),
   LL(0x85c9a2db,0xc7ca3067),LL(0x6be8606f,0xd6ae5152),LL(0xe14c651d,0x09dbcae6),LL(0x9bc32f96,0xc9536e23), LL(0x34521b03,0xa90535a9),LL(0x878756ff,0xf39c526c),LL(0x8aedf03c,0x383172ec),LL(0xefe0c034,0x20a8075e),
   LL(0x64026422,0xf22f9c62),LL(0x24b9d076,0x8dd10780),LL(0x3bef2950,0x944c742a),LL(0x88a2b00b,0x55b9502e), LL(0x86a09817,0xa59e14b4),LL(0x47bb4071,0xa39dd3ac),LL(0x3be0592f,0x55137f66),LL(0xc9e63f5b,0x07fcafd4),
   LL(0x346eb226,0x963652ee),LL(0xec2facb7,0x7dfab085),LL(0x691add26,0x273bf2b8),LL(0xf2b46c44,0x30d74540), LL(0xf2c2d065,0x05e8e73e),LL(0xd42eeac9,0xff9b8a00),LL(0x97209d22,0x2fcbd205),LL(0xde14ea2c,0xeb740ffa),
   LL(0xa8aef518,0xc71ff913),LL(0xfff4cfa2,0x7bfc74bb),LL(0xb6b36048,0x1716680c),LL(0x9ef79af1,0x121b2cce), LL(0xa01eb3d3,0xbff3c836),LL(0x5f79077b,0x50eb1c6a),LL(0xa004bbcf,0xa48c32d6),LL(0x7d64f61d,0x47a59316),
   LL(0x93102016,0x6068147f),LL(0x94d12576,0x12c5f654),LL(0xc9bc6b91,0xefb071a7),LL(0x6e23ea95,0x7c2da0c5), LL(0xd4a1dd5d,0xf4fd45b6),LL(0x9122b13c,0x3e7ad9b6),LL(0xe6f57a48,0x342ca118),LL(0x06f8288f,0x1c2e94a7),
   LL(0x5a97d231,0x99e68f07),LL(0x4d838758,0x7c80de97),LL(0x05872727,0xbce0f5d0),LL(0x19c4d016,0xbe5d95c2), LL(0x9c2492ee,0x921d5cb1),LL(0x404d6fb3,0x42192dc1),LL(0x32f988d3,0x4c84dcd1),LL(0xa17b8e85,0xde26d61f),
   LL(0x137c7408,0xc466dcb6),LL(0x36a266da,0x9a38d7b6),LL(0x83bebf1b,0x7ef5cb06),LL(0x0fd014e3,0xe5cdcbbf), LL(0xf65965a0,0x30aa376d),LL(0xebb3e95e,0x60fe88c2),LL(0x66ee6f20,0x33fd0b61),LL(0x3f41f0a0,0x8827dcdb),
   LL(0x0c56c690,0xbf8a9d24),LL(0xddb7641d,0x40265dad),LL(0x3a6b662b,0x522b05bf),LL(0xb1478c9b,0x466d1dfe), LL(0x1484469b,0xaa616962),LL(0x02df8f9f,0x0db60549),LL(0x3cb8bf51,0xc37bca02),LL(0x21371ce8,0x5effe346),
   LL(0xff112c32,0xe8f65264),LL(0x7b971fb2,0x8a9c736d),LL(0x7b75080d,0xa4f19470),LL(0x8839c59b,0xfc3f2c5a), LL(0x5aeb49c2,0x1d6c777e),LL(0xda1addfe,0xf3db034d),LL(0x5535affc,0xd76fee5a),LL(0xb92251fd,0x0853ac70),
   LL(0x8b2a29d5,0x37e3d594),LL(0x4de00ddb,0x28f1f457),LL(0xf42c328b,0x8083c1b5),LL(0xe493c73b,0xd8ef1d8f), LL(0x41dc61bd,0x96fb6260),LL(0x27ee2f8a,0xf74e8a9d),LL(0x2c946a5d,0x7c605a80),LL(0x3839ccfd,0xeed48d65),
   LL(0x3a29467a,0x9894344f),LL(0xc51eba6d,0xde81e949),LL(0xa5e5c2f2,0xdaea066b),LL(0x08c8c7b3,0x3fc8a614), LL(0x06d0de9f,0x7adff88f),LL(0x3b75ce0a,0xbbc11cf5),LL(0xfbbc87d5,0x9fbb7acc),LL(0x7badfde2,0xa1458e26),
},
/* digit=13 base_pwr=2^91 */
{
   LL(0xe039c256,0x1cb43668),LL(0x7c17fd5d,0x5f26fb8b),LL(0x79aa062b,0xeee426af),LL(0xd78fbf04,0x072002d0), LL(0xe84fb7e3,0x4c9ca237),LL(0x0c82133d,0xb401d8a1),LL(0x6d7e4181,0xaaa52592),LL(0x73dbb152,0xe9430833),
   LL(0xbe24319a,0xf92dda31),LL(0xe095a8e7,0x03f7d28b),LL(0x98782185,0xa52fe840),LL(0x29c24dbc,0x276ddafe), LL(0x1d7a64eb,0x80cd5496),LL(0x7f1dbe42,0xe4360889),LL(0x8438d2d5,0x2f81a877),LL(0x85169036,0x7e4d52a8),
   LL(0x1d59715d,0x19e3d5b1),LL(0xd788983e,0xc7eaa762),LL(0xabf1f248,0xe5a730b0),LL(0xfae3fd83,0xfbab8084), LL(0x53765b2f,0x65e50d21),LL(0xfa127f3d,0xbdd4e083),LL(0x397b1b10,0x9cf3c074),LL(0xb1b59fd3,0x59f8090c),
   LL(0x615faa8f,0x7b15fd9d),LL(0x968554ed,0x8fa1eb40),LL(0x7aa44882,0x7bb4447e),LL(0x029fff32,0x2bb2d0d1), LL(0x6caa6d2f,0x075e2a64),LL(0x22e7351b,0x8eb879de),LL(0x9a506c62,0xbcd5624e),LL(0xa87e24dc,0x218eaef0),
   LL(0x44ddfa35,0x37e56847),LL(0xdab3f747,0x9ccfc5c5),LL(0x1ee96cf4,0x9ac1df3f),LL(0x3b480b8f,0x0c0571a1), LL(0x4b3a7b3c,0x2fbeb3d5),LL(0x5dcdbb99,0x35c03669),LL(0xb2415b3a,0x52a0f5dc),LL(0x4413ed9a,0xd57759b4),
   LL(0x3d30a2c5,0x1fe647d8),LL(0xf78a81dc,0x0857f77e),LL(0x131a4a9b,0x11d5a334),LL(0x29d393f5,0xc0a94af9), LL(0xdaa6ec1a,0xbc3a5c0b),LL(0x88d2d7ed,0xba9fe493),LL(0xbb614797,0xbb4335b4),LL(0x72f83533,0x991c4d68),
   LL(0xd2f01cb3,0x53258c28),LL(0xd75db0b1,0x93d6eaa3),LL(0xe87d0db4,0x419a2b0d),LL(0xd8fe8493,0xa1e48f03), LL(0xc508b23a,0xf747faf6),LL(0x35d53549,0xf137571a),LL(0xfcf9b838,0x9f5e58e2),LL(0xa7fd3cf5,0xc7186cee),
   LL(0xe978a1d3,0x77b868ce),LL(0x7ab92d04,0xe3a68b33),LL(0x87a5b862,0x51029794),LL(0x3a61d41d,0x5f0606c3), LL(0x6f9326f1,0x2814be27),LL(0xc6fe3c2e,0x2f521c14),LL(0xacdf7351,0x17464d7d),LL(0x777f7e44,0x10f5f9d3),
   LL(0x269fb37d,0xce8e616b),LL(0x7de62de5,0xaaf73804),LL(0x4fdd4153,0xaba11175),LL(0x3770b49b,0x515759ba), LL(0xaa423a61,0x8b09ebf8),LL(0xcd41fb92,0x592245a1),LL(0x9b4c8936,0x1cba8ec1),LL(0xaf36710e,0xa87e91e3),
   LL(0x3d34a2e3,0x1fd84ce4),LL(0xb43b5d61,0xee3759ce),LL(0x619186c7,0x895bc78c),LL(0xcbb9725a,0xf19c3809), LL(0xde744b1f,0xc0be21aa),LL(0x60f8056b,0xa7d222b0),LL(0xb23efe11,0x74be6157),LL(0x0cd68253,0x6fab2b4f),
   LL(0x4bf1d725,0xad33ea5f),LL(0x4f6c950f,0x9c1d8ee2),LL(0xa377af06,0x544ee78a),LL(0x94a113e1,0x54f489bb), LL(0x992fb7e8,0x8f11d634),LL(0xa2a44347,0x0169a7aa),LL(0x95020e00,0x1d49d4af),LL(0xe08e120b,0x95945722),
   LL(0xa4d32282,0xb6e33878),LL(0x48020ae7,0xe36e029d),LL(0x37a9b750,0xe05847fb),LL(0xb29e3819,0xf876812c), LL(0xd23a17f0,0x84ad138e),LL(0xf0b3950e,0x6d7b4480),LL(0x2fd67ae0,0xdfa8aef4),LL(0x52333af6,0x8d3eea24),
   LL(0xb15d5acc,0x0d052075),LL(0xbd815bc4,0xc6d9c79f),LL(0xdfa36cf2,0x8dcafd88),LL(0x38aa9070,0x908ccbe2), LL(0xba35afce,0x638722c4),LL(0xfd6abf0b,0x5a3da8b0),LL(0xc9c335c1,0x2dce252c),LL(0x65aa799b,0x84e7f0de),
   LL(0xb99a72cb,0x2101a522),LL(0x87618016,0x06de6e67),LL(0xe6f3653e,0x5ff8c7cd),LL(0xc7a6754a,0x0a821ab5), LL(0x7cb0b5a2,0x7e3fa52b),LL(0xc9048790,0xa7fb121c),LL(0x06ce053a,0x1a725020),LL(0x04e929b0,0xb490a31f),
   LL(0x62dd61ad,0xe17be47d),LL(0x6be01371,0x781a961c),LL(0xdae3cbba,0x1063bfd3),LL(0x7f73c9ba,0x35647406), LL(0x2736a129,0xf50e957b),LL(0xed13f256,0xa6313702),LL(0x3a19fcc5,0x9436ee65),LL(0xe7a4c8b6,0xcf2bdb29),
   LL(0xc5f95cd8,0xb06b1244),LL(0xf4ab95f4,0xda8c8af0),LL(0xb9e5836d,0x1bae59c2),LL(0x3acffffc,0x07d51e7e), LL(0xc2ccbcda,0x01e15e6a),LL(0x8528c3e0,0x3bc1923f),LL(0xa49fead4,0x43324577),LL(0x2aa7a711,0x61a1b884),
   LL(0x700230ef,0xf9a86e08),LL(0xbd19adf8,0x0af585a1),LL(0xf55ad8f2,0x7645f361),LL(0x46c3614c,0x6e676223), LL(0x4e774d3f,0x23cb257c),LL(0xac102d1b,0x82a38513),LL(0x7b126aa5,0x9bcddd88),LL(0xeefd3ee4,0xe716998b),
   LL(0xfb167583,0x4239d571),LL(0xd16c8f8a,0xdd011c78),LL(0x69a27519,0x271c2895),LL(0xd2d64b6a,0x9ce0a3b7), LL(0xd5ec6738,0x8c977289),LL(0x8840ef6b,0xa3b49f9a),LL(0x9a453419,0x808c14c9),LL(0x0cf0a2d5,0x5c00295b),
   LL(0x1d4bcc76,0x524414fb),LL(0x459a88f1,0xb07691d2),LL(0xf70d110f,0x77f43263),LL(0xb7abf9f3,0x64ada5e0), LL(0x5b544cf5,0xafd0f94e),LL(0xfd2713fe,0xb4a13a15),LL(0x250c74f4,0xb99b7d6e),LL(0x20324e45,0x097f2f73),
   LL(0xaffa8208,0x994b37d8),LL(0xdc29aafc,0xc3c31b0b),LL(0x7a3a607f,0x3da74651),LL(0xfe6955d6,0xd8e1b8c1), LL(0xc8418682,0x716e1815),LL(0x7dc91d97,0x541d487f),LL(0xc6996982,0x48a04669),LL(0x83a6502e,0xf39cab15),
   LL(0xe68db055,0x025801a0),LL(0xba3338d5,0xf3569758),LL(0xee2afa84,0xb0c8c0aa),LL(0xfb6562d1,0x4f6985d3), LL(0x132ed17a,0x351f1f15),LL(0xc04365fe,0x510ed0b4),LL(0xe5b1f066,0xa3f98138),LL(0x32df03dc,0xbc9d95d6),
   LL(0x19abd09e,0xa83ccf6e),LL(0x4ff17edb,0x0b4097c1),LL(0xd64a06ce,0x58a5c478),LL(0x544a58fd,0x2ddcc3fd), LL(0x9e8153b8,0xd449503d),LL(0x7774179b,0x3324fd02),LL(0xdbd9120c,0xaf5d47c8),LL(0x34fa94db,0xeb860162),
   LL(0x972f07f4,0x5817bdd1),LL(0xd27bbceb,0xe5579e2e),LL(0x5f11e5a6,0x86847a1f),LL(0x7c3cf048,0xb39ed255), LL(0xa2f62e55,0xe1076417),LL(0x1bcf82a2,0x6b9ab38f),LL(0x7aeb29f9,0x4bb7c319),LL(0x17227a46,0xf6d17da3),
   LL(0x0f968c00,0xab53ddbd),LL(0x000c880b,0xa03da7ec),LL(0x6a9ad24d,0x7b239624),LL(0x01ec60d0,0x612c0401), LL(0x109f5df1,0x70d10493),LL(0x80af7550,0xfbda4030),LL(0xc6b9a9b3,0x30b93f95),LL(0x007d9418,0x0c74ec71),
   LL(0x6edb951f,0x94175564),LL(0x7f22c282,0x5f4a9d78),LL(0xb38d1196,0xb7870895),LL(0xa228ce7c,0xbc593df3), LL(0x6af3641a,0xc78c5bd4),LL(0x3d9b3dcc,0x7802200b),LL(0x8be33304,0x0dc73f32),LL(0x61ffb79a,0x847ed87d),
   LL(0x6d671192,0xf85c974e),LL(0xde16f60f,0x1e14100a),LL(0x95c38797,0x45cb0d5a),LL(0x9b022da4,0x18923bba), LL(0xbbe7e86e,0xef2be899),LL(0x216067bf,0x4a1510ee),LL(0x84d5ce3e,0xd98c8154),LL(0xf92a2b90,0x1af777f0),
   LL(0x4ef65724,0x9fbcb400),LL(0x3c0ca6fe,0x3e04a4c9),LL(0x55002994,0xfb3e2cb5),LL(0x5363ecab,0x1f3a93c5), LL(0x3923555b,0x1fe00efe),LL(0x1e1751ea,0x744bedd9),LL(0x6ab69357,0x3fb2db59),LL(0xf5e6618b,0x8dbd7365),
   LL(0xdf1ea40e,0x99d53099),LL(0x57d61e64,0xb3f24a0b),LL(0x596eb812,0xd088a198),LL(0x5762940b,0x22c8361b), LL(0xf9c0d95c,0x66f01f97),LL(0x8e43cdae,0x88461172),LL(0xb72b15c3,0x11599a7f),LL(0x420d95cc,0x135a7536),
   LL(0x5f7ae2f6,0x2dcdf0f7),LL(0xd7fa6da2,0x15fc6e1d),LL(0xd1d441b6,0x81ca829a),LL(0x04a106b6,0x84c10cf8), LL(0xa73fbbd0,0xa9b26c95),LL(0x4d8f6ee8,0x7f24e0cb),LL(0x1e25a043,0x48b45937),LL(0x036f3dfe,0xf8a74fca),
   LL(0xc9f84296,0x1ed46585),LL(0x3bc278b0,0x7fbaa8fb),LL(0x6c4fcbd0,0xa8e96cd4),LL(0x73b60a5f,0x940a1202), LL(0x55a4aec8,0x34aae120),LL(0xdbd742f0,0x550e9a74),LL(0x228c68ab,0x794456d7),LL(0xa4e25ec6,0x492f8868),
   LL(0xb2d8f398,0x682915ad),LL(0x5b84c953,0xf13b51cc),LL(0x5bb917d6,0xcda90ab8),LL(0x4ea3dee1,0x4b615560), LL(0x0a52c1c8,0x578b4e85),LL(0x20b75fc4,0xeab1a695),LL(0xaa0bb3c6,0x60c14f3c),LL(0xb8216094,0x220f448a),
   LL(0xb0e63d34,0x4fe7ee31),LL(0xa9e54fab,0xf4600572),LL(0xd5e7b5a4,0xc0493334),LL(0x06d54831,0x8589fb92), LL(0x6583553a,0xaa70f5cc),LL(0xe25649e5,0x0879094a),LL(0x10044652,0xcc904507),LL(0x02541c4f,0xebb0696d),
   LL(0xb9718710,0x5a171fde),LL(0xf374a9f5,0x38f1bed8),LL(0xba39bdc1,0xc8c582e1),LL(0x908cc0ce,0xfc457b0a), LL(0x883841e2,0x9a187fd4),LL(0x38725381,0x8ec25b39),LL(0x96f84395,0x2553ed05),LL(0x6f6c6897,0x095c7661),
   LL(0x4bdc5610,0x917ac85c),LL(0x179eb301,0xb2885fe4),LL(0x8b78bdcc,0x5fc65547),LL(0xe59e4699,0x4a9fc893), LL(0x3ce299af,0xbb7ff0cd),LL(0xadf38b20,0x195be9b3),LL(0xd38ddb8f,0x6a929c87),LL(0xb21a51b9,0x55fcc99c),
   LL(0x721a4593,0x2b695b4c),LL(0x768eaac2,0xed1e9a15),LL(0x7489f914,0xfb63d71c),LL(0x78118910,0xf98ba31c), LL(0x9b128eb4,0x80291373),LL(0xd448af4a,0x7801214e),LL(0x55418dd3,0xdbd2e22b),LL(0xd3998242,0xeffb3c0d),
   LL(0xc7bf3827,0xdfa6077c),LL(0x47f8238f,0xf2165bcb),LL(0x8564d554,0xfe37cf68),LL(0x0a81fb98,0xe5f825c4), LL(0xffed4d6f,0x43cc4f67),LL(0xb50a34b0,0xbc609578),LL(0x5041faf1,0x8aa8fcf9),LL(0x651773b6,0x5659f053),
   LL(0x6044d63b,0xe87582c3),LL(0x0cdb0ca0,0xa6089409),LL(0xbfb2bcf6,0x8c993e0f),LL(0x45985cfc,0xfc64a719), LL(0x83dbedba,0x15c4da80),LL(0x2be67df7,0x804ae112),LL(0xa23defde,0xda4c9658),LL(0x5156e0d3,0x12002ddd),
   LL(0x5dd21b96,0xe68eae89),LL(0xcf44624d,0x8b99f28b),LL(0x1ec8897a,0x0ae00808),LL(0x6712f76e,0xdd0a9303), LL(0x4e233de4,0x96237522),LL(0x2b36a8a5,0x192445b1),LL(0x023993d9,0xabf9ff74),LL(0x2aad4a8f,0x21f37bf4),
   LL(0xf8bd2bbd,0x340a4349),LL(0x4868195d,0x1d902cd9),LL(0xe5fdb6f1,0x3d27bbf1),LL(0x124f9f1c,0x7a5ab088), LL(0xf7a09e03,0xc466ab06),LL(0x31f2c123,0x2f8a1977),LL(0x041b6657,0xda355dc7),LL(0x8ece2a7c,0xcb840d12),
   LL(0x7db32675,0xb600ad9f),LL(0x07a06f1b,0x78fea133),LL(0xb31f6094,0x5d032269),LL(0x83ec37aa,0x07753ef5), LL(0x9c0bea78,0x03485aed),LL(0xbc3f4524,0x41bb3989),LL(0x697f726d,0x09403761),LL(0xdf394820,0x6109beb3),
   LL(0x3b6d1145,0x804111ea),LL(0xa8582654,0xb6271ea9),LL(0x24e66562,0x619615e6),LL(0xd7b6ad9c,0xa2554945), LL(0x99bfe35f,0xd9c4985e),LL(0x7b51cdf6,0x9770ccc0),LL(0x92881832,0x7c327013),LL(0x286b26d1,0x8777d45f),
   LL(0xd847999d,0x9bbeda22),LL(0xc3525d32,0x03aa33b6),LL(0x28a959a1,0x4b7b96d4),LL(0x31e5d234,0xbb3786e5), LL(0x6961f247,0xaeb5d3ce),LL(0x02f93d3f,0x20aa85af),LL(0xd7a7ae4f,0x9cd1ad3d),LL(0x781adaa8,0xbf6688f0),
   LL(0x7469cead,0xb1b40e86),LL(0x309fca48,0x1904c524),LL(0x4b54bbc7,0x9b7312af),LL(0x593affa2,0xbe24bf8f), LL(0xbd98764b,0xbe5e0790),LL(0xa26e299e,0xa0f45f17),LL(0x6b8fe4c7,0x4af0d2c2),LL(0x8ae8a3e6,0xef170db1),
   LL(0x29e0ccc1,0x0e8d61a0),LL(0x60ad36ca,0xcd53e87e),LL(0xc8173822,0x328c6623),LL(0xa496be55,0x7ee1767d), LL(0x648945af,0x89f13259),LL(0x25c8009c,0x9e45a5fd),LL(0x1f61ab8c,0xaf2febd9),LL(0x8a275385,0x43f6bc86),
   LL(0xf2142e79,0x87792348),LL(0xc6e6238a,0x17d89259),LL(0x4a839d9b,0x7536d2f6),LL(0x76a1fbdc,0x1f428fce), LL(0x0db06dfe,0x1c109601),LL(0x50a3a3cc,0xbfc16bc1),LL(0x9b30f41b,0xf9cbd9ec),LL(0x00138cce,0x5b5da0d6),
   LL(0x56ef96a7,0xec1d0a48),LL(0x982bf842,0xb47eb848),LL(0xec3f700d,0x66deae32),LL(0xaa1181e0,0x4e43c42c), LL(0xd1a4aa2a,0xa1d72a31),LL(0xc004f3ce,0x440d4668),LL(0x45fe8a7a,0x0d6a2d3b),LL(0xfb128365,0x820e52e2),
   LL(0x25e51b09,0x29ac5fcf),LL(0x2023d159,0x180cd2bf),LL(0xa1ebf90e,0xa9892171),LL(0x7c132181,0xf97c4c87), LL(0xc03dbb7e,0x9f1dc724),LL(0x018cbbe4,0xae043765),LL(0x0767d153,0xfb0b2a36),LL(0x249cbaeb,0xa8e2f4d6),
   LL(0xd95ea168,0x172a5247),LL(0x2970764a,0x1758fada),LL(0x1d978169,0xac803a51),LL(0xde77e01b,0x299cfe2e), LL(0xb0a98927,0x652a1e17),LL(0x20014495,0x2e26e1d1),LL(0x7175b56a,0x7ae0af9f),LL(0xd64b9f95,0xc2e22a80),
   LL(0xd90a060a,0x4d0ff9fb),LL(0xbaf38085,0x496a27db),LL(0xda776bcf,0x32305401),LL(0x725f209e,0xb8cdcef6), LL(0x436a0bba,0x61ba0f37),LL(0x76860049,0x263fa108),LL(0xda3542cf,0x92beb98e),LL(0xd5849538,0xa2d4d14a),
   LL(0x12e9a1bc,0x989b9d68),LL(0x5f6e3268,0x61d9075c),LL(0x99ace638,0x352c6aa9),LL(0x920f43ff,0xde4e4a55), LL(0xd673c017,0xe5e4144a),LL(0x6f6e05ea,0x667417ae),LL(0xdcd1bd56,0x613416ae),LL(0x86693711,0x5eb36201),
   LL(0x3a1aa914,0x2d7bc504),LL(0x76dc5975,0x175a1299),LL(0x3fc8125c,0xe900e0f2),LL(0x11198875,0x569ef68c), LL(0x63a113b4,0x9012db63),LL(0x98835766,0xe3bd3f56),LL(0x76412dea,0xa5c94a52),LL(0xaa735e5c,0xad9e2a09),
   LL(0x508b65e9,0x405a984c),LL(0x6df1a0d1,0xbde4a1d1),LL(0xdfba80da,0x1a9433a1),LL(0x9440ad2e,0xe9192ff9), LL(0x5099fe92,0x9f649696),LL(0x0b27a54a,0x25ddb65c),LL(0xc590da61,0x178279dd),LL(0xfbde681a,0x5479a999),
   LL(0x013fe162,0xd0e84e05),LL(0x632d471b,0xbe11dc92),LL(0xfc0e089f,0xdf0b0c45),LL(0x4c144025,0x04fb15b0), LL(0x13c99927,0xa61d5fc2),LL(0x3de2eb35,0xa033e9e0),LL(0xb8dacbb4,0xf8185d5c),LL(0x8644549d,0x9a88e265),
   LL(0x54671ff6,0xf717af62),LL(0x5fa58603,0x4bd4241b),LL(0xe67773c0,0x06fba40b),LL(0x6a2847e9,0xc1d933d2), LL(0x689e2c70,0xf4f5acf3),LL(0x46bafd31,0x92aab0e7),LL(0x3473f6e5,0x798d76aa),LL(0x93141934,0xcc6641db),
   LL(0xd31e535e,0xcae27757),LL(0x87c2ee11,0x04cc43b6),LL(0x2e029ffa,0x8d1f9675),LL(0xe4cc7a2c,0xc2150672), LL(0x8d68b013,0x3b03c1e0),LL(0xedf298f3,0xa9d6816f),LL(0xa2804464,0x1bfbb529),LL(0x5db22125,0x95a52fae),
   LL(0x0e1cb64e,0x55b32160),LL(0x7e7fc9fe,0x004828f6),LL(0x1bb0fb93,0x13394b82),LL(0x35f1a920,0xb6293a2d), LL(0xd145d2d9,0xde35ef21),LL(0xbb8fa603,0xbe6225b3),LL(0x32cf252d,0x00fc8f6b),LL(0x117cf8c2,0xa28e52e6),
   LL(0x4c371e6d,0x9d1dc89b),LL(0x36ef0f28,0xcebe0675),LL(0xa4292f81,0x5de05d09),LL(0x353e3083,0xa8303593), LL(0x7e37a9bb,0xa1715b0a),LL(0x2b8faec3,0x8c56f61e),LL(0x33c9b102,0x52507431),LL(0xa44431f0,0x0130cefc),
   LL(0xbd865cfb,0x56039fa0),LL(0xbc5f1dd7,0x4b03e578),LL(0xbabe7224,0x40edf2e4),LL(0x3a1988f6,0xc752496d), LL(0x564beb6b,0xd1572d3b),LL(0x39a1c608,0x0db1d110),LL(0x16f60126,0x568d1934),LL(0xf354af33,0x05ae9668),
   LL(0xc92544f2,0x19de6d37),LL(0xa35837d5,0xcc084353),LL(0x1a514ece,0xcbb6869c),LL(0x2e1d1066,0xb633e728), LL(0x936c581c,0xf15dd69f),LL(0x7439c4f9,0x96e7b8ce),LL(0x2e448a5b,0x5e676f48),LL(0xfd916bbb,0xb2ca7d5b),
   LL(0xf5024025,0xd55a2541),LL(0xe4c2d937,0x47bc5769),LL(0x0362189f,0x7d31b92a),LL(0xef7816f9,0x83f3086e), LL(0xb587579a,0xf9f46d94),LL(0x30e76c5f,0xec2d22d8),LL(0xb000ffcf,0x27d57461),LL(0x364ffc2c,0xbb7e65f9),
   LL(0x6652a220,0x7c7c9477),LL(0xd696c981,0x61618f89),LL(0x89effff3,0x5021701d),LL(0x7c314163,0xf2c8ff8e), LL(0x8efb4d3e,0x2da413ad),LL(0xce176d95,0x937b5adf),LL(0x2a67d51c,0x22867d34),LL(0x18eb3ac9,0x262b9b10),
   LL(0xc43ff28b,0x4e314fe4),LL(0x6a664e7a,0x76476627),LL(0xb7a565c2,0x3e90e40b),LL(0xc1acf831,0x8588993a), LL(0x8f938829,0xd7b501d6),LL(0x3edd7d4c,0x996627ee),LL(0x90cd34c7,0x37d44a62),LL(0xf3833e8d,0xa8327499),
   LL(0x4bf50353,0x2e18917d),LL(0x556765fb,0x85dd726b),LL(0x93d5ab66,0x54fe65d6),LL(0x915c25fe,0x3ddbaced), LL(0x12f22e85,0xa799d9a4),LL(0x6d06f6bc,0xe2a24867),LL(0x43ca1637,0xf4f1ee56),LL(0x61ece30a,0xfda2828b),
   LL(0xa2dee7a6,0x758c1a3e),LL(0x734b2284,0xdcde2f3c),LL(0x4eaba6ad,0xaba445d2),LL(0x76cee0a7,0x35aaf668), LL(0xe5aa049a,0x7e0b04a9),LL(0x91103e84,0xe74083ad),LL(0x40afecc3,0xbeb183ce),LL(0xea043f7a,0x6b89de9f),
},
/* digit=14 base_pwr=2^98 */
{
   LL(0xfe67ba66,0x0e299d23),LL(0x93cf2f34,0x91450760),LL(0x97fcf913,0xf45b5ea9),LL(0x8bd7ddda,0x5be00843), LL(0xd53ff04d,0x358c3e05),LL(0x5de91ef7,0xbf7ccdc3),LL(0xb69ec1a0,0xad684dbf),LL(0x801fd997,0x367e7cf2),
   LL(0xb0dc8595,0x0ca1f3b7),LL(0x9f1d9f2e,0x27de4608),LL(0xbadd82a7,0x1af3bf39),LL(0x65862448,0x79356a79), LL(0xf5f9a052,0xc0602345),LL(0x139a42f9,0x1a8b0f89),LL(0x844d40fc,0xb53eee42),LL(0x4e5b6368,0x93b0bfe5),
   LL(0xc024789c,0x5434dd02),LL(0x41b57bfc,0x90dca9ea),LL(0x243398df,0x8aa898e2),LL(0x894a94bb,0xf607c834), LL(0xc2c99b76,0xbb07be97),LL(0x18c29302,0x6576ba67),LL(0xe703a88c,0x3d79efcc),LL(0xb6a0d106,0xf259ced7),
   LL(0xc8de610b,0x0f893a5d),LL(0x67e223ce,0xe8c515fb),LL(0x4ead6dc5,0x7774bfa6),LL(0x925c728f,0x89d20f95), LL(0x098583ce,0x7a1e0966),LL(0x93f2a7d7,0xa2eedb94),LL(0x4c304d4a,0x1b282097),LL(0xc077282d,0x0842e3da),
   LL(0x3b9e2d7b,0xe4d972a3),LL(0xc48218ff,0x7cc60b27),LL(0x84149d91,0x8fc70838),LL(0x2f461ecc,0x5c04346f), LL(0x614650a9,0xebe9fdf2),LL(0xc1f666ac,0x5e35b537),LL(0x88babc83,0x645613d1),LL(0xc5e1c93e,0x88cace3a),
   LL(0x3de92e23,0x209ca375),LL(0x5fbbb6e3,0xccb03cc8),LL(0xd7b1487e,0xccb90f03),LL(0xc710941f,0xfa9c2a38), LL(0x6724ceed,0x756c3823),LL(0x192d0323,0x3a902258),LL(0xea5e038e,0xb150e519),LL(0xc7427591,0xdcba2865),
   LL(0x78890732,0xe549237f),LL(0x53fcb4d9,0xc443bef9),LL(0xeb3480d6,0x9884d8a6),LL(0x3048b186,0x8a35b6a1), LL(0x65e9a90a,0xb4e44716),LL(0x653006c0,0x45bf380d),LL(0x4fe9ae3b,0x8f3f820d),LL(0x979a3b71,0x244a35a0),
   LL(0x74cd06ff,0xa1010e9d),LL(0xaca3eeac,0x9c17c7df),LL(0x8063aa2b,0x74c86cd3),LL(0x734614ff,0x8595c4b3), LL(0x990f62cc,0xa3de00ca),LL(0xca0c3be5,0xd9bed213),LL(0xdf8ce9f5,0x7886078a),LL(0x5cd44444,0xddb27ce3),
   LL(0x58926ddd,0xed374a66),LL(0x908015b8,0x138b2d49),LL(0xde1f7ab8,0x886c6579),LL(0xc3020b7a,0x888b9aa0), LL(0x3a96e355,0xd3ec034e),LL(0xf30fbe9a,0xba65b0b8),LL(0xff21367a,0x064c8e50),LL(0x0b04b46e,0x1f508ea4),
   LL(0x747c866c,0x98561a49),LL(0x0518a062,0xbbb1e5fe),LL(0xecdc3608,0x20ff4e8b),LL(0x20184027,0x7f55cded), LL(0xf38c85f0,0x8d73ec95),LL(0x8bc3b8c3,0x5b589fdf),LL(0x0f12b66f,0xbe95dd98),LL(0x0e338e01,0xf5bd1a09),
   LL(0x5e915918,0x65163ae5),LL(0x86f8a46b,0x6158d6d9),LL(0xeeebf99c,0x8466b538),LL(0xbca477ef,0xca8761f6), LL(0x9ebbc601,0xaf3449c2),LL(0xe0c3ae2f,0xef3b0f41),LL(0x5de63752,0xaa6c577d),LL(0x64682a51,0xe9166601),
   LL(0xfc15aa1e,0x5a3097be),LL(0xb54b0745,0x40d12548),LL(0x519a5f12,0x5bad4706),LL(0xa439dee6,0xed03f717), LL(0x4a02c499,0x0794bb6c),LL(0xcffe71d2,0xf725083d),LL(0x0f3adcaf,0x2cad7519),LL(0x43729310,0x7f68ea1c),
   LL(0xb7ffd977,0xe747c8c7),LL(0x80761a22,0xec104c35),LL(0x5a3ffb83,0x8395ebaf),LL(0xe4b63db7,0xfb3261f4), LL(0xd883e544,0x53544960),LL(0x8cc2eeb8,0x13520d70),LL(0xd3d65f99,0x08f6337b),LL(0x781cf95b,0x83997db2),
   LL(0x0dbd2c01,0xce6ff106),LL(0x1f9ce934,0x4f8eea6b),LL(0x0e993921,0x546f7c4b),LL(0x5e753fc7,0x6236a324), LL(0xa16022e9,0x65a41f84),LL(0x43d1dbb2,0x0c18d878),LL(0x2d4cef9c,0x73c55640),LL(0x70444c74,0xa0428108),
   LL(0x9afdfb3c,0x68e4f15e),LL(0x5bdfb6df,0x49a56143),LL(0x5f823d97,0xa9bc1bd4),LL(0xea111c2a,0xbceb5970), LL(0xb269bbc4,0x366b455f),LL(0xe9bc5d62,0x7cd85e1e),LL(0x4f18b086,0xc743c41c),LL(0x95294fb9,0xa4b40990),
   LL(0x26ee8382,0x9c7c581d),LL(0x359d638e,0xcf17dcc5),LL(0xb728ae3d,0xee8273ab),LL(0xf821f047,0x1d112926), LL(0x50491a74,0x11498477),LL(0xfde0dfb9,0x687fa761),LL(0x7ea435ab,0x2c258022),LL(0x91ce7e3f,0x6b8bdb94),
   LL(0x3bf834aa,0x4c5b5dc9),LL(0x4f6c7e4b,0x04371819),LL(0x3736bcad,0xc284e00a),LL(0x21ae8f8d,0x0d881118), LL(0xf48c8e33,0xf9cf0f82),LL(0xa1bf40db,0xa11fd075),LL(0xdc2733e5,0xdceab0de),LL(0x8e986bd7,0xc560a8b5),
   LL(0x3929d097,0x48dd1fe2),LL(0x92f188f1,0x3885b290),LL(0xda6fcdac,0x0f2ae613),LL(0xb662a46c,0x9054303e), LL(0x0738042a,0xb6871e44),LL(0xbdaf6449,0x98e6a977),LL(0xd1c9df1b,0xd8bc0650),LL(0x36e098f9,0xef3d6451),
   LL(0xb6d72d28,0x03fbae82),LL(0xf5d84080,0x77ca9db1),LL(0xa58efc1c,0x8a112cff),LL(0xc564cb4a,0x518d761c), LL(0xf0d1b5ce,0x69b5740e),LL(0xe9eb1785,0x717039cc),LL(0x22f53382,0x3fe29f90),LL(0x6bc7c95c,0x8e54ba56),
   LL(0xf7f91d0f,0x9c806d8a),LL(0xa82a5728,0x3b61b0f1),LL(0x94d76754,0x4640032d),LL(0x47d834c6,0x273eb5de), LL(0x7b4e4d53,0x2988abf7),LL(0xde401777,0xb7ce66bf),LL(0x715071b3,0x9fba6b32),LL(0xad3a1a98,0x82413c24),
   LL(0xe0e8ad93,0x5b7fc8c4),LL(0x5fab868d,0xb5679aee),LL(0x2b3946f3,0xb1f9d2fa),LL(0x5685b50a,0x458897dc), LL(0x89d0caf3,0x1e98c930),LL(0x78642e92,0x39564c5f),LL(0x0dbdaf18,0x1b77729a),LL(0x579e82e6,0xf9170722),
   LL(0xe4515fa5,0x680c0317),LL(0xfb0c790f,0xf85cff84),LL(0x6d2e0765,0xc7a82aab),LL(0x35c82b32,0x7446bca9), LL(0x6d63184f,0x5de607aa),LL(0x262803a6,0x7c1a46a8),LL(0xaebe8035,0xd218313d),LL(0xc73c51f8,0x92113ffd),
   LL(0x12e7e46c,0x4b38e083),LL(0x56126bd5,0x69d0a37a),LL(0x73c07e04,0xfb3f324b),LL(0x8fda7267,0xa0c22f67), LL(0x4d2c7d8f,0x8f2c0051),LL(0xcbe2cae5,0xbc45ced3),LL(0xa8f0f277,0xe1c6cf07),LL(0x1eb99a98,0xbc392312),
   LL(0x3cc8ac85,0x75537b7e),LL(0xdd02753b,0x8d725f57),LL(0xb737df2f,0xfd05ff64),LL(0xf6d2531d,0x55fe8712), LL(0x6ab6b01c,0x57ce04a9),LL(0x7cd93724,0x69a02a89),LL(0xcf86699b,0x4f82ac35),LL(0x9cb4b232,0x8242d3ad),
   LL(0xd62105e5,0x713d0f65),LL(0x2d29be61,0xbb222bfa),LL(0x6cfbef09,0xf2f9a79e),LL(0xd5d6782f,0xfc24d8d3), LL(0xd4129967,0x5db77085),LL(0xdc3c2a43,0xdb81c3cc),LL(0x05d8d9a3,0x9d655fc0),LL(0x54298026,0x3f5d057a),
   LL(0x88c54694,0x1157f56d),LL(0x9b09573e,0xb26baba5),LL(0x22adffd1,0x2cab03b0),LL(0xdd69f383,0x60a412c8), LL(0x54b25039,0xed76e98b),LL(0x687e714d,0xd4ee67d3),LL(0x7b00b594,0x87739648),LL(0xc9ef709b,0xce419775),
   LL(0x1c203a40,0x40f76f85),LL(0xeafd8f91,0x30d352d6),LL(0x95578dd2,0xaf196d3d),LL(0x77cc3f3d,0xea4bb3d7), LL(0xb98e782b,0x42a5bd03),LL(0x0624920d,0xac958c40),LL(0xfc56fcc8,0xb838134c),LL(0x89572e5e,0x86ec4ccf),
   LL(0x9be47be0,0x69c43526),LL(0xcb28fea1,0x323b7dd8),LL(0x3a6c67e5,0xfa5538ba),LL(0x1d378e46,0xef921d70), LL(0x3c4b880e,0xf92961fc),LL(0x98940a67,0x3f6f914e),LL(0xfef0ff39,0xa990eb0a),LL(0xf0eeff9c,0xa6c2920f),
   LL(0x51b8d9a3,0xca804166),LL(0x0ffb0db1,0x42531bc9),LL(0xaa82e7ce,0x72ce4718),LL(0xdf574741,0x6e199913), LL(0xd5d36946,0xd5f1b13d),LL(0xf68f0194,0x8255dc65),LL(0x8710d230,0xdc9df4cd),LL(0x138c1988,0x3453c20f),
   LL(0x89a6ef01,0x9af98dc0),LL(0x9857df85,0x4dbcc3f0),LL(0x5c1ad924,0x34805601),LL(0xd0493046,0x40448da5), LL(0x4ee343e2,0xf629926d),LL(0x90e8a301,0x6343f1bd),LL(0x40815b3f,0xefc93491),LL(0xde8f66fb,0xf882a423),
   LL(0xe7db9f57,0x3a12d5f4),LL(0x3c384c27,0x7dfba38a),LL(0x6fc660b1,0x7a904bfd),LL(0x2773b21c,0xeb6c5db3), LL(0x1cdfe049,0xc350ee66),LL(0x44540f29,0x9baac0ce),LL(0xa5ec6aad,0xbc57b6ab),LL(0x0a7c1baa,0x167ce8c3),
   LL(0x53fb2b56,0xb23a03a5),LL(0x4e057f78,0x6ce141e7),LL(0x89e490d9,0x796525c3),LL(0xa31a7e75,0x0bc95725), LL(0x1220fd06,0x1ec56791),LL(0x408b0bd6,0x716e3a3c),LL(0xe8ebeba9,0x31cd6bf7),LL(0xbee6b670,0xa7326ca6),
   LL(0xcd090c43,0x3d9f851c),LL(0xf12c3988,0x561e8f13),LL(0x904b7be4,0x50490b6a),LL(0x0410737b,0x61690ce1), LL(0x0f009052,0x299e9a37),LL(0xf026092e,0x258758f0),LL(0xfdfcdc0f,0x9fa255f3),LL(0xc0e1bcd2,0xdbc9fb1f),
   LL(0x24651840,0x35f9dd6e),LL(0xa5c59abc,0xdca45a84),LL(0xecca4938,0x103d396f),LL(0xb97b3f29,0x4532da0a), LL(0x1999a6bf,0xc4135ea5),LL(0x5e6bf2ee,0x3aa9505a),LL(0x3f5be093,0xf77cef06),LL(0xa943152e,0x97d1a0f8),
   LL(0x2e1c21dd,0x2cb0ebba),LL(0x2c6797c4,0xf41b29fc),LL(0xb300101f,0xc6e17321),LL(0xd0d79a89,0x4422b0e9), LL(0x92f1bfc4,0x49e4901c),LL(0xe1e10ed9,0x06ab1f8f),LL(0xdb2926b8,0x84d35577),LL(0x356e8ec2,0xca349d39),
   LL(0x343bf1a9,0x70b63d32),LL(0x37d1a6b1,0x8fd3bd28),LL(0x316865b4,0x0454879c),LL(0xc458efa2,0xee959ff6), LL(0x9706dc3f,0x0461dcf8),LL(0x164e4b2e,0x737db0e2),LL(0x2f8843c8,0x09262680),LL(0x7745e6f6,0x54498bbc),
   LL(0xa29e24af,0x359473fa),LL(0x70aa87a1,0xfcc3c454),LL(0x00573ace,0xfd2c4bf5),LL(0x28dd1965,0xb65b514e), LL(0x2193e393,0xe46ae7cf),LL(0xf5444d97,0x60e9a4e1),LL(0x00ff38ed,0xe7594e96),LL(0x0a0e0f02,0x43d84d2f),
   LL(0xee398a21,0x8b6db141),LL(0xe3bcc5be,0xb88a56ae),LL(0x373460ea,0x0a1aa52f),LL(0x160bb19b,0x20da1a56), LL(0x65bf0384,0xfb54999d),LL(0x5d5a180e,0x71a14d24),LL(0x21737b04,0xbc44db7b),LL(0x01dd8e92,0xd84fcb18),
   LL(0xfa44b479,0x80de937b),LL(0x5c98fd4f,0x53505499),LL(0x28f08727,0x1edb12ab),LL(0xa5f3ef53,0x4c58b582), LL(0x8327f246,0xbfb236d8),LL(0x4d7df320,0xc3a3bfaa),LL(0xb96024f2,0xecd96c59),LL(0x7f4e0433,0xfc293a53),
   LL(0x5acf6e10,0x5341352b),LL(0xafe652c3,0xc50343fd),LL(0x18577a7f,0x4af3792d),LL(0xaf16823d,0xe1a4c617), LL(0x33425d0a,0x9b26d0cd),LL(0x9b7bc47f,0x306399ed),LL(0x706bb20b,0x2a792f33),LL(0x98111055,0x31219614),
   LL(0x87f5d28b,0x864ec064),LL(0x962277fd,0x11392d91),LL(0xbb6aed5f,0xb5aa7942),LL(0x47e799d9,0x080094dc), LL(0x208ba19b,0x4afa588c),LL(0x8512f284,0xd3e7570f),LL(0x02f5799a,0xcbae64e6),LL(0x514b9492,0xdeebe7ef),
   LL(0xe5c298ff,0x30300f98),LL(0x3678361f,0x17f561be),LL(0x98cb9a16,0xf52ff312),LL(0x5562d490,0x6233c3bc), LL(0x92e3a2cb,0x7bfa15a1),LL(0xe6365119,0x961bcfd1),LL(0x2c8c53b1,0x3bdd29bf),LL(0x822844ba,0x739704df),
   LL(0x7e7b754b,0x7dacfb58),LL(0xa806c9b9,0x23360791),LL(0x23504452,0xe7eb88c9),LL(0x852c1783,0x2983e996), LL(0x958d881d,0xdd4ae529),LL(0x262c7b3c,0x026bae03),LL(0x960b52d1,0x3a6f9193),LL(0x92696cfb,0xd0980f90),
   LL(0xd5f30851,0x4c1f428c),LL(0x2a4f6630,0x94dfed27),LL(0xfc5d48a4,0x4df53772),LL(0x933260ce,0xdd2d5a2f), LL(0xd44cc7a5,0x574115bd),LL(0xbd12533a,0x4ba6b20d),LL(0x243057c9,0x30e93cb8),LL(0x14de320e,0x794c486a),
   LL(0xf21496e4,0xe925d4ce),LL(0xec696331,0xf951d198),LL(0x3e8d812f,0x9810e2de),LL(0x389294ab,0xd0a47259), LL(0x0e3bab66,0x513ba2b5),LL(0xabad306f,0x462caff5),LL(0xaf04c49e,0xe2dc6d59),LL(0xe0b84b0b,0x1aeb8750),
   LL(0x2f7d0ca2,0xc034f12f),LL(0xe06acf2f,0x6d2e8128),LL(0x21facc2f,0x801f4f83),LL(0xf40ef607,0xa1170c03), LL(0x7805a99c,0xfe0a1d4f),LL(0xcc26aba5,0xbde56a36),LL(0x35531f40,0x5b1629d0),LL(0x9afa6108,0xac212c2b),
   LL(0x15697be5,0x30a06bf3),LL(0x2c63c7c1,0x6f0545dc),LL(0x7ccdadaf,0x5d8cb842),LL(0xac7015bb,0xd52e379b), LL(0xf462c23e,0xc4f56147),LL(0x46bc24b0,0xd44a4298),LL(0xe2856d4f,0xbc73d23a),LL(0x0832bcdf,0x61cedd8c),
   LL(0x99f241d7,0x60953556),LL(0x001a349d,0xee4adbd7),LL(0xaa89e491,0x0b35bf6a),LL(0x136f7546,0x7f0076f4), LL(0x9264da3d,0xd19a18ba),LL(0x62a7a28b,0x6eb2d2cd),LL(0x8761c971,0xcdba941f),LL(0xa3be4a5d,0x1550518b),
   LL(0x57d0b70c,0xd0e8e2f0),LL(0xcd133ba3,0xeea8612e),LL(0x44416aec,0x814670f0),LL(0x30775061,0x424db6c3), LL(0x16213fd1,0xd96039d1),LL(0x18a3478f,0xc61e7fa5),LL(0xcb0c5021,0xa805bdcc),LL(0x0cc616dd,0xbdd6f3a8),
   LL(0x5d97f7e2,0x06009667),LL(0xaf0bf4b6,0x31db0fc1),LL(0x5491627a,0x23680ed4),LL(0x7d741fb1,0xb99a3c66), LL(0x36b1ff92,0xe9bb5f55),LL(0x512b388d,0x29738577),LL(0x50fcf263,0xdb8a2ce7),LL(0x6c4f7b47,0x385346d4),
   LL(0x31631f9e,0xbe86c5ef),LL(0x03a57a29,0xbf91da21),LL(0x7b23f821,0xc3b1f796),LL(0x770db354,0x0f7d00d2), LL(0xd8fe79da,0x8ffc6c3b),LL(0xd525c996,0xcc5e8c40),LL(0xcfff632a,0x4640991d),LL(0x67112528,0x64d97e8c),
   LL(0x02f1cd1e,0xc232d973),LL(0x1dd212a4,0xce87eacb),LL(0xe69802f7,0x6e4c8c73),LL(0x1fffddbd,0x12ef0290), LL(0x1bcea6e2,0x941ec74e),LL(0x3cb92cbb,0xd0b54024),LL(0x7e8f9d05,0x809fb9d4),LL(0xf2992aae,0x3bf16159),
   LL(0xf8a7a838,0xad40f279),LL(0x05615660,0x11aea631),LL(0xa01f6fa1,0xbf52e6f1),LL(0x3dc2aec9,0xef046995), LL(0xd8080711,0x785dbec9),LL(0x9fdedf76,0xe1aec60a),LL(0xfa21c126,0xece797b5),LL(0x05e52732,0xc66e898f),
   LL(0x08811fdb,0x39bb69c4),LL(0x2fc7f082,0x8bfe1ef8),LL(0x174f4138,0xc8e7a393),LL(0xd58d1f98,0xfba8ad1d), LL(0xbfd2fd5b,0xbc21d0ce),LL(0x6ee60d61,0x0b839a82),LL(0xafd22253,0xaacf7658),LL(0xaae396b3,0xb526bed8),
   LL(0x38564464,0xccc1bbc2),LL(0x8c45bc73,0x9e3ff947),LL(0x58188a78,0xcde9bca3),LL(0xd73bf8f7,0x138b8ee0), LL(0x4123c489,0x5c7e234c),LL(0xfa643297,0x66e69368),LL(0x39a15fa3,0x0629eeee),LL(0xa9e2a927,0x95fab881),
   LL(0xeafbb1e1,0xb2497007),LL(0xe75b7a93,0xd75c9ce6),LL(0xefb68d78,0x3558352d),LL(0x223f6396,0xa2f26699), LL(0xe469b17a,0xeb911ecf),LL(0xe72d3ec2,0x62545779),LL(0x82cb113f,0x8ea47de7),LL(0x4e1fa98d,0xebe4b086),
   LL(0x8cdfedb1,0xec2d5ed7),LL(0xfe211a74,0xa535c077),LL(0x11d244c5,0x9678109b),LL(0xbe299a76,0xf17c8bfb), LL(0xfb11fbc4,0xb651412e),LL(0x94ab3f65,0xea0b5482),LL(0x0cf78243,0xd8dffd95),LL(0xce0361d4,0x2e719e57),
   LL(0x304ddc5b,0x9007f085),LL(0x4daba2ea,0x095e8c6d),LL(0x3f9d28a9,0x5a33cdb4),LL(0xe2283003,0x85b95cd8), LL(0xb9744733,0xbcd6c819),LL(0xfc7f5783,0x29c5f538),LL(0xd59038e4,0x6c49b2fa),LL(0x3bbe1018,0x68349cc1),
   LL(0x21830ee5,0xcc490c1d),LL(0xe9bfa297,0x36f9c4ee),LL(0x48de1a94,0x58fd7294),LL(0x4e8f2cdc,0xaadb13a8), LL(0x81313dba,0x515eaaa0),LL(0xc2152dd8,0xc76bb468),LL(0xa653dbf8,0x357f8d75),LL(0xb14ac143,0xe4d8c4d1),
   LL(0xb055cb40,0xbdb8e675),LL(0x977b5167,0x898f8e7b),LL(0xb82fb863,0xecc65651),LL(0x6d88f01f,0x56544814), LL(0x263a75a9,0xb0928e95),LL(0x1a22fcda,0xcfb6836f),LL(0x3f3bd37c,0x651d14db),LL(0xb6ad4664,0x1d3837fb),
   LL(0xff4f94ab,0x7c5fb538),LL(0x6d7fb8f2,0x7243c712),LL(0xa85c5287,0xef13d60c),LL(0x4bb8dd1b,0x18cfb7c7), LL(0x72908219,0x82f9bfe6),LL(0x9d5144ab,0x35c4592b),LL(0x9cf4b42f,0x52734f37),LL(0x8c60ddc4,0x6bac55e7),
   LL(0x94dea0f6,0xb5cd811e),LL(0xe18cc1a3,0x259ecae4),LL(0x15e660f8,0x6a0e836e),LL(0x0e02bff2,0x6c639ea6), LL(0x7e1026fd,0x8721b8cb),LL(0x63261942,0x9e73b50b),LL(0x77f01da3,0xb8c70974),LL(0x8268f57f,0x1839e6a6),
   LL(0x5150b805,0x571b9415),LL(0xf92c7097,0x1892389e),LL(0x4a084b95,0x8d69c18e),LL(0xbe5b495c,0x7014c512), LL(0x1b07523c,0x4780db36),LL(0x2c1c64fa,0x2f6219ce),LL(0x602c105a,0xc38b81b0),LL(0x5dc8e360,0xab4f4f20),
   LL(0xcf7d62d2,0x20d3c982),LL(0x23ba8150,0x1f36e29d),LL(0x92763f9e,0x48ae0bf0),LL(0x1d3a7007,0x7a527e6b), LL(0x581a85e3,0xb4a89097),LL(0xdc158be5,0x1f1a520f),LL(0x167d726e,0xf98db37d),LL(0x1113e862,0x8802786e),
},
/* digit=15 base_pwr=2^105 */
{
   LL(0x36f09ab0,0xefb2149e),LL(0x4a10bb5b,0x03f163ca),LL(0x06e20998,0xd0297045),LL(0x1b5a3bab,0x56f0af00), LL(0x70880e0d,0x7af4cfec),LL(0xbe3d913f,0x7332a66f),LL(0x7eceb4bd,0x32e6c84a),LL(0x9c228f55,0xedc4a79a),
   LL(0xc55c4496,0xc37c7dd0),LL(0x25bbabd2,0xa6a96357),LL(0xadd7f363,0x5b7e63f2),LL(0x2e73f1df,0x9dce3782), LL(0xb2b91f71,0xe1e5a16a),LL(0x5ba0163c,0xe4489823),LL(0xf6e515ad,0xf2759c32),LL(0x8615eecf,0xa5e2f1f8),
   LL(0xabded551,0x74519be7),LL(0xc8b74410,0x03d358b8),LL(0x0e10d9a9,0x4d00b10b),LL(0x28da52b7,0x6392b0b1), LL(0x0b75c904,0x6744a298),LL(0xa8f7f96c,0xc305b0ae),LL(0x182cf932,0x042e421d),LL(0x9e4636ca,0xf6fc5d50),
   LL(0xd64cc78c,0x795847c9),LL(0x9b6cb27b,0x6c50621b),LL(0xdf8022ab,0x07099bf8),LL(0xc04eda1d,0x48f862eb), LL(0xe1603c16,0xd12732ed),LL(0x5c9a9450,0x19a80e0f),LL(0xb429b4fc,0xe2257f54),LL(0x45460515,0x66d3b2c6),
   LL(0x822e37be,0x6ca4f87e),LL(0x253bda4e,0x73f237b4),LL(0x41190aeb,0xf747f3a2),LL(0x804cf284,0xf06fa36f), LL(0xfc621c12,0x0a6bbb6e),LL(0x40b80ec6,0x5d624b64),LL(0x7ba556f3,0x4b072425),LL(0x3e2d20a8,0x7fa0c354),
   LL(0xe3229d41,0xe921fa31),LL(0x94531bd4,0xa929c652),LL(0xa6d38209,0x84156027),LL(0x6bdb97bd,0xf3d69f73), LL(0x16833631,0x8906d19a),LL(0x03d51be3,0x68a34c2e),LL(0x0e511cd8,0xcb59583b),LL(0xfdc132a8,0x99ce6bfd),
   LL(0xffcdb463,0x3facdaaa),LL(0x34a38b08,0x658bbc1a),LL(0xf1a9078d,0x12a801f8),LL(0x6ab855de,0x1567bcf9), LL(0x3572359b,0xe08498e0),LL(0x8659e68b,0xcf0353e5),LL(0x7d23807c,0xbb86e9c8),LL(0x2198e8a2,0xbc08728d),
   LL(0x453cadd6,0x8de2b7bc),LL(0xbc0bc1f8,0x203900a7),LL(0xa6abd3af,0xbcd86e47),LL(0x8502effb,0x911cac12), LL(0xec965469,0x2d550242),LL(0x29e0017e,0x0e9f7692),LL(0x65979885,0x633f078f),LL(0x4cf751ef,0xfb87d449),
   LL(0xfc25419a,0xe1790e4b),LL(0x4bff3cfd,0x36467203),LL(0x25b6e83f,0xc8db6386),LL(0x6cad6fd2,0x6cc69f23), LL(0x6bc68bb9,0x0219e45a),LL(0x297f7334,0xe43d79b6),LL(0x465dc97c,0x7d445368),LL(0x2a0b949a,0x4b9eea32),
   LL(0x6102d021,0x1b96c6ba),LL(0x2f4461ea,0xeaafac78),LL(0xc49f19a8,0xd4b85c41),LL(0xcf538875,0x275c28e4), LL(0xdd2e54e0,0x35451a9d),LL(0x0605618b,0x6991adb5),LL(0x7b36cd24,0x5b8b4bcd),LL(0x56f37216,0x372a4f8c),
   LL(0xa6a5da60,0xc890bd73),LL(0xdc4c9ff0,0x6f083da0),LL(0xf0536e57,0xf4e14d94),LL(0xaaec8243,0xf9ee1eda), LL(0x8bdcf8e7,0x571241ec),LL(0x0b041e26,0xa5db8271),LL(0xe3fff040,0x9a0b9a99),LL(0x7c271202,0xcaaf21dd),
   LL(0x4f0dd2e8,0xb4e2b2e1),LL(0x0a377ac7,0xe77e7c4f),LL(0x0d7a2198,0x69202c3f),LL(0x28200eb8,0xf759b7ff), LL(0xdcfe314e,0xc87526ed),LL(0x53d5cf99,0xeb84c524),LL(0x515138b6,0xb1b52ace),LL(0x23fca3f4,0x5aa7ff8c),
   LL(0xb9791a26,0xff0b13c3),LL(0xcdd58b16,0x960022da),LL(0x57aad2de,0xdbd55c92),LL(0xf30fe619,0x3baaaaa3), LL(0x0d881efd,0x9a4b2346),LL(0x46325e2a,0x506416c0),LL(0x035c18d4,0x91381e76),LL(0xf27817b0,0xb3bb68be),
   LL(0x5116f937,0x15bfb8bf),LL(0xc1268943,0x7c64a586),LL(0x8419a2c8,0x71e25cc3),LL(0x8335f463,0x9fd6b0c4), LL(0xe8ee0e0e,0x4bf0ba3c),LL(0x298c21fa,0x6f6fba60),LL(0xae66bee0,0x57d57b39),LL(0x22672544,0x292d5130),
   LL(0xbab093b3,0xf451105d),LL(0x02839986,0x012f59b9),LL(0x3474a89c,0x8a915802),LL(0x2de03e97,0x048c919c), LL(0x91071cd5,0xc476a2b5),LL(0x034970a5,0x791ed89a),LL(0xe1b7994b,0x89bd9042),LL(0xa1057ffd,0x8eaf5179),
   LL(0xd551ee10,0x6066e2a2),LL(0x727e09a6,0x87a8f1d8),LL(0x2c01148d,0x00d08bab),LL(0x424f33fe,0x6da8e4f1), LL(0xcf9a4e71,0x466d17f0),LL(0x3bf5cb19,0xff502010),LL(0xd062ecc0,0xdccf97d8),LL(0x81d80ac4,0x80c0d9af),
   LL(0x033f2876,0xe87771d8),LL(0x7d5cc3db,0xb0186ec6),LL(0x3bc9bc1d,0x58e8bb80),LL(0x6f6ef60e,0x4d1395cc), LL(0x186244a0,0xa73c62d6),LL(0x110a5b53,0x918e5f23),LL(0x741b7eab,0xed4878ca),LL(0xdbe03e51,0x3038d71a),
   LL(0xa93c3246,0x840204b7),LL(0xa0b9b4cd,0x21ab6069),LL(0xb1d64218,0xf5fa6e2b),LL(0xf3d56191,0x1de6ad0e), LL(0xff1929c7,0x570aaa88),LL(0x640e87b5,0xc6df4c6b),LL(0xc65f0ccc,0xde8a74f2),LL(0xe6f6cc01,0x8b972fd5),
   LL(0x0b846531,0x3fff36b6),LL(0x10a5e475,0xba7e45e6),LL(0x4145b6c5,0x84a1d10e),LL(0x5e046d9d,0xf1f7f91a), LL(0x44de90d7,0x0317a692),LL(0xf199c15e,0x951a1d4a),LL(0xc9d73deb,0x91f78046),LL(0xfab8224f,0x74c82828),
   LL(0xe7560b90,0xaa6778fc),LL(0xa7e824ce,0xb4073e61),LL(0xd642eba8,0xff0d693c),LL(0x5dccef38,0x7ce2e57a), LL(0x1df1ad46,0x89c2c789),LL(0x098346fd,0x83a06922),LL(0xda2fc177,0x2d715d72),LL(0x85b6cf1d,0x7b6dd71d),
   LL(0x73fa9cb0,0xc60a6d0a),LL(0x328bf5a9,0xedd3992e),LL(0x832c8c82,0xc380ddd0),LL(0xa2a0bf50,0xd182d410), LL(0xd9a528db,0x7d9d7438),LL(0xcaf53994,0xe8b1a0e9),LL(0x0e19987c,0xddd6e5fe),LL(0x190b059d,0xacb8df03),
   LL(0x8300129f,0x53703a32),LL(0x68c43bfd,0x1f637662),LL(0x00e54051,0xbcbd1913),LL(0x7bf5a8c5,0x812fcc62), LL(0x29fb85da,0x3f969d5f),LL(0x694759e8,0x72f4e00a),LL(0x790726b7,0x426b6e52),LL(0x3bdbb209,0x617bbc87),
   LL(0x97aee317,0x511f8bb9),LL(0xe81536a8,0x812a4096),LL(0x3ac09b9b,0x137dfe59),LL(0xba8c9a7a,0x0682238f), LL(0xaeccb4bd,0x7072ead6),LL(0x692ba633,0x6a34e9aa),LL(0x6fff9d33,0xc82eaec2),LL(0x1d4d2b62,0xfb753512),
   LL(0x1d7aadab,0x1a0445ff),LL(0xd5f6a67c,0x65d38260),LL(0x91cfb26f,0x6e62fb08),LL(0x5c7d91d6,0xef1e0fa5), LL(0x33db72cd,0x47e7c7ba),LL(0xfa7c74b2,0x017cbc09),LL(0xf50a503c,0x3c931590),LL(0x616baa42,0xcac54f60),
   LL(0xb2369f0f,0x9b6cd380),LL(0x23c76151,0x97d3a70d),LL(0x9862a9c6,0x5f9dd6fc),LL(0x12312f51,0x044c4ab2), LL(0x834a2ddc,0x035ea0fd),LL(0xcc7b826d,0x49e6b862),LL(0x62fce490,0xb03d6883),LL(0xb37e36e9,0x62f2497a),
   LL(0xc6458293,0x04b005b6),LL(0xe8d10af7,0x36bb5276),LL(0x8ee617b8,0xacf2dc13),LL(0xb004b3d4,0x470d2d35), LL(0xfeeb1b77,0x06790832),LL(0x85657f9c,0x2bb75c39),LL(0xc0f60004,0xd70bd4ed),LL(0x219b018b,0xfe797ecc),
   LL(0x753aebcc,0x9b5bec2a),LL(0xc939eca5,0xdaf9f3dc),LL(0xd095ad09,0xd6bc6833),LL(0xdaa4d2fc,0x98abdd51), LL(0x8d168be5,0xd9840a31),LL(0x2325a23c,0xcf7c10e0),LL(0x7e6ecfaf,0xa5c02aa0),LL(0xb5bfdf18,0x2462e7e6),
   LL(0xa0cc3f12,0xab2d8a8b),LL(0xbc672a29,0x68dd485d),LL(0x596f2cd3,0x72039752),LL(0xa0cf3d8d,0x5d3eea67), LL(0xe6602671,0x810a1a81),LL(0x14026c0c,0x8f144a40),LL(0x76b50f85,0xbc753a6d),LL(0x645cd4a4,0xc4dc21e8),
   LL(0x521d0378,0xc5262dea),LL(0x05011c6f,0x802b8e0e),LL(0x0b4c19ea,0x1ba19cbb),LL(0xebf0aaec,0x21db64b5), LL(0x70342f9d,0x1f394ee9),LL(0x1bc44a14,0x93a10aee),LL(0x3efd0baa,0xa7eed31b),LL(0x1d154e65,0x6e7c824e),
   LL(0x9966e7ee,0xee23fa81),LL(0x05b7920d,0x64ec4aa8),LL(0x2d90aad4,0x2d44462d),LL(0xdf277ad5,0xf44dd195), LL(0xbb46b6a1,0x8d6471f1),LL(0xfd885090,0x1e65d313),LL(0x13a977b4,0x33a800f5),LL(0x0797e1ef,0xaca9d721),
   LL(0xfcff6a17,0x9a5a85a0),LL(0x1eca7cee,0x9970a3f3),LL(0xc9504be3,0xbb9f0d6b),LL(0xadd24ee2,0xe0c504be), LL(0x77fcc2f4,0x7e09d956),LL(0x65bb5fc4,0xef1a5227),LL(0x8b9286aa,0x145d4fb1),LL(0x6649028b,0x66fd0c5d),
   LL(0x1bf4581c,0x98857ceb),LL(0xaca7b166,0xe635e186),LL(0x659722ac,0x278ddd22),LL(0x1db68007,0xa0903c4c), LL(0x48f21402,0x366e4589),LL(0xb96abda2,0x31b49c14),LL(0xe0403190,0x329c4b09),LL(0xd29f43fe,0x97197ca3),
   LL(0x274983d8,0x8073dd1e),LL(0x55717c8f,0xda1a3bde),LL(0x0361f9d1,0xfd3d4da2),LL(0x4c7de1ce,0x1332d081), LL(0xaa6d0e10,0x9b7ef7a3),LL(0xf54f1c4a,0x17db2e73),LL(0x4cd35567,0xaf3dffae),LL(0xe56f4e71,0xaaa2f406),
   LL(0x7ace3fc7,0x8966759e),LL(0x45a8d8c6,0x9594eacf),LL(0x91834e0e,0x8de3bd8b),LL(0x548c0421,0xafe4ca53), LL(0xe6ee81c6,0xfdd7e856),LL(0x6b891a3a,0x8f671beb),LL(0xfae63829,0xf7a58f2b),LL(0x9c11ac9f,0x9ab186fb),
   LL(0x10b5be76,0x8d6eb369),LL(0xfb040bcd,0x046b7739),LL(0xcb73de88,0xccb4529f),LL(0xcf26be03,0x1df0fefc), LL(0xbcfcd027,0xad7757a6),LL(0xbb3165ca,0xa8786c75),LL(0x7e99a4d9,0xe9db1e34),LL(0xb06c504b,0x99ee86df),
   LL(0xc15c9f0a,0x5b7c2ddd),LL(0x4295989e,0xdf87a734),LL(0x03d08fda,0x59ece47c),LL(0xad5fc702,0xb074d3dd), LL(0x51a03776,0x20407903),LL(0x2a608007,0x2bb1f77b),LL(0xe1153185,0x25c58f4f),LL(0x766e6447,0xe6df62f6),
   LL(0xed51275a,0xefb3d1be),LL(0x2f0f483f,0x5de47dc7),LL(0x97c2bedf,0x7932d98e),LL(0x0219f8a1,0xd5c11927), LL(0xa73a294e,0x9d751200),LL(0x9dc20172,0x5f88434a),LL(0xa26f506a,0xd28d9fd3),LL(0x9d1dcd48,0xa890cd31),
   LL(0x70f4d3b4,0x0aebaec1),LL(0x0ffc8d00,0xfd1a1369),LL(0x57d57838,0xb9d9c240),LL(0x68bac361,0x45929d26), LL(0x25b15ca6,0x5a2cd060),LL(0x6e474446,0x4b3c83e1),LL(0xee1e5134,0x1aac7578),LL(0xc91e2f41,0xa418f5d6),
   LL(0x213ed68b,0x6936fc8a),LL(0x510a5224,0x860ae7ed),LL(0xdef09b53,0x63660335),LL(0xcd79c98d,0x641b2897), LL(0x01110f35,0x29bd38e1),LL(0x648b1937,0x79c26f42),LL(0x9d9164f4,0x64dae519),LL(0x0265c273,0xd85a2310),
   LL(0x4b07e2b1,0x7173dd5d),LL(0x8d9ea221,0xd144c4cb),LL(0x1105ab14,0xe8b04ea4),LL(0xfe80d8f1,0x92dda542), LL(0xcf03dce6,0xe9982fa8),LL(0x1a22cffc,0x8b5ea965),LL(0x3fad88c4,0xf7f4ea7f),LL(0x6a5ba95c,0x62db773e),
   LL(0x93f24567,0xd20f02fb),LL(0x315257ca,0xfd46c69a),LL(0x8bcab987,0x0ac74cc7),LL(0x5ceca2f5,0x46f31c01), LL(0x888b219e,0x40aedb59),LL(0xe1fccd02,0xe50ecc37),LL(0x911f816c,0x1bcd9dad),LL(0x8db9b00c,0x583cc1ec),
   LL(0xa483bf11,0xf3cd2e66),LL(0xb1b2c169,0xfa08a6f5),LL(0x4be9fa28,0xf375e245),LL(0x5b6d011f,0x99a7ffec), LL(0xc4ae62da,0x6a3ebddb),LL(0x374aef5d,0x6cea00ae),LL(0x9d4d05bc,0xab5fb98d),LL(0xd560f252,0x7cba1423),
   LL(0x208490de,0x49b2cc21),LL(0xbcfb2879,0x1ca66ec3),LL(0x1b6fb16f,0x7f1166b7),LL(0x65fe5db3,0xfff63e08), LL(0x8b2610be,0xb8345abe),LL(0x39de3df4,0xb732ed80),LL(0x211c32b4,0x0e24ed50),LL(0x848ff27d,0xd10d8a69),
   LL(0xed4de248,0xc1074398),LL(0x10488927,0xd7cedace),LL(0x85673e13,0xa4aa6bf8),LL(0x6daf30af,0xb46bae91), LL(0xfcef7ad8,0x07088472),LL(0xd4b35e97,0x61151608),LL(0xdde29986,0xbcfe8f26),LL(0xd5a34c79,0xeb84c4c7),
   LL(0x164e1214,0xc1eec55c),LL(0xa147bb03,0x891be86d),LL(0x0ba96835,0x9fab4d10),LL(0xa5c1ae9f,0xbf01e9b8), LL(0xb186ebc0,0x6b4de139),LL(0x85b91bca,0xd5c74c26),LL(0xc2d93854,0x5086a99c),LL(0xa7a9dfbc,0xeed62a7b),
   LL(0x76b7618a,0x8778ed6f),LL(0x03b66062,0xbff750a5),LL(0xb65186db,0x4cb7be22),LL(0xcc3a6d13,0x369dfbf0), LL(0x7191a321,0xc7dab26c),LL(0x40ed718e,0x9edac3f9),LL(0xd0cfd183,0xbc142b36),LL(0x7c991693,0xc8af82f6),
   LL(0x97ce0b2a,0xb3d1e4d8),LL(0xc3a55cdf,0xe6d7c87f),LL(0x68b81afe,0x35846b95),LL(0xd3c239d8,0x018d12af), LL(0x01206e15,0x2b2c6208),LL(0xa3b882c6,0xe0e42453),LL(0xa50162d5,0x854470a3),LL(0x7017a62a,0x08157478),
   LL(0x820357c7,0x18bd3fb4),LL(0x6f1458ad,0x992039ae),LL(0x25b44aa1,0x9a1df3c5),LL(0xed3d5281,0x2d780357), LL(0xc77ad4d4,0x58cf7e4d),LL(0xf9df4fc4,0xd49a7998),LL(0x1d71205e,0x4465a8b5),LL(0x649254aa,0xa0ee0ea6),
   LL(0xab7bd771,0x4b5eeecf),LL(0x35c262b9,0x6c873073),LL(0x3c9d61e7,0xdc5bd648),LL(0x321460d2,0x233d6d54), LL(0xfc195bcc,0xd20c5626),LL(0x04d78b63,0x25445958),LL(0x17ec8ef3,0xe03fcb3d),LL(0x46b8f781,0x54b690d1),
   LL(0x21230646,0x82fa2c8a),LL(0x084f418c,0xf51aabb9),LL(0x1a30ba43,0xff4fbec1),LL(0x743c9df7,0x6a5acf73), LL(0xd635b4d5,0x1da2b357),LL(0xecd5c1da,0xc3de68dd),LL(0xd61af0dd,0xa689080b),LL(0xd665bf99,0xdea5938a),
   LL(0xfe637294,0x0231d71a),LL(0xa5a81cd8,0x01968aa6),LL(0x048e63b5,0x11252d50),LL(0x6ca007e9,0xc446bc52), LL(0x96d6134b,0xef8c50a6),LL(0x9e09a05c,0x9361fbf5),LL(0xdca3291a,0xf17f85a6),LL(0xff251a21,0xb178d548),
   LL(0xa4df3915,0x87f6374b),LL(0x2fd5d608,0x566ce1bf),LL(0x7de35102,0x425cba4d),LL(0x58c5d5e2,0x6b745f8f), LL(0x63122edf,0x88402af6),LL(0x3b989a89,0x3190f9ed),LL(0xebba3156,0x4ad3d387),LL(0xc7c469a5,0xef385ad9),
   LL(0x3f642c29,0xb08281de),LL(0x910ffb88,0x20be0888),LL(0xd5292546,0xf353dd4a),LL(0x8377a262,0x3f1627de), LL(0xeefcd638,0xa5faa013),LL(0x74cc77c3,0x8f3bf626),LL(0xa348f55e,0x32618f65),LL(0x9fefeb9e,0x5787c0dc),
   LL(0xd9a23e44,0xf1673aa2),LL(0x4e10690d,0x88dfa993),LL(0x2bf91108,0x1ced1b36),LL(0x3af48649,0x9193ceca), LL(0x2d738fc5,0xfb34327d),LL(0x975fee6c,0x6697b037),LL(0xc04079a5,0x2f485da0),LL(0x2feaa1ac,0x2cdf5735),
   LL(0xbd55659e,0x76944420),LL(0x4376090c,0x7973e32b),LL(0x163b591a,0x86bb4fe1),LL(0xc196f0ca,0x10441aed), LL(0x045ad915,0x3b431f4a),LL(0xa4afacb1,0x6c11b437),LL(0x71fdbbd8,0x30b0c7db),LL(0xeda65acd,0xb642931f),
   LL(0x9c92b235,0x4baae6e8),LL(0x6b3993a1,0xa73bbd0e),LL(0x693dd031,0xd06d60ec),LL(0x7156881c,0x03cab91b), LL(0x1db3574b,0xd615862f),LL(0x64bb061a,0x485b0185),LL(0xa0181e06,0x27434988),LL(0xc1c0c757,0x2cd61ad4),
   LL(0x2ff9f403,0x3effed5a),LL(0x62239029,0x8dc98d8b),LL(0x1f17b70d,0x2206021e),LL(0xbf510015,0xafbec0ca), LL(0x80130dfa,0x9fed7164),LL(0x8a02dcf5,0x306dc2b5),LL(0xfeb10fc0,0x48f06620),LL(0x5a57cf51,0x78d1e1d5),
   LL(0x192ef710,0xadef8c5a),LL(0x3b7431f9,0x88afbd4b),LL(0x64250c9e,0x7e1f7407),LL(0xb58bec07,0x6e31318d), LL(0x24f89b4e,0xfd4fc4b8),LL(0x48c36a2a,0x65a5dd88),LL(0xf024baa7,0x4f1eccff),LL(0xcba94650,0x22a21cf2),
   LL(0x42a554f7,0x95d29dee),LL(0x002ec4ba,0x828983a5),LL(0x8badb73d,0x8112a1f7),LL(0xa27c1839,0x79ea8897), LL(0xd065fd83,0x8969a5a7),LL(0xb262a0bc,0xf49af791),LL(0xaf2b5127,0xfcdea8b6),LL(0x564c2dbc,0x10e913e1),
   LL(0xbc21ef51,0x51239d14),LL(0x4ce57292,0xe51c3ceb),LL(0x47bbcc3b,0x795ff068),LL(0xbd7e11e6,0x86b46e1e), LL(0x80041ef4,0x0ea6ba23),LL(0x6262342e,0xd72fe505),LL(0x31d294d4,0x8abc6dfd),LL(0x1278c2c9,0xbbe017a2),
   LL(0xb389328a,0xb1fcfa09),LL(0xd01771b5,0x322fbc62),LL(0x60b045bf,0x04c0d063),LL(0x10e52d01,0xdb652edc), LL(0x03ec6627,0x50ef932c),LL(0xc1ee50e3,0xde1b3b2d),LL(0xdc37a90d,0x5ab7bdc5),LL(0x31e33a96,0xfea67213),
   LL(0x4f2999aa,0x6482b5cb),LL(0xb8cbf0dd,0x38476cc6),LL(0x173405bb,0x93ebfacb),LL(0xe52369ec,0x15cdafe7), LL(0xd935b7db,0xd42d5ba4),LL(0x1c99a4cd,0x648b6004),LL(0xa3b5545b,0x785101bd),LL(0x9dd67faf,0x4bf2c38a),
   LL(0x4442449c,0xb1aadc63),LL(0x33ad4fb8,0xe0e9921a),LL(0xaa686d82,0x5c552313),LL(0x465d866c,0xdee635fa), LL(0x18ee6e8a,0xbc3c224a),LL(0xed42e02f,0xeed748a6),LL(0xd474cd08,0xe70f930a),LL(0xfff24adf,0x774ea6ec),
   LL(0xf3480d4a,0x03e2de1c),LL(0xbc8acf1a,0xf0d8edc7),LL(0x68295a9c,0xf23e3303),LL(0xc546a97d,0xfadd5f68), LL(0x96f8acb1,0x895597ad),LL(0x671bdae2,0xbddd49d5),LL(0x21dd43f4,0x16fcd528),LL(0x6619141a,0xa5a45412),
},
/* digit=16 base_pwr=2^112 */
{
   LL(0xc360e25a,0x8ce9b6bf),LL(0x075a1a78,0xe6425195),LL(0x481732f4,0x9dc756a8),LL(0x5432b57a,0x83c0440f), LL(0xd720281f,0xc670b3f1),LL(0xd135e051,0x2205910e),LL(0xdb052be7,0xded14b0e),LL(0xc568ea39,0x697b3d27),
   LL(0xfb3ff9ed,0x2e599b9a),LL(0x17f6515c,0x28c2e0ab),LL(0x474da449,0x1cbee4fd),LL(0x4f364452,0x071279a4), LL(0x01fbe855,0x97abff66),LL(0x5fda51c4,0x3ee394e8),LL(0x67597c0b,0x190385f6),LL(0xa27ee34b,0x6e9fccc6),
   LL(0x14092ebb,0x0b89de93),LL(0x428e240c,0xf17256bd),LL(0x93d2f064,0xcf89a7f3),LL(0xe1ed3b14,0x4f57841e), LL(0xe708d855,0x4ee14405),LL(0x03f1c3d0,0x856aae72),LL(0xbdd7eed5,0xc8e5424f),LL(0x73ab4270,0x3333e4ef),
   LL(0xdda492f8,0x3bc77ade),LL(0x78297205,0xc11a3aea),LL(0x34931b4c,0x5e89a3e7),LL(0x9f5694bb,0x17512e2e), LL(0x177bf8b6,0x5dc349f3),LL(0x08c7ff3e,0x232ea4ba),LL(0xf511145d,0x9c4f9d16),LL(0x33b379c3,0xccf109a3),
   LL(0xa1f25897,0xe75e7a88),LL(0xa1b5d4d8,0x7ac6961f),LL(0x08f3ed5c,0xe3e10773),LL(0x0a892dfb,0x208a54ec), LL(0x78660710,0xbe826e19),LL(0x237df2c8,0x0cf70a97),LL(0xed704da5,0x418a7340),LL(0x08ca33fd,0xa3eeb9a9),
   LL(0x169bca96,0x49d96233),LL(0x2da6aafb,0x04d286d4),LL(0xa0c2fa94,0xc09606ec),LL(0x23ff0fb3,0x8869d0d5), LL(0xd0150d65,0xa99937e5),LL(0x240c14c9,0xa92e2503),LL(0x108e2d49,0x656bf945),LL(0xa2f59e2b,0x152a733a),
   LL(0x8434a920,0xb4323d58),LL(0x622103c5,0xc0af8e93),LL(0x938dbf9a,0x667518ef),LL(0x83a9cdf2,0xa1843073), LL(0x5447ab80,0x350a94aa),LL(0xc75a3d61,0xe5e5a325),LL(0x68411a9e,0x74ba507f),LL(0x594f70c5,0x10581fc1),
   LL(0x80eb24a9,0x60e28570),LL(0x488e0cfd,0x7bedfb4d),LL(0xc259cdb8,0x721ebbd7),LL(0xbc6390a9,0x0b0da855), LL(0xde314c70,0x2b4d04db),LL(0x6c32e846,0xcdbf1fbc),LL(0xb162fc9e,0x33833eab),LL(0xb0dd3ab7,0x9939b48b),
   LL(0xcb0c9c8c,0x5aaa98a7),LL(0x81c4375c,0x75105f30),LL(0x5ef1c90f,0xceee5057),LL(0xc23a17bf,0xb31e065f), LL(0xd4b6d45a,0x5364d275),LL(0x62ec8996,0xd363f3ad),LL(0x4391c65b,0xb5d21239),LL(0xebb41b47,0x84564765),
   LL(0x37107c78,0x20d18ecc),LL(0x570c2a66,0xacff3b6b),LL(0x9bd0d845,0x22f975d9),LL(0xba178fa0,0xef0a0c46), LL(0x76b6028e,0x1a419651),LL(0x248612d4,0xc49ec674),LL(0x7338af55,0x5b6ac4f2),LL(0x7bee5a36,0x06145e62),
   LL(0xe75746b5,0x33e95d07),LL(0xc40c78be,0x1c1e1f6d),LL(0x222ff8e2,0x967833ef),LL(0xb49180ad,0x4bedcf6a), LL(0x3d7a4c8a,0x6b37e9c1),LL(0x6ddfe760,0x2748887c),LL(0xaa3a5bbc,0xf7055123),LL(0x7bbb8e74,0x954ff225),
   LL(0x97c3dfb9,0xc42b8ab1),LL(0xcf168154,0x55a549b0),LL(0xc1b50692,0xad6748e7),LL(0x6fc5cbcb,0x2775780f), LL(0xe1c9d7c8,0x4eab80b8),LL(0x3fdbcd56,0x8c69dae1),LL(0x9969eace,0x47e6b4fb),LL(0xa705cb5a,0x002f1085),
   LL(0x6d3fea55,0x4e23ca44),LL(0xf4810568,0xb4ae9c86),LL(0x2a62f27d,0x47bfb91b),LL(0xd9bac28c,0x60deb4c9), LL(0x7de6c34c,0xa892d894),LL(0x4494587d,0x4ee68259),LL(0x1a3f8a5b,0x914ee14e),LL(0x28700385,0xbb113eaa),
   LL(0x2115b4c9,0x81ca03b9),LL(0x8908cad1,0x7c163d38),LL(0xaa18179a,0xc912a118),LL(0x886e3081,0xe09ed750), LL(0x26f516ca,0xa676e3fa),LL(0x8e732f91,0x753cacf7),LL(0x833da8b4,0x51592aea),LL(0x4cbea8aa,0xc626f42f),
   LL(0xa7b56eaf,0xef9dc899),LL(0x34ef7316,0x00c0e52c),LL(0xfe818a86,0x5b1e4e24),LL(0xc538be47,0x9d31e20d), LL(0x3ed68974,0x22eb932d),LL(0x7c4e87c4,0xe44bbc08),LL(0x0dde9aef,0x4121086e),LL(0x134f4345,0x8e6b9cff),
   LL(0x711b0eb9,0x96892c1f),LL(0x780ab954,0xb905f2c8),LL(0xa20792db,0xace26309),LL(0x0684e126,0xec8ac9b3), LL(0xb40a2447,0x486ad8b6),LL(0x9fe3fb24,0x60121fc1),LL(0x1a8e3b3f,0x5626fccf),LL(0x6ad1f394,0x4e568622),
   LL(0x196aa5a1,0xda7aae0d),LL(0x1041b5fb,0xe0df8c77),LL(0x26b318b7,0x451465d9),LL(0x7ab136e9,0xc29b6e55), LL(0x71148463,0x2c2ab48b),LL(0x64454a76,0xb5738de3),LL(0x5a03abe4,0x54ccf9a0),LL(0x0427d58e,0x377c0296),
   LL(0x2bb39c1f,0x73f5f0b9),LL(0xe608d8c5,0x14373f2c),LL(0x00fbb805,0xdcbfd314),LL(0x83afdcfb,0xdf18fb20), LL(0x42b3523f,0x81a57f42),LL(0x87f650fb,0xe958532d),LL(0x8b0a7d7c,0xaa8dc8b6),LL(0x150166be,0x1b75dfb7),
   LL(0x2d7d1413,0x90e4f7c9),LL(0x9834f597,0x67e2d6b5),LL(0xa808c3e8,0x4fd4f4f9),LL(0xd5281ec1,0xaf8237e0), LL(0x84687cee,0x25ab5fdc),LL(0xa5b26c09,0xc5ded6b1),LL(0xc8ea7650,0x8e4a5aec),LL(0x14cc417f,0x23b73e5c),
   LL(0x3037bf52,0x2bfb4318),LL(0x78c725d7,0xb61e6db5),LL(0xbbb3e5d7,0x8efd4060),LL(0xdbac488e,0x2e014701), LL(0x360aa449,0xac75cf9a),LL(0x79634d08,0xb70cfd05),LL(0xfffb15ef,0xa591536d),LL(0xd07c106c,0xb2c37582),
   LL(0xf50225f9,0xb4293fdc),LL(0xb0e12b03,0xc52e175c),LL(0xd0a8bf64,0xf649c3ba),LL(0xeb8ae3c6,0x745a8fef), LL(0x58321bc3,0x30d7e5a3),LL(0x0bc4df48,0xb1732be7),LL(0xe9ea5058,0x1f217993),LL(0x3e4fd745,0xf7a71cde),
   LL(0x894c5bbb,0x86cc533e),LL(0x69d83082,0x6915c7d9),LL(0x5815c244,0xa6aa2d05),LL(0x49b22ce5,0xaeeee592), LL(0x78135486,0x89e39d13),LL(0x16b76f2f,0x3a275c1f),LL(0xe036e8f5,0xdb6bcc1b),LL(0x5e4709f5,0x4df69b21),
   LL(0x2d0f39aa,0xa188b250),LL(0x15a85947,0x622118bb),LL(0xfde0f4fa,0x2ebf520f),LL(0x4860e539,0xa40e9f29), LL(0x22b57f0f,0x7b6a51eb),LL(0x7e80644a,0x849a33b9),LL(0x1cf095fe,0x50e5d16f),LL(0xec55f002,0xd754b54e),
   LL(0x236f4a98,0x5cfbbb22),LL(0x066800bb,0x0b0c59e9),LL(0x5a9a7774,0x4ac69a8f),LL(0xd6bec948,0x2b33f804), LL(0x32e6c466,0xb3729295),LL(0x4e599c73,0x68956d0f),LL(0x155c31cc,0xa47a249f),LL(0xe1ce284e,0x24d80f0d),
   LL(0x988baf01,0xcd821dfb),LL(0xdbb16647,0xe6331a7d),LL(0x094cb960,0x1eb8ad33),LL(0xc91bbca5,0x593cca38), LL(0x26567456,0x384aac8d),LL(0xc04b6490,0x40fa0309),LL(0xdab6c8f6,0x97834cd6),LL(0x3f91e55f,0x68a7318d),
   LL(0xfc4d3157,0xa00fd04e),LL(0x2bf3bdea,0xb56f8ab2),LL(0x4fa57172,0x014f5648),LL(0x450abdb3,0x948c5860), LL(0x0ebd4f08,0x342b5df0),LL(0x0e82938e,0x3e5168cd),LL(0xb0df5dd0,0x7aedc1ce),LL(0xe5732516,0x6bbbc6d9),
   LL(0x605daaa6,0xc7bfd486),LL(0xbb9a6c9e,0x46fd72b7),LL(0xa124fb89,0xe4847fb1),LL(0xa2d8ffbc,0x75959cbd), LL(0xc8a588ee,0x42579f65),LL(0xb80b499d,0x368c92e6),LL(0x999a5df1,0xea4ef6cd),LL(0x936fe604,0xaa73bb7f),
   LL(0x6457d188,0xf347a70d),LL(0x8b7a388b,0x86eda86b),LL(0x0ccd6013,0xb7cdff06),LL(0xd0053fb2,0xbeb1b6c7), LL(0x99240a9f,0x0b022387),LL(0x776189b2,0x1bbb384f),LL(0x9066193a,0x8695e71e),LL(0x06ffac7e,0x2eb50097),
   LL(0x4a7d2caa,0x0654a9c0),LL(0xa5aaa290,0x6f3fb3d1),LL(0xff476e8f,0x835db041),LL(0xc42295e4,0x540b8b0b), LL(0x05e214f5,0xa5c73ac9),LL(0x56a0b638,0x9a74075a),LL(0xce9e680b,0x2e4b1090),LL(0x6b8d9afa,0x57a5b479),
   LL(0x26bfe65c,0x0dca48e7),LL(0x7290c307,0x097e391c),LL(0x6669e72e,0x683c462e),LL(0x062559ac,0xf505be1e), LL(0xe3a3035a,0x5fbe3ea1),LL(0x9cd50da8,0x6431ebf6),LL(0x1f6407f2,0xfd169d5c),LL(0x60fce6b8,0x8d838a95),
   LL(0x650006f0,0x2a2bfa7f),LL(0x50c0fbb2,0xdfd7dad3),LL(0xccf9ad96,0x92452495),LL(0xd95635f9,0x183bf494), LL(0x4a7bd989,0x02d5df43),LL(0xa5431095,0x505385cc),LL(0xfd43f53e,0xdd98e67d),LL(0x500c34a9,0xd61e1a6c),
   LL(0x4a8a3d62,0x5a4b46c6),LL(0x247743d2,0x8469c4d0),LL(0x88f7e433,0x2bb3a13d),LL(0x01be5849,0x62b23a10), LL(0xa63d1a4c,0xe83596b4),LL(0x7d183f3e,0x454e7fea),LL(0x17afb01c,0x643fce61),LL(0x1c4c3638,0x4e65e5e6),
   LL(0xef74c45b,0x41d85ea1),LL(0xae328506,0x2cfbfa66),LL(0x3ada7da9,0x98b078f5),LL(0xec752fbb,0xd985fe37), LL(0x5a0148b4,0xeece68fe),LL(0x2d78136d,0x6f9a55c7),LL(0xd2b729ce,0x232dccc4),LL(0x90aafbc4,0xa27e0dfd),
   LL(0x12b4603e,0x96474452),LL(0x6b706d14,0xa876c551),LL(0x69a9d412,0xdf145fcf),LL(0x2d479c34,0xe2ab75b7), LL(0x1a23ff97,0x12df9a76),LL(0x5d359d10,0xc6138992),LL(0xfa835f22,0x6e51c7ae),LL(0xc0fcc4d9,0x69a79cb1),
   LL(0x594cc7e1,0xf57f350d),LL(0x3350ab79,0x3079ca63),LL(0x9aff594a,0x226fb614),LL(0x6d59a62b,0x35afec02), LL(0x06ed2c6e,0x9bee46f4),LL(0x7d939a57,0x58da1735),LL(0x8fd1797e,0x44c50402),LL(0x5ccea6ca,0xd8853e7c),
   LL(0xa35fcd5f,0x4065508d),LL(0x495ccaeb,0x8965df8c),LL(0x12e1a962,0x0f2da850),LL(0xc1cf1cc4,0xee471b94), LL(0x0a08fb75,0xcef19bc8),LL(0x81de3591,0x704958f5),LL(0x3aef4f88,0x2867f8b2),LL(0xea9f9a5f,0x8d749384),
   LL(0x8c9049f4,0x1b385537),LL(0x7b92d8b6,0x5be948f3),LL(0xb6e2bd6b,0xd96f725d),LL(0x958c454d,0x37a222bc), LL(0x8809bf61,0xe7c61abb),LL(0x1346f18d,0x46f07fbc),LL(0xe87c0d1c,0xfb567a7a),LL(0x7ef3d07a,0x84a461c8),
   LL(0xd9278d98,0x0a5adce6),LL(0x9dfc73e1,0x24d94813),LL(0x054321c3,0x4f3528b6),LL(0x692ea706,0x2e03fdde), LL(0x47b533c0,0x10e60619),LL(0x2ca3c055,0x1a8bc73f),LL(0x1bb62b8f,0xae58d4b2),LL(0x584a24e3,0xb2045a73),
   LL(0xbd76e195,0x3ab3d5af),LL(0x6938a810,0x478dd1ad),LL(0x6ee3d5cb,0x6ffab393),LL(0x22b361e4,0xdfb693db), LL(0x51dbf1a7,0xf9694496),LL(0x08a2e762,0xcab4b4ef),LL(0xd39bba9a,0xe8c92f25),LL(0xf1464d96,0x850e61bc),
   LL(0xdc09508b,0xb7e830e3),LL(0x74317655,0xfaf6d2cf),LL(0xdf690355,0x72606ceb),LL(0xd0c3ded6,0x48bb92b3), LL(0x5c7cf892,0x65b75484),LL(0xd5d5f01f,0xf6cd7ac9),LL(0x96401d69,0xc2c30a59),LL(0xed921878,0x91268650),
   LL(0xb78c558f,0x380bf913),LL(0xc8afdaa9,0x43c0baeb),LL(0x54f169d3,0x377f61d5),LL(0xae5ff20b,0xf8da07e3), LL(0xa8a90ea8,0xb676c49d),LL(0x83a29b21,0x81c1ff2b),LL(0x2ad8d276,0x383297ac),LL(0xba89f982,0x3001122f),
   LL(0x6718e448,0xe1d794be),LL(0x7c3e6e13,0x246c1482),LL(0x5d26b5ef,0x56646ef8),LL(0x88069cdd,0x80f5091e), LL(0x724bdd38,0xc5992e2f),LL(0x8471e8c7,0x02e915b4),LL(0x0d0ff2a9,0x96ff320a),LL(0x4384d1a0,0xbf886487),
   LL(0xc93f72d6,0xbbe1e6a6),LL(0xcad800ea,0xd5f75d12),LL(0xe7acf117,0xfa40a09f),LL(0x7581a355,0x32c8cdd5), LL(0x7023c499,0x74221992),LL(0x38ec3901,0xa8afe5d7),LL(0xa90e83f0,0x5691afcb),LL(0x0b8f8eac,0x41bcaa03),
   LL(0x8d2668d5,0xe38b5ff9),LL(0x7ad81965,0x0715281a),LL(0x03c6ce11,0x1bc8fc7c),LL(0x8b650436,0xcbbee6e2), LL(0x0cdb9808,0x06b00fe8),LL(0xfe3ed315,0x17d6e066),LL(0x4d0b5018,0x2e9d38c6),LL(0x844dcaef,0xab8bfd56),
   LL(0x513aed8b,0x42894a59),LL(0x314bd07a,0xf77f3b6d),LL(0x8e42b582,0xbbdecb8f),LL(0xd2390fe6,0xf10e2fa8), LL(0x62a2f201,0xefb95022),LL(0x50ee32b0,0x4d59ea50),LL(0x6da789a8,0xd87f7728),LL(0xf79492c4,0xcf98a2cf),
   LL(0x720943c2,0xf9577239),LL(0x3990b9d0,0xba044cf5),LL(0x95f2884a,0x5aa8e823),LL(0x0278a0af,0x834de6ed), LL(0x5f25bd12,0xc8e1ee9a),LL(0x6f7ab271,0x9259ceaa),LL(0x77d00b76,0x7e6d97a2),LL(0xa437832a,0x5c0c6eea),
   LL(0x5606b81d,0x5232c20f),LL(0x0d991ee5,0xabd7b375),LL(0x8632d951,0x4d2bfe35),LL(0x98ed9364,0x78f85146), LL(0xf30c3282,0x951873f0),LL(0xa789230b,0x0da8ac80),LL(0x5398967f,0x3ac7789c),LL(0xbdda0fb5,0xa69b8f7f),
   LL(0x6add8545,0xe5db7717),LL(0x72c49b66,0x1b71cb66),LL(0x68421d77,0xd8560739),LL(0x83e3afea,0x03840fe8), LL(0x1ec69977,0xb391dad5),LL(0x307f6726,0xae243fb9),LL(0xe8ca160c,0xc88ac87b),LL(0x4ce355f4,0x5174cced),
   LL(0xe58ba37d,0x98a35966),LL(0x7817335d,0xfdcc8da2),LL(0x83fbc7bf,0x5b752830),LL(0xd9c96984,0x68e419d4), LL(0x02a40380,0x409a39f4),LL(0x1fe977bc,0x88940faf),LL(0x8f8edea6,0xc640a94b),LL(0xed11547d,0x1e22cd17),
   LL(0x59ffc3e2,0xe28568ce),LL(0xc1dee4e7,0x60aa1b55),LL(0x837cb363,0xc67497c8),LL(0x105a2bf2,0x06fb438a), LL(0x500d8e20,0x30357ec4),LL(0x0670db10,0x1ad9095d),LL(0xc73b7cfd,0x7f589a05),LL(0x880d6d28,0xf544607d),
   LL(0xa20ef103,0x17ba93b1),LL(0x6ba6577b,0xad859130),LL(0x6fa214a0,0x65c91cf6),LL(0x27990da5,0xd7d49c6c), LL(0x20bb569d,0xecd9ec8d),LL(0xeeffbc33,0xbd4b2502),LL(0x6bed0467,0x2056ca5a),LL(0x5b63728c,0x7916a1f7),
   LL(0x53a4f566,0xd4f9497d),LL(0x97b56810,0x89734664),LL(0x0494a621,0xf8e1da74),LL(0x8d011c68,0x82546a93), LL(0xc61ac162,0x1f3acb19),LL(0xabad0d3e,0x52f8fa9c),LL(0xb4b7ea43,0x15356523),LL(0xae608125,0x5a16ad61),
   LL(0x4faed184,0xb0bcb87f),LL(0x5029f45f,0x5f236b1d),LL(0x0bc6b1fc,0xd42c7607),LL(0x68aefce3,0xc644324e), LL(0x5c5d8446,0x8e191d59),LL(0x13ae1979,0xc0208077),LL(0x3ba59cc7,0xadcaee55),LL(0xa2cb81ba,0x20ed6d6b),
   LL(0xb6efcffc,0x0952ba19),LL(0x97c0b87c,0x60f12d68),LL(0x9caa30bc,0x4ee2c7c4),LL(0x97fbff4e,0x767238b7), LL(0x501b5d92,0xebc73921),LL(0xc2a37737,0x3279e3df),LL(0x6d197543,0x9fc12bc8),LL(0x0a40db4e,0xfa94dc6f),
   LL(0x530ccbbd,0x7392b41a),LL(0xea823525,0x87c82146),LL(0x05d98d0c,0xa52f984c),LL(0x5ef6974c,0x2ae57d73), LL(0x3042a6dd,0x9377f7bf),LL(0x19647a64,0xb1a007c0),LL(0x0cca9767,0xfaa9079a),LL(0xf68f72d5,0x3d81a25b),
   LL(0xff81578e,0x752067f8),LL(0x9045447d,0x78622150),LL(0x0505aa6f,0xc0c22fcf),LL(0x6bed1c77,0x1030f0a6), LL(0x1f0bd739,0x31f29f15),LL(0xe6debe85,0x2d7989c7),LL(0x8e677e98,0x5c070e72),LL(0x06e81fd5,0x0a817bd3),
   LL(0xb0f2ac95,0xc110d830),LL(0xab20e64e,0x48d0995a),LL(0x7729cd9a,0x0f3e00e1),LL(0xdd556946,0x2a570c20), LL(0x4e86214d,0x912dbcfd),LL(0xcf615498,0x2d014ee2),LL(0x3530d76e,0x55e2b1e6),LL(0xfd0fd6d1,0xc5135ae4),
   LL(0xd4f3049f,0x0066273a),LL(0xe7087477,0xbb8e9893),LL(0x14c6e5fd,0x2dba1ddb),LL(0x51f57e6c,0xdba37886), LL(0x5a72f2cf,0x5aaee0a6),LL(0x7bea5642,0x1208bfbf),LL(0x67872c37,0xf5c6aa3b),LL(0x43f93224,0xd726e083),
   LL(0x061f1658,0x1854daa5),LL(0xdf0cd2b3,0xc0016df1),LL(0x833d50de,0xc2a3f23e),LL(0xbbbd3017,0x73b681d2), LL(0x3ac343c0,0x2f046dc4),LL(0x85716421,0x9c847e7d),LL(0x0917eed4,0xe1e13c91),LL(0x63a1b9c6,0x3fc9eebd),
   LL(0x7fe02299,0x0f816a72),LL(0x294f3319,0x6335ccc2),LL(0x4745c5be,0x3820179f),LL(0x922f066e,0xe647b782), LL(0x02cafb8a,0xc22e49de),LL(0xfcc2eccc,0x299bc2ff),LL(0x6e0e8282,0x9a8feea2),LL(0xfe893205,0xa627278b),
   LL(0x7933e47b,0xa7e19733),LL(0x2e766402,0xf4ff6b13),LL(0x98440d9f,0xa4d8be0a),LL(0x38938808,0x658f5c2f), LL(0xc95b3b3e,0x90b75677),LL(0x3137b6ff,0xfa044269),LL(0x43c47c29,0x077b039b),LL(0x8a6445b2,0xcca95dd3),
   LL(0x2333fc4c,0x0b498ba4),LL(0xf736a1b1,0x274f8e68),LL(0x5f1d4b2e,0x6ca348fd),LL(0xa8f10199,0x24d3be78), LL(0xca14f530,0x8535f858),LL(0x5b982e51,0xa6e7f163),LL(0x36e1bf62,0x847c8512),LL(0x03448418,0xf6a7c58e),
   LL(0xf9374ab6,0x583f3703),LL(0x6e564145,0x864f9195),LL(0x22526d50,0x33bc3f48),LL(0x1262a496,0x9f323c80), LL(0x3f046a9a,0xaa97a7ae),LL(0xdf8a039a,0x70da183e),LL(0x52aa0ba6,0x5b68f71c),LL(0x21459c2d,0x9be0fe51),
   LL(0xcbc613e5,0xc1e17eb6),LL(0x497ea61c,0x33131d55),LL(0xaf7eded5,0x2f69d39e),LL(0xde6af11b,0x73c2f434), LL(0xa4a375fa,0x4ca52493),LL(0xb833c5c2,0x5f06787c),LL(0x3e6e71cf,0x814e091f),LL(0x8b746666,0x76451f57),
},
/* digit=17 base_pwr=2^119 */
{
   LL(0x694db7e0,0x80f9bdef),LL(0xb9fcddc6,0xedca8787),LL(0x03b8dce1,0x51981c34),LL(0x70e10ba1,0x4274dcf1), LL(0x6def6d1a,0xf72743b8),LL(0xebdb1866,0xd25b1670),LL(0x050c6f58,0xc4491e8c),LL(0x87fbd7f5,0x2be2b2ab),
   LL(0xd111f8ec,0x3e0e5c9d),LL(0xb7c4e760,0xbcc33f8d),LL(0xbd392a51,0x702f9a91),LL(0xc132e92d,0x7da4a795), LL(0x0bb1151b,0x1a0b0ae3),LL(0x02e32251,0x54febac8),LL(0x694e9e78,0xea3a5082),LL(0xe4fe40b8,0xe58ffec1),
   LL(0xd1e0cf9e,0xf85592fc),LL(0xc0e7b2e8,0xdea75f0d),LL(0xc135584e,0xc04215cf),LL(0x2f57092a,0x174fc727), LL(0xeb930bea,0xe7277877),LL(0x5eb02a5a,0x504caccb),LL(0xf5241b9b,0xf9fe08f7),LL(0x8d5ca954,0xe7fb62f4),
   LL(0x29c4120b,0xfbb8349d),LL(0xc0d0d915,0x9f94391f),LL(0x5410ba51,0xc4074fa7),LL(0x150a5911,0xa66adbf6), LL(0x34bfca38,0xc164543c),LL(0xb9e1ccfc,0xe0f27560),LL(0xe820219c,0x99da0f53),LL(0xc6b4997a,0xe8234498),
   LL(0x9d4c5423,0xcfb88b76),LL(0xb0521c49,0x9e56eb10),LL(0xbe8700a1,0x418e0b5e),LL(0xf93cb58a,0x00cbaad6), LL(0xd92a5e67,0xe923fbde),LL(0x1f347f11,0xca4979ac),LL(0x6bc0585b,0x89162d85),LL(0xac3c70e3,0xdd6254af),
   LL(0x516e19e4,0x7b23c513),LL(0xc5c4d593,0x56e2e847),LL(0x5ce71ef6,0x9f727d73),LL(0xf79a44c5,0x5b6304a6), LL(0x3ab7e433,0x6638a736),LL(0xfe742f83,0x1adea470),LL(0x5b7fc19f,0xe054b854),LL(0xba1d0698,0xf935381a),
   LL(0x799e9a74,0x546eab2d),LL(0xa949f729,0x96239e0e),LL(0x7090055a,0xca274c6b),LL(0x9020c9b0,0x835142c3), LL(0xa2e8807f,0xa405667a),LL(0x1aa3d39e,0x29f2c085),LL(0x42fc72f5,0xcc555d64),LL(0xfbeacb3c,0xe856e0e7),
   LL(0x918e4936,0xb5504f9d),LL(0xb2513982,0x65035ef6),LL(0x6f4d9cb9,0x0553a0c2),LL(0xbea85509,0x6cb10d56), LL(0xa242da11,0x48d957b7),LL(0x672b7268,0x16a4d3dd),LL(0x8502a96b,0x3d7e637c),LL(0x730d463b,0x27c7032b),
   LL(0xe4136a14,0xbdc02b18),LL(0x678e32bf,0xbacf969d),LL(0xdd9c3c03,0xc98d89a3),LL(0x23becc4f,0x7b92420a), LL(0xc64d565c,0xd4b41f78),LL(0x10f28295,0x9f969d00),LL(0xb13d051a,0xec7f7f76),LL(0xa92da585,0x08945e1e),
   LL(0x5846426f,0x55366b7d),LL(0x247d441d,0xe7d09e89),LL(0x736fbf48,0x510b404d),LL(0xe784bd7d,0x7fa003d0), LL(0x17fd9596,0x25f7614f),LL(0x35cb98db,0x49e0e0a1),LL(0x2e83a76a,0x2c65957b),LL(0xcddbe0f8,0x5d40da8d),
   LL(0x050bad24,0xf2b8c405),LL(0xc2aa4823,0x8918426d),LL(0xa38365a7,0x2aeab3dd),LL(0x7c91b690,0x72031717), LL(0x60a94120,0x8b00d699),LL(0xe99eaeec,0x478a255d),LL(0x6f60aafd,0xbf656a5f),LL(0x5dee77b3,0xdfd7cb75),
   LL(0xa595939d,0x37f68bb4),LL(0x28740217,0x03556479),LL(0x84ad7612,0x8e740e7c),LL(0x9044695f,0xd89bc843), LL(0x85a9184d,0xf7f3da5d),LL(0x9fc0b074,0x562563bb),LL(0xf88a888e,0x06d2e6aa),LL(0x161fbe7c,0x612d8643),
   LL(0xf64085e7,0x465edba7),LL(0x29aa8511,0xb230f304),LL(0xcda2d188,0x53388426),LL(0x4b666649,0x90885735), LL(0x652f54f6,0x6f02ff9a),LL(0x5fae2bf0,0x65c82294),LL(0x62f5eee3,0x7816ade0),LL(0xfcc56d70,0xdcdbdf43),
   LL(0x54530bb2,0x9fb3bba3),LL(0xcb0869ea,0xbde3ef77),LL(0x0b431163,0x89bc9046),LL(0xe4819a35,0x4d03d7d2), LL(0x43b6a782,0x33ae4f9e),LL(0x9c88a686,0x216db307),LL(0x00ffedd9,0x91dd88e0),LL(0x12bd4840,0xb280da9f),
   LL(0x1635e741,0x32a7cb8a),LL(0x78be02a7,0xfe14008a),LL(0x1b7ae030,0x3fafb334),LL(0x5add0ce9,0x7fd508e7), LL(0xd607ad51,0x72c83219),LL(0x8d40964a,0x0f229c0a),LL(0x1c878da2,0x1be2c336),LL(0xeab2ab86,0xe0c96742),
   LL(0x3e538cd7,0x458f8691),LL(0x8e08ad53,0xa7001f6c),LL(0xbf5d15ff,0x52b8c6e6),LL(0x011215dd,0x548234a4), LL(0x3d5b4045,0xff5a9d2d),LL(0x4a904190,0xb0ffeeb6),LL(0x48607f8b,0x55a3aca4),LL(0x30a0672a,0x8cbd665c),
   LL(0x42583068,0x87f834e0),LL(0xf3f6e683,0x02da2aeb),LL(0x05c12248,0x6b763e5d),LL(0x65a8aefc,0x7230378f), LL(0x71e8e5ca,0x93bd80b5),LL(0xb3b62524,0x53ab041c),LL(0x6c9c552e,0x1b860513),LL(0xd5524e66,0xe84d402c),
   LL(0xf37f5937,0xa37f3573),LL(0xd1e4fca5,0xeb0f6c7d),LL(0xac8ab0fc,0x2965a554),LL(0x274676ac,0x17fbf56c), LL(0xacf7d720,0x2e2f6bd9),LL(0x10224766,0x41fc8f88),LL(0x85d53bef,0x517a14b3),LL(0x7d76a7d1,0xdae327a5),
   LL(0xc4818267,0x6ad0a065),LL(0x37c1bbc1,0x33aa189b),LL(0x27392a92,0x64970b52),LL(0x2d1535ea,0x21699a1c), LL(0xc2d7a7fd,0xcd20779c),LL(0x99c83cf2,0xe3186059),LL(0x72c0b8c7,0x9b69440b),LL(0x7b9e0e4d,0xa81497d7),
   LL(0x1f5f82dc,0x515d5c89),LL(0x6361079e,0x9a7f67d7),LL(0x11a35330,0xa8da81e3),LL(0x4b18be1b,0xe44990c4), LL(0xaf103e59,0xc7d5ed95),LL(0x8dac9261,0xece8aba7),LL(0x9394b8d3,0xbe82b099),LL(0x16adfe83,0x6830f09a),
   LL(0x88172d01,0x250a29b4),LL(0xcaff9e02,0x8b20bd65),LL(0xe8a6329a,0xb8a7661e),LL(0xd3fce920,0x4520304d), LL(0x2b47f7ef,0xae45da1f),LL(0x5bffc540,0xe07f5288),LL(0x3464f874,0xf7997009),LL(0xa6fa1f38,0x2244c2cd),
   LL(0x94d7d9b1,0x43c41ac1),LL(0xc82e7f17,0x5bafdd82),LL(0x5fda0fca,0xdf0614c1),LL(0xa8ae37ad,0x74b043a7), LL(0x9e71734c,0x3ba6afa1),LL(0x9c450f2e,0x15d5437e),LL(0x67e242b1,0x4a5883fe),LL(0x2c1953c2,0x5143bdc2),
   LL(0xfc5e8920,0x542b8b53),LL(0x9a9cee08,0x363bf9a8),LL(0xc3486e08,0x02375f10),LL(0x8c5e70d2,0x2037543b), LL(0x625640b4,0x7109bccc),LL(0x8bc62c3b,0xcbc1051e),LL(0x803f26ea,0xf8455fed),LL(0xeb372424,0x6badceab),
   LL(0x6b53f5f9,0xa2a9ce7c),LL(0x1b176d99,0x64246595),LL(0xb95c081b,0xb1298d36),LL(0x1d9a9ee6,0x53505bb8), LL(0xf2ba70b0,0x3f6f9e61),LL(0x8afad453,0xd07e16c9),LL(0xe7eb4a6a,0x9f1694bb),LL(0x3cb0bc8e,0xdfebced9),
   LL(0x53868c8b,0x92d3dcdc),LL(0x386107a6,0x174311a2),LL(0x689b4e64,0x4109e07c),LL(0x2df3dcb6,0x30e4587f), LL(0x0811b3b2,0x841aea31),LL(0x0cce43ea,0x6144d41d),LL(0x2a9a7803,0x464c4581),LL(0x3e158930,0xd03d371f),
   LL(0xb1f3390b,0xc676d7f2),LL(0xa5b61272,0x9f7a1b8c),LL(0xc2e127a9,0x4ebebfc9),LL(0x5dd997bf,0x4602500c), LL(0x4711230f,0x7f09771c),LL(0x020f09c1,0x058eb37c),LL(0xfee5e38b,0xab693d4b),LL(0x4653cbc0,0x9289eb1f),
   LL(0xd51b9cf5,0xbecf46ab),LL(0x9f0121af,0xd2aa9c02),LL(0xe90dc274,0x36aaf7d2),LL(0x48b95a3c,0x909e4ea0), LL(0x6f32dbdb,0xe6b70496),LL(0x8b030b3e,0x672188a0),LL(0xcfb617e2,0xeeffe5b3),LL(0x7c82709e,0x87e947de),
   LL(0x1770f5a7,0xa44d2b39),LL(0x0e44eb82,0xe4d4d791),LL(0x3f69712a,0x42e69d1e),LL(0xac6a820e,0xbf11c4d6), LL(0x42c4224c,0xb5e7f3e5),LL(0x449d941c,0xd6b4e81c),LL(0x5450e878,0x5d72bd16),LL(0xee25ac54,0x6a61e28a),
   LL(0xe6f1cd95,0x33272094),LL(0x0d18673f,0x7512f30d),LL(0x5afc1464,0x32f7a4ca),LL(0x6bbb977b,0x2f095656), LL(0xa8226200,0x586f47ca),LL(0x1ac07369,0x02c868ad),LL(0xc613acbe,0x4ef2b845),LL(0x0386054c,0x43d7563e),
   LL(0xab952578,0x54da9dc7),LL(0x26e84d0b,0xb5423df2),LL(0x9b872042,0xa8b64eeb),LL(0x5990f6df,0xac205782), LL(0x21f4c77a,0x4ff696eb),LL(0xaab273af,0x1a79c3e4),LL(0x9436b3f1,0x29bc922e),LL(0xd6d9a27a,0xff807ef8),
   LL(0x778f22a0,0x82acea3d),LL(0x5b5e7469,0xfb10b2e8),LL(0x2818ee7d,0xc0b16980),LL(0xc91c1a2f,0x011afff4), LL(0xad124418,0x95a6d126),LL(0xe72e295f,0x31c081a5),LL(0xf2f4db75,0x36bb283a),LL(0x7acef462,0xd115540f),
   LL(0x33f6746c,0xc7f3a8f8),LL(0xfea990ca,0x21e46f65),LL(0xcaddb0a9,0x915fd5c5),LL(0x78614555,0xbd41f016), LL(0x426ffb58,0x346f4434),LL(0x14dbc204,0x80559436),LL(0x5a969b7f,0xf3dd20fe),LL(0xe899a39a,0x9d59e956),
   LL(0x8ad4cf4b,0xf1b0971c),LL(0x2ffb8fb8,0x03448860),LL(0x65340ba4,0xf071ac3c),LL(0xb27fd758,0x408d0596), LL(0x98c364b0,0xe7c78ea4),LL(0x051e8ab5,0xa4aac4a5),LL(0x485d9002,0xb9e1d560),LL(0x88844455,0x9acd518a),
   LL(0xd06f56c0,0xe4ca688f),LL(0xdf027972,0xa48af70d),LL(0x5e9a609d,0x691f0f04),LL(0xee61270e,0xa9dd82cd), LL(0xa0ef18d3,0x8903ca63),LL(0x3d6ca3bd,0x9fb7ee35),LL(0xabf47d03,0xa7b4a09c),LL(0x1c67de8e,0x4cdada01),
   LL(0x9355a244,0x52003749),LL(0x4f2151a9,0xe77fd2b6),LL(0x66b4efcb,0x695d6cf6),LL(0xda2cfe25,0xc5a0cacf), LL(0xef811865,0x104efe5c),LL(0x9ea5cc3d,0xf52813e8),LL(0x40b58dbc,0x855683dc),LL(0x175fcb11,0x0338ecde),
   LL(0x74921592,0xf9a05637),LL(0xb9bb9d31,0xb4f1261d),LL(0x4e9c5459,0x551429b7),LL(0x6ea71f53,0xbe182e6f), LL(0xdfc50573,0xd3a3b07c),LL(0x62be8d44,0x9ba1afda),LL(0x52ab65d3,0x9bcfd2cb),LL(0xa9571802,0xdf11d547),
   LL(0x02a2404a,0x099403ee),LL(0x21088a71,0x497406f4),LL(0x5004ae71,0x99479409),LL(0xa812c362,0xbdb42078), LL(0xd8828442,0x2b72a30f),LL(0xfcb5ed1c,0x283add27),LL(0x66a40015,0xf7c0e200),LL(0x08b295ef,0x3e3be641),
   LL(0xe038a675,0xac127dc1),LL(0x8c5c6320,0x729deff3),LL(0xa90d2c53,0xb7df8fd4),LL(0x681e7cd3,0x9b74b0ec), LL(0xdab407e5,0x5cb5a623),LL(0x76b340c6,0xcdbd3615),LL(0x7d28392c,0xa184415a),LL(0xe96f7830,0xc184c1d8),
   LL(0x81d3a80f,0xc3204f19),LL(0xc8e02432,0xfde0c841),LL(0x8149e0c1,0x78203b3e),LL(0x08053a73,0x5904bdbb), LL(0x101b6805,0x30fc1dd1),LL(0x49aa6d49,0x43c223bc),LL(0x7a174087,0x9ed67141),LL(0xd5997008,0x311469a0),
   LL(0x5e43fc61,0xb189b684),LL(0xe0d3ab57,0xf3282375),LL(0xb1181da8,0x4fa34b67),LL(0x99ee52b8,0x621ed0b2), LL(0xad990676,0x9b178de1),LL(0x56d54065,0xd51de67b),LL(0x7538c201,0x2a2c27c4),LL(0x38a40f5c,0x33856ec8),
   LL(0xbe6cdcde,0x2522fc15),LL(0x9f0c6f89,0x1e603f33),LL(0x103e30a6,0x7994edc3),LL(0x220c853e,0x033a00db), LL(0xf7bb7fd7,0xd3cfa409),LL(0x462d18f6,0x70f8781e),LL(0x687fe295,0xbbd82980),LL(0x595669f3,0x6eef4c32),
   LL(0x2f7e85c3,0x86a9303b),LL(0x71988f9b,0x5fce4621),LL(0xc138acb5,0x5b935bf6),LL(0x25661212,0x30ea7d67), LL(0xe51ab9a2,0xef1eb5f4),LL(0xae067c78,0x0587c98a),LL(0x77ca9ca6,0xb3ce1b3c),LL(0x54b5f057,0x2a553d4d),
   LL(0x4da29ec2,0xc7898236),LL(0xb9c57316,0xdbdd5d13),LL(0x2cd80d47,0xc57d6e6b),LL(0xfe9e7391,0x80b460cf), LL(0xf963c31e,0x98648cab),LL(0xcc4d32fd,0x67f9f633),LL(0xfdf7c687,0x0af42a9d),LL(0x0b015ea7,0x55f292a3),
   LL(0xcd21ab3d,0x89e468b2),LL(0xc393d392,0xe504f022),LL(0xa5013af9,0xab21e1d4),LL(0xc2c28acb,0xe3283f78), LL(0x226bf99f,0xf38b35f6),LL(0x0e291e69,0xe8354274),LL(0xb20c162d,0x61673a15),LL(0xb04fbdbe,0xc101dc75),
   LL(0x255bd617,0x8323b4c2),LL(0x6c2a9154,0x6c969693),LL(0x62679387,0xc6e65860),LL(0xb8c88e23,0x8e01db0c), LL(0x893a5559,0x33c42873),LL(0x47a3e149,0x7630f04b),LL(0xddcf35f8,0xb5d80805),LL(0x77dfe732,0x582ca080),
   LL(0x0b1894a0,0x2c7156e1),LL(0xd81c68c0,0x92034001),LL(0xc8b115b5,0xed225d00),LL(0x83b907f2,0x237f9c22), LL(0x4470e2c0,0x0ea2f32f),LL(0x58be4e95,0xb725f7c1),LL(0xb1ae5463,0x0f1dcafa),LL(0x1ba2fc04,0x59ed5187),
   LL(0xd0115d4d,0xf6e0f316),LL(0xd3691599,0x5180b12f),LL(0x527f0a41,0x157e32c9),LL(0xa8e0ecc0,0x7b0b081d), LL(0xbf4f0dd0,0x6dbaaa8a),LL(0x4d252696,0x99b289c7),LL(0xdbf864fe,0x79b7755e),LL(0x76cad3ab,0x6974e2b1),
   LL(0x06ddd657,0x35dbbee2),LL(0x2ff3a96d,0xe7cbdd11),LL(0x076be758,0x88381968),LL(0x08c91f5d,0x2d737e72), LL(0x86ec3776,0x5f83ab62),LL(0x945fa7a1,0x98aa649d),LL(0x72ef0933,0xf477ec37),LL(0x098c17b1,0x66f52b1e),
   LL(0xd803738b,0x9eec58fb),LL(0xe4e86aa4,0x91aaade7),LL(0xa5b51492,0x6b1ae617),LL(0xbbc45974,0x63272121), LL(0x862c5129,0x7e0e28f0),LL(0x3321a4a0,0x0a8f79a9),LL(0x5041c88f,0xe26d1664),LL(0x53233e3a,0x0571b805),
   LL(0xc9520711,0xd1b0ccde),LL(0x3c8b84bf,0x55a9e4ed),LL(0xa1fef314,0x9426bd39),LL(0x6eb93f2b,0x4f5f638e), LL(0x2bf9341b,0xba2a1ed3),LL(0x4d42d5a9,0xd63c1321),LL(0x316dc7c5,0xd2964a89),LL(0xca511851,0xd1759606),
   LL(0xf9e6ed35,0xd8a9201f),LL(0x6736925a,0xb7b5ee45),LL(0x99581af7,0x0a83fbbc),LL(0x64eeb051,0x3076bc40), LL(0x02dec312,0x5511c98c),LL(0x238dcb78,0x270de898),LL(0x539c08c9,0x2cf4cf9c),LL(0x38d3b06e,0xa70cb65e),
   LL(0xcfe57bbd,0xb12ec10e),LL(0x35a0c2b5,0x82c7b656),LL(0x161c67bd,0xddc7d5cd),LL(0xae3a32cc,0xe32e8985), LL(0xd11a5529,0x7aba9444),LL(0x2427fa1a,0xe964ed02),LL(0x24a1770a,0x1528392d),LL(0x12c72fcd,0xa152ce2c),
   LL(0x8ec07649,0x714553a4),LL(0x459dd453,0x18b4c290),LL(0x7b64b110,0xea32b714),LL(0x2e6f07a2,0xb871bfa5), LL(0x9e2e3c9b,0xb67112e5),LL(0x44aa90f6,0xfbf250e5),LL(0xbd539006,0xf77aedb8),LL(0xd172a66f,0x3b0cdf9a),
   LL(0xf8c51187,0xedf69fea),LL(0x741e4da7,0x05bb67ec),LL(0x08114345,0x47df0f32),LL(0xbb9792b1,0x56facb07), LL(0x8f6229e4,0xf3e007e9),LL(0x526fba0f,0x62d103f4),LL(0xb0339d79,0x4f33bef7),LL(0xb59bfec1,0x9841357b),
   LL(0xc34e6705,0xfa8dbb59),LL(0x7fdaa84c,0xc3c7180b),LL(0xa4108537,0xf95872fc),LL(0x932a3e5a,0x8750cc3b), LL(0xb7275d7d,0xb61cc69d),LL(0x2e59b2e9,0xffa0168b),LL(0x6ecbb493,0xca032abc),LL(0x2c9082d8,0x1d86dbd3),
   LL(0xe28ef5ba,0xae1e0b67),LL(0xcb18e169,0x2c9a4699),LL(0x1e6bbd20,0x0ecd0e33),LL(0xaf5e81d2,0x571b360e), LL(0x101c1d45,0xcd9fea58),LL(0x18880452,0x6651788e),LL(0x1f8dd446,0xa9972635),LL(0xe37281d0,0x44bed022),
   LL(0x33da525d,0x094b2b2d),LL(0x13144fd8,0xf193678e),LL(0xf4c1061d,0xb8ab5ba4),LL(0xdccbe0f4,0x4343b5fa), LL(0x63812713,0xa8702371),LL(0xf7611d93,0x47bf6d2d),LL(0xbd21e1d7,0x46729b8c),LL(0xd629e77d,0x7484d4e0),
   LL(0x60dbac1f,0x830e6eea),LL(0xda06a2f7,0x23d8c484),LL(0x50ca535b,0x896714b0),LL(0xebd97a9b,0xdc8d3644), LL(0xb12177b4,0x106ef9fa),LL(0x534d5d9c,0xf79bf464),LL(0xa6ab360b,0x2537a349),LL(0xa00c744f,0xc7c54253),
   LL(0xe5911a76,0xb3c7a047),LL(0x647f1ee7,0x61ffa5c8),LL(0x8f56ab42,0x15aed36f),LL(0xa3ff9ac9,0x6a0d41b0), LL(0xcc30d357,0x68f469f5),LL(0x6b72be96,0xbe9adf81),LL(0x903ad461,0x1cd926fe),LL(0xcaca441b,0x7e89e38f),
   LL(0xfacf69d4,0xf0f82de5),LL(0x4775344c,0x363b7e76),LL(0xb2e36d04,0x6894f312),LL(0x11d1c9a5,0x3c6cb4fe), LL(0x4008e1f2,0x85d9c339),LL(0x249f326c,0x5e9a85ea),LL(0x678c5e06,0xdc35c60a),LL(0x9f86fba9,0xc08b944f),
   LL(0x89f71f0f,0xde40c02c),LL(0xff3da3c0,0xad8f3e31),LL(0x42125ded,0x3ea5096b),LL(0xa7379183,0x13879cbf), LL(0x6b306a0b,0x6f4714a5),LL(0x67646c5e,0x359c2ea6),LL(0x07726368,0xfacf8943),LL(0x65ff431e,0x07a58935),
   LL(0x68754ab0,0x24d661d1),LL(0x6f429a76,0x801fce1d),LL(0xa58ce769,0xc068a85f),LL(0x5d5eca2b,0xedc35c54), LL(0xa3f660d1,0xea31276f),LL(0xb8fc7167,0xa0184ebe),LL(0x1d8db0ae,0x0f20f21a),LL(0x56c35e12,0xd96d095f),
   LL(0xf8c2a25b,0xedf402b5),LL(0x059204b6,0x1bb772b9),LL(0x19b4e34c,0x50cbeae2),LL(0x3fa0845a,0x93109d80), LL(0x8ef59fb5,0x54f7ccf7),LL(0x88070963,0x3b438fe2),LL(0x31f3ba9b,0x9e28c659),LL(0xead9da92,0x9cc31b46),
   LL(0xb733aa5f,0x3c2f0ba9),LL(0xf05af235,0xdece47cb),LL(0xa2ac82a5,0xf8e3f715),LL(0x2203f18a,0xc97ba641), LL(0x09c11060,0xc3af5504),LL(0x46af512d,0x56ea2c05),LL(0xf3f28146,0xfac28daf),LL(0x959ef494,0x87fab43a),
},
/* digit=18 base_pwr=2^126 */
{
   LL(0xd4c5105f,0x09891641),LL(0x6d7fbd65,0x1ae80f8e),LL(0xbee6bdb0,0x9d67225f),LL(0x7fc4d860,0x3b433b59), LL(0x93e85638,0x44e66db6),LL(0xe3e9862f,0xf7b59252),LL(0x665c32ec,0xdb785157),LL(0xae362f50,0x702fefd7),
   LL(0x0fefb0c3,0x3754475d),LL(0x46d7c35d,0xd48fb56b),LL(0x363798a4,0xa070b633),LL(0x8fdb98e6,0xae89f3d2), LL(0x6363d14c,0x970b89c8),LL(0x67abd27d,0x89817521),LL(0x44d5a021,0x9bf7d474),LL(0xcac72aee,0xb3083baf),
   LL(0xbe949a44,0x389741de),LL(0x546a4fa5,0x638e9388),LL(0xa0047bdc,0x3fe6419c),LL(0xaaea57ca,0x7047f648), LL(0x41fbab17,0x54e48a90),LL(0x576bdba2,0xda8e0b28),LL(0xc72afddc,0xe807eebc),LL(0xf42577bf,0x07d3336d),
   LL(0xbfe20925,0x62a8c244),LL(0x8fdce867,0x91c19ac3),LL(0xdd387063,0x5a96a5d5),LL(0x21d324f6,0x61d587d4), LL(0xa37173ea,0xe87673a2),LL(0x53778b65,0x23848008),LL(0x05bab43e,0x10f8441e),LL(0x4621efbe,0xfa11fe12),
   LL(0x81685d7b,0x047b772e),LL(0xbf34a976,0x23f27d81),LL(0x915f48ef,0xc27608e2),LL(0xa521d5c3,0x3b0b43fa), LL(0x63ca7284,0x7613fb26),LL(0x1d4db837,0x7f5729b4),LL(0x583b526b,0x87b14898),LL(0xbbadd3d1,0x00b732a6),
   LL(0x2048e396,0x8e02f426),LL(0x383d9de4,0x436b50b6),LL(0x471e85ad,0xf78d3481),LL(0xd005c8d6,0x8b01ea6a), LL(0x97015c07,0xd3c7afee),LL(0x4e3ba2ae,0x46cdf1a9),LL(0x83d3a1d2,0x7a42e501),LL(0xb541dff4,0xd54b5268),
   LL(0x4e23e9bc,0x3f24cf30),LL(0x126e3624,0x4387f816),LL(0x3b0b6d61,0x26a46a03),LL(0x8b2d777c,0xaf1bc845), LL(0x527de79c,0x25c401ba),LL(0x4261bbb6,0x0e1346d4),LL(0x287b4bc7,0x4b96c44b),LL(0x5254562f,0x658493c7),
   LL(0xb8a24a20,0x23f949fe),LL(0xf52ca53f,0x17ebfed1),LL(0xbcfb4853,0x9b691bbe),LL(0x6278a05d,0x5617ff6b), LL(0xe3c99ebd,0x241b34c5),LL(0x1784156a,0xfc64242e),LL(0x695d67df,0x4206482f),LL(0xee27c011,0xb967ce0e),
   LL(0x21c80b5d,0x65db3751),LL(0xa31ecca0,0x2e7a563c),LL(0x5238a07e,0xe56ffc4e),LL(0x32ced854,0x3d6c2966), LL(0xaf70b885,0xe99d7d1a),LL(0x2d686459,0xafc3bad9),LL(0x0cc8ba5b,0x9c78bf46),LL(0x18955aa3,0x5a439519),
   LL(0x5fe4e314,0xf8b517a8),LL(0xfcb8906f,0xe60234d0),LL(0xf2061b23,0xffe542ac),LL(0x6b4cb59c,0x287e191f), LL(0x09d877d8,0x21857ddc),LL(0x14678941,0x1c23478c),LL(0xb6e05ea4,0xbbf0c056),LL(0xb01594fe,0x82da4b53),
   LL(0xfadb8608,0xf7526791),LL(0x7b74cdf6,0x049e832d),LL(0xc2b90a34,0xa43581cc),LL(0x9360b10c,0x73639eb8), LL(0xe1e4a71b,0x4fba331f),LL(0x8072f919,0x6ffd6b93),LL(0x65679032,0x6e53271c),LL(0xf14272ce,0x67206444),
   LL(0xb2335834,0xc0f734a3),LL(0x90ef6860,0x9526205a),LL(0x04e2bb0d,0xcb8be717),LL(0x02f383fa,0x2418871e), LL(0x4082c157,0xd7177681),LL(0x29c20073,0xcc914ad0),LL(0xe587e728,0xf186c1eb),LL(0x61bcd5fd,0x6fdb3c22),
   LL(0xf2f9f8e9,0x30d014a6),LL(0x4fec49d2,0x963ece23),LL(0x9605a8d9,0x862025c5),LL(0x19f8929a,0x39874445), LL(0x12bf476a,0x01b6ff65),LL(0x09cf7d91,0x598a64d8),LL(0x93be56ca,0xd7ec7749),LL(0xcbb33615,0x10899785),
   LL(0x02eee3ad,0xb8a092fd),LL(0x30145270,0xa86b3d35),LL(0x8512b675,0x323d98c6),LL(0x62ebb40f,0x4b8bc785), LL(0x413f9cde,0x7d301f54),LL(0x2bab5664,0xa5e4fb4f),LL(0x1cbfec23,0x1d2b252d),LL(0xe177120d,0xfcd576bb),
   LL(0x83731a34,0x04427d3e),LL(0xed836e8e,0x2bb9028e),LL(0xb612ca7c,0xb36acff8),LL(0xd3d9c73a,0xb88fe5ef), LL(0xedea4eb3,0xbe2a6bc6),LL(0x488eec77,0x43b93133),LL(0xb17106e1,0xf41ff566),LL(0x654efa32,0x469e9172),
   LL(0x41c23fa3,0xb4480f04),LL(0xc1989a2e,0xb4712eb0),LL(0x93a29ca7,0x3ccbba0f),LL(0xd619428c,0x6e205c14), LL(0xb3641686,0x90db7957),LL(0x45ac8b4e,0x0432691d),LL(0xf64e0350,0x07a759ac),LL(0x9c972517,0x0514d89c),
   LL(0xa8e67fc3,0x1701147f),LL(0xab2085be,0x9e2e0b8b),LL(0xac284e57,0xd5651824),LL(0x74893664,0x890d4325), LL(0xc55e68a3,0x8a7c5e6e),LL(0x4339c85a,0xbf12e90b),LL(0xf922b655,0x31846b85),LL(0x0bf4d700,0x9a54ce4d),
   LL(0xf1a14295,0xd7f4e83a),LL(0xb285d4f9,0x916f955c),LL(0x99ffdaba,0xe57bb0e0),LL(0xeab0d152,0x28a43034), LL(0xb8a9cef8,0x0a36ffa2),LL(0xb9ec051a,0x5517407e),LL(0xea68e672,0x9c796096),LL(0xfb3c77fb,0x853db5fb),
   LL(0xe864a51a,0x21474ba9),LL(0x6e8a1b8b,0x6c267699),LL(0x94120a28,0x7c823626),LL(0x8383a5db,0xe61e9a48), LL(0x9f84216d,0x7dd75003),LL(0xad43cd85,0xab020d07),LL(0xda12c659,0x9437ae48),LL(0xe65452ad,0x6449c2eb),
   LL(0x2cf9d7c1,0xcc7c4c1c),LL(0xee95e5ab,0x1320886a),LL(0xbeae170c,0xbb7b9056),LL(0xdbc0d662,0xc8a5b250), LL(0xc11d2303,0x4ed81432),LL(0x1f03769f,0x7da66912),LL(0x84539828,0x3ac7a5fd),LL(0x3bccdd02,0x14dada94),
   LL(0x7ef6b0d1,0x8b84c321),LL(0x7c933f22,0x52a9477a),LL(0xfd440b82,0x5ef6728a),LL(0x6ce4bd5e,0x5c3bd859), LL(0xf22c2d3e,0x918b80f5),LL(0xb7bb6cc5,0x368d5040),LL(0x2695a11c,0xb66142a1),LL(0xeb19ea70,0x60ac583a),
   LL(0x0eab2437,0x317cbb98),LL(0x5e2654c8,0x8cc08c55),LL(0xe6d8307f,0xfe2d6520),LL(0x57428993,0xe9f147f3), LL(0xd2fd6cf1,0x5f9c7d14),LL(0x2d4fcbb0,0xa3ecd064),LL(0x8e7341f7,0xad83fef0),LL(0x3a63115c,0x643f23a0),
   LL(0xe65ab743,0xd38a78ab),LL(0x35edc89c,0xbf7c75b1),LL(0x530df568,0x3dd8752e),LL(0xe308c682,0xf85c4a76), LL(0xe68acf37,0x4c9955b2),LL(0xab32af85,0xa544df3d),LL(0xa25cf493,0x4b8ec3f5),LL(0x1a622feb,0x4d8f2764),
   LL(0xf0dcbc49,0x7bb4f7aa),LL(0x70bbb45b,0x7de551f9),LL(0x9f2ca2e5,0xcfd0f3e4),LL(0x1f5c76ef,0xece58709), LL(0x167d79ae,0x32920edd),LL(0xfa7d7ec1,0x039df8a2),LL(0xbb30af91,0xf46206c0),LL(0x22676b59,0x1ff5e2f5),
   LL(0x6ea51d66,0x11f4a039),LL(0x807d7a26,0x506c1445),LL(0x755a9b24,0x60da5705),LL(0x1f1a319e,0x8fc8cc32), LL(0x9433d67d,0x83642d4d),LL(0x6a7dd296,0x7fa5cb8f),LL(0x9b7bde07,0x576591db),LL(0x419716fb,0x13173d25),
   LL(0xd5b340ff,0xea30599d),LL(0xb0fe76c5,0xfc6b5297),LL(0xab8f5adc,0x1c6968c8),LL(0x901c928d,0xf723c7f5), LL(0x9773d402,0x4203c321),LL(0x1b51dd47,0xdf7c6aa3),LL(0x552be23c,0x3d49e37a),LL(0x0b5a6e87,0x57febee8),
   LL(0x7bd8e739,0xc5ecbee4),LL(0xae63bf75,0x79d44994),LL(0x38fb8923,0x168bd00f),LL(0xd0533130,0x75d48ee4), LL(0xdb5cdf33,0x554f77aa),LL(0x3c696769,0x3396e896),LL(0xd3fd674e,0x2fdddbf2),LL(0x99d0e3e5,0xbbb8f6ee),
   LL(0xcbae2f70,0x51b90651),LL(0x93aaa8eb,0xefc4bc05),LL(0xdd1df499,0x8ecd8689),LL(0x22f367a5,0x1aee99a8), LL(0xae8274c5,0x95d485b9),LL(0x7d30b39c,0x6c14d445),LL(0xbcc1ef81,0xbafea90b),LL(0xa459a2ed,0x7c5f317a),
   LL(0x4ef44227,0x01211075),LL(0xdc20f496,0xa17bed6e),LL(0x819853cd,0x0cdfe424),LL(0xf71e2ce7,0x13793298), LL(0xdbbe307b,0x3c1f3078),LL(0x76ee9936,0x6dd1c20e),LL(0x423caa20,0x23ee4b57),LL(0x8efb840e,0x4ac3793b),
   LL(0xed1f8ca0,0x934438eb),LL(0x4ebb25a2,0x3e546658),LL(0xc069896f,0xc415af0e),LL(0x9a5aa43d,0xc13eddb0), LL(0xd49eb8f6,0x7a04204f),LL(0xd74f1670,0xd0d5bdfc),LL(0x56fc0558,0x3697e286),LL(0x01cebade,0x10207371),
   LL(0x0647a82b,0x5f87e690),LL(0x8f40054f,0x908e0ed4),LL(0x79853803,0xa9f633d4),LL(0x4a28b252,0x8ed13c9a), LL(0x1f460f64,0x3e2ef676),LL(0x36d06336,0x53930b9b),LL(0x8fc4979b,0x347073ac),LL(0x5ecd5597,0x84380e0e),
   LL(0xc4fe3c39,0xe3b22c6b),LL(0x6c7bebdf,0xba4a8153),LL(0x25693459,0xf23ab6b7),LL(0x14922b11,0x53bc3770), LL(0x5afc60db,0x4645c8ab),LL(0x20b9f2a3,0xaa022355),LL(0xce0fc507,0x52a2954c),LL(0x7ce1c2e7,0x8c2731bb),
   LL(0x18a0339d,0xf39608ab),LL(0x3735436c,0xac7a658d),LL(0xcd992b4f,0xb22c2b07),LL(0xf40dcfd4,0x4e83daec), LL(0x2f39ea3e,0x8a34c7be),LL(0xb0a56d2e,0xef0c005f),LL(0x6edd8038,0x62731f6a),LL(0x4e3cb075,0x5721d740),
   LL(0xfbeeee1b,0x1ea41511),LL(0xef1d0c05,0xd1ef5e73),LL(0x73c07d35,0x42feefd1),LL(0x8a329493,0xe530a00a), LL(0xf15ebfb0,0x5d55b7fe),LL(0xd322491a,0x549de03c),LL(0x745b3237,0xf7b5f602),LL(0x1ab6e2b6,0x3632a3a2),
   LL(0x0ef59f78,0x0d3bba89),LL(0xc9e52b9a,0x0dfc6443),LL(0x72631447,0x1dc79699),LL(0xb3be20b1,0xef033917), LL(0xb1383948,0x0c92735d),LL(0xc0dd7d7d,0xc1fc29a2),LL(0x403ed068,0x6485b697),LL(0xaac93bdc,0x13bfaab3),
   LL(0x0deeaf52,0x410dc6a9),LL(0x4c641c15,0xb003fb02),LL(0x5bc504c4,0x1384978c),LL(0x864a6a77,0x37640487), LL(0x222a77da,0x05991bc6),LL(0x5e47eb11,0x62260a57),LL(0xf21b432c,0xc7af6613),LL(0xab4953e9,0x22f3acc9),
   LL(0x8e41d155,0x52934922),LL(0x3ac059ef,0x4d024568),LL(0x4d884411,0xb0201755),LL(0xa59a178f,0xce8055cf), LL(0xf6204549,0xcd77d1af),LL(0xc7066759,0xa0a00a3e),LL(0x0272c229,0x471071ef),LL(0xd3c4b6b0,0x009bcf6b),
   LL(0x22305177,0x2a2638a8),LL(0x41645bbf,0xd51d59df),LL(0xc0a7a3c0,0xa81142fd),LL(0x4c7063ee,0xa17eca6d), LL(0x60d9dcec,0x0bb887ed),LL(0x20ad2455,0xd6d28e51),LL(0xa67102ba,0xebed6308),LL(0x8bffa408,0x042c3114),
   LL(0x8aa68e30,0xfd099ac5),LL(0x1483513e,0x7a6a3d7c),LL(0xba2d8f0c,0xffcc6b75),LL(0x1e78b954,0x54dacf96), LL(0xa4a9af89,0xf645696f),LL(0x06ac98ec,0x3a411940),LL(0x22a67a20,0x41b8b3f6),LL(0x99dec626,0x2d0b1e0f),
   LL(0x40be34e8,0x27c89192),LL(0x91907f35,0xc7162b37),LL(0xa956702b,0x90188ec1),LL(0xdf93769c,0xca132f7d), LL(0x0e2025b4,0x3ece44f9),LL(0x0c62f14c,0x67aaec69),LL(0x22e3cc11,0xad741418),LL(0x7ff9a50e,0xcf9b75c3),
   LL(0x4d348272,0x02fa2b16),LL(0x9959d56d,0xbd99d61a),LL(0x18762916,0xbc4f19db),LL(0x49c1ac80,0xcc7cce50), LL(0xd846bd83,0x4d59ebaa),LL(0xa9202849,0x8775a9dc),LL(0x6e1f4ca9,0x07ec4ae1),LL(0xba893f11,0x27eb5875),
   LL(0x662cc565,0x00284d51),LL(0x0db4138d,0x82353a6b),LL(0xaa32a594,0xd9c7aaaa),LL(0xa5669c47,0xf5528b5e), LL(0x2f23c5ff,0xf3220231),LL(0x6affa3a1,0xe3e8147a),LL(0x202ddda0,0xfb423d5c),LL(0x6b871bd4,0x3d6414ac),
   LL(0xa51a168a,0x586f82e1),LL(0x48ae5448,0xb712c671),LL(0x76233eb8,0x9a2e4bd1),LL(0x78811ca9,0x0188223a), LL(0xf7c18de1,0x553c5e21),LL(0xb27bb286,0x7682e451),LL(0x0e51e929,0x3ed036b3),LL(0xec9cb34f,0xf487211b),
   LL(0x0c24efc8,0x0d094277),LL(0xbef737a4,0x0349fd04),LL(0x514cdd28,0x6d1c9dd2),LL(0x30da9521,0x29c135ff), LL(0xf78b0b6f,0xea6e4508),LL(0x678c143c,0x176f5dd2),LL(0x4be21e65,0x08148418),LL(0xe7df38c4,0x27f7525c),
   LL(0x748ab1a4,0x1fb70e09),LL(0x5efe4433,0x9cba50a0),LL(0x15f75af2,0x7846c7a6),LL(0x5ee73ea8,0x2a7c2c57), LL(0x3f0a449a,0x42e566a4),LL(0xad90fc3d,0x45474c3b),LL(0x8b61d057,0x7447be3d),LL(0x3a4ec092,0x3e9d1cf1),
   LL(0xf380a6e6,0x1603e453),LL(0x9b1437c2,0x0b86e431),LL(0xef29610a,0x7a4173f2),LL(0xf03d57f7,0x8fa729a7), LL(0x6c9c217e,0x3e186f6e),LL(0x91919524,0xbe1d3079),LL(0x153d4fb1,0x92a62a70),LL(0xd68c2f71,0x32ed3e34),
   LL(0x9eb1a8b7,0xd785027f),LL(0xc5b22fe8,0xbc37eb77),LL(0xb9d6a191,0x466b34f0),LL(0x9a05f816,0x008a89af), LL(0x7d42c10a,0x19b028fb),LL(0x49b3f6b8,0x7fe8c92f),LL(0xa5a0ade3,0x58907cc0),LL(0x559d1a7c,0xb3154f51),
   LL(0xd9790ed6,0x5066efb6),LL(0xa6aa793b,0xa77a0cbc),LL(0x223e042e,0x1a915f3c),LL(0x69c5874b,0x1c5def04), LL(0x73b6c1da,0x0e830078),LL(0xfcd8557a,0x55cf85d2),LL(0x0460f3b1,0x0f7c7c76),LL(0x46e58063,0x87052acb),
   LL(0x907eae66,0x09212b80),LL(0x4d721c89,0x3cb068e0),LL(0xdd45ac1c,0xa87941ae),LL(0x0daa0dbb,0xde8d5c0d), LL(0xe3502e6e,0xda421fdc),LL(0x4d89a084,0xc8944201),LL(0xf0c24bfb,0x7307ba5e),LL(0x20bde0ef,0xda212beb),
   LL(0xf82ce682,0xea2da24b),LL(0x07f71fe4,0x058d3816),LL(0x5ffad8de,0x35a02462),LL(0xaadcefab,0xcd7b05dc), LL(0x1d9f54ec,0xd442f8ed),LL(0xb2d3b5ca,0x8be3d618),LL(0xe06b2ce2,0xe2220ed0),LL(0x1b0da4c0,0x82699a5f),
   LL(0x71c0c3a7,0x3ff106f5),LL(0x0d34180c,0x8f580f5a),LL(0x22d7d375,0x4ebb120e),LL(0xe9513675,0x5e5782cc), LL(0x99c82a70,0x2275580c),LL(0x15ea8c4c,0xe8359fbf),LL(0x7b415e70,0x53b48db8),LL(0x100c6014,0xaacf2240),
   LL(0xe4652f1d,0x9faaccf5),LL(0xd56157b2,0xbd6fdd2a),LL(0x6261ec50,0xa4f4fb1f),LL(0x476bcd52,0x244e55ad), LL(0x047d320b,0x881c9305),LL(0x6181263f,0x1ca983d5),LL(0x278fb8ee,0x354e9a44),LL(0x396e4964,0xad2dbc0f),
   LL(0x9268b3de,0x723f3aa2),LL(0xe6e0609a,0x0d1ca29a),LL(0x6cf44252,0x794866aa),LL(0x01af87ed,0x0b59f3e3), LL(0x7f4a6c51,0xe234e5ff),LL(0x61dc2f7e,0xa8768fd2),LL(0x0a94d81f,0xdafc7332),LL(0x06938ce1,0xd7f84282),
   LL(0x0546063e,0xae0b3c0e),LL(0x5d61abc6,0x7fbadcb2),LL(0x369ac400,0xd5d7a2c9),LL(0xae67d10c,0xa5978d09), LL(0x4f85eaac,0x290f211e),LL(0xfacac681,0xe61e2ad1),LL(0x388384cd,0xae125225),LL(0xccfde30f,0xa7fb68e9),
   LL(0x3daed4c2,0x7a59b936),LL(0x2606f789,0x80a9aa40),LL(0xf6a6d90a,0xb40c1ea5),LL(0x514d5885,0x948364d3), LL(0x70985182,0x062ebc60),LL(0x33310895,0xa6db5b0e),LL(0xe329c2f5,0x64a12175),LL(0x90ea237e,0xc5f25bd2),
   LL(0x2d0a4c23,0x7915c524),LL(0x6bb3cc52,0xeb5d26e4),LL(0xc09e2c92,0x369a9116),LL(0xcf182cf8,0x0c527f92), LL(0x2aede0ac,0x9e591938),LL(0x6cc34939,0xb2922208),LL(0x99a34361,0x3c9d8962),LL(0xc1905fe6,0x3c81836d),
   LL(0xa001ec5a,0x4bfeb57f),LL(0xa0dc5dba,0xe993f5bb),LL(0x724a1380,0x47884109),LL(0x32fe9a04,0x8a0369ab), LL(0x8c927db8,0xea068d60),LL(0x94655741,0xbf5f37cf),LL(0x04b6c7ea,0x47d402a2),LL(0x6af259cb,0x4551c295),
   LL(0xed77ee8b,0x698b71e7),LL(0xf309d5c7,0xbddf7bd0),LL(0x34e780ca,0x6201c22c),LL(0x4c295ef4,0xab04f7d8), LL(0x4313a8ce,0x1c947294),LL(0x92ca4cfe,0xe532e4ac),LL(0xd0a7a97a,0x89738f80),LL(0xa580fd5b,0xec088c88),
   LL(0x42ce9e51,0x612b1ecc),LL(0xb25fdd2a,0x8f9840fd),LL(0x01e7f839,0x3cda78c0),LL(0xece05480,0x546b3d3a), LL(0x80d30916,0x271719a9),LL(0x584c20c4,0x45497107),LL(0x5bc78608,0xaf8f9478),LL(0x277e2a4c,0x28c7d484),
   LL(0x88a2ffe4,0xfce01767),LL(0x28e169a5,0xdc506a35),LL(0x7af9c93a,0x0ea10861),LL(0x03fa0e08,0x1ed24361), LL(0xa3d694e7,0x96eaaa92),LL(0xef50bc74,0xc0f43b4d),LL(0x64114db4,0xce6aa58c),LL(0x7c000fd4,0x8218e8ea),
   LL(0x185f8844,0xac815dfb),LL(0x1557abfb,0xcd7e90cb),LL(0xafbfecdf,0x23d16655),LL(0x085cac4a,0x80f3271f), LL(0xd0e62f47,0x7fc39aa7),LL(0x460a48e5,0x88d519d1),LL(0xd28f101e,0x59559ac4),LL(0xca9ae816,0x7981d9e9),
   LL(0x9ac38203,0x5c38652c),LL(0x57657fe5,0x86eaf87f),LL(0xe21f5416,0x568fc472),LL(0xe7e597b5,0x2afff39c), LL(0x256d4eab,0x3adbbb07),LL(0x8285ab89,0x22598692),LL(0x041caefe,0x35f8112a),LL(0xa5064c8b,0x95df02e3),
   LL(0xc7004bf3,0x4d63356e),LL(0xdb83c7de,0x230a08f4),LL(0x8709a7b7,0xca27b270),LL(0xcb9abd2d,0x0d1c4cc4), LL(0x7550fee8,0x8a0bc66e),LL(0x9cf7247e,0x369cd4c7),LL(0x92b5b7e7,0x75562e84),LL(0x5802af7b,0x8fed0da0),
   LL(0xe48fb889,0x6a7091c2),LL(0x7b8a9d06,0x26882c13),LL(0x1b82a0e2,0xa2498663),LL(0x3518152d,0x844ed736), LL(0xd86e27c7,0x282f476f),LL(0x04afefdc,0xa04edaca),LL(0x6119e34d,0x8b256ebc),LL(0x0787d78b,0x56a413e9),
},
/* digit=19 base_pwr=2^133 */
{
   LL(0x5a74be50,0x82ee061d),LL(0xdea16ff5,0xe41781c4),LL(0x99bfc8a2,0xe0b0c81e),LL(0x0b547e2d,0x624f4d69), LL(0xbdcc9ae4,0x3a83545d),LL(0x409b1e8e,0x2573dbb6),LL(0xa6c93539,0x482960c4),LL(0x5ae18798,0xf01059ad),
   LL(0x3112795f,0x715c9f97),LL(0x984e6ee1,0xe8244437),LL(0xecb66bcd,0x55cb4858),LL(0xabaffbee,0x7c136735), LL(0x5dbec38e,0x54661595),LL(0x388ad153,0x51c0782c),LL(0xc6e0952f,0x9ba4c53a),LL(0x1b21dfa8,0x27e6782a),
   LL(0x4ed2dbc2,0x682f903d),LL(0x7c3b2d83,0x0eba59c8),LL(0x9c7e9335,0x8e9dc84d),LL(0x0eb226d7,0x5f9b21b0), LL(0xaf267bae,0xe33bd394),LL(0xbe2e15ae,0xaa86cc25),LL(0x6a8ec500,0x4f0bf67d),LL(0xf9630658,0x5846aa44),
   LL(0xe2c2bf15,0xfeb09740),LL(0xa9e99704,0x627a2205),LL(0xc2fbc565,0xec8d73d0),LL(0xc20c8de8,0x223eed8f), LL(0xa8363b49,0x1ee32583),LL(0xc9c2b0a6,0x1a0b6cb9),LL(0x90dbc85c,0x49f7c3d2),LL(0x1ef4c1ac,0xa8dfbb97),
   LL(0x65c7c2ab,0xafb34d4c),LL(0xe2c5ea84,0x1d4610e7),LL(0x973c4ab5,0x893f6d1b),LL(0x945ba5c4,0xa3cdd7e9), LL(0x064417ee,0x60514983),LL(0xad6bdf2b,0x1459b23c),LL(0x5cf726c3,0x23b2c341),LL(0x32d6354a,0x3a829635),
   LL(0xab192c18,0x294f901f),LL(0x7030164f,0xec5fcbfe),LL(0xe2246ba6,0xe2e2fcb7),LL(0x221a1a0c,0x1e7c88b3), LL(0xc92d88c5,0x72c7dd93),LL(0x1106fb59,0x41c2148e),LL(0xa0f60f14,0x547dd4f5),LL(0x63960f31,0xed9b52b2),
   LL(0xb0a5b358,0x6c8349eb),LL(0x9e7e2ed6,0xb154c5c2),LL(0xeda462db,0xcad5eccf),LL(0x2de66b69,0xf2d6dbe4), LL(0x8665e5b2,0x426aedf3),LL(0x7b7f5723,0x488a8513),LL(0x8bcbb386,0x15cc43b3),LL(0xd791d879,0x27ad0af3),
   LL(0x846e364f,0xc16c236e),LL(0xdea50ca0,0x7f33527c),LL(0x0926b86d,0xc4810775),LL(0x0598e70c,0x6c2a3609), LL(0xf024e924,0xa6755e52),LL(0x9db4afca,0xe0fa07a4),LL(0x66831790,0x15c3ce7d),LL(0xa6cbb0d6,0x5b4ef350),
   LL(0xb6205969,0x2c4aafc4),LL(0xf6c7854f,0x42563f02),LL(0x1d983b48,0x016aced5),LL(0x99949755,0xfeb356d8), LL(0xd1a39bd7,0x8c2a2c81),LL(0xe6934ae9,0x8f44340f),LL(0x447904da,0x148cf91c),LL(0x0f51a926,0x7340185f),
   LL(0x7409ab46,0x2f8f00fb),LL(0x80e289b2,0x057e78e6),LL(0xa888e5d1,0x03e5022c),LL(0x9dede4e2,0x3c87111a), LL(0x7809460b,0x5b9b0e1c),LL(0x71c9abc7,0xe751c852),LL(0xc7cc1dc9,0x8b944e28),LL(0x1d3cfa08,0x4f201ffa),
   LL(0x3e6721ce,0x02fc905c),LL(0xd0b3674c,0xd52d70da),LL(0x18810da4,0x5dc2e5ca),LL(0x5c69dd99,0xa984b273), LL(0x84de5ca4,0x63b92527),LL(0xc852dec4,0x2f1c9872),LL(0xc2e3de09,0x18b03593),LL(0x9813dc2f,0x19d70b01),
   LL(0xa6dc1d29,0x42806b2d),LL(0xf871e144,0xd3030009),LL(0xaaf49276,0xa1feb333),LL(0xc70bc04b,0xb5583b9e), LL(0x95695f20,0x1db0be78),LL(0x89d012b5,0xfc841811),LL(0x05f61643,0x6409f272),LL(0xd5883128,0x40d34174),
   LL(0x67419833,0xd79196f5),LL(0x863b7b08,0x6059e252),LL(0x1c56700c,0x84da1817),LL(0xb28d3ec4,0x5758ee56), LL(0x013b0ea6,0x7da2771d),LL(0x54c5e9b9,0xfddf524b),LL(0x24305d80,0x7df4faf8),LL(0x3a97763f,0x58f5c1bf),
   LL(0x7c696042,0xa5af37f1),LL(0x4a2538de,0xd4cba22c),LL(0x9ea42600,0x211cb995),LL(0x7b069889,0xcd105f41), LL(0xddb81e74,0xb1e1cf19),LL(0x5157b8ca,0x472f2d89),LL(0xee9db885,0x086fb008),LL(0x0f26d131,0x365cd570),
   LL(0xa2be7053,0x284b02bb),LL(0x7ab9a6d6,0xdcbbf7c6),LL(0x20f7a530,0x4425559c),LL(0x188767c8,0x961f2dfa), LL(0x70dc80c4,0xe2fd9435),LL(0xf0784120,0x104d6b63),LL(0x53567122,0x7f592bc1),LL(0xf688ad77,0xf6bc1246),
   LL(0x0f15dde9,0x05214c05),LL(0x0d5f2b82,0xa47a76a8),LL(0x62e82b62,0xbb254d30),LL(0x3ec955ee,0x11a05fe0), LL(0x9d529b36,0x7eaff46e),LL(0x8f9e3df6,0x55ab1301),LL(0x99317698,0xc463e371),LL(0xccda47ad,0xfd251438),
   LL(0x23d695ea,0xca9c3547),LL(0x16e589b5,0x48ce626e),LL(0xb187d086,0x6b5b64c7),LL(0xb2207948,0xd02e1794), LL(0x7198111d,0x8b58e98f),LL(0xdcf9c3cc,0x90ca6305),LL(0xf34089b0,0x5691fe72),LL(0xfc7c80ff,0x60941af1),
   LL(0x22eb51e5,0xa09bc0a2),LL(0xaa9cf09a,0xc0bb7244),LL(0x80159f06,0x36a8077f),LL(0xdddc560e,0x8b5c989e), LL(0x512e1f43,0x19d2f316),LL(0xad08ff62,0x02eac554),LL(0x07d20b4e,0x012ab84c),LL(0xd6d4e4e1,0x37d1e115),
   LL(0xab7b19a8,0xb6443e1a),LL(0xdef8cd45,0xf08d067e),LL(0x685e03da,0x63adf3e9),LL(0x4792b916,0xcf15a10e), LL(0xb738a425,0xf44bcce5),LL(0x9636b2fd,0xebe131d5),LL(0x7850d605,0x94068841),LL(0xb40d749d,0x09684eaa),
   LL(0x72ba075b,0x8c3c669c),LL(0xba469015,0x89f78b55),LL(0x3e9f8ba8,0x5706aade),LL(0xb32d7ed7,0x6d8bd565), LL(0x805f08d6,0x25f4e63b),LL(0xc3bcc1b5,0x7f48200d),LL(0xb025d847,0x4e801968),LL(0x87cbe0a8,0x74afac04),
   LL(0x7e63d690,0x43ed2c2b),LL(0x0223cdb8,0xefb6bbf0),LL(0x2884d3fe,0x4fec3cae),LL(0xd75e25a4,0x065ecce6), LL(0x69f79071,0x6c2294ce),LL(0x044b8666,0x0d9a8e5f),LL(0x17b69d8f,0x5009f238),LL(0xc5dfdaf7,0x3c29f8fe),
   LL(0xebae68c4,0x9067528f),LL(0x30c5ba21,0x5b385632),LL(0x1fdd1aec,0x540df119),LL(0xcfba4c78,0xcf37825b), LL(0xbeb11454,0x77eff980),LL(0x60c1b066,0x40a1a991),LL(0xf889a1c7,0xe8018980),LL(0x76c24be0,0xb9c52ae9),
   LL(0x45650ef4,0x05fbbcce),LL(0x8aa29ac7,0xae000f10),LL(0x4f04c470,0x884b7172),LL(0x19bb5c25,0x7cd4fde2), LL(0xe8840869,0x6477b22a),LL(0x5fbd0686,0xa8868859),LL(0x1116dfba,0xf23cc02e),LL(0xd87d7776,0x76cd563f),
   LL(0xa9d82abf,0xe2a37598),LL(0xe6c170f5,0x5f188ccb),LL(0x5066b087,0x81682200),LL(0xc7155ada,0xda22c212), LL(0xfbddb479,0x151e5d3a),LL(0x6d715b99,0x4b606b84),LL(0xf997cb2e,0x4a73b54b),LL(0x3ecd8b66,0x9a1bfe43),
   LL(0x2a67d48a,0x1c312809),LL(0x031fa9e2,0xcd6a671e),LL(0x0e43a34a,0xbec3312a),LL(0x55ef47d3,0x1d935639), LL(0x8fea73ea,0x5ea02489),LL(0xa035afb2,0x8247b364),LL(0x5265b54c,0xb58300a6),LL(0x722c7148,0x3286662f),
   LL(0xb4ec4c20,0xb77fd76b),LL(0x0f3fe3fd,0xf0a12fa7),LL(0x41d8c7e8,0xf845bbf5),LL(0x5ec10aa8,0xe4d969ca), LL(0x43e232a3,0x4c0053b7),LL(0x37f8a45a,0xdc7a3fac),LL(0x20d81c8f,0x3c4261c5),LL(0xb00eab00,0xfd4b3453),
   LL(0xd36e3062,0x76d48f86),LL(0xa143ff02,0x626c5277),LL(0xaf76f42e,0x538174de),LL(0x6407ceac,0x2267aa86), LL(0x72e572d5,0xfad76351),LL(0xba7330eb,0xab861af7),LL(0x418d8657,0xa0a1c8c7),LL(0x20289a52,0x988821cb),
   LL(0xcccc18ad,0x79732522),LL(0xf1a6e027,0xaadf3f8d),LL(0x17c2354d,0xf7382c93),LL(0xd818b689,0x5ce1680c), LL(0xd9ecbee9,0x359ebbfc),LL(0x1cae62ac,0x4330689c),LL(0xc51ac38a,0xb55ce5b4),LL(0xfe238ee8,0x7921dfea),
   LL(0x271d1ca5,0x3972bef8),LL(0xe8aabd18,0x3e423bc7),LL(0x44a3e5e3,0x57b09f3f),LL(0x7b444d66,0x5da886ae), LL(0xa9964375,0x68206634),LL(0x699cd0ff,0x356a2fa3),LL(0xdba515e9,0xaf0faa24),LL(0xb321d79a,0x536e1f5c),
   LL(0x5c04e4ea,0xd3b9913a),LL(0xd6f11513,0xd549dcfe),LL(0x79fd1d94,0xee227bf5),LL(0xb43f2c67,0x9f35afee), LL(0xf1314f53,0xd2638d24),LL(0xcabcd822,0x62baf948),LL(0x4ef48db0,0x5542de29),LL(0xfc5f6bb2,0xb3eb6a04),
   LL(0x1208e16a,0x23c110ae),LL(0xf8363e24,0x1a4d15b5),LL(0x164be00b,0x30716844),LL(0xf6f4690d,0xa8e24824), LL(0x90b170cf,0x548773a2),LL(0x42f191f4,0xa1bef331),LL(0x9247aa97,0x70f418d0),LL(0x48be9147,0xea06028e),
   LL(0xdbfb894e,0xe13122f3),LL(0xce274b18,0xbe9b79f6),LL(0xca58aadf,0x85a49de5),LL(0x11487351,0x24957758), LL(0xbb939099,0x111def61),LL(0x26d13694,0x1d6a974a),LL(0xd3fc253b,0x4474b4ce),LL(0x4c5db15e,0x3a1485e6),
   LL(0x147c15b4,0xe79667b4),LL(0x7bc61301,0xe34f553b),LL(0x17094381,0x032b80f8),LL(0x723eaa21,0x55d8bafd), LL(0xf1c0e74e,0x5a987995),LL(0xebba289c,0x5a9b292e),LL(0xeb4c8251,0x413cd4b2),LL(0xd162db0a,0x98b5d243),
   LL(0x68342520,0xbb47bf66),LL(0xbaa862d1,0x08d68949),LL(0xe906abcd,0x11f349c7),LL(0xed7bf00e,0x454ce985), LL(0xb55b803b,0xacab5c9e),LL(0x31e3c16d,0xb03468ea),LL(0xd273bf12,0x5c24213d),LL(0x71587887,0x211538eb),
   LL(0x731dea2d,0x198e4a2f),LL(0x74ed7b2a,0xd5856cf2),LL(0x13a664fe,0x86a632eb),LL(0xbda41291,0x932cd909), LL(0xc0c4ddc0,0x850e95d4),LL(0x347fc2c9,0xc0f422f8),LL(0x86076bcb,0xe68cbec4),LL(0xcd6cd286,0xf9e7c0c0),
   LL(0x0f5f27ca,0x65994ddb),LL(0xa80d59ff,0xe85461fb),LL(0x66601023,0xff05481a),LL(0xfc9ebbfb,0xc665427a), LL(0x7587fd52,0xb0571a69),LL(0x8d49efce,0x935289f8),LL(0xea420688,0x61becc60),LL(0x13a786af,0xb22639d9),
   LL(0x361ecf90,0x1a8e6220),LL(0x25506463,0x001f23e0),LL(0x0a5c2b79,0xe4ae9b5d),LL(0xd8149db5,0xebc9cdad), LL(0x934aa728,0xb33164a1),LL(0xae9b60f3,0x750eb00e),LL(0x9b9cfbfd,0x5a91615b),LL(0xef45f7f6,0x97015cbf),
   LL(0xbf5151df,0xb462c4a5),LL(0xb07118f2,0x21adcc41),LL(0x043fa42c,0xd60c545b),LL(0xe96be1ab,0xfc21aa54), LL(0x4e51ea80,0xe84bc32f),LL(0x259b5d8d,0x3dae45f0),LL(0xc38f1b5e,0xbb73c7eb),LL(0xe8ae617d,0xe405a74a),
   LL(0x9f1c56bd,0xbb1ae9c6),LL(0x49f196a4,0x8c176b98),LL(0x6875092b,0xc448f311),LL(0x9f976033,0xb5afe3de), LL(0x145813e5,0xa8dafd49),LL(0xe2b34226,0x687fc4d9),LL(0x4c7ff57f,0xf2dfc92d),LL(0x401f1b46,0x004e3fc1),
   LL(0x1430c9ab,0x5afddab6),LL(0x2238e997,0x0bdd41d3),LL(0x418042ae,0xf0947430),LL(0xcdddc4cb,0x71f9adda), LL(0xc52dd907,0x7090c016),LL(0x29e2047f,0xd9bdf44d),LL(0x1b1011a6,0xe6f1fe80),LL(0xd9acdc78,0xb63accbc),
   LL(0x1272a95b,0xcfc7e235),LL(0xa6276ac8,0x0c667717),LL(0xe2d7eef7,0x3c0d3709),LL(0x9a685b3e,0x5add2b06), LL(0x14ea5d65,0x363ad32d),LL(0x8d7dd506,0xf8e01f06),LL(0x75b4aac6,0xc9ea2213),LL(0x0d353466,0xed2a2bf9),
   LL(0xe9d3a7c3,0x439d79b5),LL(0x81b7f34b,0x8e0ee5a6),LL(0x1dc4ba75,0xcf3dacf5),LL(0xeb3310c7,0x1d3d1773), LL(0x7747ae83,0xa8e67112),LL(0x197d6b40,0x31f43160),LL(0xcd961400,0x0521ccee),LL(0xf6535768,0x67246f11),
   LL(0xef0c3133,0x702fcc5a),LL(0x7e16693b,0x247cc45d),LL(0xc729b749,0xfd484e49),LL(0xb218320f,0x522cef7d), LL(0x59ab93b3,0xe56ef405),LL(0x9f181071,0x225fba11),LL(0x15330ed0,0x33bd6595),LL(0x1ddb32f7,0xc4be69d5),
   LL(0x0448087c,0x264c7668),LL(0x71432dae,0xac30903f),LL(0x00f9bf47,0x3851b266),LL(0x6cdd6d03,0x400ed311), LL(0xf8fd2424,0x045e79fe),LL(0xfa6da98b,0xfdfd974a),LL(0x0c1e673a,0x45c9f641),LL(0x5b2c5168,0x76f2e733),
   LL(0x2a601753,0x1adaebb5),LL(0xc57c2d49,0xb286514c),LL(0x1e0bfd24,0xd8769670),LL(0x04478922,0x950c547e), LL(0xe5d32bfe,0xd1d41969),LL(0x750d6c3e,0x30bc1472),LL(0xe0e27f3a,0x8f3679fe),LL(0xa4a6ee0c,0x8f64a7dc),
   LL(0x633dfb1f,0x2fe59937),LL(0x977f2547,0xea82c395),LL(0x661ea646,0xcbdfdf1a),LL(0xb9085451,0xc7ccc591), LL(0x81761e13,0x82177962),LL(0x9196885c,0xda57596f),LL(0x28ffbd70,0xbc17e849),LL(0x2671d36f,0x1e6e0a41),
   LL(0x4152fcf5,0x61ae872c),LL(0x9e77e754,0x441c87b0),LL(0xa34dff09,0xd0799dd5),LL(0x88a6b171,0x766b4e44), LL(0x11f1c792,0xdc06a512),LL(0x4be35c3e,0xea02ae93),LL(0xe90c469e,0xe5ca4d6d),LL(0x56e4ff5c,0x4df4368e),
   LL(0x4baef62e,0x7817acab),LL(0xa85b91e8,0x9f5a2202),LL(0x6ce57610,0x9666ebe6),LL(0xf73bfe03,0x32ad31f3), LL(0x25bcf4d6,0x628330a4),LL(0x515056e6,0xea950593),LL(0xe1332156,0x59811c89),LL(0x8c11b2d7,0xc89cf1fe),
   LL(0x04e60cc0,0x75b63913),LL(0x4625d375,0xce811e8d),LL(0x2d26e562,0x030e43fc),LL(0x608d36a0,0xfbb30b4b), LL(0x48528118,0x634ff82c),LL(0xcd285911,0x7c6fe085),LL(0x99358f28,0x7f2830c0),LL(0x665e6c09,0x2e60a95e),
   LL(0x9b785dbf,0x08407d3d),LL(0xa759bce7,0x530889ab),LL(0x52f61239,0xf228e0e6),LL(0x6879be3c,0x2b6d1461), LL(0x51a7bbf7,0xe6902c04),LL(0x76f24a64,0x30ad99f0),LL(0x98bc6da0,0x66d9317a),LL(0xcb596ac0,0xf4f877f3),
   LL(0x4c44f119,0xb05ff62d),LL(0xe9b77416,0x4555f536),LL(0x8caed63b,0xc7c0d059),LL(0xc358b2a9,0x0cd2b7ce), LL(0x46945fa3,0x3f33287b),LL(0xd67c8791,0xf8785b20),LL(0x9637bd08,0xc54a7a61),LL(0x18be79d7,0x54d4598c),
   LL(0xc46d7ce1,0x889e5acb),LL(0x8b085877,0x9a515bb7),LL(0x0b7a5050,0xfac1a03d),LL(0xf2926035,0x7d3e738a), LL(0x2a6cb0eb,0x861cc2ce),LL(0x8f7adc79,0x6f2e2955),LL(0x33016376,0x61c4d451),LL(0x5ad59090,0xd9fd2c80),
   LL(0xb2b836a1,0xe5a83738),LL(0x7c0d6622,0x855b41a0),LL(0x7cc19af1,0x186fe317),LL(0xfdd99acb,0x6465c1ff), LL(0x6974b99e,0x46e5c23f),LL(0xa2717cbe,0x75a7cf8b),LL(0x062be658,0x4d2ebc3f),LL(0x5f209c98,0x094b4447),
   LL(0xb940cb5a,0x4af285ed),LL(0x7cc82f10,0x6706d792),LL(0x030526fa,0xc8c8776c),LL(0xa0da9140,0xfa8e6f76), LL(0x591ee4f0,0x77ea9d34),LL(0x40274166,0x5f46e337),LL(0xea671457,0x1bdf98bb),LL(0x862a1fe2,0xd7c08b46),
   LL(0x1c08ad63,0x46cc303c),LL(0x4c845e7b,0x99543440),LL(0x48f36bf7,0x1b8fbdb5),LL(0x8c8273a7,0x5b82c392), LL(0x928435d5,0x08f712c4),LL(0x79330380,0x071cf0f1),LL(0xa8da054a,0xc74c2d24),LL(0x43c46b5c,0xcb0e7201),
   LL(0xc0b7eff3,0x0ad7337a),LL(0xc5e48b3c,0x8552225e),LL(0x73f13a5f,0xe6f78b0c),LL(0x82349cbe,0x5e70062e), LL(0xe7073969,0x6b8d5048),LL(0xc33cb3d2,0x392d2a29),LL(0x4ecaa20f,0xee4f727c),LL(0x2ccde707,0xa068c99e),
   LL(0xb87a2913,0xfcd5651f),LL(0x3cc252f0,0xea3e3c15),LL(0x3b6cd3e4,0x777d92df),LL(0xc5a732e7,0x7a414143), LL(0xa71ff493,0xa895951a),LL(0xbbd37cf6,0xfe980c92),LL(0xdecfeeff,0x45bd5e64),LL(0xa44c43e9,0x910dc2a9),
   LL(0xcca9f54d,0xcb403f26),LL(0x9303f6db,0x928bbdfb),LL(0xa9eee67c,0x3c37951e),LL(0xf79961c3,0x3bd61a52), LL(0x395c9a79,0x09a238e6),LL(0x61eb352d,0x6940ca2d),LL(0xc1875631,0x7d1e5c5e),LL(0x1e1b20d1,0x1e19742c),
   LL(0x23fc2e6e,0x4633d908),LL(0x08959149,0xa76e29a9),LL(0x84ed7da5,0x61069d9c),LL(0x5dbcad51,0x0baa11cf), LL(0x961849da,0xd01eec64),LL(0xaf3d8c28,0x93b75f1f),LL(0x1ca2ee44,0x57bc4f9f),LL(0x00e00558,0x5a26322d),
   LL(0x61a023ef,0x1888d658),LL(0xb9e5246e,0x1d72aab4),LL(0xe5563ec0,0xa9a26348),LL(0xc3439a43,0xa0971963), LL(0xadb9b5b7,0x567dd54b),LL(0xc45a524b,0x73fac1a1),LL(0xfe38e608,0x8fe97ef7),LL(0x3f384f48,0x608748d2),
   LL(0xc486094f,0xb0571794),LL(0x8bf3a8d6,0x869254a3),LL(0x310b0e25,0x148a8dd1),LL(0x9aa3f7d8,0x99ab9f3f), LL(0x6706c02e,0x0927c68a),LL(0x69790e6c,0x22b5e76c),LL(0x6c71376c,0x6c325260),LL(0x09ef6657,0x53a57690),
   LL(0xedffcf3a,0x8d63f852),LL(0x3c0a6f55,0xb4d2ed04),LL(0x12519b9e,0xdb3aa8de),LL(0x1e0a569a,0x5d38e9c4), LL(0x303747e2,0x871528bf),LL(0xf5b5c18d,0xa208e77c),LL(0xca6bf923,0x9d129c88),LL(0xbf02839f,0xbcbf197f),
   LL(0x27323194,0x9b9bf030),LL(0x339ca59d,0x3b055a8b),LL(0x0f669520,0xb46b2312),LL(0x497e5f24,0x19789f1f), LL(0xaaf01801,0x9c499468),LL(0x8b69d59c,0x72ee1190),LL(0xacf4c079,0x8bd39595),LL(0x8e0cd048,0x3ee11ece),
   LL(0x1ed66f18,0xebde86ec),LL(0xd61fce43,0x225d906b),LL(0xe8bed74d,0x5cab07d6),LL(0x27855ab7,0x16e4617f), LL(0xb2fbc3dd,0x6568aadd),LL(0x8aeddf5b,0xedb5484f),LL(0x6dcf2fad,0x878f20e8),LL(0x615f5699,0x3516497c),
},
/* digit=20 base_pwr=2^140 */
{
   LL(0xfa181e69,0xef0a3fec),LL(0x30d69a98,0x9ea02f81),LL(0x66eab95d,0xb2e9cf8e),LL(0x24720021,0x520f2beb), LL(0x1df84361,0x621c540a),LL(0x71fa6d5d,0x12037721),LL(0x0ff5f6ff,0x6e3c7b51),LL(0xabb2bef3,0x817a069b),
   LL(0xb294cda6,0x83572fb6),LL(0xb9039f34,0x6ce9bf75),LL(0x095cbb21,0x20e012f0),LL(0xd063f0da,0xa0aecc1b), LL(0xf02909e5,0x57c21c3a),LL(0x48ce9cdc,0xc7d59ecf),LL(0x8ae336f8,0x2732b844),LL(0x3f4f85f4,0x056e3723),
   LL(0x89e800ca,0x8a10b531),LL(0x145208fd,0x50fe0c17),LL(0xb714ba37,0x9e43c0d3),LL(0x34189acc,0x427d200e), LL(0xe616e2c0,0x05dee24f),LL(0xee1854c1,0x9c25f4c8),LL(0x8f342a73,0x4d3222a5),LL(0xa027c952,0x0807804f),
   LL(0x4f0d56f3,0xc222653a),LL(0xca28b805,0x961e4047),LL(0x4a73434b,0x2c03f8b0),LL(0xab712a19,0x4c966787), LL(0x864fee42,0xcc196c42),LL(0x5b0ece5c,0xc1be93da),LL(0xc131c159,0xa87d9f22),LL(0xdce45655,0x2bb6d593),
   LL(0xb809b7ce,0x22c49ec9),LL(0xe2c72c2c,0x8a41486b),LL(0xfea0bf36,0x813b9420),LL(0xa66dac69,0xb3d36ee9), LL(0x328cc987,0x6fddc08a),LL(0x3a326461,0x0a3bcd2c),LL(0xd810dbba,0x7103c49d),LL(0x4b78a4c4,0xf9d81a28),
   LL(0xe4d55941,0x3de865ad),LL(0x30384087,0xdedafa5e),LL(0x4ef18b9b,0x6f414abb),LL(0xfaee5268,0x9ee9ea42), LL(0x37a55a4a,0x260faa16),LL(0x015f93b9,0xeb19a514),LL(0x9e9c3598,0x51d7ebd2),LL(0x1932178e,0x523fc56d),
   LL(0xb98fe684,0x501d070c),LL(0x124a1458,0xd60fbe9a),LL(0x92bc6b3f,0xa45761c8),LL(0xfe6f27cb,0xf5384858), LL(0xb59e763b,0x4b0271f7),LL(0x5b5a8e5e,0x3d4606a9),LL(0x05a48292,0x1eda5d9b),LL(0xe6fec446,0xda7731d0),
   LL(0x90d45871,0xa3e33693),LL(0x06166d8d,0xe9764040),LL(0x89a90403,0xb5c33682),LL(0x72f1d637,0x4bd17983), LL(0xd5d2c53a,0xa616679e),LL(0xfdcf3b87,0x5ec4bcd8),LL(0xb66a694e,0xae6d7613),LL(0xe3fc27e5,0x7460fc76),
   LL(0x95caabee,0x70469b82),LL(0x889501e3,0xde024ca5),LL(0x076ed265,0x6bdadc06),LL(0x5a0ef8b2,0x0cb1236b), LL(0x0972ebf9,0x4065ddbf),LL(0x22aca432,0xf1dd3875),LL(0x744aff76,0xa88b97cf),LL(0xfe8e3d24,0xd1359afd),
   LL(0x91502cf3,0x52a3ba2b),LL(0x084db75d,0x2c3832a8),LL(0xde30b1c9,0x04a12ddd),LL(0xe31fd60c,0x7802eabc), LL(0xa37fddab,0x33707327),LL(0xfaafa973,0x65d6f2ab),LL(0x11e6f91a,0x3525c5b8),LL(0x5f46530b,0x76aeb0c9),
   LL(0x2f93a675,0xe8815ff6),LL(0x05f48679,0xa6ec9684),LL(0x358ae884,0x6dcbb556),LL(0xe19e3873,0x0af61472), LL(0xa5f696be,0x72334372),LL(0x6f22fb70,0xc65e57ea),LL(0x946cea90,0x268da30c),LL(0x65681b2a,0x136a8a87),
   LL(0x0f9f44d4,0xad5e81dc),LL(0x2c46585a,0xf09a6960),LL(0xc447d1b1,0xd1649164),LL(0x879dc8b1,0x3b4b36c8), LL(0x3b6b234c,0x20d4177b),LL(0x1730d9d0,0x096a2505),LL(0xef80531d,0x0611b9b8),LL(0x64bb495d,0xba904b3b),
   LL(0x93a3147a,0x1192d9d4),LL(0x9a565545,0x9f30a5dc),LL(0x6ef07212,0x90b1f9cb),LL(0x0d87fc13,0x29958546), LL(0xc17db9ba,0xd3323eff),LL(0xcb1644a8,0xcb18548c),LL(0x4f49ffbc,0x18a306d4),LL(0x4c2e8684,0x28d658f1),
   LL(0xa99f8c71,0x44ba60cd),LL(0x4bf742ff,0x67b7abdb),LL(0x914b3f99,0x66310f9c),LL(0xf412c161,0xae430a32), LL(0x88ace52f,0x1e6776d3),LL(0x52d7067d,0x4bc0fa24),LL(0x8f07cd1b,0x03c286aa),LL(0xa985b2c1,0x4cb8f38c),
   LL(0x8c3bff36,0x83ccbe80),LL(0x5263e575,0x005a0bd2),LL(0x259bdcd1,0x460d7dda),LL(0xfa5cab6b,0x4a1c5642), LL(0x9fe4fc88,0x2b7bdbb9),LL(0xcc97bbb5,0x09418e28),LL(0xa12321ae,0xd8274fb4),LL(0x5c87b64e,0xb137007d),
   LL(0xc63c4962,0x80531fe1),LL(0x981fdb25,0x50541e89),LL(0xfd4c2b6b,0xdc1291a1),LL(0xa6df4fca,0xc0693a17), LL(0x0117f203,0xb2c4604e),LL(0x0a99b8d0,0x245f1963),LL(0xc6212c44,0xaedc20aa),LL(0x520f52a8,0xb1ed4e56),
   LL(0xf8547be3,0xfe48f575),LL(0xa9e45f98,0x0a7033cd),LL(0x18c50100,0x4b45d3a9),LL(0xa61d41da,0xb2a6cd6a), LL(0x57933c6b,0x60bbb4f5),LL(0x2b0d7ffc,0xa7538ebd),LL(0x8cd626b6,0x9ea3ab8d),LL(0x3601625a,0x8273a484),
   LL(0x0168e508,0x88859845),LL(0x99a94abd,0x8cbc9bb2),LL(0xfab0a671,0x713ac792),LL(0x6c9ebffc,0xa3995b19), LL(0x1239e152,0xe711668e),LL(0xbbb8dff4,0x56892558),LL(0xdbf17963,0x8bfc7dab),LL(0xb3de1253,0x5b59fe5a),
   LL(0x34a9f7ae,0x7e3320eb),LL(0xd751efe4,0xe5e8cf72),LL(0xd9be2f37,0x7ea003bc),LL(0xb6c08ef7,0xc0f551a0), LL(0x038f6725,0x56606268),LL(0x6d92d3b6,0x1dd38e35),LL(0xc3cbd686,0x07dfce7c),LL(0x651c5da8,0x4e549e04),
   LL(0x08b19340,0x4058f93b),LL(0xcac6d89d,0xc2fae6f4),LL(0x8f159cc7,0x4bad8a8c),LL(0xcb0b601c,0x0ddba4b3), LL(0x1dd95f8c,0xda4fc7b5),LL(0xcea5c255,0x1d163cd7),LL(0x274a8c4c,0x30707d06),LL(0x2802e9ce,0x79d9e008),
   LL(0xe6ddd505,0x02a29ebf),LL(0xb50bed1a,0x37064e74),LL(0xa7327d57,0x3f6bae65),LL(0xf83920bc,0x3846f5f1), LL(0x60df1b9b,0x87c37491),LL(0x2d1da29f,0x4cfb2895),LL(0x4ed1743c,0x10a478ca),LL(0x3edd47c6,0x390c6030),
   LL(0x8c0a78de,0x8f3e5312),LL(0x1e85df70,0xccd02bda),LL(0xa61b6582,0xd6c75c03),LL(0xfc0eebd1,0x0762921c), LL(0xd85010c0,0xd34d0823),LL(0x0044cf1f,0xd73aaacb),LL(0xa3b5e78a,0xfb4159bb),LL(0xe5826f3f,0x2287c7f7),
   LL(0x580b1a01,0x4aeaf742),LL(0x60423b79,0xf080415d),LL(0xa7dea144,0xe12622cd),LL(0x59d62472,0x49ea4996), LL(0x571f3913,0xb42991ef),LL(0xf5b25a8a,0x0610f214),LL(0x30b79e8f,0x47adc585),LL(0x07a065a2,0xf90e3df6),
   LL(0x43e2e034,0x5d0a5deb),LL(0x444024aa,0x53fb5a34),LL(0x6b0c9f7f,0xa8628c68),LL(0xac563656,0x9c69c29c), LL(0xbace47b6,0x5a231feb),LL(0x9ea5a2ec,0xbdce0289),LL(0x9463853e,0x05da1fac),LL(0x509e78aa,0x96812c52),
   LL(0x57151692,0xd3fb5771),LL(0xd98e1c44,0xeb2721f8),LL(0x32399be1,0xc0506087),LL(0xd979d8b8,0xda5a5511), LL(0xc6f56780,0x737ed55d),LL(0x0dc7a7f4,0xe20d3004),LL(0xf5941a03,0x02ce7301),LL(0xed30f83a,0x91ef5215),
   LL(0x4092d85f,0x28727fc1),LL(0x5c49e41a,0x72d223c6),LL(0xba6a4d81,0xa7cf30a2),LL(0xb030d87d,0x7c086209), LL(0xfc588b09,0x04844c7d),LL(0x5874bbb0,0x728cd499),LL(0xe84c0495,0xcc1281ee),LL(0xec31958f,0x0769b5ba),
   LL(0xf99c2471,0x665c228b),LL(0x191eb110,0xf2d8a11b),LL(0xd36d7024,0x4594f494),LL(0xcdcb25a1,0x482ded8b), LL(0xdadd4885,0xc958a9d8),LL(0xf1d2b547,0x7004477e),LL(0x2a0af550,0x0a45f6ef),LL(0x2f8d6351,0x4fc739d6),
   LL(0x786f08a9,0x75cdaf27),LL(0x42c2737f,0x8700bb26),LL(0x1c4e2670,0x855a7141),LL(0x15076fef,0x810188c1), LL(0xabcd3297,0xc251d0c9),LL(0xf48108eb,0xae4c8967),LL(0x18ceed30,0xbd146de7),LL(0xc986bced,0xf9d4f07a),
   LL(0x83fa1e08,0x5ad98ed5),LL(0xbeabd1fb,0x7780d33e),LL(0x903b1196,0xe330513c),LL(0xa47bc8c4,0xba11de9e), LL(0x02c2d064,0x684334da),LL(0xa48de23b,0x7ecf360d),LL(0x0a9089d8,0x57a1b474),LL(0xff36734c,0xf28fa439),
   LL(0xea4570b3,0xf2a482cb),LL(0xa5ebcee9,0xee65d68b),LL(0xb9694cd5,0x988d0036),LL(0x37885d32,0x53edd0e9), LL(0xbeb9bc6d,0xe37e3307),LL(0x9f5c6768,0xe9abb907),LL(0x51f2160f,0x4396ccd5),LL(0x47336da6,0x2500888c),
   LL(0x926fce43,0x383f9ed9),LL(0x04da2930,0x809dd1c7),LL(0x8a4cb227,0x30f6f596),LL(0x73a56b38,0x0d700c7f), LL(0xab64a065,0x1825ea33),LL(0x1338df80,0xaab9b735),LL(0x9b63f57f,0x1516100d),LL(0x27a6a634,0x2574395a),
   LL(0x700a1acd,0xb5560fb6),LL(0xfd999681,0xe823fd73),LL(0x6cb4e1ba,0xda915d1f),LL(0x6ebe00a3,0x0d030118), LL(0x89fca8cd,0x744fb0c9),LL(0xf9da0e0b,0x970d01db),LL(0x7931d76f,0x0ad8c564),LL(0xf659b96a,0xb15737bf),
   LL(0xa8b484e7,0xdc9933e8),LL(0x7a26dec7,0xb2fdbdf9),LL(0x9f1f0136,0x2349e9a4),LL(0x70fddddb,0x7860368e), LL(0xf9ad3e18,0xd93d2c1c),LL(0x689f4e79,0x6d6c5f17),LL(0xb24ff1b6,0x7a544d91),LL(0xfe16cd8c,0x3e12a5eb),
   LL(0xa56b872f,0x543574e9),LL(0xfcf68ea2,0xa1ad550c),LL(0x3f560ef7,0x689e37d2),LL(0xc9d47a8b,0x8c54b9ca), LL(0x088ac342,0x46d40a4a),LL(0x1576c6d0,0xec450c7c),LL(0x1f9689e9,0xb589e31c),LL(0xb8781718,0xdacf2602),
   LL(0xc8cb6b42,0xa89237c6),LL(0xb96ef381,0x1326fc93),LL(0xb5f07825,0x55d56c6d),LL(0x7449e22d,0xacba2eea), LL(0x633c3000,0x74e0887a),LL(0xd7cbcf71,0xcb6cd172),LL(0xc36cf1be,0x309e81de),LL(0x60ae399b,0x07a18a6d),
   LL(0x9edce57e,0xb36c2679),LL(0xdf001d41,0x52b892f4),LL(0x16a1f2c6,0xd884ae5d),LL(0xefcc370a,0x9b329424), LL(0xbd2e21df,0x3120daf2),LL(0x02470a99,0x55298d2d),LL(0xa05db32e,0x0b78af6c),LL(0x601f5636,0x5c76a331),
   LL(0xf8a4f29c,0xaae861ff),LL(0xd68f8d49,0x70dc9240),LL(0x81b1321c,0x960e649f),LL(0x8792e4ce,0x3d2c801b), LL(0x42521876,0xf479f772),LL(0x416c79b1,0x0bed93bc),LL(0x263e5bc9,0xa67fbc05),LL(0x521db049,0x01e8e630),
   LL(0xc6f3431e,0x76f26738),LL(0xe3267541,0xe609cb02),LL(0x818c877c,0xb10cff2d),LL(0x786a13cb,0x1f0e75ce), LL(0x1158544d,0xf4fdca64),LL(0x6cb71ed0,0x5d777e89),LL(0xa9aa4755,0x3c233737),LL(0xe527ab40,0x7b453192),
   LL(0x39f05ffe,0xdb59f688),LL(0x6d82574e,0x8f4f4be0),LL(0xee292d1b,0xcce3450c),LL(0x61ccd086,0xaa448a12), LL(0xf7914967,0xabce91b3),LL(0x1908a5ed,0x4537f09b),LL(0xf51042e7,0xa812421e),LL(0xec0b3a34,0xfaf5cebc),
   LL(0x4ca6b39a,0x730ffd87),LL(0x02efd342,0x70fb72ed),LL(0xd75c8edb,0xeb4735f9),LL(0xc278aa51,0xc11f2157), LL(0xbf3bfebf,0xc459f635),LL(0x6bd9601f,0x3a1ff0b4),LL(0xc420cb73,0xc9d12823),LL(0x3c2915a3,0x3e9af3e2),
   LL(0xb41c3440,0xe0c82c72),LL(0xe3039a5f,0x175239e5),LL(0x558795a3,0xe1084b8a),LL(0xd01e5c60,0x328d0a1d), LL(0xd3788a04,0x0a495f2e),LL(0x66c11a9f,0x25d8ff16),LL(0x9ed692d6,0xf5155f05),LL(0x4f425fe4,0x954fa107),
   LL(0xe98aaa99,0xd16aabf2),LL(0x96b0f88a,0x90cd8ba0),LL(0xc154026a,0x957f4782),LL(0x52af56d2,0x54ee0734), LL(0x45b4147a,0xbcf89e54),LL(0x9a52816c,0x3d102f21),LL(0x39b62e77,0x6808517e),LL(0x69169ad8,0x92e25421),
   LL(0xbb608558,0xd721d871),LL(0xf6d4ff9b,0x60e4ebae),LL(0x41f2763e,0x0ba10819),LL(0x51ee3247,0xca2e45be), LL(0x2bfd7a5f,0x66d172ec),LL(0x74d0b12d,0x528a8f2f),LL(0xdabe70dc,0xe17f1e38),LL(0x9f93983c,0x1d5d7316),
   LL(0xdf423e31,0x51b2184a),LL(0xaedb1a10,0xcb417291),LL(0x625bcab9,0x2054ca93),LL(0xa98998f0,0x54396860), LL(0xa54ae57e,0x4e53f6c4),LL(0xee648e9d,0x0ffeb590),LL(0x6afaf6bc,0xfbbdaadc),LL(0xaa3bfb8a,0xf88ae796),
   LL(0xd2359ed9,0x209f1d44),LL(0xf3544ce2,0xac68dd03),LL(0xfd51e569,0xf378da47),LL(0x2cc80097,0xe1abd860), LL(0x343b6e3a,0x23ca18d9),LL(0xb40a1bae,0x480797e8),LL(0x533f3e67,0xd1f0c717),LL(0x06e6cdfc,0x44896970),
   LL(0x52a82e8d,0x8ca21055),LL(0x78460cdc,0xb2caf785),LL(0xe9037178,0x4c1b7b62),LL(0xdb514b58,0xefc09d2c), LL(0x9113be5c,0x5f2df9ee),LL(0xb3f9271c,0x2fbda78f),LL(0x8f83fc54,0xe09a81af),LL(0x8afb5141,0x06b13866),
   LL(0x43e3865d,0x38f6480f),LL(0x1ddf47d9,0x72dd77a8),LL(0x4c205ff7,0xf2a8e971),LL(0x9d088ad8,0x46d449d8), LL(0x185d706f,0x926619ea),LL(0xc7dd7f62,0xe47e02eb),LL(0x8cbc2031,0xe7f120a7),LL(0x998d4ac9,0xc18bef00),
   LL(0x6bdf22da,0x18f37a9c),LL(0x90dc82df,0xefbc432f),LL(0x5d703651,0xc52cef8e),LL(0xd99881a5,0x82887ba0), LL(0xb920ec1d,0x7cec9dda),LL(0xec3e8d3b,0xd0d7e8c3),LL(0x4ca88747,0x445bc395),LL(0x9fd53535,0xedeaa2e0),
   LL(0x6cc87475,0x461b1d93),LL(0x6d2383bd,0xd92a52e2),LL(0xd7903546,0xfabccb59),LL(0x3d14b112,0x6111a761), LL(0xb3d5f612,0x0ae584fe),LL(0x60e828ec,0x5ea69b8d),LL(0x54087030,0x6c078985),LL(0xac4821fe,0x649cab04),
   LL(0x8bdce214,0x25ecedcf),LL(0x86af7361,0xb5622f72),LL(0x7038b9e2,0x0e1227aa),LL(0xac20fa77,0xd0efb273), LL(0x79df975b,0x817ff88b),LL(0x1999503e,0x856bf286),LL(0x5038ec46,0xb4d5351f),LL(0xfc42af6e,0x740a52c5),
   LL(0x2cbb1a3f,0x2e38bb15),LL(0x17a83429,0xc3eb99fe),LL(0xdd66bb74,0xca4fcbf1),LL(0xcde5e8fc,0x880784d6), LL(0xb4e7a0be,0xddc84c1c),LL(0xbd15a72f,0x8780510d),LL(0x81ec30e1,0x44bcf1af),LL(0x0a61073e,0x141e50a8),
   LL(0x47be87ae,0x0d955718),LL(0xf76a4372,0x68a61417),LL(0xc607c3d3,0xf57e7e87),LL(0x5252f332,0x043afaf8), LL(0x1552a4d2,0xcc14e121),LL(0xbb4d4ab4,0xb6dee692),LL(0xa03816a4,0xb6ab74c8),LL(0x6f394a29,0x84001ae4),
   LL(0xd795fb45,0x5bed8344),LL(0xb79f55a5,0x57326e7d),LL(0x4accdffc,0xc9533ce0),LL(0x3993fa04,0x53473caf), LL(0xa13df4c8,0x7906eb93),LL(0x97cbe46f,0xa73e51f6),LL(0x0ae4ccf8,0xd1ab3ae1),LL(0x8a5b3dbc,0x25614508),
   LL(0x11a71b27,0x61eff962),LL(0x6bb7fa39,0xdf71412b),LL(0x2bd7f3ef,0xb31ba6b8),LL(0x69180d29,0xb0b9c415), LL(0x014cdde5,0xeec14552),LL(0x227b4bbb,0x702c624b),LL(0xd3e988f3,0x2b15e8c2),LL(0xa4f7fd04,0xee3bcc6d),
   LL(0x42ac6c85,0x9d00822a),LL(0x1df9f2b7,0x2db0cea6),LL(0x42de1e58,0xd7cad2ab),LL(0x2d6fbb61,0x346ed526), LL(0x1a2faf09,0xb3962995),LL(0x7c25612e,0x2fa8a580),LL(0x7cf56490,0x30ae04da),LL(0x0eea3961,0x75662908),
   LL(0x3d080847,0x3609f5c5),LL(0x5241d4f6,0xcb081d39),LL(0x77961a63,0xb4fb3810),LL(0x2abb66fc,0xc20c5984), LL(0xf902f245,0x3d40aa7c),LL(0x4e536b1e,0x9cb12736),LL(0x99b3134f,0x5eda24da),LL(0x5cd011af,0xafbd9c69),
   LL(0xc7088c7d,0x9a16e30a),LL(0x3207389f,0x5ab65710),LL(0xe7407a53,0x1b09547f),LL(0x4fdc6eab,0x2322f9d7), LL(0x7430de4d,0xc0f2f22d),LL(0xe68ca9a9,0x19382696),LL(0x918e5868,0x17f1eff1),LL(0x586f4204,0xe3b5b635),
   LL(0x3fbc4341,0x146ef980),LL(0x5b5eed4e,0x359f2c80),LL(0x7482e41d,0x9f35744e),LL(0xf3b224c2,0x9a9ac3ec), LL(0x91fc50ae,0x9161a6fe),LL(0xc613fa7c,0x89ccc66b),LL(0xc732f15a,0x89268b14),LL(0xb467ed03,0x7cd6f4e2),
   LL(0xce56b40e,0xfbf79869),LL(0xc02dde98,0xf93e094c),LL(0xedee2cd7,0xefe0c3a8),LL(0xb268fd42,0x90f3ffc0), LL(0x08241aed,0x81a7fd56),LL(0x00b1afe8,0x95ab7ad8),LL(0x3e310d52,0x40127056),LL(0x09d9fc43,0xd3ffdeb1),
   LL(0xd11a8594,0xc8f85c91),LL(0x31cf6db8,0x2e74d258),LL(0x02b5dfd0,0x829c7ca3),LL(0x69143c86,0xe389cfbe), LL(0x941768d8,0xd01b6405),LL(0x03bf825d,0x45103995),LL(0x56cd17e2,0xcc4ee166),LL(0xba037e79,0xbea3c283),
   LL(0xd9a47520,0x4e1ac06e),LL(0xaf852404,0xfbfe18aa),LL(0x8087648a,0x5615f8e2),LL(0xb9d150d9,0x7301e47e), LL(0xb299b977,0x79f9f9dd),LL(0xa5b78314,0x76697a7b),LL(0x7d7c90e7,0x10d67468),LL(0x937210b5,0x7afffe03),
   LL(0x28c22cee,0x5aef3e4b),LL(0x09fd55ae,0xefb0ecd8),LL(0x0d2a5d6a,0x4cea7132),LL(0x01db6357,0x9cfb5fa1), LL(0xf36e1ac5,0x395e0b57),LL(0x36cafb7d,0x008fa9ad),LL(0x5308c4db,0x8f6cdf70),LL(0x95ed2477,0x51527a37),
   LL(0x5bd21311,0xba0dee30),LL(0x909c90d7,0x6ed41b22),LL(0x7c8696d3,0xc5f6b758),LL(0x3ce83a80,0x0db8eaa8), LL(0xb24b4b6f,0xd297fe37),LL(0x522d1f0d,0xfe58afe8),LL(0x8c98dbd9,0x97358736),LL(0x9454a527,0x6bc226ca),
   LL(0xce53c2d0,0xa12b384e),LL(0x5e4606da,0x779d897d),LL(0x73ec12b0,0xa53e47b0),LL(0x5756f1ad,0x462dbbba), LL(0xcafe37b6,0x69fe09f2),LL(0xecce2e17,0x273d1ebf),LL(0x3cf607fd,0x8ac1d538),LL(0x12e10c25,0x8035f7ff),
},
/* digit=21 base_pwr=2^147 */
{
   LL(0x7e6c5520,0x854d34c7),LL(0xdcb9ea58,0xc27df9ef),LL(0xd686666d,0x405f2369),LL(0x0417aa85,0x29d1febf), LL(0x93470afe,0x9846819e),LL(0xe2a27f9e,0x3e6a9669),LL(0xe31e6504,0x24d008a2),LL(0x9cb7680a,0xdba7cecf),
   LL(0x338d6e43,0xecaff541),LL(0x4541d5cc,0x56f7dd73),LL(0x96bc88ca,0xb5d426de),LL(0x9ed3a2c3,0x48d94f6b), LL(0x2ef8279c,0x6354a3bb),LL(0x0b1867f2,0xd575465b),LL(0x95225151,0xef99b0ff),LL(0xf94500d8,0xf3e19d88),
   LL(0xe32dd620,0x92a83268),LL(0x627849a2,0x913ec99f),LL(0x2c378882,0xedd8fdfa),LL(0xee6f8cfe,0xaf96f33e), LL(0xdc3fa8a5,0xc06737e5),LL(0xb0b03a1d,0x236bb531),LL(0x89f037b0,0x33e59f29),LL(0xd9a12a53,0x13f9b5a7),
   LL(0x51efb310,0x0d0df6ce),LL(0x958df5be,0xcb5b2eb4),LL(0x36158e59,0xd6459e29),LL(0x1466e336,0x82aae2b9), LL(0x411aa636,0xfb658a39),LL(0xd4c0a933,0x7152ecc5),LL(0x49f026b7,0xf10c758a),LL(0xcb09311f,0xf4837f97),
   LL(0xc753c45f,0xddfb02c4),LL(0xf9c840fe,0x18ca81b6),LL(0xb0f8a3e6,0x846fd09a),LL(0xe7733dbc,0xb1162add), LL(0x236e3ab6,0x7070ad20),LL(0xb2a56326,0xf88cdaf5),LL(0x997cbc7a,0x05fc8719),LL(0x4b665272,0x442cd452),
   LL(0xb71698f5,0x7807f364),LL(0x9f7b605e,0x6ba418d2),LL(0xa03b2cbb,0xfd20b00f),LL(0xda54386f,0x883eca37), LL(0xf3437f24,0xff0be43f),LL(0xa48bb33c,0xe910b432),LL(0x329df765,0x4963a128),LL(0xbe2fe6f7,0xac1dd556),
   LL(0x24a0a3fc,0x557610f9),LL(0xe881c3f9,0x38e17bf4),LL(0xed0dac99,0x6ba84faf),LL(0x59eeb918,0xd4a222c3), LL(0x13f542b6,0xc79c1dbe),LL(0xe425d457,0x1fc65e0d),LL(0x1debb779,0xeffb754f),LL(0x9e08af60,0x638d8fd0),
   LL(0x626332d5,0x994f523a),LL(0x5561bb44,0x7bc38833),LL(0x3d845ea2,0x005ed4b0),LL(0xc2a1f08a,0xd39d3ee1), LL(0xe7676b0d,0x6561fdd3),LL(0xfb706017,0x620e35ff),LL(0xf264f9a8,0x36ce424f),LL(0xda2681f7,0xc4c3419f),
   LL(0x69beb6e8,0xfb6afd2f),LL(0x6d700d03,0x3a50b993),LL(0x0c83a14f,0xc840b2ad),LL(0x54085bef,0x573207be), LL(0x09fe7e5b,0x5af882e3),LL(0x3b40a7e1,0x957678a4),LL(0x543056e2,0x172d4bdd),LL(0x0df13c0a,0x9c1b26b4),
   LL(0xf405ff06,0x1c30861c),LL(0x486e828b,0xebac86bd),LL(0x636933fc,0xe791a971),LL(0x7aeee947,0x50e7c2be), LL(0xfa90d767,0xc3d4a095),LL(0xe670ab7b,0xae60eb7b),LL(0x397b056d,0x17633a64),LL(0x105012aa,0x93a21f33),
   LL(0xabb88643,0x663c370b),LL(0x22e21599,0x91df36d7),LL(0x8b761671,0x183ba835),LL(0x728f3bf1,0x381eea1d), LL(0x39966e6c,0xb9b2f1ba),LL(0xe7295492,0x7c464a28),LL(0x09b26b7f,0x0fd5f70a),LL(0xfbe009df,0xa9aba1f9),
   LL(0x369b87ad,0x857c1f22),LL(0x32fca556,0x3c00e5d9),LL(0x90b06466,0x1ad74cab),LL(0x550faaf2,0xa7112386), LL(0x6d9bd5f5,0x7435e198),LL(0x59c3463f,0x2dcc7e38),LL(0xca7bd4b2,0xdc7df748),LL(0x9dec2f31,0x13cd4c08),
   LL(0xe3237710,0x0d3b5df8),LL(0xcbd2f7b0,0x0dadb26e),LL(0xe4aa082b,0x9f5966ab),LL(0x350e966e,0x666ec8de), LL(0xee524216,0x1bfd1ed5),LL(0x41dab0b6,0xcd93c59b),LL(0xd186d6ba,0x658a8435),LL(0x159d1195,0x1b7d34d2),
   LL(0x22caf46b,0x5936e460),LL(0x9a96fe4f,0x6a45dd8f),LL(0xb98f474e,0xf7925434),LL(0x0053ef15,0x41410412), LL(0x41de97bf,0x71cf8d12),LL(0xbd80bef4,0xb8547b61),LL(0xc4db0037,0xb47d3970),LL(0xfef20dff,0xf1bcd328),
   LL(0x10caad67,0x31a92e09),LL(0x5531a1e1,0x1f591960),LL(0x5f4fc840,0x3bb852e0),LL(0x93a72c6c,0x63e297ca), LL(0x49abad67,0x3c2b0b2e),LL(0xed3db0d9,0x6ec405fc),LL(0x7fef1d40,0xdc14a530),LL(0x280896fc,0xccd19846),
   LL(0x9bb81648,0x00f83176),LL(0x653120d0,0xd69eb485),LL(0x4ccabc62,0xd17d75f4),LL(0xb749fcb1,0x34a07f82), LL(0xbbfb5554,0x2c3af787),LL(0x62e283f8,0xb06ed4d0),LL(0xa19213a0,0x5722889f),LL(0xdcf3c7b4,0x162b085e),
   LL(0xe0dd3eca,0xbcaecb31),LL(0xe52f13a5,0xc6237fbc),LL(0x27bac297,0xcc2b6b03),LL(0xb917f54a,0x2ae1cac5), LL(0x7845ae4f,0x474807d4),LL(0xce5972e0,0xfec7dd92),LL(0x1d7915bb,0xc3bd2541),LL(0xd94907ca,0x66f85dc4),
   LL(0xbdbcf0ca,0xd981b888),LL(0xdf279e9f,0xd75f5da6),LL(0x7054e934,0x128bbf24),LL(0x81db134b,0x3c6ff6e5), LL(0x047d26e4,0x795b7cf4),LL(0x5049ec37,0xf370f7b8),LL(0xced945af,0xc6712d4d),LL(0x095642bc,0xdf30b5ec),
   LL(0x4896246e,0x9b034c62),LL(0xee90bbd1,0x5652c016),LL(0x87fedb73,0xeb38636f),LL(0x0135a613,0x5e32f847), LL(0xcf933c83,0x0703b312),LL(0x1a7f47e6,0xd05bb76e),LL(0x949c2415,0x825e4f0c),LL(0x7250d6f8,0x569e5622),
   LL(0x6568013e,0xbbe9eb3a),LL(0x22f243fc,0x8dbd203f),LL(0xb342734a,0x9dbd7694),LL(0x46afa984,0x8f6d12f8), LL(0xc9eade29,0xb98610a2),LL(0x47dd0f18,0xbab4f323),LL(0x671c0d46,0x5779737b),LL(0xd3e0a42a,0x10b6a7c6),
   LL(0x3035b41c,0xfb19ddf3),LL(0x99c45895,0xd336343f),LL(0x54c857e5,0x61fe4938),LL(0xae4e57d5,0xc4d506be), LL(0xbbc33f75,0x3cd8c8cb),LL(0x9262c77d,0x7281f08a),LL(0xf11a2823,0x083f4ea6),LL(0x9fba2e33,0x8895041e),
   LL(0x9c438edf,0xfcdfea49),LL(0x91edba44,0x7678dcc3),LL(0xe2ba50f0,0xf07b3b87),LL(0x43948c1b,0xc13888ef), LL(0x1140af42,0xc2135ad4),LL(0x926ed1a7,0x8e5104f3),LL(0x88f6695f,0xf24430cb),LL(0x6d73c120,0x0ce0637b),
   LL(0xfe631e8f,0xb2db01e6),LL(0xd7bdd24b,0x1c5563d7),LL(0x369ad44f,0x8daea3ba),LL(0x8187a9f9,0x000c81b6), LL(0xaae1fd9a,0x5f48a951),LL(0x8d5aed8a,0xe35626c7),LL(0x0498c622,0x20952763),LL(0x773aa504,0x76d17634),
   LL(0xeb300f7a,0x36d90dda),LL(0xedb5e801,0x9dcf7dfc),LL(0x74d5244c,0x645cb268),LL(0x348e3aa2,0xa127ee79), LL(0x575f1dbb,0x488acc53),LL(0x80e6161e,0x95037e85),LL(0x292650d0,0x57e59283),LL(0x14938216,0xabe67d99),
   LL(0x3f8e1065,0x3c7f944b),LL(0x330e8924,0xed908cb6),LL(0x6f530136,0x08ee8fd5),LL(0xd7ffc169,0x2227b7d5), LL(0xb5cd6dd5,0x4f55c893),LL(0xa62796e8,0x82225e11),LL(0xcb18e12c,0x5c6cead1),LL(0x84f5a51a,0x4381ae0c),
   LL(0x7fafa4c8,0x345913d3),LL(0x0491aac0,0x3d918082),LL(0x3e69264c,0x9347871f),LL(0xb4f4f0cd,0xbea9dd3c), LL(0x3eadd3e7,0xbda5d067),LL(0x0573bcd8,0x0033c1b8),LL(0x5da2486c,0x25589379),LL(0x86abbee7,0xcb89ee5b),
   LL(0x22532e5d,0x8fe0a8f3),LL(0x727dfc4c,0xb6410ff0),LL(0x226726db,0x619b9d58),LL(0x7a2b2dc7,0x5ec25669), LL(0x4c3beb01,0xaf4d2e06),LL(0x7acea556,0x852123d0),LL(0xf783487a,0x0e9470fa),LL(0x5664b3eb,0x75a7ea04),
   LL(0x6798e4ba,0x4ad78f35),LL(0xc7d0e091,0x9214e6e5),LL(0xb1290403,0xc420b488),LL(0xfc295749,0x64049e0a), LL(0x3ae9841f,0x03ef5af1),LL(0xb0b662a6,0xdbe4ca19),LL(0xfa453458,0x46845c5f),LL(0x10b66722,0xf8dabf19),
   LL(0xcce2793b,0xb650f0aa),LL(0xc5ec47c1,0x71db851e),LL(0x3b234fa9,0x3eb78f3e),LL(0xfc0106ce,0xb0c60f35), LL(0x774eadbd,0x05427121),LL(0xce323863,0x25367faf),LL(0xcd086976,0x7541b5c9),LL(0xdc507ad1,0x4ff069e2),
   LL(0x8776e667,0x74145256),LL(0xb23c6bb5,0x6e76142c),LL(0x1b3a8a87,0xdbf30712),LL(0x98450836,0x60e7363e), LL(0xb7366d80,0x5741450e),LL(0x4837dbdf,0xe4ee14ca),LL(0x69d4316f,0xa765eb9b),LL(0x8ef43825,0x04548dca),
   LL(0x5ae888eb,0x9c9f4e4c),LL(0x56e9ac99,0x733abb51),LL(0xba6ac029,0xdaad3c20),LL(0x2ba3e38e,0x9b8dd3d3), LL(0x0bc5d11a,0xa9bb4c92),LL(0x9c5f88a3,0xf20127a7),LL(0x161d3cb8,0x4f52b06e),LL(0x6afaf0a6,0x26c1ff09),
   LL(0x7189e71f,0x32670d2f),LL(0x5ecf91e7,0xc6438748),LL(0xdb757a21,0x15758e57),LL(0x290a9ce5,0x427d09f8), LL(0x38384a7a,0x846a308f),LL(0xb0732b99,0xaac3acb4),LL(0x17845819,0x9e941009),LL(0xa7ce5e03,0x95cba111),
   LL(0xb00009c4,0x6f3d4f7f),LL(0x8ff28b5f,0xb8396c27),LL(0x1c97975d,0xb1a9ae43),LL(0xe5d9fed5,0x9d7ba8af), LL(0x34f485b6,0x338cf09f),LL(0x64122516,0xbc0ddacc),LL(0x05d471fe,0xa450da12),LL(0x628dd8c9,0x4c3a6250),
   LL(0xd1295837,0x69c7d103),LL(0x3807eb2f,0xa2893e50),LL(0xbdb41491,0xd6e1e1de),LL(0x5e138235,0xc630745b), LL(0x48661ae1,0xc892109e),LL(0xea2b2674,0x8d17e7eb),LL(0xc328d6b5,0x00ec0f87),LL(0xf079ff9e,0x6d858645),
   LL(0x19115ead,0x6cdf243e),LL(0x4bac4fcf,0x1ce1393e),LL(0x9c29f25b,0x2c960ed0),LL(0x9d388a05,0x59be4d8e), LL(0xd0def72b,0x0d46e06c),LL(0xe0342748,0xb923db5d),LL(0x936d4a3d,0xf7d3aacd),LL(0x0b0b099e,0x558519cc),
   LL(0x827097ef,0x3ea8ebf8),LL(0xd054f55d,0x259353db),LL(0x6d2ed089,0x84c89abc),LL(0x8e096a7c,0x5c548b69), LL(0x994b995d,0xd587f616),LL(0xa5845601,0x4d1531f6),LL(0x451fd9f0,0x792ab31e),LL(0x65adf6ca,0xc8b57bb2),
   LL(0x1cd5ad73,0x68440fcb),LL(0x6144da4f,0xb9c860e6),LL(0x8462beb8,0x2ab286aa),LL(0xef46797f,0xcc6b8fff), LL(0x20c8a471,0xac820da4),LL(0x77ff7faf,0x69ae05a1),LL(0xbfb5da77,0xb9163f39),LL(0x2c73ab7a,0xbd03e590),
   LL(0xb2940d9e,0x7e862b5e),LL(0x4b9af564,0x3c663d86),LL(0xbde3033d,0xd8309031),LL(0xd42c5bc6,0x298231b2), LL(0x552ad093,0x42090d2c),LL(0xff854695,0xa4799d1c),LL(0xd31f0d00,0x0a88b5d6),LL(0xa2f26b46,0xf8b40825),
   LL(0xf1bd7218,0xec29b1ed),LL(0x4b24c86e,0xd491c53b),LL(0x3395ea65,0xd2fe588f),LL(0x4456ef15,0x6f3764f7), LL(0xcdc34800,0xdb43116d),LL(0xc1e33955,0xcdbcd456),LL(0x74ab286b,0xefdb5540),LL(0xd18c5d7c,0x948c7a51),
   LL(0x7378058e,0xeb81aa37),LL(0x04411154,0x41c746a1),LL(0xfb828ac7,0xa10c73bc),LL(0x9d972b29,0x6439be91), LL(0x43a2fbad,0x4bf3b4b0),LL(0x82b5e840,0x39e6dadf),LL(0x6397bd4c,0x4f716408),LL(0x7f1eeccb,0x0f7de568),
   LL(0xd2ffbfc1,0x5865c5a1),LL(0x4ccb6451,0xf74211fa),LL(0xc0b32558,0x66368a88),LL(0x9ad7812e,0x5b539dc2), LL(0x2f3af6f6,0x579483d0),LL(0x99934ece,0x52132078),LL(0xdcc9e983,0x50b9650f),LL(0xaee42b8a,0xca989ec9),
   LL(0xd6f62f99,0x6a44c829),LL(0x4c2a7c0c,0x8f06a309),LL(0x98a0cb0a,0x4ea2b3a0),LL(0xbeee8364,0x5c547b70), LL(0x682afe11,0x461d40e1),LL(0x7b41c0a8,0x9e0fc77a),LL(0xe20d5d36,0x79e4aefd),LL(0x32dd9f63,0x2916e520),
   LL(0x3f883faf,0xf59e52e8),LL(0x2b868d35,0x396f9639),LL(0x4ca19881,0xc902a9df),LL(0xdb2401a6,0x0fc96822), LL(0x66f1c68d,0x41237587),LL(0xfb476c0d,0x10fc6de3),LL(0x841f5d90,0xf8b6b579),LL(0xfa24f44a,0x2ba8446c),
   LL(0xef4a9975,0xa237b920),LL(0x2330435f,0x60bb6004),LL(0xcfb7e7b5,0xd6f4ab5a),LL(0x83435391,0xb2ac5097), LL(0xb0d1ea67,0xf036ee2f),LL(0x74c56230,0xae779a6a),LL(0xab838ae6,0x59bff8c8),LL(0x9b38e6f0,0xcd83ca99),
   LL(0xe33deed3,0xbb27bef5),LL(0x001892a8,0xe6356f6f),LL(0x7adfbd3e,0xbf3be6cc),LL(0x33d1ac9d,0xaecbc81c), LL(0xe6e861dc,0xe4feb909),LL(0x53f5f801,0x90a247a4),LL(0x27346e57,0x01c50acb),LL(0x461acc1b,0xce29242e),
   LL(0x2f998a91,0x04dd214a),LL(0xd4baf27b,0x271ee9b1),LL(0xe8c26722,0x7e3027d1),LL(0x1820dce5,0x21d1645c), LL(0x7501779c,0x086f242c),LL(0xfa0e8009,0xf0061407),LL(0x60187129,0xf23ce477),LL(0x0fde9bd0,0x05bbdedb),
   LL(0x25d98473,0x682f4832),LL(0x5c658427,0xf207fe85),LL(0x4166ffa1,0xb6fdd7ba),LL(0x9eed799d,0x0c314056), LL(0x4107e28f,0x0db8048f),LL(0x41216840,0x74ed3871),LL(0x56a3c06e,0x74489f8f),LL(0x12777134,0x1e1c005b),
   LL(0xf37ec3c3,0xdb332a73),LL(0xdd59eba0,0xc65259bd),LL(0xdb4d3257,0x2291709c),LL(0xbd389390,0x9a793b25), LL(0xe43756f0,0xf39fe34b),LL(0x9afb56c9,0x2f76bdce),LL(0x61208b27,0x9f37867a),LL(0x089972c3,0xea1d4307),
   LL(0x8bdf623a,0x8c595330),LL(0x8441fb7d,0x5f5accda),LL(0x32ddfd95,0xfafa9418),LL(0x0fde9be7,0x6ad40c5a), LL(0xaeca8709,0x43faba89),LL(0x2c248a9d,0xc64a7cf1),LL(0x72637a76,0x16620252),LL(0x22b8d1bb,0xaee1c791),
   LL(0x21a843b2,0xf0f798fd),LL(0x8d005cb1,0x56e4ed4d),LL(0x1f0d8abe,0x355f7780),LL(0x34522326,0x197b04cf), LL(0xfd42c13f,0x41f9b31f),LL(0xb40f933d,0x5ef7feb2),LL(0x5d60bad4,0x27326f42),LL(0x8c92cf89,0x027ecdb2),
   LL(0x4e3352fe,0x04aae4d1),LL(0x73591b90,0x08414d2f),LL(0xb7da7d60,0x5ed6124e),LL(0x4d13d4ec,0xb985b931), LL(0x96bf36f9,0xa592d3ab),LL(0xbbdf51df,0x012dbed5),LL(0xdf6c177d,0xa57963c0),LL(0x87ca29cf,0x010ec869),
   LL(0xbf926dff,0xba1700f6),LL(0xf4bf6bc2,0x7c9fdbd1),LL(0x64da11f5,0xdc18dc8f),LL(0xd938ae75,0xa6074b7a), LL(0xe84f44a4,0x14270066),LL(0xd27b954e,0x99998d38),LL(0xb4f38e9a,0xc1be8ab2),LL(0x15c01016,0x8bb55bbf),
   LL(0x0ea2ab30,0xf73472b4),LL(0xf73d68dd,0xd365a340),LL(0x19c2e1eb,0xc01a7168),LL(0x34061719,0x32f49e37), LL(0x01d8b4d6,0xb73c57f1),LL(0x26b47700,0x03c8423c),LL(0xa4d8826a,0x321d0bc8),LL(0x4bc0e638,0x6004213c),
   LL(0xc1c06681,0xf78c64a1),LL(0xef018e50,0x16e0a16f),LL(0xdb42b2b3,0x31cbdf91),LL(0xe0d36f58,0xf8f4ffce), LL(0x4cc5e3e0,0xcdcc71cd),LL(0xa129e3e0,0xd55c7cfa),LL(0x0fb2cbf1,0xccdb6ba0),LL(0xc4bce3cb,0x6aba0005),
   LL(0xd232cfc4,0x501cdb30),LL(0xd58a3cef,0x9ddcf12e),LL(0x87e09149,0x02d2cf9c),LL(0x2c976257,0xdc5d7ec7), LL(0x0b50d7dd,0x6447986e),LL(0x807f112a,0x88fdbaf7),LL(0xb00ae9f6,0x58c9822a),LL(0x6d3d27e0,0x6abfb950),
   LL(0x8a429f4f,0xd0a74487),LL(0xdb516609,0x0649712b),LL(0xe769b5df,0xb826ba57),LL(0x1fc7aaf2,0x82335df2), LL(0x5c93d995,0x2389f067),LL(0x68677be6,0x59ac367a),LL(0x21d9951b,0xa77985ff),LL(0x85011cce,0x038956fb),
   LL(0xbb734e37,0x608e48cb),LL(0x2be5b26f,0xc08c0bf2),LL(0xf9b1a0d9,0x17bbdd3b),LL(0x10483319,0xeac7d898), LL(0xbc1a6dea,0xc95c4baf),LL(0x172aafdb,0xfdd0e2bf),LL(0x8235c41a,0x40373cbc),LL(0xfb6f41d5,0x14303f21),
   LL(0x0408f237,0xba063621),LL(0xecd2d1ed,0xcad3b09a),LL(0x52abb6a2,0x4667855a),LL(0xaa8b417b,0xba9157dc), LL(0x4f013efb,0xfe7f3507),LL(0xaa38c4a2,0x1b112c4b),LL(0x9ba64345,0xa1406a60),LL(0x6993c80b,0xe53cba33),
   LL(0xded40d23,0x45466063),LL(0x54908e25,0x3d5f1f4d),LL(0x403c3c31,0x9ebefe62),LL(0x0672a624,0x274ea0b5), LL(0x451d1b71,0xff818d99),LL(0x8f79cf79,0x80e82643),LL(0x73ce37f5,0xa165df13),LL(0xfe3a21fd,0xa744ef4f),
   LL(0xcf551396,0x73f1e7f5),LL(0x868c676b,0xc616898e),LL(0x8c442c36,0x671c28c7),LL(0x5e0a317d,0xcfe5e558), LL(0x7051f476,0x1242d818),LL(0x14f03442,0x56fad2a6),LL(0x0a44d0f6,0x262068bc),LL(0xce6edf4e,0xdfa2cd6e),
   LL(0xd15d1517,0x0f43813a),LL(0x377d44f5,0x61214cb2),LL(0xc639b35f,0xd399aa29),LL(0x54c51c19,0x42136d71), LL(0x08417221,0x9774711b),LL(0x52545a57,0x0a5546b3),LL(0x1150582d,0x80624c41),LL(0xfbc555bc,0x9ec5c418),
   LL(0x771849f1,0x2c87dcad),LL(0x01d7bf6f,0xb0c932c5),LL(0x89116eb2,0x6aa5cd3e),LL(0x51ca7bd3,0xd378c25a), LL(0x9e6e3e31,0xc612a0da),LL(0xb68ad5d0,0x0417a54d),LL(0x22c6edb8,0x00451e4a),LL(0xb42827ce,0x9fbfe019),
   LL(0xba9384a2,0x2fa92505),LL(0x64ad69c1,0x21b8596e),LL(0x983b35a6,0x8f4fcc49),LL(0x72754672,0xde093760), LL(0xf7bffe6d,0x2f14ccc8),LL(0x5d94263d,0x27566bff),LL(0x2df3ec30,0xb5b4e9c6),LL(0x3e6ea6ba,0x94f1d7d5),
   LL(0xaaca5e9b,0x97b7851a),LL(0x56713b97,0x518aa521),LL(0x150a61f6,0x3357e8c7),LL(0xec2c2b69,0x7842e7e2), LL(0x6868a548,0x8dffaf65),LL(0xe068fc81,0xd963bd82),LL(0x65917733,0x64da5c8b),LL(0x7b247328,0x927090ff),
},
/* digit=22 base_pwr=2^154 */
{
   LL(0xd298c241,0x214bc9a7),LL(0x56807cfd,0xe3b697ba),LL(0x4564eadb,0xef1c7802),LL(0xb48149c5,0xdde8cdcf), LL(0x5a4d2604,0x946bf0a7),LL(0x6c1538af,0x27154d7f),LL(0xde5b1fcc,0x95cc9230),LL(0x66864f82,0xd88519e9),
   LL(0x7cb1282c,0xb828dd1a),LL(0xbe46973a,0xa08d7626),LL(0xe708d6b2,0x6baf8d40),LL(0x4daeb3f3,0x72571fa1), LL(0xf22dfd98,0x85b1732f),LL(0x0087108d,0x87ab01a7),LL(0x5988207a,0xaaaafea8),LL(0x69f00755,0xccc832f8),
   LL(0x36ff3bf0,0x964d950e),LL(0xf0b34638,0x8ad20f6f),LL(0xb5d7585f,0x4d9177b3),LL(0xef3f019f,0xcf839760), LL(0x8288c545,0x582fc5b3),LL(0x13116bd1,0x2f8e4e9b),LL(0x332120ef,0xf91e1b2f),LL(0x2a17dd23,0xcf568724),
   LL(0xca8d9d1a,0x488f1185),LL(0xd987ded2,0xadf2c77d),LL(0x60c46124,0x5f3039f0),LL(0x71e095f4,0xe5d70b75), LL(0x6260e70f,0x82d58650),LL(0xf750d105,0x39d75ea7),LL(0x75bac364,0x8cf3d0b1),LL(0x21d01329,0xf3a7564d),
   LL(0x2f52d2a7,0x182f04cd),LL(0xe2df565a,0x4fde149a),LL(0xa79fb2f7,0xb80c5eec),LL(0x22ddc897,0xab491d7b), LL(0xc6312c7f,0x99d76c18),LL(0x6aa41a57,0xca0d5f3d),LL(0xd15363a0,0x71207325),LL(0xbeb252c2,0xe82aa265),
   LL(0xec3128c2,0x94ab4700),LL(0x8e383f49,0x6c76d862),LL(0xc03024eb,0xdc36b150),LL(0x53daac69,0xfb439477), LL(0x8dc79623,0xfc68764a),LL(0xb440fbb2,0x5b86995d),LL(0xccc5ee0d,0xd66879bf),LL(0x95aa8bd3,0x05228942),
   LL(0x1e6a75c1,0xb51a40a5),LL(0x0ea7d817,0x24327c76),LL(0x07774597,0x06630182),LL(0x97fa7164,0xd6fdbec3), LL(0x13c90f48,0x20c99dfb),LL(0x686ef263,0xd6ac5273),LL(0xfef64eeb,0xc6a50bdc),LL(0x86fdfc32,0xcd87b281),
   LL(0x3fcd3efc,0xb24aa43e),LL(0xb8088e9a,0xdd26c034),LL(0xbd3d46ea,0xa5ef4dc9),LL(0x8a4c6a6f,0xa2f99d58), LL(0x2f1da46c,0xddabd355),LL(0x1afacdd1,0x72c3f8ce),LL(0x92d40578,0xd90c4eee),LL(0xca623b94,0xd28bb41f),
   LL(0x745edc11,0x50fc0711),LL(0x3dc87558,0x9dd9ad7d),LL(0xb49d1e64,0xce6931fb),LL(0xc98bd0f9,0x6c77a0a2), LL(0x6baf7cb1,0x62b9a629),LL(0xccf72d22,0xcf065f91),LL(0x79639071,0x7203cce9),LL(0xf9cb732f,0x09ae4885),
   LL(0xee8314f3,0x5e7c3bec),LL(0xdbea298f,0x1c068aed),LL(0x7c80acec,0x08d381f1),LL(0xe330495b,0x03b56be8), LL(0x9222882d,0xaeffb8f2),LL(0xc4af8bf7,0x95ff38f6),LL(0x1fc57d8c,0x50e32d35),LL(0x17b444f0,0x6635be52),
   LL(0xa5177900,0x04d15276),LL(0xf6858752,0x4e1dbb47),LL(0xc615796c,0x5b475622),LL(0x691867bf,0xa6fa0387), LL(0x2844c6d0,0xed7f5d56),LL(0x03a2477d,0xc633cf9b),LL(0x2d3721d6,0xf6be5c40),LL(0xe9fd68e6,0xaf312eb7),
   LL(0xe7417ce1,0x242792d2),LL(0x970ee7f5,0xff42bc71),LL(0x5c67a41e,0x1ff4dc6d),LL(0x20882a58,0x77709b7b), LL(0xbe217f2c,0x3554731d),LL(0x5bb72177,0x2af2a8cd),LL(0x591dd059,0x58eee769),LL(0x4bba6477,0xbb2930c9),
   LL(0x7d930cfc,0x863ee047),LL(0x396fd1f4,0x4c262ad1),LL(0x039af7e1,0xf4765bc8),LL(0x5ba104f6,0x2519834b), LL(0xd105f961,0x7cd61b4c),LL(0xd63bca54,0xa5415da5),LL(0x88a1f17c,0x778280a0),LL(0x2329512c,0xc4968949),
   LL(0xcecdaa7a,0x174a9126),LL(0x0b13247b,0xfc8c7e0e),LL(0x3484c1c4,0x29c110d2),LL(0x831dfc3b,0xf8eb8757), LL(0xc0067452,0x022f0212),LL(0x7b9b926c,0x3f6f69ee),LL(0xef42daf4,0x09032da0),LL(0x83f80de4,0x79f00ade),
   LL(0x81236c97,0x6210db71),LL(0x3ee0781f,0x74f7685b),LL(0xa3e41372,0x4df7da7b),LL(0xb1a1553e,0x2aae38b1), LL(0xf6dd9d1b,0x1688e222),LL(0x5b8b6487,0x57695448),LL(0x4b2edeaa,0x478d2127),LL(0x1e85956a,0xb2818fa5),
   LL(0xf176f2c0,0x1e6addda),LL(0xe2572658,0x01ca4604),LL(0x85342ffb,0x0a404ded),LL(0x441838d6,0x8cf60f96), LL(0xc9071c4a,0x9bbc691c),LL(0x34442803,0xfd588744),LL(0x809c0d81,0x97101c85),LL(0x8c456f7f,0xa7fb754c),
   LL(0xd51805e1,0xc95f3c5c),LL(0xb299dca8,0xab4ccd39),LL(0x47eaf500,0x3e03d20b),LL(0xd7b80893,0xfa3165c1), LL(0xe160e552,0x005e8b54),LL(0x9019d11f,0xdc4972ba),LL(0x0c9a4a7a,0x21a6972e),LL(0x37840fd7,0xa52c258f),
   LL(0xc1e99d81,0xf8559ff4),LL(0xa3c617c0,0x08e1a7d6),LL(0x248c6ba7,0xb398fd43),LL(0xd1283794,0x6ffedd91), LL(0xd629d208,0x8a6a59d2),LL(0x3490530e,0xa9d141d5),LL(0x38505989,0x42f6fc18),LL(0x479d94ee,0x09bf250d),
   LL(0xb3822790,0x223ad3b1),LL(0x93b8971c,0x6c5926c0),LL(0x75f7fa62,0x609efc7e),LL(0x1ec2d989,0x45d66a6d), LL(0x987d2792,0x4422d663),LL(0x3eb31d2b,0x4a73caad),LL(0xa32cb9e6,0xf06c2ac1),LL(0x91aeba84,0xd9445c5f),
   LL(0xaf71013f,0x6af7a1d5),LL(0x0bedc946,0xe68216e5),LL(0xd27370a0,0xf4cba30b),LL(0x870421cc,0x7981afbf), LL(0x9449f0e1,0x02496a67),LL(0x0a47edae,0x86cfc4be),LL(0xb1feca22,0x3073c936),LL(0x03f8f8fb,0xf5694612),
   LL(0x901515ea,0xd063b723),LL(0x749cf038,0x4c6c77a5),LL(0xab9e5059,0x6361e360),LL(0xa76a37c0,0x596cf171), LL(0x6530ae7a,0x800f53fa),LL(0x0792a7a6,0x0f5e631e),LL(0xefdb81c9,0x5cc29c24),LL(0x3f9c40ba,0xa269e868),
   LL(0x2cb7191e,0xec14f9e1),LL(0xe5b08ea6,0x78ea1bd8),LL(0x46332bb9,0x3c65aa9b),LL(0xbf80ce25,0x84cc22b3), LL(0xd49d5bf1,0x0098e9e9),LL(0x19087da4,0xcd4ec1c6),LL(0xaef6e357,0x3c9d07c5),LL(0x9f8f64b8,0x839a0268),
   LL(0xc6d8607f,0xc5e9eb62),LL(0x6aa995e4,0x759689f5),LL(0xbbb48317,0x70464669),LL(0xe402417d,0x921474bf), LL(0x2a354c8c,0xcabe135b),LL(0x812fa4b5,0xd51e52d2),LL(0x53311fe8,0xec741096),LL(0xb864514b,0x4f774535),
   LL(0x5bde48f8,0xbcadd671),LL(0x2189bc7d,0xc9703873),LL(0xc709ee8a,0x5d45299e),LL(0x845aaff8,0xd1287ee2), LL(0xdb1dbf1f,0x7d1f8874),LL(0x990c88d6,0xea46588b),LL(0x84368313,0x60ba649a),LL(0x60d543ae,0xd5fdcbce),
   LL(0x810d5ab0,0x90b46d43),LL(0x04d7e5cc,0x6739d8f9),LL(0x0d337c33,0x021c1a58),LL(0x68e67c40,0x00a61162), LL(0x379f0a1f,0x95ef413b),LL(0xe9e2ab95,0xfe126605),LL(0x2f5f199c,0x67578b85),LL(0x2cb84913,0xf5c00329),
   LL(0x37577dd8,0xf7956430),LL(0x29c5fe88,0x83b82af4),LL(0xcdbdc132,0x9c1bea26),LL(0x9c04339e,0x589fa086), LL(0xb13799df,0x033e9538),LL(0xd295d034,0x85fa8b21),LL(0xbd9ddcca,0xdf17f73f),LL(0xddb66334,0xf32bd122),
   LL(0x858b044c,0x55ef88a7),LL(0x5aa9e397,0x1f0d69c2),LL(0x40d85559,0x55fd9cc3),LL(0x7785ddb2,0xc774df72), LL(0xd3bd2e1c,0x5dcce9f6),LL(0xa85dfed0,0xeb30da20),LL(0xd3ed09c4,0x5ed7f5bb),LL(0x82a9c1bd,0x7d42a35c),
   LL(0x9890272d,0xcf3de995),LL(0x3e713a10,0x75f3432a),LL(0xe28227b8,0x5e13479f),LL(0xfefacdc8,0xb8561ea9), LL(0x8332aafd,0xa6a297a0),LL(0x73809b62,0x9b0d8bb5),LL(0x0c63036f,0xd2fa1cfd),LL(0xbd64bda8,0x7a16eb55),
   LL(0x78e62ddc,0x3f5cf5f6),LL(0x07fd752b,0x2267c454),LL(0x5e437bbe,0x5e361b6b),LL(0x8354e075,0x95c59501), LL(0xf2b254d9,0xec725f85),LL(0x2cb52b4e,0x844b617d),LL(0xcf425fb5,0xed8554f5),LL(0x2af9f312,0xab67703e),
   LL(0x3cf48283,0x4cc34ec1),LL(0x9c8a705e,0xb09daa25),LL(0x5b7d4f84,0xd1e9d0d0),LL(0xdb38929d,0x4df6ef64), LL(0xaa21ba46,0xe16b0763),LL(0xa293f8fb,0xc6b1d178),LL(0xd520aabf,0x0ff5b602),LL(0xc339397a,0x94d671bd),
   LL(0x4f5792fa,0x7c7d98cf),LL(0x11215261,0x7c5e0d67),LL(0xa7c5a6d4,0x9b19a631),LL(0x7a45274d,0xc8511a62), LL(0xa5a60d99,0x0c16621c),LL(0xcf5e48cb,0xf7fbab88),LL(0xf7ddee08,0xab1e6ca2),LL(0xe7867f3c,0x83bd08ce),
   LL(0x2ac13e27,0xf7e48e8a),LL(0x4eb1a9f5,0x4494f6df),LL(0x981f0a62,0xedbf84eb),LL(0x536438f0,0x49badc32), LL(0x004f7571,0x50bea541),LL(0xdf1c94ee,0xbac67d10),LL(0xb727bc31,0x253d73a1),LL(0x30686e28,0xb3d01cf2),
   LL(0x55fd0b8b,0x51b77b1b),LL(0xfeec3173,0xa099d183),LL(0x670e72b7,0x202b1fb7),LL(0xa8e1635f,0xadc88b33), LL(0xf989d905,0x34e8216a),LL(0x29b58d01,0xc2e68d20),LL(0x6fe55a93,0x11f81c92),LL(0x8f296f40,0x15f1462a),
   LL(0xea3d62f2,0x1915d375),LL(0x01c8977d,0xa17765a3),LL(0xe47b26f6,0x7559710a),LL(0x535077a5,0xe0bd29c8), LL(0x08d84858,0x615f976d),LL(0x69ced5c1,0x370dfe85),LL(0xa734fa56,0xbbc7503c),LL(0x91ac4574,0xfbb9f1ec),
   LL(0x060dd7ef,0x95d7ec53),LL(0x6e657979,0xeef2dacd),LL(0xe2a08235,0x54511af3),LL(0x1f4aea3d,0x1e324aa4), LL(0xe6e67671,0x550e7e71),LL(0xbf52faf7,0xbccd5190),LL(0x223cc62a,0xf880d316),LL(0x2b32eb5d,0x0d402c7e),
   LL(0x306a5a3b,0xa40bc039),LL(0x96783a1b,0x4e0a41fd),LL(0x0253cdd4,0xa1e8d39a),LL(0xc7388638,0x6480be26), LL(0x2285f382,0xee365e1d),LL(0xec0b5c36,0x188d8d8f),LL(0x1f0f4d82,0x34ef1a48),LL(0xa487d29a,0x1a8f43e1),
   LL(0x77aefb3a,0x8168226d),LL(0x1e72c253,0xf69a751e),LL(0xe9594df1,0x8e04359a),LL(0xd14c0467,0x475ffd7d), LL(0x3844e95c,0xb5a2c2b1),LL(0xdd12ef94,0x85caf647),LL(0xf1063d00,0x1ecd2a9f),LL(0x23843311,0x1dd2e229),
   LL(0x73d17244,0x38f0e09d),LL(0x8fc653f1,0x3ede7746),LL(0xdc20e21c,0xae4459f5),LL(0x6a8599ea,0x00db2ffa), LL(0x30cfd905,0x11682c39),LL(0xa5c112a6,0x4934d074),LL(0x568bfe95,0xbdf063c5),LL(0x016c441a,0x779a440a),
   LL(0x97d6fbdc,0x0c23f218),LL(0xe0776aac,0xd3a5cd87),LL(0xd712e8db,0xcee37f72),LL(0x26f74e8d,0xfb28c70d), LL(0xb61301a0,0xffe0c728),LL(0xd3724354,0xa6282168),LL(0x768ffedc,0x7ff4cb00),LL(0x03b02de9,0xc51b3088),
   LL(0x3902dda5,0xa5a8147c),LL(0xfe6973b4,0x35d2f706),LL(0xc257457e,0x5ac2efcf),LL(0x8700611b,0x933f48d4), LL(0x4912beb2,0xc365af88),LL(0x162edf94,0x7f5a4de6),LL(0x0c32f34b,0xc646ba7c),LL(0xb2091074,0x632c6af3),
   LL(0x753e43a9,0x58d4f2e3),LL(0x24d4e23f,0x70e1d217),LL(0xafede6a6,0xb24bf729),LL(0x710c8b60,0x7f4a94d8), LL(0x8d4faa6a,0xaad90a96),LL(0xb066b690,0xd9ed0b32),LL(0x78b6dbfd,0x52fcd37b),LL(0x8bd2b431,0x0b64615e),
   LL(0xcfb9fad5,0x228e2048),LL(0x240b76bd,0xbeaa386d),LL(0x90dad7bc,0x2d6681c8),LL(0x06d38f5e,0x3e553fc3), LL(0x9d5f9750,0xf27cdb9b),LL(0xd28c5b0e,0x3e85c52a),LL(0x5247c39b,0x190795af),LL(0xbddd6828,0x547831eb),
   LL(0x4a82f424,0xf327a227),LL(0x7e47f89d,0x36919c78),LL(0x43c7392c,0xe4783919),LL(0x2316fefe,0xf101b9aa), LL(0x1c5009d2,0xbcdc9e9c),LL(0x9cd18345,0xfb55ea13),LL(0xa3ce77c7,0xf5b5e231),LL(0xd2f2cb3d,0xde6b4527),
   LL(0x9bb26f5f,0x10f6a333),LL(0x044d85b6,0x1e85db8e),LL(0x94197e54,0xc3697a08),LL(0xa7cb4ea8,0x65e18cc0), LL(0xa471fe6e,0xa38c4f50),LL(0x2f13439c,0xf031747a),LL(0xc007318b,0x53c4a6ba),LL(0x1deccb3d,0xa8da3ee5),
   LL(0x558216b1,0x0555b31c),LL(0x2f79e6c2,0x90c7810c),LL(0xfe8eed3c,0x9b669f4d),LL(0xe0fac126,0x70398ec8), LL(0xf701b235,0xa96a449e),LL(0xeb94f395,0x0ceecdb3),LL(0xd0cb7431,0x285fc368),LL(0x16a18c64,0x0d37bb52),
   LL(0xb880d2dd,0x05110d38),LL(0x65930d57,0xa60f177b),LL(0xf36235f5,0x7da34a67),LL(0x183816b9,0x47f5e17c), LL(0xdb394af4,0xc7664b57),LL(0x7036f789,0x39ba215d),LL(0x2f27b472,0x46d2ca0e),LL(0xf73a84b7,0xc42647ee),
   LL(0x64488f1d,0x44bc7545),LL(0xf4cf85d5,0xaa922708),LL(0x53e4df63,0x721a01d5),LL(0x5db46ced,0x649c0c51), LL(0x3cffcb6c,0x6bf0d64e),LL(0x50f71d96,0xe3bf93fe),LL(0xbcc194a0,0x75044558),LL(0x6afdc554,0x16ae3372),
   LL(0x5ca48f3f,0xbfc01adf),LL(0xe22a9b84,0x64352f06),LL(0xc1099e4a,0xcee54da1),LL(0xfa1b89c0,0xbbda54e8), LL(0x6f6e55fb,0x166a3df5),LL(0x20176f88,0x1ca44a24),LL(0xdfb7b5ff,0x936afd88),LL(0x8611d4a0,0xe34c2437),
   LL(0x86142103,0x7effbb75),LL(0x1f34fc4d,0x6704ba1b),LL(0x10c1b122,0x7c2a468f),LL(0x8c6aace9,0x36b3a610), LL(0x75a0d050,0xabfcc0a7),LL(0x3ce33e32,0x066f9197),LL(0x29fe09be,0xce905ef4),LL(0xa8376351,0x89ee25ba),
   LL(0xfd29dc76,0x2a3ede22),LL(0x36f17260,0x7fd32ed9),LL(0x284b4126,0x0cadcf68),LL(0xa7951fc8,0x63422f08), LL(0x0807e199,0x562b24f4),LL(0x22ad4490,0xfe9ce5d1),LL(0x0db2b1b4,0xc2f51b10),LL(0xe4541d0d,0xeb3613ff),
   LL(0x2680813b,0xbd2c4a05),LL(0x561b08d6,0x527aa55d),LL(0xa7205558,0xa9f8a40e),LL(0x243d0bec,0xe3eea56f), LL(0xa0ff58b3,0x7b853817),LL(0x1a69e627,0xb67d3f65),LL(0xa869b5d6,0x0b76bbb9),LL(0x546723ed,0xa3afeb82),
   LL(0x3e554892,0x5f24416d),LL(0x430e2a45,0x8413b53d),LL(0x9032a2a0,0x99c56aee),LL(0xeec367b1,0x09432bf6), LL(0xdaf0ecc1,0x552850c6),LL(0x5bc92048,0x49ebce55),LL(0x54811307,0xdfb66ba6),LL(0x6f298597,0x1b84f797),
   LL(0x8d1d7a0d,0x79590481),LL(0x3a6fa556,0xd9fabe03),LL(0xba9e5d35,0xa40f9c59),LL(0xf6247577,0xcb1771c1), LL(0xe9a6312b,0x542a47ca),LL(0x552dd8c5,0xa34b3560),LL(0x0d794716,0xfdf94de0),LL(0x9c623094,0xd46124a9),
   LL(0x68afe8b4,0x56b7435d),LL(0x6c0d8ea1,0x27f20540),LL(0x73186898,0x12b77e14),LL(0x7479490f,0xdbc3dd46), LL(0xc03b0c05,0x951a9842),LL(0x7921bc96,0x8b1b3bb3),LL(0x2b202e0a,0xa573b346),LL(0x47254d56,0x77e4665d),
   LL(0xd23e3984,0x08b70dfc),LL(0xebd14236,0xab86e8bc),LL(0x57114ba7,0xaa3e07f8),LL(0xab0ef4f2,0x5ac71689), LL(0x0139d9af,0x88fca384),LL(0x76644af0,0x72733f88),LL(0x65d74f4a,0xf122f72a),LL(0xa5626c7a,0x13931577),
   LL(0x70f8d5a4,0xd5b5d9eb),LL(0xd7bbb228,0x375adde7),LL(0x0c1c0b32,0x31e88b86),LL(0x173edbaa,0xd1f568c4), LL(0x5459df02,0x1592fc83),LL(0x0fcd9a7e,0x2beac0fb),LL(0x1b473b0a,0xb0a6fdb8),LL(0x0fe8fc48,0xe3224c6f),
   LL(0xe87edf5b,0x680bd00e),LL(0x20e77cf5,0x30385f02),LL(0x4d42d1b2,0xe9ab98c0),LL(0xd3816d77,0x72d191d2), LL(0x0917d9e5,0x1564daca),LL(0x1f8fed7f,0x394eab59),LL(0x7fbb3896,0xa209aa8d),LL(0xbe6ac98e,0x5564f3b9),
   LL(0xd73654ef,0xead21d05),LL(0x13d78d74,0x68d1a9c4),LL(0x6d4973a0,0x61e01708),LL(0x46e6d32a,0x83da3500), LL(0x68ae0118,0x6a3dfca4),LL(0xd02da069,0xa1b9a4c9),LL(0xebab8302,0x0b2ff9c7),LL(0x944ba436,0x98af07c3),
   LL(0x995f0f9f,0x85997326),LL(0x71b58bc6,0x467fade0),LL(0xbd625a2b,0x47e4495a),LL(0x33c3b8cd,0xfdd2d01d), LL(0xc693f9fa,0x2c38ae28),LL(0x348f7999,0x48622329),LL(0x2161f583,0x97bf738e),LL(0x565e8cc9,0x15ee2fa7),
   LL(0x5777e189,0xa1a5c845),LL(0x456f2829,0xcc10bee0),LL(0xda762bd5,0x8ad95c56),LL(0xe9d91da8,0x152e2214), LL(0x7cb23c74,0x975b0e72),LL(0xa90c66df,0xfd5d7670),LL(0x225ffc53,0xb5b5b8ad),LL(0xfaded2ae,0xab6dff73),
   LL(0x6f4cbe9d,0xebd56781),LL(0x6a574bd7,0x0ed8b249),LL(0x81a881fa,0x41c246fe),LL(0xc3db9c70,0x91564805), LL(0x5b862809,0xd7c12b08),LL(0x55858d7b,0x1facd1f1),LL(0xaf09e92a,0x7693747c),LL(0x189a425f,0x3b69dcba),
   LL(0x967365ef,0x0be28e9f),LL(0xe801f5c9,0x57300eb2),LL(0xd583352f,0x93b8ac6a),LL(0xcd05b2b7,0xa2cf1f89), LL(0x4dcc40cc,0x7c0c9b74),LL(0xada523fb,0xfee38c45),LL(0x1099cc4d,0xb49a4dec),LL(0x69f069c6,0x325c377f),
   LL(0x476cc9ff,0xe12458ce),LL(0xc6d4cb63,0x580e0b6c),LL(0x9072289b,0xd561c8b7),LL(0xa619e6da,0x0377f264), LL(0x88e591a5,0x26685362),LL(0x7523ca2b,0xa453a7bd),LL(0xc1df4533,0x8a9536d2),LL(0xbe972f79,0xc8e50f2f),
   LL(0x6d3549cf,0xd433e50f),LL(0xfacd665e,0x6f33696f),LL(0xce11fcb4,0x695bfdac),LL(0xaf7c9860,0x810ee252), LL(0x7159bb2c,0x65450fe1),LL(0x758b357b,0xf7dfbebe),LL(0xd69fea72,0x2b057e74),LL(0x92731745,0xd485717a),
},
/* digit=23 base_pwr=2^161 */
{
   LL(0xee36860c,0x896c42e8),LL(0x4113c22d,0xdaf04dfd),LL(0x44104213,0x1adbb7b7),LL(0x1fd394ea,0xe5fd5fa1), LL(0x1a4e0551,0x68235d94),LL(0x18d10151,0x6772cfbe),LL(0x09984523,0x276071e3),LL(0x5a56ba98,0xe4e879de),
   LL(0x285b9491,0xaaafafb0),LL(0x1e4c705e,0x01a0be88),LL(0x2ad9caab,0xff1d4f5d),LL(0xc37a233f,0x6e349a4a), LL(0x4a1c6a16,0xcf1c1246),LL(0x29383260,0xd99e6b66),LL(0x5f6d5471,0xea3d4366),LL(0xff8cc89b,0x36974d04),
   LL(0xcfe89d80,0xc26c49a1),LL(0xda9c8371,0xb42c026d),LL(0xdad066d2,0xca6c013a),LL(0x56a4f3ee,0xfb8f7228), LL(0xd850935b,0x08b579ec),LL(0xd631e1b3,0x34c1a74c),LL(0xac198534,0xcb5fe596),LL(0xe1f24f25,0x39ff21f6),
   LL(0x8f929057,0x27f29e14),LL(0xc0c853df,0x7a64ae06),LL(0x58e9c5ce,0x256cd183),LL(0xded092a5,0x9d9cce82), LL(0x6e93b7c7,0xcc6e5979),LL(0x31bb9e27,0xe1e47092),LL(0xaa9e29a0,0xb70b3083),LL(0x3785e644,0xbf181a75),
   LL(0x8ead09f7,0xf53f2c65),LL(0x9780d14d,0x1335e1d5),LL(0xcd1b66bc,0x69cc20e0),LL(0xbbe0bfc8,0x9b670a37), LL(0x28efbeed,0xce53dc81),LL(0x8326a6e5,0x0c74e77c),LL(0xb88e9a63,0x3604e0d2),LL(0x13dc2248,0xbab38fca),
   LL(0x5c0a3f1e,0x8ed6e8c8),LL(0x7c87c37f,0xbcad2492),LL(0x9ee3b78d,0xfdfb62bb),LL(0xcbceba46,0xeba8e477), LL(0xeeaede4b,0x37d38cb0),LL(0x7976deb6,0x0bc498e8),LL(0x6b6147fb,0xb2944c04),LL(0xf71f9609,0x8b123f35),
   LL(0xde79dc24,0xa155dcc7),LL(0x558f69cd,0xf1168a32),LL(0x0d1850df,0xbac21595),LL(0xb204c848,0x15c8295b), LL(0x7d8184ff,0xf661aa36),LL(0x30447bdb,0xc396228e),LL(0xbde4a59e,0x11cd5143),LL(0x6beab5e6,0xe3a26e3b),
   LL(0x1402b9d0,0xd3b3a13f),LL(0x2c7bc863,0x573441c3),LL(0x578c3e6e,0x4b301ec4),LL(0x0adaf57e,0xc26fc9c4), LL(0x7493cea3,0x96e71bfd),LL(0x1af81456,0xd05d4b3f),LL(0x6a8c608f,0xdaca2a8a),LL(0x0725b276,0x53ef07f6),
   LL(0x7824fc56,0x07a5fbd2),LL(0x13289077,0x34675218),LL(0xe0c48349,0x5bf69fd5),LL(0xb6aa7875,0xa613ddd3), LL(0x5450d866,0x7f78c19c),LL(0x8f84a481,0x46f4409c),LL(0x90fce239,0x9f1d1928),LL(0xb2ce44b9,0x016c4168),
   LL(0xc7435978,0xbae023f0),LL(0x20e30e19,0xb152c888),LL(0xe3fa6faf,0x9c241645),LL(0x84823e60,0x735d95c1), LL(0x03955317,0x03197573),LL(0xf03b4995,0x0b4b02a9),LL(0x70274600,0x076bf559),LL(0xaaf57508,0x32c5cc53),
   LL(0x60624129,0xe8af6d1f),LL(0x9a5e2b5e,0xb7bc5d64),LL(0x5f082d72,0x3814b048),LL(0xce19677a,0x76f267f2), LL(0xb36eed93,0x626c630f),LL(0x3bf56803,0x55230cd7),LL(0xce2736a0,0x78837949),LL(0xaa6c55f1,0x0d792d60),
   LL(0xd5c7c5d2,0x0318dbfd),LL(0x072b342d,0xb38f8da7),LL(0x7b8de38a,0x3569bddc),LL(0xa1c94842,0xf25b5887), LL(0x2946ad60,0xb2d5b284),LL(0xe9d1707e,0x854f29ad),LL(0x2c6a4509,0xaa5159dc),LL(0x57189837,0x899f94c0),
   LL(0xf4a55b03,0xcf6adc51),LL(0x35e3b2d5,0x261762de),LL(0x04827b51,0x4cc43012),LL(0xc6021442,0xcd22a113), LL(0x247c9569,0xce2fd61a),LL(0xd152beca,0x59a50973),LL(0x63a716d4,0x6c835a11),LL(0x187dedcf,0xc26455ed),
   LL(0x49ce89e7,0x27f536e0),LL(0xcc890cb5,0x18908539),LL(0xd83c2aa1,0x308909ab),LL(0x1ab73bd3,0xecd3142b), LL(0xb3f5ab84,0x6a85bf59),LL(0xf2bea4c6,0x3c320a68),LL(0x6da4541f,0xad8dc538),LL(0xb7c41186,0xeaf34eb0),
   LL(0x977c97c4,0x1c780129),LL(0xc57eb9fa,0x5ff9beeb),LL(0xc822c478,0xa24d0524),LL(0x461cd415,0xfd8eec2a), LL(0xf027458c,0xfbde194e),LL(0x1d1be115,0xb4ff5319),LL(0x4866d6f4,0x63f874d9),LL(0xb21ad0c9,0x35c75015),
   LL(0x46ac49d2,0xa6b5c9d6),LL(0x83137aa9,0x42c77c0b),LL(0x68225a38,0x24d000fc),LL(0x2fe1e907,0x0f63cfc8), LL(0xc6441f95,0x22d1b01b),LL(0xec8e448f,0x7d38f719),LL(0x787fb1ba,0x9b33fa5f),LL(0x190158df,0x94dcfda1),
   LL(0x5f6d4a09,0xc47cb339),LL(0xee52b826,0x6b4f355c),LL(0xf51b930a,0x3d100f5d),LL(0x9f668f69,0xf4512fac), LL(0x206c4c74,0x546781d5),LL(0xcb4d2e48,0xd021d4d4),LL(0xca085c2d,0x494a54c2),LL(0x520850a8,0xf1dbaca4),
   LL(0x490a1aca,0x63c79326),LL(0x41526b02,0xcb64dd9c),LL(0xa2979258,0xbb772591),LL(0x48d97846,0x3f582970), LL(0x7c213ba7,0xd66b70d1),LL(0xe8a0ced4,0xc28febb5),LL(0xc10338c1,0x6b911831),LL(0xbf0126f3,0x0d54e389),
   LL(0x4af206ee,0x7048d460),LL(0x77e97cb9,0x786c88f6),LL(0xac64802e,0xd4375ae1),LL(0xd53ec11c,0x469bcfe1), LL(0x47062230,0xfc9b340d),LL(0xc5b4a3ac,0xe743bb57),LL(0x59ef45ac,0xfe00b4aa),LL(0x59edf188,0x29a4ef23),
   LL(0xb483689b,0x40242efe),LL(0x513ac262,0x2575d3f6),LL(0x0ca6db72,0xf30037c8),LL(0x98864be2,0xc9fcce82), LL(0x0149362d,0x84a112ff),LL(0x1c4ae971,0x95e57582),LL(0x945cf86c,0x1fa4b1a8),LL(0x0b024a2f,0x4525a734),
   LL(0x8f338360,0xe76c8b62),LL(0x28edf32b,0x483ff593),LL(0x298b1aec,0x67e8e90a),LL(0x736d9a21,0x9caab338), LL(0x66892709,0x5c09d2fd),LL(0xb55a1d41,0x2496b4dc),LL(0xe24a4394,0x93f5fb1a),LL(0x6fa8f6c1,0x08c75049),
   LL(0xc905d85f,0xcaead1c2),LL(0x0733ae57,0xe9d7f790),LL(0xf07cdd94,0x24c9a65c),LL(0xa4b55931,0x7389359c), LL(0x367e45f7,0xf58709b7),LL(0xcb7e7adc,0x1f203067),LL(0xc7b72818,0x82444bff),LL(0xbaac8033,0x07303b35),
   LL(0xd13b7ea1,0x1e1ee4e4),LL(0xe0e74180,0xe6489b24),LL(0x7e70ef70,0xa5f2c610),LL(0xbdd10894,0xa1655412), LL(0x7af4194e,0x555ebefb),LL(0x8e89bd9c,0x533c1c3c),LL(0x89895856,0x735b9b57),LL(0x567f5c15,0x15fb3cd2),
   LL(0x526f09fd,0x057fed45),LL(0x8128240a,0xe8a4f10c),LL(0xff2bfd8d,0x9332efc4),LL(0xbd35aa31,0x214e77a0), LL(0x14faa40e,0x32896d73),LL(0x01e5f186,0x767867ec),LL(0x17a1813e,0xc9adf8f1),LL(0x54741795,0xcb6cda78),
   LL(0x349d51aa,0xb7521b6d),LL(0xe3c7b8e9,0xf56b5a9e),LL(0x32a096df,0xc6f1e5c9),LL(0xa3635024,0x083667c4), LL(0x18087f2f,0x365ea135),LL(0xd136e45d,0xf1b8eaac),LL(0x73aec989,0xc8a0e484),LL(0x142c9259,0xd75a324b),
   LL(0x01dae185,0xb7b4d001),LL(0x9b7a94bc,0x45434e0b),LL(0xfbd8cb0b,0xf54339af),LL(0xe98ef49e,0xdcc4569e), LL(0x09a51299,0x7789318a),LL(0xb2b025d8,0x81b4d206),LL(0xfae85792,0xf64aa418),LL(0xacd7baf7,0x3e50258f),
   LL(0x2996864b,0xdce84cdb),LL(0x1f485fa4,0xa2e67089),LL(0x534c6a5a,0xb28b2bb6),LL(0xc94b9d39,0x31a7ec6b), LL(0xd6bc20da,0x1d217766),LL(0x86761190,0x4acdb5ec),LL(0x73701063,0x68726328),LL(0x2128c29b,0x4d24ee7c),
   LL(0xa19fd868,0xc072ebd3),LL(0xdb8ddd3b,0x612e481c),LL(0x1a64d852,0xb4e1d754),LL(0xc4c6c4ab,0x00ef95ac), LL(0xaa0a6c46,0x1536d2ed),LL(0x43774790,0x61294086),LL(0x343fda10,0x54af25e8),LL(0xfd25d6f2,0x9ff9d98d),
   LL(0x468b8835,0x0746af7c),LL(0x730ecea7,0x977a31cb),LL(0xc2cf4a81,0xa5096b80),LL(0x6458c37a,0xaa986833), LL(0xa6bd9d34,0x6af29bf3),LL(0x33c5d854,0x6a62fe9b),LL(0xb7133b5e,0x50e6c304),LL(0x7d6e6848,0x04b60159),
   LL(0x5579bea4,0x4cd296df),LL(0x5ceedaf1,0x10e35ac8),LL(0xe3bcc5b1,0x04c4c5fd),LL(0x89412cf9,0x95f9ee8a), LL(0x82b6eb0f,0x2c9459ee),LL(0x95c2aadd,0x2e845765),LL(0xd327fcfe,0x774a84ae),LL(0x0368d476,0xd8c93722),
   LL(0xf83e8a3b,0x0dbd5748),LL(0x8d2495f3,0xa579aa96),LL(0xae496e9b,0x535996a0),LL(0xb7f9bcc2,0x07afbfe9), LL(0x5b7bd293,0x3ac1dc6d),LL(0x7022323d,0x3b592cff),LL(0x9c0a3e76,0xba0deb98),LL(0x4b197acb,0x18e78e9f),
   LL(0x296c36ef,0x211cde10),LL(0x82c4da77,0x7ee89672),LL(0xa57836da,0xb617d270),LL(0x9cb7560b,0xf0cd9c31), LL(0xe455fe90,0x01fdcbf7),LL(0x7e7334f3,0x3fb53cbb),LL(0x4e7de4ec,0x781e2ea4),LL(0x0b384fd0,0x8adab3ad),
   LL(0x53d64829,0x129eee2f),LL(0xa261492b,0x7a471e17),LL(0xe4cb4a2c,0xe4f9adb9),LL(0x97ba2c2d,0x3d359f6f), LL(0x0aacd697,0x346c6786),LL(0x75c2f8a8,0x92b444c3),LL(0xd85df44e,0xc79fa117),LL(0x398ddf31,0x56782372),
   LL(0xbbbab3b8,0x60e690f2),LL(0x8b04816b,0x4851f8ae),LL(0x9c92e4d2,0xc72046ab),LL(0x7cf3136b,0x518c74a1), LL(0xf9877d4c,0xff4eb50a),LL(0xa919cabb,0x14578d90),LL(0xac5eb2b6,0x8218f8c4),LL(0x542016e4,0xa3ccc547),
   LL(0x327f8349,0x025bf48e),LL(0xf43cb641,0xf3e97346),LL(0x500f1085,0xdc2bafdf),LL(0x2f063055,0x57167876), LL(0x411925a6,0x5bd914b9),LL(0xa1123de5,0x7c078d48),LL(0x182b165d,0xee6bf835),LL(0xba519727,0xb11b5e5b),
   LL(0x1eea7b85,0xe33ea76c),LL(0x92d4f85e,0x2352b461),LL(0xafe115bb,0xf101d334),LL(0x889175a3,0xfabc1294), LL(0x5233f925,0x7f6bcdc0),LL(0xe77fec55,0xe0a802db),LL(0x8069b659,0xbdb47b75),LL(0xf98fbd74,0x1c5e12de),
   LL(0x4b8457ee,0x869c58c6),LL(0x4f7ea9f7,0xa5360f69),LL(0xf460b38f,0xe576c09f),LL(0x22b7fb36,0x6b70d548), LL(0x3bfae315,0x3fd237f1),LL(0xcbdff369,0x33797852),LL(0x25b516f9,0x97df25f5),LL(0xba38ad2d,0x46f388f2),
   LL(0x89d8ddbb,0x656c4658),LL(0x70f38ee8,0x8830b26e),LL(0xde1212b0,0x4320fd5c),LL(0xe4a2edb2,0xc34f30cf), LL(0x56ab64b8,0xabb131a3),LL(0xd99c5d26,0x7f77f0cc),LL(0xbf981d94,0x66856a37),LL(0x738bd76e,0x19e76d09),
   LL(0x96238f39,0xe76c8ac3),LL(0xa830b366,0xc0a482be),LL(0x0b4eb499,0xb7b8eaff),LL(0x4bfb4865,0x8ecd83bc), LL(0xa2f3776f,0x971b2cb7),LL(0xf4b88adf,0xb42176a4),LL(0xbe1fa446,0xb9617df5),LL(0xcd031bd2,0x8b32d508),
   LL(0x53b618c0,0x1c6bd47d),LL(0x6a227923,0xc424f46c),LL(0xdd92d964,0x7303ffde),LL(0x71b5abf2,0xe9712878), LL(0xf815561d,0x8f48a632),LL(0xd3c055d1,0x85f48ff5),LL(0x7525684f,0x222a1427),LL(0x67360cc3,0xd0d841a0),
   LL(0x0b9267c6,0x4245a926),LL(0xcf07f863,0xc78913f1),LL(0x4d0d9e24,0xaa844c8e),LL(0x3d5f9017,0xa42ad522), LL(0xa2c989d5,0xbd371749),LL(0xe1f5e78e,0x928292df),LL(0x0a1ea6da,0x493b383e),LL(0x13aee529,0x5136fd8d),
   LL(0xf2c34a99,0x860c44b1),LL(0xbf5855ac,0x3b00aca4),LL(0xfaaf37be,0xabf6aaa0),LL(0x2a53ec08,0x65f43682), LL(0xa11b12e1,0x1d9a5801),LL(0xe20ed475,0x78a7ab2c),LL(0x9a41e0d5,0x0de1067e),LL(0x305023ea,0x30473f5f),
   LL(0x169c7d97,0xdd3ae09d),LL(0xcfaef9cd,0x5cd5baa4),LL(0x65a44803,0x5cd7440b),LL(0x47f364de,0xdc13966a), LL(0x2b8357c1,0x077b2be8),LL(0xe9d57c2a,0x0cb1b4c5),LL(0x05ff363e,0x7a4ceb32),LL(0xca35a9ef,0xf310fa4d),
   LL(0xf97f68c6,0xdbb7b352),LL(0x0b02cf58,0x0c773b50),LL(0x3c1f96d9,0xea2e4821),LL(0xeee01815,0xffb357b0), LL(0xe0f28039,0xb9c924cd),LL(0x46a3fbe4,0x0b36c95a),LL(0x5e46db6c,0x1faaaea4),LL(0x1928aaff,0xcae575c3),
   LL(0xa70dab86,0x7f671302),LL(0x71c58cfc,0xfcbd12a9),LL(0xbee0cb92,0xcbef9acf),LL(0xf8c1b583,0x573da0b9), LL(0x0d41d550,0x4752fcfe),LL(0x2155cffe,0xe7eec0e3),LL(0x545ae248,0x0fc39fcb),LL(0x8065f44e,0x522cb8d1),
   LL(0x70cbb96c,0x263c962a),LL(0xbcd124a9,0xe034362a),LL(0x3c2ae58d,0xf120db28),LL(0xfef6d507,0xb9a38d49), LL(0x1ff140fd,0xb1fd2a82),LL(0x20aee7e0,0xbd162f30),LL(0xcb251949,0x4e17a5d4),LL(0x4f7e1c3d,0x2aebcb83),
   LL(0x937b0527,0x608eb25f),LL(0xeb7d9997,0xf42e1e47),LL(0xb8a53a29,0xeba699c4),LL(0xe091b536,0x1f921c71), LL(0x5b26bbd5,0xcce29e7b),LL(0x3b61a680,0x7a8ef5ed),LL(0xba1f1c7e,0xe5ef8043),LL(0x18158dda,0x16ea8217),
   LL(0x599ff0f9,0x01778a2b),LL(0x8104fc6b,0x68a923d7),LL(0xda694ff3,0x5bfa44df),LL(0xf7667f12,0x4f7199db), LL(0xe46f2a79,0xc06d8ff6),LL(0xe9f8131d,0x08b5dead),LL(0xabb4ce7c,0x02519a59),LL(0xb42aec3e,0xc4f710bc),
   LL(0x78bde41a,0x3d77b057),LL(0xb4186b5a,0x6474bf80),LL(0x88c65741,0x048b3f67),LL(0x03c7c154,0xc64519de), LL(0x0edfcc4f,0xdf073846),LL(0x48f1aa6b,0x319aa737),LL(0xca909f77,0x8b9f8a02),LL(0x7580bfef,0x90258139),
   LL(0xc0c22719,0xd8bfd3ca),LL(0xc9ca151e,0xc60209e4),LL(0xd9a1a69c,0x7a744ab5),LL(0x14937f8f,0x6de5048b), LL(0xe115ac04,0x171938d8),LL(0x1c6b16d2,0x7df70940),LL(0x7f8e94e7,0xa6aeb663),LL(0x2a2cf094,0xc130388e),
   LL(0x77f54e6e,0x1850be84),LL(0x65d60fe5,0x9f258a72),LL(0x6c9146d6,0xff7ff0c0),LL(0xe63a830b,0x039aaf90), LL(0x9460342f,0x38f27a73),LL(0x3f795f8a,0x4703148c),LL(0x9681a97e,0x1bb5467b),LL(0xecaeb594,0x00931ba5),
   LL(0x786f337c,0xcdb6719d),LL(0xe704397d,0xd9c01cd2),LL(0x555c2fef,0x0f4a3f20),LL(0x7c0af223,0x00452509), LL(0x84db8e76,0x54a58047),LL(0x93c8aa06,0x3bacf1aa),LL(0xf7919422,0x11ca957c),LL(0x78cdaa40,0x50641053),
   LL(0x9f7144ae,0x7a303874),LL(0x43d4acfd,0x170c963f),LL(0x58ddd3ef,0x5e148149),LL(0x9e72dba8,0xa7bde582), LL(0x6fa68750,0x0769da8b),LL(0x572e0249,0xfa64e532),LL(0x2619ad31,0xfcaadf9d),LL(0xa7b349cd,0x87882daa),
   LL(0x6c67a775,0x9f6eb731),LL(0xefc5d0b1,0xcb10471a),LL(0xe1b806b2,0xb433750c),LL(0x57b1ae7e,0x19c5714d), LL(0xed03fd3f,0xc0dc8b7b),LL(0x31bc194e,0xdd03344f),LL(0x8c6320b5,0xa66c52a7),LL(0xd0b6fd93,0x8bc82ce3),
   LL(0xb35f1341,0xf8e13501),LL(0x25a43e42,0xe53156dd),LL(0x4daeb85c,0xd3adf27e),LL(0xbbeddeb5,0xb81d8379), LL(0x2e435867,0x1b0b546e),LL(0xeba5dd60,0x9020eb94),LL(0x8210cb9d,0x37d91161),LL(0x5c91f1cf,0x4c596b31),
   LL(0x0e0b040d,0xb228a90f),LL(0x45ff897f,0xbaf02d82),LL(0x00fa6122,0x2aac79e6),LL(0x8e36f557,0x24828817), LL(0x113ec356,0xb9521d31),LL(0x15eff1f8,0x9e48861e),LL(0xe0d41715,0x2aa1d412),LL(0x53f131b8,0x71f86203),
   LL(0x3fd19408,0xf60da8da),LL(0x278d9d99,0x4aa716dc),LL(0xa8c51c90,0x394531f7),LL(0xf59db51c,0xb560b0e8), LL(0xfa34bdad,0xa28fc992),LL(0x9cd4f8bd,0xf024fa14),LL(0x23a9d0d3,0x5cf530f7),LL(0xe28c9b56,0x615ca193),
   LL(0x6f73c51e,0x6d2a483d),LL(0xea0dc2dd,0xa4cb2412),LL(0x1eb917ff,0x50663c41),LL(0xeade299e,0x3d3a74cf), LL(0x4a7a9202,0x29b3990f),LL(0xa7b15c3d,0xa9bccf59),LL(0xa5df9208,0x66a3ccdc),LL(0x43f2f929,0x48027c14),
   LL(0x40b557f0,0xd385377c),LL(0xcd684660,0xe001c366),LL(0xe2183a27,0x1b18ed6b),LL(0x63210329,0x879738d8), LL(0xbda94882,0xa687c74b),LL(0xa684b299,0xd1bbcc48),LL(0x863b3724,0xaf6f1112),LL(0x2c8ce9f8,0x6943d1b4),
   LL(0x098cafb4,0xe044a3bb),LL(0x60d48caf,0x27ed2310),LL(0x3a31b84d,0x542b5675),LL(0xfcddbed7,0xcbf3dd50), LL(0x41b1d830,0x25031f16),LL(0xcb0c1e27,0xa7ec851d),LL(0xb5ae75db,0xac1c8fe0),LL(0x08c52120,0xb24c7557),
   LL(0x1d4636c3,0x57f811dc),LL(0x681a9939,0xf8436526),LL(0x9c81adb3,0x1f6bc6d9),LL(0x5b7d80d4,0x840f8ac3), LL(0xf4387f1a,0x731a9811),LL(0xb5156880,0x7c501cd3),LL(0xdfe68867,0xa5ca4a07),LL(0x5fcea120,0xf123d8f0),
   LL(0xd607039e,0x1fbb0e71),LL(0xcd3a4546,0x2b70e215),LL(0x53324091,0x32d2f01d),LL(0x180ab19b,0xb796ff08), LL(0x3c57c4aa,0x32d87a86),LL(0xb7c49a27,0x2aed9caf),LL(0x31630d98,0x9fb35eac),LL(0x5c3e20a3,0x338e8cdf),
   LL(0x66cde8db,0x80f16182),LL(0x2d72fd36,0x4e159980),LL(0x9b6e5072,0xd7b8f13b),LL(0x3b7b5dc1,0xf5213907), LL(0x8ce4396e,0x4d431f1d),LL(0xa7ed2142,0x37a1a680),LL(0xd01aaf6b,0xbf375696),LL(0xe63aab66,0xaa1c0c54),
   LL(0x4ed80940,0x3014368b),LL(0x7a6fcedd,0x67e6d056),LL(0xca97579f,0x7c208c49),LL(0xa23597f6,0xfe3d7a81), LL(0x7e096ae2,0x5e203202),LL(0x24b39366,0xb1f3e1e7),LL(0x2fdcdffc,0x26da26f3),LL(0x6097be83,0x79422f1d),
},
/* digit=24 base_pwr=2^168 */
{
   LL(0x9db3b381,0x263a2cfb),LL(0xd4df0a4b,0x9c3a2dee),LL(0x7d04e61f,0x728d06e9),LL(0x42449325,0x8b1adfbc), LL(0x7e053a1b,0x6ec1d939),LL(0x66daf707,0xee2be5c7),LL(0x810ac7ab,0x80ba1e14),LL(0xf530f174,0xdd2ae778),
   LL(0x205b9d8b,0x0435d97a),LL(0x056756d4,0x6eb8f064),LL(0xb6f8210e,0xd5e88a8b),LL(0xec9fd9ea,0x070ef12d), LL(0x3bcc876a,0x4d849505),LL(0xa7404ce3,0x12a75338),LL(0xb8a1db5e,0xd22b49e1),LL(0x14bfa5ad,0xec1f2051),
   LL(0xb6828f36,0xadbaeb79),LL(0x01bd5b9e,0x9d7a0258),LL(0x1e844b0c,0xeda01e0d),LL(0x887edfc9,0x4b625175), LL(0x9669b621,0x14109fdd),LL(0xf6f87b98,0x88a2ca56),LL(0x170df6bc,0xfe2eb788),LL(0xffa473f9,0x0cea06f4),
   LL(0xc4e83d33,0x43ed81b5),LL(0x5efd488b,0xd9f35879),LL(0x9deb4d0f,0x164a620f),LL(0xac6a7394,0xc6927bdb), LL(0x9f9e0f03,0x45c28df7),LL(0xfcd7e1a9,0x2868661e),LL(0xffa348f1,0x7cf4e8d0),LL(0x398538e0,0x6bd4c284),
   LL(0x289a8619,0x2618a091),LL(0x6671b173,0xef796e60),LL(0x9090c632,0x664e46e5),LL(0x1e66f8fb,0xa38062d4), LL(0x0573274e,0x6c744a20),LL(0xa9271394,0xd07b67e4),LL(0x6bdc0e20,0x391223b2),LL(0xeb0a05a7,0xbe2d93f1),
   LL(0x3f36d141,0xf23e2e53),LL(0x4dfca442,0xe84bb3d4),LL(0x6b7c023a,0xb804a48d),LL(0x76431c3b,0x1e16a8fa), LL(0xddd472e0,0x1b5452ad),LL(0x0d1ee127,0x7d405ee7),LL(0xffa27599,0x50fc6f1d),LL(0xbf391b35,0x351ac53c),
   LL(0x4444896b,0x7efa14b8),LL(0xf94027fb,0x64974d2f),LL(0xde84487d,0xefdcd0e8),LL(0x2b48989b,0x8c45b260), LL(0xd8463487,0xa8fcbbc2),LL(0x3fbc476c,0xd1b2b3f7),LL(0xc8f443c0,0x21d005b7),LL(0x40c0139c,0x518f2e67),
   LL(0x06d75fc1,0x56036e8c),LL(0x3249a89f,0x2dcf7bb7),LL(0xe245e7dd,0x81dd1d3d),LL(0xebd6e2a7,0xf578dc4b), LL(0xdf2ce7a0,0x4c028903),LL(0x9c39afac,0xaee36288),LL(0x146404ab,0xdc847c31),LL(0xa4e97818,0x6304c0d8),
   LL(0xa91f6791,0xae51dca2),LL(0x9baa9efc,0x2abe4190),LL(0x559c7ac1,0xd9d2e2f4),LL(0xfc9f773a,0xe82f4b51), LL(0x4073e81c,0xa7713027),LL(0xfbb596fc,0xc0276fac),LL(0xa684f70c,0x1d819fc9),LL(0xc9f7b1e0,0x29b47fdd),
   LL(0x459b1940,0x358de103),LL(0x5b013e93,0xec881c59),LL(0x49532ad3,0x51574c93),LL(0xb37b46de,0x2db1d445), LL(0xdf239fd8,0xc6445b87),LL(0x151d24ee,0xc718af75),LL(0xf43c6259,0xaea1c4a4),LL(0x70be02f7,0x40c0e5d7),
   LL(0x721b33f2,0x6a4590f4),LL(0xfedf04ea,0x2124f1fb),LL(0x9745efe7,0xf8e53cde),LL(0x65f046d9,0xe7e10432), LL(0xe4d0c7e6,0xc3fca28e),LL(0x87253b1b,0x847e339a),LL(0x3743e643,0x9b595348),LL(0x4fd12fc5,0xcb6a0a0b),
   LL(0x27d02dcc,0xfb6836c3),LL(0x7a68bcc2,0x5ad00982),LL(0x005e912d,0x1b24b44c),LL(0x811fdcfe,0xcc83d20f), LL(0x666fba0c,0x36527ec1),LL(0x14754635,0x69948197),LL(0x556da9c2,0xfcdcb1a8),LL(0x81a732b2,0xa5934267),
   LL(0xa714181d,0xec1214ed),LL(0x6067b341,0x609ac13b),LL(0xa545df1f,0xff4b4c97),LL(0x34d2076b,0xa1240501), LL(0x1409ca97,0x6efa0c23),LL(0x20638c43,0x254cc1a8),LL(0xdcfb46cd,0xd4e363af),LL(0x03942a27,0x62c2adc3),
   LL(0x56e46483,0xc67b9df0),LL(0x63736356,0xa55abb20),LL(0xc551bc52,0xab93c098),LL(0xb15fe64b,0x382b49f9), LL(0x4dff8d47,0x9ec221ad),LL(0x437df4d6,0x79caf615),LL(0xbb456509,0x5f13dc64),LL(0x191f0714,0xe4c589d9),
   LL(0x3fd40e09,0x27b6a8ab),LL(0x77313ea9,0xe455842e),LL(0x1f55988b,0x8b51d1e2),LL(0x062bbbfc,0x5716dd73), LL(0x4e8bf3de,0x633c11e5),LL(0x1b85be3b,0x9a0e77b6),LL(0x0911cca6,0x56510729),LL(0xefa6590f,0x27e76495),
   LL(0x070d3aab,0xe4ac8b33),LL(0x9a2cd5e5,0x2643672b),LL(0x1cfc9173,0x52eff79b),LL(0x90a7c13f,0x665ca49b), LL(0xb3efb998,0x5a8dda59),LL(0x052f1341,0x8a5b922d),LL(0x3cf9a530,0xae9ebbab),LL(0xf56da4d7,0x35986e7b),
   LL(0xff3513cc,0x3a636b5c),LL(0x3198f7dd,0xbb0cf8ba),LL(0x41f16f86,0xb8d40522),LL(0xde13a7bf,0x760575d8), LL(0x9f7aa181,0x36f74e16),LL(0xf509ed1c,0x163a3ecf),LL(0x3c40a491,0x6aead61f),LL(0xdfe8fcaa,0x158c95fc),
   LL(0x13cda46f,0xa3991b6e),LL(0x342faed0,0x79482415),LL(0x666b5970,0xf3ba5bde),LL(0xb26ab6dd,0x1d52e6bc), LL(0x8608dd3d,0x768ba1e7),LL(0xea076586,0x4930db2a),LL(0xe7dc1afa,0xd9575714),LL(0xf7c58817,0x1fc7bf7d),
   LL(0xd9eee96c,0x6b47accd),LL(0xe58cec37,0x0ca277fb),LL(0xe702c42a,0x113fe413),LL(0xc47cbe51,0xdd1764ee), LL(0x7b3ed739,0x041e7cde),LL(0x5ce9e1c0,0x50cb7459),LL(0x2925b212,0x35568513),LL(0x001b081c,0x7cff95c4),
   LL(0x8088b454,0x63ee4cbd),LL(0x9a9e0c8a,0xdb7f32f7),LL(0x6b2447cb,0xb377d418),LL(0xd370219b,0xe3e982aa), LL(0xc2a2a593,0x06ccc1e4),LL(0x0773f24f,0x72c36865),LL(0x95859423,0xa13b4da7),LL(0x75040c8f,0x8bbf1d33),
   LL(0xda50c991,0x726f0973),LL(0x822d6ee2,0x48afcd5b),LL(0x20fd7771,0xe5fc718b),LL(0xfd0807a1,0xb9e8e77d), LL(0x99a7703d,0x7f5e0f44),LL(0x618e36f3,0x6972930e),LL(0x23807bbe,0x2b7c77b8),LL(0xcb27ff50,0xe5b82405),
   LL(0xbd379062,0xba8b8be3),LL(0x2dce4a92,0xd64b7a1d),LL(0xb2952e37,0x040a73c5),LL(0xd438aeca,0x0a9e252e), LL(0xc39d3bcb,0xdd43956b),LL(0xb32b2d63,0x1a31ca00),LL(0x5c417a18,0xd67133b8),LL(0x2ef442c8,0xd08e4790),
   LL(0x255c0980,0x98cb1ae9),LL(0x2b4a739f,0x4bd86381),LL(0x1e4a45a1,0x5a5c31e1),LL(0x9cb0db2f,0x1e5d55fe), LL(0x8ff5cc29,0x74661b06),LL(0x0eb8a4f4,0x026b389f),LL(0x58848c24,0x536b21a4),LL(0x81dc72b0,0x2e5bf8ec),
   LL(0xad886aac,0x03c187d0),LL(0xb771b645,0x5c16878a),LL(0xc74045ab,0xb07dfc6f),LL(0x7800caed,0x2c6360bf), LL(0xb9c972a3,0x24295bb5),LL(0x7c9a6dba,0xc9e6f88e),LL(0x92a79aa6,0x90ffbf24),LL(0x41c26ac2,0xde29d50a),
   LL(0xd309cbe6,0x9f0af483),LL(0xe0bced4f,0x5b020d8a),LL(0xb38023e3,0x606e986d),LL(0x1abc6933,0xad8f2c9d), LL(0xe7400e93,0x19292e1d),LL(0x52be5e4d,0xfe3e18a9),LL(0x2e0680bf,0xe8e9771d),LL(0xc54db063,0x8c5bec98),
   LL(0x74a55d1f,0x2af9662a),LL(0x046f66d8,0xe3fbf28f),LL(0xd4dc4794,0xa3a72ab4),LL(0x5c7c2dd8,0x09779f45), LL(0xc3d19d8d,0xd893bdaf),LL(0x57d6a6df,0xd5a75094),LL(0x952e6255,0x8cf8fef9),LL(0xda9a8aff,0x3da67cfb),
   LL(0x2c160dcd,0x4c23f62a),LL(0x8f90eaef,0x34e6c5e3),LL(0xa9a65d5a,0x35865519),LL(0x8fd38a3d,0x07c48aae), LL(0x50068527,0xb7e7aeda),LL(0x1c90936a,0x2c09ef23),LL(0xe879324c,0x31ecfeb6),LL(0xfb0ec938,0xa0871f6b),
   LL(0xd84d835d,0xb1f0fb68),LL(0x861dc1e6,0xc90caf39),LL(0x7594f8d7,0x12e5b046),LL(0x65012b92,0x26897ae2), LL(0xa4d6755d,0xbcf68a08),LL(0x0991fbda,0x403ee41c),LL(0x3bbf17e8,0x733e343e),LL(0x679b3d65,0xd2c7980d),
   LL(0xd2e11305,0x33056232),LL(0xf3c07a6f,0x966be492),LL(0xbb15509d,0x6a8878ff),LL(0x0a9b59a4,0xff221101), LL(0xabe30129,0x6c9f564a),LL(0x336e64cf,0xc6f2c940),LL(0x8b0c8022,0x0fe75262),LL(0x6ae8db87,0xbe0267e9),
   LL(0x93bc042b,0x22e192f1),LL(0xb237c458,0xf085b534),LL(0x832c4168,0xa0d192bd),LL(0xbdf6271d,0x7a76e9e3), LL(0xb88911b5,0x52a882fa),LL(0xb4db0eb5,0xc85345e4),LL(0x81a7c3ff,0xa3be02a6),LL(0xf0ec0469,0x51889c8c),
   LL(0xa5e829e5,0x9d031369),LL(0x1607aa41,0xcbb4c6fc),LL(0x241d84c1,0x75ac59a6),LL(0x8829e0ee,0xc043f2bf), LL(0x8ea5e185,0x82a38f75),LL(0xd87cbd9f,0x8bda40b9),LL(0x2d8fc601,0x9e65e75e),LL(0xa35690b3,0x3d515f74),
   LL(0xda79e5ac,0x534acf4f),LL(0x8630215f,0x68b83b3a),LL(0xd085756e,0x5c748b2e),LL(0xe5d37cb2,0xb0317258), LL(0xc5ccc2c4,0x6735841a),LL(0x3d9d5069,0x7d7dc96b),LL(0xfd1754bd,0xa147e410),LL(0xd399ddd5,0x65296e94),
   LL(0xbc8fa5bc,0xf6b5b2d0),LL(0x500c277b,0x8a5ead67),LL(0xdfa08a5d,0x214625e6),LL(0x959cf047,0x51fdfedc), LL(0x289fca32,0x6bc9430b),LL(0x9d9bdc3f,0xe36ff0cf),LL(0x58ea0ede,0x2fe187cb),LL(0x5a900b3f,0xed66af20),
   LL(0x5fa9f4d6,0x00e0968b),LL(0x37a362e7,0x2d4066ce),LL(0xbd07e772,0xa99a9748),LL(0x06a4f1d0,0x710989c0), LL(0xce40cbd8,0xd5dedf35),LL(0x1743293d,0xab55c5f0),LL(0x8aa24e2c,0x766f1144),LL(0x605fbcb4,0x94d874f8),
   LL(0xa518001b,0xa365f0e8),LL(0x9d04ef0f,0xee605eb6),LL(0xba8d4d25,0x5a3915cd),LL(0xb5113472,0x44c0e1b8), LL(0x8b6740dc,0xcbb024e8),LL(0xee1d4f0c,0x89087a53),LL(0x1fc4e372,0xa88fa05c),LL(0xaf8b3af2,0x8bf395cb),
   LL(0xdeb8568b,0x1e71c9a1),LL(0x80fb3d32,0xa35daea0),LL(0x2cf8fb81,0xe8b6f266),LL(0x9490696a,0x6d51afe8), LL(0x51803a19,0x81beac6e),LL(0x86219080,0xe3d24b7f),LL(0xdf6f463c,0x727cfd9d),LL(0x72284ee8,0x8c6865ca),
   LL(0xb743f4ef,0x32c88b7d),LL(0xe7d11dce,0x3793909b),LL(0x2ff2ebe8,0xd398f922),LL(0xe5e49796,0x2c70ca44), LL(0xcb1131b1,0xdf4d9929),LL(0x25888e79,0x7826f298),LL(0xf1d8740a,0x4d3a112c),LL(0x270afa8b,0x00384cb6),
   LL(0x3ab48095,0xcb64125b),LL(0x62d05106,0x3451c256),LL(0xa4955845,0xd73d577d),LL(0xbf9f4433,0x39570c16), LL(0xadecf263,0xd7dfaad3),LL(0xdc76e102,0xf1c3d8d1),LL(0x54c6a836,0x5e774a58),LL(0x3e92d47b,0xdad4b672),
   LL(0xf0d796a0,0xbe7e990f),LL(0xdf0e8b02,0x5fc62478),LL(0x030c00ad,0x8aae8bf4),LL(0x9004ba0f,0x3d2db93b), LL(0xd85d5ddc,0xe48c8a79),LL(0x6bb07f34,0xe907caa7),LL(0xa39eaed5,0x58db343a),LL(0xadaf5724,0x0ea6e007),
   LL(0xd23233f3,0xe00df169),LL(0x77cb637f,0x3e322796),LL(0x1da0cf6c,0x1f897c0e),LL(0x31d6bbdd,0xa651f5d8), LL(0x1a230c76,0xdd61af19),LL(0xcdaa5e4a,0xbd527272),LL(0xd0abcd7e,0xca753636),LL(0x370bd8dc,0x78bdd37c),
   LL(0x17cd93fe,0xc23916c2),LL(0xdadce6e2,0x65b97a4d),LL(0x174e42f8,0xe04ed4eb),LL(0xbb21480a,0x1491ccaa), LL(0x23196332,0x145a8280),LL(0x587b479a,0x3c3862d7),LL(0x01dcd0ed,0x9f4a88a3),LL(0x3ea12f1f,0x4da2b7ef),
   LL(0xb126e48e,0xf8e7ae33),LL(0xf494e237,0x404a0b32),LL(0xc55acadb,0x9beac474),LL(0xcbec9fd9,0x4ee5cf3b), LL(0x7df3c8c3,0x336b33b9),LL(0xb76808fd,0xbd905fe3),LL(0xaa45c16a,0x8f436981),LL(0x3dd27b62,0x255c5bfa),
   LL(0xc3dd9b4d,0x71965cbf),LL(0xfc068a87,0xce23edbf),LL(0x745b029b,0xb78d4725),LL(0xcefdd9bd,0x74610713), LL(0x1266bf52,0x7116f75f),LL(0x18e49bb6,0x02046722),LL(0x3d6f19e3,0xdf43df9f),LL(0xe685cb2f,0xef1bc7d0),
   LL(0x7078c432,0xcddb27c1),LL(0xb77fedb7,0xe1961b9c),LL(0xc2290570,0x1edc2f5c),LL(0x19cbd886,0x2c3fefca), LL(0xc2af389a,0xcf880a36),LL(0xbda71cea,0x96c610fd),LL(0x32aa8463,0xf03977a9),LL(0x8586d90a,0x8eb7763f),
   LL(0x2a296e77,0x3f342454),LL(0x42837a35,0xc8718683),LL(0x6a09c731,0x7dc71090),LL(0x51b816db,0x54778ffb), LL(0xaf06defd,0x6b33bfec),LL(0x8592b70b,0xfe3c105f),LL(0x61da6114,0xf937fda4),LL(0x4c266ad7,0x3c13e651),
   LL(0x855938e8,0xe363a829),LL(0x9de54b72,0x2eeb5d9e),LL(0x20ccfab9,0xbeb93b0e),LL(0x25e61a25,0x3dffbb5f), LL(0x1acc093d,0x7f655e43),LL(0x3964ce61,0x0cb6cc3d),LL(0xe5e9b460,0x6ab283a1),LL(0xa1c7e72d,0x55d787c5),
   LL(0xdeadbf02,0x4d2efd47),LL(0xac459068,0x11e80219),LL(0x71f311f0,0x810c7626),LL(0x4ab6ef53,0xfa17ef8d), LL(0x93e43bff,0xaf47fd25),LL(0x0be40632,0x5cb5ff3f),LL(0x8ee61da3,0x54687106),LL(0xb08afd0f,0x7764196e),
   LL(0xf0290a8f,0x831ab3ed),LL(0xcb47c387,0xcae81966),LL(0x184efb4f,0xaad7dece),LL(0x4749110e,0xdcfc53b3), LL(0x4cb632f9,0x6698f23c),LL(0xb91f8067,0xc42a1ad6),LL(0x6284180a,0xb116a81d),LL(0xe901326f,0xebedf5f8),
   LL(0x97e3e044,0xf2274c9f),LL(0x11d09fc9,0x42018520),LL(0xd18e6e23,0x56a65f17),LL(0x352b683c,0x2ea61e2a), LL(0x575eaa94,0x27d291bc),LL(0xb8ff522d,0x9e7bc721),LL(0xa7f04d6f,0x5f7268bf),LL(0xaba41748,0x5868c73f),
   LL(0x7be0eead,0x9f85c2db),LL(0xff719135,0x511e7842),LL(0xc5ea90d7,0x5a06b1e9),LL(0x26fab631,0x0c19e283), LL(0xe9206c55,0x8af8f0cf),LL(0x3553c06a,0x89389cb4),LL(0xf65f8004,0x39dbed97),LL(0xc508991d,0x0621b037),
   LL(0x96e78cc4,0x1c52e635),LL(0x0c06b4a8,0x5385c8b2),LL(0xb0e87d03,0xd84ddfdb),LL(0x934bafad,0xc49dfb66), LL(0x59f70772,0x7071e170),LL(0x3a1db56b,0x3a073a84),LL(0x3b8af190,0x03494903),LL(0xd32920f0,0x7d882de3),
   LL(0xb2cf8940,0x91633f0a),LL(0x6f948f51,0x72b0b178),LL(0x782653c8,0x2d28dc30),LL(0xdb903a05,0x88829849), LL(0x6a19d2bb,0xb8095d0c),LL(0x86f782cb,0x4b9e7f0c),LL(0x2d907064,0x7af73988),LL(0x8b32643c,0xd12be0fe),
   LL(0x0e165dc3,0x358ed23d),LL(0x4e2378ce,0x3d47ce62),LL(0xfeb8a087,0x7e2bb0b9),LL(0xe29e10b9,0x3246e8ae), LL(0x03ce2b4d,0x459f4ec7),LL(0xbbc077cf,0xe9b4ca1b),LL(0x0e9940c1,0x2613b4f2),LL(0x047d1eb1,0xfc598bb9),
   LL(0x45036099,0x9744c62b),LL(0x167c65d8,0xa9dee742),LL(0xdabe1943,0x0c511525),LL(0x93c6c624,0xda110554), LL(0x651a3be2,0xae00a52c),LL(0x884449a6,0xcda5111d),LL(0xff33bed1,0x063c06f4),LL(0x0d3d76b4,0x73baaf9a),
   LL(0x7fc63668,0x52fb0c9d),LL(0x0c039cde,0x6886c9dd),LL(0x55b22351,0x602bd599),LL(0x360c7c13,0xb00cab02), LL(0x81b69442,0x8cb616bc),LL(0xb55c3cee,0x41486700),LL(0xf49ba278,0x71093281),LL(0x64a50710,0xad956d9c),
   LL(0x638a7e81,0x9561f28b),LL(0x5980ddc3,0x54155cdf),LL(0xd26f247a,0xb2db4a96),LL(0x4787d100,0x9d774e4e), LL(0x078637d2,0x1a9e6e2e),LL(0x5e0ae06a,0x1c363e2d),LL(0xe9cfa354,0x7493483e),LL(0x7f74b98d,0x76843cb3),
   LL(0xd4b66947,0xbaca6591),LL(0x04460a8c,0xb452ce98),LL(0x43768f55,0x6830d246),LL(0x7dff12df,0xf4197ed8), LL(0x400dd0f7,0x6521b472),LL(0x4b1e7093,0x59f5ca8f),LL(0x080338ae,0x6feff11b),LL(0xa29ca3c6,0x0ada31f6),
   LL(0x94a2c215,0x24794eb6),LL(0x05a57ab4,0xd83a43ab),LL(0x2a6f89fe,0x264a543a),LL(0xdd5ec7c2,0x2c2a3868), LL(0x8439d9b2,0xd3373940),LL(0x0acd1f11,0x715ea672),LL(0xe7e6cc19,0x42c1d235),LL(0xb990585c,0x81ce6e96),
   LL(0xd809c7bd,0x04e5dfe0),LL(0x8f1050ab,0xd7b2580c),LL(0xd8a4176f,0x6d91ad78),LL(0x4e2e897c,0x0af556ee), LL(0x921de0ac,0x162a8b73),LL(0x7ea78400,0x52ac9c22),LL(0xefce2174,0xee2a4eea),LL(0x6d637f79,0xbe61844e),
   LL(0x789a283b,0x0491f1bc),LL(0x880836f4,0x72d3ac3d),LL(0x88e5402d,0xaa1c5ea3),LL(0xd5cc473d,0x1b192421), LL(0x9dc84cac,0x5c0b9998),LL(0x9c6e75b8,0xb0a8482d),LL(0x3a191ce2,0x639961d0),LL(0x6d837930,0xda3bc865),
   LL(0x056e6f8f,0xca990653),LL(0x64d133a7,0x84861c41),LL(0x746abe40,0x8b403276),LL(0xebf8e303,0xb7b4d51a), LL(0x220a255d,0x05b43211),LL(0x02419e6e,0xc997152c),LL(0x630c2fea,0x76ff47b6),LL(0x281fdade,0x50518677),
   LL(0xcf902b0b,0x3283b8ba),LL(0x37db303b,0x8d4b4eb5),LL(0x755011bc,0xcc89f42d),LL(0xdd09d19b,0xb43d74bb), LL(0x8adba350,0x65746bc9),LL(0xb51c1927,0x364eaf8c),LL(0x10ad72ec,0x13c76596),LL(0xf8d40c20,0x30045121),
   LL(0xea7b979b,0x6d2d99b7),LL(0xe6fb3bcd,0xcd78cd74),LL(0x86cffbfe,0x11e45a9e),LL(0x637024f6,0x78a61cf4), LL(0x3d502295,0xd06bc872),LL(0x458cb288,0xf1376854),LL(0x342f8586,0xb9db26a1),LL(0x4beee09e,0xf33effcf),
   LL(0xb30cfb3a,0xd7e0c4cd),LL(0x6c9db4c8,0x6d09b8c1),LL(0x07c8d9df,0x40ba1a42),LL(0x1c52c66d,0x6fd495f7), LL(0x275264da,0xfb0e169f),LL(0xe57d8362,0x80c2b746),LL(0x49ad7222,0xedd987f7),LL(0x4398ec7b,0xfdc229af),
},
/* digit=25 base_pwr=2^175 */
{
   LL(0x52666a58,0xb0d1ed84),LL(0xe6a9c3c2,0x4bcb6e00),LL(0x26906408,0x3c57411c),LL(0x13556400,0xcfc20755), LL(0x5294dba3,0xa08b1c50),LL(0x8b7dd31e,0xa30ba286),LL(0x991eca74,0xd70ba90e),LL(0xe762c2b9,0x094e142c),
   LL(0x979f3925,0xb81d783e),LL(0xaf4c89a7,0x1efd130a),LL(0xfd1bf7fa,0x525c2144),LL(0x1b265a9e,0x4b296904), LL(0xb9db65b6,0xed8e9634),LL(0x03599d8a,0x35c82e32),LL(0x403563f3,0xdaa7a54f),LL(0x022c38ab,0x9df088ad),
   LL(0xbb3fd30a,0xe5cfb066),LL(0xeff0354e,0x429169da),LL(0x3524e36c,0x809cf852),LL(0x0155be1d,0x136f4fb3), LL(0x1fbba712,0x4826af01),LL(0x506ba1a1,0x6ef0f0b4),LL(0x77aea73e,0xd9928b31),LL(0x5eaa244e,0xe2bf6af2),
   LL(0x4237b64b,0x8d084f12),LL(0xe3ecfd07,0x688ebe99),LL(0xf6845dd8,0x57b8a70c),LL(0x5da4a325,0x808fc59c), LL(0xa3585862,0xa9032b2b),LL(0xedf29386,0xb66825d5),LL(0x431ec29b,0xb5a5a8db),LL(0x3a1e8dc8,0xbb143a98),
   LL(0x12ae381b,0x35ee94ce),LL(0x86ccda90,0x3a7f176c),LL(0x4606eaca,0xc63a657e),LL(0x43cd04df,0x9ae5a380), LL(0xed251b46,0x9bec8d15),LL(0xcaca5e64,0x1f5d6d30),LL(0x9ff20f07,0x347b3b35),LL(0xf7e4b286,0x4d65f034),
   LL(0xf111661e,0x9e93ba24),LL(0xb105eb04,0xedced484),LL(0xf424b578,0x96dc9ba1),LL(0xe83e9069,0xbf8f66b7), LL(0xd7ed8216,0x872d4df4),LL(0x8e2cbecf,0xbf07f377),LL(0x98e73754,0x4281d899),LL(0x8aab8708,0xfec85fbb),
   LL(0xa5ba5b0b,0x9a3c0dee),LL(0x42d05299,0xe6a116ce),LL(0xe9b02d42,0xae9775fe),LL(0xa1545cb6,0x72b05200), LL(0x31a3b4ea,0xbc506f7d),LL(0x8bbd9b32,0xe5893078),LL(0xe4b12a97,0xc8bc5f37),LL(0x4a73b671,0x6b000c06),
   LL(0x765fa7d0,0x13b5bf22),LL(0x1d6a5370,0x59805bf0),LL(0x4280db98,0x67a5e29d),LL(0x776b1ce3,0x4f53916f), LL(0x33ddf626,0x714ff61f),LL(0xa085d103,0x4206238e),LL(0xe5809ee3,0x1c50d4b7),LL(0x85f8eb1d,0x999f450d),
   LL(0xe4c79e9b,0x658a6051),LL(0xc66a9fea,0x1394cb73),LL(0xc6be7b23,0x27f31ed5),LL(0x5aa6f8fe,0xf4c88f36), LL(0x4aaa499e,0x0fb0721f),LL(0xe3fb2a6b,0x68b3a7d5),LL(0x3a92851d,0xa788097d),LL(0xe96f4913,0x060e7f8a),
   LL(0x1a3a93bc,0x82eebe73),LL(0xa21adc1a,0x42bbf465),LL(0xef030efd,0xc10b6fa4),LL(0x87b097bb,0x247aa4c7), LL(0xf60c77da,0x8b8dc632),LL(0xc223523e,0x6ffbc26a),LL(0x344579cf,0xa4f6ff11),LL(0x980250f6,0x5825653c),
   LL(0xbc1aa2b9,0xb2dd097e),LL(0x37a0333a,0x07889393),LL(0x37a0db38,0x1cf55e71),LL(0x792c1613,0x2648487f), LL(0x3fcef261,0xdad01336),LL(0x0eabf129,0x6239c81d),LL(0x9d276be2,0x8ee761de),LL(0x1eda6ad3,0x406a7a34),
   LL(0x4a493b31,0x4bf367ba),LL(0x9bf7f026,0x54f20a52),LL(0x9795914b,0xb696e062),LL(0x8bf236ac,0xcddab96d), LL(0xed25ea13,0x4ff2c70a),LL(0x81cbbbe7,0xfa1d09eb),LL(0x468544c5,0x88fc8c87),LL(0x696b3317,0x847a670d),
   LL(0x64bcb626,0xf133421e),LL(0x26dee0b5,0xaea638c8),LL(0xb310346c,0xd6e7680b),LL(0xd5d4ced3,0xe06f4097), LL(0x7512a30b,0x09961452),LL(0xe589a59a,0xf3d867fd),LL(0x52d0c180,0x2e73254f),LL(0x333c74ac,0x9063d8a3),
   LL(0xd314e7bc,0xeda6c595),LL(0x467899ed,0x2ee7464b),LL(0x0a1ed5d3,0x1cef423c),LL(0x69cc7613,0x217e76ea), LL(0xe7cda917,0x27ccce1f),LL(0x8a893f16,0x12d8016b),LL(0x9fc74f6b,0xbcd6de84),LL(0xf3144e61,0xfa5817e2),
   LL(0x0821ee4c,0x1f354164),LL(0x0bc61992,0x1583eab4),LL(0x1d72879f,0x7490caf6),LL(0xf76ae7b2,0x998ad9f3), LL(0xa41157f7,0x1e181950),LL(0xe8da3a7e,0xa9d7e1e6),LL(0x8426b95f,0x963784eb),LL(0x542e2a10,0x0ee4ed6e),
   LL(0xac751e7b,0xb79d4cc5),LL(0xfd4211bd,0x93f96472),LL(0xc8de4fc6,0x8c72d3d2),LL(0xdf44f064,0x7b69cbf5), LL(0xf4bf94e1,0x3da90ca2),LL(0xf12894e2,0x1a5325f8),LL(0x7917d60b,0x0a437f6c),LL(0x96c9cb5d,0x9be70486),
   LL(0xe1dc5c05,0xb4d880bf),LL(0xeebeeb57,0xd738adda),LL(0xdf0fe6a3,0x6f0119d3),LL(0x66eaaf5a,0x5c686e55), LL(0xdfd0b7ec,0x9cb10b50),LL(0x6a497c21,0xbdd0264b),LL(0x8c546c96,0xfc093514),LL(0x79dbf42a,0x58a947fa),
   LL(0x49ccd6d7,0xc0b48d4e),LL(0x88bd5580,0xff8fb02c),LL(0x07d473b2,0xc75235e9),LL(0xa2188af3,0x4fab1ac5), LL(0x97576ec0,0x030fa3bc),LL(0x0b7e7d2f,0xe8c946e8),LL(0x70305600,0x40a5c9cc),LL(0xc8b013b4,0x6d8260a9),
   LL(0x70bba85c,0x0368304f),LL(0xa4a0d311,0xad090da1),LL(0x2415eec1,0x7170e870),LL(0x8461ea47,0xbfba35fe), LL(0xc1e91938,0x6279019a),LL(0x1afc415f,0xa47638f3),LL(0xbcba0e0f,0x36c65cbb),LL(0x034e2c48,0x02160efb),
   LL(0x615cd9e4,0xe6c51073),LL(0xf1243c06,0x498ec047),LL(0xb17b3d8c,0x3e5a8809),LL(0x0cc565f1,0x5cd99e61), LL(0x7851dafe,0x81e312df),LL(0xa79061e2,0xf156f5ba),LL(0x880c590e,0x80d62b71),LL(0x0a39faa1,0xbec9746f),
   LL(0xc8ed1f7a,0x1d98a9c1),LL(0xa81d5ff2,0x09e43bb5),LL(0x0da0794a,0xd5f00f68),LL(0x661aa836,0x412050d9), LL(0x90747e40,0xa89f7c4e),LL(0xb62a3686,0x6dc05ebb),LL(0x308e3353,0xdf4de847),LL(0x9fb53bb9,0x53868fbb),
   LL(0xcfdcf7dd,0x2b09d2c3),LL(0x723fcab4,0x41a9fce3),LL(0x07f57ca3,0x73d905f7),LL(0xac8e1555,0x080f9fb1), LL(0x9ba7a531,0x7c088e84),LL(0xed9a147f,0x07d35586),LL(0xaf48c336,0x602846ab),LL(0x0ccf0e79,0x7320fd32),
   LL(0xb18bd1ff,0xaa780798),LL(0xafdd2905,0x52c2e300),LL(0x434267cd,0xf27ea3d6),LL(0x15605b5f,0x8b96d16d), LL(0x4b45706b,0x7bb31049),LL(0x743d25f8,0xe7f58b8e),LL(0x87f30076,0xe9b5e45b),LL(0x5d053d5a,0xd19448d6),
   LL(0xd3210a04,0x1ecc8cb9),LL(0xdafb5269,0x6bc7d463),LL(0x67c3489f,0x3e59b10a),LL(0x65641e1b,0x1769788c), LL(0xbd6cb838,0x8a53b82d),LL(0x236d5f22,0x7066d6e6),LL(0x6908536e,0x03aa1c61),LL(0x66ae9809,0xc971da0d),
   LL(0xc49a2fac,0x01b3a86b),LL(0x3092e77a,0x3b8420c0),LL(0x7d6fb556,0x02057300),LL(0xbff40a87,0x6941b2a1), LL(0x0658ff2a,0x140b6308),LL(0x3424ab36,0x87804363),LL(0x5751e299,0x0253bd51),LL(0x449c3e3a,0xc75bcd76),
   LL(0x7f8f875d,0x92eb4090),LL(0x56c26bbf,0x9c9d754e),LL(0x8110bbe7,0x158cea61),LL(0x745f91ea,0x62a6b802), LL(0xc6e7394b,0xa79c41aa),LL(0xad57ef10,0x445b6a83),LL(0x6ea6f40c,0x0c5277eb),LL(0x88633365,0x319fe96b),
   LL(0x385f63cb,0x0b0fc61f),LL(0x22bdd127,0x41250c84),LL(0x09e942c2,0x67d153f1),LL(0xc021ad5d,0x60920d08), LL(0x724d81a5,0x229f5746),LL(0x5bba3299,0xb7ffb892),LL(0xde413032,0x518c51a1),LL(0x3c2fd94c,0x2a9bfe77),
   LL(0x3191f4fd,0xcbcde239),LL(0xd3d6ada1,0x43093e16),LL(0x58769606,0x184579f3),LL(0xd236625c,0x2c94a8b3), LL(0x5c437d8e,0x6922b9c0),LL(0xd8d9f3c8,0x3d4ae423),LL(0x2e7090a2,0xf72c31c1),LL(0xd76a55bd,0x4ac3f5f3),
   LL(0x6b6af991,0x342508fc),LL(0x1b5cebbd,0x0d527100),LL(0xdd440dd7,0xb84740d0),LL(0x780162fd,0x748ef841), LL(0xdfc6fafb,0xa8dbfe0e),LL(0xf7300f27,0xeadfdf05),LL(0xfeba4ec9,0x7d06555f),LL(0x9e25fa97,0x12c56f83),
   LL(0xd39b8c34,0x77f84203),LL(0x3125eddb,0xed8b1be6),LL(0xf6e39dc5,0x5bbf2441),LL(0x6a5d678a,0xb00f6ee6), LL(0x57d0ea99,0xba456ecf),LL(0x17e06c43,0xdcae0f58),LL(0x0f5b4baa,0x01643de4),LL(0xd161b9be,0x2c324341),
   LL(0xe126d468,0x80177f55),LL(0x76748e09,0xed325f1f),LL(0xcfa9bdc2,0x6116004a),LL(0x3a9fb468,0x2d8607e6), LL(0x6009d660,0x0e573e27),LL(0x8d10c5a1,0x3a525d2e),LL(0x3b9009a0,0xd26cb45c),LL(0xde9d7448,0xb6b0cdc0),
   LL(0xe1337c26,0x949c9976),LL(0xd73d68e5,0x6faadebd),LL(0xf1b768d9,0x9e158614),LL(0x9cc4f069,0x22dfa557), LL(0xbe93c6d6,0xccd6da17),LL(0xa504f5b9,0x24866c61),LL(0x8d694da1,0x2121353c),LL(0x0140b8c6,0x1c6ca580),
   LL(0xe964021e,0xc245ad8c),LL(0x032b82b3,0xb83bffba),LL(0x47ef9898,0xfaa220c6),LL(0x982c948a,0x7e8d3ac6), LL(0xbc2d124a,0x1faa2091),LL(0x05b15ff4,0xbd54c3dd),LL(0xc87c6fb7,0x386bf3ab),LL(0xfdeb6f66,0xfb2b0563),
   LL(0x5b45afb4,0x4e77c557),LL(0xefb8912d,0xe9ded649),LL(0x42f6e557,0x7ec9bbf5),LL(0x62671f00,0x2570dfff), LL(0x88e084bd,0x2b3bfb78),LL(0xf37fe5b4,0xa024b238),LL(0x95649aee,0x44e7dc04),LL(0x5e7ec1d8,0x498ca255),
   LL(0xaaa07e86,0x3bc766ea),LL(0xf3608586,0x0db6facb),LL(0xbdc259c8,0xbadd2549),LL(0x041c649f,0x95af3c6e), LL(0x02e30afb,0xb36a928c),LL(0x008a88b8,0x9b5356ad),LL(0xcf1d9e9d,0x4b67a5f1),LL(0xa5d8d8ce,0xc6542e47),
   LL(0x7adfb6cc,0x73061fe8),LL(0x98678141,0xcc826fd3),LL(0x3c80515a,0x00e758b1),LL(0x41485083,0x6afe3247), LL(0xb6ae8a75,0x0fcb08b9),LL(0x4acf51e1,0xb8cf388d),LL(0x6961b9d6,0x344a5560),LL(0x6a97fd0c,0x1a6778b8),
   LL(0xecc4c7e3,0xd840fdc1),LL(0x16db68cc,0xde9fe47d),LL(0xa3e216aa,0xe95f89de),LL(0x9594a8be,0x84f1a6a4), LL(0x5a7b162b,0x7ddc7d72),LL(0xadc817a3,0xc5cfda19),LL(0x78b58d46,0x80a5d350),LL(0x82978f19,0x93365b13),
   LL(0x26a1fc90,0x2e44d225),LL(0x4d70705d,0x0d6d10d2),LL(0xd70c45f4,0xd94b6b10),LL(0xb216c079,0x0f201022), LL(0x658fde41,0xcec966c5),LL(0x7e27601d,0xa8d2bc7d),LL(0xff230be7,0xbfcce3e1),LL(0x0033ffb5,0x3394ff6b),
   LL(0x8132c9af,0xd890c509),LL(0x361e7868,0xaac4b0eb),LL(0xe82d15aa,0x5194ded3),LL(0x23ae6b7d,0x4550bd2e), LL(0xea5399d4,0x3fda318e),LL(0x91638b80,0xd989bffa),LL(0xa14aa12d,0x5ea124d0),LL(0x3667b944,0x1fb1b899),
   LL(0x44c44d6a,0x95ec7969),LL(0x57e86137,0x91df144a),LL(0x73adac44,0x915fd620),LL(0x59a83801,0x8f01732d), LL(0x3aa0a633,0xec579d25),LL(0xc9d6d59c,0x06de5e7c),LL(0xb1ef8010,0xc132f958),LL(0xe65c1a02,0x29476f96),
   LL(0xd34c3565,0x336a77c0),LL(0x1b9f1e9e,0xef1105b2),LL(0xf9e08002,0x63e6d08b),LL(0xc613809e,0x9aff2f21), LL(0x3a80e75d,0xb5754f85),LL(0x6bbda681,0xde71853e),LL(0x8197fd7a,0x86f041df),LL(0x127817fa,0x8b332e08),
   LL(0xb9c20cda,0x05d99be8),LL(0xd5cd0c98,0x89f7aad5),LL(0x5bb94183,0x7ef936fe),LL(0xb05cd7f2,0x92ca0753), LL(0x74a1e035,0x9d65db11),LL(0x13eaea92,0x02628cc8),LL(0x49e4fbf2,0xf2d9e242),LL(0xe384f8b7,0x94fdfd9b),
   LL(0x63428c6b,0x65f56054),LL(0x90b409a5,0x2f7205b2),LL(0xff45ae11,0xf778bb78),LL(0xc5ee53b2,0xa13045be), LL(0x03ef77fe,0xe00a14ff),LL(0xffef8bef,0x689cd59f),LL(0x1e9ade22,0x3578f0ed),LL(0x6268b6a8,0xe99f3ec0),
   LL(0xea1b3c3e,0xa2057d91),LL(0xb8823a4a,0x2d1a7053),LL(0x2cca451e,0xabbb336a),LL(0x2218bb5d,0xcd2466e3), LL(0xc8cb762d,0x3ac1f42f),LL(0x7690211f,0x7e312aae),LL(0x45d07450,0xebb9bd73),LL(0x46c2213f,0x207c4b82),
   LL(0x375913ec,0x99d425c1),LL(0x67908220,0x94e45e96),LL(0xcd67dbf6,0xc08f3087),LL(0xc0887056,0xa5670fbe), LL(0x66f5b8fc,0x6717b64a),LL(0x786fec28,0xd5a56aea),LL(0xc0ff4952,0xa8c3f55f),LL(0x457ac49b,0xa77fefae),
   LL(0x98379d44,0x29882d7c),LL(0x509edc8a,0xd000bdfb),LL(0xe66fe464,0xc6f95979),LL(0xfa61bde0,0x504a6115), LL(0xeffea31a,0x56b3b871),LL(0xf0c21a54,0x2d3de26d),LL(0x834753bf,0x21dbff31),LL(0x69269d86,0xe67ecf49),
   LL(0x151fe690,0x7a176952),LL(0x7f2adb5f,0x03515804),LL(0xd1b62a8d,0xee794b15),LL(0xaae454e6,0xf004ceec), LL(0xf0386fac,0x0897ea7c),LL(0xd1fca751,0x3b62ff12),LL(0x1b7a04ec,0x154181df),LL(0xfb5847ec,0x2008e04a),
   LL(0x41dbd772,0xd147148e),LL(0x22942654,0x2b419f73),LL(0xe9c544f7,0x669f30d3),LL(0xc8540149,0x52a2c223), LL(0x634dfb02,0x5da9ee14),LL(0xf47869f3,0x5f074ff0),LL(0xa3933acc,0x74ee878d),LL(0x4fe35ed1,0xe6510651),
   LL(0xf1012e7a,0xb3eb9482),LL(0xa8a566ae,0x51013cc0),LL(0x47c00d3b,0xdd5e9243),LL(0x946bb0e5,0x7fde089d), LL(0xc731b4b3,0x030754fe),LL(0x99fda062,0x12a136a4),LL(0x5a1a35bc,0x7c1064b8),LL(0x446c84ef,0xbf1f5763),
   LL(0xa16d4b34,0xed29a56d),LL(0xdca21c4f,0x7fba9d09),LL(0x6d8de486,0x66d7ac00),LL(0x73a2a5e1,0x60061987), LL(0x9da28ff0,0x8b400f86),LL(0x43c4599c,0x3133f708),LL(0xee28cb0d,0x9911c9b8),LL(0x8e0af61d,0xcd7e2874),
   LL(0x72ed91fc,0x5a85f0f2),LL(0x9cd4a373,0x85214f31),LL(0x1925253c,0x881fe5be),LL(0x91e8bc76,0xd8dc98e0), LL(0x585cc3a2,0x7120affe),LL(0x735bf97a,0x724952ed),LL(0x3eb34581,0x5581e7dc),LL(0xe52ee57d,0x5cbff4f2),
   LL(0x87d8cc7b,0x8d320a0e),LL(0xf1d280d0,0x9beaa7f3),LL(0x9beec704,0x7a0b9571),LL(0x5b7f0057,0x9126332e), LL(0x8ed3bd6d,0x01fbc1b4),LL(0xd945eb24,0x35bb2c12),LL(0x9a8ae255,0x6404694e),LL(0x8d6abfb3,0xb6092eec),
   LL(0xcc058865,0x4d76143f),LL(0x6e249922,0x7b0a5af2),LL(0x6a50d353,0x8aef9440),LL(0x64f0e07a,0xe11e4bcc), LL(0xa14a90fa,0x4472993a),LL(0xba0c51d4,0x7706e20c),LL(0x1532672d,0xf403292f),LL(0x21829382,0x52573bfa),
   LL(0x3b5bdb83,0x6a7bb6a9),LL(0xa4a72318,0x08da65c0),LL(0x63eb065f,0xc58d22aa),LL(0x1b15d685,0x1717596c), LL(0xb266d88b,0x112df0d0),LL(0x5941945a,0xf688ae97),LL(0x7c292cac,0x487386e3),LL(0x57d6985c,0x42f3b50d),
   LL(0x6a90fc34,0x6da4f998),LL(0x65ca8a8d,0xc8f257d3),LL(0x6951f762,0xc2feabca),LL(0x74c323ac,0xe1bc81d0), LL(0x251a2a12,0x1bc68f67),LL(0xbe8a70dc,0x10d86587),LL(0xf0f84d2e,0xd648af7f),LL(0x6a43ac92,0xf0aa9ebc),
   LL(0x27596893,0x69e3be04),LL(0x45bf452b,0xb6bb02a6),LL(0xf4c698c8,0x0875c11a),LL(0xbece3794,0x6652b5c7), LL(0x4f5c0499,0x7b3755fd),LL(0xb5532b38,0x6ea16558),LL(0xa2e96ef7,0xd1c69889),LL(0x61ed8f48,0x9c773c3a),
   LL(0x9b323abc,0x2b653a40),LL(0xf0e1d791,0xe26605e1),LL(0x4a87157a,0x45d41064),LL(0xcbbce616,0x8f9a78b7), LL(0xc407eddd,0xcf1e44aa),LL(0xa35b964f,0x81ddd1d8),LL(0xfd083999,0x473e339e),LL(0x8e796802,0x6c94bdde),
   LL(0x8545d185,0x5a304ada),LL(0x738bb8cb,0x82ae44ea),LL(0xdf87e10e,0x628a35e3),LL(0xa15b9fe3,0xd3624f3d), LL(0x14be4254,0xcc44209b),LL(0xbdbc2ea5,0x7d0efcbc),LL(0x04c37bbe,0x1f603362),LL(0x56a5852c,0x21f363f5),
   LL(0xa8501550,0xa1503d1c),LL(0xd8ab10bb,0x2251e0e1),LL(0x6961c51c,0xde129c96),LL(0x81910f68,0x1f7246a4), LL(0x5f2591f2,0x2eb744ee),LL(0x5e627157,0x3c47d33f),LL(0x22f3bd68,0x4d6d62c9),LL(0xcb8df856,0x6120a64b),
   LL(0x7b5d07df,0x3a9ac6c0),LL(0x7ef39783,0xa92b9558),LL(0xab3a9b4f,0xe128a134),LL(0xb1252f05,0x41c18807), LL(0x80ba9b1c,0xfc7ed089),LL(0xc532a9dd,0xac8dc6de),LL(0x55246809,0xbf829cef),LL(0x5b4ee80f,0x101b784f),
   LL(0xb6f11603,0xc09945bb),LL(0x41d2801e,0x57b09dbe),LL(0xa97534a8,0xfba5202f),LL(0xc17b9614,0x7fd8ae5f), LL(0x78308435,0xa50ba666),LL(0xd3868c4d,0x9572f77c),LL(0x2dd7aab0,0x0cef7bfd),LL(0x2c7c79ff,0xe7958e08),
   LL(0x25346689,0x81262e42),LL(0xb07c7004,0x716da290),LL(0xb7950ee3,0x35f911ea),LL(0x261d21b5,0x6fd72969), LL(0x08b640d3,0x52389803),LL(0x887f12a1,0x5b0026ee),LL(0x742e9311,0x20e21660),LL(0x5ff77ff7,0x0ef6d541),
   LL(0xf9c41135,0x969127f0),LL(0x68a64993,0xf21d60c9),LL(0xe541875c,0x656e5d0c),LL(0xa1d3c233,0xf1e0f84e), LL(0x06002d60,0x9bcca359),LL(0x06191552,0xbe2da60c),LL(0x61181ec3,0x5da8bbae),LL(0x65806f19,0x9f04b823),
   LL(0xd4b79bb8,0xf1604a7d),LL(0x52c878c8,0xaee806fb),LL(0x8d47b8e8,0x34144f11),LL(0x949f9054,0x72edf52b), LL(0x2127015a,0xebfca84e),LL(0x9cb7cef3,0x9051d0c0),LL(0x296deec8,0x86e8fe58),LL(0x41010d74,0x33b28188),
},
/* digit=26 base_pwr=2^182 */
{
   LL(0x171b445f,0x01079383),LL(0x8131ad4c,0x9bcf21e3),LL(0xc93987e8,0x8cdfe205),LL(0xc92e8c8f,0xe63f4152), LL(0x30add43d,0x729462a9),LL(0xc980f05a,0x62ebb143),LL(0x3b06e968,0x4f3954e5),LL(0x242cf6b1,0xfe1d75ad),
   LL(0xaf8685c8,0x5f95c6c7),LL(0x2f8f01aa,0xd4c1c8ce),LL(0x2574692a,0xc44bbe32),LL(0xd4a4a068,0xb8003478), LL(0x2eca3cdb,0x7c8fc6e5),LL(0xec04d399,0xea1db16b),LL(0x8f2bc5cf,0xb05bc82e),LL(0xf44793d2,0x763d517f),
   LL(0x08bd98d0,0x4451c1b8),LL(0x6575f240,0x644b1cd4),LL(0x7375d270,0x6907eb33),LL(0xfa2286bd,0x56c8bebd), LL(0xc4632b46,0xc713d2ac),LL(0xafd60242,0x17da427a),LL(0xc95c7546,0x313065b7),LL(0xbf17a3de,0xf8239898),
   LL(0x4c830320,0xf3b7963f),LL(0x903203e3,0x842c7aa0),LL(0xe7327afb,0xaf22ca0a),LL(0x967609b6,0x38e13092), LL(0x757558f1,0x73b8fb62),LL(0xf7eca8c1,0x3cc3e831),LL(0xf6331627,0xe4174474),LL(0xc3c40234,0xa77989ca),
   LL(0x44a081e0,0xe5fd17a1),LL(0xb70e296a,0xd797fb7d),LL(0x481f719c,0x2b472b30),LL(0xfe6f8c52,0x0e632a98), LL(0xc5f0c284,0x89ccd116),LL(0x2d987c62,0xf51088af),LL(0x4c2de6cf,0x2a2bccda),LL(0xf679f0f9,0x810f9efe),
   LL(0x7ffe4b3e,0xb0f394b9),LL(0xe5fa5d21,0x0b691d21),LL(0x9dfbbc75,0xb0bd7747),LL(0xfaf78b00,0xd2830fda), LL(0x52434f57,0xf78c249c),LL(0x98096dab,0x4b1f7545),LL(0x8ff8c0b3,0x73bf6f94),LL(0x454e134c,0x34aef03d),
   LL(0xb7ac7ec5,0xf8d151f4),LL(0xe50da7d5,0xd6ceb95a),LL(0xdc3a0eb8,0xa1b492b0),LL(0xb3dd2863,0x75157b69), LL(0xc5413d62,0xe2c4c74e),LL(0xbc5fc4c7,0xbe329ff7),LL(0x60fa9dda,0x835a2aea),LL(0x7445cb87,0xf117f5ad),
   LL(0xb0166f7a,0xae8317f4),LL(0xceec74e6,0xfbd3e3f7),LL(0xe0874bfd,0xfdb516ac),LL(0xc681f3a3,0x3d846019), LL(0x7c1620b0,0x0b12ee5c),LL(0x2b63c501,0xba68b4dd),LL(0x6668c51e,0xac03cd32),LL(0x4e0bcb5b,0x2a6279f7),
   LL(0x6ae85c10,0x17bd69b0),LL(0x1dfdd3a6,0x72946979),LL(0x2c078bec,0xd9a03268),LL(0xbfd68a52,0x41c6a658), LL(0x0e023900,0xcdea1024),LL(0xb10d144d,0xbaeec121),LL(0x058ab8dc,0x5a600e74),LL(0xbb89ccdd,0x1333af21),
   LL(0x3aaba1f1,0xdf25eae0),LL(0x3b7144cf,0x2cada16e),LL(0x71ab98bc,0x657ee27d),LL(0x7a6fc96e,0x99088b4c), LL(0x3549dbd4,0x05d5c0a0),LL(0xf158c3ac,0x42cbdf8f),LL(0x87edd685,0x3fb6b3b0),LL(0x86f064d0,0x22071cf6),
   LL(0xff2811e5,0xd2d6721f),LL(0xfe7fae8c,0xdb81b703),LL(0xd3f1f7bb,0x3cfb74ef),LL(0x16cdeb5d,0x0cdbcd76), LL(0x566a808c,0x4f39642a),LL(0x340064d6,0x02b74454),LL(0x0528fa6f,0xfabbadca),LL(0xd3fc0bb6,0xe4c3074c),
   LL(0xb796d219,0xb32cb8b0),LL(0x34741dd9,0xc3e95f4f),LL(0x68edf6f5,0x87212125),LL(0xa2b9cb8e,0x7a03aee4), LL(0xf53a89aa,0x0cd3c376),LL(0x948a28dc,0x0d8af9b1),LL(0x902ab04f,0xcf86a3f4),LL(0x7f42002d,0x8aacb62a),
   LL(0xf62ffd52,0x106985eb),LL(0x5797bf10,0xe670b54e),LL(0xc5e30aef,0x4b405209),LL(0x4365b5e9,0x12c97a20), LL(0x1fe32093,0x104646ce),LL(0x3907a8c9,0x13cb4ff6),LL(0xd46e726b,0x8b9f30d1),LL(0xaba0f499,0xe1985e21),
   LL(0x10a230cd,0xc573dea9),LL(0xcd30f947,0x24f46a93),LL(0xabe2010a,0xf2623fcf),LL(0x73f00e4f,0x3f278cb2), LL(0x50b920eb,0xed55c67d),LL(0x8e760571,0xf1cb9a2d),LL(0x0895b709,0x7c50d109),LL(0x190d4369,0x4207cf07),
   LL(0xc4127fe1,0x3b027e81),LL(0x3ae9c566,0xa9f8b9ad),LL(0xacbfbba5,0x5ab10851),LL(0x569556f5,0xa747d648), LL(0x2ba97bf7,0xcc172b5c),LL(0xbcfa3324,0x15e0f77d),LL(0x7686279d,0xa345b797),LL(0xe38003d3,0x5a723480),
   LL(0x8f5fcda8,0xfd8e139f),LL(0xbdee5bfd,0xf3e558c4),LL(0xe33f9f77,0xd76cbaf4),LL(0x71771969,0x3a4c97a4), LL(0xf6dce6a7,0xda27e84b),LL(0x13e6c2d1,0xff373d96),LL(0xd759a6e9,0xf115193c),LL(0x63d2262c,0x3f9b7025),
   LL(0x317cd062,0xd9764a31),LL(0x199f8332,0x30779d8e),LL(0x16b11b0b,0xd8074106),LL(0x78aeaed8,0x7917ab9f), LL(0x28fb1d8e,0xb67a9cbe),LL(0x136eda33,0x2e313563),LL(0xa371a86c,0x010b7069),LL(0x6744e6b7,0x44d90fa2),
   LL(0xd6b3e243,0x68190867),LL(0x59048c48,0x9fe6cd9d),LL(0x95731538,0xb900b028),LL(0x32cae04f,0xa012062f), LL(0x9399d082,0x8107c8bc),LL(0x41df12e2,0x47e8c54a),LL(0xb6ef3f73,0x14ba5117),LL(0x81362f0b,0x22260bea),
   LL(0x1a18cc20,0x90ea261e),LL(0x2321d636,0x2192999f),LL(0xe311b6a0,0xef64d314),LL(0x3b54a1f5,0xd7401e4c), LL(0x6fbca2ba,0x19019983),LL(0x8fbffc4b,0x46ad3293),LL(0x3786bf40,0xa142d3f6),LL(0xb67039fc,0xeb5cbc26),
   LL(0x252bd479,0x9cb0ae6c),LL(0x12b5848f,0x05e0f88a),LL(0xa5c97663,0x78f6d2b2),LL(0xc162225c,0x6f6e149b), LL(0xde601a89,0xe602235c),LL(0xf373be1f,0xd17bbe98),LL(0xa8471827,0xcaf49a5b),LL(0x18aaa116,0x7e1a0a85),
   LL(0x270580c3,0x6c833196),LL(0xf1c98a14,0x1e233839),LL(0xae34e0a5,0x67b2f7b4),LL(0xd8ce7289,0x47ac8745), LL(0x100dd467,0x2b74779a),LL(0x4ee50d09,0x274a4337),LL(0x83608bc9,0x603dcf13),LL(0xc89e8388,0xcd9da6c3),
   LL(0x355116ac,0x2660199f),LL(0xb6d18eed,0xcc38bb59),LL(0x2f4bc071,0x3075f31f),LL(0x265dc57e,0x9774457f), LL(0xc6db88bb,0x06a6a9c8),LL(0x4ec98e04,0x6429d07f),LL(0x05ecaa8b,0x8d05e57b),LL(0x7872ea7b,0x20f140b1),
   LL(0xca494693,0xdf8c0f09),LL(0xf252e909,0x48d3a020),LL(0x57b14b12,0x4c5c29af),LL(0xbf47ad1c,0x7e6fa37d), LL(0x49a0c938,0x66e7b506),LL(0x6be5f41f,0xb72c0d48),LL(0xb2359412,0x6a6242b8),LL(0x8e859480,0xcd35c774),
   LL(0x87baa627,0x12536fea),LL(0xf72aa680,0x58c1fec1),LL(0x601e5dc9,0x6c29b637),LL(0xde9e01b9,0x9e3c3c1c), LL(0x2bcfe0b0,0xefc8127b),LL(0x2a12f50d,0x35107102),LL(0x4879b397,0x6ccd6cb1),LL(0xf8a82f21,0xf792f804),
   LL(0xa9b46402,0x509d4804),LL(0xc10f0850,0xedddf85d),LL(0x4b6208aa,0x928410dc),LL(0x391012dc,0xf6229c46), LL(0x7727b9b6,0xc5a7c41e),LL(0xaa444842,0x289e4e4b),LL(0xe9a947ea,0x049ba1d9),LL(0x83c8debc,0x44f9e47f),
   LL(0x611f8b8e,0xfa77a1fe),LL(0xf518f427,0xfd2e416a),LL(0x114ebac3,0xc5fffa70),LL(0x5d89697b,0xfe57c4e9), LL(0xb1aaf613,0xfdd053ac),LL(0xea585a45,0x31df210f),LL(0x24985034,0x318cc10e),LL(0x5f1d6130,0x1a38efd1),
   LL(0x0b1e9e21,0xbf86f237),LL(0x1dbe88aa,0xb258514d),LL(0x90c1baf9,0x1e38a588),LL(0xbdb9b692,0x2936a01e), LL(0x6dd5b20c,0xd576de98),LL(0x70f98ecf,0xb586bf71),LL(0xc42d2fd7,0xcccf0f12),LL(0xfb35bd7b,0x8717e61c),
   LL(0x35e6fc06,0x8b1e5722),LL(0x0b3e13d5,0x3477728f),LL(0xaa8a7372,0x150c294d),LL(0x3bfa528a,0xc0291d43), LL(0xcec5a196,0xc6c8bc67),LL(0x5c2e8a7c,0xdeeb31e4),LL(0xfb6e1c51,0xba93e244),LL(0x2e28e156,0xb9f8b71b),
   LL(0x968a2ab9,0xce65a287),LL(0x46bbcb1f,0xe3c5ce69),LL(0xe7ae3f30,0xf8c835b9),LL(0xff72b82b,0x16bbee26), LL(0xfd42cd22,0x665e2017),LL(0xf8b1d2a0,0x1e139970),LL(0x79204932,0x125cda29),LL(0x49c3bee5,0x7aee94a5),
   LL(0x89821a66,0x68c70160),LL(0x8f981669,0xf7c37678),LL(0x48cc3645,0xd90829fc),LL(0xd70addfc,0x346af049), LL(0x370bf29c,0x2057b232),LL(0x42e650ee,0xf90c73ce),LL(0xa126ab90,0xe03386ea),LL(0x975a087b,0x0e266e7e),
   LL(0x0fca65d9,0x80578eb9),LL(0x16af45b8,0x7e2989ea),LL(0xcac75a4e,0x7438212d),LL(0x4fef36b8,0x38c7ca39), LL(0xd402676a,0x8650c494),LL(0xf72c7c48,0x26ab5a66),LL(0xce3a464e,0x4e6cb426),LL(0x2b72f841,0xf8f99896),
   LL(0x1a335cc8,0x8c318491),LL(0x6a5913e4,0x563459ba),LL(0xc7b32919,0x1b920d61),LL(0xa02425ad,0x805ab8b6), LL(0x8d006086,0x2ac512da),LL(0xbcf5c0fd,0x6ca4846a),LL(0xac2138d7,0xafea51d8),LL(0x344cd443,0xcb647545),
   LL(0xbd7d9040,0x0429ee8f),LL(0x819b9c96,0xee66a2de),LL(0xdea7d744,0x54f9ec25),LL(0x671721bb,0x2ffea642), LL(0x114344ea,0x4f19dbd1),LL(0xfd0dbc8b,0x04304536),LL(0x29ec7f91,0x014b50aa),LL(0xbb06014d,0xb5fc22fe),
   LL(0x1ee682e0,0x60d963a9),LL(0xfe85c727,0xdf48abc0),LL(0x2e707c2d,0x0cadba13),LL(0xa645aeff,0xde608d3a), LL(0xedafd883,0x05f1c28b),LL(0xbd94de1f,0x3c362ede),LL(0x13593e41,0x8dd0629d),LL(0x766d6eaf,0x0a5e736f),
   LL(0xf68cf9d1,0xbfa92311),LL(0xc1797556,0xa4f9ef87),LL(0x5601c209,0x10d75a1f),LL(0x09b07361,0x651c374c), LL(0x88b5cead,0x49950b58),LL(0x6fa9dbaa,0x0ef00058),LL(0x4e15f33a,0xf51ddc26),LL(0x2ef46140,0x1f8b5ca6),
   LL(0xee9523f0,0x343ac0a3),LL(0x975ea978,0xbb75eab2),LL(0x107387f4,0x1bccf332),LL(0x9ab0062e,0x790f9259), LL(0x1e4f6a5f,0xf1a363ad),LL(0x62519a50,0x06e08b84),LL(0x7265f1ee,0x60915187),LL(0x93ae985e,0x6a80ca34),
   LL(0xaaba4864,0x81b29768),LL(0x8d52a7d6,0xb13cabf2),LL(0x8ead03f1,0xb5c36348),LL(0x81c7c1c0,0xc932ad95), LL(0xcae1e27b,0x5452708e),LL(0x1b0df648,0x9dac4269),LL(0xdfcdb8bc,0x233e3f0c),LL(0xec540174,0xe6ceccdf),
   LL(0x95081181,0xbd0d845e),LL(0x699355d5,0xcc8a7920),LL(0xc3b375a8,0x111c0f6d),LL(0xfd51e0dc,0xfd95bc6b), LL(0x6888523a,0x4a106a26),LL(0xcb01a06d,0x4d142bd6),LL(0xadb9b397,0x79bfd289),LL(0xe9863914,0x0bdbfb94),
   LL(0x1660f6a6,0x29d8a229),LL(0x551c042d,0x7f6abcd6),LL(0x0ac3ffe8,0x13039deb),LL(0xec8523fb,0xa01be628), LL(0x0ca1c328,0x6ea34103),LL(0xb903928e,0xc74114bd),LL(0x9e9144b0,0x8aa4ff4e),LL(0x7f9a4b17,0x7064091f),
   LL(0xe447f2c4,0xa3f4f521),LL(0x604291f0,0x81b8da7a),LL(0x7d5926de,0xd680bc46),LL(0x34a1202f,0x84f21fd5), LL(0x4e9df3d8,0x1d1e3181),LL(0x39ab8d34,0x1ca4861a),LL(0x5b19aa4a,0x809ddeec),LL(0x4d329366,0x59f72f7e),
   LL(0x386d5087,0xa2f93f41),LL(0xdd67d64f,0x40bf739c),LL(0x66702158,0xb4494205),LL(0x73b1e178,0xc33c65be), LL(0x38ca6153,0xcdcd657c),LL(0xdc791976,0x97f4519a),LL(0xcd6e1f39,0xcc7c7f29),LL(0x7e3c3932,0x38de9cfb),
   LL(0x7b793f85,0xe448eba3),LL(0xf067e914,0xe9f8dbf9),LL(0xf114ae87,0xc0390266),LL(0xcd6a8e2a,0x39ed75a7), LL(0x7ffba390,0xadb14848),LL(0x6af9bc09,0x67f8cb8b),LL(0x9c7476db,0x322c3848),LL(0x52a538d6,0xa320fecf),
   LL(0xb2aced2b,0xe0493002),LL(0x616bd430,0xdfba1809),LL(0xc331be70,0x531c4644),LL(0x90d2e450,0xbc04d32e), LL(0x0f9f142d,0x1805a0d1),LL(0x47ee5a23,0x2c44a0c5),LL(0x3989b4e3,0x31875a43),LL(0x0c063481,0x6b1949fd),
   LL(0xbe0f4492,0x2dfb9e08),LL(0xe9d5e517,0x3ff0da03),LL(0xf79466a8,0x03dbe9a1),LL(0x15ea9932,0x0b87bcd0), LL(0xab1f58ab,0xeb64fc83),LL(0x817edc8a,0x6d9598da),LL(0x1d3b67e5,0x699cff66),LL(0x92635853,0x645c0f29),
   LL(0xeabaf21c,0x253cdd82),LL(0x2241659e,0x82b9602a),LL(0x2d9f7091,0x2cae07ec),LL(0x8b48cd9b,0xbe4c720c), LL(0x6f08d6c9,0x6ce5bc03),LL(0xaf10bf40,0x36e8a997),LL(0x3e10ff12,0x83422d21),LL(0xbcc12494,0x7b26d3eb),
   LL(0xc9469ad6,0xb240d2d0),LL(0x30afa05b,0xc4a11b4d),LL(0xdd6ba286,0x4b604ace),LL(0x3ee2864c,0x18486600), LL(0x8d9ce5be,0x5869d6ba),LL(0xff4bfb0d,0x0d8f68c5),LL(0x5700cf73,0xb69f210b),LL(0x6d37c135,0x61f6653a),
   LL(0x5aff5a48,0xff3d432b),LL(0x72ba3a69,0x0d81c4b9),LL(0xfa1899ef,0xee879ae9),LL(0x2d6acafd,0xbac7e2a0), LL(0x1c664399,0xd6d93f6c),LL(0x5bcb135d,0x4c288de1),LL(0x9dab7cbf,0x83031dab),LL(0x3abbf5f0,0xfe23feb0),
   LL(0xcdedca85,0x9f1b2466),LL(0x1a09538c,0x140bb710),LL(0x5e11115d,0xac8ae851),LL(0x6f03f59e,0x0d63ff67), LL(0x7d234afb,0x755e5551),LL(0x7e208fc1,0x61c2db4e),LL(0xf28a4b5d,0xaa9859ce),LL(0x34af030f,0xbdd6d4fc),
   LL(0x3be01cb1,0xd1c4a26d),LL(0x243aa07c,0x9ba14ffc),LL(0xb2503502,0xf95cd3a9),LL(0x7d2a93ab,0xe379bc06), LL(0xd4ca8d68,0x3efc18e9),LL(0x80bb412a,0x083558ec),LL(0x9645a968,0xd903b940),LL(0x9ba6054f,0xa499f0b6),
   LL(0xb8349abe,0x208b573c),LL(0x30b4fc1c,0x3baab3e5),LL(0xcb524990,0x87e978ba),LL(0xccdf0e80,0x3524194e), LL(0x7d4bcc42,0x62711725),LL(0xb90109ba,0xe90a3d9b),LL(0x1323e1e0,0x3b1bdd57),LL(0x5eae1599,0xb78e9bd5),
   LL(0x9e03d278,0x0794b746),LL(0xd70e6297,0x80178605),LL(0x99c97855,0x171792f8),LL(0xf5a86b5c,0x11b393ee), LL(0xd8884f27,0x48ef6582),LL(0xbf19ba5f,0xbd44737a),LL(0xa42062c6,0x8698de4c),LL(0x61ce9c54,0x8975eb80),
   LL(0xd7fe71f3,0xd50e57c7),LL(0xbc97ce38,0x15342190),LL(0x4df07b63,0x51bda2de),LL(0x200eb87d,0xba12aeae), LL(0xa9b4f8f6,0xabe135d2),LL(0xfad6d99c,0x04619d65),LL(0x7994937c,0x4a6683a7),LL(0x6f94f09a,0x7a778c8b),
   LL(0x20a71b89,0x8c508623),LL(0x1c229165,0x241a2aed),LL(0xaaf83a99,0x352be595),LL(0x1562bac8,0x9fbfee7f), LL(0x5c4017e3,0xeaf658b9),LL(0x15120b86,0x1dc7f9e0),LL(0x4c034d6f,0xd84f13dd),LL(0xeaea3038,0x283dd737),
   LL(0xcd85d6a2,0x197f2609),LL(0xfae60177,0x6ebbc345),LL(0x4e12fede,0xb80f031b),LL(0x07a2186b,0xde55d0c2), LL(0x24dcdd5a,0x1fb3e37f),LL(0x7ed191fb,0x8d602da5),LL(0x76023e0d,0x108fb056),LL(0x459c20c0,0x70178c71),
   LL(0x3fe54cf0,0xfad5a386),LL(0x02bbb475,0xa4a3ec4f),LL(0x919d94d7,0x1aa5ec20),LL(0xa81e4ab3,0x5d3b63b5), LL(0x5ad3d2af,0x7fa733d8),LL(0xd1ac7a37,0xfbc586dd),LL(0x40779614,0x282925de),LL(0xe74a242a,0xfe0ffffb),
   LL(0x906151e5,0x3f39e67f),LL(0x55e10649,0xcea27f5f),LL(0xc17cf7b7,0xdca1d4e1),LL(0x2fe2362d,0x0c326d12), LL(0x7dd35df3,0x05f7ac33),LL(0xc396dbdf,0x0c3b7639),LL(0x03b7db1c,0x0912f5ac),LL(0x5c9ed4a9,0x9dea4b70),
   LL(0xaae3f639,0x475e6e53),LL(0xfc278bac,0xfaba0e7c),LL(0x9490375f,0x16f9e221),LL(0xa5a7ed0a,0xaebf9746), LL(0xf41ad5d6,0x45f9af3f),LL(0xb2e99224,0x03c4623c),LL(0xb3cf56aa,0x82c5bb5c),LL(0x34567ed3,0x64311819),
   LL(0x8be489ac,0xec57f211),LL(0xb9a1104b,0x2821895d),LL(0x6064e007,0x610dc875),LL(0x5b20d0fe,0x8e526f3f), LL(0x5b645aee,0x6e71ca77),LL(0x800e10ff,0x3d1dcb9f),LL(0x189cf6de,0x36b51162),LL(0x6bb17353,0x2c5a3e30),
   LL(0x2a6c6fbf,0xc186cd3e),LL(0x4bf97906,0xa74516fa),LL(0x279d6901,0x5b4b8f4b),LL(0x2b573743,0x0c4e57b4), LL(0xb6e386b6,0x75fdb229),LL(0x99deac27,0xb46793fd),LL(0xcf712629,0xeeec47ea),LL(0xcbc3b2dd,0xe965f3c4),
   LL(0x425c6559,0x8dd1fb83),LL(0x0af06fda,0x7fc00ee6),LL(0x33d956df,0xe98c9225),LL(0x4fbdc8a2,0x0f1ef335), LL(0xb79b8ea2,0x2abb5145),LL(0xbdbff288,0x40fd2945),LL(0xd7185db7,0x6a814ac4),LL(0xc084609a,0xc4329d6f),
   LL(0xed1be45d,0xc9ba7b52),LL(0xe4cd2c74,0x891dd20d),LL(0x824139b1,0x5a4d4a7f),LL(0xb873c710,0x66c17716), LL(0x2843c4e0,0x5e5bc141),LL(0xb97eb5bf,0xd5ac4817),LL(0x450c95c7,0xc0f8af54),LL(0x318406c5,0xc91b3fa0),
   LL(0xab9d97f8,0x360c340a),LL(0x90a2d611,0xfb57bd07),LL(0xa6a6f7e5,0x4339ae3c),LL(0x2feb8a10,0x9c1fcd2a), LL(0xc7ea7432,0x972bcca9),LL(0x308076f6,0x1b0b924c),LL(0x2a5b4ca5,0x80b2814a),LL(0x61ef3b29,0x2f78f55b),
   LL(0xc18a414f,0xf838744a),LL(0x903d0a86,0xc611eaae),LL(0x2a453f55,0x94dabc16),LL(0x14efb279,0xe6f2e3da), LL(0x9320dc3c,0x5b7a6017),LL(0x8df6b5a4,0x692e382f),LL(0x2d40fa90,0x3f5e15e0),LL(0x643dd318,0xc87883ae),
   LL(0x53544774,0x511053e4),LL(0x3adba2bc,0x834d0ecc),LL(0xbae371f5,0x4215d7f7),LL(0x6c8663bc,0xfcfd57bf), LL(0xd6901b1d,0xded2383d),LL(0xb5587dc3,0x3b49fbb4),LL(0x07625f62,0xfd44a08d),LL(0x9de9b762,0x3ee4d65b),
},
/* digit=27 base_pwr=2^189 */
{
   LL(0x0d63d1fa,0x64e5137d),LL(0x02a9d89f,0x658fc052),LL(0x50436309,0x48894874),LL(0xd598da61,0xe9ae30f8), LL(0x818baf91,0x2ed710d1),LL(0x8b6a0c20,0xe27e9e06),LL(0x1c1a6b44,0x1e28dcfb),LL(0xd6ac57dc,0x883acb64),
   LL(0xc2c6ff70,0x8735728d),LL(0xc5dc2235,0x79d6122f),LL(0x19e277f9,0x23f5d003),LL(0xdded8cc7,0x7ee84e25), LL(0x63cd880a,0x91a8afb0),LL(0x3574af60,0x3f3ea7c6),LL(0x02de7f42,0x0cfcdc84),LL(0xb31aa152,0x62d0792f),
   LL(0x8a5807ce,0x8e1b4e43),LL(0xe4109a7e,0xad283893),LL(0xafd59dda,0xc30cc9cb),LL(0x3d8d8093,0xf65f36c6), LL(0xa60d32b2,0xdf31469e),LL(0x3e8191c8,0xee93df4b),LL(0x355bdeb5,0x9c1017c5),LL(0x8616aa28,0xd2623185),
   LL(0xdec31a21,0xb02c83f9),LL(0x6ad9d573,0x988c8b23),LL(0xa57be365,0x53e983ae),LL(0x646f834e,0xe968734d), LL(0x5da6309b,0x9137ea8f),LL(0xc1f1ce16,0x10f3a624),LL(0xca440921,0x782a9ea2),LL(0x5b46f1b5,0xdf94739e),
   LL(0xcce85c9b,0x9f9be006),LL(0xa4c7c2d3,0x360e70d6),LL(0xaefa1e60,0x2cd5beea),LL(0x8c3d2b6d,0x64cf63c0), LL(0xe1cf6f90,0xfb107fa3),LL(0xd5e044e6,0xb7e937c6),LL(0xce34db9f,0x74e8ca78),LL(0x3e210bd0,0x4f8b36c1),
   LL(0x34a35ea8,0x1df165a4),LL(0x4d4412f6,0x3418e0f7),LL(0x518836c3,0x5af1f8af),LL(0x130e1965,0x42ceef4d), LL(0x543a1957,0x5560ca0b),LL(0x886cb123,0xc33761e5),LL(0xfe98ed30,0x66624b1f),LL(0x1090997d,0xf772f4bf),
   LL(0x4885d410,0xf4e540bb),LL(0x9ba5f8d7,0x7287f810),LL(0xde98dfb1,0x22d0d865),LL(0xbcfbb8a3,0x49ff51a1), LL(0x6bc3012e,0xb6b6fa53),LL(0x170d541d,0x3d31fd72),LL(0x4b0f4966,0x8018724f),LL(0x87dbde07,0x79e7399f),
   LL(0xf4f8b16a,0x56f8410e),LL(0xc47b266a,0x97241afe),LL(0x6d9c87c1,0x0a406b8e),LL(0xcd42ab1b,0x803f3e02), LL(0x04dbec69,0x7f0309a8),LL(0x3bbad05f,0xa83b85f7),LL(0xad8e197f,0xc6097273),LL(0x5067adc1,0xc097440e),
   LL(0x3524ff16,0x730eafb6),LL(0x823fc6ce,0xd7f9b51e),LL(0x443e4ac0,0x27bd0d32),LL(0x4d66f217,0x40c59ad9), LL(0x17c387a4,0x6c33136f),LL(0xeb86804d,0x5043b8d5),LL(0x675a73c9,0x74970312),LL(0xf16669b6,0x838fdb31),
   LL(0x418e7ddd,0xc507b6dd),LL(0x472f19d6,0x39888d93),LL(0x0c27eb4d,0x7eae26be),LL(0xfbabb884,0x17b53ed3), LL(0x2b01ae4f,0xfc27021b),LL(0xcf488682,0x88462e87),LL(0x215e2d87,0xbee096ec),LL(0xd242e29b,0xeb2fea9a),
   LL(0xb821fc28,0x5d985b5f),LL(0xdc1e2ad2,0x89d2e197),LL(0x9030ba62,0x55b566b8),LL(0x4f41b1c6,0xe3fd41b5), LL(0xb9a96d61,0xb738ac2e),LL(0x369443f4,0x7f8567ca),LL(0xf803a440,0x8698622d),LL(0x8fe2f4dc,0x2b586236),
   LL(0x56b95bce,0xbbcc00c7),LL(0x616da680,0x5ec03906),LL(0x72214252,0x79162ee6),LL(0x86a892d2,0x43132b63), LL(0x2f3263bf,0x4bdd3ff2),LL(0x9cd0a142,0xd5b3733c),LL(0x44415ccb,0x592eaa82),LL(0x8d5474ea,0x663e8924),
   LL(0x5236344e,0x8058a25e),LL(0xbda76ee6,0x82e8df9d),LL(0x11cc3d22,0xdcf6efd8),LL(0x3b4ab529,0x00089cda), LL(0xbd38a3db,0x91d3a071),LL(0xef72b925,0x4ea97fc0),LL(0xea3edf75,0x0c9fc15b),LL(0xa4348ed3,0x5a6297cd),
   LL(0xce7c42d4,0x0d38ab35),LL(0x82feab10,0x9fd493ef),LL(0x82111b45,0x46056b6d),LL(0x73efc5c3,0xda11dae1), LL(0x5545a7fb,0xdc740278),LL(0x40d507e6,0xbdb2601c),LL(0x7066fa58,0x121dfeeb),LL(0x39ae8c2a,0x214369a8),
   LL(0x06e0956c,0x195709cb),LL(0x010cd34b,0x4c9d254f),LL(0x0471a532,0xf51e13f7),LL(0x1e73054d,0xe19d6791), LL(0xdb5c7be3,0xf702a628),LL(0xb24dde05,0xc7141218),LL(0xf29b2e2e,0xdc18233c),LL(0x85342dba,0x3a6bd1e8),
   LL(0xb311898c,0x3f747fa0),LL(0xcd0eac65,0xe2a272e4),LL(0xf914d0bc,0x4bba5851),LL(0xc4a43ee3,0x7a1a9660), LL(0xa1c8cde9,0xe5a367ce),LL(0x7271abe3,0x9d958ba9),LL(0x3d1615cd,0xf3ff7eb6),LL(0xf5ae20b0,0xa2280dce),
   LL(0xcf640147,0x56dba5c1),LL(0x5e83d118,0xea5a2e3d),LL(0xda24c511,0x04cd6b6d),LL(0xe854d214,0x1c0f4671), LL(0x69565381,0x91a6b7a9),LL(0xdecf1f5b,0xdc966240),LL(0xfcf5d009,0x1b22d21c),LL(0x9021dbd5,0x2a05f641),
   LL(0xd4312483,0x8c0ed566),LL(0x643e216f,0x5179a95d),LL(0x17044493,0xcc185fec),LL(0x54991a21,0xb3063339), LL(0x0081a726,0xd801ecdb),LL(0x4fa89bbb,0x0149b0c6),LL(0x4391b6b9,0xafe9065a),LL(0xd633f3a3,0xedc92786),
   LL(0xae6a8e13,0xe408c24a),LL(0x9f3897ab,0x85833fde),LL(0xd81a0715,0x43800e7e),LL(0xb44ffc5f,0xde08e346), LL(0xcdeff2e0,0x7094184c),LL(0x165eaed1,0x49f9387b),LL(0x777c468a,0x635d6129),LL(0x538c2dd8,0x8c0dcfd1),
   LL(0x7a6a308b,0xd6d9d9e3),LL(0x4c2767d3,0x62375830),LL(0xf38cbeb6,0x874a8bc6),LL(0xccb6fd9e,0xd94d3f1a), LL(0xba21f248,0x92a9735b),LL(0x6cd1efb0,0x272ad0e5),LL(0x05b03284,0x7437b69c),LL(0x6948c225,0xe7f04702),
   LL(0xcba2ecec,0x8a56c04a),LL(0xe3a73e41,0x0c181270),LL(0x03e93725,0x6cb34e9d),LL(0x496521a9,0xf77c8713), LL(0xfa7f9f90,0x94569183),LL(0x8c9707ad,0xf2e7aa4c),LL(0x26c1c9a3,0xced2c9ba),LL(0x40197507,0x9109fe96),
   LL(0xe9adfe1c,0x9ae868a9),LL(0x314e39bb,0x3984403d),LL(0xf2fe378f,0xb5875720),LL(0xba44a628,0x33f901e0), LL(0x3652438c,0xea1125fe),LL(0x9dd1f20b,0xae9ec4e6),LL(0xbebf7fbd,0x1e740d9e),LL(0x42dbe79c,0x6dbd3ddc),
   LL(0xedd36776,0x62082aec),LL(0xe9859039,0xf612c478),LL(0x032f7065,0xa493b201),LL(0x4ff9b211,0xebd4d8f2), LL(0xaac4cb32,0x3f23a0aa),LL(0x15ed4005,0xea3aadb7),LL(0xafa27e63,0xacf17ea4),LL(0xc11fd66c,0x56125c1a),
   LL(0x3794f8dc,0x266344a4),LL(0x483c5c36,0xdcca923a),LL(0x3f9d10a0,0x2d6b6bbf),LL(0x81d9bdf3,0xb320c5ca), LL(0x47b50a95,0x620e28ff),LL(0xcef03371,0x933e3b01),LL(0x99100153,0xf081bf85),LL(0xc3a8c8d6,0x183be9a0),
   LL(0xd6bbe24d,0x4e3ddc5a),LL(0x53843795,0xc6c74630),LL(0x65ec2d4c,0x78193dd7),LL(0xcd3c89b2,0xb8df26cc), LL(0x5a483f8d,0x98dbe399),LL(0x7dd3313a,0x72d8a957),LL(0xab0bd375,0x65087294),LL(0x7c259d16,0xfcd89248),
   LL(0x7613aa81,0x8a9443d7),LL(0x85fe6584,0x80100800),LL(0x7fb10288,0x70fc4dbc),LL(0xe86beee8,0xf58280d3), LL(0x7c978c38,0x14fdd82f),LL(0x0de44d7b,0xdf1204c1),LL(0x4160252f,0xa08a1c84),LL(0xc17646a5,0x591554ca),
   LL(0xa05bd525,0x214a37d6),LL(0x07957b3c,0x48d5f09b),LL(0xd7109bc9,0x0247cdcb),LL(0x30599ce7,0x40f9e4bb), LL(0xf46ad2ec,0xc325fa03),LL(0xc3e3f9ee,0x00f766cf),LL(0xd43a4577,0xab556668),LL(0x3ee03b93,0x68d30a61),
   LL(0x77b46a08,0x7ddc81ea),LL(0xc7480699,0xcf5a6477),LL(0x6633f683,0x43a8cb34),LL(0x92363c60,0x1b867e6b), LL(0x1f60558e,0x43921114),LL(0x2f41450e,0xcdbcdd63),LL(0xcc630e8b,0x7fc04601),LL(0x97038b43,0xea7c66d5),
   LL(0x04e99fd8,0x7259b8a5),LL(0x4785549a,0x98a8dd12),LL(0x840552e1,0x0e459a7c),LL(0x4bb0909e,0xcdfcf4d0), LL(0x53758da7,0x34a86db2),LL(0xeac997e1,0xe643bb83),LL(0x530c5b7e,0x96400bd7),LL(0xb41c8b52,0x9f97af87),
   LL(0xfbeee3f9,0x34fc8820),LL(0x49091afd,0x93e53490),LL(0x9a31f35c,0x764b9be5),LL(0x57e3d924,0x71f37864), LL(0x943aa75e,0x02fb34e0),LL(0xab8ff6e4,0xa18c9c58),LL(0x33cf0d19,0x080f31b1),LL(0x083518a7,0x5c9682db),
   LL(0xb709c3de,0x873d4ca6),LL(0x3575b8f0,0x64a84262),LL(0x020154bb,0x6275da1f),LL(0xd17cf1ab,0x97678caa), LL(0x951a95c3,0x8779795f),LL(0x50fccc08,0xdd35b163),LL(0x33d8f031,0x32709627),LL(0x498dd85c,0x3c5ab10a),
   LL(0x41dca566,0xb6c185c3),LL(0xd8622aa3,0x7de7feda),LL(0x901b6dfb,0x99e84d92),LL(0x7c4ad288,0x30a02b0e), LL(0x2fd3cf36,0xc7c81daa),LL(0xdf89e59f,0xd1319547),LL(0xcd496733,0xb2be8184),LL(0x93d3412b,0xd5f449eb),
   LL(0x25fe531d,0x7ea41b1b),LL(0x6a1d5646,0xf9797432),LL(0x2bde501a,0x86067f72),LL(0x0c85e89c,0xf91481c0), LL(0xf8b05bc6,0xca8ee465),LL(0x02e83cda,0x1844e1cf),LL(0xb4dbe33b,0xca82114a),LL(0x4eabfde2,0x0f9f8769),
   LL(0x38b27fe2,0x4936b1c0),LL(0xaba402df,0x63b6359b),LL(0x656bdbab,0x40c0ea2f),LL(0x6580c39c,0x9c992a89), LL(0x2a60aed1,0x600e8f15),LL(0xe0bf49df,0xeb089ca4),LL(0x2d42d99a,0x9c233d7d),LL(0x4c6bc2fa,0x648d3f95),
   LL(0xe1add3f3,0xdcc383a8),LL(0x4f64a348,0xf42c0c6a),LL(0x0030dbdb,0x2abd176f),LL(0x7d6c215e,0x4de501a3), LL(0x4b9a64bc,0x4a107c1f),LL(0x2496cd59,0xa77f0ad3),LL(0x7688dffb,0xfb78ac62),LL(0x67937d8e,0x7025a2ca),
   LL(0xd1a8f4e7,0xfde8b2d1),LL(0x7354927c,0xf5b3da47),LL(0xd9205735,0xe48606a3),LL(0xe177b917,0xac477cc6), LL(0xa883239a,0xfb1f73d2),LL(0xcc8b8357,0xe12572f6),LL(0xfb1f4f86,0x9d355e9c),LL(0xd9f3ec6e,0x89b795f8),
   LL(0xb54398dc,0x27be56f1),LL(0x3fedeed5,0x1890efd7),LL(0x9c6d0140,0x62f77f1f),LL(0x596f0ee4,0x7ef0e314), LL(0xcc61dab3,0x50ca6631),LL(0xf4866e4f,0x4a39801d),LL(0xae363b39,0x66c8d032),LL(0x2ead66aa,0x22c591e5),
   LL(0xde02a53e,0x954ba308),LL(0xd389f357,0x2a6c060f),LL(0xfbf40b66,0xe6cfcde8),LL(0xc6340ce1,0x8e02fc56), LL(0x73adb4ba,0xe4957795),LL(0xa7b03805,0x7b86122c),LL(0x0c8e6fa6,0x63f83512),LL(0x057d7804,0x83660ea0),
   LL(0x21ba473c,0xbad79105),LL(0xded5389d,0xb6c50bee),LL(0xaa7c9bc0,0xee2caf4d),LL(0x8c4e98a7,0xd97b8de4), LL(0xab3bbddb,0xa9f63e70),LL(0x2597815a,0x3898aabf),LL(0xac15b3d9,0x7659af89),LL(0x703ce784,0xedf7725b),
   LL(0xe085116b,0x25470fab),LL(0x87285310,0x04a43375),LL(0xe2bfd52f,0x4e39187e),LL(0x7d9ebc74,0x36166b44), LL(0xfd4b322c,0x92ad433c),LL(0xba79ab51,0x726aa817),LL(0xc1db15eb,0xf96eacd8),LL(0x0476be63,0xfaf71e91),
   LL(0x641fad98,0xdd69a640),LL(0x29622559,0xb7995918),LL(0xde4199dc,0x03c6daa5),LL(0xad545eb4,0x92cadc97), LL(0x256534e4,0x1028238b),LL(0x8595409a,0x73e80ce6),LL(0xd05dc59b,0x690d4c66),LL(0x981dee80,0xc95f7b8f),
   LL(0xd856ac25,0xf4337014),LL(0xac524dca,0x441bd9dd),LL(0x5f0499f5,0x640b3d85),LL(0xd5fda182,0x39cf84a9), LL(0xb2aa95a0,0x04e7b055),LL(0x0ddf1860,0x29e33f0a),LL(0x423f6b43,0x082e74b5),LL(0x0aaa2b0f,0x217edeb9),
   LL(0x83cbea55,0x58b83f35),LL(0xbc185d70,0xc485ee4d),LL(0x1e5f6992,0x833ff03b),LL(0xcf0c0dd5,0xb5b9b9cc), LL(0x4e9e8a50,0x7caaee8e),LL(0x6269dafd,0x462e907b),LL(0xfbe791c6,0x6ed5cee9),LL(0xed430790,0x68ca3259),
   LL(0x13b5ba88,0x2b72bdf2),LL(0x35ef0ac4,0x60294c8a),LL(0x19b99b08,0x9c3230ed),LL(0x6c2589aa,0x560fff17), LL(0xd6770374,0x552b8487),LL(0x9a56f685,0xa373202d),LL(0x45f175d9,0xd3e7f907),LL(0xd080d810,0x3c2f315f),
   LL(0x7b9520e8,0x1130e9dd),LL(0x0af037b5,0xc078f9e2),LL(0x1e9c104c,0x38cd2ec7),LL(0xc472fe92,0x0f684368), LL(0x6247e7ef,0xd3f1b5ed),LL(0x396dfe21,0xb32d33a9),LL(0x4a9aa2c2,0x46f59cf4),LL(0xff0f7e41,0x69cd5168),
   LL(0x4b3234da,0x3f59da0f),LL(0xb4579ebe,0xcf0b0235),LL(0x6d2476c7,0x6d1cbb25),LL(0x9dc30f08,0x4f0837e6), LL(0x906f6e98,0x9a4075bb),LL(0xc761e7d1,0x253bb434),LL(0x6e73af10,0xde2e645f),LL(0x0c5f131c,0xb89a4060),
   LL(0xb8cc037f,0xd12840c5),LL(0x7405bb47,0x3d093a5b),LL(0x206348b8,0x6202c253),LL(0xc55a3ca7,0xbf5d57fc), LL(0x8c3bef48,0x89f6c90c),LL(0x5a0a960a,0x23ac7623),LL(0x552b42ab,0xdfbd3d6b),LL(0x132061f6,0x3ef22458),
   LL(0xc97e6516,0xd74e9bda),LL(0xc230f49e,0x88779360),LL(0x1e74ea49,0xa6ec1de3),LL(0x3fb645a2,0x581dcee5), LL(0x8f483f14,0xbaef2391),LL(0xd137d13b,0x6d2dddfc),LL(0xd2743a42,0x54cde50e),LL(0xe4d97e67,0x89a34fc5),
   LL(0x12e08ce5,0x13f1f5b3),LL(0xa7f0b2ca,0xa80540b8),LL(0x01982805,0x854bcf77),LL(0x233bea04,0xb8653ffd), LL(0x02b0b4c9,0x8e7b8787),LL(0x9acb170a,0x2675261f),LL(0x930c14e5,0x061a9d90),LL(0xdef0abea,0xb59b30e0),
   LL(0x0200ec7d,0x1dc19ea6),LL(0x0bce132b,0xb6f4a3f9),LL(0xf13e27e0,0xb8d5de90),LL(0x1fade16f,0xbaee5ef0), LL(0xe4c6cf38,0x6f406aaa),LL(0xd1369815,0xab4cfe06),LL(0xefd550c6,0x0dcffe87),LL(0x75ff7d39,0x9d4f59c7),
   LL(0x51deb6ad,0xb02553b1),LL(0xb1877749,0x812399a4),LL(0xca6006e1,0xce90f71f),LL(0xb02b6e77,0xc32363a6), LL(0xdc36c64d,0x02284fbe),LL(0xa7e1ae61,0x86c81e31),LL(0xb909d94a,0x2576c7e5),LL(0x818b2bb0,0x8b6f7d02),
   LL(0x56faa38a,0xeca3ed07),LL(0x9305bb54,0xa3790e6c),LL(0x7bc73061,0xd784eeda),LL(0x6dd50614,0xbd56d369), LL(0x229a8aa9,0xd6575949),LL(0x4595ec28,0xdcca8f47),LL(0x06ab4fe6,0x814305c1),LL(0x24f43f16,0xc8c39768),
   LL(0x523f2b36,0xe2a45f36),LL(0x920d93bb,0x995c6493),LL(0x90f1632b,0xf8afdab7),LL(0x1c295954,0x79ebbecd), LL(0x79592f48,0xc7bb3ddb),LL(0x5f88e998,0x67216a7b),LL(0xbc01193e,0xd91f098b),LL(0xb1db83fc,0xf7d928a5),
   LL(0xe991f600,0x55e38417),LL(0x2981a934,0x2a91113e),LL(0x06b13bde,0xcbc9d648),LL(0x0755ff44,0xb011b6ac), LL(0x045ec613,0x6f4cb518),LL(0xc2f5930a,0x522d2d31),LL(0x382e65de,0x5acae1af),LL(0x27bc966f,0x57643067),
   LL(0x1c7193f0,0x5e12705d),LL(0x3be8858e,0xf0f32f47),LL(0x96c6dfc7,0x785c3d7d),LL(0xbf31795d,0xd75b4a20), LL(0x342659d4,0x91acf17b),LL(0x44f0378f,0xe596ea34),LL(0xce52129d,0x4515708f),LL(0x79f2f585,0x17387e1e),
   LL(0x49dee168,0x72cfd2e9),LL(0x3e2af239,0x1ae05223),LL(0x1d94066a,0x009e75be),LL(0x38abf413,0x6cca31c7), LL(0x9bc49908,0xb50bd61d),LL(0xf5e2bc1e,0x4a9b4a8c),LL(0x946f83ac,0xeb6cc5f7),LL(0xebffab28,0x27da93fc),
   LL(0x4821c8c5,0xea314c96),LL(0xa83c15f4,0x8de49ded),LL(0x7af33004,0x7a64cf20),LL(0xc9627e10,0x45f1bfeb), LL(0x54b9df60,0x878b0626),LL(0xa95c0b33,0x5e4fdc3c),LL(0xc2035d8e,0xe54a37ca),LL(0x80f20b8c,0x9087cda9),
   LL(0x8319ade4,0x36f61c23),LL(0xde8cfdf8,0x766f287a),LL(0x346f3705,0x48821948),LL(0x16e4f4a2,0x49a7b853), LL(0x5cedadfd,0xb9b3f8a7),LL(0x8db2a815,0x8f562815),LL(0x01f68f95,0xc0b7d554),LL(0x688a208e,0x12971e27),
   LL(0xd0ff34fc,0xc9f8b696),LL(0x1222718c,0x20824de2),LL(0x0c95284d,0x7213cf9f),LL(0xdc158240,0xe2ad741b), LL(0x54043ccf,0x0ee3a6df),LL(0xd84412b3,0x16ff479b),LL(0xdfc98af0,0xf6c74ee0),LL(0x52fcd2fb,0xa78a169f),
   LL(0x99c930e9,0xd8ae8746),LL(0x49e117a5,0x1d33e858),LL(0x6624759f,0x7581fcb4),LL(0x5bedc01d,0xde50644f), LL(0xcaf3155e,0xbeec5d00),LL(0xbc73e75f,0x672d66ac),LL(0x270b01db,0x86b9d8c6),LL(0x50f55b79,0xd249ef83),
   LL(0x73978fe3,0x6131d6d4),LL(0x754b00a1,0xcc4e4542),LL(0x57dfcfe9,0x4e05df05),LL(0x51ef6bf0,0x94b29cdd), LL(0x9bc7edf2,0xe4530cff),LL(0xd3da65f3,0x8ac236fd),LL(0xc8eb0b48,0x0faf7d5f),LL(0x660eb039,0x4d2de14c),
   LL(0x60430e54,0xc006bba7),LL(0xda3289ab,0x10a2d0d6),LL(0xd7979c59,0x9c037a5d),LL(0xa116d944,0x04d1f3d3), LL(0x8a0983cd,0x9ff22473),LL(0xc883cabb,0x28e25b38),LL(0x47a58995,0xe968dba5),LL(0x774eebdf,0x2c80b505),
   LL(0x4a953beb,0xee763b71),LL(0x1642e7f6,0x502e223f),LL(0x61d5e722,0x6fe4b641),LL(0xdbef5316,0x9d37c5b0), LL(0xf8330bc7,0x0115ed70),LL(0x75a72789,0x139850e6),LL(0xffceccc2,0x27d7faec),LL(0x4fd9f7f6,0x3016a860),
   LL(0x4cd8f64c,0xc492ec64),LL(0x279d7b51,0x58a2d790),LL(0x1fc75256,0x0ced1fc5),LL(0x8f433017,0x3e658aed), LL(0x05da59eb,0x0b61942e),LL(0x0ddc3722,0xba3d60a3),LL(0x742e7f87,0x7c311cd1),LL(0xf6b01b6e,0x6473ffee),
},
/* digit=28 base_pwr=2^196 */
{
   LL(0x692ac542,0x8303604f),LL(0x227b91d3,0xf079ffe1),LL(0x15aaf9bd,0x19f63e63),LL(0xf1f344fb,0xf99ee565), LL(0xd6219199,0x8a1d661f),LL(0xd48ce41c,0x8c883bc6),LL(0x3c74d904,0x1065118f),LL(0x0faf8b1b,0x713889ee),
   LL(0x81a1b3be,0x972b3f8f),LL(0xce2764a0,0x4f3ce145),LL(0x28c4f5f7,0xe2d0f1cc),LL(0xc7f3985b,0xdeee0c0d), LL(0xd39e25c3,0x7df4adc0),LL(0xc467a080,0x40619820),LL(0x61cf5a58,0x440ebc93),LL(0x422ad600,0x527729a6),
   LL(0xb1b76ba6,0xca6c0937),LL(0x4d2026dc,0x1a2eab85),LL(0x19d9ae0a,0xb1715e15),LL(0xbac4a026,0xf1ad9199), LL(0x07ea7b0e,0x35b3dfb8),LL(0x3ed9eb89,0xedf5496f),LL(0x2d6d08ab,0x8932e5ff),LL(0x25bd2731,0xf314874e),
   LL(0x3f73f449,0xefb26a75),LL(0x8d44fc79,0x1d1c94f8),LL(0x3bc0dc4d,0x49f0fbc5),LL(0x3698a0d0,0xb747ea0b), LL(0x228d291e,0x5218c3fe),LL(0x43c129d6,0x35b804b5),LL(0xd1acc516,0xfac859b8),LL(0x95d6e668,0x6c10697d),
   LL(0x0876fd4e,0xc38e438f),LL(0x83d2f383,0x45f0c307),LL(0xb10934cb,0x203cc2ec),LL(0x2c9d46ee,0x6a8f2439), LL(0x65ccde7b,0xf16b431b),LL(0x27e76a6f,0x41e2cd18),LL(0x4e3484d7,0xb9c8cf8f),LL(0x8315244a,0x64426efd),
   LL(0xfc94dea3,0x1c0a8e44),LL(0xdad6a0b0,0x34c8cdbf),LL(0x04113cef,0x919c3840),LL(0x15490ffa,0xfd32fba4), LL(0x795dcfb7,0x58d190f6),LL(0x83588baf,0xfef01b03),LL(0xca1fc1c0,0x9e6d1d63),LL(0xf0a41ac9,0x53173f96),
   LL(0xba16f73b,0x2b1d402a),LL(0x8cf9b9fc,0x2fb31014),LL(0x446ef7bf,0x2d51e60e),LL(0xb91e1745,0xc731021b), LL(0x4fee99d4,0x9d3b4724),LL(0xfac5c1ea,0x4bca48b6),LL(0xbbea9af7,0x70f5f514),LL(0x974c283a,0x751f55a5),
   LL(0xcb452fdb,0x6e30251a),LL(0x50f30650,0x31ee6965),LL(0x933548d9,0xb0b3e508),LL(0xf4b0ef5b,0xb8949a4f), LL(0x3c88f3bd,0x208b8326),LL(0xdb1d9989,0xab147c30),LL(0x44d4df03,0xed6515fd),LL(0xe72eb0c5,0x17a12f75),
   LL(0x36cf69db,0x3b59796d),LL(0x56670c18,0x1219eee9),LL(0x7a070d8e,0xfe3341f7),LL(0xa327f90c,0x9b70130b), LL(0x0ae18e0e,0x36a32462),LL(0x46c0a638,0x2021a623),LL(0xc62eb0d4,0x251b5817),LL(0x4c762293,0x87bfbcdf),
   LL(0xcdd61d64,0xf78ab505),LL(0xc8c18857,0x8c7a53fc),LL(0x16147515,0xa653ce6f),LL(0xea7d52d5,0x9c923aa5), LL(0x5c18871f,0xc24709cb),LL(0x73b3cc74,0x7d53bec8),LL(0xfdd1d4c4,0x59264aff),LL(0x240da582,0x5555917e),
   LL(0x548f5a0e,0xcae8bbda),LL(0x3bbfbbe1,0x1910eaba),LL(0x7677afc3,0xae579685),LL(0x73ff0b5c,0x49ea61f1), LL(0x4f7c3922,0x78655478),LL(0x20c68eef,0x95d337cd),LL(0xdf779ab9,0x68f1e1e5),LL(0xb5cf69a8,0x14b491b0),
   LL(0x28e3fe89,0x7a6cbbe0),LL(0xc5aac0eb,0xe7e1fee4),LL(0x697e5140,0x7f47eda5),LL(0xb454921f,0x4f450137), LL(0x95cd8185,0xdb625f84),LL(0xcdb2e583,0x74be0ba1),LL(0xdd5e6de4,0xaee4fd7c),LL(0xe8101739,0x4251437d),
   LL(0xac620366,0x686d72a0),LL(0xb6d59344,0x4be3fb9c),LL(0xa1eb75b9,0x6e8b44e7),LL(0x91a5c10c,0x84e39da3), LL(0xb38f0409,0x37cc1490),LL(0x2c2ade82,0x02951943),LL(0x1190a2d8,0x9b688783),LL(0x231182ba,0x25627d14),
   LL(0x658a6d87,0x6eb550aa),LL(0xcf9c7325,0x1405aaa7),LL(0x5c8748c9,0xd147142e),LL(0x53ede0e0,0x7f637e4f), LL(0x14ffad2c,0xf8ca2776),LL(0xbafb6791,0xe58fb1bd),LL(0xbf8f93fc,0x17158c23),LL(0x0a4a4655,0x7f15b373),
   LL(0xd842ca72,0x39d4add2),LL(0x3ed96305,0xa71e4391),LL(0x6700be14,0x5bb09cbe),LL(0xd8befcf6,0x68d69d54), LL(0x37183bcf,0xa45f5367),LL(0x3370dff7,0x7152b7bb),LL(0xbf12525b,0xcf887baa),LL(0xd6d1e3cd,0xe7ac7bdd),
   LL(0x81fdad90,0x25914f78),LL(0x0d2cf6ab,0xcf638f56),LL(0xcc054de5,0xb90bc03f),LL(0x18b06350,0x932811a7), LL(0x9bbd11ff,0x2f00b330),LL(0xb4044974,0x76108a6f),LL(0xa851d266,0x801bb9e0),LL(0xbf8990c1,0x0dd099be),
   LL(0xabe32986,0x58c5aaaa),LL(0x50d59c27,0x0fe9dd2a),LL(0x8d307305,0x84951ff4),LL(0x86529b78,0x6c23f829), LL(0x0b136a79,0x50bb2218),LL(0x77a20996,0x7e2174de),LL(0xc0bb4da6,0x6f00a4b9),LL(0xefdde8da,0x89a25a17),
   LL(0xc11ee01d,0xf728a27e),LL(0xe5f10dfb,0xf900553a),LL(0x02ec893c,0x189a83c8),LL(0x23f66d77,0x3ca5bdc1), LL(0x97eada9f,0x98781537),LL(0x10256230,0x59c50ab3),LL(0x323c69b3,0x346042d9),LL(0x2c460449,0x1b715a6d),
   LL(0x6ae06e0b,0xa41dd476),LL(0x9d42e25f,0xcdd7888e),LL(0x56b25a20,0x0f395f74),LL(0x8700e27e,0xeadfe0ae), LL(0x69950093,0xb09d52a9),LL(0x327f8d40,0x3525d9cb),LL(0x67df886a,0xb8235a94),LL(0x035faec2,0x77e4b0dd),
   LL(0x517d7061,0x115eb20a),LL(0x6c2df683,0x77fe3433),LL(0xcdc6fc67,0x6870ddc7),LL(0x0b87de83,0xb1610588), LL(0xd9c4ddbe,0x343584ca),LL(0x3d754be2,0xb3164f1c),LL(0xc1e6c894,0x0731ed3a),LL(0x4f6b904c,0x26327dec),
   LL(0x97b5cd32,0x9d49c6de),LL(0xb5eceecd,0x40835dae),LL(0xd9ded7fe,0xc66350ed),LL(0x7a678804,0x8aeebb5c), LL(0x5b8ee9ec,0x51d42fb7),LL(0x8e3ca118,0xd7a17bdd),LL(0x2ef4400e,0x40d7511a),LL(0x875a66f4,0xc48990ac),
   LL(0x2199e347,0x8de07d2a),LL(0x2a39e051,0xbee75556),LL(0x916e51dc,0x56918786),LL(0x4a2d89ec,0xeb191313), LL(0x37d341ed,0x6679610d),LL(0x56d51c2b,0x434fbb41),LL(0xd7492dba,0xe54b7ee7),LL(0x59021493,0xaa33a79a),
   LL(0xe4bd6d3d,0x49fc5054),LL(0x5ab551d0,0x09540f04),LL(0x4942d3a6,0x8acc9085),LL(0x2d28323b,0x231af02f), LL(0x0992c163,0x93458cac),LL(0x888e3bb4,0x1fef8e71),LL(0xbe8c268c,0x27578da5),LL(0xe805ec00,0xcc8be792),
   LL(0xc61c3855,0x29267bae),LL(0x58c1fd3b,0xebff429d),LL(0x8c0b93b8,0x22d886c0),LL(0x2ddb8953,0xca5e00b2), LL(0xc3fed8b7,0xcf330117),LL(0x819c01f6,0xd49ac6fa),LL(0x3c0fbd54,0x6ddaa6bd),LL(0x8049a2cf,0x91743068),
   LL(0xaff2ef81,0xd67f981e),LL(0x2818ae80,0xc3654d35),LL(0x1b2aa892,0x81d05044),LL(0x3d099328,0x2db067bf), LL(0x703dcc97,0xe7c79e86),LL(0xe133e215,0xe66f9b37),LL(0xe39a7a5c,0xcdf119a6),LL(0x876f1b61,0x47c60de3),
   LL(0xd860f1b2,0x6e405939),LL(0xf5ed4d4a,0x3e9a1dbc),LL(0xc9b6bcbd,0x3f23619e),LL(0x734e4497,0x5ee790cf), LL(0x5bdaf9bb,0xf0a834b1),LL(0x4ca295f0,0x02cedda7),LL(0xcb8e378c,0x4619aa2b),LL(0xcc987ea4,0xe5613244),
   LL(0x76b23a50,0x0bc022cc),LL(0x0a6c21ce,0x4a2793ad),LL(0x89cac3f5,0x38328780),LL(0xcba26d56,0x29176f1b), LL(0x4f6f59eb,0x06296187),LL(0x8bdc658e,0x86e9bca9),LL(0x57e30402,0x2ca9c4d3),LL(0x516a09bb,0x5438b216),
   LL(0x7672765a,0x0a6a063c),LL(0x0547b9bf,0x37a3ce64),LL(0x98b1a633,0x42c099c8),LL(0x05ee6961,0xb5ab800d), LL(0x11a5acd6,0xf1963f59),LL(0x46201063,0xbaee6157),LL(0xa596210a,0x36d9a649),LL(0x1ba7138c,0xaed04363),
   LL(0xa4a82b76,0xcf817d1c),LL(0xf3806be9,0x5586960e),LL(0x09dc6bb5,0x7ab67c89),LL(0x114fe7eb,0x52ace7a0), LL(0xcbbc9b70,0xcd987618),LL(0x604ca5e1,0x4f06fd5a),LL(0x6dbde133,0x90af14ca),LL(0x948a3264,0x1afe4322),
   LL(0xc44b2c6c,0xa70d2ca6),LL(0x0ef87dfe,0xab726799),LL(0x2e696377,0x310f64dc),LL(0x4c8126a0,0x49b42e68), LL(0xcea0b176,0x0ea444c3),LL(0xcb269182,0x53a8ddf7),LL(0xbbba9dcb,0xf3e674eb),LL(0xd8669d33,0x0d2878a8),
   LL(0xd019b6a3,0x04b935d5),LL(0x406f1e46,0xbb5cf88e),LL(0x5b57c111,0xa1912d16),LL(0x19ebfd78,0x9803fc21), LL(0xc07764a9,0x4f231c9e),LL(0xb75bd055,0xd93286ee),LL(0x8ee6c9de,0x83a9457d),LL(0x6087ec90,0x04695915),
   LL(0x58d6cd46,0x14c6dd8a),LL(0x8e6634d2,0x9cb633b5),LL(0xf81bc328,0xc1305047),LL(0x26a177e5,0x12ede0e2), LL(0x065a6f4f,0x332cca62),LL(0x67be487b,0xc3a47ecd),LL(0x0f47ed1c,0x741eb187),LL(0xe7598b14,0x99e66e58),
   LL(0x63d0ff12,0x6f0544ca),LL(0xb610a05f,0xe5efc784),LL(0x7cad7b47,0xf72917b1),LL(0xf2cac0c0,0x3ff6ea20), LL(0xf21db8b7,0xcc23791b),LL(0xd7d93565,0x7dac70b1),LL(0x694bdaad,0x682cda1d),LL(0x1023516d,0xeb88bb8c),
   LL(0xdfdbeb1b,0xc4c634b4),LL(0xb4ee4dea,0x22f5ca72),LL(0xe6524821,0x1045a368),LL(0x052b18b2,0xed9e8a3f), LL(0xb961f49a,0x9b7f2cb1),LL(0x7b009670,0x7fee2ec1),LL(0x22507a6d,0x350d8754),LL(0x4db55f1d,0x561bd711),
   LL(0x320bbcaf,0x4c189ccc),LL(0xdf1de48c,0x568434cf),LL(0x0fa8f128,0x6af1b00e),LL(0x8907583c,0xf0ba9d02), LL(0x32ff9f60,0x735a4004),LL(0xc25dcf33,0x3dd8e4b6),LL(0x42c74cef,0xf2230f16),LL(0x013fa8ad,0xd8117623),
   LL(0xf51fe76e,0x36822876),LL(0x11d62589,0x8a6811cc),LL(0x46225718,0xc3fc7e65),LL(0xc82fdbcd,0xb7df2c9f), LL(0xdd7b205b,0x3b1d4e52),LL(0x47a2e414,0xb6959478),LL(0xefa91148,0x05e4d793),LL(0xfd2e9675,0xb47ed446),
   LL(0x04c9d9bf,0x1a7098b9),LL(0x1b793048,0x661e2881),LL(0xb01ee461,0xb1a16966),LL(0x2954746f,0xbc521308), LL(0x2477de50,0xc909a0fc),LL(0x7dbd51ef,0xd80bb41c),LL(0x53294905,0xa85be7ec),LL(0x83958f97,0x6d465b18),
   LL(0xfb6840fd,0x16f6f330),LL(0x3401e6c8,0xfaaeb214),LL(0xccb5b4f8,0xaf83d30f),LL(0x266dec4b,0x22885739), LL(0x7bc467df,0x51b4367c),LL(0xd842d27a,0x926562e3),LL(0x0fea14a6,0xdfcb6614),LL(0xf2734cd9,0xeb394dae),
   LL(0x11c0be98,0x3eeae5d2),LL(0x814e8165,0xb1e6ed11),LL(0xe52bce1c,0x191086bc),LL(0xa75a04da,0x14b74cc6), LL(0x8c060985,0x63cf1186),LL(0x2dbd7f7c,0x071047de),LL(0xce0942ca,0x4e433b8b),LL(0xd8fec61d,0xecbac447),
   LL(0xebf3232f,0x8f0ed0e2),LL(0xc52a2edd,0xfff80f9e),LL(0x75b55fdb,0xad9ab433),LL(0xe42e0c11,0x73ca7820), LL(0xe6251b46,0x6dace0a0),LL(0x4c0d932d,0x89bc6b5c),LL(0x095da19a,0x3438cd77),LL(0x8d48bdfb,0x2f24a939),
   LL(0x766561b7,0x99b47e46),LL(0x0ed0322a,0x736600e6),LL(0x638e1865,0x06a47cb1),LL(0xcb136000,0x927c1c2d), LL(0x0cc5df69,0x29542337),LL(0x09d649a9,0x99b37c02),LL(0x6aefdb27,0xc5f0043c),LL(0x1be95c27,0x6cdd9987),
   LL(0x390420d2,0x69850931),LL(0x0983efa4,0x299c40ac),LL(0xaf39aead,0x3a05e778),LL(0x43a45193,0x84274408), LL(0x91a711a0,0x6bcd0fb9),LL(0x9f52ab17,0x461592c8),LL(0xda3c6ed6,0xb49302b4),LL(0x330d7067,0xc51fddc7),
   LL(0xda50d531,0x94babeb6),LL(0xa6a7b9da,0x521b840d),LL(0x404bdc89,0x5305151e),LL(0xd0d07449,0x1bcde201), LL(0x3b76a59a,0xf427a78b),LL(0x07791a1b,0xf84841ce),LL(0xbf91ed1c,0xebd314be),LL(0xbf172943,0x8e61d34c),
   LL(0x5541b892,0x1d5dc451),LL(0xfc9d9e54,0xb186ee41),LL(0xd5bf610d,0x9d9f345e),LL(0xf6acca9f,0x3e7ba65d), LL(0xa8369486,0x9dda787a),LL(0x8eb5ba53,0x09f9dab7),LL(0xd6481bc3,0x5afb2033),LL(0xafa62104,0x76f4ce30),
   LL(0xf4f066b5,0xa8fa00cf),LL(0x461dafc2,0x89ab5143),LL(0xa3389998,0x44339ed7),LL(0xbc214903,0x2ff862f1), LL(0xb05556e3,0x2c88f985),LL(0x3467081e,0xcd96058e),LL(0xedc637ea,0x7d6a4176),LL(0x36a5acdc,0xe1743d09),
   LL(0x7eb37726,0x66fd72e2),LL(0x1481a037,0xf7fa264e),LL(0x45f4aa79,0x9fbd3bde),LL(0x767c3e22,0xed1e0147), LL(0x82e7abe2,0x7621f979),LL(0x45f633f8,0x19eedc72),LL(0x6137bf3a,0xe69b155e),LL(0x414ee94e,0xa0ad13ce),
   LL(0x1c0e651a,0x93e3d524),LL(0x02ce227e,0xab1a6e2a),LL(0x4ab27eca,0xe7af1797),LL(0xbd444f39,0x245446de), LL(0x56c07613,0x59e22a21),LL(0xf4275498,0x43deafce),LL(0x67fd0946,0x10834ccb),LL(0x47406edf,0xa75841e5),
   LL(0x7b0ac93d,0xebd6a677),LL(0x78f5e0d7,0xa6e37b0d),LL(0x76f5492b,0x2516c096),LL(0x9ac05f3a,0x1e4bf888), LL(0x4df0ba2b,0xcdb42ce0),LL(0x5062341b,0x935d5cfd),LL(0x82acac20,0x8a303333),LL(0x5198b00e,0x429438c4),
   LL(0x049d33fa,0x1d083bc9),LL(0x946f67ff,0x58b82dda),LL(0x67a1d6a3,0xac3e2db8),LL(0x1798aac8,0x62e6bead), LL(0xde46c58c,0xfc85980f),LL(0x69c8d7be,0xa7f69379),LL(0x837b35ec,0x23557927),LL(0xe0790c0c,0x06a933d8),
   LL(0x077ff55d,0x827c0e9b),LL(0xbb26e680,0x53977798),LL(0x1d9cb54f,0x59530874),LL(0x4aac53ef,0xcca3f449), LL(0xa07eda0f,0x11dc5c87),LL(0xfd6400c8,0xc138bccf),LL(0x13e5da72,0x549680d3),LL(0x4540617e,0xc93eed82),
   LL(0x4d0b75c0,0xfd3db157),LL(0x6386075b,0x9716eb42),LL(0x817b2c16,0x0639605c),LL(0xf1e4f201,0x09915109), LL(0x5cca6c3b,0x35c9a928),LL(0x3505c900,0xb25f7d1a),LL(0x630480c4,0xeb9f7d20),LL(0x2a1a501c,0xc3c7b8c6),
   LL(0x5a1f8e24,0x3f99183c),LL(0x9dd255f0,0xfdb118fa),LL(0xc27f62a6,0xb9b18b90),LL(0x396ec191,0xe8f732f7), LL(0x0be786ab,0x524a2d91),LL(0x0ac5a0f5,0x5d32adef),LL(0x9725f694,0x9b53d4d6),LL(0x0510ba89,0x032a76c6),
   LL(0xebeb1544,0x840391a3),LL(0x3ed73ac3,0x44b7b88c),LL(0x256cb8b3,0xd24bae7a),LL(0xe394cb12,0x7ceb151a), LL(0x5bc1e6a8,0xbd6b66d0),LL(0x090f07bf,0xec70cecb),LL(0x7d937589,0x270644ed),LL(0x5f1dccfe,0xee9e1a3d),
   LL(0x745b98d2,0xb0d40a84),LL(0x2556ed40,0xda429a21),LL(0x85148cb9,0xf676eced),LL(0xded18936,0x5a22d40c), LL(0x70e8a4ce,0x3bc4b9e5),LL(0x9eae0379,0xbfd1445b),LL(0x1a0bd47e,0xf23f2c0c),LL(0xe1845531,0xa9c0bb31),
   LL(0x0a4c3f6b,0x9ddc4d60),LL(0x2c15ef44,0xbdfaad79),LL(0x7f484acc,0xce55a236),LL(0x055b1f15,0x08653ca7), LL(0x538873a3,0x2efa8724),LL(0xace1c7e7,0x09299e5d),LL(0xade332ba,0x07afab66),LL(0x92dd71b7,0x9be1fdf6),
   LL(0x5758b11c,0xa49b5d59),LL(0xc8654f40,0x0b852893),LL(0x52379447,0xb63ef6f4),LL(0x105e690c,0xd4957d29), LL(0x646559b0,0x7d484363),LL(0x49788a8e,0xf4a8273c),LL(0x34ce54a9,0xee406cb8),LL(0xf86fda9b,0x1e1c260f),
   LL(0xcf6a4a81,0xe150e228),LL(0x1b488772,0x1fa3b6a3),LL(0xc5a9c15b,0x1e6ff110),LL(0x8ad6aa47,0xc6133b91), LL(0x9dffa978,0x8ac5d55c),LL(0x5f3965f2,0xba1d1c1d),LL(0x7732b52f,0xf969f4e0),LL(0xa5172a07,0xfceecdb5),
   LL(0x10f2b8f5,0xb0120a5f),LL(0x5c4c2f63,0xc83a6cdf),LL(0xf8f9c213,0x4d47a491),LL(0xd3f1bbd5,0xd9e1cce5), LL(0xaba7e372,0x0d91bc7c),LL(0xdfd1a2db,0xfcdc74c8),LL(0x374618e5,0x05efa800),LL(0x15a7925e,0x11216969),
   LL(0xf6021c5d,0xd4c89823),LL(0xeff14423,0x880d5e84),LL(0x6dcd1396,0x6523bc5a),LL(0x113c978b,0xd1acfdfc), LL(0xbbb66840,0xb0c164e8),LL(0x72b58459,0xf7f4301e),LL(0xa638e8ec,0xc29ad4a6),LL(0x46b78699,0xf5ab8961),
   LL(0x0e954750,0x9dbd7974),LL(0x64f9d2c6,0x0121de88),LL(0xd985232e,0x2e597b42),LL(0x53451777,0x55b6c3c5), LL(0x519cb9fb,0xbb53e547),LL(0x8428600d,0xf134019f),LL(0xe081791a,0x5a473176),LL(0x35fb0c08,0x2f3e2263),
   LL(0x73d273b0,0xb28c3017),LL(0x7721ef9a,0xccd21076),LL(0xb650dc39,0x054cc292),LL(0x6188045e,0x662246de), LL(0x6b83c0d1,0x904b52fa),LL(0x97e9cd46,0xa72df267),LL(0x899725e4,0x886b43cd),LL(0xd849ff22,0x2b651688),
   LL(0x02f34533,0x60479b79),LL(0x0c77c148,0x5e354c14),LL(0xa8537c78,0xb4bb7581),LL(0xefe1495f,0x188043d7), LL(0x8c1d5026,0x9ba12f42),LL(0x93d4aaab,0x2e0c8a26),LL(0xaa57c450,0xbdba7b8b),LL(0x9bbdafef,0x140c9ad6),
   LL(0x25ac0f18,0x2067aa42),LL(0x04d1fbf3,0xf7b1295b),LL(0xa4b04824,0x14829111),LL(0x33bd5e91,0x2ce3f192), LL(0x8f2e1b72,0x9c7a1d55),LL(0x302aa243,0xfe932286),LL(0xd4be9554,0x497ca7b4),LL(0xe0547a6e,0xb8e821b8),
   LL(0x67e573e0,0xfb2838be),LL(0x4084c44b,0x05891db9),LL(0x96c1c2c5,0x91311373),LL(0xd958444b,0x6aebfa3f), LL(0xe56e55c1,0xac9cdce9),LL(0x2caa46d0,0x7148ced3),LL(0xb61fe8eb,0x2e10c7ef),LL(0xff97cf4d,0x9fd835da),
},
/* digit=29 base_pwr=2^203 */
{
   LL(0x081e9387,0xa36da109),LL(0x8c935828,0xfb9780d7),LL(0xe540b015,0xd5940332),LL(0xe0f466fa,0xc9d7b51b), LL(0xd6d9f671,0xfaadcd41),LL(0xb1a2ac17,0xba6c1e28),LL(0xed201e5f,0x066a7833),LL(0xf90f462b,0x19d99719),
   LL(0x060b5f61,0xf431f462),LL(0x7bd057c2,0xa56f46b4),LL(0x47e1bf65,0x348dca6c),LL(0x41bcf1ff,0x9a38783e), LL(0xda710718,0x7a5d33a9),LL(0x2e0aeaf6,0x5a779987),LL(0x2d29d187,0xca87314d),LL(0xc687d733,0xfa0edc3e),
   LL(0x6a31e09b,0x9df33621),LL(0xc1350e35,0xde89e44d),LL(0x4ca0cf52,0x29214871),LL(0x0b88a538,0xdf379672), LL(0x2591d61b,0xc92a510a),LL(0x585b447b,0x79aa87d7),LL(0xe5287f77,0xf67db604),LL(0x5efe7a80,0x1697c8bf),
   LL(0xcb198ac7,0x1c894849),LL(0x0f264665,0xa884a93d),LL(0x9b200678,0x2da964ef),LL(0x009834e6,0x3c351b87), LL(0xe2c4b44b,0xafb2ef9f),LL(0x3326790c,0x580f6c47),LL(0x0b02264a,0xb8480521),LL(0x42a194e2,0x8ba6f9e2),
   LL(0x8fb54738,0xfc87975f),LL(0x27c3ead3,0x35160788),LL(0xb74a085a,0x834116d2),LL(0xa62fe996,0x53c99a73), LL(0x5b81c51b,0x87585be0),LL(0xbe0852b7,0x925bafa8),LL(0xa84d19a7,0x76a4fafd),LL(0x585206d4,0x39a45982),
   LL(0x5eb03c0e,0x499b6ab6),LL(0x72bc3fde,0xf19b7954),LL(0x6e3a80d2,0xa86b5b9c),LL(0x6d42819f,0xe4377508), LL(0xbb3ee8a3,0xc1663650),LL(0xb132075f,0x75eb14fc),LL(0x7ad834f6,0xa8ccc906),LL(0xe6e92ffd,0xea6a2474),
   LL(0x0f8d6758,0x9d72fd95),LL(0x408c07dd,0xcb84e101),LL(0xa5e23221,0xb9114bfd),LL(0xe94e742c,0x358b5fe2), LL(0x95f40e75,0x1c0577ec),LL(0x3d73f3d6,0xf0155451),LL(0xbd1b9b66,0x9d55cd67),LL(0xaf8d63c7,0x63e86e78),
   LL(0xd3c095f1,0x39d934ab),LL(0xe4b76d71,0x04b261be),LL(0xe73e6984,0x1d2e6970),LL(0x5e5fcb11,0x879fb23b), LL(0xdfd75490,0x11506c72),LL(0x61bcf1c1,0x3a97d085),LL(0xbf5e7007,0x43201d82),LL(0x798232a7,0x7f0ac52f),
   LL(0x6eb564d4,0x2715cbc4),LL(0x9e570e29,0x8d6c752c),LL(0x9ef5fd5d,0xf80247c8),LL(0xd53eb514,0xc3c66b46), LL(0x0f87de56,0x9666b401),LL(0xc6c603b5,0xce62c06f),LL(0x7e4fc942,0xae7b4c60),LL(0x663a9c19,0x38ac0b77),
   LL(0x4b049136,0xcb4d20ee),LL(0x356a4613,0x8b63bf12),LL(0x70e08128,0x1221aef6),LL(0x4acb6b16,0xe62d8c51), LL(0x379e7896,0x71f64a67),LL(0xcafd7fa5,0xb25237a2),LL(0x3841ba6a,0xf077bd98),LL(0x3cd16e7e,0xc4ac0244),
   LL(0x21fea4ca,0x548ba869),LL(0xf3dfdac1,0xd36d0817),LL(0xf4685faf,0x09d8d71f),LL(0xc52c459a,0x8eff66be), LL(0x0b57235e,0x182faee7),LL(0x0106712b,0xee3c39b1),LL(0xc0fcdcb0,0x5107331f),LL(0xa51054ba,0x669fb9dc),
   LL(0x319d7682,0xb25101fb),LL(0x0a982fee,0xb0293129),LL(0x0261b344,0x51c1c9b9),LL(0xbfd371fa,0x0e008c5b), LL(0x0278ca33,0xd866dd1c),LL(0xe5aa53b1,0x666f76a6),LL(0x6013a2cf,0xe5cfb779),LL(0xa3521836,0x1d3a1aad),
   LL(0x73faa485,0xcedd2531),LL(0xc0a76878,0xc8ee6c4f),LL(0x2a11667d,0xddbccfc9),LL(0x1c2f695a,0x1a418ea9), LL(0x51f73971,0xdb11bd92),LL(0xda2ed89f,0x3e4b3c82),LL(0xe73e0319,0x9a44f3f4),LL(0x303431af,0xd1e3de0f),
   LL(0x50f75f9c,0x3c5604ff),LL(0x7e752b22,0x1d8eddf3),LL(0x3c9a1118,0x0ef074dd),LL(0xccb86d7b,0xd0ffc172), LL(0x037d90f2,0xabd1ece3),LL(0x6055856c,0xe3f307d6),LL(0x7e4c6daf,0x422f9328),LL(0x334879a0,0x902aac66),
   LL(0x94cdfade,0xb6a1e7bf),LL(0x7fc6d634,0x6c97e1ed),LL(0xa2fb63f8,0x662ad24d),LL(0xa5928405,0xf81be1b9), LL(0xd14b4206,0x86d765e4),LL(0x8fa0db65,0xbecc2e0e),LL(0xb17fc76c,0xa28838e0),LL(0xe37cf24e,0xe49a602a),
   LL(0x567193ec,0x76b4131a),LL(0xe5f6e70b,0xaf3c305a),LL(0x031eebdd,0x9587bd39),LL(0x71bbe831,0x5709def8), LL(0x0eb2b669,0x57059983),LL(0x875b7029,0x4d80ce1b),LL(0x0364ac16,0x838a7da8),LL(0xbe1c83ab,0x2f431d23),
   LL(0xf9294dd3,0xe56812a6),LL(0x9b4b0d77,0xb448d01f),LL(0x04e8305c,0xf3ae6061),LL(0x94d8c63e,0x2bead645), LL(0x84fd8b07,0x0a85434d),LL(0xf7a9dee5,0x537b983f),LL(0xef55bd85,0xedcc5f18),LL(0x21c6cf8b,0x2041af62),
   LL(0xb940c71e,0x8e52874c),LL(0xdb5f4b3a,0x211935a9),LL(0x301b1dc3,0x94350492),LL(0x29958620,0x33d2646d), LL(0xef911404,0x16b0d64b),LL(0x9a3c5ef4,0x9d1f25ea),LL(0x4a352c78,0x20f200eb),LL(0x4bd0b428,0x43929f2c),
   LL(0xc7196e29,0xa5656667),LL(0x9391be48,0x7992c2f0),LL(0x9ee0cd6e,0xaaa97cbd),LL(0x3dc8c9bf,0x51b0310c), LL(0xdd9f22cb,0x237f8acf),LL(0xb585d584,0xbb1d81a1),LL(0x8c416388,0x8d5d85f5),LL(0x42fe474f,0x0d6e5a5a),
   LL(0x38235d4e,0xe7812766),LL(0x496e3298,0x1c62bd67),LL(0x3f175bc8,0x8378660c),LL(0x17afdd4d,0x4d04e189), LL(0x85a8068c,0x32a81601),LL(0x92b29a85,0xdb58e4e1),LL(0xc70d8a3b,0xe8a65b86),LL(0x98a0403b,0x5f0e6f4e),
   LL(0x69ed2370,0x08129684),LL(0x0871ee26,0x34dc30bd),LL(0x7c9c5b05,0x3a5ce948),LL(0x43a90c87,0x7d487b80), LL(0xdd0e7179,0x4089ba37),LL(0xb4041811,0x45f80191),LL(0x98747ba5,0x1c3e1058),LL(0x6e1ae592,0x98c4e13a),
   LL(0xe82c9f9e,0xd44636e6),LL(0xc33a1043,0x711db87c),LL(0xaa8aec05,0x6f431263),LL(0x2744a4aa,0x43ff120d), LL(0xae77779b,0xd3bd892f),LL(0x8cdc9f82,0xf0fe0cc9),LL(0xf1c5b1bc,0xca5f7fe6),LL(0x44929a72,0xcc63a682),
   LL(0x09dbe19a,0xc7eaba0c),LL(0x6b5c73c2,0x2f3585ad),LL(0x0ae50c30,0x8ab8924b),LL(0x638b30ba,0x17fcd27a), LL(0x10b3d5a5,0xaf414d34),LL(0x2a9accf1,0x09c107d2),LL(0x946a6242,0x15dac49f),LL(0xd707d642,0xaec3df2a),
   LL(0x3f894ae0,0x2c2492b7),LL(0xb75f18ce,0xf59df3e5),LL(0x8f53cad0,0x7cb740d2),LL(0xc4f01294,0x3eb585fb), LL(0x32c7f717,0x17da0c86),LL(0xaf943f4c,0xeb8c795b),LL(0xf67c51d2,0x4ee23fb5),LL(0x68889949,0xef187575),
   LL(0x0389168b,0xa6b4bdb2),LL(0xea577d03,0xc4ecd258),LL(0x55743082,0x3a63782b),LL(0xc72f08cd,0x6f678f4c), LL(0x65e58dd8,0x553511cf),LL(0xd402c0cd,0xd53b4e3e),LL(0xa037c14c,0x37de3e29),LL(0xc05712aa,0x86b6c516),
   LL(0xb38dff6f,0x2834da3e),LL(0xea636be8,0xbe012c52),LL(0x61dd37f8,0x292d238c),LL(0x8f8142db,0x0e54523f), LL(0x036a05d8,0xe31eb436),LL(0x1e93c0ff,0x83e3cdff),LL(0x50821ddf,0x3fd2fe0f),LL(0xff9eb33b,0xc8e19b0d),
   LL(0xb569a5fe,0xc8cc943f),LL(0xd4342d75,0xad0090d4),LL(0xcaeca000,0x82090b4b),LL(0x1bd410eb,0xca39687f), LL(0x65959d77,0xe7bb0df7),LL(0x9c964999,0x39d78218),LL(0xb2415451,0xd87f62e8),LL(0xbed76108,0xe5efb774),
   LL(0xe822f0d0,0x3ea011a4),LL(0x5a8704f8,0xbc647ad1),LL(0x50c6820f,0xbb315b35),LL(0xb7e76bec,0x863dec3d), LL(0xf017bfc7,0x01ff5d3a),LL(0x976b8229,0x20054439),LL(0x0bbd0d3b,0x067fca37),LL(0x7f5e3d0f,0xf63dde64),
   LL(0x2a4c94e9,0x22dbefb3),LL(0x96f8278a,0xafbff0fe),LL(0x3503793d,0x80aea0b1),LL(0x5f06cd29,0xb2238029), LL(0x8ec3feca,0x65703e57),LL(0x393e7053,0x06c38314),LL(0x7c6734c4,0xa0b751eb),LL(0xc59f0f1e,0xd2e8a435),
   LL(0x5e9ca895,0x147d9052),LL(0x972072df,0x2f4dd31e),LL(0xe6c6755c,0xa16fda8e),LL(0xcf196558,0xc66826ff), LL(0x0cf43895,0x1f1a76a3),LL(0x83c3097b,0xa9d604e0),LL(0x66390e0e,0xe1908309),LL(0xb3c85eff,0xa50bf753),
   LL(0xf6a70251,0x0696bdde),LL(0x3c6ab16a,0x548b801b),LL(0xa4d08762,0x37fcf704),LL(0xdff76c4e,0x090b3def), LL(0x69cb9158,0x87e8cb89),LL(0x995ece43,0x44a90744),LL(0x0ad9fbf5,0xf85395f4),LL(0x4fb0c82d,0x49b0f6c5),
   LL(0xadf7cccf,0x75d9bc15),LL(0xdfa1e1b0,0x81a3e5d6),LL(0x249bc17e,0x8c39e444),LL(0x8ea7fd43,0xf37dccb2), LL(0x907fba12,0xda654873),LL(0x4a372904,0x35daa6da),LL(0x6283a6c5,0x0564cfc6),LL(0x4a9395bf,0xd09fa4f6),
   LL(0xaeb19a36,0x688e9ec9),LL(0xc7bfbfb4,0xd913f1ce),LL(0x61c2faa6,0x797b9a3c),LL(0x6a0a9c12,0x2f979bec), LL(0x359679ec,0xb5969d0f),LL(0x079b0460,0xebcf523d),LL(0x10fab870,0xfd6b0008),LL(0x9373a39c,0x3f2edcda),
   LL(0x6f568431,0x0d64f9a7),LL(0x02f8898c,0xf848c27c),LL(0x260b5bd5,0xf418ade1),LL(0x6973dee8,0xc1f3e323), LL(0x26c185dd,0x46e9319c),LL(0x546f0ac4,0x6d85b7d8),LL(0x247f9d57,0x427965f2),LL(0xb0035f48,0xb519b636),
   LL(0xab87d59c,0x6b6163a9),LL(0x39caaa11,0xff9f58c3),LL(0x3177387b,0x4ac39cde),LL(0x873e77f9,0x5f6557c2), LL(0x36a83041,0x67504006),LL(0x75ef196c,0x9b1c96ca),LL(0xb08c7940,0xf34283de),LL(0x1128c316,0x7ea09644),
   LL(0x6aa39dff,0xb510b3b5),LL(0x9f8e4d8c,0x59b43da2),LL(0x9e4c4b9f,0xa8ce31fd),LL(0xc1303c01,0x0e20be26), LL(0xe8ee47c9,0x18187182),LL(0x7db98101,0xd9687cdb),LL(0xa1e14ff6,0x7a520e4d),LL(0x8836d572,0x429808ba),
   LL(0x4944b663,0xa37ca60d),LL(0xa3f91ae5,0xf901f7a9),LL(0x9e36e3b1,0xe4e3e76e),LL(0x29d93250,0x9aa219cf), LL(0x056a2512,0x347fe275),LL(0xde65d95c,0xa4d643d9),LL(0x699fc3ed,0x9669d396),LL(0xcf8c6bbe,0xb598dee2),
   LL(0xdda9e5c6,0x682ac1e5),LL(0xcaa9fc95,0x4e0d3c72),LL(0x772bea44,0x17faaade),LL(0xab0009c8,0x5ef8428c), LL(0x460ff016,0xcc4ce47a),LL(0x725281cb,0xda6d12bf),LL(0x0223aad2,0x44c67848),LL(0x36256e28,0x6e342afa),
   LL(0x93a37c04,0x1400bb0b),LL(0xdd10bd96,0x62b1bc9b),LL(0x0dac46b7,0x7251adeb),LL(0x7be4ef51,0x7d33b92e), LL(0xe61fa29a,0x28b2a94b),LL(0x06422233,0x4b2be13f),LL(0x330d8d37,0x36d6d062),LL(0xb28ca005,0x5ef80e1e),
   LL(0x6d16768e,0x174d4699),LL(0x628bf217,0x9fc4ff6a),LL(0x154e490d,0x77705a94),LL(0x8d2d997a,0x9d96dd28), LL(0xce5d72c4,0x77e2d9d8),LL(0xc11c714f,0x9d06c5a4),LL(0x79e4a03e,0x02aa5136),LL(0x030ff28b,0x1386b3c2),
   LL(0xfb283f61,0xfe82e8a6),LL(0xf3abc3fb,0x7df203e5),LL(0x3a4d3622,0xeec7c351),LL(0xdf762761,0xf7d17dbf), LL(0x522055f0,0xc3956e44),LL(0x8fa748db,0xde3012db),LL(0xbf1dcc14,0xca9fcb63),LL(0xbe4e2f3a,0xa56d9dcf),
   LL(0x8bcec9c2,0xb86186b6),LL(0x680b9f06,0x7cf24df9),LL(0xc0d29281,0xc46b45ea),LL(0x07b10e12,0xfff42bc5), LL(0x4d289427,0x12263c40),LL(0xb4848ec4,0x3d5f1899),LL(0xd040800c,0x11f97010),LL(0x300feb20,0xb4c5f529),
   LL(0xde94fdcb,0xcc543f8f),LL(0xc7c2f05e,0xe96af739),LL(0x882692e1,0xaa5e0036),LL(0x950d4ae9,0x09c75b68), LL(0xb5932a7a,0x62f63df2),LL(0xde0979ad,0x2658252e),LL(0xb5e69631,0x2a19343f),LL(0x525b666b,0x718c7501),
   LL(0xea40dc3a,0x26a42d69),LL(0xaecc018f,0xdc84ad22),LL(0x3270f04a,0x25c36c7b),LL(0x50fa72ed,0x46ba6d47), LL(0x93e58a8e,0x6c37d1c5),LL(0x120c088c,0xa2394731),LL(0xcb6e86da,0xc3be4263),LL(0x7126d038,0x2c417d36),
   LL(0x8b6f8efa,0x5b70f9c5),LL(0x37718536,0x671a2faa),LL(0xb539c92b,0xd3ced3c6),LL(0xa31203c2,0xe56f1bd9), LL(0x9ff3c8eb,0x8b096ec4),LL(0x43491cea,0x2deae432),LL(0x17943794,0x2465c6eb),LL(0x20586843,0x5d267e66),
   LL(0xb07159d0,0x9d3d116d),LL(0xc1896210,0xae07a67f),LL(0xbb961579,0x8fc84d87),LL(0x1c1f8dd6,0x30009e49), LL(0xe3132819,0x8a8caf22),LL(0xf23ab4ff,0xcffa197c),LL(0x205dd687,0x58103a44),LL(0x0ded67a2,0x57b796c3),
   LL(0xa1779ad7,0x0b9c3a6c),LL(0x357c09c5,0xa33cfe2e),LL(0x3db4a57e,0x2ea29315),LL(0x8ebeb52e,0x91959695), LL(0xe546c879,0x118db9a6),LL(0x6295c8d6,0x8e996df4),LL(0x55ec806b,0xdd990484),LL(0x165c1035,0x24f291ca),
   LL(0x440e2229,0xcca523bb),LL(0x73ef4d04,0x324673a2),LL(0x3e11ec39,0xaf3adf34),LL(0xdc5968d3,0x6136d7f1), LL(0xb053a927,0x7a7b2899),LL(0xae067ecd,0x3eaa2661),LL(0x02779cd9,0x8549b9c8),LL(0xc53385ea,0x061d7940),
   LL(0xf06d18bd,0x3e0ba883),LL(0xb2700843,0x4ba6de53),LL(0x591a9e4d,0xb966b668),LL(0x7f4fa0ed,0x93f67567), LL(0x4347237b,0x5a02711b),LL(0xe794608e,0xbc041e2f),LL(0x70f73d8c,0x55af10f5),LL(0xbb7564f7,0xd2d4d4f7),
   LL(0xb3e93ce7,0xd7d27a89),LL(0x5d3a2c1b,0xf7b5a875),LL(0x255b218a,0xb29e68a0),LL(0x8af76754,0xb533837e), LL(0x579fab2e,0xd1b05a73),LL(0xecd74385,0xb41055a1),LL(0x445e9115,0xb2369274),LL(0xf520274e,0x2972a7c4),
   LL(0xf678e68a,0x6c08334e),LL(0x99b057ed,0x4e4160f0),LL(0x52ccb69a,0x3cfe11b8),LL(0x21c8f772,0x2fd1823a), LL(0x3298f055,0xdf7f072f),LL(0xfec74a6e,0x8c0566f9),LL(0x5bb4d041,0xe549e019),LL(0x9208d850,0x7c3930ba),
   LL(0xaaa2902b,0xe07141fc),LL(0xe4f69ad3,0x539ad799),LL(0x813f9ffd,0xa6453f94),LL(0x375bc2f7,0xc58d3c48), LL(0x5dc64e96,0xb3326fad),LL(0xb240e354,0x3aafcaa9),LL(0xaca1e7a9,0x1d1b0903),LL(0x1211b8a0,0x4ceb9767),
   LL(0xe32a858e,0xeca83e49),LL(0xae907bad,0x4c32892e),LL(0x2eb9b494,0xd5b42ab6),LL(0x1eabae1b,0x7fde3ee2), LL(0xcaf54957,0x13b5ab09),LL(0xe5f5d5d5,0xbfb028be),LL(0x2003e2c0,0x928a0650),LL(0x67476843,0x90793aac),
   LL(0xc81710a0,0x5e942e79),LL(0x27ccadd4,0x557e4a36),LL(0x4bcf6d0c,0x72a2bc56),LL(0x26d7b80c,0x09ee5f43), LL(0xd4292f19,0x6b70dbe9),LL(0x63f16b18,0x56f74c26),LL(0x35fbb42a,0xc23db0f7),LL(0x6ae10040,0xb606bdf6),
   LL(0x044573ac,0x1eb15d4d),LL(0x556b0ba4,0x7dc3cf86),LL(0xc60df6f7,0x97af9a33),LL(0xa716ce8c,0x0b1ef85c), LL(0xc96958be,0x2922f884),LL(0x35690963,0x7c32fa94),LL(0xeaa00061,0x2d7f667c),LL(0x3547365c,0xeaaf7c17),
   LL(0x87032d58,0x1eb4de46),LL(0x5e2c79e0,0xc54f3d83),LL(0x5d04ef23,0x07818df4),LL(0x673d41b4,0x55faa9c8), LL(0x89b95355,0xced64f6f),LL(0xb7415c84,0x4860d2ea),LL(0x050ebad3,0x5fdb9bd2),LL(0x6685a5bf,0xdb53e0cc),
   LL(0x9feb6593,0xb830c031),LL(0x6accff17,0xdd87f310),LL(0x9f555c10,0x2303ebab),LL(0x287e7065,0x94603695), LL(0x2e83358c,0xf88311c3),LL(0xeefb0178,0x508dd9b4),LL(0x2dba8652,0x7ca23706),LL(0x0047abe5,0x62aac5a3),
   LL(0x8b1ea7b3,0x9a61d2a0),LL(0xae8b1485,0xd495ab63),LL(0x87052f99,0x38740f84),LL(0xb2974eea,0x178ebe5b), LL(0x5b36d17f,0x030bbcca),LL(0xaaf86eea,0xb5e4cce3),LL(0x68f8e9e0,0xb51a0220),LL(0x09eb3e75,0xa4348796),
   LL(0xeef1a752,0xbe592309),LL(0x6f2aa1ed,0x5d7162d7),LL(0x0f007dd2,0xaebfb5ed),LL(0xc89edd22,0x255e14b2), LL(0x0303b697,0xba85e072),LL(0xf05720ff,0xc5d17e25),LL(0x5128ebb6,0x02b58d6e),LL(0xd754e113,0x2c80242d),
   LL(0xabfae1ca,0x919fca5f),LL(0x1a21459b,0x937afaac),LL(0x1f66a4d2,0x9e0ca91c),LL(0x23ec1331,0x194cc7f3), LL(0x8aa11690,0xad25143a),LL(0x09b59e08,0xbe40ad8d),LL(0xe750860a,0x37d60d9b),LL(0xc6bf434c,0x6c53b008),
   LL(0x1356eb80,0xb572415d),LL(0x9578ded8,0xb8bf9da3),LL(0x5e8fb38b,0x22658e36),LL(0x5af8cb22,0x9b70ce22), LL(0x829a8180,0x7c00018a),LL(0xb81ed295,0x84329f93),LL(0x5f3cea83,0x7c343ea2),LL(0x67586536,0x38f8655f),
   LL(0x1d3ec517,0xa661a0d0),LL(0x512321ae,0x98744652),LL(0xeca92598,0x084ca591),LL(0x1dcb3feb,0xa9bb9dc9), LL(0x78b4c240,0x14c54355),LL(0x610cafdc,0x5ed62a3b),LL(0x1b38846b,0x07512f37),LL(0xb0e38161,0x571bb70a),
   LL(0x2da705d2,0xb556b95b),LL(0xb1a08f98,0x3ef8ada6),LL(0xddecfbe5,0x85302ca7),LL(0x943105cd,0x0e530573), LL(0x21a9255d,0x60554d55),LL(0xf2f3802a,0x63a32fa1),LL(0xcd477875,0x35c8c5b0),LL(0x6ad42da1,0x97f458ea),
   LL(0xeb6b242d,0x832d7080),LL(0x3b71e246,0xd30bd023),LL(0xbe31139d,0x7027991b),LL(0x462e4e53,0x68797e91), LL(0x6b4e185a,0x423fe20a),LL(0x42d9b707,0x82f2c67e),LL(0x4cf7811b,0x25c81768),LL(0x045bb95d,0xbd53005e),
},
/* digit=30 base_pwr=2^210 */
{
   LL(0x9d8e68fd,0xe5f649be),LL(0x1b044320,0xdb0f0533),LL(0xe0c33398,0xf6fde9b3),LL(0x66c8cfae,0x92f4209b), LL(0x1a739d4b,0xe9d1afcc),LL(0xa28ab8de,0x09aea75f),LL(0xeac6f1d0,0x14375fb5),LL(0x708f7aa5,0x6420b560),
   LL(0x6254dc41,0x9eae499c),LL(0x7a837e7e,0x7e293924),LL(0x090524a7,0x74aec08c),LL(0x8d6f55f2,0xf82b9219), LL(0x1402cec5,0x493c962e),LL(0xfa2f30e7,0x9f17ca17),LL(0xe9b879cb,0xbcd783e8),LL(0x5a6f145f,0xea3d8c14),
   LL(0x5e0dee6e,0xdede15e7),LL(0xdc628aa2,0x74f24872),LL(0x7861bb93,0xd3e9c4fe),LL(0x6187b2e0,0x56d4822a), LL(0xc59826f9,0xb66417cf),LL(0x2408169e,0xca260969),LL(0xc79ef885,0xedf69d06),LL(0xdc7d138f,0x00031f8a),
   LL(0x0ebcf726,0x103c46e6),LL(0x6231470e,0x4482b831),LL(0x487c2109,0x6f6dfaca),LL(0x62e666ef,0x2e0ace97), LL(0x1f8d1f42,0x3246a9d3),LL(0x574944d2,0x1b1e83f1),LL(0xa57f334b,0x13dfa63a),LL(0x9f025d81,0x0cf8daed),
   LL(0x00ee11c1,0x30d78ea8),LL(0xb5e3dd75,0xeb053cd4),LL(0xd58c43c5,0x9b65b13e),LL(0xbd151663,0xc3ad49bd), LL(0xb6427990,0x99fd8e41),LL(0x707eae1e,0x12cf15bd),LL(0x1aabb71e,0x29ad4f1b),LL(0x07545d0e,0x5143e74d),
   LL(0xc88bdee1,0x30266336),LL(0x5876767c,0x25f29306),LL(0xc6731996,0x9c078571),LL(0xed552951,0xc88690b2), LL(0x852705b4,0x274f2c2d),LL(0x4e09552d,0xb0bf8d44),LL(0x986575d1,0x7628beeb),LL(0x7f864651,0x407be238),
   LL(0xa639fc6b,0x0e5e3049),LL(0x86003625,0xe75c35d9),LL(0x5dcc1646,0x0cf35bd8),LL(0x6c26273a,0x8bcaced2), LL(0xb5536742,0xe22ecf1d),LL(0x1a9e068b,0x013dd897),LL(0x8a7909c5,0x17f411cb),LL(0x861dd506,0x5757ac98),
   LL(0x1e935abb,0x85de1f0d),LL(0x154de37a,0xdefd10b4),LL(0x369cebb5,0xb8d9e392),LL(0x761324be,0x54d5ef9b), LL(0x74f17e26,0x4d6341ba),LL(0x78c1dde4,0xc0a0e3c8),LL(0x87d918fd,0xa6d77581),LL(0x02ca3a13,0x66876015),
   LL(0xf36658f0,0xc7313e9c),LL(0x71f8057e,0xc433ef1c),LL(0x1b6a835a,0x85326246),LL(0x7c86394c,0xc8f05398), LL(0xe983c4a1,0xff398cdf),LL(0x03b7b931,0xbf5e8162),LL(0xb7b9045b,0x93193c46),LL(0xa4a6e46b,0x1e4ebf5d),
   LL(0x43a24fe7,0xf9942a60),LL(0xffb3492b,0x29c1191e),LL(0x902fde05,0x9f662449),LL(0x6713c32d,0xc792a7ac), LL(0xb737982c,0x2fd88ad8),LL(0xa21e60e3,0x7e3a0319),LL(0x7383591a,0x09b0de44),LL(0x8310a456,0x6df141ee),
   LL(0xe6d6f471,0xaec1a039),LL(0x1198d12e,0x14b2ba0f),LL(0x3aeee5ac,0xebc1a160),LL(0xe0b964ce,0x401f4836), LL(0x4fd03f66,0x2ee43796),LL(0xdd8f3f12,0x3fdb4e49),LL(0x29380f18,0x6ef267f6),LL(0x8da64d16,0x3e8e9670),
   LL(0x207674f1,0xbc19180c),LL(0x33ae8fdb,0x112e09a7),LL(0x6aaeb71e,0x99667554),LL(0xe101b1c7,0x79432af1), LL(0xde2ddec6,0xd5eb558f),LL(0x5357753f,0x81392d1f),LL(0x3ae1158a,0xa7a76b97),LL(0x4a899991,0x416fbbff),
   LL(0x0d4a9dcf,0x9e65fdfd),LL(0x944ddf12,0x7bc29e48),LL(0x3c856866,0xbc1a92d9),LL(0x6e98dfe2,0x273c6905), LL(0xcdfaa6b8,0x69fce418),LL(0x5061c69f,0x606bd823),LL(0x6af75e27,0x42d495a0),LL(0x6d873a1f,0x8ed3d505),
   LL(0x6ab25b6a,0xaf552841),LL(0x2b1a4523,0xc6c0ffc7),LL(0x21c99e03,0xab18827b),LL(0x9034691b,0x060e8648), LL(0x93c7f398,0x5207f90f),LL(0x82f8d10b,0x9f4a96cb),LL(0x3ad0f9e3,0xdd71cd79),LL(0xfc3a54f5,0x84f435d2),
   LL(0x8e33787f,0x4b03c55b),LL(0xa6384673,0xef42f975),LL(0x5051b9f0,0xff7304f7),LL(0x741c87c2,0x18aca1dc), LL(0x2d4bfe80,0x56f120a7),LL(0x053e732c,0xfd823b3d),LL(0x7537ca16,0x11bccfe4),LL(0x1b5a996b,0xdf6c9c74),
   LL(0x904fc3fa,0xee7332c7),LL(0xc7e3636a,0x14a23f45),LL(0xf091d9aa,0xc38659c3),LL(0xb12d8540,0x4a995e5d), LL(0xf3a5598a,0x20a53bec),LL(0xb1eaa995,0x56534b17),LL(0xbf04e03c,0x9ed3dca4),LL(0xd8d56268,0x716c563a),
   LL(0x1d6178e7,0x27ba77a4),LL(0x68a1ff8e,0xe4c80c40),LL(0x0a13f63d,0x75011099),LL(0xa61d46f3,0x7bf33521), LL(0x10b365bb,0x0aff218e),LL(0x0fd7ea75,0x81021804),LL(0xa4b3a925,0x05a3fd8a),LL(0x9b3db4e6,0xb829e75f),
   LL(0x4d53e5fb,0x6bdc75a5),LL(0xd52717e3,0x04a5dc02),LL(0xe9a42ec2,0x86af502f),LL(0x2630e382,0x8867e8fb), LL(0xbec9889b,0xbf845c6e),LL(0xcb47c98d,0x54f491f2),LL(0x790c2a12,0xa3091fba),LL(0xc20f708b,0xd7f6fd78),
   LL(0xacde5e17,0xa569ac30),LL(0x6852b4d7,0xd0f996d0),LL(0x4609ae54,0xe51d4bb5),LL(0x0daed061,0x3fa37d17), LL(0x34b8fb41,0x62a88684),LL(0x9efb64f1,0x99a2acbd),LL(0x6448e1f2,0xb75c1a5e),LL(0x42b5a069,0xfa99951a),
   LL(0x2f3b26e7,0x6d956e89),LL(0xda875247,0xf4709860),LL(0x2482dda3,0x3ad15179),LL(0x017d82f0,0xd64110e3), LL(0xfad414e4,0x14928d2c),LL(0x2ed02b24,0x2b155f58),LL(0xcb821bf1,0x481a141b),LL(0x4f81f5da,0x12e3c770),
   LL(0x9fff8381,0xe49c5de5),LL(0x5bbec894,0x11053232),LL(0x454d88c4,0xa0d051cc),LL(0x1f8e531b,0x4f6db89c), LL(0xca563a44,0x34fe3fd6),LL(0x58da8ab9,0x7f5c2215),LL(0x9474f0a1,0x8445016d),LL(0xcb7d8a0a,0x17d34d61),
   LL(0x1c474019,0x8e9d3910),LL(0xd52ceefb,0xcaff2629),LL(0xc1622c2b,0xf9cf3e32),LL(0xe9071a05,0xd4b95e3c), LL(0x1594438c,0xfbbca61f),LL(0x04aadedf,0x1eb6e6a6),LL(0x68e14940,0x853027f4),LL(0xdfabda9c,0x221d322a),
   LL(0xb7cb179a,0xed8ea9f6),LL(0xb7934dcc,0xdc7b764d),LL(0x5e09180d,0xfcb13940),LL(0xb47dc2dd,0x6629a6bf), LL(0x9f5a915e,0xbfc55e4e),LL(0x6204441e,0xb1db9d37),LL(0x930c5f53,0xf82d68cf),LL(0xcbb605b1,0x17d3a142),
   LL(0x308780f2,0xdd5944ea),LL(0x3845f5e4,0xdc8de761),LL(0x7624d7a3,0x6beaba7d),LL(0x304df11e,0x1e709afd), LL(0x02170456,0x95364376),LL(0xc8f94b64,0xbf204b3a),LL(0x5680ca68,0x4e53af7c),LL(0xe0c67574,0x0526074a),
   LL(0xecd92af6,0x95d8cef8),LL(0x6cd1745a,0xe6b9fa7a),LL(0xa325c3e4,0x3d546d3d),LL(0x9ae93aae,0x1f57691d), LL(0x9d2e1a33,0xe891f3fe),LL(0xac063d35,0xd430093f),LL(0x5513a327,0xeda59b12),LL(0x5536f18f,0xdc2134f3),
   LL(0x5c210286,0xaa51fe2c),LL(0x1cab658c,0x3f68aaee),LL(0xf9357292,0x5a23a00b),LL(0x7efdabed,0x9a626f39), LL(0x199d78e3,0xfe2b3bf3),LL(0x71bbc345,0xb7a2af77),LL(0x1e59802c,0x3d19827a),LL(0xb487a51c,0x823bbc15),
   LL(0x99d0a422,0x856139f2),LL(0xf456c6fb,0x9ac3df65),LL(0x701f8bd6,0xaddf65c6),LL(0x3758df87,0x149f321e), LL(0x721b7eba,0xb1ecf714),LL(0x31a3312a,0xe17df098),LL(0xd5c4d581,0xdb2fd6ec),LL(0x8fcea1b3,0xfd02996f),
   LL(0x7882f14f,0xe29fa63e),LL(0x07c6cadc,0xc9f6dc35),LL(0xb882bed0,0x46f22d6f),LL(0xd118e52c,0x1a45755b), LL(0x7c4608cf,0x9f2c7c27),LL(0x568012c2,0x7ccbdf32),LL(0x61729b0e,0xfcb0aedd),LL(0xf7d75dbf,0x7ca2ca9e),
   LL(0x6f640f62,0xf58fecb1),LL(0x39f51946,0xe274b92b),LL(0x6288af44,0x7f4dfc04),LL(0xeac329e5,0x0a91f32a), LL(0xd6aaba31,0x43ad274b),LL(0x0f6884f9,0x719a1640),LL(0xdaf91e20,0x685d29f6),LL(0x27e49d52,0x5ec1cc33),
   LL(0x3b54a059,0x38f4de96),LL(0xefbcfdb3,0x0e0015e5),LL(0x4dbb8da6,0x177d23d9),LL(0x97a617ad,0x98724aa2), LL(0xfdb6558e,0x30f0885b),LL(0xc7899a96,0xf9f7a28a),LL(0x872dc112,0xd2ae8ac8),LL(0x73c3c459,0xfa0642ca),
   LL(0xe7dfc8d6,0x15296981),LL(0x1fb5b94a,0x67cd4450),LL(0x0eddfd37,0x0ec71cf1),LL(0x9a8eddc7,0xc7e5eeb3), LL(0x81d95028,0x02ac8e3d),LL(0x70b0e35d,0x0088f172),LL(0xe1881fe3,0xec041fab),LL(0xd99e7faa,0x62cf71b8),
   LL(0xe0f222c2,0x5043dea7),LL(0x72e65142,0x309d42ac),LL(0x9216cd30,0x94fe9ddd),LL(0x0f87feec,0xd6539c7d), LL(0x432ac7d7,0x03c5a57c),LL(0x327fda10,0x72692cf0),LL(0x280698de,0xec28c85f),LL(0x7ec283b1,0x2331fb46),
   LL(0x2867e633,0xd34bfa32),LL(0x0a9cc815,0x78709a82),LL(0x875e2fa5,0xb7fe6964),LL(0x9e98bfb5,0x25cc064f), LL(0x493a65c5,0x9eb0151c),LL(0x53182464,0x5fb5d941),LL(0xf04618e2,0x69e6f130),LL(0xf89c8ab6,0xa8ecec22),
   LL(0xb96209bd,0xcd6ac88b),LL(0xb3e1c9e0,0x65fa8cdb),LL(0x4a8d8eac,0xa47d22f5),LL(0x8d33f963,0x83895cdf), LL(0xb56cd3d1,0xa8adca59),LL(0xdaf38232,0x10c8350b),LL(0xa5080a9f,0x2b161fb3),LL(0x3af65b3a,0xbe7f5c64),
   LL(0x97403a11,0x2c754039),LL(0x121b96af,0x94626cf7),LL(0x6a983ec2,0x431de7c4),LL(0x52cc3df7,0x3780dd3a), LL(0x2baf8e3b,0xe28a0e46),LL(0x51d299ae,0xabe68aad),LL(0x647a2408,0x603eb8f9),LL(0x5c750981,0x14c61ed6),
   LL(0xc53352e7,0x88b34414),LL(0x1337d46e,0x5a34889c),LL(0xf95f2bc8,0x612c1560),LL(0xd4807a3a,0x8a3f8441), LL(0x5224da68,0x680d9e97),LL(0xc3eb00e9,0x60cd6e88),LL(0x9a6bc375,0x3875a98e),LL(0x4fd554c2,0xdc80f924),
   LL(0x6ac77407,0x6c4b3415),LL(0x25420681,0xa1e5ea8f),LL(0x4607a458,0x541bfa14),LL(0x96d7fbf9,0x5dbc7e7a), LL(0x31590a47,0x646a851b),LL(0x15ee6df8,0x039e85ba),LL(0xd7b43fc0,0xd19fa231),LL(0x299a0e04,0x84bc8be8),
   LL(0xf20df03a,0x2b9d2936),LL(0x8608d472,0x24054382),LL(0x9149202a,0x76b6ba04),LL(0x3670e7b7,0xb21c3831), LL(0xd6fdee10,0xddd93059),LL(0x78488e71,0x9da47ad3),LL(0xa0fcfb25,0x99cc1dfd),LL(0x64696954,0x42abde10),
   LL(0x17eab9fe,0x14cc15fc),LL(0xd3e70972,0xd6e863e4),LL(0x6432112c,0x29a7765c),LL(0x5b0774d8,0x88660001), LL(0x2c088eae,0x3729175a),LL(0x8230b8d4,0x13afbcae),LL(0x915f4379,0x44768151),LL(0xd8d22812,0xf086431a),
   LL(0xc298b974,0x37461955),LL(0xf8711e04,0x905fb5f0),LL(0xfe969d18,0x787abf3a),LL(0x6f6a494e,0x392167c2), LL(0x28c511da,0xfc7a0d2d),LL(0xb66a262d,0xf127c7dc),LL(0xfd63fdf0,0xf9c4bb95),LL(0x3913ef46,0x90016589),
   LL(0x11aa600d,0x74d2a73c),LL(0x9fb5ab52,0x2f5379bd),LL(0x7fb70068,0xe49e53a4),LL(0x404aa9a7,0x68dd39e5), LL(0x2ecaa9c3,0xb9b0cf57),LL(0xe824826b,0xba0e103b),LL(0x4631a3c4,0x60c2198b),LL(0xfa8966a2,0xc5ff84ab),
   LL(0xac95aff8,0x2d6ebe22),LL(0xb5a46d09,0x1c9bb6db),LL(0x53ee4f8d,0x419062da),LL(0xbb97efef,0x7b9042d0), LL(0x830cf6bd,0x0f87f080),LL(0x6ec8a6c6,0x4861d19a),LL(0x202f01aa,0xd3a0daa1),LL(0xf25afbd5,0xb0111674),
   LL(0x1afb20d9,0x6d00d6cf),LL(0x40671bc5,0x13695000),LL(0x2485ea9b,0x913ab0dc),LL(0x9eef61ac,0x1f2bed06), LL(0x6d799e20,0x850c8217),LL(0x3271c2de,0x93415f37),LL(0x6c4f5910,0x5afb06e9),LL(0xc4e9e421,0x688a52df),
   LL(0xe2a9a6db,0x30495ba3),LL(0x58f9268b,0x4601303d),LL(0x7eb0f04f,0xbe3b0dad),LL(0x4456936d,0x4ea47250), LL(0xd33fd3e7,0x8caf8798),LL(0xeb433708,0x1ccd8a89),LL(0x87fd50ad,0x9effe3e8),LL(0x6b29c4df,0xbe240a56),
   LL(0xca0e7ebd,0xec4ffd98),LL(0xe748616e,0xf586783a),LL(0xc77baa99,0xa5b00d8f),LL(0xb4f34c9c,0x0acada29), LL(0x0fe723ac,0x36dad67d),LL(0x39c36c1e,0x1d8e53a5),LL(0x1f4bea41,0xe4dd342d),LL(0xebc9e4e0,0x64fd5e35),
   LL(0x57908805,0x96f01f90),LL(0x5ed480dd,0xb5b9ea3d),LL(0x3efd2dd0,0x366c5dc2),LL(0x6e9dfa27,0xed2fe305), LL(0x6e9197e2,0x4575e892),LL(0xab502a5d,0x11719c09),LL(0xe81f213f,0x264c7bec),LL(0x55f5c457,0x741b9241),
   LL(0x49a5f4f4,0x78ac7b68),LL(0x9fc45b7d,0xf91d70a2),LL(0xb0f5f355,0x39b05544),LL(0xeef930d9,0x11f06bce), LL(0x038d05e1,0xdb84d25d),LL(0xbacc1d51,0x04838ee5),LL(0x9e8ee00b,0x9da3ce86),LL(0xc36eda1f,0xc3412057),
   LL(0x64d9c2f4,0xae80b913),LL(0xa010a8ff,0x7468bac3),LL(0x37359d41,0xdfd20037),LL(0x15efeacc,0x1a0f5ab8), LL(0x659d0ce0,0x7c25ad2f),LL(0x6785cff1,0x4011bcbb),LL(0x7e2192c7,0x128b9912),LL(0x13ccb0e8,0xa549d8e1),
   LL(0xc85438b1,0x805588d8),LL(0xbc25cb27,0x5680332d),LL(0x1a4bfdf4,0xdcd1bc96),LL(0x706f6566,0x779ff428), LL(0xf059987a,0x8bbee998),LL(0xcc686de7,0xf6ce8cf2),LL(0x953cfdb2,0xf8ad3c4a),LL(0x2205da36,0xd1d426d9),
   LL(0xc781a241,0xb3c0f13f),LL(0xd75362a8,0x3e89360e),LL(0xc8a91184,0xccd05863),LL(0xefa8a7f4,0x9bd0c9b7), LL(0x8a912a4b,0x97ee4d53),LL(0xbcf518fd,0xde5e15f8),LL(0xc467e1e0,0x6a055bf8),LL(0x1587e256,0x10be4b4b),
   LL(0x668621c9,0xd90c14f2),LL(0xab9c92c1,0xd5518f51),LL(0xd6d47b3c,0x8e6a0100),LL(0x66716175,0xcbe980dd), LL(0xddd83683,0x500d3f10),LL(0x99cac73c,0x3b6cb35d),LL(0x6083d550,0x53730c8b),LL(0xdf0a1987,0xcf159767),
   LL(0x43ad73b3,0x84bfcf53),LL(0x4f035a94,0x1b528c20),LL(0x33eeac69,0x4294edf7),LL(0x817f3240,0xb6283e83), LL(0x0a5f25b1,0xc3fdc959),LL(0x5844ee22,0xefaf8aa5),LL(0xdbdde4de,0xde269ba5),LL(0xc56133bf,0xe3347160),
   LL(0x8d9ea9f8,0xc1184219),LL(0xf3fc1ab5,0x090de5db),LL(0x0bf22cda,0x404c37b1),LL(0xf5618894,0x7de20ec8), LL(0xecdaecab,0x754c588e),LL(0x88342743,0x6ca4b0ed),LL(0xf4a938ec,0x76f08bdd),LL(0x91493ccb,0xd182de89),
   LL(0xc8a4186a,0xd652c53e),LL(0x946d8e33,0xb3e878db),LL(0x5f37663c,0x088453c0),LL(0xb407748b,0x5cd9daaa), LL(0x586d5e72,0xa1f5197f),LL(0xc443ca59,0x47500be8),LL(0xe2652424,0x78ef35b2),LL(0x6dd7767d,0x09c5d26f),
   LL(0xa74d3f7b,0x7175a79a),LL(0xcf5ea459,0x0428fd8d),LL(0xa5d1746d,0x511cb97c),LL(0xe71d1278,0x36363939), LL(0x10350bf4,0xcf2df955),LL(0x60aae782,0xb3817439),LL(0x3e688809,0xa748c0e4),LL(0xd7a5a006,0x98021fbf),
   LL(0x0e367a98,0x9076a70c),LL(0x0f62b7c2,0xbea1bc15),LL(0x30fe0343,0x2645a68c),LL(0x699dc14f,0xacaffa78), LL(0x457bf9c4,0xf4469964),LL(0x0d2ead83,0x0db6407b),LL(0xb2c6f3eb,0x68d56cad),LL(0xf376356c,0x3b512e73),
   LL(0xfce10408,0xe43b0e1f),LL(0x5a5e257d,0x89ddc003),LL(0x0362e5b3,0xb0ae0d12),LL(0xb0519161,0x07f983c7), LL(0x5d5231e7,0xc2e94d15),LL(0x0b4f9513,0xcff22aed),LL(0x6ad0b0b5,0xb02588dd),LL(0x11d0dcd5,0xb967d1ac),
   LL(0xcf777b6c,0x8dac6bc6),LL(0x4c6d1959,0x0062bdbd),LL(0x0ef5cc85,0x53da71b5),LL(0x4006f14f,0x07012c7d), LL(0xac47800d,0x4617f962),LL(0xc102ed75,0x53365f2b),LL(0x4ab8c9d3,0xb422efcb),LL(0x34af31c9,0x195cb26b),
   LL(0x05f2c4ce,0x3a926e29),LL(0x9856966c,0xbd2bdecb),LL(0x85527015,0x5d16ab3a),LL(0x4486c231,0x9f81609e), LL(0xda350002,0xd8b96b2c),LL(0xfa1b7d36,0xbd054690),LL(0xe71d79bc,0xdc90ebf5),LL(0x08964e4e,0xf241b6f9),
   LL(0x2fe3cd4c,0x7c838643),LL(0xb4bc633c,0xe0f33acb),LL(0x3d139f1f,0xb4a9ecec),LL(0xdc4a1f49,0x05ce69cd), LL(0xf5f98aaf,0xa19d1b16),LL(0x6f23e0ef,0x45bb71d6),LL(0x46cdfdd3,0x33789fcd),LL(0xcee040ca,0x9b8e2978),
   LL(0xae0a6828,0x9c69b246),LL(0x7078d5aa,0xba533d24),LL(0x7bb4fbdb,0x7a2e42c0),LL(0x7035385c,0xcfb4879a), LL(0x3281705b,0x8c3dd30b),LL(0x404fe081,0x7e361c6c),LL(0x3f604edf,0x7b21649c),LL(0xe52ffe47,0x5dbf6a3f),
   LL(0x4b54d9bf,0xc41b7c23),LL(0x3511c3d9,0x1374e681),LL(0xc1b2b758,0x1863bf16),LL(0x1e9e6a96,0x90e78507), LL(0x5d86f174,0xab4bf98d),LL(0x85e96fe4,0xd74e0bd3),LL(0xcac5d344,0x8afde39f),LL(0xbd91b847,0x90946dbc),
   LL(0xfe1a838c,0xf5b42358),LL(0x620ac9d8,0x05aae6c5),LL(0xa1ce5a0b,0x8e193bd8),LL(0x4dabfd72,0x8f710571), LL(0x182caaac,0x8d8fdd48),LL(0x040745cf,0x8c4aeefa),LL(0xf3b93e6d,0x73c6c30a),LL(0x16f42011,0x991241f3),
   LL(0xe457a477,0xa0158eea),LL(0xee6ddc05,0xd19857db),LL(0x18c41671,0xb3265224),LL(0x3c2c0d58,0x3ffdfc7e), LL(0x26ee7cda,0x3a3a5254),LL(0xdf02c3a8,0x341b0869),LL(0x723bbfc8,0xa023bf42),LL(0x14452691,0x3d15002a),
},
/* digit=31 base_pwr=2^217 */
{
   LL(0x85edfa30,0x5ef7324c),LL(0x87d4f3da,0x25976554),LL(0xdcb50c86,0x352f5bc0),LL(0x4832a96c,0x8f6927b0), LL(0x55f2f94c,0xd08ee1ba),LL(0x344b45fa,0x6a996f99),LL(0xa8aa455d,0xe133cb8d),LL(0x758dc1f7,0x5d0721ec),
   LL(0x79e5fb67,0x6ba7a920),LL(0x70aa725e,0xe1331feb),LL(0x7df5d837,0x5080ccf5),LL(0x7ff72e21,0xe4cae01d), LL(0x0412a77d,0xd9243ee6),LL(0xdf449025,0x06ff7cac),LL(0x23ef5a31,0xbe75f7cd),LL(0x0ddef7a8,0xbc957822),
   LL(0xb0ce1c55,0x8cf7230c),LL(0x0bbfb607,0x5b534d05),LL(0x0e16363b,0xee1ef113),LL(0xb4999e82,0x27e0aa7a), LL(0x79362c41,0xce1dac2d),LL(0x91bb6cb0,0x67920c90),LL(0x2223df24,0x1e648d63),LL(0xe32e8f28,0x0f7d9eef),
   LL(0xfa833834,0x6943f39a),LL(0xa6328562,0x22951722),LL(0x4170fc10,0x81d63dd5),LL(0xaecc2e6d,0x9f5fa58f), LL(0xe77d9a3b,0xb66c8725),LL(0x6384ebe0,0x11235cea),LL(0x5845e24a,0x06a8c118),LL(0xebd093b1,0x0137b286),
   LL(0x44ace150,0xc589e1ce),LL(0x4381e97c,0xe0f8d3d9),LL(0x62c5a4b8,0x59e99b11),LL(0xfd0ec9f9,0x90d262f7), LL(0x283e13c9,0xfbc854c9),LL(0xaedc7085,0x2d04fde7),LL(0x47dcbecb,0x057d7765),LL(0x9a76fa5f,0x8dbdf591),
   LL(0x0de1e578,0xd0150695),LL(0xe9f72bc6,0x2e1463e7),LL(0x1b39eca5,0xffa68441),LL(0x7c037f2f,0x673c8530), LL(0x747f91da,0xd0d6a600),LL(0xc9cb78e9,0xb08d43e1),LL(0x27b5cef5,0x0fc0c644),LL(0xa60a2fd6,0x5c1d160a),
   LL(0x28c8e13b,0xf98cae53),LL(0xb2eddcd1,0x375f10c4),LL(0x5cce06ad,0xd4eb8b7f),LL(0x80a2e1ef,0xb4669f45), LL(0x5bbd8699,0xd593f9d0),LL(0xe7976d13,0x5528a4c9),LL(0x1c7e28d3,0x3923e095),LL(0x3f6bb577,0xb9293790),
   LL(0xc42bd6d2,0xdb567d6a),LL(0xbb1f96ae,0x6df86468),LL(0x4843b28e,0x0efe5b1a),LL(0x6379b240,0x961bbb05), LL(0x70a6a26b,0xb6caf5f0),LL(0x328e6e39,0x70686c0d),LL(0x895fc8d3,0x80da06cf),LL(0xb363fdc9,0x804d8810),
   LL(0x207f1670,0xbe22877b),LL(0x4e615291,0x9b0dd188),LL(0x97a3c2bf,0x625ae8dc),LL(0x439b86e8,0x08584ef7), LL(0xdcd898ff,0xde7190a5),LL(0x2058ee3d,0x26286c40),LL(0x5f87b1c1,0x3db0b217),LL(0x102a6db5,0xcc334771),
   LL(0x2f770fb1,0xd99de954),LL(0x4cd7535e,0x97c1c620),LL(0x3f09cefc,0xd3b6c448),LL(0x5a63b4f8,0xd725af15), LL(0xc01e20ec,0x0c95d24f),LL(0x9ae7121f,0xdfd37494),LL(0xec77b7ec,0x7d6ddb72),LL(0x0353a4ae,0xfe079d3b),
   LL(0x2e6ac8d2,0x3066e70a),LL(0x106e5c05,0x9c6b5a43),LL(0xede59b8c,0x52d3c6f5),LL(0xfccec9ae,0x30d6a5c3), LL(0x4fc0a9ef,0xedec7c22),LL(0x95c16ced,0x190ff083),LL(0x94de0fde,0xbe12ec8f),LL(0x852d3433,0x0d131ab8),
   LL(0x85701291,0x42ace07e),LL(0x194061a8,0x94793ed9),LL(0xd7f4a485,0x30e83ed6),LL(0xf9eeff4d,0x9eec7269), LL(0x0c9d8005,0x90acba59),LL(0x1e79b9d1,0x5feca458),LL(0x1d506a1e,0x8fbe5427),LL(0x2439cfa7,0xa32b2c8e),
   LL(0x73dd0b4e,0x1671c173),LL(0x44a054c6,0x37a28214),LL(0x4e8b53f1,0x81760a1b),LL(0xf9f93b9e,0xa6c04224), LL(0xcf671e3c,0x18784b34),LL(0xcda9b994,0x81bbecd2),LL(0xb2ab3848,0x38831979),LL(0xf2e03c2d,0xef54feb7),
   LL(0xfb8088fa,0xcf197ca7),LL(0x4ddc96c5,0x01427247),LL(0x30777176,0xa2d2550a),LL(0x4d0cf71d,0x53469898), LL(0x3a2aaac6,0x6ce937b8),LL(0x5af38d9b,0xe9f91dc3),LL(0xc8bf2899,0x2598ad83),LL(0xb5536c16,0x8e706ac9),
   LL(0xf688dc98,0x40dc7495),LL(0x124c4afc,0x26490cd7),LL(0x1f18775c,0xe651ec84),LL(0xb4fdaf4a,0x393ea6c3), LL(0x7f338e0d,0x1e1f3343),LL(0x6053e7b5,0x39fb832b),LL(0x619e14d5,0x46e702da),LL(0xcdeef6e0,0x859cacd1),
   LL(0x4462007d,0x63b99ce7),LL(0x4cb5f5b7,0xb8ab48a5),LL(0xf55edde7,0x9ec673d2),LL(0x8cfaefda,0xd1567f74), LL(0x0887bcec,0x46381b6b),LL(0xe178f3c2,0x694497ce),LL(0x1e6266cb,0x5e6525e3),LL(0x697d6413,0x5931de26),
   LL(0x0e58d493,0x87f8df7c),LL(0x58b73f12,0xb1ae5ed0),LL(0xdea0c34d,0xc368f784),LL(0x859a91a0,0x9bd0a120), LL(0xcc863c68,0xb00d88b7),LL(0x3d1f4d65,0x3a1cc11e),LL(0x0aa85593,0xea38e0e7),LL(0x7dc4aee8,0x37f13e98),
   LL(0xbc947bad,0x10d38667),LL(0x2a36ee2e,0x738e07ce),LL(0xc577fcac,0xc93470cd),LL(0x2782470d,0xdee1b616), LL(0x2e793d12,0x36a25e67),LL(0xe0f186da,0xd6aa6cae),LL(0x80e07af7,0x474d0fd9),LL(0xba8a5cd4,0xf7cdc47d),
   LL(0xab15247f,0x28af6d9d),LL(0x493a537f,0x7c789c10),LL(0x23a334e7,0x7ac9b110),LL(0x12c9c277,0x0236ac09), LL(0x1d7a5144,0xa7e5bd25),LL(0xf13ec4ec,0x098b9c2a),LL(0xd3f0abca,0x3639daca),LL(0xa23960f9,0x642da81a),
   LL(0x4f7269b1,0x7d2e5c05),LL(0xe287c385,0xfcf30777),LL(0xf2a46f21,0x10edc84f),LL(0x4f43fa36,0x35441757), LL(0xfd703431,0xf1327899),LL(0x16dd587a,0xa438d7a6),LL(0xe9c8352d,0x65c34c57),LL(0x5cc5a24e,0xa728edab),
   LL(0x42531689,0xaed78abc),LL(0x010963ef,0x0a51a0e8),LL(0xd717d9b3,0x5776fa0a),LL(0x7dd3428b,0xf356c239), LL(0x8d3a3dac,0x29903fff),LL(0x3d94491f,0x409597fa),LL(0xbf4a56a4,0x4cd7a5ff),LL(0x8adab462,0xe5096474),
   LL(0x5c3427b0,0xa97b5126),LL(0xd282c9bd,0x6401405c),LL(0x222c5c45,0x3629f8d7),LL(0xe8d50aed,0xb1c02c16), LL(0xd9635bc9,0xbea2ed75),LL(0x6e24552f,0x226790c7),LL(0x65f1d066,0x3c33f2a3),LL(0x6dfccc2e,0x2a43463e),
   LL(0xdb483761,0x8cc3453a),LL(0x65d5672b,0xe7cc6085),LL(0xde3efc87,0x277ed6cb),LL(0x69234eaf,0x19f2f368), LL(0x5c0b800b,0x9aaf4317),LL(0x8b6da6e2,0x1f1e7c89),LL(0xb94ec75e,0x6cfb4715),LL(0x453118c2,0xd590dd5f),
   LL(0x1f17a34c,0x14e49da1),LL(0x235a1456,0x5420ab39),LL(0x2f50363b,0xb7637241),LL(0xc3fabb6e,0x7b15d623), LL(0xe274e49c,0xa0ef40b1),LL(0x96b1860a,0x5cf50744),LL(0x66afe5a4,0xd6583fbf),LL(0xf47e3e9a,0x44240510),
   LL(0x11b2d595,0x99254343),LL(0xeec8df57,0xf1367499),LL(0x3e73dd05,0x3cb12c61),LL(0x7dac102a,0xd248c033), LL(0xa77739f5,0xcf154f13),LL(0x23d2af42,0xbf4288cb),LL(0x32e4a1cf,0xaa64c9b6),LL(0xc8a208f3,0xee8c07a8),
   LL(0x6fe8393f,0xe10d4999),LL(0xe91f3a32,0x0f809a3f),LL(0x802f63c8,0x61096d1c),LL(0x57750d3d,0x289e1462), LL(0x9889feea,0xed06167e),LL(0xe0993909,0xd5c9c0e2),LL(0x56508ac6,0x46fca0d8),LL(0x4f1b8e83,0x91826047),
   LL(0x9a4a2751,0x4f2c877a),LL(0xcae6fead,0x71bd0072),LL(0x06aa1941,0x38df8dcc),LL(0x63beeaa8,0x5a074b4c), LL(0xc1cec8ed,0xd6d65934),LL(0xaabc03bd,0xa6ecb49e),LL(0xde8a8415,0xaade91c2),LL(0x691136e0,0xcfb0efdf),
   LL(0x23ab3495,0x11af45ee),LL(0x0b77463d,0xa132df88),LL(0x815d06f4,0x8923c15c),LL(0x0d61a436,0xc3ceb3f5), LL(0xe88fb1da,0xaf52291d),LL(0x1da12179,0xea057974),LL(0xd2fef720,0xb0d7218c),LL(0x8e1d8845,0x6c0899c9),
   LL(0x752ddad7,0x98157504),LL(0xa1a68a97,0xd60bd74f),LL(0xf658fb99,0x7047a3a9),LL(0x5f8511e4,0x1f5d86d6), LL(0x4b5a6d88,0xb8a4bc42),LL(0x1abefa7d,0x69eb2c33),LL(0x13c9c510,0x95bf39e8),LL(0xd48aab43,0xf571960a),
   LL(0x704e23c6,0x7e8cfbcf),LL(0x28aaa65b,0xc71b7d22),LL(0x245e3c83,0xa041b2bd),LL(0xd21854ff,0x69b98834), LL(0x963bfeec,0x89d227a3),LL(0xde7da7cb,0x99947aaa),LL(0xee68a9b1,0x1d9ee9db),LL(0x698ec368,0x0a08f003),
   LL(0x78ef2487,0xe9ea4094),LL(0x02cfec26,0xc8d2d415),LL(0xb7dcf328,0xc52f9a6e),LL(0x85b6a937,0x0ed489e3), LL(0xbef3366e,0x9b94986b),LL(0xedddddb8,0x0de59c70),LL(0xeadddbe2,0xffdb748c),LL(0x8266ea40,0x9b9784bb),
   LL(0x1a93507a,0x142b5502),LL(0x8d3c06cf,0xb4cd1187),LL(0x91ec3f40,0xdf70e76a),LL(0x4e7553c2,0x484e81ad), LL(0x272e9d6e,0x830f87b5),LL(0xc6ff514a,0xea1c93e5),LL(0xc4192a8e,0x67cc2adc),LL(0x42f4535a,0xc77e27e2),
   LL(0xd2b713c5,0x9cdbab36),LL(0xcf7b0cd3,0x86274ea0),LL(0x09af826b,0x784680f3),LL(0x0c72dea3,0xbfcc837a), LL(0xd6529b73,0xa8bdfe9d),LL(0x63a88002,0x708aa228),LL(0xc91d45b9,0x6c7a9a54),LL(0xfd004f56,0xdf1a38bb),
   LL(0xb8bad853,0x2e8c9a26),LL(0x3723eae7,0x2d52cea3),LL(0x56ca2830,0x054d6d81),LL(0x9a8dc411,0xa3317d14), LL(0xfd4ddeda,0xa08662fe),LL(0xb55d792b,0xed2a153a),LL(0xbfc6e944,0x7035c16a),LL(0x00171cf3,0xb6bc5834),
   LL(0x83d102b6,0xe27152b3),LL(0x0646b848,0xfe695a47),LL(0x916e6d37,0xa5bb09d8),LL(0x0d17015e,0xb4269d64), LL(0x0a1d2285,0x8d8156a1),LL(0x46d26d72,0xfeef6c51),LL(0x4c5434a7,0x9dac57c8),LL(0x59d39e31,0x0282e5be),
   LL(0x721c486d,0xedfff181),LL(0xbc58824e,0x301baf10),LL(0x00570031,0x8136a6aa),LL(0x1cddde68,0x55aaf78c), LL(0x59c63952,0x26829371),LL(0x8bc25baf,0x3a3bd274),LL(0xb7e52dc3,0xecdf8657),LL(0xfd78e6c8,0x2dd8c087),
   LL(0xf5531461,0x20553274),LL(0x5d95499b,0x8b4a1281),LL(0x1a80f9d2,0xe2c8763a),LL(0x4ddec758,0xd1dbe32b), LL(0x30c34169,0xaf12210d),LL(0x78baa533,0xba74a953),LL(0xa438f254,0x3d133c6e),LL(0x201bef5b,0xa431531a),
   LL(0xf669d7ec,0x15295e22),LL(0x357fb515,0xca374f64),LL(0xeaa3fdb3,0x8a8406ff),LL(0xdf3f2da8,0x106ae448), LL(0x33c8e9a1,0x8f9b0a90),LL(0x71ad5885,0x234645e2),LL(0x1c0aed14,0x3d083224),LL(0x7a942d46,0xf10a7d3e),
   LL(0x40d5c9be,0x7c11deee),LL(0xba84ed98,0xb2bae7ff),LL(0xaad58ddd,0x93e97139),LL(0x3f6d1fa3,0x3d872796), LL(0x8569ff13,0x483aca81),LL(0x9a600f72,0x8b89a5fb),LL(0xc06f2b86,0x4cbc27c3),LL(0x63ad9c0b,0x22130713),
   LL(0x48ac2840,0xb5358b1e),LL(0xecba9477,0x18311294),LL(0xa6946b43,0xda58f990),LL(0x9ab41819,0x3098baf9), LL(0x4198da52,0x66c4c158),LL(0x146bfd1b,0xab4fc17c),LL(0xbf36a908,0x2f0a4c3c),LL(0x58cf7838,0x2ae9e34b),
   LL(0x3fa11b1f,0xf411529e),LL(0x974af2b4,0x21e43677),LL(0xc230793b,0x7c20958e),LL(0x16e840f3,0x710ea885), LL(0xc5dc67cf,0xfc0b21fc),LL(0x88405718,0x08d51647),LL(0xcfe49eb7,0xd955c21f),LL(0x56dd4a1f,0x9722a5d5),
   LL(0xc861baa5,0xc9ef50e2),LL(0x9505ac3e,0xc0c21a5d),LL(0x8b7c063f,0xaf6b9a33),LL(0x2f4779c1,0xc6370339), LL(0x638167c3,0x22df99c7),LL(0x795db30c,0xfe6ffe76),LL(0xa4854989,0x2b822d33),LL(0x30563aa5,0xfef031dd),
   LL(0xd57c667f,0x16b09f82),LL(0xcc0b76f1,0xc70312ce),LL(0xc9118aec,0xbf04a9e6),LL(0x3409d133,0x82fcb419), LL(0xab45d44d,0x1a8ab385),LL(0x617b83a3,0xfba07222),LL(0x58e81b52,0xb05f50dd),LL(0x21ce5aff,0x1d8db553),
   LL(0xe344a873,0x3097b8d4),LL(0xfe36d53e,0x7d8d116d),LL(0x7875e750,0x6db22f58),LL(0x43e144ea,0x2dc5e373), LL(0xe799eb95,0xc05f32e6),LL(0x6899e6ec,0xe9e5f4df),LL(0x1fab23d5,0xbdc3bd68),LL(0x73af60e6,0xb72b8ab7),
   LL(0x2cecc84a,0x8db27ae0),LL(0x7bdb871c,0x600016d8),LL(0xd7c46f58,0x42a44b13),LL(0xc3a77d39,0xb8919727), LL(0xdafd6088,0xcfc6bbbd),LL(0x6bd20d39,0x1a740146),LL(0x98c41072,0x8c747abd),LL(0xbdf68ea1,0x4c91e765),
   LL(0x08819a78,0x7c95e5ca),LL(0xc9587921,0xcf48b729),LL(0xdebbcc7d,0x091c7c5f),LL(0xf0e05149,0x6f287404), LL(0x26cd44ec,0xf83b5ac2),LL(0xcfea250e,0x88ae32a6),LL(0x1d06ebc5,0x6ac5047a),LL(0xd434f781,0xc7e550b4),
   LL(0x5c727bd2,0x61ab1cf2),LL(0x1cf915b0,0x2e4badb1),LL(0xf69d3920,0x1b4dadec),LL(0xf14c1dfe,0xe61b1ca6), LL(0xbd6bd51f,0x90b479cc),LL(0x8045ec30,0x8024e401),LL(0x25ef0e62,0xcab29ca3),LL(0x49e4ebc0,0x4f2e9416),
   LL(0x0ccced58,0x45eb40ec),LL(0x0da44f98,0x25cd4b9c),LL(0x871812c6,0x43e06458),LL(0x16cef651,0x99f80d55), LL(0xce6dc153,0x571340c9),LL(0xd8665521,0x138d5117),LL(0x4e07014d,0xacdb45bc),LL(0x84b60b91,0x2f34bb38),
   LL(0x2ae8921e,0xf44a4fd2),LL(0x892ba1e2,0xb039288e),LL(0xb1c180b2,0x9da50174),LL(0x1693dc87,0x6b70ab66), LL(0xe7057481,0x7e9babc9),LL(0x9c80dc41,0x4581ddef),LL(0x51294682,0x0c890da9),LL(0x3f4736e5,0x0b5629d3),
   LL(0xb06f5b41,0x2340c79e),LL(0x4e243469,0xa42e84ce),LL(0x045a71a9,0xf9a20135),LL(0xd27b6fb6,0xefbfb415), LL(0x9d33cd6f,0x25ebea23),LL(0xaa6c0af8,0x9caedb88),LL(0xd9ce6f96,0x53dc7e9a),LL(0x51e0b15a,0x3897f9fd),
   LL(0x8e5d788e,0xf51cb1f8),LL(0xe1d490ee,0x1aec7ba8),LL(0xcc58cb3c,0x265991e0),LL(0x9fc3ad31,0x9f306e8c), LL(0x5040a0ac,0x5fed006e),LL(0xfb476f2e,0xca9d5043),LL(0xbeea7a23,0xa19c06e8),LL(0x0edabb63,0xd2865801),
   LL(0x6967469a,0xdb92293f),LL(0x8d8a8ed8,0x2894d839),LL(0xbbc77122,0x87c9e406),LL(0x2ea3a26a,0x8671c6f1), LL(0xd7de9853,0xe42df8d6),LL(0xb1f2bcc7,0x2e3ce346),LL(0x899d50cf,0xda601dfc),LL(0xfb1b598f,0xbfc913de),
   LL(0xe61f7908,0x81c4909f),LL(0x9bbc7b29,0x192e304f),LL(0xc104b338,0xc3ed8738),LL(0x783f5d61,0xedbe9e47), LL(0x2db30660,0x0c06e9be),LL(0xc0eb7d8e,0xda3e613f),LL(0x322e096e,0xd8fa3e97),LL(0xd336e247,0xfebd91e8),
   LL(0xdf655a49,0x8f13ccc4),LL(0x5eb20210,0xa9e00dfc),LL(0xc656b6ea,0x84631d0f),LL(0xd8c0d947,0x93a058cd), LL(0x67bd3448,0x6846904a),LL(0xf394fd5c,0x4a3d4e1a),LL(0xdb225f52,0xc102c1a5),LL(0xfc4f5e9a,0xe3455bba),
   LL(0x4b9ad1ce,0x6b36985b),LL(0x5bb7f793,0xa9818536),LL(0x48b1a416,0x6c25e1d0),LL(0x3c81bee7,0x1381dd53), LL(0x7a4a7620,0xd2a30d61),LL(0x39b8944c,0xc8412926),LL(0x7a97c33a,0x3c1c6fbe),LL(0x938664e7,0x941e541d),
   LL(0x4a34f239,0x417499e8),LL(0xb90402d5,0x15fdb83c),LL(0x433aa832,0xb75f46bf),LL(0x63215db1,0xb61e15af), LL(0xa127f89a,0xaabe59d4),LL(0x07e816da,0x5d541e0c),LL(0xa618b692,0xaaba0659),LL(0x17266026,0x55327733),
   LL(0x95f57552,0xaf53a0fc),LL(0x6cacb0c9,0x32947650),LL(0xc821be01,0x253ff58d),LL(0xa06f1146,0xb0309531), LL(0x05c2e54d,0x59bbbdf5),LL(0x26e8dd22,0x158f27ad),LL(0x397e1e53,0xcc5b7ffb),LL(0x7fc1e50d,0xae03f65b),
   LL(0x9c95f0f9,0xa9784ebd),LL(0x24640771,0x5ed9deb2),LL(0x035561c4,0x31244af7),LL(0x7ee857de,0x87332f3a), LL(0x2b9e0d88,0x09e16e9e),LL(0x56a06049,0x52d910f4),LL(0xa9592f48,0x507ed477),LL(0x2365d678,0x85cb917b),
   LL(0x4c8998d1,0xf8511c93),LL(0x730ea58f,0x2186a3f1),LL(0xb2029db0,0x50189626),LL(0x02ceb75a,0x9137a6d9), LL(0x748bc82c,0x2fe17f37),LL(0x80469f8c,0x87c2e931),LL(0xbf891aa2,0x850f71cd),LL(0x75ec3d8d,0x0ca1b89b),
   LL(0x5e1cd3cd,0x516c43aa),LL(0x9a887c28,0x89397808),LL(0xddea1f9f,0x0059c699),LL(0x8e6868f7,0x7737d6fa), LL(0x60f1524b,0x6d93746a),LL(0xba052aa7,0x36985e55),LL(0xed923ea5,0x41b1d322),LL(0x25852a11,0x3429759f),
   LL(0x092e9f41,0xbeca6ec3),LL(0x62256bbd,0x3a238c66),LL(0x70ad487d,0xd82958ea),LL(0x65610d93,0x4ac8aaf9), LL(0x5e4ccab0,0x3fa101b1),LL(0x9de14bfb,0x9bf430f2),LL(0x6531899d,0xa10f5cc6),LL(0xea8ce17d,0x590005fb),
   LL(0x24544cb6,0xc437912f),LL(0xd79ac2e3,0x9987b71a),LL(0xc058a212,0x13e3d9dd),LL(0xd2de9606,0x00075aac), LL(0x6cac8369,0x80ab508b),LL(0xf54f6c89,0x87842be7),LL(0x6bc532a4,0xa7ad663d),LL(0x78a91bc8,0x67813de7),
   LL(0xc3427239,0x5dcb61ce),LL(0xc56934d9,0x5f3c7cf0),LL(0xe3191591,0xc079e0fb),LL(0xb01aada7,0xe40896bd), LL(0x0492d25f,0x8d466791),LL(0xe7408276,0x8aeb30c9),LL(0x9287aacc,0xe9437495),LL(0x79fe03d4,0x23d4708d),
   LL(0xd0c05199,0x8cda9cf2),LL(0xfae78454,0x502fbc22),LL(0xf572a182,0xc0bda9df),LL(0x6158b372,0x5f9b71b8), LL(0x2b82dd07,0xe0f33a59),LL(0x9523032e,0x76302735),LL(0xc4505a32,0x7fe1a721),LL(0xf796409f,0x7b6e3e82),
},
/* digit=32 base_pwr=2^224 */
{
   LL(0x35d0b34a,0xe3417bc0),LL(0x8327c0a7,0x440b386b),LL(0xac0362d1,0x8fb7262d),LL(0xe0cdf943,0x2c41114c), LL(0xad95a0b1,0x2ba5cef1),LL(0x67d54362,0xc09b37a8),LL(0x01e486c9,0x26d6cdd2),LL(0x42ff9297,0x20477abf),
   LL(0x292a9287,0xa004dcb3),LL(0x77b092c7,0xddc15cf6),LL(0x806c0605,0x083a8464),LL(0x3db997b0,0x4a68df70), LL(0x05bf7dd0,0x9c134e45),LL(0x8ccf7f8c,0xa4e63d39),LL(0x41b5f8af,0xa6e6517f),LL(0xad7bc1cc,0xaa8b9342),
   LL(0x1e706ad9,0x126f35b5),LL(0xc3a9ebdf,0xb99cebb4),LL(0xbf608d90,0xa75389af),LL(0xc6c89858,0x76113c4f), LL(0x97e2b5aa,0x80de8eb0),LL(0x63b91304,0x7e1022cc),LL(0x6ccc066c,0x3bdab605),LL(0xb2edf900,0x33cbb144),
   LL(0x7af715d2,0xc4176471),LL(0xd0134a96,0xe2f7f594),LL(0xa41ec956,0x2c1873ef),LL(0x77821304,0xe4e7b4f6), LL(0x88d5374a,0xe5c8ff97),LL(0x80823d5b,0x2b915e63),LL(0xb2ee8fe2,0xea6bc755),LL(0xe7112651,0x6657624c),
   LL(0xdace5aca,0x157af101),LL(0x11a6a267,0xc4fdbcf2),LL(0xc49c8609,0xdaddf340),LL(0xe9604a65,0x97e49f52), LL(0x937e2ad5,0x9be8e790),LL(0x326e17f1,0x846e2508),LL(0x0bbbc0dc,0x3f38007a),LL(0xb11e16d6,0xcf03603f),
   LL(0x7442f1d5,0xd6f800e0),LL(0x66e0e3ab,0x475607d1),LL(0xb7c64047,0x82807f16),LL(0xa749883d,0x8858e1e3), LL(0x8231ee10,0x5859120b),LL(0x638a1ece,0x1b80e7eb),LL(0xc6aa73a4,0xcb72525a),LL(0x844423ac,0xa7cdea3d),
   LL(0xf8ae7c38,0x5ed0c007),LL(0x3d740192,0x6db07a5c),LL(0x5fe36db3,0xbe5e9c2a),LL(0x76e95046,0xd5b9d57a), LL(0x8eba20f2,0x54ac32e7),LL(0x71b9a352,0xef11ca8f),LL(0xff98a658,0x305e373e),LL(0x823eb667,0xffe5a100),
   LL(0xe51732d2,0x57477b11),LL(0x2538fc0e,0xdfd6eb28),LL(0x3b39eec5,0x5c43b0cc),LL(0xcb36cc57,0x6af12778), LL(0x06c425ae,0x70b0852d),LL(0x5c221b9b,0x6df92f8c),LL(0xce826d9c,0x6c8d4f9e),LL(0xb49359c3,0xf59aba7b),
   LL(0xda64309d,0x5c8ed8d5),LL(0x91b30704,0x61a6de56),LL(0x2f9b5808,0xd6b52f6a),LL(0x98c958a7,0x0eee4194), LL(0x771e4caa,0xcddd9aab),LL(0x78bc21be,0x83965dfd),LL(0xb3b504f5,0x02affce3),LL(0x561c8291,0x30847a21),
   LL(0x52bfda05,0xd2eb2cf1),LL(0x6197b98c,0xe0e4c4e9),LL(0xf8a1726f,0x1d35076c),LL(0x2db11e3d,0x6c06085b), LL(0x4463ba14,0x15c0c4d7),LL(0x0030238c,0x9d292f83),LL(0x3727536d,0x1311ee8b),LL(0xbeaedc1e,0xfeea86ef),
   LL(0x66131e2e,0xb9d18cd3),LL(0x80fe2682,0xf31d974f),LL(0xe4160289,0xb6e49e0f),LL(0x08e92799,0x7c48ec0b), LL(0xd1989aa7,0x818111d8),LL(0xebf926f9,0xb34fa0aa),LL(0xa245474a,0xdb5fe2f5),LL(0x3c7ca756,0xf80a6ebb),
   LL(0xafa05dd8,0xa7f96054),LL(0xfcaf119e,0x26dfcf21),LL(0x0564bb59,0xe20ef2e3),LL(0x61cb02b8,0xef4dca50), LL(0x65d30672,0xcda7838a),LL(0xfd657e86,0x8b08d534),LL(0x46d595c8,0x4c5b4395),LL(0x425cb836,0x39b58725),
   LL(0x3de9abe3,0x8ea61059),LL(0x9cdc03be,0x40434881),LL(0xcfedce8c,0x9b261245),LL(0xcf5234a1,0x78c318b4), LL(0xfde24c99,0x510bcf16),LL(0xa2c2ff5d,0x2a77cb75),LL(0x27960fb4,0x9c895c2b),LL(0xb0eda42b,0xd30ce975),
   LL(0x1a62cc26,0xfda85393),LL(0x50c0e052,0x23c69b96),LL(0xbfc633f3,0xa227df15),LL(0x1bae7d48,0x2ac78848), LL(0x187d073d,0x487878f9),LL(0x967f807d,0x6c2be919),LL(0x336e6d8f,0x765861d8),LL(0xce528a43,0x88b8974c),
   LL(0xff57d051,0x09521177),LL(0xfb6a1961,0x2ff38037),LL(0xa3d76ad4,0xfc0aba74),LL(0x25a7ec17,0x7c764803), LL(0x48879bc8,0x7532d75f),LL(0x58ce6bc1,0xea7eacc0),LL(0x8e896c16,0xc82176b4),LL(0x2c750fed,0x9a30e0b2),
   LL(0x421d3aa4,0xc37e2c2e),LL(0xe84fa840,0xf926407c),LL(0x1454e41c,0x18abc03d),LL(0x3f7af644,0x26605ecd), LL(0xd6a5eabf,0x242341a6),LL(0x216b668e,0x1edb84f4),LL(0x04010102,0xd836edb8),LL(0x945e1d8c,0x5b337ce7),
   LL(0xc055dc14,0xd2075c77),LL(0x81d89cdf,0x2a0ffa25),LL(0x6ffdcbaf,0x8ce815ea),LL(0xfb648867,0xa3428878), LL(0x884655fb,0x277699cf),LL(0x364d3e41,0xfa5b5bd6),LL(0x441e1cb7,0x01f680c6),LL(0xb70a7d67,0x3fd61e66),
   LL(0xcc78cf66,0x666ba2dc),LL(0x6fdbff77,0xb3018174),LL(0x168d4668,0x8d4dd0db),LL(0x1dab3a2a,0x259455d0), LL(0xcde3acec,0xf58564c5),LL(0x13adb276,0x77141925),LL(0x8a303f65,0x527d725d),LL(0xe6f38f7b,0x55deb6c9),
   LL(0xb1fa70fb,0xfd5bb657),LL(0xd8073a00,0xfa07f50f),LL(0xbca02500,0xf72e3aa7),LL(0x9975740d,0xf68f895d), LL(0x5cae2a6a,0x30112060),LL(0x02874842,0x01bd7218),LL(0x7ce47bd3,0x3d423891),LL(0x789544f6,0xa66663c1),
   LL(0x3272d838,0x864d05d7),LL(0xfa6295c5,0xe22924f9),LL(0x6c2fda32,0x8189593f),LL(0xb184b544,0x330d7189), LL(0xbde1f714,0x79efa62c),LL(0xe5cb1a63,0x35771c94),LL(0x641c8332,0x2f4826b8),LL(0xc8cee854,0x00a894fb),
   LL(0x36194d40,0xb4b9a39b),LL(0x77612601,0xe857a7c5),LL(0x4ecf2f58,0xf4209dd2),LL(0x5a033487,0x82b9e66d), LL(0xe4e8b9dd,0xc1e36934),LL(0xa42377d7,0xd2372c9d),LL(0x0e3ae43b,0x51dc94c7),LL(0x04474f6f,0x4c57761e),
   LL(0x1058a318,0xdcdacd0a),LL(0x78053a9a,0x369cf3f5),LL(0x31c68de2,0xc6c3de50),LL(0x3c4b6d9f,0x4653a576), LL(0xaa4e5c97,0x1688dd5a),LL(0xb7ab3c74,0x5be80aa1),LL(0xbc65c283,0x70cefe7c),LL(0x06867091,0x57f95f13),
   LL(0x4415503b,0xa39114e2),LL(0x4cbb17e9,0xc08ff7c6),LL(0xd7dec966,0x1eff674d),LL(0x53376f63,0x6d4690af), LL(0xea74237b,0xff6fe32e),LL(0xcd57508e,0xc436d17e),LL(0xedcc40fe,0x15aa28e1),LL(0x581bbb44,0x0d769c04),
   LL(0x34eaacda,0xc240b6de),LL(0x2ba0f1de,0xd9e116e8),LL(0x79438e55,0xcbe45ec7),LL(0x96f752d7,0x91787c9d), LL(0xf129ac2f,0x897f532b),LL(0x5a36e22c,0xd307b7c8),LL(0x749fb8f3,0x91940675),LL(0x157fdb28,0xd14f95d0),
   LL(0x6ae55043,0xfe51d029),LL(0x44a87de1,0x8931e98f),LL(0x09e4fee2,0xe57f1cc6),LL(0x4e072d92,0x0d063b67), LL(0xed0e4316,0x70a998b9),LL(0x306aca46,0xe74a736b),LL(0x4fda97c7,0xecf0fbf2),LL(0x3e178d93,0xa40f65cb),
   LL(0x16df4285,0x16253604),LL(0xd0c56ae2,0xb0c9babb),LL(0xcfc5cfc3,0x73032b19),LL(0x09752056,0xe497e5c3), LL(0x164bda96,0x12096bb4),LL(0xa0b74da1,0x1ee42419),LL(0x403826ba,0x8fc36243),LL(0xdc09e660,0x0c8f0069),
   LL(0xc27253c9,0x8667e981),LL(0x92b36a45,0x05a6aefb),LL(0x9cb7bb46,0xa62c4b36),LL(0x11f7027b,0x8394f375), LL(0x5f109d0f,0x747bc79c),LL(0x5b8cc60a,0xcad88a76),LL(0x58f09e68,0x80c5a66b),LL(0xf6127eac,0xe753d451),
   LL(0x5b0ec6f5,0xc44b74a1),LL(0x5289b2b8,0x47989fe4),LL(0x58d6fc73,0x745f8484),LL(0xf61c70ab,0xec362a6f), LL(0xb3a8ad41,0x070c98a7),LL(0x7b63db51,0x73a20fc0),LL(0xf44c35f4,0xed2c2173),LL(0x9acc9dca,0x8a56149d),
   LL(0x9ac6e0f4,0x98f17881),LL(0xa413b5ed,0x360fdeaf),LL(0xa300b0fd,0x0625b8f4),LL(0x5b3222d3,0xf1f4d76a), LL(0x587f76b8,0x9d6f5109),LL(0x2317fdb5,0x8b4ee08d),LL(0x8c68b095,0x88089bb7),LL(0x5808d9b9,0x95570e9a),
   LL(0x35d33ae7,0xa395c36f),LL(0x50bb5a94,0x200ea123),LL(0x0bafe84b,0x20c789bd),LL(0x0919276a,0x243ef52d), LL(0xe23ae233,0x3934c577),LL(0xa460d1ec,0xb93807af),LL(0xf8fa76a4,0xb72a53b1),LL(0xc3ca4491,0xd8914cb0),
   LL(0x3fb42622,0x2e128494),LL(0x500907d5,0x3b2700ac),LL(0x1a95ec63,0xf370fb09),LL(0x31b6dfbd,0xf8f30be2), LL(0x69e55f15,0xf2b2f8d2),LL(0xcc1323e9,0x1fead851),LL(0xd9e5eef6,0xfa366010),LL(0xe316107e,0x64d487b0),
   LL(0xd23ddc82,0x4c076b86),LL(0x7e0143f0,0x03fd344c),LL(0x317af2c5,0xa95362ff),LL(0xe18b7a4f,0x0add3db7), LL(0x8260e01b,0x9c673e3f),LL(0x54a1cc91,0xfbeb49e5),LL(0x92f2e433,0x91351bf2),LL(0x851141eb,0xc755e7ec),
   LL(0x29607745,0xc9a95139),LL(0xa26f2b28,0x0ca07420),LL(0x4bc6f9dd,0xcb2790e7),LL(0xadcaffc0,0x345bbb58), LL(0xbe0f27a2,0xc65ea38c),LL(0x641fcb56,0x67c24d7c),LL(0xa9e2c757,0x2c25f0a7),LL(0x16f16c49,0x93f5cdb0),
   LL(0xc5ee30a1,0x2ca5a9d7),LL(0xb909b729,0xd1593635),LL(0xdadeff48,0x804ce9f3),LL(0xb07c30c3,0xec464751), LL(0x9e49af6a,0x89d65ff3),LL(0x6f3d01bc,0xf2d6238a),LL(0x0bced843,0x1095561e),LL(0xc8a13fd8,0x51789e12),
   LL(0x763231df,0xd633f929),LL(0xe7cbddef,0x46df9f7d),LL(0xcb265da8,0x01c889c0),LL(0xaf4336d2,0xfce1ad10), LL(0xfc6a0a7e,0x8d110df6),LL(0x6da425dc,0xdd431b98),LL(0x1834aabe,0xcdc4aeab),LL(0x8439b7fc,0x84deb124),
   LL(0x3c2a5998,0x8796f169),LL(0x7947190d,0x9b9247b4),LL(0x11597014,0x55b9d9a5),LL(0x7b1566ee,0x7e9dd70d), LL(0xcbcd5e64,0x94ad78f7),LL(0x9bd4c032,0x0359ac17),LL(0x7cc222ae,0x3b11baaf),LL(0xba78e812,0xa6a6e284),
   LL(0x24cea1a0,0x8392053f),LL(0x33621491,0xc97bce4a),LL(0x35399ee9,0x7eb1db34),LL(0xece81ad1,0x473f78ef), LL(0xf63d3d0d,0x41d72fe0),LL(0xafab62fc,0xe620b880),LL(0x93158383,0x92096bc9),LL(0x8f896f6c,0x41a21357),
   LL(0xc7dcfcab,0x1b5ee2fa),LL(0x9546e007,0x650acfde),LL(0xb1b02e07,0xc081b749),LL(0xf9eca03d,0xda9e41a0), LL(0x175a54ab,0x013ba727),LL(0xea5d8d10,0xca0cd190),LL(0x95fd96a9,0x85ea52c0),LL(0xbc5c3940,0x2c591b9f),
   LL(0x2bad4d5f,0x6fb4d4e4),LL(0xfef0059b,0xfa4c3590),LL(0xf5122294,0x6a10218a),LL(0xa85751d1,0x9a78a81a), LL(0xa98e84e7,0x04f20579),LL(0x4997e5b5,0xfe1242c0),LL(0xca21e1e4,0xe77a273b),LL(0x9411939d,0xfcc8b1ef),
   LL(0x92d0487a,0xe20ea302),LL(0x294b91fe,0x1442dbec),LL(0xbb6b0e8f,0x1f7a4afe),LL(0x6889c318,0x1700ef74), LL(0x70f1fc62,0xf5bbffc3),LL(0x69c79cca,0x3b31d4b6),LL(0xa7f6340d,0xe8bc2aab),LL(0xa725e10a,0xb0b08ab4),
   LL(0xae340050,0x44f05701),LL(0x1cf0c569,0xba4b3016),LL(0xfbe19a51,0x5aa29f83),LL(0xb71d752e,0x1b9ed428), LL(0xeb4819f5,0x1666e54e),LL(0x9e18b75b,0x616cdfed),LL(0x3ee27b0b,0x112ed5be),LL(0x44c7de4d,0xfbf28319),
   LL(0xe0e60d84,0xd685ec85),LL(0x1db7ee78,0x68037e30),LL(0x003c4d6e,0x5b65bdcd),LL(0x93e29a6a,0x33e7363a), LL(0x08d0756c,0x995b3a61),LL(0x2faf134b,0xd727f85c),LL(0x1d337823,0xfac6edf7),LL(0x0439b8b4,0x99b9aa50),
   LL(0xe2b4e075,0x722eb104),LL(0x437c4926,0x49987295),LL(0x46a9b82d,0xb1e4c0e4),LL(0x57a006f5,0xd0cb3197), LL(0xd7808c56,0xf3de0f7d),LL(0x51f89772,0xb5c54d8f),LL(0xadbd31aa,0x500a114a),LL(0x295f6cab,0x9afaaaa6),
   LL(0x04cf667a,0x94705e21),LL(0x9d3935d7,0xfc2a811b),LL(0x6d09267c,0x560b0280),LL(0xf780e53b,0xf19ed119), LL(0x067b6269,0xf0227c09),LL(0x5caef599,0x967b8533),LL(0x68efeebc,0x155b9243),LL(0xc497bae6,0xcd6d34f5),
   LL(0x6cceb370,0x1dd8d5d3),LL(0xa78d7bf9,0x2aeac579),LL(0x70b67a62,0x5d65017d),LL(0x17c53f67,0x70c8e44f), LL(0x86a34d09,0xd1fc0950),LL(0xe7134907,0xe0fca256),LL(0x80fdd315,0xe24fa29c),LL(0xd87499ad,0x2c4acd03),
   LL(0x3b5a9ba6,0xbaaf7517),LL(0x12e51a51,0xb9cbe1f6),LL(0x5e154897,0xd88edae3),LL(0x77b66ca0,0xe4309c3c), LL(0xf67f3746,0xf5555805),LL(0xa36401ff,0x85fc37ba),LL(0xd9499a53,0xdf86e2ca),LL(0xecbc955b,0x6270b2a3),
   LL(0x974ad33b,0xafae64f5),LL(0xfe7b2df1,0x04d85977),LL(0x4ab03f73,0x2a3db3ff),LL(0x8702740a,0x0b87878a), LL(0x5a061732,0x6d263f01),LL(0xa32a1901,0xc25430ce),LL(0xdb155018,0xf7ebab3d),LL(0x63a9b78e,0x3a86f693),
   LL(0xda9f3804,0x349ae368),LL(0xa164349c,0x470f07fe),LL(0x8562baa5,0xd52f4cc9),LL(0x2b290df3,0xc74a9e86), LL(0x43471a24,0xd3a1aa35),LL(0xb8194511,0x239446be),LL(0x81dcd44d,0xbec2dd00),LL(0xc42ac82d,0xca3d7f0f),
   LL(0xfdaf4520,0x1f3db085),LL(0x4549daf2,0xbb6d3e80),LL(0x19ad5c42,0xf5969d8a),LL(0xdbfd1511,0x7052b13d), LL(0x682b9060,0x11890d1b),LL(0xac34452c,0xa71d3883),LL(0x783805b4,0xa438055b),LL(0x4725b23e,0x43241277),
   LL(0x4901bbed,0xf20cf96e),LL(0xf432a2bb,0x6419c710),LL(0xdfa9cd7d,0x57a0fbb9),LL(0x00daa249,0x589111e4), LL(0x7b60554e,0x19809a33),LL(0xede283a4,0xea5f8887),LL(0x503bfd35,0x2d713802),LL(0x585d2a53,0x151bb0af),
   LL(0x43b30ca8,0x40b08f74),LL(0xd9934583,0xe10b5bba),LL(0xb51110ad,0xe8a546d6),LL(0x28e0b6c5,0x1dd50e66), LL(0xcff2b821,0x292e9d54),LL(0x47281760,0x3882555d),LL(0x3724d6e3,0x134838f8),LL(0x22ddcda1,0xf2c679e0),
   LL(0x6d2a5768,0x40ee8815),LL(0x1c1e7e2d,0x7f227bd2),LL(0xd04ff443,0x487ba134),LL(0xc614e54b,0x76e2ff3d), LL(0xa3177ec7,0x36b88d6f),LL(0x2328fff5,0xbf731d51),LL(0x49ba158e,0x758caea2),LL(0x02938188,0x5ab8ff4c),
   LL(0x35edc56d,0x33e16056),LL(0x7e940d79,0x5a69d349),LL(0x03866dcb,0x6c4fd001),LL(0x4893cdef,0x20a38f57), LL(0xfac3a15b,0xfbf3e790),LL(0x7a4f8e6b,0x6ed7ea2e),LL(0xbc3aca86,0xa663eb4f),LL(0x080d53f7,0x22061ea5),
   LL(0xf546783f,0x2480dfe6),LL(0x5a0a641e,0xd38bc6da),LL(0x2ede8965,0xfb093cd1),LL(0xacb455cf,0x89654db4), LL(0x26e1adee,0x413cbf9a),LL(0x373294d4,0x291f3764),LL(0x648083fe,0x00797257),LL(0x208cc341,0x25f504d3),
   LL(0xc3a0ee43,0x635a8e5e),LL(0x679898ff,0x70aaebca),LL(0x5dc63d56,0x9ee9f547),LL(0xffb34d00,0xce987966), LL(0x5e26310a,0xf9f86b19),LL(0x382a8ca8,0x9e435484),LL(0xc2352fe4,0x253bcb81),LL(0x4474b571,0xa4eac8b0),
   LL(0xc1ad8cf8,0xc1b97512),LL(0x99e0b697,0x193b4e9e),LL(0x01e85df0,0x939d2716),LL(0xcd44eafd,0x4fb265b3), LL(0xe51e1ae2,0x321e7dcd),LL(0xe3d8b096,0x8e3a8ca6),LL(0x52604998,0x8de46cb0),LL(0x39072aa7,0x91099ad8),
   LL(0x93aa96b8,0x2617f91c),LL(0x7fca2e13,0x0fc8716b),LL(0x95328723,0xa7106f5e),LL(0x262e6522,0xd1c9c40b), LL(0x42b7c094,0xb9bafe86),LL(0x1543c021,0x1873439d),LL(0x5cbefd5d,0xe1baa5de),LL(0x521e8aff,0xa363fc5e),
   LL(0xf862eaac,0xefe6320d),LL(0x22c647dc,0x14419c63),LL(0x4e46d428,0x0e06707c),LL(0x4a178f8f,0xcb6c834f), LL(0xd30f917c,0x0f993a45),LL(0x9879afee,0xd4c4b049),LL(0x70500063,0xb6142a1e),LL(0xa5d9d605,0x7c9b41c3),
   LL(0x2f8ba2c7,0xbc00fc2f),LL(0x7c67aa28,0x0966eb2f),LL(0x5a786972,0x13f7b516),LL(0x8a2fbba0,0x3bfb7557), LL(0x5a2b9620,0x131c4f23),LL(0x6faf46be,0xbff3ed27),LL(0x7e172323,0x9b4473d1),LL(0x339f6246,0x421e8878),
   LL(0x25a41632,0x0fa8587a),LL(0xa35b6c93,0xc0814124),LL(0x59ebb8db,0x2b18a9f5),LL(0x76edb29c,0x264e3357), LL(0xc87c51e2,0xaf245ccd),LL(0x501e6214,0x16b3015b),LL(0x0a3882ce,0xbb31c560),LL(0xfec11e04,0x6961bb94),
   LL(0xeff7a3a0,0x3b825b8d),LL(0xb1df7326,0xbec33738),LL(0x99604a1f,0x68ad747c),LL(0x9a3bd499,0xd154c934), LL(0x1cc7a906,0xac33506f),LL(0x6c560e8f,0x73bb5392),LL(0x263e3944,0x6428fcbe),LL(0x1c387434,0xc11828d5),
   LL(0x3e4b12ff,0x3cd04be1),LL(0x2d88667c,0xc3aad9f9),LL(0x248120cf,0xc52ddcf8),LL(0x2a389532,0x985a892e), LL(0x3bb85fa0,0xfbb4b21b),LL(0x8dfc6269,0xf95375e0),LL(0x7ee2acea,0xfb4fb06c),LL(0x309c4d1f,0x6785426e),
   LL(0xd8ceb147,0x659b17c8),LL(0xb70a5554,0x9b649eee),LL(0xac6bc634,0x6b7fa0b5),LL(0x1d6e732f,0xd99fe2c7), LL(0x8d3abba2,0x30e6e762),LL(0xa797b799,0x18fee6e7),LL(0xc696464d,0x5c9d360d),LL(0x27bfde12,0xe3baeb48),
   LL(0xf23206d5,0x2bf5db47),LL(0x1d260152,0x2f6d3420),LL(0x3f8ff89a,0x17b87653),LL(0x378fa458,0x5157c30c), LL(0x2d4fb936,0x7517c5c5),LL(0xe6518cdc,0xef22f7ac),LL(0xbf847a64,0xdeb483e6),LL(0x92e0fa89,0xf5084558),
},
/* digit=33 base_pwr=2^231 */
{
   LL(0xdf7304d4,0xab9659d8),LL(0xff210e8e,0xb71bcf1b),LL(0xd73fbd60,0xa9a2438b),LL(0x5d11b4de,0x4595cd1f), LL(0x4835859d,0x9c0d329a),LL(0x7dbb6e56,0x4a0f0d2d),LL(0xdf928a4e,0xc6038e5e),LL(0x8f5ad154,0xc9429621),
   LL(0xf23f2d92,0x91213462),LL(0x60b94078,0x6cab71bd),LL(0x176cde20,0x6bdd0a63),LL(0xee4d54bc,0x54c9b20c), LL(0x9f2ac02f,0x3cd2d8aa),LL(0x206eedb0,0x03f8e617),LL(0x93086434,0xc7f68e16),LL(0x92dd3db9,0x831469c5),
   LL(0x8f981354,0x8521df24),LL(0x3588a259,0x587e23ec),LL(0xd7a0992c,0xcbedf281),LL(0x38961407,0x06930a55), LL(0xbe5bbe21,0x09320deb),LL(0x2491817f,0xa7ffa5b5),LL(0x09065160,0xe6c8b4d9),LL(0xfff6d2a9,0xac4f3992),
   LL(0x3ae9c1bd,0x7aa7a158),LL(0xe37ce240,0xe0af6d98),LL(0x28ab38b4,0xe54342d9),LL(0x0a1c98ca,0xe8b75007), LL(0xe02358f2,0xefce86af),LL(0xea921228,0x31b8b856),LL(0x0a1c67fc,0x052a1912),LL(0xe3aead59,0xb4069ea4),
   LL(0x7fa03cb3,0x3232d6e2),LL(0x0fdd7d88,0xdb938e5b),LL(0x2ccbfc5d,0x04c1d2cd),LL(0xaf3a580f,0xd2f45c12), LL(0x7883e614,0x592620b5),LL(0xbe7c5f26,0x5fd27e68),LL(0x1567e1e3,0x139e45a9),LL(0x44d8aaaf,0x2cc71d2d),
   LL(0xe36d0757,0x4a9090cd),LL(0xd9a29382,0xf722d7b1),LL(0x04b48ddf,0xfb7fb04c),LL(0xebe16f43,0x628ad2a7), LL(0x20226040,0xcd3fbfb5),LL(0x5104b6c4,0x6c34ecb1),LL(0xc903c188,0x30c0754e),LL(0x2d23cab0,0xec336b08),
   LL(0x1e206ee5,0x473d62a2),LL(0x8c49a633,0xf1e27480),LL(0xe9f6b2c3,0x87ab956c),LL(0x62b606ea,0x61830b48), LL(0xe78e815f,0x67cd6846),LL(0x4c02082a,0xfe40139f),LL(0x952ec365,0x52bbbfcb),LL(0x6b9836ab,0x74c11642),
   LL(0x558df019,0x9f51439e),LL(0xac712b27,0x230da4ba),LL(0x55185a24,0x518919e3),LL(0x84b78f50,0x4dcefcdd), LL(0xa47d4c5a,0xa7d90fb2),LL(0xb30e009e,0x55ac9abf),LL(0x74eed273,0xfd2fc359),LL(0xdbea8faf,0xb72d824c),
   LL(0x4513e2ca,0xce721a74),LL(0x38240b2c,0x0b418612),LL(0xd5baa450,0x05199968),LL(0x2b0e8c25,0xeb1757ed), LL(0x3dfac6d5,0x6ebc3e28),LL(0x48a237f5,0xb2431e2e),LL(0x52f61499,0x2acb5e23),LL(0xe06c936b,0x5558a2a7),
   LL(0xcbb13d1b,0xd213f923),LL(0x5bfb9bfe,0x98799f42),LL(0x701144a9,0x1ae8ddc9),LL(0x4c5595ee,0x0b8b3bb6), LL(0x3ecebb21,0x0ea9ef2e),LL(0x3671f9a7,0x17cb6c4b),LL(0x726f1d1f,0x47ef464f),LL(0x6943a276,0x171b9484),
   LL(0x7ef0329c,0x51a4ae2d),LL(0x91c4402a,0x08509222),LL(0xafd45bbc,0x64a61d35),LL(0x3035a851,0x38f096fe), LL(0xa1dec027,0xc7468b74),LL(0x4fc7dcba,0xe8cf10e7),LL(0xf4a06353,0xea35ff40),LL(0x8b77dd66,0x0b4c0dfa),
   LL(0xde7e5c19,0x779b8552),LL(0xc1c0256c,0xfab28609),LL(0xabd4743d,0x64f58eee),LL(0x7b6cc93b,0x4e8ef838), LL(0x4cb1bf3d,0xee650d26),LL(0x73dedf61,0x4c1f9d09),LL(0xbfb70ced,0xaef7c9d7),LL(0x1641de1e,0x1ec0507e),
   LL(0xcde45079,0xcd7e5cc7),LL(0x516ac9e4,0xde173c9a),LL(0xc170315c,0x517a8494),LL(0x91d8e8fb,0x438fd905), LL(0xc7d9630b,0x5145c506),LL(0xf47d4d75,0x6457a87b),LL(0x0d9a80e8,0xd31646bf),LL(0xcef3aabe,0x453add2b),
   LL(0xa607419d,0xc9941109),LL(0xbb6bca80,0xfaa71e62),LL(0x07c431f3,0x34158c13),LL(0x992bc47a,0x594abebc), LL(0xeb78399f,0x6dfea691),LL(0x3f42cba4,0x48aafb35),LL(0x077c04f0,0xedcd65af),LL(0xe884491a,0x1a29a366),
   LL(0x1c21f2bf,0x023a40e5),LL(0xa5057aee,0xf99a513c),LL(0xbcab072e,0xa3fe7e25),LL(0x40e32bcf,0x8568d2e1), LL(0xd3f69d9f,0x904594eb),LL(0x07affab1,0x181a9733),LL(0xb6e330f4,0xe4d68d76),LL(0xc75a7fc1,0x87a6dafb),
   LL(0xef7d9289,0x549db2b5),LL(0x197f015a,0x2480d4a8),LL(0xc40493b6,0x61d5590b),LL(0x6f780331,0x3a55b52e), LL(0x309eadb0,0x40eb8115),LL(0x92e5c625,0xdea7de5a),LL(0xcc6a3d5a,0x64d631f0),LL(0x93e8dd61,0x9d5e9d7c),
   LL(0x206d3ffc,0xf297bef5),LL(0x7d808bd4,0x23d5e033),LL(0xd24cf5ba,0x4a4f6912),LL(0x09cdaa8a,0xe4d8163b), LL(0xd3082e8e,0x0e0de9ef),LL(0x0192f360,0x4fe1246c),LL(0x4b8eee0a,0x1f900150),LL(0xf1da391b,0x5219da81),
   LL(0xf7ea25aa,0x7bf6a5c1),LL(0xfbb07d5f,0xd165e6bf),LL(0x89e78671,0xe3539361),LL(0x2bac4219,0xa3fcac89), LL(0xf0baa8ab,0xdfab6fd4),LL(0xe2c1c2e5,0x5a4adac1),LL(0x40d85849,0x6cd75e31),LL(0x19b39181,0xce263fea),
   LL(0x07032c72,0xcb6803d3),LL(0x790968c8,0x7f40d5ce),LL(0xdce978f0,0xa6de86bd),LL(0x368f751c,0x25547c4f), LL(0x65fb2a9e,0xb1e685fd),LL(0x1eb9179c,0xce69336f),LL(0x12504442,0xb15d1c27),LL(0xb911a06b,0xb7df465c),
   LL(0x315980cd,0xb8d804a3),LL(0xfa3bebf7,0x693bc492),LL(0x2253c504,0x3578aeee),LL(0xcd2474a2,0x158de498), LL(0xcfda8368,0x1331f5c7),LL(0x78d7177e,0xd2d7bbb3),LL(0xf3c1e46e,0xdf61133a),LL(0xd30e7be8,0x5836ce7d),
   LL(0x94f834cb,0x83084f19),LL(0x429ed782,0xd35653d4),LL(0x59e58243,0xa542f16f),LL(0x0470a22d,0xc2b52f65), LL(0x18f23d96,0xe3b6221b),LL(0x3f5252b4,0xcb05abac),LL(0x87d61402,0xca00938b),LL(0x411933e4,0x2f186cdd),
   LL(0x9a29a5c5,0xe042ece5),LL(0x3b6c8402,0xb19b3c07),LL(0x19d92684,0xc97667c7),LL(0xebc66372,0xb5624622), LL(0x3c04fa02,0x0cb96e65),LL(0x8eaa39aa,0x83a7176c),LL(0xeaa1633f,0x2033561d),LL(0x4533df73,0x45a9d086),
   LL(0x3dc090bc,0xe0542c1d),LL(0xaa59c167,0x82c996ef),LL(0x0ee7fc4d,0xe3f735e8),LL(0x7c35db79,0x7b179393), LL(0xf8c5dbfd,0xb6419e25),LL(0x1f327b04,0x4d9d7a1e),LL(0x298dfca8,0x979f6f9b),LL(0x8de9366a,0xc7c5dff1),
   LL(0x04c82bdd,0x1b7a588d),LL(0xf8319dfd,0x68005534),LL(0xd8eb9580,0xde8a55b5),LL(0x8d5bca81,0x5ea886da), LL(0x252a0b4d,0xe8530a01),LL(0x35eaa0a1,0x1bffb4fe),LL(0xd8e99563,0x2ad828b1),LL(0x95f9cd87,0x7de96ef5),
   LL(0xd77d970c,0x4abb2d0c),LL(0xd33ef9cb,0x03cfb933),LL(0x8b211fe9,0xb0547c01),LL(0xa56ed1c6,0x2fe64809), LL(0xc2ac98cc,0xcb7d5624),LL(0x1a393e33,0x2a1372c0),LL(0x29660521,0xc8d1ec1c),LL(0xb37ac3e9,0xf3d31b04),
   LL(0x5ece6e7c,0xa29ae9df),LL(0x0facfb55,0x0603ac8f),LL(0xdda233a5,0xcfe85b7a),LL(0xbd75f0b8,0xe618919f), LL(0x99bf1603,0xf555a3d2),LL(0xf184255a,0x1f43afc9),LL(0x319a3e02,0xdcdaf341),LL(0x03903a39,0xd3b117ef),
   LL(0x65d1d131,0xe095da13),LL(0xc37ad03e,0x86f16367),LL(0x462cd8dd,0x5f37389e),LL(0xd67a60e6,0xc103fa04), LL(0xf4b478f0,0x57c34344),LL(0xe117c98d,0xce91edd8),LL(0x231fc12e,0x001777b0),LL(0xb207bccb,0x11ae47f2),
   LL(0x20f8a242,0xd983cf8d),LL(0xf22e1ad8,0x7aff5b1d),LL(0x7fc4feb3,0x68fd11d0),LL(0xb0f1c3e1,0x5d53ae90), LL(0xec041803,0x50fb7905),LL(0x14404888,0x85e3c977),LL(0xac628d8f,0x0e67faed),LL(0x6668532c,0x2e865150),
   LL(0x6a67a6b0,0x15acaaa4),LL(0xb25cec41,0xf4cdee25),LL(0xe4c6701e,0x49ee565a),LL(0xfc7d63d8,0x2a04ca66), LL(0xef0543fb,0xeb105018),LL(0xd1b0d81d,0xf709a4f5),LL(0x2915d333,0x5b906ee6),LL(0x96f1f0ab,0xf4a87412),
   LL(0x4d82f4c2,0xb6b82fa7),LL(0x6804efb3,0x90725a60),LL(0xadc3425e,0xbc82ec46),LL(0x2787843e,0xb7b80581), LL(0xdd1fc74c,0xdf46d91c),LL(0xe783a6c4,0xdc1c62cb),LL(0x1a04cbba,0x59d1b9f3),LL(0x95e40764,0xd87f6f72),
   LL(0x317f4a76,0x02b4cfc1),LL(0x91036bce,0x8d2703eb),LL(0xa5e72a56,0x98206cc6),LL(0xcf53fb0f,0x57be9ed1), LL(0xef0b17ac,0x09374571),LL(0xd9181b38,0x74b2655e),LL(0x89935d0e,0xc8f80ea8),LL(0x91529936,0xc0d9e942),
   LL(0x1e84e0e5,0x19686041),LL(0xaea34c93,0xa5db84d3),LL(0x7073a732,0xf9d5bb19),LL(0x6bcfd7c0,0xb8d2fe56), LL(0xf3eb82fa,0x45775f36),LL(0xfdff8b58,0x8cb20ccc),LL(0x8374c110,0x1659b65f),LL(0x330c789a,0xb8b4a422),
   LL(0x6fe8208b,0x75e3c3ea),LL(0x286e78fe,0xbd74b9e4),LL(0xd7d93a1a,0x0be2e81b),LL(0xdd0a5aae,0x7ed06e27), LL(0x6be8b800,0x721f5a58),LL(0xd846db28,0x428299d1),LL(0x5be88ed3,0x95cb8e6b),LL(0x1c034e11,0xc3186b23),
   LL(0x8977d99b,0xa6312c9e),LL(0x83f531e7,0xbe944331),LL(0x18d3b1d4,0x8232c0c2),LL(0xe1247b73,0x617aae8b), LL(0x282aec3b,0x40153fc4),LL(0xf7b8f823,0xc6063d2f),LL(0x3304f94c,0x68f10e58),LL(0xee676346,0x31efae74),
   LL(0x40a9b97c,0xbadb6c6d),LL(0x4f666256,0x14702c63),LL(0x5184b2e3,0xdeb954f1),LL(0x94b6ca40,0x5184a526), LL(0x003c32ea,0xfff05337),LL(0x205974c7,0x5aa374dd),LL(0x4b0dd71a,0x9a763854),LL(0xdeb947ec,0x459cd27f),
   LL(0x459c2b92,0xa6e28161),LL(0x75ee8ef5,0x2f020fa8),LL(0x30b06310,0xb132ec2d),LL(0xbc6a4530,0xc3e15899), LL(0xaa3f451a,0xdc5f53fe),LL(0xc2d9acac,0x3a3c7f23),LL(0x6b27e58b,0x2ec2f892),LL(0xd742799f,0x68466ee7),
   LL(0x1fa26613,0x98324dd4),LL(0xbdc29d63,0xa2dc6dab),LL(0xd712d657,0xf9675faa),LL(0x21fd8d15,0x813994be), LL(0xfd4f7553,0x5ccbb722),LL(0xf3a36b20,0x5135ff8b),LL(0x69559df5,0x44be28af),LL(0x9d41bf30,0x40b65bed),
   LL(0x3734e520,0xd98bf2a4),LL(0x209bdcba,0x5e3abbe3),LL(0xbc945b35,0x77c76553),LL(0xc6ef14aa,0x5331c093), LL(0x76b60c80,0x518ffe29),LL(0x7ace16f8,0x2285593b),LL(0xbe2b9784,0xab1f64cc),LL(0xab2421b6,0xe8f2c0d9),
   LL(0xc1df065c,0x617d7174),LL(0x5f6578fa,0xafeeb5ab),LL(0x263b54a8,0x16ff1329),LL(0xc990dce3,0x45c55808), LL(0xecc8c177,0x42eab6c0),LL(0x5982ecaa,0x799ea9b5),LL(0xb607ef8e,0xf65da244),LL(0x32a3fc2c,0x8ab226ce),
   LL(0x7ea973dc,0x745741e5),LL(0x20888f2e,0x5c00ca70),LL(0x45fd9cf1,0x7cdce3cf),LL(0x5507f872,0x8a741ef1), LL(0x196b4cec,0x47c51c2f),LL(0xc97ea618,0x70d08e43),LL(0x15b18a2b,0x930da15c),LL(0x2f610514,0x33b6c678),
   LL(0x07ac9794,0xc662e4f8),LL(0xba06cb79,0x1eccf050),LL(0xe7d954e5,0x1ff08623),LL(0x24cf71c3,0x6ef2c5fb), LL(0x67978453,0xb2c063d2),LL(0x1d654af8,0xa0cf3796),LL(0x7ebdaa37,0x7cb242ea),LL(0xb86747e0,0x206e0b10),
   LL(0xd5ecfefc,0x481dae5f),LL(0xc2bff8fc,0x07084fd8),LL(0xea324596,0x8040a01a),LL(0xd4de4036,0x4c646980), LL(0xd65abfc3,0x9eb8ab4e),LL(0x13541ec7,0xe01cb91f),LL(0xfd695012,0x8f029adb),LL(0x3c7569ec,0x9ae28483),
   LL(0xa66d80a1,0xa5614c9e),LL(0x75f5f911,0x680a3e44),LL(0xceba4fc1,0x0c07b14d),LL(0xa13071c1,0x891c285b), LL(0x799ece3c,0xcac67ceb),LL(0x41e07e27,0x29b910a9),LL(0xf2e43123,0x66bdb409),LL(0x7ac9ecbe,0x06f8b137),
   LL(0x38547090,0x5981fafd),LL(0x85e3415d,0x19ab8b9f),LL(0xc7e31b27,0xfc28c194),LL(0x6fbcbb42,0x843be0aa), LL(0xa6db836c,0xf3b1ed43),LL(0x01a45c05,0x2a1330e4),LL(0x95c1a377,0x4f19f3c5),LL(0x44b5ee33,0xa85f39d0),
   LL(0x4ae52834,0x3da18e6d),LL(0x7423dcb0,0x5a403b39),LL(0xf2374aef,0xbb555e0a),LL(0x1e8ca111,0x2ad599c4), LL(0x014b3bf8,0x1b3a2fb9),LL(0xf66d5007,0x73092684),LL(0xc4340102,0x079f1426),LL(0x8fddf4de,0x1827cf81),
   LL(0xf10ff927,0xc83605f6),LL(0x23739fc6,0xd3871451),LL(0xcac1c2cc,0x6d163450),LL(0xa2ec1ac5,0x6b521296), LL(0x6e3cb4a5,0x0606c4f9),LL(0x778abff7,0xe47d3f41),LL(0xbe8e3a45,0x425a8d5e),LL(0xa6102160,0x53ea9e97),
   LL(0x39cbb688,0x477a106e),LL(0xf3386d32,0x532401d2),LL(0xb1b9b421,0x8e564f64),LL(0x81dad33f,0xca9b8388), LL(0x2093913e,0xb1422b4e),LL(0x69bc8112,0x533d2f92),LL(0xebe7b2c7,0x3fa017be),LL(0xcaf197c6,0xb2767c4a),
   LL(0xaedbae9f,0xc925ff87),LL(0x36880a54,0x7daf0eb9),LL(0x9c4d0e71,0x9284ddf5),LL(0x316f8cf5,0x1581cf93), LL(0x3ac1f452,0x3eeca887),LL(0xfb6aeffe,0xb417fce9),LL(0xeefb8dc3,0xa5918046),LL(0x02209400,0x73d318ac),
   LL(0x728693e5,0xe800400f),LL(0x339927ed,0xe87d814b),LL(0x57ea9910,0x93e94d3b),LL(0x2245fb69,0xff8a35b6), LL(0x7f200d34,0x043853d7),LL(0x0f653ce1,0x470f1e68),LL(0x59a06379,0x81ac05bd),LL(0x03930c29,0xa14052c2),
   LL(0x26bc2797,0x6b72fab5),LL(0x99f16771,0x13670d16),LL(0x1e3e48d1,0x00170052),LL(0xb7adf678,0x978fe401), LL(0xd41c5dd4,0x55ecfb92),LL(0xc7b27da5,0x5ff8e247),LL(0x013fb606,0xe7518272),LL(0x2f547a3c,0x5768d7e5),
   LL(0x60017a5f,0xbb24eaa3),LL(0x9c64ce9b,0x6b18e6e4),LL(0x103dde07,0xc225c655),LL(0x7592f7ea,0xfc3672ae), LL(0xd06283a1,0x9606ad77),LL(0xe4d59d99,0x542fc650),LL(0x2a40e7c2,0xabb57c49),LL(0xa8db9f55,0xac948f13),
   LL(0xb04465c3,0x6d4c9682),LL(0x6468bd15,0xe3d062fa),LL(0x5f318d7e,0xa51729ac),LL(0x9eb6fc95,0x1fc87df6), LL(0x0591f652,0x63d146a8),LL(0x589621aa,0xa861b8f7),LL(0xce31348c,0x59f5f15a),LL(0x440da6da,0x8f663391),
   LL(0xb591ffa3,0xcfa778ac),LL(0x4cdfebce,0x027ca9c5),LL(0x444ea6b3,0xbe8e05a5),LL(0xa78d8254,0x8aab4e69), LL(0xb474d6b8,0x2437f04f),LL(0x045b3855,0x6597ffd4),LL(0xca47ecaa,0xbb0aea4e),LL(0x85c7ebfc,0x568aae83),
   LL(0xc73b2383,0x0e966e64),LL(0xd17d8762,0x49eb3447),LL(0x8da05dab,0xde107821),LL(0x016b7236,0x443d8baa), LL(0xea7610d6,0x163b63a5),LL(0xce1ca979,0xe47e4185),LL(0x80baa132,0xae648b65),LL(0x0e0d5b64,0xebf53de2),
   LL(0xd3c8c1ca,0x8d3bfcb4),LL(0x5d04b309,0x0d914ef3),LL(0x3de7d395,0x55ef6415),LL(0x26b850e8,0xbde1666f), LL(0xd449ab19,0xdbe1ca6e),LL(0xe89a2672,0x8902b322),LL(0xdacb7a53,0xb1674b7e),LL(0xf52523ff,0x8e9faf6e),
   LL(0x9a85788b,0x6ba535da),LL(0xbd0626d4,0xd21f03ae),LL(0xe873dc64,0x099f8c47),LL(0x018ec97e,0xcda8564d), LL(0xde92c68c,0x3e8d7a5c),LL(0x73323cc4,0x78e035a1),LL(0xf880ff7c,0x3ef26275),LL(0x273eedaa,0xa4ee3dff),
   LL(0xaf4e18f8,0x58823507),LL(0x0672f328,0x967ec9b5),LL(0x559d3186,0x9ded19d9),LL(0x6cdce39c,0x5e2ab3de), LL(0x11c226df,0xabad6e4d),LL(0x87723014,0xf9783f43),LL(0x1a885719,0x9a49a0cf),LL(0x90da9dbf,0xfc0c1a5a),
   LL(0x571d92ac,0x8bbaec49),LL(0x4692517f,0x569e85fe),LL(0xa14ea4af,0x8333b014),LL(0x12e5c5ad,0x32f2a62f), LL(0x06d89b85,0x98c2ce3a),LL(0x2ff77a08,0xb90741aa),LL(0x01f795a2,0x2530defc),LL(0x84b3c199,0xd6e5ba0b),
   LL(0x12e4c936,0x7d8e8451),LL(0xbd0be17b,0xae419f7d),LL(0x22262bc9,0xa583fc8c),LL(0x91bfe2bd,0x6b842ac7), LL(0x440d6827,0x33cef4e9),LL(0xef81fb14,0x5f69f4de),LL(0x234fbb92,0xf16cf6f6),LL(0xd9e7e158,0x76ae3fc3),
   LL(0xe9740b33,0x4e89f6c2),LL(0x4962d6a1,0x677bc85d),LL(0x68d10d15,0x6c6d8a7f),LL(0x0257b1cd,0x5f9a7224), LL(0x4ad85961,0x7096b916),LL(0xe657ab4a,0x5f8c47f7),LL(0xf7461d7e,0xde57d7d0),LL(0x80ce5ee2,0x7eb6094d),
   LL(0x34190547,0x0b1e1dfd),LL(0xf05dd150,0x8a394f43),LL(0x97df44e6,0x0a9eb24d),LL(0x87675719,0x78ca06bf), LL(0x6ffeec22,0x6f0b3462),LL(0x36cdd8fb,0x9d91bcea),LL(0xa105be47,0xac83363c),LL(0x069710e3,0x81ba76c1),
   LL(0x28c682c6,0x3d1b24cb),LL(0x8612575b,0x27f25228),LL(0xe8e66e98,0xb587c779),LL(0x405eb1fe,0x7b0c03e9), LL(0x15b548e7,0xfdf0d030),LL(0x38b36af7,0xa8be76e0),LL(0x4f310c40,0x4cdab04a),LL(0xf47ecaec,0x6287223e),
   LL(0x8b399320,0x678e6055),LL(0xc01e4646,0x61fe3fa6),LL(0x03261a5e,0xc482866b),LL(0x5c2f244a,0xdfcf45b8), LL(0x2f684b43,0x8fab9a51),LL(0xc7220a66,0xf796c654),LL(0xf5afa58f,0x1d90707e),LL(0x4fdbe0de,0x2c421d97),
   LL(0xaf2ebc2f,0xc4f4cda3),LL(0xcb4efe24,0xa0af843d),LL(0x9ccd10b1,0x53b857c1),LL(0x914d3e04,0xddc9d1eb), LL(0x62771deb,0x7bdec8bb),LL(0x91c5aa81,0x829277aa),LL(0x832391ae,0x7af18dd6),LL(0xc71a84ca,0x1740f316),
},
/* digit=34 base_pwr=2^238 */
{
   LL(0xeeaf8c49,0x8928e99a),LL(0x6e24d728,0xee7aa73d),LL(0xe72b156c,0x4c5007c2),LL(0xed408a1d,0x5fcf57c5), LL(0xb6057604,0x9f719e39),LL(0xc2868bbf,0x7d343c01),LL(0x7e103e2d,0x2cca254b),LL(0xf131bea2,0xe6eb38a9),
   LL(0x8be762b4,0xb33e624f),LL(0x058e3413,0x2a9ee4d1),LL(0x67d805fa,0x968e6369),LL(0x7db8bfd7,0x9848949b), LL(0xd23a8417,0x5308d7e5),LL(0xf3e29da5,0x892f3b1d),LL(0x3dee471f,0xc95c139e),LL(0xd757e089,0x8631594d),
   LL(0xde918dcc,0xe0c82a3c),LL(0x26fdcf4b,0x2e7b5994),LL(0x32cb1b2d,0x82c50249),LL(0x7657ae07,0xea613a9d), LL(0xf1fdc9f7,0xc2eb5f6c),LL(0x879fe682,0xb6eae8b8),LL(0x591cbc7f,0x253dfee0),LL(0x3e1290e6,0x000da713),
   LL(0x1f095615,0x1083e2ea),LL(0x14e68c33,0x0a28ad77),LL(0x3d8818be,0x6bfc0252),LL(0xf35850cd,0xb585113a), LL(0x30df8aa1,0x7d935f0b),LL(0x4ab7e3ac,0xaddda07c),LL(0x552f00cb,0x92c34299),LL(0x2909df6c,0xc33ed1de),
   LL(0x80e87766,0x22c2195d),LL(0x9ddf4ac0,0x9e99e6d8),LL(0x65e74934,0x09642e4e),LL(0xff1ff241,0x2610ffa2), LL(0x751c8159,0x4d1d47d4),LL(0xaf3a9363,0x697b4985),LL(0x87477c33,0x0318ca46),LL(0x9441eff3,0xa90cb565),
   LL(0x36f024cb,0x58bb3848),LL(0x36016168,0x85be1f77),LL(0xdc7e07f1,0x6c59587c),LL(0xaf1d8f02,0x191be071), LL(0xcca5e55c,0xbf169fa5),LL(0xf7d04eac,0x3864ba3c),LL(0x8d7d05db,0x915e367f),LL(0xa6549e5d,0xb48a876d),
   LL(0x580e40a2,0xef89c656),LL(0x728068bc,0xf194ed8c),LL(0xa47990c9,0x74528045),LL(0x5e1a4649,0xf53fc7d7), LL(0x78593e7d,0xbec5ae9b),LL(0x41db65d7,0x2cac4ee3),LL(0x04a3d39b,0xa8c1eb24),LL(0x03f8f3ef,0x53b7d634),
   LL(0x3e07113c,0x2dc40d48),LL(0x7d8b63ae,0x6e4a5d39),LL(0x79684c2b,0x5582a94b),LL(0x622da26c,0x932b33d4), LL(0x0dbbf08d,0xf534f651),LL(0x64c23a52,0x211d07c9),LL(0xee5bdc9b,0x0eeece0f),LL(0xf7015558,0xdf178168),
   LL(0x0a712229,0xd4294635),LL(0x09273f8c,0x93cbe448),LL(0x8f13bc83,0x00b095ef),LL(0x8798978c,0xbb741972), LL(0x56dbe6e7,0x9d7309a2),LL(0x5a5d39ec,0xe578ec56),LL(0x851f9a31,0x3961151b),LL(0xe5709eb4,0x2da7715d),
   LL(0x53dfabf0,0x867f3017),LL(0xb8e39259,0x728d2078),LL(0x815d9958,0x5c75a0cd),LL(0x16603be1,0xf84867a6), LL(0x70e35b1c,0xc865b13d),LL(0x19b03e2c,0x02414468),LL(0xac1f3121,0xe46041da),LL(0x6f028a7c,0x7c9017ad),
   LL(0x0a482873,0xabc96de9),LL(0xb77e54d4,0x4265d6b1),LL(0xa57d88e7,0x68c38e79),LL(0x9ce82de3,0xd461d766), LL(0x64a7e489,0x817a9ec5),LL(0xa0def5f2,0xcc5675cd),LL(0x985d494e,0x9a00e785),LL(0x1b03514a,0xc626833f),
   LL(0x83cdd60e,0xabe7905a),LL(0xa1170184,0x50602fb5),LL(0xb023642a,0x689886cd),LL(0xa6e1fb00,0xd568d090), LL(0x0259217f,0x5b1922c7),LL(0xc43141e4,0x93831cd9),LL(0x0c95f86e,0xdfca3587),LL(0x568ae828,0xdec2057a),
   LL(0xf98a759a,0xc44ea599),LL(0xf7c23c1d,0x55a0a7a2),LL(0x94c4f687,0xd5ffb6e6),LL(0x12848478,0x3563cce2), LL(0xe7b1fbe1,0x812b3517),LL(0x4f7338e0,0x8a7dc979),LL(0x52d048db,0x211ecee9),LL(0xc86ea3b8,0x2eea4056),
   LL(0xba772b34,0xd8cb68a7),LL(0x5f4e2541,0xe16ed341),LL(0x0fec14db,0x9b32f6a6),LL(0x391698be,0xeee376f7), LL(0x83674c02,0xe9a7aa17),LL(0x5843022a,0x65832f97),LL(0x5ba4990f,0x29f3a8da),LL(0xfb8e3216,0x79a59c3a),
   LL(0xbd19bb16,0x9cdc4d2e),LL(0xb3262d86,0xc6c7cfd0),LL(0x969c0b47,0xd4ce14d0),LL(0x13e56128,0x1fa352b7), LL(0x973db6d3,0x383d55b8),LL(0xe8e5b7bf,0x71836850),LL(0xe6bb571f,0xc7714596),LL(0x2d5b2dd2,0x259df31f),
   LL(0x913cc16d,0x568f8925),LL(0xe1a26f5a,0x18bc5b6d),LL(0xf5f499ae,0xdfa413be),LL(0xc3f0ae84,0xf8835dec), LL(0x65a40ab0,0xb6e60bd8),LL(0x194b377e,0x65596439),LL(0x92084a69,0xbcd85625),LL(0x4f23ede0,0x5ce433b9),
   LL(0x6ad65143,0xe8e8f04f),LL(0xd6e14af6,0x11511827),LL(0x8295c0c7,0x3d390a10),LL(0x621eba16,0x71e29ee4), LL(0x63717b46,0xa588fc09),LL(0xe06ad4a2,0x02be02fe),LL(0x04c22b22,0x931558c6),LL(0x12f3c849,0xbb4d4bd6),
   LL(0x20efd662,0x54a4f496),LL(0xc5952d14,0x92ba6d20),LL(0xcc9784c2,0x2db8ea1e),LL(0x4b353644,0x81cc10ca), LL(0x4b4d7f6c,0x40b570ad),LL(0x84a1dcd2,0x5c9f1d96),LL(0x3147e797,0x01379f81),LL(0x2bd499f5,0xe5c6097b),
   LL(0x328e5e20,0x40dcafa6),LL(0x54815550,0xf7b5244a),LL(0x47bfc978,0xb9a4f118),LL(0xd25825b1,0x0ea0e79f), LL(0x646c7ecf,0xa50f96eb),LL(0x446dea9d,0xeb811493),LL(0xdfabcf69,0x2af04677),LL(0xc713f6e8,0xbe3a068f),
   LL(0x42e06189,0x860d523d),LL(0x4e3aff13,0xbf077941),LL(0xc1b20650,0x0b616dca),LL(0x2131300d,0xe66dd6d1), LL(0xff99abde,0xd4a0fd67),LL(0xc7aac50d,0xc9903550),LL(0x7c46b2d7,0x022ecf8b),LL(0x3abf92af,0x3333b1e8),
   LL(0x6c491c14,0x11cc113c),LL(0x80dd3f88,0x05976688),LL(0x29d932ed,0xf5b4d9e7),LL(0xa2c38b6d,0xe982aad8), LL(0x8be0dcf0,0x6f925347),LL(0x65ca53f2,0x700080ae),LL(0x443ca77f,0xd8131156),LL(0xec51f984,0xe92d6942),
   LL(0x85dfe9ae,0xd2a08af8),LL(0x4d2a86ca,0xd825d9a5),LL(0x39dff020,0x2c53988d),LL(0x430cdc40,0xf38b135a), LL(0x62a7150b,0x0c918ae0),LL(0x0c340e9b,0xf31fd8de),LL(0x4dbbf02e,0xafa0e7ae),LL(0x5eba6239,0x5847fb2a),
   LL(0xdccbac8b,0x6b1647dc),LL(0x06f485c8,0xb642aa78),LL(0x7038ecdf,0x873f3765),LL(0xfa49d3fe,0x2ce5e865), LL(0xc98c4400,0xea223788),LL(0xf1fa5279,0x8104a8cd),LL(0x06becfd7,0xbcf7cc7a),LL(0xc8f974ae,0x49424316),
   LL(0x84d6365d,0xc0da65e7),LL(0x8f759fb8,0xbcb7443f),LL(0x7ae81930,0x35c712b1),LL(0x4c6e08ab,0x80428dff), LL(0xa4faf843,0xf19dafef),LL(0xffa9855f,0xced8538d),LL(0xbe3ac7ce,0x20ac409c),LL(0x882da71e,0x358c1fb6),
   LL(0xfd349961,0xafa9c0e5),LL(0x8421c2fc,0x2b2cfa51),LL(0xf3a28d38,0x2a80db17),LL(0x5d138e7e,0xa8aba539), LL(0x6e96eb8d,0x52012d1d),LL(0xcbaf9622,0x65d8dea0),LL(0xb264f56c,0x57735447),LL(0x1b6c8da2,0xbeebef3f),
   LL(0xce785254,0xfc346d98),LL(0xbb64a161,0xd50e8d72),LL(0x49794add,0xc03567c7),LL(0x752c7ef6,0x15a76065), LL(0x961f23d6,0x59f3a222),LL(0x73ecc0b0,0x378e4438),LL(0x5a82fde4,0xc74be434),LL(0xd8b9cf34,0xae509af2),
   LL(0x577f44a1,0x4a61ee46),LL(0xb611deeb,0xe09b748c),LL(0xf5f7b884,0xc0481b2c),LL(0x61acfa6b,0x35626678), LL(0xbf8d21e6,0x37f4c518),LL(0xb205a76d,0x22d96531),LL(0x954073c0,0x37fb85e1),LL(0x65b3a567,0xbceafe4f),
   LL(0xbe42a582,0xefecdef7),LL(0x65046be6,0xd3fc6080),LL(0x09e8dba9,0xc9af13c8),LL(0x641491ff,0x1e6c9847), LL(0xd30c31f7,0x3b574925),LL(0xac2a2122,0xb7eb72ba),LL(0xef0859e7,0x776a0dac),LL(0x21900942,0x06fec314),
   LL(0xf8c22049,0x2464bc10),LL(0x875ebf69,0x9bfbcce7),LL(0x4336326b,0xd7a88e2a),LL(0x5bc2acfa,0xda05261c), LL(0xeba7efc8,0xc29f5bdc),LL(0x25dbbf2e,0x471237ca),LL(0x2975f127,0xa72773f2),LL(0x04d0b326,0xdc744e8e),
   LL(0xa56edb73,0x38a7ed16),LL(0x2c007e70,0x64357e37),LL(0x5080b400,0xa167d15b),LL(0x23de4be1,0x07b41164), LL(0x74c89883,0xb2d91e32),LL(0x2882e7ed,0x3c162821),LL(0x7503e482,0xad6b36ba),LL(0x0ea34331,0x48434e8e),
   LL(0x2c7ae0b9,0x79f4f24f),LL(0x1939b44a,0xc46fbf81),LL(0x56595eb1,0x76fefae8),LL(0xcd5f29c7,0x417b66ab), LL(0xc5ceec20,0x5f2332b2),LL(0xe1a1cae2,0xd69661ff),LL(0x9b0286e6,0x5ede7e52),LL(0xe276b993,0x9d062529),
   LL(0x7e50122b,0x324794b0),LL(0x4af07ca5,0xdd744f8b),LL(0xd63fc97b,0x30a12f08),LL(0x76626d9d,0x39650f1a), LL(0x1fa38477,0x101b47f7),LL(0xd4dc124f,0x3d815f19),LL(0xb26eb58a,0x1569ae95),LL(0x95fb1887,0xc3cde188),
   LL(0xf9539a48,0x54e9f37b),LL(0x7408c1a5,0xb0100e06),LL(0xea580cbb,0x821d9811),LL(0x86e50c56,0x8af52d35), LL(0xdbbf698b,0xdfbd9d47),LL(0x03dc1c73,0x2961a1ea),LL(0xe76a5df8,0x203d38f8),LL(0x6def707a,0x08a53a68),
   LL(0x1bee45d4,0x26eefb48),LL(0x3c688036,0xb3cee346),LL(0xc42f2469,0x463c5315),LL(0x81378162,0x19d84d2e), LL(0x1c4d349f,0x22d7c3c5),LL(0x163d59c5,0x65965844),LL(0xb8abceae,0xcf198c56),LL(0x628559d5,0x6fb1fb1b),
   LL(0x07bf8fe3,0x8bbffd06),LL(0x3467734b,0x46259c58),LL(0x35f7f0d3,0xd8953cea),LL(0xd65b0ff1,0x1f0bece2), LL(0xf3c72914,0xf7d5b4b3),LL(0x3cb53389,0x29e8ea95),LL(0x836b6d46,0x4a365626),LL(0xea174fde,0xe849f910),
   LL(0xf4737f21,0x7ec62fbb),LL(0x6209f5ac,0xd8dba5ab),LL(0xa5f9adbe,0x24b5d7a9),LL(0xa61dc768,0x707d28f7), LL(0xcaa999ea,0x7711460b),LL(0x1c92e4cc,0xba7b174d),LL(0x18d4bf2d,0x3c4bab66),LL(0xeb8bd279,0xb8f0c980),
   LL(0x324b4737,0x024bea9a),LL(0x32a83bca,0xfba9e423),LL(0xa232dced,0x6e635643),LL(0x2571c8ba,0x99619367), LL(0x54b7032b,0xe8c9f357),LL(0x2442d54a,0xf936b3ba),LL(0x8290c65a,0x2263f0f0),LL(0xee2c7fdb,0x48989780),
   LL(0x13d4f95e,0xadc5d55a),LL(0xad9b8500,0x737cff85),LL(0x8a73f43d,0x271c557b),LL(0xe18bc476,0xbed617a4), LL(0x7dfd8ab2,0x66245401),LL(0x3a2870aa,0xae7b89ae),LL(0x23a7e545,0x1b555f53),LL(0xbe057e4c,0x6791e247),
   LL(0x324fa34d,0x860136ad),LL(0x4cbeae28,0xea111447),LL(0xbedd3299,0x023a4270),LL(0xc1c35c34,0x3d5c3a7f), LL(0x8d0412d2,0xb0f6db67),LL(0xfcdc6b9a,0xd92625e2),LL(0x4e28a982,0x92ae5ccc),LL(0x47a3ce7e,0xea251c36),
   LL(0x790691bf,0x9d658932),LL(0x06b736ae,0xed610589),LL(0xc0d63b6e,0x712c2f04),LL(0xc63d488f,0x5cf06fd5), LL(0xd9588e41,0x97363fac),LL(0x2b93257e,0x1f9bf762),LL(0x667acace,0xa9d1ffc4),LL(0x0a061ecf,0x1cf4a1aa),
   LL(0xdc1818d0,0x40e48a49),LL(0xa3621ab0,0x0643ff39),LL(0xe39ef639,0x5768640c),LL(0x04d86854,0x1fc099ea), LL(0xeccd28fd,0x9130b9c3),LL(0x7eec54ab,0xd743cbd2),LL(0xe5b475b6,0x052b146f),LL(0x900a7d1f,0x058d9a82),
   LL(0x91262b72,0x65e02292),LL(0xbb0edf03,0x96f924f9),LL(0xfe206842,0x5cfa59c8),LL(0x5eafa720,0xf6037004), LL(0x18d7dd96,0x5f30699e),LL(0xcbab2495,0x381e8782),LL(0xdd8be949,0x91669b46),LL(0x26aae8ef,0xb40606f5),
   LL(0xfc6751a4,0x2812b839),LL(0xfba800ef,0x16196214),LL(0x4c1a2875,0x4398d5ca),LL(0x653d8349,0x720c00ee), LL(0xd820007c,0xc2699eb0),LL(0xa39b5825,0x880ee660),LL(0x471f6984,0x70694694),LL(0xe3dda99a,0xf7d16ea8),
   LL(0xc0519a23,0x28d675b2),LL(0x4f6952e3,0x9ebf94fe),LL(0xa2294a8a,0xf28bb767),LL(0xfe0af3f5,0x85512b4d), LL(0x99b16a0d,0x18958ba8),LL(0xba7548a7,0x95c2430c),LL(0xa16be615,0xb30d1b10),LL(0x85bfb74c,0xe3ebbb97),
   LL(0x18549fdb,0xa3273cfe),LL(0x4fcdb792,0xf6e200bf),LL(0x83aba56c,0x54a76e18),LL(0x89ef6aa2,0x73ec66f6), LL(0xd1b9a305,0x8d17add7),LL(0xb7ae1b9d,0xa959c5b9),LL(0x6bcc094a,0x88643522),LL(0xd7d429b9,0xcc5616c4),
   LL(0xe6a33f7c,0xa6dada01),LL(0x9d4e70ad,0xc6217a07),LL(0x09c15b7c,0xd619a818),LL(0x0e80c854,0xea06b329), LL(0xa5f5e7b9,0x174811ce),LL(0x787c65f4,0x66dfc310),LL(0x3316ab54,0x4ea7bd69),LL(0x1dcc0f70,0xc12c4acb),
   LL(0x1e407dd9,0xe4308d1a),LL(0x91afa997,0xe8a3587c),LL(0xab77b7a5,0xea296c12),LL(0x673c0d52,0xb5ad49e4), LL(0x7006085a,0x40f9b2b2),LL(0x87bf6ec2,0xa88ff340),LL(0x4e3066a6,0x978603b1),LL(0xb5e486e2,0xb3f99fc2),
   LL(0xb2e63645,0x07b53f5e),LL(0x84c84232,0xbe57e547),LL(0x7214d5cf,0xd779c216),LL(0x029a3aca,0x617969cd), LL(0x8a7017a0,0xd17668cd),LL(0xbe9b7ee8,0x77b4d19a),LL(0x9c161776,0x58fd0e93),LL(0xd5968a72,0xa8c4f4ef),
   LL(0x67b3de77,0x296071cc),LL(0x634f7905,0xae3c0b8e),LL(0x8a7100c9,0x67e440c2),LL(0xeb4b9b42,0xbb8c3c1b), LL(0xc51b3583,0x6d71e8ea),LL(0x9525e642,0x7591f5af),LL(0x13f509f3,0xf73a2f7b),LL(0x5619ac9b,0x618487aa),
   LL(0x9d61718a,0x3a72e5f7),LL(0x7592d28c,0x00413bcc),LL(0x963c35cf,0x7d9b11d3),LL(0xb90a46ed,0x77623bcf), LL(0xdcdd2a50,0xdeef273b),LL(0x0601846e,0x4a741f9b),LL(0x0ec6e929,0x33b89e51),LL(0x8b7f22cd,0xcb02319f),
   LL(0x084bae24,0xbbe1500d),LL(0x343d2693,0x2f0ae8d7),LL(0x7cdef811,0xacffb5f2),LL(0x263fb94f,0xaa0c030a), LL(0xa0f442de,0x6eef0d61),LL(0x27b139d3,0xf92e1817),LL(0x0ad8bc28,0x1ae6deb7),LL(0xc0514130,0xa89e38dc),
   LL(0xd2fdca23,0x81eeb865),LL(0xcc8ef895,0x5a15ee08),LL(0x01905614,0x768fa10a),LL(0x880ee19b,0xeff5b8ef), LL(0xcb1c8a0e,0xf0c0cabb),LL(0xb8c838f9,0x2e1ee9cd),LL(0x8a4a14c0,0x0587d8b8),LL(0x2ff698e5,0xf6f27896),
   LL(0x89ee6256,0xed38ef1c),LL(0x6b353b45,0xf44ee1fe),LL(0x70e903b3,0x9115c0c7),LL(0x818f31df,0xc78ec0a1), LL(0xb7dccbc6,0x6c003324),LL(0x163bbc25,0xd96dd1f3),LL(0x5cedd805,0x33aa82dd),LL(0x7f7eb2f1,0x123aae4f),
   LL(0xa26262cd,0x1723fcf5),LL(0x0060ebd5,0x1f7f4d5d),LL(0xb2eaa3af,0xf19c5c01),LL(0x9790accf,0x2ccb9b14), LL(0x52324aa6,0x1f9c1cad),LL(0x7247df54,0x63200526),LL(0xbac96f82,0x5732fe42),LL(0x01a1c384,0x52fe771f),
   LL(0xb1001684,0x546ca13d),LL(0xa1709f75,0xb56b4eee),LL(0xd5db8672,0x266545a9),LL(0x1e8f3cfb,0xed971c90), LL(0xe3a07b29,0x4e7d8691),LL(0xe4b696b9,0x7570d9ec),LL(0x7bc7e9ae,0xdc5fa067),LL(0xc82c4844,0x68b44caf),
   LL(0xbf44da80,0x519d34b3),LL(0x5ab32e66,0x283834f9),LL(0x6278a000,0x6e608797),LL(0x627312f6,0x1e62960e), LL(0xe6901c55,0x9b87b27b),LL(0x24fdbc1f,0x80e78538),LL(0x2facc27d,0xbbbc0951),LL(0xac143b5a,0x06394239),
   LL(0x376c1944,0x35bb4a40),LL(0x63da1511,0x7cb62694),LL(0xb7148a3b,0xafd29161),LL(0x4e2ea2ee,0xa6f9d9ed), LL(0x880dd212,0x15dc2ca2),LL(0xa61139a9,0x903c3813),LL(0x6c0f8785,0x2aa7b46d),LL(0x901c60ff,0x36ce2871),
   LL(0xe10d9c12,0xc683b028),LL(0x032f33d3,0x7573baa2),LL(0x67a31b58,0x87a9b1f6),LL(0xf4ffae12,0xfd3ed11a), LL(0x0cb2748e,0x83dcaa9a),LL(0x5d6fdf16,0x8239f018),LL(0x72753941,0xba67b49c),LL(0xc321cb36,0x2beec455),
   LL(0x3f8b84ce,0x88015606),LL(0x8d38c86f,0x76417083),LL(0x598953dd,0x054f1ca7),LL(0x4e8e7429,0xc939e110), LL(0x5a914f2f,0x9b1ac2b3),LL(0xe74b8f9c,0x39e35ed3),LL(0x781b2fb0,0xd0debdb2),LL(0x2d997ba2,0x1585638f),
   LL(0x9e2fce99,0x9c4b646e),LL(0x1e80857f,0x68a21081),LL(0x3643b52a,0x06d54e44),LL(0x0d8eb843,0xde8d6d63), LL(0x42146a0a,0x70321563),LL(0x5eaa3622,0x8ba826f2),LL(0x86138787,0x227a58bd),LL(0x10281d37,0x43b6c03c),
   LL(0xb54dde39,0x6326afbb),LL(0xdb6f2d5f,0x744e5e8a),LL(0xcff158e1,0x48b2a99a),LL(0xef87918f,0xa93c8fa0), LL(0xde058c5c,0x2182f956),LL(0x936f9e7a,0x216235d2),LL(0xd2e31e67,0xace0c0db),LL(0xf23ac3e7,0xc96449bf),
   LL(0x170693bd,0x7e9a2874),LL(0xa45e6335,0xa28e14fd),LL(0x56427344,0x5757f6b3),LL(0xacf8edf9,0x822e4556), LL(0xe6a285cd,0x2b7a6ee2),LL(0xa9df3af0,0x5866f211),LL(0xf845b844,0x40dde2dd),LL(0x110e5e49,0x986c3726),
   LL(0xf7172277,0x73680c2a),LL(0x0cccb244,0x57b94f0f),LL(0x2d438ca7,0xbdff7267),LL(0xcf4663fd,0xbad1ce11), LL(0xd8f71cae,0x9813ed9d),LL(0x961fdaa6,0xf43272a6),LL(0xbd6d1637,0xbeff0119),LL(0x30361978,0xfebc4f91),
   LL(0x2f41deff,0x02b37a95),LL(0xe63b89b7,0x0e44a59a),LL(0x143ff951,0x673257dc),LL(0xd752baf4,0x19c02205), LL(0xc4b7d692,0x46c23069),LL(0xfd1502ac,0x2e6392c3),LL(0x1b220846,0x6057b1a2),LL(0x0c1b5b63,0xe51ff946),
},
/* digit=35 base_pwr=2^245 */
{
   LL(0x566c5c43,0x6e85cb51),LL(0x3597f046,0xcff9c919),LL(0x4994d94a,0x9354e90c),LL(0x2147927d,0xe0a39332), LL(0x0dc1eb2b,0x8427fac1),LL(0x2ff319fa,0x88cfd8c2),LL(0x01965274,0xe2d4e684),LL(0x67aaa746,0xfa2e067d),
   LL(0x3e5f9f11,0xb6d92a7f),LL(0xd6cb3b8e,0x9afe153a),LL(0xddf800bd,0x4d1a6dd7),LL(0xcaf17e19,0xf6c13cc0), LL(0x325fc3ee,0x15f6c58e),LL(0xa31dc3b2,0x71095400),LL(0xafa3d3e7,0x168e7c07),LL(0x94c7ae2d,0x3f8417a1),
   LL(0x813b230d,0xec234772),LL(0x17344427,0x634d0f5f),LL(0xd77fc56a,0x11548ab1),LL(0xce06af77,0x7fab1750), LL(0x4f7c4f83,0xb62c10a7),LL(0x220a67d9,0xa7d2edc4),LL(0x921209a0,0x1c404170),LL(0xface59f0,0x0b9815a0),
   LL(0x319540c3,0x2842589b),LL(0xa283d6f8,0x18490f59),LL(0xdaae9fcb,0xa2731f84),LL(0xc3683ba0,0x3db6d960), LL(0x14611069,0xc85c63bb),LL(0x0788bf05,0xb19436af),LL(0x347460d2,0x905459df),LL(0xe11a7db1,0x73f6e094),
   LL(0xb6357f37,0xdc7f938e),LL(0x2bd8aa62,0xc5d00f79),LL(0x2ca979fc,0xc878dcb9),LL(0xeb023a99,0x37e83ed9), LL(0x1560bf3d,0x6b23e273),LL(0x1d0fae61,0x1086e459),LL(0x9a9414bd,0x78248316),LL(0xf0ea9ea1,0x1b956bc0),
   LL(0xc31b9c38,0x7b85bb91),LL(0x48ef57b5,0x0c5aa90b),LL(0xaf3bab6f,0xdedeb169),LL(0x2d373685,0xe610ad73), LL(0x02ba8e15,0xf13870df),LL(0x8ca7f771,0x0337edb6),LL(0xb62c036c,0xe4acf747),LL(0xb6b94e81,0xd921d576),
   LL(0x2c422f7a,0xdbc86439),LL(0xed348898,0xfb635362),LL(0xc45bfcd1,0x83084668),LL(0x2b315e11,0xc357c9e3), LL(0x5b2e5b8c,0xb173b540),LL(0xe102b9a4,0x7e946931),LL(0x7b0fb199,0x17c890eb),LL(0xd61b662b,0xec225a83),
   LL(0xee3c76cb,0xf306a3c8),LL(0xd32a1f6e,0x3cf11623),LL(0x6863e956,0xe6d5ab64),LL(0x5c005c26,0x3b8a4cbe), LL(0x9ce6bb27,0xdcd529a5),LL(0x04d4b16f,0xc4afaa52),LL(0x7923798d,0xb0624a26),LL(0x6b307fab,0x85e56df6),
   LL(0x2bf29698,0x0281893c),LL(0xd7ce7603,0x91fc19a4),LL(0xad9a558f,0x75a5dca3),LL(0x4d50bf77,0x40ceb3fa), LL(0xbc9ba369,0x1baf6060),LL(0x597888c2,0x927e1037),LL(0x86a34c07,0xd936bf19),LL(0xc34ae980,0xd4cf10c1),
   LL(0x859dd614,0x3a3e5334),LL(0x18d0c8ee,0x9c475b5b),LL(0x07cd51d5,0x63080d1f),LL(0xb88b4326,0xc9c0d0a6), LL(0xc234296f,0x1ac98691),LL(0x94887fb6,0x2a0a83a4),LL(0x0cea9cf2,0x56511427),LL(0xa24802f5,0x5230a6e8),
   LL(0x72e3d5c1,0xf7a2bf0f),LL(0x4f21439e,0x37717446),LL(0x9ce30334,0xfedcbf25),LL(0x7ce202f9,0xe0030a78), LL(0x1202e9ca,0x6f2d9ebf),LL(0x75e6e591,0xe79dde6c),LL(0xf1dac4f8,0xf52072af),LL(0xbb9b404d,0x6c8d087e),
   LL(0xbce913af,0xad0fc73d),LL(0x458a07cb,0x909e587b),LL(0xd4f00c8a,0x1300da84),LL(0xb54466ac,0x425cd048), LL(0x90e9d8bf,0xb59cb9be),LL(0x3e431b0e,0x991616db),LL(0x531aecff,0xd3aa117a),LL(0x59f4dc3b,0x91af92d3),
   LL(0xe93fda29,0x9b1ec292),LL(0xe97d91bc,0x76bb6c17),LL(0xaface1e6,0x7509d95f),LL(0xbe855ae3,0x3653fe47), LL(0x0f680e75,0x73180b28),LL(0xeeb6c26c,0x75eefd1b),LL(0xb66d4236,0xa4cdf29f),LL(0x6b5821d8,0x2d70a997),
   LL(0x20445c36,0x7a3ee207),LL(0x59877174,0x71d1ac82),LL(0x949f73e9,0x0fc539f7),LL(0x982e3081,0xd05cf3d7), LL(0x7b1c7129,0x8758e20b),LL(0x569e61f2,0xffadcc20),LL(0x59544c2d,0xb05d3a2f),LL(0x9fff5e53,0xbe16f5c1),
   LL(0xaad58135,0x73cf65b8),LL(0x037aa5be,0x622c2119),LL(0x646fd6a0,0x79373b3f),LL(0x0d3978cf,0x0e029db5), LL(0x94fba037,0x8bdfc437),LL(0x620797a6,0xaefbd687),LL(0xbd30d38e,0x3fa5382b),LL(0x585d7464,0x7627cfbf),
   LL(0x4e4ca463,0xb2330fef),LL(0x3566cc63,0xbcef7287),LL(0xcf780900,0xd161d2ca),LL(0x5b54827d,0x135dc539), LL(0x27bf1bc6,0x638f052e),LL(0x07dfa06c,0x10a224f0),LL(0x6d3321da,0xe973586d),LL(0x26152c8f,0x8b0c5738),
   LL(0x34606074,0x07ef4f2a),LL(0xa0f7047a,0x80fe7fe8),LL(0xe1a0e306,0x3d1a8152),LL(0x88da5222,0x32cf43d8), LL(0x5f02ffe6,0xbf89a95f),LL(0x806ad3ea,0x3d9eb9a4),LL(0x79c8e55e,0x012c17bb),LL(0x99c81dac,0xfdcd1a74),
   LL(0xb9556098,0x7043178b),LL(0x801c3886,0x4090a1df),LL(0x9b67b912,0x759800ff),LL(0x232620c8,0x3e5c0304), LL(0x70dceeca,0x4b9d3c4b),LL(0x181f648e,0xbb2d3c15),LL(0x6e33345c,0xf981d837),LL(0x0cf2297a,0xb626289b),
   LL(0x8baebdcf,0x766ac659),LL(0x75df01e5,0x1a28ae09),LL(0x375876d8,0xb71283da),LL(0x607b9800,0x4865a96d), LL(0x237936b2,0x25dd1bcd),LL(0x60417494,0x332f4f4b),LL(0x370a2147,0xd0923d68),LL(0xdc842203,0x497f5dfb),
   LL(0x32be5e0f,0x9dc74cbd),LL(0x17a01375,0x7475bcb7),LL(0x50d872b1,0x438477c9),LL(0xffe1d63d,0xcec67879), LL(0xd8578c70,0x9b006014),LL(0x78bb6b8b,0xc9ad99a8),LL(0x11fb3806,0x6799008e),LL(0xcd44cab3,0xcfe81435),
   LL(0x2f4fb344,0xa2ee1582),LL(0x483fa6eb,0xb8823450),LL(0x652c7749,0x622d323d),LL(0xbeb0a15b,0xd8474a98), LL(0x5d1c00d0,0xe43c154d),LL(0x0e3e7aac,0x7fd581d9),LL(0x2525ddf8,0x2b44c619),LL(0xb8ae9739,0x67a033eb),
   LL(0x9ef2d2e4,0x113ffec1),LL(0xd5a0ea7f,0x1bf6767e),LL(0x03714c0a,0x57fff75e),LL(0x0a23e9ee,0xa23c422e), LL(0x540f83af,0xdd5f6b2d),LL(0x55ea46a7,0xc2c2c27e),LL(0x672a1208,0xeb6b4246),LL(0xae634f7a,0xd13599f7),
   LL(0xd7b32c6e,0xcf914b5c),LL(0xeaf61814,0x61a5a640),LL(0x208a1bbb,0x8dc3df8b),LL(0xb6d79aa5,0xef627fd6), LL(0xc4c86bc8,0x44232ffc),LL(0x061539fe,0xe6f9231b),LL(0x958b9533,0x1d04f25a),LL(0x49e8c885,0x180cf934),
   LL(0x9884aaf7,0x89689595),LL(0x07b348a6,0xb1959be3),LL(0x3c147c87,0x96250e57),LL(0xdd0c61f8,0xae0efb3a), LL(0xca8c325e,0xed00745e),LL(0xecff3f70,0x3c911696),LL(0x319ad41d,0x73acbc65),LL(0xf0b1c7ef,0x7b01a020),
   LL(0x63a1483f,0xea32b293),LL(0x7a248f96,0x89eabe71),LL(0x343157e5,0x9c6231d3),LL(0xdf3c546d,0x93a375e5), LL(0x6a2afe69,0xe76e9343),LL(0xe166c88e,0xc4f89100),LL(0x4f872093,0x248efd0d),LL(0x8fe0ea61,0xae0eb3ea),
   LL(0x9d79046e,0xaf89790d),LL(0x6cee0976,0x4d650f2d),LL(0x43071eca,0xa3935d9a),LL(0x283b0bfe,0x66fcd2c9), LL(0x696605f1,0x0e665eb5),LL(0xa54cd38d,0xe77e5d07),LL(0x43d950cf,0x90ee050a),LL(0xd32e69b5,0x86ddebda),
   LL(0xfddf7415,0x6ad94a3d),LL(0x3f6e8d5a,0xf7fa1309),LL(0xe9957f75,0xc4831d1d),LL(0xd5817447,0x7de28501), LL(0x9e2aeb6b,0x6f1d7078),LL(0xf67a53c2,0xba2b9ff4),LL(0xdf9defc3,0x36963767),LL(0x0d38022c,0x479deed3),
   LL(0x3a8631e8,0xd2edb89b),LL(0x7a213746,0x8de855de),LL(0xb00c5f11,0xb2056cb7),LL(0x2c9b85e4,0xdeaefbd0), LL(0xd150892d,0x03f39a8d),LL(0x218b7985,0x37b84686),LL(0xb7375f1a,0x36296dd8),LL(0xb78e898e,0x472cd4b1),
   LL(0xe9f05de9,0x15dff651),LL(0x2ce98ba9,0xd4045069),LL(0x9b38024c,0x8466a7ae),LL(0xe5a6b5ef,0xb910e700), LL(0xb3aa8f0d,0xae1c56ea),LL(0x7eee74a6,0xbab2a507),LL(0x4b4c4620,0x0dca11e2),LL(0x4c47d1f4,0xfd896e2e),
   LL(0x308fbd93,0xeb45ae53),LL(0x02c36fda,0x46cd5a2e),LL(0xbaa48385,0x6a3d4e90),LL(0x9dbe9960,0xdd55e62e), LL(0x2a81ede7,0xa1406aa0),LL(0xf9274ea7,0x6860dd14),LL(0x80414f86,0xcfdcb0c2),LL(0x22f94327,0xff410b10),
   LL(0x49ad467b,0x5a33cc38),LL(0x0a7335f1,0xefb48b6c),LL(0xb153a360,0x14fb54a4),LL(0xb52469cc,0x604aa9d2), LL(0x754e48e9,0x5e9dc486),LL(0x37471e8e,0x693cb455),LL(0x8d3b37b6,0xfb2fd7cd),LL(0xcf09ff07,0x63345e16),
   LL(0x23a5d896,0x9910ba6b),LL(0x7fe4364e,0x1fe19e35),LL(0x9a33c677,0x6e1da8c3),LL(0x29fd9fd0,0x15b4488b), LL(0x1a1f22bf,0x1f439254),LL(0xab8163e8,0x920a8a70),LL(0x07e5658e,0x3fd1b249),LL(0xb6ec839b,0xf2c4f79c),
   LL(0x4aa38d1b,0x1abbc3d0),LL(0xb5d9510e,0x3b0db35c),LL(0x3e60dec0,0x1754ac78),LL(0xea099b33,0x53272fd7), LL(0x07a8e107,0x5fb0494f),LL(0x6a8191fa,0x4a89e137),LL(0x3c4ad544,0xa113b7f6),LL(0x6cb9897b,0x88a2e909),
   LL(0xb44a3f84,0x17d55de3),LL(0x17c6c690,0xacb2f344),LL(0x10232390,0x32088168),LL(0x6c733bf7,0xf2e8a61f), LL(0x9c2d7652,0xa774aab6),LL(0xed95c5bc,0xfb5307e3),LL(0x4981f110,0xa05c73c2),LL(0xa39458c9,0x1baae31c),
   LL(0xcbea62e7,0x1def185b),LL(0xeaf63059,0xe8ac9eae),LL(0x9921851c,0x098a8cfd),LL(0x3abe2f5b,0xd959c3f1), LL(0x20e40ae5,0xa4f19525),LL(0x07a24aa1,0x320789e3),LL(0x7392b2bc,0x259e6927),LL(0x1918668b,0x58f6c667),
   LL(0xc55d2d8b,0xce1db2bb),LL(0xf4f6ca56,0x41d58bb7),LL(0x8f877614,0x7650b680),LL(0xf4c349ed,0x905e16ba), LL(0xf661acac,0xed415140),LL(0xcb2270af,0x3b8784f0),LL(0x8a402cba,0x3bc280ac),LL(0x0937921a,0xd53f7146),
   LL(0xe5681e83,0xc03c8ee5),LL(0xf6ac9e4a,0x62126105),LL(0x936b1a38,0x9503a53f),LL(0x782fecbd,0x3d45e2d4), LL(0x76e8ae98,0x69a5c439),LL(0xbfb4b00e,0xb53b2eeb),LL(0x72386c89,0xf1674712),LL(0x4268bce4,0x30ca34a2),
   LL(0x78341730,0x7f1ed86c),LL(0xb525e248,0x8ef5beb8),LL(0xb74fbf38,0xbbc489fd),LL(0x91a0b382,0x38a92a0e), LL(0x22433ccf,0x7a77ba3f),LL(0xa29f05a9,0xde8362d6),LL(0x61189afc,0x7f6a30ea),LL(0x59ef114f,0x693b5505),
   LL(0xcd1797a1,0x50266bc0),LL(0xf4b7af2d,0xea17b47e),LL(0x3df9483e,0xd6c4025c),LL(0xa37b18c9,0x8cbb9d9f), LL(0x4d8424cf,0x91cbfd9c),LL(0xab1c3506,0xdb7048f1),LL(0x028206a3,0x9eaf641f),LL(0x25bdf6ce,0xf986f3f9),
   LL(0x224c08dc,0x262143b5),LL(0x81b50c91,0x2bbb09b4),LL(0xaca8c84f,0xc16ed709),LL(0xb2850ca8,0xa6210d9d), LL(0x09cb54d6,0x6d8df67a),LL(0x500919a4,0x91eef6e0),LL(0x0f132857,0x90f61381),LL(0xf8d5028b,0x9acede47),
   LL(0x90b771c3,0x844d1b71),LL(0xba6426be,0x563b71e4),LL(0xbdb802ff,0x2efa2e83),LL(0xab5b4a41,0x3410cbab), LL(0x30da84dd,0x555b2d26),LL(0xee1cc29a,0xd0711ae9),LL(0x2f547792,0xcf3e8c60),LL(0xdc678b35,0x03d7d5de),
   LL(0xced806b8,0x071a2fa8),LL(0x697f1478,0x222e6134),LL(0xabfcdbbf,0xdc16fd5d),LL(0x121b53b8,0x44912ebf), LL(0x2496c27c,0xac943674),LL(0x1ffc26b0,0x8ea3176c),LL(0x13debf2c,0xb6e224ac),LL(0xf372a832,0x524cc235),
   LL(0x9f6f1b18,0xd706e1d8),LL(0x44cce35b,0x2552f005),LL(0xa88e31fc,0x8c8326c2),LL(0xf9552047,0xb5468b2c), LL(0x3ff90f2b,0xce683e88),LL(0x2f0a5423,0x77947bdf),LL(0xed56e328,0xd0a1b28b),LL(0xc20134ac,0xaee35253),
   LL(0x3567962f,0x7e98367d),LL(0x8188bffb,0x379ed61f),LL(0xfaf130a1,0x73bba348),LL(0x904ed734,0x6c1f75e1), LL(0x3b4a79fc,0x18956642),LL(0x54ef4493,0xf20bc83d),LL(0x9111eca1,0x836d425d),LL(0x009a8dcf,0xe5b5c318),
   LL(0x13221bc5,0x3360b25d),LL(0x6b3eeaf7,0x707baad2),LL(0x743a95a1,0xd7279ed8),LL(0x969e809f,0x7450a875), LL(0xe5d0338f,0x32b6bd53),LL(0x2b883bbc,0x1e77f7af),LL(0x1063ecd0,0x90da12cc),LL(0xc315be47,0xe2697b58),
   LL(0xda85d534,0x2771a5bd),LL(0xff980eea,0x53e78c1f),LL(0x900385e7,0xadf1cf84),LL(0xc9387b62,0x7d3b14f6), LL(0xcb8f2bd2,0x170e74b0),LL(0x827fa993,0x2d50b486),LL(0xf6f32bab,0xcdbe8c9a),LL(0xc3b93ab8,0x55e906b0),
   LL(0x8fe280d1,0x747f22fc),LL(0xb2e114ab,0xcd8e0de5),LL(0xe10b68b0,0x5ab7dbeb),LL(0xa480d4b2,0x9dc63a9c), LL(0x4be1495f,0x78d4bc3b),LL(0x9359122d,0x25eb3db8),LL(0x0809cbdc,0x3f8ac05b),LL(0xd37c702f,0xbf4187bb),
   LL(0x1416a6a5,0x84cea069),LL(0x43ef881c,0x8f860c79),LL(0x38038a5d,0x41311f8a),LL(0xfc612067,0xe78c2ec0), LL(0x5ad73581,0x494d2e81),LL(0x59604097,0xb4cc9e00),LL(0xf3612cba,0xff558aec),LL(0x9e36c39e,0x35beef7a),
   LL(0xdbcf41b9,0x1845c7cf),LL(0xaea997c0,0x5703662a),LL(0xe402f6d8,0x8b925afe),LL(0x4dd72162,0xd0a1b1ae), LL(0x03c41c4b,0x9f47b375),LL(0x0391d042,0xa023829b),LL(0x503b8b0a,0x5f5045c3),LL(0x98c010e5,0x123c2688),
   LL(0x36ba06ee,0x324ec0cc),LL(0x3dd2cc0c,0xface3115),LL(0xf333e91f,0xb364f3be),LL(0x28e832b0,0xef8aff73), LL(0x2d05841b,0x1e9bad04),LL(0x356a21e2,0x42f0e3df),LL(0x4add627e,0xa3270bcb),LL(0xd322e711,0xb09a8158),
   LL(0x0fee104a,0x86e326a1),LL(0x3703f65d,0xad7788f8),LL(0x47bc4833,0x7e765430),LL(0x2b9b893a,0x6cee582b), LL(0xe8f55a7b,0x9cd2a167),LL(0xd9e4190d,0xefbee3c6),LL(0xd40c2e9d,0x33ee7185),LL(0xa380b548,0x844cc9c5),
   LL(0x66926e04,0x323f8ecd),LL(0x8110c1ba,0x0001e38f),LL(0xfc6a7f07,0x8dbcac12),LL(0x0cec0827,0xd65e1d58), LL(0xbe76ca2d,0xd2cd4141),LL(0xe892f33a,0x7895cf5c),LL(0x367139d2,0x956d230d),LL(0xd012c4c1,0xa91abd3e),
   LL(0x87eb36bf,0x34fa4883),LL(0x914b8fb4,0xc5f07102),LL(0xadb9c95f,0x90f0e579),LL(0x28888195,0xfe6ea8cb), LL(0xedfa9284,0x7b9b5065),LL(0x2b8c8d65,0x6c510bd2),LL(0xcbe8aafd,0xd7b8ebef),LL(0x96b1da07,0xedb3af98),
   LL(0x6295d426,0x28ff779d),LL(0x3fa3ad7b,0x0c4f6ac7),LL(0x8b8e2604,0xec44d054),LL(0x8b0050e1,0x9b32a66d), LL(0xf0476ce2,0x1f943366),LL(0xa602c7b4,0x7554d953),LL(0x524f2809,0xbe35aca6),LL(0xfd4edbea,0xb6881229),
   LL(0x508efb63,0xe8cd0c8f),LL(0x6abcefc7,0x9eb5b5c8),LL(0xb441ab4f,0xf5621f5f),LL(0xb76a2b22,0x79e6c046), LL(0xe37a1f69,0x74a4792c),LL(0x03542b60,0xcbd252cb),LL(0xb3c20bd3,0x785f65d5),LL(0x4fabc60c,0x8dea6143),
   LL(0xde673629,0x45e21446),LL(0x703c2d21,0x57f7aa1e),LL(0x98c868c7,0xa0e99b7f),LL(0x8b641676,0x4e42f66d), LL(0x91077896,0x602884dc),LL(0xc2c9885b,0xa0d690cf),LL(0x3b9a5187,0xfeb4da33),LL(0x153c87ee,0x5f789598),
   LL(0x52b16dba,0x2192dd47),LL(0x3524c1b1,0xdeefc0e6),LL(0xe4383693,0x465ea76e),LL(0x361b8d98,0x79401711), LL(0xf21a15cb,0xa5f9ace9),LL(0xefee9aeb,0x73d26163),LL(0xe677016c,0xcca844b3),LL(0x57eaee06,0x6c122b07),
   LL(0x15f09690,0xb782dce7),LL(0x2dfc0fc9,0x508b9b12),LL(0x65d89fc6,0x9015ab4b),LL(0xd6d5bb0f,0x5e79dab7), LL(0x6c775aa2,0x64f021f0),LL(0x37c7eca1,0xdf09d8cc),LL(0xef2fa506,0x9a761367),LL(0x5b81eec6,0xed4ca476),
   LL(0x10bbb8b5,0x262ede36),LL(0x0641ada3,0x0737ce83),LL(0xe9831ccc,0x4c94288a),LL(0x8065e635,0x487fc1ce), LL(0xb8bb3659,0xb13d7ab3),LL(0x855e4120,0xdea5df3e),LL(0x85eb0244,0xb9a18573),LL(0xa7cfe0a3,0x1a1b8ea3),
   LL(0x67b0867c,0x3b837119),LL(0x9d364520,0x8d5e0d08),LL(0xd930f0e3,0x52dccc1e),LL(0xbf20bbaf,0xefbbcec7), LL(0x0263ad10,0x99cffcab),LL(0xfcd18f8a,0xd8199e6d),LL(0xe9f10617,0x64e2773f),LL(0x08704848,0x0079e8e1),
   LL(0x8a342283,0x1169989f),LL(0xa83012e6,0x8097799c),LL(0x8a6a9001,0xece966cb),LL(0x072ac7fc,0x93b3afef), LL(0x2db3d5ba,0xe6893a2a),LL(0x89bf4fdc,0x263dc462),LL(0xe0396673,0x8852dfc9),LL(0x3af362b6,0x7ac70895),
   LL(0x5c2f342b,0xbb9cce4d),LL(0xb52d7aae,0xbf80907a),LL(0x2161bcd0,0x97f3d3cd),LL(0x0962744d,0xb25b0834), LL(0x6c3a1dda,0xc5b18ea5),LL(0x06c92317,0xfe4ec7eb),LL(0xad1c4afe,0xb787b890),LL(0x0ede801a,0xdccd9a92),
   LL(0xdb58da1f,0x9ac6ddda),LL(0xb8cae6ee,0x22bbc12f),LL(0x815c4a43,0xc6f8bced),LL(0xf96480c7,0x8105a92c), LL(0x7a859d51,0x0dc3dbf3),LL(0x3041196b,0xe3ec7ce6),LL(0x0d1067c9,0xd9f64b25),LL(0x3d1f8dd8,0xf2321321),
   LL(0x76497ee8,0x8b5c619c),LL(0xc717370e,0x5d2b0ac6),LL(0x4fcf68e1,0x98204cb6),LL(0x62bc6792,0x0bdec211), LL(0xa63b1011,0x6973ccef),LL(0xe0de1ac5,0xf9e3fa97),LL(0x3d0e0c8b,0x5efb693e),LL(0xd2d4fcb4,0x037248e9),
},
/* digit=36 base_pwr=2^252 */
{
   LL(0x1ec34f9e,0x80802dc9),LL(0x33810603,0xd8772d35),LL(0x530cb4f3,0x3f06d66c),LL(0xc475c129,0x7be5ed0d), LL(0x31e82b10,0xcb9e3c19),LL(0xc9ff6b4c,0xc63d2857),LL(0x92a1b45e,0xb92118c6),LL(0x7285bbca,0x0aec4414),
   LL(0x1e29a3ef,0xfc189ae7),LL(0x4c93302e,0xcbe906f0),LL(0xceaae10e,0xd0107914),LL(0xb68e19f8,0xb7a23f34), LL(0xefd2119d,0xe9d875c2),LL(0xfcadc9c8,0x03198c6e),LL(0x4da17113,0x65591bf6),LL(0x3d443038,0x3cf0bbf8),
   LL(0x2b724759,0xae485bb7),LL(0xb2d4c63a,0x945353e1),LL(0xde7d6f2c,0x82159d07),LL(0x4ec5b109,0x389caef3), LL(0xdb65ef14,0x4a8ebb53),LL(0xdd99de43,0x2dc2cb7e),LL(0x83f2405f,0x816fa3ed),LL(0xc14208a3,0x73429bb9),
   LL(0xb01e6e27,0xb618d590),LL(0xe180b2dc,0x047e2ccd),LL(0x04aea4a9,0xd1b299b5),LL(0x9fa403a4,0x412c9e1e), LL(0x79407552,0x88d28a36),LL(0xf332b8e3,0x49c50136),LL(0xe668de19,0x3a1b6fcc),LL(0x75122b97,0x178851bc),
   LL(0xfb85fa4c,0xb1e13752),LL(0x383c8ce9,0xd61257ce),LL(0xd2f74dae,0xd43da670),LL(0xbf846bbb,0xa35aa23f), LL(0x4421fc83,0x5e74235d),LL(0xc363473b,0xf6df8ee0),LL(0x3c4aa158,0x34d7f52a),LL(0x9bc6d22e,0x50d05aab),
   LL(0xa64785f4,0x8c56e735),LL(0x5f29cd07,0xbc56637b),LL(0x3ee35067,0x53b2bb80),LL(0xdc919270,0x50235a0f), LL(0xf2c4aa65,0x191ab6d8),LL(0x8396023b,0xc3475831),LL(0xf0f805ba,0x80400ba5),LL(0x5ec0f80f,0x8881065b),
   LL(0xcc1b5e83,0xc370e522),LL(0x860b8bfb,0xde2d4ad1),LL(0x67b256df,0xad364df0),LL(0xe0138997,0x8f12502e), LL(0x7783920a,0x503fa0dc),LL(0xc0bc866a,0xe80014ad),LL(0xd3064ba6,0x3f89b744),LL(0xcba5dba5,0x03511dcd),
   LL(0x95a7b1a2,0x197dd46d),LL(0x3c6341fb,0x9c4e7ad6),LL(0x484c2ece,0x426eca29),LL(0xde7f4f8a,0x9211e489), LL(0xc78ef1f4,0x14997f6e),LL(0x06574586,0x2b2c0910),LL(0x1c3eede8,0x17286a6e),LL(0x0f60e018,0x25f92e47),
   LL(0x31890a36,0x805c5646),LL(0x57feea5b,0x703ef600),LL(0xaf3c3030,0x389f747c),LL(0x54dd3739,0xe0e5daeb), LL(0xc9c9f155,0xfe24a4c3),LL(0xb5393962,0x7e4bf176),LL(0xaf20bf29,0x37183de2),LL(0xf95a8c3b,0x4a1bd7b5),
   LL(0x46191d3d,0xa83b9699),LL(0x7b87f257,0x281fc8dd),LL(0x54107588,0xb18e2c13),LL(0x9b2bafe8,0x6372def7), LL(0x0d8972ca,0xdaf4bb48),LL(0x56167a3f,0x3f2dd4b7),LL(0x84310cf4,0x1eace32d),LL(0xe42700aa,0xe3bcefaf),
   LL(0xd785e73d,0x5fe5691e),LL(0x2ea60467,0xa5db5ab6),LL(0xdfc6514a,0x02e23d41),LL(0xe03c3665,0x35e8048e), LL(0x1adaa0f8,0x3f8b118f),LL(0x84ce1a5a,0x28ec3b45),LL(0x2c6646b8,0xe8cacc6e),LL(0xdbd0e40f,0x1343d185),
   LL(0xcaaa358c,0xe5d7f844),LL(0x9924182a,0x1a1db7e4),LL(0x9c875d9a,0xd64cd42d),LL(0x042eeec8,0xb37b515f), LL(0x7b165fbe,0x4d4dd409),LL(0xe206eff3,0xfc322ed9),LL(0x59b7e17e,0x7dee4102),LL(0x8236ca00,0x55a481c0),
   LL(0xc23fc975,0x8c885312),LL(0x05d6297b,0x15715806),LL(0xf78edd39,0xa078868e),LL(0x03c45e52,0x956b31e0), LL(0xff7b33a6,0x470275d5),LL(0x0c7e673f,0xc8d5dc3a),LL(0x7e2f2598,0x419227b4),LL(0x4c14a975,0x8b37b634),
   LL(0x8b11888c,0xd0667ed6),LL(0x803e25dc,0x5e0e8c3e),LL(0xb987a24a,0x34e5d0dc),LL(0xae920323,0x9f40ac3b), LL(0x34e0f63a,0x5463de95),LL(0x6b6328f9,0xa128bf92),LL(0xda64f1b7,0x491ccd7c),LL(0xc47bde35,0x7ef1ec27),
   LL(0xa36a2737,0xa857240f),LL(0x63621bc1,0x35dc1366),LL(0xd4fb6897,0x7a3a6453),LL(0xc929319d,0x80f1a439), LL(0xf8cb0ba0,0xfc18274b),LL(0x8078c5eb,0xb0b53766),LL(0x1e01d0ef,0xfb0d4924),LL(0x372ab09c,0x50d7c67d),
   LL(0x3aeac968,0xb4e370af),LL(0xc4b63266,0xe4f7fee9),LL(0xe3ac5664,0xb4acd4c2),LL(0xceb38cbf,0xf8910bd2), LL(0xc9c0726e,0x1c3ae50c),LL(0xd97b40bf,0x15309569),LL(0xfd5a5a1b,0x70884b7f),LL(0xef8314cd,0x3890896a),
   LL(0xa5618c93,0x58e1515c),LL(0x77d942d1,0xe665432b),LL(0xb6f767a8,0xb32181bf),LL(0x3a604110,0x753794e8), LL(0xe8c0dbcc,0x09afeb7c),LL(0x598673a3,0x31e02613),LL(0x7d46db00,0x5d98e557),LL(0x9d985b28,0xfc21fb8c),
   LL(0xb0843e0b,0xc9040116),LL(0x69b04531,0x53b1b3a8),LL(0x85d7d830,0xdd1649f0),LL(0xcb7427e8,0xbb3bcc87), LL(0xc93dce83,0x77261100),LL(0xa1922a2a,0x7e79da61),LL(0xf3149ce8,0x587a2b02),LL(0xde92ec83,0x147e1384),
   LL(0xaf077f30,0x484c83d3),LL(0x0658b53a,0xea78f844),LL(0x027aec53,0x912076c2),LL(0x93c8177d,0xf34714e3), LL(0xc2376c84,0x37ef5d15),LL(0x3d1aa783,0x8315b659),LL(0xef852a90,0x3a75c484),LL(0x16086bd4,0x0ba0c58a),
   LL(0x529a6d48,0x29688d7a),LL(0xc2f19203,0x9c7f250d),LL(0x682e2df9,0x123042fb),LL(0xad8121bc,0x2b7587e7), LL(0xe0182a65,0x30fc0233),LL(0xe3e1128a,0xb82ecf87),LL(0x93fb098f,0x71682861),LL(0x85e9e6a7,0x043e21ae),
   LL(0x66c834ea,0xab5b49d6),LL(0x47414287,0x3be43e18),LL(0x219a2a47,0xf40fb859),LL(0xcc58df3c,0x0e6559e9), LL(0x0c6615b4,0xfe1dfe8e),LL(0x56459d70,0x14abc8fd),LL(0x05de0386,0x7be0fa8e),LL(0xe9035c7c,0x8e63ef68),
   LL(0x53b31e91,0x116401b4),LL(0x4436b4d8,0x0cba7ad4),LL(0x107afd66,0x9151f9a0),LL(0x1f0ee4c4,0xafaca8d0), LL(0x9ee9761c,0x75fe5c1d),LL(0xf0c0588f,0x3497a16b),LL(0x0304804c,0x3ee2bebd),LL(0xc2c990b9,0xa8fb9a60),
   LL(0x39251114,0xd14d32fe),LL(0xcac73366,0x36bf25bc),LL(0xdba7495c,0xc9562c66),LL(0x46ad348b,0x324d301b), LL(0xd670407e,0x9f46620c),LL(0xe3733a01,0x0ea8d4f1),LL(0xb0c324e0,0xd396d532),LL(0x03c317cd,0x5b211a0e),
   LL(0x5ffe7b37,0x090d7d20),LL(0x1747d2da,0x3b7f3efb),LL(0xb54fc519,0xa2cb525f),LL(0xf66a971e,0x6e220932), LL(0xb486d440,0xddc160df),LL(0x3fe13465,0x7fcfec46),LL(0x76e4c151,0x83da7e4e),LL(0xd8d302b5,0xd6fa48a1),
   LL(0x5872cd88,0xc6304f26),LL(0x278b90a1,0x806c1d3c),LL(0xcaf0bc1c,0x3553e725),LL(0xbb9d8d5c,0xff59e603), LL(0x7a0b85dd,0xa4550f32),LL(0x93ecc217,0xdec5720a),LL(0x69d62213,0x0b88b741),LL(0x5b365955,0x7212f245),
   LL(0xb5cae787,0x20764111),LL(0x1dfd3124,0x13cb7f58),LL(0x1175aefb,0x2dca77da),LL(0xffaae775,0xeb75466b), LL(0xdb6cff32,0x74d76f3b),LL(0x61fcda9a,0x7440f37a),LL(0xb525028b,0x1bb3ac92),LL(0xa1975f29,0x20fbf8f7),
   LL(0xdf83097f,0x982692e1),LL(0x554b0800,0x28738f6c),LL(0xa2ce2f2f,0xdc703717),LL(0x40814194,0x7913b93c), LL(0x1fe89636,0x04924593),LL(0xf78834a6,0x7b98443f),LL(0x5114a5a1,0x11c6ab01),LL(0xffba5f4c,0x60deb383),
   LL(0x01a982e6,0x4caa54c6),LL(0x3491cd26,0x1dd35e11),LL(0x7cbd6b05,0x973c315f),LL(0x52494724,0xcab00775), LL(0x6565e15a,0x04659b1f),LL(0x8c8fb026,0xbf30f529),LL(0xa8a0de37,0xfc21641b),LL(0xfa5e5114,0xe9c7a366),
   LL(0x52f03ad8,0xdb849ca5),LL(0x024e35c0,0xc7e8dbe9),LL(0xcfc3c789,0xa1a2bbac),LL(0x9c26f262,0xbf733e7d), LL(0xb8444823,0x882ffbf5),LL(0x6bf8483b,0xb7224e88),LL(0x65bef640,0x53023b8b),LL(0xd4d5f8cd,0xaabfec91),
   LL(0x079ea1bd,0xa40e1510),LL(0xd05d5d26,0x1ad9addc),LL(0x13e68d4f,0xdb3f2eab),LL(0x640f803f,0x1cff1ae2), LL(0xd4cee117,0xe0e7b749),LL(0x4036d909,0x8e9f275b),LL(0x8f4d4c38,0xce34e31d),LL(0xd75130fc,0x22b37f69),
   LL(0xb4014604,0x83e0f1fd),LL(0x89415078,0xa8ce9919),LL(0x41792efe,0x82375b75),LL(0x97d4515b,0x4f59bf5c), LL(0x923a277d,0xac4f324f),LL(0x650f3406,0xd9bc9b7d),LL(0x8a39bc51,0xc6fa87d1),LL(0x5ccc108f,0x82588530),
   LL(0x82e4c634,0x5ced3c9f),LL(0x3a4464f8,0x8efb8314),LL(0x7a1dca25,0xe706381b),LL(0x5a2a412b,0x6cd15a3c), LL(0xbfcd8fb5,0x9347a8fd),LL(0x6e54cd22,0x31db2eef),LL(0xf8d8932f,0xc4aeb11e),LL(0x344411af,0x11e7c1ed),
   LL(0xdc9a151e,0x2653050c),LL(0x3bb0a859,0x9edbfc08),LL(0xfd5691e7,0x926c81c7),LL(0x6f39019a,0x9c1b2342), LL(0x7f8474b9,0x64a81c8b),LL(0x01761819,0x90657c07),LL(0x55e0375a,0x390b3331),LL(0xb6ebc47d,0xc676c626),
   LL(0xb7d6dee8,0x51623247),LL(0x79659313,0x0948d927),LL(0xe9ab35ed,0x99700161),LL(0x8ddde408,0x06cc32b4), LL(0x061ef338,0x6f2fd664),LL(0xc202e9ed,0x1606fa02),LL(0x929ba99b,0x55388bc1),LL(0x1e81df69,0xc4428c5e),
   LL(0xf91b0b2a,0xce2028ae),LL(0xf03dfd3f,0xce870a23),LL(0x0affe8ed,0x66ec2c87),LL(0x284d0c00,0xb205fb46), LL(0x44cefa48,0xbf5dffe7),LL(0xa19876d7,0xb6fc37a8),LL(0x08b72863,0xbecfa84c),LL(0x2576374f,0xd7205ff5),
   LL(0x8887de41,0x80330d32),LL(0x869ea534,0x5de0df0c),LL(0x3c56ea17,0x13f42753),LL(0x452b1a78,0xeb1f6069), LL(0xe30ea15c,0x50474396),LL(0xc1494125,0x575816a1),LL(0xfe6bb38f,0xbe1ce55b),LL(0x96ae30f7,0xb901a948),
   LL(0xd8fc3548,0xe5af0f08),LL(0xd73bfd08,0x5010b5d0),LL(0x53fe655a,0x993d2880),LL(0x1c1309fd,0x99f2630b), LL(0xb4e3b76f,0xd8677baf),LL(0xb840784b,0x14e51ddc),LL(0xbf0092ce,0x326c750c),LL(0xf528320f,0xc83d306b),
   LL(0x77d4715c,0xc4456715),LL(0x6b703235,0xd30019f9),LL(0xd669e986,0x207ccb2e),LL(0xf6dbfc28,0x57c824af), LL(0xd8f92a23,0xf0eb532f),LL(0x9bb98fd2,0x4a557fd4),LL(0xc1e6199a,0xa57acea7),LL(0x8b94b1ed,0x0c663820),
   LL(0xf83a9266,0x9b42be8f),LL(0x0101bd45,0xc7741c97),LL(0x07bd9ceb,0x95770c11),LL(0x8b2e0744,0x1f50250a), LL(0x1477b654,0xf762eec8),LL(0x15efe59a,0xc65b900e),LL(0x9546a897,0x88c96148),LL(0xc30b4d7c,0x7e8025b3),
   LL(0x12045cf9,0xae4065ef),LL(0x9ccce8bd,0x6fcb2caf),LL(0xf2cf6525,0x1fa0ba4e),LL(0xcb72c312,0xf683125d), LL(0xe312410e,0xa01da4ea),LL(0x6cd8e830,0x67e28677),LL(0x98fb3f07,0xabd95752),LL(0xeef649a5,0x05f11e11),
   LL(0x9d3472c2,0xba47faef),LL(0xc77d1345,0x3adff697),LL(0xdd15afee,0x4761fa04),LL(0xb9e69462,0x64f1f61a), LL(0x9bfb9093,0xfa691fab),LL(0xa1133dfe,0x3df8ae8f),LL(0x58cc710d,0xcd5f8967),LL(0x16c7fe79,0xfbb88d50),
   LL(0xe88c50d1,0x8e011b4c),LL(0xa8771c4f,0x7532e807),LL(0xe2278ee4,0x64c78a48),LL(0x3845072a,0x0b283e83), LL(0x49e69274,0x98a6f291),LL(0x1868b21c,0xb96e9668),LL(0xb1a8908e,0x38f0adc2),LL(0x1feb829d,0x90afcff7),
   LL(0x210b0856,0x9915a383),LL(0xdef04889,0xa5a80602),LL(0x7c64d509,0x800e9af9),LL(0xb8996f6f,0x81382d0b), LL(0x81927e27,0x490eba53),LL(0x4af50182,0x46c63b32),LL(0xd3ad62ce,0x784c5fd9),LL(0xf8ae8736,0xe4fa1870),
   LL(0xd7466b25,0x4ec9d0bc),LL(0xdb235c65,0x84ddbe1a),LL(0x163c1688,0x5e2645ee),LL(0x00eba747,0x570bd00e), LL(0x128bfa0f,0xfa51b629),LL(0x6c1d3b68,0x92fce1bd),LL(0xb66778b1,0x3e7361dc),LL(0x5561d2bb,0x9c7d249d),
   LL(0x0bbc6229,0xa40b28bf),LL(0xdfd91497,0x1c83c05e),LL(0xf083df05,0x5f9f5154),LL(0xeee66c9d,0xbac38b3c), LL(0xec0dfcfd,0xf71db7e3),LL(0x8b0a8416,0xf2ecda8e),LL(0x7812aa66,0x52fddd86),LL(0x4e6f4272,0x2896ef10),
   LL(0x0fe9a745,0xff27186a),LL(0x49ca70db,0x08249fcd),LL(0x441cac49,0x7425a2e6),LL(0xece5ff57,0xf4a0885a), LL(0x7d7ead58,0x6e2cb731),LL(0x1898d104,0xf96cf7d6),LL(0x4f2c9a89,0xafe67c9d),LL(0x1c7bf5bc,0x89895a50),
   LL(0x573cecfa,0xdc7cb8e5),LL(0xd15f03e6,0x66497eae),LL(0x3f084420,0x6bc0de69),LL(0xacd532b0,0x323b9b36), LL(0x0115a3c1,0xcfed390a),LL(0x2d65ca0e,0x9414c40b),LL(0x2f530c78,0x641406bd),LL(0x833438f2,0x29369a44),
   LL(0x903fa271,0x996884f5),LL(0xb9da921e,0xe6da0fd2),LL(0x5db01e54,0xa6f2f269),LL(0x6876214e,0x1ee3e9bd), LL(0xe27a9497,0xa26e181c),LL(0x8e215e04,0x36d254e4),LL(0x252cabca,0x42f32a6c),LL(0x80b57614,0x99481487),
   LL(0x40d9cae1,0x4c4dfe69),LL(0x11a10f09,0x05869580),LL(0x3491b64b,0xca287b57),LL(0x3fd4a53b,0x77862d5d), LL(0x50349126,0xbf94856e),LL(0x71c5268f,0x2be30bd1),LL(0xcbb650a6,0x10393f19),LL(0x778cf9fd,0x639531fe),
   LL(0xb2935359,0x02556a11),LL(0xaf8c126e,0xda38aa96),LL(0x0960167f,0x47dbe6c2),LL(0x501901cd,0x37bbabb6), LL(0x2c947778,0xb6e979e0),LL(0x7a1a1dc6,0xd69a5175),LL(0x9d9faf0c,0xc3ed5095),LL(0x1d5fa5f0,0x4dd9c096),
   LL(0x64f16ea8,0xa0c4304d),LL(0x7e718623,0x8b1cac16),LL(0x7c67f03e,0x0b576546),LL(0xcbd88c01,0x559cf5ad), LL(0x0e2af19a,0x074877bb),LL(0xa1228c92,0x1f717ec1),LL(0x326e8920,0x70bcb800),LL(0x4f312804,0xec6e2c5c),
   LL(0x3fca4752,0x426aea7d),LL(0x2211f62a,0xf12c0949),LL(0x7be7b6b5,0x24beecd8),LL(0x36d7a27d,0xb77eaf4c), LL(0xfda78fd3,0x154c2781),LL(0x264eeabe,0x848a83b0),LL(0x4ffe2bc4,0x81287ef0),LL(0xb6b6fc2a,0x7b6d88c6),
   LL(0xce417d99,0x805fb947),LL(0x8b916cc4,0x4b93dcc3),LL(0x21273323,0x72e65bb3),LL(0x6ea9886e,0xbcc1badd), LL(0x4bc5ee85,0x0e223011),LL(0xc18ee1e4,0xa561be74),LL(0xa6bcf1f1,0x762fd2d4),LL(0x95231489,0x50e6a5a4),
   LL(0xa00b500b,0xca96001f),LL(0x5d7dcdf5,0x5c098cfc),LL(0x8c446a85,0xa64e2d2e),LL(0x971f3c62,0xbae9bcf1), LL(0x8435a2c5,0x4ec22683),LL(0x4bad4643,0x8ceaed6c),LL(0xccccf4e3,0xe9f8fb47),LL(0x1ce3b21e,0xbd4f3fa4),
   LL(0xa3db3292,0xd79fb110),LL(0xb536c66a,0xe28a37da),LL(0x8e49e6a9,0x279ce87b),LL(0xfdcec8e3,0x70ccfe8d), LL(0x3ba464b2,0x2193e4e0),LL(0xaca9a398,0x0f39d60e),LL(0xf82c12ab,0x7d7932af),LL(0x91e7e0f7,0xd8ff50ed),
   LL(0xfa28a7e0,0xea961058),LL(0x0bf5ec74,0xc726cf25),LL(0xdb229666,0xe74d55c8),LL(0xa57f5799,0x0bd9abbf), LL(0x4dfc47b3,0x7479ef07),LL(0x0c52f91d,0xd9c65fc3),LL(0x36a8bde2,0x8e0283fe),LL(0x7d4b7280,0xa32a8b5e),
   LL(0x12e83233,0x6a677c61),LL(0xdcc9bf28,0x0fbb3512),LL(0x0d780f61,0x562e8ea5),LL(0x1dc4e89c,0x0db8b22b), LL(0x89be0144,0x0a6fd1fb),LL(0xca57113b,0x8c77d246),LL(0xff09c91c,0x4639075d),LL(0x5060824c,0x5b47b17f),
   LL(0x16287b52,0x58aea2b0),LL(0xd0cd8eb0,0xa1343520),LL(0xc5d58573,0x6148b4d0),LL(0x291c68ae,0xdd2b6170), LL(0x1da3b3b7,0xa61b3929),LL(0x08c4ac10,0x5f946d79),LL(0x7217d583,0x4105d4a5),LL(0x25e6de5e,0x5061da3d),
   LL(0xec1b4991,0x3113940d),LL(0x36f485ae,0xf12195e1),LL(0x731a2ee0,0xa7507fb2),LL(0x6e9e196e,0x95057a8e), LL(0x2e130136,0xa3c2c911),LL(0x33c60d15,0x97dfbb36),LL(0xb300ee2b,0xcaf3c581),LL(0xf4bac8b8,0x77f25d90),
   LL(0x6d840cd6,0xdb1c4f98),LL(0xe634288c,0x471d62c0),LL(0xcec8a161,0x8ec2f85e),LL(0xfa6f4ae2,0x41f37cbc), LL(0x4b709985,0x6793a20f),LL(0xefa8985b,0x7a7bd33b),LL(0x938e6446,0x2c6a3fbd),LL(0x2a8d47c1,0x19042619),
   LL(0xcc36975f,0x16848667),LL(0x9d5f1dfb,0x02acf168),LL(0x613baa94,0x62d41ad4),LL(0x9f684670,0xb56fbb92), LL(0xe9e40569,0xce610d0d),LL(0x35489fef,0x7b99c65f),LL(0x3df18b97,0x0c88ad1b),LL(0x5d0e9edb,0x81b7d9be),
   LL(0xc716cc0a,0xd85218c0),LL(0x85691c49,0xf4b5ff90),LL(0xce356ac6,0xa4fd666b),LL(0x4b327a7a,0x17c72895), LL(0xda6be7de,0xf93d5085),LL(0x3301d34e,0xff71530e),LL(0xd8f448e8,0x4cd96442),LL(0x2ed18ffa,0x9283d331),
   LL(0x2a849870,0x4d33dd99),LL(0x41576335,0xa716964b),LL(0x179be0e5,0xff5e3a9b),LL(0x83b13632,0x5b9d6b1b), LL(0xa52f313b,0x3b8bd7d4),LL(0x637a4660,0xc9dd95a0),LL(0x0b3e218f,0x30035962),LL(0xc7b28a3c,0xce1481a3),
   LL(0x43228d83,0xab41b43a),LL(0x4ad63f99,0x24ae1c30),LL(0x46a51229,0x8e525f1a),LL(0xcd26d2b4,0x14af860f), LL(0x3f714aa1,0xd6baef61),LL(0xeb78795e,0xf51865ad),LL(0xe6a9d694,0xd3e21fce),LL(0x8a37b527,0x82ceb1dd),
}
};
#endif /* _DISABLE_ECP_256R1_HARDCODED_BP_TBL_ */
#endif  /* _IPP_DATA */


IPP_OWN_DEFN (const cpPrecompAP*, gfpec_precom_nistP256r1_fun, (void))
{
   static cpPrecompAP t = {
      /* w */                  7,
      /* select function */    p256r1_select_ap_w7,
      /* precomputed data */   (BNU_CHUNK_T*)ec_p256r1_precomputed
   };
   return &t;
}
