(*****************************************************************************

  Liquidsoap, a programmable audio stream generator.
  Copyright 2003-2022 Savonet team

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details, fully stated in the COPYING
  file at the root of the liquidsoap distribution.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA

 *****************************************************************************)

let log = Log.make ["decoder"; "image"; "metadata"]

(** Configuration keys. *)
let mime_types =
  Dtools.Conf.list
    ~p:(Decoder.conf_mime_types#plug "image_metadata")
    "Mime-types used for decoding metadata using native parser."
    ~d:["image/png"; "image/jpeg"]

let conf_image =
  Dtools.Conf.void
    ~p:(Decoder.conf_decoder#plug "image_metadata")
    "Native image metadata parser settings."

let file_extensions =
  Dtools.Conf.list
    ~p:(Decoder.conf_file_extensions#plug "image_metadata")
    "File extensions used for decoding metadata using native parser."
    ~d:["png"; "jpg"; "jpeg"]

let get_tags fname =
  try
    if
      not
        (Decoder.test_file ~log ~mimes:mime_types#get
           ~extensions:file_extensions#get fname)
    then raise Metadata.Invalid;
    Metadata.Image.parse_file fname
  with
    | Metadata.Invalid -> []
    | e ->
        let bt = Printexc.get_backtrace () in
        Utils.log_exception ~log ~bt
          (Printf.sprintf "Error while decoding file tags: %s"
             (Printexc.to_string e));
        raise Not_found

let () = Request.mresolvers#register "IMAGEMD" get_tags
