/***************************** LICENSE START ***********************************

 Copyright 2012 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#ifndef MvQLayerModel_H
#define MvQLayerModel_H

#include <QAbstractItemModel>
#include <QList>
#include <QSortFilterProxyModel>
#include <QStyledItemDelegate>
#include <QStyleOptionViewItem>
#include <QTreeView>

#include "Layer.h"
#include "MvQDragDrop.h"

class MgQLayerItem;
class MgQPlotScene;
class MgQSceneItem;
class MvQLayerContentsIcon;

using namespace std;

class MvQLayerDelegate : public QStyledItemDelegate
{
public:
    MvQLayerDelegate(QWidget* parent = 0);
    void paint(QPainter* painter, const QStyleOptionViewItem& option,
               const QModelIndex& index) const;
    QSize sizeHint(const QStyleOptionViewItem& option, const QModelIndex& index) const;

private:
    int itemHeight_;
};


class MvQLayerModel : public QAbstractItemModel
{
    Q_OBJECT

public:
    MvQLayerModel(MgQPlotScene*);
    ~MvQLayerModel();

    enum CustomItemRole
    {
        VisibleRole = Qt::UserRole + 1,
        LevelRole   = Qt::UserRole + 2
    };

    int columnCount(const QModelIndex& parent = QModelIndex()) const;
    int rowCount(const QModelIndex& parent = QModelIndex()) const;

    QVariant data(const QModelIndex&, int role = Qt::DisplayRole) const;
    bool setData(const QModelIndex&, const QVariant&, int role = Qt::EditRole);
    //QVariant headerData(int,Qt::Orientation,int role = Qt::DisplayRole ) const;

    QModelIndex index(int, int, const QModelIndex& parent = QModelIndex()) const;
    QModelIndex parent(const QModelIndex&) const;

    // Layers functions
    QList<MgQLayerItem*>& layers() { return layers_; }
    MgQLayerItem* layer(const QModelIndex&) const;
    MgQLayerItem* layerFromAnyLevel(const QModelIndex&) const;

    string layerId(const QModelIndex&);
    QString layerName(const QModelIndex&);
    void layerMetaData(const QModelIndex&, MetaDataCollector&);
    QString layerMetaData(const QModelIndex&, QString);

    void layersAreAboutToChange();
    void resetLayers(MgQSceneItem*);

    void moveUp(const QModelIndex& index);
    void moveDown(const QModelIndex& index);
    void moveTop(const QModelIndex& index);
    void moveBottom(const QModelIndex& index);

    Qt::ItemFlags flags(const QModelIndex&) const;

    Qt::DropActions supportedDropActions() const;
    QStringList mimeTypes() const;
    QMimeData* mimeData(const QModelIndexList&) const;
    bool dropMimeData(const QMimeData* data, Qt::DropAction action,
                      int row, int column, const QModelIndex& parent);

    void setTransparency(const QModelIndex&, int);
    int transparency(const QModelIndex&);
    MgQLayerItem* rowToLayer(int) const;
    int layerToRow(MgQLayerItem*) const;
    int indexToLevel(const QModelIndex&) const;
    MvQLayerContentsIcon* indexToIcon(const QModelIndex&) const;

    int idToParentRow(int) const;

    // Set/unset highlighting layers for drag&drop operations
    void setHighlightedLayer(const QModelIndex&, int);

    int presentableId()
    {
        return presentableId_;
    }

    // Check if icon is to be shown
    bool checkIconVisibility(MvQLayerContentsIcon*, int& dataId) const;
    bool showIcon(MvQLayerContentsIcon*) const;

    // Check if layer is to be shown
    bool checkLayerVisibility(MgQLayerItem*) const;

signals:
    void layerVisibilityChanged(QString, bool);
    void layerTransparencyChanged(QString, int);
    void layerStackingOrderChanged(QList<QPair<QString, int> >);
    void layerUpdate();
    void layerDragFinished();

protected:
    void clearIcons();
    MvQLayerContentsIcon* layerIcon(MgQLayerItem*, int) const;
    int rowToStackLevel(int row);
    QString label(MgQLayerItem*, const int, const int) const;
    int idToLevel(int) const;
    void moveLayer(int, int);

    MgQPlotScene* scene_;
    MgQSceneItem* sceneItem_;
    QList<MgQLayerItem*> layers_;
    int higlightedId_;
    Qt::CheckState state_;
    mutable QHash<MgQLayerItem*, QList<MvQLayerContentsIcon*> > icons_;

    int presentableId_;  // current presentable/scene id
};


//----------------------------------------------------------------------------

// Filter icons to be shown
class MvQLayerFilterModel : public QSortFilterProxyModel
{
public:
    MvQLayerFilterModel(QObject* parent = 0);

    bool filterAcceptsRow(int, const QModelIndex&) const;
};


//----------------------------------------------------------------------------

// Handle external drag&drop
class MvQLayerView : public QTreeView
{
    Q_OBJECT

public:
    explicit MvQLayerView(QWidget* parent = 0);

    void setModel(MvQLayerModel*, MvQLayerFilterModel*);

protected:
    void dragEnterEvent(QDragEnterEvent*);
    void dragLeaveEvent(QDragLeaveEvent*);
    void dragMoveEvent(QDragMoveEvent*);
    void dropEvent(QDropEvent*);

    // Handle MvQDropTarget facility
    void checkDropTarget(QDropEvent*);
    void removeDropTarget();

    MvQLayerModel* model_;
    MvQLayerFilterModel* filterModel_;
    int layerId_;

signals:
    void iconDropped(const MvQDrop&, const QModelIndex&);
};

#endif
