/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var logger = require( 'debug' );
var hasOwnProp = require( '@stdlib/assert/has-own-property' );


// VARIABLES //

var debug = logger( 'group-by-async:limit' );


// MAIN //

/**
* Invokes an indicator function once for each element in a collection, limiting the number of concurrently pending functions.
*
* ## Notes
*
* -   We need to cache the collection value to prevent the edge case where, during the invocation of the indicator function, the element at index `i` is swapped for some other value. For some, that might be a feature; here, we take the stance that one should be less clever.
* -   Checking for an "own" property is necessary to guard against the edge case where an indicator function returns a group identifier which matches a method or property on the `Object` prototype.
*
*
* @private
* @param {Collection} collection - input collection
* @param {Options} opts - function options
* @param {*} [opts.thisArg] - execution context
* @param {PositiveInteger} [opts.limit] - maximum number of pending function invocations
* @param {string} [options.returns] - output format
* @param {Function} indicator - indicator function
* @param {Callback} done - function to invoke upon completion or upon encountering an error
* @returns {void}
*/
function limit( collection, opts, indicator, done ) {
	var maxIndex;
	var count;
	var flg;
	var lim;
	var len;
	var idx;
	var out;
	var i;

	len = collection.length;
	debug( 'Collection length: %d', len );

	out = {};
	if ( len === 0 ) {
		debug( 'Finished processing a collection.' );
		return done( null, out );
	}
	if ( len < opts.limit ) {
		lim = len;
	} else {
		lim = opts.limit;
	}
	debug( 'Concurrency limit: %d', lim );
	debug( 'Number of arguments: %d', indicator.length );

	maxIndex = len - 1;
	count = 0;
	idx = -1;
	for ( i = 0; i < lim; i++ ) {
		// This guard is necessary to protect against synchronous functions which exhaust all collection elements...
		if ( idx < maxIndex ) {
			next(); // eslint-disable-line callback-return
		}
	}
	/**
	* Callback to invoke a provided function for the next element in a collection.
	*
	* @private
	*/
	function next() {
		var v;
		var j;

		idx += 1;
		j = idx;
		v = collection[ j ];

		debug( 'Collection element %d: %s.', j, JSON.stringify( v ) );
		if ( indicator.length === 2 ) {
			indicator.call( opts.thisArg, v, cb );
		} else if ( indicator.length === 3 ) {
			indicator.call( opts.thisArg, v, j, cb );
		} else {
			indicator.call( opts.thisArg, v, j, collection, cb );
		}
		/**
		* Callback invoked once a provided function finishes processing a collection element.
		*
		* @private
		* @param {*} [error] - error
		* @param {*} [group] - group
		* @returns {void}
		*/
		function cb( error, group ) {
			if ( flg ) {
				// Prevent further processing of collection elements:
				return;
			}
			if ( error ) {
				flg = true;
				return clbk( error );
			}
			debug( 'Collection element %d group: %s.', j, group );

			// Default is to return values...
			if ( opts.returns === 'indices' ) {
				if ( hasOwnProp( out, group ) ) {
					out[ group ].push( j );
				} else {
					out[ group ] = [ j ];
				}
			} else if ( opts.returns === '*' ) {
				if ( hasOwnProp( out, group ) ) {
					out[ group ].push( [ j, v ] );
				} else {
					out[ group ] = [ [ j, v ] ];
				}
			} else if ( hasOwnProp( out, group ) ) {
				out[ group ].push( v );
			} else {
				out[ group ] = [ v ];
			}
			clbk();
		}
	}

	/**
	* Callback invoked once ready to process the next collection element.
	*
	* @private
	* @param {*} [error] - error
	* @returns {void}
	*/
	function clbk( error ) {
		if ( error ) {
			debug( 'Encountered an error: %s', error.message );
			return done( error );
		}
		count += 1;
		debug( 'Processed %d of %d collection elements.', count, len );
		if ( idx < maxIndex ) {
			return next();
		}
		if ( count === len ) {
			debug( 'Finished processing a collection.' );
			return done( null, out );
		}
	}
}


// EXPORTS //

module.exports = limit;
