/**********************************************************************
 * $test_invoke_options example -- PLI application using VPI routines
 *
 * C source to test for aany simulation invocation option, including
 * user-defined options.  The system function will return 1 if an
 * invocation option was present when simulation was invoked, and will
 * return 0 if the option did not exist.
 *
 * Usage: reg flag;
          flag = $test_invoke_options("<invocation_option>");
 *   Where <invocation_option can be any + or - option.
 *
 * Debug tip: Uncomment the line "#define PLIbook_verbose" to enable
 * this application to list all simulation invocation options.
 *
 * For the book, "The Verilog PLI Handbook" by Stuart Sutherland
 *  Book copyright 1999, Kluwer Academic Publishers, Norwell, MA, USA
 *   Contact: www.wkap.il
 *  Example copyright 1998, Sutherland HDL Inc, Portland, Oregon, USA
 *   Contact: www.sutherland.com or (503) 692-0898
 *********************************************************************/

#define PLIbook_verbose 1 /* uncomment to list all invoke options */

#include <stdlib.h>    /* ANSI C standard library */
#include <stdio.h>     /* ANSI C standard input/output library */
#include "vpi_user.h"  /* IEEE 1364 PLI VPI routine library  */

/* prototypes of routines in this PLI application */
int  PLIbook_TestInvokeOptions_calltf(), 
     PLIbook_TestInvokeOptions_compiletf(), 
     PLIbook_TestInvokeOptions_sizetf(),
     PLIbook_GetOptions(); 
void PLIbook_ScanCommandFile();

/**********************************************************************
 * VPI Registration Data
 *********************************************************************/
void PLIbook_TestInvokeOptions_register()
{
  s_vpi_systf_data tf_data;
  tf_data.type        = vpiSysFunc;
  tf_data.sysfunctype = vpiSysFuncSized;
  tf_data.tfname      = "$test_invoke_options";
  tf_data.calltf      = PLIbook_TestInvokeOptions_calltf;
  tf_data.compiletf   = PLIbook_TestInvokeOptions_compiletf;
  tf_data.sizetf      = PLIbook_TestInvokeOptions_sizetf;
  tf_data.user_data   = NULL;
  vpi_register_systf(&tf_data);
}

/**********************************************************************
 * sizetf application
 *********************************************************************/
int PLIbook_TestInvokeOptions_sizetf(char *user_data)
{
  return(1);    /* $test_invoke_options returns 1-bit value */
}

/**********************************************************************
 * calltf routine
 *********************************************************************/
int PLIbook_TestInvokeOptions_calltf(char *user_data)
{
  vpiHandle        systf_h, arg_itr, arg_h;
  char            *option_name;
  s_vpi_value      value_s;
  s_vpi_vlog_info  sim_info;
  int              found;

  vpi_get_vlog_info(&sim_info);

  /* get system function arg--compiletf already verified correctness */
  systf_h = vpi_handle(vpiSysTfCall, NULL);
  arg_itr = vpi_iterate(vpiArgument, systf_h);
  arg_h   = vpi_scan(arg_itr);
  vpi_free_object(arg_itr); /* free iterator -- did not scan to null */

  /* read target option name from first tfarg */
  value_s.format = vpiStringVal;
  vpi_get_value(arg_h, &value_s);
  option_name = value_s.value.str;

  /* test for target option and return true/false to system function */
  found = PLIbook_GetOptions(option_name,sim_info.argc,sim_info.argv);
  value_s.format = vpiIntVal;
  value_s.value.integer = found;
  vpi_put_value(systf_h, &value_s, NULL, vpiNoDelay);
  return(0);
}

int PLIbook_optfound = 0; /* global variable for option found flag */
int PLIbook_indent = 0;   /* global variable to format text indenting */

int PLIbook_GetOptions(char *option, int argc, char **argv)
{
  int i;
  PLIbook_optfound = 0;
  PLIbook_indent = 0;
  for (i=0; i<argc; i++) {
    #ifdef PLIbook_verbose
      vpi_printf("%s\n", *argv); 
    #endif
    if (strcmp(*argv, option) == 0) {
      PLIbook_optfound = 1;
    }
    if (strcmp(*argv, "-f") == 0) {
      argv++;  /* next arg is address to array of strings */
      i++;
      PLIbook_ScanCommandFile(option, (char **)*argv);
    }
    argv++; /* increment to next argument */
  }
  return(PLIbook_optfound);
}

void PLIbook_ScanCommandFile(char *option, char **arg)
{
  int i;

  #ifdef PLIbook_verbose
    PLIbook_indent += 4; /* increase text indentation */
  #endif
  while ( *arg != NULL ) { /* loop until null termination */
    #ifdef PLIbook_verbose
      for (i=0; i<=PLIbook_indent; i++)
        vpi_printf(" ");
      vpi_printf("%s\n", *arg); 
    #endif
    if (strcmp(*arg, option) == 0) {
      PLIbook_optfound = 1;
    }
    if (strcmp(*arg, "-f") == 0) {
      arg++;  /* next arg is address to array of strings */
      PLIbook_ScanCommandFile(option, (char **)*arg);
    }
    arg++;
  }
  #ifdef PLIbook_verbose
    PLIbook_indent -= 4; /* decrease text indentation */
  #endif
  return;
}

/**********************************************************************
 * compiletf routine
 *********************************************************************/
int PLIbook_TestInvokeOptions_compiletf(char *user_data)
{
  vpiHandle   systf_h, arg_itr, arg_h;
  int         err = 0;
  
  systf_h = vpi_handle(vpiSysTfCall, NULL);
  arg_itr = vpi_iterate(vpiArgument, systf_h);
  if (arg_itr == NULL) {
    vpi_printf("ERROR: $test_invoke_options requires 1 argument\n");
    tf_dofinish();
    return(0);
  }
  arg_h = vpi_scan(arg_itr);  /* get handle for first tfarg */
  if (vpi_get(vpiType, arg_h) != vpiConstant) {
    vpi_printf("$test_invoke_options arg must be a quoted name\n");
    err = 1;
  }
  else if (vpi_get(vpiConstType, arg_h) != vpiStringConst) {
    vpi_printf("$test_invoke_options arg must be a string\n");
    err = 1;
  }
  if (vpi_scan(arg_itr) != NULL) {
    vpi_printf("test_invoke_options requires only 1 argument\n");
    vpi_free_object(arg_itr);
    err = 1;
  }
  if (err)
    tf_dofinish();

  return(0);
}

/*********************************************************************/
