// Copyright 2017 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

package org.chromium.components.payments;

import androidx.annotation.VisibleForTesting;

import org.chromium.base.ThreadUtils;
import org.chromium.base.annotations.CalledByNative;
import org.chromium.base.annotations.JNINamespace;
import org.chromium.base.annotations.NativeMethods;
import org.chromium.content_public.browser.WebContents;
import org.chromium.url.GURL;
import org.chromium.url.Origin;
import org.chromium.url.URI;

/**
 * See comment in:
 * components/payments/core/payment_manifest_downloader.h
 */
@JNINamespace("payments")
public class PaymentManifestDownloader {
    /** Interface for the callback to invoke when finished downloading. */
    public interface ManifestDownloadCallback {
        /**
         * Called on successful download of a payment method manifest.
         *
         * @param paymentMethodManifestUrl The URL of the payment method manifest after all
         * redirects and the optional HTTP Link rel=payment-method-manifest header have been
         * followed.
         * @param paymentMethodManifestOrigin The origin of the payment method manifest after all
         * redirects and the optional HTTP Link rel=payment-method-manifest header have been
         * followed.
         * @param content The successfully downloaded payment method manifest.
         */
        @CalledByNative("ManifestDownloadCallback")
        void onPaymentMethodManifestDownloadSuccess(
                URI paymentMethodManifestUrl, Origin paymentMethodManifestOrigin, String content);

        /**
         * Called on successful download of a web app manifest.
         *
         * @param content The successfully downloaded web app manifest.
         */
        @CalledByNative("ManifestDownloadCallback")
        void onWebAppManifestDownloadSuccess(String content);

        /**
         * Called on failed download.
         *
         * @param errorMessage The error message, which could be empty or null.
         */
        @CalledByNative("ManifestDownloadCallback")
        void onManifestDownloadFailure(String errorMessage);
    }

    private long mNativeObject;

    /**
     * Initializes the native downloader.
     *
     * @param webContents The web contents to use as the context for the downloads. If this goes
     *                    away, pending downloads are cancelled.
     */
    public void initialize(WebContents webContents) {
        ThreadUtils.assertOnUiThread();
        assert mNativeObject == 0;
        mNativeObject = PaymentManifestDownloaderJni.get().init(webContents);
    }

    /** @return Whether the native downloader is initialized. */
    public boolean isInitialized() {
        ThreadUtils.assertOnUiThread();
        return mNativeObject != 0;
    }

    /**
     * Downloads the payment method manifest file asynchronously.
     *
     * @param merchantOrigin The origin of the iframe that invoked the PaymentRequest API.
     * @param methodName     The payment method name that is a URI with HTTPS scheme.
     * @param callback       The callback to invoke when finished downloading.
     */
    public void downloadPaymentMethodManifest(
            Origin merchantOrigin, URI methodName, ManifestDownloadCallback callback) {
        ThreadUtils.assertOnUiThread();
        assert mNativeObject != 0;
        assert merchantOrigin != null;
        PaymentManifestDownloaderJni.get().downloadPaymentMethodManifest(mNativeObject,
                PaymentManifestDownloader.this, merchantOrigin, methodName, callback);
    }

    /**
     * Downloads the web app manifest file asynchronously.
     *
     * @param paymentMethodManifestOrigin The origin of the payment method manifest that is pointing
     *                                    to this web app manifest.
     * @param webAppManifestUri           The web app manifest URI with HTTPS scheme.
     * @param callback                    The callback to invoke when finished downloading.
     */
    public void downloadWebAppManifest(Origin paymentMethodManifestOrigin, URI webAppManifestUri,
            ManifestDownloadCallback callback) {
        ThreadUtils.assertOnUiThread();
        assert mNativeObject != 0;
        assert paymentMethodManifestOrigin != null;
        PaymentManifestDownloaderJni.get().downloadWebAppManifest(mNativeObject,
                PaymentManifestDownloader.this, paymentMethodManifestOrigin, webAppManifestUri,
                callback);
    }

    /** Destroys the native downloader. */
    public void destroy() {
        ThreadUtils.assertOnUiThread();
        assert mNativeObject != 0;
        PaymentManifestDownloaderJni.get().destroy(mNativeObject, PaymentManifestDownloader.this);
        mNativeObject = 0;
    }

    /** @return An opaque origin to be used in tests. */
    @VisibleForTesting
    public static Origin createOpaqueOriginForTest() {
        return PaymentManifestDownloaderJni.get().createOpaqueOriginForTest();
    }

    /**
     * Converts GURL to URI through string serialization. Needed because C++ knows only how to
     * create Java GURL objects, but web payments uses URI, which is a subclass of GURL, so casting
     * is not possible.
     *
     * TODO(crbug.com/1065577): Use GURL direclly everywhere in web payments.
     *
     * @param gurl The GURL to convert. Cannot be null. Must be valid.
     * @return The equivalent URI.
     */
    @CalledByNative
    public static URI convertGURLToURI(GURL gurl) {
        return URI.create(gurl.getPossiblyInvalidSpec());
    }

    @NativeMethods
    interface Natives {
        long init(WebContents webContents);
        void downloadPaymentMethodManifest(long nativePaymentManifestDownloaderAndroid,
                PaymentManifestDownloader caller, Origin merchantOrigin, URI methodName,
                ManifestDownloadCallback callback);
        void downloadWebAppManifest(long nativePaymentManifestDownloaderAndroid,
                PaymentManifestDownloader caller, Origin paymentMethodManifestOrigin,
                URI webAppManifestUri, ManifestDownloadCallback callback);
        void destroy(long nativePaymentManifestDownloaderAndroid, PaymentManifestDownloader caller);
        Origin createOpaqueOriginForTest();
    }
}
