/*
 * avrdude - A Downloader/Uploader for AVR device programmers
 * Copyright (C) 2007 Dick Streefland, adapted for 5.4 by Limor Fried
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * Driver for "usbtiny"-type programmers
 * Please see http://www.xs4all.nl/~dicks/avr/usbtiny/
 *        and http://www.ladyada.net/make/usbtinyisp/
 * For example schematics and detailed documentation
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <libintl.h>
#include <errno.h>
#include <sys/time.h>
#include <unistd.h>

//#include "avrdude.h"
//#include "avr.h"
//#include "pgm.h"
#include "usbtiny.h"
#include "s51dude.h"

#ifdef WIN32
#include <windows.h>
#define sleep(seconds)      Sleep(seconds * 1000) /* from mingw.org list */
#define usleep(useconds)    Sleep(useconds / 1000)
//El de windows:
//Sleep (1000) Espera 1 segundo
//Sleep (10000) Espera 10 segundo

//El de linux:
//sleep (1)  Espera 1 segundo
//sleep (10) Espera 10 segundos
//usleep (1000) Espera 1ms
//usleep (1) Espera 1us
#endif

typedef	unsigned int	uint_t;
typedef	unsigned long	ulong_t;

extern usb_dev_handle *usb_handle;


// ----------------------------------------------------------------------

// Wrapper for simple usb_control_msg messages
int usb_control (unsigned int requestid, unsigned int val, unsigned int index )
{
  int nbytes;
  nbytes = usb_control_msg( usb_handle,
			    USB_ENDPOINT_IN | USB_TYPE_VENDOR | USB_RECIP_DEVICE,
			    requestid,
			    val, index,           // 2 bytes each of data
			    NULL, 0,              // no data buffer in control messge
			    USB_TIMEOUT );        // default timeout
  if(nbytes < 0){
    print_error (_("ERROR: usbtiny_transmit. Check permissions on the device, or the cable connection"));
    exit(1);
  }

  return nbytes;
}

// Wrapper for simple usb_control_msg messages to receive data from programmer
int usb_in (unsigned int requestid, unsigned int val, unsigned int index,
		   unsigned char* buffer, int buflen, int bitclk )
{
  int nbytes;
  int timeout;

  // calculate the amout of time we expect the process to take by
  // figuring the bit-clock time and buffer size and adding to the standard USB timeout.
  timeout = USB_TIMEOUT + (buflen * bitclk) / 1000;

  nbytes = usb_control_msg( usb_handle,
			    USB_ENDPOINT_IN | USB_TYPE_VENDOR | USB_RECIP_DEVICE,
			    requestid,
			    val, index,
			    (char *)buffer, buflen,
			    timeout);
  if (nbytes != buflen) {
    fprintf(stderr, _("%s: error: usbtiny_receive: (expected %d, got %d)\n"),
	    usb_strerror(), buflen, nbytes);
    exit(1);
  }

  return nbytes;
}

// Wrapper for simple usb_control_msg messages to send data to programmer
int usb_out (unsigned int requestid, unsigned int val, unsigned int index,
		    unsigned char* buffer, int buflen, int bitclk )
{
  int nbytes;
  int timeout;

  // calculate the amout of time we expect the process to take by
  // figuring the bit-clock time and buffer size and adding to the standard USB timeout.
  timeout = USB_TIMEOUT + (buflen * bitclk) / 1000;

  nbytes = usb_control_msg( usb_handle,
			    USB_ENDPOINT_OUT | USB_TYPE_VENDOR | USB_RECIP_DEVICE,
			    requestid,
			    val, index,
			    (char *)buffer, buflen,
			    timeout);
  if (nbytes != buflen) {
    fprintf(stderr, "%s: error: usbtiny_send: (expected %d, got %d)\n",
	    usb_strerror(), buflen, nbytes);
    exit(1);
  }

  return nbytes;
}

// ----------------------------------------------------------------------

/* Find a device with the correct VID/PID match for USBtiny */
int	usbtiny_open()
{
  struct usb_bus      *bus;
  struct usb_device   *dev = 0;

  usb_init();                    // initialize the libusb system
  usb_find_busses();             // have libusb scan all the usb busses available
  usb_find_devices();            // have libusb scan all the usb devices available

  usb_handle = NULL;

  // now we iterate through all the busses and devices
	for ( bus = usb_busses; bus; bus = bus->next ) {
		for	( dev = bus->devices; dev; dev = dev->next ) {
			if (dev->descriptor.idVendor == USBTINY_VENDOR		// found match?
				&& dev->descriptor.idProduct == USBTINY_PRODUCT ) {
				usb_handle = usb_open(dev);           // attempt to connect to device
				// wrong permissions or something?
				if (!usb_handle) {
					print_error(_("Warning: cannot open USB device."));
					continue;
				}
			}
		}
	}

	if (usb_handle) return 1;

	// If we got here, we must have found a good USB device
	return 0;
}

/* Clean up the handle for the usbtiny */
void usbtiny_close ( void )
{
	if (! usb_handle) {
		return;                // not a valid handle, bail!
	}
	usb_close(usb_handle);   // ask libusb to clean up
	usb_handle = NULL;
}

void usbtiny_powerup()
{
	usb_control( USBTINY_POWERUP, sck_period, RESET_HIGH );
	sleep(1);
}

void usbtiny_send_protocol()
{
    usb_control(PROTOCOL_MSG, options.target->protocol, 0);
    sleep(1);
}

/* Tell the USBtiny to release the output pins, etc */
void usbtiny_powerdown()
{
	if (!usb_handle) {
		return;                 // wasn't connected in the first place
	}
	usb_control(USBTINY_POWERDOWN, 0, 0);      // Send USB control command to device
}

/* Send a 4-byte SPI command to the USBtinyISP for execution
   This procedure is used by higher-level Avrdude procedures */
int usbtiny_trasnparent_spi(unsigned char cmd[4], unsigned char res[SPI_CMD_LEN])
{
  int nbytes;

  // Make sure its empty so we don't read previous calls if it fails
  // 4 is the fixed size of the SPI command.
  memset(res, '\0', SPI_CMD_LEN );

  nbytes = usb_in(USBTINY_SPI,
		   (cmd[1] << 8) | cmd[0],  // convert to 16-bit words
		   (cmd[3] << 8) | cmd[2],  //  "
			res, SPI_CMD_LEN, 8 * sck_period );
  if (debug_flag) {
    // print out the data we sent and received
    printf( "CMD: [%02x %02x %02x %02x] [%02x %02x %02x %02x]\n",
	    cmd[0], cmd[1], cmd[2], cmd[3],
	    res[0], res[1], res[2], res[3] );
  }
  return ((nbytes == SPI_CMD_LEN) &&      // should have read 4 bytes
	  res[2] == cmd[1]);                   // AVR's do a delayed-echo thing
}
