#
# Tcl Library for TkRev
#

#
# subimport.tcl is similar to import.tcl except that it is used for
# importing to an existing module.
# By: Eugene Lee, Aerospace Corporation, 10/16/03
#

# Called from "Import To An Existing Module"
proc cvs_subimport_setup {} {
  global cwd
  global incvs
  global insvn
  global ingit
  global inrcs
  global cvsglb
  
  gen_log:log T "ENTER"
  
  # Make sure we're not in a directory that's already under revision control
  lassign [vcs_detect [pwd]] incvs insvn inrcs ingit
  if {$incvs} {
    cvsok "This directory is already in CVS.\nCan\'t import here!" .import
    gen_log:log T "LEAVE"
    return
  }
  if {$insvn} {
    cvsok "There are Subversion directories here.\nPlease remove them first." .import
    gen_log:log T "LEAVE"
    return
  }
  if {$ingit} {
    cvsok "There are Git directories here.\nPlease remove them first." .import
    gen_log:log T "LEAVE"
    return
  }
  if {$inrcs} {
    cvsok "There are RCS directories here.\nPlease remove them first." .import
    gen_log:log T "LEAVE"
    return
  }
  
  if {[winfo exists .subimport]} {
    wm deiconify .subimport
    raise .subimport
    grab set .subimport
    gen_log:log T "LEAVE"
    return
  }
  
  # Set some defaults
  set cvsglb(existmodule) ""
  set cvsglb(newcode) [file tail $cwd]
  set cvsglb(newdir) $cvsglb(newcode)
  set cvsglb(newdesc) ""
  set cvsglb(newvers) "1.1.1"

  toplevel .subimport
  grab set .subimport
  
  frame .subimport.top
  
  message .subimport.top.explain -justify left -width 500 -relief groove \
    -text "This will import the current directory and its sub-directories\
          into an Existing CVS module."
  label .subimport.top.lnewcode -text "Module Name" -anchor w
  label .subimport.top.lnewdir -text "Path relative to \$CVSROOT" -anchor w
  label .subimport.top.lnewdesc -text "Descriptive Title" -anchor w
  label .subimport.top.lnewvers -text "Version Number" -anchor w
  label .subimport.top.tnewcode -textvariable cvsglb(existmodule) \
    -relief sunken -width 40 -anchor w
  label .subimport.top.tnewdir -textvariable cvsglb(newdir) \
    -relief sunken -width 40 -anchor w
  entry .subimport.top.tnewvers -textvariable cvsglb(newvers) -width 40
  button .subimport.top.bnewcode -text "Browse ..." -command "moduleDialog"
  
  grid .subimport.top.explain -column 0 -row 0 -columnspan 3 -sticky ew
  grid .subimport.top.lnewcode -column 0 -row 1 -sticky w
  grid .subimport.top.tnewcode -column 1 -row 1 -sticky w
  grid .subimport.top.bnewcode -column 2 -row 1 -sticky e
  grid .subimport.top.lnewdir -column 0 -row 2 -sticky w
  grid .subimport.top.tnewdir -column 1 -row 2 -sticky w
  grid .subimport.top.lnewvers -column 0 -row 3 -sticky w
  grid .subimport.top.tnewvers -column 1 -row 3 -sticky ew
  
  frame .subimport.down -relief groove -borderwidth 2
  button .subimport.down.ok -text "OK" \
    -command {
      if {! [cvs_import_errorcheck]} {
        grab release .subimport
        wm withdraw .subimport
        cvs_subimport_do
      }
  }
  button .subimport.down.quit -text "Cancel" -command {
    grab release .subimport
    wm withdraw .subimport
  }
  
  pack .subimport.down -side bottom -expand yes -fill x
  pack .subimport.top -side top -expand yes -fill x
  pack .subimport.down.ok -side left -expand yes
  pack .subimport.down.quit -side left -expand yes
  
  
  wm title .subimport "Import To An Existing Module"
  wm minsize .subimport 1 1
  
  # The module browser has to be running for the module selection dialog to work
  if {! [info exists modlist_sorted]} {
    modbrowse_run
  }
  
  gen_log:log T "LEAVE"
}

proc cvs_subimport_do {} {
  global cvs
  global cvsglb
  global cvscfg
  global cwd
  global modlist_sorted
  global modval
  
  gen_log:log T "ENTER"
  set imdir [pwd]
  
  # Error checks
  if {$cvsglb(existmodule) == ""} {
    cvsok "You must select an existing module from the repository." .subimport
    raise .subimport
    grab set .subimport
    return 1
  }
  if {$cvsglb(newdir) == ""} {
    cvsok "You must select an existing module from the repository." .subimport
    raise .subimport
    grab set .subimport
    return 1
  }
  if {$cvsglb(newvers) == ""} {
    cvsok "You must type in a version number." .subimport
    raise .subimport
    grab set .subimport
    return 1
  }
  
  wm withdraw .subimport
  # After no more errors
  
  # See if all apropriate Directories in newdirname exist.  CVS import will
  # create them, but we'll want to make a #D entry.
  set cvsglb(newdir) [string trimleft $cvsglb(newdir) "/"]
  set pathname [file dirname $cvsglb(newdir)]
  set need_Dir 0
  if {$pathname != "."} {
    foreach idx $modlist_sorted {
      lappend knowndirs [lindex $idx 0]
    }
    gen_log:log D "looking for $pathname in known directories ($knowndirs)"
    if {$pathname ni $knowndirs} {
      set need_Dir 1
    }
  }
  
  # Make a baseline tag
  set versions [split $cvsglb(newvers) ".,/ -"]
  set baseline "baseline-[join $versions {_}]"
  
  set v [viewer::new "Import Module"]
  
  set commandline "$cvs -d $cvscfg(cvsroot) import -m \"Imported using TkRev\" $cvsglb(newdir) VENDOR $baseline"
  $v\::log "\nCVS Import\n"
  $v\::do "$commandline"
  $v\::wait
  update
  
  # No need to update the modules file.
  
  cd ../
  gen_log:log F "CD [pwd]"
  set commandline "$cvs -d $cvscfg(cvsroot) -Q release -d CVSROOT"
  $v\::do "$commandline"
  $v\::wait
  cd $cwd
  gen_log:log F "CD [pwd]"
  
  # Now check out the new module
  cd ..
  gen_log:log F "CD [pwd]"
  set ckmoddir $cwd
  # save later for use in checking out
  # We have to move the original stuff entirely out of the way.
  # Otherwise checkout won't do the whole tree.
  gen_log:log F "MOVE $imdir $imdir.orig"
  file rename $imdir $imdir.orig
  set $cwd $cwd.orig
  set commandline "$cvs -d $cvscfg(cvsroot) checkout -r$baseline \"$cvsglb(existmodule)\""
  $v\::log "\nCVS Checkout\n"
  $v\::do "$commandline"
  $v\::wait
  
  set cwd $imdir.orig
  
  # cd to the checked out module. $cwd is the correct directory to cd to
  # only if the name of the existing module is the same as the directory name
  # where the source code is in. If the existing module name is different modify
  # ckmoddir
  if {$cvsglb(existmodule) != [file tail $ckmoddir]} {
    set ckmoddir [file join [file dirname $ckmoddir] $cvsglb(existmodule)]
  }
  change_dir $ckmoddir
  gen_log:log F "CD [pwd]"
  
  if {$cvscfg(auto_status)} {
    setup_dir
  }
  gen_log:log T "LEAVE"
}

proc getExistModDialog {} {
  global modval
  global ExModList ExModDirList
  
  set ExModList ""
  set ExModDirList ""
  foreach {key value} [array get modval] {
    if {$key != ""} {
      lappend ExModList $key
      lappend ExModDirList $value
    }
  }
  set ExModList [lsort $ExModList]
}

proc moduleDialog {} {
  global ExModList ExModDirList
  set w .modDialog
  grab release .subimport
  catch {destroy $w}
  toplevel $w
  wm title $w "Select An Existing Module"
  wm minsize $w 28 3
  grab set $w
  
  frame $w.buttons
  pack $w.buttons -side bottom -fill x -pady 2m
  button $w.buttons.ok -text Ok -command {
    destroy .modDialog
    raise .subimport
    grab set .subimport
  }
  button $w.buttons.cancel -text Cancel -command {
    grab release .modDialog
    wm withdraw .modDialog
  }
  pack $w.buttons.ok -side left -expand 1
  pack $w.buttons.cancel -side left -expand 1
  
  frame $w.frame -borderwidth .5c
  pack $w.frame -side top -expand yes -fill y
  
  ttk::scrollbar $w.frame.scroll -command "$w.frame.list yview"
  listbox $w.frame.list -yscrollcommand "$w.frame.scroll set" -setgrid 1 -height 5
  pack $w.frame.scroll -side right -fill y
  pack $w.frame.list -side left -expand 1 -fill both
  
  getExistModDialog
  set nModule [llength $ExModList]
  for {set i 0} {$i < $nModule} {incr i} {
    $w.frame.list insert end [lindex $ExModList $i]
  }
  
  bind $w.frame.list <Button-1> {
    set cvsglb(existmodule) [%W get [%W nearest %y]]
    set tmp [%W get [%W nearest %y]]
    set cvsglb(newdir) $tmp
    set index [lsearch -exact $ExModList $tmp]
    set cvsglb(newdir) [lindex $ExModDirList $index]
  }
}

