#pragma once

#include "frontend_fwd.hpp"

#include <string_view>

#include "h-basic.hpp"

/**
 * User interface operations supported by the front end.
 */
class Frontend {

public:
	/**
	 * Result polling/waiting for an event.
	 */
	enum class event_result_t {
		SUCCESS,
		FAILURE,
	};

	/**
	 * Initialize the underlying user interface element.
	 */
	virtual void init() = 0;

	/**
	 * Destroy the underlying user interface element.
	 */
	virtual void nuke() = 0;

	/**
	 * Poll or wait for event and process it.
	 */
	virtual void process_event(bool wait) = 0;

	/**
	 * Does this user interface draw its own cursor?
	 *
	 * The return value is assumed to be constant over the
	 * lifetime of the user interface.
	 */
	virtual bool soft_cursor() const = 0;

	/**
	 * Does this user interface have an "icky" lower right corner,
	 * i.e. does it _not_ support putting the cursor in that corner?
	 *
	 * The return value is assumed to be constant over the
	 * lifetime of the user interface.
	 */
	virtual bool icky_corner() const = 0;

	/**
	 * Flush any pending events.
	 */
	virtual void flush_events() = 0;

	/**
	 * Process any pending events.
	 */
	virtual void process_queued_events() = 0;

	/**
	 * Clear contents of the user interface.
	 */
	virtual void clear() = 0;

	/**
	 * Flush pending output.
	 */
	virtual void flush_output() = 0;

	/**
	 * Make a noise, e.g. a system beep or similar.
	 */
	virtual void noise() = 0;

	/**
	 * React to changes generated by the game, e.g. window
	 * size, or colors.
	 */
	virtual void react() = 0;

	/**
	 * Rename the main window.
	 */
	virtual void rename_main_window(std::string_view) = 0;

	/**
	 * Draw the cursor at position x, y.
	 */
	virtual void draw_cursor(int x, int y) = 0;

	/**
	 * Draw the given text string s at position x, y
	 * using attributes a.
	 */
	virtual void draw_text(int x, int y, int n, byte a, const char *s) = 0;

	/**
	 * Destroy.
	 */
	virtual ~Frontend() = default;

};
