##########################################################################
# This file is part of Vacuum Magic
# Copyright (C) 2008-2009 by UPi <upi at sourceforge.net>
##########################################################################

use strict;

our ($ScreenWidth, $ScreenHeight);

##########################################################################
package Joystick;
##########################################################################

use vars qw(@Joysticks @JoystickButtons);

sub InitJoystick {
  my ($numJoysticks, $joystick, $numButtons, $i);

  $numJoysticks = &SDL::NumJoysticks();
  for ($i = 0; $i < $numJoysticks; ++$i) {
    print STDERR "Found joystick " , $i+1 , ": " , &SDL::JoystickName($i), "\n";
    $joystick = &SDL::JoystickOpen($i);
    next unless $joystick;
    $numButtons = &SDL::JoystickNumButtons($joystick);
    next unless $numButtons;
    push @Joysticks, $joystick;
    push @JoystickButtons, $numButtons;
    print STDERR "Joystick opened, $numButtons buttons.\n";
  }
}

sub ReadJoystick {
  my ($i, $button, $buttonPressed);

  $i = 0;
  foreach my $joystick (@Joysticks) {
    ++$i;
    my $axis = &SDL::JoystickGetAxis($joystick, 0);
    if ($axis <= -10000) {
      $::Events{"L$i"} = $::MenuEvents{LEFT} = 1 unless $::Keys{"L$i"};
      $::Keys{"L$i"} = 1;
      $::Keys{"R$i"} = 0;
    } elsif ($axis >= 10000) {
      $::Events{"R$i"} = $::MenuEvents{RIGHT} = 1 unless $::Keys{"R$i"};
      $::Keys{"R$i"} = 1;
      $::Keys{"L$i"} = 0;
    } else {
      $::Keys{"L$i"} = 0;
      $::Keys{"R$i"} = 0;
    }
    $axis = &SDL::JoystickGetAxis($joystick, 1);
    if ($axis <= -10000) {
      $::Events{"U$i"} = $::MenuEvents{UP} = 1 unless $::Keys{"U$i"};
      $::Keys{"U$i"} = 1;
      $::Keys{"D$i"} = 0;
    } elsif ($axis >= 10000) {
      $::Events{"D$i"} = $::MenuEvents{DOWN} = 1 unless $::Keys{"D$i"};
      $::Keys{"D$i"} = 1;
      $::Keys{"U$i"} = 0;
    } else {
      $::Keys{"D$i"} = 0;
      $::Keys{"U$i"} = 0;
    }
    $buttonPressed = 0;
    for ($button = 0; $button < $JoystickButtons[$i-1]; ++$button) {
      if (&SDL::JoystickGetButton($joystick, $button)) {
        $buttonPressed = 1; last;
      }
    }
    if ($buttonPressed and not $::Keys{"B$i"}) {
      $::Events{"B$i"} = $::MenuEvents{BUTTON} = 1;
    }
    $::Keys{"B$i"} = $buttonPressed;
    delete $::Events{"B$i"}  unless $buttonPressed;     # This is a workaround for very short-lived button presses
  }
}



##########################################################################
package main;
##########################################################################

sub Pause {
  my $event = new SDL::Event;

  my ($paused, $pausedWidth, $continue, $continueWidth);
  $paused = T("GAME PAUSED");
  $continue = T("Press P to continue");
  $pausedWidth = $GlossyFont->TextWidth($paused);
  $continueWidth = $MenuFont->TextWidth($continue);

  while (1) { # Paused, wait for keypress

    $Game->DrawGame();
    glColor(0,0,0,0.5);
    glDisable(GL_TEXTURE_2D);
    glBegin(GL_QUADS);
      glVertex(0,0); glVertex($ScreenWidth, 0);
      glVertex($ScreenWidth, $ScreenHeight); glVertex(0, $ScreenHeight);
    glEnd();
    glEnable(GL_TEXTURE_2D);
    $GlossyFont->pre_output();
    $GlossyFont->output(400 - int($pausedWidth/2), 300, $paused);
    $GlossyFont->post_output();
    $MenuFont->pre_output();
    $MenuFont->output(400 - int($continueWidth/2), 250, $continue);
    $MenuFont->post_output();
    $App->sync();

    $event->wait();
    do {
      last if $event->type() == SDL_KEYDOWN and $event->key_sym == SDLK_p;
      if ($event->type() == SDL_KEYDOWN and $event->key_sym == SDLK_ESCAPE) { $Game->{abortgame} = 1; last; }
      $Game->Exit() if $event->type() == SDL_QUIT;
    }  while ($event->poll());
  }
  %Keys = (); %Events = ();
  &GameTimer::ResetTimer();
}

sub HandleEvents {
  my ($event, $type);

  $event = new SDL::Event;
  while (1) {
    last unless $event->poll();
    $type = $event->type();

    if ($type == SDL_QUIT) {
      $Game->Exit();
    } elsif ($type == SDL_KEYDOWN) {
      my $keypressed = $event->key_sym;
      if ($keypressed == SDLK_ESCAPE) {
        $Game->{abortgame} = 1;
      } elsif ($keypressed == SDLK_F1) {
        &SaveScreenshot();
      } elsif ($keypressed == SDLK_F2) {
        &DumpObjects();
      } elsif ($keypressed == SDLK_F3) {
        $Game->OnPlayerRespawn();
      } elsif ($keypressed == SDLK_p and not $UnicodeMode) {
        &Pause();
      } else {
        die  if $Keys{$keypressed};
        $Keys{$keypressed} = 1;
        $Events{$keypressed} = 1;
        $MenuEvents{UP} = 1 if $keypressed == SDLK_UP();
        $MenuEvents{DOWN} = 1 if $keypressed == SDLK_DOWN();
        $MenuEvents{LEFT} = 1 if $keypressed == SDLK_LEFT();
        $MenuEvents{RIGHT} = 1 if $keypressed == SDLK_RIGHT();
        $MenuEvents{BUTTON} = 1 if $keypressed == SDLK_RETURN();
        $LastUnicodeKey = $event->key_unicode() if $UnicodeMode;
      }
    } elsif ($type == SDL_KEYUP) {
      my $keypressed = $event->key_sym;
      $Keys{$keypressed} = 0;
      delete $Events{$keypressed};    # This is a workaround for very short-lived key presses
    } elsif ($type == SDL_VIDEORESIZE) {
      # This apparently doesn't work so well..
      glViewport(0, 0, $event->resize_w, $event->resize_h);
    }
  }
  &Joystick::ReadJoystick();
}

1;
