/* ###
 * IP: GHIDRA
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/* Generated by Together */

package ghidra.program.model.util;

import java.io.*;
import java.util.Iterator;
import java.util.NoSuchElementException;

import ghidra.program.model.address.*;
import ghidra.util.LongIterator;
import ghidra.util.datastruct.NoSuchIndexException;
import ghidra.util.prop.PropertySet;
import ghidra.util.prop.PropertyVisitor;

/**
 * PropertyMap is used to store values for a fixed property at
 *   address locations given as longs. The values for the property
 *   must be homogeneous, i.e. all have the same type, and are
 *   determined by which subclass of PropertyMap is instantiated.
 *   For any long the property
 *   manager can be used to tell if the property exists there and
 *   what its value is. It also maintains information that allows it
 *   to efficiently search for the next and previous occurence of the
 *   property relative to a given address.
 *   The subclass provides the createPage() method that dictates
  *  the type of PropertyPage that will be managed.
 */
public abstract class DefaultPropertyMap implements PropertyMap {

	protected PropertySet propertyMgr;
	protected AddressMapImpl addrMap;
	protected String description;

	/**
	 * Construct a PropertyMap
	 * @param propertyMgr property manager that manages storage of
	 * properties
	 */
	public DefaultPropertyMap(PropertySet propertyMgr) {
		this.propertyMgr = propertyMgr;
		this.addrMap = new AddressMapImpl();
	}

	/**
	 * Get the name for this property manager.
	 */
	@Override
	public String getName() {
		return propertyMgr.getName();
	}

	/**
	 * Set the description for this property.
	 *
	 * @param description property description
	 */
	public void setDescription(String description) {
		this.description = description;
	}

	/**
	 * Return the property description.
	 * 
	 * @return the property description
	 */
	public String getDescription() {
		return description;
	}

	/**
	 * Given two addresses, indicate whether there is an address in
	 * that range (inclusive) having the property.<p>
	 * @param start the start of the range.
	 * @param end the end of the range.
	 *
	 * @return boolean true if at least one address in the range
	 * has the property, false otherwise.
	 */
	@Override
	public boolean intersects(Address start, Address end) {
		return propertyMgr.intersects(addrMap.getKey(start), addrMap.getKey(end));
	}

	/*
	 * @see ghidra.program.model.util.PropertyMap#intersects(ghidra.program.model.address.AddressSetView)
	 */
	@Override
	public boolean intersects(AddressSetView set) {
		AddressRangeIterator ranges = set.getAddressRanges();
		while (ranges.hasNext()) {
			AddressRange range = ranges.next();
			if (intersects(range.getMinAddress(), range.getMaxAddress())) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Removes all property values within a given range.
	 * @param start begin range
	 * @param end end range, inclusive
	 * @return true if any property value was removed; return
	 * 		false otherwise.
	 */
	@Override
	public boolean removeRange(Address start, Address end) {
		return propertyMgr.removeRange(addrMap.getKey(start), addrMap.getKey(end));
	}

	/**
	 * Remove the property value at the given address.
	 * @return true if the property value was removed, false
	 *   otherwise.
	 * @param addr the address where the property should be removed
	 */
	@Override
	public boolean remove(Address addr) {
		return propertyMgr.remove(addrMap.getKey(addr));
	}

	/**
	 * returns whether there is a property value at addr.
	 * @param addr the address in question
	 */
	@Override
	public boolean hasProperty(Address addr) {
		return propertyMgr.hasProperty(addrMap.getKey(addr));

	}

	/**
	 * Get the next address where the property value exists.
	 * @param addr the address from which to begin the search (exclusive).
	 * @throws NoSuchIndexException thrown if there is no address with
	 *   a property value after the given address.
	 */
	@Override
	public Address getNextPropertyAddress(Address addr) {

		try {
			long index = propertyMgr.getNextPropertyIndex(addrMap.getKey(addr));
			return addrMap.decodeAddress(index);
		}
		catch (NoSuchIndexException e) {
		}
		return null;

	}

	/**
	 * Get the previous Address where a property value exists.
	 * @param addr the address from which
	 * 		to begin the search (exclusive).
	 * @throws NoSuchIndexException when there is no address
	 * 		with a property value before the given address.
	 */
	@Override
	public Address getPreviousPropertyAddress(Address addr) {
		try {
			long index = propertyMgr.getPreviousPropertyIndex(addrMap.getKey(addr));
			return addrMap.decodeAddress(index);
		}
		catch (NoSuchIndexException e) {
		}
		return null;
	}

	/**
	 * Get the first Address where a property value exists.
	 */
	@Override
	public Address getFirstPropertyAddress() {
		try {
			long index = propertyMgr.getFirstPropertyIndex();
			return addrMap.decodeAddress(index);
		}
		catch (NoSuchIndexException e) {
		}
		return null;

	}

	/**
	 * Get the last Address where a property value exists.
	 * @exception NoSuchIndexException
	 *                   thrown if there is no address having the property value.
	 */
	@Override
	public Address getLastPropertyAddress() {
		try {
			long index = propertyMgr.getLastPropertyIndex();
			return addrMap.decodeAddress(index);
		}
		catch (NoSuchIndexException e) {
		}
		return null;
	}

	/**
	 * Get the number of properties in the map.
	 */
	@Override
	public int getSize() {
		return propertyMgr.getSize();
	}

	/** Returns an iterator over addresses that have a property value within the
	 * given address range.
	 * @param start the first address in the range.
	 * @param end the last address in the range.
	 * @exception TypeMismatchException thrown if the property does not
	 * have values of type <CODE>Object</CODE>.
	 */
	@Override
	public AddressIterator getPropertyIterator(Address start, Address end) {
		return new AddressPropertyIterator(start, end);
	}

	/**
	 * @see ghidra.program.model.util.PropertyMap#getPropertyIterator(ghidra.program.model.address.Address, ghidra.program.model.address.Address, boolean)
	 */
	@Override
	public AddressIterator getPropertyIterator(Address start, Address end, boolean forward) {
		return new AddressPropertyIterator(start, end, forward);
	}

	/** Returns an iterator over addresses that have a property value within the
	 * property map.
	 * @exception TypeMismatchException thrown if the property does not
	 * have values of type <CODE>Object</CODE>.
	 */
	@Override
	public AddressIterator getPropertyIterator() {
		return new AddressPropertyIterator();
	}

	/**
	 * Returns an iterator over the addresses that have a property value and
	 * are in the given address set.
	 */
	@Override
	public AddressIterator getPropertyIterator(AddressSetView asv) {
		return new AddressSetPropertyIterator(asv, true);
	}

	/**
	 * @see ghidra.program.model.util.PropertyMap#getPropertyIterator(ghidra.program.model.address.AddressSetView, boolean)
	 */
	@Override
	public AddressIterator getPropertyIterator(AddressSetView asv, boolean forward) {
		return new AddressSetPropertyIterator(asv, forward);
	}

	/**
	 * @see ghidra.program.model.util.PropertyMap#getPropertyIterator(ghidra.program.model.address.Address, boolean)
	 */
	@Override
	public AddressIterator getPropertyIterator(Address start, boolean forward) {
		return new AddressPropertyIterator(start, forward);
	}

	/**
	 * 
	 * @see ghidra.program.model.util.PropertyMap#applyValue(ghidra.util.prop.PropertyVisitor, ghidra.program.model.address.Address)
	 */
	@Override
	public void applyValue(PropertyVisitor visitor, Address addr) {
		propertyMgr.applyValue(visitor, addrMap.getKey(addr));
	}

	/**
	 * 
	 * @see ghidra.program.model.util.PropertyMap#moveRange(ghidra.program.model.address.Address, ghidra.program.model.address.Address, ghidra.program.model.address.Address)
	 */
	@Override
	public void moveRange(Address start, Address end, Address newStart) {
		propertyMgr.moveRange(addrMap.getKey(start), addrMap.getKey(end), addrMap.getKey(newStart));
	}

	/**
	 * Save the properties in the given range to output stream.
	 * @param oos output stream to write to
	 * @param start start address in the range
	 * @param end end address in the range 
	 * @throws IOException if there a problem doing the write
	 */
	public void saveProperties(ObjectOutputStream oos, Address start, Address end)
			throws IOException {
		propertyMgr.saveProperties(oos, addrMap.getKey(start), addrMap.getKey(end));
	}

	/**
	 * Restore properties from the given input stream.
	 * @param ois input stream
	 * @throws IOException if there is a problem reading from the stream
	 * @throws ClassNotFoundException if the class for the object being
	 * read is not in the class path
	 */
	public void restoreProperties(ObjectInputStream ois) throws IOException, ClassNotFoundException {
		propertyMgr.restoreProperties(ois);
	}

	/**
	 * Write all properties in the map to the given output stream.
	 * @throws IOException if there is a problem writing to the stream
	 */
	public void saveAll(ObjectOutputStream out) throws IOException {
		propertyMgr.saveAll(out);
	}

	/**
	 * Restore properties read from the given input stream.
	 * @param in input stream 
	 * @throws IOException if there is a problem reading from the stream
	 * @throws ClassNotFoundException if the class for the object being
	 * read is not in the class path
	 */
	public void restoreAll(ObjectInputStream in) throws IOException, ClassNotFoundException {
		propertyMgr.restoreAll(in);
	}

	private class AddressPropertyIterator implements AddressIterator {

		private LongIterator iter;
		private boolean forward = true;

		AddressPropertyIterator() {
			iter = propertyMgr.getPropertyIterator();
		}

		AddressPropertyIterator(Address start, boolean forward) {
			iter = propertyMgr.getPropertyIterator(addrMap.getKey(start), forward);
			this.forward = forward;
		}

		AddressPropertyIterator(Address start, Address end) {
			iter = propertyMgr.getPropertyIterator(addrMap.getKey(start), addrMap.getKey(end));

		}

		AddressPropertyIterator(Address start, Address end, boolean forward) {
			iter =
				propertyMgr.getPropertyIterator(addrMap.getKey(start), addrMap.getKey(end), forward);
			this.forward = forward;

		}

		/**
		 * @see java.util.Iterator#remove()
		 */
		@Override
		public void remove() {
			throw new UnsupportedOperationException();
		}

		/**
		 * @see AddressIterator#hasNext()
		 */
		@Override
		public boolean hasNext() {
			if (forward) {
				return iter.hasNext();
			}
			return iter.hasPrevious();
		}

		/**
		* @see AddressIterator#next()
		*/
		@Override
		public Address next() {
			try {
				if (forward) {
					return addrMap.decodeAddress(iter.next());
				}
				return addrMap.decodeAddress(iter.previous());
			}
			catch (NoSuchElementException e) {
			}
			return null;
		}

		@Override
		public Iterator<Address> iterator() {
			return this;
		}
	}

	private class AddressSetPropertyIterator implements AddressIterator {

		private AddressPropertyIterator iter;
		private AddressRangeIterator ranges;
		private Address nextAddr;
		private AddressRange curRange;
		private boolean atStart;

		AddressSetPropertyIterator(AddressSetView addrSet, boolean atStart) {
			ranges = addrSet.getAddressRanges(atStart);
			this.atStart = atStart;
		}

		/**
		 * @see java.util.Iterator#remove()
		 */
		@Override
		public void remove() {
			throw new UnsupportedOperationException();
		}

		/**
		 * @see AddressIterator#hasNext()
		 */
		@Override
		public boolean hasNext() {
			if (nextAddr == null) {
				nextAddr = findNext();
			}
			return nextAddr != null;
		}

		/**
		 * @see AddressIterator#hasPrevious()
		 */
		public boolean hasPrevious() {
			throw new UnsupportedOperationException();
//			nextAddr = null;
//			if (prevAddr == null) {
//				prevAddr = findPrevious();
//			}
//			return prevAddr != null;
		}

		/**
		 * @see AddressIterator#next()
		 */
		@Override
		public Address next() {
			if (hasNext()) {
				Address addr = nextAddr;
				nextAddr = null;
				return addr;
			}
			return null;
		}

		/**
		 * @see AddressIterator#previous()
		 */
		public Address previous() {
			throw new UnsupportedOperationException();
//			if (hasPrevious()) {
//				Address addr = prevAddr;
//				prevAddr = null;
//				return addr;
//			}
//			return null;
		}

		private Address findNext() {
			if ((iter != null) && iter.hasNext()) {
				return iter.next();
			}
			while (ranges.hasNext()) {
				AddressRange range = ranges.next();
				if (range == curRange) {
					continue;
				}
				curRange = range;
				iter =
					new AddressPropertyIterator(range.getMinAddress(), range.getMaxAddress(),
						atStart);
				if (iter.hasNext()) {
					return iter.next();
				}
			}
			return null;
		}

//		private Address findPrevious() {
//			if ((iter != null) && iter.hasPrevious()) {
//				return iter.previous();
//			}
//			while(ranges.hasPrevious()) {
//				AddressRange range = ranges.previous();
//				if (range == curRange) {
//					continue;
//				}
//				curRange = range;
//				iter = new AddressPropertyIterator(range.getMinAddress(),
//												   range.getMaxAddress(), 
//												   atStart);
//				if (iter.hasPrevious()) {
//					return iter.previous();
//				}
//			}
//			return null;
//		}

		@Override
		public Iterator<Address> iterator() {
			return this;
		}
	}

}
