/*
 * Decompiled with CFR 0.152.
 */
package com.epam.parso.impl;

import com.epam.parso.Column;
import com.epam.parso.ColumnFormat;
import com.epam.parso.ColumnMissingInfo;
import com.epam.parso.SasFileProperties;
import com.epam.parso.date.OutputDateType;
import com.epam.parso.date.SasTemporalFormatter;
import com.epam.parso.impl.BinDecompressor;
import com.epam.parso.impl.CharDecompressor;
import com.epam.parso.impl.Decompressor;
import com.epam.parso.impl.PageType;
import com.epam.parso.impl.SasFileConstants;
import java.io.DataInputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class SasFileParser {
    private static final Logger LOGGER = LoggerFactory.getLogger(SasFileParser.class);
    private static final Map<Long, SubheaderIndexes> SUBHEADER_SIGNATURE_TO_INDEX;
    private static final Map<String, Decompressor> LITERALS_TO_DECOMPRESSOR;
    private static final int MAX_PAGE_LENGTH = 10000000;
    private static final byte[] SKIP_BYTE_BUFFER;
    private final DataInputStream sasFileStream;
    private final Boolean byteOutput;
    private final OutputDateType outputDateType;
    private final List<SubheaderPointer> currentPageDataSubheaderPointers = new ArrayList<SubheaderPointer>();
    private final SasFileProperties sasFileProperties = new SasFileProperties();
    private final List<byte[]> columnsNamesBytes = new ArrayList<byte[]>();
    private final List<String> columnsNamesList = new ArrayList<String>();
    private final List<Class<?>> columnsTypesList = new ArrayList();
    private final List<Long> columnsDataOffset = new ArrayList<Long>();
    private final List<Integer> columnsDataLength = new ArrayList<Integer>();
    private final List<Column> columns = new ArrayList<Column>();
    private final Map<SubheaderIndexes, ProcessingSubheader> subheaderIndexToClass;
    private String encoding = "US-ASCII";
    private byte[] cachedPage;
    private int currentPageType;
    private int currentPageBlockCount;
    private int currentPageSubheadersCount;
    private int currentFilePosition;
    private int currentColumnNumber;
    private int currentRowInFileIndex;
    private int currentRowOnPageIndex;
    private Object[] currentRow;
    private boolean eof;
    private int fileLabelOffset;
    private int compressionMethodOffset;
    private int compressionMethodLength;
    private int fileLabelLength;
    private final List<ColumnMissingInfo> columnMissingInfoList = new ArrayList<ColumnMissingInfo>();
    private String deletedMarkers = "";
    private final SasTemporalFormatter sasTemporalFormatter = new SasTemporalFormatter();

    private SasFileParser(Builder builder) {
        this.sasFileStream = new DataInputStream(builder.sasFileStream);
        this.byteOutput = builder.byteOutput;
        this.outputDateType = builder.outputDateType;
        HashMap<SubheaderIndexes, ProcessingSubheader> tmpMap = new HashMap<SubheaderIndexes, ProcessingSubheader>();
        tmpMap.put(SubheaderIndexes.ROW_SIZE_SUBHEADER_INDEX, new RowSizeSubheader());
        tmpMap.put(SubheaderIndexes.COLUMN_SIZE_SUBHEADER_INDEX, new ColumnSizeSubheader());
        tmpMap.put(SubheaderIndexes.SUBHEADER_COUNTS_SUBHEADER_INDEX, new SubheaderCountsSubheader());
        tmpMap.put(SubheaderIndexes.COLUMN_TEXT_SUBHEADER_INDEX, new ColumnTextSubheader());
        tmpMap.put(SubheaderIndexes.COLUMN_NAME_SUBHEADER_INDEX, new ColumnNameSubheader());
        tmpMap.put(SubheaderIndexes.COLUMN_ATTRIBUTES_SUBHEADER_INDEX, new ColumnAttributesSubheader());
        tmpMap.put(SubheaderIndexes.FORMAT_AND_LABEL_SUBHEADER_INDEX, new FormatAndLabelSubheader());
        tmpMap.put(SubheaderIndexes.COLUMN_LIST_SUBHEADER_INDEX, new ColumnListSubheader());
        tmpMap.put(SubheaderIndexes.DATA_SUBHEADER_INDEX, new DataSubheader());
        this.subheaderIndexToClass = Collections.unmodifiableMap(tmpMap);
        try {
            this.getMetadataFromSasFile(builder.encoding);
        }
        catch (IOException e) {
            LOGGER.error(e.getMessage(), (Throwable)e);
        }
    }

    private void getMetadataFromSasFile(String encoding) throws IOException {
        boolean endOfMetadata = false;
        this.processSasFileHeader(encoding);
        this.cachedPage = new byte[this.sasFileProperties.getPageLength()];
        while (!endOfMetadata) {
            try {
                this.sasFileStream.readFully(this.cachedPage, 0, this.sasFileProperties.getPageLength());
            }
            catch (EOFException ex) {
                this.eof = true;
                break;
            }
            endOfMetadata = this.processSasFilePageMeta();
        }
    }

    private void processSasFileHeader(String builderEncoding) throws IOException {
        Integer[] lengthForAlign;
        int align1 = 0;
        int align2 = 0;
        Long[] offsetForAlign = new Long[]{32L, 35L};
        List<byte[]> varsForAlign = this.getBytesFromFile(offsetForAlign, lengthForAlign = new Integer[]{1, 1});
        if (varsForAlign.get(0)[0] == 51) {
            align2 = 4;
            this.sasFileProperties.setU64(true);
        }
        if (varsForAlign.get(1)[0] == 51) {
            align1 = 4;
        }
        int totalAlign = align1 + align2;
        Long[] offset = new Long[]{37L, 70L, 92L, 156L, 164L + (long)align1, 172L + (long)align1, 196L + (long)align1, 200L + (long)align1, 204L + (long)align1, 216L + (long)totalAlign, 224L + (long)totalAlign, 240L + (long)totalAlign, 256L + (long)totalAlign, 272L + (long)totalAlign};
        Integer[] length = new Integer[]{1, 1, 64, 8, 8, 8, 4, 4, 4 + align2, 8, 16, 16, 16, 16};
        List<byte[]> vars = this.getBytesFromFile(offset, length);
        this.sasFileProperties.setEndianness(vars.get(0)[0]);
        if (!this.isSasFileValid()) {
            throw new IOException("Can not read metadata from sas7bdat file.");
        }
        String fileEncoding = SasFileConstants.SAS_CHARACTER_ENCODINGS.get(vars.get(1)[0]);
        this.encoding = builderEncoding != null ? builderEncoding : (fileEncoding != null ? fileEncoding : this.encoding);
        this.sasFileProperties.setEncoding(fileEncoding);
        this.sasFileProperties.setName(this.bytesToString(vars.get(2)).trim());
        this.sasFileProperties.setFileType(this.bytesToString(vars.get(3)).trim());
        this.sasFileProperties.setDateCreated(this.bytesToDateTime(vars.get(4)));
        this.sasFileProperties.setDateModified(this.bytesToDateTime(vars.get(5)));
        this.sasFileProperties.setHeaderLength(this.bytesToInt(vars.get(6)));
        int pageLength = this.bytesToInt(vars.get(7));
        if (pageLength > 10000000) {
            throw new IOException("Page limit (" + pageLength + ") exceeds maximum: " + 10000000);
        }
        this.sasFileProperties.setPageLength(pageLength);
        this.sasFileProperties.setPageCount(this.bytesToLong(vars.get(8)));
        this.sasFileProperties.setSasRelease(this.bytesToString(vars.get(9)).trim());
        this.sasFileProperties.setServerType(this.bytesToString(vars.get(10)).trim());
        this.sasFileProperties.setOsType(this.bytesToString(vars.get(11)).trim());
        if (vars.get(13)[0] != 0) {
            this.sasFileProperties.setOsName(this.bytesToString(vars.get(13)).trim());
        } else {
            this.sasFileProperties.setOsName(this.bytesToString(vars.get(12)).trim());
        }
        if (this.sasFileStream != null) {
            this.skipBytes(this.sasFileProperties.getHeaderLength() - this.currentFilePosition);
            this.currentFilePosition = 0;
        }
    }

    private void skipBytes(long numberOfBytesToSkip) throws IOException {
        long actuallySkipped;
        long remainBytes;
        long readBytes;
        for (remainBytes = numberOfBytesToSkip; remainBytes > 0L; remainBytes -= readBytes) {
            try {
                readBytes = this.sasFileStream.read(SKIP_BYTE_BUFFER, 0, (int)Math.min(remainBytes, (long)SKIP_BYTE_BUFFER.length));
                if (readBytes >= 0L) continue;
                break;
            }
            catch (IOException e) {
                throw new IOException("There are no available bytes in the input stream.");
            }
        }
        if ((actuallySkipped = numberOfBytesToSkip - remainBytes) != numberOfBytesToSkip) {
            throw new IOException("Expected to skip " + numberOfBytesToSkip + " to the end of the header, but skipped " + actuallySkipped + " instead.");
        }
    }

    private boolean isSasFileValid() {
        return this.sasFileProperties.getEndianness() == 1 || this.sasFileProperties.getEndianness() == 0;
    }

    private boolean processSasFilePageMeta() throws IOException {
        int bitOffset = this.sasFileProperties.isU64() ? 32 : 16;
        this.readPageHeader();
        ArrayList<SubheaderPointer> subheaderPointers = new ArrayList<SubheaderPointer>();
        if (PageType.PAGE_TYPE_META.contains(this.currentPageType) || PageType.PAGE_TYPE_MIX.contains(this.currentPageType)) {
            this.processPageMetadata(bitOffset, subheaderPointers);
        }
        return PageType.PAGE_TYPE_DATA.contains(this.currentPageType) || PageType.PAGE_TYPE_MIX.contains(this.currentPageType) || this.currentPageDataSubheaderPointers.size() != 0;
    }

    private void processPageMetadata(int bitOffset, List<SubheaderPointer> subheaderPointers) throws IOException {
        subheaderPointers.clear();
        for (int subheaderPointerIndex = 0; subheaderPointerIndex < this.currentPageSubheadersCount; ++subheaderPointerIndex) {
            try {
                SubheaderPointer currentSubheaderPointer = this.processSubheaderPointers((long)bitOffset + 8L, subheaderPointerIndex);
                subheaderPointers.add(currentSubheaderPointer);
                if (currentSubheaderPointer.compression == 1) continue;
                long subheaderSignature = this.readSubheaderSignature(currentSubheaderPointer.offset);
                SubheaderIndexes subheaderIndex = this.chooseSubheaderClass(subheaderSignature, currentSubheaderPointer.compression, currentSubheaderPointer.type);
                if (subheaderIndex != null) {
                    if (subheaderIndex != SubheaderIndexes.DATA_SUBHEADER_INDEX) {
                        LOGGER.debug("Subheader process function name: {}", (Object)subheaderIndex);
                        this.subheaderIndexToClass.get((Object)subheaderIndex).processSubheader(subheaderPointers.get(subheaderPointerIndex).offset, subheaderPointers.get(subheaderPointerIndex).length);
                        continue;
                    }
                    this.currentPageDataSubheaderPointers.add(subheaderPointers.get(subheaderPointerIndex));
                    continue;
                }
                LOGGER.debug("Unknown subheader signature");
                continue;
            }
            catch (Exception e) {
                LOGGER.warn("Encountered broken page metadata. Skipping subheader.");
            }
        }
    }

    private long readSubheaderSignature(Long subheaderPointerOffset) throws IOException {
        int intOrLongLength = this.sasFileProperties.isU64() ? 8 : 4;
        Long[] subheaderOffsetMass = new Long[]{subheaderPointerOffset};
        Integer[] subheaderLengthMass = new Integer[]{intOrLongLength};
        List<byte[]> subheaderSignatureMass = this.getBytesFromFile(subheaderOffsetMass, subheaderLengthMass);
        return this.bytesToLong(subheaderSignatureMass.get(0));
    }

    private SubheaderIndexes chooseSubheaderClass(long subheaderSignature, int compression, int type) {
        SubheaderIndexes subheaderIndex = SUBHEADER_SIGNATURE_TO_INDEX.get(subheaderSignature);
        if (this.sasFileProperties.isCompressed() && subheaderIndex == null && (compression == 4 || compression == 0) && type == 1) {
            subheaderIndex = SubheaderIndexes.DATA_SUBHEADER_INDEX;
        }
        return subheaderIndex;
    }

    private SubheaderPointer processSubheaderPointers(long subheaderPointerOffset, int subheaderPointerIndex) throws IOException {
        int intOrLongLength = this.sasFileProperties.isU64() ? 8 : 4;
        int subheaderPointerLength = this.sasFileProperties.isU64() ? 24 : 12;
        long totalOffset = subheaderPointerOffset + (long)subheaderPointerLength * (long)subheaderPointerIndex;
        Long[] offset = new Long[]{totalOffset, totalOffset + (long)intOrLongLength, totalOffset + 2L * (long)intOrLongLength, totalOffset + 2L * (long)intOrLongLength + 1L};
        Integer[] length = new Integer[]{intOrLongLength, intOrLongLength, 1, 1};
        List<byte[]> vars = this.getBytesFromFile(offset, length);
        long subheaderOffset = this.bytesToLong(vars.get(0));
        long subheaderLength = this.bytesToLong(vars.get(1));
        byte subheaderCompression = vars.get(2)[0];
        byte subheaderType = vars.get(3)[0];
        return new SubheaderPointer(subheaderOffset, subheaderLength, subheaderCompression, subheaderType);
    }

    private boolean matchCompressionMethod(String compressionMethod) {
        if (compressionMethod == null) {
            LOGGER.warn("Null provided as the file compression literal, assuming no compression");
            return false;
        }
        if (LITERALS_TO_DECOMPRESSOR.containsKey(compressionMethod)) {
            return true;
        }
        LOGGER.debug("No supported compression literal found, assuming no compression");
        return false;
    }

    Integer getOffset() {
        return this.currentRowInFileIndex;
    }

    public Object[] readNext() throws IOException {
        return this.readNext(null);
    }

    public Object[] readNext(List<String> columnNames) throws IOException {
        if ((long)this.currentRowInFileIndex++ >= this.sasFileProperties.getRowCount() || this.eof) {
            return null;
        }
        int bitOffset = this.sasFileProperties.isU64() ? 32 : 16;
        this.currentRow = null;
        switch (this.currentPageType) {
            case 0: 
            case 128: 
            case 16384: {
                if (this.currentPageDataSubheaderPointers.size() == 0 && this.currentPageType == 128) {
                    this.readNextPage();
                    this.currentRowOnPageIndex = 0;
                }
                SubheaderPointer currentSubheaderPointer = this.currentPageDataSubheaderPointers.get(this.currentRowOnPageIndex++);
                ((ProcessingDataSubheader)this.subheaderIndexToClass.get((Object)SubheaderIndexes.DATA_SUBHEADER_INDEX)).processSubheader(currentSubheaderPointer.offset, currentSubheaderPointer.length, columnNames);
                if (this.currentRowOnPageIndex != this.currentPageDataSubheaderPointers.size()) break;
                this.readNextPage();
                this.currentRowOnPageIndex = 0;
                break;
            }
            case 512: {
                int subheaderPointerLength = this.sasFileProperties.isU64() ? 24 : 12;
                int alignCorrection = (bitOffset + 8 + this.currentPageSubheadersCount * subheaderPointerLength) % 8;
                this.currentRow = this.processByteArrayWithData((long)(bitOffset + 8 + alignCorrection + this.currentPageSubheadersCount * subheaderPointerLength) + (long)this.currentRowOnPageIndex++ * this.sasFileProperties.getRowLength(), this.sasFileProperties.getRowLength(), columnNames);
                if ((long)this.currentRowOnPageIndex != Math.min(this.sasFileProperties.getRowCount(), this.sasFileProperties.getMixPageRowCount())) break;
                this.readNextPage();
                this.currentRowOnPageIndex = 0;
                break;
            }
            case 640: {
                if (Objects.equals(this.deletedMarkers, "")) {
                    this.readDeletedInfo();
                    LOGGER.info(this.deletedMarkers);
                }
                int subheaderPointerLength = this.sasFileProperties.isU64() ? 24 : 12;
                int alignCorrection = (bitOffset + 8 + this.currentPageSubheadersCount * subheaderPointerLength) % 8;
                if (this.deletedMarkers.charAt(this.currentRowOnPageIndex) == '0') {
                    this.currentRow = this.processByteArrayWithData((long)(bitOffset + 8 + alignCorrection + this.currentPageSubheadersCount * subheaderPointerLength) + (long)this.currentRowOnPageIndex++ * this.sasFileProperties.getRowLength(), this.sasFileProperties.getRowLength(), columnNames);
                } else {
                    ++this.currentRowOnPageIndex;
                }
                if ((long)this.currentRowOnPageIndex != Math.min(this.sasFileProperties.getRowCount(), this.sasFileProperties.getMixPageRowCount())) break;
                this.readNextPage();
                this.currentRowOnPageIndex = 0;
                break;
            }
            case 256: {
                this.currentRow = this.processByteArrayWithData((long)(bitOffset + 8) + (long)this.currentRowOnPageIndex++ * this.sasFileProperties.getRowLength(), this.sasFileProperties.getRowLength(), columnNames);
                if (this.currentRowOnPageIndex != this.currentPageBlockCount) break;
                this.readNextPage();
                this.currentRowOnPageIndex = 0;
                break;
            }
            case 384: {
                if (Objects.equals(this.deletedMarkers, "")) {
                    this.readDeletedInfo();
                    LOGGER.info(this.deletedMarkers);
                    LOGGER.info(Integer.toString(this.deletedMarkers.length()));
                    LOGGER.info(Integer.toString(this.currentPageBlockCount));
                }
                if (this.deletedMarkers.charAt(this.currentRowOnPageIndex) == '0') {
                    this.currentRow = this.processByteArrayWithData((long)(bitOffset + 8) + (long)this.currentRowOnPageIndex++ * this.sasFileProperties.getRowLength(), this.sasFileProperties.getRowLength(), columnNames);
                } else {
                    ++this.currentRowOnPageIndex;
                }
                if (this.currentRowOnPageIndex != this.currentPageBlockCount) break;
                this.readNextPage();
                this.currentRowOnPageIndex = 0;
                break;
            }
        }
        if (this.currentRow == null) {
            return null;
        }
        return Arrays.copyOf(this.currentRow, this.currentRow.length);
    }

    private void readNextPage() throws IOException {
        this.deletedMarkers = "";
        this.processNextPage();
        while (!(PageType.PAGE_TYPE_META.contains(this.currentPageType) || PageType.PAGE_TYPE_MIX.contains(this.currentPageType) || PageType.PAGE_TYPE_DATA.contains(this.currentPageType))) {
            if (this.eof) {
                return;
            }
            this.processNextPage();
        }
    }

    private void processNextPage() throws IOException {
        int bitOffset = this.sasFileProperties.isU64() ? 32 : 16;
        this.currentPageDataSubheaderPointers.clear();
        try {
            this.sasFileStream.readFully(this.cachedPage, 0, this.sasFileProperties.getPageLength());
        }
        catch (EOFException ex) {
            this.eof = true;
            return;
        }
        this.readPageHeader();
        if (PageType.PAGE_TYPE_META.contains(this.currentPageType) || PageType.PAGE_TYPE_AMD.contains(this.currentPageType) || PageType.PAGE_TYPE_MIX.contains(this.currentPageType)) {
            ArrayList<SubheaderPointer> subheaderPointers = new ArrayList<SubheaderPointer>();
            this.processPageMetadata(bitOffset, subheaderPointers);
            this.readDeletedInfo();
            if (PageType.PAGE_TYPE_AMD.contains(this.currentPageType)) {
                this.processMissingColumnInfo();
            }
        }
    }

    private void processMissingColumnInfo() throws UnsupportedEncodingException {
        for (ColumnMissingInfo columnMissingInfo : this.columnMissingInfoList) {
            String missedInfo = this.bytesToString(this.columnsNamesBytes.get(columnMissingInfo.getTextSubheaderIndex()), columnMissingInfo.getOffset(), columnMissingInfo.getLength()).intern();
            Column column = this.columns.get(columnMissingInfo.getColumnId());
            switch (columnMissingInfo.getMissingInfoType()) {
                case NAME: {
                    column.setName(missedInfo);
                    break;
                }
                case FORMAT: {
                    column.setFormat(new ColumnFormat(missedInfo));
                    break;
                }
                case LABEL: {
                    column.setLabel(missedInfo);
                    break;
                }
            }
        }
    }

    private void readPageHeader() throws IOException {
        int bitOffset = this.sasFileProperties.isU64() ? 32 : 16;
        Long[] offset = new Long[]{(long)bitOffset + 0L, (long)bitOffset + 2L, (long)bitOffset + 4L};
        Integer[] length = new Integer[]{2, 2, 2};
        List<byte[]> vars = this.getBytesFromFile(offset, length);
        this.currentPageType = this.bytesToShort(vars.get(0));
        LOGGER.debug("Page type: {}", (Object)this.currentPageType);
        this.currentPageBlockCount = this.bytesToShort(vars.get(1));
        LOGGER.debug("Block count: {}", (Object)this.currentPageBlockCount);
        this.currentPageSubheadersCount = this.bytesToShort(vars.get(2));
        LOGGER.debug("Subheader count: {}", (Object)this.currentPageSubheadersCount);
    }

    private void readDeletedInfo() throws IOException {
        byte[] x;
        int bitOffset;
        int subheaderPointerLength;
        long deletedPointerOffset;
        if (this.sasFileProperties.isU64()) {
            deletedPointerOffset = 24L;
            subheaderPointerLength = 24;
            bitOffset = 40;
        } else {
            deletedPointerOffset = 12L;
            subheaderPointerLength = 12;
            bitOffset = 24;
        }
        int alignCorrection = (bitOffset + 8 + this.currentPageSubheadersCount * subheaderPointerLength) % 8;
        List<byte[]> vars = this.getBytesFromFile(new Long[]{deletedPointerOffset}, new Integer[]{4});
        long currentPageDeletedPointer = this.bytesToInt(vars.get(0));
        long deletedMapOffset = (long)bitOffset + currentPageDeletedPointer + (long)alignCorrection + (long)(this.currentPageSubheadersCount * subheaderPointerLength) + (long)(this.currentPageBlockCount - this.currentPageSubheadersCount) * this.sasFileProperties.getRowLength();
        List<byte[]> bytes = this.getBytesFromFile(new Long[]{deletedMapOffset}, new Integer[]{(int)Math.ceil((double)(this.currentPageBlockCount - this.currentPageSubheadersCount) / 8.0)});
        for (byte b : x = bytes.get(0)) {
            this.deletedMarkers = this.deletedMarkers + String.format("%8s", Integer.toString(b & 0xFF, 2)).replace(" ", "0");
        }
    }

    private Object[] processByteArrayWithData(long rowOffset, long rowLength, List<String> columnNames) {
        int offset;
        byte[] source;
        Object[] rowElements = columnNames != null ? new Object[columnNames.size()] : new Object[(int)this.sasFileProperties.getColumnsCount()];
        if (this.sasFileProperties.isCompressed() && rowLength < this.sasFileProperties.getRowLength()) {
            Decompressor decompressor = LITERALS_TO_DECOMPRESSOR.get(this.sasFileProperties.getCompressionMethod());
            source = decompressor.decompressRow((int)rowOffset, (int)rowLength, (int)this.sasFileProperties.getRowLength(), this.cachedPage);
            offset = 0;
        } else {
            source = this.cachedPage;
            offset = (int)rowOffset;
        }
        int currentColumnIndex = 0;
        while ((long)currentColumnIndex < this.sasFileProperties.getColumnsCount() && this.columnsDataLength.get(currentColumnIndex) != 0) {
            if (columnNames == null) {
                rowElements[currentColumnIndex] = this.processElement(source, offset, currentColumnIndex);
            } else {
                String name = this.columns.get(currentColumnIndex).getName();
                if (columnNames.contains(name)) {
                    rowElements[columnNames.indexOf((Object)name)] = this.processElement(source, offset, currentColumnIndex);
                }
            }
            ++currentColumnIndex;
        }
        return rowElements;
    }

    private Object processElement(byte[] source, int offset, int currentColumnIndex) {
        int length = this.columnsDataLength.get(currentColumnIndex);
        if (this.columns.get(currentColumnIndex).getType() == Number.class) {
            byte[] temp = Arrays.copyOfRange(source, offset + (int)this.columnsDataOffset.get(currentColumnIndex).longValue(), offset + (int)this.columnsDataOffset.get(currentColumnIndex).longValue() + length);
            if (this.columnsDataLength.get(currentColumnIndex) <= 2) {
                return this.bytesToShort(temp);
            }
            if (this.columns.get(currentColumnIndex).getFormat().getName().isEmpty()) {
                return this.convertByteArrayToNumber(temp);
            }
            ColumnFormat columnFormat = this.columns.get(currentColumnIndex).getFormat();
            String sasDateFormat = columnFormat.getName();
            if (SasTemporalFormatter.isDateTimeFormat(sasDateFormat)) {
                return this.bytesToDateTime(temp, this.outputDateType, columnFormat);
            }
            if (SasTemporalFormatter.isDateFormat(sasDateFormat)) {
                return this.bytesToDate(temp, this.outputDateType, columnFormat);
            }
            if (SasTemporalFormatter.isTimeFormat(sasDateFormat)) {
                return this.bytesToTime(temp, this.outputDateType, columnFormat);
            }
            return this.convertByteArrayToNumber(temp);
        }
        byte[] bytes = this.trimBytesArray(source, offset + this.columnsDataOffset.get(currentColumnIndex).intValue(), length);
        if (this.byteOutput.booleanValue()) {
            return bytes;
        }
        try {
            return bytes == null ? null : this.bytesToString(bytes);
        }
        catch (UnsupportedEncodingException e) {
            LOGGER.error(e.getMessage(), (Throwable)e);
            return null;
        }
    }

    private List<byte[]> getBytesFromFile(Long[] offset, Integer[] length) throws IOException {
        ArrayList<byte[]> vars = new ArrayList<byte[]>();
        if (this.cachedPage == null) {
            for (int i = 0; i < offset.length; ++i) {
                byte[] temp = new byte[length[i].intValue()];
                this.skipBytes(offset[i] - (long)this.currentFilePosition);
                try {
                    this.sasFileStream.readFully(temp, 0, length[i]);
                }
                catch (EOFException e) {
                    this.eof = true;
                }
                this.currentFilePosition = (int)offset[i].longValue() + length[i];
                vars.add(temp);
            }
        } else {
            for (int i = 0; i < offset.length; ++i) {
                if ((long)this.cachedPage.length < offset[i]) {
                    throw new IOException("There are no available bytes in the input stream.");
                }
                vars.add(Arrays.copyOfRange(this.cachedPage, (int)offset[i].longValue(), (int)offset[i].longValue() + length[i]));
            }
        }
        return vars;
    }

    private long correctLongProcess(ByteBuffer byteBuffer) {
        if (this.sasFileProperties.isU64()) {
            return byteBuffer.getLong();
        }
        return byteBuffer.getInt();
    }

    private ByteBuffer byteArrayToByteBuffer(byte[] data) {
        ByteBuffer byteBuffer = ByteBuffer.wrap(data);
        if (this.sasFileProperties.getEndianness() == 0) {
            return byteBuffer;
        }
        return byteBuffer.order(ByteOrder.LITTLE_ENDIAN);
    }

    private Object convertByteArrayToNumber(byte[] mass) {
        double resultDouble = this.bytesToDouble(mass);
        if (Double.isNaN(resultDouble) || resultDouble < 1.0E-300 && resultDouble > 0.0) {
            return null;
        }
        long resultLong = Math.round(resultDouble);
        if (Math.abs(resultDouble - (double)resultLong) >= 1.0E-14) {
            return resultDouble;
        }
        return resultLong;
    }

    private int bytesToShort(byte[] bytes) {
        return this.byteArrayToByteBuffer(bytes).getShort();
    }

    private int bytesToInt(byte[] bytes) {
        return this.byteArrayToByteBuffer(bytes).getInt();
    }

    private long bytesToLong(byte[] bytes) {
        return this.correctLongProcess(this.byteArrayToByteBuffer(bytes));
    }

    private String bytesToString(byte[] bytes) throws UnsupportedEncodingException {
        return new String(bytes, this.encoding);
    }

    private String bytesToString(byte[] bytes, int offset, int length) throws UnsupportedEncodingException, StringIndexOutOfBoundsException {
        return new String(bytes, offset, length, this.encoding);
    }

    private Date bytesToDateTime(byte[] bytes) {
        double doubleSeconds = this.bytesToDouble(bytes);
        if (Double.isNaN(doubleSeconds)) {
            return null;
        }
        return this.sasTemporalFormatter.formatSasSecondsAsJavaDate(doubleSeconds);
    }

    private Object bytesToDateTime(byte[] bytes, OutputDateType outputDateType, ColumnFormat columnFormat) {
        double doubleSeconds = this.bytesToDouble(bytes);
        return this.sasTemporalFormatter.formatSasDateTime(doubleSeconds, outputDateType, columnFormat.getName(), columnFormat.getWidth(), columnFormat.getPrecision());
    }

    private Object bytesToTime(byte[] bytes, OutputDateType outputDateType, ColumnFormat columnFormat) {
        double doubleSeconds = this.bytesToDouble(bytes);
        return this.sasTemporalFormatter.formatSasTime(doubleSeconds, outputDateType, columnFormat.getName(), columnFormat.getWidth(), columnFormat.getPrecision());
    }

    private Object bytesToDate(byte[] bytes, OutputDateType outputDateType, ColumnFormat columnFormat) {
        double doubleDays = this.bytesToDouble(bytes);
        return this.sasTemporalFormatter.formatSasDate(doubleDays, outputDateType, columnFormat.getName(), columnFormat.getWidth(), columnFormat.getPrecision());
    }

    private double bytesToDouble(byte[] bytes) {
        ByteBuffer original = this.byteArrayToByteBuffer(bytes);
        if (bytes.length < 8) {
            ByteBuffer byteBuffer = ByteBuffer.allocate(8);
            if (this.sasFileProperties.getEndianness() == 1) {
                byteBuffer.position(8 - bytes.length);
            }
            byteBuffer.put(original);
            byteBuffer.order(original.order());
            byteBuffer.position(0);
            original = byteBuffer;
        }
        return original.getDouble();
    }

    private byte[] trimBytesArray(byte[] source, int offset, int length) {
        int lengthFromBegin;
        for (lengthFromBegin = offset + length; lengthFromBegin > offset && (source[lengthFromBegin - 1] == 32 || source[lengthFromBegin - 1] == 0 || source[lengthFromBegin - 1] == 9); --lengthFromBegin) {
        }
        if (lengthFromBegin - offset != 0) {
            return Arrays.copyOfRange(source, offset, lengthFromBegin);
        }
        return null;
    }

    List<Column> getColumns() {
        return this.columns;
    }

    public SasFileProperties getSasFileProperties() {
        return this.sasFileProperties;
    }

    static /* synthetic */ Object[] access$3002(SasFileParser x0, Object[] x1) {
        x0.currentRow = x1;
        return x1;
    }

    static {
        LITERALS_TO_DECOMPRESSOR = new HashMap<String, Decompressor>();
        SKIP_BYTE_BUFFER = new byte[4096];
        HashMap<Long, SubheaderIndexes> tmpMap = new HashMap<Long, SubheaderIndexes>();
        tmpMap.put(-134744073L, SubheaderIndexes.ROW_SIZE_SUBHEADER_INDEX);
        tmpMap.put(-151587082L, SubheaderIndexes.COLUMN_SIZE_SUBHEADER_INDEX);
        tmpMap.put(-1024L, SubheaderIndexes.SUBHEADER_COUNTS_SUBHEADER_INDEX);
        tmpMap.put(-3L, SubheaderIndexes.COLUMN_TEXT_SUBHEADER_INDEX);
        tmpMap.put(-1L, SubheaderIndexes.COLUMN_NAME_SUBHEADER_INDEX);
        tmpMap.put(-4L, SubheaderIndexes.COLUMN_ATTRIBUTES_SUBHEADER_INDEX);
        tmpMap.put(-1026L, SubheaderIndexes.FORMAT_AND_LABEL_SUBHEADER_INDEX);
        tmpMap.put(-2L, SubheaderIndexes.COLUMN_LIST_SUBHEADER_INDEX);
        tmpMap.put(0xF7F7F7F7L, SubheaderIndexes.ROW_SIZE_SUBHEADER_INDEX);
        tmpMap.put(0xF6F6F6F6L, SubheaderIndexes.COLUMN_SIZE_SUBHEADER_INDEX);
        tmpMap.put(-578721386864836608L, SubheaderIndexes.ROW_SIZE_SUBHEADER_INDEX);
        tmpMap.put(-651061559686070272L, SubheaderIndexes.COLUMN_SIZE_SUBHEADER_INDEX);
        tmpMap.put(-578721382569870338L, SubheaderIndexes.ROW_SIZE_SUBHEADER_INDEX);
        tmpMap.put(-651061555391104002L, SubheaderIndexes.COLUMN_SIZE_SUBHEADER_INDEX);
        tmpMap.put(0xFCFFFFFFFFFFFFL, SubheaderIndexes.SUBHEADER_COUNTS_SUBHEADER_INDEX);
        tmpMap.put(-144115188075855873L, SubheaderIndexes.COLUMN_TEXT_SUBHEADER_INDEX);
        tmpMap.put(-1L, SubheaderIndexes.COLUMN_NAME_SUBHEADER_INDEX);
        tmpMap.put(-216172782113783809L, SubheaderIndexes.COLUMN_ATTRIBUTES_SUBHEADER_INDEX);
        tmpMap.put(-73183493944770561L, SubheaderIndexes.FORMAT_AND_LABEL_SUBHEADER_INDEX);
        tmpMap.put(-72057594037927937L, SubheaderIndexes.COLUMN_LIST_SUBHEADER_INDEX);
        SUBHEADER_SIGNATURE_TO_INDEX = Collections.unmodifiableMap(tmpMap);
        LITERALS_TO_DECOMPRESSOR.put("SASYZCRL", CharDecompressor.INSTANCE);
        LITERALS_TO_DECOMPRESSOR.put("SASYZCR2", BinDecompressor.INSTANCE);
    }

    class DataSubheader
    implements ProcessingDataSubheader {
        DataSubheader() {
        }

        @Override
        public void processSubheader(long subheaderOffset, long subheaderLength) throws IOException {
            SasFileParser.access$3002(SasFileParser.this, SasFileParser.this.processByteArrayWithData(subheaderOffset, subheaderLength, null));
        }

        @Override
        public void processSubheader(long subheaderOffset, long subheaderLength, List<String> columnNames) throws IOException {
            SasFileParser.access$3002(SasFileParser.this, SasFileParser.this.processByteArrayWithData(subheaderOffset, subheaderLength, columnNames));
        }
    }

    static class ColumnListSubheader
    implements ProcessingSubheader {
        ColumnListSubheader() {
        }

        @Override
        public void processSubheader(long subheaderOffset, long subheaderLength) throws IOException {
        }
    }

    class FormatAndLabelSubheader
    implements ProcessingSubheader {
        FormatAndLabelSubheader() {
        }

        @Override
        public void processSubheader(long subheaderOffset, long subheaderLength) throws IOException {
            int intOrLongLength = SasFileParser.this.sasFileProperties.isU64() ? 8 : 4;
            Long[] offset = new Long[]{subheaderOffset + 0L + (long)(3 * intOrLongLength), subheaderOffset + 2L + (long)(3 * intOrLongLength), subheaderOffset + 22L + (long)(3 * intOrLongLength), subheaderOffset + 24L + (long)(3 * intOrLongLength), subheaderOffset + 26L + (long)(3 * intOrLongLength), subheaderOffset + 28L + (long)(3 * intOrLongLength), subheaderOffset + 30L + (long)(3 * intOrLongLength), subheaderOffset + 32L + (long)(3 * intOrLongLength)};
            Integer[] length = new Integer[]{2, 2, 2, 2, 2, 2, 2, 2};
            List vars = SasFileParser.this.getBytesFromFile(offset, length);
            int columnFormatWidth = SasFileParser.this.bytesToShort((byte[])vars.get(0));
            int columnFormatPrecision = SasFileParser.this.bytesToShort((byte[])vars.get(1));
            int textSubheaderIndexForFormat = SasFileParser.this.bytesToShort((byte[])vars.get(2));
            int columnFormatOffset = SasFileParser.this.bytesToShort((byte[])vars.get(3));
            int columnFormatLength = SasFileParser.this.bytesToShort((byte[])vars.get(4));
            int textSubheaderIndexForLabel = SasFileParser.this.bytesToShort((byte[])vars.get(5));
            int columnLabelOffset = SasFileParser.this.bytesToShort((byte[])vars.get(6));
            int columnLabelLength = SasFileParser.this.bytesToShort((byte[])vars.get(7));
            String columnLabel = "";
            String columnFormatName = "";
            if (textSubheaderIndexForLabel < SasFileParser.this.columnsNamesBytes.size()) {
                columnLabel = SasFileParser.this.bytesToString((byte[])SasFileParser.this.columnsNamesBytes.get(textSubheaderIndexForLabel), columnLabelOffset, columnLabelLength).intern();
            } else {
                SasFileParser.this.columnMissingInfoList.add(new ColumnMissingInfo(SasFileParser.this.columns.size(), textSubheaderIndexForLabel, columnLabelOffset, columnLabelLength, ColumnMissingInfo.MissingInfoType.LABEL));
            }
            if (textSubheaderIndexForFormat < SasFileParser.this.columnsNamesBytes.size()) {
                columnFormatName = SasFileParser.this.bytesToString((byte[])SasFileParser.this.columnsNamesBytes.get(textSubheaderIndexForFormat), columnFormatOffset, columnFormatLength).intern();
            } else {
                SasFileParser.this.columnMissingInfoList.add(new ColumnMissingInfo(SasFileParser.this.columns.size(), textSubheaderIndexForFormat, columnFormatOffset, columnFormatLength, ColumnMissingInfo.MissingInfoType.FORMAT));
            }
            LOGGER.debug("Column format: {}", (Object)columnFormatName);
            ColumnFormat columnFormat = new ColumnFormat(columnFormatName, columnFormatWidth, columnFormatPrecision);
            SasFileParser.this.columns.add(new Column(SasFileParser.this.currentColumnNumber + 1, (String)SasFileParser.this.columnsNamesList.get(SasFileParser.this.columns.size()), columnLabel, columnFormat, (Class)SasFileParser.this.columnsTypesList.get(SasFileParser.this.columns.size()), (Integer)SasFileParser.this.columnsDataLength.get(SasFileParser.this.currentColumnNumber++)));
        }
    }

    class ColumnAttributesSubheader
    implements ProcessingSubheader {
        ColumnAttributesSubheader() {
        }

        @Override
        public void processSubheader(long subheaderOffset, long subheaderLength) throws IOException {
            int intOrLongLength = SasFileParser.this.sasFileProperties.isU64() ? 8 : 4;
            long columnAttributesVectorsCount = (subheaderLength - (long)(2 * intOrLongLength) - 12L) / (long)(intOrLongLength + 8);
            int i = 0;
            while ((long)i < columnAttributesVectorsCount) {
                Long[] offset = new Long[]{subheaderOffset + (long)intOrLongLength + 8L + (long)(i * (intOrLongLength + 8)), subheaderOffset + (long)(2 * intOrLongLength) + 8L + (long)(i * (intOrLongLength + 8)), subheaderOffset + (long)(2 * intOrLongLength) + 14L + (long)(i * (intOrLongLength + 8))};
                Integer[] length = new Integer[]{intOrLongLength, 4, 1};
                List vars = SasFileParser.this.getBytesFromFile(offset, length);
                SasFileParser.this.columnsDataOffset.add(SasFileParser.this.bytesToLong((byte[])vars.get(0)));
                SasFileParser.this.columnsDataLength.add(SasFileParser.this.bytesToInt((byte[])vars.get(1)));
                SasFileParser.this.columnsTypesList.add(((byte[])vars.get(2))[0] == 1 ? Number.class : String.class);
                ++i;
            }
        }
    }

    class ColumnNameSubheader
    implements ProcessingSubheader {
        ColumnNameSubheader() {
        }

        @Override
        public void processSubheader(long subheaderOffset, long subheaderLength) throws IOException {
            int intOrLongLength = SasFileParser.this.sasFileProperties.isU64() ? 8 : 4;
            long columnNamePointersCount = (subheaderLength - (long)(2 * intOrLongLength) - 12L) / 8L;
            int i = 0;
            while ((long)i < columnNamePointersCount) {
                Long[] offset = new Long[]{subheaderOffset + (long)intOrLongLength + (long)(8 * (i + 1)) + 0L, subheaderOffset + (long)intOrLongLength + (long)(8 * (i + 1)) + 2L, subheaderOffset + (long)intOrLongLength + (long)(8 * (i + 1)) + 4L};
                Integer[] length = new Integer[]{2, 2, 2};
                List vars = SasFileParser.this.getBytesFromFile(offset, length);
                int textSubheaderIndex = SasFileParser.this.bytesToShort((byte[])vars.get(0));
                int columnNameOffset = SasFileParser.this.bytesToShort((byte[])vars.get(1));
                int columnNameLength = SasFileParser.this.bytesToShort((byte[])vars.get(2));
                if (textSubheaderIndex < SasFileParser.this.columnsNamesBytes.size()) {
                    SasFileParser.this.columnsNamesList.add(SasFileParser.this.bytesToString((byte[])SasFileParser.this.columnsNamesBytes.get(textSubheaderIndex), columnNameOffset, columnNameLength).intern());
                } else {
                    SasFileParser.this.columnsNamesList.add(new String(new char[columnNameLength]));
                    SasFileParser.this.columnMissingInfoList.add(new ColumnMissingInfo(i, textSubheaderIndex, columnNameOffset, columnNameLength, ColumnMissingInfo.MissingInfoType.NAME));
                }
                ++i;
            }
        }
    }

    class ColumnTextSubheader
    implements ProcessingSubheader {
        ColumnTextSubheader() {
        }

        @Override
        public void processSubheader(long subheaderOffset, long subheaderLength) throws IOException {
            int intOrLongLength = SasFileParser.this.sasFileProperties.isU64() ? 8 : 4;
            Long[] offset = new Long[]{subheaderOffset + (long)intOrLongLength};
            Integer[] length = new Integer[]{2};
            List vars = SasFileParser.this.getBytesFromFile(offset, length);
            short textBlockSize = SasFileParser.this.byteArrayToByteBuffer((byte[])vars.get(0)).getShort();
            offset[0] = subheaderOffset + (long)intOrLongLength;
            length[0] = textBlockSize;
            vars = SasFileParser.this.getBytesFromFile(offset, length);
            SasFileParser.this.columnsNamesBytes.add(vars.get(0));
            if (SasFileParser.this.columnsNamesBytes.size() == 1) {
                byte[] columnName = (byte[])SasFileParser.this.columnsNamesBytes.get(0);
                String compressionMethod = SasFileParser.this.bytesToString(columnName, SasFileParser.this.compressionMethodOffset, SasFileParser.this.compressionMethodLength);
                if (SasFileParser.this.matchCompressionMethod(compressionMethod)) {
                    SasFileParser.this.sasFileProperties.setCompressionMethod(compressionMethod);
                }
                SasFileParser.this.sasFileProperties.setFileLabel(SasFileParser.this.bytesToString(columnName, SasFileParser.this.fileLabelOffset, SasFileParser.this.fileLabelLength));
            }
        }
    }

    static class SubheaderCountsSubheader
    implements ProcessingSubheader {
        SubheaderCountsSubheader() {
        }

        @Override
        public void processSubheader(long subheaderOffset, long subheaderLength) throws IOException {
        }
    }

    class ColumnSizeSubheader
    implements ProcessingSubheader {
        ColumnSizeSubheader() {
        }

        @Override
        public void processSubheader(long subheaderOffset, long subheaderLength) throws IOException {
            int intOrLongLength = SasFileParser.this.sasFileProperties.isU64() ? 8 : 4;
            Long[] offset = new Long[]{subheaderOffset + (long)intOrLongLength};
            Integer[] length = new Integer[]{intOrLongLength};
            List vars = SasFileParser.this.getBytesFromFile(offset, length);
            SasFileParser.this.sasFileProperties.setColumnsCount(SasFileParser.this.bytesToLong((byte[])vars.get(0)));
        }
    }

    class RowSizeSubheader
    implements ProcessingSubheader {
        RowSizeSubheader() {
        }

        @Override
        public void processSubheader(long subheaderOffset, long subheaderLength) throws IOException {
            int intOrLongLength = SasFileParser.this.sasFileProperties.isU64() ? 8 : 4;
            Long[] offset = new Long[]{subheaderOffset + (long)(5 * intOrLongLength), subheaderOffset + (long)(6 * intOrLongLength), subheaderOffset + (long)(15 * intOrLongLength), subheaderOffset + 24L + (long)(82 * intOrLongLength), subheaderOffset + 26L + (long)(82 * intOrLongLength), subheaderOffset + (long)(8 * intOrLongLength), subheaderOffset + 36L + (long)(82 * intOrLongLength), subheaderOffset + 38L + (long)(82 * intOrLongLength)};
            Integer[] length = new Integer[]{intOrLongLength, intOrLongLength, intOrLongLength, 2, 2, intOrLongLength, 2, 2};
            List vars = SasFileParser.this.getBytesFromFile(offset, length);
            if (SasFileParser.this.sasFileProperties.getRowLength() == 0L) {
                SasFileParser.this.sasFileProperties.setRowLength(SasFileParser.this.bytesToLong((byte[])vars.get(0)));
            }
            if (SasFileParser.this.sasFileProperties.getRowCount() == 0L) {
                SasFileParser.this.sasFileProperties.setRowCount(SasFileParser.this.bytesToLong((byte[])vars.get(1)));
            }
            if (SasFileParser.this.sasFileProperties.getMixPageRowCount() == 0L) {
                SasFileParser.this.sasFileProperties.setMixPageRowCount(SasFileParser.this.bytesToLong((byte[])vars.get(2)));
            }
            SasFileParser.this.fileLabelOffset = SasFileParser.this.bytesToShort((byte[])vars.get(3));
            SasFileParser.this.fileLabelLength = SasFileParser.this.bytesToShort((byte[])vars.get(4));
            if (SasFileParser.this.sasFileProperties.getDeletedRowCount() == 0L) {
                SasFileParser.this.sasFileProperties.setDeletedRowCount(SasFileParser.this.bytesToLong((byte[])vars.get(5)));
            }
            SasFileParser.this.compressionMethodOffset = SasFileParser.this.bytesToShort((byte[])vars.get(6));
            SasFileParser.this.compressionMethodLength = SasFileParser.this.bytesToShort((byte[])vars.get(7));
        }
    }

    static class SubheaderPointer {
        private final long offset;
        private final long length;
        private final byte compression;
        private final byte type;

        SubheaderPointer(long offset, long length, byte compression, byte type) {
            this.offset = offset;
            this.length = length;
            this.compression = compression;
            this.type = type;
        }
    }

    public static class Builder {
        private InputStream sasFileStream;
        private String encoding;
        private OutputDateType outputDateType = OutputDateType.JAVA_DATE_LEGACY;
        private Boolean byteOutput = false;

        private Builder() {
        }

        public Builder(InputStream sasFileStream) {
            this.sasFileStream = sasFileStream;
        }

        public Builder encoding(String val) {
            this.encoding = val;
            return this;
        }

        public Builder outputDateType(OutputDateType val) {
            if (val != null) {
                this.outputDateType = val;
            }
            return this;
        }

        public Builder byteOutput(Boolean val) {
            this.byteOutput = val;
            return this;
        }

        public SasFileParser build() {
            return new SasFileParser(this);
        }
    }

    private static interface ProcessingDataSubheader
    extends ProcessingSubheader {
        public void processSubheader(long var1, long var3, List<String> var5) throws IOException;
    }

    private static interface ProcessingSubheader {
        public void processSubheader(long var1, long var3) throws IOException;
    }

    private static enum SubheaderIndexes {
        ROW_SIZE_SUBHEADER_INDEX,
        COLUMN_SIZE_SUBHEADER_INDEX,
        SUBHEADER_COUNTS_SUBHEADER_INDEX,
        COLUMN_TEXT_SUBHEADER_INDEX,
        COLUMN_NAME_SUBHEADER_INDEX,
        COLUMN_ATTRIBUTES_SUBHEADER_INDEX,
        FORMAT_AND_LABEL_SUBHEADER_INDEX,
        COLUMN_LIST_SUBHEADER_INDEX,
        DATA_SUBHEADER_INDEX;

    }
}

