/*
 * Decompiled with CFR 0.152.
 */
package org.infinispan.quarkus.hibernate.cache;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.Expiry;
import com.github.benmanes.caffeine.cache.Ticker;
import java.time.Duration;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Predicate;
import org.infinispan.quarkus.hibernate.cache.InternalCache;
import org.infinispan.quarkus.hibernate.cache.InternalCacheConfig;
import org.infinispan.quarkus.hibernate.cache.Time;
import org.infinispan.quarkus.hibernate.cache.VersionedEntry;
import org.jboss.logging.Logger;

final class CaffeineCache
implements InternalCache {
    private static final Logger log = Logger.getLogger(CaffeineCache.class);
    private static final boolean trace = log.isTraceEnabled();
    private static final Ticker TICKER = Ticker.systemTicker();
    static final Time.NanosService TIME_SERVICE = () -> ((Ticker)TICKER).read();
    private final Cache cache;
    private final String cacheName;

    CaffeineCache(String cacheName, InternalCacheConfig config, Time.NanosService nanosTimeService) {
        Duration maxIdle = config.maxIdle;
        long objectCount = config.objectCount;
        this.cacheName = cacheName;
        Caffeine cacheBuilder = Caffeine.newBuilder().ticker(nanosTimeService::nanoTime);
        if (!Time.isForever(maxIdle)) {
            cacheBuilder.expireAfter((Expiry)new CacheExpiryPolicy(maxIdle));
        }
        if (objectCount >= 0L) {
            cacheBuilder.maximumSize(objectCount);
        }
        this.cache = cacheBuilder.build();
    }

    @Override
    public Object getOrNull(Object key) {
        Object value = this.cache.getIfPresent(key);
        if (trace) {
            log.tracef("Cache get(key=%s) returns: %s", key, value);
        }
        return value;
    }

    @Override
    public void putIfAbsent(Object key, Object value) {
        if (trace) {
            log.tracef("Cache put if absent key=%s value=%s", key, value);
        }
        this.cache.asMap().putIfAbsent(key, value);
    }

    @Override
    public void put(Object key, Object value) {
        if (trace) {
            log.tracef("Cache put key=%s value=%s", key, value);
        }
        this.cache.put(key, value);
    }

    @Override
    public Object compute(Object key, BiFunction<Object, Object, Object> remappingFunction) {
        Object result = this.cache.asMap().compute(key, remappingFunction);
        if (trace) {
            log.tracef("Computing function=%s on key=%s returns: %s", remappingFunction, key, result);
        }
        return result;
    }

    @Override
    public void invalidate(Object key) {
        if (trace) {
            log.tracef("Cache invalidate key %s", key);
        }
        this.cache.invalidate(key);
    }

    @Override
    public long size(Predicate<Map.Entry> filter) {
        this.cache.cleanUp();
        if (filter == null) {
            long size = this.cache.estimatedSize();
            if (trace) {
                log.tracef("Cache(%s) size estimated at %d elements", (Object)this.cacheName, (Object)size);
            }
            return size;
        }
        long size = this.cache.asMap().entrySet().stream().filter(filter).count();
        if (trace) {
            log.tracef("Cache(%s) size for entries matching filter(%s) is %d elements", (Object)this.cacheName, filter, (Object)size);
        }
        return size;
    }

    @Override
    public void forEach(Predicate<Map.Entry> filter, Consumer<Map.Entry> action) {
        this.cache.asMap().entrySet().stream().filter(filter).forEach(action);
    }

    @Override
    public void stop() {
        if (trace) {
            log.tracef("Cleanup cache %s", (Object)this.cacheName);
        }
        this.cache.cleanUp();
    }

    @Override
    public void invalidateAll() {
        if (trace) {
            log.tracef("Invalidate all in cache %s", (Object)this.cacheName);
        }
        this.cache.invalidateAll();
    }

    private static final class CacheExpiryPolicy
    implements Expiry {
        private final long maxIdleNanos;

        public CacheExpiryPolicy(Duration maxIdle) {
            this.maxIdleNanos = maxIdle == null ? -1L : maxIdle.toNanos();
        }

        public long expireAfterCreate(Object key, Object value, long currentTime) {
            return this.calculateExpiration(value, currentTime);
        }

        private long calculateExpiration(Object value, long currentTime) {
            if (value instanceof VersionedEntry) {
                VersionedEntry versioned = (VersionedEntry)value;
                if (this.maxIdleNanos > 0L) {
                    long idleDeadline = currentTime + this.maxIdleNanos;
                    long versionedLifespan = versioned.getLifespanNanos();
                    log.tracef("Expire after create, either idle deadline %d (ns) or versioned entry lifespan %d (ns)", idleDeadline, versionedLifespan);
                    return Math.min(idleDeadline, versionedLifespan);
                }
                return versioned.getLifespanNanos();
            }
            if (this.maxIdleNanos > 0L) {
                return this.maxIdleNanos;
            }
            return Long.MAX_VALUE;
        }

        public long expireAfterUpdate(Object key, Object value, long currentTime, long currentDuration) {
            return this.calculateExpiration(value, currentTime);
        }

        public long expireAfterRead(Object key, Object value, long currentTime, long currentDuration) {
            if (this.maxIdleNanos > 0L) {
                return this.maxIdleNanos;
            }
            return Long.MAX_VALUE;
        }
    }
}

