/*
 * Decompiled with CFR 0.152.
 */
package ghidra.app.plugin.core.decompile.actions;

import ghidra.app.plugin.core.decompile.actions.ASTGraphSelectionHandler;
import ghidra.app.services.GraphService;
import ghidra.program.model.address.Address;
import ghidra.program.model.graph.GraphData;
import ghidra.program.model.graph.GraphDisplay;
import ghidra.program.model.graph.GraphSelectionHandler;
import ghidra.program.model.graph.GraphVertex;
import ghidra.program.model.lang.Register;
import ghidra.program.model.listing.Program;
import ghidra.program.model.pcode.HighFunction;
import ghidra.program.model.pcode.HighVariable;
import ghidra.program.model.pcode.PcodeBlock;
import ghidra.program.model.pcode.PcodeBlockBasic;
import ghidra.program.model.pcode.PcodeOp;
import ghidra.program.model.pcode.PcodeOpAST;
import ghidra.program.model.pcode.Varnode;
import ghidra.util.Msg;
import ghidra.util.NumericUtilities;
import ghidra.util.exception.CancelledException;
import ghidra.util.exception.GraphException;
import ghidra.util.task.Task;
import ghidra.util.task.TaskListener;
import ghidra.util.task.TaskMonitor;
import java.util.Iterator;

public class ASTGraphTask
extends Task {
    static final int CONTROL_FLOW_GRAPH = 0;
    static final int DATA_FLOW_GRAPH = 1;
    private static final String[] GRAPH_TYPES = new String[]{"AST Control Flow", "AST Data Flow"};
    private static final String CODE_ATTRIBUTE = "Code";
    private static final String SYMBOLS_ATTRIBUTE = "Symbols";
    private static final String VERTEX_TYPE_ATTRIBUTE = "VertexType";
    private static final String ENTRY_NODE = "Entry";
    private static final String BODY_NODE = "Body";
    private static final String EXIT_NODE = "Exit";
    private static final String SWITCH_NODE = "Switch";
    private static final String BAD_NODE = "Bad";
    private static final String DATA_NODE = "Data";
    private GraphService graphService;
    private boolean newGraph;
    private int codeLimitPerBlock;
    private Address location;
    private HighFunction hfunction;
    private int graphType;
    private int uniqueNum = 0;
    private TaskListener listener;

    public ASTGraphTask(GraphService graphService, boolean newGraph, int codeLimitPerBlock, Address location, HighFunction hfunction, int graphType) {
        super("Graph " + GRAPH_TYPES[graphType], true, false, true);
        this.graphService = graphService;
        this.newGraph = newGraph;
        this.codeLimitPerBlock = codeLimitPerBlock;
        this.location = location;
        this.hfunction = hfunction;
        this.graphType = graphType;
        this.listener = new TaskListener(){

            public void taskCancelled(Task task) {
            }

            public void taskCompleted(Task task) {
                try {
                    GraphDisplay graphDisplay = ASTGraphTask.this.graphService.getGraphDisplay(false);
                    if (graphDisplay != null) {
                        graphDisplay.popup();
                    }
                }
                catch (GraphException graphException) {
                    // empty catch block
                }
            }
        };
        this.addTaskListener(this.listener);
    }

    public void run(TaskMonitor monitor) {
        GraphData graph = this.graphService.createGraphContent();
        if (graph == null) {
            return;
        }
        ASTGraphSelectionHandler handler = null;
        try {
            monitor.setMessage("Computing Graph...");
            if (this.graphType == 1) {
                this.createDataFlowGraph(graph, monitor);
            } else {
                this.createControlFlowGraph(graph, monitor);
            }
            handler = new ASTGraphSelectionHandler(this.graphService, this.hfunction, this.graphType);
        }
        catch (CancelledException e1) {
            return;
        }
        try {
            monitor.setMessage("Obtaining handle to graph provider...");
            GraphDisplay display = this.graphService.getGraphDisplay(this.newGraph);
            if (monitor.isCancelled()) {
                return;
            }
            monitor.setCancelEnabled(false);
            if (!this.newGraph) {
                display.clear();
            }
            display.setSelectionHandler((GraphSelectionHandler)handler);
            monitor.setMessage("Rendering Graph...");
            display.defineVertexAttribute(CODE_ATTRIBUTE);
            display.defineVertexAttribute(SYMBOLS_ATTRIBUTE);
            display.setGraphData(graph);
            display.setVertexLabel(CODE_ATTRIBUTE, 0, 12, true, this.graphType == 0 ? this.codeLimitPerBlock + 1 : 1);
            if (this.location != null) {
                display.locate((Object)this.location, false);
            }
        }
        catch (GraphException e) {
            Msg.showError((Object)((Object)this), null, (String)"Graph Error", (Object)e.getMessage());
        }
    }

    protected void createDataFlowGraph(GraphData graph, TaskMonitor monitor) throws CancelledException {
        Iterator opIter = this.hfunction.getPcodeOps();
        while (opIter.hasNext()) {
            monitor.checkCanceled();
            this.graphOpData(graph, (PcodeOpAST)opIter.next(), monitor);
        }
    }

    private void graphOpData(GraphData graph, PcodeOpAST op, TaskMonitor monitor) throws CancelledException {
        if (op == null || op.getOpcode() == 61) {
            return;
        }
        GraphVertex opVertex = this.getOpVertex(graph, op, monitor);
        Varnode output = op.getOutput();
        if (output != null) {
            opVertex = this.getOpVertex(graph, op, monitor);
            GraphVertex outVertex = this.getDataVertex(graph, output, monitor);
            graph.createEdge(Integer.toString(++this.uniqueNum), opVertex, outVertex);
        }
        int start = 0;
        int stop = op.getNumInputs() - 1;
        switch (op.getOpcode()) {
            case 2: 
            case 3: 
            case 4: 
            case 7: {
                start = 1;
                break;
            }
            case 61: {
                stop = 1;
            }
        }
        for (int i = start; i <= stop; ++i) {
            monitor.checkCanceled();
            Varnode input = op.getInput(i);
            if (input == null) continue;
            if (opVertex == null) {
                opVertex = this.getOpVertex(graph, op, monitor);
            }
            GraphVertex inVertex = this.getDataVertex(graph, input, monitor);
            graph.createEdge(Integer.toString(++this.uniqueNum), inVertex, opVertex);
        }
    }

    private GraphVertex getOpVertex(GraphData graph, PcodeOpAST op, TaskMonitor monitor) {
        String key = "O_" + Integer.toString(op.getSeqnum().getTime());
        GraphVertex vertex = graph.getVertex(key);
        if (vertex == null) {
            vertex = graph.createVertex(key, key);
            this.setOpVertexAttributes(vertex, op);
        }
        return vertex;
    }

    private void setOpVertexAttributes(GraphVertex vertex, PcodeOpAST op) {
        vertex.setAttribute(CODE_ATTRIBUTE, this.formatOpMnemonic((PcodeOp)op));
        String vertexType = BODY_NODE;
        switch (op.getOpcode()) {
            case 4: 
            case 5: 
            case 6: 
            case 7: 
            case 8: {
                vertexType = SWITCH_NODE;
                break;
            }
            case 10: {
                vertexType = EXIT_NODE;
            }
        }
        vertex.setAttribute(VERTEX_TYPE_ATTRIBUTE, vertexType);
    }

    private GraphVertex getDataVertex(GraphData graph, Varnode node, TaskMonitor monitor) {
        Object key;
        GraphVertex vertex = null;
        HighVariable var = node.getHigh();
        if (var != null) {
            key = "V_" + var.getName();
            vertex = graph.getVertex((String)key);
        } else {
            key = Integer.toString(++this.uniqueNum);
        }
        if (vertex == null) {
            vertex = graph.createVertex((String)key, (String)key);
            this.setVarnodeVertexAttributes(vertex, node);
        }
        return vertex;
    }

    private void setVarnodeVertexAttributes(GraphVertex vertex, Varnode node) {
        Object label = "";
        HighVariable var = node.getHigh();
        if (var != null) {
            label = var.getName() + ": ";
        }
        label = (String)label + this.translateVarnode(node, false);
        vertex.setAttribute(CODE_ATTRIBUTE, (String)label);
        vertex.setAttribute(VERTEX_TYPE_ATTRIBUTE, DATA_NODE);
    }

    protected void createControlFlowGraph(GraphData graph, TaskMonitor monitor) throws CancelledException {
        Iterator pblockIter = this.hfunction.getBasicBlocks().iterator();
        while (pblockIter.hasNext()) {
            monitor.checkCanceled();
            this.graphPcodeBlock(graph, (PcodeBlock)pblockIter.next(), monitor);
        }
    }

    private void graphPcodeBlock(GraphData graph, PcodeBlock pblock, TaskMonitor monitor) throws CancelledException {
        if (pblock == null) {
            return;
        }
        GraphVertex fromVertex = this.getBlockVertex(graph, pblock, monitor);
        int outCnt = pblock.getOutSize();
        for (int i = 0; i < outCnt; ++i) {
            monitor.checkCanceled();
            PcodeBlock outPBlock = pblock.getOut(i);
            GraphVertex toVertex = this.getBlockVertex(graph, outPBlock, monitor);
            graph.createEdge(Integer.toString(++this.uniqueNum), fromVertex, toVertex);
        }
    }

    private GraphVertex getBlockVertex(GraphData graph, PcodeBlock pblock, TaskMonitor monitor) {
        String key = Integer.toString(pblock.getIndex());
        GraphVertex vertex = graph.getVertex(key);
        if (vertex == null) {
            vertex = graph.createVertex(key, key);
            if (pblock instanceof PcodeBlockBasic) {
                this.setBlockVertexAttributes(vertex, (PcodeBlockBasic)pblock);
            } else {
                vertex.setAttribute(CODE_ATTRIBUTE, "<???>");
                vertex.setAttribute(VERTEX_TYPE_ATTRIBUTE, BAD_NODE);
            }
        }
        return vertex;
    }

    private void setBlockVertexAttributes(GraphVertex vertex, PcodeBlockBasic basicBlk) {
        StringBuffer buf = new StringBuffer();
        int cnt = 0;
        Iterator opIter = basicBlk.getIterator();
        while (opIter.hasNext()) {
            PcodeOp op = (PcodeOp)opIter.next();
            if (buf.length() != 0) {
                buf.append('\n');
            }
            this.formatOp(op, buf);
            if (++cnt != this.codeLimitPerBlock) continue;
            buf.append("\n...");
            break;
        }
        vertex.setAttribute(CODE_ATTRIBUTE, buf.toString());
        String vertexType = BODY_NODE;
        if (basicBlk.getInSize() == 0) {
            vertexType = ENTRY_NODE;
        } else {
            switch (basicBlk.getOutSize()) {
                case 0: {
                    vertexType = EXIT_NODE;
                    break;
                }
                case 1: {
                    vertexType = BODY_NODE;
                    break;
                }
                default: {
                    vertexType = SWITCH_NODE;
                }
            }
        }
        vertex.setAttribute(VERTEX_TYPE_ATTRIBUTE, vertexType);
    }

    private String formatOpMnemonic(PcodeOp op) {
        Object str = op.getMnemonic();
        Varnode output = op.getOutput();
        String size = null;
        if (output != null) {
            switch (output.getSize()) {
                case 1: {
                    size = "b";
                    break;
                }
                case 2: {
                    size = "w";
                    break;
                }
                case 4: {
                    size = "d";
                    break;
                }
                case 8: {
                    size = "q";
                }
            }
            if (size != null) {
                str = (String)str + "." + size;
            }
        }
        return str;
    }

    private void formatOp(PcodeOp op, StringBuffer buf) {
        Varnode output = op.getOutput();
        if (output != null) {
            buf.append(this.translateVarnode(output, true));
            buf.append(" = ");
        }
        buf.append(this.formatOpMnemonic(op));
        buf.append(" ");
        Varnode[] inputs = op.getInputs();
        for (int i = 0; i < inputs.length; ++i) {
            if (i != 0) {
                buf.append(",");
            }
            buf.append(this.translateVarnode(inputs[i], true));
        }
    }

    private String translateVarnode(Varnode node, boolean useVarName) {
        if (node == null) {
            return "null";
        }
        Program p = this.hfunction.getFunction().getProgram();
        Address addr = node.getAddress();
        if (node.isConstant()) {
            return "#" + NumericUtilities.toHexString((long)addr.getOffset(), (int)node.getSize());
        }
        if (node.isUnique()) {
            return "u_" + Long.toHexString(addr.getOffset());
        }
        if (addr.isRegisterAddress()) {
            Register r = p.getRegister(addr, node.getSize());
            if (r == null) {
                r = p.getRegister(addr);
            }
            if (r != null) {
                return r.getName();
            }
        } else {
            if (addr.isStackAddress()) {
                HighVariable var;
                if (useVarName && (var = node.getHigh()) != null) {
                    return var.getName();
                }
                return "Stack[" + NumericUtilities.toSignedHexString((long)addr.getOffset()) + "]";
            }
            if (addr.isMemoryAddress()) {
                return addr.toString(true);
            }
        }
        return node.toString();
    }
}

