"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.BasicAuthRoutes = void 0;

var _configSchema = require("@osd/config-schema");

var _security_cookie = require("../../../session/security_cookie");

var _common = require("../../../../common");

var _tenant_resolver = require("../../../multitenancy/tenant_resolver");

var _encode_uri_query = require("../../../../../../src/plugins/opensearch_dashboards_utils/common/url/encode_uri_query");

/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
class BasicAuthRoutes {
  constructor(router, config, sessionStorageFactory, securityClient, coreSetup) {
    this.router = router;
    this.config = config;
    this.sessionStorageFactory = sessionStorageFactory;
    this.securityClient = securityClient;
    this.coreSetup = coreSetup;
  }

  setupRoutes() {
    // bootstrap an empty page so that browser app can render the login page
    // using client side routing.
    this.coreSetup.http.resources.register({
      path: _common.LOGIN_PAGE_URI,
      validate: false,
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      this.sessionStorageFactory.asScoped(request).clear();
      const clearOldVersionCookie = (0, _security_cookie.clearOldVersionCookieValue)(this.config);
      return response.renderAnonymousCoreApp({
        headers: {
          'set-cookie': clearOldVersionCookie
        }
      });
    }); // login using username and password

    this.router.post({
      path: _common.API_AUTH_LOGIN,
      validate: {
        body: _configSchema.schema.object({
          username: _configSchema.schema.string(),
          password: _configSchema.schema.string()
        })
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      var _this$config$multiten, _this$config$multiten2;

      const forbiddenUsernames = this.config.auth.forbidden_usernames;

      if (forbiddenUsernames.indexOf(request.body.username) > -1) {
        context.security_plugin.logger.error(`Denied login for forbidden username ${request.body.username}`);
        return response.badRequest({
          // Cannot login using forbidden user name.
          body: 'Invalid username or password'
        });
      }

      let user;

      try {
        user = await this.securityClient.authenticate(request, {
          username: request.body.username,
          password: request.body.password
        });
      } catch (error) {
        context.security_plugin.logger.error(`Failed authentication: ${error}`);
        return response.unauthorized({
          headers: {
            'www-authenticate': error.message
          }
        });
      }

      this.sessionStorageFactory.asScoped(request).clear();
      const encodedCredentials = Buffer.from(`${request.body.username}:${request.body.password}`).toString('base64');
      const sessionStorage = {
        username: user.username,
        credentials: {
          authHeaderValue: `Basic ${encodedCredentials}`
        },
        authType: 'basicauth',
        isAnonymousAuth: false,
        expiryTime: Date.now() + this.config.session.ttl
      };

      if ((_this$config$multiten = this.config.multitenancy) !== null && _this$config$multiten !== void 0 && _this$config$multiten.enabled) {
        const selectTenant = (0, _tenant_resolver.resolveTenant)(request, user.username, user.roles, user.tenants, this.config, sessionStorage);
        sessionStorage.tenant = selectTenant;
      }

      this.sessionStorageFactory.asScoped(request).set(sessionStorage);
      return response.ok({
        body: {
          username: user.username,
          tenants: user.tenants,
          roles: user.roles,
          backendroles: user.backendRoles,
          selectedTenants: (_this$config$multiten2 = this.config.multitenancy) !== null && _this$config$multiten2 !== void 0 && _this$config$multiten2.enabled ? sessionStorage.tenant : undefined
        }
      });
    }); // logout

    this.router.post({
      path: _common.API_AUTH_LOGOUT,
      validate: false,
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      this.sessionStorageFactory.asScoped(request).clear();
      return response.ok({
        body: {}
      });
    }); // anonymous auth

    this.router.get({
      path: _common.ANONYMOUS_AUTH_LOGIN,
      validate: false,
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      if (this.config.auth.anonymous_auth_enabled) {
        var _this$config$multiten3;

        let user; // If the request contains no redirect path, simply redirect to basepath.

        let redirectUrl = this.coreSetup.http.basePath.serverBasePath ? this.coreSetup.http.basePath.serverBasePath : '/';
        const requestQuery = request.url.searchParams;
        const nextUrl = requestQuery === null || requestQuery === void 0 ? void 0 : requestQuery.get('nextUrl');

        if (nextUrl) {
          redirectUrl = nextUrl;
        }

        context.security_plugin.logger.info('The Redirect Path is ' + redirectUrl);

        try {
          user = await this.securityClient.authenticateWithHeaders(request, {});
        } catch (error) {
          context.security_plugin.logger.error(`Failed authentication: ${error}. Redirecting to Login Page`);
          return response.redirected({
            headers: {
              location: `${this.coreSetup.http.basePath.serverBasePath}${_common.LOGIN_PAGE_URI}${nextUrl ? '?nextUrl=' + (0, _encode_uri_query.encodeUriQuery)(redirectUrl) : ''}`
            }
          });
        }

        this.sessionStorageFactory.asScoped(request).clear();
        const sessionStorage = {
          username: user.username,
          authType: 'basicauth',
          isAnonymousAuth: true,
          expiryTime: Date.now() + this.config.session.ttl
        };

        if ((_this$config$multiten3 = this.config.multitenancy) !== null && _this$config$multiten3 !== void 0 && _this$config$multiten3.enabled) {
          const selectTenant = (0, _tenant_resolver.resolveTenant)(request, user.username, user.roles, user.tenants, this.config, sessionStorage);
          sessionStorage.tenant = selectTenant;
        }

        this.sessionStorageFactory.asScoped(request).set(sessionStorage);
        return response.redirected({
          headers: {
            location: `${redirectUrl}`
          }
        });
      } else {
        context.security_plugin.logger.error('Anonymous auth is disabled. Redirecting to Login Page');
        return response.redirected({
          headers: {
            location: `${this.coreSetup.http.basePath.serverBasePath}${_common.LOGIN_PAGE_URI}`
          }
        });
      }
    });
  }

}

exports.BasicAuthRoutes = BasicAuthRoutes;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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