;;; xwem-minibuffer.el --- XWEM minibuffer support.

;; Copyright (C) 2003-2005 by XWEM Org.

;; Author: Zajcev Evgeny <zevlg@yandex.ru>
;; Created: Thu Dec  4 15:13:12 MSK 2003
;; Keywords: xwem, xlib
;; X-CVS: $Id: xwem-minibuffer.el,v 1.12 2005-04-04 19:54:13 lg Exp $

;; This file is part of XWEM.

;; XWEM is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; XWEM is distributed in the hope that it will be useful, but WITHOUT
;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
;; or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
;; License for more details.

;; You should have received a copy of the GNU General Public License
;; along with XEmacs; see the file COPYING.  If not, write to the Free
;; Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
;; 02111-1307, USA.

;;; Synched up with: Not in FSF

;;; Commentary:

;; XWEM minibuffer stuff.  XWEM minibuffer is Emacs frame with
;; 'minibuffer property set to 'only.  It is used for various thigs,
;; such as messages displaying, system tray, etc.

;;; Code:

(require 'xwem-load)
(require 'xwem-focus)
(require 'xwem-manage)

(eval-when-compile
  (defvar x-emacs-application-class nil))

;; Customization
(defgroup xwem-minibuffer nil
  "Group to customize XWEM minibuffer."
  :prefix "xwem-minibuffer-"
  :group 'xwem-modes)

(defcustom xwem-minibuffer-name "xwem-minibuffer"
  "*Minibuffer name to be used by XWEM."
  :type 'string
  :group 'xwem-minibuffer)

(defcustom xwem-minibuffer-bgcol "gray80"
  "*Background color to be used in `xwem-minib-frame'."
  :type 'color
  :set (lambda (sym val)
         (set sym val)
         (when (and xwem-minibuffer
                    (X-Win-p (xwem-minib-xwin xwem-minibuffer)))
           (XSetWindowBackground
            (xwem-dpy) (xwem-minib-xwin xwem-minibuffer)
            (XAllocColor (xwem-dpy) (XDefaultColormap (xwem-dpy))
                         (xwem-make-color xwem-minibuffer-bgcol)))
           (XClearArea (xwem-dpy) (xwem-minib-xwin xwem-minibuffer) 0 0
                       (X-Geom-width (xwem-minib-xgeom xwem-minibuffer))
                       (X-Geom-height (xwem-minib-xgeom xwem-minibuffer)) nil)))
  :initialize 'custom-initialize-default
  :group 'xwem-minibuffer)

(defcustom xwem-minibuffer-font (face-font-name 'default)
  "*Font to be used in `xwem-minib-frame'.  May be nil or string."
  :type '(restricted-sexp :match-alternatives ('nil try-font-name))
  :set (lambda (sym val)
         (set sym val)
         (when (and xwem-minibuffer
                    (xwem-minib-frame xwem-minibuffer))
           (set-face-font 'default xwem-minibuffer-font
                          (xwem-minib-frame xwem-minibuffer))))
  :initialize 'custom-initialize-default
  :group 'xwem-minibuffer)

(defcustom xwem-minibuffer-height 1
  "Height of `xwem-minibuffer'."
  :type 'number
  :set (lambda (sym val)
         (set sym val)
         ;; DO NOT RELY on `set-frame-height'
         (let ((frame (and xwem-minibuffer (xwem-minib-frame xwem-minibuffer)))
               (mcl (and xwem-minibuffer (xwem-minib-cl xwem-minibuffer))))
           (when (and frame mcl)
             (setq st (/ (frame-pixel-height frame) (frame-height frame))
                   nsz (* st xwem-minibuffer-height))
             (xwem-client-resize mcl nil nsz))))
  :initialize 'custom-initialize-default
  :group 'xwem-minibuffer)

(defcustom xwem-minibuffer-width 80
  "*Usable width of `xwem-minibuffer' frame."
  :type 'number
  :set (lambda (sym val)
         (set sym val)
         ;; DO NOT RELY on `set-frame-width'
         (let ((frame (and xwem-minibuffer (xwem-minib-frame xwem-minibuffer)))
               (mcl (and xwem-minibuffer (xwem-minib-cl xwem-minibuffer))))
           (when (and frame mcl)
             (setq st (/ (frame-pixel-width frame) (frame-width frame))
                   nsz (* st xwem-minibuffer-width))
             (xwem-client-resize mcl nsz nil))))
  :initialize 'custom-initialize-default
  :group 'xwem-minibuffer)

;;;###xwem-autoload
(defcustom xwem-minibuffer-border-width 2
  "Border width for `xwem-minibuffer'."
  :type 'number
  :set (lambda (sym val)
         (set sym val)
         (let ((cl (and xwem-minibuffer (xwem-minib-cl xwem-minibuffer))))
           (when (xwem-cl-p cl)
             (xwem-client-set-property
              cl 'x-border-width xwem-minibuffer-border-width))))
  :initialize 'custom-initialize-default
  :group 'xwem-minibuffer)

(defcustom xwem-minibuffer-passive-border-color "blue3"
  "Border color for `xwem-minibuffer'."
  :type 'color
  :set (lambda (sym val)
         (set sym val)
         (let ((cl (and xwem-minibuffer (xwem-minib-cl xwem-minibuffer))))
           (when (xwem-cl-p cl)
             (xwem-set-face-foreground
              'x-border-face xwem-minibuffer-passive-border-color nil cl)
             (xwem-client-set-property
              cl 'x-border-color
              (xwem-face-foreground
               'x-border-face (and (xwem-cl-selected-p cl) '(selected)) cl)))))
  :initialize 'custom-initialize-default
  :group 'xwem-minibuffer)

(defcustom xwem-minibuffer-active-border-color "blue"
  "Border color for `xwem-minibuffer' when it focused."
  :type 'color
  :set (lambda (sym val)
         (set sym val)
         (let ((cl (and xwem-minibuffer (xwem-minib-cl xwem-minibuffer))))
           (when (xwem-cl-p cl)
             (xwem-set-face-foreground
              'x-border-face xwem-minibuffer-active-border-color
              '(selected) cl)
             (xwem-client-set-property
              cl 'x-border-color
              (xwem-face-foreground
               'x-border-face (and (xwem-cl-selected-p cl)
                                   '(selected)) cl)))))
  :initialize 'custom-initialize-default
  :group 'xwem-minibuffer)

(defcustom xwem-minibuffer-hide-cursor-mode t
  "*Non-nil mean that Emacs cursor will be invisible in `xwem-minibuffer'.
When `xwem-minibuffer' loses focus Emacs cursor hides, and unhides
when it focused."
  :type 'boolean
  :group 'xwem-minibuffer)

(defcustom xwem-minibuffer-hide-show-parameter 0
  "*Animation delay parameter, when hiding/showing xwem minibuffer."
  :type 'number
  :group 'xwem-minibuffer)

(defcustom xwem-minibuffer-autohide-timeout nil
  "*Non-nil mean xwem minibuffer will be autohided, after that many seconds."
  :type '(choice (const :tag "Disabled" nil)
                 (number :tag "Seconds"))
  :set (lambda (sym value)
         (set sym value)
         (let ((mcl (xwem-minib-cl xwem-minibuffer)))
           (when mcl
             (if value
                 (xwem-minibuffer-enable-autohide-timer mcl)
               (xwem-activate mcl)
               (xwem-minibuffer-disable-autohide-timer mcl)))))
  :initialize 'custom-initialize-default
  :group 'xwem-minibuffer)

(defcustom xwem-minibiffer-always-on-top-rank 20
  "*Always on top rank or nil."
  :type '(choice (const :tag "No rank" nil)
                 (number :tag "Rank"))
  :set (lambda (sym value)
         (set sym value)
         (let ((xwin (xwem-minib-xwin xwem-minibiffer)))
           (when xwin
             (if value
                 (xwem-misc-set-xwin-always-on-top xwin value)
               (xwem-misc-unset-always-on-top xwin)))))
  :initialize 'custom-initialize-default
  :group 'xwem-minibuffer)

(defcustom xwem-minibuffer-raise-when-active t
  "*Non-nil mean xwem minibuffer is raised when activated."
  :type 'boolean
  :group 'xwem-minibuffer)
  
(defcustom xwem-minibuffer-emacs-frames-has-minibuffer t
  "*Non-nil mean Emacs frames will have their own minibuffers."
  :type 'boolean
  :group 'xwem-minibuffer)

(defcustom xwem-minibuffer-set-default-minibuffer-frame t
  "*Non-nil mean that xwem minibuffer frame will be set as `default-minibuffer-frame'.
Modify this variable only if you know what you are doing."
  :type 'boolean
  :group 'xwem-minibuffer)

;;;###xwem-autoload
(defcustom xwem-minibuffer-outer-border-width 1
  "*Outer border width for xwem minibuffer."
  :type 'number
  :group 'xwem-minibuffer)

(defcustom xwem-minibuffer-outer-border-color "black"
  "*Outer border color for xwem minibuffer."
  :type 'color
  :group 'xwem-minibuffer)

;;;###autoload
(defcustom xwem-minibuffer-focusin-hook nil
  "*Hooks called when xwem minibuffer got focus."
  :type 'hook
  :group 'xwem-minibuffer
  :group 'xwem-hooks)

;;;###autoload
(defcustom xwem-minibuffer-focusout-hook nil
  "*Hooks called when xwem minibuffer lose focus."
  :type 'hook
  :group 'xwem-minibuffer
  :group 'xwem-hooks)

(defcustom xwem-minib-resize-exact t
  "*If non-`nil', make minibuffer frame exactly the size needed to display all its contents.
Otherwise, the minibuffer frame can temporarily increase in size but
never get smaller while it is active."
  :type 'boolean
  :group 'xwem-minibuffer)

(defcustom xwem-minib-specifiers
  '((default-toolbar-visible-p . nil)

    ;; Gutters
    (default-gutter-visible-p . t)
    (top-gutter . nil)
    (top-gutter-border-width . 1)

    (menubar-visible-p . nil)
    (horizontal-scrollbar-visible-p . nil)
    ((face-font 'default) . xwem-minibuffer-font))
  "*Alist of specifiers to be set for xwem minibuffer."
  :type '(repeat (cons (sexp :tag "Specifier sexp")
                       (sexp :tag "Specifier value sexp")))
  :set (lambda (sym value)
         (set sym value)
         (when (and xwem-minibuffer
                    (frame-live-p (xwem-minib-frame xwem-minibuffer)))
           (xwem-minib-apply-specifiers (xwem-minib-frame xwem-minibuffer))))
  :group 'xwem-minibuffer)

;;; Internal variables

;; Variables
;;;###xwem-autoload
(defvar xwem-minibuffer nil
  "Default xwem minibuffer.")

(defvar xwem-minibuffer-map 
  (let ((map (make-sparse-keymap)))
    (define-key map (xwem-kbd "H-g") 'minibuffer-keyboard-quit)
    (define-key map (xwem-kbd "C-<button1>") 'xwem-client-imove)
    map)
  "Keymap used while in xwem.")


(defun xwem-minib-apply-specifiers (frame)
  "Apply `xwem-minib-specifiers' to FRAME."
  (mapc #'(lambda (spc)
            (set-specifier (eval (car spc)) (eval (cdr spc))
                           frame nil 'remove-locale))
        xwem-minib-specifiers))

(defun xwem-minib-create ()
  "Create minibuffer that will be used by xwem, or use existen."
  (let ((mframe (or (xwem-misc-find-frame xwem-minibuffer-name
                                          (minibuffer-frame-list))
                    (make-frame minibuffer-frame-plist
                                (default-x-device))))
        (dd default-directory))

    (setf (xwem-minib-frame xwem-minibuffer) mframe)
    ;; Set specifiers values for MFRAME
    (xwem-minib-apply-specifiers mframe)
    (redraw-frame mframe t)             ; KEEP THIS!

    ;; Hack over default-directory for minibuffer buffer
    (with-current-buffer
        (window-buffer (frame-root-window mframe))
      (setq default-directory dd))

    mframe))

(defmacro xwem-cl-minibuffer (cl)
  `(xwem-cl-get-sys-prop ,cl 'xwem-minibuffer))
(defsetf xwem-cl-minibuffer (cl) (minib)
  `(xwem-cl-put-sys-prop ,cl 'xwem-minibuffer ,minib))

;;; Minibuffer focus model
(define-xwem-focus-mode minibuffer (cl action &optional xev)
  "Focus mode for xwem minibuffer"
  (let ((mb (xwem-cl-minibuffer cl)))
    (when (and (xwem-minib-p mb)
               (X-Event-p xev)
               (not (member (X-Event-xfocus-mode xev)
                            (list X-NotifyVirtual X-NotifyNonlinearVirtual))))
      (cond ((eq action 'focus-in)
             ;; XWEM Minibuffer activates
             (run-hook-with-args 'xwem-minibuffer-focusin-hook mb))
            ((eq action 'focus-out)
             ;; XWEM Minibuffer deactivates
             (run-hook-with-args 'xwem-minibuffer-focusout-hook mb))))))
  
;;;; ---- XWEM Minibuffer manage type ----
(defun xwem-minibuffer-client-p (cl)
  "Return non-nil if CL is minibuffer client."
  (xwem-cl-minibuffer cl))

(defun xwem-manage-minibuffer (cl)
  "Manage method for xwem minibuffers."
  (let* ((xgeom (make-X-Geom :x 0       ; XXX
                             :y (- (X-Geom-height (xwem-rootgeom))
                                   xwem-minibuffer-outer-border-width
                                   xwem-minibuffer-outer-border-width)
                             :width (- (X-Geom-width (xwem-rootgeom))
                                       xwem-minibuffer-outer-border-width
                                       xwem-minibuffer-outer-border-width)
                             :height (X-Geom-height-with-borders
                                      (xwem-cl-xgeom cl))
                             :border-width xwem-minibuffer-outer-border-width))
         (minib (make-xwem-minib
                 :frame (xwem-misc-find-emacs-frame cl)
                 :cl cl
                 :xgeom xgeom)))

    (setf (xwem-minib-xwin minib)
          (XCreateWindow (xwem-dpy) nil
                         (X-Geom-x xgeom)
                         (X-Geom-y xgeom)
                         (X-Geom-width xgeom)
                         (X-Geom-height xgeom)
                         (X-Geom-border-width xgeom)
                         nil nil nil
                         (make-X-Attr :override-redirect t
                                      :background-pixel
                                      (XAllocNamedColor
                                       (xwem-dpy) (XDefaultColormap (xwem-dpy))
                                       xwem-minibuffer-bgcol))))
    ;; Setup window a little
    (when xwem-minibuffer-outer-border-color
      (XSetWindowBorder (xwem-dpy) (xwem-minib-xwin minib)
                        (XAllocNamedColor
                         (xwem-dpy) (XDefaultColormap (xwem-dpy))
                         xwem-minibuffer-outer-border-color)))
                                          
    ;; Save CL's minibuffer
    (setf (xwem-cl-minibuffer cl) minib)

    ;; Setup x-border-face for minibuffer
    (xwem-set-face-foreground 'x-border-face xwem-minibuffer-active-border-color
                              '(selected) cl)
    (xwem-set-face-foreground 'x-border-face xwem-minibuffer-passive-border-color
                              nil cl)
    (xwem-client-set-property cl 'x-border-width xwem-minibuffer-border-width)
    (xwem-client-set-property cl 'x-border-color
                              (xwem-face-foreground 'x-border-face
                                                    (and (xwem-cl-selected-p cl)
                                                         '(selected)) cl))

    ;; Reparent xwem minib client to parent
    ;; XXX XXX
    (setf (X-Geom-x (xwem-cl-xgeom cl)) 0)
    (setf (X-Geom-y (xwem-cl-xgeom cl)) 0)
    (XReparentWindow (xwem-dpy) (xwem-minib-cl-xwin minib)
                     (xwem-minib-xwin minib) 0 0)

    ;; Set minibuffer focus model
    (xwem-focus-mode-set cl 'minibuffer)

    ;; Install minibuffer local keymap
    (xwem-use-local-map xwem-minibuffer-map cl)

    ;; Finnally refit cl and map parent
    (xwem-refit cl)
    (XMapWindow (xwem-dpy) (xwem-minib-xwin minib))

    ;; Set always on top rank (if any)
    (when xwem-minibiffer-always-on-top-rank
      (xwem-misc-set-xwin-always-on-top
       (xwem-minib-xwin minib) xwem-minibiffer-always-on-top-rank))

    ;; Set default minibuffer, if not already set
    (unless (xwem-cl-p (xwem-minib-cl xwem-minibuffer))
      (setq xwem-minibuffer minib)
      (when xwem-minibuffer-set-default-minibuffer-frame
        (setq default-minibuffer-frame (xwem-minib-frame xwem-minibuffer))))

    ;; Now activate minibuffer
    (xwem-activate cl)
    (xwem-minib-apply-state-1 minib )))

(define-xwem-deffered xwem-minib-apply-pxgeom (minib)
  "Apply MINIB's parent geometry to life."
  (let ((pxgeom (xwem-minib-xgeom minib)))
    (XMoveResizeWindow (xwem-dpy)
                       (xwem-minib-xwin minib)
                       (X-Geom-x pxgeom)
                       (X-Geom-y pxgeom)
                       (X-Geom-width pxgeom)
                       (X-Geom-height pxgeom))))

;;
;; Some bug here:

;;  Do `(xwem-refit (xwem-minib-cl xwem-minibuffer))' - xwem minib
;;  will change its width. (ONLY when xwem-minib-resize-mode is on)

(defun xwem-refit-minibuffer (cl)
  "Refit xwem minibuffer client CL."
  (xwem-debug 'xwem-misc "Minib: Refiting .. to %S" '(xwem-cl-new-xgeom cl))

  (let ((cl-xgeom (xwem-cl-xgeom cl))
        (cl-nx (and (xwem-cl-new-xgeom cl)
                    (X-Geom-x (xwem-cl-new-xgeom cl))))
        (pxgeom (xwem-minib-xgeom (xwem-cl-minibuffer cl))))
    ;; Adjust geometry a little to fill into xwem-minib-xwin and apply
    ;; changes to life
    (xwem-cl-apply-new-xgeom cl)
    (when cl-nx
      ;; CL has new x location - handle it
      (setf (X-Geom-x pxgeom) (X-Geom-x cl-xgeom))
      (setf (X-Geom-width pxgeom)
            (- (X-Geom-width (xwem-rootgeom))
               (X-Geom-x pxgeom)
               xwem-minibuffer-outer-border-width
               xwem-minibuffer-outer-border-width))
      (xwem-minib-apply-pxgeom (xwem-cl-minibuffer cl)))
    (xwem-cl-correct-size-for-size
     cl
     (make-X-Geom :x 0 :y 0
                  :width (X-Geom-width-with-borders cl-xgeom)
                  :height (X-Geom-height-with-borders cl-xgeom)
                  :border-width (X-Geom-border-width cl-xgeom))
     'left 'top)
    (xwem-cl-apply-xgeom cl)

    ;; Check maybe parent need to be resized/moved?
    (unless (= (X-Geom-height-with-borders cl-xgeom)
               (X-Geom-height pxgeom))
      (when (eq (xwem-cl-state cl) 'active)
        (decf (X-Geom-y pxgeom)
              (- (X-Geom-height-with-borders cl-xgeom)
                 (X-Geom-height pxgeom))))
      (setf (X-Geom-height pxgeom)
            (X-Geom-height-with-borders cl-xgeom))
      (xwem-minib-apply-pxgeom (xwem-cl-minibuffer cl)))))

(defun xwem-minibuffer-autohide-timer (cl)
  (when (and (numberp xwem-minibuffer-autohide-timeout)
             (not (xwem-cl-selected-p cl)))
    (xwem-deactivate cl))
  (xwem-cl-rem-sys-prop cl 'auto-hide-timer))

(defun xwem-minibuffer-disable-autohide-timer (cl)
  (let ((tmr (xwem-cl-get-sys-prop cl 'auto-hide-timer)))
    (when tmr
      (disable-timeout tmr)
      (xwem-cl-rem-sys-prop cl 'auto-hide-timer))))
  
(defun xwem-minibuffer-enable-autohide-timer (cl)
  (when (numberp xwem-minibuffer-autohide-timeout)
    (xwem-minibuffer-disable-autohide-timer cl)
    (xwem-cl-put-sys-prop cl 'auto-hide-timer
      (add-timeout xwem-minibuffer-autohide-timeout
                   'xwem-minibuffer-autohide-timer cl))))
  
(define-xwem-deffered xwem-minib-apply-state (minib)
  "Apply xwem minibuffer MINIB's state to life."
  (let* ((pgeom (xwem-minib-xgeom minib))
         (cyo (X-Geom-height pgeom))
         (state (xwem-cl-state (xwem-minib-cl minib)))
         (param (and (numberp xwem-minibuffer-hide-show-parameter)
                     (not (zerop xwem-minibuffer-hide-show-parameter))
                     xwem-minibuffer-hide-show-parameter))
         (i 0))
    (when (not (eq state (xwem-minib-get-prop minib 'saved-state)))
      (xwem-debug 'xwem-misc "Minib: new state .. %S" 'state)

      (while (< i cyo)
        (setf (X-Geom-y (xwem-minib-xgeom minib))
              (funcall (if (eq state 'active) '- '+)
                       (X-Geom-y (xwem-minib-xgeom minib)) 1))
        (XMoveWindow (xwem-dpy) (xwem-minib-xwin minib)
                     (X-Geom-x (xwem-minib-xgeom minib))
                     (X-Geom-y (xwem-minib-xgeom minib)))
        (XFlush (xwem-dpy))
        (when param
          (sit-for param t))
        (setq i (1+ i)))
      (xwem-minib-put-prop minib 'saved-state state)

      (when (and xwem-minibuffer-raise-when-active
                 (eq state 'active))
        (xwem-misc-raise-xwin (xwem-minib-xwin minib))))

    (when (eq state 'active)
      (xwem-minibuffer-enable-autohide-timer (xwem-minib-cl minib)))))

(defun xwem-activate-minibuffer (cl &optional type)
  "Activate xwem minibuffer CL."
  (cond ((eq type 'activate)
         (xwem-minib-apply-state (xwem-cl-minibuffer cl)))

        ((eq type 'select)
         (when xwem-minibuffer-hide-cursor-mode
           (set-frame-property (xwem-minib-frame (xwem-cl-minibuffer cl))
                               'text-cursor-visible-p t))

         (xwem-minibuffer-disable-autohide-timer cl))))
  
(defun xwem-deactivate-minibuffer (cl &optional type)
  "Deactivate xwem minibuffer client CL."
  (cond ((eq type 'deactivate)
         (xwem-minib-apply-state (xwem-cl-minibuffer cl)))

        ((eq type 'deselect)
         (when xwem-minibuffer-hide-cursor-mode
           (set-frame-property (xwem-minib-frame (xwem-cl-minibuffer cl))
                               'text-cursor-visible-p nil))
         (when xwem-minibuffer-autohide-timeout
           (xwem-deactivate cl)))))

(defun xwem-iconify-minibuffer (cl)
  "Iconify xwem minibuffer client CL."
  (xwem-deactivate cl))

;; Events handler
(define-xwem-deffered xwem-minib-focusin-autoraise (minib)
  "Mainly for use in `xwem-minibuffer-focusout-hook'."
  (xwem-misc-raise-xwin (xwem-minib-xwin minib)))

(define-xwem-deffered xwem-minib-focusout-autolower (minib)
  "Mainly for use in `xwem-minibuffer-focusout-hook'."
  (xwem-misc-lower-xwin (xwem-minib-xwin minib)))

;;;###autoload(autoload 'xwem-minibuffer-activate "xwem-minibuffer" "" t)
(define-xwem-command xwem-minibuffer-activate ()
  "Switch to xwem minibuffer if it is in active state."
  (xwem-interactive)
  (if (eq (active-minibuffer-window)
          (frame-selected-window (xwem-minib-frame xwem-minibuffer)))
      (xwem-select-client (xwem-minib-cl xwem-minibuffer))
    (xwem-activate (xwem-minib-cl xwem-minibuffer))))

;;;###autoload
(defun xwem-minibuffer-init ()
  "Initialize xwem minibuffer."
  (xwem-message 'init "Initializing minibuffer ...")

  (setq minibuffer-frame-plist
        (xwem-misc-merge-plists
         minibuffer-frame-plist
         `(name ,xwem-minibuffer-name
                wait-for-wm nil
                height ,xwem-minibuffer-height
                text-cursor-visible-p ,(not xwem-minibuffer-hide-cursor-mode)
                width ,xwem-minibuffer-width
                minibuffer only)))

  ;; Adjust initial frame params
  (unless xwem-minibuffer-emacs-frames-has-minibuffer
    (setq initial-frame-plist
          (plist-put initial-frame-plist 'minibuffer nil)))
  (setq initial-frame-plist
        (plist-put initial-frame-plist 'wait-for-wm nil))

  ;; Adjust default frame params
  (unless xwem-minibuffer-emacs-frames-has-minibuffer
    (setq default-x-frame-plist
          (plist-put default-x-frame-plist 'minibuffer nil)))
  (setq default-x-frame-plist
        (plist-put default-x-frame-plist 'wait-for-wm nil))

  ;; Create XEmacs minibuffer only frame for xwem minibuffer
  (setq xwem-minibuffer (make-xwem-minib))
  (setf (xwem-minib-frame xwem-minibuffer)
        (make-initial-minibuffer-frame nil))
  (xwem-message 'init "Initializing minibuffer ... done"))

;;; Resize-minibuffer mode
(defvar xwem-minib-rsz-saved-height nil)

(defun xwem-minib-rsz-count-window-lines (&optional start end)
  "Return number of window lines occupied by text in region.
The number of window lines may be greater than the number of actual lines
in the buffer if any wrap on the display due to their length.

Optional arguments START and END default to point-min and point-max,
respectively."
  (or start (setq start (point-min)))
  (or end   (setq end   (point-max)))
  (if (= start end)
      0
    (save-excursion
      (save-restriction
        (widen)
        (narrow-to-region start end)
        (goto-char start)
        (vertical-motion (buffer-size))))))

(defun xwem-minib-rsz-restore ()
  "Restore xwem minibuffer size."
  (xwem-minib-rsz-resize xwem-minib-rsz-saved-height)
  (xwem-misc-lower-xwin (xwem-minib-xwin xwem-minibuffer)))

(defun xwem-minib-rsz-check ()
  "Called in `pre-command-hook'.
Check is next command is actually keyboard quit.
If yes, then restore xwem minibuffer size."
  (when (member this-command '(minibuffer-keyboard-quit keyboard-quit))
    (xwem-minib-rsz-restore)))

(defun xwem-minib-rsz-resize (&optional new-height)
  "Resize xwem minibuffer to fit either NEW-HEIGHT.
If NEW-HEIGHT is ommited, current window height considered."
  (let* ((frame (xwem-minib-frame xwem-minibuffer))
         (height (frame-height frame))
         (lines (or new-height (1+ (xwem-minib-rsz-count-window-lines))))
         w n)
    (cond ((or new-height xwem-minib-resize-exact (> lines height))
           (setq w (frame-pixel-height frame))
           (set-frame-size frame (frame-width frame) lines)
           (setq n (frame-pixel-height frame))

           (xwem-misc-raise-xwin (xwem-minib-xwin xwem-minibuffer))

           ;; Adjust xwem minibuffer cl height
           (xwem-client-resize (xwem-minib-cl xwem-minibuffer) nil
                               (+ (X-Geom-height
                                   (xwem-minib-cl-xgeom xwem-minibuffer))
                                  (- n w)))
           ))))

(defun xwem-minib-rsz-setup ()
  "Setup xwem resize minibuffer mode for xwem minibuffer."
  (when (eq (window-frame (minibuffer-window))
            (xwem-minib-frame xwem-minibuffer))
    (setq xwem-minib-rsz-saved-height
          (frame-property (selected-frame) 'height))

    (make-local-hook 'minibuffer-exit-hook)
    (add-hook 'minibuffer-exit-hook 'xwem-minib-rsz-restore nil t)
    (make-local-hook 'pre-command-hook)
    (add-hook 'pre-command-hook 'xwem-minib-rsz-check nil t)
    (make-local-hook 'post-command-hook)
    (add-hook 'post-command-hook 'xwem-minib-rsz-resize nil t)

    (unless (and (boundp 'icomplete-mode)
                 (eval 'icomplete-mode)) ; shutup compiler
      (xwem-minib-rsz-resize))
    ))

;;;###autoload(autoload 'xwem-minib-resize-mode "xwem-minibuffer" "" t)
(define-xwem-command xwem-minib-resize-mode (arg)
  "Start/stop xwem minibuffer auto-resize mode.
With prefix ARG start, without - stop."
  (xwem-interactive "P")
  (if arg
      (add-hook 'minibuffer-setup-hook 'xwem-minib-rsz-setup)
    (remove-hook 'minibuffer-setup-hook 'xwem-minib-rsz-setup)))

;;; xwem minibuffer modeline using top gutter
(defvar xwem-modeline-format
  '("--"
    (symbol-name (xwem-cl-manage-type cl))
    ": "
    "["
    (or (car (xwem-client-application cl))
        "unknown")
    "]"

    ;; Some additional info
    " "
    (if (xwem-cl-marked-p cl) "*" "-")
    (let ((reg (xwem-client-property cl 'register)))
      (if reg (list (format "%c" reg) 'bold) "-"))
    " "

    (list (xwem-client-name cl) 'modeline-buffer-id)
    "   "
    ;; Minor modes
    "("
    (mapconcat 'identity
               (delq nil (mapcar #'(lambda (mm)
                                     (and (symbol-value (car mm))
                                          (cadr mm)))
                                 xwem-minor-mode-alist)) " ")
    ")"
    "----"
    (let ((usz (xwem-cl-get-usize cl)))
      (format "%dx%d" (car usz) (cdr usz)))
    "--")
  "Modeline format.")
(xwem-make-variable-client-local 'xwem-modeline-format)

(defun xwem-modeline-regenerate ()
  "Regenerate modeline string."
  (mapconcat #'(lambda (me)
                 (let ((cl (xwem-cl-selected)))
                   (setq cl cl)         ; shutup compiler
                   (condition-case nil
                       (let ((str "") (faces nil))
                         (setq me (eval me))
                         (if (listp me)
                             (setq str (copy-sequence (car me))
                                   faces (cdr me))
                           (setq str (copy-sequence me)
                                 faces nil))
                         (xwem-str-with-faces str (append '(modeline) faces)))
                     (t "<error>"))))
             xwem-modeline-format ""))

(define-xwem-deffered xwem-modeline-redraw (&optional cl)
  "Redraw xwem modeline."
  (if (xwem-cl-alive-p cl)
      (when (xwem-cl-selected-p cl)
        (xwem-modeline-redraw))

    (let* ((str (xwem-modeline-regenerate))
           (mw (frame-width (xwem-minib-frame xwem-minibuffer))))
      (set-specifier top-gutter (substring str 0 (and (> (length str) mw) mw))
                     (xwem-minib-frame xwem-minibuffer))
      )))

;;;###autoload(autoload 'xwem-modeline-enable "xwem-minibuffer" nil t)
(define-xwem-command xwem-modeline-enable ()
  "Enable modeline."
  (xwem-interactive)

  (add-hook 'xwem-cl-change-hook 'xwem-modeline-redraw)
  (add-hook 'xwem-client-select-hook 'xwem-modeline-redraw)

  (set-specifier top-gutter-visible-p t
                 (xwem-minib-frame xwem-minibuffer))
  ;; Start showing gutter
  (xwem-modeline-redraw-1)
  ;; Fix xwem minibuffer height size
  (xwem-client-resize
   (xwem-minib-cl xwem-minibuffer) nil
   (+ (X-Geom-height-with-borders (xwem-minib-cl-xgeom xwem-minibuffer))
      (gutter-pixel-height 'top (xwem-minib-frame xwem-minibuffer))
      (specifier-instance top-gutter-border-width
                          (xwem-minib-frame xwem-minibuffer))
      (specifier-instance top-gutter-border-width
                          (xwem-minib-frame xwem-minibuffer)))))

;;;###autoload(autoload 'xwem-modeline-disable "xwem-minibuffer" nil t)
(define-xwem-command xwem-modeline-disable ()
  "Disable modeline."
  (xwem-interactive)

  (remove-hook 'xwem-cl-change-hook 'xwem-modeline-redraw)
  (remove-hook 'xwem-client-select-hook 'xwem-modeline-redraw)

  ;; Fix xwem minibuffer height size
  (xwem-client-resize
   (xwem-minib-cl xwem-minibuffer) nil
   (- (X-Geom-height-with-borders (xwem-minib-cl-xgeom xwem-minibuffer))
      (gutter-pixel-height 'top (xwem-minib-frame xwem-minibuffer))
      (specifier-instance top-gutter-border-width
                          (xwem-minib-frame xwem-minibuffer))
      (specifier-instance top-gutter-border-width
                          (xwem-minib-frame xwem-minibuffer))))
  ;; Stop showing gutter
  (set-specifier top-gutter nil
                 (xwem-minib-frame xwem-minibuffer)))


(provide 'xwem-minibuffer)

;;;; On-load actions:
;; Define application
(add-to-list 'xwem-applications-alist
             `("xemacs-xwem-minibuffer"
               (and (class-name ,(concat "^" x-emacs-application-class "$"))
                    (class-inst ,(concat "^" xwem-minibuffer-name "$")))))

;; Add manage type
(define-xwem-manage-model minibuffer
  "Managing model for xwem minibuffer."
  :cl-properties '(dummy-client-p t
                                  skip-deselect t
                                  override-skip-deselect t)
  :match-spec '(application "xemacs-xwem-minibuffer")

  :manage-method 'xwem-manage-minibuffer
  :activate-method 'xwem-activate-minibuffer
  :deactivate-method 'xwem-deactivate-minibuffer
  :refit-method 'xwem-refit-minibuffer
  :iconify-method 'xwem-iconify-minibuffer)

;; - Before init hook
(if xwem-started
    (xwem-minib-create)
  (add-hook 'xwem-config-read-hook 'xwem-minibuffer-init)
  (add-hook 'xwem-before-init-hook 'xwem-minib-create))

;;; xwem-minibuffer.el ends here
