"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AppAnalyticsAdaptor = void 0;

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class AppAnalyticsAdaptor {
  constructor() {
    _defineProperty(this, "fetchApps", async client => {
      try {
        const response = await client.callAsCurrentUser('observability.getObject', {
          objectType: 'application'
        });
        return response.observabilityObjectList.map(object => {
          return {
            id: object.objectId,
            dateCreated: object.createdTimeMs,
            dateModified: object.lastUpdatedTimeMs,
            name: object.application.name,
            description: object.application.description,
            baseQuery: object.application.baseQuery,
            servicesEntities: object.application.servicesEntities.map(rec => decodeURI(rec)),
            traceGroups: object.application.traceGroups.map(rec => decodeURI(rec)),
            panelId: object.application.panelId,
            availability: {
              name: '',
              color: '',
              availabilityVisId: object.application.availabilityVisId || ''
            }
          };
        });
      } catch (err) {
        throw new Error('Fetch All Applications Error: ' + err);
      }
    });

    _defineProperty(this, "fetchAppById", async (client, appId) => {
      try {
        const response = await client.callAsCurrentUser('observability.getObjectById', {
          objectId: appId
        });
        const app = response.observabilityObjectList[0];
        return {
          id: appId,
          dateCreated: app.createdTimeMs,
          dateModified: app.lastUpdatedTimeMs,
          name: app.application.name,
          description: app.application.description,
          baseQuery: app.application.baseQuery,
          servicesEntities: app.application.servicesEntities.map(rec => decodeURI(rec)),
          traceGroups: app.application.traceGroups.map(rec => decodeURI(rec)),
          panelId: app.application.panelId,
          availability: {
            name: '',
            color: '',
            availabilityVisId: app.application.availabilityVisId || ''
          }
        };
      } catch (err) {
        throw new Error('Fetch Application By Id Error: ' + err);
      }
    });

    _defineProperty(this, "createNewApp", async (client, appBody) => {
      try {
        const response = await client.callAsCurrentUser('observability.createObject', {
          body: {
            application: appBody
          }
        });
        return response.objectId;
      } catch (err) {
        throw new Error('Create New Application Error: ' + err);
      }
    });

    _defineProperty(this, "renameApp", async (client, appId, name) => {
      const updateApplicationBody = {
        name
      };

      try {
        const response = await client.callAsCurrentUser('observability.updateObjectById', {
          objectId: appId,
          body: {
            application: updateApplicationBody
          }
        });
        return response.objectId;
      } catch (err) {
        throw new Error('Rename Application Error: ' + err);
      }
    });

    _defineProperty(this, "updateApp", async (client, appId, updateAppBody) => {
      try {
        const response = await client.callAsCurrentUser('observability.updateObjectById', {
          objectId: appId,
          body: {
            application: updateAppBody
          }
        });
        return response.objectId;
      } catch (err) {
        throw new Error('Update Panel Error: ' + err);
      }
    });

    _defineProperty(this, "deleteApp", async (client, appList) => {
      try {
        const response = await client.callAsCurrentUser('observability.deleteObjectByIdList', {
          objectIdList: appList
        });
        return {
          status: 'OK',
          message: response
        };
      } catch (err) {
        throw new Error('Delete Application Error: ' + err);
      }
    });
  }

}

exports.AppAnalyticsAdaptor = AppAnalyticsAdaptor;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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