"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerAppAnalyticsRouter = registerAppAnalyticsRouter;

var _configSchema = require("@osd/config-schema");

var _application_analytics = require("../../../common/constants/application_analytics");

var _app_analytics_adaptor = require("../../../server/adaptors/application_analytics/app_analytics_adaptor");

var _metrics_helper = require("../../common/metrics/metrics_helper");

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

/* eslint-disable no-console */
function registerAppAnalyticsRouter(router) {
  const appAnalyticsBackend = new _app_analytics_adaptor.AppAnalyticsAdaptor(); // Fetches all existing applications

  router.get({
    path: `${_application_analytics.APP_ANALYTICS_API_PREFIX}/`,
    validate: {}
  }, async (context, request, response) => {
    (0, _metrics_helper.addRequestToMetric)('application_analytics', 'get', 'count');
    const opensearchClient = context.observability_plugin.observabilityClient.asScoped(request);
    let applicationsData = [];

    try {
      applicationsData = await appAnalyticsBackend.fetchApps(opensearchClient);
      return response.ok({
        body: {
          data: applicationsData
        }
      });
    } catch (err) {
      (0, _metrics_helper.addRequestToMetric)('application_analytics', 'get', err);
      console.error('Error occurred while fetching applications', err);
      return response.custom({
        statusCode: err.statusCode || 500,
        body: err.message
      });
    }
  }); // Fetch application by id

  router.get({
    path: `${_application_analytics.APP_ANALYTICS_API_PREFIX}/{appId}`,
    validate: {
      params: _configSchema.schema.object({
        appId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    (0, _metrics_helper.addRequestToMetric)('application_analytics', 'get', 'count');
    const opensearchClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const appObject = await appAnalyticsBackend.fetchAppById(opensearchClient, request.params.appId);
      return response.ok({
        body: appObject
      });
    } catch (err) {
      (0, _metrics_helper.addRequestToMetric)('application_analytics', 'get', err);
      console.error('Error occurred while fetching application', err);
      return response.custom({
        statusCode: err.statusCode || 500,
        body: err.message
      });
    }
  }); // Create a new application

  router.post({
    path: `${_application_analytics.APP_ANALYTICS_API_PREFIX}/`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        description: _configSchema.schema.maybe(_configSchema.schema.string()),
        baseQuery: _configSchema.schema.string(),
        servicesEntities: _configSchema.schema.arrayOf(_configSchema.schema.string()),
        traceGroups: _configSchema.schema.arrayOf(_configSchema.schema.string()),
        availabilityVisId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    (0, _metrics_helper.addRequestToMetric)('application_analytics', 'create', 'count');
    const opensearchClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const newAppId = await appAnalyticsBackend.createNewApp(opensearchClient, request.body);
      return response.ok({
        body: {
          message: 'Application Created',
          newAppId
        }
      });
    } catch (err) {
      (0, _metrics_helper.addRequestToMetric)('application_analytics', 'get', err);
      console.error('Error occurred while creating a new application', err);
      return response.custom({
        statusCode: err.statusCode || 500,
        body: err.message
      });
    }
  }); // Renames an existing application

  router.put({
    path: `${_application_analytics.APP_ANALYTICS_API_PREFIX}/rename`,
    validate: {
      body: _configSchema.schema.object({
        appId: _configSchema.schema.string(),
        name: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    (0, _metrics_helper.addRequestToMetric)('application_analytics', 'update', 'count');
    const opensearchClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      await appAnalyticsBackend.renameApp(opensearchClient, request.body.appId, request.body.name);
      return response.ok({
        body: {
          message: 'Application Renamed'
        }
      });
    } catch (err) {
      (0, _metrics_helper.addRequestToMetric)('application_analytics', 'update', err);
      console.error('Error occurred while renaming an existing application', err);
      return response.custom({
        statusCode: err.statusCode || 500,
        body: err.message
      });
    }
  }); // Updates an existing application

  router.put({
    path: `${_application_analytics.APP_ANALYTICS_API_PREFIX}/`,
    validate: {
      body: _configSchema.schema.object({
        appId: _configSchema.schema.string(),
        updateBody: _configSchema.schema.object({
          name: _configSchema.schema.maybe(_configSchema.schema.string()),
          description: _configSchema.schema.maybe(_configSchema.schema.string()),
          servicesEntities: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
          traceGroups: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
          panelId: _configSchema.schema.maybe(_configSchema.schema.string()),
          availabilityVisId: _configSchema.schema.maybe(_configSchema.schema.string())
        })
      })
    }
  }, async (context, request, response) => {
    (0, _metrics_helper.addRequestToMetric)('application_analytics', 'update', 'count');
    const opensearchClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const updatedAppId = await appAnalyticsBackend.updateApp(opensearchClient, request.body.appId, request.body.updateBody);
      return response.ok({
        body: {
          message: 'Application Updated',
          updatedAppId
        }
      });
    } catch (err) {
      (0, _metrics_helper.addRequestToMetric)('application_analytics', 'update', err);
      console.error('Error occurred while updating an existing application', err);
      return response.custom({
        statusCode: err.statusCode || 500,
        body: err.message
      });
    }
  }); // Delete applications

  router.delete({
    path: `${_application_analytics.APP_ANALYTICS_API_PREFIX}/{appList}`,
    validate: {
      params: _configSchema.schema.object({
        appList: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    (0, _metrics_helper.addRequestToMetric)('application_analytics', 'delete', 'count');
    const opensearchClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const delResponse = await appAnalyticsBackend.deleteApp(opensearchClient, request.params.appList);
      return response.ok({
        body: delResponse
      });
    } catch (err) {
      (0, _metrics_helper.addRequestToMetric)('application_analytics', 'delete', err);
      return response.custom({
        statusCode: err.statusCode || 500,
        body: err.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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