"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.deserializeFactory = void 0;
/*
 * This public interface for parsing the AWS Encryption SDK Message Header Format
 * is provided for the use of the Encryption SDK for JavaScript only.  It can be used
 * as a reference but is not intended to be use by any packages other than the
 * Encryption SDK for JavaScript.
 *
 * See: https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/message-format.html#header-structure
 */
const material_management_1 = require("@aws-crypto/material-management");
const decode_encryption_context_1 = require("./decode_encryption_context");
const deserialize_encrypted_data_keys_1 = require("./deserialize_encrypted_data_keys");
const deserialize_header_v1_1 = require("./deserialize_header_v1");
const deserialize_header_v2_1 = require("./deserialize_header_v2");
// To deal with Browser and Node.js I inject a function to handle utf8 encoding.
function deserializeFactory(toUtf8, SdkSuite) {
    const decodeEncryptionContext = (0, decode_encryption_context_1.decodeEncryptionContextFactory)(toUtf8);
    const deserializeEncryptedDataKeys = (0, deserialize_encrypted_data_keys_1.deserializeEncryptedDataKeysFactory)(toUtf8);
    const deserializeHeaderV1 = (0, deserialize_header_v1_1.deserializeHeaderV1Factory)({
        decodeEncryptionContext,
        deserializeEncryptedDataKeys,
        SdkSuite,
    });
    const deserializeHeaderV2 = (0, deserialize_header_v2_1.deserializeHeaderV2Factory)({
        decodeEncryptionContext,
        deserializeEncryptedDataKeys,
        SdkSuite,
    });
    /* The first byte holds the message format version.
     * So this maps a version to a deserializer.
     */
    const deserializeMap = new Map([
        /* I have no idea why someone
         * is going to call me with an empty buffer.
         * But since that is clearly not enough data
         * the right thing seems to be to ask for more data.
         * An unknown version can't be invalid.
         */
        [undefined, (_) => false],
        [1, deserializeHeaderV1],
        [2, deserializeHeaderV2],
    ]);
    return {
        deserializeMessageHeader,
        deserializeEncryptedDataKeys,
        decodeEncryptionContext,
    };
    function deserializeMessageHeader(messageBuffer, deserializeOptions = { maxEncryptedDataKeys: false }) {
        const messageFormatVersion = messageBuffer[0];
        const deserializer = deserializeMap.get(messageFormatVersion);
        /* Precondition: A valid deserializer must exist. */
        (0, material_management_1.needs)(deserializer, 'Not a supported message format version.');
        return deserializer(messageBuffer, deserializeOptions);
    }
}
exports.deserializeFactory = deserializeFactory;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGVzZXJpYWxpemVfZmFjdG9yeS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9kZXNlcmlhbGl6ZV9mYWN0b3J5LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQSxvRUFBb0U7QUFDcEUsc0NBQXNDOzs7QUFFdEM7Ozs7Ozs7R0FPRztBQUVILHlFQUF1RTtBQU12RSwyRUFBNEU7QUFDNUUsdUZBQXVGO0FBQ3ZGLG1FQUFvRTtBQUNwRSxtRUFBb0U7QUFFcEUsZ0ZBQWdGO0FBQ2hGLFNBQWdCLGtCQUFrQixDQUNoQyxNQUFxQyxFQUNyQyxRQUEwQztJQUUxQyxNQUFNLHVCQUF1QixHQUFHLElBQUEsMERBQThCLEVBQUMsTUFBTSxDQUFDLENBQUE7SUFDdEUsTUFBTSw0QkFBNEIsR0FDaEMsSUFBQSxxRUFBbUMsRUFBQyxNQUFNLENBQUMsQ0FBQTtJQUU3QyxNQUFNLG1CQUFtQixHQUFHLElBQUEsa0RBQTBCLEVBQUM7UUFDckQsdUJBQXVCO1FBQ3ZCLDRCQUE0QjtRQUM1QixRQUFRO0tBQ1QsQ0FBQyxDQUFBO0lBRUYsTUFBTSxtQkFBbUIsR0FBRyxJQUFBLGtEQUEwQixFQUFDO1FBQ3JELHVCQUF1QjtRQUN2Qiw0QkFBNEI7UUFDNUIsUUFBUTtLQUNULENBQUMsQ0FBQTtJQUVGOztPQUVHO0lBQ0gsTUFBTSxjQUFjLEdBQUcsSUFBSSxHQUFHLENBQUM7UUFDN0I7Ozs7O1dBS0c7UUFDSCxDQUFDLFNBQVMsRUFBRSxDQUFDLENBQWEsRUFBc0IsRUFBRSxDQUFDLEtBQUssQ0FBQztRQUN6RCxDQUFDLENBQUMsRUFBRSxtQkFBbUIsQ0FBQztRQUN4QixDQUFDLENBQUMsRUFBRSxtQkFBbUIsQ0FBQztLQUN6QixDQUFDLENBQUE7SUFFRixPQUFPO1FBQ0wsd0JBQXdCO1FBQ3hCLDRCQUE0QjtRQUM1Qix1QkFBdUI7S0FDeEIsQ0FBQTtJQUVELFNBQVMsd0JBQXdCLENBQy9CLGFBQXlCLEVBQ3pCLHFCQUF5QyxFQUFFLG9CQUFvQixFQUFFLEtBQUssRUFBRTtRQUV4RSxNQUFNLG9CQUFvQixHQUFHLGFBQWEsQ0FBQyxDQUFDLENBQUMsQ0FBQTtRQUM3QyxNQUFNLFlBQVksR0FBRyxjQUFjLENBQUMsR0FBRyxDQUFDLG9CQUFvQixDQUFDLENBQUE7UUFDN0Qsb0RBQW9EO1FBQ3BELElBQUEsMkJBQUssRUFBQyxZQUFZLEVBQUUseUNBQXlDLENBQUMsQ0FBQTtRQUU5RCxPQUFPLFlBQVksQ0FBQyxhQUFhLEVBQUUsa0JBQWtCLENBQUMsQ0FBQTtJQUN4RCxDQUFDO0FBQ0gsQ0FBQztBQXBERCxnREFvREMifQ==