(ns plumbing.map
  "Common operations on maps (both Clojure immutable and mutable Java stuff)"
  (:refer-clojure :exclude [flatten])
  (:require
   [plumbing.core :as plumbing :include-macros true]
   [plumbing.fnk.schema :as schema :include-macros true]
          [clojure.set :as set]))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Clojure immutable maps

(defn safe-select-keys
  "Like select-keys, but asserts that all keys are present."
  [m ks]
  (doseq [k ks]
    (assert (contains? m k)))
  (select-keys m ks))

(defn merge-disjoint
  "Like merge, but throws with any key overlap between maps"
  ([] {})
  ([m] m)
  ([m1 m2]
     (doseq [k (keys m1)]
       (schema/assert-iae (not (contains? m2 k)) "Duplicate key %s" k))
     (into (or m2 {}) m1))
  ([m1 m2 & maps]
     (reduce merge-disjoint m1 (cons m2 maps))))

(defn merge-with-key
  "Like merge-with, but the merging function takes the key being merged
   as the first argument"
  [f & maps]
  (when (some identity maps)
    (let [merge-entry (fn [m e]
                        (let [k (key e) v (val e)]
                          (if (contains? m k)
                            (assoc m k (f k (get m k) v))
                            (assoc m k v))))
          merge2 (fn [m1 m2]
                   (reduce merge-entry (or m1 {}) (seq m2)))]
      (reduce merge2 maps))))

(defn flatten
  "Transform a nested map into a seq of [keyseq leaf-val] pairs"
  [m]
  (when m
    ((fn flatten-helper [keyseq m]
       (when m
         (if (map? m)
           (mapcat (fn [[k v]] (flatten-helper (conj keyseq k) v)) m)
           [[keyseq m]])))
     [] m)))

(defn unflatten
  "Transform a seq of [keyseq leaf-val] pairs into a nested map.
   If one keyseq is a prefix of another, you're on your own."
  [s]
  (reduce (fn [m [ks v]] (if (seq ks) (assoc-in m ks v) v)) {} s))


;; TODO: make sure we're safe with false here -- pretty sure we're not.  Same for nil.
(defn map-leaves-and-path
  "Takes a nested map and returns a nested map with the same shape, where each
   (non-map) leaf v is transformed to (f key-seq v).
   key-seq is the sequence of keys to reach this leaf, starting at the root."
  ([f m] (when m (map-leaves-and-path f [] m)))
  ([f ks m]
     (if-not (map? m)
       (f ks m)
       (plumbing/for-map [[k v] m]
         k
         (map-leaves-and-path f (conj ks k) v)))))

(defn keep-leaves-and-path
  "Takes a nested map and returns a nested map with the same shape, where each
   (non-map) leaf v is transformed to (f key-seq v), or removed if it returns nil.
   key-seq is the sequence of keys to reach this leaf, starting at the root.
   Empty maps produced by this pruning are themselves pruned from the output."
  ([f m] (keep-leaves-and-path f [] m))
  ([f ks m]
     (if-not (map? m)
       (f ks m)
       (plumbing/for-map [[k ov] m
                          :let [nv (keep-leaves-and-path f (conj ks k) ov)]
                          :when (not (or (nil? nv) (and (map? nv) (empty? nv))))]
         k nv))))

(defn map-leaves
  "Takes a nested map and returns a nested map with the same shape, where each
   (non-map) leaf v is transformed to (f v)."
  ([f m] (map-leaves-and-path (fn [_ l] (f l)) m)))

(defn keep-leaves
  "Takes a nested map and returns a nested map with the same shape, where each
   (non-map) leaf v is transformed to (f v), or removed if it returns nil.
   Empty maps produced by this pruning are themselves pruned from the output."
  ([f m] (keep-leaves-and-path (fn [_ l] (f l)) m)))

                     
                                                                           
                

                          
                        

                         
          
                                 
                                                                               
                                               

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Java mutable Maps

     
   
                   
                                                               
                           
                                 
                                  
                                             

                
                                                                              
                                                                         
                                                     

                                                        
                      
                                      
                       
                                   
                             
                    

                
                                                                     
                                  
                                    
                                
                    

                   
                                                                         
                                                                              
                                   
                          
                                                           
                                 


                                  
                                                                     
                 
                                 
                                
                         
         

                                       
                                                                        
                   
                                 
                                   
                             
          

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Ops on graphs represented as maps.

     
                      
                                                                             
                                                                                
                                                                  
                                                                                  
                                                                               
                                 
                                                        
                               
                             
                                  
                       
                                                                         
                                     
                             
                   
                                 
                                
                          
                                        
                        
                         
                                      
                         
                               
                                 
                                                                                                          
                  

      
(defn topological-sort
  [child-map & [include-leaves?]]
  (let [child-map (if include-leaves?
                    (reduce
                     (fn [e k]
                       (let [leaves (remove #(contains? e %)  (get e k))]
                         (reduce (fn [e leaf] (assoc e leaf nil)) e leaves)))
                     child-map
                     (keys child-map))
                    child-map)

        dfs (fn [s g r pred]
              (if (pred s)
                (let [n          (peek s)
                      neighbors  (vec (drop-while #(not (contains? g %)) (get g n)))]
                  (if (seq neighbors)
                    (recur (conj s (peek neighbors))
                           (assoc g n (pop neighbors))
                           r
                           pred)
                    (recur (pop s)
                           (dissoc g n)
                           (conj r n)
                           pred)))
                [s g r pred]))

        first-duplicate (fn [coll]
                          (loop [[f & r :as coll] coll
                                 acc  #{}]
                            (if (seq coll)
                              (if (contains? acc f)
                                [f] ;; We package the result to avoid confusion
                                (recur r (conj acc f)))           ;; around nil
                              nil)))

        sorted (loop [[stack g r pred :as tuple] [[] child-map () seq]]
                 (let [ks (keys g)]
                   (if (seq ks)
                     (recur (apply dfs (assoc tuple 0 [(first ks)])))
                     r)))]

    (if-let [cycle-start (first-duplicate sorted)]
      (let [[cycle _ _ _] (dfs cycle-start child-map () #(and (seq %)
                                                              (or (= (count %) 1)
                                                                  (not= (first cycle-start) (peek %)))))]
        (throw (ex-info (str "Graph contains a cycle.") {:cycle cycle})))
      sorted)))

;;;;;;;;;;;; This file autogenerated from src/plumbing/map.cljx
