"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
exports.registerOpenSearchRoutes = registerOpenSearchRoutes;
var _lodash = require("lodash");
var _adHelpers = require("./utils/adHelpers");
var _helpers = require("../utils/helpers");
var _opensearchHelpers = require("./utils/opensearchHelpers");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */
function registerOpenSearchRoutes(apiRouter, opensearchService) {
  apiRouter.get('/_indices', opensearchService.getIndices);
  apiRouter.get('/_indices/{dataSourceId}', opensearchService.getIndices);
  apiRouter.get('/_aliases', opensearchService.getAliases);
  apiRouter.get('/_aliases/{dataSourceId}', opensearchService.getAliases);
  apiRouter.get('/_mappings', opensearchService.getMapping);
  apiRouter.get('/_mappings/{dataSourceId}', opensearchService.getMapping);
  apiRouter.post('/_search', opensearchService.executeSearch);
  apiRouter.put('/create_index', opensearchService.createIndex);
  apiRouter.put('/create_index/{dataSourceId}', opensearchService.createIndex);
  apiRouter.post('/bulk', opensearchService.bulk);
  apiRouter.post('/bulk/{dataSourceId}', opensearchService.bulk);
  apiRouter.post('/delete_index', opensearchService.deleteIndex);
  apiRouter.get('/_remote/info', opensearchService.getClustersInfo);
  apiRouter.get('/_remote/info/', opensearchService.getClustersInfo);
  apiRouter.get('/_remote/info/{dataSourceId}', opensearchService.getClustersInfo);
  apiRouter.get('/_indices_and_aliases', opensearchService.getIndicesAndAliases);
  apiRouter.get('/_indices_and_aliases/{dataSourceId}', opensearchService.getIndicesAndAliases);
}
class OpenSearchService {
  constructor(client, dataSourceEnabled) {
    _defineProperty(this, "client", void 0);
    _defineProperty(this, "dataSourceEnabled", void 0);
    _defineProperty(this, "executeSearch", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          index,
          query,
          size = 0,
          sort = undefined,
          collapse = undefined,
          aggs = undefined,
          rawQuery = undefined
        } = request.body;
        const requestBody = rawQuery ? rawQuery : {
          query: query,
          ...(sort !== undefined && {
            sort: sort
          }),
          ...(collapse !== undefined && {
            collapse: collapse
          }),
          ...(aggs !== undefined && {
            aggs: aggs
          })
        };
        const params = {
          index,
          size,
          body: requestBody
        };
        const results = await this.client.asScoped(request).callAsCurrentUser('search', params);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: results
          }
        });
      } catch (err) {
        console.error('Anomaly detector - Unable to execute search', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getIndices", async (context, request, opensearchDashboardsResponse) => {
      const {
        index,
        clusters
      } = request.query;
      const {
        dataSourceId = ''
      } = request.params;
      try {
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        let indices = [];
        let resolve_resp;
        let response = await callWithRequest('cat.indices', {
          index,
          format: 'json',
          h: 'health,index'
        });
        response = response.map(item => ({
          ...item,
          localCluster: true
        }));

        // only call cat indices
        if (clusters != '') {
          if (index == '') {
            resolve_resp = await callWithRequest('transport.request', {
              method: 'GET',
              path: '/_resolve/index/' + clusters + ':*'
            });
          } else {
            resolve_resp = await callWithRequest('transport.request', {
              method: 'GET',
              path: '/_resolve/index/' + clusters + ':' + index
            });
          }
          indices = resolve_resp.indices.map(item => ({
            index: item.name,
            format: 'json',
            health: 'undefined',
            localCluster: false
          }));
          response = response.concat(indices);
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              indices: response
            }
          }
        });
      } catch (err) {
        // In case no matching indices is found it throws an error.
        if (err.statusCode === 404 && (0, _lodash.get)(err, 'body.error.type', '') === 'index_not_found_exception') {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                indices: []
              }
            }
          });
        }
        console.log('Anomaly detector - Unable to get indices', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getAliases", async (context, request, opensearchDashboardsResponse) => {
      const {
        alias
      } = request.query;
      const {
        dataSourceId = ''
      } = request.params;
      try {
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('cat.aliases', {
          alias,
          format: 'json',
          h: 'alias,index'
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              aliases: response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get aliases', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "createIndex", async (context, request, opensearchDashboardsResponse) => {
      const {
        dataSourceId = ''
      } = request.params;

      //@ts-ignore
      const index = request.body.index;
      //@ts-ignore
      const body = request.body.body;
      const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
      try {
        await callWithRequest('indices.create', {
          index: index,
          body: body
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to create index', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
      try {
        const response = await callWithRequest('cat.indices', {
          index,
          format: 'json',
          h: 'health,index'
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              indices: response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get indices', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "bulk", async (context, request, opensearchDashboardsResponse) => {
      const {
        dataSourceId = ''
      } = request.params;
      const body = request.body;
      try {
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('bulk', {
          body: body
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to perform bulk action', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "deleteIndex", async (context, request, opensearchDashboardsResponse) => {
      const index = request.query;
      try {
        await callWithRequest('indices.delete', {
          index: index
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to perform delete index action', err);
        // Ignore the error if it's an index_not_found_exception
        if (!(0, _adHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: false,
              error: (0, _adHelpers.getErrorMessage)(err)
            }
          });
        }
      }
      try {
        const response = await this.client.asScoped(request).callAsCurrentUser('cat.indices', {
          index,
          format: 'json',
          h: 'health,index'
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              indices: response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get indices', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getMapping", async (context, request, opensearchDashboardsResponse) => {
      let {
        indices
      } = request.query;
      // If indices is not an array, convert it to an array, server framework auto converts single item in string array to a string
      if (!Array.isArray(indices)) {
        indices = [indices];
      }
      const {
        dataSourceId = ''
      } = request.params;
      try {
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        let mappings = {};
        let remoteMappings = {};
        let localIndices = indices.filter(index => !index.includes(':'));
        let remoteIndices = indices.filter(index => index.includes(':'));
        if (localIndices.length > 0) {
          mappings = await callWithRequest('indices.getMapping', {
            index: localIndices
          });
        }

        // make call to fields_caps
        if (remoteIndices.length) {
          const fieldCapsResponse = await callWithRequest('transport.request', {
            method: 'GET',
            path: remoteIndices.toString() + '/_field_caps?fields=*&include_unmapped'
          });
          remoteMappings = (0, _opensearchHelpers.convertFieldCapsToMappingStructure)(fieldCapsResponse);
        }
        Object.assign(mappings, remoteMappings);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              mappings: mappings
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get mappings', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    // we use this to retrieve indices and aliases from both the local cluster and remote clusters
    // 3 different OS APIs are called here, _cat/indices, _cat/aliases and _resolve/index
    _defineProperty(this, "getIndicesAndAliases", async (context, request, opensearchDashboardsResponse) => {
      const {
        indexOrAliasQuery,
        clusters,
        queryForLocalCluster
      } = request.query;
      const {
        dataSourceId = ''
      } = request.params;
      try {
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        let indicesResponse = [];
        let aliasesResponse = [];
        if (queryForLocalCluster == 'true') {
          indicesResponse = await callWithRequest('cat.indices', {
            index: indexOrAliasQuery,
            format: 'json',
            h: 'health,index'
          });
          indicesResponse = indicesResponse.map(item => ({
            ...item,
            localCluster: true
          }));
          aliasesResponse = await callWithRequest('cat.aliases', {
            alias: indexOrAliasQuery,
            format: 'json',
            h: 'alias,index'
          });
          aliasesResponse = aliasesResponse.map(item => ({
            ...item,
            localCluster: true
          }));
        }

        // only call cat indices and cat aliases
        if (clusters != '') {
          let remoteIndices = [];
          let remoteAliases = [];
          let resolveResponse;
          const resolveIndexQuery = indexOrAliasQuery == '' ? clusters.split(',').map(cluster => `${cluster}:*`).join(',') : clusters.split(',').map(cluster => `${cluster}:${indexOrAliasQuery}`).join(',');
          resolveResponse = await callWithRequest('transport.request', {
            method: 'GET',
            path: '/_resolve/index/' + resolveIndexQuery
          });
          remoteIndices = resolveResponse.indices.map(item => ({
            index: item.name,
            format: 'json',
            health: 'undefined',
            localCluster: false
          }));
          remoteAliases = resolveResponse.aliases.map(item => ({
            alias: item.name,
            index: item.indices,
            format: 'json',
            localCluster: false
          }));
          indicesResponse = indicesResponse.concat(remoteIndices);
          aliasesResponse = aliasesResponse.concat(remoteAliases);
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              aliases: aliasesResponse,
              indices: indicesResponse
            }
          }
        });
      } catch (err) {
        // In case no matching indices is found it throws an error.
        if (err.statusCode === 404 && (0, _lodash.get)(err, 'body.error.type', '') === 'index_not_found_exception') {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                indices: [],
                aliases: []
              }
            }
          });
        }
        console.log('Anomaly detector - Unable to get indices and aliases', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getClustersInfo", async (context, request, opensearchDashboardsResponse) => {
      const {
        dataSourceId = ''
      } = request.params;
      try {
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        let clustersResponse = [];
        try {
          const remoteInfo = await callWithRequest('transport.request', {
            method: 'GET',
            path: '/_remote/info'
          });
          clustersResponse = Object.keys(remoteInfo).map(key => ({
            name: key,
            localCluster: false
          }));
        } catch (remoteErr) {
          console.warn('Failed to fetch remote cluster info, proceeding with local datasource info only.', remoteErr);
        }
        const clusterHealth = await callWithRequest('cat.health', {
          format: 'json',
          h: 'cluster'
        });
        clustersResponse.push({
          name: clusterHealth[0].cluster,
          localCluster: true
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              clusters: clustersResponse
            }
          }
        });
      } catch (err) {
        console.error('Alerting - OpensearchService - getClusterHealth:', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    this.client = client;
    this.dataSourceEnabled = dataSourceEnabled;
  }
}
exports.default = OpenSearchService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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