"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerDslRoute = registerDslRoute;
var _configSchema = require("@osd/config-schema");
var _common = require("../../common");
var _metrics = require("../metrics");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const performance = require('perf_hooks').performance;
function registerDslRoute(router, openSearchServiceSetup, dataSourceEnabled) {
  router.post({
    path: _common.ServiceEndpoints.GetSearchResults,
    validate: {
      body: _configSchema.schema.any()
    }
  }, async (context, request, response) => {
    const {
      query1,
      dataSourceId1,
      query2,
      dataSourceId2
    } = request.body;
    const actionName = query1 && query2 ? _metrics.METRIC_ACTION.COMPARISON_SEARCH : _metrics.METRIC_ACTION.SINGLE_SEARCH;
    const resBody = {};
    if (query1) {
      const {
        index,
        pipeline,
        size,
        ...rest
      } = query1;
      const params = pipeline !== '' ? {
        index,
        size,
        body: rest,
        search_pipeline: pipeline
      } : {
        index,
        size,
        body: rest
      };
      const start = performance.now();
      try {
        let resp;
        const invalidCharactersPattern = /[\s,:\"*+\/\\|?#><]/;
        if (index !== index.toLowerCase() || index.startsWith('_') || index.startsWith('-') || invalidCharactersPattern.test(index)) {
          resBody.errorMessage1 = {
            statusCode: 400,
            body: 'Invalid Index or missing'
          };
        }
        if (pipeline !== '*' && pipeline !== '_none' && pipeline !== '' && !/^[a-zA-Z0-9_\-*]+(,[a-zA-Z0-9_\-*]+)*$/.test(pipeline)) {
          resBody.errorMessage1 = {
            statusCode: 400,
            body: 'Invalid Pipepline'
          };
        }
        if (dataSourceEnabled && dataSourceId1) {
          const client = context.dataSource.opensearch.legacy.getClient(dataSourceId1);
          resp = await client.callAPI('search', params);
        } else {
          resp = await context.core.opensearch.legacy.client.callAsCurrentUser('search', params);
        }
        const end = performance.now();
        context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, actionName, 200, end - start);
        resBody.result1 = resp;
      } catch (error) {
        var _error$body, _error$body2;
        const end = performance.now();
        context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, actionName, error.statusCode, end - start);
        if (error.statusCode !== 404) console.error(error);

        // Template: Error: {{Error.type}} – {{Error.reason}}
        const errorMessage = `Error: ${(_error$body = error.body) === null || _error$body === void 0 || (_error$body = _error$body.error) === null || _error$body === void 0 ? void 0 : _error$body.type} - ${(_error$body2 = error.body) === null || _error$body2 === void 0 || (_error$body2 = _error$body2.error) === null || _error$body2 === void 0 ? void 0 : _error$body2.reason}`;
        resBody.errorMessage1 = {
          statusCode: error.statusCode || 500,
          body: errorMessage
        };
      }
    }
    if (query2) {
      const {
        index,
        pipeline,
        size,
        ...rest
      } = query2;
      const params = pipeline !== '' ? {
        index,
        size,
        body: rest,
        search_pipeline: pipeline
      } : {
        index,
        size,
        body: rest
      };
      const start = performance.now();
      try {
        let resp;
        const invalidCharactersPattern = /[\s,:\"*+\/\\|?#><]/;
        if (index !== index.toLowerCase() || index.startsWith('_') || index.startsWith('-') || invalidCharactersPattern.test(index)) {
          throw new Error("Index invalid or missing.");
        }
        if (pipeline !== '*' && pipeline !== '_none' && pipeline !== '' && !/^[a-zA-Z0-9_\-*]+(,[a-zA-Z0-9_\-*]+)*$/.test(pipeline)) {
          resBody.errorMessage1 = {
            statusCode: 400,
            body: 'Invalid Pipepline'
          };
        }
        if (dataSourceEnabled && dataSourceId2) {
          const client = context.dataSource.opensearch.legacy.getClient(dataSourceId2);
          resp = await client.callAPI('search', params);
        } else {
          resp = await context.core.opensearch.legacy.client.callAsCurrentUser('search', params);
        }
        const end = performance.now();
        context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, actionName, 200, end - start);
        resBody.result2 = resp;
      } catch (error) {
        var _error$body3, _error$body4;
        const end = performance.now();
        if (error.statusCode !== 404) console.error(error);
        context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, actionName, error.statusCode, end - start);

        // Template: Error: {{Error.type}} – {{Error.reason}}
        const errorMessage = `Error: ${(_error$body3 = error.body) === null || _error$body3 === void 0 || (_error$body3 = _error$body3.error) === null || _error$body3 === void 0 ? void 0 : _error$body3.type} - ${(_error$body4 = error.body) === null || _error$body4 === void 0 || (_error$body4 = _error$body4.error) === null || _error$body4 === void 0 ? void 0 : _error$body4.reason}`;
        resBody.errorMessage2 = {
          statusCode: error.statusCode || 500,
          body: errorMessage
        };
      }
    }
    return response.ok({
      body: resBody
    });
  });

  // Get Indices
  router.get({
    path: `${_common.ServiceEndpoints.GetIndexes}/{dataSourceId?}`,
    validate: {
      params: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        }))
      })
    }
  }, async (context, request, response) => {
    const params = {
      format: 'json'
    };
    const start = performance.now();
    try {
      const dataSourceId = request.params.dataSourceId;
      let callApi;
      if (dataSourceEnabled && dataSourceId) {
        callApi = context.dataSource.opensearch.legacy.getClient(dataSourceId).callAPI;
      } else {
        callApi = context.core.opensearch.legacy.client.callAsCurrentUser;
      }
      const resp = await callApi('cat.indices', params);
      const end = performance.now();
      context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, _metrics.METRIC_ACTION.FETCH_INDEX, 200, end - start);
      return response.ok({
        body: resp
      });
    } catch (error) {
      const end = performance.now();
      context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, _metrics.METRIC_ACTION.FETCH_INDEX, error.statusCode, end - start);
      if (error.statusCode !== 404) console.error(error);
      return response.custom({
        statusCode: error.statusCode || 400,
        body: error.message
      });
    }
  });

  // Get Pipelines
  router.get({
    path: `${_common.ServiceEndpoints.GetPipelines}/{dataSourceId?}`,
    validate: {
      params: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        }))
      })
    }
  }, async (context, request, response) => {
    const params = {
      format: 'json'
    };
    const start = performance.now();
    let resBody = {};
    let resp;
    try {
      const dataSourceId = request.params.dataSourceId;
      if (dataSourceEnabled && dataSourceId) {
        resp = (await context.dataSource.opensearch.getClient(dataSourceId)).transport;
        resp = await resp.request({
          method: 'GET',
          path: `${_common.SEARCH_API}/pipeline`
        });
      } else {
        resp = await context.core.opensearch.client.asCurrentUser.transport.request({
          method: 'GET',
          path: `${_common.SEARCH_API}/pipeline`
        });
      }
      resBody = resp.body;
      const end = performance.now();
      context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, _metrics.METRIC_ACTION.FETCH_PIPELINE, 200, end - start);
      return response.ok({
        body: resBody
      });
    } catch (error) {
      const end = performance.now();
      context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, _metrics.METRIC_ACTION.FETCH_PIPELINE, error.statusCode, end - start);
      if (error.statusCode !== 404) console.error(error);
      return response.customError({
        statusCode: 404,
        body: error
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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