"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AuthenticationType = void 0;
var _opensearch_security_client = require("../../backend/opensearch_security_client");
var _tenant_resolver = require("../../multitenancy/tenant_resolver");
var _errors = require("../../errors");
var _common = require("../../../common");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
class AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, coreSetup, logger) {
    this.config = config;
    this.sessionStorageFactory = sessionStorageFactory;
    this.router = router;
    this.esClient = esClient;
    this.coreSetup = coreSetup;
    this.logger = logger;
    _defineProperty(this, "type", void 0);
    _defineProperty(this, "securityClient", void 0);
    _defineProperty(this, "authHandler", async (request, response, toolkit) => {
      var _this$config$multiten;
      // Skip authentication for APIs that do not require it
      if (this.authNotRequired(request)) {
        return toolkit.authenticated();
      }
      const authState = {};
      const authHeaders = {};
      let cookie;
      let authInfo;

      // If the request contains authentication data (e.g. Authorization header or JWT in url parameters), use that to authenticate the request.
      if (this.requestIncludesAuthInfo(request)) {
        try {
          // Build the auth headers from the request
          const additionalAuthHeader = await this.getAdditionalAuthHeader(request);
          Object.assign(authHeaders, additionalAuthHeader);
          authInfo = await this.securityClient.authinfo(request, additionalAuthHeader);
          cookie = this.getCookie(request, authInfo);

          // Set the tenant from the cookie
          const browserCookie = await this.sessionStorageFactory.asScoped(request).get();
          if (browserCookie && (0, _tenant_resolver.isValidTenant)(browserCookie.tenant)) {
            cookie.tenant = browserCookie.tenant;
          }

          // Save the cookie
          this.sessionStorageFactory.asScoped(request).set(cookie);
        } catch (error) {
          return response.unauthorized({
            body: error.message
          });
        }
      } else {
        // If the request does not contain authentication data, check for a stored cookie.
        try {
          cookie = await this.sessionStorageFactory.asScoped(request).get();
        } catch (error) {
          this.logger.error(`Error parsing cookie: ${error.message}`);
          cookie = undefined;
        }

        // If the cookie is not valid, clear the cookie and send the request to the authentication workflow
        if (!cookie || !(await this.isValidCookie(cookie, request))) {
          // Clear the cookie
          this.sessionStorageFactory.asScoped(request).clear();

          // For assets, we can still pass it to resource handler as notHandled.
          // Marking it as authenticated may result in login pop up when auth challenge
          // is enabled.
          if (request.url.pathname && request.url.pathname.startsWith('/bundles/')) {
            return toolkit.notHandled();
          }

          // Allow optional authentication
          if (this.authOptional(request)) {
            return toolkit.authenticated();
          }

          // Send the request to the authentication workflow
          return this.handleUnauthedRequest(request, response, toolkit);
        }

        // If the cookie is still valid, update the cookie with a new expiry time.
        if (this.config.session.keepalive) {
          cookie.expiryTime = this.getKeepAliveExpiry(cookie, request);
          this.sessionStorageFactory.asScoped(request).set(cookie);
        }
        // At this point we have a valid cookie.

        // Build the auth headers from the cookie.
        const authHeadersFromCookie = this.buildAuthHeaderFromCookie(cookie, request);
        Object.assign(authHeaders, authHeadersFromCookie);
        const additionalAuthHeader = await this.getAdditionalAuthHeader(request);
        Object.assign(authHeaders, additionalAuthHeader);
      }

      // resolve tenant if necessary
      if ((_this$config$multiten = this.config.multitenancy) !== null && _this$config$multiten !== void 0 && _this$config$multiten.enabled) {
        try {
          const tenant = await this.resolveTenant(request, cookie, authHeaders, authInfo);
          // return 401 if no tenant available
          if (!(0, _tenant_resolver.isValidTenant)(tenant)) {
            return response.badRequest({
              body: 'No available tenant for current user, please reach out to your system administrator'
            });
          }
          authState.selectedTenant = tenant;

          // set tenant in header
          if (this.config.multitenancy.enabled && this.config.multitenancy.enable_aggregation_view) {
            // Store all saved objects in a single kibana index.
            Object.assign(authHeaders, {
              securitytenant: _common.GLOBAL_TENANT_SYMBOL
            });
          } else {
            Object.assign(authHeaders, {
              securitytenant: tenant
            });
          }

          // set tenant to cookie
          if (tenant !== cookie.tenant) {
            cookie.tenant = tenant;
            this.sessionStorageFactory.asScoped(request).set(cookie);
          }
        } catch (error) {
          this.logger.error(`Failed to resolve user tenant: ${error}`);
          if (error instanceof _errors.UnauthenticatedError) {
            if (request.url.pathname && request.url.pathname.startsWith('/bundles/')) {
              return toolkit.notHandled();
            }
            this.sessionStorageFactory.asScoped(request).clear();
            return this.handleUnauthedRequest(request, response, toolkit);
          }
          throw error;
        }
      }
      if (!authInfo) {
        authInfo = await this.securityClient.authinfo(request, authHeaders);
      }
      authState.authInfo = authInfo;
      return toolkit.authenticated({
        requestHeaders: authHeaders,
        state: authState
      });
    });
    this.securityClient = new _opensearch_security_client.SecurityClient(esClient);
    this.type = '';
    this.config = config;
  }
  authNotRequired(request) {
    const pathname = request.url.pathname;
    if (!pathname) {
      return false;
    }
    // allow requests to ignored routes
    if (AuthenticationType.ROUTES_TO_IGNORE.includes(pathname)) {
      return true;
    }
    // allow requests to routes that doesn't require authentication
    if (this.config.auth.unauthenticated_routes.indexOf(pathname) > -1) {
      // TODO: use opensearch-dashboards server user
      return true;
    }
    return false;
  }
  authOptional(request) {
    const pathname = request.url.pathname;
    if (!pathname) {
      return false;
    }
    return AuthenticationType.ROUTES_AUTH_OPTIONAL.includes(pathname);
  }
  async resolveTenant(request, cookie, authHeader, authInfo) {
    if (!authInfo) {
      try {
        authInfo = await this.securityClient.authinfo(request, authHeader);
      } catch (error) {
        throw new _errors.UnauthenticatedError(error);
      }
    }
    const dashboardsInfo = await this.securityClient.dashboardsinfo(request, authHeader);
    return (0, _tenant_resolver.resolveTenant)({
      request,
      username: authInfo.user_name,
      roles: authInfo.roles,
      availableTenants: authInfo.tenants,
      config: this.config,
      cookie,
      multitenancyEnabled: dashboardsInfo.multitenancy_enabled,
      privateTenantEnabled: dashboardsInfo.private_tenant_enabled,
      defaultTenant: dashboardsInfo.default_tenant
    });
  }
  getKeepAliveExpiry(cookie, request) {
    return Date.now() + this.config.session.ttl;
  }
  isPageRequest(request) {
    const path = request.url.pathname || '/';
    return path.startsWith('/app/') || path === '/' || path.startsWith('/goto/');
  }

  // abstract functions for concrete auth types to implement
}
exports.AuthenticationType = AuthenticationType;
_defineProperty(AuthenticationType, "ROUTES_TO_IGNORE", ['/app/login']);
_defineProperty(AuthenticationType, "ROUTES_AUTH_OPTIONAL", ['/api/core/capabilities']);
_defineProperty(AuthenticationType, "REST_API_CALL_HEADER", 'osd-xsrf');
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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