"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerData2SummaryRoutes = registerData2SummaryRoutes;
exports.registerSummaryAssistantRoutes = registerSummaryAssistantRoutes;
var _configSchema = require("@osd/config-schema");
var _llm = require("../../common/constants/llm");
var _get_opensearch_client_transport = require("../utils/get_opensearch_client_transport");
var _get_agent = require("./get_agent");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const SUMMARY_AGENT_CONFIG_ID = 'os_summary';
const LOG_PATTERN_SUMMARY_AGENT_CONFIG_ID = 'os_summary_with_log_pattern';
const OS_INSIGHT_AGENT_CONFIG_ID = 'os_insight';
const DATA2SUMMARY_AGENT_CONFIG_ID = 'os_data2summary';
function registerSummaryAssistantRoutes(router, assistantService) {
  router.post({
    path: _llm.SUMMARY_ASSISTANT_API.SUMMARIZE,
    validate: {
      body: _configSchema.schema.object({
        summaryType: _configSchema.schema.string(),
        insightType: _configSchema.schema.maybe(_configSchema.schema.string()),
        question: _configSchema.schema.string(),
        context: _configSchema.schema.maybe(_configSchema.schema.string()),
        index: _configSchema.schema.maybe(_configSchema.schema.string()),
        dsl: _configSchema.schema.maybe(_configSchema.schema.string()),
        topNLogPatternData: _configSchema.schema.maybe(_configSchema.schema.string())
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, router.handleLegacyErrors(async (context, req, res) => {
    const client = await (0, _get_opensearch_client_transport.getOpenSearchClientTransport)({
      context,
      dataSourceId: req.query.dataSourceId
    });
    const assistantClient = assistantService.getScopedClient(req, context);
    const agentConfigId = req.body.index && req.body.dsl && req.body.topNLogPatternData ? LOG_PATTERN_SUMMARY_AGENT_CONFIG_ID : SUMMARY_AGENT_CONFIG_ID;
    const response = await assistantClient.executeAgentByConfigName(agentConfigId, {
      context: req.body.context,
      question: req.body.question,
      index: req.body.index,
      input: req.body.dsl,
      topNLogPatternData: req.body.topNLogPatternData
    });
    let summary;
    let insightAgentIdExists = false;
    try {
      if (req.body.insightType) {
        insightAgentIdExists = !!(await detectInsightAgentId(req.body.insightType, req.body.summaryType, client));
      }
    } catch (e) {
      context.assistant_plugin.logger.debug(`Cannot find insight agent for ${req.body.insightType}`, e);
    }
    try {
      summary = response.body.inference_results[0].output[0].result;
      return res.ok({
        body: {
          summary,
          insightAgentIdExists
        }
      });
    } catch (e) {
      return res.badRequest({
        body: e
      });
    }
  }));
  router.post({
    path: _llm.SUMMARY_ASSISTANT_API.INSIGHT,
    validate: {
      body: _configSchema.schema.object({
        summaryType: _configSchema.schema.string(),
        insightType: _configSchema.schema.string(),
        summary: _configSchema.schema.string(),
        context: _configSchema.schema.string(),
        question: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, router.handleLegacyErrors(async (context, req, res) => {
    const client = await (0, _get_opensearch_client_transport.getOpenSearchClientTransport)({
      context,
      dataSourceId: req.query.dataSourceId
    });
    const assistantClient = assistantService.getScopedClient(req, context);
    const insightAgentId = await detectInsightAgentId(req.body.insightType, req.body.summaryType, client);
    const response = await assistantClient.executeAgent(insightAgentId, {
      context: req.body.context,
      summary: req.body.summary,
      question: req.body.question
    });
    try {
      return res.ok({
        body: response.body.inference_results[0].output[0].result
      });
    } catch (e) {
      return res.badRequest({
        body: e
      });
    }
  }));
}
function detectInsightAgentId(insightType, summaryType, client) {
  // We have separate agent for os_insight and user_insight. And for user_insight, we can
  // only get it by searching on name since it is not stored in agent config.
  if (insightType === 'os_insight') {
    return (0, _get_agent.getAgentIdByConfigName)(OS_INSIGHT_AGENT_CONFIG_ID, client);
  } else if (insightType === 'user_insight' && summaryType === 'alerts') {
    return (0, _get_agent.searchAgent)({
      name: 'KB_For_Alert_Insight'
    }, client);
  }
  return undefined;
}
function registerData2SummaryRoutes(router, assistantService) {
  router.post({
    path: _llm.SUMMARY_ASSISTANT_API.DATA2SUMMARY,
    validate: {
      body: _configSchema.schema.object({
        sample_data: _configSchema.schema.string(),
        sample_count: _configSchema.schema.maybe(_configSchema.schema.number()),
        total_count: _configSchema.schema.maybe(_configSchema.schema.number()),
        question: _configSchema.schema.maybe(_configSchema.schema.string()),
        ppl: _configSchema.schema.maybe(_configSchema.schema.string())
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, router.handleLegacyErrors(async (context, req, res) => {
    const assistantClient = assistantService.getScopedClient(req, context);
    try {
      const response = await assistantClient.executeAgentByConfigName(DATA2SUMMARY_AGENT_CONFIG_ID, {
        sample_data: req.body.sample_data,
        total_count: req.body.total_count,
        sample_count: req.body.sample_count,
        ppl: req.body.ppl,
        question: req.body.question
      });
      const result = response.body.inference_results[0].output[0].result;
      return res.ok({
        body: result
      });
    } catch (e) {
      return res.badRequest({
        body: e
      });
    }
  }));
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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