"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerText2VizRoutes = registerText2VizRoutes;
var _configSchema = require("@osd/config-schema");
var _llm = require("../../common/constants/llm");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const inputSchema = _configSchema.schema.string({
  maxLength: _llm.TEXT2VEGA_INPUT_SIZE_LIMIT,
  validate(value) {
    if (!value || value.trim().length === 0) {
      return "can't be empty or blank.";
    }
  }
});
function registerText2VizRoutes(router, assistantService) {
  router.post({
    path: _llm.TEXT2VIZ_API.TEXT2VEGA,
    validate: {
      body: _configSchema.schema.object({
        input_question: inputSchema,
        input_instruction: _configSchema.schema.maybe(_configSchema.schema.string({
          maxLength: _llm.TEXT2VEGA_INPUT_SIZE_LIMIT
        })),
        ppl: _configSchema.schema.string(),
        dataSchema: _configSchema.schema.string(),
        sampleData: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, router.handleLegacyErrors(async (context, req, res) => {
    const assistantClient = assistantService.getScopedClient(req, context);
    try {
      const agentConfigName = req.body.input_instruction ? _llm.TEXT2VEGA_WITH_INSTRUCTIONS_AGENT_CONFIG_ID : _llm.TEXT2VEGA_RULE_BASED_AGENT_CONFIG_ID;
      const response = await assistantClient.executeAgentByConfigName(agentConfigName, {
        input_question: req.body.input_question,
        input_instruction: req.body.input_instruction,
        ppl: req.body.ppl,
        dataSchema: req.body.dataSchema,
        sampleData: req.body.sampleData
      });
      let textContent = response.body.inference_results[0].output[0].result;

      // extra content between tag <vega-lite></vega-lite>
      const startTag = '<vega-lite>';
      const endTag = '</vega-lite>';
      const startIndex = textContent.indexOf(startTag);
      const endIndex = textContent.indexOf(endTag);
      if (startIndex !== -1 && endIndex !== -1 && startIndex < endIndex) {
        // Extract the content between the tags
        textContent = textContent.substring(startIndex + startTag.length, endIndex).trim();
      }

      // extract json object
      const jsonMatch = textContent.match(/\{.*\}/s);
      if (jsonMatch) {
        let result = JSON.parse(jsonMatch[0]);
        // sometimes llm returns {response: <schema>} instead of <schema>
        if (result.response) {
          result = JSON.parse(result.response);
        }
        // Sometimes the response contains width and height which is not needed, here delete the these fields
        delete result.width;
        delete result.height;

        // make sure $schema field always been added, sometimes, LLM 'forgot' to add this field
        result.$schema = 'https://vega.github.io/schema/vega-lite/v5.json';
        return res.ok({
          body: result
        });
      }
      return res.badRequest();
    } catch (e) {
      return res.badRequest();
    }
  }));
  router.post({
    path: _llm.TEXT2VIZ_API.TEXT2PPL,
    validate: {
      body: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        question: inputSchema
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, router.handleLegacyErrors(async (context, req, res) => {
    const assistantClient = assistantService.getScopedClient(req, context);
    try {
      const response = await assistantClient.executeAgentByConfigName(_llm.TEXT2PPL_AGENT_CONFIG_ID, {
        question: req.body.question,
        index: req.body.index
      });
      const result = JSON.parse(response.body.inference_results[0].output[0].result);
      return res.ok({
        body: result
      });
    } catch (e) {
      return res.badRequest();
    }
  }));
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJuYW1lcyI6WyJfY29uZmlnU2NoZW1hIiwicmVxdWlyZSIsIl9sbG0iLCJpbnB1dFNjaGVtYSIsInNjaGVtYSIsInN0cmluZyIsIm1heExlbmd0aCIsIlRFWFQyVkVHQV9JTlBVVF9TSVpFX0xJTUlUIiwidmFsaWRhdGUiLCJ2YWx1ZSIsInRyaW0iLCJsZW5ndGgiLCJyZWdpc3RlclRleHQyVml6Um91dGVzIiwicm91dGVyIiwiYXNzaXN0YW50U2VydmljZSIsInBvc3QiLCJwYXRoIiwiVEVYVDJWSVpfQVBJIiwiVEVYVDJWRUdBIiwiYm9keSIsIm9iamVjdCIsImlucHV0X3F1ZXN0aW9uIiwiaW5wdXRfaW5zdHJ1Y3Rpb24iLCJtYXliZSIsInBwbCIsImRhdGFTY2hlbWEiLCJzYW1wbGVEYXRhIiwicXVlcnkiLCJkYXRhU291cmNlSWQiLCJoYW5kbGVMZWdhY3lFcnJvcnMiLCJjb250ZXh0IiwicmVxIiwicmVzIiwiYXNzaXN0YW50Q2xpZW50IiwiZ2V0U2NvcGVkQ2xpZW50IiwiYWdlbnRDb25maWdOYW1lIiwiVEVYVDJWRUdBX1dJVEhfSU5TVFJVQ1RJT05TX0FHRU5UX0NPTkZJR19JRCIsIlRFWFQyVkVHQV9SVUxFX0JBU0VEX0FHRU5UX0NPTkZJR19JRCIsInJlc3BvbnNlIiwiZXhlY3V0ZUFnZW50QnlDb25maWdOYW1lIiwidGV4dENvbnRlbnQiLCJpbmZlcmVuY2VfcmVzdWx0cyIsIm91dHB1dCIsInJlc3VsdCIsInN0YXJ0VGFnIiwiZW5kVGFnIiwic3RhcnRJbmRleCIsImluZGV4T2YiLCJlbmRJbmRleCIsInN1YnN0cmluZyIsImpzb25NYXRjaCIsIm1hdGNoIiwiSlNPTiIsInBhcnNlIiwid2lkdGgiLCJoZWlnaHQiLCIkc2NoZW1hIiwib2siLCJiYWRSZXF1ZXN0IiwiZSIsIlRFWFQyUFBMIiwiaW5kZXgiLCJxdWVzdGlvbiIsIlRFWFQyUFBMX0FHRU5UX0NPTkZJR19JRCJdLCJzb3VyY2VzIjpbInRleHQydml6X3JvdXRlcy50cyJdLCJzb3VyY2VzQ29udGVudCI6WyIvKlxuICogQ29weXJpZ2h0IE9wZW5TZWFyY2ggQ29udHJpYnV0b3JzXG4gKiBTUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMFxuICovXG5cbmltcG9ydCB7IHNjaGVtYSB9IGZyb20gJ0Bvc2QvY29uZmlnLXNjaGVtYSc7XG5pbXBvcnQgeyBJUm91dGVyIH0gZnJvbSAnLi4vLi4vLi4vLi4vc3JjL2NvcmUvc2VydmVyJztcbmltcG9ydCB7XG4gIFRFWFQyUFBMX0FHRU5UX0NPTkZJR19JRCxcbiAgVEVYVDJWRUdBX1JVTEVfQkFTRURfQUdFTlRfQ09ORklHX0lELFxuICBURVhUMlZFR0FfSU5QVVRfU0laRV9MSU1JVCxcbiAgVEVYVDJWRUdBX1dJVEhfSU5TVFJVQ1RJT05TX0FHRU5UX0NPTkZJR19JRCxcbiAgVEVYVDJWSVpfQVBJLFxufSBmcm9tICcuLi8uLi9jb21tb24vY29uc3RhbnRzL2xsbSc7XG5pbXBvcnQgeyBBc3Npc3RhbnRTZXJ2aWNlU2V0dXAgfSBmcm9tICcuLi9zZXJ2aWNlcy9hc3Npc3RhbnRfc2VydmljZSc7XG5cbmNvbnN0IGlucHV0U2NoZW1hID0gc2NoZW1hLnN0cmluZyh7XG4gIG1heExlbmd0aDogVEVYVDJWRUdBX0lOUFVUX1NJWkVfTElNSVQsXG4gIHZhbGlkYXRlKHZhbHVlKSB7XG4gICAgaWYgKCF2YWx1ZSB8fCB2YWx1ZS50cmltKCkubGVuZ3RoID09PSAwKSB7XG4gICAgICByZXR1cm4gXCJjYW4ndCBiZSBlbXB0eSBvciBibGFuay5cIjtcbiAgICB9XG4gIH0sXG59KTtcblxuZXhwb3J0IGZ1bmN0aW9uIHJlZ2lzdGVyVGV4dDJWaXpSb3V0ZXMocm91dGVyOiBJUm91dGVyLCBhc3Npc3RhbnRTZXJ2aWNlOiBBc3Npc3RhbnRTZXJ2aWNlU2V0dXApIHtcbiAgcm91dGVyLnBvc3QoXG4gICAge1xuICAgICAgcGF0aDogVEVYVDJWSVpfQVBJLlRFWFQyVkVHQSxcbiAgICAgIHZhbGlkYXRlOiB7XG4gICAgICAgIGJvZHk6IHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgIGlucHV0X3F1ZXN0aW9uOiBpbnB1dFNjaGVtYSxcbiAgICAgICAgICBpbnB1dF9pbnN0cnVjdGlvbjogc2NoZW1hLm1heWJlKHNjaGVtYS5zdHJpbmcoeyBtYXhMZW5ndGg6IFRFWFQyVkVHQV9JTlBVVF9TSVpFX0xJTUlUIH0pKSxcbiAgICAgICAgICBwcGw6IHNjaGVtYS5zdHJpbmcoKSxcbiAgICAgICAgICBkYXRhU2NoZW1hOiBzY2hlbWEuc3RyaW5nKCksXG4gICAgICAgICAgc2FtcGxlRGF0YTogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICB9KSxcbiAgICAgICAgcXVlcnk6IHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgIGRhdGFTb3VyY2VJZDogc2NoZW1hLm1heWJlKHNjaGVtYS5zdHJpbmcoKSksXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICB9LFxuICAgIHJvdXRlci5oYW5kbGVMZWdhY3lFcnJvcnMoYXN5bmMgKGNvbnRleHQsIHJlcSwgcmVzKSA9PiB7XG4gICAgICBjb25zdCBhc3Npc3RhbnRDbGllbnQgPSBhc3Npc3RhbnRTZXJ2aWNlLmdldFNjb3BlZENsaWVudChyZXEsIGNvbnRleHQpO1xuICAgICAgdHJ5IHtcbiAgICAgICAgY29uc3QgYWdlbnRDb25maWdOYW1lID0gcmVxLmJvZHkuaW5wdXRfaW5zdHJ1Y3Rpb25cbiAgICAgICAgICA/IFRFWFQyVkVHQV9XSVRIX0lOU1RSVUNUSU9OU19BR0VOVF9DT05GSUdfSURcbiAgICAgICAgICA6IFRFWFQyVkVHQV9SVUxFX0JBU0VEX0FHRU5UX0NPTkZJR19JRDtcbiAgICAgICAgY29uc3QgcmVzcG9uc2UgPSBhd2FpdCBhc3Npc3RhbnRDbGllbnQuZXhlY3V0ZUFnZW50QnlDb25maWdOYW1lKGFnZW50Q29uZmlnTmFtZSwge1xuICAgICAgICAgIGlucHV0X3F1ZXN0aW9uOiByZXEuYm9keS5pbnB1dF9xdWVzdGlvbixcbiAgICAgICAgICBpbnB1dF9pbnN0cnVjdGlvbjogcmVxLmJvZHkuaW5wdXRfaW5zdHJ1Y3Rpb24sXG4gICAgICAgICAgcHBsOiByZXEuYm9keS5wcGwsXG4gICAgICAgICAgZGF0YVNjaGVtYTogcmVxLmJvZHkuZGF0YVNjaGVtYSxcbiAgICAgICAgICBzYW1wbGVEYXRhOiByZXEuYm9keS5zYW1wbGVEYXRhLFxuICAgICAgICB9KTtcblxuICAgICAgICBsZXQgdGV4dENvbnRlbnQgPSByZXNwb25zZS5ib2R5LmluZmVyZW5jZV9yZXN1bHRzWzBdLm91dHB1dFswXS5yZXN1bHQ7XG5cbiAgICAgICAgLy8gZXh0cmEgY29udGVudCBiZXR3ZWVuIHRhZyA8dmVnYS1saXRlPjwvdmVnYS1saXRlPlxuICAgICAgICBjb25zdCBzdGFydFRhZyA9ICc8dmVnYS1saXRlPic7XG4gICAgICAgIGNvbnN0IGVuZFRhZyA9ICc8L3ZlZ2EtbGl0ZT4nO1xuXG4gICAgICAgIGNvbnN0IHN0YXJ0SW5kZXggPSB0ZXh0Q29udGVudC5pbmRleE9mKHN0YXJ0VGFnKTtcbiAgICAgICAgY29uc3QgZW5kSW5kZXggPSB0ZXh0Q29udGVudC5pbmRleE9mKGVuZFRhZyk7XG5cbiAgICAgICAgaWYgKHN0YXJ0SW5kZXggIT09IC0xICYmIGVuZEluZGV4ICE9PSAtMSAmJiBzdGFydEluZGV4IDwgZW5kSW5kZXgpIHtcbiAgICAgICAgICAvLyBFeHRyYWN0IHRoZSBjb250ZW50IGJldHdlZW4gdGhlIHRhZ3NcbiAgICAgICAgICB0ZXh0Q29udGVudCA9IHRleHRDb250ZW50LnN1YnN0cmluZyhzdGFydEluZGV4ICsgc3RhcnRUYWcubGVuZ3RoLCBlbmRJbmRleCkudHJpbSgpO1xuICAgICAgICB9XG5cbiAgICAgICAgLy8gZXh0cmFjdCBqc29uIG9iamVjdFxuICAgICAgICBjb25zdCBqc29uTWF0Y2ggPSB0ZXh0Q29udGVudC5tYXRjaCgvXFx7LipcXH0vcyk7XG4gICAgICAgIGlmIChqc29uTWF0Y2gpIHtcbiAgICAgICAgICBsZXQgcmVzdWx0ID0gSlNPTi5wYXJzZShqc29uTWF0Y2hbMF0pO1xuICAgICAgICAgIC8vIHNvbWV0aW1lcyBsbG0gcmV0dXJucyB7cmVzcG9uc2U6IDxzY2hlbWE+fSBpbnN0ZWFkIG9mIDxzY2hlbWE+XG4gICAgICAgICAgaWYgKHJlc3VsdC5yZXNwb25zZSkge1xuICAgICAgICAgICAgcmVzdWx0ID0gSlNPTi5wYXJzZShyZXN1bHQucmVzcG9uc2UpO1xuICAgICAgICAgIH1cbiAgICAgICAgICAvLyBTb21ldGltZXMgdGhlIHJlc3BvbnNlIGNvbnRhaW5zIHdpZHRoIGFuZCBoZWlnaHQgd2hpY2ggaXMgbm90IG5lZWRlZCwgaGVyZSBkZWxldGUgdGhlIHRoZXNlIGZpZWxkc1xuICAgICAgICAgIGRlbGV0ZSByZXN1bHQud2lkdGg7XG4gICAgICAgICAgZGVsZXRlIHJlc3VsdC5oZWlnaHQ7XG5cbiAgICAgICAgICAvLyBtYWtlIHN1cmUgJHNjaGVtYSBmaWVsZCBhbHdheXMgYmVlbiBhZGRlZCwgc29tZXRpbWVzLCBMTE0gJ2ZvcmdvdCcgdG8gYWRkIHRoaXMgZmllbGRcbiAgICAgICAgICByZXN1bHQuJHNjaGVtYSA9ICdodHRwczovL3ZlZ2EuZ2l0aHViLmlvL3NjaGVtYS92ZWdhLWxpdGUvdjUuanNvbic7XG4gICAgICAgICAgcmV0dXJuIHJlcy5vayh7IGJvZHk6IHJlc3VsdCB9KTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gcmVzLmJhZFJlcXVlc3QoKTtcbiAgICAgIH0gY2F0Y2ggKGUpIHtcbiAgICAgICAgcmV0dXJuIHJlcy5iYWRSZXF1ZXN0KCk7XG4gICAgICB9XG4gICAgfSlcbiAgKTtcblxuICByb3V0ZXIucG9zdChcbiAgICB7XG4gICAgICBwYXRoOiBURVhUMlZJWl9BUEkuVEVYVDJQUEwsXG4gICAgICB2YWxpZGF0ZToge1xuICAgICAgICBib2R5OiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBpbmRleDogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICAgIHF1ZXN0aW9uOiBpbnB1dFNjaGVtYSxcbiAgICAgICAgfSksXG4gICAgICAgIHF1ZXJ5OiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBkYXRhU291cmNlSWQ6IHNjaGVtYS5tYXliZShzY2hlbWEuc3RyaW5nKCkpLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfSxcbiAgICByb3V0ZXIuaGFuZGxlTGVnYWN5RXJyb3JzKGFzeW5jIChjb250ZXh0LCByZXEsIHJlcykgPT4ge1xuICAgICAgY29uc3QgYXNzaXN0YW50Q2xpZW50ID0gYXNzaXN0YW50U2VydmljZS5nZXRTY29wZWRDbGllbnQocmVxLCBjb250ZXh0KTtcbiAgICAgIHRyeSB7XG4gICAgICAgIGNvbnN0IHJlc3BvbnNlID0gYXdhaXQgYXNzaXN0YW50Q2xpZW50LmV4ZWN1dGVBZ2VudEJ5Q29uZmlnTmFtZShURVhUMlBQTF9BR0VOVF9DT05GSUdfSUQsIHtcbiAgICAgICAgICBxdWVzdGlvbjogcmVxLmJvZHkucXVlc3Rpb24sXG4gICAgICAgICAgaW5kZXg6IHJlcS5ib2R5LmluZGV4LFxuICAgICAgICB9KTtcblxuICAgICAgICBjb25zdCByZXN1bHQgPSBKU09OLnBhcnNlKHJlc3BvbnNlLmJvZHkuaW5mZXJlbmNlX3Jlc3VsdHNbMF0ub3V0cHV0WzBdLnJlc3VsdCk7XG4gICAgICAgIHJldHVybiByZXMub2soeyBib2R5OiByZXN1bHQgfSk7XG4gICAgICB9IGNhdGNoIChlKSB7XG4gICAgICAgIHJldHVybiByZXMuYmFkUmVxdWVzdCgpO1xuICAgICAgfVxuICAgIH0pXG4gICk7XG59XG4iXSwibWFwcGluZ3MiOiI7Ozs7OztBQUtBLElBQUFBLGFBQUEsR0FBQUMsT0FBQTtBQUVBLElBQUFDLElBQUEsR0FBQUQsT0FBQTtBQVBBO0FBQ0E7QUFDQTtBQUNBOztBQWFBLE1BQU1FLFdBQVcsR0FBR0Msb0JBQU0sQ0FBQ0MsTUFBTSxDQUFDO0VBQ2hDQyxTQUFTLEVBQUVDLCtCQUEwQjtFQUNyQ0MsUUFBUUEsQ0FBQ0MsS0FBSyxFQUFFO0lBQ2QsSUFBSSxDQUFDQSxLQUFLLElBQUlBLEtBQUssQ0FBQ0MsSUFBSSxDQUFDLENBQUMsQ0FBQ0MsTUFBTSxLQUFLLENBQUMsRUFBRTtNQUN2QyxPQUFPLDBCQUEwQjtJQUNuQztFQUNGO0FBQ0YsQ0FBQyxDQUFDO0FBRUssU0FBU0Msc0JBQXNCQSxDQUFDQyxNQUFlLEVBQUVDLGdCQUF1QyxFQUFFO0VBQy9GRCxNQUFNLENBQUNFLElBQUksQ0FDVDtJQUNFQyxJQUFJLEVBQUVDLGlCQUFZLENBQUNDLFNBQVM7SUFDNUJWLFFBQVEsRUFBRTtNQUNSVyxJQUFJLEVBQUVmLG9CQUFNLENBQUNnQixNQUFNLENBQUM7UUFDbEJDLGNBQWMsRUFBRWxCLFdBQVc7UUFDM0JtQixpQkFBaUIsRUFBRWxCLG9CQUFNLENBQUNtQixLQUFLLENBQUNuQixvQkFBTSxDQUFDQyxNQUFNLENBQUM7VUFBRUMsU0FBUyxFQUFFQztRQUEyQixDQUFDLENBQUMsQ0FBQztRQUN6RmlCLEdBQUcsRUFBRXBCLG9CQUFNLENBQUNDLE1BQU0sQ0FBQyxDQUFDO1FBQ3BCb0IsVUFBVSxFQUFFckIsb0JBQU0sQ0FBQ0MsTUFBTSxDQUFDLENBQUM7UUFDM0JxQixVQUFVLEVBQUV0QixvQkFBTSxDQUFDQyxNQUFNLENBQUM7TUFDNUIsQ0FBQyxDQUFDO01BQ0ZzQixLQUFLLEVBQUV2QixvQkFBTSxDQUFDZ0IsTUFBTSxDQUFDO1FBQ25CUSxZQUFZLEVBQUV4QixvQkFBTSxDQUFDbUIsS0FBSyxDQUFDbkIsb0JBQU0sQ0FBQ0MsTUFBTSxDQUFDLENBQUM7TUFDNUMsQ0FBQztJQUNIO0VBQ0YsQ0FBQyxFQUNEUSxNQUFNLENBQUNnQixrQkFBa0IsQ0FBQyxPQUFPQyxPQUFPLEVBQUVDLEdBQUcsRUFBRUMsR0FBRyxLQUFLO0lBQ3JELE1BQU1DLGVBQWUsR0FBR25CLGdCQUFnQixDQUFDb0IsZUFBZSxDQUFDSCxHQUFHLEVBQUVELE9BQU8sQ0FBQztJQUN0RSxJQUFJO01BQ0YsTUFBTUssZUFBZSxHQUFHSixHQUFHLENBQUNaLElBQUksQ0FBQ0csaUJBQWlCLEdBQzlDYyxnREFBMkMsR0FDM0NDLHlDQUFvQztNQUN4QyxNQUFNQyxRQUFRLEdBQUcsTUFBTUwsZUFBZSxDQUFDTSx3QkFBd0IsQ0FBQ0osZUFBZSxFQUFFO1FBQy9FZCxjQUFjLEVBQUVVLEdBQUcsQ0FBQ1osSUFBSSxDQUFDRSxjQUFjO1FBQ3ZDQyxpQkFBaUIsRUFBRVMsR0FBRyxDQUFDWixJQUFJLENBQUNHLGlCQUFpQjtRQUM3Q0UsR0FBRyxFQUFFTyxHQUFHLENBQUNaLElBQUksQ0FBQ0ssR0FBRztRQUNqQkMsVUFBVSxFQUFFTSxHQUFHLENBQUNaLElBQUksQ0FBQ00sVUFBVTtRQUMvQkMsVUFBVSxFQUFFSyxHQUFHLENBQUNaLElBQUksQ0FBQ087TUFDdkIsQ0FBQyxDQUFDO01BRUYsSUFBSWMsV0FBVyxHQUFHRixRQUFRLENBQUNuQixJQUFJLENBQUNzQixpQkFBaUIsQ0FBQyxDQUFDLENBQUMsQ0FBQ0MsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDQyxNQUFNOztNQUVyRTtNQUNBLE1BQU1DLFFBQVEsR0FBRyxhQUFhO01BQzlCLE1BQU1DLE1BQU0sR0FBRyxjQUFjO01BRTdCLE1BQU1DLFVBQVUsR0FBR04sV0FBVyxDQUFDTyxPQUFPLENBQUNILFFBQVEsQ0FBQztNQUNoRCxNQUFNSSxRQUFRLEdBQUdSLFdBQVcsQ0FBQ08sT0FBTyxDQUFDRixNQUFNLENBQUM7TUFFNUMsSUFBSUMsVUFBVSxLQUFLLENBQUMsQ0FBQyxJQUFJRSxRQUFRLEtBQUssQ0FBQyxDQUFDLElBQUlGLFVBQVUsR0FBR0UsUUFBUSxFQUFFO1FBQ2pFO1FBQ0FSLFdBQVcsR0FBR0EsV0FBVyxDQUFDUyxTQUFTLENBQUNILFVBQVUsR0FBR0YsUUFBUSxDQUFDakMsTUFBTSxFQUFFcUMsUUFBUSxDQUFDLENBQUN0QyxJQUFJLENBQUMsQ0FBQztNQUNwRjs7TUFFQTtNQUNBLE1BQU13QyxTQUFTLEdBQUdWLFdBQVcsQ0FBQ1csS0FBSyxDQUFDLFNBQVMsQ0FBQztNQUM5QyxJQUFJRCxTQUFTLEVBQUU7UUFDYixJQUFJUCxNQUFNLEdBQUdTLElBQUksQ0FBQ0MsS0FBSyxDQUFDSCxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDckM7UUFDQSxJQUFJUCxNQUFNLENBQUNMLFFBQVEsRUFBRTtVQUNuQkssTUFBTSxHQUFHUyxJQUFJLENBQUNDLEtBQUssQ0FBQ1YsTUFBTSxDQUFDTCxRQUFRLENBQUM7UUFDdEM7UUFDQTtRQUNBLE9BQU9LLE1BQU0sQ0FBQ1csS0FBSztRQUNuQixPQUFPWCxNQUFNLENBQUNZLE1BQU07O1FBRXBCO1FBQ0FaLE1BQU0sQ0FBQ2EsT0FBTyxHQUFHLGlEQUFpRDtRQUNsRSxPQUFPeEIsR0FBRyxDQUFDeUIsRUFBRSxDQUFDO1VBQUV0QyxJQUFJLEVBQUV3QjtRQUFPLENBQUMsQ0FBQztNQUNqQztNQUNBLE9BQU9YLEdBQUcsQ0FBQzBCLFVBQVUsQ0FBQyxDQUFDO0lBQ3pCLENBQUMsQ0FBQyxPQUFPQyxDQUFDLEVBQUU7TUFDVixPQUFPM0IsR0FBRyxDQUFDMEIsVUFBVSxDQUFDLENBQUM7SUFDekI7RUFDRixDQUFDLENBQ0gsQ0FBQztFQUVEN0MsTUFBTSxDQUFDRSxJQUFJLENBQ1Q7SUFDRUMsSUFBSSxFQUFFQyxpQkFBWSxDQUFDMkMsUUFBUTtJQUMzQnBELFFBQVEsRUFBRTtNQUNSVyxJQUFJLEVBQUVmLG9CQUFNLENBQUNnQixNQUFNLENBQUM7UUFDbEJ5QyxLQUFLLEVBQUV6RCxvQkFBTSxDQUFDQyxNQUFNLENBQUMsQ0FBQztRQUN0QnlELFFBQVEsRUFBRTNEO01BQ1osQ0FBQyxDQUFDO01BQ0Z3QixLQUFLLEVBQUV2QixvQkFBTSxDQUFDZ0IsTUFBTSxDQUFDO1FBQ25CUSxZQUFZLEVBQUV4QixvQkFBTSxDQUFDbUIsS0FBSyxDQUFDbkIsb0JBQU0sQ0FBQ0MsTUFBTSxDQUFDLENBQUM7TUFDNUMsQ0FBQztJQUNIO0VBQ0YsQ0FBQyxFQUNEUSxNQUFNLENBQUNnQixrQkFBa0IsQ0FBQyxPQUFPQyxPQUFPLEVBQUVDLEdBQUcsRUFBRUMsR0FBRyxLQUFLO0lBQ3JELE1BQU1DLGVBQWUsR0FBR25CLGdCQUFnQixDQUFDb0IsZUFBZSxDQUFDSCxHQUFHLEVBQUVELE9BQU8sQ0FBQztJQUN0RSxJQUFJO01BQ0YsTUFBTVEsUUFBUSxHQUFHLE1BQU1MLGVBQWUsQ0FBQ00sd0JBQXdCLENBQUN3Qiw2QkFBd0IsRUFBRTtRQUN4RkQsUUFBUSxFQUFFL0IsR0FBRyxDQUFDWixJQUFJLENBQUMyQyxRQUFRO1FBQzNCRCxLQUFLLEVBQUU5QixHQUFHLENBQUNaLElBQUksQ0FBQzBDO01BQ2xCLENBQUMsQ0FBQztNQUVGLE1BQU1sQixNQUFNLEdBQUdTLElBQUksQ0FBQ0MsS0FBSyxDQUFDZixRQUFRLENBQUNuQixJQUFJLENBQUNzQixpQkFBaUIsQ0FBQyxDQUFDLENBQUMsQ0FBQ0MsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDQyxNQUFNLENBQUM7TUFDOUUsT0FBT1gsR0FBRyxDQUFDeUIsRUFBRSxDQUFDO1FBQUV0QyxJQUFJLEVBQUV3QjtNQUFPLENBQUMsQ0FBQztJQUNqQyxDQUFDLENBQUMsT0FBT2dCLENBQUMsRUFBRTtNQUNWLE9BQU8zQixHQUFHLENBQUMwQixVQUFVLENBQUMsQ0FBQztJQUN6QjtFQUNGLENBQUMsQ0FDSCxDQUFDO0FBQ0gifQ==