package App::Manoc::Search::Result;

use Moose;

our $VERSION = '2.99.2'; ##TRIAL VERSION

use App::Manoc::Search::Item::IpAddr;
use App::Manoc::Search::Item::MacAddr;
use App::Manoc::Search::Item::WinLogon;
use App::Manoc::Search::Widget::ApplyRole;


has query => ( is => 'ro' );


has item_by_match => (
    is      => 'ro',
    isa     => 'HashRef[String]',
    default => sub { {} },
);


has message => (
    is  => 'rw',
    isa => 'Str',
);


has groups => (
    is      => 'ro',
    isa     => 'ArrayRef',
    writer  => '_groups',
    default => sub { [] },
);

my %GROUP_ITEM = (
    'logon'   => 'App::Manoc::Search::Item::WinLogon',
    'ipaddr'  => 'App::Manoc::Search::Item::IpAddr',
    'macaddr' => 'App::Manoc::Search::Item::MacAddr',
);


sub add_item {
    my ( $self, $item ) = @_;

    my $match = $item->match;

    my $query_type = $self->query->query_type;

    my $group = $self->item_by_match->{$match};
    if ( !defined($group) ) {
        my $class = $GROUP_ITEM{$query_type};
        $class ||= 'App::Manoc::Search::Item::Group';

        $group = $class->new( { match => $match } );
        $self->item_by_match->{$match} = $group;

        push @{ $self->{groups} }, $group;
    }

    $group->add_item($item);
}


sub items {
    my $self = shift;
    $self->sort_items;

    return $self->groups;
}


sub sort_items {
    my $self = shift;

    foreach my $group ( values %{ $self->item_by_match } ) {
        $group->sort_items;
    }

    my $groups = $self->groups;
    my @g = sort { $a->key cmp $b->key } @$groups;
    $self->_groups( \@g );
}


sub load_widgets {
    my $self = shift;

    foreach my $group ( values %{ $self->item_by_match } ) {
        $group->load_widgets(@_);
    }
}

no Moose;
__PACKAGE__->meta->make_immutable;

__END__

=pod

=head1 NAME

App::Manoc::Search::Result

=head1 VERSION

version 2.99.2

=head1 ATTRIBUTES

=head2 query

=head2 item_by_match

=head2 message

=head2 groups

The list of L<App::Manoc::Search::Item::Group> objects which will
contain the result items.

=head1 METHODS

=head2 add_item

Add a new L<App::Manoc::Search::Item> to the result. Creates a new
group if the item has a new match.

=head2 items

Return all the result items, sorted by groups.

=head2 sort_items

Sort the result group by key.

=head2 load_widgets

Load widget for all result items.

=head1 AUTHORS

=over 4

=item *

Gabriele Mambrini <gmambro@cpan.org>

=item *

Enrico Liguori

=back

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2017 by Gabriele Mambrini.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
