# vim:set filetype=perl sw=4 et:
# Before `make install' is performed this script should be runnable with
# `make test'. After `make install' it should work as `perl 1.t'

#########################

use Test::More tests => 672;
use Carp;

use Lingua::Zompist::Verdurian 'noun';

sub form_ok {
    croak 'usage: form_ok($noun, $is, $should)' unless @_ >= 3;
    my($noun, $is, $should) = @_;

    is($is->[0], $should->[0], "nom.sg. of $noun");
    is($is->[1], $should->[1], "gen.sg. of $noun");
    is($is->[2], $should->[2], "acc.sg. of $noun");
    is($is->[3], $should->[3], "dat.sg. of $noun");
    is($is->[4], $should->[4], "nom.pl. of $noun");
    is($is->[5], $should->[5], "gen.pl. of $noun");
    is($is->[6], $should->[6], "acc.pl. of $noun");
    is($is->[7], $should->[7], "dat.pl. of $noun");
}

form_ok('redh', noun('redh'), [ qw( redh redhei redh redhn
                                    redh redhi redhi redhin ) ]);
form_ok('dasco', noun('dasco'), [ qw( dasco dascei dascam dascon
                                      dascoi dasco dascom dascoin ) ]);
form_ok('dalu', noun('dalu'), [ qw( dalu dalui dalum dalun
                                    dal dalu dalom daluin ) ]);
form_ok('katy', noun('katy'), [ qw( katy katii katim katn
                                    kat katu katom katuin ) ]);
form_ok('ciy', noun('ciy'), [ qw( ciy cii ciim cin
                                    ci ci ciom ciuin ) ]);
form_ok('esta', noun('esta'), [ qw( esta estei esta estan
                                    estai esta estam estain ) ]);

form_ok('rana', noun('rana'), [ qw( rana rane rana ranan
                                    ran rani ranem ranen ) ]);
form_ok('lavsia', noun('lavsia'), [ qw( lavsia lavs lavsiam lavsian
                                          lavsi lavsi lavisem lavisen ) ]);
form_ok('casi', noun('casi'), [ qw( casi cas casa casin
                                    cas casi casem casin ) ]);
form_ok('leve', noun('leve'), [ qw( leve levei leva leven
                                    lev levi levem leven ) ]);
form_ok('gur', noun('gur'), [ qw( gur guri gur guren
                                    gur guri gurem guren ) ]);

# final accent remains on the ending
form_ok('akn', noun('akn'), [ qw( akn aknei aknm aknn
                                    aknoi akno aknm aknoin ) ]);
form_ok('pis', noun('pis'), [ qw( pis pis pis pisn
                                    pis pisi pism pisn ) ]);

# irregular plural stem
# First those from morphology.htm (as of 2001-10-22)
form_ok('brca', noun('brca'), [ qw( brca brce brca brcan
                                      brs brsi brsem brsen ) ]);
form_ok('kud', noun('kud'), [ qw( kud kudei kud kudn
                                  kuz kuzi kuzi kuzin ) ]);
form_ok('log', noun('log'), [ qw( log logei log logn
                                  lozhi lozhi lozhi lozhin ) ]);
form_ok('rhit', noun('rhit'), [ qw( rhit rhitei rhit rhitn
                                    rhichi rhichi rhichi rhichin ) ]);
form_ok('verat', noun('verat'), [ qw( verat veratei verat veratn
                                      veradhi veradhi veradhi veradhin ) ]);

# Now those from the source code
# (These are derived from the dictionary)
# Some checks might be made twice; oh well :)
form_ok('aklog', noun('aklog'), [ qw( aklog aklogei aklog aklogn
                                      aklozhi aklozhi aklozhi aklozhin ) ]);
form_ok('nselcud', noun('nselcud'), [ qw( nselcud nselcudei nselcud anselcudn
                                            nselcuz nselcuzi nselcuzi nselcuzin ) ]);
form_ok('barsc', noun('barsc'), [ qw( barsc barscei barsc barsucn
                                        barsus barssi barsusi barsusin ) ]);
form_ok('belac', noun('belac'), [ qw( belac belacei belac belacn
                                      belas belasi belasi belasin ) ]);
form_ok('boc', noun('boc'), [ qw( boc bocei boc bocn
                                  bos bosi bosi bosin ) ]);
form_ok('brca', noun('brca'), [ qw( brca brce brca brcan
                                      brs brsi brsem brsen ) ]);
form_ok('bt', noun('bt'), [ qw( bt btei bt btn
                                  bs bsi bsi bsin ) ]);
form_ok('chedesnaga', noun('chedesnaga'), [ qw( chedesnaga chedesnage chedesnaga chedesnagan
                                                chedesnazhi chedesnazhi chedesnazhem chedesnazhen ) ]);
form_ok('chuca', noun('chuca'), [ qw( chuca chuce chuca chucan
                                      chus chusi chusem chusen ) ]);
form_ok('dosic', noun('dosic'), [ qw( dosic dosicei dosic dosicn
                                      dosis dosisi dosisi dosisin ) ]);
form_ok('drac', noun('drac'), [ qw( drac dracei drac dracn
                                    dras drasi drasi drasin ) ]);
form_ok('dushic', noun('dushic'), [ qw( dushic dushicei dushic dushicn
                                        dushis dushisi dushisi dushisin ) ]);
form_ok('dhac', noun('dhac'), [ qw( dhac dhacei dhac dhacn
                                    dhas dhasi dhasi dhasin ) ]);
form_ok('dhiec', noun('dhiec'), [ qw( dhiec dhiecei dhiec dhiecn
                                      dhies dhiesi dhiesi dhiesin ) ]);
form_ok('ecelg', noun('ecelg'), [ qw( ecelg ecelgei ecelg ecelogn
                                        ecelozhi ecelzhi ecelozhi ecelozhin ) ]);
form_ok('etalg', noun('etalg'), [ qw( etalg etalgei etalg etalogn
                                        etalozhi etalzhi etalozhi etalozhin ) ]);
form_ok('ferica', noun('ferica'), [ qw( ferica ferice ferica ferican
                                        feris ferisi ferisem ferisen ) ]);
form_ok('fifachic', noun('fifachic'), [ qw( fifachic fifachicei fifachic fifachicn
                                            fifachis fifachisi fifachisi fifachisin ) ]);
form_ok('formica', noun('formica'), [ qw( formica formice formica formican
                                          formis formisi formisem formisen ) ]);
form_ok('glca', noun('glca'), [ qw( glca glce glca glcan
                                      gls glsi glsem glsen ) ]);
form_ok('gorat', noun('gorat'), [ qw( gorat goratei gorat goratn
                                      goradhi goradhi goradhi goradhin ) ]);
form_ok('grak', noun('grak'), [ qw( grak grakei grak grakn
                                    grah grahi grahi grahin ) ]);
form_ok('gut', noun('gut'), [ qw( gut gutei gut gutn
                                  gudhi gudhi gudhi gudhin ) ]);
form_ok('huca', noun('huca'), [ qw( huca huce huca hucan
                                    hus husi husem husen ) ]);
form_ok('ktlog', noun('ktlog'), [ qw( ktlog ktlogei ktlog ktlogn
                                        ktlozhi ktlozhi ktlozhi ktlozhin ) ]);
form_ok('kud', noun('kud'), [ qw( kud kudei kud kudn
                                  kuz kuzi kuzi kuzin ) ]);
form_ok('lertlog', noun('lertlog'), [ qw( lertlog lertlogei lertlog lertlogn
                                          lertlozhi lertlozhi lertlozhi lertlozhin ) ]);
form_ok('log', noun('log'), [ qw( log logei log logn
                                  lozhi lozhi lozhi lozhin ) ]);
form_ok('matica', noun('matica'), [ qw( matica matice matica matican
                                        matis matisi matisem matisen ) ]);
form_ok('meca', noun('meca'), [ qw( meca mece meca mecan
                                    mes mesi mesem mesen ) ]);
form_ok('mevlog', noun('mevlog'), [ qw( mevlog mevlogei mevlog mevlogn
                                        mevlozhi mevlozhi mevlozhi mevlozhin ) ]);
form_ok('morut', noun('morut'), [ qw( morut morutei morut morutn
                                      morudhi morudhi morudhi morudhin ) ]);
form_ok('naga', noun('naga'), [ qw( naga nage naga nagan
                                    nazhi nazhi nazhem nazhen ) ]);
form_ok('nid', noun('nid'), [ qw( nid nidei nid nidn
                                  niz nizi nizi nizin ) ]);
form_ok('pag', noun('pag'), [ qw( pag pagei pag pagn
                                  pazhi pazhi pazhi pazhin ) ]);
form_ok('prolog', noun('prolog'), [ qw( prolog prologei prolog prologn
                                        prolozhi prolozhi prolozhi prolozhin ) ]);
form_ok('rak', noun('rak'), [ qw( rak rakei rak rakn
                                  rah rahi rahi rahin ) ]);
form_ok('rog', noun('rog'), [ qw( rog rogei rog rogn
                                  rozhi rozhi rozhi rozhin ) ]);
form_ok('rhit', noun('rhit'), [ qw( rhit rhitei rhit rhitn
                                    rhichi rhichi rhichi rhichin ) ]);
form_ok('sfica', noun('sfica'), [ qw( sfica sfice sfica sfican
                                      sfis sfisi sfisem sfisen ) ]);
form_ok('shank', noun('shank'), [ qw( shank shankei shank shankn
                                      shanh shanhi shanhi shanhin ) ]);
form_ok('smeric', noun('smeric'), [ qw( smeric smericei smeric smericn
                                        smeris smerisi smerisi smerisin ) ]);
form_ok('verat', noun('verat'), [ qw( verat veratei verat veratn
                                      veradhi veradhi veradhi veradhin ) ]);
form_ok('yag', noun('yag'), [ qw( yag yagei yag yagn
                                  yazhi yazhi yazhi yazhin ) ]);

# Test generic conjugation
form_ok('ggg', noun('ggg'), [ qw( ggg gggei ggg gggn
                                  ggg gggi gggi gggin ) ]);
form_ok('gggia', noun('gggia'), [ qw( gggia ggg gggiam gggian
                                      gggi gggi gggem gggen ) ]);
form_ok('ggga', noun('ggga'), [ qw( ggga ggge ggga gggan
                                    ggg gggi gggem gggen ) ]);
form_ok('ggg', noun('ggg'), [ qw( ggg ggg ggg gggn
                                    ggg gggi gggm gggn ) ]);
form_ok('gggo', noun('gggo'), [ qw( gggo gggei gggam gggon
                                    gggoi gggo gggom gggoin ) ]);
form_ok('ggg', noun('ggg'), [ qw( ggg gggei gggm gggn
                                    gggoi gggo gggm gggoin ) ]);
form_ok('gggu', noun('gggu'), [ qw( gggu gggui gggum gggun
                                    ggg gggu gggom ggguin ) ]);
form_ok('ggg', noun('ggg'), [ qw( ggg gggui gggm gggn
                                    ggg gggu gggm ggguin ) ]);
form_ok('gggiy', noun('gggiy'), [ qw( gggiy gggii gggiim gggin
                                      gggi gggi gggiom gggiuin ) ]);
form_ok('gggy', noun('gggy'), [ qw( gggy gggii gggiim gggin
                                      gggi ggg gggiom ggguin ) ]);
form_ok('gggy', noun('gggy'), [ qw( gggy gggii gggim gggn
                                    ggg gggu gggom ggguin ) ]);
# Can't test generic masculine in -a, but see further down
form_ok('gggi', noun('gggi'), [ qw( gggi ggg ggga gggin
                                    ggg gggi gggem gggin ) ]);
form_ok('ggg', noun('ggg'), [ qw( ggg ggg ggg gggn
                                    ggg gggi gggm gggn ) ]);
form_ok('ggge', noun('ggge'), [ qw( ggge gggei ggga gggen
                                    ggg gggi gggem gggen ) ]);
form_ok('ggg', noun('ggg'), [ qw( ggg gggei ggg gggn
                                    ggg gggi gggm gggn ) ]);
form_ok('ggg', noun('ggg'), [ qw( ggg gggi ggg gggen
                                    ggg gggi gggem gggen ) ]);

# Test known masculine nouns in -a
form_ok('creza', noun('creza'), [ qw( creza crezei creza crezan
                                      crezai creza crezam crezain ) ]);
form_ok('Erva', noun('Erva'), [ qw( Erva Ervei Erva Ervan
                                      Ervai Erva Ervam Ervain ) ]);
form_ok('esta', noun('esta'), [ qw( esta estei esta estan
                                    estai esta estam estain ) ]);
form_ok('hezhiosa', noun('hezhiosa'), [ qw( hezhiosa hezhiosei hezhiosa hezhiosan
                                            hezhiosai hezhiosa hezhiosam hezhiosain ) ]);
form_ok('rhena', noun('rhena'), [ qw( rhena rhenei rhena rhenan
                                      rhenai rhena rhenam rhenain ) ]);
form_ok('didha', noun('didha'), [ qw( didha didhei didha didhan
                                      didhai didha didham didhain ) ]);
