package TaskPipe::Tool::Command_GenerateSchema;

use Moose;
use TaskPipe::SchemaManager;
use TaskPipe::JobManager;
use TaskPipe::Tool::Options;
use DateTime;
extends 'TaskPipe::Tool::Command';
with 'MooseX::ConfigCascade';
with 'TaskPipe::Role::MooseType_ScopeMode';

has schema_manager => (is => 'rw', isa => 'TaskPipe::SchemaManager', lazy => 1, default => sub{
    my $sm = TaskPipe::SchemaManager->new;
    return $sm;
});
        


has option_specs => (is => 'ro', isa => 'ArrayRef', lazy => 1, default => sub{
    my $settings = ref( $_[0]->schema_manager->settings );
    my %items;

    if ( $_[0]->scope eq 'project' ){
        $items{global} = [
            'project_root',
            'project'
        ];
        $items{project} = [
            'lib_dir',
        ]
    } else {
        $items{global} = [
            'global_root',
            'global_lib_dir'
        ]
    }

    my $specs = [{
        module => __PACKAGE__,
        is_config => 0
    }, {
        module => $settings,
        items => [
            'method',
            'type',
            'name',
            'host',
            'username',
            'password',
            'module'
        ],
        is_config => 1
    }, {
        module => 'TaskPipe::PathSettings::Global',
        items => $items{global},
        is_config => 1
    }];

    if ( $items{project} ){
        push @$specs, {
            module => 'TaskPipe::PathSettings::Project',
            items => $items{project},
            is_config => 1
        }
    }

    return $specs;
});




sub execute{
    my $self = shift;

    $self->run_info->scope( $self->scope );
    $self->schema_manager->gen_schema;

}



=pod

=head1 NAME

TaskPipe::Tool::Command_GenerateSchema - command to generate taskpipe schema files

=head1 PURPOSE

generate the schema files needed by taskpipe to interact with the database

=head1 DESCRIPTION

TaskPipe uses the L<DBIx::Class> ORM to interface with your database(s). TaskPipe uses one general purpose I<global> database and one I<project> database for each project. See the help for C<deploy tables> for more information on the organisation of database tables.

=head2 Global Schema Files

After running C<setup> you will need to run C<generate schema> to create the schema files needed for your global database. You need to have already successfully deployed global tables (see C<deploy tables>). Then run

    taskpipe generate schema --scope=global

=head2 Project Schema Files

Each time you make changes to your project database, you need to run C<generate schema> over your project database before you can run your plan again.

Schema files are generated for

=over

=item *

the tables in your project which taskpipe autogenerated, and which taskpipe uses for caching (these are the C<cache> tables

=item *

the tables you created manually as part of your project (these are your C<plan> tables).

=back

To run C<generate schema> over your project database, make sure your tables are deployed (see C<deploy tables>). Then run

    taskpipe generate schema --project=myproject

If you set a default project in your global config file, and you want to run C<generate schema> over the default project then you can just type

    taskpipe generate schema

=head1 OPTIONS

=over

=item scope

Specify C<project> to generate the schema over the tables in your project (both cache and plan tables) - or C<global> to generate the global schema

=back

=cut

has scope => (is => 'rw', isa => 'ScopeMode', default => 'project');

=head1 AUTHOR

Tom Gracey <tomgracey@gmail.com>

=head1 COPYRIGHT AND LICENSE

Copyright (c) Tom Gracey 2018

TaskPipe is free software, licensed under

    The GNU Public License Version 3

=cut

__PACKAGE__->meta->make_immutable;
1;
