package App::Manoc::Controller::ServerHW;
#ABSTRACT: ServerHW controller

use Moose;

our $VERSION = '2.99.3'; ##TRIAL VERSION

use namespace::autoclean;

use App::Manoc::Form::ServerHW;
use App::Manoc::Form::CSVImport::ServerHW;

BEGIN { extends 'Catalyst::Controller'; }
with "App::Manoc::ControllerRole::CommonCRUD";

__PACKAGE__->config(
    # define PathPart
    action => {
        setup => {
            PathPart => 'serverhw',
        }
    },
    class            => 'ManocDB::ServerHW',
    form_class       => 'App::Manoc::Form::ServerHW',
    view_object_perm => undef,

    create_page_title => 'Create server hardware',
    edit_page_title   => 'Edit server hardware',

    object_list_options => {
        prefetch => [ { 'hwasset' => { 'rack' => 'building' } }, 'server' ]
    },

    csv_columns => [
        'model',        'vendor',        'inventory',     'serial',
        'ram_memory',   'cpu_model',     'proc_freq',     'n_procs',
        'n_cores_proc', 'storage1_size', 'storage2_size', 'notes'
    ],

);


before 'create' => sub {
    my ( $self, $c ) = @_;

    if ( my $copy_id = $c->req->query_parameters->{'copy'} ) {
        my $original = $c->stash->{resultset}->find($copy_id);
        if ($original) {
            $c->debug and $c->log->debug("copy server from $copy_id");

            my %cols = $original->get_columns;
            delete $cols{'hwasset_id'};
            delete $cols{'id'};
            foreach (qw(model vendor)) {
                $cols{$_} = $original->hwasset->get_column($_);
            }

            $cols{nics} = [];
            foreach my $nic ( $original->nics ) {
                push @{ $cols{nics} },
                    {
                    name     => $nic->name,
                    nic_type => $nic->nic_type,
                    };
            }

            $c->stash( form_defaults => \%cols );
        }
    }

    if ( my $nwinfo_id = $c->req->query_parameters->{'nwinfo'} ) {
        my $nwinfo = $c->model('ManocDB::ServerNWInfo')->find($nwinfo_id);
        if ($nwinfo) {
            my %cols;
            $cols{model}      = $nwinfo->model;
            $cols{vendor}     = $nwinfo->vendor;
            $cols{serial}     = $nwinfo->serial;
            $cols{n_procs}    = $nwinfo->n_procs;
            $cols{cpu_model}  = $nwinfo->cpu_model;
            $cols{ram_memory} = $nwinfo->ram_memory;

            $c->stash( form_defaults => \%cols );
        }
    }

};


before 'edit' => sub {
    my ( $self, $c ) = @_;

    my $object    = $c->stash->{object};
    my $object_pk = $c->stash->{object_pk};

    # decommissioned objects cannot be edited
    if ( $object->is_decommissioned ) {
        $c->res->redirect( $c->uri_for_action( 'serverhw/view', [$object_pk] ) );
        $c->detach();
    }
};


sub import_csv : Chained('base') : PathPart('importcsv') : Args(0) {
    my ( $self, $c ) = @_;

    $c->require_permission( 'serverhw', 'create' );
    my $rs = $c->stash->{resultset};

    my $upload;
    $c->req->method eq 'POST' and $upload = $c->req->upload('file');

    my $form = App::Manoc::Form::CSVImport::ServerHW->new(
        ctx       => $c,
        resultset => $rs,
    );
    $c->stash->{form} = $form;

    my %process_params;
    $process_params{params} = $c->req->parameters;
    $upload and $process_params{params}->{file} = $upload;
    my $process_status = $form->process(%process_params);

    return unless $process_status;
}


sub decommission : Chained('object') : PathPart('decommission') : Args(0) {
    my ( $self, $c ) = @_;

    my $object = $c->stash->{object};
    $c->require_permission( 'serverhw', 'edit' );

    if ( $object->in_use ) {
        $c->response->redirect(
            $c->uri_for_action( 'serverhw/view', [ $c->stash->{object_pk} ] ) );
        $c->detach();
    }

    if ( $c->req->method eq 'POST' ) {
        $object->decommission;
        $object->update();
        $c->flash( message => "Server decommissioned" );
        $c->response->redirect(
            $c->uri_for_action( 'serverhw/view', [ $c->stash->{object_pk} ] ) );
        $c->detach();
    }

    # show confirm page
    $c->stash(
        title           => 'Decommission server hardware',
        confirm_message => 'Decommission server hardware ' . $object->label . '?',
        template        => 'generic_confirm.tt',
    );
}


sub restore : Chained('object') : PathPart('restore') : Args(0) {
    my ( $self, $c ) = @_;

    my $serverhw = $c->stash->{object};
    $c->require_permission( $serverhw, 'edit' );

    if ( !$serverhw->is_decommissioned ) {
        $c->response->redirect( $c->uri_for_action( 'serverhw/view', [ $serverhw->id ] ) );
        $c->detach();
    }

    if ( $c->req->method eq 'POST' ) {
        $serverhw->restore;
        $serverhw->update();
        $c->flash( message => "Asset restored" );
        $c->response->redirect( $c->uri_for_action( 'serverhw/view', [ $serverhw->id ] ) );
        $c->detach();
    }

    # show confirm page
    $c->stash(
        title           => 'Restore server hardware',
        confirm_message => 'Restore ' . $serverhw->label . '?',
        template        => 'generic_confirm.tt',
    );
}

__PACKAGE__->meta->make_immutable;

1;
# Local Variables:
# mode: cperl
# indent-tabs-mode: nil
# cperl-indent-level: 4
# cperl-indent-parens-as-block: t
# End:

__END__

=pod

=head1 NAME

App::Manoc::Controller::ServerHW - ServerHW controller

=head1 VERSION

version 2.99.3

=head1 ACTIONS

=head2 create

=head2 edit

=head2 import_csv

Import a server hardware list from a CSV file

=head2 decommission

=head2 restore

=head1 AUTHORS

=over 4

=item *

Gabriele Mambrini <gmambro@cpan.org>

=item *

Enrico Liguori

=back

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2017 by Gabriele Mambrini.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
