#!perl -w
#
#    sjis.t
#
#    Copyright (c) 2000 Chetan Patil
#
#    This package is free software and is provided ``as is'' without
#    express or implied warranty. It may be used, redistributed and/or
#    modified under the terms of the Perl Artistic License
#    (see http://www.perl.com/perl/misc/Artistic.html)
#

require 5.6.0;
use strict;

use ExtUtils::testlib;
use utf8;

use ICU;
use CharacterSet::SJIS;

my ($read_characters, $read_encoding, $read_shouldpass, $read_shouldfail, $read_testcase);
my (@tests, %characters, %encoding, %pass, %fail, %lines);
my $testname;
while (<DATA>) {
    if (/^TestName=(.+)$/) {
        $read_testcase = 1;
        $testname = $1;
        chomp($testname);
        push @tests, $testname;
        next;
    }
    if (/^Characters=(.*)$/) {
        next unless (defined $testname);
        $read_characters = 1;
        next unless (defined $1);
        $characters{$testname} = $1;
        chomp($characters{$testname});
        next;
    }
    if (/^CharacterEncoding=(.+)$/) {
        next unless (defined $testname);
        $read_encoding = 1;
        next unless (defined $1);
        $encoding{$testname} = $1;
        chomp($encoding{$testname});
        next;
    }
    if (/^ShouldPass=(.*)$/) {
        next unless (defined $testname);
        $read_shouldpass = 1;
        next unless (defined $1);
        $pass{$testname} = $1;
        chomp($pass{$testname});
        next;
    }
    if (/^ShouldFail=(.*)$/) {
        next unless (defined $testname);
        $read_shouldfail = 1;
        next unless (defined $1);
        $fail{$testname} = $1;
        chomp($fail{$testname});
        next;
    }
    if (/^END_TEST (\d+)$/) {
        my ($line) = $1;
        $lines{$testname} = $line;
        unless (
                (defined $read_testcase) &&
                (defined $read_characters) &&
                (defined $read_encoding)   &&
                (defined $read_shouldpass) &&
                (defined $read_shouldfail) 
                ) {
            die "Error while parsing DATA at $line (hint: inspect the Test structure immediately above this line)\n";
        }
        $read_testcase = undef;
        $read_characters = undef;
        $read_encoding   = undef;
        $read_shouldpass = undef;
        $read_shouldfail = undef;
        $testname = undef;
        next;
    }
}

my $num_test = scalar(@tests);
print "1..$num_test\n";
print "Begining SJIS tests:\n";

my $sjis = new CharacterSet::SJIS;

my ($func, $funcname);
$num_test = 1;
foreach my $test (@tests) {
    my $ok = 0; my $fail = 0; my $error = ""; my $ret;
    {
        no strict "refs";
        foreach $funcname (split(",",$pass{$test})) {            
            next unless ((defined $funcname) && ($funcname ne ''));
            # print STDERR "\nDoing $funcname for $test\n";
            $func = get_method_ref($sjis, $funcname);
            eval {
                $ret = &$func($characters{$test});
            };
            if ($@) { $fail++; $error .= "$funcname ($@) ";}
            if ($ret) { $ok++; } else { $fail++; $error .= "\n\n\n**** shouldPass $funcname (failed test) ";}
        }
        foreach $funcname (split(",",$fail{$test})) {
            next unless ((defined $funcname) && ($funcname ne ''));
            # print STDERR "\nDoing $funcname $test\n";
            $func = get_method_ref($sjis, $funcname);
            eval {
                $ret = &$func($characters{$test});
            };
            if ($@) { $fail++; $error .= "$funcname ($@) ";}
            if ($ret == 0) { $ok++; } else { $fail++; $error .= "\n\n\n**** shouldFail $funcname (failed test) "}
        }
    }
    if ($fail) {
        print STDERR "$test failed - $error\n";
        print "not ok $num_test\n";
    } else {
        print "ok $num_test\n";
    }
    $num_test++;
}

sub get_method_ref {
    my ($self, $method) = @_;
    return sub { return $self->$method(@_) };
}
__DATA__
TestName=Single-Pass-1(Singlebyte ASCII Only)
Characters=0123456789!"#$%&'()=-*<>?_*+:;@ABCGHIJKLMXYZaopqrstuvwxyz
CharacterEncoding=UTF8
ShouldPass=isSingle,isMultiKanji
ShouldFail=isDoubleKatakana
END_TEST __LINE__
TestName=Single-Fail-1(Singlebyte ASCII + Singlebyte Katakana)
Characters=XYZ@a1.co.jp;  ｱｲｳｴｵﾜｦﾝ
CharacterEncoding=UTF8
ShouldPass=
ShouldFail=isSingle,isDoubleKatakana,isMultiKanji
END_TEST __LINE__
TestName=Single-Fail-2(Singlebyte ASCII + Doublebyte Katakana)
Characters=XYZ@a1.co.jp;  Doublebyte Katakana アイウエオ
CharacterEncoding=UTF8
ShouldPass=isMultiKanji
ShouldFail=isSingle,isDoubleKatakana
END_TEST __LINE__
TestName=Single-Fail-3(Singlebyte ASCII + Doublebyte NonKanji)
Characters=XYZ@a1.co.jp;  Doublebyte NonKanji Ａ１あア●√
CharacterEncoding=UTF8
ShouldPass=isMultiKanji
ShouldFail=isSingle,isDoubleKatakana
END_TEST __LINE__
TestName=Single-Fail-4(Singlebyte ASCII + Doublebyte JIS-Level1)
Characters=XYZ@a1.co.jp;  Doublebyte JIS-Level1 日本
CharacterEncoding=UTF8
ShouldPass=isMultiKanji
ShouldFail=isSingle,isDoubleKatakana
END_TEST __LINE__
TestName=Single-Fail-5(Singlebyte ASCII + Doublebyte JIS-Level2)
Characters=XYZ@a1.co.jp;  Doublebyte JIS-Level2 梵鶯
CharacterEncoding=UTF8
ShouldPass=isMultiKanji
ShouldFail=isSingle,isDoubleKatakana
END_TEST __LINE__
TestName=Single-Fail-6(Singlebyte ASCII + Doublebyte Other)
Characters=XYZ@a1.co.jp;  Doublebyte Other ⅸ髙
CharacterEncoding=CharacterEncoding=UTF8
ShouldPass=
ShouldFail=isSingle,isMultiKanji,isDoubleKatakana
END_TEST __LINE__
TestName=Multi-Pass-1(Singlebyte ASCII Only)
Characters=0123456789!"#$%&'()=-*<>?_*+:;@ABCGHIJKLMXYZaopqrstuvwxyz
CharacterEncoding=UTF8
ShouldPass=isSingle,isMultiKanji
ShouldFail=isDoubleKatakana
END_TEST __LINE__
TestName=Multi-Pass-2(Doublebyte NonKanji JIS-Level1 JIS-Level2)
Characters=ＤＯＵＢＬＥ　ＯＮＬＹ Ａ１あア●√　漢字
CharacterEncoding=UTF8
ShouldPass=isMultiKanji
ShouldFail=isSingle,isDoubleKatakana
END_TEST __LINE__
TestName=Multi-Pass-3(Singlebyte ASCII + Doublebyte Katakana)
Characters=XYZ@a1.co.jp;  Doublebyte Katakana アイウエオ
CharacterEncoding=UTF8
ShouldPass=isMultiKanji
ShouldFail=isSingle,isDoubleKatakana
END_TEST __LINE__
TestName=Multi-Pass-4(Singlebyte ASCII + Doublebyte NonKanji)
Characters=XYZ@a1.co.jp; Ａ１あア●√
CharacterEncoding=UTF8
ShouldPass=isMultiKanji
ShouldFail=isSingle,isDoubleKatakana
END_TEST __LINE__
TestName=Multi-Pass-5(Singlebyte ASCII + Doublebyte NonKanji JIS-Level1)
Characters=XYZ@a1.co.jp;  Doublebyte NonKanji Ａ１あア●√　JIS-Level1 日本
CharacterEncoding=UTF8
ShouldPass=isMultiKanji
ShouldFail=isSingle,isDoubleKatakana
END_TEST __LINE__
TestName=Multi-Pass-6(Singlebyte ASCII + Doublebyte NonKanji JIS-Level1 JIS-Level2)
Characters=XYZ@a1.co.jp;  Doublebyte NonKanji Ａ１あア●√　JIS-Level1 日本　JIS-Level2 梵鶯
CharacterEncoding=UTF8
ShouldPass=isMultiKanji
ShouldFail=isSingle,isDoubleKatakana
END_TEST __LINE__
TestName=Multi-Fail-1(Singlebyte + Singlebyte Katakana)
Characters=XYZ@a1.co.jp;  Singlebyte Katakana ｱｲｳｴｵﾜｵﾝ
CharacterEncoding=UTF8
ShouldPass=
ShouldFail=isSingle,isMultiKanji,isDoubleKatakana
END_TEST __LINE__
TestName=Multi-Fail-2(Doublebyte + Singlebyte Katakana)
Characters=XYZ@a1.co.jp;  Double/Singlebyte Katakana アｱイｲウｳエｴオｵﾜｦﾝ
CharacterEncoding=CharacterEncoding=UTF8
ShouldPass=
ShouldFail=isSingle,isMultiKanji,isDoubleKatakana
END_TEST __LINE__
TestName=Multi-Fail-3(Multibyte + Doublebyte Other)
Characters=XYZ@a1.co.jp; Ａ１あア●√ Doublebyte Other ⅸ髙
CharacterEncoding=UTF8
ShouldPass=
ShouldFail=isSingle,isMultiKanji,isDoubleKatakana
END_TEST __LINE__
TestName=Katakana-Pass-1(Doublebyte Katakana Only)
Characters=アイウエオワヲンパバ
CharacterEncoding=CharacterEncoding=UTF8
ShouldPass=isMultiKanji,isDoubleKatakana
ShouldFail=isSingle
END_TEST __LINE__
TestName=Katakana-Fail-1(Doublebyte Katakana + Singlebyte ASCII)
Characters=0123456789!"#$%&'()=-*<>?_*+:;@ABCGHIJKLMXYZaopqrstuvwxyz
CharacterEncoding=UTF8
ShouldPass=isSingle,isMultiKanji
ShouldFail=isDoubleKatakana
END_TEST __LINE__
TestName=Katakana-Fail-2(Doublebyte Katakana + Doublebyte NonKanji)
Characters=Ａ１あア●√
CharacterEncoding=CharacterEncoding=UTF8
ShouldPass=isMultiKanji
ShouldFail=isSingle,isDoubleKatakana
END_TEST __LINE__
TestName=Katakana-Fail-3(Doublebyte Katakana + Doublebyte JIS-Level1)
Characters=アイウ日本エオ
CharacterEncoding=UTF8
ShouldPass=isMultiKanji
ShouldFail=isSingle,isDoubleKatakana
END_TEST __LINE__
TestName=Katakana-Fail-4(Doublebyte Katakana + Doublebyte JIS-Level2)
Characters=アイウ梵鶯エオ
CharacterEncoding=UTF8
ShouldPass=isMultiKanji
ShouldFail=isSingle,isDoubleKatakana
END_TEST __LINE__
TestName=Katakana-Fail-5(Doublebyte Katakana + Doublebyte Katakana)
Characters=アｱイｲウｳエｴオｵﾜｦﾝワヲン
CharacterEncoding=UTF8
ShouldPass=
ShouldFail=isSingle,isMultiKanji,isDoubleKatakana
END_TEST __LINE__
TestName=Katakana-Fail-6(Doublebyte Katakana +Doublebyte Other)
Characters=アイウエオⅸ髙ワヲン
CharacterEncoding=UTF8
ShouldPass=
ShouldFail=isSingle,isMultiKanji,isDoubleKatakana
END_TEST __LINE__
