use Test::Lib;
use My::Test;
use Statocles::Util qw( trim dircopy run_editor uniq_by derp read_stdin );
use Statocles::Link;
my $SHARE_DIR = path( __DIR__, 'share' );

use constant Win32 => $^O =~ /Win32/;

subtest 'trim' => sub {
    my $foo;
    my @warnings;
    local $SIG{__WARN__} = sub { push @warnings, @_ };

    $foo = ' foo',
    is trim $foo, 'foo', 'leading whitespace is trimmed';

    $foo = "foo\t\n";
    is trim $foo, 'foo', 'trailing whitespace is trimmed';

    $foo = ' foo ';
    is trim $foo, 'foo', 'leading and trailing whitespace is trimmed';

    $foo = '';
    is trim $foo, '', 'empty string is passed through';

    $foo = undef;
    is trim $foo, undef, 'undef is passed through';

    local $_ = " foo ";
    trim;
    is $_, "foo", '$_ is trimmed';

    is scalar @warnings, 0, 'no warnings about anything'
        or diag "Got warnings: " . join "\n", @warnings;
};

subtest 'dircopy' => sub {
    my $tmp_source = tempdir;
    $tmp_source->child( qw( fuzz.txt ) )->spew( 'World' );
    $tmp_source->child( qw( foo bar.txt ) )->touchpath->spew( 'Hello' );

    my $tmp_dest = tempdir;
    dircopy $tmp_source, $tmp_dest;
    ok $tmp_dest->child( 'fuzz.txt' )->is_file;
    ok $tmp_dest->child( 'foo' )->is_dir;
    ok $tmp_dest->child( qw( foo bar.txt ) )->is_file;

    subtest 'dir does not exist yet' => sub {
        my $tmp_dest = tempdir;

        dircopy $tmp_source, $tmp_dest->child( 'missing' );
        ok $tmp_dest->child( qw( missing fuzz.txt ) )->is_file;
        ok $tmp_dest->child( qw( missing foo ) )->is_dir;
        ok $tmp_dest->child( qw( missing foo bar.txt ) )->is_file;

    };
};

subtest 'run_editor' => sub {
    my $editor = join ' ', map qq{"$_"}, $^X, $SHARE_DIR->child( 'bin', 'editor.pl' );
    subtest 'no editor found' => sub {
        local $ENV{EDITOR};
        my $tmp = tempdir;
        ok !run_editor( $tmp->child( 'index.markdown' ) ), 'no editor, so return false';
    };

    subtest 'editor found' => sub {
        my $path = $SHARE_DIR->child(qw( store docs required.markdown ));
        local $ENV{EDITOR} = $editor;
        local $ENV{STATOCLES_TEST_EDITOR_CONTENT} = "".$path;
        my $tmp = tempdir;
        ok my $content = run_editor( $tmp->child( 'index.markdown' ) ), 'editor invoked, so return true';
        is $content, $path->slurp_utf8, 'run_editor returns edited content';
    };

    subtest 'editor set but invalid' => sub {
        local $ENV{EDITOR} = "HOPEFULLY_DOES_NOT_EXIST";
        my $tmp = tempdir;
        throws_ok {
            run_editor( $tmp->child( 'index.markdown' ) );
        } qr{Editor "HOPEFULLY_DOES_NOT_EXIST" exited with error \(non-zero\) status: \d+\n};
    };

    if (!Win32) {
        subtest 'editor dies by signal' => sub {
            local $ENV{EDITOR} = $editor . " --signal TERM";
            my $tmp = tempdir;
            throws_ok {
                run_editor( $tmp->child( 'index.markdown' ) );
            } qr[Editor "\Q$ENV{EDITOR}\E" exited with error \(non-zero\) status: \d+\n];
        };
    }

    subtest 'editor nonzero exit' => sub {
        local $ENV{EDITOR} = $editor . " --exit 1";
        my $tmp = tempdir;
        throws_ok {
            run_editor( $tmp->child( 'index.markdown' ) );
        } qr[Editor "\Q$ENV{EDITOR}\E" exited with error \(non-zero\) status: \d+\n];
    };
};

subtest 'read_stdin' => sub {
    local $TODO;
    $TODO = 'Only working intermittently on Travis for unknown reason on 5.12-18'
        if $ENV{TRAVIS} and ($] >= 5.012 and $] < 5.020);
    my $content = "Content on STDIN\n";
    open my $stdin, '<', \$content or die "Could not create scalar filehandle: $!";
    local *STDIN = $stdin;
    is read_stdin(), $content, 'STDIN content is correct';
};

subtest 'uniq_by' => sub {
    my @links = map { Statocles::Link->new( href => $_ ) } qw(
        /foo.html
        /bar.html
        /baz.html
        /foo.html
    );

    cmp_deeply [ uniq_by { $_->href } @links ], [ @links[0,1,2] ];
};

subtest 'derp' => sub {
    my @warnings;
    local $SIG{__WARN__} = sub { push @warnings, @_ };

    subtest 'one argument' => sub {
        subtest 'first call warns' => sub {
            derp 'Foo';
            is scalar @warnings, 1, '1 warning issued';
            is $warnings[-1], "Foo. See Statocles::Help::Upgrading\n",
                'derp message directs to upgrading guide';
            @warnings = ();
        };

        subtest 'second call with same args does not warn' => sub {
            derp 'Foo';
            is scalar @warnings, 0, "doesn't warn for same text a second time";
            @warnings = ();
        };
    };

    subtest 'many arguments' => sub {
        subtest 'first call warns' => sub {
            derp 'Foo %s', 'Bar';
            is scalar @warnings, 1, '1 warning issued';
            is $warnings[-1], "Foo Bar. See Statocles::Help::Upgrading\n",
                'derp message directs to upgrading guide';
            @warnings = ();
        };

        subtest 'second call with same args does not warn' => sub {
            derp 'Foo %s', 'Bar';
            is scalar @warnings, 0, "doesn't warn for same args a second time";
            @warnings = ();
        };

        subtest 'second call with different args warns' => sub {
            derp 'Foo %s', 'Baz';
            is scalar @warnings, 1, '1 warning issued';
            is $warnings[-1], "Foo Baz. See Statocles::Help::Upgrading\n",
                'derp message directs to upgrading guide';
            @warnings = ();
        };
    };

};

done_testing;
