#=head1 NAME

C<Parse::Lex> - Gnrateur d'analyseurs lexicaux

#=head1 SYNOPSIS

	require 5.005;

	use Parse::Lex;
	@token = (
	  qw(
	     ADDOP    [-+]
	     LEFTP    [\(]
	     RIGHTP   [\)]
	     INTEGER  [1-9][0-9]*
	     NEWLINE  \n
	     
	    ),
	  qw(STRING),   [qw(" (?:[^"]+|"")* ")],
	  qw(ERROR  .*), sub {
	    die qq!can\'t analyze: "$_[1]"!;
	  }
	 );

	Parse::Lex->trace;  # Class method
	$lexer = Parse::Lex->new(@token);
	$lexer->from(\*DATA);
	print "Tokenization of DATA:\n";

	TOKEN:while (1) {
	  $token = $lexer->next;
	  if (not $lexer->eoi) {
	    print "Line $.\t";
	    print "Type: ", $token->name, "\t";
	    print "Content:->", $token->text, "<-\n";
	  } else {
	    last TOKEN;
	  }
	}

	__END__
	1+2-5
        "a multiline
        string with an embedded "" in it"
        an invalid string with a "" in it"

#=head1 DESCRIPTION

Les classes C<Parse::Lex> et C<Parse::CLex> permettent de crer des
analyseurs lexicaux. Elles exploitent des techniques d'analyse
diffrentes :

1. C<Parse::Lex> fait progresser l'analyse en dplaant un pointeur
dans les chanes de caractres  analyser (utilisation de C<pos()>
associ  C<\G>),

2. C<Parse::CLex> fait progresser l'analyse en consommant
les donnes reconnues (utilisation de C<s///>).

Les analyseurs de la classe C<Parse::CLex> ne permettent pas
d'utiliser des expressions rgulires avec ancrage. De plus les
sous-classes de C<Parse::Token> ne sont pas implmentes pour ce type
d'analyseur.

Un analyseur lexical est spcifi au moyen d'un liste de tokens passe
en argument  la mthode C<new()>.  Les tokens sont des instances de la
classe C<Parse::Token>, livr avec C<Parse::Lex>. La dfinition d'un
token comporte ordinairement deux arguments : un nom symbolique (comme
C<INTEGER>), suivi d'une expression rgulire. Si une fonction anonyme
est donne en troisime argument, elle est excute lorsque le token
est reconnu.  Elle reoit en argument l'instance C<Parse::Token>, suivie
de la chane reconnue par l'expression rgulire. Le scalaire retourn
par la fonction anonyme dfinit la chane de caractres place dans
l'instance C<Parse::Token>.

L'ordre dans lequel l'analyseur lexical examine les expressions
rgulires est dtermin par l'ordre dans lequel ces expressions sont
passes en argument  la mthode C<new()>. Le token retourn par
l'analyseur lexical correspond  la premire expression qui s'apparie
(la stratgie est diffrente de celle utilis par Lex qui retourne la
plus longue chane de toutes celles qu'il est possible de
reconnatre). 

Les analyseurs lexicaux peuvent reconnatre des tokens disposs sur
plusieurs enregistrements. Si la dfinition du token comporte
plusieurs expressions rgulires, et est place dans un tableau
anonyme, l'analyseur lit autant d'enregistrements que ncessaire pour
reconnatre le token (voir la documentation de la classe
C<Parse::Token>). Lorsque que le motif du dbut est trouv,
l'analyseur en cherche la fin, et si besoin est, lit de nouveaux
enregistrements. Il n'y a pas de rebroussement en cas d'chec.

L'analyseur peut tre utilis pour analyser une chane de caractres
isole ou un flot de donnes provenant d'une entre quelconque.
Lorsque l'analyseur a puis les donnes il retourne une instance
C<Parse::Token> dont le nom est C<EOI> (End Of Input).

#=head2 Conditions

Il est possible d'associer des conditions de dclenchement aux rgles
de reconnaissance des tokens qui composent votre analyseur lexical (
la manire de ce l'on trouve dans FLEX). Grce aux conditions la rgle
qui russit n'est plus obligatoirement la premire qui s'apparie.

Toute dsignation de symbole peut tre prcde par la spcification
de conditions d'activation de la rgle de reconnaissance associe. Par
exemple :

	qw(C1:TERMINAL_1  REGEXP), sub { # associated action },
	qw(TERMINAL_2  REGEXP), sub { # associated action },


Le symbole C<TERMINAL_1> ne sera reconnu que si la condition C<C1> est active.
L'activation/dsactivation s'opre respectivement aux moyens des
mthodes C<start(NOM_CONDITION)> et C<end(NOM_CONDITION)>.

C<start('INITIAL')> permet de remettre  zro l'automate d'analyse.

Les conditions peuvent tre combines aux moyens d'oprateurs ET/OU
comme suit : 

	C1:SYMBOL      condition C1

	C1:C2:SYMBOL   condition C1 ET condition C2

	C1,C2:SYMBOL   condition C1 OU condition C2


Il existe deux types de conditions : les conditions I<inclusives> et
les conditions I<exclusives>, respectivement dclares par les
mthodes de classe C<inclusive()> et C<exclusive()>. Avec une
condition inclusive les rgles actives sont celles qui comportent la
condition, ainsi que celles qui n'en comportent pas du tout.  Avec une
condition exclusive, seules les rgles qui comportent cette condition
sont actives. Toutes les autres sont dsactives.

Exemple (emprunt  la documentation de FLEX).

 use Parse::Lex;
 @token = (
	  'EXPECT', 'expect-floats', sub {
	    $lexer->start('expect'); 
	    $_[1] 
	  },
	  'expect:FLOAT', '\d+\.\d+', sub { 
	    print "found a float: $_[1]\n";
	    $_[1] 
	  },
	  'expect:NEWLINE', '\n', sub { 
	    $lexer->end('expect') ;
	    $_[1] 
	  },
	  'NEWLINE2', '\n',
	  'INT', '\d+', sub {
	    print "found an integer: $_[1] \n";
	    $_[1] 
	  },
	  'DOT', '\.', sub {
	    print "found a dot\n";
	    $_[1] 
	  },
	 );

 Parse::Lex->exclusive('expect');
 $lexer = Parse::Lex->new(@token);

La condition spciale C<ALL> est toujours vrifie.

#=head2 Mthodes

#=over 4

#=item analyze EXPR

Analyse C<EXPR> et retourne une liste de couples composs d'un nom de
token suivi du texte reconnu. C<EXPR> peut tre une chane de caractres
ou une rfrence  un filehandle.

Exemples.

 @tokens = Parse::Lex->new(qw(PLUS [+] NUMBER \d+))->analyze("3+3+3");
 @tokens = Parse::Lex->new(qw(PLUS [+] NUMBER \d+))->analyze(\*STREAM);

#=item buffer EXPR

#=item buffer

Retourne le contenu du buffer interne  l'analyseur lexical. Avec une
expression en argument, place le rsultat de l'expression dans le
buffer.

Il n'est pas conseill de changer directement le contenu du buffer
sans changer la position du pointeur qui suit l'analyse (C<pos()>) et
la valeur de la longueur du buffer (C<length()>).

#=item configure(HASH)

Mthode d'instance permettant de spcifier un analyseur lexical. Cette
mthode accepte la liste des attributs-valeurs suivants :

#=over 10

#=item From => EXPR

Cet attribut joue le mme rle que la mthode C<from(EXPR)>.
C<EXPR> peut tre un filehandle ou une chane de caractres.

#=item Tokens => ARRAY_REF

C<ARRAY_REF> doit contenir la liste des attributs-valeurs spcifiant
les tokens  reconnatre (voir la documentation de C<Parse::Token>).

#=item Skip => REGEX

Cet attribut joue le mme rle que la mthode C<skip(REGEX)>. C<REGEX>
dcrit les motifs  sauter lors de l'analyse.

#=over 4

#=item end EXPR

Dsactive la condition C<EXPR>.

#=item eoi

Retourne VRAI lorsqu'il n'y a plus de donnes  analyser.

#=item every SUB

vite de devoir crire une boucle de lecture pour analyser un flot de
donnes.  C<SUB> est une fonction anonyme excute aprs la
reconnaissance de chaque token. Par exemple, pour analyser la chane
C<"1+2"> vous pouvez crire : 

	use Parse::Lex;

	$lexer = Parse::Lex->new(
	  qw(
	     ADDOP [-+]
	     INTEGER \d+
	    ));

	$lexer->from("1+2");
	$lexer->every (sub { 
	  print $_[0]->name, "\t";
	  print $_[0]->text, "\n"; 
	});

Le premier argument de la fonction anonyme est l'instance
C<Parse::Token> reconnue.

#=item exclusive LISTE

Mthode de classe dclarant comme I<exclusives> les conditions
prsentes dans C<LISTE>.

#=item flush

Si la conservation des chanes consommes est active, C<flush()>
retourne et vide le buffer contenant les chanes de caractres
reconnues jusqu'ici. Utile seulement si vous tes dans le mode activ
par C<hold()>.

#=item from EXPR

#=item from

C<from(EXPR)> permet d'indiquer les donnes  analyser ou la source
des donnes  analyser.  L'argument de cette mthode est donc, soit
une chane de caractre (ou une liste), soit une rfrence  un
filehandle. Avec un argument C<from()> retourne l'objet receveur. Sans
argument retourne le filehandle s'il est dfini, sinon C<undef>.

Par dfaut on suppose que les donnes sont lues sur C<STDIN>.

Exemples.

	$handle = new IO::File;
	$handle->open("< filename");
	$lexer->from($handle);

	$lexer->from(\*DATA);
	$lexer->from('les donnes  analyser');

#=item getSub

C<getSub()> retourne la fonction anonyme qui effectue l'analyse
lexicale.

Exemple.

	my $token = '';
	my $sub = $lexer->getSub;
	while (($token = &$sub()) ne $Token::EOI) {
	  print $token->name, "\t";
	  print $token->text, "\n";
	}
    
   # or 
	
	my $token = '';
	local *tokenizer = $lexer->getSub;
	while (($token = tokenizer()) ne $Token::EOI) {
	  print $token->name, "\t";
	  print $token->text, "\n";
	}

#=item getToken

Synonyme de la mthode C<token()>.

#=item hold EXPR

#=item hold

Active/dsactive la conservation des chanes analyses (et consommes
dans le cas de C<Parse::CLex>).  Retourne la valeur courante. Peut
tre utilise comme mthode de classe.

On peut obtenir le contenu du buffer au moyen de la mthode C<flush()>
qui a galement pour effet de vider le buffer. 

#=item inclusive LISTE

Mthode de classe dclarant comme I<inclusives> les conditions
prsentes dans C<LISTE>.

#=item length EXPR

#=item length

Retourne la longueur de l'enregistrement courant.  C<length
EXPR> fixe la longueur de cet enregistrement.

#=item line EXPR

#=item line

Retourne le numro de l'enregistrement courant.  C<line EXPR>
permet de fixer ce numro.  Retourne toujours 1 si on analyse une
chane de caractres. La mthode C<readline()> incrmente le numro de
ligne.

#=item name EXPR

#=item name

Permet de donner un nom  un analyseur lexical. C<name()> permet de
connatre ce nom.

#=item next

Provoque la recherche du prochain token. Retourne l'instance C<Parse::Token>
reconnue.  Retourne l'instance C<Token::EOI> en fin de donnes.

Exemples.

	$lexer = Parse::Lex->new(@token);
	print $lexer->next->name;   # print the token type
	print $lexer->next->text;   # print the token content

#=item nextis SCALAR_REF

Variable de la mthode C<next()>. Les tokens sont placs dans
C<SCALAR_REF>. La mthode retourne 1 tant que le token n'est C<EOI>.

Exemple.

	while($lexer->nextis(\$token)) {
	   print $token->text();
	}

#=item new LISTE

Cre et retourne un nouvel analyseur lexical. L'argument de la mthode
est une liste d'instances de la classe C<Parse::Token> ou de triplets
permettant de les crer. Ces triplets sont constitus du nom
symbolique du token, de l'expression rgulire ncessaire  sa
reconnaissance et ventuellement d'une fonction anonyme excute lors
de la reconnaissance du token. Pour chaque triplet, une instance de
type C<Parse::Token> est cre dans le package appelant.

#=item offset

Retourne le nombre de caractres dj consomm depuis le dbut du flot
de donnes analys.

#=item pos EXPR

#=item pos

C<pos EXPR> fixe la position de dbut du prochain token  reconnatre
dans l'enregistrement courant (ne fonctionne pas avec les analyseurs
de la classe C<Parse::CLex>). C<pos()> retourne le nombre de caractres
dj consomms dans l'enregistrement courant.

#=item readline

Effectue la lecture des donnes sur l'entre spcifie par la
mthode C<from()>. Retourne le rsultat de la lecture.

Exemple.

	use Parse::Lex;

	$lexer = Parse::Lex->new();
	while (not $lexer->eoi) {
	  print $lexer->readline() # read and print one line
	}

#=item reset

Vide le buffer interne  l'analyseur lexical et efface tout token dj
reconnu.

#=item restart

Rinitialise l'automate d'analyse. La seule condition active devient
la condition C<INITIAL>.

#=item setToken TOKEN

Force le token  C<TOKEN>. Utile pour requalifier un token 
l'intrieur de la fonction anonyme associe  ce token.

#=item skip EXPR

#=item skip

C<EXPR> est une expression rgulire dfinissant un motif inter-token
(par dfaut C<[ \t]+>). C<skip('')> permet de supprimer ce
motif. C<skip()> retourne la valeur du motif. C<skip()> peut tre
utilise comme mthode de classe.  

Le changement du motif  "sauter" provoque une recompilation de
l'analyseur lexical.

Exemple.

  Parse::Lex->skip('\s*#(?s:.*)|\s+');
  @tokens = Parse::Lex->new('INTEGER' => '\d+')->analyze(\*DATA);
  print "@tokens\n"; # print INTEGER 1 INTEGER 2 INTEGER 3 INTEGER 4 EOI 
  __END__
  1 # first string to skip
  2
  3# second string to skip
  4


#=item start EXPR

Active la condition EXPR.

#=item state EXPR

C<state EXPR> permet de connatre l'tat de la condition reprsente
par EXPR.

#=item token

Retourne l'instance correspondant au dernier token reconnu. En l'absence
de token lu, retourne un token spcial dont le nom est C<DEFAULT>.

#=item tokenClass EXPR

#=item tokenClass 

Indique quelle est la classe des tokens  crer  partir de la liste
passe en argument  la mthode C<new()>. Sans argument retourne le
nom de cette classe.  Par dfaut la classe est C<Parse::Token>.

#=item trace OUTPUT

#=item trace 

Mthode de classe qui active le mode trace.  L'activation du mode
trace doit avoir lieu avant la cration de l'analyseur lexical. Le
mode peut tre ensuite dsactiv par un nouvel appel de la mthode.

C<OUTPUT> peut tre un nom de fichier ou une rfrence  un
filehandle vers laquelle la trace va tre redirige.

#=back

#=head1 GESTION DES ERREURS

Pour traiter les cas de non reconnaissance de token vous pouvez
dfinir un token spcifique en fin de la liste des tokens composant
notre analyseur lexical.  Si la recherche de ce token russie il est
alors possible d'appeler une fonction de traitement des erreurs :

     qw(ERROR  (?s:.*)), sub {
       print STDERR "ERROR: buffer content->", $_[0]->lexer->buffer, "<-\n";
       die qq!can\'t analyze: "$_[1]"!;
     }

#=head1 EXEMPLES

ctokenizer.pl - Segmentation d'un flot de donnes au moyen
de la classe C<Parse::CLex>.

tokenizer.pl - Segmentation d'un flot de donnes au moyen
de la classe C<Parse::Lex>.

every.pl - Utilisation de la mthode C<every()>.

sexp.pl - Interprte d'expressions arithmtiques prfixes.

sexpcond.pl - Interprte d'expressions arithmtiques prfixes avec
utilisation des conditions.

#=head1 BUGS

Les analyseurs de la classe C<Parse::CLex> ne permettent pas
d'utiliser des expressions rgulires avec ancrage.

#=head1 VOIR EGALEMENT

C<Parse::Token>, C<Parse::LexEvent>, C<Parse::YYLex>.

#=head1 AUTEUR

Philippe Verdret 

#=head1 REMERCIEMENTS

La version 2.0 doit beaucoup aux suggestions de Vladimir Alexiev.
Ocrat a largement contribu  l'amlioration de cette documentation.
Merci galement aux nombreuses personnes qui m'ont envoy des rapports
de bugs et parfois des corrections.

#=head1 REFERENCES

Friedl, J.E.F. Mastering Regular Expressions. O'Reilly & Associates
1996.

Mason, T. & Brown, D. - Lex & Yacc. O'Reilly & Associates, Inc. 1990.

FLEX - A Scanner generator (voir par exemple ftp://ftp.ee.lbl.gov/)

#=head1 COPYRIGHT

Copyright (c) 1995-1999 Philippe Verdret. All rights reserved.
This module is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

