package Module::CoreList;
use strict;
use vars qw/$VERSION %released %version %families %upstream
	    %bug_tracker %deprecated/;
$VERSION = '2.50_02';

=head1 NAME

Module::CoreList - what modules shipped with versions of perl

=head1 SYNOPSIS

 use Module::CoreList;

 print $Module::CoreList::version{5.00503}{CPAN}; # prints 1.48

 print Module::CoreList->first_release('File::Spec');         # prints 5.00405
 print Module::CoreList->first_release_by_date('File::Spec'); # prints 5.005
 print Module::CoreList->first_release('File::Spec', 0.82);   # prints 5.006001

 print join ', ', Module::CoreList->find_modules(qr/Data/);
    # prints 'Data::Dumper'
 print join ', ', Module::CoreList->find_modules(qr/test::h.*::.*s/i, 5.008008);
    # prints 'Test::Harness::Assert, Test::Harness::Straps'

 print join ", ", @{ $Module::CoreList::families{5.005} };
    # prints "5.005, 5.00503, 5.00504"

=head1 DESCRIPTION

Module::CoreList provides information on which core and dual-life modules shipped
with each version of L<perl>.

It provides a number of mechanisms for querying this information.

There is a utility called L<corelist> provided with this module
which is a convenient way of querying from the command-line.

There is a functional programming API available for programmers to query
information.

Programmers may also query the contained hash structures to find relevant
information.

=head1 FUNCTIONS API

These are the functions that are available, they may either be called as functions or class methods:

  Module::CoreList::first_release('File::Spec'); # as a function

  Module::CoreList->first_release('File::Spec'); # class method

=over

=item C<first_release( MODULE )>

Behaviour since version 2.11

Requires a MODULE name as an argument, returns the perl version when that module first
appeared in core as ordered by perl version number or undef ( in scalar context )
or an empty list ( in list context ) if that module is not in core.

=item C<first_release_by_date( MODULE )>

Requires a MODULE name as an argument, returns the perl version when that module first
appeared in core as ordered by release date or undef ( in scalar context )
or an empty list ( in list context ) if that module is not in core.

=item C<find_modules( REGEX, [ LIST OF PERLS ] )>

Takes a regex as an argument, returns a list of modules that match the regex given.
If only a regex is provided applies to all modules in all perl versions. Optionally
you may provide a list of perl versions to limit the regex search.

=item C<find_version( PERL_VERSION )>

Takes a perl version as an argument. Returns that perl version if it exists or C<undef>
otherwise.

=item C<is_deprecated( MODULE, PERL_VERSION )>

Available in version 2.22 and above.

Returns true if MODULE is marked as deprecated in PERL_VERSION.  If PERL_VERSION is
omitted, it defaults to the current version of Perl.

=item C<removed_from( MODULE )>

Available in version 2.32 and above

Takes a module name as an argument, returns the first perl version where that module
was removed from core. Returns undef if the given module was never in core or remains
in core.

=item C<removed_from_by_date( MODULE )>

Available in version 2.32 and above

Takes a module name as an argument, returns the first perl version by release date where that module
was removed from core. Returns undef if the given module was never in core or remains
in core.

=back

=head1 DATA STRUCTURES

These are the hash data structures that are available:

=over

=item C<%Module::CoreList::version>

A hash of hashes that is keyed on perl version as indicated
in $].  The second level hash is module => version pairs.

Note, it is possible for the version of a module to be unspecified,
whereby the value is C<undef>, so use C<exists $version{$foo}{$bar}> if
that's what you're testing for.

Starting with 2.10, the special module name C<Unicode> refers to the version of
the Unicode Character Database bundled with Perl.

=item C<%Module::CoreList::released>

Keyed on perl version this contains ISO
formatted versions of the release dates, as gleaned from L<perlhist>.

=item C<%Module::CoreList::families>

New, in 1.96, a hash that
clusters known perl releases by their major versions.

=item C<%Module::CoreList::deprecated>

A hash of hashes keyed on perl version and on module name.
If a module is defined it indicates that that module is
deprecated in that perl version and is scheduled for removal
from core at some future point.

=item C<%Module::CoreList::upstream>

A hash that contains information on where patches should be directed
for each core module.

UPSTREAM indicates where patches should go. C<undef> implies
that this hasn't been discussed for the module at hand.
C<blead> indicates that the copy of the module in the blead
sources is to be considered canonical, C<cpan> means that the
module on CPAN is to be patched first. C<first-come> means
that blead can be patched freely if it is in sync with the
latest release on CPAN.

=item C<%Module::CoreList::bug_tracker>

A hash that contains information on the appropriate bug tracker
for each core module.

BUGS is an email or url to post bug reports.  For modules with
UPSTREAM => 'blead', use perl5-porters@perl.org.  rt.cpan.org
appears to automatically provide a URL for CPAN modules; any value
given here overrides the default:
http://rt.cpan.org/Public/Dist/Display.html?Name=$ModuleName

=back

=head1 CAVEATS

Module::CoreList currently covers the 5.000, 5.001, 5.002, 5.003_07,
5.004, 5.004_05, 5.005, 5.005_03, 5.005_04, 5.6.0, 5.6.1, 5.6.2, 5.7.3,
5.8.0, 5.8.1, 5.8.2, 5.8.3, 5.8.4, 5.8.5, 5.8.6, 5.8.7, 5.8.8, 5.8.9,
5.9.0, 5.9.1, 5.9.2, 5.9.3, 5.9.4, 5.9.5, 5.10.0, 5.10.1, 5.11.0, 5.11.1,
5.11.2, 5.11.3, 5.11.4, 5.11.5, 5.12.0, 5.12.1, 5.12.2, 5.12.3, 5.12.4,
5.12.5, 5.13.0, 5.13.1, 5.13.2, 5.13.3, 5.13.4, 5.13.5, 5.13.6, 5.13.7,
5.13.8, 5.13.9, 5.13.10, 5.13.11, 5.14.0 and 5.14.1 releases of perl.

=head1 HISTORY

Moved to Changes file.

=head1 AUTHOR

Richard Clamp E<lt>richardc@unixbeard.netE<gt>

Currently maintained by the perl 5 porters E<lt>perl5-porters@perl.orgE<gt>.

=head1 LICENSE

Copyright (C) 2002-2009 Richard Clamp.  All Rights Reserved.

This module is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.

=head1 SEE ALSO

L<corelist>, L<Module::Info>, L<perl>

=cut

my $dumpinc = 0;
sub import {
    my $self = shift;
    my $what = shift || '';
    if ($what eq 'dumpinc') {
        $dumpinc = 1;
    }
}

END {
    print "---INC---\n", join "\n" => keys %INC
      if $dumpinc;
}


sub first_release_raw {
    my $module = shift;
    $module = shift if eval { $module->isa(__PACKAGE__) }
      and scalar @_ and $_[0] =~ m#\A[a-zA-Z_][0-9a-zA-Z_]*(?:(::|')[0-9a-zA-Z_]+)*\z#;
    my $version = shift;

    my @perls = $version
        ? grep { exists $version{$_}{ $module } &&
                        $version{$_}{ $module } ge $version } keys %version
        : grep { exists $version{$_}{ $module }             } keys %version;

    return @perls;
}

sub first_release_by_date {
    my @perls = &first_release_raw;
    return unless @perls;
    return (sort { $released{$a} cmp $released{$b} } @perls)[0];
}

sub first_release {
    my @perls = &first_release_raw;
    return unless @perls;
    return (sort { $a cmp $b } @perls)[0];
}

sub find_modules {
    my $regex = shift;
    $regex = shift if eval { $regex->isa(__PACKAGE__) };
    my @perls = @_;
    @perls = keys %version unless @perls;

    my %mods;
    foreach (@perls) {
        while (my ($k, $v) = each %{$version{$_}}) {
            $mods{$k}++ if $k =~ $regex;
        }
    }
    return sort keys %mods
}

sub find_version {
    my $v = shift;
    $v = shift if eval { $v->isa(__PACKAGE__) };
    return $version{$v} if defined $version{$v};
    return undef;
}

sub is_deprecated {
    my $module = shift;
    $module = shift if eval { $module->isa(__PACKAGE__) }
      and scalar @_ and $_[0] =~ m#\A[a-zA-Z_][0-9a-zA-Z_]*(?:(::|')[0-9a-zA-Z_]+)*\z#;
    my $perl_version = shift;
    $perl_version ||= $];
    return unless $module && exists $deprecated{$perl_version}{$module};
    return $deprecated{$perl_version}{$module};
}

sub removed_from {
  my @perls = &removed_raw;
  return shift @perls;
}

sub removed_from_by_date {
  my @perls = sort { $released{$a} cmp $released{$b} } &removed_raw;
  return shift @perls;
}

sub removed_raw {
  my $mod = shift;
  $mod = shift if eval { $mod->isa(__PACKAGE__) }
      and scalar @_ and $_[0] =~ m#\A[a-zA-Z_][0-9a-zA-Z_]*(?:(::|')[0-9a-zA-Z_]+)*\z#;
  return unless my @perls = sort { $a cmp $b } first_release_raw($mod);
  my $last = pop @perls;
  my @removed = grep { $_ > $last } sort { $a cmp $b } keys %version;
  return @removed;
}

# When things escaped.
# NB. If you put version numbers with trailing zeroes here, you
# should also add an alias for the numerical ($]) version; see
# just before the __END__ of this module.
%released = (
    5.000    => '1994-10-17',
    5.001    => '1995-03-14',
    5.002    => '1996-02-29',
    5.00307  => '1996-10-10',
    5.004    => '1997-05-15',
    5.005    => '1998-07-22',
    5.00503  => '1999-03-28',
    5.00405  => '1999-04-29',
    5.006    => '2000-03-22',
    5.006001 => '2001-04-08',
    5.007003 => '2002-03-05',
    5.008    => '2002-07-19',
    5.008001 => '2003-09-25',
    5.009    => '2003-10-27',
    5.008002 => '2003-11-05',
    5.006002 => '2003-11-15',
    5.008003 => '2004-01-14',
    5.00504  => '2004-02-23',
    5.009001 => '2004-03-16',
    5.008004 => '2004-04-21',
    5.008005 => '2004-07-19',
    5.008006 => '2004-11-27',
    5.009002 => '2005-04-01',
    5.008007 => '2005-05-30',
    5.009003 => '2006-01-28',
    5.008008 => '2006-01-31',
    5.009004 => '2006-08-15',
    5.009005 => '2007-07-07',
    5.010000 => '2007-12-18',
    5.008009 => '2008-12-14',
    5.010001 => '2009-08-22',
    5.011000 => '2009-10-02',
    5.011001 => '2009-10-20',
    5.011002 => '2009-11-20',
    5.011003 => '2009-12-20',
    5.011004 => '2010-01-20',
    5.011005 => '2010-02-20',
    5.012000 => '2010-04-12',
    5.013000 => '2010-04-20',
    5.012001 => '2010-05-16',
    5.013001 => '2010-05-20',
    5.013002 => '2010-06-22',
    5.013003 => '2010-07-20',
    5.013004 => '2010-08-20',
    5.012002 => '2010-09-06',
    5.013005 => '2010-09-19',
    5.013006 => '2010-10-20',
    5.013007 => '2010-11-20',
    5.013008 => '2010-12-20',
    5.012003 => '2011-01-21',
    5.013009 => '2011-01-20',
    5.013010 => '2011-02-20',
    5.013011 => '2011-03-20',
    5.014000 => '2011-05-14',
    5.012004 => '2011-06-20',
    5.014001 => '2011-06-16',
    5.012005 => '2012-11-10',
  );

for my $version ( sort { $a <=> $b } keys %released ) {
    my $family = int ($version * 1000) / 1000;
    push @{ $families{ $family }} , $version;
}


%version = (
    5.000 => {
        'AnyDBM_File'           => undef,  # lib/AnyDBM_File.pm
        'AutoLoader'            => undef,  # lib/AutoLoader.pm
        'AutoSplit'             => undef,  # lib/AutoSplit.pm
        'Benchmark'             => undef,  # lib/Benchmark.pm
        'Carp'                  => undef,  # lib/Carp.pm
        'Cwd'                   => undef,  # lib/Cwd.pm
        'DB_File'               => undef,  # ext/DB_File/DB_File.pm
        'DynaLoader'            => undef,  # ext/DynaLoader/DynaLoader.pm
        'English'               => undef,  # lib/English.pm
        'Env'                   => undef,  # lib/Env.pm
        'Exporter'              => undef,  # lib/Exporter.pm
        'ExtUtils::MakeMaker'   => undef,  # lib/ExtUtils/MakeMaker.pm
        'Fcntl'                 => undef,  # ext/Fcntl/Fcntl.pm
        'File::Basename'        => undef,  # lib/File/Basename.pm
        'File::CheckTree'       => undef,  # lib/File/CheckTree.pm
        'File::Find'            => undef,  # lib/File/Find.pm
        'FileHandle'            => undef,  # lib/FileHandle.pm
        'GDBM_File'             => undef,  # ext/GDBM_File/GDBM_File.pm
        'Getopt::Long'          => undef,  # lib/Getopt/Long.pm
        'Getopt::Std'           => undef,  # lib/Getopt/Std.pm
        'I18N::Collate'         => undef,  # lib/I18N/Collate.pm
        'IPC::Open2'            => undef,  # lib/IPC/Open2.pm
        'IPC::Open3'            => undef,  # lib/IPC/Open3.pm
        'Math::BigFloat'        => undef,  # lib/Math/BigFloat.pm
        'Math::BigInt'          => undef,  # lib/Math/BigInt.pm
        'Math::Complex'         => undef,  # lib/Math/Complex.pm
        'NDBM_File'             => undef,  # ext/NDBM_File/NDBM_File.pm
        'Net::Ping'             => undef,  # lib/Net/Ping.pm
        'ODBM_File'             => undef,  # ext/ODBM_File/ODBM_File.pm
        'POSIX'                 => undef,  # ext/POSIX/POSIX.pm
        'SDBM_File'             => undef,  # ext/SDBM_File/SDBM_File.pm
        'Search::Dict'          => undef,  # lib/Search/Dict.pm
        'Shell'                 => undef,  # lib/Shell.pm
        'Socket'                => undef,  # ext/Socket/Socket.pm
        'Sys::Hostname'         => undef,  # lib/Sys/Hostname.pm
        'Sys::Syslog'           => undef,  # lib/Sys/Syslog.pm
        'Term::Cap'             => undef,  # lib/Term/Cap.pm
        'Term::Complete'        => undef,  # lib/Term/Complete.pm
        'Test::Harness'         => undef,  # lib/Test/Harness.pm
        'Text::Abbrev'          => undef,  # lib/Text/Abbrev.pm
        'Text::ParseWords'      => undef,  # lib/Text/ParseWords.pm
        'Text::Soundex'         => undef,  # lib/Text/Soundex.pm
        'Text::Tabs'            => undef,  # lib/Text/Tabs.pm
        'TieHash'               => undef,  # lib/TieHash.pm
        'Time::Local'           => undef,  # lib/Time/Local.pm
        'integer'               => undef,  # lib/integer.pm
        'less'                  => undef,  # lib/less.pm
        'sigtrap'               => undef,  # lib/sigtrap.pm
        'strict'                => undef,  # lib/strict.pm
        'subs'                  => undef,  # lib/subs.pm
    },

    5.001 => {
        'AnyDBM_File'           => undef,  # lib/AnyDBM_File.pm
        'AutoLoader'            => undef,  # lib/AutoLoader.pm
        'AutoSplit'             => undef,  # lib/AutoSplit.pm
        'Benchmark'             => undef,  # lib/Benchmark.pm
        'Carp'                  => undef,  # lib/Carp.pm
        'Cwd'                   => undef,  # lib/Cwd.pm
        'DB_File'               => undef,  # ext/DB_File/DB_File.pm
        'DynaLoader'            => undef,  # ext/DynaLoader/DynaLoader.pm
        'English'               => undef,  # lib/English.pm
        'Env'                   => undef,  # lib/Env.pm
        'Exporter'              => undef,  # lib/Exporter.pm
        'ExtUtils::Liblist'     => undef,  # lib/ExtUtils/Liblist.pm
        'ExtUtils::MakeMaker'   => undef,  # lib/ExtUtils/MakeMaker.pm
        'ExtUtils::Manifest'    => undef,  # lib/ExtUtils/Manifest.pm
        'ExtUtils::Mkbootstrap' => undef,  # lib/ExtUtils/Mkbootstrap.pm
        'Fcntl'                 => undef,  # ext/Fcntl/Fcntl.pm
        'File::Basename'        => undef,  # lib/File/Basename.pm
        'File::CheckTree'       => undef,  # lib/File/CheckTree.pm
        'File::Find'            => undef,  # lib/File/Find.pm
        'File::Path'            => undef,  # lib/File/Path.pm
        'FileHandle'            => undef,  # lib/FileHandle.pm
        'GDBM_File'             => undef,  # ext/GDBM_File/GDBM_File.pm
        'Getopt::Long'          => undef,  # lib/Getopt/Long.pm
        'Getopt::Std'           => undef,  # lib/Getopt/Std.pm
        'I18N::Collate'         => undef,  # lib/I18N/Collate.pm
        'IPC::Open2'            => undef,  # lib/IPC/Open2.pm
        'IPC::Open3'            => undef,  # lib/IPC/Open3.pm
        'Math::BigFloat'        => undef,  # lib/Math/BigFloat.pm
        'Math::BigInt'          => undef,  # lib/Math/BigInt.pm
        'Math::Complex'         => undef,  # lib/Math/Complex.pm
        'NDBM_File'             => undef,  # ext/NDBM_File/NDBM_File.pm
        'Net::Ping'             => undef,  # lib/Net/Ping.pm
        'ODBM_File'             => undef,  # ext/ODBM_File/ODBM_File.pm
        'POSIX'                 => undef,  # ext/POSIX/POSIX.pm
        'SDBM_File'             => undef,  # ext/SDBM_File/SDBM_File.pm
        'Search::Dict'          => undef,  # lib/Search/Dict.pm
        'Shell'                 => undef,  # lib/Shell.pm
        'Socket'                => undef,  # ext/Socket/Socket.pm
        'SubstrHash'            => undef,  # lib/SubstrHash.pm
        'Sys::Hostname'         => undef,  # lib/Sys/Hostname.pm
        'Sys::Syslog'           => undef,  # lib/Sys/Syslog.pm
        'Term::Cap'             => undef,  # lib/Term/Cap.pm
        'Term::Complete'        => undef,  # lib/Term/Complete.pm
        'Test::Harness'         => undef,  # lib/Test/Harness.pm
        'Text::Abbrev'          => undef,  # lib/Text/Abbrev.pm
        'Text::ParseWords'      => undef,  # lib/Text/ParseWords.pm
        'Text::Soundex'         => undef,  # lib/Text/Soundex.pm
        'Text::Tabs'            => undef,  # lib/Text/Tabs.pm
        'TieHash'               => undef,  # lib/TieHash.pm
        'Time::Local'           => undef,  # lib/Time/Local.pm
        'integer'               => undef,  # lib/integer.pm
        'less'                  => undef,  # lib/less.pm
        'lib'                   => undef,  # lib/lib.pm
        'sigtrap'               => undef,  # lib/sigtrap.pm
        'strict'                => undef,  # lib/strict.pm
        'subs'                  => undef,  # lib/subs.pm
    },

    5.002 => {
        'AnyDBM_File'           => undef,  # lib/AnyDBM_File.pm
        'AutoLoader'            => undef,  # lib/AutoLoader.pm
        'AutoSplit'             => undef,  # lib/AutoSplit.pm
        'Benchmark'             => undef,  # lib/Benchmark.pm
        'Carp'                  => undef,  # lib/Carp.pm
        'Cwd'                   => undef,  # lib/Cwd.pm
        'DB_File'               => '1.01',  # ext/DB_File/DB_File.pm
        'Devel::SelfStubber'    => '1.01',  # lib/Devel/SelfStubber.pm
        'DirHandle'             => undef,  # lib/DirHandle.pm
        'DynaLoader'            => '1.00',  # ext/DynaLoader/DynaLoader.pm
        'English'               => undef,  # lib/English.pm
        'Env'                   => undef,  # lib/Env.pm
        'Exporter'              => undef,  # lib/Exporter.pm
        'ExtUtils::Install'     => undef,  # lib/ExtUtils/Install.pm
        'ExtUtils::Liblist'     => undef,  # lib/ExtUtils/Liblist.pm
        'ExtUtils::MM_OS2'      => undef,  # lib/ExtUtils/MM_OS2.pm
        'ExtUtils::MM_Unix'     => undef,  # lib/ExtUtils/MM_Unix.pm
        'ExtUtils::MM_VMS'      => undef,  # lib/ExtUtils/MM_VMS.pm
        'ExtUtils::MakeMaker'   => '5.21',  # lib/ExtUtils/MakeMaker.pm
        'ExtUtils::Manifest'    => '1.22',  # lib/ExtUtils/Manifest.pm
        'ExtUtils::Mkbootstrap' => undef,  # lib/ExtUtils/Mkbootstrap.pm
        'ExtUtils::Mksymlists'  => '1.00',  # lib/ExtUtils/Mksymlists.pm
        'Fcntl'                 => '1.00',  # ext/Fcntl/Fcntl.pm
        'File::Basename'        => undef,  # lib/File/Basename.pm
        'File::CheckTree'       => undef,  # lib/File/CheckTree.pm
        'File::Copy'            => '1.5',  # lib/File/Copy.pm
        'File::Find'            => undef,  # lib/File/Find.pm
        'File::Path'            => '1.01',  # lib/File/Path.pm
        'FileCache'             => undef,  # lib/FileCache.pm
        'FileHandle'            => '1.00',  # ext/FileHandle/FileHandle.pm
        'GDBM_File'             => '1.00',  # ext/GDBM_File/GDBM_File.pm
        'Getopt::Long'          => '2.01',  # lib/Getopt/Long.pm
        'Getopt::Std'           => undef,  # lib/Getopt/Std.pm
        'I18N::Collate'         => undef,  # lib/I18N/Collate.pm
        'IPC::Open2'            => undef,  # lib/IPC/Open2.pm
        'IPC::Open3'            => undef,  # lib/IPC/Open3.pm
        'Math::BigFloat'        => undef,  # lib/Math/BigFloat.pm
        'Math::BigInt'          => undef,  # lib/Math/BigInt.pm
        'Math::Complex'         => undef,  # lib/Math/Complex.pm
        'NDBM_File'             => '1.00',  # ext/NDBM_File/NDBM_File.pm
        'Net::Ping'             => '1',  # lib/Net/Ping.pm
        'ODBM_File'             => '1.00',  # ext/ODBM_File/ODBM_File.pm
        'POSIX'                 => '1.00',  # ext/POSIX/POSIX.pm
        'Pod::Functions'        => undef,  # lib/Pod/Functions.pm
        'Pod::Text'             => undef,  # lib/Pod/Text.pm
        'SDBM_File'             => '1.00',  # ext/SDBM_File/SDBM_File.pm
        'Safe'                  => '1.00',  # ext/Safe/Safe.pm
        'Search::Dict'          => undef,  # lib/Search/Dict.pm
        'SelectSaver'           => undef,  # lib/SelectSaver.pm
        'SelfLoader'            => '1.06',  # lib/SelfLoader.pm
        'Shell'                 => undef,  # lib/Shell.pm
        'Socket'                => '1.5',  # ext/Socket/Socket.pm
        'Symbol'                => undef,  # lib/Symbol.pm
        'Sys::Hostname'         => undef,  # lib/Sys/Hostname.pm
        'Sys::Syslog'           => undef,  # lib/Sys/Syslog.pm
        'Term::Cap'             => undef,  # lib/Term/Cap.pm
        'Term::Complete'        => undef,  # lib/Term/Complete.pm
        'Term::ReadLine'        => undef,  # lib/Term/ReadLine.pm
        'Test::Harness'         => '1.07',  # lib/Test/Harness.pm
        'Text::Abbrev'          => undef,  # lib/Text/Abbrev.pm
        'Text::ParseWords'      => undef,  # lib/Text/ParseWords.pm
        'Text::Soundex'         => undef,  # lib/Text/Soundex.pm
        'Text::Tabs'            => undef,  # lib/Text/Tabs.pm
        'Text::Wrap'            => undef,  # lib/Text/Wrap.pm
        'Tie::Hash'             => undef,  # lib/Tie/Hash.pm
        'Tie::Scalar'           => undef,  # lib/Tie/Scalar.pm
        'Tie::SubstrHash'       => undef,  # lib/Tie/SubstrHash.pm
        'Time::Local'           => undef,  # lib/Time/Local.pm
        'diagnostics'           => undef,  # lib/diagnostics.pm
        'integer'               => undef,  # lib/integer.pm
        'less'                  => undef,  # lib/less.pm
        'lib'                   => undef,  # lib/lib.pm
        'overload'              => undef,  # lib/overload.pm
        'sigtrap'               => undef,  # lib/sigtrap.pm
        'strict'                => undef,  # lib/strict.pm
        'subs'                  => undef,  # lib/subs.pm
        'vars'                  => undef,  # lib/vars.pm
    },

    5.00307 => {
        'AnyDBM_File'           => undef, #./lib/AnyDBM_File.pm
        'AutoLoader'            => undef, #./lib/AutoLoader.pm
        'AutoSplit'             => undef, #./lib/AutoSplit.pm
        'Benchmark'             => undef, #./lib/Benchmark.pm
        'Carp'                  => undef, #./lib/Carp.pm
        'Config'                => undef,
        'Cwd'                   => undef, #./lib/Cwd.pm
        'DB_File'               => '1.03', #./lib/DB_File.pm
        'Devel::SelfStubber'    => '1.01', #./lib/Devel/SelfStubber.pm
        'diagnostics'           => undef, #./lib/diagnostics.pm
        'DirHandle'             => undef, #./lib/DirHandle.pm
        'DynaLoader'            => '1.00', #./ext/DynaLoader/DynaLoader.pm
        'English'               => undef, #./lib/English.pm
        'Env'                   => undef, #./lib/Env.pm
        'Exporter'              => undef, #./lib/Exporter.pm
        'ExtUtils::Embed'       => '1.18', #./lib/ExtUtils/Embed.pm
        'ExtUtils::Install'     => '1.15 ', #./lib/ExtUtils/Install.pm
        'ExtUtils::Liblist'     => '1.20 ', #./lib/ExtUtils/Liblist.pm
        'ExtUtils::MakeMaker'   => '5.38', #./lib/ExtUtils/MakeMaker.pm
        'ExtUtils::Manifest'    => '1.27', #./lib/ExtUtils/Manifest.pm
        'ExtUtils::Mkbootstrap' => '1.13 ', #./lib/ExtUtils/Mkbootstrap.pm
        'ExtUtils::Mksymlists'  => '1.12 ', #./lib/ExtUtils/Mksymlists.pm
        'ExtUtils::MM_OS2'      => undef, #./lib/ExtUtils/MM_OS2.pm
        'ExtUtils::MM_Unix'     => '1.107 ', #./lib/ExtUtils/MM_Unix.pm
        'ExtUtils::MM_VMS'      => undef, #./lib/ExtUtils/MM_VMS.pm
        'ExtUtils::testlib'     => '1.11 ', #./lib/ExtUtils/testlib.pm
        'Fatal'                 => undef, #./lib/Fatal.pm
        'Fcntl'                 => '1.00', #./ext/Fcntl/Fcntl.pm
        'File::Basename'        => '2.4', #./lib/File/Basename.pm
        'File::CheckTree'       => undef, #./lib/File/CheckTree.pm
        'File::Copy'            => '1.5', #./lib/File/Copy.pm
        'File::Find'            => undef, #./lib/File/Find.pm
        'File::Path'            => '1.01', #./lib/File/Path.pm
        'FileCache'             => undef, #./lib/FileCache.pm
        'FileHandle'            => '1.00', #./ext/FileHandle/FileHandle.pm
        'FindBin'               => '1.04', #./lib/FindBin.pm
        'GDBM_File'             => '1.00', #./ext/GDBM_File/GDBM_File.pm
        'Getopt::Long'          => '2.04', #./lib/Getopt/Long.pm
        'Getopt::Std'           => undef, #./lib/Getopt/Std.pm
        'I18N::Collate'         => undef, #./lib/I18N/Collate.pm
        'integer'               => undef, #./lib/integer.pm
        'IO'                    => undef, #./ext/IO/IO.pm
        'IO::File'              => '1.05', #./ext/IO/lib/IO/File.pm
        'IO::Handle'            => '1.12', #./ext/IO/lib/IO/Handle.pm
        'IO::Pipe'              => '1.07', #./ext/IO/lib/IO/Pipe.pm
        'IO::Seekable'          => '1.05', #./ext/IO/lib/IO/Seekable.pm
        'IO::Select'            => '1.09', #./ext/IO/lib/IO/Select.pm
        'IO::Socket'            => '1.13', #./ext/IO/lib/IO/Socket.pm
        'IPC::Open2'            => undef, #./lib/IPC/Open2.pm
        'IPC::Open3'            => undef, #./lib/IPC/Open3.pm
        'less'                  => undef, #./lib/less.pm
        'lib'                   => undef, #./lib/lib.pm
        'Math::BigFloat'        => undef, #./lib/Math/BigFloat.pm
        'Math::BigInt'          => undef, #./lib/Math/BigInt.pm
        'Math::Complex'         => undef, #./lib/Math/Complex.pm
        'NDBM_File'             => '1.00', #./ext/NDBM_File/NDBM_File.pm
        'Net::Ping'             => '1.01', #./lib/Net/Ping.pm
        'ODBM_File'             => '1.00', #./ext/ODBM_File/ODBM_File.pm
        'Opcode'                => '1.01', #./ext/Opcode/Opcode.pm
        'ops'                   => undef, #./ext/Opcode/ops.pm
        'OS2::ExtAttr'          => '0.01', #./os2/OS2/ExtAttr/ExtAttr.pm
        'OS2::PrfDB'            => '0.02', #./os2/OS2/PrfDB/PrfDB.pm
        'OS2::Process'          => undef, #./os2/OS2/Process/Process.pm
        'OS2::REXX'             => undef, #./os2/OS2/REXX/REXX.pm
        'overload'              => undef, #./lib/overload.pm
        'Pod::Functions'        => undef, #./lib/Pod/Functions.pm
        'Pod::Text'             => undef, #./lib/Pod/Text.pm
        'POSIX'                 => '1.00', #./ext/POSIX/POSIX.pm
        'Safe'                  => '2.06', #./ext/Opcode/Safe.pm
        'SDBM_File'             => '1.00', #./ext/SDBM_File/SDBM_File.pm
        'Search::Dict'          => undef, #./lib/Search/Dict.pm
        'SelectSaver'           => undef, #./lib/SelectSaver.pm
        'SelfLoader'            => '1.06', #./lib/SelfLoader.pm
        'Shell'                 => undef, #./lib/Shell.pm
        'sigtrap'               => '1.01', #./lib/sigtrap.pm
        'Socket'                => '1.5', #./ext/Socket/Socket.pm
        'strict'                => undef, #./lib/strict.pm
        'subs'                  => undef, #./lib/subs.pm
        'Symbol'                => undef, #./lib/Symbol.pm
        'Sys::Hostname'         => undef, #./lib/Sys/Hostname.pm
        'Sys::Syslog'           => undef, #./lib/Sys/Syslog.pm
        'Term::Cap'             => undef, #./lib/Term/Cap.pm
        'Term::Complete'        => undef, #./lib/Term/Complete.pm
        'Term::ReadLine'        => undef, #./lib/Term/ReadLine.pm
        'Test::Harness'         => '1.13', #./lib/Test/Harness.pm
        'Text::Abbrev'          => undef, #./lib/Text/Abbrev.pm
        'Text::ParseWords'      => undef, #./lib/Text/ParseWords.pm
        'Text::Soundex'         => undef, #./lib/Text/Soundex.pm
        'Text::Tabs'            => '96.051501', #./lib/Text/Tabs.pm
        'Text::Wrap'            => '96.041801', #./lib/Text/Wrap.pm
        'Tie::Hash'             => undef, #./lib/Tie/Hash.pm
        'Tie::Scalar'           => undef, #./lib/Tie/Scalar.pm
        'Tie::SubstrHash'       => undef, #./lib/Tie/SubstrHash.pm
        'Time::Local'           => undef, #./lib/Time/Local.pm
        'UNIVERSAL'             => undef, #./lib/UNIVERSAL.pm
        'vars'                  => undef, #./lib/vars.pm
        'VMS::Filespec'         => undef, #./vms/ext/Filespec.pm
        'VMS::Stdio'            => '2.0', #./vms/ext/Stdio/Stdio.pm
    },

    5.004   => {
        'AnyDBM_File'           => undef, #./lib/AnyDBM_File.pm
        'AutoLoader'            => undef, #./lib/AutoLoader.pm
        'AutoSplit'             => undef, #./lib/AutoSplit.pm
        'autouse'               => '1.01', #./lib/autouse.pm
        'Benchmark'             => undef, #./lib/Benchmark.pm
        'blib'                  => undef, #./lib/blib.pm
        'Bundle::CPAN'          => '0.02', #./lib/Bundle/CPAN.pm
        'Carp'                  => undef, #./lib/Carp.pm
        'CGI'                   => '2.36', #./lib/CGI.pm
        'CGI::Apache'           => '1.01', #./lib/CGI/Apache.pm
        'CGI::Carp'             => '1.06', #./lib/CGI/Carp.pm
        'CGI::Fast'             => '1.00a', #./lib/CGI/Fast.pm
        'CGI::Push'             => '1.00', #./lib/CGI/Push.pm
        'CGI::Switch'           => '0.05', #./lib/CGI/Switch.pm
        'Class::Struct'         => undef, #./lib/Class/Struct.pm
        'Config'                => undef,
        'constant'              => '1.00', #./lib/constant.pm
        'CPAN'                  => '1.2401', #./lib/CPAN.pm
        'CPAN::FirstTime'       => '1.18 ', #./lib/CPAN/FirstTime.pm
        'CPAN::Nox'             => undef, #./lib/CPAN/Nox.pm
        'Cwd'                   => '2.00', #./lib/Cwd.pm
        'DB_File'               => '1.14', #./ext/DB_File/DB_File.pm
        'Devel::SelfStubber'    => '1.01', #./lib/Devel/SelfStubber.pm
        'diagnostics'           => undef, #./lib/diagnostics.pm
        'DirHandle'             => undef, #./lib/DirHandle.pm
        'DynaLoader'            => '1.02', #./ext/DynaLoader/DynaLoader.pm
        'English'               => undef, #./lib/English.pm
        'Env'                   => undef, #./lib/Env.pm
        'Exporter'              => undef, #./lib/Exporter.pm
        'ExtUtils::Command'     => '1.00', #./lib/ExtUtils/Command.pm
        'ExtUtils::Embed'       => '1.2501', #./lib/ExtUtils/Embed.pm
        'ExtUtils::Install'     => '1.16 ', #./lib/ExtUtils/Install.pm
        'ExtUtils::Liblist'     => '1.2201 ', #./lib/ExtUtils/Liblist.pm
        'ExtUtils::MakeMaker'   => '5.4002', #./lib/ExtUtils/MakeMaker.pm
        'ExtUtils::Manifest'    => '1.33 ', #./lib/ExtUtils/Manifest.pm
        'ExtUtils::Mkbootstrap' => '1.13 ', #./lib/ExtUtils/Mkbootstrap.pm
        'ExtUtils::Mksymlists'  => '1.13 ', #./lib/ExtUtils/Mksymlists.pm
        'ExtUtils::MM_OS2'      => undef, #./lib/ExtUtils/MM_OS2.pm
        'ExtUtils::MM_Unix'     => '1.114 ', #./lib/ExtUtils/MM_Unix.pm
        'ExtUtils::MM_VMS'      => undef, #./lib/ExtUtils/MM_VMS.pm
        'ExtUtils::MM_Win32'    => undef, #./lib/ExtUtils/MM_Win32.pm
        'ExtUtils::testlib'     => '1.11 ', #./lib/ExtUtils/testlib.pm
        'ExtUtils::XSSymSet'    => '1.0', #./vms/ext/XSSymSet.pm
        'Fcntl'                 => '1.03', #./ext/Fcntl/Fcntl.pm
        'File::Basename'        => '2.5', #./lib/File/Basename.pm
        'File::CheckTree'       => undef, #./lib/File/CheckTree.pm
        'File::Compare'         => '1.1001', #./lib/File/Compare.pm
        'File::Copy'            => '2.02', #./lib/File/Copy.pm
        'File::Find'            => undef, #./lib/File/Find.pm
        'File::Path'            => '1.04', #./lib/File/Path.pm
        'File::stat'            => undef, #./lib/File/stat.pm
        'FileCache'             => undef, #./lib/FileCache.pm
        'FileHandle'            => '2.00', #./lib/FileHandle.pm
        'FindBin'               => '1.04', #./lib/FindBin.pm
        'GDBM_File'             => '1.00', #./ext/GDBM_File/GDBM_File.pm
        'Getopt::Long'          => '2.10', #./lib/Getopt/Long.pm
        'Getopt::Std'           => undef, #./lib/Getopt/Std.pm
        'I18N::Collate'         => undef, #./lib/I18N/Collate.pm
        'integer'               => undef, #./lib/integer.pm
        'IO'                    => undef, #./ext/IO/IO.pm
        'IO::File'              => '1.0602', #./ext/IO/lib/IO/File.pm
        'IO::Handle'            => '1.1504', #./ext/IO/lib/IO/Handle.pm
        'IO::Pipe'              => '1.0901', #./ext/IO/lib/IO/Pipe.pm
        'IO::Seekable'          => '1.06', #./ext/IO/lib/IO/Seekable.pm
        'IO::Select'            => '1.10', #./ext/IO/lib/IO/Select.pm
        'IO::Socket'            => '1.1602', #./ext/IO/lib/IO/Socket.pm
        'IPC::Open2'            => '1.01', #./lib/IPC/Open2.pm
        'IPC::Open3'            => '1.0101', #./lib/IPC/Open3.pm
        'less'                  => undef, #./lib/less.pm
        'lib'                   => undef, #./lib/lib.pm
        'locale'                => undef, #./lib/locale.pm
        'Math::BigFloat'        => undef, #./lib/Math/BigFloat.pm
        'Math::BigInt'          => undef, #./lib/Math/BigInt.pm
        'Math::Complex'         => '1.01', #./lib/Math/Complex.pm
        'Math::Trig'            => '1', #./lib/Math/Trig.pm
        'NDBM_File'             => '1.00', #./ext/NDBM_File/NDBM_File.pm
        'Net::hostent'          => undef, #./lib/Net/hostent.pm
        'Net::netent'           => undef, #./lib/Net/netent.pm
        'Net::Ping'             => '2.02', #./lib/Net/Ping.pm
        'Net::protoent'         => undef, #./lib/Net/protoent.pm
        'Net::servent'          => undef, #./lib/Net/servent.pm
        'ODBM_File'             => '1.00', #./ext/ODBM_File/ODBM_File.pm
        'Opcode'                => '1.04', #./ext/Opcode/Opcode.pm
        'ops'                   => undef, #./ext/Opcode/ops.pm
        'Safe'                  => '2.06', #./ext/Opcode/Safe.pm
        'OS2::ExtAttr'          => '0.01', #./os2/OS2/ExtAttr/ExtAttr.pm
        'OS2::PrfDB'            => '0.02', #./os2/OS2/PrfDB/PrfDB.pm
        'OS2::Process'          => undef, #./os2/OS2/Process/Process.pm
        'OS2::REXX'             => undef, #./os2/OS2/REXX/REXX.pm
        'overload'              => undef, #./lib/overload.pm
        'Pod::Functions'        => undef, #./lib/Pod/Functions.pm
        'Pod::Html'             => undef, #./lib/Pod/Html.pm
        'Pod::Text'             => '1.0203', #./lib/Pod/Text.pm
        'POSIX'                 => '1.02', #./ext/POSIX/POSIX.pm
        'SDBM_File'             => '1.00', #./ext/SDBM_File/SDBM_File.pm
        'Search::Dict'          => undef, #./lib/Search/Dict.pm
        'SelectSaver'           => undef, #./lib/SelectSaver.pm
        'SelfLoader'            => '1.07', #./lib/SelfLoader.pm
        'Shell'                 => undef, #./lib/Shell.pm
        'sigtrap'               => '1.02', #./lib/sigtrap.pm
        'Socket'                => '1.6', #./ext/Socket/Socket.pm
        'strict'                => undef, #./lib/strict.pm
        'subs'                  => undef, #./lib/subs.pm
        'Symbol'                => '1.02', #./lib/Symbol.pm
        'Sys::Hostname'         => undef, #./lib/Sys/Hostname.pm
        'Sys::Syslog'           => undef, #./lib/Sys/Syslog.pm
        'Term::Cap'             => undef, #./lib/Term/Cap.pm
        'Term::Complete'        => undef, #./lib/Term/Complete.pm
        'Term::ReadLine'        => undef, #./lib/Term/ReadLine.pm
        'Test::Harness'         => '1.1502', #./lib/Test/Harness.pm
        'Text::Abbrev'          => undef, #./lib/Text/Abbrev.pm
        'Text::ParseWords'      => undef, #./lib/Text/ParseWords.pm
        'Text::Soundex'         => undef, #./lib/Text/Soundex.pm
        'Text::Tabs'            => '96.121201', #./lib/Text/Tabs.pm
        'Text::Wrap'            => '97.011701', #./lib/Text/Wrap.pm
        'Tie::Hash'             => undef, #./lib/Tie/Hash.pm
        'Tie::RefHash'          => undef, #./lib/Tie/RefHash.pm
        'Tie::Scalar'           => undef, #./lib/Tie/Scalar.pm
        'Tie::SubstrHash'       => undef, #./lib/Tie/SubstrHash.pm
        'Time::gmtime'          => '1.01', #./lib/Time/gmtime.pm
        'Time::Local'           => undef, #./lib/Time/Local.pm
        'Time::localtime'       => '1.01', #./lib/Time/localtime.pm
        'Time::tm'              => undef, #./lib/Time/tm.pm
        'UNIVERSAL'             => undef, #./lib/UNIVERSAL.pm
        'User::grent'           => undef, #./lib/User/grent.pm
        'User::pwent'           => undef, #./lib/User/pwent.pm
        'vars'                  => undef, #./lib/vars.pm
        'VMS::DCLsym'           => '1.01', #./vms/ext/DCLsym/DCLsym.pm
        'VMS::Filespec'         => undef, #./vms/ext/Filespec.pm
        'VMS::Stdio'            => '2.02', #./vms/ext/Stdio/Stdio.pm
        'vmsish'                => undef, #./vms/ext/vmsish.pm
    },

    5.005   => {
        'AnyDBM_File'           => undef, #./lib/AnyDBM_File.pm
        'attrs'                 => '1.0', #./ext/attrs/attrs.pm
        'AutoLoader'            => undef, #./lib/AutoLoader.pm
        'AutoSplit'             => '1.0302', #./lib/AutoSplit.pm
        'autouse'               => '1.01', #./lib/autouse.pm
        'B'                     => undef, #./ext/B/B.pm
        'B::Asmdata'            => undef, #./ext/B/B/Asmdata.pm
        'B::Assembler'          => undef, #./ext/B/B/Assembler.pm
        'B::Bblock'             => undef, #./ext/B/B/Bblock.pm
        'B::Bytecode'           => undef, #./ext/B/B/Bytecode.pm
        'B::C'                  => undef, #./ext/B/B/C.pm
        'B::CC'                 => undef, #./ext/B/B/CC.pm
        'B::Debug'              => undef, #./ext/B/B/Debug.pm
        'B::Deparse'            => '0.56', #./ext/B/B/Deparse.pm
        'B::Disassembler'       => undef, #./ext/B/B/Disassembler.pm
        'B::Lint'               => undef, #./ext/B/B/Lint.pm
        'B::Showlex'            => undef, #./ext/B/B/Showlex.pm
        'B::Stackobj'           => undef, #./ext/B/B/Stackobj.pm
        'B::Terse'              => undef, #./ext/B/B/Terse.pm
        'B::Xref'               => undef, #./ext/B/B/Xref.pm
        'base'                  => undef, #./lib/base.pm
        'Benchmark'             => undef, #./lib/Benchmark.pm
        'blib'                  => '1.00', #./lib/blib.pm
        'Carp'                  => undef, #./lib/Carp.pm
        'CGI'                   => '2.42', #./lib/CGI.pm
        'CGI::Apache'           => '1.1', #./lib/CGI/Apache.pm
        'CGI::Carp'             => '1.101', #./lib/CGI/Carp.pm
        'CGI::Cookie'           => '1.06', #./lib/CGI/Cookie.pm
        'CGI::Fast'             => '1.00a', #./lib/CGI/Fast.pm
        'CGI::Push'             => '1.01', #./lib/CGI/Push.pm
        'CGI::Switch'           => '0.06', #./lib/CGI/Switch.pm
        'Class::Struct'         => undef, #./lib/Class/Struct.pm
        'Config'                => undef,
        'constant'              => '1.00', #./lib/constant.pm
        'CPAN'                  => '1.3901', #./lib/CPAN.pm
        'CPAN::FirstTime'       => '1.29 ', #./lib/CPAN/FirstTime.pm
        'CPAN::Nox'             => undef, #./lib/CPAN/Nox.pm
        'Cwd'                   => '2.01', #./lib/Cwd.pm
        'Data::Dumper'          => '2.09', #./ext/Data/Dumper/Dumper.pm
        'DB_File'               => '1.60', #./ext/DB_File/DB_File.pm
        'Devel::SelfStubber'    => '1.01', #./lib/Devel/SelfStubber.pm
        'DynaLoader'            => '1.03',
        'diagnostics'           => undef, #./lib/diagnostics.pm
        'DirHandle'             => undef, #./lib/DirHandle.pm
        'English'               => undef, #./lib/English.pm
        'Env'                   => undef, #./lib/Env.pm
        'Exporter'              => undef, #./lib/Exporter.pm
        'ExtUtils::Command'     => '1.01', #./lib/ExtUtils/Command.pm
        'ExtUtils::Embed'       => '1.2505', #./lib/ExtUtils/Embed.pm
        'ExtUtils::Install'     => '1.28 ', #./lib/ExtUtils/Install.pm
        'ExtUtils::Installed'   => '0.02', #./lib/ExtUtils/Installed.pm
        'ExtUtils::Liblist'     => '1.25 ', #./lib/ExtUtils/Liblist.pm
        'ExtUtils::MakeMaker'   => '5.4301', #./lib/ExtUtils/MakeMaker.pm
        'ExtUtils::Manifest'    => '1.33 ', #./lib/ExtUtils/Manifest.pm
        'ExtUtils::Mkbootstrap' => '1.13 ', #./lib/ExtUtils/Mkbootstrap.pm
        'ExtUtils::Mksymlists'  => '1.17 ', #./lib/ExtUtils/Mksymlists.pm
        'ExtUtils::MM_OS2'      => undef, #./lib/ExtUtils/MM_OS2.pm
        'ExtUtils::MM_Unix'     => '1.12601 ', #./lib/ExtUtils/MM_Unix.pm
        'ExtUtils::MM_VMS'      => undef, #./lib/ExtUtils/MM_VMS.pm
        'ExtUtils::MM_Win32'    => undef, #./lib/ExtUtils/MM_Win32.pm
        'ExtUtils::Packlist'    => '0.03', #./lib/ExtUtils/Packlist.pm
        'ExtUtils::testlib'     => '1.11 ', #./lib/ExtUtils/testlib.pm
        'ExtUtils::XSSymSet'    => '1.0', #./vms/ext/XSSymSet.pm
        'Fatal'                 => '1.02', #./lib/Fatal.pm
        'Fcntl'                 => '1.03', #./ext/Fcntl/Fcntl.pm
        'fields'                => '0.02', #./lib/fields.pm
        'File::Basename'        => '2.6', #./lib/File/Basename.pm
        'File::CheckTree'       => undef, #./lib/File/CheckTree.pm
        'File::Compare'         => '1.1001', #./lib/File/Compare.pm
        'File::Copy'            => '2.02', #./lib/File/Copy.pm
        'File::DosGlob'         => undef, #./lib/File/DosGlob.pm
        'File::Find'            => undef, #./lib/File/Find.pm
        'File::Path'            => '1.0401', #./lib/File/Path.pm
        'File::Spec'            => '0.6', #./lib/File/Spec.pm
        'File::Spec::Mac'       => '1.0', #./lib/File/Spec/Mac.pm
        'File::Spec::OS2'       => undef, #./lib/File/Spec/OS2.pm
        'File::Spec::Unix'      => undef, #./lib/File/Spec/Unix.pm
        'File::Spec::VMS'       => undef, #./lib/File/Spec/VMS.pm
        'File::Spec::Win32'     => undef, #./lib/File/Spec/Win32.pm
        'File::stat'            => undef, #./lib/File/stat.pm
        'FileCache'             => undef, #./lib/FileCache.pm
        'FileHandle'            => '2.00', #./lib/FileHandle.pm
        'FindBin'               => '1.41', #./lib/FindBin.pm
        'GDBM_File'             => '1.00', #./ext/GDBM_File/GDBM_File.pm
        'Getopt::Long'          => '2.17', #./lib/Getopt/Long.pm
        'Getopt::Std'           => undef, #./lib/Getopt/Std.pm
        'I18N::Collate'         => undef, #./lib/I18N/Collate.pm
        'integer'               => undef, #./lib/integer.pm
        'IO'                    => undef, #./ext/IO/IO.pm
        'IO::File'              => '1.06021', #./ext/IO/lib/IO/File.pm
        'IO::Handle'            => '1.1505', #./ext/IO/lib/IO/Handle.pm
        'IO::Pipe'              => '1.0901', #./ext/IO/lib/IO/Pipe.pm
        'IO::Seekable'          => '1.06', #./ext/IO/lib/IO/Seekable.pm
        'IO::Select'            => '1.10', #./ext/IO/lib/IO/Select.pm
        'IO::Socket'            => '1.1603', #./ext/IO/lib/IO/Socket.pm
        'IPC::Open2'            => '1.01', #./lib/IPC/Open2.pm
        'IPC::Open3'            => '1.0102', #./lib/IPC/Open3.pm
        'IPC::Msg'              => '1.00', #./ext/IPC/SysV/Msg.pm
        'IPC::Semaphore'        => '1.00', #./ext/IPC/SysV/Semaphore.pm
        'IPC::SysV'             => '1.03', #./ext/IPC/SysV/SysV.pm
        'less'                  => undef, #./lib/less.pm
        'lib'                   => undef, #./lib/lib.pm
        'locale'                => undef, #./lib/locale.pm
        'Math::BigFloat'        => undef, #./lib/Math/BigFloat.pm
        'Math::BigInt'          => undef, #./lib/Math/BigInt.pm
        'Math::Complex'         => '1.25', #./lib/Math/Complex.pm
        'Math::Trig'            => '1', #./lib/Math/Trig.pm
        'NDBM_File'             => '1.01', #./ext/NDBM_File/NDBM_File.pm
        'Net::hostent'          => undef, #./lib/Net/hostent.pm
        'Net::netent'           => undef, #./lib/Net/netent.pm
        'Net::Ping'             => '2.02', #./lib/Net/Ping.pm
        'Net::protoent'         => undef, #./lib/Net/protoent.pm
        'Net::servent'          => undef, #./lib/Net/servent.pm
        'O'                     => undef, #./ext/B/O.pm
        'ODBM_File'             => '1.00', #./ext/ODBM_File/ODBM_File.pm
        'Opcode'                => '1.04', #./ext/Opcode/Opcode.pm
        'ops'                   => undef, #./ext/Opcode/ops.pm
        'Safe'                  => '2.06', #./ext/Opcode/Safe.pm
        'OS2::ExtAttr'          => '0.01', #./os2/OS2/ExtAttr/ExtAttr.pm
        'OS2::PrfDB'            => '0.02', #./os2/OS2/PrfDB/PrfDB.pm
        'OS2::Process'          => '0.2', #./os2/OS2/Process/Process.pm
        'OS2::REXX'             => undef, #./os2/OS2/REXX/REXX.pm
        'overload'              => undef, #./lib/overload.pm
        'Pod::Functions'        => undef, #./lib/Pod/Functions.pm
        'Pod::Html'             => '1.01', #./lib/Pod/Html.pm
        'Pod::Text'             => '1.0203', #./lib/Pod/Text.pm
        'POSIX'                 => '1.02', #./ext/POSIX/POSIX.pm
        're'                    => '0.02', #./ext/re/re.pm
        'SDBM_File'             => '1.00', #./ext/SDBM_File/SDBM_File.pm
        'Search::Dict'          => undef, #./lib/Search/Dict.pm
        'SelectSaver'           => undef, #./lib/SelectSaver.pm
        'SelfLoader'            => '1.08', #./lib/SelfLoader.pm
        'Shell'                 => undef, #./lib/Shell.pm
        'sigtrap'               => '1.02', #./lib/sigtrap.pm
        'Socket'                => '1.7', #./ext/Socket/Socket.pm
        'strict'                => '1.01', #./lib/strict.pm
        'subs'                  => undef, #./lib/subs.pm
        'Symbol'                => '1.02', #./lib/Symbol.pm
        'Sys::Hostname'         => undef, #./lib/Sys/Hostname.pm
        'Sys::Syslog'           => undef, #./lib/Sys/Syslog.pm
        'Term::Cap'             => undef, #./lib/Term/Cap.pm
        'Term::Complete'        => undef, #./lib/Term/Complete.pm
        'Term::ReadLine'        => undef, #./lib/Term/ReadLine.pm
        'Test'                  => '1.04', #./lib/Test.pm
        'Test::Harness'         => '1.1602', #./lib/Test/Harness.pm
        'Text::Abbrev'          => undef, #./lib/Text/Abbrev.pm
        'Text::ParseWords'      => '3.1', #./lib/Text/ParseWords.pm
        'Text::Soundex'         => undef, #./lib/Text/Soundex.pm
        'Text::Tabs'            => '96.121201', #./lib/Text/Tabs.pm
        'Text::Wrap'            => '97.02', #./lib/Text/Wrap.pm
        'Thread'                => '1.0', #./ext/Thread/Thread.pm
        'Thread::Queue'         => undef, #./ext/Thread/Thread/Queue.pm
        'Thread::Semaphore'     => undef, #./ext/Thread/Thread/Semaphore.pm
        'Thread::Signal'        => undef, #./ext/Thread/Thread/Signal.pm
        'Thread::Specific'      => undef, #./ext/Thread/Thread/Specific.pm
        'Tie::Array'            => '1.00', #./lib/Tie/Array.pm
        'Tie::Handle'           => undef, #./lib/Tie/Handle.pm
        'Tie::Hash'             => undef, #./lib/Tie/Hash.pm
        'Tie::RefHash'          => undef, #./lib/Tie/RefHash.pm
        'Tie::Scalar'           => undef, #./lib/Tie/Scalar.pm
        'Tie::SubstrHash'       => undef, #./lib/Tie/SubstrHash.pm
        'Time::gmtime'          => '1.01', #./lib/Time/gmtime.pm
        'Time::Local'           => undef, #./lib/Time/Local.pm
        'Time::localtime'       => '1.01', #./lib/Time/localtime.pm
        'Time::tm'              => undef, #./lib/Time/tm.pm
        'UNIVERSAL'             => undef, #./lib/UNIVERSAL.pm
        'User::grent'           => undef, #./lib/User/grent.pm
        'User::pwent'           => undef, #./lib/User/pwent.pm
        'vars'                  => undef, #./lib/vars.pm
        'VMS::DCLsym'           => '1.01', #./vms/ext/DCLsym/DCLsym.pm
        'VMS::Filespec'         => undef, #./vms/ext/Filespec.pm
        'VMS::Stdio'            => '2.1', #./vms/ext/Stdio/Stdio.pm
        'vmsish'                => undef, #./vms/ext/vmsish.pm
    },

    5.00503   => {
        'AnyDBM_File'           => undef,
        'attrs'                 => '1.0',
        'AutoLoader'            => undef,
        'AutoSplit'             => 1.0303,
        'autouse'               => 1.01,
        'B::Asmdata'            => undef,
        'B::Assembler'          => undef,
        'B::Bblock'             => undef,
        'B::Bytecode'           => undef,
        'B::C'                  => undef,
        'B::CC'                 => undef,
        'B::Debug'              => undef,
        'B::Deparse'            => 0.56,
        'B::Disassembler'       => undef,
        'B::Lint'               => undef,
        'B'                     => undef,
        'B::Showlex'            => undef,
        'B::Stackobj'           => undef,
        'B::Terse'              => undef,
        'B::Xref'               => undef,
        'base'                  => undef,
        'Benchmark'             => undef,
        'blib'                  => '1.00',
        'Carp'                  => undef,
        'CGI'                   => 2.46,
        'CGI::Apache'           => 1.1,
        'CGI::Carp'             => 1.13,
        'CGI::Cookie'           => 1.06,
        'CGI::Fast'             => 1.01,
        'CGI::Push'             => 1.01,
        'CGI::Switch'           => 0.06,
        'Class::Struct'         => undef,
        'Config'                => undef,
        'constant'              => '1.00',
        'CPAN::FirstTime'       => 1.36 ,
        'CPAN'                  => 1.48,
        'CPAN::Nox'             => '1.00',
        'Cwd'                   => 2.01,
        'Data::Dumper'          => 2.101,
        'DB_File'               => 1.65,
        'Devel::SelfStubber'    => 1.01,
        'diagnostics'           => undef,
        'DirHandle'             => undef,
        'Dumpvalue'             => undef,
        'DynaLoader'            => 1.03,
        'English'               => undef,
        'Env'                   => undef,
        'Exporter'              => undef,
        'ExtUtils::Command'     => 1.01,
        'ExtUtils::Embed'       => 1.2505,
        'ExtUtils::Install'     => 1.28 ,
        'ExtUtils::Installed'   => 0.02,
        'ExtUtils::Liblist'     => 1.25 ,
        'ExtUtils::MakeMaker'   => 5.4302,
        'ExtUtils::Manifest'    => 1.33 ,
        'ExtUtils::Mkbootstrap' => 1.14 ,
        'ExtUtils::Mksymlists'  => 1.17 ,
        'ExtUtils::MM_OS2'      => undef,
        'ExtUtils::MM_Unix'     => 1.12602 ,
        'ExtUtils::MM_VMS'      => undef,
        'ExtUtils::MM_Win32'    => undef,
        'ExtUtils::Packlist'    => 0.03,
        'ExtUtils::testlib'     => 1.11 ,
        'ExtUtils::XSSymSet'    => '1.0',
        'Fatal'                 => 1.02,
        'Fcntl'                 => 1.03,
        'fields'                => 0.02,
        'File::Basename'        => 2.6,
        'File::CheckTree'       => undef,
        'File::Compare'         => 1.1001,
        'File::Copy'            => 2.02,
        'File::DosGlob'         => undef,
        'File::Find'            => undef,
        'File::Path'            => 1.0401,
        'File::Spec'            => 0.6,
        'File::Spec::Mac'       => '1.0',
        'File::Spec::OS2'       => undef,
        'File::Spec::Unix'      => undef,
        'File::Spec::VMS'       => undef,
        'File::Spec::Win32'     => undef,
        'File::stat'            => undef,
        'FileCache'             => undef,
        'FileHandle'            => '2.00',
        'FindBin'               => 1.42,
        'GDBM_File'             => '1.00',
        'Getopt::Long'          => 2.19,
        'Getopt::Std'           => 1.01,
        'I18N::Collate'         => undef,
        'integer'               => undef,
        'IO'                    => undef,
        'IO::File'              => 1.06021,
        'IO::Handle'            => 1.1505,
        'IO::Pipe'              => 1.0902,
        'IO::Seekable'          => 1.06,
        'IO::Select'            => '1.10',
        'IO::Socket'            => 1.1603,
        'IPC::Msg'              => '1.00',
        'IPC::Open2'            => 1.01,
        'IPC::Open3'            => 1.0103,
        'IPC::Semaphore'        => '1.00',
        'IPC::SysV'             => 1.03,
        'less'                  => undef,
        'lib'                   => undef,
        'locale'                => undef,
        'Math::BigFloat'        => undef,
        'Math::BigInt'          => undef,
        'Math::Complex'         => 1.26,
        'Math::Trig'            => 1,
        'NDBM_File'             => 1.01,
        'Net::hostent'          => undef,
        'Net::netent'           => undef,
        'Net::Ping'             => 2.02,
        'Net::protoent'         => undef,
        'Net::servent'          => undef,
        'O'                     => undef,
        'ODBM_File'             => '1.00',
        'Opcode'                => 1.04,
        'ops'                   => undef,
        'OS2::ExtAttr'          => 0.01,
        'OS2::PrfDB'            => 0.02,
        'OS2::Process'          => 0.2,
        'OS2::REXX'             => undef,
        'overload'              => undef,
        'Pod::Functions'        => undef,
        'Pod::Html'             => 1.01,
        'Pod::Text'             => 1.0203,
        'POSIX'                 => 1.02,
        're'                    => 0.02,
        'Safe'                  => 2.06,
        'SDBM_File'             => '1.00',
        'Search::Dict'          => undef,
        'SelectSaver'           => undef,
        'SelfLoader'            => 1.08,
        'Shell'                 => undef,
        'sigtrap'               => 1.02,
        'Socket'                => 1.7,
        'strict'                => 1.01,
        'subs'                  => undef,
        'Symbol'                => 1.02,
        'Sys::Hostname'         => undef,
        'Sys::Syslog'           => undef,
        'Term::Cap'             => undef,
        'Term::Complete'        => undef,
        'Term::ReadLine'        => undef,
        'Test'                  => 1.122,
        'Test::Harness'         => 1.1602,
        'Text::Abbrev'          => undef,
        'Text::ParseWords'      => 3.1,
        'Text::Soundex'         => undef,
        'Text::Tabs'            => 96.121201,
        'Text::Wrap'            => 98.112902,
        'Thread'                => '1.0',
        'Thread::Queue'         => undef,
        'Thread::Semaphore'     => undef,
        'Thread::Specific'      => undef,
        'Thread::Signal'        => undef,
        'Tie::Array'            => '1.00',
        'Tie::Handle'           => undef,
        'Tie::Hash'             => undef,
        'Tie::RefHash'          => undef,
        'Tie::Scalar'           => undef,
        'Tie::SubstrHash'       => undef,
        'Time::gmtime'          => 1.01,
        'Time::Local'           => undef,
        'Time::localtime'       => 1.01,
        'Time::tm'              => undef,
        'UNIVERSAL'             => undef,
        'User::grent'           => undef,
        'User::pwent'           => undef,
        'vars'                  => undef,
        'VMS::DCLsym'           => 1.01,
        'VMS::Filespec'         => undef,
        'VMS::Stdio'            => 2.1,
        'vmsish'                => undef,
    },

    5.00405   => {
        'AnyDBM_File'           => undef, #./lib/AnyDBM_File.pm
        'attrs'                 => '0.1', #./lib/attrs.pm
        'AutoLoader'            => '5.56', #./lib/AutoLoader.pm
        'AutoSplit'             => '1.0303', #./lib/AutoSplit.pm
        'autouse'               => '1.01', #./lib/autouse.pm
        'base'                  => undef, #./lib/base.pm
        'Benchmark'             => undef, #./lib/Benchmark.pm
        'blib'                  => '1.00', #./lib/blib.pm
        'Bundle::CPAN'          => '0.03', #./lib/Bundle/CPAN.pm
        'Carp'                  => undef, #./lib/Carp.pm
        'CGI'                   => '2.42', #./lib/CGI.pm
        'CGI::Apache'           => '1.1', #./lib/CGI/Apache.pm
        'CGI::Carp'             => '1.10', #./lib/CGI/Carp.pm
        'CGI::Cookie'           => '1.06', #./lib/CGI/Cookie.pm
        'CGI::Fast'             => '1.00a', #./lib/CGI/Fast.pm
        'CGI::Push'             => '1.01', #./lib/CGI/Push.pm
        'CGI::Switch'           => '0.06', #./lib/CGI/Switch.pm
        'Class::Struct'         => undef, #./lib/Class/Struct.pm
        'Config'                => undef,
        'constant'              => '1.00', #./lib/constant.pm
        'CPAN'                  => '1.40', #./lib/CPAN.pm
        'CPAN::FirstTime'       => '1.30 ', #./lib/CPAN/FirstTime.pm
        'CPAN::Nox'             => undef, #./lib/CPAN/Nox.pm
        'Cwd'                   => '2.01', #./lib/Cwd.pm
        'DB_File'               => '1.15', #./ext/DB_File/DB_File.pm
        'Devel::SelfStubber'    => '1.01', #./lib/Devel/SelfStubber.pm
        'diagnostics'           => undef, #./lib/diagnostics.pm
        'DirHandle'             => undef, #./lib/DirHandle.pm
        'DynaLoader'            => '1.03',
        'English'               => undef, #./lib/English.pm
        'Env'                   => undef, #./lib/Env.pm
        'Exporter'              => undef, #./lib/Exporter.pm
        'ExtUtils::Command'     => '1.01', #./lib/ExtUtils/Command.pm
        'ExtUtils::Embed'       => '1.2505', #./lib/ExtUtils/Embed.pm
        'ExtUtils::Install'     => '1.28 ', #./lib/ExtUtils/Install.pm
        'ExtUtils::Liblist'     => '1.25 ', #./lib/ExtUtils/Liblist.pm
        'ExtUtils::MakeMaker'   => '5.42', #./lib/ExtUtils/MakeMaker.pm
        'ExtUtils::Manifest'    => '1.33 ', #./lib/ExtUtils/Manifest.pm
        'ExtUtils::Mkbootstrap' => '1.14 ', #./lib/ExtUtils/Mkbootstrap.pm
        'ExtUtils::Mksymlists'  => '1.16 ', #./lib/ExtUtils/Mksymlists.pm
        'ExtUtils::MM_OS2'      => undef, #./lib/ExtUtils/MM_OS2.pm
        'ExtUtils::MM_Unix'     => '1.118 ', #./lib/ExtUtils/MM_Unix.pm
        'ExtUtils::MM_VMS'      => undef, #./lib/ExtUtils/MM_VMS.pm
        'ExtUtils::MM_Win32'    => undef, #./lib/ExtUtils/MM_Win32.pm
        'ExtUtils::testlib'     => '1.11 ', #./lib/ExtUtils/testlib.pm
        'ExtUtils::XSSymSet'    => '1.0', #./vms/ext/XSSymSet.pm
        'Fcntl'                 => '1.03', #./ext/Fcntl/Fcntl.pm
        'File::Basename'        => '2.6', #./lib/File/Basename.pm
        'File::CheckTree'       => undef, #./lib/File/CheckTree.pm
        'File::Compare'         => '1.1001', #./lib/File/Compare.pm
        'File::Copy'            => '2.02', #./lib/File/Copy.pm
        'File::DosGlob'         => undef, #./lib/File/DosGlob.pm
        'File::Find'            => undef, #./lib/File/Find.pm
        'File::Path'            => '1.0402', #./lib/File/Path.pm
        'File::Spec'            => '0.6', #./lib/File/Spec.pm
        'File::Spec::Mac'       => '1.0', #./lib/File/Spec/Mac.pm
        'File::Spec::OS2'       => undef, #./lib/File/Spec/OS2.pm
        'File::Spec::Unix'      => undef, #./lib/File/Spec/Unix.pm
        'File::Spec::VMS'       => undef, #./lib/File/Spec/VMS.pm
        'File::Spec::Win32'     => undef, #./lib/File/Spec/Win32.pm
        'File::stat'            => undef, #./lib/File/stat.pm
        'FileCache'             => undef, #./lib/FileCache.pm
        'FileHandle'            => '2.00', #./lib/FileHandle.pm
        'FindBin'               => '1.41', #./lib/FindBin.pm
        'GDBM_File'             => '1.00', #./ext/GDBM_File/GDBM_File.pm
        'Getopt::Long'          => '2.19', #./lib/Getopt/Long.pm
        'Getopt::Std'           => undef, #./lib/Getopt/Std.pm
        'I18N::Collate'         => undef, #./lib/I18N/Collate.pm
        'integer'               => undef, #./lib/integer.pm
        'IO'                    => undef, #./ext/IO/IO.pm
        'IO::File'              => '1.06021', #./ext/IO/lib/IO/File.pm
        'IO::Handle'            => '1.1504', #./ext/IO/lib/IO/Handle.pm
        'IO::Pipe'              => '1.0901', #./ext/IO/lib/IO/Pipe.pm
        'IO::Seekable'          => '1.06', #./ext/IO/lib/IO/Seekable.pm
        'IO::Select'            => '1.10', #./ext/IO/lib/IO/Select.pm
        'IO::Socket'            => '1.1603', #./ext/IO/lib/IO/Socket.pm
        'IPC::Open2'            => '1.01', #./lib/IPC/Open2.pm
        'IPC::Open3'            => '1.0103', #./lib/IPC/Open3.pm
        'less'                  => undef, #./lib/less.pm
        'lib'                   => undef, #./lib/lib.pm
        'locale'                => undef, #./lib/locale.pm
        'Math::BigFloat'        => undef, #./lib/Math/BigFloat.pm
        'Math::BigInt'          => undef, #./lib/Math/BigInt.pm
        'Math::Complex'         => '1.25', #./lib/Math/Complex.pm
        'Math::Trig'            => '1', #./lib/Math/Trig.pm
        'NDBM_File'             => '1.01', #./ext/NDBM_File/NDBM_File.pm
        'Net::hostent'          => undef, #./lib/Net/hostent.pm
        'Net::netent'           => undef, #./lib/Net/netent.pm
        'Net::Ping'             => '2.02', #./lib/Net/Ping.pm
        'Net::protoent'         => undef, #./lib/Net/protoent.pm
        'Net::servent'          => undef, #./lib/Net/servent.pm
        'ODBM_File'             => '1.00', #./ext/ODBM_File/ODBM_File.pm
        'Opcode'                => '1.04', #./ext/Opcode/Opcode.pm
        'ops'                   => undef, #./ext/Opcode/ops.pm
        'OS2::ExtAttr'          => '0.01', #./os2/OS2/ExtAttr/ExtAttr.pm
        'OS2::PrfDB'            => '0.02', #./os2/OS2/PrfDB/PrfDB.pm
        'OS2::Process'          => undef, #./os2/OS2/Process/Process.pm
        'OS2::REXX'             => undef, #./os2/OS2/REXX/REXX.pm
        'overload'              => undef, #./lib/overload.pm
        'Pod::Functions'        => undef, #./lib/Pod/Functions.pm
        'Pod::Html'             => '1.0101', #./lib/Pod/Html.pm
        'Pod::Text'             => '1.0204', #./lib/Pod/Text.pm
        'POSIX'                 => '1.02', #./ext/POSIX/POSIX.pm
        're'                    => undef, #./lib/re.pm
        'Safe'                  => '2.06', #./ext/Opcode/Safe.pm
        'SDBM_File'             => '1.00', #./ext/SDBM_File/SDBM_File.pm
        'Search::Dict'          => undef, #./lib/Search/Dict.pm
        'SelectSaver'           => undef, #./lib/SelectSaver.pm
        'SelfLoader'            => '1.08', #./lib/SelfLoader.pm
        'Shell'                 => undef, #./lib/Shell.pm
        'sigtrap'               => '1.02', #./lib/sigtrap.pm
        'Socket'                => '1.7', #./ext/Socket/Socket.pm
        'strict'                => '1.01', #./lib/strict.pm
        'subs'                  => undef, #./lib/subs.pm
        'Symbol'                => '1.02', #./lib/Symbol.pm
        'Sys::Hostname'         => undef, #./lib/Sys/Hostname.pm
        'Sys::Syslog'           => undef, #./lib/Sys/Syslog.pm
        'Term::Cap'             => undef, #./lib/Term/Cap.pm
        'Term::Complete'        => undef, #./lib/Term/Complete.pm
        'Term::ReadLine'        => undef, #./lib/Term/ReadLine.pm
        'Test'                  => '1.04', #./lib/Test.pm
        'Test::Harness'         => '1.1602', #./lib/Test/Harness.pm
        'Text::Abbrev'          => undef, #./lib/Text/Abbrev.pm
        'Text::ParseWords'      => '3.1001', #./lib/Text/ParseWords.pm
        'Text::Soundex'         => undef, #./lib/Text/Soundex.pm
        'Text::Tabs'            => '96.121201', #./lib/Text/Tabs.pm
        'Text::Wrap'            => '98.112902', #./lib/Text/Wrap.pm
        'Tie::Handle'           => undef, #./lib/Tie/Handle.pm
        'Tie::Hash'             => undef, #./lib/Tie/Hash.pm
        'Tie::RefHash'          => undef, #./lib/Tie/RefHash.pm
        'Tie::Scalar'           => undef, #./lib/Tie/Scalar.pm
        'Tie::SubstrHash'       => undef, #./lib/Tie/SubstrHash.pm
        'Time::gmtime'          => '1.01', #./lib/Time/gmtime.pm
        'Time::Local'           => undef, #./lib/Time/Local.pm
        'Time::localtime'       => '1.01', #./lib/Time/localtime.pm
        'Time::tm'              => undef, #./lib/Time/tm.pm
        'UNIVERSAL'             => undef, #./lib/UNIVERSAL.pm
        'User::grent'           => undef, #./lib/User/grent.pm
        'User::pwent'           => undef, #./lib/User/pwent.pm
        'vars'                  => undef, #./lib/vars.pm
        'VMS::DCLsym'           => '1.01', #./vms/ext/DCLsym/DCLsym.pm
        'VMS::Filespec'         => undef, #./vms/ext/Filespec.pm
        'VMS::Stdio'            => '2.02', #./vms/ext/Stdio/Stdio.pm
        'vmsish'                => undef, #./vms/ext/vmsish.pm
    },

    5.00504 => {
        'AnyDBM_File'           => undef,  #lib/AnyDBM_File.pm
        'attrs'                 => '1.0',  #lib/attrs.pm
        'AutoLoader'            => undef,  #lib/AutoLoader.pm
        'AutoSplit'             => '1.0303',  #lib/AutoSplit.pm
        'autouse'               => '1.01',  #lib/autouse.pm
        'base'                  => undef,  #lib/base.pm
        'B::Asmdata'            => undef,  #lib/B/Asmdata.pm
        'B::Assembler'          => undef,  #lib/B/Assembler.pm
        'B::Bblock'             => undef,  #lib/B/Bblock.pm
        'B::Bytecode'           => undef,  #lib/B/Bytecode.pm
        'B::CC'                 => undef,  #lib/B/CC.pm
        'B::C'                  => undef,  #lib/B/C.pm
        'B::Debug'              => undef,  #lib/B/Debug.pm
        'B::Deparse'            => '0.56',  #lib/B/Deparse.pm
        'B::Disassembler'       => undef,  #lib/B/Disassembler.pm
        'Benchmark'             => undef,  #lib/Benchmark.pm
        'blib'                  => '1.00',  #lib/blib.pm
        'B::Lint'               => undef,  #lib/B/Lint.pm
        'B::Showlex'            => undef,  #lib/B/Showlex.pm
        'B::Stackobj'           => undef,  #lib/B/Stackobj.pm
        'B::Terse'              => undef,  #lib/B/Terse.pm
        'B'                     => undef,  #lib/B.pm
        'B::Xref'               => undef,  #lib/B/Xref.pm
        'Carp'                  => undef,  #lib/Carp.pm
        'CGI'                   => '2.46',  #lib/CGI.pm
        'CGI::Apache'           => '1.1',  #lib/CGI/Apache.pm
        'CGI::Carp'             => '1.13',  #lib/CGI/Carp.pm
        'CGI::Cookie'           => '1.06',  #lib/CGI/Cookie.pm
        'CGI::Fast'             => '1.01',  #lib/CGI/Fast.pm
        'CGI::Push'             => '1.01',  #lib/CGI/Push.pm
        'CGI::Switch'           => '0.06',  #lib/CGI/Switch.pm
        'Class::Struct'         => undef,  #lib/Class/Struct.pm
        'Config'                => undef,  #lib/Config.pm
        'constant'              => '1.00',  #lib/constant.pm
        'CPAN'                  => '1.48',  #lib/CPAN.pm
        'CPAN::FirstTime'       => '1.36 ',  #lib/CPAN/FirstTime.pm
        'CPAN::Nox'             => '1.00',  #lib/CPAN/Nox.pm
        'Cwd'                   => '2.01',  #lib/Cwd.pm
        'Data::Dumper'          => '2.101',  #lib/Data/Dumper.pm
        'DB_File'               => '1.807',  #lib/DB_File.pm
        'Devel::SelfStubber'    => '1.01',  #lib/Devel/SelfStubber.pm
        'diagnostics'           => undef,  #lib/diagnostics.pm
        'DirHandle'             => undef,  #lib/DirHandle.pm
        'Dumpvalue'             => undef,  #lib/Dumpvalue.pm
        'DynaLoader'            => '1.03',  #lib/DynaLoader.pm
        'English'               => undef,  #lib/English.pm
        'Env'                   => undef,  #lib/Env.pm
        'Errno'                 => '1.111',  #lib/Errno.pm
        'Exporter'              => undef,  #lib/Exporter.pm
        'ExtUtils::Command'     => '1.01',  #lib/ExtUtils/Command.pm
        'ExtUtils::Embed'       => '1.2505',  #lib/ExtUtils/Embed.pm
        'ExtUtils::Install'     => '1.28 ',  #lib/ExtUtils/Install.pm
        'ExtUtils::Installed'   => '0.02',  #lib/ExtUtils/Installed.pm
        'ExtUtils::Liblist'     => '1.25 ',  #lib/ExtUtils/Liblist.pm
        'ExtUtils::MakeMaker'   => '5.4302',  #lib/ExtUtils/MakeMaker.pm
        'ExtUtils::Manifest'    => '1.33 ',  #lib/ExtUtils/Manifest.pm
        'ExtUtils::Miniperl'    => undef,  #lib/ExtUtils/Miniperl.pm
        'ExtUtils::Mkbootstrap' => '1.14 ',  #lib/ExtUtils/Mkbootstrap.pm
        'ExtUtils::Mksymlists'  => '1.17 ',  #lib/ExtUtils/Mksymlists.pm
        'ExtUtils::MM_OS2'      => undef,  #lib/ExtUtils/MM_OS2.pm
        'ExtUtils::MM_Unix'     => '1.12602 ',  #lib/ExtUtils/MM_Unix.pm
        'ExtUtils::MM_VMS'      => undef,  #lib/ExtUtils/MM_VMS.pm
        'ExtUtils::MM_Win32'    => undef,  #lib/ExtUtils/MM_Win32.pm
        'ExtUtils::Packlist'    => '0.03',  #lib/ExtUtils/Packlist.pm
        'ExtUtils::testlib'     => '1.11 ',  #lib/ExtUtils/testlib.pm
        'ExtUtils::XSSymSet'    => '1.0',  #vms/ext/XSSymSet.pm
        'Fatal'                 => '1.02',  #lib/Fatal.pm
        'Fcntl'                 => '1.03',  #lib/Fcntl.pm
        'fields'                => '0.02',  #lib/fields.pm
        'File::Basename'        => '2.6',  #lib/File/Basename.pm
        'FileCache'             => undef,  #lib/FileCache.pm
        'File::CheckTree'       => undef,  #lib/File/CheckTree.pm
        'File::Compare'         => '1.1002',  #lib/File/Compare.pm
        'File::Copy'            => '2.02',  #lib/File/Copy.pm
        'File::DosGlob'         => undef,  #lib/File/DosGlob.pm
        'File::Find'            => undef,  #lib/File/Find.pm
        'FileHandle'            => '2.00',  #lib/FileHandle.pm
        'File::Path'            => '1.0401',  #lib/File/Path.pm
        'File::Spec'            => '0.8',  #lib/File/Spec.pm
        'File::Spec::Functions' => undef,  #lib/File/Spec/Functions.pm
        'File::Spec::Mac'       => undef,  #lib/File/Spec/Mac.pm
        'File::Spec::OS2'       => undef,  #lib/File/Spec/OS2.pm
        'File::Spec::Unix'      => undef,  #lib/File/Spec/Unix.pm
        'File::Spec::VMS'       => undef,  #lib/File/Spec/VMS.pm
        'File::Spec::Win32'     => undef,  #lib/File/Spec/Win32.pm
        'File::stat'            => undef,  #lib/File/stat.pm
        'FindBin'               => '1.42',  #lib/FindBin.pm
        'GDBM_File'             => '1.00',  #lib/GDBM_File.pm
        'Getopt::Long'          => '2.20',  #lib/Getopt/Long.pm
        'Getopt::Std'           => '1.01',  #lib/Getopt/Std.pm
        'I18N::Collate'         => undef,  #lib/I18N/Collate.pm
        'integer'               => undef,  #lib/integer.pm
        'IO::File'              => '1.06021',  #lib/IO/File.pm
        'IO::Handle'            => '1.1505',  #lib/IO/Handle.pm
        'IO::Pipe'              => '1.0902',  #lib/IO/Pipe.pm
        'IO::Seekable'          => '1.06',  #lib/IO/Seekable.pm
        'IO::Select'            => '1.10',  #lib/IO/Select.pm
        'IO::Socket'            => '1.1603',  #lib/IO/Socket.pm
        'IO'                    => undef,  #lib/IO.pm
        'IPC::Msg'              => '1.00',  #lib/IPC/Msg.pm
        'IPC::Open2'            => '1.01',  #lib/IPC/Open2.pm
        'IPC::Open3'            => '1.0103',  #lib/IPC/Open3.pm
        'IPC::Semaphore'        => '1.00',  #lib/IPC/Semaphore.pm
        'IPC::SysV'             => '1.03',  #lib/IPC/SysV.pm
        'less'                  => undef,  #lib/less.pm
        'lib'                   => undef,  #lib/lib.pm
        'locale'                => undef,  #lib/locale.pm
        'Math::BigFloat'        => undef,  #lib/Math/BigFloat.pm
        'Math::BigInt'          => undef,  #lib/Math/BigInt.pm
        'Math::Complex'         => '1.26',  #lib/Math/Complex.pm
        'Math::Trig'            => '1',  #lib/Math/Trig.pm
        'NDBM_File'             => '1.01',  #ext/NDBM_File/NDBM_File.pm
        'Net::hostent'          => undef,  #lib/Net/hostent.pm
        'Net::netent'           => undef,  #lib/Net/netent.pm
        'Net::Ping'             => '2.02',  #lib/Net/Ping.pm
        'Net::protoent'         => undef,  #lib/Net/protoent.pm
        'Net::servent'          => undef,  #lib/Net/servent.pm
        'ODBM_File'             => '1.00', #ext/ODBM_File/ODBM_File.pm
        'Opcode'                => '1.04',  #lib/Opcode.pm
        'ops'                   => undef,  #lib/ops.pm
        'O'                     => undef,  #lib/O.pm
        'OS2::ExtAttr'          => '0.01',  #os2/OS2/ExtAttr/ExtAttr.pm
        'OS2::PrfDB'            => '0.02',  #os2/OS2/PrfDB/PrfDB.pm
        'OS2::Process'          => '0.2',  #os2/OS2/Process/Process.pm
        'OS2::REXX'             => undef,  #os2/OS2/REXX/REXX.pm
        'overload'              => undef,  #lib/overload.pm
        'Pod::Functions'        => undef,  #lib/Pod/Functions.pm
        'Pod::Html'             => '1.02',  #lib/Pod/Html.pm
        'Pod::Text'             => '1.0203',  #lib/Pod/Text.pm
        'POSIX'                 => '1.02',  #lib/POSIX.pm
        're'                    => '0.02',  #lib/re.pm
        'Safe'                  => '2.06',  #lib/Safe.pm
        'SDBM_File'             => '1.00',  #lib/SDBM_File.pm
        'Search::Dict'          => undef,  #lib/Search/Dict.pm
        'SelectSaver'           => undef,  #lib/SelectSaver.pm
        'SelfLoader'            => '1.08',  #lib/SelfLoader.pm
        'Shell'                 => undef,  #lib/Shell.pm
        'sigtrap'               => '1.02',  #lib/sigtrap.pm
        'Socket'                => '1.7',  #lib/Socket.pm
        'strict'                => '1.01',  #lib/strict.pm
        'subs'                  => undef,  #lib/subs.pm
        'Symbol'                => '1.02',  #lib/Symbol.pm
        'Sys::Hostname'         => undef,  #lib/Sys/Hostname.pm
        'Sys::Syslog'           => undef,  #lib/Sys/Syslog.pm
        'Term::Cap'             => undef,  #lib/Term/Cap.pm
        'Term::Complete'        => undef,  #lib/Term/Complete.pm
        'Term::ReadLine'        => undef,  #lib/Term/ReadLine.pm
        'Test'                  => '1.122',  #lib/Test.pm
        'Test::Harness'         => '1.1602',  #lib/Test/Harness.pm
        'Text::Abbrev'          => undef,  #lib/Text/Abbrev.pm
        'Text::ParseWords'      => '3.1',  #lib/Text/ParseWords.pm
        'Text::Soundex'         => undef,  #lib/Text/Soundex.pm
        'Text::Tabs'            => '96.121201',  #lib/Text/Tabs.pm
        'Text::Wrap'            => '98.112902',  #lib/Text/Wrap.pm
        'Thread'                => '1.0',  #ext/Thread/Thread.pm
        'Thread::Queue'         => undef,  #ext/Thread/Thread/Queue.pm
        'Thread::Semaphore'     => undef,  #ext/Thread/Thread/Semaphore.pm
        'Thread::Signal'        => undef,  #ext/Thread/Thread/Signal.pm
        'Thread::Specific'      => undef,  #ext/Thread/Thread/Specific.pm
        'Tie::Array'            => '1.00',  #lib/Tie/Array.pm
        'Tie::Handle'           => undef,  #lib/Tie/Handle.pm
        'Tie::Hash'             => undef,  #lib/Tie/Hash.pm
        'Tie::RefHash'          => undef,  #lib/Tie/RefHash.pm
        'Tie::Scalar'           => undef,  #lib/Tie/Scalar.pm
        'Tie::SubstrHash'       => undef,  #lib/Tie/SubstrHash.pm
        'Time::gmtime'          => '1.01',  #lib/Time/gmtime.pm
        'Time::localtime'       => '1.01',  #lib/Time/localtime.pm
        'Time::Local'           => undef,  #lib/Time/Local.pm
        'Time::tm'              => undef,  #lib/Time/tm.pm
        'UNIVERSAL'             => undef,  #lib/UNIVERSAL.pm
        'User::grent'           => undef,  #lib/User/grent.pm
        'User::pwent'           => undef,  #lib/User/pwent.pm
        'vars'                  => undef,  #lib/vars.pm
        'VMS::DCLsym'           => '1.01',  #vms/ext/DCLsym/DCLsym.pm
        'VMS::Filespec'         => undef,  #vms/ext/Filespec.pm
        'VMS::Stdio'            => '2.1',  #vms/ext/Stdio/Stdio.pm
        'vmsish'                => undef,  #vms/ext/vmsish.pm
    },

    5.006   => {
        'AnyDBM_File'           => undef, #./lib/AnyDBM_File.pm
        'AutoLoader'            => '5.57', #./lib/AutoLoader.pm
        'AutoSplit'             => '1.0305', #./lib/AutoSplit.pm
        'B'                     => undef, #./ext/B/B.pm
        'B::Asmdata'            => undef, #./ext/B/B/Asmdata.pm
        'B::Assembler'          => undef, #./ext/B/B/Assembler.pm
        'B::Bblock'             => undef, #./ext/B/B/Bblock.pm
        'B::Bytecode'           => undef, #./ext/B/B/Bytecode.pm
        'B::C'                  => undef, #./ext/B/B/C.pm
        'B::CC'                 => undef, #./ext/B/B/CC.pm
        'B::Debug'              => undef, #./ext/B/B/Debug.pm
        'B::Deparse'            => '0.59', #./ext/B/B/Deparse.pm
        'B::Disassembler'       => undef, #./ext/B/B/Disassembler.pm
        'B::Lint'               => undef, #./ext/B/B/Lint.pm
        'B::Showlex'            => undef, #./ext/B/B/Showlex.pm
        'B::Stackobj'           => undef, #./ext/B/B/Stackobj.pm
        'B::Stash'              => undef, #./ext/B/B/Stash.pm
        'B::Terse'              => undef, #./ext/B/B/Terse.pm
        'B::Xref'               => undef, #./ext/B/B/Xref.pm
        'Benchmark'             => '1', #./lib/Benchmark.pm
        'ByteLoader'            => '0.03', #./ext/ByteLoader/ByteLoader.pm
        'CGI'                   => '2.56', #./lib/CGI.pm
        'CGI::Apache'           => undef, #./lib/CGI/Apache.pm
        'CGI::Carp'             => '1.14', #./lib/CGI/Carp.pm
        'CGI::Cookie'           => '1.12', #./lib/CGI/Cookie.pm
        'CGI::Fast'             => '1.02', #./lib/CGI/Fast.pm
        'CGI::Pretty'           => '1.03', #./lib/CGI/Pretty.pm
        'CGI::Push'             => '1.01', #./lib/CGI/Push.pm
        'CGI::Switch'           => undef, #./lib/CGI/Switch.pm
        'CPAN'                  => '1.52', #./lib/CPAN.pm
        'CPAN::FirstTime'       => '1.38 ', #./lib/CPAN/FirstTime.pm
        'CPAN::Nox'             => '1.00', #./lib/CPAN/Nox.pm
        'Carp'                  => undef, #./lib/Carp.pm
        'Carp::Heavy'           => undef, #./lib/Carp/Heavy.pm
        'Class::Struct'         => '0.58', #./lib/Class/Struct.pm
        'Config'                => undef,
        'Cwd'                   => '2.02', #./lib/Cwd.pm
        'DB'                    => '1.0', #./lib/DB.pm
        'DB_File'               => '1.72', #./ext/DB_File/DB_File.pm
        'Data::Dumper'          => '2.101', #./ext/Data/Dumper/Dumper.pm
        'Devel::DProf'          => '20000000.00_00', #./ext/Devel/DProf/DProf.pm
        'Devel::Peek'           => '1.00_01', #./ext/Devel/Peek/Peek.pm
        'Devel::SelfStubber'    => '1.01', #./lib/Devel/SelfStubber.pm
        'DirHandle'             => undef, #./lib/DirHandle.pm
        'Dumpvalue'             => undef, #./lib/Dumpvalue.pm
        'DynaLoader'            => '1.04',
        'English'               => undef, #./lib/English.pm
        'Env'                   => undef, #./lib/Env.pm
        'Exporter'              => '5.562', #./lib/Exporter.pm
        'Exporter::Heavy'       => undef, #./lib/Exporter/Heavy.pm
        'ExtUtils::Command'     => '1.01', #./lib/ExtUtils/Command.pm
        'ExtUtils::Embed'       => '1.2505', #./lib/ExtUtils/Embed.pm
        'ExtUtils::Install'     => '1.28 ', #./lib/ExtUtils/Install.pm
        'ExtUtils::Installed'   => '0.02', #./lib/ExtUtils/Installed.pm
        'ExtUtils::Liblist'     => '1.25 ', #./lib/ExtUtils/Liblist.pm
        'ExtUtils::MM_Cygwin'   => undef, #./lib/ExtUtils/MM_Cygwin.pm
        'ExtUtils::MM_OS2'      => undef, #./lib/ExtUtils/MM_OS2.pm
        'ExtUtils::MM_Unix'     => '1.12603 ', #./lib/ExtUtils/MM_Unix.pm
        'ExtUtils::MM_VMS'      => undef, #./lib/ExtUtils/MM_VMS.pm
        'ExtUtils::MM_Win32'    => undef, #./lib/ExtUtils/MM_Win32.pm
        'ExtUtils::MakeMaker'   => '5.45', #./lib/ExtUtils/MakeMaker.pm
        'ExtUtils::Manifest'    => '1.33 ', #./lib/ExtUtils/Manifest.pm
        'ExtUtils::Miniperl'    => undef,
        'ExtUtils::Mkbootstrap' => '1.14 ', #./lib/ExtUtils/Mkbootstrap.pm
        'ExtUtils::Mksymlists'  => '1.17 ', #./lib/ExtUtils/Mksymlists.pm
        'ExtUtils::Packlist'    => '0.03', #./lib/ExtUtils/Packlist.pm
        'ExtUtils::XSSymSet'    => '1.0', #./vms/ext/XSSymSet.pm
        'ExtUtils::testlib'     => '1.11 ', #./lib/ExtUtils/testlib.pm
        'Fatal'                 => '1.02', #./lib/Fatal.pm
        'Fcntl'                 => '1.03', #./ext/Fcntl/Fcntl.pm
        'File::Basename'        => '2.6', #./lib/File/Basename.pm
        'File::CheckTree'       => undef, #./lib/File/CheckTree.pm
        'File::Compare'         => '1.1002', #./lib/File/Compare.pm
        'File::Copy'            => '2.03', #./lib/File/Copy.pm
        'File::DosGlob'         => undef, #./lib/File/DosGlob.pm
        'File::Find'            => undef, #./lib/File/Find.pm
        'File::Glob'            => '0.991', #./ext/File/Glob/Glob.pm
        'File::Path'            => '1.0403', #./lib/File/Path.pm
        'File::Spec'            => '0.8', #./lib/File/Spec.pm
        'File::Spec::Functions' => undef, #./lib/File/Spec/Functions.pm
        'File::Spec::Mac'       => undef, #./lib/File/Spec/Mac.pm
        'File::Spec::OS2'       => undef, #./lib/File/Spec/OS2.pm
        'File::Spec::Unix'      => undef, #./lib/File/Spec/Unix.pm
        'File::Spec::VMS'       => undef, #./lib/File/Spec/VMS.pm
        'File::Spec::Win32'     => undef, #./lib/File/Spec/Win32.pm
        'File::stat'            => undef, #./lib/File/stat.pm
        'FileCache'             => undef, #./lib/FileCache.pm
        'FileHandle'            => '2.00', #./lib/FileHandle.pm
        'FindBin'               => '1.42', #./lib/FindBin.pm
        'GDBM_File'             => '1.03', #./ext/GDBM_File/GDBM_File.pm
        'Getopt::Long'          => '2.23', #./lib/Getopt/Long.pm
        'Getopt::Std'           => '1.02', #./lib/Getopt/Std.pm
        'I18N::Collate'         => undef, #./lib/I18N/Collate.pm
        'IO'                    => '1.20', #./ext/IO/IO.pm
        'IO::Dir'               => '1.03', #./ext/IO/lib/IO/Dir.pm
        'IO::File'              => '1.08', #./ext/IO/lib/IO/File.pm
        'IO::Handle'            => '1.21', #./ext/IO/lib/IO/Handle.pm
        'IO::Pipe'              => '1.121', #./ext/IO/lib/IO/Pipe.pm
        'IO::Poll'              => '0.01', #./ext/IO/lib/IO/Poll.pm
        'IO::Seekable'          => '1.08', #./ext/IO/lib/IO/Seekable.pm
        'IO::Select'            => '1.14', #./ext/IO/lib/IO/Select.pm
        'IO::Socket'            => '1.26', #./ext/IO/lib/IO/Socket.pm
        'IO::Socket::INET'      => '1.25', #./ext/IO/lib/IO/Socket/INET.pm
        'IO::Socket::UNIX'      => '1.20', #./ext/IO/lib/IO/Socket/UNIX.pm
        'IPC::Open2'            => '1.01', #./lib/IPC/Open2.pm
        'IPC::Open3'            => '1.0103', #./lib/IPC/Open3.pm
        'IPC::Msg'              => '1.00', #./ext/IPC/SysV/Msg.pm
        'IPC::Semaphore'        => '1.00', #./ext/IPC/SysV/Semaphore.pm
        'IPC::SysV'             => '1.03', #./ext/IPC/SysV/SysV.pm
        'JNI'                   => '0.01', #./jpl/JNI/JNI.pm
        'JPL::AutoLoader'       => undef, #./jpl/JPL/AutoLoader.pm
        'JPL::Class'            => undef, #./jpl/JPL/Class.pm
        'JPL::Compile'          => undef, #./jpl/JPL/Compile.pm
        'Math::BigFloat'        => undef, #./lib/Math/BigFloat.pm
        'Math::BigInt'          => undef, #./lib/Math/BigInt.pm
        'Math::Complex'         => '1.26', #./lib/Math/Complex.pm
        'Math::Trig'            => '1', #./lib/Math/Trig.pm
        'NDBM_File'             => '1.03', #./ext/NDBM_File/NDBM_File.pm
        'Net::Ping'             => '2.02', #./lib/Net/Ping.pm
        'Net::hostent'          => undef, #./lib/Net/hostent.pm
        'Net::netent'           => undef, #./lib/Net/netent.pm
        'Net::protoent'         => undef, #./lib/Net/protoent.pm
        'Net::servent'          => undef, #./lib/Net/servent.pm
        'O'                     => undef, #./ext/B/O.pm
        'ODBM_File'             => '1.02', #./ext/ODBM_File/ODBM_File.pm
        'OS2::ExtAttr'          => '0.01', #./os2/OS2/ExtAttr/ExtAttr.pm
        'OS2::PrfDB'            => '0.02', #./os2/OS2/PrfDB/PrfDB.pm
        'OS2::Process'          => '0.2', #./os2/OS2/Process/Process.pm
        'OS2::REXX'             => undef, #./os2/OS2/REXX/REXX.pm
        'OS2::DLL'              => undef, #./os2/OS2/REXX/DLL/DLL.pm
        'Opcode'                => '1.04', #./ext/Opcode/Opcode.pm
        'POSIX'                 => '1.03', #./ext/POSIX/POSIX.pm
        'Pod::Checker'          => '1.098', #./lib/Pod/Checker.pm
        'Pod::Find'             => '0.12', #./lib/Pod/Find.pm
        'Pod::Functions'        => undef, #./lib/Pod/Functions.pm
        'Pod::Html'             => '1.03', #./lib/Pod/Html.pm
        'Pod::InputObjects'     => '1.12', #./lib/Pod/InputObjects.pm
        'Pod::Man'              => '1.02', #./lib/Pod/Man.pm
        'Pod::ParseUtils'       => '0.2', #./lib/Pod/ParseUtils.pm
        'Pod::Parser'           => '1.12', #./lib/Pod/Parser.pm
        'Pod::Plainer'          => '0.01', #./lib/Pod/Plainer.pm
        'Pod::Select'           => '1.12', #./lib/Pod/Select.pm
        'Pod::Text'             => '2.03', #./lib/Pod/Text.pm
        'Pod::Text::Color'      => '0.05', #./lib/Pod/Text/Color.pm
        'Pod::Text::Termcap'    => '0.04', #./lib/Pod/Text/Termcap.pm
        'Pod::Usage'            => '1.12', #./lib/Pod/Usage.pm
        'SDBM_File'             => '1.02', #./ext/SDBM_File/SDBM_File.pm
        'Safe'                  => '2.06', #./ext/Opcode/Safe.pm
        'Search::Dict'          => undef, #./lib/Search/Dict.pm
        'SelectSaver'           => undef, #./lib/SelectSaver.pm
        'SelfLoader'            => '1.0901', #./lib/SelfLoader.pm
        'Shell'                 => '0.2', #./lib/Shell.pm
        'Socket'                => '1.72', #./ext/Socket/Socket.pm
        'Symbol'                => '1.02', #./lib/Symbol.pm
        'Sys::Hostname'         => '1.1', #./ext/Sys/Hostname/Hostname.pm
        'Sys::Syslog'           => '0.01', #./ext/Sys/Syslog/Syslog.pm
        'Term::ANSIColor'       => '1.01', #./lib/Term/ANSIColor.pm
        'Term::Cap'             => undef, #./lib/Term/Cap.pm
        'Term::Complete'        => undef, #./lib/Term/Complete.pm
        'Term::ReadLine'        => undef, #./lib/Term/ReadLine.pm
        'Test'                  => '1.13', #./lib/Test.pm
        'Test::Harness'         => '1.1604', #./lib/Test/Harness.pm
        'Text::Abbrev'          => undef, #./lib/Text/Abbrev.pm
        'Text::ParseWords'      => '3.2', #./lib/Text/ParseWords.pm
        'Text::Soundex'         => '1.0', #./lib/Text/Soundex.pm
        'Text::Tabs'            => '98.112801', #./lib/Text/Tabs.pm
        'Text::Wrap'            => '98.112902', #./lib/Text/Wrap.pm
        'Thread'                => '1.0', #./ext/Thread/Thread.pm
        'Thread::Queue'         => undef, #./ext/Thread/Thread/Queue.pm
        'Thread::Semaphore'     => undef, #./ext/Thread/Thread/Semaphore.pm
        'Thread::Signal'        => undef, #./ext/Thread/Thread/Signal.pm
        'Thread::Specific'      => undef, #./ext/Thread/Thread/Specific.pm
        'Tie::Array'            => '1.01', #./lib/Tie/Array.pm
        'Tie::Handle'           => '1.0', #./lib/Tie/Handle.pm
        'Tie::Hash'             => undef, #./lib/Tie/Hash.pm
        'Tie::RefHash'          => undef, #./lib/Tie/RefHash.pm
        'Tie::Scalar'           => undef, #./lib/Tie/Scalar.pm
        'Tie::SubstrHash'       => undef, #./lib/Tie/SubstrHash.pm
        'Time::Local'           => undef, #./lib/Time/Local.pm
        'Time::gmtime'          => '1.01', #./lib/Time/gmtime.pm
        'Time::localtime'       => '1.01', #./lib/Time/localtime.pm
        'Time::tm'              => undef, #./lib/Time/tm.pm
        'UNIVERSAL'             => undef, #./lib/UNIVERSAL.pm
        'User::grent'           => undef, #./lib/User/grent.pm
        'User::pwent'           => undef, #./lib/User/pwent.pm
        'VMS::DCLsym'           => '1.01', #./vms/ext/DCLsym/DCLsym.pm
        'VMS::Filespec'         => undef, #./vms/ext/Filespec.pm
        'VMS::Stdio'            => '2.2', #./vms/ext/Stdio/Stdio.pm
        'XSLoader'              => '0.01',
        'attributes'            => '0.03', #./lib/attributes.pm
        'attrs'                 => '1.0', #./ext/attrs/attrs.pm
        'autouse'               => '1.02', #./lib/autouse.pm
        'base'                  => '1.01', #./lib/base.pm
        'blib'                  => '1.00', #./lib/blib.pm
        'bytes'                 => undef, #./lib/bytes.pm
        'charnames'             => undef, #./lib/charnames.pm
        'constant'              => '1.02', #./lib/constant.pm
        'diagnostics'           => '1.0', #./lib/diagnostics.pm
        'fields'                => '1.01', #./lib/fields.pm
        'filetest'              => undef, #./lib/filetest.pm
        'integer'               => undef, #./lib/integer.pm
        'less'                  => undef, #./lib/less.pm
        'lib'                   => '0.5564', #./lib/lib.pm
        'locale'                => undef, #./lib/locale.pm
        'open'                  => undef, #./lib/open.pm
        'ops'                   => undef, #./ext/Opcode/ops.pm
        'overload'              => undef, #./lib/overload.pm
        're'                    => '0.02', #./ext/re/re.pm
        'sigtrap'               => '1.02', #./lib/sigtrap.pm
        'strict'                => '1.01', #./lib/strict.pm
        'subs'                  => undef, #./lib/subs.pm
        'utf8'                  => undef, #./lib/utf8.pm
        'vars'                  => undef, #./lib/vars.pm
        'vmsish'                => undef, #./vms/ext/vmsish.pm
        'warnings'              => undef, #./lib/warnings.pm
        'warnings::register'    => undef, #./lib/warnings/register.pm
    },

    5.006001   => {
        'AnyDBM_File'           => undef,
        'attributes'            => 0.03,
        'attrs'                 => '1.0',
        'AutoLoader'            => 5.58,
        'AutoSplit'             => 1.0305,
        'autouse'               => 1.02,
        'B::Asmdata'            => undef,
        'B::Assembler'          => 0.02,
        'B::Bblock'             => undef,
        'B::Bytecode'           => undef,
        'B::C'                  => undef,
        'B::CC'                 => undef,
        'B::Concise'            => 0.51,
        'B::Debug'              => undef,
        'B::Deparse'            => 0.6,
        'B::Disassembler'       => undef,
        'B::Lint'               => undef,
        'B'                     => undef,
        'B::Showlex'            => undef,
        'B::Stackobj'           => undef,
        'B::Stash'              => undef,
        'B::Terse'              => undef,
        'B::Xref'               => undef,
        'base'                  => 1.01,
        'Benchmark'             => 1,
        'blib'                  => '1.00',
        'ByteLoader'            => 0.04,
        'bytes'                 => undef,
        'Carp'                  => undef,
        'Carp::Heavy'           => undef,
        'CGI'                   => 2.752,
        'CGI::Apache'           => undef,
        'CGI::Carp'             => '1.20',
        'CGI::Cookie'           => 1.18,
        'CGI::Fast'             => 1.02,
        'CGI::Pretty'           => 1.05,
        'CGI::Push'             => 1.04,
        'CGI::Switch'           => undef,
        'CGI::Util'             => 1.1,
        'charnames'             => undef,
        'Class::Struct'         => 0.59,
        'Config'                => undef,
        'constant'              => 1.02,
        'CPAN::FirstTime'       => 1.53 ,
        'CPAN'                  => '1.59_54',
        'CPAN::Nox'             => '1.00',
        'Cwd'                   => 2.04,
        'Data::Dumper'          => 2.102,
        'DB'                    => '1.0',
        'DB_File'               => 1.75,
        'Devel::DProf'          => '20000000.00_00',
        'Devel::Peek'           => '1.00_01',
        'Devel::SelfStubber'    => 1.01,
        'diagnostics'           => '1.0', # really v1.0, but that causes breakage
        'DirHandle'             => undef,
        'Dumpvalue'             => undef,
        'DynaLoader'            => 1.04,
        'English'               => undef,
        'Env'                   => undef,
        'Exporter'              => 5.562,
        'Exporter::Heavy'       => undef,
        'ExtUtils::Command'     => 1.01,
        'ExtUtils::Embed'       => 1.2505,
        'ExtUtils::Install'     => 1.28 ,
        'ExtUtils::Installed'   => 0.02,
        'ExtUtils::Liblist'     => 1.26 ,
        'ExtUtils::MakeMaker'   => 5.45,
        'ExtUtils::Manifest'    => 1.33 ,
        'ExtUtils::Miniperl'    => undef,
        'ExtUtils::Mkbootstrap' => 1.14 ,
        'ExtUtils::Mksymlists'  => 1.17 ,
        'ExtUtils::MM_Cygwin'   => undef,
        'ExtUtils::MM_OS2'      => undef,
        'ExtUtils::MM_Unix'     => 1.12603 ,
        'ExtUtils::MM_VMS'      => undef,
        'ExtUtils::MM_Win32'    => undef,
        'ExtUtils::Packlist'    => 0.03,
        'ExtUtils::testlib'     => 1.11 ,
        'ExtUtils::XSSymSet'    => '1.0',
        'Fatal'                 => 1.02,
        'Fcntl'                 => 1.03,
        'fields'                => 1.01,
        'File::Basename'        => 2.6,
        'File::CheckTree'       => undef,
        'File::Compare'         => 1.1002,
        'File::Copy'            => 2.03,
        'File::DosGlob'         => undef,
        'File::Find'            => undef,
        'File::Glob'            => 0.991,
        'File::Path'            => 1.0404,
        'File::Spec'            => 0.82,
        'File::Spec::Epoc'      => undef,
        'File::Spec::Functions' => 1.1,
        'File::Spec::Mac'       => 1.2,
        'File::Spec::OS2'       => 1.1,
        'File::Spec::Unix'      => 1.2,
        'File::Spec::VMS'       => 1.1,
        'File::Spec::Win32'     => 1.2,
        'File::stat'            => undef,
        'File::Temp'            => 0.12,
        'FileCache'             => undef,
        'FileHandle'            => '2.00',
        'filetest'              => undef,
        'FindBin'               => 1.42,
        'GDBM_File'             => 1.05,
        'Getopt::Long'          => 2.25,
        'Getopt::Std'           => 1.02,
        'I18N::Collate'         => undef,
        'integer'               => undef,
        'IO'                    => '1.20',
        'IO::Dir'               => 1.03,
        'IO::File'              => 1.08,
        'IO::Handle'            => 1.21,
        'IO::Pipe'              => 1.121,
        'IO::Poll'              => 0.05,
        'IO::Seekable'          => 1.08,
        'IO::Select'            => 1.14,
        'IO::Socket'            => 1.26,
        'IO::Socket::INET'      => 1.25,
        'IO::Socket::UNIX'      => '1.20',
        'IPC::Msg'              => '1.00',
        'IPC::Open2'            => 1.01,
        'IPC::Open3'            => 1.0103,
        'IPC::Semaphore'        => '1.00',
        'IPC::SysV'             => 1.03,
        'JNI'                   => 0.1,
        'JPL::AutoLoader'       => undef,
        'JPL::Class'            => undef,
        'JPL::Compile'          => undef,
        'less'                  => undef,
        'lib'                   => 0.5564,
        'locale'                => undef,
        'Math::BigFloat'        => 0.02,
        'Math::BigInt'          => 0.01,
        'Math::Complex'         => 1.31,
        'Math::Trig'            => 1,
        'NDBM_File'             => 1.04,
        'Net::hostent'          => undef,
        'Net::netent'           => undef,
        'Net::Ping'             => 2.02,
        'Net::protoent'         => undef,
        'Net::servent'          => undef,
        'O'                     => undef,
        'ODBM_File'             => 1.03,
        'Opcode'                => 1.04,
        'open'                  => undef,
        'ops'                   => undef,
        'OS2::DLL'              => undef,
        'OS2::ExtAttr'          => 0.01,
        'OS2::PrfDB'            => 0.02,
        'OS2::Process'          => 0.2,
        'OS2::REXX'             => '1.00',
        'overload'              => undef,
        'Pod::Checker'          => 1.2,
        'Pod::Find'             => 0.21,
        'Pod::Functions'        => undef,
        'Pod::Html'             => 1.03,
        'Pod::LaTeX'            => 0.53,
        'Pod::Man'              => 1.15,
        'Pod::InputObjects'     => 1.13,
        'Pod::Parser'           => 1.13,
        'Pod::ParseUtils'       => 0.22,
        'Pod::Plainer'          => 0.01,
        'Pod::Select'           => 1.13,
        'Pod::Text'             => 2.08,
        'Pod::Text::Color'      => 0.06,
        'Pod::Text::Overstrike' => 1.01,
        'Pod::Text::Termcap'    => 1,
        'Pod::Usage'            => 1.14,
        'POSIX'                 => 1.03,
        're'                    => 0.02,
        'Safe'                  => 2.06,
        'SDBM_File'             => 1.03,
        'Search::Dict'          => undef,
        'SelectSaver'           => undef,
        'SelfLoader'            => 1.0902,
        'Shell'                 => 0.3,
        'sigtrap'               => 1.02,
        'Socket'                => 1.72,
        'strict'                => 1.01,
        'subs'                  => undef,
        'Symbol'                => 1.02,
        'Sys::Hostname'         => 1.1,
        'Sys::Syslog'           => 0.01,
        'Term::ANSIColor'       => 1.03,
        'Term::Cap'             => undef,
        'Term::Complete'        => undef,
        'Term::ReadLine'        => undef,
        'Test'                  => 1.15,
        'Test::Harness'         => 1.1604,
        'Text::Abbrev'          => undef,
        'Text::ParseWords'      => 3.2,
        'Text::Soundex'         => '1.0',
        'Text::Tabs'            => 98.112801,
        'Text::Wrap'            => 2001.0131,
        'Thread'                => '1.0',
        'Thread::Queue'         => undef,
        'Thread::Semaphore'     => undef,
        'Thread::Signal'        => undef,
        'Thread::Specific'      => undef,
        'Tie::Array'            => 1.01,
        'Tie::Handle'           => '4.0',
        'Tie::Hash'             => undef,
        'Tie::RefHash'          => 1.3,
        'Tie::Scalar'           => undef,
        'Tie::SubstrHash'       => undef,
        'Time::gmtime'          => 1.01,
        'Time::Local'           => undef,
        'Time::localtime'       => 1.01,
        'Time::tm'              => undef,
        'UNIVERSAL'             => undef,
        'User::grent'           => undef,
        'User::pwent'           => undef,
        'utf8'                  => undef,
        'vars'                  => undef,
        'VMS::DCLsym'           => 1.01,
        'VMS::Filespec'         => undef,
        'VMS::Stdio'            => 2.2,
        'vmsish'                => undef,
        'warnings'              => undef,
        'warnings::register'    => undef,
        'XSLoader'              => '0.01',
    },

    5.006002 => {
        'AnyDBM_File'           => undef,  #lib/AnyDBM_File.pm
        'attributes'            => '0.03',  #lib/attributes.pm
        'attrs'                 => '1.0',  #lib/attrs.pm
        'AutoLoader'            => '5.58',  #lib/AutoLoader.pm
        'AutoSplit'             => '1.0305',  #lib/AutoSplit.pm
        'autouse'               => '1.02',  #lib/autouse.pm
        'B'                     => undef,  #lib/B.pm
        'B::Asmdata'            => undef,  #lib/B/Asmdata.pm
        'B::Assembler'          => '0.02',  #lib/B/Assembler.pm
        'B::Bblock'             => undef,  #lib/B/Bblock.pm
        'B::Bytecode'           => undef,  #lib/B/Bytecode.pm
        'B::C'                  => undef,  #lib/B/C.pm
        'B::CC'                 => undef,  #lib/B/CC.pm
        'B::Concise'            => '0.51',  #lib/B/Concise.pm
        'B::Debug'              => undef,  #lib/B/Debug.pm
        'B::Deparse'            => '0.6',  #lib/B/Deparse.pm
        'B::Disassembler'       => undef,  #lib/B/Disassembler.pm
        'B::Lint'               => undef,  #lib/B/Lint.pm
        'B::Showlex'            => undef,  #lib/B/Showlex.pm
       'B::Stackobj'           => undef,  #lib/B/Stackobj.pm
        'B::Stash'              => undef,  #lib/B/Stash.pm
        'B::Terse'              => undef,  #lib/B/Terse.pm
        'B::Xref'               => undef,  #lib/B/Xref.pm
        'base'                  => '1.01',  #lib/base.pm
        'Benchmark'             => '1',  #lib/Benchmark.pm
        'blib'                  => '1.00',  #lib/blib.pm
        'ByteLoader'            => '0.04',  #lib/ByteLoader.pm
        'bytes'                 => undef,  #lib/bytes.pm
        'Carp'                  => undef,  #lib/Carp.pm
        'Carp::Heavy'           => undef,  #lib/Carp/Heavy.pm
        'CGI'                   => '2.752',  #lib/CGI.pm
        'CGI::Apache'           => undef,  #lib/CGI/Apache.pm
        'CGI::Carp'             => '1.20',  #lib/CGI/Carp.pm
        'CGI::Cookie'           => '1.18',  #lib/CGI/Cookie.pm
        'CGI::Fast'             => '1.02',  #lib/CGI/Fast.pm
        'CGI::Pretty'           => '1.05',  #lib/CGI/Pretty.pm
        'CGI::Push'             => '1.04',  #lib/CGI/Push.pm
        'CGI::Switch'           => undef,  #lib/CGI/Switch.pm
        'CGI::Util'             => '1.1',  #lib/CGI/Util.pm
        'charnames'             => undef,  #lib/charnames.pm
        'Class::Struct'         => '0.59',  #lib/Class/Struct.pm
        'Config'                => undef,  #lib/Config.pm
        'constant'              => '1.02',  #lib/constant.pm
        'CPAN'                  => '1.59_54',  #lib/CPAN.pm
        'CPAN::FirstTime'       => '1.53 ',  #lib/CPAN/FirstTime.pm
        'CPAN::Nox'             => '1.00',  #lib/CPAN/Nox.pm
        'Cwd'                   => '2.04', #lib/Cwd.pm
        'Data::Dumper'          => '2.121', #lib/Data/Dumper.pm
        'DB'                    => '1.0', #lib/DB.pm
        'DB_File'               => '1.806', #lib/DB_File.pm
        'Devel::DProf'          => '20000000.00_00', #lib/Devel/DProf.pm
        'Devel::Peek'           => '1.00_01', #lib/Devel/Peek.pm
        'Devel::SelfStubber'    => '1.01', #lib/Devel/SelfStubber.pm
        'diagnostics'           => '1.0', #lib/diagnostics.pm
        'DirHandle'             => undef, #lib/DirHandle.pm
        'Dumpvalue'             => undef, #lib/Dumpvalue.pm
        'DynaLoader'            => '1.04', #lib/DynaLoader.pm
        'English'               => undef, #lib/English.pm
        'Env'                   => undef, #lib/Env.pm
        'Errno'                 => '1.111', #lib/Errno.pm
        'Exporter'              => '5.562', #lib/Exporter.pm
        'Exporter::Heavy'       => undef, #lib/Exporter/Heavy.pm
        'ExtUtils::Command'     => '1.05', #lib/ExtUtils/Command.pm
        'ExtUtils::Command::MM' => '0.03', #lib/ExtUtils/Command/MM.pm
        'ExtUtils::Embed'       => '1.2505', #lib/ExtUtils/Embed.pm
        'ExtUtils::Install'     => '1.32', #lib/ExtUtils/Install.pm
        'ExtUtils::Installed'   => '0.08', #lib/ExtUtils/Installed.pm
        'ExtUtils::Liblist'     => '1.01', #lib/ExtUtils/Liblist.pm
        'ExtUtils::Liblist::Kid'=> '1.3', #lib/ExtUtils/Liblist/Kid.pm
        'ExtUtils::MakeMaker'   => '6.17', #lib/ExtUtils/MakeMaker.pm
        'ExtUtils::MakeMaker::bytes'=> '0.01', #lib/ExtUtils/MakeMaker/bytes.pm
        'ExtUtils::MakeMaker::vmsish'=> '0.01', #lib/ExtUtils/MakeMaker/vmsish.pm
        'ExtUtils::Manifest'    => '1.42', #lib/ExtUtils/Manifest.pm
        'ExtUtils::Miniperl'    => undef, #lib/ExtUtils/Miniperl.pm
        'ExtUtils::Mkbootstrap' => '1.15', #lib/ExtUtils/Mkbootstrap.pm
        'ExtUtils::Mksymlists'  => '1.19', #lib/ExtUtils/Mksymlists.pm
        'ExtUtils::MM'          => '0.04', #lib/ExtUtils/MM.pm
        'ExtUtils::MM_Any'      => '0.07', #lib/ExtUtils/MM_Any.pm
        'ExtUtils::MM_BeOS'     => '1.04', #lib/ExtUtils/MM_BeOS.pm
        'ExtUtils::MM_Cygwin'   => '1.06', #lib/ExtUtils/MM_Cygwin.pm
        'ExtUtils::MM_DOS'      => '0.02', #lib/ExtUtils/MM_DOS.pm
        'ExtUtils::MM_MacOS'    => '1.07', #lib/ExtUtils/MM_MacOS.pm
        'ExtUtils::MM_NW5'      => '2.06', #lib/ExtUtils/MM_NW5.pm
        'ExtUtils::MM_OS2'      => '1.04', #lib/ExtUtils/MM_OS2.pm
        'ExtUtils::MM_Unix'     => '1.42', #lib/ExtUtils/MM_Unix.pm
        'ExtUtils::MM_UWIN'     => '0.02', #lib/ExtUtils/MM_UWIN.pm
        'ExtUtils::MM_VMS'      => '5.70', #lib/ExtUtils/MM_VMS.pm
        'ExtUtils::MM_Win32'    => '1.09', #lib/ExtUtils/MM_Win32.pm
        'ExtUtils::MM_Win95'    => '0.03', #lib/ExtUtils/MM_Win95.pm
        'ExtUtils::MY'          => '0.01', #lib/ExtUtils/MY.pm
        'ExtUtils::Packlist'    => '0.04', #lib/ExtUtils/Packlist.pm
        'ExtUtils::testlib'     => '1.15', #lib/ExtUtils/testlib.pm
        'ExtUtils::XSSymSet'    => '1.0',  #vms/ext/XSSymSet.pm
        'Fatal'                 => '1.02', #lib/Fatal.pm
        'Fcntl'                 => '1.03', #lib/Fcntl.pm
        'fields'                => '1.01', #lib/fields.pm
        'File::Basename'        => '2.6', #lib/File/Basename.pm
        'File::CheckTree'       => undef, #lib/File/CheckTree.pm
        'File::Compare'         => '1.1002', #lib/File/Compare.pm
        'File::Copy'            => '2.03', #lib/File/Copy.pm
        'File::DosGlob'         => undef, #lib/File/DosGlob.pm
        'File::Find'            => undef, #lib/File/Find.pm
        'File::Glob'            => '0.991', #lib/File/Glob.pm
        'File::Path'            => '1.0404', #lib/File/Path.pm
        'File::Spec'            => '0.86', #lib/File/Spec.pm
        'File::Spec::Cygwin'    => '1.1', #lib/File/Spec/Cygwin.pm
        'File::Spec::Epoc'      => '1.1', #lib/File/Spec/Epoc.pm
        'File::Spec::Functions' => '1.3', #lib/File/Spec/Functions.pm
        'File::Spec::Mac'       => '1.4', #lib/File/Spec/Mac.pm
        'File::Spec::OS2'       => '1.2', #lib/File/Spec/OS2.pm
        'File::Spec::Unix'      => '1.5', #lib/File/Spec/Unix.pm
        'File::Spec::VMS'       => '1.4', #lib/File/Spec/VMS.pm
        'File::Spec::Win32'     => '1.4', #lib/File/Spec/Win32.pm
        'File::stat'            => undef, #lib/File/stat.pm
        'File::Temp'            => '0.14', #lib/File/Temp.pm
        'FileCache'             => undef, #lib/FileCache.pm
        'FileHandle'            => '2.00', #lib/FileHandle.pm
        'filetest'              => undef, #lib/filetest.pm
        'FindBin'               => '1.42', #lib/FindBin.pm
        'GDBM_File'             => '1.05', #ext/GDBM_File/GDBM_File.pm
        'Getopt::Long'          => '2.25', #lib/Getopt/Long.pm
        'Getopt::Std'           => '1.02', #lib/Getopt/Std.pm
        'I18N::Collate'         => undef, #lib/I18N/Collate.pm
        'if'                    => '0.03', #lib/if.pm
        'integer'               => undef, #lib/integer.pm
        'IO'                    => '1.20', #lib/IO.pm
        'IO::Dir'               => '1.03', #lib/IO/Dir.pm
        'IO::File'              => '1.08', #lib/IO/File.pm
        'IO::Handle'            => '1.21', #lib/IO/Handle.pm
        'IO::Pipe'              => '1.121', #lib/IO/Pipe.pm
        'IO::Poll'              => '0.05', #lib/IO/Poll.pm
        'IO::Seekable'          => '1.08', #lib/IO/Seekable.pm
        'IO::Select'            => '1.14', #lib/IO/Select.pm
        'IO::Socket'            => '1.26', #lib/IO/Socket.pm
        'IO::Socket::INET'      => '1.25', #lib/IO/Socket/INET.pm
        'IO::Socket::UNIX'      => '1.20', #lib/IO/Socket/UNIX.pm
        'IPC::Msg'              => '1.00', #lib/IPC/Msg.pm
        'IPC::Open2'            => '1.01', #lib/IPC/Open2.pm
        'IPC::Open3'            => '1.0103', #lib/IPC/Open3.pm
        'IPC::Semaphore'        => '1.00', #lib/IPC/Semaphore.pm
        'IPC::SysV'             => '1.03', #lib/IPC/SysV.pm
        'JNI'                   => '0.1', #jpl/JNI/JNI.pm
        'JPL::AutoLoader'       => undef, #jpl/JPL/AutoLoader.pm
        'JPL::Class'            => undef, #jpl/JPL/Class.pm
        'JPL::Compile'          => undef, #jpl/JPL/Compile.pm
        'less'                  => undef, #lib/less.pm
        'lib'                   => '0.5564', #lib/lib.pm
        'locale'                => undef, #lib/locale.pm
        'Math::BigFloat'        => '0.02', #lib/Math/BigFloat.pm
        'Math::BigInt'          => '0.01', #lib/Math/BigInt.pm
        'Math::Complex'         => '1.31', #lib/Math/Complex.pm
        'Math::Trig'            => '1', #lib/Math/Trig.pm
        'NDBM_File'             => '1.04',  #ext/NDBM_File/NDBM_File.pm
        'Net::hostent'          => undef, #lib/Net/hostent.pm
        'Net::netent'           => undef, #lib/Net/netent.pm
        'Net::Ping'             => '2.02', #lib/Net/Ping.pm
        'Net::protoent'         => undef, #lib/Net/protoent.pm
        'Net::servent'          => undef, #lib/Net/servent.pm
        'O'                     => undef, #lib/O.pm
        'ODBM_File'             => '1.03', #ext/ODBM_File/ODBM_File.pm
        'Opcode'                => '1.04', #lib/Opcode.pm
        'open'                  => undef, #lib/open.pm
        'ops'                   => '1.00', #lib/ops.pm
        'OS2::DLL'              => undef, #os2/OS2/REXX/DLL/DLL.pm
        'OS2::ExtAttr'          => '0.01', #os2/OS2/ExtAttr/ExtAttr.pm
        'OS2::PrfDB'            => '0.02', #os2/OS2/PrfDB/PrfDB.pm
        'OS2::Process'          => '0.2', #os2/OS2/Process/Process.pm
        'OS2::REXX'             => '1.00', #os2/OS2/REXX/REXX.pm
        'overload'              => undef, #lib/overload.pm
        'Pod::Checker'          => '1.2', #lib/Pod/Checker.pm
        'Pod::Find'             => '0.21', #lib/Pod/Find.pm
        'Pod::Functions'        => undef, #lib/Pod/Functions.pm
        'Pod::Html'             => '1.03', #lib/Pod/Html.pm
        'Pod::InputObjects'     => '1.13', #lib/Pod/InputObjects.pm
        'Pod::LaTeX'            => '0.53', #lib/Pod/LaTeX.pm
        'Pod::Man'              => '1.15', #lib/Pod/Man.pm
        'Pod::Parser'           => '1.13', #lib/Pod/Parser.pm
        'Pod::ParseUtils'       => '0.22', #lib/Pod/ParseUtils.pm
        'Pod::Plainer'          => '0.01', #lib/Pod/Plainer.pm
        'Pod::Select'           => '1.13', #lib/Pod/Select.pm
        'Pod::Text'             => '2.08', #lib/Pod/Text.pm
        'Pod::Text::Color'      => '0.06', #lib/Pod/Text/Color.pm
        'Pod::Text::Overstrike' => '1.01', #lib/Pod/Text/Overstrike.pm
        'Pod::Text::Termcap'    => '1', #lib/Pod/Text/Termcap.pm
        'Pod::Usage'            => '1.14', #lib/Pod/Usage.pm
        'POSIX'                 => '1.03', #lib/POSIX.pm
        're'                    => '0.02', #lib/re.pm
        'Safe'                  => '2.10', #lib/Safe.pm
        'SDBM_File'             => '1.03', #lib/SDBM_File.pm
        'Search::Dict'          => undef, #lib/Search/Dict.pm
        'SelectSaver'           => undef, #lib/SelectSaver.pm
        'SelfLoader'            => '1.0902', #lib/SelfLoader.pm
        'Shell'                 => '0.3', #lib/Shell.pm
        'sigtrap'               => '1.02', #lib/sigtrap.pm
        'Socket'                => '1.72', #lib/Socket.pm
        'strict'                => '1.01', #lib/strict.pm
        'subs'                  => undef, #lib/subs.pm
        'Symbol'                => '1.02', #lib/Symbol.pm
        'Sys::Hostname'         => '1.1', #lib/Sys/Hostname.pm
        'Sys::Syslog'           => '0.01', #lib/Sys/Syslog.pm
        'Term::ANSIColor'       => '1.03', #lib/Term/ANSIColor.pm
        'Term::Cap'             => undef, #lib/Term/Cap.pm
        'Term::Complete'        => undef, #lib/Term/Complete.pm
        'Term::ReadLine'        => undef, #lib/Term/ReadLine.pm
        'Test'                  => '1.24', #lib/Test.pm
        'Test::Builder'         => '0.17', #lib/Test/Builder.pm
        'Test::Harness'         => '2.30', #lib/Test/Harness.pm
        'Test::Harness::Assert' => '0.01', #lib/Test/Harness/Assert.pm
        'Test::Harness::Iterator'=> '0.01', #lib/Test/Harness/Iterator.pm
        'Test::Harness::Straps' => '0.15', #lib/Test/Harness/Straps.pm
        'Test::More'            => '0.47', #lib/Test/More.pm
        'Test::Simple'          => '0.47', #lib/Test/Simple.pm
        'Text::Abbrev'          => undef, #lib/Text/Abbrev.pm
        'Text::ParseWords'      => '3.2', #lib/Text/ParseWords.pm
        'Text::Soundex'         => '1.0', #lib/Text/Soundex.pm
        'Text::Tabs'            => '98.112801', #lib/Text/Tabs.pm
        'Text::Wrap'            => '2001.0131', #lib/Text/Wrap.pm
        'Thread'                => '1.0', #ext/Thread/Thread.pm
        'Thread::Queue'         => undef, #ext/Thread/Thread/Queue.pm
        'Thread::Semaphore'     => undef, #ext/Thread/Thread/Semaphore.pm
        'Thread::Signal'        => undef, #ext/Thread/Thread/Signal.pm
        'Thread::Specific'      => undef, #ext/Thread/Thread/Specific.pm
        'Tie::Array'            => '1.01', #lib/Tie/Array.pm
        'Tie::Handle'           => '4.0', #lib/Tie/Handle.pm
        'Tie::Hash'             => undef, #lib/Tie/Hash.pm
        'Tie::RefHash'          => '1.3', #lib/Tie/RefHash.pm
        'Tie::Scalar'           => undef, #lib/Tie/Scalar.pm
        'Tie::SubstrHash'       => undef, #lib/Tie/SubstrHash.pm
        'Time::gmtime'          => '1.01', #lib/Time/gmtime.pm
        'Time::Local'           => undef, #lib/Time/Local.pm
        'Time::localtime'       => '1.01', #lib/Time/localtime.pm
        'Time::tm'              => undef, #lib/Time/tm.pm
        'Unicode'               => '3.0.1', # lib/unicore/version
        'UNIVERSAL'             => undef, #lib/UNIVERSAL.pm
        'User::grent'           => undef, #lib/User/grent.pm
        'User::pwent'           => undef, #lib/User/pwent.pm
        'utf8'                  => undef, #lib/utf8.pm
        'vars'                  => undef, #lib/vars.pm
        'VMS::DCLsym'           => '1.01', #vms/ext/DCLsym/DCLsym.pm
        'VMS::Filespec'         => undef, #vms/ext/Filespec.pm
        'VMS::Stdio'            => '2.2', #vms/ext/Stdio/Stdio.pm
        'vmsish'                => undef, #vms/ext/vmsish.pm
        'warnings'              => undef, #lib/warnings.pm
        'warnings::register'    => undef, #lib/warnings/register.pm
        'XSLoader'              => '0.01', #lib/XSLoader.pm
    },

    5.007003   => {
        'AnyDBM_File'           => '1.00',
        'Attribute::Handlers'   => '0.76',
        'attributes'            => '0.04_01',
        'attrs'                 => '1.01',
        'AutoLoader'            => '5.59',
        'AutoSplit'             => '1.0307',
        'autouse'               => '1.03',
        'B::Asmdata'            => '1.00',
        'B::Assembler'          => '0.04',
        'B::Bblock'             => '1.00',
        'B::Bytecode'           => '1.00',
        'B::C'                  => '1.01',
        'B::CC'                 => '1.00',
        'B::Concise'            => '0.52',
        'B::Debug'              => '1.00',
        'B::Deparse'            => '0.63',
        'B::Disassembler'       => '1.01',
        'B::Lint'               => '1.00',
        'B'                     => '1.00',
        'B::Showlex'            => '1.00',
        'B::Stackobj'           => '1.00',
        'B::Stash'              => '1.00',
        'B::Terse'              => '1.00',
        'B::Xref'               => '1.00',
        'base'                  => '1.02',
        'Benchmark'             => '1.04',
        'blib'                  => '1.01',
        'ByteLoader'            => '0.04',
        'bytes'                 => '1.00',
        'Carp'                  => '1.01',
        'Carp::Heavy'           => undef,
        'CGI'                   => '2.80',
        'CGI::Apache'           => '1.00',
        'CGI::Carp'             => '1.22',
        'CGI::Cookie'           => '1.20',
        'CGI::Fast'             => '1.04',
        'CGI::Pretty'           => '1.05_00',
        'CGI::Push'             => '1.04',
        'CGI::Switch'           => '1.00',
        'CGI::Util'             => '1.3',
        'charnames'             => '1.01',
        'Class::ISA'            => '0.32',
        'Class::Struct'         => '0.61',
        'Config'                => undef,
        'constant'              => '1.04',
        'CPAN::FirstTime'       => '1.54 ',
        'CPAN'                  => '1.59_56',
        'CPAN::Nox'             => '1.00_01',
        'Cwd'                   => '2.06',
        'Data::Dumper'          => '2.12',
        'DB'                    => '1.0',
        'DB_File'               => '1.804',
        'Devel::DProf'          => '20000000.00_01',
        'Devel::Peek'           => '1.00_03',
        'Devel::PPPort'         => '2.0002',
        'Devel::SelfStubber'    => '1.03',
        'diagnostics'           => '1.1',
        'Digest'                => '1.00',
        'Digest::MD5'           => '2.16',
        'DirHandle'             => '1.00',
        'Dumpvalue'             => '1.10',
        'DynaLoader'            => 1.04,
        'Encode'                => '0.40',
        'Encode::CN'            => '0.02',
        'Encode::CN::HZ'        => undef,
        'Encode::Encoding'      => '0.02',
        'Encode::Internal'      => '0.30',
        'Encode::iso10646_1'    => '0.30',
        'Encode::JP'            => '0.02',
        'Encode::JP::Constants' => '1.02',
        'Encode::JP::H2Z'       => '0.77',
        'Encode::JP::ISO_2022_JP' => undef,
        'Encode::JP::JIS'       => undef,
        'Encode::JP::Tr'        => '0.77',
        'Encode::KR'            => '0.02',
        'Encode::Tcl'           => '1.01',
        'Encode::Tcl::Escape'   => '1.01',
        'Encode::Tcl::Extended' => '1.01',
        'Encode::Tcl::HanZi'    => '1.01',
        'Encode::Tcl::Table'    => '1.01',
        'Encode::TW'            => '0.02',
        'Encode::Unicode'       => '0.30',
        'Encode::usc2_le'       => '0.30',
        'Encode::utf8'          => '0.30',
        'Encode::XS'            => '0.40',
        'encoding'              => '1.00',
        'English'               => '1.00',
        'Env'                   => '1.00',
        'Exporter'              => '5.566',
        'Exporter::Heavy'       => '5.562',
        'ExtUtils::Command'     => '1.02',
        'ExtUtils::Constant'    => '0.11',
        'ExtUtils::Embed'       => '1.250601',
        'ExtUtils::Install'     => '1.29',
        'ExtUtils::Installed'   => '0.04',
        'ExtUtils::Liblist'     => '1.2701',
        'ExtUtils::MakeMaker'   => '5.48_03',
        'ExtUtils::Manifest'    => '1.35',
        'ExtUtils::Miniperl'    => undef,
        'ExtUtils::Mkbootstrap' => '1.1401',
        'ExtUtils::Mksymlists'  => '1.18',
        'ExtUtils::MM_BeOS'     => '1.00',
        'ExtUtils::MM_Cygwin'   => '1.00',
        'ExtUtils::MM_OS2'      => '1.00',
        'ExtUtils::MM_Unix'     => '1.12607',
        'ExtUtils::MM_VMS'      => '5.56',
        'ExtUtils::MM_Win32'    => '1.00_02',
        'ExtUtils::Packlist'    => '0.04',
        'ExtUtils::testlib'     => '1.1201',
        'ExtUtils::XSSymSet'    => '1.0',
        'Fatal'                 => '1.03',
        'Fcntl'                 => '1.04',
        'fields'                => '1.02',
        'File::Basename'        => '2.71',
        'File::CheckTree'       => '4.1',
        'File::Compare'         => '1.1003',
        'File::Copy'            => '2.05',
        'File::DosGlob'         => '1.00',
        'File::Find'            => '1.04',
        'File::Glob'            => '1.01',
        'File::Path'            => '1.05',
        'File::Spec'            => '0.83',
        'File::Spec::Cygwin'    => '1.0',
        'File::Spec::Epoc'      => '1.00',
        'File::Spec::Functions' => '1.2',
        'File::Spec::Mac'       => '1.3',
        'File::Spec::OS2'       => '1.1',
        'File::Spec::Unix'      => '1.4',
        'File::Spec::VMS'       => '1.2',
        'File::Spec::Win32'     => '1.3',
        'File::stat'            => '1.00',
        'File::Temp'            => '0.13',
        'FileCache'             => '1.00',
        'FileHandle'            => '2.01',
        'filetest'              => '1.00',
        'Filter::Simple'        => '0.77',
        'Filter::Util::Call'    => '1.06',
        'FindBin'               => '1.43',
        'GDBM_File'             => '1.06',
        'Getopt::Long'          => '2.28',
        'Getopt::Std'           => '1.03',
        'I18N::Collate'         => '1.00',
        'I18N::Langinfo'        => '0.01',
        'I18N::LangTags'        => '0.27',
        'I18N::LangTags::List'  => '0.25',
        'if'                    => '0.01',
        'integer'               => '1.00',
        'IO'                    => '1.20',
        'IO::Dir'               => '1.03_00',
        'IO::File'              => '1.09',
        'IO::Handle'            => '1.21_00',
        'IO::Pipe'              => '1.122',
        'IO::Poll'              => '0.06',
        'IO::Seekable'          => '1.08_00',
        'IO::Select'            => '1.15',
        'IO::Socket'            => '1.27',
        'IO::Socket::INET'      => '1.26',
        'IO::Socket::UNIX'      => '1.20_00',
        'IPC::Msg'              => '1.00_00',
        'IPC::Open2'            => '1.01',
        'IPC::Open3'            => '1.0104',
        'IPC::Semaphore'        => '1.00_00',
        'IPC::SysV'             => '1.03_00',
        'JNI'                   => '0.1',
        'JPL::AutoLoader'       => undef,
        'JPL::Class'            => undef,
        'JPL::Compile'          => undef,
        'less'                  => '0.01',
        'lib'                   => '0.5564',
        'List::Util'            => '1.06_00',
        'locale'                => '1.00',
        'Locale::Constants'     => '2.01',
        'Locale::Country'       => '2.01',
        'Locale::Currency'      => '2.01',
        'Locale::Language'      => '2.01',
        'Locale::Maketext'      => '1.03',
        'Locale::Script'        => '2.01',
        'Math::BigFloat'        => '1.30',
        'Math::BigInt'          => '1.54',
        'Math::BigInt::Calc'    => '0.25',
        'Math::Complex'         => '1.34',
        'Math::Trig'            => '1.01',
        'Memoize'               => '0.66',
        'Memoize::AnyDBM_File'  => '0.65',
        'Memoize::Expire'       => '0.66',
        'Memoize::ExpireFile'   => '0.65',
        'Memoize::ExpireTest'   => '0.65',
        'Memoize::NDBM_File'    => '0.65',
        'Memoize::SDBM_File'    => '0.65',
        'Memoize::Storable'     => '0.65',
        'MIME::Base64'          => '2.12',
        'MIME::QuotedPrint'     => '2.03',
        'NDBM_File'             => '1.04',
        'Net::Cmd'              => '2.21',
        'Net::Config'           => '1.10',
        'Net::Domain'           => '2.17',
        'Net::FTP'              => '2.64',
        'Net::FTP::A'           => '1.15',
        'Net::FTP::dataconn'    => '0.10',
        'Net::FTP::E'           => '0.01',
        'Net::FTP::I'           => '1.12',
        'Net::FTP::L'           => '0.01',
        'Net::hostent'          => '1.00',
        'Net::netent'           => '1.00',
        'Net::Netrc'            => '2.12',
        'Net::NNTP'             => '2.21',
        'Net::Ping'             => '2.12',
        'Net::POP3'             => '2.23',
        'Net::protoent'         => '1.00',
        'Net::servent'          => '1.00',
        'Net::SMTP'             => '2.21',
        'Net::Time'             => '2.09',
        'NEXT'                  => '0.50',
        'O'                     => '1.00',
        'ODBM_File'             => '1.03',
        'Opcode'                => '1.05',
        'open'                  => '1.01',
        'ops'                   => '1.00',
        'OS2::DLL'              => '1.00',
        'OS2::ExtAttr'          => '0.01',
        'OS2::PrfDB'            => '0.02',
        'OS2::Process'          => '1.0',
        'OS2::REXX'             => '1.01',
        'overload'              => '1.00',
        'PerlIO'                => '1.00',
        'PerlIO::Scalar'        => '0.01',
        'PerlIO::Via'           => '0.01',
        'Pod::Checker'          => '1.3',
        'Pod::Find'             => '0.22',
        'Pod::Functions'        => '1.01',
        'Pod::Html'             => '1.04',
        'Pod::LaTeX'            => '0.54',
        'Pod::Man'              => '1.32',
        'Pod::InputObjects'     => '1.13',
        'Pod::ParseLink'        => '1.05',
        'Pod::Parser'           => '1.13',
        'Pod::ParseUtils'       => '0.22',
        'Pod::Plainer'          => '0.01',
        'Pod::Select'           => '1.13',
        'Pod::Text'             => '2.18',
        'Pod::Text::Color'      => '1.03',
        'Pod::Text::Overstrike' => '1.08',
        'Pod::Text::Termcap'    => '1.09',
        'Pod::Usage'            => '1.14',
        'POSIX'                 => '1.05',
        're'                    => '0.03',
        'Safe'                  => '2.07',
        'Scalar::Util'          => undef,
        'SDBM_File'             => '1.03',
        'Search::Dict'          => '1.02',
        'SelectSaver'           => '1.00',
        'SelfLoader'            => '1.0903',
        'Shell'                 => '0.4',
        'sigtrap'               => '1.02',
        'Socket'                => '1.75',
        'sort'                  => '1.00',
        'Storable'              => '1.015',
        'strict'                => '1.02',
        'subs'                  => '1.00',
        'Switch'                => '2.06',
        'Symbol'                => '1.04',
        'Sys::Hostname'         => '1.1',
        'Sys::Syslog'           => '0.02',
        'Term::ANSIColor'       => '1.04',
        'Term::Cap'             => '1.07',
        'Term::Complete'        => '1.4',
        'Term::ReadLine'        => '1.00',
        'Test'                  => '1.18',
        'Test::Builder'         => '0.11',
        'Test::Harness'         => '2.01',
        'Test::Harness::Assert' => '0.01',
        'Test::Harness::Iterator'=> '0.01',
        'Test::Harness::Straps' => '0.08',
        'Test::More'            => '0.41',
        'Test::Simple'          => '0.41',
        'Text::Abbrev'          => '1.00',
        'Text::Balanced'        => '1.89',
        'Text::ParseWords'      => '3.21',
        'Text::Soundex'         => '1.01',
        'Text::Tabs'            => '98.112801',
        'Text::Wrap'            => '2001.0929',
        'Thread'                => '2.00',
        'Thread::Queue'         => '1.00',
        'Thread::Semaphore'     => '1.00',
        'Thread::Signal'        => '1.00',
        'Thread::Specific'      => '1.00',
        'threads'               => '0.05',
        'threads::shared'       => '0.90',
        'Tie::Array'            => '1.02',
        'Tie::File'             => '0.17',
        'Tie::Hash'             => '1.00',
        'Tie::Handle'           => '4.1',
        'Tie::Memoize'          => '1.0',
        'Tie::RefHash'          => '1.3_00',
        'Tie::Scalar'           => '1.00',
        'Tie::SubstrHash'       => '1.00',
        'Time::gmtime'          => '1.02',
        'Time::HiRes'           => '1.20_00',
        'Time::Local'           => '1.04',
        'Time::localtime'       => '1.02',
        'Time::tm'              => '1.00',
        'Unicode::Collate'      => '0.10',
        'Unicode::Normalize'    => '0.14',
        'Unicode::UCD'          => '0.2',
        'UNIVERSAL'             => '1.00',
        'User::grent'           => '1.00',
        'User::pwent'           => '1.00',
        'utf8'                  => '1.00',
        'vars'                  => '1.01',
        'VMS::DCLsym'           => '1.02',
        'VMS::Filespec'         => '1.1',
        'VMS::Stdio'            => '2.3',
        'vmsish'                => '1.00',
        'warnings'              => '1.00',
        'warnings::register'    => '1.00',
        'XS::Typemap'           => '0.01',
        'XSLoader'              => '0.01',
    },

    5.008   => {
        'AnyDBM_File'           => '1.00', #./lib/AnyDBM_File.pm
        'Attribute::Handlers'   => '0.77', #./lib/Attribute/Handlers.pm
        'attributes'            => '0.05', #./lib/attributes.pm
        'attrs'                 => '1.01', #./ext/attrs/attrs.pm
        'AutoLoader'            => '5.59', #./lib/AutoLoader.pm
        'AutoSplit'             => '1.0307', #./lib/AutoSplit.pm
        'autouse'               => '1.03', #./lib/autouse.pm
        'B'                     => '1.01', #./ext/B/B.pm
        'B::Asmdata'            => '1.00', #./ext/B/B/Asmdata.pm
        'B::Assembler'          => '0.04', #./ext/B/B/Assembler.pm
        'B::Bblock'             => '1.00', #./ext/B/B/Bblock.pm
        'B::Bytecode'           => '1.00', #./ext/B/B/Bytecode.pm
        'B::C'                  => '1.01', #./ext/B/B/C.pm
        'B::CC'                 => '1.00', #./ext/B/B/CC.pm
        'B::Concise'            => '0.52', #./ext/B/B/Concise.pm
        'B::Debug'              => '1.00', #./ext/B/B/Debug.pm
        'B::Deparse'            => '0.63', #./ext/B/B/Deparse.pm
        'B::Disassembler'       => '1.01', #./ext/B/B/Disassembler.pm
        'B::Lint'               => '1.01', #./ext/B/B/Lint.pm
        'B::Showlex'            => '1.00', #./ext/B/B/Showlex.pm
        'B::Stackobj'           => '1.00', #./ext/B/B/Stackobj.pm
        'B::Stash'              => '1.00', #./ext/B/B/Stash.pm
        'B::Terse'              => '1.00', #./ext/B/B/Terse.pm
        'B::Xref'               => '1.01', #./ext/B/B/Xref.pm
        'base'                  => '1.03', #./lib/base.pm
        'Benchmark'             => '1.04', #./lib/Benchmark.pm
        'bigint'                => '0.02', #./lib/bigint.pm
        'bignum'                => '0.11', #./lib/bignum.pm
        'bigrat'                => '0.04', #./lib/bigrat.pm
        'blib'                  => '1.02', #./lib/blib.pm
        'ByteLoader'            => '0.04', #./ext/ByteLoader/ByteLoader.pm
        'bytes'                 => '1.00', #./lib/bytes.pm
        'Carp'                  => '1.01', #./lib/Carp.pm
        'Carp::Heavy'           => 'undef', #./lib/Carp/Heavy.pm
        'CGI'                   => '2.81', #./lib/CGI.pm
        'CGI::Apache'           => '1.00', #./lib/CGI/Apache.pm
        'CGI::Carp'             => '1.23', #./lib/CGI/Carp.pm
        'CGI::Cookie'           => '1.20', #./lib/CGI/Cookie.pm
        'CGI::Fast'             => '1.04', #./lib/CGI/Fast.pm
        'CGI::Pretty'           => '1.05_00', #./lib/CGI/Pretty.pm
        'CGI::Push'             => '1.04', #./lib/CGI/Push.pm
        'CGI::Switch'           => '1.00', #./lib/CGI/Switch.pm
        'CGI::Util'             => '1.3', #./lib/CGI/Util.pm
        'charnames'             => '1.01', #./lib/charnames.pm
        'Class::ISA'            => '0.32', #./lib/Class/ISA.pm
        'Class::Struct'         => '0.61', #./lib/Class/Struct.pm
        'constant'              => '1.04', #./lib/constant.pm
        'Config'                => undef,
        'CPAN'                  => '1.61', #./lib/CPAN.pm
        'CPAN::FirstTime'       => '1.56 ', #./lib/CPAN/FirstTime.pm
        'CPAN::Nox'             => '1.02', #./lib/CPAN/Nox.pm
        'Cwd'                   => '2.06', #./lib/Cwd.pm
        'Data::Dumper'          => '2.12', #./ext/Data/Dumper/Dumper.pm
        'DB'                    => '1.0', #./lib/DB.pm
        'DB_File'               => '1.804', #./ext/DB_File/DB_File.pm
        'Devel::DProf'          => '20000000.00_01', #./ext/Devel/DProf/DProf.pm
        'Devel::Peek'           => '1.00_03', #./ext/Devel/Peek/Peek.pm
        'Devel::PPPort'         => '2.0002', #./ext/Devel/PPPort/PPPort.pm
        'Devel::SelfStubber'    => '1.03', #./lib/Devel/SelfStubber.pm
        'diagnostics'           => '1.1', #./lib/diagnostics.pm
        'Digest'                => '1.00', #./lib/Digest.pm
        'Digest::MD5'           => '2.20', #./ext/Digest/MD5/MD5.pm
        'DirHandle'             => '1.00', #./lib/DirHandle.pm
        'Dumpvalue'             => '1.11', #./lib/Dumpvalue.pm
        'DynaLoader'            => '1.04',
        'Encode'                => '1.75', #./ext/Encode/Encode.pm
        'Encode::Alias'         => '1.32', #./ext/Encode/lib/Encode/Alias.pm
        'Encode::Byte'          => '1.22', #./ext/Encode/Byte/Byte.pm
        'Encode::CJKConstants'  => '1.00', #./ext/Encode/lib/Encode/CJKConstants.pm
        'Encode::CN'            => '1.24', #./ext/Encode/CN/CN.pm
        'Encode::CN::HZ'        => '1.04', #./ext/Encode/lib/Encode/CN/HZ.pm
        'Encode::Config'        => '1.06', #./ext/Encode/lib/Encode/Config.pm
        'Encode::EBCDIC'        => '1.21', #./ext/Encode/EBCDIC/EBCDIC.pm
        'Encode::Encoder'       => '0.05', #./ext/Encode/lib/Encode/Encoder.pm
        'Encode::Encoding'      => '1.30', #./ext/Encode/lib/Encode/Encoding.pm
        'Encode::Guess'         => '1.06', #./ext/Encode/lib/Encode/Guess.pm
        'Encode::JP::H2Z'       => '1.02', #./ext/Encode/lib/Encode/JP/H2Z.pm
        'Encode::JP::JIS7'      => '1.08', #./ext/Encode/lib/Encode/JP/JIS7.pm
        'Encode::JP'            => '1.25', #./ext/Encode/JP/JP.pm
        'Encode::KR'            => '1.22', #./ext/Encode/KR/KR.pm
        'Encode::KR::2022_KR'   => '1.05', #./ext/Encode/lib/Encode/KR/2022_KR.pm
        'Encode::MIME::Header'  => '1.05', #./ext/Encode/lib/Encode/MIME/Header.pm
        'Encode::Symbol'        => '1.22', #./ext/Encode/Symbol/Symbol.pm
        'Encode::TW'            => '1.26', #./ext/Encode/TW/TW.pm
        'Encode::Unicode'       => '1.37', #./ext/Encode/Unicode/Unicode.pm
        'encoding'              => '1.35', #./ext/Encode/encoding.pm
        'English'               => '1.00', #./lib/English.pm
        'Env'                   => '1.00', #./lib/Env.pm
        'Exporter'              => '5.566', #./lib/Exporter.pm
        'Exporter::Heavy'       => '5.566', #./lib/Exporter/Heavy.pm
        'ExtUtils::Command'     => '1.04', #./lib/ExtUtils/Command.pm
        'ExtUtils::Command::MM' => '0.01', #./lib/ExtUtils/Command/MM.pm
        'ExtUtils::Constant'    => '0.12', #./lib/ExtUtils/Constant.pm
        'ExtUtils::Embed'       => '1.250601', #./lib/ExtUtils/Embed.pm
        'ExtUtils::Install'     => '1.29', #./lib/ExtUtils/Install.pm
        'ExtUtils::Installed'   => '0.06', #./lib/ExtUtils/Installed.pm
        'ExtUtils::Liblist'     => '1.00', #./lib/ExtUtils/Liblist.pm
        'ExtUtils::Liblist::Kid'=> '1.29', #./lib/ExtUtils/Liblist/Kid.pm
        'ExtUtils::MakeMaker'   => '6.03', #./lib/ExtUtils/MakeMaker.pm
        'ExtUtils::Manifest'    => '1.38', #./lib/ExtUtils/Manifest.pm
        'ExtUtils::Miniperl'    => undef,
        'ExtUtils::Mkbootstrap' => '1.15', #./lib/ExtUtils/Mkbootstrap.pm
        'ExtUtils::Mksymlists'  => '1.19', #./lib/ExtUtils/Mksymlists.pm
        'ExtUtils::MM'          => '0.04', #./lib/ExtUtils/MM.pm
        'ExtUtils::MM_Any'      => '0.04', #./lib/ExtUtils/MM_Any.pm
        'ExtUtils::MM_BeOS'     => '1.03', #./lib/ExtUtils/MM_BeOS.pm
        'ExtUtils::MM_Cygwin'   => '1.04', #./lib/ExtUtils/MM_Cygwin.pm
        'ExtUtils::MM_DOS'      => '0.01', #./lib/ExtUtils/MM_DOS.pm
        'ExtUtils::MM_MacOS'    => '1.03', #./lib/ExtUtils/MM_MacOS.pm
        'ExtUtils::MM_NW5'      => '2.05', #./lib/ExtUtils/MM_NW5.pm
        'ExtUtils::MM_OS2'      => '1.03', #./lib/ExtUtils/MM_OS2.pm
        'ExtUtils::MM_Unix'     => '1.33', #./lib/ExtUtils/MM_Unix.pm
        'ExtUtils::MM_UWIN'     => '0.01', #./lib/ExtUtils/MM_UWIN.pm
        'ExtUtils::MM_VMS'      => '5.65', #./lib/ExtUtils/MM_VMS.pm
        'ExtUtils::MM_Win32'    => '1.05', #./lib/ExtUtils/MM_Win32.pm
        'ExtUtils::MM_Win95'    => '0.02', #./lib/ExtUtils/MM_Win95.pm
        'ExtUtils::MY'          => '0.01', #./lib/ExtUtils/MY.pm
        'ExtUtils::Packlist'    => '0.04', #./lib/ExtUtils/Packlist.pm
        'ExtUtils::testlib'     => '1.15', #./lib/ExtUtils/testlib.pm
        'ExtUtils::XSSymSet'    => '1.0', #./vms/ext/XSSymSet.pm
        'Fatal'                 => '1.03', #./lib/Fatal.pm
        'Fcntl'                 => '1.04', #./ext/Fcntl/Fcntl.pm
        'fields'                => '1.02', #./lib/fields.pm
        'File::Basename'        => '2.71', #./lib/File/Basename.pm
        'File::CheckTree'       => '4.2', #./lib/File/CheckTree.pm
        'File::Compare'         => '1.1003', #./lib/File/Compare.pm
        'File::Copy'            => '2.05', #./lib/File/Copy.pm
        'File::DosGlob'         => '1.00', #./lib/File/DosGlob.pm
        'File::Find'            => '1.04', #./lib/File/Find.pm
        'File::Glob'            => '1.01', #./ext/File/Glob/Glob.pm
        'File::Path'            => '1.05', #./lib/File/Path.pm
        'File::Spec'            => '0.83', #./lib/File/Spec.pm
        'File::Spec::Cygwin'    => '1.0', #./lib/File/Spec/Cygwin.pm
        'File::Spec::Epoc'      => '1.00', #./lib/File/Spec/Epoc.pm
        'File::Spec::Functions' => '1.2', #./lib/File/Spec/Functions.pm
        'File::Spec::Mac'       => '1.3', #./lib/File/Spec/Mac.pm
        'File::Spec::OS2'       => '1.1', #./lib/File/Spec/OS2.pm
        'File::Spec::Unix'      => '1.4', #./lib/File/Spec/Unix.pm
        'File::Spec::VMS'       => '1.2', #./lib/File/Spec/VMS.pm
        'File::Spec::Win32'     => '1.3', #./lib/File/Spec/Win32.pm
        'File::stat'            => '1.00', #./lib/File/stat.pm
        'File::Temp'            => '0.13', #./lib/File/Temp.pm
        'FileCache'             => '1.021', #./lib/FileCache.pm
        'FileHandle'            => '2.01', #./lib/FileHandle.pm
        'filetest'              => '1.00', #./lib/filetest.pm
        'Filter::Simple'        => '0.78', #./lib/Filter/Simple.pm
        'Filter::Util::Call'    => '1.06', #./ext/Filter/Util/Call/Call.pm
        'FindBin'               => '1.43', #./lib/FindBin.pm
        'GDBM_File'             => '1.06', #./ext/GDBM_File/GDBM_File.pm
        'Getopt::Long'          => '2.32', #./lib/Getopt/Long.pm
        'Getopt::Std'           => '1.03', #./lib/Getopt/Std.pm
        'Hash::Util'            => '0.04', #./lib/Hash/Util.pm
        'I18N::Collate'         => '1.00', #./lib/I18N/Collate.pm
        'I18N::Langinfo'        => '0.01', #./ext/I18N/Langinfo/Langinfo.pm
        'I18N::LangTags'        => '0.27', #./lib/I18N/LangTags.pm
        'I18N::LangTags::List'  => '0.25', #./lib/I18N/LangTags/List.pm
        'if'                    => '0.01', #./lib/if.pm
        'integer'               => '1.00', #./lib/integer.pm
        'IO'                    => '1.20', #./ext/IO/IO.pm
        'IO::Dir'               => '1.03_00', #./ext/IO/lib/IO/Dir.pm
        'IO::File'              => '1.09', #./ext/IO/lib/IO/File.pm
        'IO::Handle'            => '1.21_00', #./ext/IO/lib/IO/Handle.pm
        'IO::Pipe'              => '1.122', #./ext/IO/lib/IO/Pipe.pm
        'IO::Poll'              => '0.06', #./ext/IO/lib/IO/Poll.pm
        'IO::Seekable'          => '1.08_00', #./ext/IO/lib/IO/Seekable.pm
        'IO::Select'            => '1.15', #./ext/IO/lib/IO/Select.pm
        'IO::Socket'            => '1.27', #./ext/IO/lib/IO/Socket.pm
        'IO::Socket::INET'      => '1.26', #./ext/IO/lib/IO/Socket/INET.pm
        'IO::Socket::UNIX'      => '1.20_00', #./ext/IO/lib/IO/Socket/UNIX.pm
        'IPC::Open2'            => '1.01', #./lib/IPC/Open2.pm
        'IPC::Open3'            => '1.0104', #./lib/IPC/Open3.pm
        'IPC::Msg'              => '1.00_00', #./ext/IPC/SysV/Msg.pm
        'IPC::Semaphore'        => '1.00_00', #./ext/IPC/SysV/Semaphore.pm
        'IPC::SysV'             => '1.03_00', #./ext/IPC/SysV/SysV.pm
        'JNI'                   => '0.1', #./jpl/JNI/JNI.pm
        'JPL::AutoLoader'       => undef, #./jpl/JPL/AutoLoader.pm
        'JPL::Class'            => undef, #./jpl/JPL/Class.pm
        'JPL::Compile'          => undef, #./jpl/JPL/Compile.pm
        'less'                  => '0.01', #./lib/less.pm
        'lib'                   => '0.5564',
        'List::Util'            => '1.07_00', #./ext/List/Util/lib/List/Util.pm
        'locale'                => '1.00', #./lib/locale.pm
        'Locale::Constants'     => '2.01', #./lib/Locale/Constants.pm
        'Locale::Country'       => '2.04', #./lib/Locale/Country.pm
        'Locale::Currency'      => '2.01', #./lib/Locale/Currency.pm
        'Locale::Language'      => '2.01', #./lib/Locale/Language.pm
        'Locale::Maketext'      => '1.03', #./lib/Locale/Maketext.pm
        'Locale::Script'        => '2.01', #./lib/Locale/Script.pm
        'Math::BigFloat'        => '1.35', #./lib/Math/BigFloat.pm
        'Math::BigFloat::Trace' => '0.01', #./lib/Math/BigFloat/Trace.pm
        'Math::BigInt'          => '1.60', #./lib/Math/BigInt.pm
        'Math::BigInt::Calc'    => '0.30', #./lib/Math/BigInt/Calc.pm
        'Math::BigInt::Trace'   => '0.01', #./lib/Math/BigInt/Trace.pm
        'Math::BigRat'          => '0.07', #./lib/Math/BigRat.pm
        'Math::Complex'         => '1.34', #./lib/Math/Complex.pm
        'Math::Trig'            => '1.01', #./lib/Math/Trig.pm
        'Memoize'               => '1.01', #./lib/Memoize.pm
        'Memoize::AnyDBM_File'  => '0.65', #./lib/Memoize/AnyDBM_File.pm
        'Memoize::Expire'       => '1.00', #./lib/Memoize/Expire.pm
        'Memoize::ExpireFile'   => '1.01', #./lib/Memoize/ExpireFile.pm
        'Memoize::ExpireTest'   => '0.65', #./lib/Memoize/ExpireTest.pm
        'Memoize::NDBM_File'    => '0.65', #./lib/Memoize/NDBM_File.pm
        'Memoize::SDBM_File'    => '0.65', #./lib/Memoize/SDBM_File.pm
        'Memoize::Storable'     => '0.65', #./lib/Memoize/Storable.pm
        'MIME::Base64'          => '2.12', #./ext/MIME/Base64/Base64.pm
        'MIME::QuotedPrint'     => '2.03', #./ext/MIME/Base64/QuotedPrint.pm
        'NDBM_File'             => '1.04', #./ext/NDBM_File/NDBM_File.pm
        'Net::Cmd'              => '2.21', #./lib/Net/Cmd.pm
        'Net::Config'           => '1.10', #./lib/Net/Config.pm
        'Net::Domain'           => '2.17', #./lib/Net/Domain.pm
        'Net::FTP'              => '2.65', #./lib/Net/FTP.pm
        'Net::FTP::A'           => '1.15', #./lib/Net/FTP/A.pm
        'Net::FTP::dataconn'    => '0.11', #./lib/Net/FTP/dataconn.pm
        'Net::FTP::E'           => '0.01', #./lib/Net/FTP/E.pm
        'Net::FTP::I'           => '1.12', #./lib/Net/FTP/I.pm
        'Net::FTP::L'           => '0.01', #./lib/Net/FTP/L.pm
        'Net::hostent'          => '1.00', #./lib/Net/hostent.pm
        'Net::netent'           => '1.00', #./lib/Net/netent.pm
        'Net::Netrc'            => '2.12', #./lib/Net/Netrc.pm
        'Net::NNTP'             => '2.21', #./lib/Net/NNTP.pm
        'Net::Ping'             => '2.19', #./lib/Net/Ping.pm
        'Net::POP3'             => '2.23', #./lib/Net/POP3.pm
        'Net::protoent'         => '1.00', #./lib/Net/protoent.pm
        'Net::servent'          => '1.00', #./lib/Net/servent.pm
        'Net::SMTP'             => '2.24', #./lib/Net/SMTP.pm
        'Net::Time'             => '2.09', #./lib/Net/Time.pm
        'NEXT'                  => '0.50', #./lib/NEXT.pm
        'O'                     => '1.00', #./ext/B/O.pm
        'ODBM_File'             => '1.03', #./ext/ODBM_File/ODBM_File.pm
        'Opcode'                => '1.05', #./ext/Opcode/Opcode.pm
        'open'                  => '1.01', #./lib/open.pm
        'ops'                   => '1.00', #./ext/Opcode/ops.pm
        'OS2::DLL'              => '1.00', #./os2/OS2/REXX/DLL/DLL.pm
        'OS2::ExtAttr'          => '0.01', #./os2/OS2/ExtAttr/ExtAttr.pm
        'OS2::PrfDB'            => '0.02', #./os2/OS2/PrfDB/PrfDB.pm
        'OS2::Process'          => '1.0', #./os2/OS2/Process/Process.pm
        'OS2::REXX'             => '1.01', #./os2/OS2/REXX/REXX.pm
        'overload'              => '1.00', #./lib/overload.pm
        'PerlIO'                => '1.01', #./lib/PerlIO.pm
        'PerlIO::encoding'      => '0.06', #./ext/PerlIO/encoding/encoding.pm
        'PerlIO::scalar'        => '0.01', #./ext/PerlIO/scalar/scalar.pm
        'PerlIO::via'           => '0.01', #./ext/PerlIO/via/via.pm
        'PerlIO::via::QuotedPrint'=> '0.04', #./lib/PerlIO/via/QuotedPrint.pm
        'Pod::Checker'          => '1.3', #./lib/Pod/Checker.pm
        'Pod::Find'             => '0.22', #./lib/Pod/Find.pm
        'Pod::Functions'        => '1.01', #./lib/Pod/Functions.pm
        'Pod::Html'             => '1.04', #./lib/Pod/Html.pm
        'Pod::InputObjects'     => '1.13', #./lib/Pod/InputObjects.pm
        'Pod::LaTeX'            => '0.54', #./lib/Pod/LaTeX.pm
        'Pod::Man'              => '1.33', #./lib/Pod/Man.pm
        'Pod::ParseLink'        => '1.05', #./lib/Pod/ParseLink.pm
        'Pod::Parser'           => '1.13', #./lib/Pod/Parser.pm
        'Pod::ParseUtils'       => '0.22', #./lib/Pod/ParseUtils.pm
        'Pod::Plainer'          => '0.01', #./lib/Pod/Plainer.pm
        'Pod::Select'           => '1.13', #./lib/Pod/Select.pm
        'Pod::Text'             => '2.19', #./lib/Pod/Text.pm
        'Pod::Text::Color'      => '1.03', #./lib/Pod/Text/Color.pm
        'Pod::Text::Overstrike' => '1.08', #./lib/Pod/Text/Overstrike.pm
        'Pod::Text::Termcap'    => '1.09', #./lib/Pod/Text/Termcap.pm
        'Pod::Usage'            => '1.14', #./lib/Pod/Usage.pm
        'POSIX'                 => '1.05', #./ext/POSIX/POSIX.pm
        're'                    => '0.03', #./ext/re/re.pm
        'Safe'                  => '2.07', #./ext/Opcode/Safe.pm
        'Scalar::Util'          => 'undef', #./ext/List/Util/lib/Scalar/Util.pm
        'SDBM_File'             => '1.03', #./ext/SDBM_File/SDBM_File.pm
        'Search::Dict'          => '1.02', #./lib/Search/Dict.pm
        'SelectSaver'           => '1.00', #./lib/SelectSaver.pm
        'SelfLoader'            => '1.0903', #./lib/SelfLoader.pm
        'Shell'                 => '0.4', #./lib/Shell.pm
        'sigtrap'               => '1.02', #./lib/sigtrap.pm
        'Socket'                => '1.75', #./ext/Socket/Socket.pm
        'sort'                  => '1.01', #./lib/sort.pm
        'Storable'              => '2.04', #./ext/Storable/Storable.pm
        'strict'                => '1.02', #./lib/strict.pm
        'subs'                  => '1.00', #./lib/subs.pm
        'Switch'                => '2.09', #./lib/Switch.pm
        'Symbol'                => '1.04', #./lib/Symbol.pm
        'Sys::Hostname'         => '1.1', #./ext/Sys/Hostname/Hostname.pm
        'Sys::Syslog'           => '0.03', #./ext/Sys/Syslog/Syslog.pm
        'Term::ANSIColor'       => '1.04', #./lib/Term/ANSIColor.pm
        'Term::Cap'             => '1.07', #./lib/Term/Cap.pm
        'Term::Complete'        => '1.4', #./lib/Term/Complete.pm
        'Term::ReadLine'        => '1.00', #./lib/Term/ReadLine.pm
        'Test'                  => '1.20', #./lib/Test.pm
        'Test::Builder'         => '0.15', #./lib/Test/Builder.pm
        'Test::Harness'         => '2.26', #./lib/Test/Harness.pm
        'Test::Harness::Assert' => '0.01', #./lib/Test/Harness/Assert.pm
        'Test::Harness::Iterator'=> '0.01', #./lib/Test/Harness/Iterator.pm
        'Test::Harness::Straps' => '0.14', #./lib/Test/Harness/Straps.pm
        'Test::More'            => '0.45', #./lib/Test/More.pm
        'Test::Simple'          => '0.45', #./lib/Test/Simple.pm
        'Text::Abbrev'          => '1.00', #./lib/Text/Abbrev.pm
        'Text::Balanced'        => '1.89', #./lib/Text/Balanced.pm
        'Text::ParseWords'      => '3.21', #./lib/Text/ParseWords.pm
        'Text::Soundex'         => '1.01', #./lib/Text/Soundex.pm
        'Text::Tabs'            => '98.112801', #./lib/Text/Tabs.pm
        'Text::Wrap'            => '2001.0929', #./lib/Text/Wrap.pm
        'Thread'                => '2.00', #./lib/Thread.pm
        'Thread::Queue'         => '2.00', #./lib/Thread/Queue.pm
        'Thread::Semaphore'     => '2.00', #./lib/Thread/Semaphore.pm
        'Thread::Signal'        => '1.00', #./ext/Thread/Thread/Signal.pm
        'Thread::Specific'      => '1.00', #./ext/Thread/Thread/Specific.pm
        'threads'               => '0.99', #./ext/threads/threads.pm
        'threads::shared'       => '0.90', #./ext/threads/shared/shared.pm
        'Tie::Array'            => '1.02', #./lib/Tie/Array.pm
        'Tie::File'             => '0.93', #./lib/Tie/File.pm
        'Tie::Handle'           => '4.1', #./lib/Tie/Handle.pm
        'Tie::Hash'             => '1.00', #./lib/Tie/Hash.pm
        'Tie::Memoize'          => '1.0', #./lib/Tie/Memoize.pm
        'Tie::RefHash'          => '1.30', #./lib/Tie/RefHash.pm
        'Tie::Scalar'           => '1.00', #./lib/Tie/Scalar.pm
        'Tie::SubstrHash'       => '1.00', #./lib/Tie/SubstrHash.pm
        'Time::gmtime'          => '1.02', #./lib/Time/gmtime.pm
        'Time::HiRes'           => '1.20_00', #./ext/Time/HiRes/HiRes.pm
        'Time::Local'           => '1.04', #./lib/Time/Local.pm
        'Time::localtime'       => '1.02', #./lib/Time/localtime.pm
        'Time::tm'              => '1.00', #./lib/Time/tm.pm
        'Unicode'               => '3.2.0', # lib/unicore/version
        'Unicode::Collate'      => '0.12', #./lib/Unicode/Collate.pm
        'Unicode::Normalize'    => '0.17', #./ext/Unicode/Normalize/Normalize.pm
        'Unicode::UCD'          => '0.2', #./lib/Unicode/UCD.pm
        'UNIVERSAL'             => '1.00', #./lib/UNIVERSAL.pm
        'User::grent'           => '1.00', #./lib/User/grent.pm
        'User::pwent'           => '1.00', #./lib/User/pwent.pm
        'utf8'                  => '1.00', #./lib/utf8.pm
        'vars'                  => '1.01', #./lib/vars.pm
        'VMS::DCLsym'           => '1.02', #./vms/ext/DCLsym/DCLsym.pm
        'VMS::Filespec'         => '1.1', #./vms/ext/Filespec.pm
        'VMS::Stdio'            => '2.3', #./vms/ext/Stdio/Stdio.pm
        'vmsish'                => '1.00', #./lib/vmsish.pm
        'warnings'              => '1.00', #./lib/warnings.pm
        'warnings::register'    => '1.00', #./lib/warnings/register.pm
        'XS::APItest'           => '0.01', #./ext/XS/APItest/APItest.pm
        'XS::Typemap'           => '0.01', #./ext/XS/Typemap/Typemap.pm
        'XSLoader'              => '0.01',
    },

    5.008001 => {
        'AnyDBM_File'           => '1.00', #./lib/AnyDBM_File.pm
        'Attribute::Handlers'   => '0.78', #./lib/Attribute/Handlers.pm
        'attributes'            => '0.06', #./lib/attributes.pm
        'attrs'                 => '1.01', #./lib/attrs.pm
        'AutoLoader'            => '5.60', #./lib/AutoLoader.pm
        'AutoSplit'             => '1.04', #./lib/AutoSplit.pm
        'autouse'               => '1.03', #./lib/autouse.pm
        'B'                     => '1.02', #./lib/B.pm
        'B::Asmdata'            => '1.01', #./lib/B/Asmdata.pm
        'B::Assembler'          => '0.06', #./lib/B/Assembler.pm
        'B::Bblock'             => '1.02', #./lib/B/Bblock.pm
        'B::Bytecode'           => '1.01', #./lib/B/Bytecode.pm
        'B::C'                  => '1.02', #./lib/B/C.pm
        'B::CC'                 => '1.00', #./lib/B/CC.pm
        'B::Concise'            => '0.56', #./lib/B/Concise.pm
        'B::Debug'              => '1.01', #./lib/B/Debug.pm
        'B::Deparse'            => '0.64', #./lib/B/Deparse.pm
        'B::Disassembler'       => '1.03', #./lib/B/Disassembler.pm
        'B::Lint'               => '1.02', #./lib/B/Lint.pm
        'B::Showlex'            => '1.00', #./lib/B/Showlex.pm
        'B::Stackobj'           => '1.00', #./lib/B/Stackobj.pm
        'B::Stash'              => '1.00', #./lib/B/Stash.pm
        'B::Terse'              => '1.02', #./lib/B/Terse.pm
        'B::Xref'               => '1.01', #./lib/B/Xref.pm
        'base'                  => '2.03', #./lib/base.pm
        'Benchmark'             => '1.051', #./lib/Benchmark.pm
        'bigint'                => '0.04', #./lib/bigint.pm
        'bignum'                => '0.14', #./lib/bignum.pm
        'bigrat'                => '0.06', #./lib/bigrat.pm
        'blib'                  => '1.02', #./lib/blib.pm
        'ByteLoader'            => '0.05', #./lib/ByteLoader.pm
        'bytes'                 => '1.01', #./lib/bytes.pm
        'Carp'                  => '1.01', #./lib/Carp.pm
        'Carp::Heavy'           => '1.01', #./lib/Carp/Heavy.pm
        'CGI'                   => '3.00', #./lib/CGI.pm
        'CGI::Apache'           => '1.00', #./lib/CGI/Apache.pm
        'CGI::Carp'             => '1.26', #./lib/CGI/Carp.pm
        'CGI::Cookie'           => '1.24', #./lib/CGI/Cookie.pm
        'CGI::Fast'             => '1.041', #./lib/CGI/Fast.pm
        'CGI::Pretty'           => '1.07_00', #./lib/CGI/Pretty.pm
        'CGI::Push'             => '1.04', #./lib/CGI/Push.pm
        'CGI::Switch'           => '1.00', #./lib/CGI/Switch.pm
        'CGI::Util'             => '1.31', #./lib/CGI/Util.pm
        'charnames'             => '1.02', #./lib/charnames.pm
        'Class::ISA'            => '0.32', #./lib/Class/ISA.pm
        'Class::Struct'         => '0.63', #./lib/Class/Struct.pm
        'Config'                => undef, #./lib/Config.pm
        'constant'              => '1.04', #./lib/constant.pm
        'CPAN'                  => '1.76_01', #./lib/CPAN.pm
        'CPAN::FirstTime'       => '1.60 ', #./lib/CPAN/FirstTime.pm
        'CPAN::Nox'             => '1.03', #./lib/CPAN/Nox.pm
        'Cwd'                   => '2.08', #./lib/Cwd.pm
        'Data::Dumper'          => '2.121', #./lib/Data/Dumper.pm
        'DB'                    => '1.0', #./lib/DB.pm
        'DB_File'               => '1.806', #./lib/DB_File.pm
        'Devel::DProf'          => '20030813.00', #./lib/Devel/DProf.pm
        'Devel::Peek'           => '1.01', #./lib/Devel/Peek.pm
        'Devel::PPPort'         => '2.007', #./lib/Devel/PPPort.pm
        'Devel::SelfStubber'    => '1.03', #./lib/Devel/SelfStubber.pm
        'diagnostics'           => '1.11', #./lib/diagnostics.pm
        'Digest'                => '1.02', #./lib/Digest.pm
        'Digest::MD5'           => '2.27', #./lib/Digest/MD5.pm
        'DirHandle'             => '1.00', #./lib/DirHandle.pm
        'Dumpvalue'             => '1.11', #./lib/Dumpvalue.pm
        'DynaLoader'            => '1.04', #./lib/DynaLoader.pm
        'Encode'                => '1.9801', #./lib/Encode.pm
        'Encode::Alias'         => '1.38', #./lib/Encode/Alias.pm
        'Encode::Byte'          => '1.23', #./lib/Encode/Byte.pm
        'Encode::CJKConstants'  => '1.02', #./lib/Encode/CJKConstants.pm
        'Encode::CN'            => '1.24', #./lib/Encode/CN.pm
        'Encode::CN::HZ'        => '1.05', #./lib/Encode/CN/HZ.pm
        'Encode::Config'        => '1.07', #./lib/Encode/Config.pm
        'Encode::EBCDIC'        => '1.21', #./lib/Encode/EBCDIC.pm
        'Encode::Encoder'       => '0.07', #./lib/Encode/Encoder.pm
        'Encode::Encoding'      => '1.33', #./lib/Encode/Encoding.pm
        'Encode::Guess'         => '1.09', #./lib/Encode/Guess.pm
        'Encode::JP'            => '1.25', #./lib/Encode/JP.pm
        'Encode::JP::H2Z'       => '1.02', #./lib/Encode/JP/H2Z.pm
        'Encode::JP::JIS7'      => '1.12', #./lib/Encode/JP/JIS7.pm
        'Encode::KR'            => '1.23', #./lib/Encode/KR.pm
        'Encode::KR::2022_KR'   => '1.06', #./lib/Encode/KR/2022_KR.pm
        'Encode::MIME::Header'  => '1.09', #./lib/Encode/MIME/Header.pm
        'Encode::Symbol'        => '1.22', #./lib/Encode/Symbol.pm
        'Encode::TW'            => '1.26', #./lib/Encode/TW.pm
        'Encode::Unicode'       => '1.40', #./lib/Encode/Unicode.pm
        'Encode::Unicode::UTF7' => '0.02', #./lib/Encode/Unicode/UTF7.pm
        'encoding'              => '1.47', #./lib/encoding.pm
        'English'               => '1.01', #./lib/English.pm
        'Env'                   => '1.00', #./lib/Env.pm
        'Errno'                 => '1.09_00', #./lib/Errno.pm
        'Exporter'              => '5.567', #./lib/Exporter.pm
        'Exporter::Heavy'       => '5.567', #./lib/Exporter/Heavy.pm
        'ExtUtils::Command'     => '1.05', #./lib/ExtUtils/Command.pm
        'ExtUtils::Command::MM' => '0.03', #./lib/ExtUtils/Command/MM.pm
        'ExtUtils::Constant'    => '0.14', #./lib/ExtUtils/Constant.pm
        'ExtUtils::Embed'       => '1.250601', #./lib/ExtUtils/Embed.pm
        'ExtUtils::Install'     => '1.32', #./lib/ExtUtils/Install.pm
        'ExtUtils::Installed'   => '0.08', #./lib/ExtUtils/Installed.pm
        'ExtUtils::Liblist'     => '1.01', #./lib/ExtUtils/Liblist.pm
        'ExtUtils::Liblist::Kid'=> '1.3', #./lib/ExtUtils/Liblist/Kid.pm
        'ExtUtils::MakeMaker'   => '6.17', #./lib/ExtUtils/MakeMaker.pm
        'ExtUtils::MakeMaker::bytes'=> '0.01', #./lib/ExtUtils/MakeMaker/bytes.pm
        'ExtUtils::MakeMaker::vmsish'=> '0.01', #./lib/ExtUtils/MakeMaker/vmsish.pm
        'ExtUtils::Manifest'    => '1.42', #./lib/ExtUtils/Manifest.pm
        'ExtUtils::Miniperl'    => undef, #./lib/ExtUtils/Miniperl.pm
        'ExtUtils::Mkbootstrap' => '1.15', #./lib/ExtUtils/Mkbootstrap.pm
        'ExtUtils::Mksymlists'  => '1.19', #./lib/ExtUtils/Mksymlists.pm
        'ExtUtils::MM'          => '0.04', #./lib/ExtUtils/MM.pm
        'ExtUtils::MM_Any'      => '0.07', #./lib/ExtUtils/MM_Any.pm
        'ExtUtils::MM_BeOS'     => '1.04', #./lib/ExtUtils/MM_BeOS.pm
        'ExtUtils::MM_Cygwin'   => '1.06', #./lib/ExtUtils/MM_Cygwin.pm
        'ExtUtils::MM_DOS'      => '0.02', #./lib/ExtUtils/MM_DOS.pm
        'ExtUtils::MM_MacOS'    => '1.07', #./lib/ExtUtils/MM_MacOS.pm
        'ExtUtils::MM_NW5'      => '2.06', #./lib/ExtUtils/MM_NW5.pm
        'ExtUtils::MM_OS2'      => '1.04', #./lib/ExtUtils/MM_OS2.pm
        'ExtUtils::MM_Unix'     => '1.42', #./lib/ExtUtils/MM_Unix.pm
        'ExtUtils::MM_UWIN'     => '0.02', #./lib/ExtUtils/MM_UWIN.pm
        'ExtUtils::MM_VMS'      => '5.70', #./lib/ExtUtils/MM_VMS.pm
        'ExtUtils::MM_Win32'    => '1.09', #./lib/ExtUtils/MM_Win32.pm
        'ExtUtils::MM_Win95'    => '0.03', #./lib/ExtUtils/MM_Win95.pm
        'ExtUtils::MY'          => '0.01', #./lib/ExtUtils/MY.pm
        'ExtUtils::Packlist'    => '0.04', #./lib/ExtUtils/Packlist.pm
        'ExtUtils::testlib'     => '1.15', #./lib/ExtUtils/testlib.pm
        'ExtUtils::XSSymSet'    => '1.0',  #./vms/ext/XSSymSet.pm
        'Fatal'                 => '1.03', #./lib/Fatal.pm
        'Fcntl'                 => '1.05', #./lib/Fcntl.pm
        'fields'                => '2.03', #./lib/fields.pm
        'File::Basename'        => '2.72', #./lib/File/Basename.pm
        'File::CheckTree'       => '4.2', #./lib/File/CheckTree.pm
        'File::Compare'         => '1.1003', #./lib/File/Compare.pm
        'File::Copy'            => '2.06', #./lib/File/Copy.pm
        'File::DosGlob'         => '1.00', #./lib/File/DosGlob.pm
        'File::Find'            => '1.05', #./lib/File/Find.pm
        'File::Glob'            => '1.02', #./lib/File/Glob.pm
        'File::Path'            => '1.06', #./lib/File/Path.pm
        'File::Spec'            => '0.86', #./lib/File/Spec.pm
        'File::Spec::Cygwin'    => '1.1', #./lib/File/Spec/Cygwin.pm
        'File::Spec::Epoc'      => '1.1', #./lib/File/Spec/Epoc.pm
        'File::Spec::Functions' => '1.3', #./lib/File/Spec/Functions.pm
        'File::Spec::Mac'       => '1.4', #./lib/File/Spec/Mac.pm
        'File::Spec::OS2'       => '1.2', #./lib/File/Spec/OS2.pm
        'File::Spec::Unix'      => '1.5', #./lib/File/Spec/Unix.pm
        'File::Spec::VMS'       => '1.4', #./lib/File/Spec/VMS.pm
        'File::Spec::Win32'     => '1.4', #./lib/File/Spec/Win32.pm
        'File::stat'            => '1.00', #./lib/File/stat.pm
        'File::Temp'            => '0.14', #./lib/File/Temp.pm
        'FileCache'             => '1.03', #./lib/FileCache.pm
        'FileHandle'            => '2.01', #./lib/FileHandle.pm
        'filetest'              => '1.01', #./lib/filetest.pm
        'Filter::Simple'        => '0.78', #./lib/Filter/Simple.pm
        'Filter::Util::Call'    => '1.0601', #./lib/Filter/Util/Call.pm
        'FindBin'               => '1.43', #./lib/FindBin.pm
        'GDBM_File'             => '1.07', #./ext/GDBM_File/GDBM_File.pm
        'Getopt::Long'          => '2.34', #./lib/Getopt/Long.pm
        'Getopt::Std'           => '1.04', #./lib/Getopt/Std.pm
        'Hash::Util'            => '0.05', #./lib/Hash/Util.pm
        'I18N::Collate'         => '1.00', #./lib/I18N/Collate.pm
        'I18N::Langinfo'        => '0.02', #./lib/I18N/Langinfo.pm
        'I18N::LangTags'        => '0.28', #./lib/I18N/LangTags.pm
        'I18N::LangTags::List'  => '0.26', #./lib/I18N/LangTags/List.pm
        'if'                    => '0.03', #./lib/if.pm
        'integer'               => '1.00', #./lib/integer.pm
        'IO'                    => '1.21', #./lib/IO.pm
        'IO::Dir'               => '1.04', #./lib/IO/Dir.pm
        'IO::File'              => '1.10', #./lib/IO/File.pm
        'IO::Handle'            => '1.23', #./lib/IO/Handle.pm
        'IO::Pipe'              => '1.122', #./lib/IO/Pipe.pm
        'IO::Poll'              => '0.06', #./lib/IO/Poll.pm
        'IO::Seekable'          => '1.09', #./lib/IO/Seekable.pm
        'IO::Select'            => '1.16', #./lib/IO/Select.pm
        'IO::Socket'            => '1.28', #./lib/IO/Socket.pm
        'IO::Socket::INET'      => '1.27', #./lib/IO/Socket/INET.pm
        'IO::Socket::UNIX'      => '1.21', #./lib/IO/Socket/UNIX.pm
        'IPC::Msg'              => '1.02', #./lib/IPC/Msg.pm
        'IPC::Open2'            => '1.01', #./lib/IPC/Open2.pm
        'IPC::Open3'            => '1.0105', #./lib/IPC/Open3.pm
        'IPC::Semaphore'        => '1.02', #./lib/IPC/Semaphore.pm
        'IPC::SysV'             => '1.04', #./lib/IPC/SysV.pm
        'JNI'                   => '0.2', #./jpl/JNI/JNI.pm
        'JPL::AutoLoader'       => undef, #./jpl/JPL/AutoLoader.pm
        'JPL::Class'            => undef, #./jpl/JPL/Class.pm
        'JPL::Compile'          => undef, #./jpl/JPL/Compile.pm
        'less'                  => '0.01', #./lib/less.pm
        'lib'                   => '0.5565', #./lib/lib.pm
        'List::Util'            => '1.13', #./lib/List/Util.pm
        'locale'                => '1.00', #./lib/locale.pm
        'Locale::Constants'     => '2.01', #./lib/Locale/Constants.pm
        'Locale::Country'       => '2.61', #./lib/Locale/Country.pm
        'Locale::Currency'      => '2.21', #./lib/Locale/Currency.pm
        'Locale::Language'      => '2.21', #./lib/Locale/Language.pm
        'Locale::Maketext'      => '1.06', #./lib/Locale/Maketext.pm
        'Locale::Maketext::Guts'=> undef, #./lib/Locale/Maketext/Guts.pm
        'Locale::Maketext::GutsLoader'=> undef, #./lib/Locale/Maketext/GutsLoader.pm
        'Locale::Script'        => '2.21', #./lib/Locale/Script.pm
        'Math::BigFloat'        => '1.40', #./lib/Math/BigFloat.pm
        'Math::BigFloat::Trace' => '0.01', #./lib/Math/BigFloat/Trace.pm
        'Math::BigInt'          => '1.66', #./lib/Math/BigInt.pm
        'Math::BigInt::Calc'    => '0.36', #./lib/Math/BigInt/Calc.pm
        'Math::BigInt::Scalar'  => '0.11', #./lib/Math/BigInt/Scalar.pm
        'Math::BigInt::Trace'   => '0.01', #./lib/Math/BigInt/Trace.pm
        'Math::BigRat'          => '0.10', #./lib/Math/BigRat.pm
        'Math::Complex'         => '1.34', #./lib/Math/Complex.pm
        'Math::Trig'            => '1.02', #./lib/Math/Trig.pm
        'Memoize'               => '1.01', #./lib/Memoize.pm
        'Memoize::AnyDBM_File'  => '0.65', #./lib/Memoize/AnyDBM_File.pm
        'Memoize::Expire'       => '1.00', #./lib/Memoize/Expire.pm
        'Memoize::ExpireFile'   => '1.01', #./lib/Memoize/ExpireFile.pm
        'Memoize::ExpireTest'   => '0.65', #./lib/Memoize/ExpireTest.pm
        'Memoize::NDBM_File'    => '0.65', #./lib/Memoize/NDBM_File.pm
        'Memoize::SDBM_File'    => '0.65', #./lib/Memoize/SDBM_File.pm
        'Memoize::Storable'     => '0.65', #./lib/Memoize/Storable.pm
        'MIME::Base64'          => '2.20', #./lib/MIME/Base64.pm
        'MIME::QuotedPrint'     => '2.20', #./lib/MIME/QuotedPrint.pm
        'NDBM_File'             => '1.05', #./ext/NDBM_File/NDBM_File.pm
        'Net::Cmd'              => '2.24', #./lib/Net/Cmd.pm
        'Net::Config'           => '1.10', #./lib/Net/Config.pm
        'Net::Domain'           => '2.18', #./lib/Net/Domain.pm
        'Net::FTP'              => '2.71', #./lib/Net/FTP.pm
        'Net::FTP::A'           => '1.16', #./lib/Net/FTP/A.pm
        'Net::FTP::dataconn'    => '0.11', #./lib/Net/FTP/dataconn.pm
        'Net::FTP::E'           => '0.01', #./lib/Net/FTP/E.pm
        'Net::FTP::I'           => '1.12', #./lib/Net/FTP/I.pm
        'Net::FTP::L'           => '0.01', #./lib/Net/FTP/L.pm
        'Net::hostent'          => '1.01', #./lib/Net/hostent.pm
        'Net::netent'           => '1.00', #./lib/Net/netent.pm
        'Net::Netrc'            => '2.12', #./lib/Net/Netrc.pm
        'Net::NNTP'             => '2.22', #./lib/Net/NNTP.pm
        'Net::Ping'             => '2.31', #./lib/Net/Ping.pm
        'Net::POP3'             => '2.24', #./lib/Net/POP3.pm
        'Net::protoent'         => '1.00', #./lib/Net/protoent.pm
        'Net::servent'          => '1.01', #./lib/Net/servent.pm
        'Net::SMTP'             => '2.26', #./lib/Net/SMTP.pm
        'Net::Time'             => '2.09', #./lib/Net/Time.pm
        'NEXT'                  => '0.60', #./lib/NEXT.pm
        'O'                     => '1.00', #./lib/O.pm
        'ODBM_File'             => '1.04', #./ext/ODBM_File/ODBM_File.pm
        'Opcode'                => '1.05', #./lib/Opcode.pm
        'open'                  => '1.02', #./lib/open.pm
        'ops'                   => '1.00', #./lib/ops.pm
        'OS2::ExtAttr'          => '0.02', #./os2/OS2/ExtAttr/ExtAttr.pm
        'OS2::PrfDB'            => '0.03', #./os2/OS2/PrfDB/PrfDB.pm
        'OS2::Process'          => '1.01', #./os2/OS2/Process/Process.pm
        'OS2::DLL'              => '1.01', #./os2/OS2/REXX/DLL/DLL.pm
        'OS2::REXX'             => '1.02', #./os2/OS2/REXX/REXX.pm
        'overload'              => '1.01', #./lib/overload.pm
        'PerlIO'                => '1.02', #./lib/PerlIO.pm
        'PerlIO::encoding'      => '0.07', #./lib/PerlIO/encoding.pm
        'PerlIO::scalar'        => '0.02', #./lib/PerlIO/scalar.pm
        'PerlIO::via'           => '0.02', #./lib/PerlIO/via.pm
        'PerlIO::via::QuotedPrint'=> '0.05', #./lib/PerlIO/via/QuotedPrint.pm
        'Pod::Checker'          => '1.41', #./lib/Pod/Checker.pm
        'Pod::Find'             => '0.24', #./lib/Pod/Find.pm
        'Pod::Functions'        => '1.02', #./lib/Pod/Functions.pm
        'Pod::Html'             => '1.0501', #./lib/Pod/Html.pm
        'Pod::InputObjects'     => '1.14', #./lib/Pod/InputObjects.pm
        'Pod::LaTeX'            => '0.55', #./lib/Pod/LaTeX.pm
        'Pod::Man'              => '1.37', #./lib/Pod/Man.pm
        'Pod::ParseLink'        => '1.06', #./lib/Pod/ParseLink.pm
        'Pod::Parser'           => '1.13', #./lib/Pod/Parser.pm
        'Pod::ParseUtils'       => '0.3', #./lib/Pod/ParseUtils.pm
        'Pod::Perldoc'          => '3.10', #./lib/Pod/Perldoc.pm
        'Pod::Perldoc::BaseTo'  => undef, #./lib/Pod/Perldoc/BaseTo.pm
        'Pod::Perldoc::GetOptsOO'=> undef, #./lib/Pod/Perldoc/GetOptsOO.pm
        'Pod::Perldoc::ToChecker'=> undef, #./lib/Pod/Perldoc/ToChecker.pm
        'Pod::Perldoc::ToMan'   => undef, #./lib/Pod/Perldoc/ToMan.pm
        'Pod::Perldoc::ToNroff' => undef, #./lib/Pod/Perldoc/ToNroff.pm
        'Pod::Perldoc::ToPod'   => undef, #./lib/Pod/Perldoc/ToPod.pm
        'Pod::Perldoc::ToRtf'   => undef, #./lib/Pod/Perldoc/ToRtf.pm
        'Pod::Perldoc::ToText'  => undef, #./lib/Pod/Perldoc/ToText.pm
        'Pod::Perldoc::ToTk'    => 'undef', #./lib/Pod/Perldoc/ToTk.pm
        'Pod::Perldoc::ToXml'   => undef, #./lib/Pod/Perldoc/ToXml.pm
        'Pod::Plainer'          => '0.01', #./lib/Pod/Plainer.pm
        'Pod::PlainText'        => '2.01', #./lib/Pod/PlainText.pm
        'Pod::Select'           => '1.13', #./lib/Pod/Select.pm
        'Pod::Text'             => '2.21', #./lib/Pod/Text.pm
        'Pod::Text::Color'      => '1.04', #./lib/Pod/Text/Color.pm
        'Pod::Text::Overstrike' => '1.1', #./lib/Pod/Text/Overstrike.pm
        'Pod::Text::Termcap'    => '1.11', #./lib/Pod/Text/Termcap.pm
        'Pod::Usage'            => '1.16', #./lib/Pod/Usage.pm
        'POSIX'                 => '1.06', #./lib/POSIX.pm
        're'                    => '0.04', #./lib/re.pm
        'Safe'                  => '2.10', #./lib/Safe.pm
        'Scalar::Util'          => '1.13', #./lib/Scalar/Util.pm
        'SDBM_File'             => '1.04', #./lib/SDBM_File.pm
        'Search::Dict'          => '1.02', #./lib/Search/Dict.pm
        'SelectSaver'           => '1.00', #./lib/SelectSaver.pm
        'SelfLoader'            => '1.0904', #./lib/SelfLoader.pm
        'Shell'                 => '0.5', #./lib/Shell.pm
        'sigtrap'               => '1.02', #./lib/sigtrap.pm
        'Socket'                => '1.76', #./lib/Socket.pm
        'sort'                  => '1.02', #./lib/sort.pm
        'Storable'              => '2.08', #./lib/Storable.pm
        'strict'                => '1.03', #./lib/strict.pm
        'subs'                  => '1.00', #./lib/subs.pm
        'Switch'                => '2.10', #./lib/Switch.pm
        'Symbol'                => '1.05', #./lib/Symbol.pm
        'Sys::Hostname'         => '1.11', #./lib/Sys/Hostname.pm
        'Sys::Syslog'           => '0.04', #./lib/Sys/Syslog.pm
        'Term::ANSIColor'       => '1.07', #./lib/Term/ANSIColor.pm
        'Term::Cap'             => '1.08', #./lib/Term/Cap.pm
        'Term::Complete'        => '1.401', #./lib/Term/Complete.pm
        'Term::ReadLine'        => '1.01', #./lib/Term/ReadLine.pm
        'Test'                  => '1.24', #./lib/Test.pm
        'Test::Builder'         => '0.17', #./lib/Test/Builder.pm
        'Test::Harness'         => '2.30', #./lib/Test/Harness.pm
        'Test::Harness::Assert' => '0.01', #./lib/Test/Harness/Assert.pm
        'Test::Harness::Iterator'=> '0.01', #./lib/Test/Harness/Iterator.pm
        'Test::Harness::Straps' => '0.15', #./lib/Test/Harness/Straps.pm
        'Test::More'            => '0.47', #./lib/Test/More.pm
        'Test::Simple'          => '0.47', #./lib/Test/Simple.pm
        'Text::Abbrev'          => '1.01', #./lib/Text/Abbrev.pm
        'Text::Balanced'        => '1.95', #./lib/Text/Balanced.pm
        'Text::ParseWords'      => '3.21', #./lib/Text/ParseWords.pm
        'Text::Soundex'         => '1.01', #./lib/Text/Soundex.pm
        'Text::Tabs'            => '98.112801', #./lib/Text/Tabs.pm
        'Text::Wrap'            => '2001.09291', #./lib/Text/Wrap.pm
        'Thread'                => '2.00', #./lib/Thread.pm
        'Thread::Queue'         => '2.00', #./lib/Thread/Queue.pm
        'Thread::Semaphore'     => '2.01', #./lib/Thread/Semaphore.pm
        'Thread::Signal'        => '1.00', #./ext/Thread/Thread/Signal.pm
        'Thread::Specific'      => '1.00', #./ext/Thread/Thread/Specific.pm
        'threads'               => '1.00', #./lib/threads.pm
        'threads::shared'       => '0.91', #./lib/threads/shared.pm
        'Tie::Array'            => '1.03', #./lib/Tie/Array.pm
        'Tie::File'             => '0.97', #./lib/Tie/File.pm
        'Tie::Handle'           => '4.1', #./lib/Tie/Handle.pm
        'Tie::Hash'             => '1.00', #./lib/Tie/Hash.pm
        'Tie::Memoize'          => '1.0', #./lib/Tie/Memoize.pm
        'Tie::RefHash'          => '1.31', #./lib/Tie/RefHash.pm
        'Tie::Scalar'           => '1.00', #./lib/Tie/Scalar.pm
        'Tie::SubstrHash'       => '1.00', #./lib/Tie/SubstrHash.pm
        'Time::gmtime'          => '1.02', #./lib/Time/gmtime.pm
        'Time::HiRes'           => '1.51', #./lib/Time/HiRes.pm
        'Time::Local'           => '1.07', #./lib/Time/Local.pm
        'Time::localtime'       => '1.02', #./lib/Time/localtime.pm
        'Time::tm'              => '1.00', #./lib/Time/tm.pm
        'Unicode'               => '4.0.0', # lib/unicore/version
        'Unicode::Collate'      => '0.28', #./lib/Unicode/Collate.pm
        'Unicode::Normalize'    => '0.23', #./lib/Unicode/Normalize.pm
        'Unicode::UCD'          => '0.21', #./lib/Unicode/UCD.pm
        'UNIVERSAL'             => '1.01', #./lib/UNIVERSAL.pm
        'User::grent'           => '1.00', #./lib/User/grent.pm
        'User::pwent'           => '1.00', #./lib/User/pwent.pm
        'utf8'                  => '1.02', #./lib/utf8.pm
        'vars'                  => '1.01', #./lib/vars.pm
        'VMS::DCLsym'           => '1.02', #./vms/ext/DCLsym/DCLsym.pm
        'VMS::Filespec'         => '1.11', #./vms/ext/Filespec.pm
        'VMS::Stdio'            => '2.3', #./vms/ext/Stdio/Stdio.pm
        'vmsish'                => '1.01', #./lib/vmsish.pm
        'warnings'              => '1.03', #./lib/warnings.pm
        'warnings::register'    => '1.00', #./lib/warnings/register.pm
        'XS::APItest'           => '0.02', #./lib/XS/APItest.pm
        'XS::Typemap'           => '0.01', #./lib/XS/Typemap.pm
        'XSLoader'              => '0.02', #./lib/XSLoader.pm
    },

    5.008002 => {
        'AnyDBM_File' => '1.00',  #AnyDBM_File.pm
        'Attribute::Handlers' => 0.78, #Attribute\Handlers.pm
        'attributes' => 0.06,   #attributes.pm
        'attrs' => 1.01,        #attrs.pm
        'AutoLoader' => '5.60',   #AutoLoader.pm
        'AutoSplit' => 1.04,    #AutoSplit.pm
        'autouse' => 1.03,      #autouse.pm
        'B' => 1.02,            #B.pm
        'B::Asmdata' => 1.01,   #B\Asmdata.pm
        'B::Assembler' => 0.06, #B\Assembler.pm
        'B::Bblock' => 1.02,    #B\Bblock.pm
        'B::Bytecode' => 1.01,  #B\Bytecode.pm
        'B::C' => 1.02,         #B\C.pm
        'B::CC' => '1.00',        #B\CC.pm
        'B::Concise' => 0.56,   #B\Concise.pm
        'B::Debug' => 1.01,     #B\Debug.pm
        'B::Deparse' => 0.64,   #B\Deparse.pm
        'B::Disassembler' => 1.03, #B\Disassembler.pm
        'B::Lint' => 1.02,      #B\Lint.pm
        'B::Showlex' => '1.00',   #B\Showlex.pm
        'B::Stackobj' => '1.00',  #B\Stackobj.pm
        'B::Stash' => '1.00',     #B\Stash.pm
        'B::Terse' => 1.02,     #B\Terse.pm
        'B::Xref' => 1.01,      #B\Xref.pm
        'base' => 2.03,         #base.pm
        'Benchmark' => 1.051,   #Benchmark.pm
        'bigint' => 0.04,       #bigint.pm
        'bignum' => 0.14,       #bignum.pm
        'bigrat' => 0.06,       #bigrat.pm
        'blib' => 1.02,         #blib.pm
        'ByteLoader' => 0.05,   #ByteLoader.pm
        'bytes' => 1.01,        #bytes.pm
        'Carp' => 1.01,         #Carp.pm
        'Carp::Heavy' => 1.01,  #Carp\Heavy.pm
        'CGI' => '3.00',          #CGI.pm
        'CGI::Apache' => '1.00',  #CGI\Apache.pm
        'CGI::Carp' => 1.26,    #CGI\Carp.pm
        'CGI::Cookie' => 1.24,  #CGI\Cookie.pm
        'CGI::Fast' => 1.041,   #CGI\Fast.pm
        'CGI::Pretty' => '1.07_00', #CGI\Pretty.pm
        'CGI::Push' => 1.04,    #CGI\Push.pm
        'CGI::Switch' => '1.00',  #CGI\Switch.pm
        'CGI::Util' => 1.31,    #CGI\Util.pm
        'charnames' => 1.02,    #charnames.pm
        'Class::ISA' => 0.32,   #Class\ISA.pm
        'Class::Struct' => 0.63, #Class\Struct.pm
        'Config' => undef,      #Config.pm
        'constant' => 1.04,     #constant.pm
        'CPAN' => '1.76_01',      #CPAN.pm
        'CPAN::FirstTime' => '1.60 ', #CPAN\FirstTime.pm
        'CPAN::Nox' => 1.03,    #CPAN\Nox.pm
        'Cwd' => 2.08,          #Cwd.pm
        'Data::Dumper' => 2.121, #Data\Dumper.pm
        'DB' => '1.0',            #DB.pm
        'Devel::DProf' => '20030813.00', #Devel\DProf.pm
        'Devel::Peek' => 1.01,  #Devel\Peek.pm
        'Devel::PPPort' => 2.009, #Devel\PPPort.pm
        'Devel::SelfStubber' => 1.03, #Devel\SelfStubber.pm
        'diagnostics' => 1.11,  #diagnostics.pm
        'Digest' => 1.02,       #Digest.pm
        'Digest::MD5' => '2.30',  #Digest\MD5.pm
        'DirHandle' => '1.00',    #DirHandle.pm
        'Dumpvalue' => 1.11,    #Dumpvalue.pm
        'DynaLoader' => 1.04,   #DynaLoader.pm
        'Encode' => 1.9801,     #Encode.pm
        'Encode::Alias' => 1.38, #Encode\Alias.pm
        'Encode::Byte' => 1.23, #Encode\Byte.pm
        'Encode::CJKConstants' => 1.02, #Encode\CJKConstants.pm
        'Encode::CN' => 1.24,   #Encode\CN.pm
        'Encode::CN::HZ' => 1.05, #Encode\CN\HZ.pm
        'Encode::Config' => 1.07, #Encode\Config.pm
        'Encode::EBCDIC' => 1.21, #Encode\EBCDIC.pm
        'Encode::Encoder' => 0.07, #Encode\Encoder.pm
        'Encode::Encoding' => 1.33, #Encode\Encoding.pm
        'Encode::Guess' => 1.09, #Encode\Guess.pm
        'Encode::JP' => 1.25,   #Encode\JP.pm
        'Encode::JP::H2Z' => 1.02, #Encode\JP\H2Z.pm
        'Encode::JP::JIS7' => 1.12, #Encode\JP\JIS7.pm
        'Encode::KR' => 1.23,   #Encode\KR.pm
        'Encode::KR::2022_KR' => 1.06, #Encode\KR\2022_KR.pm
        'Encode::MIME::Header' => 1.09, #Encode\MIME\Header.pm
        'Encode::Symbol' => 1.22, #Encode\Symbol.pm
        'Encode::TW' => 1.26,   #Encode\TW.pm
        'Encode::Unicode' => '1.40', #Encode\Unicode.pm
        'Encode::Unicode::UTF7' => 0.02, #Encode\Unicode\UTF7.pm
        'encoding' => 1.47,     #encoding.pm
        'English' => 1.01,      #English.pm
        'Env' => '1.00',          #Env.pm
        'Errno' => '1.09_00',     #Errno.pm
        'Exporter' => 5.567,    #Exporter.pm
        'Exporter::Heavy' => 5.567, #Exporter\Heavy.pm
        'ExtUtils::Command' => 1.05, #ExtUtils\Command.pm
        'ExtUtils::Command::MM' => 0.03, #ExtUtils\Command\MM.pm
        'ExtUtils::Constant' => 0.14, #ExtUtils\Constant.pm
        'ExtUtils::Embed' => 1.250601, #ExtUtils\Embed.pm
        'ExtUtils::Install' => 1.32, #ExtUtils\Install.pm
        'ExtUtils::Installed' => 0.08, #ExtUtils\Installed.pm
        'ExtUtils::Liblist' => 1.01, #ExtUtils\Liblist.pm
        'ExtUtils::Liblist::Kid' => 1.3, #ExtUtils\Liblist\Kid.pm
        'ExtUtils::MakeMaker' => 6.17, #ExtUtils\MakeMaker.pm
        'ExtUtils::MakeMaker::bytes' => 0.01, #ExtUtils\MakeMaker\bytes.pm
        'ExtUtils::MakeMaker::vmsish' => 0.01, #ExtUtils\MakeMaker\vmsish.pm
        'ExtUtils::Manifest' => 1.42, #ExtUtils\Manifest.pm
        'ExtUtils::Miniperl' => undef, #ExtUtils\Miniperl.pm
        'ExtUtils::Mkbootstrap' => 1.15, #ExtUtils\Mkbootstrap.pm
        'ExtUtils::Mksymlists' => 1.19, #ExtUtils\Mksymlists.pm
        'ExtUtils::MM' => 0.04, #ExtUtils\MM.pm
        'ExtUtils::MM_Any' => 0.07, #ExtUtils\MM_Any.pm
        'ExtUtils::MM_BeOS' => 1.04, #ExtUtils\MM_BeOS.pm
        'ExtUtils::MM_Cygwin' => 1.06, #ExtUtils\MM_Cygwin.pm
        'ExtUtils::MM_DOS' => 0.02, #ExtUtils\MM_DOS.pm
        'ExtUtils::MM_MacOS' => 1.07, #ExtUtils\MM_MacOS.pm
        'ExtUtils::MM_NW5' => 2.06, #ExtUtils\MM_NW5.pm
        'ExtUtils::MM_OS2' => 1.04, #ExtUtils\MM_OS2.pm
        'ExtUtils::MM_Unix' => 1.42, #ExtUtils\MM_Unix.pm
        'ExtUtils::MM_UWIN' => 0.02, #ExtUtils\MM_UWIN.pm
        'ExtUtils::MM_VMS' => '5.70', #ExtUtils\MM_VMS.pm
        'ExtUtils::MM_Win32' => 1.09, #ExtUtils\MM_Win32.pm
        'ExtUtils::MM_Win95' => 0.03, #ExtUtils\MM_Win95.pm
        'ExtUtils::MY' => 0.01, #ExtUtils\MY.pm
        'ExtUtils::Packlist' => 0.04, #ExtUtils\Packlist.pm
        'ExtUtils::testlib' => 1.15, #ExtUtils\testlib.pm
        'ExtUtils::XSSymSet' => '1.0',  #vms\ext\XSSymSet.pm
        'Fatal' => 1.03,        #Fatal.pm
        'Fcntl' => 1.05,        #Fcntl.pm
        'fields' => 2.03,       #fields.pm
        'File::Basename' => 2.72, #File\Basename.pm
        'File::CheckTree' => 4.2, #File\CheckTree.pm
        'File::Compare' => 1.1003, #File\Compare.pm
        'File::Copy' => 2.06,   #File\Copy.pm
        'File::DosGlob' => '1.00', #File\DosGlob.pm
        'File::Find' => 1.05,   #File\Find.pm
        'File::Glob' => 1.02,   #File\Glob.pm
        'File::Path' => 1.06,   #File\Path.pm
        'File::Spec' => 0.86,   #File\Spec.pm
        'File::Spec::Cygwin' => 1.1, #File\Spec\Cygwin.pm
        'File::Spec::Epoc' => 1.1, #File\Spec\Epoc.pm
        'File::Spec::Functions' => 1.3, #File\Spec\Functions.pm
        'File::Spec::Mac' => 1.4, #File\Spec\Mac.pm
        'File::Spec::OS2' => 1.2, #File\Spec\OS2.pm
        'File::Spec::Unix' => 1.5, #File\Spec\Unix.pm
        'File::Spec::VMS' => 1.4, #File\Spec\VMS.pm
        'File::Spec::Win32' => 1.4, #File\Spec\Win32.pm
        'File::stat' => '1.00',   #File\stat.pm
        'File::Temp' => 0.14,   #File\Temp.pm
        'FileCache' => 1.03,    #FileCache.pm
        'FileHandle' => 2.01,   #FileHandle.pm
        'filetest' => 1.01,     #filetest.pm
        'Filter::Simple' => 0.78, #Filter\Simple.pm
        'Filter::Util::Call' => 1.0601, #Filter\Util\Call.pm
        'FindBin' => 1.43,      #FindBin.pm
        'GDBM_File' => '1.07', #ext\GDBM_File\GDBM_File.pm
        'Getopt::Long' => 2.34, #Getopt\Long.pm
        'Getopt::Std' => 1.04,  #Getopt\Std.pm
        'Hash::Util' => 0.05,   #Hash\Util.pm
        'I18N::Collate' => '1.00', #I18N\Collate.pm
        'I18N::Langinfo' => '0.02', #I18N\Langinfo.pm
        'I18N::LangTags' => 0.29, #I18N\LangTags.pm
        'I18N::LangTags::List' => 0.29, #I18N\LangTags\List.pm
        'if' => 0.03,           #if.pm
        'integer' => '1.00',      #integer.pm
        'IO' => 1.21,           #IO.pm
        'IO::Dir' => 1.04,      #IO\Dir.pm
        'IO::File' => '1.10',     #IO\File.pm
        'IO::Handle' => 1.23,   #IO\Handle.pm
        'IO::Pipe' => 1.122,    #IO\Pipe.pm
        'IO::Poll' => 0.06,     #IO\Poll.pm
        'IO::Seekable' => 1.09, #IO\Seekable.pm
        'IO::Select' => 1.16,   #IO\Select.pm
        'IO::Socket' => 1.28,   #IO\Socket.pm
        'IO::Socket::INET' => 1.27, #IO\Socket\INET.pm
        'IO::Socket::UNIX' => 1.21, #IO\Socket\UNIX.pm
        'IPC::Msg' => 1.02,     #IPC\Msg.pm
        'IPC::Open2' => 1.01,   #IPC\Open2.pm
        'IPC::Open3' => 1.0105, #IPC\Open3.pm
        'IPC::Semaphore' => 1.02, #IPC\Semaphore.pm
        'IPC::SysV' => 1.04,    #IPC\SysV.pm
        'JNI' => '0.2',         #jpl\JNI\JNI.pm
        'JPL::AutoLoader' => undef, #jpl\JPL\AutoLoader.pm
        'JPL::Class' => undef,  #jpl\JPL\Class.pm
        'JPL::Compile' => undef, #jpl\JPL\Compile.pm
        'less' => 0.01,         #less.pm
        'lib' => 0.5565,        #lib.pm
        'List::Util' => 1.13,   #List\Util.pm
        'locale' => '1.00',       #locale.pm
        'Locale::Constants' => 2.01, #Locale\Constants.pm
        'Locale::Country' => 2.61, #Locale\Country.pm
        'Locale::Currency' => 2.21, #Locale\Currency.pm
        'Locale::Language' => 2.21, #Locale\Language.pm
        'Locale::Maketext' => 1.06, #Locale\Maketext.pm
        'Locale::Maketext::Guts' => undef, #Locale\Maketext\Guts.pm
        'Locale::Maketext::GutsLoader' => undef, #Locale\Maketext\GutsLoader.pm
        'Locale::Script' => 2.21, #Locale\Script.pm
        'Math::BigFloat' => '1.40', #Math\BigFloat.pm
        'Math::BigFloat::Trace' => 0.01, #Math\BigFloat\Trace.pm
        'Math::BigInt' => 1.66, #Math\BigInt.pm
        'Math::BigInt::Calc' => 0.36, #Math\BigInt\Calc.pm
        'Math::BigInt::Scalar' => 0.11, #Math\BigInt\Scalar.pm
        'Math::BigInt::Trace' => 0.01, #Math\BigInt\Trace.pm
        'Math::BigRat' => '0.10', #Math\BigRat.pm
        'Math::Complex' => 1.34, #Math\Complex.pm
        'Math::Trig' => 1.02,   #Math\Trig.pm
        'Memoize' => 1.01,      #Memoize.pm
        'Memoize::AnyDBM_File' => 0.65, #Memoize\AnyDBM_File.pm
        'Memoize::Expire' => '1.00', #Memoize\Expire.pm
        'Memoize::ExpireFile' => 1.01, #Memoize\ExpireFile.pm
        'Memoize::ExpireTest' => 0.65, #Memoize\ExpireTest.pm
        'Memoize::NDBM_File' => 0.65, #Memoize\NDBM_File.pm
        'Memoize::SDBM_File' => 0.65, #Memoize\SDBM_File.pm
        'Memoize::Storable' => 0.65, #Memoize\Storable.pm
        'MIME::Base64' => 2.21, #MIME\Base64.pm
        'MIME::QuotedPrint' => 2.21, #MIME\QuotedPrint.pm
        'NDBM_File' => '1.05',  #ext\NDBM_File\NDBM_File.pm
        'Net::Cmd' => 2.24,     #Net\Cmd.pm
        'Net::Config' => '1.10',  #Net\Config.pm
        'Net::Domain' => 2.19,  #Net\Domain.pm
        'Net::FTP' => 2.72,     #Net\FTP.pm
        'Net::FTP::A' => 1.16,  #Net\FTP\A.pm
        'Net::FTP::dataconn' => 0.11, #Net\FTP\dataconn.pm
        'Net::FTP::E' => 0.01,  #Net\FTP\E.pm
        'Net::FTP::I' => 1.12,  #Net\FTP\I.pm
        'Net::FTP::L' => 0.01,  #Net\FTP\L.pm
        'Net::hostent' => 1.01, #Net\hostent.pm
        'Net::netent' => '1.00',  #Net\netent.pm
        'Net::Netrc' => 2.12,   #Net\Netrc.pm
        'Net::NNTP' => 2.22,    #Net\NNTP.pm
        'Net::Ping' => 2.31,    #Net\Ping.pm
        'Net::POP3' => 2.24,    #Net\POP3.pm
        'Net::protoent' => '1.00', #Net\protoent.pm
        'Net::servent' => 1.01, #Net\servent.pm
        'Net::SMTP' => 2.26,    #Net\SMTP.pm
        'Net::Time' => 2.09,    #Net\Time.pm
        'NEXT' => '0.60',         #NEXT.pm
        'O' => '1.00',            #O.pm
        'ODBM_File' => '1.04',  #ext\ODBM_File\ODBM_File.pm
        'Opcode' => 1.05,       #Opcode.pm
        'open' => 1.02,         #open.pm
        'ops' => '1.00',          #ops.pm
        'OS2::DLL' => '1.01',   #os2\OS2\REXX\DLL\DLL.pm
        'OS2::ExtAttr' => '0.02', #os2\OS2\ExtAttr\ExtAttr.pm
        'OS2::PrfDB' => '0.03', #os2\OS2\PrfDB\PrfDB.pm
        'OS2::Process' => '1.01', #os2\OS2\Process\Process.pm
        'OS2::REXX' => '1.02',  #os2\OS2\REXX\REXX.pm
        'overload' => 1.01,     #overload.pm
        'PerlIO' => 1.02,       #PerlIO.pm
        'PerlIO::encoding' => 0.07, #PerlIO\encoding.pm
        'PerlIO::scalar' => 0.02, #PerlIO\scalar.pm
        'PerlIO::via' => 0.02,  #PerlIO\via.pm
        'PerlIO::via::QuotedPrint' => 0.05, #PerlIO\via\QuotedPrint.pm
        'Pod::Checker' => 1.41, #Pod\Checker.pm
        'Pod::Find' => 0.24,    #Pod\Find.pm
        'Pod::Functions' => 1.02, #Pod\Functions.pm
        'Pod::Html' => 1.0501,  #Pod\Html.pm
        'Pod::InputObjects' => 1.14, #Pod\InputObjects.pm
        'Pod::LaTeX' => 0.55,   #Pod\LaTeX.pm
        'Pod::Man' => 1.37,     #Pod\Man.pm
        'Pod::ParseLink' => 1.06, #Pod\ParseLink.pm
        'Pod::Parser' => 1.13,  #Pod\Parser.pm
        'Pod::ParseUtils' => 0.3, #Pod\ParseUtils.pm
        'Pod::Perldoc' => 3.11, #Pod\Perldoc.pm
        'Pod::Perldoc::BaseTo' => undef, #Pod\Perldoc\BaseTo.pm
        'Pod::Perldoc::GetOptsOO' => undef, #Pod\Perldoc\GetOptsOO.pm
        'Pod::Perldoc::ToChecker' => undef, #Pod\Perldoc\ToChecker.pm
        'Pod::Perldoc::ToMan' => undef, #Pod\Perldoc\ToMan.pm
        'Pod::Perldoc::ToNroff' => undef, #Pod\Perldoc\ToNroff.pm
        'Pod::Perldoc::ToPod' => undef, #Pod\Perldoc\ToPod.pm
        'Pod::Perldoc::ToRtf' => undef, #Pod\Perldoc\ToRtf.pm
        'Pod::Perldoc::ToText' => undef, #Pod\Perldoc\ToText.pm
        'Pod::Perldoc::ToTk' => undef, #Pod\Perldoc\ToTk.pm
        'Pod::Perldoc::ToXml' => undef, #Pod\Perldoc\ToXml.pm
        'Pod::Plainer' => 0.01, #Pod\Plainer.pm
        'Pod::PlainText' => 2.01, #Pod\PlainText.pm
        'Pod::Select' => 1.13,  #Pod\Select.pm
        'Pod::Text' => 2.21,    #Pod\Text.pm
        'Pod::Text::Color' => 1.04, #Pod\Text\Color.pm
        'Pod::Text::Overstrike' => 1.1, #Pod\Text\Overstrike.pm
        'Pod::Text::Termcap' => 1.11, #Pod\Text\Termcap.pm
        'Pod::Usage' => 1.16,   #Pod\Usage.pm
        'POSIX' => 1.06,        #POSIX.pm
        're' => 0.04,           #re.pm
        'Safe' => '2.10',         #Safe.pm
        'Scalar::Util' => 1.13, #Scalar\Util.pm
        'SDBM_File' => 1.04,    #SDBM_File.pm
        'Search::Dict' => 1.02, #Search\Dict.pm
        'SelectSaver' => '1.00',  #SelectSaver.pm
        'SelfLoader' => 1.0904, #SelfLoader.pm
        'Shell' => 0.5,         #Shell.pm
        'sigtrap' => 1.02,      #sigtrap.pm
        'Socket' => 1.76,       #Socket.pm
        'sort' => 1.02,         #sort.pm
        'Storable' => 2.08,     #Storable.pm
        'strict' => 1.03,       #strict.pm
        'subs' => '1.00',         #subs.pm
        'Switch' => '2.10',       #Switch.pm
        'Symbol' => 1.05,       #Symbol.pm
        'Sys::Hostname' => 1.11, #Sys\Hostname.pm
        'Sys::Syslog' => '0.04', #ext\Sys\Syslog\Syslog.pm
        'Term::ANSIColor' => 1.07, #Term\ANSIColor.pm
        'Term::Cap' => 1.08,    #Term\Cap.pm
        'Term::Complete' => 1.401, #Term\Complete.pm
        'Term::ReadLine' => 1.01, #Term\ReadLine.pm
        'Test' => 1.24,         #Test.pm
        'Test::Builder' => 0.17, #Test\Builder.pm
        'Test::Harness' => '2.30', #Test\Harness.pm
        'Test::Harness::Assert' => 0.01, #Test\Harness\Assert.pm
        'Test::Harness::Iterator' => 0.01, #Test\Harness\Iterator.pm
        'Test::Harness::Straps' => 0.15, #Test\Harness\Straps.pm
        'Test::More' => 0.47,   #Test\More.pm
        'Test::Simple' => 0.47, #Test\Simple.pm
        'Text::Abbrev' => 1.01, #Text\Abbrev.pm
        'Text::Balanced' => 1.95, #Text\Balanced.pm
        'Text::ParseWords' => 3.21, #Text\ParseWords.pm
        'Text::Soundex' => 1.01, #Text\Soundex.pm
        'Text::Tabs' => 98.112801, #Text\Tabs.pm
        'Text::Wrap' => 2001.09291, #Text\Wrap.pm
        'Thread' => '2.00',       #Thread.pm
        'Thread::Queue' => '2.00', #Thread\Queue.pm
        'Thread::Semaphore' => 2.01, #Thread\Semaphore.pm
        'Thread::Signal' => '1.00', #Thread\Signal.pm
        'Thread::Specific' => '1.00', #Thread\Specific.pm
        'threads' => '1.00',      #threads.pm
        'threads::shared' => 0.91, #threads\shared.pm
        'Tie::Array' => 1.03,   #Tie\Array.pm
        'Tie::File' => 0.97,    #Tie\File.pm
        'Tie::Handle' => 4.1,   #Tie\Handle.pm
        'Tie::Hash' => '1.00',    #Tie\Hash.pm
        'Tie::Memoize' => '1.0',  #Tie\Memoize.pm
        'Tie::RefHash' => 1.31, #Tie\RefHash.pm
        'Tie::Scalar' => '1.00',  #Tie\Scalar.pm
        'Tie::SubstrHash' => '1.00', #Tie\SubstrHash.pm
        'Time::gmtime' => 1.02, #Time\gmtime.pm
        'Time::HiRes' => 1.52,  #Time\HiRes.pm
        'Time::Local' => 1.07,  #Time\Local.pm
        'Time::localtime' => 1.02, #Time\localtime.pm
        'Time::tm' => '1.00',     #Time\tm.pm
        'Unicode' => '4.0.0', # lib/unicore/version
        'Unicode::Collate' => '0.30', #Unicode\Collate.pm
        'Unicode::Normalize' => 0.25, #Unicode\Normalize.pm
        'Unicode::UCD' => 0.21, #Unicode\UCD.pm
        'UNIVERSAL' => 1.01,    #UNIVERSAL.pm
        'User::grent' => '1.00',  #User\grent.pm
        'User::pwent' => '1.00',  #User\pwent.pm
        'utf8' => 1.02,         #utf8.pm
        'vars' => 1.01,         #vars.pm
        'VMS::DCLsym' => '1.02', #vms\ext\DCLsym\DCLsym.pm
        'VMS::Filespec' => '1.11', #vms\ext\Filespec.pm
        'VMS::Stdio' => '2.3',  #vms\ext\Stdio\Stdio.pm
        'vmsish' => 1.01,       #vmsish.pm
        'warnings' => 1.03,     #warnings.pm
        'warnings::register' => '1.00', #warnings\register.pm
        'XS::APItest' => 0.02,  #XS\APItest.pm
        'XS::Typemap' => 0.01,  #XS\Typemap.pm
        'XSLoader' => 0.02,     #XSLoader.pm
    },

    5.008003 => {
        'AnyDBM_File'           => '1.00',  #lib/AnyDBM_File.pm
        'Attribute::Handlers'   => '0.78',  #lib/Attribute/Handlers.pm
        'attributes'            => '0.06',  #lib/attributes.pm
        'attrs'                 => '1.01',  #lib/attrs.pm
        'AutoLoader'            => '5.60',  #lib/AutoLoader.pm
        'AutoSplit'             => '1.04',  #lib/AutoSplit.pm
        'autouse'               => '1.03',  #lib/autouse.pm
        'B'                     => '1.02',  #lib/B.pm
        'base'                  => '2.04',  #lib/base.pm
        'B::Asmdata'            => '1.01',  #lib/B/Asmdata.pm
        'B::Assembler'          => '0.06',  #lib/B/Assembler.pm
        'B::Bblock'             => '1.02',  #lib/B/Bblock.pm
        'B::Bytecode'           => '1.01',  #lib/B/Bytecode.pm
        'B::C'                  => '1.02',  #lib/B/C.pm
        'B::CC'                 => '1.00',  #lib/B/CC.pm
        'B::Concise'            => '0.56',  #lib/B/Concise.pm
        'B::Debug'              => '1.01',  #lib/B/Debug.pm
        'B::Deparse'            => '0.64',  #lib/B/Deparse.pm
        'B::Disassembler'       => '1.03',  #lib/B/Disassembler.pm
        'Benchmark'             => '1.052',  #lib/Benchmark.pm
        'bigint'                => '0.04',  #lib/bigint.pm
        'bignum'                => '0.14',  #lib/bignum.pm
        'bigrat'                => '0.06',  #lib/bigrat.pm
        'blib'                  => '1.02',  #lib/blib.pm
        'B::Lint'               => '1.02',  #lib/B/Lint.pm
        'B::Showlex'            => '1.00',  #lib/B/Showlex.pm
        'B::Stackobj'           => '1.00',  #lib/B/Stackobj.pm
        'B::Stash'              => '1.00',  #lib/B/Stash.pm
        'B::Terse'              => '1.02',  #lib/B/Terse.pm
        'B::Xref'               => '1.01',  #lib/B/Xref.pm
        'ByteLoader'            => '0.05',  #lib/ByteLoader.pm
        'bytes'                 => '1.01',  #lib/bytes.pm
        'Carp'                  => '1.01',  #lib/Carp.pm
        'Carp::Heavy'           => '1.01',  #lib/Carp/Heavy.pm
        'CGI'                   => '3.01',  #lib/CGI.pm
        'CGI::Apache'           => '1.00',  #lib/CGI/Apache.pm
        'CGI::Carp'             => '1.27',  #lib/CGI/Carp.pm
        'CGI::Cookie'           => '1.24',  #lib/CGI/Cookie.pm
        'CGI::Fast'             => '1.05',  #lib/CGI/Fast.pm
        'CGI::Pretty'           => '1.08',  #lib/CGI/Pretty.pm
        'CGI::Push'             => '1.04',  #lib/CGI/Push.pm
        'CGI::Switch'           => '1.00',  #lib/CGI/Switch.pm
        'CGI::Util'             => '1.4',  #lib/CGI/Util.pm
        'charnames'             => '1.02',  #lib/charnames.pm
        'Class::ISA'            => '0.32',  #lib/Class/ISA.pm
        'Class::Struct'         => '0.63',  #lib/Class/Struct.pm
        'Config'                => undef,  #lib/Config.pm
        'constant'              => '1.04',  #lib/constant.pm
        'CPAN'                  => '1.76_01',  #lib/CPAN.pm
        'CPAN::FirstTime'       => '1.60 ',  #lib/CPAN/FirstTime.pm
        'CPAN::Nox'             => '1.03',  #lib/CPAN/Nox.pm
        'Cwd'                   => '2.12',  #lib/Cwd.pm
        'Data::Dumper'          => '2.121',  #lib/Data/Dumper.pm
        'DB'                    => '1.0',  #lib/DB.pm
        'DB_File'               => '1.808',  #lib/DB_File.pm
        'Devel::DProf'          => '20030813.00',  #lib/Devel/DProf.pm
        'Devel::Peek'           => '1.01',  #lib/Devel/Peek.pm
        'Devel::PPPort'         => '2.011',  #lib/Devel/PPPort.pm
        'Devel::SelfStubber'    => '1.03',  #lib/Devel/SelfStubber.pm
        'diagnostics'           => '1.12',  #lib/diagnostics.pm
        'Digest'                => '1.05',  #lib/Digest.pm
        'Digest::base'          => '1.00',  #lib/Digest/base.pm
        'Digest::MD5'           => '2.33',  #lib/Digest/MD5.pm
        'DirHandle'             => '1.00',  #lib/DirHandle.pm
        'Dumpvalue'             => '1.11',  #lib/Dumpvalue.pm
        'DynaLoader'            => '1.04',  #lib/DynaLoader.pm
        'Encode'                => '1.99',  #lib/Encode.pm
        'Encode::Alias'         => '1.38',  #lib/Encode/Alias.pm
        'Encode::Byte'          => '1.23',  #lib/Encode/Byte.pm
        'Encode::CJKConstants'  => '1.02',  #lib/Encode/CJKConstants.pm
        'Encode::CN'            => '1.24',  #lib/Encode/CN.pm
        'Encode::CN::HZ'        => '1.05',  #lib/Encode/CN/HZ.pm
        'Encode::Config'        => '1.07',  #lib/Encode/Config.pm
        'Encode::EBCDIC'        => '1.21',  #lib/Encode/EBCDIC.pm
        'Encode::Encoder'       => '0.07',  #lib/Encode/Encoder.pm
        'Encode::Encoding'      => '1.33',  #lib/Encode/Encoding.pm
        'Encode::Guess'         => '1.09',  #lib/Encode/Guess.pm
        'Encode::JP'            => '1.25',  #lib/Encode/JP.pm
        'Encode::JP::H2Z'       => '1.02',  #lib/Encode/JP/H2Z.pm
        'Encode::JP::JIS7'      => '1.12',  #lib/Encode/JP/JIS7.pm
        'Encode::KR'            => '1.23',  #lib/Encode/KR.pm
        'Encode::KR::2022_KR'   => '1.06',  #lib/Encode/KR/2022_KR.pm
        'Encode::MIME::Header'  => '1.09',  #lib/Encode/MIME/Header.pm
        'Encode::Symbol'        => '1.22',  #lib/Encode/Symbol.pm
        'Encode::TW'            => '1.26',  #lib/Encode/TW.pm
        'Encode::Unicode'       => '1.40',  #lib/Encode/Unicode.pm
        'Encode::Unicode::UTF7' => '0.02',  #lib/Encode/Unicode/UTF7.pm
        'encoding'              => '1.48',  #lib/encoding.pm
        'English'               => '1.01',  #lib/English.pm
        'Env'                   => '1.00',  #lib/Env.pm
        'Errno'                 => '1.09_00',  #lib/Errno.pm
        'Exporter'              => '5.57',  #lib/Exporter.pm
        'Exporter::Heavy'       => '5.567',  #lib/Exporter/Heavy.pm
        'ExtUtils::Command'     => '1.05',  #lib/ExtUtils/Command.pm
        'ExtUtils::Command::MM' => '0.03',  #lib/ExtUtils/Command/MM.pm
        'ExtUtils::Constant'    => '0.14',  #lib/ExtUtils/Constant.pm
        'ExtUtils::Embed'       => '1.250601',  #lib/ExtUtils/Embed.pm
        'ExtUtils::Install'     => '1.32',  #lib/ExtUtils/Install.pm
        'ExtUtils::Installed'   => '0.08',  #lib/ExtUtils/Installed.pm
        'ExtUtils::Liblist'     => '1.01',  #lib/ExtUtils/Liblist.pm
        'ExtUtils::Liblist::Kid'=> '1.3',  #lib/ExtUtils/Liblist/Kid.pm
        'ExtUtils::MakeMaker'   => '6.17',  #lib/ExtUtils/MakeMaker.pm
        'ExtUtils::MakeMaker::bytes'=> '0.01',  #lib/ExtUtils/MakeMaker/bytes.pm
        'ExtUtils::MakeMaker::vmsish'=> '0.01',  #lib/ExtUtils/MakeMaker/vmsish.pm
        'ExtUtils::Manifest'    => '1.42',  #lib/ExtUtils/Manifest.pm
        'ExtUtils::Miniperl'    => undef,  #lib/ExtUtils/Miniperl.pm
        'ExtUtils::Mkbootstrap' => '1.15',  #lib/ExtUtils/Mkbootstrap.pm
        'ExtUtils::Mksymlists'  => '1.19',  #lib/ExtUtils/Mksymlists.pm
        'ExtUtils::MM'          => '0.04',  #lib/ExtUtils/MM.pm
        'ExtUtils::MM_Any'      => '0.07',  #lib/ExtUtils/MM_Any.pm
        'ExtUtils::MM_BeOS'     => '1.04',  #lib/ExtUtils/MM_BeOS.pm
        'ExtUtils::MM_Cygwin'   => '1.06',  #lib/ExtUtils/MM_Cygwin.pm
        'ExtUtils::MM_DOS'      => '0.02',  #lib/ExtUtils/MM_DOS.pm
        'ExtUtils::MM_MacOS'    => '1.07',  #lib/ExtUtils/MM_MacOS.pm
        'ExtUtils::MM_NW5'      => '2.06',  #lib/ExtUtils/MM_NW5.pm
        'ExtUtils::MM_OS2'      => '1.04',  #lib/ExtUtils/MM_OS2.pm
        'ExtUtils::MM_Unix'     => '1.42',  #lib/ExtUtils/MM_Unix.pm
        'ExtUtils::MM_UWIN'     => '0.02',  #lib/ExtUtils/MM_UWIN.pm
        'ExtUtils::MM_VMS'      => '5.70',  #lib/ExtUtils/MM_VMS.pm
        'ExtUtils::MM_Win32'    => '1.09',  #lib/ExtUtils/MM_Win32.pm
        'ExtUtils::MM_Win95'    => '0.03',  #lib/ExtUtils/MM_Win95.pm
        'ExtUtils::MY'          => '0.01',  #lib/ExtUtils/MY.pm
        'ExtUtils::Packlist'    => '0.04',  #lib/ExtUtils/Packlist.pm
        'ExtUtils::testlib'     => '1.15',  #lib/ExtUtils/testlib.pm
        'ExtUtils::XSSymSet'    => '1.0',  #vms/ext/XSSymSet.pm
        'Fatal'                 => '1.03',  #lib/Fatal.pm
        'Fcntl'                 => '1.05',  #lib/Fcntl.pm
        'fields'                => '2.03',  #lib/fields.pm
        'File::Basename'        => '2.72',  #lib/File/Basename.pm
        'FileCache'             => '1.03',  #lib/FileCache.pm
        'File::CheckTree'       => '4.3',  #lib/File/CheckTree.pm
        'File::Compare'         => '1.1003',  #lib/File/Compare.pm
        'File::Copy'            => '2.07',  #lib/File/Copy.pm
        'File::DosGlob'         => '1.00',  #lib/File/DosGlob.pm
        'File::Find'            => '1.06',  #lib/File/Find.pm
        'File::Glob'            => '1.02',  #lib/File/Glob.pm
        'FileHandle'            => '2.01',  #lib/FileHandle.pm
        'File::Path'            => '1.06',  #lib/File/Path.pm
        'File::Spec'            => '0.87',  #lib/File/Spec.pm
        'File::Spec::Cygwin'    => '1.1',  #lib/File/Spec/Cygwin.pm
        'File::Spec::Epoc'      => '1.1',  #lib/File/Spec/Epoc.pm
        'File::Spec::Functions' => '1.3',  #lib/File/Spec/Functions.pm
        'File::Spec::Mac'       => '1.4',  #lib/File/Spec/Mac.pm
        'File::Spec::OS2'       => '1.2',  #lib/File/Spec/OS2.pm
        'File::Spec::Unix'      => '1.5',  #lib/File/Spec/Unix.pm
        'File::Spec::VMS'       => '1.4',  #lib/File/Spec/VMS.pm
        'File::Spec::Win32'     => '1.4',  #lib/File/Spec/Win32.pm
        'File::stat'            => '1.00',  #lib/File/stat.pm
        'File::Temp'            => '0.14',  #lib/File/Temp.pm
        'filetest'              => '1.01',  #lib/filetest.pm
        'Filter::Simple'        => '0.78',  #lib/Filter/Simple.pm
        'Filter::Util::Call'    => '1.0601',  #lib/Filter/Util/Call.pm
        'FindBin'               => '1.44',  #lib/FindBin.pm
        'GDBM_File'             => '1.07',  #lib/GDBM_File.pm
        'Getopt::Long'          => '2.34',  #lib/Getopt/Long.pm
        'Getopt::Std'           => '1.05',  #lib/Getopt/Std.pm
        'Hash::Util'            => '0.05',  #lib/Hash/Util.pm
        'I18N::Collate'         => '1.00',  #lib/I18N/Collate.pm
        'I18N::Langinfo'        => '0.02',  #lib/I18N/Langinfo.pm
        'I18N::LangTags'        => '0.29',  #lib/I18N/LangTags.pm
        'I18N::LangTags::List'  => '0.29',  #lib/I18N/LangTags/List.pm
        'if'                    => '0.03',  #lib/if.pm
        'integer'               => '1.00',  #lib/integer.pm
        'IO'                    => '1.21',  #lib/IO.pm
        'IO::Dir'               => '1.04',  #lib/IO/Dir.pm
        'IO::File'              => '1.10',  #lib/IO/File.pm
        'IO::Handle'            => '1.23',  #lib/IO/Handle.pm
        'IO::Pipe'              => '1.122',  #lib/IO/Pipe.pm
        'IO::Poll'              => '0.06',  #lib/IO/Poll.pm
        'IO::Seekable'          => '1.09',  #lib/IO/Seekable.pm
        'IO::Select'            => '1.16',  #lib/IO/Select.pm
        'IO::Socket'            => '1.28',  #lib/IO/Socket.pm
        'IO::Socket::INET'      => '1.27',  #lib/IO/Socket/INET.pm
        'IO::Socket::UNIX'      => '1.21',  #lib/IO/Socket/UNIX.pm
        'IPC::Msg'              => '1.02',  #lib/IPC/Msg.pm
        'IPC::Open2'            => '1.01',  #lib/IPC/Open2.pm
        'IPC::Open3'            => '1.0105',  #lib/IPC/Open3.pm
        'IPC::Semaphore'        => '1.02',  #lib/IPC/Semaphore.pm
        'IPC::SysV'             => '1.04',  #lib/IPC/SysV.pm
        'JNI'                   => '0.2',  #jpl/JNI/JNI.pm
        'JPL::AutoLoader'       => undef,  #jpl/JPL/AutoLoader.pm
        'JPL::Class'            => undef,  #jpl/JPL/Class.pm
        'JPL::Compile'          => undef,  #jpl/JPL/Compile.pm
        'less'                  => '0.01',  #lib/less.pm
        'lib'                   => '0.5565',  #lib/lib.pm
        'List::Util'            => '1.13',  #lib/List/Util.pm
        'locale'                => '1.00',  #lib/locale.pm
        'Locale::Constants'     => '2.01',  #lib/Locale/Constants.pm
        'Locale::Country'       => '2.61',  #lib/Locale/Country.pm
        'Locale::Currency'      => '2.21',  #lib/Locale/Currency.pm
        'Locale::Language'      => '2.21',  #lib/Locale/Language.pm
        'Locale::Maketext'      => '1.06',  #lib/Locale/Maketext.pm
        'Locale::Maketext::GutsLoader'=> undef,  #lib/Locale/Maketext/GutsLoader.pm
        'Locale::Maketext::Guts'=> undef,  #lib/Locale/Maketext/Guts.pm
        'Locale::Script'        => '2.21',  #lib/Locale/Script.pm
        'Math::BigFloat'        => '1.42',  #lib/Math/BigFloat.pm
        'Math::BigFloat::Trace' => '0.01',  #lib/Math/BigFloat/Trace.pm
        'Math::BigInt'          => '1.68',  #lib/Math/BigInt.pm
        'Math::BigInt::Calc'    => '0.38',  #lib/Math/BigInt/Calc.pm
        'Math::BigInt::CalcEmu' => '0.02',  #lib/Math/BigInt/CalcEmu.pm
        'Math::BigInt::Trace'   => '0.01',  #lib/Math/BigInt/Trace.pm
        'Math::BigRat'          => '0.10',  #lib/Math/BigRat.pm
        'Math::Complex'         => '1.34',  #lib/Math/Complex.pm
        'Math::Trig'            => '1.02',  #lib/Math/Trig.pm
        'Memoize'               => '1.01',  #lib/Memoize.pm
        'Memoize::AnyDBM_File'  => '0.65',  #lib/Memoize/AnyDBM_File.pm
        'Memoize::Expire'       => '1.00',  #lib/Memoize/Expire.pm
        'Memoize::ExpireFile'   => '1.01',  #lib/Memoize/ExpireFile.pm
        'Memoize::ExpireTest'   => '0.65',  #lib/Memoize/ExpireTest.pm
        'Memoize::NDBM_File'    => '0.65',  #lib/Memoize/NDBM_File.pm
        'Memoize::SDBM_File'    => '0.65',  #lib/Memoize/SDBM_File.pm
        'Memoize::Storable'     => '0.65',  #lib/Memoize/Storable.pm
        'MIME::Base64'          => '2.21',  #lib/MIME/Base64.pm
        'MIME::QuotedPrint'     => '2.21',  #lib/MIME/QuotedPrint.pm
        'NDBM_File'             => '1.05',  #lib/NDBM_File.pm
        'Net::Cmd'              => '2.24',  #lib/Net/Cmd.pm
        'Net::Config'           => '1.10',  #lib/Net/Config.pm
        'Net::Domain'           => '2.19',  #lib/Net/Domain.pm
        'Net::FTP'              => '2.72',  #lib/Net/FTP.pm
        'Net::FTP::A'           => '1.16',  #lib/Net/FTP/A.pm
        'Net::FTP::dataconn'    => '0.11',  #lib/Net/FTP/dataconn.pm
        'Net::FTP::E'           => '0.01',  #lib/Net/FTP/E.pm
        'Net::FTP::I'           => '1.12',  #lib/Net/FTP/I.pm
        'Net::FTP::L'           => '0.01',  #lib/Net/FTP/L.pm
        'Net::hostent'          => '1.01',  #lib/Net/hostent.pm
        'Net::netent'           => '1.00',  #lib/Net/netent.pm
        'Net::Netrc'            => '2.12',  #lib/Net/Netrc.pm
        'Net::NNTP'             => '2.22',  #lib/Net/NNTP.pm
        'Net::Ping'             => '2.31',  #lib/Net/Ping.pm
        'Net::POP3'             => '2.24',  #lib/Net/POP3.pm
        'Net::protoent'         => '1.00',  #lib/Net/protoent.pm
        'Net::servent'          => '1.01',  #lib/Net/servent.pm
        'Net::SMTP'             => '2.26',  #lib/Net/SMTP.pm
        'Net::Time'             => '2.09',  #lib/Net/Time.pm
        'NEXT'                  => '0.60',  #lib/NEXT.pm
        'O'                     => '1.00',  #lib/O.pm
        'ODBM_File'             => '1.04',  #ext/ODBM_File/ODBM_File.pm
        'Opcode'                => '1.05',  #lib/Opcode.pm
        'open'                  => '1.02',  #lib/open.pm
        'ops'                   => '1.00',  #lib/ops.pm
        'OS2::DLL'              => '1.02',  #os2/OS2/REXX/DLL/DLL.pm
        'OS2::ExtAttr'          => '0.02',  #os2/OS2/ExtAttr/ExtAttr.pm
        'OS2::PrfDB'            => '0.03',  #os2/OS2/PrfDB/PrfDB.pm
        'OS2::Process'          => '1.01',  #os2/OS2/Process/Process.pm
        'OS2::REXX'             => '1.02',  #os2/OS2/REXX/REXX.pm
        'overload'              => '1.01',  #lib/overload.pm
        'PerlIO'                => '1.03',  #lib/PerlIO.pm
        'PerlIO::encoding'      => '0.07',  #lib/PerlIO/encoding.pm
        'PerlIO::scalar'        => '0.02',  #lib/PerlIO/scalar.pm
        'PerlIO::via'           => '0.02',  #lib/PerlIO/via.pm
        'PerlIO::via::QuotedPrint'=> '0.06',  #lib/PerlIO/via/QuotedPrint.pm
        'Pod::Checker'          => '1.41',  #lib/Pod/Checker.pm
        'Pod::Find'             => '0.24',  #lib/Pod/Find.pm
        'Pod::Functions'        => '1.02',  #lib/Pod/Functions.pm
        'Pod::Html'             => '1.0502',  #lib/Pod/Html.pm
        'Pod::InputObjects'     => '1.14',  #lib/Pod/InputObjects.pm
        'Pod::LaTeX'            => '0.55',  #lib/Pod/LaTeX.pm
        'Pod::Man'              => '1.37',  #lib/Pod/Man.pm
        'Pod::ParseLink'        => '1.06',  #lib/Pod/ParseLink.pm
        'Pod::Parser'           => '1.14',  #lib/Pod/Parser.pm
        'Pod::ParseUtils'       => '0.3',  #lib/Pod/ParseUtils.pm
        'Pod::Perldoc'          => '3.12',  #lib/Pod/Perldoc.pm
        'Pod::Perldoc::BaseTo'  => undef,  #lib/Pod/Perldoc/BaseTo.pm
        'Pod::Perldoc::GetOptsOO'=> undef,  #lib/Pod/Perldoc/GetOptsOO.pm
        'Pod::Perldoc::ToChecker'=> undef,  #lib/Pod/Perldoc/ToChecker.pm
        'Pod::Perldoc::ToMan'   => undef,  #lib/Pod/Perldoc/ToMan.pm
        'Pod::Perldoc::ToNroff' => undef,  #lib/Pod/Perldoc/ToNroff.pm
        'Pod::Perldoc::ToPod'   => undef,  #lib/Pod/Perldoc/ToPod.pm
        'Pod::Perldoc::ToRtf'   => undef,  #lib/Pod/Perldoc/ToRtf.pm
        'Pod::Perldoc::ToText'  => undef,  #lib/Pod/Perldoc/ToText.pm
        'Pod::Perldoc::ToTk'    => 'undef',  #lib/Pod/Perldoc/ToTk.pm
        'Pod::Perldoc::ToXml'   => undef,  #lib/Pod/Perldoc/ToXml.pm
        'Pod::Plainer'          => '0.01',  #lib/Pod/Plainer.pm
        'Pod::PlainText'        => '2.02',  #lib/Pod/PlainText.pm
        'Pod::Select'           => '1.13',  #lib/Pod/Select.pm
        'Pod::Text'             => '2.21',  #lib/Pod/Text.pm
        'Pod::Text::Color'      => '1.04',  #lib/Pod/Text/Color.pm
        'Pod::Text::Overstrike' => '1.1',  #lib/Pod/Text/Overstrike.pm
        'Pod::Text::Termcap'    => '1.11',  #lib/Pod/Text/Termcap.pm
        'Pod::Usage'            => '1.16',  #lib/Pod/Usage.pm
        'POSIX'                 => '1.07',  #lib/POSIX.pm
        're'                    => '0.04',  #lib/re.pm
        'Safe'                  => '2.10',  #lib/Safe.pm
        'Scalar::Util'          => '1.13',  #lib/Scalar/Util.pm
        'SDBM_File'             => '1.04',  #lib/SDBM_File.pm
        'Search::Dict'          => '1.02',  #lib/Search/Dict.pm
        'SelectSaver'           => '1.00',  #lib/SelectSaver.pm
        'SelfLoader'            => '1.0904',  #lib/SelfLoader.pm
        'Shell'                 => '0.5',  #lib/Shell.pm
        'sigtrap'               => '1.02',  #lib/sigtrap.pm
        'Socket'                => '1.76',  #lib/Socket.pm
        'sort'                  => '1.02',  #lib/sort.pm
        'Storable'              => '2.09',  #lib/Storable.pm
        'strict'                => '1.03',  #lib/strict.pm
        'subs'                  => '1.00',  #lib/subs.pm
        'Switch'                => '2.10',  #lib/Switch.pm
        'Symbol'                => '1.05',  #lib/Symbol.pm
        'Sys::Hostname'         => '1.11',  #lib/Sys/Hostname.pm
        'Sys::Syslog'           => '0.04',  #lib/Sys/Syslog.pm
        'Term::ANSIColor'       => '1.07',  #lib/Term/ANSIColor.pm
        'Term::Cap'             => '1.08',  #lib/Term/Cap.pm
        'Term::Complete'        => '1.401',  #lib/Term/Complete.pm
        'Term::ReadLine'        => '1.01',  #lib/Term/ReadLine.pm
        'Test'                  => '1.24',  #lib/Test.pm
        'Test::Builder'         => '0.17',  #lib/Test/Builder.pm
        'Test::Harness'         => '2.40',  #lib/Test/Harness.pm
        'Test::Harness::Assert' => '0.02',  #lib/Test/Harness/Assert.pm
        'Test::Harness::Iterator'=> '0.02',  #lib/Test/Harness/Iterator.pm
        'Test::Harness::Straps' => '0.19',  #lib/Test/Harness/Straps.pm
        'Test::More'            => '0.47',  #lib/Test/More.pm
        'Test::Simple'          => '0.47',  #lib/Test/Simple.pm
        'Text::Abbrev'          => '1.01',  #lib/Text/Abbrev.pm
        'Text::Balanced'        => '1.95',  #lib/Text/Balanced.pm
        'Text::ParseWords'      => '3.21',  #lib/Text/ParseWords.pm
        'Text::Soundex'         => '1.01',  #lib/Text/Soundex.pm
        'Text::Tabs'            => '98.112801',  #lib/Text/Tabs.pm
        'Text::Wrap'            => '2001.09291',  #lib/Text/Wrap.pm
        'Thread'                => '2.00',  #lib/Thread.pm
        'Thread::Queue'         => '2.00',  #lib/Thread/Queue.pm
        'threads'               => '1.01',  #lib/threads.pm
        'Thread::Semaphore'     => '2.01',  #lib/Thread/Semaphore.pm
        'Thread::Signal'        => '1.00', #./ext/Thread/Thread/Signal.pm
        'Thread::Specific'      => '1.00', #./ext/Thread/Thread/Specific.pm
        'threads::shared'       => '0.92',  #lib/threads/shared.pm
        'Tie::Array'            => '1.03',  #lib/Tie/Array.pm
        'Tie::File'             => '0.97',  #lib/Tie/File.pm
        'Tie::Handle'           => '4.1',  #lib/Tie/Handle.pm
        'Tie::Hash'             => '1.01',  #lib/Tie/Hash.pm
        'Tie::Memoize'          => '1.0',  #lib/Tie/Memoize.pm
        'Tie::RefHash'          => '1.31',  #lib/Tie/RefHash.pm
        'Tie::Scalar'           => '1.00',  #lib/Tie/Scalar.pm
        'Tie::SubstrHash'       => '1.00',  #lib/Tie/SubstrHash.pm
        'Time::gmtime'          => '1.02',  #lib/Time/gmtime.pm
        'Time::HiRes'           => '1.52',  #lib/Time/HiRes.pm
        'Time::Local'           => '1.07',  #lib/Time/Local.pm
        'Time::localtime'       => '1.02',  #lib/Time/localtime.pm
        'Time::tm'              => '1.00',  #lib/Time/tm.pm
        'Unicode'               => '4.0.0', # lib/unicore/version
        'Unicode::Collate'      => '0.33',  #lib/Unicode/Collate.pm
        'Unicode::Normalize'    => '0.28',  #lib/Unicode/Normalize.pm
        'Unicode::UCD'          => '0.21',  #lib/Unicode/UCD.pm
        'UNIVERSAL'             => '1.01',  #lib/UNIVERSAL.pm
        'User::grent'           => '1.00',  #lib/User/grent.pm
        'User::pwent'           => '1.00',  #lib/User/pwent.pm
        'utf8'                  => '1.02',  #lib/utf8.pm
        'vars'                  => '1.01',  #lib/vars.pm
        'VMS::DCLsym'           => '1.02',  #vms/ext/DCLsym/DCLsym.pm
        'VMS::Filespec'         => '1.11',  #vms/ext/Filespec.pm
        'VMS::Stdio'            => '2.3',  #vms/ext/Stdio/Stdio.pm
        'vmsish'                => '1.01',  #lib/vmsish.pm
        'warnings'              => '1.03',  #lib/warnings.pm
        'warnings::register'    => '1.00',  #lib/warnings/register.pm
        'XS::APItest'           => '0.03',  #lib/XS/APItest.pm
        'XSLoader'              => '0.02',  #lib/XSLoader.pm
        'XS::Typemap'           => '0.01',  #lib/XS/Typemap.pm
    },

    5.009 => {
        'AnyDBM_File'           => '1.00',  #lib/AnyDBM_File.pm
        'assertions'            => '0.01',  #lib/assertions.pm
        'assertions::activate'  => '0.01',  #lib/assertions/activate.pm
        'Attribute::Handlers'   => '0.78',  #lib/Attribute/Handlers.pm
        'attributes'            => '0.06',  #lib/attributes.pm
        'attrs'                 => '1.01',  #lib/attrs.pm
        'AutoLoader'            => '5.60',  #lib/AutoLoader.pm
        'AutoSplit'             => '1.04',  #lib/AutoSplit.pm
        'autouse'               => '1.03',  #lib/autouse.pm
        'B'                     => '1.03',  #lib/B.pm
        'B::Asmdata'            => '1.01',  #lib/B/Asmdata.pm
        'B::Assembler'          => '0.06',  #lib/B/Assembler.pm
        'B::Bblock'             => '1.02',  #lib/B/Bblock.pm
        'B::Bytecode'           => '1.01',  #lib/B/Bytecode.pm
        'B::C'                  => '1.03',  #lib/B/C.pm
        'B::CC'                 => '1.00',  #lib/B/CC.pm
        'B::Concise'            => '0.57',  #lib/B/Concise.pm
        'B::Debug'              => '1.01',  #lib/B/Debug.pm
        'B::Deparse'            => '0.65',  #lib/B/Deparse.pm
        'B::Disassembler'       => '1.03',  #lib/B/Disassembler.pm
        'B::Lint'               => '1.02',  #lib/B/Lint.pm
        'B::Showlex'            => '1.00',  #lib/B/Showlex.pm
        'B::Stackobj'           => '1.00',  #lib/B/Stackobj.pm
        'B::Stash'              => '1.00',  #lib/B/Stash.pm
        'B::Terse'              => '1.02',  #lib/B/Terse.pm
        'B::Xref'               => '1.01',  #lib/B/Xref.pm
        'base'                  => '2.03',  #lib/base.pm
        'Benchmark'             => '1.051',  #lib/Benchmark.pm
        'bigint'                => '0.04',  #lib/bigint.pm
        'bignum'                => '0.14',  #lib/bignum.pm
        'bigrat'                => '0.06',  #lib/bigrat.pm
        'blib'                  => '1.02',  #lib/blib.pm
        'ByteLoader'            => '0.05',  #lib/ByteLoader.pm
        'bytes'                 => '1.01',  #lib/bytes.pm
        'Carp'                  => '1.01',  #lib/Carp.pm
        'Carp::Heavy'           => '1.01',  #lib/Carp/Heavy.pm
        'CGI'                   => '3.00',  #lib/CGI.pm
        'CGI::Apache'           => '1.00',  #lib/CGI/Apache.pm
        'CGI::Carp'             => '1.26',  #lib/CGI/Carp.pm
        'CGI::Cookie'           => '1.24',  #lib/CGI/Cookie.pm
        'CGI::Fast'             => '1.041',  #lib/CGI/Fast.pm
        'CGI::Pretty'           => '1.07_00',  #lib/CGI/Pretty.pm
        'CGI::Push'             => '1.04',  #lib/CGI/Push.pm
        'CGI::Switch'           => '1.00',  #lib/CGI/Switch.pm
        'CGI::Util'             => '1.31',  #lib/CGI/Util.pm
        'charnames'             => '1.02',  #lib/charnames.pm
        'Class::ISA'            => '0.32',  #lib/Class/ISA.pm
        'Class::Struct'         => '0.63',  #lib/Class/Struct.pm
        'Config'                => undef,  #lib/Config.pm
        'constant'              => '1.04',  #lib/constant.pm
        'CPAN'                  => '1.76_01',  #lib/CPAN.pm
        'CPAN::FirstTime'       => '1.60 ',  #lib/CPAN/FirstTime.pm
        'CPAN::Nox'             => '1.03',  #lib/CPAN/Nox.pm
        'Cwd'                   => '2.08',  #lib/Cwd.pm
        'Data::Dumper'          => '2.121',  #lib/Data/Dumper.pm
        'DB'                    => '1.0',  #lib/DB.pm
        'DB_File'               => '1.806',  #lib/DB_File.pm
        'Devel::DProf'          => '20030813.00',  #lib/Devel/DProf.pm
        'Devel::Peek'           => '1.01',  #lib/Devel/Peek.pm
        'Devel::PPPort'         => '2.008',  #lib/Devel/PPPort.pm
        'Devel::SelfStubber'    => '1.03',  #lib/Devel/SelfStubber.pm
        'diagnostics'           => '1.11',  #lib/diagnostics.pm
        'Digest'                => '1.02',  #lib/Digest.pm
        'Digest::MD5'           => '2.30',  #lib/Digest/MD5.pm
        'DirHandle'             => '1.00',  #lib/DirHandle.pm
        'Dumpvalue'             => '1.11',  #lib/Dumpvalue.pm
        'DynaLoader'            => '1.04',  #lib/DynaLoader.pm
        'Encode'                => '1.9801',  #lib/Encode.pm
        'Encode::Alias'         => '1.38',  #lib/Encode/Alias.pm
        'Encode::Byte'          => '1.23',  #lib/Encode/Byte.pm
        'Encode::CJKConstants'  => '1.02',  #lib/Encode/CJKConstants.pm
        'Encode::CN'            => '1.24',  #lib/Encode/CN.pm
        'Encode::CN::HZ'        => '1.05',  #lib/Encode/CN/HZ.pm
        'Encode::Config'        => '1.07',  #lib/Encode/Config.pm
        'Encode::EBCDIC'        => '1.21',  #lib/Encode/EBCDIC.pm
        'Encode::Encoder'       => '0.07',  #lib/Encode/Encoder.pm
        'Encode::Encoding'      => '1.33',  #lib/Encode/Encoding.pm
        'Encode::Guess'         => '1.09',  #lib/Encode/Guess.pm
        'Encode::JP'            => '1.25',  #lib/Encode/JP.pm
        'Encode::JP::H2Z'       => '1.02',  #lib/Encode/JP/H2Z.pm
        'Encode::JP::JIS7'      => '1.12',  #lib/Encode/JP/JIS7.pm
        'Encode::KR'            => '1.23',  #lib/Encode/KR.pm
        'Encode::KR::2022_KR'   => '1.06',  #lib/Encode/KR/2022_KR.pm
        'Encode::MIME::Header'  => '1.09',  #lib/Encode/MIME/Header.pm
        'Encode::Symbol'        => '1.22',  #lib/Encode/Symbol.pm
        'Encode::TW'            => '1.26',  #lib/Encode/TW.pm
        'Encode::Unicode'       => '1.40',  #lib/Encode/Unicode.pm
        'Encode::Unicode::UTF7' => '0.02',  #lib/Encode/Unicode/UTF7.pm
        'encoding'              => '1.47',  #lib/encoding.pm
        'English'               => '1.02',  #lib/English.pm
        'Env'                   => '1.00',  #lib/Env.pm
        'Errno'                 => '1.09_00',  #lib/Errno.pm
        'Exporter'              => '5.567',  #lib/Exporter.pm
        'Exporter::Heavy'       => '5.567',  #lib/Exporter/Heavy.pm
        'ExtUtils::Command'     => '1.05',  #lib/ExtUtils/Command.pm
        'ExtUtils::Command::MM' => '0.03',  #lib/ExtUtils/Command/MM.pm
        'ExtUtils::Constant'    => '0.14',  #lib/ExtUtils/Constant.pm
        'ExtUtils::Embed'       => '1.250601',  #lib/ExtUtils/Embed.pm
        'ExtUtils::Install'     => '1.32',  #lib/ExtUtils/Install.pm
        'ExtUtils::Installed'   => '0.08',  #lib/ExtUtils/Installed.pm
        'ExtUtils::Liblist'     => '1.01',  #lib/ExtUtils/Liblist.pm
        'ExtUtils::Liblist::Kid'=> '1.3',  #lib/ExtUtils/Liblist/Kid.pm
        'ExtUtils::MakeMaker'   => '6.17',  #lib/ExtUtils/MakeMaker.pm
        'ExtUtils::MakeMaker::bytes'=> '0.01',  #lib/ExtUtils/MakeMaker/bytes.pm
        'ExtUtils::MakeMaker::vmsish'=> '0.01',  #lib/ExtUtils/MakeMaker/vmsish.pm
        'ExtUtils::Manifest'    => '1.42',  #lib/ExtUtils/Manifest.pm
        'ExtUtils::Miniperl'    => undef,  #lib/ExtUtils/Miniperl.pm
        'ExtUtils::Mkbootstrap' => '1.15',  #lib/ExtUtils/Mkbootstrap.pm
        'ExtUtils::Mksymlists'  => '1.19',  #lib/ExtUtils/Mksymlists.pm
        'ExtUtils::MM'          => '0.04',  #lib/ExtUtils/MM.pm
        'ExtUtils::MM_Any'      => '0.07',  #lib/ExtUtils/MM_Any.pm
        'ExtUtils::MM_BeOS'     => '1.04',  #lib/ExtUtils/MM_BeOS.pm
        'ExtUtils::MM_Cygwin'   => '1.06',  #lib/ExtUtils/MM_Cygwin.pm
        'ExtUtils::MM_DOS'      => '0.02',  #lib/ExtUtils/MM_DOS.pm
        'ExtUtils::MM_MacOS'    => '1.07',  #lib/ExtUtils/MM_MacOS.pm
        'ExtUtils::MM_NW5'      => '2.06',  #lib/ExtUtils/MM_NW5.pm
        'ExtUtils::MM_OS2'      => '1.04',  #lib/ExtUtils/MM_OS2.pm
        'ExtUtils::MM_Unix'     => '1.42',  #lib/ExtUtils/MM_Unix.pm
        'ExtUtils::MM_UWIN'     => '0.02',  #lib/ExtUtils/MM_UWIN.pm
        'ExtUtils::MM_VMS'      => '5.70',  #lib/ExtUtils/MM_VMS.pm
        'ExtUtils::MM_Win32'    => '1.09',  #lib/ExtUtils/MM_Win32.pm
        'ExtUtils::MM_Win95'    => '0.03',  #lib/ExtUtils/MM_Win95.pm
        'ExtUtils::MY'          => '0.01',  #lib/ExtUtils/MY.pm
        'ExtUtils::Packlist'    => '0.04',  #lib/ExtUtils/Packlist.pm
        'ExtUtils::testlib'     => '1.15',  #lib/ExtUtils/testlib.pm
        'ExtUtils::XSSymSet'    => '1.0',  #vms/ext/XSSymSet.pm
        'Fatal'                 => '1.04',  #lib/Fatal.pm
        'Fcntl'                 => '1.05',  #lib/Fcntl.pm
        'fields'                => '2.03',  #lib/fields.pm
        'File::Basename'        => '2.72',  #lib/File/Basename.pm
        'File::CheckTree'       => '4.2',  #lib/File/CheckTree.pm
        'File::Compare'         => '1.1003',  #lib/File/Compare.pm
        'File::Copy'            => '2.06',  #lib/File/Copy.pm
        'File::DosGlob'         => '1.00',  #lib/File/DosGlob.pm
        'File::Find'            => '1.05',  #lib/File/Find.pm
        'File::Glob'            => '1.02',  #lib/File/Glob.pm
        'File::Path'            => '1.06',  #lib/File/Path.pm
        'File::Spec'            => '0.86',  #lib/File/Spec.pm
        'File::Spec::Cygwin'    => '1.1',  #lib/File/Spec/Cygwin.pm
        'File::Spec::Epoc'      => '1.1',  #lib/File/Spec/Epoc.pm
        'File::Spec::Functions' => '1.3',  #lib/File/Spec/Functions.pm
        'File::Spec::Mac'       => '1.4',  #lib/File/Spec/Mac.pm
        'File::Spec::OS2'       => '1.2',  #lib/File/Spec/OS2.pm
        'File::Spec::Unix'      => '1.5',  #lib/File/Spec/Unix.pm
        'File::Spec::VMS'       => '1.4',  #lib/File/Spec/VMS.pm
        'File::Spec::Win32'     => '1.4',  #lib/File/Spec/Win32.pm
        'File::stat'            => '1.00',  #lib/File/stat.pm
        'File::Temp'            => '0.14',  #lib/File/Temp.pm
        'FileCache'             => '1.03',  #lib/FileCache.pm
        'FileHandle'            => '2.01',  #lib/FileHandle.pm
        'filetest'              => '1.01',  #lib/filetest.pm
        'Filter::Simple'        => '0.78',  #lib/Filter/Simple.pm
        'Filter::Util::Call'    => '1.0601',  #lib/Filter/Util/Call.pm
        'FindBin'               => '1.43',  #lib/FindBin.pm
        'GDBM_File'             => '1.07',  #ext/GDBM_File/GDBM_File.pm
        'Getopt::Long'          => '2.34',  #lib/Getopt/Long.pm
        'Getopt::Std'           => '1.04',  #lib/Getopt/Std.pm
        'Hash::Util'            => '0.05',  #lib/Hash/Util.pm
        'I18N::Collate'         => '1.00',  #lib/I18N/Collate.pm
        'I18N::Langinfo'        => '0.02',  #lib/I18N/Langinfo.pm
        'I18N::LangTags'        => '0.29',  #lib/I18N/LangTags.pm
        'I18N::LangTags::List'  => '0.29',  #lib/I18N/LangTags/List.pm
        'if'                    => '0.03',  #lib/if.pm
        'integer'               => '1.00',  #lib/integer.pm
        'IO'                    => '1.21',  #lib/IO.pm
        'IO::Dir'               => '1.04',  #lib/IO/Dir.pm
        'IO::File'              => '1.10',  #lib/IO/File.pm
        'IO::Handle'            => '1.23',  #lib/IO/Handle.pm
        'IO::Pipe'              => '1.122',  #lib/IO/Pipe.pm
        'IO::Poll'              => '0.06',  #lib/IO/Poll.pm
        'IO::Seekable'          => '1.09',  #lib/IO/Seekable.pm
        'IO::Select'            => '1.16',  #lib/IO/Select.pm
        'IO::Socket'            => '1.28',  #lib/IO/Socket.pm
        'IO::Socket::INET'      => '1.27',  #lib/IO/Socket/INET.pm
        'IO::Socket::UNIX'      => '1.21',  #lib/IO/Socket/UNIX.pm
        'IPC::Msg'              => '1.02',  #lib/IPC/Msg.pm
        'IPC::Open2'            => '1.01',  #lib/IPC/Open2.pm
        'IPC::Open3'            => '1.0105',  #lib/IPC/Open3.pm
        'IPC::Semaphore'        => '1.02',  #lib/IPC/Semaphore.pm
        'IPC::SysV'             => '1.04',  #lib/IPC/SysV.pm
        'JNI'                   => '0.2',  #jpl/JNI/JNI.pm
        'JPL::AutoLoader'       => undef,  #jpl/JPL/AutoLoader.pm
        'JPL::Class'            => undef,  #jpl/JPL/Class.pm
        'JPL::Compile'          => undef,  #jpl/JPL/Compile.pm
        'less'                  => '0.01',  #lib/less.pm
        'lib'                   => '0.5565',  #lib/lib.pm
        'List::Util'            => '1.13',  #lib/List/Util.pm
        'locale'                => '1.00',  #lib/locale.pm
        'Locale::Constants'     => '2.01',  #lib/Locale/Constants.pm
        'Locale::Country'       => '2.61',  #lib/Locale/Country.pm
        'Locale::Currency'      => '2.21',  #lib/Locale/Currency.pm
        'Locale::Language'      => '2.21',  #lib/Locale/Language.pm
        'Locale::Maketext'      => '1.06',  #lib/Locale/Maketext.pm
        'Locale::Maketext::Guts'=> undef,  #lib/Locale/Maketext/Guts.pm
        'Locale::Maketext::GutsLoader'=> undef,  #lib/Locale/Maketext/GutsLoader.pm
        'Locale::Script'        => '2.21',  #lib/Locale/Script.pm
        'Math::BigFloat'        => '1.40',  #lib/Math/BigFloat.pm
        'Math::BigFloat::Trace' => '0.01',  #lib/Math/BigFloat/Trace.pm
        'Math::BigInt'          => '1.66',  #lib/Math/BigInt.pm
        'Math::BigInt::Calc'    => '0.36',  #lib/Math/BigInt/Calc.pm
        'Math::BigInt::Scalar'  => '0.11',  #lib/Math/BigInt/Scalar.pm
        'Math::BigInt::Trace'   => '0.01',  #lib/Math/BigInt/Trace.pm
        'Math::BigRat'          => '0.10',  #lib/Math/BigRat.pm
        'Math::Complex'         => '1.34',  #lib/Math/Complex.pm
        'Math::Trig'            => '1.02',  #lib/Math/Trig.pm
        'Memoize'               => '1.01',  #lib/Memoize.pm
        'Memoize::AnyDBM_File'  => '0.65',  #lib/Memoize/AnyDBM_File.pm
        'Memoize::Expire'       => '1.00',  #lib/Memoize/Expire.pm
        'Memoize::ExpireFile'   => '1.01',  #lib/Memoize/ExpireFile.pm
        'Memoize::ExpireTest'   => '0.65',  #lib/Memoize/ExpireTest.pm
        'Memoize::NDBM_File'    => '0.65',  #lib/Memoize/NDBM_File.pm
        'Memoize::SDBM_File'    => '0.65',  #lib/Memoize/SDBM_File.pm
        'Memoize::Storable'     => '0.65',  #lib/Memoize/Storable.pm
        'MIME::Base64'          => '2.21',  #lib/MIME/Base64.pm
        'MIME::QuotedPrint'     => '2.21',  #lib/MIME/QuotedPrint.pm
        'NDBM_File'             => '1.05',  #ext/NDBM_File/NDBM_File.pm
        'Net::Cmd'              => '2.24',  #lib/Net/Cmd.pm
        'Net::Config'           => '1.10',  #lib/Net/Config.pm
        'Net::Domain'           => '2.19',  #lib/Net/Domain.pm
        'Net::FTP'              => '2.72',  #lib/Net/FTP.pm
        'Net::FTP::A'           => '1.16',  #lib/Net/FTP/A.pm
        'Net::FTP::dataconn'    => '0.11',  #lib/Net/FTP/dataconn.pm
        'Net::FTP::E'           => '0.01',  #lib/Net/FTP/E.pm
        'Net::FTP::I'           => '1.12',  #lib/Net/FTP/I.pm
        'Net::FTP::L'           => '0.01',  #lib/Net/FTP/L.pm
        'Net::hostent'          => '1.01',  #lib/Net/hostent.pm
        'Net::netent'           => '1.00',  #lib/Net/netent.pm
        'Net::Netrc'            => '2.12',  #lib/Net/Netrc.pm
        'Net::NNTP'             => '2.22',  #lib/Net/NNTP.pm
        'Net::Ping'             => '2.31',  #lib/Net/Ping.pm
        'Net::POP3'             => '2.24',  #lib/Net/POP3.pm
        'Net::protoent'         => '1.00',  #lib/Net/protoent.pm
        'Net::servent'          => '1.01',  #lib/Net/servent.pm
        'Net::SMTP'             => '2.26',  #lib/Net/SMTP.pm
        'Net::Time'             => '2.09',  #lib/Net/Time.pm
        'NEXT'                  => '0.60',  #lib/NEXT.pm
        'O'                     => '1.00',  #lib/O.pm
        'ODBM_File'             => '1.04',  #ext/ODBM_File/ODBM_File.pm
        'Opcode'                => '1.06',  #lib/Opcode.pm
        'open'                  => '1.02',  #lib/open.pm
        'ops'                   => '1.00',  #lib/ops.pm
        'OS2::DLL'              => '1.02',  #os2/OS2/REXX/DLL/DLL.pm
        'OS2::ExtAttr'          => '0.02',  #os2/OS2/ExtAttr/ExtAttr.pm
        'OS2::PrfDB'            => '0.03',  #os2/OS2/PrfDB/PrfDB.pm
        'OS2::Process'          => '1.01',  #os2/OS2/Process/Process.pm
        'OS2::REXX'             => '1.02',  #os2/OS2/REXX/REXX.pm
        'overload'              => '1.02',  #lib/overload.pm
        'PerlIO'                => '1.02',  #lib/PerlIO.pm
        'PerlIO::encoding'      => '0.07',  #lib/PerlIO/encoding.pm
        'PerlIO::scalar'        => '0.02',  #lib/PerlIO/scalar.pm
        'PerlIO::via'           => '0.02',  #lib/PerlIO/via.pm
        'PerlIO::via::QuotedPrint'=> '0.05',  #lib/PerlIO/via/QuotedPrint.pm
        'Pod::Checker'          => '1.41',  #lib/Pod/Checker.pm
        'Pod::Find'             => '0.24',  #lib/Pod/Find.pm
        'Pod::Functions'        => '1.02',  #lib/Pod/Functions.pm
        'Pod::Html'             => '1.0501',  #lib/Pod/Html.pm
        'Pod::InputObjects'     => '1.14',  #lib/Pod/InputObjects.pm
        'Pod::LaTeX'            => '0.55',  #lib/Pod/LaTeX.pm
        'Pod::Man'              => '1.37',  #lib/Pod/Man.pm
        'Pod::ParseLink'        => '1.06',  #lib/Pod/ParseLink.pm
        'Pod::Parser'           => '1.13',  #lib/Pod/Parser.pm
        'Pod::ParseUtils'       => '0.3',  #lib/Pod/ParseUtils.pm
        'Pod::Perldoc'          => '3.11',  #lib/Pod/Perldoc.pm
        'Pod::Perldoc::BaseTo'  => undef,  #lib/Pod/Perldoc/BaseTo.pm
        'Pod::Perldoc::GetOptsOO'=> undef,  #lib/Pod/Perldoc/GetOptsOO.pm
        'Pod::Perldoc::ToChecker'=> undef,  #lib/Pod/Perldoc/ToChecker.pm
        'Pod::Perldoc::ToMan'   => undef,  #lib/Pod/Perldoc/ToMan.pm
        'Pod::Perldoc::ToNroff' => undef,  #lib/Pod/Perldoc/ToNroff.pm
        'Pod::Perldoc::ToPod'   => undef,  #lib/Pod/Perldoc/ToPod.pm
        'Pod::Perldoc::ToRtf'   => undef,  #lib/Pod/Perldoc/ToRtf.pm
        'Pod::Perldoc::ToText'  => undef,  #lib/Pod/Perldoc/ToText.pm
        'Pod::Perldoc::ToTk'    => 'undef',  #lib/Pod/Perldoc/ToTk.pm
        'Pod::Perldoc::ToXml'   => undef,  #lib/Pod/Perldoc/ToXml.pm
        'Pod::Plainer'          => '0.01',  #lib/Pod/Plainer.pm
        'Pod::PlainText'        => '2.01',  #lib/Pod/PlainText.pm
        'Pod::Select'           => '1.13',  #lib/Pod/Select.pm
        'Pod::Text'             => '2.21',  #lib/Pod/Text.pm
        'Pod::Text::Color'      => '1.04',  #lib/Pod/Text/Color.pm
        'Pod::Text::Overstrike' => '1.1',  #lib/Pod/Text/Overstrike.pm
        'Pod::Text::Termcap'    => '1.11',  #lib/Pod/Text/Termcap.pm
        'Pod::Usage'            => '1.16',  #lib/Pod/Usage.pm
        'POSIX'                 => '1.06',  #lib/POSIX.pm
        're'                    => '0.04',  #lib/re.pm
        'Safe'                  => '2.10',  #lib/Safe.pm
        'Scalar::Util'          => '1.13',  #lib/Scalar/Util.pm
        'SDBM_File'             => '1.04',  #lib/SDBM_File.pm
        'Search::Dict'          => '1.02',  #lib/Search/Dict.pm
        'SelectSaver'           => '1.00',  #lib/SelectSaver.pm
        'SelfLoader'            => '1.0904',  #lib/SelfLoader.pm
        'Shell'                 => '0.5',  #lib/Shell.pm
        'sigtrap'               => '1.02',  #lib/sigtrap.pm
        'Socket'                => '1.76',  #lib/Socket.pm
        'sort'                  => '1.02',  #lib/sort.pm
        'Storable'              => '2.08',  #lib/Storable.pm
        'strict'                => '1.03',  #lib/strict.pm
        'subs'                  => '1.00',  #lib/subs.pm
        'Switch'                => '2.10',  #lib/Switch.pm
        'Symbol'                => '1.05',  #lib/Symbol.pm
        'Sys::Hostname'         => '1.11',  #lib/Sys/Hostname.pm
        'Sys::Syslog'           => '0.04',  #lib/Sys/Syslog.pm
        'Term::ANSIColor'       => '1.07',  #lib/Term/ANSIColor.pm
        'Term::Cap'             => '1.08',  #lib/Term/Cap.pm
        'Term::Complete'        => '1.401',  #lib/Term/Complete.pm
        'Term::ReadLine'        => '1.01',  #lib/Term/ReadLine.pm
        'Test'                  => '1.24',  #lib/Test.pm
        'Test::Builder'         => '0.17',  #lib/Test/Builder.pm
        'Test::Harness'         => '2.30',  #lib/Test/Harness.pm
        'Test::Harness::Assert' => '0.01',  #lib/Test/Harness/Assert.pm
        'Test::Harness::Iterator'=> '0.01',  #lib/Test/Harness/Iterator.pm
        'Test::Harness::Straps' => '0.15',  #lib/Test/Harness/Straps.pm
        'Test::More'            => '0.47',  #lib/Test/More.pm
        'Test::Simple'          => '0.47',  #lib/Test/Simple.pm
        'Text::Abbrev'          => '1.01',  #lib/Text/Abbrev.pm
        'Text::Balanced'        => '1.95',  #lib/Text/Balanced.pm
        'Text::ParseWords'      => '3.21',  #lib/Text/ParseWords.pm
        'Text::Soundex'         => '1.01',  #lib/Text/Soundex.pm
        'Text::Tabs'            => '98.112801',  #lib/Text/Tabs.pm
        'Text::Wrap'            => '2001.09291',  #lib/Text/Wrap.pm
        'Thread'                => '2.00',  #lib/Thread.pm
        'Thread::Queue'         => '2.00',  #lib/Thread/Queue.pm
        'Thread::Semaphore'     => '2.01',  #lib/Thread/Semaphore.pm
        'Thread::Signal'        => '1.00', #./ext/Thread/Thread/Signal.pm
        'Thread::Specific'      => '1.00', #./ext/Thread/Thread/Specific.pm
        'threads'               => '1.00',  #lib/threads.pm
        'threads::shared'       => '0.91',  #lib/threads/shared.pm
        'Tie::Array'            => '1.03',  #lib/Tie/Array.pm
        'Tie::File'             => '0.97',  #lib/Tie/File.pm
        'Tie::Handle'           => '4.1',  #lib/Tie/Handle.pm
        'Tie::Hash'             => '1.00',  #lib/Tie/Hash.pm
        'Tie::Memoize'          => '1.0',  #lib/Tie/Memoize.pm
        'Tie::RefHash'          => '1.31',  #lib/Tie/RefHash.pm
        'Tie::Scalar'           => '1.00',  #lib/Tie/Scalar.pm
        'Tie::SubstrHash'       => '1.00',  #lib/Tie/SubstrHash.pm
        'Time::gmtime'          => '1.02',  #lib/Time/gmtime.pm
        'Time::HiRes'           => '1.51',  #lib/Time/HiRes.pm
        'Time::Local'           => '1.07',  #lib/Time/Local.pm
        'Time::localtime'       => '1.02',  #lib/Time/localtime.pm
        'Time::tm'              => '1.00',  #lib/Time/tm.pm
        'Unicode'               => '4.0.0', #lib/unicore/version
        'Unicode::Collate'      => '0.28',  #lib/Unicode/Collate.pm
        'Unicode::Normalize'    => '0.23',  #lib/Unicode/Normalize.pm
        'Unicode::UCD'          => '0.21',  #lib/Unicode/UCD.pm
        'UNIVERSAL'             => '1.01',  #lib/UNIVERSAL.pm
        'User::grent'           => '1.00',  #lib/User/grent.pm
        'User::pwent'           => '1.00',  #lib/User/pwent.pm
        'utf8'                  => '1.02',  #lib/utf8.pm
        'vars'                  => '1.01',  #lib/vars.pm
        'version'               => '0.29',  #lib/version.pm
        'VMS::DCLsym'           => '1.02',  #vms/ext/DCLsym/DCLsym.pm
        'VMS::Filespec'         => '1.11',  #vms/ext/Filespec.pm
        'VMS::Stdio'            => '2.3',  #vms/ext/Stdio/Stdio.pm
        'vmsish'                => '1.01',  #lib/vmsish.pm
        'warnings'              => '1.03',  #lib/warnings.pm
        'warnings::register'    => '1.00',  #lib/warnings/register.pm
        'XS::APItest'           => '0.02',  #lib/XS/APItest.pm
        'XS::Typemap'           => '0.01',  #lib/XS/Typemap.pm
        'XSLoader'              => '0.03',  #lib/XSLoader.pm
    },

    5.009001 => {
        'AnyDBM_File'           => '1.00',  #lib/AnyDBM_File.pm
        'assertions'            => '0.01',  #lib/assertions.pm
        'assertions::activate'  => '0.01',  #lib/assertions/activate.pm
        'Attribute::Handlers'   => '0.78_01',  #lib/Attribute/Handlers.pm
        'attributes'            => '0.06',  #lib/attributes.pm
        'attrs'                 => '1.01',  #lib/attrs.pm
        'AutoLoader'            => '5.60',  #lib/AutoLoader.pm
        'AutoSplit'             => '1.04',  #lib/AutoSplit.pm
        'autouse'               => '1.03',  #lib/autouse.pm
        'B'                     => '1.05',  #lib/B.pm
        'base'                  => '2.04',  #lib/base.pm
        'B::Asmdata'            => '1.01',  #lib/B/Asmdata.pm
        'B::Assembler'          => '0.06',  #lib/B/Assembler.pm
        'B::Bblock'             => '1.02',  #lib/B/Bblock.pm
        'B::Bytecode'           => '1.01',  #lib/B/Bytecode.pm
        'B::C'                  => '1.04',  #lib/B/C.pm
        'B::CC'                 => '1.00',  #lib/B/CC.pm
        'B::Concise'            => '0.59',  #lib/B/Concise.pm
        'B::Debug'              => '1.02',  #lib/B/Debug.pm
        'B::Deparse'            => '0.65',  #lib/B/Deparse.pm
        'B::Disassembler'       => '1.03',  #lib/B/Disassembler.pm
        'Benchmark'             => '1.06',  #lib/Benchmark.pm
        'bigint'                => '0.05',  #lib/bigint.pm
        'bignum'                => '0.15',  #lib/bignum.pm
        'bigrat'                => '0.06',  #lib/bigrat.pm
        'blib'                  => '1.02',  #lib/blib.pm
        'B::Lint'               => '1.02',  #lib/B/Lint.pm
        'B::Showlex'            => '1.00',  #lib/B/Showlex.pm
        'B::Stackobj'           => '1.00',  #lib/B/Stackobj.pm
        'B::Stash'              => '1.00',  #lib/B/Stash.pm
        'B::Terse'              => '1.02',  #lib/B/Terse.pm
        'B::Xref'               => '1.01',  #lib/B/Xref.pm
        'ByteLoader'            => '0.05',  #lib/ByteLoader.pm
        'bytes'                 => '1.01',  #lib/bytes.pm
        'Carp'                  => '1.02',  #lib/Carp.pm
        'Carp::Heavy'           => '1.01',  #lib/Carp/Heavy.pm
        'CGI'                   => '3.04',  #lib/CGI.pm
        'CGI::Apache'           => '1.00',  #lib/CGI/Apache.pm
        'CGI::Carp'             => '1.27',  #lib/CGI/Carp.pm
        'CGI::Cookie'           => '1.24',  #lib/CGI/Cookie.pm
        'CGI::Fast'             => '1.05',  #lib/CGI/Fast.pm
        'CGI::Pretty'           => '1.08',  #lib/CGI/Pretty.pm
        'CGI::Push'             => '1.04',  #lib/CGI/Push.pm
        'CGI::Switch'           => '1.00',  #lib/CGI/Switch.pm
        'CGI::Util'             => '1.4',  #lib/CGI/Util.pm
        'charnames'             => '1.03',  #lib/charnames.pm
        'Class::ISA'            => '0.32',  #lib/Class/ISA.pm
        'Class::Struct'         => '0.63',  #lib/Class/Struct.pm
        'Config'                => undef,  #lib/Config.pm
        'constant'              => '1.04',  #lib/constant.pm
        'CPAN'                  => '1.76_01',  #lib/CPAN.pm
        'CPAN::FirstTime'       => '1.60 ',  #lib/CPAN/FirstTime.pm
        'CPAN::Nox'             => '1.03',  #lib/CPAN/Nox.pm
        'Cwd'                   => '2.17',  #lib/Cwd.pm
        'Data::Dumper'          => '2.121',  #lib/Data/Dumper.pm
        'DB'                    => '1.0',  #lib/DB.pm
        'DB_File'               => '1.808_01',  #lib/DB_File.pm
        'DBM_Filter'            => '0.01',  #lib/DBM_Filter.pm
        'DBM_Filter::compress'  => '0.01',  #lib/DBM_Filter/compress.pm
        'DBM_Filter::encode'    => '0.01',  #lib/DBM_Filter/encode.pm
        'DBM_Filter::int32'     => '0.01',  #lib/DBM_Filter/int32.pm
        'DBM_Filter::null'      => '0.01',  #lib/DBM_Filter/null.pm
        'DBM_Filter::utf8'      => '0.01',  #lib/DBM_Filter/utf8.pm
        'Devel::DProf'          => '20030813.00',  #lib/Devel/DProf.pm
        'Devel::Peek'           => '1.01',  #lib/Devel/Peek.pm
        'Devel::PPPort'         => '2.011_01',  #lib/Devel/PPPort.pm
        'Devel::SelfStubber'    => '1.03',  #lib/Devel/SelfStubber.pm
        'diagnostics'           => '1.12',  #lib/diagnostics.pm
        'Digest'                => '1.05',  #lib/Digest.pm
        'Digest::base'          => '1.00',  #lib/Digest/base.pm
        'Digest::MD5'           => '2.33',  #lib/Digest/MD5.pm
        'DirHandle'             => '1.00',  #lib/DirHandle.pm
        'Dumpvalue'             => '1.11',  #lib/Dumpvalue.pm
        'DynaLoader'            => '1.04',  #lib/DynaLoader.pm
        'Encode'                => '1.99_01',  #lib/Encode.pm
        'Encode::Alias'         => '1.38',  #lib/Encode/Alias.pm
        'Encode::Byte'          => '1.23',  #lib/Encode/Byte.pm
        'Encode::CJKConstants'  => '1.02',  #lib/Encode/CJKConstants.pm
        'Encode::CN'            => '1.24',  #lib/Encode/CN.pm
        'Encode::CN::HZ'        => '1.0501',  #lib/Encode/CN/HZ.pm
        'Encode::Config'        => '1.07',  #lib/Encode/Config.pm
        'Encode::EBCDIC'        => '1.21',  #lib/Encode/EBCDIC.pm
        'Encode::Encoder'       => '0.07',  #lib/Encode/Encoder.pm
        'Encode::Encoding'      => '1.33',  #lib/Encode/Encoding.pm
        'Encode::Guess'         => '1.09',  #lib/Encode/Guess.pm
        'Encode::JP'            => '1.25',  #lib/Encode/JP.pm
        'Encode::JP::H2Z'       => '1.02',  #lib/Encode/JP/H2Z.pm
        'Encode::JP::JIS7'      => '1.12',  #lib/Encode/JP/JIS7.pm
        'Encode::KR'            => '1.23',  #lib/Encode/KR.pm
        'Encode::KR::2022_KR'   => '1.06',  #lib/Encode/KR/2022_KR.pm
        'Encode::MIME::Header'  => '1.09',  #lib/Encode/MIME/Header.pm
        'Encode::Symbol'        => '1.22',  #lib/Encode/Symbol.pm
        'Encode::TW'            => '1.26',  #lib/Encode/TW.pm
        'Encode::Unicode'       => '1.40',  #lib/Encode/Unicode.pm
        'Encode::Unicode::UTF7' => '0.02',  #lib/Encode/Unicode/UTF7.pm
        'encoding'              => '1.48',  #lib/encoding.pm
        'English'               => '1.02',  #lib/English.pm
        'Env'                   => '1.00',  #lib/Env.pm
        'Errno'                 => '1.09_00',  #lib/Errno.pm
        'Exporter'              => '5.58',  #lib/Exporter.pm
        'Exporter::Heavy'       => '5.567',  #lib/Exporter/Heavy.pm
        'ExtUtils::Command'     => '1.07',  #lib/ExtUtils/Command.pm
        'ExtUtils::Command::MM' => '0.03',  #lib/ExtUtils/Command/MM.pm
        'ExtUtils::Constant'    => '0.14',  #lib/ExtUtils/Constant.pm
        'ExtUtils::Embed'       => '1.250601',  #lib/ExtUtils/Embed.pm
        'ExtUtils::Install'     => '1.32',  #lib/ExtUtils/Install.pm
        'ExtUtils::Installed'   => '0.08',  #lib/ExtUtils/Installed.pm
        'ExtUtils::Liblist'     => '1.01',  #lib/ExtUtils/Liblist.pm
        'ExtUtils::Liblist::Kid'=> '1.3',  #lib/ExtUtils/Liblist/Kid.pm
        'ExtUtils::MakeMaker'   => '6.21_02',  #lib/ExtUtils/MakeMaker.pm
        'ExtUtils::MakeMaker::bytes'=> '0.01',  #lib/ExtUtils/MakeMaker/bytes.pm
        'ExtUtils::MakeMaker::vmsish'=> '0.01',  #lib/ExtUtils/MakeMaker/vmsish.pm
        'ExtUtils::Manifest'    => '1.43',  #lib/ExtUtils/Manifest.pm
        'ExtUtils::Miniperl'    => undef,  #lib/ExtUtils/Miniperl.pm
        'ExtUtils::Mkbootstrap' => '1.15',  #lib/ExtUtils/Mkbootstrap.pm
        'ExtUtils::Mksymlists'  => '1.19',  #lib/ExtUtils/Mksymlists.pm
        'ExtUtils::MM'          => '0.04',  #lib/ExtUtils/MM.pm
        'ExtUtils::MM_Any'      => '0.0901',  #lib/ExtUtils/MM_Any.pm
        'ExtUtils::MM_BeOS'     => '1.04',  #lib/ExtUtils/MM_BeOS.pm
        'ExtUtils::MM_Cygwin'   => '1.07',  #lib/ExtUtils/MM_Cygwin.pm
        'ExtUtils::MM_DOS'      => '0.02',  #lib/ExtUtils/MM_DOS.pm
        'ExtUtils::MM_MacOS'    => '1.07',  #lib/ExtUtils/MM_MacOS.pm
        'ExtUtils::MM_NW5'      => '2.07_01',  #lib/ExtUtils/MM_NW5.pm
        'ExtUtils::MM_OS2'      => '1.04',  #lib/ExtUtils/MM_OS2.pm
        'ExtUtils::MM_Unix'     => '1.45_01',  #lib/ExtUtils/MM_Unix.pm
        'ExtUtils::MM_UWIN'     => '0.02',  #lib/ExtUtils/MM_UWIN.pm
        'ExtUtils::MM_VMS'      => '5.71_01',  #lib/ExtUtils/MM_VMS.pm
        'ExtUtils::MM_Win32'    => '1.10_01',  #lib/ExtUtils/MM_Win32.pm
        'ExtUtils::MM_Win95'    => '0.03',  #lib/ExtUtils/MM_Win95.pm
        'ExtUtils::MY'          => '0.01',  #lib/ExtUtils/MY.pm
        'ExtUtils::Packlist'    => '0.04',  #lib/ExtUtils/Packlist.pm
        'ExtUtils::testlib'     => '1.15',  #lib/ExtUtils/testlib.pm
        'ExtUtils::XSSymSet'    => '1.0',  #vms/ext/XSSymSet.pm
        'Fatal'                 => '1.04',  #lib/Fatal.pm
        'Fcntl'                 => '1.05',  #lib/Fcntl.pm
        'fields'                => '2.03',  #lib/fields.pm
        'File::Basename'        => '2.72',  #lib/File/Basename.pm
        'FileCache'             => '1.03',  #lib/FileCache.pm
        'File::CheckTree'       => '4.3',  #lib/File/CheckTree.pm
        'File::Compare'         => '1.1003',  #lib/File/Compare.pm
        'File::Copy'            => '2.07',  #lib/File/Copy.pm
        'File::DosGlob'         => '1.00',  #lib/File/DosGlob.pm
        'File::Find'            => '1.07',  #lib/File/Find.pm
        'File::Glob'            => '1.02',  #lib/File/Glob.pm
        'FileHandle'            => '2.01',  #lib/FileHandle.pm
        'File::Path'            => '1.06',  #lib/File/Path.pm
        'File::Spec'            => '0.87',  #lib/File/Spec.pm
        'File::Spec::Cygwin'    => '1.1',  #lib/File/Spec/Cygwin.pm
        'File::Spec::Epoc'      => '1.1',  #lib/File/Spec/Epoc.pm
        'File::Spec::Functions' => '1.3',  #lib/File/Spec/Functions.pm
        'File::Spec::Mac'       => '1.4',  #lib/File/Spec/Mac.pm
        'File::Spec::OS2'       => '1.2',  #lib/File/Spec/OS2.pm
        'File::Spec::Unix'      => '1.5',  #lib/File/Spec/Unix.pm
        'File::Spec::VMS'       => '1.4',  #lib/File/Spec/VMS.pm
        'File::Spec::Win32'     => '1.4',  #lib/File/Spec/Win32.pm
        'File::stat'            => '1.00',  #lib/File/stat.pm
        'File::Temp'            => '0.14',  #lib/File/Temp.pm
        'filetest'              => '1.01',  #lib/filetest.pm
        'Filter::Simple'        => '0.78',  #lib/Filter/Simple.pm
        'Filter::Util::Call'    => '1.0601',  #lib/Filter/Util/Call.pm
        'FindBin'               => '1.44',  #lib/FindBin.pm
        'GDBM_File'             => '1.07',  #lib/GDBM_File.pm
        'Getopt::Long'          => '2.3401',  #lib/Getopt/Long.pm
        'Getopt::Std'           => '1.05',  #lib/Getopt/Std.pm
        'Hash::Util'            => '0.05',  #lib/Hash/Util.pm
        'I18N::Collate'         => '1.00',  #lib/I18N/Collate.pm
        'I18N::Langinfo'        => '0.02',  #lib/I18N/Langinfo.pm
        'I18N::LangTags'        => '0.29',  #lib/I18N/LangTags.pm
        'I18N::LangTags::List'  => '0.29',  #lib/I18N/LangTags/List.pm
        'if'                    => '0.0401',  #lib/if.pm
        'integer'               => '1.00',  #lib/integer.pm
        'IO'                    => '1.21',  #lib/IO.pm
        'IO::Dir'               => '1.04',  #lib/IO/Dir.pm
        'IO::File'              => '1.10',  #lib/IO/File.pm
        'IO::Handle'            => '1.23',  #lib/IO/Handle.pm
        'IO::Pipe'              => '1.122',  #lib/IO/Pipe.pm
        'IO::Poll'              => '0.06',  #lib/IO/Poll.pm
        'IO::Seekable'          => '1.09',  #lib/IO/Seekable.pm
        'IO::Select'            => '1.16',  #lib/IO/Select.pm
        'IO::Socket'            => '1.28',  #lib/IO/Socket.pm
        'IO::Socket::INET'      => '1.27',  #lib/IO/Socket/INET.pm
        'IO::Socket::UNIX'      => '1.21',  #lib/IO/Socket/UNIX.pm
        'IPC::Msg'              => '1.02',  #lib/IPC/Msg.pm
        'IPC::Open2'            => '1.01',  #lib/IPC/Open2.pm
        'IPC::Open3'            => '1.0105',  #lib/IPC/Open3.pm
        'IPC::Semaphore'        => '1.02',  #lib/IPC/Semaphore.pm
        'IPC::SysV'             => '1.04',  #lib/IPC/SysV.pm
        'JNI'                   => '0.2',  #jpl/JNI/JNI.pm
        'JPL::AutoLoader'       => undef,  #jpl/JPL/AutoLoader.pm
        'JPL::Class'            => undef,  #jpl/JPL/Class.pm
        'JPL::Compile'          => undef,  #jpl/JPL/Compile.pm
        'less'                  => '0.01',  #lib/less.pm
        'lib'                   => '0.5565',  #lib/lib.pm
        'List::Util'            => '1.13',  #lib/List/Util.pm
        'locale'                => '1.00',  #lib/locale.pm
        'Locale::Constants'     => '2.01',  #lib/Locale/Constants.pm
        'Locale::Country'       => '2.61',  #lib/Locale/Country.pm
        'Locale::Currency'      => '2.21',  #lib/Locale/Currency.pm
        'Locale::Language'      => '2.21',  #lib/Locale/Language.pm
        'Locale::Maketext'      => '1.08',  #lib/Locale/Maketext.pm
        'Locale::Maketext::GutsLoader'=> undef,  #lib/Locale/Maketext/GutsLoader.pm
        'Locale::Maketext::Guts'=> undef,  #lib/Locale/Maketext/Guts.pm
        'Locale::Script'        => '2.21',  #lib/Locale/Script.pm
        'Math::BigFloat'        => '1.44',  #lib/Math/BigFloat.pm
        'Math::BigFloat::Trace' => '0.01',  #lib/Math/BigFloat/Trace.pm
        'Math::BigInt'          => '1.70',  #lib/Math/BigInt.pm
        'Math::BigInt::Calc'    => '0.40',  #lib/Math/BigInt/Calc.pm
        'Math::BigInt::CalcEmu' => '0.04',  #lib/Math/BigInt/CalcEmu.pm
        'Math::BigInt::Trace'   => '0.01',  #lib/Math/BigInt/Trace.pm
        'Math::BigRat'          => '0.12',  #lib/Math/BigRat.pm
        'Math::Complex'         => '1.34',  #lib/Math/Complex.pm
        'Math::Trig'            => '1.02',  #lib/Math/Trig.pm
        'Memoize'               => '1.01_01',  #lib/Memoize.pm
        'Memoize::AnyDBM_File'  => '0.65',  #lib/Memoize/AnyDBM_File.pm
        'Memoize::Expire'       => '1.00',  #lib/Memoize/Expire.pm
        'Memoize::ExpireFile'   => '1.01',  #lib/Memoize/ExpireFile.pm
        'Memoize::ExpireTest'   => '0.65',  #lib/Memoize/ExpireTest.pm
        'Memoize::NDBM_File'    => '0.65',  #lib/Memoize/NDBM_File.pm
        'Memoize::SDBM_File'    => '0.65',  #lib/Memoize/SDBM_File.pm
        'Memoize::Storable'     => '0.65',  #lib/Memoize/Storable.pm
        'MIME::Base64'          => '3.00_01',  #lib/MIME/Base64.pm
        'MIME::QuotedPrint'     => '3.00',  #lib/MIME/QuotedPrint.pm
        'NDBM_File'             => '1.05',  #lib/NDBM_File.pm
        'Net::Cmd'              => '2.24',  #lib/Net/Cmd.pm
        'Net::Config'           => '1.10',  #lib/Net/Config.pm
        'Net::Domain'           => '2.19',  #lib/Net/Domain.pm
        'Net::FTP'              => '2.72',  #lib/Net/FTP.pm
        'Net::FTP::A'           => '1.16',  #lib/Net/FTP/A.pm
        'Net::FTP::dataconn'    => '0.11',  #lib/Net/FTP/dataconn.pm
        'Net::FTP::E'           => '0.01',  #lib/Net/FTP/E.pm
        'Net::FTP::I'           => '1.12',  #lib/Net/FTP/I.pm
        'Net::FTP::L'           => '0.01',  #lib/Net/FTP/L.pm
        'Net::hostent'          => '1.01',  #lib/Net/hostent.pm
        'Net::netent'           => '1.00',  #lib/Net/netent.pm
        'Net::Netrc'            => '2.12',  #lib/Net/Netrc.pm
        'Net::NNTP'             => '2.22',  #lib/Net/NNTP.pm
        'Net::Ping'             => '2.31',  #lib/Net/Ping.pm
        'Net::POP3'             => '2.24',  #lib/Net/POP3.pm
        'Net::protoent'         => '1.00',  #lib/Net/protoent.pm
        'Net::servent'          => '1.01',  #lib/Net/servent.pm
        'Net::SMTP'             => '2.26',  #lib/Net/SMTP.pm
        'Net::Time'             => '2.09',  #lib/Net/Time.pm
        'NEXT'                  => '0.60',  #lib/NEXT.pm
        'O'                     => '1.00',  #lib/O.pm
        'ODBM_File'             => '1.04',  #ext/ODBM_File/ODBM_File.pm
        'Opcode'                => '1.06',  #lib/Opcode.pm
        'open'                  => '1.02',  #lib/open.pm
        'ops'                   => '1.00',  #lib/ops.pm
        'OS2::DLL'              => '1.02',  #os2/OS2/REXX/DLL/DLL.pm
        'OS2::ExtAttr'          => '0.02',  #os2/OS2/ExtAttr/ExtAttr.pm
        'OS2::PrfDB'            => '0.03',  #os2/OS2/PrfDB/PrfDB.pm
        'OS2::Process'          => '1.01',  #os2/OS2/Process/Process.pm
        'OS2::REXX'             => '1.02',  #os2/OS2/REXX/REXX.pm
        'overload'              => '1.02',  #lib/overload.pm
        'PerlIO'                => '1.03',  #lib/PerlIO.pm
        'PerlIO::encoding'      => '0.07',  #lib/PerlIO/encoding.pm
        'PerlIO::scalar'        => '0.02',  #lib/PerlIO/scalar.pm
        'PerlIO::via'           => '0.02',  #lib/PerlIO/via.pm
        'PerlIO::via::QuotedPrint'=> '0.06',  #lib/PerlIO/via/QuotedPrint.pm
        'Pod::Checker'          => '1.41',  #lib/Pod/Checker.pm
        'Pod::Find'             => '0.24',  #lib/Pod/Find.pm
        'Pod::Functions'        => '1.02',  #lib/Pod/Functions.pm
        'Pod::Html'             => '1.0502',  #lib/Pod/Html.pm
        'Pod::InputObjects'     => '1.14',  #lib/Pod/InputObjects.pm
        'Pod::LaTeX'            => '0.55',  #lib/Pod/LaTeX.pm
        'Pod::Man'              => '1.37',  #lib/Pod/Man.pm
        'Pod::ParseLink'        => '1.06',  #lib/Pod/ParseLink.pm
        'Pod::Parser'           => '1.14',  #lib/Pod/Parser.pm
        'Pod::ParseUtils'       => '0.3',  #lib/Pod/ParseUtils.pm
        'Pod::Perldoc'          => '3.12',  #lib/Pod/Perldoc.pm
        'Pod::Perldoc::BaseTo'  => undef,  #lib/Pod/Perldoc/BaseTo.pm
        'Pod::Perldoc::GetOptsOO'=> undef,  #lib/Pod/Perldoc/GetOptsOO.pm
        'Pod::Perldoc::ToChecker'=> undef,  #lib/Pod/Perldoc/ToChecker.pm
        'Pod::Perldoc::ToMan'   => undef,  #lib/Pod/Perldoc/ToMan.pm
        'Pod::Perldoc::ToNroff' => undef,  #lib/Pod/Perldoc/ToNroff.pm
        'Pod::Perldoc::ToPod'   => undef,  #lib/Pod/Perldoc/ToPod.pm
        'Pod::Perldoc::ToRtf'   => undef,  #lib/Pod/Perldoc/ToRtf.pm
        'Pod::Perldoc::ToText'  => undef,  #lib/Pod/Perldoc/ToText.pm
        'Pod::Perldoc::ToTk'    => 'undef',  #lib/Pod/Perldoc/ToTk.pm
        'Pod::Perldoc::ToXml'   => undef,  #lib/Pod/Perldoc/ToXml.pm
        'Pod::Plainer'          => '0.01',  #lib/Pod/Plainer.pm
        'Pod::PlainText'        => '2.02',  #lib/Pod/PlainText.pm
        'Pod::Select'           => '1.13',  #lib/Pod/Select.pm
        'Pod::Text'             => '2.21',  #lib/Pod/Text.pm
        'Pod::Text::Color'      => '1.04',  #lib/Pod/Text/Color.pm
        'Pod::Text::Overstrike' => '1.1',  #lib/Pod/Text/Overstrike.pm
        'Pod::Text::Termcap'    => '1.11',  #lib/Pod/Text/Termcap.pm
        'Pod::Usage'            => '1.16',  #lib/Pod/Usage.pm
        'POSIX'                 => '1.07',  #lib/POSIX.pm
        're'                    => '0.04',  #lib/re.pm
        'Safe'                  => '2.10',  #lib/Safe.pm
        'Scalar::Util'          => '1.13',  #lib/Scalar/Util.pm
        'SDBM_File'             => '1.04',  #lib/SDBM_File.pm
        'Search::Dict'          => '1.02',  #lib/Search/Dict.pm
        'SelectSaver'           => '1.00',  #lib/SelectSaver.pm
        'SelfLoader'            => '1.0904',  #lib/SelfLoader.pm
        'Shell'                 => '0.5.2',  #lib/Shell.pm
        'sigtrap'               => '1.02',  #lib/sigtrap.pm
        'Socket'                => '1.77',  #lib/Socket.pm
        'sort'                  => '1.02',  #lib/sort.pm
        'Storable'              => '2.11',  #lib/Storable.pm
        'strict'                => '1.03',  #lib/strict.pm
        'subs'                  => '1.00',  #lib/subs.pm
        'Switch'                => '2.10',  #lib/Switch.pm
        'Symbol'                => '1.05',  #lib/Symbol.pm
        'Sys::Hostname'         => '1.11',  #lib/Sys/Hostname.pm
        'Sys::Syslog'           => '0.05',  #lib/Sys/Syslog.pm
        'Term::ANSIColor'       => '1.08',  #lib/Term/ANSIColor.pm
        'Term::Cap'             => '1.08',  #lib/Term/Cap.pm
        'Term::Complete'        => '1.401',  #lib/Term/Complete.pm
        'Term::ReadLine'        => '1.01',  #lib/Term/ReadLine.pm
        'Test'                  => '1.24',  #lib/Test.pm
        'Test::Builder'         => '0.17',  #lib/Test/Builder.pm
        'Test::Harness'         => '2.40',  #lib/Test/Harness.pm
        'Test::Harness::Assert' => '0.02',  #lib/Test/Harness/Assert.pm
        'Test::Harness::Iterator'=> '0.02',  #lib/Test/Harness/Iterator.pm
        'Test::Harness::Straps' => '0.19',  #lib/Test/Harness/Straps.pm
        'Test::More'            => '0.47',  #lib/Test/More.pm
        'Test::Simple'          => '0.47',  #lib/Test/Simple.pm
        'Text::Abbrev'          => '1.01',  #lib/Text/Abbrev.pm
        'Text::Balanced'        => '1.95',  #lib/Text/Balanced.pm
        'Text::ParseWords'      => '3.21',  #lib/Text/ParseWords.pm
        'Text::Soundex'         => '1.01',  #lib/Text/Soundex.pm
        'Text::Tabs'            => '98.112801',  #lib/Text/Tabs.pm
        'Text::Wrap'            => '2001.09291',  #lib/Text/Wrap.pm
        'Thread'                => '2.00',  #lib/Thread.pm
        'Thread::Queue'         => '2.00',  #lib/Thread/Queue.pm
        'threads'               => '1.02',  #lib/threads.pm
        'Thread::Semaphore'     => '2.01',  #lib/Thread/Semaphore.pm
        'Thread::Signal'        => '1.00', #./ext/Thread/Thread/Signal.pm
        'Thread::Specific'      => '1.00', #./ext/Thread/Thread/Specific.pm
        'threads::shared'       => '0.92',  #lib/threads/shared.pm
        'Tie::Array'            => '1.03',  #lib/Tie/Array.pm
        'Tie::File'             => '0.97',  #lib/Tie/File.pm
        'Tie::Handle'           => '4.1',  #lib/Tie/Handle.pm
        'Tie::Hash'             => '1.01',  #lib/Tie/Hash.pm
        'Tie::Memoize'          => '1.0',  #lib/Tie/Memoize.pm
        'Tie::RefHash'          => '1.31',  #lib/Tie/RefHash.pm
        'Tie::Scalar'           => '1.00',  #lib/Tie/Scalar.pm
        'Tie::SubstrHash'       => '1.00',  #lib/Tie/SubstrHash.pm
        'Time::gmtime'          => '1.02',  #lib/Time/gmtime.pm
        'Time::HiRes'           => '1.56',  #lib/Time/HiRes.pm
        'Time::Local'           => '1.07_94',  #lib/Time/Local.pm
        'Time::localtime'       => '1.02',  #lib/Time/localtime.pm
        'Time::tm'              => '1.00',  #lib/Time/tm.pm
        'Unicode'               => '4.0.0', #lib/unicore/version
        'Unicode::Collate'      => '0.33',  #lib/Unicode/Collate.pm
        'Unicode::Normalize'    => '0.28',  #lib/Unicode/Normalize.pm
        'Unicode::UCD'          => '0.21',  #lib/Unicode/UCD.pm
        'UNIVERSAL'             => '1.02',  #lib/UNIVERSAL.pm
        'User::grent'           => '1.00',  #lib/User/grent.pm
        'User::pwent'           => '1.00',  #lib/User/pwent.pm
        'utf8'                  => '1.02',  #lib/utf8.pm
        'vars'                  => '1.01',  #lib/vars.pm
        'version'               => '0.36',  #lib/version.pm
        'VMS::DCLsym'           => '1.02',  #vms/ext/DCLsym/DCLsym.pm
        'VMS::Filespec'         => '1.11',  #vms/ext/Filespec.pm
        'VMS::Stdio'            => '2.3',  #vms/ext/Stdio/Stdio.pm
        'vmsish'                => '1.01',  #lib/vmsish.pm
        'warnings'              => '1.03',  #lib/warnings.pm
        'warnings::register'    => '1.00',  #lib/warnings/register.pm
        'XS::APItest'           => '0.03',  #lib/XS/APItest.pm
        'XSLoader'              => '0.03',  #lib/XSLoader.pm
        'XS::Typemap'           => '0.01',  #lib/XS/Typemap.pm
    },

    5.008004 => {
        'AnyDBM_File'           => '1.00',  #lib/AnyDBM_File.pm
        'attributes'            => '0.06',  #lib/attributes.pm
        'AutoLoader'            => '5.60',  #lib/AutoLoader.pm
        'AutoSplit'             => '1.04',  #lib/AutoSplit.pm
        'autouse'               => '1.03',  #lib/autouse.pm
        'base'                  => '2.05',  #lib/base.pm
        'Benchmark'             => '1.06',  #lib/Benchmark.pm
        'bigint'                => '0.05',  #lib/bigint.pm
        'bignum'                => '0.15',  #lib/bignum.pm
        'bigrat'                => '0.06',  #lib/bigrat.pm
        'blib'                  => '1.02',  #lib/blib.pm
        'bytes'                 => '1.01',  #lib/bytes.pm
        'Carp'                  => '1.02',  #lib/Carp.pm
        'CGI'                   => '3.04',  #lib/CGI.pm
        'charnames'             => '1.03',  #lib/charnames.pm
        'constant'              => '1.04',  #lib/constant.pm
        'CPAN'                  => '1.76_01',  #lib/CPAN.pm
        'Cwd'                   => '2.17',  #lib/Cwd.pm
        'DBM_Filter'            => '0.01',  #lib/DBM_Filter.pm
        'DB'                    => '1.0',  #lib/DB.pm
        'diagnostics'           => '1.12',  #lib/diagnostics.pm
        'Digest'                => '1.06',  #lib/Digest.pm
        'DirHandle'             => '1.00',  #lib/DirHandle.pm
        'Dumpvalue'             => '1.11',  #lib/Dumpvalue.pm
        'English'               => '1.01',  #lib/English.pm
        'Env'                   => '1.00',  #lib/Env.pm
        'Exporter'              => '5.58',  #lib/Exporter.pm
        'Fatal'                 => '1.03',  #lib/Fatal.pm
        'fields'                => '2.03',  #lib/fields.pm
        'FileCache'             => '1.03',  #lib/FileCache.pm
        'FileHandle'            => '2.01',  #lib/FileHandle.pm
        'filetest'              => '1.01',  #lib/filetest.pm
        'FindBin'               => '1.44',  #lib/FindBin.pm
        'if'                    => '0.03',  #lib/if.pm
        'integer'               => '1.00',  #lib/integer.pm
        'less'                  => '0.01',  #lib/less.pm
        'locale'                => '1.00',  #lib/locale.pm
        'Memoize'               => '1.01',  #lib/Memoize.pm
        'NEXT'                  => '0.60',  #lib/NEXT.pm
        'open'                  => '1.03',  #lib/open.pm
        'overload'              => '1.01',  #lib/overload.pm
        'PerlIO'                => '1.03',  #lib/PerlIO.pm
        'SelectSaver'           => '1.00',  #lib/SelectSaver.pm
        'SelfLoader'            => '1.0904',  #lib/SelfLoader.pm
        'Shell'                 => '0.5.2',  #lib/Shell.pm
        'sigtrap'               => '1.02',  #lib/sigtrap.pm
        'sort'                  => '1.02',  #lib/sort.pm
        'strict'                => '1.03',  #lib/strict.pm
        'subs'                  => '1.00',  #lib/subs.pm
        'Switch'                => '2.10',  #lib/Switch.pm
        'Symbol'                => '1.05',  #lib/Symbol.pm
        'Test'                  => '1.24',  #lib/Test.pm
        'Thread'                => '2.00',  #lib/Thread.pm
        'Unicode'               => '4.0.1', # lib/unicore/version
        'UNIVERSAL'             => '1.01',  #lib/UNIVERSAL.pm
        'utf8'                  => '1.03',  #lib/utf8.pm
        'vars'                  => '1.01',  #lib/vars.pm
        'vmsish'                => '1.01',  #lib/vmsish.pm
        'warnings'              => '1.03',  #lib/warnings.pm
        'Config'                => undef,  #lib/Config.pm
        'lib'                   => '0.5565',  #lib/lib.pm
        're'                    => '0.04',  #lib/re.pm
        'XSLoader'              => '0.02',  #lib/XSLoader.pm
        'DynaLoader'            => '1.05',  #lib/DynaLoader.pm
        'attrs'                 => '1.01',  #lib/attrs.pm
        'B'                     => '1.02',  #lib/B.pm
        'O'                     => '1.00',  #lib/O.pm
        'ByteLoader'            => '0.05',  #lib/ByteLoader.pm
        'DB_File'               => '1.808',  #lib/DB_File.pm
        'Encode'                => '1.99_01',  #lib/Encode.pm
        'encoding'              => '1.48',  #lib/encoding.pm
        'Fcntl'                 => '1.05',  #lib/Fcntl.pm
        'GDBM_File'             => '1.07',  #lib/GDBM_File.pm
        'IO'                    => '1.21',  #lib/IO.pm
        'NDBM_File'             => '1.05',  #lib/NDBM_File.pm
        'Safe'                  => '2.10',  #lib/Safe.pm
        'Opcode'                => '1.05',  #lib/Opcode.pm
        'ops'                   => '1.00',  #lib/ops.pm
        'POSIX'                 => '1.08',  #lib/POSIX.pm
        'SDBM_File'             => '1.04',  #lib/SDBM_File.pm
        'Socket'                => '1.77',  #lib/Socket.pm
        'Storable'              => '2.12',  #lib/Storable.pm
        'threads'               => '1.03',  #lib/threads.pm
        'Errno'                 => '1.09_00',  #lib/Errno.pm
        'Attribute::Handlers'   => '0.78_01',  #lib/Attribute/Handlers.pm
        'Carp::Heavy'           => '1.01',  #lib/Carp/Heavy.pm
        'CGI::Apache'           => '1.00',  #lib/CGI/Apache.pm
        'CGI::Carp'             => '1.27',  #lib/CGI/Carp.pm
        'CGI::Cookie'           => '1.24',  #lib/CGI/Cookie.pm
        'CGI::Fast'             => '1.05',  #lib/CGI/Fast.pm
        'CGI::Pretty'           => '1.08',  #lib/CGI/Pretty.pm
        'CGI::Push'             => '1.04',  #lib/CGI/Push.pm
        'CGI::Switch'           => '1.00',  #lib/CGI/Switch.pm
        'CGI::Util'             => '1.4',  #lib/CGI/Util.pm
        'Class::ISA'            => '0.32',  #lib/Class/ISA.pm
        'Class::Struct'         => '0.63',  #lib/Class/Struct.pm
        'CPAN::FirstTime'       => '1.60 ',  #lib/CPAN/FirstTime.pm
        'CPAN::Nox'             => '1.03',  #lib/CPAN/Nox.pm
        'DBM_Filter::compress'  => '0.01',  #lib/DBM_Filter/compress.pm
        'DBM_Filter::encode'    => '0.01',  #lib/DBM_Filter/encode.pm
        'DBM_Filter::int32'     => '0.01',  #lib/DBM_Filter/int32.pm
        'DBM_Filter::null'      => '0.01',  #lib/DBM_Filter/null.pm
        'DBM_Filter::utf8'      => '0.01',  #lib/DBM_Filter/utf8.pm
        'Devel::SelfStubber'    => '1.03',  #lib/Devel/SelfStubber.pm
        'Devel::DProf'          => '20030813.00',  #lib/Devel/DProf.pm
        'Devel::Peek'           => '1.01',  #lib/Devel/Peek.pm
        'Devel::PPPort'         => '2.011',  #lib/Devel/PPPort.pm
        'Digest::base'          => '1.00',  #lib/Digest/base.pm
        'Digest::MD5'           => '2.33',  #lib/Digest/MD5.pm
        'Exporter::Heavy'       => '5.57',  #lib/Exporter/Heavy.pm
        'ExtUtils::Command'     => '1.05',  #lib/ExtUtils/Command.pm
        'ExtUtils::Constant'    => '0.14',  #lib/ExtUtils/Constant.pm
        'ExtUtils::Embed'       => '1.250601',  #lib/ExtUtils/Embed.pm
        'ExtUtils::Installed'   => '0.08',  #lib/ExtUtils/Installed.pm
        'ExtUtils::Install'     => '1.32',  #lib/ExtUtils/Install.pm
        'ExtUtils::Liblist'     => '1.01',  #lib/ExtUtils/Liblist.pm
        'ExtUtils::MakeMaker'   => '6.17',  #lib/ExtUtils/MakeMaker.pm
        'ExtUtils::Manifest'    => '1.42',  #lib/ExtUtils/Manifest.pm
        'ExtUtils::Mkbootstrap' => '1.15',  #lib/ExtUtils/Mkbootstrap.pm
        'ExtUtils::Mksymlists'  => '1.19',  #lib/ExtUtils/Mksymlists.pm
        'ExtUtils::MM_Any'      => '0.07',  #lib/ExtUtils/MM_Any.pm
        'ExtUtils::MM_BeOS'     => '1.04',  #lib/ExtUtils/MM_BeOS.pm
        'ExtUtils::MM_Cygwin'   => '1.06',  #lib/ExtUtils/MM_Cygwin.pm
        'ExtUtils::MM_DOS'      => '0.02',  #lib/ExtUtils/MM_DOS.pm
        'ExtUtils::MM_MacOS'    => '1.07',  #lib/ExtUtils/MM_MacOS.pm
        'ExtUtils::MM_NW5'      => '2.07_02',  #lib/ExtUtils/MM_NW5.pm
        'ExtUtils::MM'          => '0.04',  #lib/ExtUtils/MM.pm
        'ExtUtils::MM_OS2'      => '1.04',  #lib/ExtUtils/MM_OS2.pm
        'ExtUtils::MM_Unix'     => '1.42',  #lib/ExtUtils/MM_Unix.pm
        'ExtUtils::MM_UWIN'     => '0.02',  #lib/ExtUtils/MM_UWIN.pm
        'ExtUtils::MM_VMS'      => '5.70',  #lib/ExtUtils/MM_VMS.pm
        'ExtUtils::MM_Win32'    => '1.09',  #lib/ExtUtils/MM_Win32.pm
        'ExtUtils::MM_Win95'    => '0.0301',  #lib/ExtUtils/MM_Win95.pm
        'ExtUtils::MY'          => '0.01',  #lib/ExtUtils/MY.pm
        'ExtUtils::Packlist'    => '0.04',  #lib/ExtUtils/Packlist.pm
        'ExtUtils::testlib'     => '1.15',  #lib/ExtUtils/testlib.pm
        'ExtUtils::Miniperl'    => undef,  #lib/ExtUtils/Miniperl.pm
        'ExtUtils::Command::MM' => '0.03',  #lib/ExtUtils/Command/MM.pm
        'ExtUtils::Liblist::Kid'=> '1.3001',  #lib/ExtUtils/Liblist/Kid.pm
        'ExtUtils::MakeMaker::bytes'=> '0.01',  #lib/ExtUtils/MakeMaker/bytes.pm
        'ExtUtils::MakeMaker::vmsish'=> '0.01',  #lib/ExtUtils/MakeMaker/vmsish.pm
        'File::Basename'        => '2.72',  #lib/File/Basename.pm
        'File::CheckTree'       => '4.3',  #lib/File/CheckTree.pm
        'File::Compare'         => '1.1003',  #lib/File/Compare.pm
        'File::Copy'            => '2.07',  #lib/File/Copy.pm
        'File::DosGlob'         => '1.00',  #lib/File/DosGlob.pm
        'File::Find'            => '1.07',  #lib/File/Find.pm
        'File::Path'            => '1.06',  #lib/File/Path.pm
        'File::Spec'            => '0.87',  #lib/File/Spec.pm
        'File::stat'            => '1.00',  #lib/File/stat.pm
        'File::Temp'            => '0.14',  #lib/File/Temp.pm
        'File::Glob'            => '1.02',  #lib/File/Glob.pm
        'File::Spec::Cygwin'    => '1.1',  #lib/File/Spec/Cygwin.pm
        'File::Spec::Epoc'      => '1.1',  #lib/File/Spec/Epoc.pm
        'File::Spec::Functions' => '1.3',  #lib/File/Spec/Functions.pm
        'File::Spec::Mac'       => '1.4',  #lib/File/Spec/Mac.pm
        'File::Spec::OS2'       => '1.2',  #lib/File/Spec/OS2.pm
        'File::Spec::Unix'      => '1.5',  #lib/File/Spec/Unix.pm
        'File::Spec::VMS'       => '1.4',  #lib/File/Spec/VMS.pm
        'File::Spec::Win32'     => '1.4',  #lib/File/Spec/Win32.pm
        'Filter::Simple'        => '0.78',  #lib/Filter/Simple.pm
        'Filter::Util::Call'    => '1.0601',  #lib/Filter/Util/Call.pm
        'Getopt::Long'          => '2.34',  #lib/Getopt/Long.pm
        'Getopt::Std'           => '1.05',  #lib/Getopt/Std.pm
        'Hash::Util'            => '0.05',  #lib/Hash/Util.pm
        'I18N::Collate'         => '1.00',  #lib/I18N/Collate.pm
        'I18N::LangTags'        => '0.29',  #lib/I18N/LangTags.pm
        'I18N::Langinfo'        => '0.02',  #lib/I18N/Langinfo.pm
        'I18N::LangTags::List'  => '0.29',  #lib/I18N/LangTags/List.pm
        'IPC::Open2'            => '1.01',  #lib/IPC/Open2.pm
        'IPC::Open3'            => '1.0106',  #lib/IPC/Open3.pm
        'IPC::Msg'              => '1.02',  #lib/IPC/Msg.pm
        'IPC::Semaphore'        => '1.02',  #lib/IPC/Semaphore.pm
        'IPC::SysV'             => '1.04',  #lib/IPC/SysV.pm
        'Locale::Constants'     => '2.01',  #lib/Locale/Constants.pm
        'Locale::Country'       => '2.61',  #lib/Locale/Country.pm
        'Locale::Currency'      => '2.21',  #lib/Locale/Currency.pm
        'Locale::Language'      => '2.21',  #lib/Locale/Language.pm
        'Locale::Maketext'      => '1.08',  #lib/Locale/Maketext.pm
        'Locale::Script'        => '2.21',  #lib/Locale/Script.pm
        'Locale::Maketext::GutsLoader'=> undef,  #lib/Locale/Maketext/GutsLoader.pm
        'Locale::Maketext::Guts'=> undef,  #lib/Locale/Maketext/Guts.pm
        'Math::BigFloat'        => '1.44',  #lib/Math/BigFloat.pm
        'Math::BigInt'          => '1.70',  #lib/Math/BigInt.pm
        'Math::BigRat'          => '0.12',  #lib/Math/BigRat.pm
        'Math::Complex'         => '1.34',  #lib/Math/Complex.pm
        'Math::Trig'            => '1.02',  #lib/Math/Trig.pm
        'Math::BigFloat::Trace' => '0.01',  #lib/Math/BigFloat/Trace.pm
        'Math::BigInt::CalcEmu' => '0.04',  #lib/Math/BigInt/CalcEmu.pm
        'Math::BigInt::Calc'    => '0.40',  #lib/Math/BigInt/Calc.pm
        'Math::BigInt::Trace'   => '0.01',  #lib/Math/BigInt/Trace.pm
        'Memoize::AnyDBM_File'  => '0.65',  #lib/Memoize/AnyDBM_File.pm
        'Memoize::ExpireFile'   => '1.01',  #lib/Memoize/ExpireFile.pm
        'Memoize::Expire'       => '1.00',  #lib/Memoize/Expire.pm
        'Memoize::ExpireTest'   => '0.65',  #lib/Memoize/ExpireTest.pm
        'Memoize::NDBM_File'    => '0.65',  #lib/Memoize/NDBM_File.pm
        'Memoize::SDBM_File'    => '0.65',  #lib/Memoize/SDBM_File.pm
        'Memoize::Storable'     => '0.65',  #lib/Memoize/Storable.pm
        'Net::Cmd'              => '2.24',  #lib/Net/Cmd.pm
        'Net::Config'           => '1.10',  #lib/Net/Config.pm
        'Net::Domain'           => '2.19',  #lib/Net/Domain.pm
        'Net::FTP'              => '2.72',  #lib/Net/FTP.pm
        'Net::hostent'          => '1.01',  #lib/Net/hostent.pm
        'Net::netent'           => '1.00',  #lib/Net/netent.pm
        'Net::Netrc'            => '2.12',  #lib/Net/Netrc.pm
        'Net::NNTP'             => '2.22',  #lib/Net/NNTP.pm
        'Net::Ping'             => '2.31',  #lib/Net/Ping.pm
        'Net::POP3'             => '2.24',  #lib/Net/POP3.pm
        'Net::protoent'         => '1.00',  #lib/Net/protoent.pm
        'Net::servent'          => '1.01',  #lib/Net/servent.pm
        'Net::SMTP'             => '2.26',  #lib/Net/SMTP.pm
        'Net::Time'             => '2.09',  #lib/Net/Time.pm
        'Net::FTP::A'           => '1.16',  #lib/Net/FTP/A.pm
        'Net::FTP::dataconn'    => '0.11',  #lib/Net/FTP/dataconn.pm
        'Net::FTP::E'           => '0.01',  #lib/Net/FTP/E.pm
        'Net::FTP::I'           => '1.12',  #lib/Net/FTP/I.pm
        'Net::FTP::L'           => '0.01',  #lib/Net/FTP/L.pm
        'PerlIO::encoding'      => '0.07',  #lib/PerlIO/encoding.pm
        'PerlIO::scalar'        => '0.02',  #lib/PerlIO/scalar.pm
        'PerlIO::via'           => '0.02',  #lib/PerlIO/via.pm
        'PerlIO::via::QuotedPrint'=> '0.06',  #lib/PerlIO/via/QuotedPrint.pm
        'Pod::Checker'          => '1.41',  #lib/Pod/Checker.pm
        'Pod::Find'             => '0.24',  #lib/Pod/Find.pm
        'Pod::Functions'        => '1.02',  #lib/Pod/Functions.pm
        'Pod::Html'             => '1.0502',  #lib/Pod/Html.pm
        'Pod::InputObjects'     => '1.14',  #lib/Pod/InputObjects.pm
        'Pod::LaTeX'            => '0.55',  #lib/Pod/LaTeX.pm
        'Pod::Man'              => '1.37',  #lib/Pod/Man.pm
        'Pod::ParseLink'        => '1.06',  #lib/Pod/ParseLink.pm
        'Pod::Parser'           => '1.14',  #lib/Pod/Parser.pm
        'Pod::ParseUtils'       => '0.3',  #lib/Pod/ParseUtils.pm
        'Pod::Perldoc'          => '3.12',  #lib/Pod/Perldoc.pm
        'Pod::Plainer'          => '0.01',  #lib/Pod/Plainer.pm
        'Pod::PlainText'        => '2.02',  #lib/Pod/PlainText.pm
        'Pod::Select'           => '1.13',  #lib/Pod/Select.pm
        'Pod::Text'             => '2.21',  #lib/Pod/Text.pm
        'Pod::Usage'            => '1.16',  #lib/Pod/Usage.pm
        'Pod::Perldoc::BaseTo'  => undef,  #lib/Pod/Perldoc/BaseTo.pm
        'Pod::Perldoc::GetOptsOO'=> undef,  #lib/Pod/Perldoc/GetOptsOO.pm
        'Pod::Perldoc::ToChecker'=> undef,  #lib/Pod/Perldoc/ToChecker.pm
        'Pod::Perldoc::ToMan'   => undef,  #lib/Pod/Perldoc/ToMan.pm
        'Pod::Perldoc::ToNroff' => undef,  #lib/Pod/Perldoc/ToNroff.pm
        'Pod::Perldoc::ToPod'   => undef,  #lib/Pod/Perldoc/ToPod.pm
        'Pod::Perldoc::ToRtf'   => undef,  #lib/Pod/Perldoc/ToRtf.pm
        'Pod::Perldoc::ToText'  => undef,  #lib/Pod/Perldoc/ToText.pm
        'Pod::Perldoc::ToTk'    => 'undef',  #lib/Pod/Perldoc/ToTk.pm
        'Pod::Perldoc::ToXml'   => undef,  #lib/Pod/Perldoc/ToXml.pm
        'Pod::Text::Color'      => '1.04',  #lib/Pod/Text/Color.pm
        'Pod::Text::Overstrike' => '1.1',  #lib/Pod/Text/Overstrike.pm
        'Pod::Text::Termcap'    => '1.11',  #lib/Pod/Text/Termcap.pm
        'Search::Dict'          => '1.02',  #lib/Search/Dict.pm
        'Term::ANSIColor'       => '1.08',  #lib/Term/ANSIColor.pm
        'Term::Cap'             => '1.08',  #lib/Term/Cap.pm
        'Term::Complete'        => '1.401',  #lib/Term/Complete.pm
        'Term::ReadLine'        => '1.01',  #lib/Term/ReadLine.pm
        'Test::Builder'         => '0.17',  #lib/Test/Builder.pm
        'Test::Harness'         => '2.40',  #lib/Test/Harness.pm
        'Test::More'            => '0.47',  #lib/Test/More.pm
        'Test::Simple'          => '0.47',  #lib/Test/Simple.pm
        'Test::Harness::Assert' => '0.02',  #lib/Test/Harness/Assert.pm
        'Test::Harness::Iterator'=> '0.02',  #lib/Test/Harness/Iterator.pm
        'Test::Harness::Straps' => '0.19',  #lib/Test/Harness/Straps.pm
        'Text::Abbrev'          => '1.01',  #lib/Text/Abbrev.pm
        'Text::Balanced'        => '1.95',  #lib/Text/Balanced.pm
        'Text::ParseWords'      => '3.21',  #lib/Text/ParseWords.pm
        'Text::Soundex'         => '1.01',  #lib/Text/Soundex.pm
        'Text::Tabs'            => '98.112801',  #lib/Text/Tabs.pm
        'Text::Wrap'            => '2001.09291',  #lib/Text/Wrap.pm
        'Thread::Queue'         => '2.00',  #lib/Thread/Queue.pm
        'Thread::Semaphore'     => '2.01',  #lib/Thread/Semaphore.pm
        'Tie::Array'            => '1.03',  #lib/Tie/Array.pm
        'Tie::File'             => '0.97',  #lib/Tie/File.pm
        'Tie::Handle'           => '4.1',  #lib/Tie/Handle.pm
        'Tie::Hash'             => '1.01',  #lib/Tie/Hash.pm
        'Tie::Memoize'          => '1.0',  #lib/Tie/Memoize.pm
        'Tie::RefHash'          => '1.31',  #lib/Tie/RefHash.pm
        'Tie::Scalar'           => '1.00',  #lib/Tie/Scalar.pm
        'Tie::SubstrHash'       => '1.00',  #lib/Tie/SubstrHash.pm
        'Time::gmtime'          => '1.02',  #lib/Time/gmtime.pm
        'Time::Local'           => '1.07',  #lib/Time/Local.pm
        'Time::localtime'       => '1.02',  #lib/Time/localtime.pm
        'Time::tm'              => '1.00',  #lib/Time/tm.pm
        'Time::HiRes'           => '1.59',  #lib/Time/HiRes.pm
        'Unicode::Collate'      => '0.33',  #lib/Unicode/Collate.pm
        'Unicode::UCD'          => '0.22',  #lib/Unicode/UCD.pm
        'Unicode::Normalize'    => '0.28',  #lib/Unicode/Normalize.pm
        'User::grent'           => '1.00',  #lib/User/grent.pm
        'User::pwent'           => '1.00',  #lib/User/pwent.pm
        'warnings::register'    => '1.00',  #lib/warnings/register.pm
        'B::Stash'              => '1.00',  #lib/B/Stash.pm
        'B::Asmdata'            => '1.01',  #lib/B/Asmdata.pm
        'B::C'                  => '1.02',  #lib/B/C.pm
        'B::Deparse'            => '0.66',  #lib/B/Deparse.pm
        'B::Debug'              => '1.01',  #lib/B/Debug.pm
        'B::Bblock'             => '1.02',  #lib/B/Bblock.pm
        'B::Assembler'          => '0.07',  #lib/B/Assembler.pm
        'B::Terse'              => '1.02',  #lib/B/Terse.pm
        'B::CC'                 => '1.00',  #lib/B/CC.pm
        'B::Concise'            => '0.60',  #lib/B/Concise.pm
        'B::Lint'               => '1.02',  #lib/B/Lint.pm
        'B::Showlex'            => '1.00',  #lib/B/Showlex.pm
        'B::Bytecode'           => '1.01',  #lib/B/Bytecode.pm
        'B::Disassembler'       => '1.03',  #lib/B/Disassembler.pm
        'B::Xref'               => '1.01',  #lib/B/Xref.pm
        'B::Stackobj'           => '1.00',  #lib/B/Stackobj.pm
        'Data::Dumper'          => '2.121',  #lib/Data/Dumper.pm
        'Encode::Alias'         => '1.38',  #lib/Encode/Alias.pm
        'Encode::Encoding'      => '1.33',  #lib/Encode/Encoding.pm
        'Encode::Guess'         => '1.09',  #lib/Encode/Guess.pm
        'Encode::Config'        => '1.07',  #lib/Encode/Config.pm
        'Encode::Encoder'       => '0.07',  #lib/Encode/Encoder.pm
        'Encode::CJKConstants'  => '1.02',  #lib/Encode/CJKConstants.pm
        'Encode::Byte'          => '1.23',  #lib/Encode/Byte.pm
        'Encode::CN'            => '1.24',  #lib/Encode/CN.pm
        'Encode::EBCDIC'        => '1.21',  #lib/Encode/EBCDIC.pm
        'Encode::JP'            => '1.25',  #lib/Encode/JP.pm
        'Encode::KR'            => '1.23',  #lib/Encode/KR.pm
        'Encode::Symbol'        => '1.22',  #lib/Encode/Symbol.pm
        'Encode::TW'            => '1.26',  #lib/Encode/TW.pm
        'Encode::Unicode'       => '1.40',  #lib/Encode/Unicode.pm
        'Encode::JP::H2Z'       => '1.02',  #lib/Encode/JP/H2Z.pm
        'Encode::JP::JIS7'      => '1.12',  #lib/Encode/JP/JIS7.pm
        'Encode::Unicode::UTF7' => '0.02',  #lib/Encode/Unicode/UTF7.pm
        'Encode::KR::2022_KR'   => '1.06',  #lib/Encode/KR/2022_KR.pm
        'Encode::MIME::Header'  => '1.09',  #lib/Encode/MIME/Header.pm
        'Encode::CN::HZ'        => '1.0501',  #lib/Encode/CN/HZ.pm
        'IO::Pipe'              => '1.123',  #lib/IO/Pipe.pm
        'IO::File'              => '1.10',  #lib/IO/File.pm
        'IO::Select'            => '1.16',  #lib/IO/Select.pm
        'IO::Socket'            => '1.28',  #lib/IO/Socket.pm
        'IO::Poll'              => '0.06',  #lib/IO/Poll.pm
        'IO::Dir'               => '1.04',  #lib/IO/Dir.pm
        'IO::Handle'            => '1.24',  #lib/IO/Handle.pm
        'IO::Seekable'          => '1.09',  #lib/IO/Seekable.pm
        'IO::Socket::INET'      => '1.27',  #lib/IO/Socket/INET.pm
        'IO::Socket::UNIX'      => '1.21',  #lib/IO/Socket/UNIX.pm
        'List::Util'            => '1.13',  #lib/List/Util.pm
        'Scalar::Util'          => '1.13',  #lib/Scalar/Util.pm
        'MIME::QuotedPrint'     => '3.01',  #lib/MIME/QuotedPrint.pm
        'MIME::Base64'          => '3.01',  #lib/MIME/Base64.pm
        'Sys::Hostname'         => '1.11',  #lib/Sys/Hostname.pm
        'Sys::Syslog'           => '0.05',  #lib/Sys/Syslog.pm
        'XS::APItest'           => '0.03',  #lib/XS/APItest.pm
        'XS::Typemap'           => '0.01',  #lib/XS/Typemap.pm
        'threads::shared'       => '0.92',  #lib/threads/shared.pm
        'ExtUtils::XSSymSet'    => '1.0',  #vms/ext/XSSymSet.pm
        'JNI'                   => '0.2',  #jpl/JNI/JNI.pm
        'JPL::AutoLoader'       => undef,  #jpl/JPL/AutoLoader.pm
        'JPL::Class'            => undef,  #jpl/JPL/Class.pm
        'JPL::Compile'          => undef,  #jpl/JPL/Compile.pm
        'ODBM_File'             => '1.05',  #ext/ODBM_File/ODBM_File.pm
        'OS2::DLL'              => '1.02',  #os2/OS2/REXX/DLL/DLL.pm
        'OS2::ExtAttr'          => '0.02',  #os2/OS2/ExtAttr/ExtAttr.pm
        'OS2::PrfDB'            => '0.03',  #os2/OS2/PrfDB/PrfDB.pm
        'OS2::Process'          => '1.01',  #os2/OS2/Process/Process.pm
        'OS2::REXX'             => '1.02',  #os2/OS2/REXX/REXX.pm
        'Thread::Signal'        => '1.00', #./ext/Thread/Thread/Signal.pm
        'Thread::Specific'      => '1.00', #./ext/Thread/Thread/Specific.pm
        'VMS::DCLsym'           => '1.02',  #vms/ext/DCLsym/DCLsym.pm
        'VMS::Filespec'         => '1.11',  #vms/ext/Filespec.pm
        'VMS::Stdio'            => '2.3',  #vms/ext/Stdio/Stdio.pm
    },

    5.008005 => {
        'AnyDBM_File'           => '1.00',  #lib/AnyDBM_File.pm
        'attributes'            => '0.06',  #lib/attributes.pm
        'AutoLoader'            => '5.60',  #lib/AutoLoader.pm
        'AutoSplit'             => '1.04',  #lib/AutoSplit.pm
        'autouse'               => '1.04',  #lib/autouse.pm
        'base'                  => '2.06',  #lib/base.pm
        'Benchmark'             => '1.06',  #lib/Benchmark.pm
        'bigint'                => '0.05',  #lib/bigint.pm
        'bignum'                => '0.15',  #lib/bignum.pm
        'bigrat'                => '0.06',  #lib/bigrat.pm
        'blib'                  => '1.02',  #lib/blib.pm
        'bytes'                 => '1.01',  #lib/bytes.pm
        'Carp'                  => '1.03',  #lib/Carp.pm
        'CGI'                   => '3.05',  #lib/CGI.pm
        'charnames'             => '1.04',  #lib/charnames.pm
        'constant'              => '1.04',  #lib/constant.pm
        'CPAN'                  => '1.76_01',  #lib/CPAN.pm
        'Cwd'                   => '2.19',  #lib/Cwd.pm
        'DBM_Filter'            => '0.01',  #lib/DBM_Filter.pm
        'DB'                    => '1.0',  #lib/DB.pm
        'diagnostics'           => '1.13',  #lib/diagnostics.pm
        'Digest'                => '1.08',  #lib/Digest.pm
        'DirHandle'             => '1.00',  #lib/DirHandle.pm
        'Dumpvalue'             => '1.11',  #lib/Dumpvalue.pm
        'English'               => '1.01',  #lib/English.pm
        'Env'                   => '1.00',  #lib/Env.pm
        'Exporter'              => '5.58',  #lib/Exporter.pm
        'Fatal'                 => '1.03',  #lib/Fatal.pm
        'fields'                => '2.03',  #lib/fields.pm
        'FileCache'             => '1.04_01',  #lib/FileCache.pm
        'FileHandle'            => '2.01',  #lib/FileHandle.pm
        'filetest'              => '1.01',  #lib/filetest.pm
        'FindBin'               => '1.44',  #lib/FindBin.pm
        'if'                    => '0.03',  #lib/if.pm
        'integer'               => '1.00',  #lib/integer.pm
        'less'                  => '0.01',  #lib/less.pm
        'locale'                => '1.00',  #lib/locale.pm
        'Memoize'               => '1.01',  #lib/Memoize.pm
        'NEXT'                  => '0.60',  #lib/NEXT.pm
        'open'                  => '1.03',  #lib/open.pm
        'overload'              => '1.01',  #lib/overload.pm
        'PerlIO'                => '1.03',  #lib/PerlIO.pm
        'SelectSaver'           => '1.00',  #lib/SelectSaver.pm
        'SelfLoader'            => '1.0904',  #lib/SelfLoader.pm
        'Shell'                 => '0.6',  #lib/Shell.pm
        'sigtrap'               => '1.02',  #lib/sigtrap.pm
        'sort'                  => '1.02',  #lib/sort.pm
        'strict'                => '1.03',  #lib/strict.pm
        'subs'                  => '1.00',  #lib/subs.pm
        'Switch'                => '2.10',  #lib/Switch.pm
        'Symbol'                => '1.05',  #lib/Symbol.pm
        'Test'                  => '1.25',  #lib/Test.pm
        'Thread'                => '2.00',  #lib/Thread.pm
        'UNIVERSAL'             => '1.01',  #lib/UNIVERSAL.pm
        'utf8'                  => '1.04',  #lib/utf8.pm
        'vars'                  => '1.01',  #lib/vars.pm
        'vmsish'                => '1.01',  #lib/vmsish.pm
        'warnings'              => '1.03',  #lib/warnings.pm
        'Config'                => undef,  #lib/Config.pm
        'lib'                   => '0.5565',  #lib/lib.pm
        're'                    => '0.04',  #lib/re.pm
        'XSLoader'              => '0.02',  #lib/XSLoader.pm
        'DynaLoader'            => '1.05',  #lib/DynaLoader.pm
        'attrs'                 => '1.01',  #lib/attrs.pm
        'B'                     => '1.02',  #lib/B.pm
        'O'                     => '1.00',  #lib/O.pm
        'ByteLoader'            => '0.05',  #lib/ByteLoader.pm
        'DB_File'               => '1.809',  #lib/DB_File.pm
        'Encode'                => '2.01',  #lib/Encode.pm
        'encoding'              => '2.00',  #lib/encoding.pm
        'Fcntl'                 => '1.05',  #lib/Fcntl.pm
        'GDBM_File'             => '1.07',  #lib/GDBM_File.pm
        'IO'                    => '1.21',  #lib/IO.pm
        'NDBM_File'             => '1.05',  #lib/NDBM_File.pm
        'Safe'                  => '2.11',  #lib/Safe.pm
        'Opcode'                => '1.05',  #lib/Opcode.pm
        'ops'                   => '1.00',  #lib/ops.pm
        'POSIX'                 => '1.08',  #lib/POSIX.pm
        'SDBM_File'             => '1.04',  #lib/SDBM_File.pm
        'Socket'                => '1.77',  #lib/Socket.pm
        'Storable'              => '2.13',  #lib/Storable.pm
        'threads'               => '1.05',  #lib/threads.pm
        'Errno'                 => '1.09_00',  #lib/Errno.pm
        'Attribute::Handlers'   => '0.78_01',  #lib/Attribute/Handlers.pm
        'Carp::Heavy'           => '1.01',  #lib/Carp/Heavy.pm
        'CGI::Apache'           => '1.00',  #lib/CGI/Apache.pm
        'CGI::Carp'             => '1.28',  #lib/CGI/Carp.pm
        'CGI::Cookie'           => '1.24',  #lib/CGI/Cookie.pm
        'CGI::Fast'             => '1.05',  #lib/CGI/Fast.pm
        'CGI::Pretty'           => '1.08',  #lib/CGI/Pretty.pm
        'CGI::Push'             => '1.04',  #lib/CGI/Push.pm
        'CGI::Switch'           => '1.00',  #lib/CGI/Switch.pm
        'CGI::Util'             => '1.5',  #lib/CGI/Util.pm
        'Class::ISA'            => '0.32',  #lib/Class/ISA.pm
        'Class::Struct'         => '0.63',  #lib/Class/Struct.pm
        'CPAN::FirstTime'       => '1.60 ',  #lib/CPAN/FirstTime.pm
        'CPAN::Nox'             => '1.03',  #lib/CPAN/Nox.pm
        'DBM_Filter::compress'  => '0.01',  #lib/DBM_Filter/compress.pm
        'DBM_Filter::encode'    => '0.01',  #lib/DBM_Filter/encode.pm
        'DBM_Filter::int32'     => '0.01',  #lib/DBM_Filter/int32.pm
        'DBM_Filter::null'      => '0.01',  #lib/DBM_Filter/null.pm
        'DBM_Filter::utf8'      => '0.01',  #lib/DBM_Filter/utf8.pm
        'Devel::SelfStubber'    => '1.03',  #lib/Devel/SelfStubber.pm
        'Devel::DProf'          => '20030813.00',  #lib/Devel/DProf.pm
        'Devel::Peek'           => '1.01',  #lib/Devel/Peek.pm
        'Devel::PPPort'         => '2.011',  #lib/Devel/PPPort.pm
        'Digest::base'          => '1.00',  #lib/Digest/base.pm
        'Digest::MD5'           => '2.33',  #lib/Digest/MD5.pm
        'Exporter::Heavy'       => '5.57',  #lib/Exporter/Heavy.pm
        'ExtUtils::Command'     => '1.05',  #lib/ExtUtils/Command.pm
        'ExtUtils::Constant'    => '0.14',  #lib/ExtUtils/Constant.pm
        'ExtUtils::Embed'       => '1.250601',  #lib/ExtUtils/Embed.pm
        'ExtUtils::Installed'   => '0.08',  #lib/ExtUtils/Installed.pm
        'ExtUtils::Install'     => '1.32',  #lib/ExtUtils/Install.pm
        'ExtUtils::Liblist'     => '1.01',  #lib/ExtUtils/Liblist.pm
        'ExtUtils::MakeMaker'   => '6.17',  #lib/ExtUtils/MakeMaker.pm
        'ExtUtils::Manifest'    => '1.42',  #lib/ExtUtils/Manifest.pm
        'ExtUtils::Mkbootstrap' => '1.15',  #lib/ExtUtils/Mkbootstrap.pm
        'ExtUtils::Mksymlists'  => '1.19',  #lib/ExtUtils/Mksymlists.pm
        'ExtUtils::MM_Any'      => '0.07',  #lib/ExtUtils/MM_Any.pm
        'ExtUtils::MM_BeOS'     => '1.04',  #lib/ExtUtils/MM_BeOS.pm
        'ExtUtils::MM_Cygwin'   => '1.06',  #lib/ExtUtils/MM_Cygwin.pm
        'ExtUtils::MM_DOS'      => '0.02',  #lib/ExtUtils/MM_DOS.pm
        'ExtUtils::MM_MacOS'    => '1.07',  #lib/ExtUtils/MM_MacOS.pm
        'ExtUtils::MM_NW5'      => '2.07_02',  #lib/ExtUtils/MM_NW5.pm
        'ExtUtils::MM'          => '0.04',  #lib/ExtUtils/MM.pm
        'ExtUtils::MM_OS2'      => '1.04',  #lib/ExtUtils/MM_OS2.pm
        'ExtUtils::MM_Unix'     => '1.42',  #lib/ExtUtils/MM_Unix.pm
        'ExtUtils::MM_UWIN'     => '0.02',  #lib/ExtUtils/MM_UWIN.pm
        'ExtUtils::MM_VMS'      => '5.70',  #lib/ExtUtils/MM_VMS.pm
        'ExtUtils::MM_Win32'    => '1.09',  #lib/ExtUtils/MM_Win32.pm
        'ExtUtils::MM_Win95'    => '0.0301',  #lib/ExtUtils/MM_Win95.pm
        'ExtUtils::MY'          => '0.01',  #lib/ExtUtils/MY.pm
        'ExtUtils::Packlist'    => '0.04',  #lib/ExtUtils/Packlist.pm
        'ExtUtils::testlib'     => '1.15',  #lib/ExtUtils/testlib.pm
        'ExtUtils::Miniperl'    => undef,  #lib/ExtUtils/Miniperl.pm
        'ExtUtils::Command::MM' => '0.03',  #lib/ExtUtils/Command/MM.pm
        'ExtUtils::Liblist::Kid'=> '1.3001',  #lib/ExtUtils/Liblist/Kid.pm
        'ExtUtils::MakeMaker::bytes'=> '0.01',  #lib/ExtUtils/MakeMaker/bytes.pm
        'ExtUtils::MakeMaker::vmsish'=> '0.01',  #lib/ExtUtils/MakeMaker/vmsish.pm
        'File::Basename'        => '2.73',  #lib/File/Basename.pm
        'File::CheckTree'       => '4.3',  #lib/File/CheckTree.pm
        'File::Compare'         => '1.1003',  #lib/File/Compare.pm
        'File::Copy'            => '2.08',  #lib/File/Copy.pm
        'File::DosGlob'         => '1.00',  #lib/File/DosGlob.pm
        'File::Find'            => '1.07',  #lib/File/Find.pm
        'File::Path'            => '1.06',  #lib/File/Path.pm
        'File::Spec'            => '0.87',  #lib/File/Spec.pm
        'File::stat'            => '1.00',  #lib/File/stat.pm
        'File::Temp'            => '0.14',  #lib/File/Temp.pm
        'File::Glob'            => '1.03',  #lib/File/Glob.pm
        'File::Spec::Cygwin'    => '1.1',  #lib/File/Spec/Cygwin.pm
        'File::Spec::Epoc'      => '1.1',  #lib/File/Spec/Epoc.pm
        'File::Spec::Functions' => '1.3',  #lib/File/Spec/Functions.pm
        'File::Spec::Mac'       => '1.4',  #lib/File/Spec/Mac.pm
        'File::Spec::OS2'       => '1.2',  #lib/File/Spec/OS2.pm
        'File::Spec::Unix'      => '1.5',  #lib/File/Spec/Unix.pm
        'File::Spec::VMS'       => '1.4',  #lib/File/Spec/VMS.pm
        'File::Spec::Win32'     => '1.4',  #lib/File/Spec/Win32.pm
        'Filter::Simple'        => '0.78',  #lib/Filter/Simple.pm
        'Filter::Util::Call'    => '1.0601',  #lib/Filter/Util/Call.pm
        'Getopt::Long'          => '2.34',  #lib/Getopt/Long.pm
        'Getopt::Std'           => '1.05',  #lib/Getopt/Std.pm
        'Hash::Util'            => '0.05',  #lib/Hash/Util.pm
        'I18N::Collate'         => '1.00',  #lib/I18N/Collate.pm
        'I18N::LangTags'        => '0.33',  #lib/I18N/LangTags.pm
        'I18N::Langinfo'        => '0.02',  #lib/I18N/Langinfo.pm
        'I18N::LangTags::Detect'=> '1.03',  #lib/I18N/LangTags/Detect.pm
        'I18N::LangTags::List'  => '0.29',  #lib/I18N/LangTags/List.pm
        'IPC::Open2'            => '1.01',  #lib/IPC/Open2.pm
        'IPC::Open3'            => '1.0106',  #lib/IPC/Open3.pm
        'IPC::Msg'              => '1.02',  #lib/IPC/Msg.pm
        'IPC::Semaphore'        => '1.02',  #lib/IPC/Semaphore.pm
        'IPC::SysV'             => '1.04',  #lib/IPC/SysV.pm
        'Locale::Constants'     => '2.07',  #lib/Locale/Constants.pm
        'Locale::Country'       => '2.07',  #lib/Locale/Country.pm
        'Locale::Currency'      => '2.07',  #lib/Locale/Currency.pm
        'Locale::Language'      => '2.07',  #lib/Locale/Language.pm
        'Locale::Maketext'      => '1.09',  #lib/Locale/Maketext.pm
        'Locale::Script'        => '2.07',  #lib/Locale/Script.pm
        'Locale::Maketext::GutsLoader'=> undef,  #lib/Locale/Maketext/GutsLoader.pm
        'Locale::Maketext::Guts'=> undef,  #lib/Locale/Maketext/Guts.pm
        'Math::BigFloat'        => '1.44',  #lib/Math/BigFloat.pm
        'Math::BigInt'          => '1.70',  #lib/Math/BigInt.pm
        'Math::BigRat'          => '0.12',  #lib/Math/BigRat.pm
        'Math::Complex'         => '1.34',  #lib/Math/Complex.pm
        'Math::Trig'            => '1.02',  #lib/Math/Trig.pm
        'Math::BigFloat::Trace' => '0.01',  #lib/Math/BigFloat/Trace.pm
        'Math::BigInt::CalcEmu' => '0.04',  #lib/Math/BigInt/CalcEmu.pm
        'Math::BigInt::Calc'    => '0.40',  #lib/Math/BigInt/Calc.pm
        'Math::BigInt::Trace'   => '0.01',  #lib/Math/BigInt/Trace.pm
        'Memoize::AnyDBM_File'  => '0.65',  #lib/Memoize/AnyDBM_File.pm
        'Memoize::ExpireFile'   => '1.01',  #lib/Memoize/ExpireFile.pm
        'Memoize::Expire'       => '1.00',  #lib/Memoize/Expire.pm
        'Memoize::ExpireTest'   => '0.65',  #lib/Memoize/ExpireTest.pm
        'Memoize::NDBM_File'    => '0.65',  #lib/Memoize/NDBM_File.pm
        'Memoize::SDBM_File'    => '0.65',  #lib/Memoize/SDBM_File.pm
        'Memoize::Storable'     => '0.65',  #lib/Memoize/Storable.pm
        'Net::Cmd'              => '2.26',  #lib/Net/Cmd.pm
        'Net::Config'           => '1.10',  #lib/Net/Config.pm
        'Net::Domain'           => '2.19',  #lib/Net/Domain.pm
        'Net::FTP'              => '2.75',  #lib/Net/FTP.pm
        'Net::hostent'          => '1.01',  #lib/Net/hostent.pm
        'Net::netent'           => '1.00',  #lib/Net/netent.pm
        'Net::Netrc'            => '2.12',  #lib/Net/Netrc.pm
        'Net::NNTP'             => '2.23',  #lib/Net/NNTP.pm
        'Net::Ping'             => '2.31',  #lib/Net/Ping.pm
        'Net::POP3'             => '2.28',  #lib/Net/POP3.pm
        'Net::protoent'         => '1.00',  #lib/Net/protoent.pm
        'Net::servent'          => '1.01',  #lib/Net/servent.pm
        'Net::SMTP'             => '2.29',  #lib/Net/SMTP.pm
        'Net::Time'             => '2.10',  #lib/Net/Time.pm
        'Net::FTP::A'           => '1.16',  #lib/Net/FTP/A.pm
        'Net::FTP::dataconn'    => '0.11',  #lib/Net/FTP/dataconn.pm
        'Net::FTP::E'           => '0.01',  #lib/Net/FTP/E.pm
        'Net::FTP::I'           => '1.12',  #lib/Net/FTP/I.pm
        'Net::FTP::L'           => '0.01',  #lib/Net/FTP/L.pm
        'PerlIO::encoding'      => '0.07',  #lib/PerlIO/encoding.pm
        'PerlIO::scalar'        => '0.02',  #lib/PerlIO/scalar.pm
        'PerlIO::via'           => '0.02',  #lib/PerlIO/via.pm
        'PerlIO::via::QuotedPrint'=> '0.06',  #lib/PerlIO/via/QuotedPrint.pm
        'Pod::Checker'          => '1.42',  #lib/Pod/Checker.pm
        'Pod::Find'             => '0.2401',  #lib/Pod/Find.pm
        'Pod::Functions'        => '1.02',  #lib/Pod/Functions.pm
        'Pod::Html'             => '1.0502',  #lib/Pod/Html.pm
        'Pod::InputObjects'     => '1.14',  #lib/Pod/InputObjects.pm
        'Pod::LaTeX'            => '0.56',  #lib/Pod/LaTeX.pm
        'Pod::Man'              => '1.37',  #lib/Pod/Man.pm
        'Pod::ParseLink'        => '1.06',  #lib/Pod/ParseLink.pm
        'Pod::Parser'           => '1.14',  #lib/Pod/Parser.pm
        'Pod::ParseUtils'       => '1.2',  #lib/Pod/ParseUtils.pm
        'Pod::Perldoc'          => '3.13',  #lib/Pod/Perldoc.pm
        'Pod::Plainer'          => '0.01',  #lib/Pod/Plainer.pm
        'Pod::PlainText'        => '2.02',  #lib/Pod/PlainText.pm
        'Pod::Select'           => '1.13',  #lib/Pod/Select.pm
        'Pod::Text'             => '2.21',  #lib/Pod/Text.pm
        'Pod::Usage'            => '1.16',  #lib/Pod/Usage.pm
        'Pod::Perldoc::BaseTo'  => undef,  #lib/Pod/Perldoc/BaseTo.pm
        'Pod::Perldoc::GetOptsOO'=> undef,  #lib/Pod/Perldoc/GetOptsOO.pm
        'Pod::Perldoc::ToChecker'=> undef,  #lib/Pod/Perldoc/ToChecker.pm
        'Pod::Perldoc::ToMan'   => undef,  #lib/Pod/Perldoc/ToMan.pm
        'Pod::Perldoc::ToNroff' => undef,  #lib/Pod/Perldoc/ToNroff.pm
        'Pod::Perldoc::ToPod'   => undef,  #lib/Pod/Perldoc/ToPod.pm
        'Pod::Perldoc::ToRtf'   => undef,  #lib/Pod/Perldoc/ToRtf.pm
        'Pod::Perldoc::ToText'  => undef,  #lib/Pod/Perldoc/ToText.pm
        'Pod::Perldoc::ToTk'    => 'undef',  #lib/Pod/Perldoc/ToTk.pm
        'Pod::Perldoc::ToXml'   => undef,  #lib/Pod/Perldoc/ToXml.pm
        'Pod::Text::Color'      => '1.04',  #lib/Pod/Text/Color.pm
        'Pod::Text::Overstrike' => '1.1',  #lib/Pod/Text/Overstrike.pm
        'Pod::Text::Termcap'    => '1.11',  #lib/Pod/Text/Termcap.pm
        'Search::Dict'          => '1.02',  #lib/Search/Dict.pm
        'Term::ANSIColor'       => '1.08',  #lib/Term/ANSIColor.pm
        'Term::Cap'             => '1.09',  #lib/Term/Cap.pm
        'Term::Complete'        => '1.401',  #lib/Term/Complete.pm
        'Term::ReadLine'        => '1.01',  #lib/Term/ReadLine.pm
        'Test::Builder'         => '0.17',  #lib/Test/Builder.pm
        'Test::Harness'         => '2.42',  #lib/Test/Harness.pm
        'Test::More'            => '0.47',  #lib/Test/More.pm
        'Test::Simple'          => '0.47',  #lib/Test/Simple.pm
        'Test::Harness::Assert' => '0.02',  #lib/Test/Harness/Assert.pm
        'Test::Harness::Iterator'=> '0.02',  #lib/Test/Harness/Iterator.pm
        'Test::Harness::Straps' => '0.19',  #lib/Test/Harness/Straps.pm
        'Text::Abbrev'          => '1.01',  #lib/Text/Abbrev.pm
        'Text::Balanced'        => '1.95',  #lib/Text/Balanced.pm
        'Text::ParseWords'      => '3.22',  #lib/Text/ParseWords.pm
        'Text::Soundex'         => '1.01',  #lib/Text/Soundex.pm
        'Text::Tabs'            => '98.112801',  #lib/Text/Tabs.pm
        'Text::Wrap'            => '2001.09292',  #lib/Text/Wrap.pm
        'Thread::Queue'         => '2.00',  #lib/Thread/Queue.pm
        'Thread::Semaphore'     => '2.01',  #lib/Thread/Semaphore.pm
        'Tie::Array'            => '1.03',  #lib/Tie/Array.pm
        'Tie::File'             => '0.97',  #lib/Tie/File.pm
        'Tie::Handle'           => '4.1',  #lib/Tie/Handle.pm
        'Tie::Hash'             => '1.01',  #lib/Tie/Hash.pm
        'Tie::Memoize'          => '1.0',  #lib/Tie/Memoize.pm
        'Tie::RefHash'          => '1.31',  #lib/Tie/RefHash.pm
        'Tie::Scalar'           => '1.00',  #lib/Tie/Scalar.pm
        'Tie::SubstrHash'       => '1.00',  #lib/Tie/SubstrHash.pm
        'Time::gmtime'          => '1.02',  #lib/Time/gmtime.pm
        'Time::Local'           => '1.10',  #lib/Time/Local.pm
        'Time::localtime'       => '1.02',  #lib/Time/localtime.pm
        'Time::tm'              => '1.00',  #lib/Time/tm.pm
        'Time::HiRes'           => '1.59',  #lib/Time/HiRes.pm
        'Unicode'               => '4.0.1', # lib/unicore/version
        'Unicode::Collate'      => '0.40',  #lib/Unicode/Collate.pm
        'Unicode::UCD'          => '0.22',  #lib/Unicode/UCD.pm
        'Unicode::Normalize'    => '0.30',  #lib/Unicode/Normalize.pm
        'User::grent'           => '1.00',  #lib/User/grent.pm
        'User::pwent'           => '1.00',  #lib/User/pwent.pm
        'warnings::register'    => '1.00',  #lib/warnings/register.pm
        'B::Stash'              => '1.00',  #lib/B/Stash.pm
        'B::Asmdata'            => '1.01',  #lib/B/Asmdata.pm
        'B::C'                  => '1.02',  #lib/B/C.pm
        'B::Deparse'            => '0.67',  #lib/B/Deparse.pm
        'B::Debug'              => '1.01',  #lib/B/Debug.pm
        'B::Bblock'             => '1.02',  #lib/B/Bblock.pm
        'B::Assembler'          => '0.07',  #lib/B/Assembler.pm
        'B::Terse'              => '1.02',  #lib/B/Terse.pm
        'B::CC'                 => '1.00',  #lib/B/CC.pm
        'B::Concise'            => '0.61',  #lib/B/Concise.pm
        'B::Lint'               => '1.02',  #lib/B/Lint.pm
        'B::Showlex'            => '1.00',  #lib/B/Showlex.pm
        'B::Bytecode'           => '1.01',  #lib/B/Bytecode.pm
        'B::Disassembler'       => '1.03',  #lib/B/Disassembler.pm
        'B::Xref'               => '1.01',  #lib/B/Xref.pm
        'B::Stackobj'           => '1.00',  #lib/B/Stackobj.pm
        'Data::Dumper'          => '2.121',  #lib/Data/Dumper.pm
        'Encode::Alias'         => '2.00',  #lib/Encode/Alias.pm
        'Encode::Encoding'      => '2.00',  #lib/Encode/Encoding.pm
        'Encode::Guess'         => '2.00',  #lib/Encode/Guess.pm
        'Encode::Config'        => '2.00',  #lib/Encode/Config.pm
        'Encode::Encoder'       => '2.00',  #lib/Encode/Encoder.pm
        'Encode::CJKConstants'  => '2.00',  #lib/Encode/CJKConstants.pm
        'Encode::Byte'          => '2.00',  #lib/Encode/Byte.pm
        'Encode::CN'            => '2.00',  #lib/Encode/CN.pm
        'Encode::EBCDIC'        => '2.00',  #lib/Encode/EBCDIC.pm
        'Encode::JP'            => '2.00',  #lib/Encode/JP.pm
        'Encode::KR'            => '2.00',  #lib/Encode/KR.pm
        'Encode::Symbol'        => '2.00',  #lib/Encode/Symbol.pm
        'Encode::TW'            => '2.00',  #lib/Encode/TW.pm
        'Encode::Unicode'       => '2.00',  #lib/Encode/Unicode.pm
        'Encode::JP::H2Z'       => '2.00',  #lib/Encode/JP/H2Z.pm
        'Encode::JP::JIS7'      => '2.00',  #lib/Encode/JP/JIS7.pm
        'Encode::Unicode::UTF7' => '2.01',  #lib/Encode/Unicode/UTF7.pm
        'Encode::KR::2022_KR'   => '2.00',  #lib/Encode/KR/2022_KR.pm
        'Encode::MIME::Header'  => '2.00',  #lib/Encode/MIME/Header.pm
        'Encode::CN::HZ'        => '2.01',  #lib/Encode/CN/HZ.pm
        'IO::Pipe'              => '1.123',  #lib/IO/Pipe.pm
        'IO::File'              => '1.10',  #lib/IO/File.pm
        'IO::Select'            => '1.16',  #lib/IO/Select.pm
        'IO::Socket'            => '1.28',  #lib/IO/Socket.pm
        'IO::Poll'              => '0.06',  #lib/IO/Poll.pm
        'IO::Dir'               => '1.04',  #lib/IO/Dir.pm
        'IO::Handle'            => '1.24',  #lib/IO/Handle.pm
        'IO::Seekable'          => '1.09',  #lib/IO/Seekable.pm
        'IO::Socket::INET'      => '1.27',  #lib/IO/Socket/INET.pm
        'IO::Socket::UNIX'      => '1.21',  #lib/IO/Socket/UNIX.pm
        'List::Util'            => '1.14',  #lib/List/Util.pm
        'Scalar::Util'          => '1.14',  #lib/Scalar/Util.pm
        'MIME::QuotedPrint'     => '3.01',  #lib/MIME/QuotedPrint.pm
        'MIME::Base64'          => '3.01',  #lib/MIME/Base64.pm
        'Sys::Hostname'         => '1.11',  #lib/Sys/Hostname.pm
        'Sys::Syslog'           => '0.05',  #lib/Sys/Syslog.pm
        'XS::APItest'           => '0.04',  #lib/XS/APItest.pm
        'XS::Typemap'           => '0.01',  #lib/XS/Typemap.pm
        'threads::shared'       => '0.92',  #lib/threads/shared.pm
        'ExtUtils::XSSymSet'    => '1.0',  #vms/ext/XSSymSet.pm
        'JNI'                   => '0.2',  #jpl/JNI/JNI.pm
        'JPL::AutoLoader'       => undef,  #jpl/JPL/AutoLoader.pm
        'JPL::Class'            => undef,  #jpl/JPL/Class.pm
        'JPL::Compile'          => undef,  #jpl/JPL/Compile.pm
        'ODBM_File'             => '1.05',  #ext/ODBM_File/ODBM_File.pm
        'OS2::DLL'              => '1.02',  #os2/OS2/REXX/DLL/DLL.pm
        'OS2::ExtAttr'          => '0.02',  #os2/OS2/ExtAttr/ExtAttr.pm
        'OS2::PrfDB'            => '0.03',  #os2/OS2/PrfDB/PrfDB.pm
        'OS2::Process'          => '1.01',  #os2/OS2/Process/Process.pm
        'OS2::REXX'             => '1.02',  #os2/OS2/REXX/REXX.pm
        'Thread::Signal'        => '1.00', #./ext/Thread/Thread/Signal.pm
        'Thread::Specific'      => '1.00', #./ext/Thread/Thread/Specific.pm
        'VMS::DCLsym'           => '1.02',  #vms/ext/DCLsym/DCLsym.pm
        'VMS::Filespec'         => '1.11',  #vms/ext/Filespec.pm
        'VMS::Stdio'            => '2.3',  #vms/ext/Stdio/Stdio.pm
    },

    5.008006 => {
        'AnyDBM_File'           => '1.00',  #lib/AnyDBM_File.pm
        'Attribute::Handlers'   => '0.78_01',  #lib/Attribute/Handlers.pm
        'attributes'            => '0.06',  #lib/attributes.pm
        'attrs'                 => '1.01',  #lib/attrs.pm
        'AutoLoader'            => '5.60',  #lib/AutoLoader.pm
        'AutoSplit'             => '1.04',  #lib/AutoSplit.pm
        'autouse'               => '1.04',  #lib/autouse.pm
        'B'                     => '1.07',  #lib/B.pm
        'base'                  => '2.06',  #lib/base.pm
        'B::Asmdata'            => '1.01',  #lib/B/Asmdata.pm
        'B::Assembler'          => '0.07',  #lib/B/Assembler.pm
        'B::Bblock'             => '1.02',  #lib/B/Bblock.pm
        'B::Bytecode'           => '1.01',  #lib/B/Bytecode.pm
        'B::C'                  => '1.04',  #lib/B/C.pm
        'B::CC'                 => '1.00',  #lib/B/CC.pm
        'B::Concise'            => '0.64',  #lib/B/Concise.pm
        'B::Debug'              => '1.02',  #lib/B/Debug.pm
        'B::Deparse'            => '0.69',  #lib/B/Deparse.pm
        'B::Disassembler'       => '1.03',  #lib/B/Disassembler.pm
        'Benchmark'             => '1.06',  #lib/Benchmark.pm
        'bigint'                => '0.05',  #lib/bigint.pm
        'bignum'                => '0.15',  #lib/bignum.pm
        'bigrat'                => '0.06',  #lib/bigrat.pm
        'blib'                  => '1.02',  #lib/blib.pm
        'B::Lint'               => '1.03',  #lib/B/Lint.pm
        'B::Showlex'            => '1.02',  #lib/B/Showlex.pm
        'B::Stackobj'           => '1.00',  #lib/B/Stackobj.pm
        'B::Stash'              => '1.00',  #lib/B/Stash.pm
        'B::Terse'              => '1.02',  #lib/B/Terse.pm
        'B::Xref'               => '1.01',  #lib/B/Xref.pm
        'ByteLoader'            => '0.05',  #lib/ByteLoader.pm
        'bytes'                 => '1.01',  #lib/bytes.pm
        'Carp'                  => '1.03',  #lib/Carp.pm
        'Carp::Heavy'           => '1.03',  #lib/Carp/Heavy.pm
        'CGI'                   => '3.05',  #lib/CGI.pm
        'CGI::Apache'           => '1.00',  #lib/CGI/Apache.pm
        'CGI::Carp'             => '1.28',  #lib/CGI/Carp.pm
        'CGI::Cookie'           => '1.24',  #lib/CGI/Cookie.pm
        'CGI::Fast'             => '1.05',  #lib/CGI/Fast.pm
        'CGI::Pretty'           => '1.08',  #lib/CGI/Pretty.pm
        'CGI::Push'             => '1.04',  #lib/CGI/Push.pm
        'CGI::Switch'           => '1.00',  #lib/CGI/Switch.pm
        'CGI::Util'             => '1.5',  #lib/CGI/Util.pm
        'charnames'             => '1.04',  #lib/charnames.pm
        'Class::ISA'            => '0.32',  #lib/Class/ISA.pm
        'Class::Struct'         => '0.63',  #lib/Class/Struct.pm
        'Config'                => undef,  #lib/Config.pm
        'constant'              => '1.04',  #lib/constant.pm
        'CPAN'                  => '1.76_01',  #lib/CPAN.pm
        'CPAN::FirstTime'       => '1.60 ',  #lib/CPAN/FirstTime.pm
        'CPAN::Nox'             => '1.03',  #lib/CPAN/Nox.pm
        'Cwd'                   => '3.01',  #lib/Cwd.pm
        'Data::Dumper'          => '2.121_02',  #lib/Data/Dumper.pm
        'DB'                    => '1.0',  #lib/DB.pm
        'DB_File'               => '1.810',  #lib/DB_File.pm
        'DBM_Filter'            => '0.01',  #lib/DBM_Filter.pm
        'DBM_Filter::compress'  => '0.01',  #lib/DBM_Filter/compress.pm
        'DBM_Filter::encode'    => '0.01',  #lib/DBM_Filter/encode.pm
        'DBM_Filter::int32'     => '0.01',  #lib/DBM_Filter/int32.pm
        'DBM_Filter::null'      => '0.01',  #lib/DBM_Filter/null.pm
        'DBM_Filter::utf8'      => '0.01',  #lib/DBM_Filter/utf8.pm
        'Devel::DProf'          => '20030813.00',  #lib/Devel/DProf.pm
        'Devel::Peek'           => '1.02',  #lib/Devel/Peek.pm
        'Devel::PPPort'         => '3.03',  #lib/Devel/PPPort.pm
        'Devel::SelfStubber'    => '1.03',  #lib/Devel/SelfStubber.pm
        'diagnostics'           => '1.14',  #lib/diagnostics.pm
        'Digest'                => '1.08',  #lib/Digest.pm
        'Digest::base'          => '1.00',  #lib/Digest/base.pm
        'Digest::MD5'           => '2.33',  #lib/Digest/MD5.pm
        'DirHandle'             => '1.00',  #lib/DirHandle.pm
        'Dumpvalue'             => '1.11',  #lib/Dumpvalue.pm
        'DynaLoader'            => '1.05',  #lib/DynaLoader.pm
        'Encode'                => '2.08',  #lib/Encode.pm
        'Encode::Alias'         => '2.02',  #lib/Encode/Alias.pm
        'Encode::Byte'          => '2.00',  #lib/Encode/Byte.pm
        'Encode::CJKConstants'  => '2.00',  #lib/Encode/CJKConstants.pm
        'Encode::CN'            => '2.00',  #lib/Encode/CN.pm
        'Encode::CN::HZ'        => '2.01',  #lib/Encode/CN/HZ.pm
        'Encode::Config'        => '2.00',  #lib/Encode/Config.pm
        'Encode::EBCDIC'        => '2.00',  #lib/Encode/EBCDIC.pm
        'Encode::Encoder'       => '2.00',  #lib/Encode/Encoder.pm
        'Encode::Encoding'      => '2.02',  #lib/Encode/Encoding.pm
        'Encode::Guess'         => '2.00',  #lib/Encode/Guess.pm
        'Encode::JP'            => '2.01',  #lib/Encode/JP.pm
        'Encode::JP::H2Z'       => '2.00',  #lib/Encode/JP/H2Z.pm
        'Encode::JP::JIS7'      => '2.00',  #lib/Encode/JP/JIS7.pm
        'Encode::KR'            => '2.00',  #lib/Encode/KR.pm
        'Encode::KR::2022_KR'   => '2.00',  #lib/Encode/KR/2022_KR.pm
        'Encode::MIME::Header'  => '2.00',  #lib/Encode/MIME/Header.pm
        'Encode::Symbol'        => '2.00',  #lib/Encode/Symbol.pm
        'Encode::TW'            => '2.00',  #lib/Encode/TW.pm
        'Encode::Unicode'       => '2.02',  #lib/Encode/Unicode.pm
        'Encode::Unicode::UTF7' => '2.01',  #lib/Encode/Unicode/UTF7.pm
        'encoding'              => '2.01',  #lib/encoding.pm
        'English'               => '1.01',  #lib/English.pm
        'Env'                   => '1.00',  #lib/Env.pm
        'Errno'                 => '1.09_00',  #lib/Errno.pm
        'Exporter'              => '5.58',  #lib/Exporter.pm
        'Exporter::Heavy'       => '5.58',  #lib/Exporter/Heavy.pm
        'ExtUtils::Command'     => '1.05',  #lib/ExtUtils/Command.pm
        'ExtUtils::Command::MM' => '0.03',  #lib/ExtUtils/Command/MM.pm
        'ExtUtils::Constant'    => '0.1401',  #lib/ExtUtils/Constant.pm
        'ExtUtils::Embed'       => '1.250601',  #lib/ExtUtils/Embed.pm
        'ExtUtils::Install'     => '1.32',  #lib/ExtUtils/Install.pm
        'ExtUtils::Installed'   => '0.08',  #lib/ExtUtils/Installed.pm
        'ExtUtils::Liblist'     => '1.01',  #lib/ExtUtils/Liblist.pm
        'ExtUtils::Liblist::Kid'=> '1.3001',  #lib/ExtUtils/Liblist/Kid.pm
        'ExtUtils::MakeMaker'   => '6.17',  #lib/ExtUtils/MakeMaker.pm
        'ExtUtils::MakeMaker::bytes'=> '0.01',  #lib/ExtUtils/MakeMaker/bytes.pm
        'ExtUtils::MakeMaker::vmsish'=> '0.01',  #lib/ExtUtils/MakeMaker/vmsish.pm
        'ExtUtils::Manifest'    => '1.42',  #lib/ExtUtils/Manifest.pm
        'ExtUtils::Miniperl'    => undef,  #lib/ExtUtils/Miniperl.pm
        'ExtUtils::Mkbootstrap' => '1.15',  #lib/ExtUtils/Mkbootstrap.pm
        'ExtUtils::Mksymlists'  => '1.19',  #lib/ExtUtils/Mksymlists.pm
        'ExtUtils::MM'          => '0.04',  #lib/ExtUtils/MM.pm
        'ExtUtils::MM_Any'      => '0.07',  #lib/ExtUtils/MM_Any.pm
        'ExtUtils::MM_BeOS'     => '1.04',  #lib/ExtUtils/MM_BeOS.pm
        'ExtUtils::MM_Cygwin'   => '1.06',  #lib/ExtUtils/MM_Cygwin.pm
        'ExtUtils::MM_DOS'      => '0.02',  #lib/ExtUtils/MM_DOS.pm
        'ExtUtils::MM_MacOS'    => '1.07',  #lib/ExtUtils/MM_MacOS.pm
        'ExtUtils::MM_NW5'      => '2.07_02',  #lib/ExtUtils/MM_NW5.pm
        'ExtUtils::MM_OS2'      => '1.04',  #lib/ExtUtils/MM_OS2.pm
        'ExtUtils::MM_Unix'     => '1.42',  #lib/ExtUtils/MM_Unix.pm
        'ExtUtils::MM_UWIN'     => '0.02',  #lib/ExtUtils/MM_UWIN.pm
        'ExtUtils::MM_VMS'      => '5.70',  #lib/ExtUtils/MM_VMS.pm
        'ExtUtils::MM_Win32'    => '1.09',  #lib/ExtUtils/MM_Win32.pm
        'ExtUtils::MM_Win95'    => '0.0301',  #lib/ExtUtils/MM_Win95.pm
        'ExtUtils::MY'          => '0.01',  #lib/ExtUtils/MY.pm
        'ExtUtils::Packlist'    => '0.04',  #lib/ExtUtils/Packlist.pm
        'ExtUtils::testlib'     => '1.15',  #lib/ExtUtils/testlib.pm
        'ExtUtils::XSSymSet'    => '1.0',  #vms/ext/XSSymSet.pm
        'Fatal'                 => '1.03',  #lib/Fatal.pm
        'Fcntl'                 => '1.05',  #lib/Fcntl.pm
        'fields'                => '2.03',  #lib/fields.pm
        'File::Basename'        => '2.73',  #lib/File/Basename.pm
        'FileCache'             => '1.04_01',  #lib/FileCache.pm
        'File::CheckTree'       => '4.3',  #lib/File/CheckTree.pm
        'File::Compare'         => '1.1003',  #lib/File/Compare.pm
        'File::Copy'            => '2.08',  #lib/File/Copy.pm
        'File::DosGlob'         => '1.00',  #lib/File/DosGlob.pm
        'File::Find'            => '1.07',  #lib/File/Find.pm
        'File::Glob'            => '1.03',  #lib/File/Glob.pm
        'FileHandle'            => '2.01',  #lib/FileHandle.pm
        'File::Path'            => '1.06',  #lib/File/Path.pm
        'File::Spec'            => '3.01',  #lib/File/Spec.pm
        'File::Spec::Cygwin'    => '1.1',  #lib/File/Spec/Cygwin.pm
        'File::Spec::Epoc'      => '1.1',  #lib/File/Spec/Epoc.pm
        'File::Spec::Functions' => '1.3',  #lib/File/Spec/Functions.pm
        'File::Spec::Mac'       => '1.4',  #lib/File/Spec/Mac.pm
        'File::Spec::OS2'       => '1.2',  #lib/File/Spec/OS2.pm
        'File::Spec::Unix'      => '1.5',  #lib/File/Spec/Unix.pm
        'File::Spec::VMS'       => '1.4',  #lib/File/Spec/VMS.pm
        'File::Spec::Win32'     => '1.5',  #lib/File/Spec/Win32.pm
        'File::stat'            => '1.00',  #lib/File/stat.pm
        'File::Temp'            => '0.14',  #lib/File/Temp.pm
        'filetest'              => '1.01',  #lib/filetest.pm
        'Filter::Simple'        => '0.78',  #lib/Filter/Simple.pm
        'Filter::Util::Call'    => '1.0601',  #lib/Filter/Util/Call.pm
        'FindBin'               => '1.44',  #lib/FindBin.pm
        'GDBM_File'             => '1.07',  #lib/GDBM_File.pm
        'Getopt::Long'          => '2.34',  #lib/Getopt/Long.pm
        'Getopt::Std'           => '1.05',  #lib/Getopt/Std.pm
        'Hash::Util'            => '0.05',  #lib/Hash/Util.pm
        'I18N::Collate'         => '1.00',  #lib/I18N/Collate.pm
        'I18N::Langinfo'        => '0.02',  #lib/I18N/Langinfo.pm
        'I18N::LangTags'        => '0.35',  #lib/I18N/LangTags.pm
        'I18N::LangTags::Detect'=> '1.03',  #lib/I18N/LangTags/Detect.pm
        'I18N::LangTags::List'  => '0.35',  #lib/I18N/LangTags/List.pm
        'if'                    => '0.03',  #lib/if.pm
        'integer'               => '1.00',  #lib/integer.pm
        'IO'                    => '1.21',  #lib/IO.pm
        'IO::Dir'               => '1.04',  #lib/IO/Dir.pm
        'IO::File'              => '1.10',  #lib/IO/File.pm
        'IO::Handle'            => '1.24',  #lib/IO/Handle.pm
        'IO::Pipe'              => '1.123',  #lib/IO/Pipe.pm
        'IO::Poll'              => '0.06',  #lib/IO/Poll.pm
        'IO::Seekable'          => '1.09',  #lib/IO/Seekable.pm
        'IO::Select'            => '1.16',  #lib/IO/Select.pm
        'IO::Socket'            => '1.28',  #lib/IO/Socket.pm
        'IO::Socket::INET'      => '1.27',  #lib/IO/Socket/INET.pm
        'IO::Socket::UNIX'      => '1.21',  #lib/IO/Socket/UNIX.pm
        'IPC::Msg'              => '1.02',  #lib/IPC/Msg.pm
        'IPC::Open2'            => '1.01',  #lib/IPC/Open2.pm
        'IPC::Open3'            => '1.0106',  #lib/IPC/Open3.pm
        'IPC::Semaphore'        => '1.02',  #lib/IPC/Semaphore.pm
        'IPC::SysV'             => '1.04',  #lib/IPC/SysV.pm
        'JNI'                   => '0.2',  #jpl/JNI/JNI.pm
        'JPL::AutoLoader'       => undef,  #jpl/JPL/AutoLoader.pm
        'JPL::Class'            => undef,  #jpl/JPL/Class.pm
        'JPL::Compile'          => undef,  #jpl/JPL/Compile.pm
        'less'                  => '0.01',  #lib/less.pm
        'lib'                   => '0.5565',  #lib/lib.pm
        'List::Util'            => '1.14',  #lib/List/Util.pm
        'locale'                => '1.00',  #lib/locale.pm
        'Locale::Constants'     => '2.07',  #lib/Locale/Constants.pm
        'Locale::Country'       => '2.07',  #lib/Locale/Country.pm
        'Locale::Currency'      => '2.07',  #lib/Locale/Currency.pm
        'Locale::Language'      => '2.07',  #lib/Locale/Language.pm
        'Locale::Maketext'      => '1.09',  #lib/Locale/Maketext.pm
        'Locale::Maketext::GutsLoader'=> undef,  #lib/Locale/Maketext/GutsLoader.pm
        'Locale::Maketext::Guts'=> undef,  #lib/Locale/Maketext/Guts.pm
        'Locale::Script'        => '2.07',  #lib/Locale/Script.pm
        'Math::BigFloat'        => '1.47',  #lib/Math/BigFloat.pm
        'Math::BigFloat::Trace' => '0.01',  #lib/Math/BigFloat/Trace.pm
        'Math::BigInt'          => '1.73',  #lib/Math/BigInt.pm
        'Math::BigInt::Calc'    => '0.43',  #lib/Math/BigInt/Calc.pm
        'Math::BigInt::CalcEmu' => '0.04',  #lib/Math/BigInt/CalcEmu.pm
        'Math::BigInt::Trace'   => '0.01',  #lib/Math/BigInt/Trace.pm
        'Math::BigRat'          => '0.13',  #lib/Math/BigRat.pm
        'Math::Complex'         => '1.34',  #lib/Math/Complex.pm
        'Math::Trig'            => '1.02',  #lib/Math/Trig.pm
        'Memoize'               => '1.01',  #lib/Memoize.pm
        'Memoize::AnyDBM_File'  => '0.65',  #lib/Memoize/AnyDBM_File.pm
        'Memoize::Expire'       => '1.00',  #lib/Memoize/Expire.pm
        'Memoize::ExpireFile'   => '1.01',  #lib/Memoize/ExpireFile.pm
        'Memoize::ExpireTest'   => '0.65',  #lib/Memoize/ExpireTest.pm
        'Memoize::NDBM_File'    => '0.65',  #lib/Memoize/NDBM_File.pm
        'Memoize::SDBM_File'    => '0.65',  #lib/Memoize/SDBM_File.pm
        'Memoize::Storable'     => '0.65',  #lib/Memoize/Storable.pm
        'MIME::Base64'          => '3.05',  #lib/MIME/Base64.pm
        'MIME::QuotedPrint'     => '3.03',  #lib/MIME/QuotedPrint.pm
        'NDBM_File'             => '1.05',  #lib/NDBM_File.pm
        'Net::Cmd'              => '2.26',  #lib/Net/Cmd.pm
        'Net::Config'           => '1.10',  #lib/Net/Config.pm
        'Net::Domain'           => '2.19',  #lib/Net/Domain.pm
        'Net::FTP'              => '2.75',  #lib/Net/FTP.pm
        'Net::FTP::A'           => '1.16',  #lib/Net/FTP/A.pm
        'Net::FTP::dataconn'    => '0.11',  #lib/Net/FTP/dataconn.pm
        'Net::FTP::E'           => '0.01',  #lib/Net/FTP/E.pm
        'Net::FTP::I'           => '1.12',  #lib/Net/FTP/I.pm
        'Net::FTP::L'           => '0.01',  #lib/Net/FTP/L.pm
        'Net::hostent'          => '1.01',  #lib/Net/hostent.pm
        'Net::netent'           => '1.00',  #lib/Net/netent.pm
        'Net::Netrc'            => '2.12',  #lib/Net/Netrc.pm
        'Net::NNTP'             => '2.23',  #lib/Net/NNTP.pm
        'Net::Ping'             => '2.31',  #lib/Net/Ping.pm
        'Net::POP3'             => '2.28',  #lib/Net/POP3.pm
        'Net::protoent'         => '1.00',  #lib/Net/protoent.pm
        'Net::servent'          => '1.01',  #lib/Net/servent.pm
        'Net::SMTP'             => '2.29',  #lib/Net/SMTP.pm
        'Net::Time'             => '2.10',  #lib/Net/Time.pm
        'NEXT'                  => '0.60',  #lib/NEXT.pm
        'O'                     => '1.00',  #lib/O.pm
        'ODBM_File'             => '1.05',  #ext/ODBM_File/ODBM_File.pm
        'Opcode'                => '1.05',  #lib/Opcode.pm
        'open'                  => '1.04',  #lib/open.pm
        'ops'                   => '1.00',  #lib/ops.pm
        'OS2::DLL'              => '1.02',  #os2/OS2/REXX/DLL/DLL.pm
        'OS2::ExtAttr'          => '0.02',  #os2/OS2/ExtAttr/ExtAttr.pm
        'OS2::PrfDB'            => '0.03',  #os2/OS2/PrfDB/PrfDB.pm
        'OS2::Process'          => '1.01',  #os2/OS2/Process/Process.pm
        'OS2::REXX'             => '1.02',  #os2/OS2/REXX/REXX.pm
        'overload'              => '1.02',  #lib/overload.pm
        'PerlIO'                => '1.03',  #lib/PerlIO.pm
        'PerlIO::encoding'      => '0.07',  #lib/PerlIO/encoding.pm
        'PerlIO::scalar'        => '0.02',  #lib/PerlIO/scalar.pm
        'PerlIO::via'           => '0.02',  #lib/PerlIO/via.pm
        'PerlIO::via::QuotedPrint'=> '0.06',  #lib/PerlIO/via/QuotedPrint.pm
        'Pod::Checker'          => '1.42',  #lib/Pod/Checker.pm
        'Pod::Find'             => '0.2401',  #lib/Pod/Find.pm
        'Pod::Functions'        => '1.02',  #lib/Pod/Functions.pm
        'Pod::Html'             => '1.0502',  #lib/Pod/Html.pm
        'Pod::InputObjects'     => '1.14',  #lib/Pod/InputObjects.pm
        'Pod::LaTeX'            => '0.56',  #lib/Pod/LaTeX.pm
        'Pod::Man'              => '1.37',  #lib/Pod/Man.pm
        'Pod::ParseLink'        => '1.06',  #lib/Pod/ParseLink.pm
        'Pod::Parser'           => '1.14',  #lib/Pod/Parser.pm
        'Pod::ParseUtils'       => '1.2',  #lib/Pod/ParseUtils.pm
        'Pod::Perldoc'          => '3.13',  #lib/Pod/Perldoc.pm
        'Pod::Perldoc::BaseTo'  => undef,  #lib/Pod/Perldoc/BaseTo.pm
        'Pod::Perldoc::GetOptsOO'=> undef,  #lib/Pod/Perldoc/GetOptsOO.pm
        'Pod::Perldoc::ToChecker'=> undef,  #lib/Pod/Perldoc/ToChecker.pm
        'Pod::Perldoc::ToMan'   => undef,  #lib/Pod/Perldoc/ToMan.pm
        'Pod::Perldoc::ToNroff' => undef,  #lib/Pod/Perldoc/ToNroff.pm
        'Pod::Perldoc::ToPod'   => undef,  #lib/Pod/Perldoc/ToPod.pm
        'Pod::Perldoc::ToRtf'   => undef,  #lib/Pod/Perldoc/ToRtf.pm
        'Pod::Perldoc::ToText'  => undef,  #lib/Pod/Perldoc/ToText.pm
        'Pod::Perldoc::ToTk'    => 'undef',  #lib/Pod/Perldoc/ToTk.pm
        'Pod::Perldoc::ToXml'   => undef,  #lib/Pod/Perldoc/ToXml.pm
        'Pod::Plainer'          => '0.01',  #lib/Pod/Plainer.pm
        'Pod::PlainText'        => '2.02',  #lib/Pod/PlainText.pm
        'Pod::Select'           => '1.13',  #lib/Pod/Select.pm
        'Pod::Text'             => '2.21',  #lib/Pod/Text.pm
        'Pod::Text::Color'      => '1.04',  #lib/Pod/Text/Color.pm
        'Pod::Text::Overstrike' => '1.1',  #lib/Pod/Text/Overstrike.pm
        'Pod::Text::Termcap'    => '1.11',  #lib/Pod/Text/Termcap.pm
        'Pod::Usage'            => '1.16',  #lib/Pod/Usage.pm
        'POSIX'                 => '1.08',  #lib/POSIX.pm
        're'                    => '0.04',  #lib/re.pm
        'Safe'                  => '2.11',  #lib/Safe.pm
        'Scalar::Util'          => '1.14',  #lib/Scalar/Util.pm
        'SDBM_File'             => '1.04',  #lib/SDBM_File.pm
        'Search::Dict'          => '1.02',  #lib/Search/Dict.pm
        'SelectSaver'           => '1.00',  #lib/SelectSaver.pm
        'SelfLoader'            => '1.0904',  #lib/SelfLoader.pm
        'Shell'                 => '0.6',  #lib/Shell.pm
        'sigtrap'               => '1.02',  #lib/sigtrap.pm
        'Socket'                => '1.77',  #lib/Socket.pm
        'sort'                  => '1.02',  #lib/sort.pm
        'Storable'              => '2.13',  #lib/Storable.pm
        'strict'                => '1.03',  #lib/strict.pm
        'subs'                  => '1.00',  #lib/subs.pm
        'Switch'                => '2.10',  #lib/Switch.pm
        'Symbol'                => '1.05',  #lib/Symbol.pm
        'Sys::Hostname'         => '1.11',  #lib/Sys/Hostname.pm
        'Sys::Syslog'           => '0.05',  #lib/Sys/Syslog.pm
        'Term::ANSIColor'       => '1.08',  #lib/Term/ANSIColor.pm
        'Term::Cap'             => '1.09',  #lib/Term/Cap.pm
        'Term::Complete'        => '1.401',  #lib/Term/Complete.pm
        'Term::ReadLine'        => '1.01',  #lib/Term/ReadLine.pm
        'Test'                  => '1.25',  #lib/Test.pm
        'Test::Builder'         => '0.17',  #lib/Test/Builder.pm
        'Test::Harness'         => '2.42',  #lib/Test/Harness.pm
        'Test::Harness::Assert' => '0.02',  #lib/Test/Harness/Assert.pm
        'Test::Harness::Iterator'=> '0.02',  #lib/Test/Harness/Iterator.pm
        'Test::Harness::Straps' => '0.19',  #lib/Test/Harness/Straps.pm
        'Test::More'            => '0.47',  #lib/Test/More.pm
        'Test::Simple'          => '0.47',  #lib/Test/Simple.pm
        'Text::Abbrev'          => '1.01',  #lib/Text/Abbrev.pm
        'Text::Balanced'        => '1.95',  #lib/Text/Balanced.pm
        'Text::ParseWords'      => '3.23',  #lib/Text/ParseWords.pm
        'Text::Soundex'         => '1.01',  #lib/Text/Soundex.pm
        'Text::Tabs'            => '98.112801',  #lib/Text/Tabs.pm
        'Text::Wrap'            => '2001.09292',  #lib/Text/Wrap.pm
        'Thread'                => '2.00',  #lib/Thread.pm
        'Thread::Queue'         => '2.00',  #lib/Thread/Queue.pm
        'threads'               => '1.05',  #lib/threads.pm
        'Thread::Semaphore'     => '2.01',  #lib/Thread/Semaphore.pm
        'Thread::Signal'        => '1.00',  #ext/Thread/Thread/Signal.pm
        'Thread::Specific'      => '1.00',  #ext/Thread/Thread/Specific.pm
        'threads::shared'       => '0.92',  #lib/threads/shared.pm
        'Tie::Array'            => '1.03',  #lib/Tie/Array.pm
        'Tie::File'             => '0.97',  #lib/Tie/File.pm
        'Tie::Handle'           => '4.1',  #lib/Tie/Handle.pm
        'Tie::Hash'             => '1.01',  #lib/Tie/Hash.pm
        'Tie::Memoize'          => '1.0',  #lib/Tie/Memoize.pm
        'Tie::RefHash'          => '1.31',  #lib/Tie/RefHash.pm
        'Tie::Scalar'           => '1.00',  #lib/Tie/Scalar.pm
        'Tie::SubstrHash'       => '1.00',  #lib/Tie/SubstrHash.pm
        'Time::gmtime'          => '1.02',  #lib/Time/gmtime.pm
        'Time::HiRes'           => '1.65',  #lib/Time/HiRes.pm
        'Time::Local'           => '1.10',  #lib/Time/Local.pm
        'Time::localtime'       => '1.02',  #lib/Time/localtime.pm
        'Time::tm'              => '1.00',  #lib/Time/tm.pm
        'Unicode::Collate'      => '0.40',  #lib/Unicode/Collate.pm
        'Unicode::Normalize'    => '0.30',  #lib/Unicode/Normalize.pm
        'Unicode::UCD'          => '0.22',  #lib/Unicode/UCD.pm
        'UNIVERSAL'             => '1.01',  #lib/UNIVERSAL.pm
        'Unicode'               => '4.0.1', # lib/unicore/version
        'User::grent'           => '1.00',  #lib/User/grent.pm
        'User::pwent'           => '1.00',  #lib/User/pwent.pm
        'utf8'                  => '1.04',  #lib/utf8.pm
        'vars'                  => '1.01',  #lib/vars.pm
        'VMS::DCLsym'           => '1.02',  #vms/ext/DCLsym/DCLsym.pm
        'VMS::Filespec'         => '1.11',  #vms/ext/Filespec.pm
        'vmsish'                => '1.01',  #lib/vmsish.pm
        'VMS::Stdio'            => '2.3',  #vms/ext/Stdio/Stdio.pm
        'warnings'              => '1.03',  #lib/warnings.pm
        'warnings::register'    => '1.00',  #lib/warnings/register.pm
        'XS::APItest'           => '0.05',  #lib/XS/APItest.pm
        'XSLoader'              => '0.02',  #lib/XSLoader.pm
        'XS::Typemap'           => '0.01',  #lib/XS/Typemap.pm
    },

    5.009002 => {
	'AnyDBM_File'           => '1.00',
	'Attribute::Handlers'   => '0.78_01',
	'AutoLoader'            => '5.60',
	'AutoSplit'             => '1.04',
	'B'                     => '1.07',
	'B::Asmdata'            => '1.01',
	'B::Assembler'          => '0.07',
	'B::Bblock'             => '1.02',
	'B::Bytecode'           => '1.01',
	'B::C'                  => '1.04',
	'B::CC'                 => '1.00',
	'B::Concise'            => '0.64',
	'B::Debug'              => '1.02',
	'B::Deparse'            => '0.69',
	'B::Disassembler'       => '1.03',
	'B::Lint'               => '1.03',
	'B::Showlex'            => '1.02',
	'B::Stackobj'           => '1.00',
	'B::Stash'              => '1.00',
	'B::Terse'              => '1.02',
	'B::Xref'               => '1.01',
	'Benchmark'             => '1.07',
	'ByteLoader'            => '0.05',
	'CGI'                   => '3.07',
	'CGI::Apache'           => '1.00',
	'CGI::Carp'             => '1.29',
	'CGI::Cookie'           => '1.25',
	'CGI::Fast'             => '1.05',
	'CGI::Pretty'           => '1.08',
	'CGI::Push'             => '1.04',
	'CGI::Switch'           => '1.00',
	'CGI::Util'             => '1.5',
	'CPAN'                  => '1.76_01',
	'CPAN::FirstTime'       => '1.60 ',
	'CPAN::Nox'             => '1.03',
	'Carp'                  => '1.04',
	'Carp::Heavy'           => '1.04',
	'Class::ISA'            => '0.33',
	'Class::Struct'         => '0.63',
        'Config'                => undef,
	'Config::Extensions'    => '0.01',
	'Cwd'                   => '3.05',
	'DB'                    => '1.0',
	'DBM_Filter'            => '0.01',
	'DBM_Filter::compress'  => '0.01',
	'DBM_Filter::encode'    => '0.01',
	'DBM_Filter::int32'     => '0.01',
	'DBM_Filter::null'      => '0.01',
	'DBM_Filter::utf8'      => '0.01',
	'DB_File'               => '1.811',
	'DCLsym'                => '1.02',
	'Data::Dumper'          => '2.121_04',
	'Devel::DProf'          => '20030813.00',
	'Devel::PPPort'         => '3.06',
	'Devel::Peek'           => '1.02',
	'Devel::SelfStubber'    => '1.03',
	'Digest'                => '1.10',
	'Digest::MD5'           => '2.33',
	'Digest::base'          => '1.00',
	'Digest::file'          => '0.01',
	'DirHandle'             => '1.00',
	'Dumpvalue'             => '1.11',
	'DynaLoader'            => '1.07',
	'Encode'                => '2.09',
	'Encode::Alias'         => '2.02',
	'Encode::Byte'          => '2.00',
	'Encode::CJKConstants'  => '2.00',
	'Encode::CN'            => '2.00',
	'Encode::CN::HZ'        => '2.01',
	'Encode::Config'        => '2.00',
	'Encode::EBCDIC'        => '2.00',
	'Encode::Encoder'       => '2.00',
	'Encode::Encoding'      => '2.02',
	'Encode::Guess'         => '2.00',
	'Encode::JP'            => '2.01',
	'Encode::JP::H2Z'       => '2.00',
	'Encode::JP::JIS7'      => '2.00',
	'Encode::KR'            => '2.00',
	'Encode::KR::2022_KR'   => '2.00',
	'Encode::MIME::Header'  => '2.00',
	'Encode::Symbol'        => '2.00',
	'Encode::TW'            => '2.00',
	'Encode::Unicode'       => '2.02',
	'Encode::Unicode::UTF7' => '2.01',
	'English'               => '1.03',
	'Env'                   => '1.00',
	'Errno'                 => '1.09_01',
	'Exporter'              => '5.59',
	'Exporter::Heavy'       => '5.59',
	'ExtUtils::Command'     => '1.07',
	'ExtUtils::Command::MM' => '0.03_01',
	'ExtUtils::Constant'    => '0.16',
	'ExtUtils::Constant::Base'=> '0.01',
	'ExtUtils::Constant::Utils'=> '0.01',
	'ExtUtils::Constant::XS'=> '0.01',
	'ExtUtils::Embed'       => '1.26',
	'ExtUtils::Install'     => '1.32',
	'ExtUtils::Installed'   => '0.08',
	'ExtUtils::Liblist'     => '1.01',
	'ExtUtils::Liblist::Kid'=> '1.3',
	'ExtUtils::MM'          => '0.04',
	'ExtUtils::MM_Any'      => '0.10',
	'ExtUtils::MM_BeOS'     => '1.04',
	'ExtUtils::MM_Cygwin'   => '1.07',
	'ExtUtils::MM_DOS'      => '0.02',
	'ExtUtils::MM_MacOS'    => '1.08',
	'ExtUtils::MM_NW5'      => '2.07',
	'ExtUtils::MM_OS2'      => '1.04',
	'ExtUtils::MM_UWIN'     => '0.02',
	'ExtUtils::MM_Unix'     => '1.46_01',
	'ExtUtils::MM_VMS'      => '5.71',
	'ExtUtils::MM_Win32'    => '1.10',
	'ExtUtils::MM_Win95'    => '0.03',
	'ExtUtils::MY'          => '0.01',
	'ExtUtils::MakeMaker'   => '6.25',
	'ExtUtils::MakeMaker::bytes'=> '0.01',
	'ExtUtils::MakeMaker::vmsish'=> '0.01',
	'ExtUtils::Manifest'    => '1.44',
        'ExtUtils::Miniperl'    => undef,
	'ExtUtils::Mkbootstrap' => '1.15',
	'ExtUtils::Mksymlists'  => '1.19',
	'ExtUtils::Packlist'    => '0.04',
	'ExtUtils::testlib'     => '1.15',
	'Fatal'                 => '1.04',
	'Fcntl'                 => '1.05',
	'File::Basename'        => '2.73',
	'File::CheckTree'       => '4.3',
	'File::Compare'         => '1.1003',
	'File::Copy'            => '2.08',
	'File::DosGlob'         => '1.00',
	'File::Find'            => '1.09',
	'File::Glob'            => '1.04',
	'File::Path'            => '1.06',
	'File::Spec'            => '3.05',
	'File::Spec::Cygwin'    => '1.1',
	'File::Spec::Epoc'      => '1.1',
	'File::Spec::Functions' => '1.3',
	'File::Spec::Mac'       => '1.4',
	'File::Spec::OS2'       => '1.2',
	'File::Spec::Unix'      => '1.5',
	'File::Spec::VMS'       => '1.4',
	'File::Spec::Win32'     => '1.5',
	'File::Temp'            => '0.16',
	'File::stat'            => '1.00',
	'FileCache'             => '1.04_01',
	'FileHandle'            => '2.01',
	'VMS::Filespec'         => '1.11',
	'Filter::Simple'        => '0.78',
	'Filter::Util::Call'    => '1.0601',
	'FindBin'               => '1.44',
	'GDBM_File'             => '1.07',
	'Getopt::Long'          => '2.3401',
	'Getopt::Std'           => '1.05',
	'Hash::Util'            => '0.05',
	'I18N::Collate'         => '1.00',
	'I18N::LangTags'        => '0.35',
	'I18N::LangTags::Detect'=> '1.03',
	'I18N::LangTags::List'  => '0.35',
	'I18N::Langinfo'        => '0.02',
	'IO'                    => '1.21',
	'IO::Dir'               => '1.04',
	'IO::File'              => '1.10',
	'IO::Handle'            => '1.24',
	'IO::Pipe'              => '1.123',
	'IO::Poll'              => '0.06',
	'IO::Seekable'          => '1.09',
	'IO::Select'            => '1.16',
	'IO::Socket'            => '1.28',
	'IO::Socket::INET'      => '1.27',
	'IO::Socket::UNIX'      => '1.21',
	'IPC::Msg'              => '1.02',
	'IPC::Open2'            => '1.01',
	'IPC::Open3'            => '1.0106',
	'IPC::Semaphore'        => '1.02',
	'IPC::SysV'             => '1.04',
	'List::Util'            => '1.14',
	'Locale::Constants'     => '2.07',
	'Locale::Country'       => '2.07',
	'Locale::Currency'      => '2.07',
	'Locale::Language'      => '2.07',
	'Locale::Maketext'      => '1.09',
	'Locale::Maketext::Guts'=> undef,
	'Locale::Maketext::GutsLoader'=> undef,
	'Locale::Script'        => '2.07',
	'MIME::Base64'          => '3.05',
	'MIME::QuotedPrint'     => '3.03',
	'Math::BigFloat'        => '1.49',
	'Math::BigFloat::Trace' => '0.01',
	'Math::BigInt'          => '1.75',
	'Math::BigInt::Calc'    => '0.45',
	'Math::BigInt::CalcEmu' => '0.05',
	'Math::BigInt::Trace'   => '0.01',
	'Math::BigRat'          => '0.14',
	'Math::Complex'         => '1.34',
	'Math::Trig'            => '1.02',
	'Memoize'               => '1.01_01',
	'Memoize::AnyDBM_File'  => '0.65',
	'Memoize::Expire'       => '1.00',
	'Memoize::ExpireFile'   => '1.01',
	'Memoize::ExpireTest'   => '0.65',
	'Memoize::NDBM_File'    => '0.65',
	'Memoize::SDBM_File'    => '0.65',
	'Memoize::Storable'     => '0.65',
	'Module::CoreList'      => '1.99',
	'NDBM_File'             => '1.05',
	'NEXT'                  => '0.60_01',
	'Net::Cmd'              => '2.26',
	'Net::Config'           => '1.10',
	'Net::Domain'           => '2.19',
	'Net::FTP'              => '2.75',
	'Net::FTP::A'           => '1.16',
	'Net::FTP::E'           => '0.01',
	'Net::FTP::I'           => '1.12',
	'Net::FTP::L'           => '0.01',
	'Net::FTP::dataconn'    => '0.11',
	'Net::NNTP'             => '2.23',
	'Net::Netrc'            => '2.12',
	'Net::POP3'             => '2.28',
	'Net::Ping'             => '2.31',
	'Net::SMTP'             => '2.29',
	'Net::Time'             => '2.10',
	'Net::hostent'          => '1.01',
	'Net::netent'           => '1.00',
	'Net::protoent'         => '1.00',
	'Net::servent'          => '1.01',
	'O'                     => '1.00',
	'ODBM_File'             => '1.05',
	'Opcode'                => '1.06',
	'POSIX'                 => '1.08',
	'PerlIO'                => '1.03',
	'PerlIO::encoding'      => '0.07',
	'PerlIO::scalar'        => '0.02',
	'PerlIO::via'           => '0.02',
	'PerlIO::via::QuotedPrint'=> '0.06',
	'Pod::Checker'          => '1.42',
	'Pod::Find'             => '1.3',
	'Pod::Functions'        => '1.02',
	'Pod::Html'             => '1.0502',
	'Pod::InputObjects'     => '1.3',
	'Pod::LaTeX'            => '0.58',
	'Pod::Man'              => '1.37',
	'Pod::ParseLink'        => '1.06',
	'Pod::ParseUtils'       => '1.3',
	'Pod::Parser'           => '1.3',
	'Pod::Perldoc'          => '3.14',
	'Pod::Perldoc::BaseTo'  => undef,
	'Pod::Perldoc::GetOptsOO'=> undef,
	'Pod::Perldoc::ToChecker'=> undef,
	'Pod::Perldoc::ToMan'   => undef,
	'Pod::Perldoc::ToNroff' => undef,
	'Pod::Perldoc::ToPod'   => undef,
	'Pod::Perldoc::ToRtf'   => undef,
	'Pod::Perldoc::ToText'  => undef,
	'Pod::Perldoc::ToTk'    => undef,
	'Pod::Perldoc::ToXml'   => undef,
	'Pod::PlainText'        => '2.02',
	'Pod::Plainer'          => '0.01',
	'Pod::Select'           => '1.3',
	'Pod::Text'             => '2.21',
	'Pod::Text::Color'      => '1.04',
	'Pod::Text::Overstrike' => '1.1',
	'Pod::Text::Termcap'    => '1.11',
	'Pod::Usage'            => '1.3',
	'SDBM_File'             => '1.04',
	'Safe'                  => '2.11',
	'Scalar::Util'          => '1.14_1',
	'Search::Dict'          => '1.02',
	'SelectSaver'           => '1.01',
	'SelfLoader'            => '1.0904',
	'Shell'                 => '0.6',
	'Socket'                => '1.77',
	'Stdio'                 => '2.3',
	'Storable'              => '2.14',
	'Switch'                => '2.10',
	'Symbol'                => '1.05',
	'Sys::Hostname'         => '1.11',
	'Sys::Syslog'           => '0.06',
	'Term::ANSIColor'       => '1.09',
	'Term::Cap'             => '1.09',
	'Term::Complete'        => '1.402',
	'Term::ReadLine'        => '1.01',
	'Test'                  => '1.25',
	'Test::Builder'         => '0.22',
	'Test::Harness'         => '2.46',
	'Test::Harness::Assert' => '0.02',
	'Test::Harness::Iterator'=> '0.02',
	'Test::Harness::Straps' => '0.20_01',
	'Test::More'            => '0.54',
	'Test::Simple'          => '0.54',
	'Text::Abbrev'          => '1.01',
	'Text::Balanced'        => '1.95_01',
	'Text::ParseWords'      => '3.24',
	'Text::Soundex'         => '1.01',
	'Text::Tabs'            => '98.112801',
	'Text::Wrap'            => '2001.09292',
	'Thread'                => '2.00',
	'Thread::Queue'         => '2.00',
	'Thread::Semaphore'     => '2.01',
	'Thread::Signal'        => '1.00',
	'Thread::Specific'      => '1.00',
	'Tie::Array'            => '1.03',
	'Tie::File'             => '0.97',
	'Tie::Handle'           => '4.1',
	'Tie::Hash'             => '1.01',
	'Tie::Memoize'          => '1.0',
	'Tie::RefHash'          => '1.32',
	'Tie::Scalar'           => '1.00',
	'Tie::SubstrHash'       => '1.00',
	'Time::HiRes'           => '1.66',
	'Time::Local'           => '1.11',
	'Time::gmtime'          => '1.02',
	'Time::localtime'       => '1.02',
	'Time::tm'              => '1.00',
	'UNIVERSAL'             => '1.02',
        'Unicode'               => '4.0.1',
	'Unicode::Collate'      => '0.40',
	'Unicode::Normalize'    => '0.30',
	'Unicode::UCD'          => '0.22',
	'User::grent'           => '1.00',
	'User::pwent'           => '1.00',
	'Win32'                 => '0.23',
	'XS::APItest'           => '0.05',
	'XS::Typemap'           => '0.01',
	'XSLoader'              => '0.03',
	'XSSymSet'              => '1.0',
	'assertions'            => '0.01',
	'assertions::activate'  => '0.01',
	'attributes'            => '0.06',
	'attrs'                 => '1.01',
	'autouse'               => '1.04',
	'base'                  => '2.06',
	'bigint'                => '0.06',
	'bignum'                => '0.16',
	'bigrat'                => '0.07',
	'blib'                  => '1.02',
	'bytes'                 => '1.01',
	'charnames'             => '1.04',
	'constant'              => '1.05',
	'diagnostics'           => '1.14',
	'encoding'              => '2.01',
	'encoding::warnings'    => '0.05',
	'fields'                => '2.03',
	'filetest'              => '1.01',
	'if'                    => '0.0401',
	'integer'               => '1.00',
	'less'                  => '0.01',
	'lib'                   => '0.5565',
	'locale'                => '1.00',
	'open'                  => '1.04',
	'ops'                   => '1.00',
	'overload'              => '1.03',
	're'                    => '0.05',
	'sigtrap'               => '1.02',
	'sort'                  => '1.02',
	'strict'                => '1.03',
	'subs'                  => '1.00',
	'threads'               => '1.05',
	'threads::shared'       => '0.92',
	'utf8'                  => '1.04',
	'vars'                  => '1.01',
	'version'               => '0.42',
	'vmsish'                => '1.01',
	'warnings'              => '1.04',
	'warnings::register'    => '1.00',
    },

    5.008007 => {
	'AnyDBM_File'           => '1.00',
	'Attribute::Handlers'   => '0.78_01',
	'AutoLoader'            => '5.60',
	'AutoSplit'             => '1.04',
	'B'                     => '1.09',
	'B::Asmdata'            => '1.01',
	'B::Assembler'          => '0.07',
	'B::Bblock'             => '1.02',
	'B::Bytecode'           => '1.01',
	'B::C'                  => '1.04',
	'B::CC'                 => '1.00',
	'B::Concise'            => '0.65',
	'B::Debug'              => '1.02',
	'B::Deparse'            => '0.7',
	'B::Disassembler'       => '1.04',
	'B::Lint'               => '1.03',
	'B::Showlex'            => '1.02',
	'B::Stackobj'           => '1.00',
	'B::Stash'              => '1.00',
	'B::Terse'              => '1.03',
	'B::Xref'               => '1.01',
	'Benchmark'             => '1.07',
	'ByteLoader'            => '0.05',
	'CGI'                   => '3.10',
	'CGI::Apache'           => '1.00',
	'CGI::Carp'             => '1.29',
	'CGI::Cookie'           => '1.25',
	'CGI::Fast'             => '1.05',
	'CGI::Pretty'           => '1.08',
	'CGI::Push'             => '1.04',
	'CGI::Switch'           => '1.00',
	'CGI::Util'             => '1.5',
	'CPAN'                  => '1.76_01',
	'CPAN::FirstTime'       => '1.60 ',
	'CPAN::Nox'             => '1.03',
	'Carp'                  => '1.04',
	'Carp::Heavy'           => '1.04',
	'Class::ISA'            => '0.33',
	'Class::Struct'         => '0.63',
        'Config'                => undef,
	'Cwd'                   => '3.05',
	'DB'                    => '1.0',
	'DBM_Filter'            => '0.01',
	'DBM_Filter::compress'  => '0.01',
	'DBM_Filter::encode'    => '0.01',
	'DBM_Filter::int32'     => '0.01',
	'DBM_Filter::null'      => '0.01',
	'DBM_Filter::utf8'      => '0.01',
	'DB_File'               => '1.811',
	'DCLsym'                => '1.02',
	'Data::Dumper'          => '2.121_04',
	'Devel::DProf'          => '20050310.00',
	'Devel::PPPort'         => '3.06',
	'Devel::Peek'           => '1.02',
	'Devel::SelfStubber'    => '1.03',
	'Digest'                => '1.10',
	'Digest::MD5'           => '2.33',
	'Digest::base'          => '1.00',
	'Digest::file'          => '0.01',
	'DirHandle'             => '1.00',
	'Dumpvalue'             => '1.11',
	'DynaLoader'            => '1.05',
	'Encode'                => '2.10',
	'Encode::Alias'         => '2.03',
	'Encode::Byte'          => '2.00',
	'Encode::CJKConstants'  => '2.00',
	'Encode::CN'            => '2.00',
	'Encode::CN::HZ'        => '2.01',
	'Encode::Config'        => '2.00',
	'Encode::EBCDIC'        => '2.00',
	'Encode::Encoder'       => '2.00',
	'Encode::Encoding'      => '2.02',
	'Encode::Guess'         => '2.00',
	'Encode::JP'            => '2.01',
	'Encode::JP::H2Z'       => '2.00',
	'Encode::JP::JIS7'      => '2.00',
	'Encode::KR'            => '2.00',
	'Encode::KR::2022_KR'   => '2.00',
	'Encode::MIME::Header'  => '2.00',
	'Encode::Symbol'        => '2.00',
	'Encode::TW'            => '2.00',
	'Encode::Unicode'       => '2.02',
	'Encode::Unicode::UTF7' => '2.01',
	'English'               => '1.01',
	'Env'                   => '1.00',
	'Errno'                 => '1.09_01',
	'Exporter'              => '5.58',
	'Exporter::Heavy'       => '5.58',
	'ExtUtils::Command'     => '1.05',
	'ExtUtils::Command::MM' => '0.03',
	'ExtUtils::Constant'    => '0.16',
	'ExtUtils::Constant::Base'=> '0.01',
	'ExtUtils::Constant::Utils'=> '0.01',
	'ExtUtils::Constant::XS'=> '0.01',
	'ExtUtils::Embed'       => '1.250601',
	'ExtUtils::Install'     => '1.32',
	'ExtUtils::Installed'   => '0.08',
	'ExtUtils::Liblist'     => '1.01',
	'ExtUtils::Liblist::Kid'=> '1.3001',
	'ExtUtils::MM'          => '0.04',
	'ExtUtils::MM_Any'      => '0.07',
	'ExtUtils::MM_BeOS'     => '1.04',
	'ExtUtils::MM_Cygwin'   => '1.06',
	'ExtUtils::MM_DOS'      => '0.02',
	'ExtUtils::MM_MacOS'    => '1.07',
	'ExtUtils::MM_NW5'      => '2.07_02',
	'ExtUtils::MM_OS2'      => '1.04',
	'ExtUtils::MM_UWIN'     => '0.02',
	'ExtUtils::MM_Unix'     => '1.42',
	'ExtUtils::MM_VMS'      => '5.70',
	'ExtUtils::MM_Win32'    => '1.09',
	'ExtUtils::MM_Win95'    => '0.0301',
	'ExtUtils::MY'          => '0.01',
	'ExtUtils::MakeMaker'   => '6.17',
	'ExtUtils::MakeMaker::bytes'=> '0.01',
	'ExtUtils::MakeMaker::vmsish'=> '0.01',
	'ExtUtils::Manifest'    => '1.42',
        'ExtUtils::Miniperl'    => undef,
	'ExtUtils::Mkbootstrap' => '1.15',
	'ExtUtils::Mksymlists'  => '1.19',
	'ExtUtils::Packlist'    => '0.04',
	'ExtUtils::testlib'     => '1.15',
	'Fatal'                 => '1.03',
	'Fcntl'                 => '1.05',
	'File::Basename'        => '2.73',
	'File::CheckTree'       => '4.3',
	'File::Compare'         => '1.1003',
	'File::Copy'            => '2.08',
	'File::DosGlob'         => '1.00',
	'File::Find'            => '1.09',
	'File::Glob'            => '1.04',
	'File::Path'            => '1.07',
	'File::Spec'            => '3.05',
	'File::Spec::Cygwin'    => '1.1',
	'File::Spec::Epoc'      => '1.1',
	'File::Spec::Functions' => '1.3',
	'File::Spec::Mac'       => '1.4',
	'File::Spec::OS2'       => '1.2',
	'File::Spec::Unix'      => '1.5',
	'File::Spec::VMS'       => '1.4',
	'File::Spec::Win32'     => '1.5',
	'File::Temp'            => '0.16',
	'File::stat'            => '1.00',
	'FileCache'             => '1.05',
	'FileHandle'            => '2.01',
	'VMS::Filespec'         => '1.11',
	'Filter::Simple'        => '0.78',
	'Filter::Util::Call'    => '1.0601',
	'FindBin'               => '1.44',
	'GDBM_File'             => '1.07',
	'Getopt::Long'          => '2.34',
	'Getopt::Std'           => '1.05',
	'Hash::Util'            => '0.05',
	'I18N::Collate'         => '1.00',
	'I18N::LangTags'        => '0.35',
	'I18N::LangTags::Detect'=> '1.03',
	'I18N::LangTags::List'  => '0.35',
	'I18N::Langinfo'        => '0.02',
	'IO'                    => '1.21',
	'IO::Dir'               => '1.04',
	'IO::File'              => '1.11',
	'IO::Handle'            => '1.24',
	'IO::Pipe'              => '1.123',
	'IO::Poll'              => '0.06',
	'IO::Seekable'          => '1.09',
	'IO::Select'            => '1.16',
	'IO::Socket'            => '1.28',
	'IO::Socket::INET'      => '1.28',
	'IO::Socket::UNIX'      => '1.21',
	'IPC::Msg'              => '1.02',
	'IPC::Open2'            => '1.01',
	'IPC::Open3'            => '1.0106',
	'IPC::Semaphore'        => '1.02',
	'IPC::SysV'             => '1.04',
	'List::Util'            => '1.14',
	'Locale::Constants'     => '2.07',
	'Locale::Country'       => '2.07',
	'Locale::Currency'      => '2.07',
	'Locale::Language'      => '2.07',
	'Locale::Maketext'      => '1.09',
	'Locale::Maketext::Guts'=> undef,
	'Locale::Maketext::GutsLoader'=> undef,
	'Locale::Script'        => '2.07',
	'MIME::Base64'          => '3.05',
	'MIME::QuotedPrint'     => '3.03',
	'Math::BigFloat'        => '1.51',
	'Math::BigFloat::Trace' => '0.01',
	'Math::BigInt'          => '1.77',
	'Math::BigInt::Calc'    => '0.47',
	'Math::BigInt::CalcEmu' => '0.05',
	'Math::BigInt::Trace'   => '0.01',
	'Math::BigRat'          => '0.15',
	'Math::Complex'         => '1.34',
	'Math::Trig'            => '1.02',
	'Memoize'               => '1.01',
	'Memoize::AnyDBM_File'  => '0.65',
	'Memoize::Expire'       => '1.00',
	'Memoize::ExpireFile'   => '1.01',
	'Memoize::ExpireTest'   => '0.65',
	'Memoize::NDBM_File'    => '0.65',
	'Memoize::SDBM_File'    => '0.65',
	'Memoize::Storable'     => '0.65',
	'NDBM_File'             => '1.05',
	'NEXT'                  => '0.60',
	'Net::Cmd'              => '2.26',
	'Net::Config'           => '1.10',
	'Net::Domain'           => '2.19',
	'Net::FTP'              => '2.75',
	'Net::FTP::A'           => '1.16',
	'Net::FTP::E'           => '0.01',
	'Net::FTP::I'           => '1.12',
	'Net::FTP::L'           => '0.01',
	'Net::FTP::dataconn'    => '0.11',
	'Net::NNTP'             => '2.23',
	'Net::Netrc'            => '2.12',
	'Net::POP3'             => '2.28',
	'Net::Ping'             => '2.31',
	'Net::SMTP'             => '2.29',
	'Net::Time'             => '2.10',
	'Net::hostent'          => '1.01',
	'Net::netent'           => '1.00',
	'Net::protoent'         => '1.00',
	'Net::servent'          => '1.01',
	'O'                     => '1.00',
	'ODBM_File'             => '1.05',
	'Opcode'                => '1.05',
	'POSIX'                 => '1.08',
	'PerlIO'                => '1.03',
	'PerlIO::encoding'      => '0.07',
	'PerlIO::scalar'        => '0.02',
	'PerlIO::via'           => '0.02',
	'PerlIO::via::QuotedPrint'=> '0.06',
	'Pod::Checker'          => '1.42',
	'Pod::Find'             => '1.3',
	'Pod::Functions'        => '1.02',
	'Pod::Html'             => '1.0503',
	'Pod::InputObjects'     => '1.3',
	'Pod::LaTeX'            => '0.58',
	'Pod::Man'              => '1.37',
	'Pod::ParseLink'        => '1.06',
	'Pod::ParseUtils'       => '1.3',
	'Pod::Parser'           => '1.3',
	'Pod::Perldoc'          => '3.14',
	'Pod::Perldoc::BaseTo'  => undef,
	'Pod::Perldoc::GetOptsOO'=> undef,
	'Pod::Perldoc::ToChecker'=> undef,
	'Pod::Perldoc::ToMan'   => undef,
	'Pod::Perldoc::ToNroff' => undef,
	'Pod::Perldoc::ToPod'   => undef,
	'Pod::Perldoc::ToRtf'   => undef,
	'Pod::Perldoc::ToText'  => undef,
	'Pod::Perldoc::ToTk'    => undef,
	'Pod::Perldoc::ToXml'   => undef,
	'Pod::PlainText'        => '2.02',
	'Pod::Plainer'          => '0.01',
	'Pod::Select'           => '1.3',
	'Pod::Text'             => '2.21',
	'Pod::Text::Color'      => '1.04',
	'Pod::Text::Overstrike' => '1.1',
	'Pod::Text::Termcap'    => '1.11',
	'Pod::Usage'            => '1.3',
	'SDBM_File'             => '1.04',
	'Safe'                  => '2.11',
	'Scalar::Util'          => '1.14',
	'Search::Dict'          => '1.02',
	'SelectSaver'           => '1.01',
	'SelfLoader'            => '1.0904',
	'Shell'                 => '0.6',
	'Socket'                => '1.77',
	'Stdio'                 => '2.3',
	'Storable'              => '2.13',
	'Switch'                => '2.10',
	'Symbol'                => '1.06',
	'Sys::Hostname'         => '1.11',
	'Sys::Syslog'           => '0.06',
	'Term::ANSIColor'       => '1.09',
	'Term::Cap'             => '1.09',
	'Term::Complete'        => '1.402',
	'Term::ReadLine'        => '1.01',
	'Test'                  => '1.25',
	'Test::Builder'         => '0.22',
	'Test::Harness'         => '2.48',
	'Test::Harness::Assert' => '0.02',
	'Test::Harness::Iterator'=> '0.02',
	'Test::Harness::Point'  => '0.01',
	'Test::Harness::Straps' => '0.23',
	'Test::More'            => '0.54',
	'Test::Simple'          => '0.54',
	'Text::Abbrev'          => '1.01',
	'Text::Balanced'        => '1.95',
	'Text::ParseWords'      => '3.24',
	'Text::Soundex'         => '1.01',
	'Text::Tabs'            => '98.112801',
	'Text::Wrap'            => '2001.09293',
	'Thread'                => '2.00',
	'Thread::Queue'         => '2.00',
	'Thread::Semaphore'     => '2.01',
	'Thread::Signal'        => '1.00',
	'Thread::Specific'      => '1.00',
	'Tie::Array'            => '1.03',
	'Tie::File'             => '0.97',
	'Tie::Handle'           => '4.1',
	'Tie::Hash'             => '1.01',
	'Tie::Memoize'          => '1.0',
	'Tie::RefHash'          => '1.32',
	'Tie::Scalar'           => '1.00',
	'Tie::SubstrHash'       => '1.00',
	'Time::HiRes'           => '1.66',
	'Time::Local'           => '1.11',
	'Time::gmtime'          => '1.02',
	'Time::localtime'       => '1.02',
	'Time::tm'              => '1.00',
	'UNIVERSAL'             => '1.01',
        'Unicode'               => '4.1.0', # lib/unicore/version
	'Unicode::Collate'      => '0.40',
	'Unicode::Normalize'    => '0.32',
	'Unicode::UCD'          => '0.23',
	'User::grent'           => '1.00',
	'User::pwent'           => '1.00',
	'Win32'                 => '0.24',
	'XS::APItest'           => '0.06',
	'XS::Typemap'           => '0.01',
	'XSLoader'              => '0.02',
	'XSSymSet'              => '1.0',
	'attributes'            => '0.06',
	'attrs'                 => '1.01',
	'autouse'               => '1.04',
	'base'                  => '2.07',
	'bigint'                => '0.07',
	'bignum'                => '0.17',
	'bigrat'                => '0.08',
	'blib'                  => '1.02',
	'bytes'                 => '1.02',
	'charnames'             => '1.04',
	'constant'              => '1.05',
	'diagnostics'           => '1.14',
	'encoding'              => '2.01',
	'fields'                => '2.03',
	'filetest'              => '1.01',
	'if'                    => '0.03',
	'integer'               => '1.00',
	'less'                  => '0.01',
	'lib'                   => '0.5565',
	'locale'                => '1.00',
	'open'                  => '1.04',
	'ops'                   => '1.00',
	'overload'              => '1.03',
	're'                    => '0.04',
	'sigtrap'               => '1.02',
	'sort'                  => '1.02',
	'strict'                => '1.03',
	'subs'                  => '1.00',
	'threads'               => '1.05',
	'threads::shared'       => '0.93',
	'utf8'                  => '1.05',
	'vars'                  => '1.01',
	'vmsish'                => '1.01',
	'warnings'              => '1.03',
	'warnings::register'    => '1.00',
    },

    5.009003 => {
	'AnyDBM_File'           => '1.00',
	'Archive::Tar'          => '1.26_01',
	'Archive::Tar::Constant'=> '0.02',
	'Archive::Tar::File'    => '0.02',
	'Attribute::Handlers'   => '0.78_02',
	'AutoLoader'            => '5.60',
	'AutoSplit'             => '1.04_01',
	'B'                     => '1.10',
	'B::Asmdata'            => '1.01',
	'B::Assembler'          => '0.07',
	'B::Bblock'             => '1.02',
	'B::Bytecode'           => '1.01',
	'B::C'                  => '1.04',
	'B::CC'                 => '1.00',
	'B::Concise'            => '0.67',
	'B::Debug'              => '1.02',
	'B::Deparse'            => '0.73',
	'B::Disassembler'       => '1.05',
	'B::Lint'               => '1.04',
	'B::Showlex'            => '1.02',
	'B::Stackobj'           => '1.00',
	'B::Stash'              => '1.00',
	'B::Terse'              => '1.03',
	'B::Xref'               => '1.01',
	'Benchmark'             => '1.07',
	'ByteLoader'            => '0.06',
	'CGI'                   => '3.15_01',
	'CGI::Apache'           => '1.00',
	'CGI::Carp'             => '1.29',
	'CGI::Cookie'           => '1.26',
	'CGI::Fast'             => '1.05',
	'CGI::Pretty'           => '1.08',
	'CGI::Push'             => '1.04',
	'CGI::Switch'           => '1.00',
	'CGI::Util'             => '1.5',
	'CPAN'                  => '1.83_58',
	'CPAN::Debug'           => '4.44',
	'CPAN::FirstTime'       => '4.50',
	'CPAN::HandleConfig'    => '4.31',
	'CPAN::Nox'             => '2.31',
	'CPAN::Tarzip'          => '3.36',
	'CPAN::Version'         => '2.55',
	'Carp'                  => '1.05',
	'Carp::Heavy'           => '1.05',
	'Class::ISA'            => '0.33',
	'Class::Struct'         => '0.63',
	'Compress::Zlib'        => '2.000_07',
	'Compress::Zlib::Common'=> '2.000_07',
	'Compress::Zlib::Compress::Gzip::Constants'=> '2.000_07',
	'Compress::Zlib::Compress::Zip::Constants'=> '1.00',
	'Compress::Zlib::CompressPlugin::Deflate'=> '2.000_05',
	'Compress::Zlib::CompressPlugin::Identity'=> '2.000_05',
	'Compress::Zlib::File::GlobMapper'=> '0.000_02',
	'Compress::Zlib::FileConstants'=> '2.000_07',
	'Compress::Zlib::IO::Compress::Base'=> '2.000_05',
	'Compress::Zlib::IO::Compress::Deflate'=> '2.000_07',
	'Compress::Zlib::IO::Compress::Gzip'=> '2.000_07',
	'Compress::Zlib::IO::Compress::RawDeflate'=> '2.000_07',
	'Compress::Zlib::IO::Compress::Zip'=> '2.000_04',
	'Compress::Zlib::IO::Uncompress::AnyInflate'=> '2.000_07',
	'Compress::Zlib::IO::Uncompress::AnyUncompress'=> '2.000_05',
	'Compress::Zlib::IO::Uncompress::Base'=> '2.000_05',
	'Compress::Zlib::IO::Uncompress::Gunzip'=> '2.000_07',
	'Compress::Zlib::IO::Uncompress::Inflate'=> '2.000_07',
	'Compress::Zlib::IO::Uncompress::RawInflate'=> '2.000_07',
	'Compress::Zlib::IO::Uncompress::Unzip'=> '2.000_05',
	'Compress::Zlib::ParseParameters'=> '2.000_07',
	'Compress::Zlib::UncompressPlugin::Identity'=> '2.000_05',
	'Compress::Zlib::UncompressPlugin::Inflate'=> '2.000_05',
        'Config'                => undef,
	'Config::Extensions'    => '0.01',
	'Cwd'                   => '3.15',
	'DB'                    => '1.01',
	'DBM_Filter'            => '0.01',
	'DBM_Filter::compress'  => '0.01',
	'DBM_Filter::encode'    => '0.01',
	'DBM_Filter::int32'     => '0.01',
	'DBM_Filter::null'      => '0.01',
	'DBM_Filter::utf8'      => '0.01',
	'DB_File'               => '1.814',
	'DCLsym'                => '1.02',
	'Data::Dumper'          => '2.121_08',
	'Devel::DProf'          => '20050603.00',
	'Devel::PPPort'         => '3.08',
	'Devel::Peek'           => '1.03',
	'Devel::SelfStubber'    => '1.03',
	'Digest'                => '1.14',
	'Digest::MD5'           => '2.36',
	'Digest::SHA'           => '5.32',
	'Digest::base'          => '1.00',
	'Digest::file'          => '1.00',
	'DirHandle'             => '1.01',
	'Dumpvalue'             => '1.12',
	'DynaLoader'            => '1.07',
	'Encode'                => '2.14',
	'Encode::Alias'         => '2.04',
	'Encode::Byte'          => '2.00',
	'Encode::CJKConstants'  => '2.00',
	'Encode::CN'            => '2.00',
	'Encode::CN::HZ'        => '2.02',
	'Encode::Config'        => '2.01',
	'Encode::EBCDIC'        => '2.00',
	'Encode::Encoder'       => '2.00',
	'Encode::Encoding'      => '2.02',
	'Encode::Guess'         => '2.00',
	'Encode::JP'            => '2.01',
	'Encode::JP::H2Z'       => '2.00',
	'Encode::JP::JIS7'      => '2.00',
	'Encode::KR'            => '2.00',
	'Encode::KR::2022_KR'   => '2.00',
	'Encode::MIME::Header'  => '2.02',
	'Encode::MIME::Header::ISO_2022_JP'=> '1.01',
	'Encode::Symbol'        => '2.00',
	'Encode::TW'            => '2.00',
	'Encode::Unicode'       => '2.02',
	'Encode::Unicode::UTF7' => '2.01',
	'English'               => '1.04',
	'Env'                   => '1.00',
	'Errno'                 => '1.09_01',
	'Exporter'              => '5.59',
	'Exporter::Heavy'       => '5.59',
	'ExtUtils::CBuilder'    => '0.15',
	'ExtUtils::CBuilder::Base'=> '0.12',
	'ExtUtils::CBuilder::Platform::Unix'=> '0.12',
	'ExtUtils::CBuilder::Platform::VMS'=> '0.12',
	'ExtUtils::CBuilder::Platform::Windows'=> '0.12',
	'ExtUtils::CBuilder::Platform::aix'=> '0.12',
	'ExtUtils::CBuilder::Platform::cygwin'=> '0.12',
	'ExtUtils::CBuilder::Platform::darwin'=> '0.12',
	'ExtUtils::CBuilder::Platform::dec_osf'=> '0.01',
	'ExtUtils::CBuilder::Platform::os2'=> '0.13',
	'ExtUtils::Command'     => '1.09',
	'ExtUtils::Command::MM' => '0.05_01',
	'ExtUtils::Constant'    => '0.2',
	'ExtUtils::Constant::Base'=> '0.02',
	'ExtUtils::Constant::ProxySubs'=> '0.01',
	'ExtUtils::Constant::Utils'=> '0.01',
	'ExtUtils::Constant::XS'=> '0.02',
	'ExtUtils::Embed'       => '1.26',
	'ExtUtils::Install'     => '1.33',
	'ExtUtils::Installed'   => '0.08',
	'ExtUtils::Liblist'     => '1.01',
	'ExtUtils::Liblist::Kid'=> '1.3',
	'ExtUtils::MM'          => '0.05',
	'ExtUtils::MM_AIX'      => '0.03',
	'ExtUtils::MM_Any'      => '0.13_01',
	'ExtUtils::MM_BeOS'     => '1.05',
	'ExtUtils::MM_Cygwin'   => '1.08',
	'ExtUtils::MM_DOS'      => '0.02',
	'ExtUtils::MM_MacOS'    => '1.08',
	'ExtUtils::MM_NW5'      => '2.08',
	'ExtUtils::MM_OS2'      => '1.05',
	'ExtUtils::MM_QNX'      => '0.02',
	'ExtUtils::MM_UWIN'     => '0.02',
	'ExtUtils::MM_Unix'     => '1.50_01',
	'ExtUtils::MM_VMS'      => '5.73',
	'ExtUtils::MM_VOS'      => '0.02',
	'ExtUtils::MM_Win32'    => '1.12',
	'ExtUtils::MM_Win95'    => '0.04',
	'ExtUtils::MY'          => '0.01',
	'ExtUtils::MakeMaker'   => '6.30_01',
	'ExtUtils::MakeMaker::Config'=> '0.02',
	'ExtUtils::MakeMaker::bytes'=> '0.01',
	'ExtUtils::MakeMaker::vmsish'=> '0.01',
	'ExtUtils::Manifest'    => '1.46',
        'ExtUtils::Miniperl'    => undef,
	'ExtUtils::Mkbootstrap' => '1.15',
	'ExtUtils::Mksymlists'  => '1.19',
	'ExtUtils::Packlist'    => '0.04',
	'ExtUtils::ParseXS'     => '2.15_02',
	'ExtUtils::testlib'     => '1.15',
	'Fatal'                 => '1.04',
	'Fcntl'                 => '1.05',
	'File::Basename'        => '2.74',
	'File::CheckTree'       => '4.3',
	'File::Compare'         => '1.1005',
	'File::Copy'            => '2.09',
	'File::DosGlob'         => '1.00',
	'File::Find'            => '1.10',
	'File::Glob'            => '1.05',
	'File::Path'            => '1.08',
	'File::Spec'            => '3.15',
	'File::Spec::Cygwin'    => '1.1',
	'File::Spec::Epoc'      => '1.1',
	'File::Spec::Functions' => '1.3',
	'File::Spec::Mac'       => '1.4',
	'File::Spec::OS2'       => '1.2',
	'File::Spec::Unix'      => '1.5',
	'File::Spec::VMS'       => '1.4',
	'File::Spec::Win32'     => '1.6',
	'File::Temp'            => '0.16_01',
	'File::stat'            => '1.00',
	'FileCache'             => '1.06',
	'FileHandle'            => '2.01',
	'VMS::Filespec'         => '1.11',
	'Filter::Simple'        => '0.82',
	'Filter::Util::Call'    => '1.0601',
	'FindBin'               => '1.47',
	'GDBM_File'             => '1.08',
	'Getopt::Long'          => '2.35',
	'Getopt::Std'           => '1.05',
	'Hash::Util'            => '0.05',
	'I18N::Collate'         => '1.00',
	'I18N::LangTags'        => '0.35',
	'I18N::LangTags::Detect'=> '1.03',
	'I18N::LangTags::List'  => '0.35',
	'I18N::Langinfo'        => '0.02',
	'IO'                    => '1.22',
	'IO::Dir'               => '1.05',
	'IO::File'              => '1.13_01',
	'IO::Handle'            => '1.26',
	'IO::Pipe'              => '1.13',
	'IO::Poll'              => '0.07',
	'IO::Seekable'          => '1.10',
	'IO::Select'            => '1.17',
	'IO::Socket'            => '1.29_01',
	'IO::Socket::INET'      => '1.29_02',
	'IO::Socket::UNIX'      => '1.22_01',
	'IO::Zlib'              => '1.04_02',
	'IPC::Msg'              => '1.02',
	'IPC::Open2'            => '1.02',
	'IPC::Open3'            => '1.02',
	'IPC::Semaphore'        => '1.02',
	'IPC::SysV'             => '1.04',
	'List::Util'            => '1.18',
	'Locale::Constants'     => '2.07',
	'Locale::Country'       => '2.07',
	'Locale::Currency'      => '2.07',
	'Locale::Language'      => '2.07',
	'Locale::Maketext'      => '1.10_01',
	'Locale::Maketext::Guts'=> undef,
	'Locale::Maketext::GutsLoader'=> undef,
	'Locale::Script'        => '2.07',
	'MIME::Base64'          => '3.07',
	'MIME::QuotedPrint'     => '3.07',
	'Math::BigFloat'        => '1.51',
	'Math::BigFloat::Trace' => '0.01',
	'Math::BigInt'          => '1.77',
	'Math::BigInt::Calc'    => '0.47',
	'Math::BigInt::CalcEmu' => '0.05',
	'Math::BigInt::FastCalc'=> '0.10',
	'Math::BigInt::Trace'   => '0.01',
	'Math::BigRat'          => '0.15',
	'Math::Complex'         => '1.35',
	'Math::Trig'            => '1.03',
	'Memoize'               => '1.01_01',
	'Memoize::AnyDBM_File'  => '0.65',
	'Memoize::Expire'       => '1.00',
	'Memoize::ExpireFile'   => '1.01',
	'Memoize::ExpireTest'   => '0.65',
	'Memoize::NDBM_File'    => '0.65',
	'Memoize::SDBM_File'    => '0.65',
	'Memoize::Storable'     => '0.65',
	'Module::CoreList'      => '2.02',
	'Moped::Msg'            => '0.01',
	'NDBM_File'             => '1.06',
	'NEXT'                  => '0.60_01',
	'Net::Cmd'              => '2.26_01',
	'Net::Config'           => '1.10',
	'Net::Domain'           => '2.19_01',
	'Net::FTP'              => '2.75',
	'Net::FTP::A'           => '1.16',
	'Net::FTP::E'           => '0.01',
	'Net::FTP::I'           => '1.12',
	'Net::FTP::L'           => '0.01',
	'Net::FTP::dataconn'    => '0.11',
	'Net::NNTP'             => '2.23',
	'Net::Netrc'            => '2.12',
	'Net::POP3'             => '2.28',
	'Net::Ping'             => '2.31_04',
	'Net::SMTP'             => '2.29',
	'Net::Time'             => '2.10',
	'Net::hostent'          => '1.01',
	'Net::netent'           => '1.00',
	'Net::protoent'         => '1.00',
	'Net::servent'          => '1.01',
	'O'                     => '1.00',
	'ODBM_File'             => '1.06',
	'Opcode'                => '1.08',
	'POSIX'                 => '1.10',
	'PerlIO'                => '1.04',
	'PerlIO::encoding'      => '0.09',
	'PerlIO::scalar'        => '0.04',
	'PerlIO::via'           => '0.03',
	'PerlIO::via::QuotedPrint'=> '0.06',
	'Pod::Checker'          => '1.43',
	'Pod::Escapes'          => '1.04',
	'Pod::Find'             => '1.34',
	'Pod::Functions'        => '1.03',
	'Pod::Html'             => '1.0504',
	'Pod::InputObjects'     => '1.3',
	'Pod::LaTeX'            => '0.58',
	'Pod::Man'              => '2.04',
	'Pod::ParseLink'        => '1.06',
	'Pod::ParseUtils'       => '1.33',
	'Pod::Parser'           => '1.32',
	'Pod::Perldoc'          => '3.14_01',
	'Pod::Perldoc::BaseTo'  => undef,
	'Pod::Perldoc::GetOptsOO'=> undef,
	'Pod::Perldoc::ToChecker'=> undef,
	'Pod::Perldoc::ToMan'   => undef,
	'Pod::Perldoc::ToNroff' => undef,
	'Pod::Perldoc::ToPod'   => undef,
	'Pod::Perldoc::ToRtf'   => undef,
	'Pod::Perldoc::ToText'  => undef,
	'Pod::Perldoc::ToTk'    => undef,
	'Pod::Perldoc::ToXml'   => undef,
	'Pod::PlainText'        => '2.02',
	'Pod::Plainer'          => '0.01',
	'Pod::Select'           => '1.3',
	'Pod::Simple'           => '3.04',
	'Pod::Simple::BlackBox' => undef,
	'Pod::Simple::Checker'  => '2.02',
	'Pod::Simple::Debug'    => undef,
	'Pod::Simple::DumpAsText'=> '2.02',
	'Pod::Simple::DumpAsXML'=> '2.02',
	'Pod::Simple::HTML'     => '3.03',
	'Pod::Simple::HTMLBatch'=> '3.02',
	'Pod::Simple::HTMLLegacy'=> '5.01',
	'Pod::Simple::LinkSection'=> undef,
	'Pod::Simple::Methody'  => '2.02',
	'Pod::Simple::Progress' => '1.01',
	'Pod::Simple::PullParser'=> '2.02',
	'Pod::Simple::PullParserEndToken'=> undef,
	'Pod::Simple::PullParserStartToken'=> undef,
	'Pod::Simple::PullParserTextToken'=> undef,
	'Pod::Simple::PullParserToken'=> '2.02',
	'Pod::Simple::RTF'      => '2.02',
	'Pod::Simple::Search'   => '3.04',
	'Pod::Simple::SimpleTree'=> '2.02',
	'Pod::Simple::Text'     => '2.02',
	'Pod::Simple::TextContent'=> '2.02',
	'Pod::Simple::TiedOutFH'=> undef,
	'Pod::Simple::Transcode'=> undef,
	'Pod::Simple::TranscodeDumb'=> '2.02',
	'Pod::Simple::TranscodeSmart'=> undef,
	'Pod::Simple::XMLOutStream'=> '2.02',
	'Pod::Text'             => '3.01',
	'Pod::Text::Color'      => '2.01',
	'Pod::Text::Overstrike' => '2',
	'Pod::Text::Termcap'    => '2.01',
	'Pod::Usage'            => '1.33_01',
	'SDBM_File'             => '1.05',
	'Safe'                  => '2.12',
	'Scalar::Util'          => '1.18',
	'Search::Dict'          => '1.02',
	'SelectSaver'           => '1.01',
	'SelfLoader'            => '1.0905',
	'Shell'                 => '0.6',
	'Socket'                => '1.78',
	'Stdio'                 => '2.3',
	'Storable'              => '2.15_02',
	'Switch'                => '2.10_01',
	'Symbol'                => '1.06',
	'Sys::Hostname'         => '1.11',
	'Sys::Syslog'           => '0.13',
	'Term::ANSIColor'       => '1.10',
	'Term::Cap'             => '1.09',
	'Term::Complete'        => '1.402',
	'Term::ReadLine'        => '1.02',
	'Test'                  => '1.25',
	'Test::Builder'         => '0.32',
	'Test::Builder::Module' => '0.03',
	'Test::Builder::Tester' => '1.02',
	'Test::Builder::Tester::Color'=> undef,
	'Test::Harness'         => '2.56',
	'Test::Harness::Assert' => '0.02',
	'Test::Harness::Iterator'=> '0.02',
	'Test::Harness::Point'  => '0.01',
	'Test::Harness::Straps' => '0.26',
	'Test::More'            => '0.62',
	'Test::Simple'          => '0.62',
	'Text::Abbrev'          => '1.01',
	'Text::Balanced'        => '1.95_01',
	'Text::ParseWords'      => '3.24',
	'Text::Soundex'         => '1.01',
	'Text::Tabs'            => '2005.0824',
	'Text::Wrap'            => '2005.082401',
	'Thread'                => '2.00',
	'Thread::Queue'         => '2.00',
	'Thread::Semaphore'     => '2.01',
	'Thread::Signal'        => '1.00',
	'Thread::Specific'      => '1.00',
	'Tie::Array'            => '1.03',
	'Tie::File'             => '0.97_01',
	'Tie::Handle'           => '4.1',
	'Tie::Hash'             => '1.02',
	'Tie::Memoize'          => '1.0',
	'Tie::RefHash'          => '1.32',
	'Tie::Scalar'           => '1.00',
	'Tie::SubstrHash'       => '1.00',
	'Time::HiRes'           => '1.86',
	'Time::Local'           => '1.11',
	'Time::gmtime'          => '1.02',
	'Time::localtime'       => '1.02',
	'Time::tm'              => '1.00',
	'UNIVERSAL'             => '1.03',
        'Unicode'               => '4.1.0',
	'Unicode::Collate'      => '0.52',
	'Unicode::Normalize'    => '0.32',
	'Unicode::UCD'          => '0.24',
	'User::grent'           => '1.01',
	'User::pwent'           => '1.00',
	'Win32'                 => '0.2601',
	'XS::APItest'           => '0.09',
	'XS::Typemap'           => '0.02',
	'XSLoader'              => '0.06',
	'XSSymSet'              => '1.0',
	'assertions'            => '0.02',
	'assertions::activate'  => '0.02',
	'assertions::compat'    => undef,
	'attributes'            => '0.06',
	'attrs'                 => '1.02',
	'autouse'               => '1.05',
	'base'                  => '2.07',
	'bigint'                => '0.07',
	'bignum'                => '0.17',
	'bigrat'                => '0.08',
	'blib'                  => '1.03',
	'bytes'                 => '1.02',
	'charnames'             => '1.05',
	'constant'              => '1.07',
	'diagnostics'           => '1.15',
	'encoding'              => '2.02',
	'encoding::warnings'    => '0.05',
	'feature'               => '1.00',
	'fields'                => '2.03',
	'filetest'              => '1.01',
	'if'                    => '0.05',
	'integer'               => '1.00',
	'less'                  => '0.01',
	'lib'                   => '0.5565',
	'locale'                => '1.00',
	'open'                  => '1.05',
	'ops'                   => '1.01',
	'overload'              => '1.04',
	're'                    => '0.06',
	'sigtrap'               => '1.02',
	'sort'                  => '2.00',
	'strict'                => '1.03',
	'subs'                  => '1.00',
	'threads'               => '1.07',
	'threads::shared'       => '0.94',
	'utf8'                  => '1.06',
	'vars'                  => '1.01',
	'version'               => '0.53',
	'vmsish'                => '1.02',
	'warnings'              => '1.05',
	'warnings::register'    => '1.01',
    },

    5.008008 => {
	'AnyDBM_File'           => '1.00',
	'Attribute::Handlers'   => '0.78_02',
	'AutoLoader'            => '5.60',
	'AutoSplit'             => '1.04',
	'B'                     => '1.09_01',
	'B::Asmdata'            => '1.01',
	'B::Assembler'          => '0.07',
	'B::Bblock'             => '1.02_01',
	'B::Bytecode'           => '1.01_01',
	'B::C'                  => '1.04_01',
	'B::CC'                 => '1.00_01',
	'B::Concise'            => '0.66',
	'B::Debug'              => '1.02_01',
	'B::Deparse'            => '0.71',
	'B::Disassembler'       => '1.05',
	'B::Lint'               => '1.03',
	'B::Showlex'            => '1.02',
	'B::Stackobj'           => '1.00',
	'B::Stash'              => '1.00',
	'B::Terse'              => '1.03_01',
	'B::Xref'               => '1.01',
	'Benchmark'             => '1.07',
	'ByteLoader'            => '0.06',
	'CGI'                   => '3.15',
	'CGI::Apache'           => '1.00',
	'CGI::Carp'             => '1.29',
	'CGI::Cookie'           => '1.26',
	'CGI::Fast'             => '1.05',
	'CGI::Pretty'           => '1.08',
	'CGI::Push'             => '1.04',
	'CGI::Switch'           => '1.00',
	'CGI::Util'             => '1.5',
	'CPAN'                  => '1.76_02',
	'CPAN::FirstTime'       => '1.60 ',
	'CPAN::Nox'             => '1.03',
	'Carp'                  => '1.04',
	'Carp::Heavy'           => '1.04',
	'Class::ISA'            => '0.33',
	'Class::Struct'         => '0.63',
        'Config'                => undef,
	'Cwd'                   => '3.12',
	'DB'                    => '1.01',
	'DBM_Filter'            => '0.01',
	'DBM_Filter::compress'  => '0.01',
	'DBM_Filter::encode'    => '0.01',
	'DBM_Filter::int32'     => '0.01',
	'DBM_Filter::null'      => '0.01',
	'DBM_Filter::utf8'      => '0.01',
	'DB_File'               => '1.814',
	'DCLsym'                => '1.02',
	'Data::Dumper'          => '2.121_08',
	'Devel::DProf'          => '20050603.00',
	'Devel::PPPort'         => '3.06_01',
	'Devel::Peek'           => '1.03',
	'Devel::SelfStubber'    => '1.03',
	'Digest'                => '1.14',
	'Digest::MD5'           => '2.36',
	'Digest::base'          => '1.00',
	'Digest::file'          => '1.00',
	'DirHandle'             => '1.00',
	'Dumpvalue'             => '1.12',
	'DynaLoader'            => '1.05',
	'Encode'                => '2.12',
	'Encode::Alias'         => '2.04',
	'Encode::Byte'          => '2.00',
	'Encode::CJKConstants'  => '2.00',
	'Encode::CN'            => '2.00',
	'Encode::CN::HZ'        => '2.01',
	'Encode::Config'        => '2.01',
	'Encode::EBCDIC'        => '2.00',
	'Encode::Encoder'       => '2.00',
	'Encode::Encoding'      => '2.02',
	'Encode::Guess'         => '2.00',
	'Encode::JP'            => '2.01',
	'Encode::JP::H2Z'       => '2.00',
	'Encode::JP::JIS7'      => '2.00',
	'Encode::KR'            => '2.00',
	'Encode::KR::2022_KR'   => '2.00',
	'Encode::MIME::Header'  => '2.01',
	'Encode::MIME::Header::ISO_2022_JP'=> '1.01',
	'Encode::Symbol'        => '2.00',
	'Encode::TW'            => '2.00',
	'Encode::Unicode'       => '2.02',
	'Encode::Unicode::UTF7' => '2.01',
	'English'               => '1.02',
	'Env'                   => '1.00',
	'Errno'                 => '1.09_01',
	'Exporter'              => '5.58',
	'Exporter::Heavy'       => '5.58',
	'ExtUtils::Command'     => '1.09',
	'ExtUtils::Command::MM' => '0.05',
	'ExtUtils::Constant'    => '0.17',
	'ExtUtils::Constant::Base'=> '0.01',
	'ExtUtils::Constant::Utils'=> '0.01',
	'ExtUtils::Constant::XS'=> '0.01',
	'ExtUtils::Embed'       => '1.26',
	'ExtUtils::Install'     => '1.33',
	'ExtUtils::Installed'   => '0.08',
	'ExtUtils::Liblist'     => '1.01',
	'ExtUtils::Liblist::Kid'=> '1.3',
	'ExtUtils::MM'          => '0.05',
	'ExtUtils::MM_AIX'      => '0.03',
	'ExtUtils::MM_Any'      => '0.13',
	'ExtUtils::MM_BeOS'     => '1.05',
	'ExtUtils::MM_Cygwin'   => '1.08',
	'ExtUtils::MM_DOS'      => '0.02',
	'ExtUtils::MM_MacOS'    => '1.08',
	'ExtUtils::MM_NW5'      => '2.08',
	'ExtUtils::MM_OS2'      => '1.05',
	'ExtUtils::MM_QNX'      => '0.02',
	'ExtUtils::MM_UWIN'     => '0.02',
	'ExtUtils::MM_Unix'     => '1.50',
	'ExtUtils::MM_VMS'      => '5.73',
	'ExtUtils::MM_VOS'      => '0.02',
	'ExtUtils::MM_Win32'    => '1.12',
	'ExtUtils::MM_Win95'    => '0.04',
	'ExtUtils::MY'          => '0.01',
	'ExtUtils::MakeMaker'   => '6.30',
	'ExtUtils::MakeMaker::Config'=> '0.02',
	'ExtUtils::MakeMaker::bytes'=> '0.01',
	'ExtUtils::MakeMaker::vmsish'=> '0.01',
	'ExtUtils::Manifest'    => '1.46',
        'ExtUtils::Miniperl'    => undef,
	'ExtUtils::Mkbootstrap' => '1.15',
	'ExtUtils::Mksymlists'  => '1.19',
	'ExtUtils::Packlist'    => '0.04',
	'ExtUtils::testlib'     => '1.15',
	'Fatal'                 => '1.03',
	'Fcntl'                 => '1.05',
	'File::Basename'        => '2.74',
	'File::CheckTree'       => '4.3',
	'File::Compare'         => '1.1003',
	'File::Copy'            => '2.09',
	'File::DosGlob'         => '1.00',
	'File::Find'            => '1.10',
	'File::Glob'            => '1.05',
	'File::Path'            => '1.08',
	'File::Spec'            => '3.12',
	'File::Spec::Cygwin'    => '1.1',
	'File::Spec::Epoc'      => '1.1',
	'File::Spec::Functions' => '1.3',
	'File::Spec::Mac'       => '1.4',
	'File::Spec::OS2'       => '1.2',
	'File::Spec::Unix'      => '1.5',
	'File::Spec::VMS'       => '1.4',
	'File::Spec::Win32'     => '1.6',
	'File::Temp'            => '0.16',
	'File::stat'            => '1.00',
	'FileCache'             => '1.06',
	'FileHandle'            => '2.01',
	'VMS::Filespec'         => '1.11',
	'Filter::Simple'        => '0.82',
	'Filter::Util::Call'    => '1.0601',
	'FindBin'               => '1.47',
	'GDBM_File'             => '1.08',
	'Getopt::Long'          => '2.35',
	'Getopt::Std'           => '1.05',
	'Hash::Util'            => '0.05',
	'I18N::Collate'         => '1.00',
	'I18N::LangTags'        => '0.35',
	'I18N::LangTags::Detect'=> '1.03',
	'I18N::LangTags::List'  => '0.35',
	'I18N::Langinfo'        => '0.02',
	'IO'                    => '1.22',
	'IO::Dir'               => '1.05',
	'IO::File'              => '1.13',
	'IO::Handle'            => '1.25',
	'IO::Pipe'              => '1.13',
	'IO::Poll'              => '0.07',
	'IO::Seekable'          => '1.10',
	'IO::Select'            => '1.17',
	'IO::Socket'            => '1.29',
	'IO::Socket::INET'      => '1.29',
	'IO::Socket::UNIX'      => '1.22',
	'IPC::Msg'              => '1.02',
	'IPC::Open2'            => '1.02',
	'IPC::Open3'            => '1.02',
	'IPC::Semaphore'        => '1.02',
	'IPC::SysV'             => '1.04',
	'List::Util'            => '1.18',
	'Locale::Constants'     => '2.07',
	'Locale::Country'       => '2.07',
	'Locale::Currency'      => '2.07',
	'Locale::Language'      => '2.07',
	'Locale::Maketext'      => '1.09',
	'Locale::Maketext::Guts'=> undef,
	'Locale::Maketext::GutsLoader'=> undef,
	'Locale::Script'        => '2.07',
	'MIME::Base64'          => '3.07',
	'MIME::QuotedPrint'     => '3.07',
	'Math::BigFloat'        => '1.51',
	'Math::BigFloat::Trace' => '0.01',
	'Math::BigInt'          => '1.77',
	'Math::BigInt::Calc'    => '0.47',
	'Math::BigInt::CalcEmu' => '0.05',
	'Math::BigInt::Trace'   => '0.01',
	'Math::BigRat'          => '0.15',
	'Math::Complex'         => '1.35',
	'Math::Trig'            => '1.03',
	'Memoize'               => '1.01',
	'Memoize::AnyDBM_File'  => '0.65',
	'Memoize::Expire'       => '1.00',
	'Memoize::ExpireFile'   => '1.01',
	'Memoize::ExpireTest'   => '0.65',
	'Memoize::NDBM_File'    => '0.65',
	'Memoize::SDBM_File'    => '0.65',
	'Memoize::Storable'     => '0.65',
	'NDBM_File'             => '1.06',
	'NEXT'                  => '0.60',
	'Net::Cmd'              => '2.26',
	'Net::Config'           => '1.10',
	'Net::Domain'           => '2.19',
	'Net::FTP'              => '2.75',
	'Net::FTP::A'           => '1.16',
	'Net::FTP::E'           => '0.01',
	'Net::FTP::I'           => '1.12',
	'Net::FTP::L'           => '0.01',
	'Net::FTP::dataconn'    => '0.11',
	'Net::NNTP'             => '2.23',
	'Net::Netrc'            => '2.12',
	'Net::POP3'             => '2.28',
	'Net::Ping'             => '2.31',
	'Net::SMTP'             => '2.29',
	'Net::Time'             => '2.10',
	'Net::hostent'          => '1.01',
	'Net::netent'           => '1.00',
	'Net::protoent'         => '1.00',
	'Net::servent'          => '1.01',
	'O'                     => '1.00',
	'ODBM_File'             => '1.06',
	'Opcode'                => '1.06',
	'POSIX'                 => '1.09',
	'PerlIO'                => '1.04',
	'PerlIO::encoding'      => '0.09',
	'PerlIO::scalar'        => '0.04',
	'PerlIO::via'           => '0.03',
	'PerlIO::via::QuotedPrint'=> '0.06',
	'Pod::Checker'          => '1.43',
	'Pod::Find'             => '1.34',
	'Pod::Functions'        => '1.03',
	'Pod::Html'             => '1.0504',
	'Pod::InputObjects'     => '1.3',
	'Pod::LaTeX'            => '0.58',
	'Pod::Man'              => '1.37',
	'Pod::ParseLink'        => '1.06',
	'Pod::ParseUtils'       => '1.33',
	'Pod::Parser'           => '1.32',
	'Pod::Perldoc'          => '3.14',
	'Pod::Perldoc::BaseTo'  => undef,
	'Pod::Perldoc::GetOptsOO'=> undef,
	'Pod::Perldoc::ToChecker'=> undef,
	'Pod::Perldoc::ToMan'   => undef,
	'Pod::Perldoc::ToNroff' => undef,
	'Pod::Perldoc::ToPod'   => undef,
	'Pod::Perldoc::ToRtf'   => undef,
	'Pod::Perldoc::ToText'  => undef,
	'Pod::Perldoc::ToTk'    => undef,
	'Pod::Perldoc::ToXml'   => undef,
	'Pod::PlainText'        => '2.02',
	'Pod::Plainer'          => '0.01',
	'Pod::Select'           => '1.3',
	'Pod::Text'             => '2.21',
	'Pod::Text::Color'      => '1.04',
	'Pod::Text::Overstrike' => '1.1',
	'Pod::Text::Termcap'    => '1.11',
	'Pod::Usage'            => '1.33',
	'SDBM_File'             => '1.05',
	'Safe'                  => '2.12',
	'Scalar::Util'          => '1.18',
	'Search::Dict'          => '1.02',
	'SelectSaver'           => '1.01',
	'SelfLoader'            => '1.0904',
	'Shell'                 => '0.6',
	'Socket'                => '1.78',
	'Stdio'                 => '2.3',
	'Storable'              => '2.15',
	'Switch'                => '2.10_01',
	'Symbol'                => '1.06',
	'Sys::Hostname'         => '1.11',
	'Sys::Syslog'           => '0.13',
	'Term::ANSIColor'       => '1.10',
	'Term::Cap'             => '1.09',
	'Term::Complete'        => '1.402',
	'Term::ReadLine'        => '1.02',
	'Test'                  => '1.25',
	'Test::Builder'         => '0.32',
	'Test::Builder::Module' => '0.02',
	'Test::Builder::Tester' => '1.02',
	'Test::Builder::Tester::Color'=> undef,
	'Test::Harness'         => '2.56',
	'Test::Harness::Assert' => '0.02',
	'Test::Harness::Iterator'=> '0.02',
	'Test::Harness::Point'  => '0.01',
	'Test::Harness::Straps' => '0.26',
	'Test::More'            => '0.62',
	'Test::Simple'          => '0.62',
	'Text::Abbrev'          => '1.01',
	'Text::Balanced'        => '1.95',
	'Text::ParseWords'      => '3.24',
	'Text::Soundex'         => '1.01',
	'Text::Tabs'            => '2005.0824',
	'Text::Wrap'            => '2005.082401',
	'Thread'                => '2.00',
	'Thread::Queue'         => '2.00',
	'Thread::Semaphore'     => '2.01',
	'Thread::Signal'        => '1.00',
	'Thread::Specific'      => '1.00',
	'Tie::Array'            => '1.03',
	'Tie::File'             => '0.97',
	'Tie::Handle'           => '4.1',
	'Tie::Hash'             => '1.02',
	'Tie::Memoize'          => '1.0',
	'Tie::RefHash'          => '1.32',
	'Tie::Scalar'           => '1.00',
	'Tie::SubstrHash'       => '1.00',
	'Time::HiRes'           => '1.86',
	'Time::Local'           => '1.11',
	'Time::gmtime'          => '1.02',
	'Time::localtime'       => '1.02',
	'Time::tm'              => '1.00',
	'UNIVERSAL'             => '1.01',
        'Unicode'               => '4.1.0',
	'Unicode::Collate'      => '0.52',
	'Unicode::Normalize'    => '0.32',
	'Unicode::UCD'          => '0.24',
	'User::grent'           => '1.01',
	'User::pwent'           => '1.00',
	'Win32'                 => '0.2601',
	'XS::APItest'           => '0.08',
	'XS::Typemap'           => '0.02',
	'XSLoader'              => '0.06',
	'XSSymSet'              => '1.0',
	'attributes'            => '0.06',
	'attrs'                 => '1.02',
	'autouse'               => '1.05',
	'base'                  => '2.07',
	'bigint'                => '0.07',
	'bignum'                => '0.17',
	'bigrat'                => '0.08',
	'blib'                  => '1.03',
	'bytes'                 => '1.02',
	'charnames'             => '1.05',
	'constant'              => '1.05',
	'diagnostics'           => '1.15',
	'encoding'              => '2.02',
	'fields'                => '2.03',
	'filetest'              => '1.01',
	'if'                    => '0.05',
	'integer'               => '1.00',
	'less'                  => '0.01',
	'lib'                   => '0.5565',
	'locale'                => '1.00',
	'open'                  => '1.05',
	'ops'                   => '1.01',
	'overload'              => '1.04',
	're'                    => '0.05',
	'sigtrap'               => '1.02',
	'sort'                  => '1.02',
	'strict'                => '1.03',
	'subs'                  => '1.00',
	'threads'               => '1.07',
	'threads::shared'       => '0.94',
	'utf8'                  => '1.06',
	'vars'                  => '1.01',
	'vmsish'                => '1.02',
	'warnings'              => '1.05',
	'warnings::register'    => '1.01',
    },

    5.009004 => {
	'AnyDBM_File'           => '1.00',
	'Archive::Tar'          => '1.30_01',
	'Archive::Tar::Constant'=> '0.02',
	'Archive::Tar::File'    => '0.02',
	'Attribute::Handlers'   => '0.78_02',
	'AutoLoader'            => '5.61',
	'AutoSplit'             => '1.04_01',
	'B'                     => '1.11',
	'B::Asmdata'            => '1.01',
	'B::Assembler'          => '0.07',
	'B::Bblock'             => '1.02',
	'B::Bytecode'           => '1.02',
	'B::C'                  => '1.05',
	'B::CC'                 => '1.00',
	'B::Concise'            => '0.69',
	'B::Debug'              => '1.02',
	'B::Deparse'            => '0.76',
	'B::Disassembler'       => '1.05',
	'B::Lint'               => '1.08',
	'B::Showlex'            => '1.02',
	'B::Stackobj'           => '1.00',
	'B::Stash'              => '1.00',
	'B::Terse'              => '1.03',
	'B::Xref'               => '1.01',
	'Benchmark'             => '1.08',
	'ByteLoader'            => '0.06',
	'CGI'                   => '3.20',
	'CGI::Apache'           => '1.00',
	'CGI::Carp'             => '1.29',
	'CGI::Cookie'           => '1.27',
	'CGI::Fast'             => '1.07',
	'CGI::Pretty'           => '1.08',
	'CGI::Push'             => '1.04',
	'CGI::Switch'           => '1.00',
	'CGI::Util'             => '1.5',
	'CPAN'                  => '1.87_55',
	'CPAN::Debug'           => '5.400561',
	'CPAN::FirstTime'       => '5.400742',
	'CPAN::HandleConfig'    => '5.400740',
	'CPAN::Nox'             => '5.400561',
	'CPAN::Tarzip'          => '5.400714',
	'CPAN::Version'         => '5.400561',
	'Carp'                  => '1.05',
	'Carp::Heavy'           => '1.05',
	'Class::ISA'            => '0.33',
	'Class::Struct'         => '0.63',
	'Compress::Raw::Zlib'   => '2.000_13',
	'Compress::Zlib'        => '2.000_13',
        'Config'                => undef,
	'Config::Extensions'    => '0.01',
	'Cwd'                   => '3.19',
	'DB'                    => '1.01',
	'DBM_Filter'            => '0.01',
	'DBM_Filter::compress'  => '0.01',
	'DBM_Filter::encode'    => '0.01',
	'DBM_Filter::int32'     => '0.01',
	'DBM_Filter::null'      => '0.01',
	'DBM_Filter::utf8'      => '0.01',
	'DB_File'               => '1.814',
	'DCLsym'                => '1.02',
	'Data::Dumper'          => '2.121_08',
	'Devel::DProf'          => '20050603.00',
	'Devel::PPPort'         => '3.10',
	'Devel::Peek'           => '1.03',
	'Devel::SelfStubber'    => '1.03',
	'Digest'                => '1.15',
	'Digest::MD5'           => '2.36',
	'Digest::SHA'           => '5.43',
	'Digest::base'          => '1.00',
	'Digest::file'          => '1.00',
	'DirHandle'             => '1.01',
	'Dumpvalue'             => '1.12',
	'DynaLoader'            => '1.07',
	'Encode'                => '2.18_01',
	'Encode::Alias'         => '2.06',
	'Encode::Byte'          => '2.02',
	'Encode::CJKConstants'  => '2.02',
	'Encode::CN'            => '2.02',
	'Encode::CN::HZ'        => '2.04',
	'Encode::Config'        => '2.03',
	'Encode::EBCDIC'        => '2.02',
	'Encode::Encoder'       => '2.01',
	'Encode::Encoding'      => '2.04',
	'Encode::Guess'         => '2.02',
	'Encode::JP'            => '2.03',
	'Encode::JP::H2Z'       => '2.02',
	'Encode::JP::JIS7'      => '2.02',
	'Encode::KR'            => '2.02',
	'Encode::KR::2022_KR'   => '2.02',
	'Encode::MIME::Header'  => '2.04',
	'Encode::MIME::Header::ISO_2022_JP'=> '1.03',
	'Encode::Symbol'        => '2.02',
	'Encode::TW'            => '2.02',
	'Encode::Unicode'       => '2.03',
	'Encode::Unicode::UTF7' => '2.04',
	'English'               => '1.04',
	'Env'                   => '1.00',
	'Errno'                 => '1.09_01',
	'Exporter'              => '5.59',
	'Exporter::Heavy'       => '5.59',
	'ExtUtils::CBuilder'    => '0.18',
	'ExtUtils::CBuilder::Base'=> '0.12',
	'ExtUtils::CBuilder::Platform::Unix'=> '0.12',
	'ExtUtils::CBuilder::Platform::VMS'=> '0.12',
	'ExtUtils::CBuilder::Platform::Windows'=> '0.12_01',
	'ExtUtils::CBuilder::Platform::aix'=> '0.12',
	'ExtUtils::CBuilder::Platform::cygwin'=> '0.12',
	'ExtUtils::CBuilder::Platform::darwin'=> '0.12',
	'ExtUtils::CBuilder::Platform::dec_osf'=> '0.01',
	'ExtUtils::CBuilder::Platform::os2'=> '0.13',
	'ExtUtils::Command'     => '1.09',
	'ExtUtils::Command::MM' => '0.05_01',
	'ExtUtils::Constant'    => '0.2',
	'ExtUtils::Constant::Base'=> '0.03',
	'ExtUtils::Constant::ProxySubs'=> '0.03',
	'ExtUtils::Constant::Utils'=> '0.01',
	'ExtUtils::Constant::XS'=> '0.02',
	'ExtUtils::Embed'       => '1.26',
	'ExtUtils::Install'     => '1.41',
	'ExtUtils::Installed'   => '1.41',
	'ExtUtils::Liblist'     => '1.01',
	'ExtUtils::Liblist::Kid'=> '1.3',
	'ExtUtils::MM'          => '0.05',
	'ExtUtils::MM_AIX'      => '0.03',
	'ExtUtils::MM_Any'      => '0.13_02',
	'ExtUtils::MM_BeOS'     => '1.05',
	'ExtUtils::MM_Cygwin'   => '1.08',
	'ExtUtils::MM_DOS'      => '0.02',
	'ExtUtils::MM_MacOS'    => '1.08',
	'ExtUtils::MM_NW5'      => '2.08_01',
	'ExtUtils::MM_OS2'      => '1.05',
	'ExtUtils::MM_QNX'      => '0.02',
	'ExtUtils::MM_UWIN'     => '0.02',
	'ExtUtils::MM_Unix'     => '1.5003',
	'ExtUtils::MM_VMS'      => '5.73_03',
	'ExtUtils::MM_VOS'      => '0.02',
	'ExtUtils::MM_Win32'    => '1.12_02',
	'ExtUtils::MM_Win95'    => '0.04_01',
	'ExtUtils::MY'          => '0.01',
	'ExtUtils::MakeMaker'   => '6.30_02',
	'ExtUtils::MakeMaker::Config'=> '0.02',
	'ExtUtils::MakeMaker::bytes'=> '0.01',
	'ExtUtils::MakeMaker::vmsish'=> '0.01',
	'ExtUtils::Manifest'    => '1.46_01',
        'ExtUtils::Miniperl'    => undef,
	'ExtUtils::Mkbootstrap' => '1.15_01',
	'ExtUtils::Mksymlists'  => '1.19_01',
	'ExtUtils::Packlist'    => '1.41',
	'ExtUtils::ParseXS'     => '2.15_02',
	'ExtUtils::testlib'     => '1.15',
	'Fatal'                 => '1.04',
	'Fcntl'                 => '1.05',
	'File::Basename'        => '2.75',
	'File::CheckTree'       => '4.3',
	'File::Compare'         => '1.1005',
	'File::Copy'            => '2.09',
	'File::DosGlob'         => '1.00',
	'File::Find'            => '1.11',
	'File::Glob'            => '1.05',
	'File::GlobMapper'      => '0.000_02',
	'File::Path'            => '1.08',
	'File::Spec'            => '3.19',
	'File::Spec::Cygwin'    => '1.1',
	'File::Spec::Epoc'      => '1.1',
	'File::Spec::Functions' => '1.3',
	'File::Spec::Mac'       => '1.4',
	'File::Spec::OS2'       => '1.2',
	'File::Spec::Unix'      => '1.5',
	'File::Spec::VMS'       => '1.4',
	'File::Spec::Win32'     => '1.6',
	'File::Temp'            => '0.16_01',
	'File::stat'            => '1.00',
	'FileCache'             => '1.07',
	'FileHandle'            => '2.01',
	'VMS::Filespec'         => '1.11',
	'Filter::Simple'        => '0.82',
	'Filter::Util::Call'    => '1.0601',
	'FindBin'               => '1.47',
	'GDBM_File'             => '1.08',
	'Getopt::Long'          => '2.3501',
	'Getopt::Std'           => '1.05',
	'Hash::Util'            => '0.07',
	'Hash::Util::FieldHash' => '0.01',
	'I18N::Collate'         => '1.00',
	'I18N::LangTags'        => '0.35',
	'I18N::LangTags::Detect'=> '1.03',
	'I18N::LangTags::List'  => '0.35',
	'I18N::Langinfo'        => '0.02',
	'IO'                    => '1.23_01',
	'IO::Compress::Adapter::Deflate'=> '2.000_13',
	'IO::Compress::Adapter::Identity'=> '2.000_13',
	'IO::Compress::Base'    => '2.000_13',
	'IO::Compress::Base::Common'=> '2.000_13',
	'IO::Compress::Deflate' => '2.000_13',
	'IO::Compress::Gzip'    => '2.000_13',
	'IO::Compress::Gzip::Constants'=> '2.000_13',
	'IO::Compress::RawDeflate'=> '2.000_13',
	'IO::Compress::Zip'     => '2.000_13',
	'IO::Compress::Zip::Constants'=> '2.000_13',
	'IO::Compress::Zlib::Constants'=> '2.000_13',
	'IO::Compress::Zlib::Extra'=> '2.000_13',
	'IO::Dir'               => '1.06',
	'IO::File'              => '1.14',
	'IO::Handle'            => '1.27',
	'IO::Pipe'              => '1.13',
	'IO::Poll'              => '0.07',
	'IO::Seekable'          => '1.10',
	'IO::Select'            => '1.17',
	'IO::Socket'            => '1.30_01',
	'IO::Socket::INET'      => '1.31',
	'IO::Socket::UNIX'      => '1.23',
	'IO::Uncompress::Adapter::Identity'=> '2.000_13',
	'IO::Uncompress::Adapter::Inflate'=> '2.000_13',
	'IO::Uncompress::AnyInflate'=> '2.000_13',
	'IO::Uncompress::AnyUncompress'=> '2.000_13',
	'IO::Uncompress::Base'  => '2.000_13',
	'IO::Uncompress::Gunzip'=> '2.000_13',
	'IO::Uncompress::Inflate'=> '2.000_13',
	'IO::Uncompress::RawInflate'=> '2.000_13',
	'IO::Uncompress::Unzip' => '2.000_13',
	'IO::Zlib'              => '1.04_02',
	'IPC::Msg'              => '1.02',
	'IPC::Open2'            => '1.02',
	'IPC::Open3'            => '1.02',
	'IPC::Semaphore'        => '1.02',
	'IPC::SysV'             => '1.04',
	'List::Util'            => '1.18',
	'Locale::Constants'     => '2.07',
	'Locale::Country'       => '2.07',
	'Locale::Currency'      => '2.07',
	'Locale::Language'      => '2.07',
	'Locale::Maketext'      => '1.10_01',
	'Locale::Maketext::Guts'=> undef,
	'Locale::Maketext::GutsLoader'=> undef,
	'Locale::Script'        => '2.07',
	'MIME::Base64'          => '3.07_01',
	'MIME::QuotedPrint'     => '3.07',
	'Math::BigFloat'        => '1.51',
	'Math::BigFloat::Trace' => '0.01',
	'Math::BigInt'          => '1.77',
	'Math::BigInt::Calc'    => '0.47',
	'Math::BigInt::CalcEmu' => '0.05',
	'Math::BigInt::FastCalc'=> '0.10',
	'Math::BigInt::Trace'   => '0.01',
	'Math::BigRat'          => '0.15',
	'Math::Complex'         => '1.36',
	'Math::Trig'            => '1.04',
	'Memoize'               => '1.01_01',
	'Memoize::AnyDBM_File'  => '0.65',
	'Memoize::Expire'       => '1.00',
	'Memoize::ExpireFile'   => '1.01',
	'Memoize::ExpireTest'   => '0.65',
	'Memoize::NDBM_File'    => '0.65',
	'Memoize::SDBM_File'    => '0.65',
	'Memoize::Storable'     => '0.65',
	'Module::Build'         => '0.2805',
	'Module::Build::Base'   => undef,
	'Module::Build::Compat' => '0.03',
	'Module::Build::ConfigData'=> undef,
	'Module::Build::Cookbook'=> undef,
	'Module::Build::ModuleInfo'=> undef,
	'Module::Build::Notes'  => undef,
	'Module::Build::PPMMaker'=> undef,
	'Module::Build::Platform::Amiga'=> undef,
	'Module::Build::Platform::Default'=> undef,
	'Module::Build::Platform::EBCDIC'=> undef,
	'Module::Build::Platform::MPEiX'=> undef,
	'Module::Build::Platform::MacOS'=> undef,
	'Module::Build::Platform::RiscOS'=> undef,
	'Module::Build::Platform::Unix'=> undef,
	'Module::Build::Platform::VMS'=> undef,
	'Module::Build::Platform::VOS'=> undef,
	'Module::Build::Platform::Windows'=> undef,
	'Module::Build::Platform::aix'=> undef,
	'Module::Build::Platform::cygwin'=> undef,
	'Module::Build::Platform::darwin'=> undef,
	'Module::Build::Platform::os2'=> undef,
	'Module::Build::PodParser'=> undef,
	'Module::Build::Version'=> '0',
	'Module::Build::YAML'   => '0.50',
	'Module::CoreList'      => '2.08',
	'Module::Load'          => '0.10',
	'Module::Loaded'        => '0.01',
	'Moped::Msg'            => '0.01',
	'NDBM_File'             => '1.06',
	'NEXT'                  => '0.60_01',
	'Net::Cmd'              => '2.26_01',
	'Net::Config'           => '1.10',
	'Net::Domain'           => '2.19_01',
	'Net::FTP'              => '2.75',
	'Net::FTP::A'           => '1.16',
	'Net::FTP::E'           => '0.01',
	'Net::FTP::I'           => '1.12',
	'Net::FTP::L'           => '0.01',
	'Net::FTP::dataconn'    => '0.11',
	'Net::NNTP'             => '2.23',
	'Net::Netrc'            => '2.12',
	'Net::POP3'             => '2.28',
	'Net::Ping'             => '2.31_04',
	'Net::SMTP'             => '2.29',
	'Net::Time'             => '2.10',
	'Net::hostent'          => '1.01',
	'Net::netent'           => '1.00',
	'Net::protoent'         => '1.00',
	'Net::servent'          => '1.01',
	'O'                     => '1.00',
	'ODBM_File'             => '1.06',
	'Opcode'                => '1.08',
	'POSIX'                 => '1.10',
	'Package::Constants'    => '0.01',
	'PerlIO'                => '1.04',
	'PerlIO::encoding'      => '0.09',
	'PerlIO::scalar'        => '0.04',
	'PerlIO::via'           => '0.03',
	'PerlIO::via::QuotedPrint'=> '0.06',
	'Pod::Checker'          => '1.43',
	'Pod::Escapes'          => '1.04',
	'Pod::Find'             => '1.34',
	'Pod::Functions'        => '1.03',
	'Pod::Html'             => '1.07',
	'Pod::InputObjects'     => '1.3',
	'Pod::LaTeX'            => '0.58',
	'Pod::Man'              => '2.09',
	'Pod::ParseLink'        => '1.06',
	'Pod::ParseUtils'       => '1.33',
	'Pod::Parser'           => '1.32',
	'Pod::Perldoc'          => '3.14_01',
	'Pod::Perldoc::BaseTo'  => undef,
	'Pod::Perldoc::GetOptsOO'=> undef,
	'Pod::Perldoc::ToChecker'=> undef,
	'Pod::Perldoc::ToMan'   => undef,
	'Pod::Perldoc::ToNroff' => undef,
	'Pod::Perldoc::ToPod'   => undef,
	'Pod::Perldoc::ToRtf'   => undef,
	'Pod::Perldoc::ToText'  => undef,
	'Pod::Perldoc::ToTk'    => undef,
	'Pod::Perldoc::ToXml'   => undef,
	'Pod::PlainText'        => '2.02',
	'Pod::Plainer'          => '0.01',
	'Pod::Select'           => '1.3',
	'Pod::Simple'           => '3.04',
	'Pod::Simple::BlackBox' => undef,
	'Pod::Simple::Checker'  => '2.02',
	'Pod::Simple::Debug'    => undef,
	'Pod::Simple::DumpAsText'=> '2.02',
	'Pod::Simple::DumpAsXML'=> '2.02',
	'Pod::Simple::HTML'     => '3.03',
	'Pod::Simple::HTMLBatch'=> '3.02',
	'Pod::Simple::HTMLLegacy'=> '5.01',
	'Pod::Simple::LinkSection'=> undef,
	'Pod::Simple::Methody'  => '2.02',
	'Pod::Simple::Progress' => '1.01',
	'Pod::Simple::PullParser'=> '2.02',
	'Pod::Simple::PullParserEndToken'=> undef,
	'Pod::Simple::PullParserStartToken'=> undef,
	'Pod::Simple::PullParserTextToken'=> undef,
	'Pod::Simple::PullParserToken'=> '2.02',
	'Pod::Simple::RTF'      => '2.02',
	'Pod::Simple::Search'   => '3.04',
	'Pod::Simple::SimpleTree'=> '2.02',
	'Pod::Simple::Text'     => '2.02',
	'Pod::Simple::TextContent'=> '2.02',
	'Pod::Simple::TiedOutFH'=> undef,
	'Pod::Simple::Transcode'=> undef,
	'Pod::Simple::TranscodeDumb'=> '2.02',
	'Pod::Simple::TranscodeSmart'=> undef,
	'Pod::Simple::XMLOutStream'=> '2.02',
	'Pod::Text'             => '3.07',
	'Pod::Text::Color'      => '2.03',
	'Pod::Text::Overstrike' => '2',
	'Pod::Text::Termcap'    => '2.03',
	'Pod::Usage'            => '1.33_01',
	'SDBM_File'             => '1.06',
	'Safe'                  => '2.12',
	'Scalar::Util'          => '1.18',
	'Search::Dict'          => '1.02',
	'SelectSaver'           => '1.01',
	'SelfLoader'            => '1.0905',
	'Shell'                 => '0.7',
	'Socket'                => '1.78',
	'Stdio'                 => '2.3',
	'Storable'              => '2.15_02',
	'Switch'                => '2.10_01',
	'Symbol'                => '1.06',
	'Sys::Hostname'         => '1.11',
	'Sys::Syslog'           => '0.17',
	'Term::ANSIColor'       => '1.11',
	'Term::Cap'             => '1.09',
	'Term::Complete'        => '1.402',
	'Term::ReadLine'        => '1.02',
	'Test'                  => '1.25',
	'Test::Builder'         => '0.33',
	'Test::Builder::Module' => '0.03',
	'Test::Builder::Tester' => '1.04',
	'Test::Builder::Tester::Color'=> undef,
	'Test::Harness'         => '2.62',
	'Test::Harness::Assert' => '0.02',
	'Test::Harness::Iterator'=> '0.02',
	'Test::Harness::Point'  => '0.01',
	'Test::Harness::Straps' => '0.26',
	'Test::Harness::Util'   => '0.01',
	'Test::More'            => '0.64',
	'Test::Simple'          => '0.64',
	'Text::Abbrev'          => '1.01',
	'Text::Balanced'        => '1.98_01',
	'Text::ParseWords'      => '3.25',
	'Text::Soundex'         => '1.01',
	'Text::Tabs'            => '2007.071101',
	'Text::Wrap'            => '2006.0711',
	'Thread'                => '2.00',
	'Thread::Queue'         => '2.00',
	'Thread::Semaphore'     => '2.01',
	'Thread::Signal'        => '1.00',
	'Thread::Specific'      => '1.00',
	'Tie::Array'            => '1.03',
	'Tie::File'             => '0.97_01',
	'Tie::Handle'           => '4.1',
	'Tie::Hash'             => '1.02',
	'Tie::Memoize'          => '1.0',
	'Tie::RefHash'          => '1.34_01',
	'Tie::Scalar'           => '1.00',
	'Tie::SubstrHash'       => '1.00',
	'Time::HiRes'           => '1.87',
	'Time::Local'           => '1.13',
	'Time::gmtime'          => '1.03',
	'Time::localtime'       => '1.02',
	'Time::tm'              => '1.00',
	'UNIVERSAL'             => '1.04',
        'Unicode'               => '4.1.0',
	'Unicode::Collate'      => '0.52',
	'Unicode::Normalize'    => '1.01',
	'Unicode::UCD'          => '0.24',
	'User::grent'           => '1.01',
	'User::pwent'           => '1.00',
	'Win32'                 => '0.2601',
	'Win32API::File'        => '0.1001',
	'Win32API::File::ExtUtils::Myconst2perl'=> '1',
	'XS::APItest'           => '0.09',
	'XS::Typemap'           => '0.02',
	'XSLoader'              => '0.06',
	'XSSymSet'              => '1.0',
	'assertions'            => '0.03',
	'assertions::activate'  => '0.02',
	'assertions::compat'    => '0.02',
	'attributes'            => '0.06',
	'attrs'                 => '1.02',
	'autouse'               => '1.06',
	'base'                  => '2.07',
	'bigint'                => '0.07',
	'bignum'                => '0.17',
	'bigrat'                => '0.08',
	'blib'                  => '1.03',
	'bytes'                 => '1.02',
	'charnames'             => '1.05',
	'constant'              => '1.07',
	'diagnostics'           => '1.16',
	'encoding'              => '2.04',
	'encoding::warnings'    => '0.10',
	'feature'               => '1.01',
	'fields'                => '2.03',
	'filetest'              => '1.01',
	'if'                    => '0.05',
	'integer'               => '1.00',
	'less'                  => '0.01',
	'lib'                   => '0.5565',
	'locale'                => '1.00',
	'open'                  => '1.05',
	'ops'                   => '1.01',
	'overload'              => '1.04',
	're'                    => '0.0601',
	'sigtrap'               => '1.02',
	'sort'                  => '2.00',
	'strict'                => '1.03',
	'subs'                  => '1.00',
	'threads'               => '1.38',
	'threads::shared'       => '0.94_01',
	'utf8'                  => '1.06',
	'vars'                  => '1.01',
	'version'               => '0.67',
	'vmsish'                => '1.02',
	'warnings'              => '1.05',
	'warnings::register'    => '1.01',
    },

    5.009005 => {
	'AnyDBM_File'           => '1.00',
	'Archive::Extract'      => '0.22_01',
	'Archive::Tar'          => '1.32',
	'Archive::Tar::Constant'=> '0.02',
	'Archive::Tar::File'    => '0.02',
	'Attribute::Handlers'   => '0.78_06',
	'AutoLoader'            => '5.63',
	'AutoSplit'             => '1.05',
	'B'                     => '1.16',
	'B::Concise'            => '0.72',
	'B::Debug'              => '1.05',
	'B::Deparse'            => '0.82',
	'B::Lint'               => '1.09',
	'B::Showlex'            => '1.02',
	'B::Terse'              => '1.05',
	'B::Xref'               => '1.01',
	'Benchmark'             => '1.1',
	'CGI'                   => '3.29',
	'CGI::Apache'           => '1.00',
	'CGI::Carp'             => '1.29',
	'CGI::Cookie'           => '1.28',
	'CGI::Fast'             => '1.07',
	'CGI::Pretty'           => '1.08',
	'CGI::Push'             => '1.04',
	'CGI::Switch'           => '1.00',
	'CGI::Util'             => '1.5_01',
	'CPAN'                  => '1.9102',
	'CPAN::Debug'           => '5.400955',
	'CPAN::FirstTime'       => '5.401669',
	'CPAN::HandleConfig'    => '5.401744',
	'CPAN::Kwalify'         => '5.401418',
	'CPAN::Nox'             => '5.400844',
	'CPAN::Queue'           => '5.401704',
	'CPAN::Tarzip'          => '5.401717',
	'CPAN::Version'         => '5.401387',
	'CPANPLUS'              => '0.81_01',
	'CPANPLUS::Backend'     => undef,
	'CPANPLUS::Backend::RV' => undef,
	'CPANPLUS::Config'      => undef,
	'CPANPLUS::Configure'   => undef,
	'CPANPLUS::Configure::Setup'=> undef,
	'CPANPLUS::Dist'        => undef,
	'CPANPLUS::Dist::Base'  => '0.01',
	'CPANPLUS::Dist::Build' => '0.06_01',
	'CPANPLUS::Dist::Build::Constants'=> '0.01',
	'CPANPLUS::Dist::MM'    => undef,
	'CPANPLUS::Dist::Sample'=> undef,
	'CPANPLUS::Error'       => undef,
	'CPANPLUS::Internals'   => '0.81_01',
	'CPANPLUS::Internals::Constants'=> '0.01',
	'CPANPLUS::Internals::Constants::Report'=> '0.01',
	'CPANPLUS::Internals::Extract'=> undef,
	'CPANPLUS::Internals::Fetch'=> undef,
	'CPANPLUS::Internals::Report'=> undef,
	'CPANPLUS::Internals::Search'=> undef,
	'CPANPLUS::Internals::Source'=> undef,
	'CPANPLUS::Internals::Utils'=> undef,
	'CPANPLUS::Internals::Utils::Autoflush'=> undef,
	'CPANPLUS::Module'      => undef,
	'CPANPLUS::Module::Author'=> undef,
	'CPANPLUS::Module::Author::Fake'=> undef,
	'CPANPLUS::Module::Checksums'=> undef,
	'CPANPLUS::Module::Fake'=> undef,
	'CPANPLUS::Module::Signature'=> undef,
	'CPANPLUS::Selfupdate'  => undef,
	'CPANPLUS::Shell'       => undef,
	'CPANPLUS::Shell::Classic'=> '0.0562',
	'CPANPLUS::Shell::Default'=> '0.81_01',
	'CPANPLUS::Shell::Default::Plugins::Remote'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Source'=> undef,
	'CPANPLUS::inc'         => undef,
	'Carp'                  => '1.07',
	'Carp::Heavy'           => '1.07',
	'Class::ISA'            => '0.33',
	'Class::Struct'         => '0.63',
	'Compress::Raw::Zlib'   => '2.005',
	'Compress::Zlib'        => '2.005',
	'Config'                => undef,
	'Config::Extensions'    => '0.01',
	'Cwd'                   => '3.25',
	'DB'                    => '1.01',
	'DBM_Filter'            => '0.02',
	'DBM_Filter::compress'  => '0.01',
	'DBM_Filter::encode'    => '0.01',
	'DBM_Filter::int32'     => '0.01',
	'DBM_Filter::null'      => '0.01',
	'DBM_Filter::utf8'      => '0.01',
	'DB_File'               => '1.815',
	'DCLsym'                => '1.03',
	'Data::Dumper'          => '2.121_13',
	'Devel::DProf'          => '20050603.00',
	'Devel::InnerPackage'   => '0.3',
	'Devel::PPPort'         => '3.11_01',
	'Devel::Peek'           => '1.03',
	'Devel::SelfStubber'    => '1.03',
	'Digest'                => '1.15',
	'Digest::MD5'           => '2.36_01',
	'Digest::SHA'           => '5.44',
	'Digest::base'          => '1.00',
	'Digest::file'          => '1.00',
	'DirHandle'             => '1.01',
	'Dumpvalue'             => '1.12',
	'DynaLoader'            => '1.08',
	'Encode'                => '2.23',
	'Encode::Alias'         => '2.07',
	'Encode::Byte'          => '2.03',
	'Encode::CJKConstants'  => '2.02',
	'Encode::CN'            => '2.02',
	'Encode::CN::HZ'        => '2.04',
	'Encode::Config'        => '2.04',
	'Encode::EBCDIC'        => '2.02',
	'Encode::Encoder'       => '2.01',
	'Encode::Encoding'      => '2.05',
	'Encode::GSM0338'       => '2.00',
	'Encode::Guess'         => '2.02',
	'Encode::JP'            => '2.03',
	'Encode::JP::H2Z'       => '2.02',
	'Encode::JP::JIS7'      => '2.03',
	'Encode::KR'            => '2.02',
	'Encode::KR::2022_KR'   => '2.02',
	'Encode::MIME::Header'  => '2.05',
	'Encode::MIME::Header::ISO_2022_JP'=> '1.03',
	'Encode::MIME::Name'    => '1.01',
	'Encode::Symbol'        => '2.02',
	'Encode::TW'            => '2.02',
	'Encode::Unicode'       => '2.05',
	'Encode::Unicode::UTF7' => '2.04',
	'English'               => '1.04',
	'Env'                   => '1.00',
	'Errno'                 => '1.10',
	'Exporter'              => '5.60',
	'Exporter::Heavy'       => '5.60',
	'ExtUtils::CBuilder'    => '0.19',
	'ExtUtils::CBuilder::Base'=> '0.12',
	'ExtUtils::CBuilder::Platform::Unix'=> '0.12',
	'ExtUtils::CBuilder::Platform::VMS'=> '0.12',
	'ExtUtils::CBuilder::Platform::Windows'=> '0.13',
	'ExtUtils::CBuilder::Platform::aix'=> '0.12',
	'ExtUtils::CBuilder::Platform::cygwin'=> '0.12',
	'ExtUtils::CBuilder::Platform::darwin'=> '0.12',
	'ExtUtils::CBuilder::Platform::dec_osf'=> '0.01',
	'ExtUtils::CBuilder::Platform::os2'=> '0.13',
	'ExtUtils::Command'     => '1.13',
	'ExtUtils::Command::MM' => '0.07',
	'ExtUtils::Constant'    => '0.2',
	'ExtUtils::Constant::Base'=> '0.04',
	'ExtUtils::Constant::ProxySubs'=> '0.03',
	'ExtUtils::Constant::Utils'=> '0.01',
	'ExtUtils::Constant::XS'=> '0.02',
	'ExtUtils::Embed'       => '1.26',
	'ExtUtils::Install'     => '1.41_01',
	'ExtUtils::Installed'   => '1.41',
	'ExtUtils::Liblist'     => '1.03',
	'ExtUtils::Liblist::Kid'=> '1.33',
	'ExtUtils::MM'          => '0.07',
	'ExtUtils::MM_AIX'      => '0.05',
	'ExtUtils::MM_Any'      => '0.15',
	'ExtUtils::MM_BeOS'     => '1.07',
	'ExtUtils::MM_Cygwin'   => '1.1',
	'ExtUtils::MM_DOS'      => '0.04',
	'ExtUtils::MM_MacOS'    => '1.1',
	'ExtUtils::MM_NW5'      => '2.1',
	'ExtUtils::MM_OS2'      => '1.07',
	'ExtUtils::MM_QNX'      => '0.04',
	'ExtUtils::MM_UWIN'     => '0.04',
	'ExtUtils::MM_Unix'     => '1.54_01',
	'ExtUtils::MM_VMS'      => '5.76',
	'ExtUtils::MM_VOS'      => '0.04',
	'ExtUtils::MM_Win32'    => '1.15',
	'ExtUtils::MM_Win95'    => '0.06',
	'ExtUtils::MY'          => '0.03',
	'ExtUtils::MakeMaker'   => '6.36',
	'ExtUtils::MakeMaker::Config'=> '0.04',
	'ExtUtils::MakeMaker::bytes'=> '0.03',
	'ExtUtils::MakeMaker::vmsish'=> '0.03',
	'ExtUtils::Manifest'    => '1.51_01',
	'ExtUtils::Miniperl'    => undef,
	'ExtUtils::Mkbootstrap' => '1.17',
	'ExtUtils::Mksymlists'  => '1.21',
	'ExtUtils::Packlist'    => '1.41',
	'ExtUtils::ParseXS'     => '2.18',
	'ExtUtils::testlib'     => '1.17',
	'Fatal'                 => '1.05',
	'Fcntl'                 => '1.06',
	'File::Basename'        => '2.76',
	'File::CheckTree'       => '4.3',
	'File::Compare'         => '1.1005',
	'File::Copy'            => '2.10',
	'File::DosGlob'         => '1.00',
	'File::Fetch'           => '0.10',
	'File::Find'            => '1.11',
	'File::Glob'            => '1.06',
	'File::GlobMapper'      => '0.000_02',
	'File::Path'            => '2.01',
	'File::Spec'            => '3.25',
	'File::Spec::Cygwin'    => '1.1_01',
	'File::Spec::Epoc'      => '1.1',
	'File::Spec::Functions' => '1.3',
	'File::Spec::Mac'       => '1.4',
	'File::Spec::OS2'       => '1.2',
	'File::Spec::Unix'      => '1.5',
	'File::Spec::VMS'       => '1.4_01',
	'File::Spec::Win32'     => '1.6',
	'File::Temp'            => '0.18',
	'File::stat'            => '1.00',
	'FileCache'             => '1.07',
	'FileHandle'            => '2.01',
	'VMS::Filespec'         => '1.11',
	'Filter::Simple'        => '0.82',
	'Filter::Util::Call'    => '1.0602',
	'FindBin'               => '1.49',
	'GDBM_File'             => '1.08',
	'Getopt::Long'          => '2.36',
	'Getopt::Std'           => '1.05',
	'Hash::Util'            => '0.07',
	'Hash::Util::FieldHash' => '1.01',
	'I18N::Collate'         => '1.00',
	'I18N::LangTags'        => '0.35',
	'I18N::LangTags::Detect'=> '1.03',
	'I18N::LangTags::List'  => '0.35',
	'I18N::Langinfo'        => '0.02',
	'IO'                    => '1.23_01',
	'IO::Compress::Adapter::Deflate'=> '2.005',
	'IO::Compress::Adapter::Identity'=> '2.005',
	'IO::Compress::Base'    => '2.005',
	'IO::Compress::Base::Common'=> '2.005',
	'IO::Compress::Deflate' => '2.005',
	'IO::Compress::Gzip'    => '2.005',
	'IO::Compress::Gzip::Constants'=> '2.005',
	'IO::Compress::RawDeflate'=> '2.005',
	'IO::Compress::Zip'     => '2.005',
	'IO::Compress::Zip::Constants'=> '2.005',
	'IO::Compress::Zlib::Constants'=> '2.005',
	'IO::Compress::Zlib::Extra'=> '2.005',
	'IO::Dir'               => '1.06',
	'IO::File'              => '1.14',
	'IO::Handle'            => '1.27',
	'IO::Pipe'              => '1.13',
	'IO::Poll'              => '0.07',
	'IO::Seekable'          => '1.10',
	'IO::Select'            => '1.17',
	'IO::Socket'            => '1.30_01',
	'IO::Socket::INET'      => '1.31',
	'IO::Socket::UNIX'      => '1.23',
	'IO::Uncompress::Adapter::Identity'=> '2.005',
	'IO::Uncompress::Adapter::Inflate'=> '2.005',
	'IO::Uncompress::AnyInflate'=> '2.005',
	'IO::Uncompress::AnyUncompress'=> '2.005',
	'IO::Uncompress::Base'  => '2.005',
	'IO::Uncompress::Gunzip'=> '2.005',
	'IO::Uncompress::Inflate'=> '2.005',
	'IO::Uncompress::RawInflate'=> '2.005',
	'IO::Uncompress::Unzip' => '2.005',
	'IO::Zlib'              => '1.05_01',
	'IPC::Cmd'              => '0.36_01',
	'IPC::Msg'              => '1.02',
	'IPC::Open2'            => '1.02',
	'IPC::Open3'            => '1.02',
	'IPC::Semaphore'        => '1.02',
	'IPC::SysV'             => '1.04',
	'List::Util'            => '1.19',
	'Locale::Constants'     => '2.07',
	'Locale::Country'       => '2.07',
	'Locale::Currency'      => '2.07',
	'Locale::Language'      => '2.07',
	'Locale::Maketext'      => '1.10_01',
	'Locale::Maketext::Guts'=> undef,
	'Locale::Maketext::GutsLoader'=> undef,
	'Locale::Maketext::Simple'=> '0.18',
	'Locale::Script'        => '2.07',
	'Log::Message'          => '0.01',
	'Log::Message::Config'  => '0.01',
	'Log::Message::Handlers'=> undef,
	'Log::Message::Item'    => undef,
	'Log::Message::Simple'  => '0.0201',
	'MIME::Base64'          => '3.07_01',
	'MIME::QuotedPrint'     => '3.07',
	'Math::BigFloat'        => '1.58',
	'Math::BigFloat::Trace' => '0.01',
	'Math::BigInt'          => '1.87',
	'Math::BigInt::Calc'    => '0.51',
	'Math::BigInt::CalcEmu' => '0.05',
	'Math::BigInt::FastCalc'=> '0.15_01',
	'Math::BigInt::Trace'   => '0.01',
	'Math::BigRat'          => '0.19',
	'Math::Complex'         => '1.37',
	'Math::Trig'            => '1.04',
	'Memoize'               => '1.01_02',
	'Memoize::AnyDBM_File'  => '0.65',
	'Memoize::Expire'       => '1.00',
	'Memoize::ExpireFile'   => '1.01',
	'Memoize::ExpireTest'   => '0.65',
	'Memoize::NDBM_File'    => '0.65',
	'Memoize::SDBM_File'    => '0.65',
	'Memoize::Storable'     => '0.65',
	'Module::Build'         => '0.2808',
	'Module::Build::Base'   => undef,
	'Module::Build::Compat' => '0.03',
	'Module::Build::Config' => undef,
	'Module::Build::ConfigData'=> undef,
	'Module::Build::Cookbook'=> undef,
	'Module::Build::ModuleInfo'=> undef,
	'Module::Build::Notes'  => undef,
	'Module::Build::PPMMaker'=> undef,
	'Module::Build::Platform::Amiga'=> undef,
	'Module::Build::Platform::Default'=> undef,
	'Module::Build::Platform::EBCDIC'=> undef,
	'Module::Build::Platform::MPEiX'=> undef,
	'Module::Build::Platform::MacOS'=> undef,
	'Module::Build::Platform::RiscOS'=> undef,
	'Module::Build::Platform::Unix'=> undef,
	'Module::Build::Platform::VMS'=> undef,
	'Module::Build::Platform::VOS'=> undef,
	'Module::Build::Platform::Windows'=> undef,
	'Module::Build::Platform::aix'=> undef,
	'Module::Build::Platform::cygwin'=> undef,
	'Module::Build::Platform::darwin'=> undef,
	'Module::Build::Platform::os2'=> undef,
	'Module::Build::PodParser'=> undef,
	'Module::Build::Version'=> '0.7203',
	'Module::Build::YAML'   => '0.50',
	'Module::CoreList'      => '2.12',
	'Module::Load'          => '0.10',
	'Module::Load::Conditional'=> '0.16',
	'Module::Loaded'        => '0.01',
	'Module::Pluggable'     => '3.6',
	'Module::Pluggable::Object'=> '3.6',
	'Moped::Msg'            => '0.01',
	'NDBM_File'             => '1.07',
	'NEXT'                  => '0.60_01',
	'Net::Cmd'              => '2.28',
	'Net::Config'           => '1.11',
	'Net::Domain'           => '2.20',
	'Net::FTP'              => '2.77',
	'Net::FTP::A'           => '1.18',
	'Net::FTP::E'           => '0.01',
	'Net::FTP::I'           => '1.12',
	'Net::FTP::L'           => '0.01',
	'Net::FTP::dataconn'    => '0.11',
	'Net::NNTP'             => '2.24',
	'Net::Netrc'            => '2.12',
	'Net::POP3'             => '2.29',
	'Net::Ping'             => '2.31_04',
	'Net::SMTP'             => '2.31',
	'Net::Time'             => '2.10',
	'Net::hostent'          => '1.01',
	'Net::netent'           => '1.00',
	'Net::protoent'         => '1.00',
	'Net::servent'          => '1.01',
	'O'                     => '1.00',
	'ODBM_File'             => '1.07',
	'Object::Accessor'      => '0.32',
	'Opcode'                => '1.09',
	'POSIX'                 => '1.13',
	'Package::Constants'    => '0.01',
	'Params::Check'         => '0.26',
	'PerlIO'                => '1.04',
	'PerlIO::encoding'      => '0.10',
	'PerlIO::scalar'        => '0.05',
	'PerlIO::via'           => '0.04',
	'PerlIO::via::QuotedPrint'=> '0.06',
	'Pod::Checker'          => '1.43',
	'Pod::Escapes'          => '1.04',
	'Pod::Find'             => '1.34',
	'Pod::Functions'        => '1.03',
	'Pod::Html'             => '1.08',
	'Pod::InputObjects'     => '1.3',
	'Pod::LaTeX'            => '0.58',
	'Pod::Man'              => '2.12',
	'Pod::ParseLink'        => '1.06',
	'Pod::ParseUtils'       => '1.35',
	'Pod::Parser'           => '1.35',
	'Pod::Perldoc'          => '3.14_01',
	'Pod::Perldoc::BaseTo'  => undef,
	'Pod::Perldoc::GetOptsOO'=> undef,
	'Pod::Perldoc::ToChecker'=> undef,
	'Pod::Perldoc::ToMan'   => undef,
	'Pod::Perldoc::ToNroff' => undef,
	'Pod::Perldoc::ToPod'   => undef,
	'Pod::Perldoc::ToRtf'   => undef,
	'Pod::Perldoc::ToText'  => undef,
	'Pod::Perldoc::ToTk'    => undef,
	'Pod::Perldoc::ToXml'   => undef,
	'Pod::PlainText'        => '2.02',
	'Pod::Plainer'          => '0.01',
	'Pod::Select'           => '1.35',
	'Pod::Simple'           => '3.05',
	'Pod::Simple::BlackBox' => undef,
	'Pod::Simple::Checker'  => '2.02',
	'Pod::Simple::Debug'    => undef,
	'Pod::Simple::DumpAsText'=> '2.02',
	'Pod::Simple::DumpAsXML'=> '2.02',
	'Pod::Simple::HTML'     => '3.03',
	'Pod::Simple::HTMLBatch'=> '3.02',
	'Pod::Simple::HTMLLegacy'=> '5.01',
	'Pod::Simple::LinkSection'=> undef,
	'Pod::Simple::Methody'  => '2.02',
	'Pod::Simple::Progress' => '1.01',
	'Pod::Simple::PullParser'=> '2.02',
	'Pod::Simple::PullParserEndToken'=> undef,
	'Pod::Simple::PullParserStartToken'=> undef,
	'Pod::Simple::PullParserTextToken'=> undef,
	'Pod::Simple::PullParserToken'=> '2.02',
	'Pod::Simple::RTF'      => '2.02',
	'Pod::Simple::Search'   => '3.04',
	'Pod::Simple::SimpleTree'=> '2.02',
	'Pod::Simple::Text'     => '2.02',
	'Pod::Simple::TextContent'=> '2.02',
	'Pod::Simple::TiedOutFH'=> undef,
	'Pod::Simple::Transcode'=> undef,
	'Pod::Simple::TranscodeDumb'=> '2.02',
	'Pod::Simple::TranscodeSmart'=> undef,
	'Pod::Simple::XMLOutStream'=> '2.02',
	'Pod::Text'             => '3.08',
	'Pod::Text::Color'      => '2.03',
	'Pod::Text::Overstrike' => '2',
	'Pod::Text::Termcap'    => '2.03',
	'Pod::Usage'            => '1.35',
	'SDBM_File'             => '1.06',
	'Safe'                  => '2.12',
	'Scalar::Util'          => '1.19',
	'Search::Dict'          => '1.02',
	'SelectSaver'           => '1.01',
	'SelfLoader'            => '1.11',
	'Shell'                 => '0.72_01',
	'Socket'                => '1.79',
	'Stdio'                 => '2.3',
	'Storable'              => '2.16',
	'Switch'                => '2.13',
	'Symbol'                => '1.06',
	'Sys::Hostname'         => '1.11',
	'Sys::Syslog'           => '0.18_01',
	'Term::ANSIColor'       => '1.12',
	'Term::Cap'             => '1.09',
	'Term::Complete'        => '1.402',
	'Term::ReadLine'        => '1.02',
	'Term::UI'              => '0.14_01',
	'Term::UI::History'     => undef,
	'Test'                  => '1.25',
	'Test::Builder'         => '0.70',
	'Test::Builder::Module' => '0.68',
	'Test::Builder::Tester' => '1.07',
	'Test::Builder::Tester::Color'=> undef,
	'Test::Harness'         => '2.64',
	'Test::Harness::Assert' => '0.02',
	'Test::Harness::Iterator'=> '0.02',
	'Test::Harness::Point'  => '0.01',
	'Test::Harness::Results'=> '0.01',
	'Test::Harness::Straps' => '0.26',
	'Test::Harness::Util'   => '0.01',
	'Test::More'            => '0.70',
	'Test::Simple'          => '0.70',
	'Text::Abbrev'          => '1.01',
	'Text::Balanced'        => '2.0.0',
	'Text::ParseWords'      => '3.25',
	'Text::Soundex'         => '3.02',
	'Text::Tabs'            => '2007.1117',
	'Text::Wrap'            => '2006.1117',
	'Thread'                => '3.02',
	'Thread::Queue'         => '2.00',
	'Thread::Semaphore'     => '2.01',
	'Tie::Array'            => '1.03',
	'Tie::File'             => '0.97_02',
	'Tie::Handle'           => '4.1',
	'Tie::Hash'             => '1.02',
	'Tie::Hash::NamedCapture'=> '0.06',
	'Tie::Memoize'          => '1.1',
	'Tie::RefHash'          => '1.37',
	'Tie::Scalar'           => '1.00',
	'Tie::SubstrHash'       => '1.00',
	'Time::HiRes'           => '1.9707',
	'Time::Local'           => '1.17',
	'Time::Piece'           => '1.11_02',
	'Time::Piece::Seconds'  => undef,
	'Time::Seconds'         => undef,
	'Time::gmtime'          => '1.03',
	'Time::localtime'       => '1.02',
	'Time::tm'              => '1.00',
	'UNIVERSAL'             => '1.04',
	'Unicode'               => '5.0.0',
	'Unicode::Collate'      => '0.52',
	'Unicode::Normalize'    => '1.02',
	'Unicode::UCD'          => '0.25',
	'User::grent'           => '1.01',
	'User::pwent'           => '1.00',
	'Win32'                 => '0.30',
	'Win32API::File'        => '0.1001_01',
	'Win32API::File::ExtUtils::Myconst2perl'=> '1',
	'Win32CORE'             => '0.02',
	'XS::APItest'           => '0.12',
	'XS::Typemap'           => '0.02',
	'XSLoader'              => '0.08',
	'XSSymSet'              => '1.1',
	'attributes'            => '0.08',
	'attrs'                 => '1.02',
	'autouse'               => '1.06',
	'base'                  => '2.12',
	'bigint'                => '0.22',
	'bignum'                => '0.22',
	'bigrat'                => '0.22',
	'blib'                  => '1.03',
	'bytes'                 => '1.03',
	'charnames'             => '1.06',
	'constant'              => '1.10',
	'diagnostics'           => '1.17',
	'encoding'              => '2.06',
	'encoding::warnings'    => '0.11',
	'feature'               => '1.10',
	'fields'                => '2.12',
	'filetest'              => '1.01',
	'if'                    => '0.05',
	'integer'               => '1.00',
	'less'                  => '0.02',
	'lib'                   => '0.5565',
	'locale'                => '1.00',
	'mro'                   => '1.00',
	'open'                  => '1.05',
	'ops'                   => '1.01',
	'overload'              => '1.06',
	're'                    => '0.08',
	'sigtrap'               => '1.04',
	'sort'                  => '2.01',
	'strict'                => '1.04',
	'subs'                  => '1.00',
	'threads'               => '1.63',
	'threads::shared'       => '1.12',
	'utf8'                  => '1.07',
	'vars'                  => '1.01',
	'version'               => '0.7203',
	'vmsish'                => '1.02',
	'warnings'              => '1.06',
	'warnings::register'    => '1.01',
    },

    5.010000 => {
	'AnyDBM_File'           => '1.00',
	'Archive::Extract'      => '0.24',
	'Archive::Tar'          => '1.38',
	'Archive::Tar::Constant'=> '0.02',
	'Archive::Tar::File'    => '0.02',
	'Attribute::Handlers'   => '0.79',
	'AutoLoader'            => '5.63',
	'AutoSplit'             => '1.05',
	'B'                     => '1.17',
	'B::Concise'            => '0.74',
	'B::Debug'              => '1.05',
	'B::Deparse'            => '0.83',
	'B::Lint'               => '1.09',
	'B::Showlex'            => '1.02',
	'B::Terse'              => '1.05',
	'B::Xref'               => '1.01',
	'Benchmark'             => '1.1',
	'CGI'                   => '3.29',
	'CGI::Apache'           => '1.00',
	'CGI::Carp'             => '1.29',
	'CGI::Cookie'           => '1.28',
	'CGI::Fast'             => '1.07',
	'CGI::Pretty'           => '1.08',
	'CGI::Push'             => '1.04',
	'CGI::Switch'           => '1.00',
	'CGI::Util'             => '1.5_01',
	'CPAN'                  => '1.9205',
	'CPAN::API::HOWTO'      => undef,
	'CPAN::Debug'           => '5.402212',
	'CPAN::DeferedCode'     => '5.50',
	'CPAN::FirstTime'       => '5.402229',
	'CPAN::HandleConfig'    => '5.402212',
	'CPAN::Kwalify'         => '5.401418',
	'CPAN::Nox'             => '5.402411',
	'CPAN::Queue'           => '5.402212',
	'CPAN::Tarzip'          => '5.402213',
	'CPAN::Version'         => '5.5',
	'CPANPLUS'              => '0.84',
	'CPANPLUS::Backend'     => undef,
	'CPANPLUS::Backend::RV' => undef,
	'CPANPLUS::Config'      => undef,
	'CPANPLUS::Configure'   => undef,
	'CPANPLUS::Configure::Setup'=> undef,
	'CPANPLUS::Dist'        => undef,
	'CPANPLUS::Dist::Base'  => '0.01',
	'CPANPLUS::Dist::Build' => '0.06_02',
	'CPANPLUS::Dist::Build::Constants'=> '0.01',
	'CPANPLUS::Dist::MM'    => undef,
	'CPANPLUS::Dist::Sample'=> undef,
	'CPANPLUS::Error'       => undef,
	'CPANPLUS::Internals'   => '0.84',
	'CPANPLUS::Internals::Constants'=> '0.01',
	'CPANPLUS::Internals::Constants::Report'=> '0.01',
	'CPANPLUS::Internals::Extract'=> undef,
	'CPANPLUS::Internals::Fetch'=> undef,
	'CPANPLUS::Internals::Report'=> undef,
	'CPANPLUS::Internals::Search'=> undef,
	'CPANPLUS::Internals::Source'=> undef,
	'CPANPLUS::Internals::Utils'=> undef,
	'CPANPLUS::Internals::Utils::Autoflush'=> undef,
	'CPANPLUS::Module'      => undef,
	'CPANPLUS::Module::Author'=> undef,
	'CPANPLUS::Module::Author::Fake'=> undef,
	'CPANPLUS::Module::Checksums'=> undef,
	'CPANPLUS::Module::Fake'=> undef,
	'CPANPLUS::Module::Signature'=> undef,
	'CPANPLUS::Selfupdate'  => undef,
	'CPANPLUS::Shell'       => undef,
	'CPANPLUS::Shell::Classic'=> '0.0562',
	'CPANPLUS::Shell::Default'=> '0.84',
	'CPANPLUS::Shell::Default::Plugins::CustomSource'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Remote'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Source'=> undef,
	'CPANPLUS::inc'         => undef,
	'Carp'                  => '1.08',
	'Carp::Heavy'           => '1.08',
	'Class::ISA'            => '0.33',
	'Class::Struct'         => '0.63',
	'Compress::Raw::Zlib'   => '2.008',
	'Compress::Zlib'        => '2.008',
	'Config'                => undef,
	'Config::Extensions'    => '0.01',
	'Cwd'                   => '3.2501',
	'DB'                    => '1.01',
	'DBM_Filter'            => '0.02',
	'DBM_Filter::compress'  => '0.01',
	'DBM_Filter::encode'    => '0.01',
	'DBM_Filter::int32'     => '0.01',
	'DBM_Filter::null'      => '0.01',
	'DBM_Filter::utf8'      => '0.01',
	'DB_File'               => '1.816_1',
	'DCLsym'                => '1.03',
	'Data::Dumper'          => '2.121_14',
	'Devel::DProf'          => '20050603.00',
	'Devel::InnerPackage'   => '0.3',
	'Devel::PPPort'         => '3.13',
	'Devel::Peek'           => '1.03',
	'Devel::SelfStubber'    => '1.03',
	'Digest'                => '1.15',
	'Digest::MD5'           => '2.36_01',
	'Digest::SHA'           => '5.45',
	'Digest::base'          => '1.00',
	'Digest::file'          => '1.00',
	'DirHandle'             => '1.01',
	'Dumpvalue'             => '1.12',
	'DynaLoader'            => '1.08',
	'Encode'                => '2.23',
	'Encode::Alias'         => '2.07',
	'Encode::Byte'          => '2.03',
	'Encode::CJKConstants'  => '2.02',
	'Encode::CN'            => '2.02',
	'Encode::CN::HZ'        => '2.04',
	'Encode::Config'        => '2.04',
	'Encode::EBCDIC'        => '2.02',
	'Encode::Encoder'       => '2.01',
	'Encode::Encoding'      => '2.05',
	'Encode::GSM0338'       => '2.00',
	'Encode::Guess'         => '2.02',
	'Encode::JP'            => '2.03',
	'Encode::JP::H2Z'       => '2.02',
	'Encode::JP::JIS7'      => '2.03',
	'Encode::KR'            => '2.02',
	'Encode::KR::2022_KR'   => '2.02',
	'Encode::MIME::Header'  => '2.05',
	'Encode::MIME::Header::ISO_2022_JP'=> '1.03',
	'Encode::MIME::Name'    => '1.01',
	'Encode::Symbol'        => '2.02',
	'Encode::TW'            => '2.02',
	'Encode::Unicode'       => '2.05',
	'Encode::Unicode::UTF7' => '2.04',
	'English'               => '1.04',
	'Env'                   => '1.00',
	'Errno'                 => '1.10',
	'Exporter'              => '5.62',
	'Exporter::Heavy'       => '5.62',
	'ExtUtils::CBuilder'    => '0.21',
	'ExtUtils::CBuilder::Base'=> '0.21',
	'ExtUtils::CBuilder::Platform::Unix'=> '0.21',
	'ExtUtils::CBuilder::Platform::VMS'=> '0.22',
	'ExtUtils::CBuilder::Platform::Windows'=> '0.21',
	'ExtUtils::CBuilder::Platform::aix'=> '0.21',
	'ExtUtils::CBuilder::Platform::cygwin'=> '0.21',
	'ExtUtils::CBuilder::Platform::darwin'=> '0.21',
	'ExtUtils::CBuilder::Platform::dec_osf'=> '0.21',
	'ExtUtils::CBuilder::Platform::os2'=> '0.21',
	'ExtUtils::Command'     => '1.13',
	'ExtUtils::Command::MM' => '6.42',
	'ExtUtils::Constant'    => '0.2',
	'ExtUtils::Constant::Base'=> '0.04',
	'ExtUtils::Constant::ProxySubs'=> '0.05',
	'ExtUtils::Constant::Utils'=> '0.01',
	'ExtUtils::Constant::XS'=> '0.02',
	'ExtUtils::Embed'       => '1.27',
	'ExtUtils::Install'     => '1.44',
	'ExtUtils::Installed'   => '1.43',
	'ExtUtils::Liblist'     => '6.42',
	'ExtUtils::Liblist::Kid'=> '6.42',
	'ExtUtils::MM'          => '6.42',
	'ExtUtils::MM_AIX'      => '6.42',
	'ExtUtils::MM_Any'      => '6.42',
	'ExtUtils::MM_BeOS'     => '6.42',
	'ExtUtils::MM_Cygwin'   => '6.42',
	'ExtUtils::MM_DOS'      => '6.42',
	'ExtUtils::MM_MacOS'    => '6.42',
	'ExtUtils::MM_NW5'      => '6.42',
	'ExtUtils::MM_OS2'      => '6.42',
	'ExtUtils::MM_QNX'      => '6.42',
	'ExtUtils::MM_UWIN'     => '6.42',
	'ExtUtils::MM_Unix'     => '6.42',
	'ExtUtils::MM_VMS'      => '6.42',
	'ExtUtils::MM_VOS'      => '6.42',
	'ExtUtils::MM_Win32'    => '6.42',
	'ExtUtils::MM_Win95'    => '6.42',
	'ExtUtils::MY'          => '6.42',
	'ExtUtils::MakeMaker'   => '6.42',
	'ExtUtils::MakeMaker::Config'=> '6.42',
	'ExtUtils::MakeMaker::bytes'=> '6.42',
	'ExtUtils::MakeMaker::vmsish'=> '6.42',
	'ExtUtils::Manifest'    => '1.51_01',
        'ExtUtils::Miniperl'    => undef,
	'ExtUtils::Mkbootstrap' => '6.42',
	'ExtUtils::Mksymlists'  => '6.42',
	'ExtUtils::Packlist'    => '1.43',
	'ExtUtils::ParseXS'     => '2.18_02',
	'ExtUtils::testlib'     => '6.42',
	'Fatal'                 => '1.05',
	'Fcntl'                 => '1.06',
	'File::Basename'        => '2.76',
	'File::CheckTree'       => '4.3',
	'File::Compare'         => '1.1005',
	'File::Copy'            => '2.11',
	'File::DosGlob'         => '1.00',
	'File::Fetch'           => '0.14',
	'File::Find'            => '1.12',
	'File::Glob'            => '1.06',
	'File::GlobMapper'      => '0.000_02',
	'File::Path'            => '2.04',
	'File::Spec'            => '3.2501',
	'File::Spec::Cygwin'    => '3.2501',
	'File::Spec::Epoc'      => '3.2501',
	'File::Spec::Functions' => '3.2501',
	'File::Spec::Mac'       => '3.2501',
	'File::Spec::OS2'       => '3.2501',
	'File::Spec::Unix'      => '3.2501',
	'File::Spec::VMS'       => '3.2501',
	'File::Spec::Win32'     => '3.2501',
	'File::Temp'            => '0.18',
	'File::stat'            => '1.00',
	'FileCache'             => '1.07',
	'FileHandle'            => '2.01',
	'VMS::Filespec'         => '1.12',
	'Filter::Simple'        => '0.82',
	'Filter::Util::Call'    => '1.07',
	'FindBin'               => '1.49',
	'GDBM_File'             => '1.08',
	'Getopt::Long'          => '2.37',
	'Getopt::Std'           => '1.05',
	'Hash::Util'            => '0.07',
	'Hash::Util::FieldHash' => '1.03',
	'I18N::Collate'         => '1.00',
	'I18N::LangTags'        => '0.35',
	'I18N::LangTags::Detect'=> '1.03',
	'I18N::LangTags::List'  => '0.35',
	'I18N::Langinfo'        => '0.02',
	'IO'                    => '1.23_01',
	'IO::Compress::Adapter::Deflate'=> '2.008',
	'IO::Compress::Adapter::Identity'=> '2.008',
	'IO::Compress::Base'    => '2.008',
	'IO::Compress::Base::Common'=> '2.008',
	'IO::Compress::Deflate' => '2.008',
	'IO::Compress::Gzip'    => '2.008',
	'IO::Compress::Gzip::Constants'=> '2.008',
	'IO::Compress::RawDeflate'=> '2.008',
	'IO::Compress::Zip'     => '2.008',
	'IO::Compress::Zip::Constants'=> '2.008',
	'IO::Compress::Zlib::Constants'=> '2.008',
	'IO::Compress::Zlib::Extra'=> '2.008',
	'IO::Dir'               => '1.06',
	'IO::File'              => '1.14',
	'IO::Handle'            => '1.27',
	'IO::Pipe'              => '1.13',
	'IO::Poll'              => '0.07',
	'IO::Seekable'          => '1.10',
	'IO::Select'            => '1.17',
	'IO::Socket'            => '1.30_01',
	'IO::Socket::INET'      => '1.31',
	'IO::Socket::UNIX'      => '1.23',
	'IO::Uncompress::Adapter::Identity'=> '2.008',
	'IO::Uncompress::Adapter::Inflate'=> '2.008',
	'IO::Uncompress::AnyInflate'=> '2.008',
	'IO::Uncompress::AnyUncompress'=> '2.008',
	'IO::Uncompress::Base'  => '2.008',
	'IO::Uncompress::Gunzip'=> '2.008',
	'IO::Uncompress::Inflate'=> '2.008',
	'IO::Uncompress::RawInflate'=> '2.008',
	'IO::Uncompress::Unzip' => '2.008',
	'IO::Zlib'              => '1.07',
	'IPC::Cmd'              => '0.40_1',
	'IPC::Msg'              => '1.02',
	'IPC::Open2'            => '1.02',
	'IPC::Open3'            => '1.02',
	'IPC::Semaphore'        => '1.02',
	'IPC::SysV'             => '1.05',
	'List::Util'            => '1.19',
	'Locale::Constants'     => '2.07',
	'Locale::Country'       => '2.07',
	'Locale::Currency'      => '2.07',
	'Locale::Language'      => '2.07',
	'Locale::Maketext'      => '1.12',
	'Locale::Maketext::Guts'=> undef,
	'Locale::Maketext::GutsLoader'=> undef,
	'Locale::Maketext::Simple'=> '0.18',
	'Locale::Script'        => '2.07',
	'Log::Message'          => '0.01',
	'Log::Message::Config'  => '0.01',
	'Log::Message::Handlers'=> undef,
	'Log::Message::Item'    => undef,
	'Log::Message::Simple'  => '0.04',
	'MIME::Base64'          => '3.07_01',
	'MIME::QuotedPrint'     => '3.07',
	'Math::BigFloat'        => '1.59',
	'Math::BigFloat::Trace' => '0.01',
	'Math::BigInt'          => '1.88',
	'Math::BigInt::Calc'    => '0.52',
	'Math::BigInt::CalcEmu' => '0.05',
	'Math::BigInt::FastCalc'=> '0.16',
	'Math::BigInt::Trace'   => '0.01',
	'Math::BigRat'          => '0.21',
	'Math::Complex'         => '1.37',
	'Math::Trig'            => '1.04',
	'Memoize'               => '1.01_02',
	'Memoize::AnyDBM_File'  => '0.65',
	'Memoize::Expire'       => '1.00',
	'Memoize::ExpireFile'   => '1.01',
	'Memoize::ExpireTest'   => '0.65',
	'Memoize::NDBM_File'    => '0.65',
	'Memoize::SDBM_File'    => '0.65',
	'Memoize::Storable'     => '0.65',
	'Module::Build'         => '0.2808_01',
	'Module::Build::Base'   => '0.2808_01',
	'Module::Build::Compat' => '0.2808_01',
	'Module::Build::Config' => '0.2808_01',
	'Module::Build::ConfigData'=> undef,
	'Module::Build::Cookbook'=> undef,
	'Module::Build::Dumper' => undef,
	'Module::Build::ModuleInfo'=> '0.2808_01',
	'Module::Build::Notes'  => '0.2808_01',
	'Module::Build::PPMMaker'=> '0.2808_01',
	'Module::Build::Platform::Amiga'=> '0.2808_01',
	'Module::Build::Platform::Default'=> '0.2808_01',
	'Module::Build::Platform::EBCDIC'=> '0.2808_01',
	'Module::Build::Platform::MPEiX'=> '0.2808_01',
	'Module::Build::Platform::MacOS'=> '0.2808_01',
	'Module::Build::Platform::RiscOS'=> '0.2808_01',
	'Module::Build::Platform::Unix'=> '0.2808_01',
	'Module::Build::Platform::VMS'=> '0.2808_01',
	'Module::Build::Platform::VOS'=> '0.2808_01',
	'Module::Build::Platform::Windows'=> '0.2808_01',
	'Module::Build::Platform::aix'=> '0.2808_01',
	'Module::Build::Platform::cygwin'=> '0.2808_01',
	'Module::Build::Platform::darwin'=> '0.2808_01',
	'Module::Build::Platform::os2'=> '0.2808_01',
	'Module::Build::PodParser'=> '0.2808_01',
	'Module::Build::Version'=> '0.7203',
	'Module::Build::YAML'   => '0.50',
	'Module::CoreList'      => '2.12',
	'Module::Load'          => '0.12',
	'Module::Load::Conditional'=> '0.22',
	'Module::Loaded'        => '0.01',
	'Module::Pluggable'     => '3.6',
	'Module::Pluggable::Object'=> '3.6',
	'Moped::Msg'            => '0.01',
	'NDBM_File'             => '1.07',
	'NEXT'                  => '0.60_01',
	'Net::Cmd'              => '2.29',
	'Net::Config'           => '1.11',
	'Net::Domain'           => '2.20',
	'Net::FTP'              => '2.77',
	'Net::FTP::A'           => '1.18',
	'Net::FTP::E'           => '0.01',
	'Net::FTP::I'           => '1.12',
	'Net::FTP::L'           => '0.01',
	'Net::FTP::dataconn'    => '0.11',
	'Net::NNTP'             => '2.24',
	'Net::Netrc'            => '2.12',
	'Net::POP3'             => '2.29',
	'Net::Ping'             => '2.33',
	'Net::SMTP'             => '2.31',
	'Net::Time'             => '2.10',
	'Net::hostent'          => '1.01',
	'Net::netent'           => '1.00',
	'Net::protoent'         => '1.00',
	'Net::servent'          => '1.01',
	'O'                     => '1.00',
	'ODBM_File'             => '1.07',
	'Object::Accessor'      => '0.32',
	'Opcode'                => '1.11',
	'POSIX'                 => '1.13',
	'Package::Constants'    => '0.01',
	'Params::Check'         => '0.26',
	'PerlIO'                => '1.04',
	'PerlIO::encoding'      => '0.10',
	'PerlIO::scalar'        => '0.05',
	'PerlIO::via'           => '0.04',
	'PerlIO::via::QuotedPrint'=> '0.06',
	'Pod::Checker'          => '1.43_01',
	'Pod::Escapes'          => '1.04',
	'Pod::Find'             => '1.34',
	'Pod::Functions'        => '1.03',
	'Pod::Html'             => '1.08',
	'Pod::InputObjects'     => '1.3',
	'Pod::LaTeX'            => '0.58',
	'Pod::Man'              => '2.16',
	'Pod::ParseLink'        => '1.06',
	'Pod::ParseUtils'       => '1.35',
	'Pod::Parser'           => '1.35',
	'Pod::Perldoc'          => '3.14_02',
	'Pod::Perldoc::BaseTo'  => undef,
	'Pod::Perldoc::GetOptsOO'=> undef,
	'Pod::Perldoc::ToChecker'=> undef,
	'Pod::Perldoc::ToMan'   => undef,
	'Pod::Perldoc::ToNroff' => undef,
	'Pod::Perldoc::ToPod'   => undef,
	'Pod::Perldoc::ToRtf'   => undef,
	'Pod::Perldoc::ToText'  => undef,
	'Pod::Perldoc::ToTk'    => undef,
	'Pod::Perldoc::ToXml'   => undef,
	'Pod::PlainText'        => '2.02',
	'Pod::Plainer'          => '0.01',
	'Pod::Select'           => '1.35',
	'Pod::Simple'           => '3.05',
	'Pod::Simple::BlackBox' => undef,
	'Pod::Simple::Checker'  => '2.02',
	'Pod::Simple::Debug'    => undef,
	'Pod::Simple::DumpAsText'=> '2.02',
	'Pod::Simple::DumpAsXML'=> '2.02',
	'Pod::Simple::HTML'     => '3.03',
	'Pod::Simple::HTMLBatch'=> '3.02',
	'Pod::Simple::HTMLLegacy'=> '5.01',
	'Pod::Simple::LinkSection'=> undef,
	'Pod::Simple::Methody'  => '2.02',
	'Pod::Simple::Progress' => '1.01',
	'Pod::Simple::PullParser'=> '2.02',
	'Pod::Simple::PullParserEndToken'=> undef,
	'Pod::Simple::PullParserStartToken'=> undef,
	'Pod::Simple::PullParserTextToken'=> undef,
	'Pod::Simple::PullParserToken'=> '2.02',
	'Pod::Simple::RTF'      => '2.02',
	'Pod::Simple::Search'   => '3.04',
	'Pod::Simple::SimpleTree'=> '2.02',
	'Pod::Simple::Text'     => '2.02',
	'Pod::Simple::TextContent'=> '2.02',
	'Pod::Simple::TiedOutFH'=> undef,
	'Pod::Simple::Transcode'=> undef,
	'Pod::Simple::TranscodeDumb'=> '2.02',
	'Pod::Simple::TranscodeSmart'=> undef,
	'Pod::Simple::XMLOutStream'=> '2.02',
	'Pod::Text'             => '3.08',
	'Pod::Text::Color'      => '2.03',
	'Pod::Text::Overstrike' => '2',
	'Pod::Text::Termcap'    => '2.03',
	'Pod::Usage'            => '1.35',
	'SDBM_File'             => '1.06',
	'Safe'                  => '2.12',
	'Scalar::Util'          => '1.19',
	'Search::Dict'          => '1.02',
	'SelectSaver'           => '1.01',
	'SelfLoader'            => '1.11',
	'Shell'                 => '0.72_01',
	'Socket'                => '1.80',
	'Stdio'                 => '2.3',
	'Storable'              => '2.18',
	'Switch'                => '2.13',
	'Symbol'                => '1.06',
	'Sys::Hostname'         => '1.11',
	'Sys::Syslog'           => '0.22',
	'Sys::Syslog::win32::Win32'=> undef,
	'Term::ANSIColor'       => '1.12',
	'Term::Cap'             => '1.12',
	'Term::Complete'        => '1.402',
	'Term::ReadLine'        => '1.03',
	'Term::UI'              => '0.18',
	'Term::UI::History'     => undef,
	'Test'                  => '1.25',
	'Test::Builder'         => '0.72',
	'Test::Builder::Module' => '0.72',
	'Test::Builder::Tester' => '1.09',
	'Test::Builder::Tester::Color'=> undef,
	'Test::Harness'         => '2.64',
	'Test::Harness::Assert' => '0.02',
	'Test::Harness::Iterator'=> '0.02',
	'Test::Harness::Point'  => '0.01',
	'Test::Harness::Results'=> '0.01',
	'Test::Harness::Straps' => '0.26_01',
	'Test::Harness::Util'   => '0.01',
	'Test::More'            => '0.72',
	'Test::Simple'          => '0.72',
	'Text::Abbrev'          => '1.01',
	'Text::Balanced'        => '2.0.0',
	'Text::ParseWords'      => '3.26',
	'Text::Soundex'         => '3.03',
	'Text::Tabs'            => '2007.1117',
	'Text::Wrap'            => '2006.1117',
	'Thread'                => '3.02',
	'Thread::Queue'         => '2.00',
	'Thread::Semaphore'     => '2.01',
	'Tie::Array'            => '1.03',
	'Tie::File'             => '0.97_02',
	'Tie::Handle'           => '4.1',
	'Tie::Hash'             => '1.02',
	'Tie::Hash::NamedCapture'=> '0.06',
	'Tie::Memoize'          => '1.1',
	'Tie::RefHash'          => '1.37',
	'Tie::Scalar'           => '1.00',
	'Tie::StdHandle'        => undef,
	'Tie::SubstrHash'       => '1.00',
	'Time::HiRes'           => '1.9711',
	'Time::Local'           => '1.18',
	'Time::Piece'           => '1.12',
	'Time::Piece::Seconds'  => undef,
	'Time::Seconds'         => undef,
	'Time::gmtime'          => '1.03',
	'Time::localtime'       => '1.02',
	'Time::tm'              => '1.00',
	'UNIVERSAL'             => '1.04',
	'Unicode'               => '5.0.0',
	'Unicode::Collate'      => '0.52',
	'Unicode::Normalize'    => '1.02',
	'Unicode::UCD'          => '0.25',
	'User::grent'           => '1.01',
	'User::pwent'           => '1.00',
	'Win32'                 => '0.34',
	'Win32API::File'        => '0.1001_01',
	'Win32API::File::ExtUtils::Myconst2perl'=> '1',
	'Win32CORE'             => '0.02',
	'XS::APItest'           => '0.12',
	'XS::Typemap'           => '0.02',
	'XSLoader'              => '0.08',
	'XSSymSet'              => '1.1',
	'attributes'            => '0.08',
	'attrs'                 => '1.02',
	'autouse'               => '1.06',
	'base'                  => '2.13',
	'bigint'                => '0.22',
	'bignum'                => '0.22',
	'bigrat'                => '0.22',
	'blib'                  => '1.03',
	'bytes'                 => '1.03',
	'charnames'             => '1.06',
	'constant'              => '1.13',
	'diagnostics'           => '1.17',
	'encoding'              => '2.06',
	'encoding::warnings'    => '0.11',
	'feature'               => '1.11',
	'fields'                => '2.13',
	'filetest'              => '1.02',
	'if'                    => '0.05',
	'integer'               => '1.00',
	'less'                  => '0.02',
	'lib'                   => '0.5565',
	'locale'                => '1.00',
	'mro'                   => '1.00',
	'open'                  => '1.06',
	'ops'                   => '1.01',
	'overload'              => '1.06',
	're'                    => '0.08',
	'sigtrap'               => '1.04',
	'sort'                  => '2.01',
	'strict'                => '1.04',
	'subs'                  => '1.00',
	'threads'               => '1.67',
	'threads::shared'       => '1.14',
	'utf8'                  => '1.07',
	'vars'                  => '1.01',
	'version'               => '0.74',
	'vmsish'                => '1.02',
	'warnings'              => '1.06',
	'warnings::register'    => '1.01',
    },

    5.008009 => {
	'AnyDBM_File'           => '1.00',
	'Attribute::Handlers'   => '0.78_03',
	'AutoLoader'            => '5.67',
	'AutoSplit'             => '1.06',
	'B'                     => '1.19',
	'B::Asmdata'            => '1.02',
	'B::Assembler'          => '0.08',
	'B::Bblock'             => '1.02_01',
	'B::Bytecode'           => '1.01_01',
	'B::C'                  => '1.05',
	'B::CC'                 => '1.00_01',
	'B::Concise'            => '0.76',
	'B::Debug'              => '1.05',
	'B::Deparse'            => '0.87',
	'B::Disassembler'       => '1.05',
	'B::Lint'               => '1.11',
	'B::Lint::Debug'        => undef,
	'B::Showlex'            => '1.02',
	'B::Stackobj'           => '1.00',
	'B::Stash'              => '1.00',
	'B::Terse'              => '1.05',
	'B::Xref'               => '1.01',
	'Benchmark'             => '1.1',
	'ByteLoader'            => '0.06',
	'CGI'                   => '3.42',
	'CGI::Apache'           => '1.00',
	'CGI::Carp'             => '1.30_01',
	'CGI::Cookie'           => '1.29',
	'CGI::Fast'             => '1.07',
	'CGI::Pretty'           => '1.08',
	'CGI::Push'             => '1.04',
	'CGI::Switch'           => '1.00',
	'CGI::Util'             => '1.5_01',
	'CPAN'                  => '1.9301',
	'CPAN::Debug'           => '5.5',
	'CPAN::DeferedCode'     => '5.50',
	'CPAN::Distroprefs'     => '6',
	'CPAN::FirstTime'       => '5.5_01',
	'CPAN::HandleConfig'    => '5.5',
	'CPAN::Kwalify'         => '5.50',
	'CPAN::Nox'             => '5.50',
	'CPAN::Queue'           => '5.5',
	'CPAN::Tarzip'          => '5.5',
	'CPAN::Version'         => '5.5',
	'Carp'                  => '1.10',
	'Carp::Heavy'           => '1.10',
	'Class::ISA'            => '0.33',
	'Class::Struct'         => '0.63',
	'Config'                => undef,
	'Cwd'                   => '3.29',
	'DB'                    => '1.01',
	'DBM_Filter'            => '0.02',
	'DBM_Filter::compress'  => '0.02',
	'DBM_Filter::encode'    => '0.02',
	'DBM_Filter::int32'     => '0.02',
	'DBM_Filter::null'      => '0.02',
	'DBM_Filter::utf8'      => '0.02',
	'DB_File'               => '1.817',
	'DCLsym'                => '1.03',
	'Data::Dumper'          => '2.121_17',
	'Devel::DProf'          => '20080331.00',
	'Devel::InnerPackage'   => '0.3',
	'Devel::PPPort'         => '3.14',
	'Devel::Peek'           => '1.04',
	'Devel::SelfStubber'    => '1.03',
	'Digest'                => '1.15',
	'Digest::MD5'           => '2.37',
	'Digest::base'          => '1.00',
	'Digest::file'          => '1.00',
	'DirHandle'             => '1.02',
	'Dumpvalue'             => '1.12',
	'DynaLoader'            => '1.09',
	'Encode'                => '2.26',
	'Encode::Alias'         => '2.10',
	'Encode::Byte'          => '2.03',
	'Encode::CJKConstants'  => '2.02',
	'Encode::CN'            => '2.02',
	'Encode::CN::HZ'        => '2.05',
	'Encode::Config'        => '2.05',
	'Encode::EBCDIC'        => '2.02',
	'Encode::Encoder'       => '2.01',
	'Encode::Encoding'      => '2.05',
	'Encode::GSM0338'       => '2.01',
	'Encode::Guess'         => '2.02',
	'Encode::JP'            => '2.03',
	'Encode::JP::H2Z'       => '2.02',
	'Encode::JP::JIS7'      => '2.04',
	'Encode::KR'            => '2.02',
	'Encode::KR::2022_KR'   => '2.02',
	'Encode::MIME::Header'  => '2.05',
	'Encode::MIME::Header::ISO_2022_JP'=> '1.03',
	'Encode::MIME::Name'    => '1.01',
	'Encode::Symbol'        => '2.02',
	'Encode::TW'            => '2.02',
	'Encode::Unicode'       => '2.05',
	'Encode::Unicode::UTF7' => '2.04',
	'English'               => '1.03',
	'Env'                   => '1.00',
	'Errno'                 => '1.10',
	'Exporter'              => '5.63',
	'Exporter::Heavy'       => '5.63',
	'ExtUtils::Command'     => '1.15',
	'ExtUtils::Command::MM' => '6.48',
	'ExtUtils::Constant'    => '0.21',
	'ExtUtils::Constant::Base'=> '0.04',
	'ExtUtils::Constant::ProxySubs'=> '0.06',
	'ExtUtils::Constant::Utils'=> '0.02',
	'ExtUtils::Constant::XS'=> '0.02',
	'ExtUtils::Embed'       => '1.28',
	'ExtUtils::Install'     => '1.50_01',
	'ExtUtils::Installed'   => '1.43',
	'ExtUtils::Liblist'     => '6.48',
	'ExtUtils::Liblist::Kid'=> '6.48',
	'ExtUtils::MM'          => '6.48',
	'ExtUtils::MM_AIX'      => '6.48',
	'ExtUtils::MM_Any'      => '6.48',
	'ExtUtils::MM_BeOS'     => '6.48',
	'ExtUtils::MM_Cygwin'   => '6.48',
	'ExtUtils::MM_DOS'      => '6.48',
	'ExtUtils::MM_Darwin'   => '6.48',
	'ExtUtils::MM_MacOS'    => '6.48',
	'ExtUtils::MM_NW5'      => '6.48',
	'ExtUtils::MM_OS2'      => '6.48',
	'ExtUtils::MM_QNX'      => '6.48',
	'ExtUtils::MM_UWIN'     => '6.48',
	'ExtUtils::MM_Unix'     => '6.48',
	'ExtUtils::MM_VMS'      => '6.48',
	'ExtUtils::MM_VOS'      => '6.48',
	'ExtUtils::MM_Win32'    => '6.48',
	'ExtUtils::MM_Win95'    => '6.48',
	'ExtUtils::MY'          => '6.48',
	'ExtUtils::MakeMaker'   => '6.48',
	'ExtUtils::MakeMaker::Config'=> '6.48',
	'ExtUtils::MakeMaker::bytes'=> '6.48',
	'ExtUtils::MakeMaker::vmsish'=> '6.48',
	'ExtUtils::Manifest'    => '1.55',
	'ExtUtils::Miniperl'    => undef,
	'ExtUtils::Mkbootstrap' => '6.48',
	'ExtUtils::Mksymlists'  => '6.48',
	'ExtUtils::Packlist'    => '1.43',
	'ExtUtils::ParseXS'     => '2.19',
	'ExtUtils::testlib'     => '6.48',
	'Fatal'                 => '1.06',
	'Fcntl'                 => '1.06',
	'File::Basename'        => '2.77',
	'File::CheckTree'       => '4.4',
	'File::Compare'         => '1.1005',
	'File::Copy'            => '2.13',
	'File::DosGlob'         => '1.01',
	'File::Find'            => '1.13',
	'File::Glob'            => '1.06',
	'File::Path'            => '2.07_02',
	'File::Spec'            => '3.29',
	'File::Spec::Cygwin'    => '3.29',
	'File::Spec::Epoc'      => '3.29',
	'File::Spec::Functions' => '3.29',
	'File::Spec::Mac'       => '3.29',
	'File::Spec::OS2'       => '3.29',
	'File::Spec::Unix'      => '3.29',
	'File::Spec::VMS'       => '3.29',
	'File::Spec::Win32'     => '3.29',
	'File::Temp'            => '0.20',
	'File::stat'            => '1.01',
	'FileCache'             => '1.07',
	'FileHandle'            => '2.01',
	'VMS::Filespec'         => '1.11',
	'Filter::Simple'        => '0.83',
	'Filter::Util::Call'    => '1.07',
	'FindBin'               => '1.49',
	'GDBM_File'             => '1.09',
	'Getopt::Long'          => '2.37',
	'Getopt::Std'           => '1.06',
	'Hash::Util'            => '0.06',
	'I18N::Collate'         => '1.00',
	'I18N::LangTags'        => '0.35',
	'I18N::LangTags::Detect'=> '1.03',
	'I18N::LangTags::List'  => '0.35',
	'I18N::Langinfo'        => '0.02',
	'IO'                    => '1.23',
	'IO::Dir'               => '1.06',
	'IO::File'              => '1.14',
	'IO::Handle'            => '1.27',
	'IO::Pipe'              => '1.13',
	'IO::Poll'              => '0.07',
	'IO::Seekable'          => '1.10',
	'IO::Select'            => '1.17',
	'IO::Socket'            => '1.30',
	'IO::Socket::INET'      => '1.31',
	'IO::Socket::UNIX'      => '1.23',
	'IPC::Msg'              => '2.00',
	'IPC::Open2'            => '1.03',
	'IPC::Open3'            => '1.03',
	'IPC::Semaphore'        => '2.00',
	'IPC::SharedMem'        => '2.00',
	'IPC::SysV'             => '2.00',
	'IPC::lib::IPC::Msg'    => '2.00',
	'IPC::lib::IPC::Semaphore'=> '2.00',
	'IPC::lib::IPC::SharedMem'=> '2.00',
	'List::Util'            => '1.19',
	'Locale::Constants'     => '2.07',
	'Locale::Country'       => '2.07',
	'Locale::Currency'      => '2.07',
	'Locale::Language'      => '2.07',
	'Locale::Maketext'      => '1.13',
	'Locale::Maketext::Guts'=> '1.13',
	'Locale::Maketext::GutsLoader'=> '1.13',
	'Locale::Script'        => '2.07',
	'MIME::Base64'          => '3.07',
	'MIME::QuotedPrint'     => '3.07',
	'Math::BigFloat'        => '1.60',
	'Math::BigFloat::Trace' => '0.01',
	'Math::BigInt'          => '1.89',
	'Math::BigInt::Calc'    => '0.52',
	'Math::BigInt::CalcEmu' => '0.05',
	'Math::BigInt::Trace'   => '0.01',
	'Math::BigRat'          => '0.22',
	'Math::Complex'         => '1.54',
	'Math::Trig'            => '1.18',
	'Memoize'               => '1.01',
	'Memoize::AnyDBM_File'  => '0.65',
	'Memoize::Expire'       => '1.00',
	'Memoize::ExpireFile'   => '1.01',
	'Memoize::ExpireTest'   => '0.65',
	'Memoize::NDBM_File'    => '0.65',
	'Memoize::SDBM_File'    => '0.65',
	'Memoize::Storable'     => '0.65',
	'Module::CoreList'      => '2.17',
	'Module::Pluggable'     => '3.8',
	'Module::Pluggable::Object'=> '3.6',
	'Module::Pluggable::lib::Devel::InnerPackage'=> '0.3',
	'NDBM_File'             => '1.07',
	'NEXT'                  => '0.61',
	'Net::Cmd'              => '2.29',
	'Net::Config'           => '1.11',
	'Net::Domain'           => '2.20',
	'Net::FTP'              => '2.77',
	'Net::FTP::A'           => '1.18',
	'Net::FTP::E'           => '0.01',
	'Net::FTP::I'           => '1.12',
	'Net::FTP::L'           => '0.01',
	'Net::FTP::dataconn'    => '0.11',
	'Net::NNTP'             => '2.24',
	'Net::Netrc'            => '2.12',
	'Net::POP3'             => '2.29',
	'Net::Ping'             => '2.35',
	'Net::SMTP'             => '2.31',
	'Net::Time'             => '2.10',
	'Net::hostent'          => '1.01',
	'Net::netent'           => '1.00',
	'Net::protoent'         => '1.00',
	'Net::servent'          => '1.01',
	'O'                     => '1.01',
	'ODBM_File'             => '1.07',
	'Opcode'                => '1.0601',
	'POSIX'                 => '1.15',
	'PerlIO'                => '1.05',
	'PerlIO::encoding'      => '0.11',
	'PerlIO::scalar'        => '0.06',
	'PerlIO::via'           => '0.05',
	'PerlIO::via::QuotedPrint'=> '0.06',
	'Pod::Checker'          => '1.43',
	'Pod::Find'             => '1.34',
	'Pod::Functions'        => '1.03',
	'Pod::Html'             => '1.09',
	'Pod::InputObjects'     => '1.3',
	'Pod::LaTeX'            => '0.58',
	'Pod::Man'              => '1.37',
	'Pod::ParseLink'        => '1.06',
	'Pod::ParseUtils'       => '1.35',
	'Pod::Parser'           => '1.35',
	'Pod::Perldoc'          => '3.14',
	'Pod::Perldoc::BaseTo'  => undef,
	'Pod::Perldoc::GetOptsOO'=> undef,
	'Pod::Perldoc::ToChecker'=> undef,
	'Pod::Perldoc::ToMan'   => undef,
	'Pod::Perldoc::ToNroff' => undef,
	'Pod::Perldoc::ToPod'   => undef,
	'Pod::Perldoc::ToRtf'   => undef,
	'Pod::Perldoc::ToText'  => undef,
	'Pod::Perldoc::ToTk'    => undef,
	'Pod::Perldoc::ToXml'   => undef,
	'Pod::PlainText'        => '2.02',
	'Pod::Plainer'          => '0.01',
	'Pod::Select'           => '1.35',
	'Pod::Text'             => '2.21',
	'Pod::Text::Color'      => '1.04',
	'Pod::Text::Overstrike' => '1.1',
	'Pod::Text::Termcap'    => '1.11',
	'Pod::Usage'            => '1.35',
	'SDBM_File'             => '1.06',
	'Safe'                  => '2.16',
	'Scalar::Util'          => '1.19',
	'Search::Dict'          => '1.02',
	'SelectSaver'           => '1.01',
	'SelfLoader'            => '1.17',
	'Shell'                 => '0.72',
	'Socket'                => '1.81',
	'Stdio'                 => '2.4',
	'Storable'              => '2.19',
	'Switch'                => '2.13',
	'Symbol'                => '1.06',
	'Sys::Hostname'         => '1.11',
	'Sys::Syslog'           => '0.27',
	'Sys::Syslog::win32::Win32'=> undef,
	'Term::ANSIColor'       => '1.12',
	'Term::Cap'             => '1.12',
	'Term::Complete'        => '1.402',
	'Term::ReadLine'        => '1.03',
	'Test'                  => '1.25',
	'Test::Builder'         => '0.80',
	'Test::Builder::Module' => '0.80',
	'Test::Builder::Tester' => '1.13',
	'Test::Builder::Tester::Color'=> undef,
	'Test::Harness'         => '2.64',
	'Test::Harness::Assert' => '0.02',
	'Test::Harness::Iterator'=> '0.02',
	'Test::Harness::Point'  => '0.01',
	'Test::Harness::Results'=> '0.01_01',
	'Test::Harness::Straps' => '0.26_01',
	'Test::Harness::Util'   => '0.01',
	'Test::More'            => '0.80',
	'Test::Simple'          => '0.80',
	'Text::Abbrev'          => '1.01',
	'Text::Balanced'        => '1.98',
	'Text::ParseWords'      => '3.27',
	'Text::Soundex'         => '3.03',
	'Text::Tabs'            => '2007.1117',
	'Text::Wrap'            => '2006.1117',
	'Thread'                => '2.01',
	'Thread::Queue'         => '2.11',
	'Thread::Semaphore'     => '2.09',
	'Thread::Signal'        => '1.00',
	'Thread::Specific'      => '1.00',
	'Tie::Array'            => '1.03',
	'Tie::File'             => '0.97',
	'Tie::Handle'           => '4.2',
	'Tie::Hash'             => '1.03',
	'Tie::Memoize'          => '1.1',
	'Tie::RefHash'          => '1.38',
	'Tie::Scalar'           => '1.01',
	'Tie::StdHandle'        => '4.2',
	'Tie::SubstrHash'       => '1.00',
	'Time::HiRes'           => '1.9715',
	'Time::Local'           => '1.1901',
	'Time::gmtime'          => '1.03',
	'Time::localtime'       => '1.02',
	'Time::tm'              => '1.00',
	'UNIVERSAL'             => '1.01',
	'Unicode'               => '5.1.0',
	'Unicode::Collate'      => '0.52',
	'Unicode::Normalize'    => '1.02',
	'Unicode::UCD'          => '0.25',
	'User::grent'           => '1.01',
	'User::pwent'           => '1.00',
	'Win32'                 => '0.38',
	'Win32API::File'        => '0.1001_01',
	'Win32API::File::ExtUtils::Myconst2perl'=> '1',
	'Win32CORE'             => '0.02',
	'XS::APItest'           => '0.15',
	'XS::Typemap'           => '0.03',
	'XSLoader'              => '0.10',
	'XSSymSet'              => '1.1',
	'attributes'            => '0.09',
	'attrs'                 => '1.02',
	'autouse'               => '1.06',
	'base'                  => '2.13',
	'bigint'                => '0.23',
	'bignum'                => '0.23',
	'bigrat'                => '0.23',
	'blib'                  => '1.04',
	'bytes'                 => '1.02',
	'charnames'             => '1.06',
	'constant'              => '1.17',
	'diagnostics'           => '1.16',
	'encoding'              => '2.6_01',
	'fields'                => '2.12',
	'filetest'              => '1.02',
	'if'                    => '0.05',
	'integer'               => '1.00',
	'less'                  => '0.01',
	'lib'                   => '0.61',
	'locale'                => '1.00',
	'open'                  => '1.06',
	'ops'                   => '1.02',
	'overload'              => '1.06',
	're'                    => '0.0601',
	'sigtrap'               => '1.04',
	'sort'                  => '1.02',
	'strict'                => '1.03',
	'subs'                  => '1.00',
	'threads'               => '1.71',
	'threads::shared'       => '1.27',
	'utf8'                  => '1.07',
	'vars'                  => '1.01',
	'vmsish'                => '1.02',
	'warnings'              => '1.05_01',
	'warnings::register'    => '1.01',
    },

    5.010001 => {
	'AnyDBM_File'           => '1.00',
	'App::Prove'            => '3.17',
	'App::Prove::State'     => '3.17',
	'App::Prove::State::Result'=> '3.17',
	'App::Prove::State::Result::Test'=> '3.17',
	'Archive::Extract'      => '0.34',
	'Archive::Tar'          => '1.52',
	'Archive::Tar::Constant'=> '0.02',
	'Archive::Tar::File'    => '0.02',
	'Attribute::Handlers'   => '0.85',
	'AutoLoader'            => '5.68',
	'AutoSplit'             => '1.06',
	'B'                     => '1.22',
	'B::Concise'            => '0.76',
	'B::Debug'              => '1.11',
	'B::Deparse'            => '0.89',
	'B::Lint'               => '1.11',
	'B::Lint::Debug'        => undef,
	'B::Showlex'            => '1.02',
	'B::Terse'              => '1.05',
	'B::Xref'               => '1.02',
	'Benchmark'             => '1.11',
	'CGI'                   => '3.43',
	'CGI::Apache'           => '1.00',
	'CGI::Carp'             => '1.30_01',
	'CGI::Cookie'           => '1.29',
	'CGI::Fast'             => '1.07',
	'CGI::Pretty'           => '1.08',
	'CGI::Push'             => '1.04',
	'CGI::Switch'           => '1.00',
	'CGI::Util'             => '1.5_01',
	'CPAN'                  => '1.9402',
	'CPAN::Author'          => '5.5',
	'CPAN::Bundle'          => '5.5',
	'CPAN::CacheMgr'        => '5.5',
	'CPAN::Complete'        => '5.5',
	'CPAN::Debug'           => '5.5',
	'CPAN::DeferredCode'    => '5.50',
	'CPAN::Distribution'    => '1.93',
	'CPAN::Distroprefs'     => '6',
	'CPAN::Distrostatus'    => '5.5',
	'CPAN::Exception::RecursiveDependency'=> '5.5',
	'CPAN::Exception::blocked_urllist'=> '1.0',
	'CPAN::Exception::yaml_not_installed'=> '5.5',
	'CPAN::FTP'             => '5.5001',
	'CPAN::FTP::netrc'      => '1.00',
	'CPAN::FirstTime'       => '5.53',
	'CPAN::HandleConfig'    => '5.5',
	'CPAN::Index'           => '1.93',
	'CPAN::InfoObj'         => '5.5',
	'CPAN::Kwalify'         => '5.50',
	'CPAN::LWP::UserAgent'  => '1.00',
	'CPAN::Module'          => '5.5',
	'CPAN::Nox'             => '5.50',
	'CPAN::Prompt'          => '5.5',
	'CPAN::Queue'           => '5.5',
	'CPAN::Shell'           => '5.5',
	'CPAN::Tarzip'          => '5.501',
	'CPAN::URL'             => '5.5',
	'CPAN::Version'         => '5.5',
	'CPANPLUS'              => '0.88',
	'CPANPLUS::Backend'     => undef,
	'CPANPLUS::Backend::RV' => undef,
	'CPANPLUS::Config'      => undef,
	'CPANPLUS::Configure'   => undef,
	'CPANPLUS::Configure::Setup'=> undef,
	'CPANPLUS::Dist'        => undef,
	'CPANPLUS::Dist::Autobundle'=> undef,
	'CPANPLUS::Dist::Base'  => undef,
	'CPANPLUS::Dist::Build' => '0.36',
	'CPANPLUS::Dist::Build::Constants'=> '0.36',
	'CPANPLUS::Dist::MM'    => undef,
	'CPANPLUS::Dist::Sample'=> undef,
	'CPANPLUS::Error'       => undef,
	'CPANPLUS::Internals'   => '0.88',
	'CPANPLUS::Internals::Constants'=> undef,
	'CPANPLUS::Internals::Constants::Report'=> undef,
	'CPANPLUS::Internals::Extract'=> undef,
	'CPANPLUS::Internals::Fetch'=> undef,
	'CPANPLUS::Internals::Report'=> undef,
	'CPANPLUS::Internals::Search'=> undef,
	'CPANPLUS::Internals::Source'=> undef,
	'CPANPLUS::Internals::Source::Memory'=> undef,
	'CPANPLUS::Internals::Source::SQLite'=> undef,
	'CPANPLUS::Internals::Source::SQLite::Tie'=> undef,
	'CPANPLUS::Internals::Utils'=> undef,
	'CPANPLUS::Internals::Utils::Autoflush'=> undef,
	'CPANPLUS::Module'      => undef,
	'CPANPLUS::Module::Author'=> undef,
	'CPANPLUS::Module::Author::Fake'=> undef,
	'CPANPLUS::Module::Checksums'=> undef,
	'CPANPLUS::Module::Fake'=> undef,
	'CPANPLUS::Module::Signature'=> undef,
	'CPANPLUS::Selfupdate'  => undef,
	'CPANPLUS::Shell'       => undef,
	'CPANPLUS::Shell::Classic'=> '0.0562',
	'CPANPLUS::Shell::Default'=> '0.88',
	'CPANPLUS::Shell::Default::Plugins::CustomSource'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Remote'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Source'=> undef,
	'Carp'                  => '1.11',
	'Carp::Heavy'           => '1.11',
	'Class::ISA'            => '0.33',
	'Class::Struct'         => '0.63',
	'Compress::Raw::Bzip2'  => '2.020',
	'Compress::Raw::Zlib'   => '2.020',
	'Compress::Zlib'        => '2.020',
	'Config'                => undef,
	'Config::Extensions'    => '0.01',
	'Cwd'                   => '3.30',
	'DB'                    => '1.02',
	'DBM_Filter'            => '0.02',
	'DBM_Filter::compress'  => '0.02',
	'DBM_Filter::encode'    => '0.02',
	'DBM_Filter::int32'     => '0.02',
	'DBM_Filter::null'      => '0.02',
	'DBM_Filter::utf8'      => '0.02',
	'DB_File'               => '1.820',
	'DCLsym'                => '1.03',
	'Data::Dumper'          => '2.124',
	'Devel::DProf'          => '20080331.00',
	'Devel::InnerPackage'   => '0.3',
	'Devel::PPPort'         => '3.19',
	'Devel::Peek'           => '1.04',
	'Devel::SelfStubber'    => '1.03',
	'Digest'                => '1.16',
	'Digest::MD5'           => '2.39',
	'Digest::SHA'           => '5.47',
	'Digest::base'          => '1.16',
	'Digest::file'          => '1.16',
	'DirHandle'             => '1.03',
	'Dumpvalue'             => '1.13',
	'DynaLoader'            => '1.10',
	'Encode'                => '2.35',
	'Encode::Alias'         => '2.12',
	'Encode::Byte'          => '2.03',
	'Encode::CJKConstants'  => '2.02',
	'Encode::CN'            => '2.02',
	'Encode::CN::HZ'        => '2.05',
	'Encode::Config'        => '2.05',
	'Encode::EBCDIC'        => '2.02',
	'Encode::Encoder'       => '2.01',
	'Encode::Encoding'      => '2.05',
	'Encode::GSM0338'       => '2.01',
	'Encode::Guess'         => '2.03',
	'Encode::JP'            => '2.03',
	'Encode::JP::H2Z'       => '2.02',
	'Encode::JP::JIS7'      => '2.04',
	'Encode::KR'            => '2.02',
	'Encode::KR::2022_KR'   => '2.02',
	'Encode::MIME::Header'  => '2.11',
	'Encode::MIME::Header::ISO_2022_JP'=> '1.03',
	'Encode::MIME::Name'    => '1.01',
	'Encode::Symbol'        => '2.02',
	'Encode::TW'            => '2.02',
	'Encode::Unicode'       => '2.06',
	'Encode::Unicode::UTF7' => '2.04',
	'English'               => '1.04',
	'Env'                   => '1.00',
	'Errno'                 => '1.11',
	'Exporter'              => '5.63',
	'Exporter::Heavy'       => '5.63',
	'ExtUtils::CBuilder'    => '0.2602',
	'ExtUtils::CBuilder::Base'=> '0.2602',
	'ExtUtils::CBuilder::Platform::Unix'=> '0.2602',
	'ExtUtils::CBuilder::Platform::VMS'=> '0.2602',
	'ExtUtils::CBuilder::Platform::Windows'=> '0.2602',
	'ExtUtils::CBuilder::Platform::aix'=> '0.2602',
	'ExtUtils::CBuilder::Platform::cygwin'=> '0.2602',
	'ExtUtils::CBuilder::Platform::darwin'=> '0.2602',
	'ExtUtils::CBuilder::Platform::dec_osf'=> '0.2602',
	'ExtUtils::CBuilder::Platform::os2'=> '0.2602',
	'ExtUtils::Command'     => '1.16',
	'ExtUtils::Command::MM' => '6.55_02',
	'ExtUtils::Constant'    => '0.22',
	'ExtUtils::Constant::Base'=> '0.04',
	'ExtUtils::Constant::ProxySubs'=> '0.06',
	'ExtUtils::Constant::Utils'=> '0.02',
	'ExtUtils::Constant::XS'=> '0.03',
	'ExtUtils::Embed'       => '1.28',
	'ExtUtils::Install'     => '1.54',
	'ExtUtils::Installed'   => '1.999_001',
	'ExtUtils::Liblist'     => '6.55_02',
	'ExtUtils::Liblist::Kid'=> '6.5502',
	'ExtUtils::MM'          => '6.55_02',
	'ExtUtils::MM_AIX'      => '6.55_02',
	'ExtUtils::MM_Any'      => '6.55_02',
	'ExtUtils::MM_BeOS'     => '6.55_02',
	'ExtUtils::MM_Cygwin'   => '6.55_02',
	'ExtUtils::MM_DOS'      => '6.5502',
	'ExtUtils::MM_Darwin'   => '6.55_02',
	'ExtUtils::MM_MacOS'    => '6.5502',
	'ExtUtils::MM_NW5'      => '6.55_02',
	'ExtUtils::MM_OS2'      => '6.55_02',
	'ExtUtils::MM_QNX'      => '6.55_02',
	'ExtUtils::MM_UWIN'     => '6.5502',
	'ExtUtils::MM_Unix'     => '6.55_02',
	'ExtUtils::MM_VMS'      => '6.55_02',
	'ExtUtils::MM_VOS'      => '6.55_02',
	'ExtUtils::MM_Win32'    => '6.55_02',
	'ExtUtils::MM_Win95'    => '6.55_02',
	'ExtUtils::MY'          => '6.5502',
	'ExtUtils::MakeMaker'   => '6.55_02',
	'ExtUtils::MakeMaker::Config'=> '6.55_02',
	'ExtUtils::Manifest'    => '1.56',
	'ExtUtils::Miniperl'    => undef,
	'ExtUtils::Mkbootstrap' => '6.55_02',
	'ExtUtils::Mksymlists'  => '6.55_02',
	'ExtUtils::Packlist'    => '1.43',
	'ExtUtils::ParseXS'     => '2.2002',
	'ExtUtils::XSSymSet'    => '1.1',
	'ExtUtils::testlib'     => '6.5502',
	'Fatal'                 => '2.06_01',
	'Fcntl'                 => '1.06',
	'File::Basename'        => '2.77',
	'File::CheckTree'       => '4.4',
	'File::Compare'         => '1.1006',
	'File::Copy'            => '2.14',
	'File::DosGlob'         => '1.01',
	'File::Fetch'           => '0.20',
	'File::Find'            => '1.14',
	'File::Glob'            => '1.06',
	'File::GlobMapper'      => '1.000',
	'File::Path'            => '2.07_03',
	'File::Spec'            => '3.30',
	'File::Spec::Cygwin'    => '3.30',
	'File::Spec::Epoc'      => '3.30',
	'File::Spec::Functions' => '3.30',
	'File::Spec::Mac'       => '3.30',
	'File::Spec::OS2'       => '3.30',
	'File::Spec::Unix'      => '3.30',
	'File::Spec::VMS'       => '3.30',
	'File::Spec::Win32'     => '3.30',
	'File::Temp'            => '0.22',
	'File::stat'            => '1.01',
	'FileCache'             => '1.08',
	'FileHandle'            => '2.02',
	'VMS::Filespec'         => '1.12',
	'Filter::Simple'        => '0.84',
	'Filter::Util::Call'    => '1.08',
	'FindBin'               => '1.50',
	'GDBM_File'             => '1.09',
	'Getopt::Long'          => '2.38',
	'Getopt::Std'           => '1.06',
	'Hash::Util'            => '0.07',
	'Hash::Util::FieldHash' => '1.04',
	'I18N::Collate'         => '1.01',
	'I18N::LangTags'        => '0.35',
	'I18N::LangTags::Detect'=> '1.03',
	'I18N::LangTags::List'  => '0.35',
	'I18N::Langinfo'        => '0.02',
	'IO'                    => '1.25',
	'IO::Compress::Adapter::Bzip2'=> '2.020',
	'IO::Compress::Adapter::Deflate'=> '2.020',
	'IO::Compress::Adapter::Identity'=> '2.020',
	'IO::Compress::Base'    => '2.020',
	'IO::Compress::Base::Common'=> '2.020',
	'IO::Compress::Bzip2'   => '2.020',
	'IO::Compress::Deflate' => '2.020',
	'IO::Compress::Gzip'    => '2.020',
	'IO::Compress::Gzip::Constants'=> '2.020',
	'IO::Compress::RawDeflate'=> '2.020',
	'IO::Compress::Zip'     => '2.020',
	'IO::Compress::Zip::Constants'=> '2.020',
	'IO::Compress::Zlib::Constants'=> '2.020',
	'IO::Compress::Zlib::Extra'=> '2.020',
	'IO::Dir'               => '1.07',
	'IO::File'              => '1.14',
	'IO::Handle'            => '1.28',
	'IO::Pipe'              => '1.13',
	'IO::Poll'              => '0.07',
	'IO::Seekable'          => '1.10',
	'IO::Select'            => '1.17',
	'IO::Socket'            => '1.31',
	'IO::Socket::INET'      => '1.31',
	'IO::Socket::UNIX'      => '1.23',
	'IO::Uncompress::Adapter::Bunzip2'=> '2.020',
	'IO::Uncompress::Adapter::Identity'=> '2.020',
	'IO::Uncompress::Adapter::Inflate'=> '2.020',
	'IO::Uncompress::AnyInflate'=> '2.020',
	'IO::Uncompress::AnyUncompress'=> '2.020',
	'IO::Uncompress::Base'  => '2.020',
	'IO::Uncompress::Bunzip2'=> '2.020',
	'IO::Uncompress::Gunzip'=> '2.020',
	'IO::Uncompress::Inflate'=> '2.020',
	'IO::Uncompress::RawInflate'=> '2.020',
	'IO::Uncompress::Unzip' => '2.020',
	'IO::Zlib'              => '1.09',
	'IPC::Cmd'              => '0.46',
	'IPC::Msg'              => '2.01',
	'IPC::Open2'            => '1.03',
	'IPC::Open3'            => '1.04',
	'IPC::Semaphore'        => '2.01',
	'IPC::SharedMem'        => '2.01',
	'IPC::SysV'             => '2.01',
	'List::Util'            => '1.21',
	'List::Util::PP'        => '1.21',
	'List::Util::XS'        => '1.21',
	'Locale::Constants'     => '2.07',
	'Locale::Country'       => '2.07',
	'Locale::Currency'      => '2.07',
	'Locale::Language'      => '2.07',
	'Locale::Maketext'      => '1.13',
	'Locale::Maketext::Guts'=> '1.13',
	'Locale::Maketext::GutsLoader'=> '1.13',
	'Locale::Maketext::Simple'=> '0.18',
	'Locale::Script'        => '2.07',
	'Log::Message'          => '0.02',
	'Log::Message::Config'  => '0.01',
	'Log::Message::Handlers'=> undef,
	'Log::Message::Item'    => undef,
	'Log::Message::Simple'  => '0.04',
	'MIME::Base64'          => '3.08',
	'MIME::QuotedPrint'     => '3.08',
	'Math::BigFloat'        => '1.60',
	'Math::BigFloat::Trace' => '0.01',
	'Math::BigInt'          => '1.89',
	'Math::BigInt::Calc'    => '0.52',
	'Math::BigInt::CalcEmu' => '0.05',
	'Math::BigInt::FastCalc'=> '0.19',
	'Math::BigInt::Trace'   => '0.01',
	'Math::BigRat'          => '0.22',
	'Math::Complex'         => '1.56',
	'Math::Trig'            => '1.2',
	'Memoize'               => '1.01_03',
	'Memoize::AnyDBM_File'  => '0.65',
	'Memoize::Expire'       => '1.00',
	'Memoize::ExpireFile'   => '1.01',
	'Memoize::ExpireTest'   => '0.65',
	'Memoize::NDBM_File'    => '0.65',
	'Memoize::SDBM_File'    => '0.65',
	'Memoize::Storable'     => '0.65',
	'Module::Build'         => '0.340201',
	'Module::Build::Base'   => '0.340201',
	'Module::Build::Compat' => '0.340201',
	'Module::Build::Config' => '0.340201',
	'Module::Build::ConfigData'=> undef,
	'Module::Build::Cookbook'=> '0.340201',
	'Module::Build::Dumper' => '0.340201',
	'Module::Build::ModuleInfo'=> '0.340201',
	'Module::Build::Notes'  => '0.340201',
	'Module::Build::PPMMaker'=> '0.340201',
	'Module::Build::Platform::Amiga'=> '0.340201',
	'Module::Build::Platform::Default'=> '0.340201',
	'Module::Build::Platform::EBCDIC'=> '0.340201',
	'Module::Build::Platform::MPEiX'=> '0.340201',
	'Module::Build::Platform::MacOS'=> '0.340201',
	'Module::Build::Platform::RiscOS'=> '0.340201',
	'Module::Build::Platform::Unix'=> '0.340201',
	'Module::Build::Platform::VMS'=> '0.340201',
	'Module::Build::Platform::VOS'=> '0.340201',
	'Module::Build::Platform::Windows'=> '0.340201',
	'Module::Build::Platform::aix'=> '0.340201',
	'Module::Build::Platform::cygwin'=> '0.340201',
	'Module::Build::Platform::darwin'=> '0.340201',
	'Module::Build::Platform::os2'=> '0.340201',
	'Module::Build::PodParser'=> '0.340201',
	'Module::Build::Version'=> '0.77',
	'Module::Build::YAML'   => '0.50',
	'Module::CoreList'      => '2.18',
	'Module::Load'          => '0.16',
	'Module::Load::Conditional'=> '0.30',
	'Module::Loaded'        => '0.02',
	'Module::Pluggable'     => '3.9',
	'Module::Pluggable::Object'=> '3.9',
	'Moped::Msg'            => '0.01',
	'NDBM_File'             => '1.08',
	'NEXT'                  => '0.64',
	'Net::Cmd'              => '2.29',
	'Net::Config'           => '1.11',
	'Net::Domain'           => '2.20',
	'Net::FTP'              => '2.77',
	'Net::FTP::A'           => '1.18',
	'Net::FTP::E'           => '0.01',
	'Net::FTP::I'           => '1.12',
	'Net::FTP::L'           => '0.01',
	'Net::FTP::dataconn'    => '0.11',
	'Net::NNTP'             => '2.24',
	'Net::Netrc'            => '2.12',
	'Net::POP3'             => '2.29',
	'Net::Ping'             => '2.36',
	'Net::SMTP'             => '2.31',
	'Net::Time'             => '2.10',
	'Net::hostent'          => '1.01',
	'Net::netent'           => '1.00',
	'Net::protoent'         => '1.00',
	'Net::servent'          => '1.01',
	'O'                     => '1.01',
	'ODBM_File'             => '1.07',
	'Object::Accessor'      => '0.34',
	'Opcode'                => '1.11',
	'POSIX'                 => '1.17',
	'Package::Constants'    => '0.02',
	'Params::Check'         => '0.26',
	'Parse::CPAN::Meta'     => '1.39',
	'PerlIO'                => '1.06',
	'PerlIO::encoding'      => '0.11',
	'PerlIO::scalar'        => '0.07',
	'PerlIO::via'           => '0.07',
	'PerlIO::via::QuotedPrint'=> '0.06',
	'Pod::Checker'          => '1.45',
	'Pod::Escapes'          => '1.04',
	'Pod::Find'             => '1.35',
	'Pod::Functions'        => '1.03',
	'Pod::Html'             => '1.09',
	'Pod::InputObjects'     => '1.31',
	'Pod::LaTeX'            => '0.58',
	'Pod::Man'              => '2.22',
	'Pod::ParseLink'        => '1.09',
	'Pod::ParseUtils'       => '1.36',
	'Pod::Parser'           => '1.37',
	'Pod::Perldoc'          => '3.14_04',
	'Pod::Perldoc::BaseTo'  => undef,
	'Pod::Perldoc::GetOptsOO'=> undef,
	'Pod::Perldoc::ToChecker'=> undef,
	'Pod::Perldoc::ToMan'   => undef,
	'Pod::Perldoc::ToNroff' => undef,
	'Pod::Perldoc::ToPod'   => undef,
	'Pod::Perldoc::ToRtf'   => undef,
	'Pod::Perldoc::ToText'  => undef,
	'Pod::Perldoc::ToTk'    => undef,
	'Pod::Perldoc::ToXml'   => undef,
	'Pod::PlainText'        => '2.04',
	'Pod::Plainer'          => '0.01',
	'Pod::Select'           => '1.36',
	'Pod::Simple'           => '3.07',
	'Pod::Simple::BlackBox' => undef,
	'Pod::Simple::Checker'  => '2.02',
	'Pod::Simple::Debug'    => undef,
	'Pod::Simple::DumpAsText'=> '2.02',
	'Pod::Simple::DumpAsXML'=> '2.02',
	'Pod::Simple::HTML'     => '3.03',
	'Pod::Simple::HTMLBatch'=> '3.02',
	'Pod::Simple::HTMLLegacy'=> '5.01',
	'Pod::Simple::LinkSection'=> undef,
	'Pod::Simple::Methody'  => '2.02',
	'Pod::Simple::Progress' => '1.01',
	'Pod::Simple::PullParser'=> '2.02',
	'Pod::Simple::PullParserEndToken'=> undef,
	'Pod::Simple::PullParserStartToken'=> undef,
	'Pod::Simple::PullParserTextToken'=> undef,
	'Pod::Simple::PullParserToken'=> '2.02',
	'Pod::Simple::RTF'      => '2.02',
	'Pod::Simple::Search'   => '3.04',
	'Pod::Simple::SimpleTree'=> '2.02',
	'Pod::Simple::Text'     => '2.02',
	'Pod::Simple::TextContent'=> '2.02',
	'Pod::Simple::TiedOutFH'=> undef,
	'Pod::Simple::Transcode'=> undef,
	'Pod::Simple::TranscodeDumb'=> '2.02',
	'Pod::Simple::TranscodeSmart'=> undef,
	'Pod::Simple::XHTML'    => '3.04',
	'Pod::Simple::XMLOutStream'=> '2.02',
	'Pod::Text'             => '3.13',
	'Pod::Text::Color'      => '2.05',
	'Pod::Text::Overstrike' => '2.03',
	'Pod::Text::Termcap'    => '2.05',
	'Pod::Usage'            => '1.36',
	'SDBM_File'             => '1.06',
	'Safe'                  => '2.18',
	'Scalar::Util'          => '1.21',
	'Scalar::Util::PP'      => '1.21',
	'Search::Dict'          => '1.02',
	'SelectSaver'           => '1.02',
	'SelfLoader'            => '1.17',
	'Shell'                 => '0.72_01',
	'Socket'                => '1.82',
	'Stdio'                 => '2.4',
	'Storable'              => '2.20',
	'Switch'                => '2.14',
	'Symbol'                => '1.07',
	'Sys::Hostname'         => '1.11',
	'Sys::Syslog'           => '0.27',
	'Sys::Syslog::win32::Win32'=> undef,
	'TAP::Base'             => '3.17',
	'TAP::Formatter::Base'  => '3.17',
	'TAP::Formatter::Color' => '3.17',
	'TAP::Formatter::Console'=> '3.17',
	'TAP::Formatter::Console::ParallelSession'=> '3.17',
	'TAP::Formatter::Console::Session'=> '3.17',
	'TAP::Formatter::File'  => '3.17',
	'TAP::Formatter::File::Session'=> '3.17',
	'TAP::Formatter::Session'=> '3.17',
	'TAP::Harness'          => '3.17',
	'TAP::Object'           => '3.17',
	'TAP::Parser'           => '3.17',
	'TAP::Parser::Aggregator'=> '3.17',
	'TAP::Parser::Grammar'  => '3.17',
	'TAP::Parser::Iterator' => '3.17',
	'TAP::Parser::Iterator::Array'=> '3.17',
	'TAP::Parser::Iterator::Process'=> '3.17',
	'TAP::Parser::Iterator::Stream'=> '3.17',
	'TAP::Parser::IteratorFactory'=> '3.17',
	'TAP::Parser::Multiplexer'=> '3.17',
	'TAP::Parser::Result'   => '3.17',
	'TAP::Parser::Result::Bailout'=> '3.17',
	'TAP::Parser::Result::Comment'=> '3.17',
	'TAP::Parser::Result::Plan'=> '3.17',
	'TAP::Parser::Result::Pragma'=> '3.17',
	'TAP::Parser::Result::Test'=> '3.17',
	'TAP::Parser::Result::Unknown'=> '3.17',
	'TAP::Parser::Result::Version'=> '3.17',
	'TAP::Parser::Result::YAML'=> '3.17',
	'TAP::Parser::ResultFactory'=> '3.17',
	'TAP::Parser::Scheduler'=> '3.17',
	'TAP::Parser::Scheduler::Job'=> '3.17',
	'TAP::Parser::Scheduler::Spinner'=> '3.17',
	'TAP::Parser::Source'   => '3.17',
	'TAP::Parser::Source::Perl'=> '3.17',
	'TAP::Parser::Utils'    => '3.17',
	'TAP::Parser::YAMLish::Reader'=> '3.17',
	'TAP::Parser::YAMLish::Writer'=> '3.17',
	'Term::ANSIColor'       => '2.00',
	'Term::Cap'             => '1.12',
	'Term::Complete'        => '1.402',
	'Term::ReadLine'        => '1.04',
	'Term::UI'              => '0.20',
	'Term::UI::History'     => undef,
	'Test'                  => '1.25_02',
	'Test::Builder'         => '0.92',
	'Test::Builder::Module' => '0.92',
	'Test::Builder::Tester' => '1.18',
	'Test::Builder::Tester::Color'=> '1.18',
	'Test::Harness'         => '3.17',
	'Test::More'            => '0.92',
	'Test::Simple'          => '0.92',
	'Text::Abbrev'          => '1.01',
	'Text::Balanced'        => '2.0.0',
	'Text::ParseWords'      => '3.27',
	'Text::Soundex'         => '3.03',
	'Text::Tabs'            => '2009.0305',
	'Text::Wrap'            => '2009.0305',
	'Thread'                => '3.02',
	'Thread::Queue'         => '2.11',
	'Thread::Semaphore'     => '2.09',
	'Tie::Array'            => '1.03',
	'Tie::File'             => '0.97_02',
	'Tie::Handle'           => '4.2',
	'Tie::Hash'             => '1.03',
	'Tie::Hash::NamedCapture'=> '0.06',
	'Tie::Memoize'          => '1.1',
	'Tie::RefHash'          => '1.38',
	'Tie::Scalar'           => '1.01',
	'Tie::StdHandle'        => '4.2',
	'Tie::SubstrHash'       => '1.00',
	'Time::HiRes'           => '1.9719',
	'Time::Local'           => '1.1901',
	'Time::Piece'           => '1.15',
	'Time::Piece::Seconds'  => undef,
	'Time::Seconds'         => undef,
	'Time::gmtime'          => '1.03',
	'Time::localtime'       => '1.02',
	'Time::tm'              => '1.00',
	'UNIVERSAL'             => '1.05',
	'Unicode'               => '5.1.0',
	'Unicode::Collate'      => '0.52',
	'Unicode::Normalize'    => '1.03',
	'Unicode::UCD'          => '0.27',
	'User::grent'           => '1.01',
	'User::pwent'           => '1.00',
	'Win32'                 => '0.39',
	'Win32API::File'        => '0.1101',
	'Win32API::File::ExtUtils::Myconst2perl'=> '1',
	'Win32CORE'             => '0.02',
	'XS::APItest'           => '0.15',
	'XS::Typemap'           => '0.03',
	'XSLoader'              => '0.10',
	'attributes'            => '0.09',
	'attrs'                 => '1.03',
	'autodie'               => '2.06_01',
	'autodie::exception'    => '2.06_01',
	'autodie::exception::system'=> '2.06_01',
	'autodie::hints'        => '2.06_01',
	'autouse'               => '1.06',
	'base'                  => '2.14',
	'bigint'                => '0.23',
	'bignum'                => '0.23',
	'bigrat'                => '0.23',
	'blib'                  => '1.04',
	'bytes'                 => '1.03',
	'charnames'             => '1.07',
	'constant'              => '1.17',
	'diagnostics'           => '1.17',
	'encoding'              => '2.6_01',
	'encoding::warnings'    => '0.11',
	'feature'               => '1.13',
	'fields'                => '2.14',
	'filetest'              => '1.02',
	'if'                    => '0.05',
	'integer'               => '1.00',
	'less'                  => '0.02',
	'lib'                   => '0.62',
	'locale'                => '1.00',
	'mro'                   => '1.01',
	'open'                  => '1.07',
	'ops'                   => '1.02',
	'overload'              => '1.07',
	'overload::numbers'     => undef,
	'overloading'           => '0.01',
	'parent'                => '0.221',
	're'                    => '0.09',
	'sigtrap'               => '1.04',
	'sort'                  => '2.01',
	'strict'                => '1.04',
	'subs'                  => '1.00',
	'threads'               => '1.72',
	'threads::shared'       => '1.29',
	'utf8'                  => '1.07',
	'vars'                  => '1.01',
	'version'               => '0.77',
	'vmsish'                => '1.02',
	'warnings'              => '1.06',
	'warnings::register'    => '1.01',
    },

    5.011000 => {
	'AnyDBM_File'           => '1.00',
	'App::Prove'            => '3.17',
	'App::Prove::State'     => '3.17',
	'App::Prove::State::Result'=> '3.17',
	'App::Prove::State::Result::Test'=> '3.17',
	'Archive::Extract'      => '0.34',
	'Archive::Tar'          => '1.54',
	'Archive::Tar::Constant'=> '0.02',
	'Archive::Tar::File'    => '0.02',
	'Attribute::Handlers'   => '0.87',
	'AutoLoader'            => '5.70',
	'AutoSplit'             => '1.06',
	'B'                     => '1.22',
	'B::Concise'            => '0.76',
	'B::Debug'              => '1.11',
	'B::Deparse'            => '0.91',
	'B::Lint'               => '1.11_01',
	'B::Lint::Debug'        => '0.01',
	'B::Showlex'            => '1.02',
	'B::Terse'              => '1.05',
	'B::Xref'               => '1.02',
	'Benchmark'             => '1.11',
	'CGI'                   => '3.45',
	'CGI::Apache'           => '1.01',
	'CGI::Carp'             => '3.45',
	'CGI::Cookie'           => '1.29',
	'CGI::Fast'             => '1.07',
	'CGI::Pretty'           => '3.44',
	'CGI::Push'             => '1.04',
	'CGI::Switch'           => '1.01',
	'CGI::Util'             => '3.45',
	'CPAN'                  => '1.94_51',
	'CPAN::Author'          => '5.5',
	'CPAN::Bundle'          => '5.5',
	'CPAN::CacheMgr'        => '5.5',
	'CPAN::Complete'        => '5.5',
	'CPAN::Debug'           => '5.5',
	'CPAN::DeferredCode'    => '5.50',
	'CPAN::Distribution'    => '1.94',
	'CPAN::Distroprefs'     => '6',
	'CPAN::Distrostatus'    => '5.5',
	'CPAN::Exception::RecursiveDependency'=> '5.5',
	'CPAN::Exception::blocked_urllist'=> '1.0',
	'CPAN::Exception::yaml_not_installed'=> '5.5',
	'CPAN::FTP'             => '5.5002',
	'CPAN::FTP::netrc'      => '1.00',
	'CPAN::FirstTime'       => '5.53',
	'CPAN::HandleConfig'    => '5.5',
	'CPAN::Index'           => '1.94',
	'CPAN::InfoObj'         => '5.5',
	'CPAN::Kwalify'         => '5.50',
	'CPAN::LWP::UserAgent'  => '1.94',
	'CPAN::Module'          => '5.5',
	'CPAN::Nox'             => '5.50',
	'CPAN::Prompt'          => '5.5',
	'CPAN::Queue'           => '5.5',
	'CPAN::Shell'           => '5.5',
	'CPAN::Tarzip'          => '5.501',
	'CPAN::URL'             => '5.5',
	'CPAN::Version'         => '5.5',
	'CPANPLUS'              => '0.88',
	'CPANPLUS::Backend'     => undef,
	'CPANPLUS::Backend::RV' => undef,
	'CPANPLUS::Config'      => undef,
	'CPANPLUS::Configure'   => undef,
	'CPANPLUS::Configure::Setup'=> undef,
	'CPANPLUS::Dist'        => undef,
	'CPANPLUS::Dist::Autobundle'=> undef,
	'CPANPLUS::Dist::Base'  => undef,
	'CPANPLUS::Dist::Build' => '0.40',
	'CPANPLUS::Dist::Build::Constants'=> '0.40',
	'CPANPLUS::Dist::MM'    => undef,
	'CPANPLUS::Dist::Sample'=> undef,
	'CPANPLUS::Error'       => undef,
	'CPANPLUS::Internals'   => '0.88',
	'CPANPLUS::Internals::Constants'=> undef,
	'CPANPLUS::Internals::Constants::Report'=> undef,
	'CPANPLUS::Internals::Extract'=> undef,
	'CPANPLUS::Internals::Fetch'=> undef,
	'CPANPLUS::Internals::Report'=> undef,
	'CPANPLUS::Internals::Search'=> undef,
	'CPANPLUS::Internals::Source'=> undef,
	'CPANPLUS::Internals::Source::Memory'=> undef,
	'CPANPLUS::Internals::Source::SQLite'=> undef,
	'CPANPLUS::Internals::Source::SQLite::Tie'=> undef,
	'CPANPLUS::Internals::Utils'=> undef,
	'CPANPLUS::Internals::Utils::Autoflush'=> undef,
	'CPANPLUS::Module'      => undef,
	'CPANPLUS::Module::Author'=> undef,
	'CPANPLUS::Module::Author::Fake'=> undef,
	'CPANPLUS::Module::Checksums'=> undef,
	'CPANPLUS::Module::Fake'=> undef,
	'CPANPLUS::Module::Signature'=> undef,
	'CPANPLUS::Selfupdate'  => undef,
	'CPANPLUS::Shell'       => undef,
	'CPANPLUS::Shell::Classic'=> '0.0562',
	'CPANPLUS::Shell::Default'=> '0.88',
	'CPANPLUS::Shell::Default::Plugins::CustomSource'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Remote'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Source'=> undef,
	'Carp'                  => '1.12',
	'Carp::Heavy'           => '1.12',
	'Class::ISA'            => '0.36',
	'Class::Struct'         => '0.63',
	'Compress::Raw::Bzip2'  => '2.021',
	'Compress::Raw::Zlib'   => '2.021',
	'Compress::Zlib'        => '2.021',
	'Config'                => undef,
	'Config::Extensions'    => '0.01',
	'Cwd'                   => '3.3002',
	'DB'                    => '1.02',
	'DBM_Filter'            => '0.02',
	'DBM_Filter::compress'  => '0.02',
	'DBM_Filter::encode'    => '0.02',
	'DBM_Filter::int32'     => '0.02',
	'DBM_Filter::null'      => '0.02',
	'DBM_Filter::utf8'      => '0.02',
	'DB_File'               => '1.820',
	'Data::Dumper'          => '2.125',
	'Devel::DProf'          => '20080331.00',
	'Devel::DProf::dprof::V'=> undef,
	'Devel::InnerPackage'   => '0.3',
	'Devel::PPPort'         => '3.19',
	'Devel::Peek'           => '1.04',
	'Devel::SelfStubber'    => '1.03',
	'Digest'                => '1.16',
	'Digest::MD5'           => '2.39',
	'Digest::SHA'           => '5.47',
	'Digest::base'          => '1.16',
	'Digest::file'          => '1.16',
	'DirHandle'             => '1.03',
	'Dumpvalue'             => '1.13',
	'DynaLoader'            => '1.10',
	'Encode'                => '2.37',
	'Encode::Alias'         => '2.12',
	'Encode::Byte'          => '2.03',
	'Encode::CJKConstants'  => '2.02',
	'Encode::CN'            => '2.02',
	'Encode::CN::HZ'        => '2.05',
	'Encode::Config'        => '2.05',
	'Encode::EBCDIC'        => '2.02',
	'Encode::Encoder'       => '2.01',
	'Encode::Encoding'      => '2.05',
	'Encode::GSM0338'       => '2.01',
	'Encode::Guess'         => '2.03',
	'Encode::JP'            => '2.03',
	'Encode::JP::H2Z'       => '2.02',
	'Encode::JP::JIS7'      => '2.04',
	'Encode::KR'            => '2.02',
	'Encode::KR::2022_KR'   => '2.02',
	'Encode::MIME::Header'  => '2.11',
	'Encode::MIME::Header::ISO_2022_JP'=> '1.03',
	'Encode::MIME::Name'    => '1.01',
	'Encode::Symbol'        => '2.02',
	'Encode::TW'            => '2.02',
	'Encode::Unicode'       => '2.06',
	'Encode::Unicode::UTF7' => '2.04',
	'English'               => '1.04',
	'Env'                   => '1.00',
	'Errno'                 => '1.11',
	'Exporter'              => '5.64',
	'Exporter::Heavy'       => '5.64',
	'ExtUtils::CBuilder'    => '0.2602',
	'ExtUtils::CBuilder::Base'=> '0.2602',
	'ExtUtils::CBuilder::Platform::Unix'=> '0.2602',
	'ExtUtils::CBuilder::Platform::VMS'=> '0.2602',
	'ExtUtils::CBuilder::Platform::Windows'=> '0.2602',
	'ExtUtils::CBuilder::Platform::aix'=> '0.2602',
	'ExtUtils::CBuilder::Platform::cygwin'=> '0.2602',
	'ExtUtils::CBuilder::Platform::darwin'=> '0.2602',
	'ExtUtils::CBuilder::Platform::dec_osf'=> '0.2602',
	'ExtUtils::CBuilder::Platform::os2'=> '0.2602',
	'ExtUtils::Command'     => '1.16',
	'ExtUtils::Command::MM' => '6.55_02',
	'ExtUtils::Constant'    => '0.22',
	'ExtUtils::Constant::Base'=> '0.04',
	'ExtUtils::Constant::ProxySubs'=> '0.06',
	'ExtUtils::Constant::Utils'=> '0.02',
	'ExtUtils::Constant::XS'=> '0.03',
	'ExtUtils::Embed'       => '1.28',
	'ExtUtils::Install'     => '1.54',
	'ExtUtils::Installed'   => '1.999_001',
	'ExtUtils::Liblist'     => '6.55_02',
	'ExtUtils::Liblist::Kid'=> '6.5502',
	'ExtUtils::MM'          => '6.55_02',
	'ExtUtils::MM_AIX'      => '6.55_02',
	'ExtUtils::MM_Any'      => '6.55_02',
	'ExtUtils::MM_BeOS'     => '6.55_02',
	'ExtUtils::MM_Cygwin'   => '6.55_02',
	'ExtUtils::MM_DOS'      => '6.5502',
	'ExtUtils::MM_Darwin'   => '6.55_02',
	'ExtUtils::MM_MacOS'    => '6.5502',
	'ExtUtils::MM_NW5'      => '6.55_02',
	'ExtUtils::MM_OS2'      => '6.55_02',
	'ExtUtils::MM_QNX'      => '6.55_02',
	'ExtUtils::MM_UWIN'     => '6.5502',
	'ExtUtils::MM_Unix'     => '6.55_02',
	'ExtUtils::MM_VMS'      => '6.55_02',
	'ExtUtils::MM_VOS'      => '6.55_02',
	'ExtUtils::MM_Win32'    => '6.55_02',
	'ExtUtils::MM_Win95'    => '6.55_02',
	'ExtUtils::MY'          => '6.5502',
	'ExtUtils::MakeMaker'   => '6.55_02',
	'ExtUtils::MakeMaker::Config'=> '6.55_02',
	'ExtUtils::Manifest'    => '1.56',
	'ExtUtils::Miniperl'    => undef,
	'ExtUtils::Mkbootstrap' => '6.55_02',
	'ExtUtils::Mksymlists'  => '6.55_02',
	'ExtUtils::Packlist'    => '1.43',
	'ExtUtils::ParseXS'     => '2.200403',
	'ExtUtils::XSSymSet'    => '1.1',
	'ExtUtils::testlib'     => '6.5502',
	'Fatal'                 => '2.06_01',
	'Fcntl'                 => '1.06',
	'File::Basename'        => '2.78',
	'File::CheckTree'       => '4.4',
	'File::Compare'         => '1.1006',
	'File::Copy'            => '2.16',
	'File::DosGlob'         => '1.01',
	'File::Fetch'           => '0.20',
	'File::Find'            => '1.14',
	'File::Glob'            => '1.06',
	'File::GlobMapper'      => '1.000',
	'File::Path'            => '2.07_03',
	'File::Spec'            => '3.30',
	'File::Spec::Cygwin'    => '3.30',
	'File::Spec::Epoc'      => '3.30',
	'File::Spec::Functions' => '3.30',
	'File::Spec::Mac'       => '3.30',
	'File::Spec::OS2'       => '3.30',
	'File::Spec::Unix'      => '3.30',
	'File::Spec::VMS'       => '3.30',
	'File::Spec::Win32'     => '3.30',
	'File::Temp'            => '0.22',
	'File::stat'            => '1.02',
	'FileCache'             => '1.08',
	'FileHandle'            => '2.02',
	'VMS::Filespec'         => '1.12',
	'Filter::Simple'        => '0.84',
	'Filter::Util::Call'    => '1.08',
	'FindBin'               => '1.50',
	'GDBM_File'             => '1.09',
	'Getopt::Long'          => '2.38',
	'Getopt::Std'           => '1.06',
	'Hash::Util'            => '0.07',
	'Hash::Util::FieldHash' => '1.04',
	'I18N::Collate'         => '1.01',
	'I18N::LangTags'        => '0.35',
	'I18N::LangTags::Detect'=> '1.03',
	'I18N::LangTags::List'  => '0.35',
	'I18N::Langinfo'        => '0.02',
	'IO'                    => '1.25_01',
	'IO::Compress::Adapter::Bzip2'=> '2.021',
	'IO::Compress::Adapter::Deflate'=> '2.021',
	'IO::Compress::Adapter::Identity'=> '2.021',
	'IO::Compress::Base'    => '2.021',
	'IO::Compress::Base::Common'=> '2.021',
	'IO::Compress::Bzip2'   => '2.021',
	'IO::Compress::Deflate' => '2.021',
	'IO::Compress::Gzip'    => '2.021',
	'IO::Compress::Gzip::Constants'=> '2.021',
	'IO::Compress::RawDeflate'=> '2.021',
	'IO::Compress::Zip'     => '2.021',
	'IO::Compress::Zip::Constants'=> '2.021',
	'IO::Compress::Zlib::Constants'=> '2.021',
	'IO::Compress::Zlib::Extra'=> '2.021',
	'IO::Dir'               => '1.07',
	'IO::File'              => '1.14',
	'IO::Handle'            => '1.28',
	'IO::Pipe'              => '1.13',
	'IO::Poll'              => '0.07',
	'IO::Seekable'          => '1.10',
	'IO::Select'            => '1.17',
	'IO::Socket'            => '1.31',
	'IO::Socket::INET'      => '1.31',
	'IO::Socket::UNIX'      => '1.23',
	'IO::Uncompress::Adapter::Bunzip2'=> '2.021',
	'IO::Uncompress::Adapter::Identity'=> '2.021',
	'IO::Uncompress::Adapter::Inflate'=> '2.021',
	'IO::Uncompress::AnyInflate'=> '2.021',
	'IO::Uncompress::AnyUncompress'=> '2.021',
	'IO::Uncompress::Base'  => '2.021',
	'IO::Uncompress::Bunzip2'=> '2.021',
	'IO::Uncompress::Gunzip'=> '2.021',
	'IO::Uncompress::Inflate'=> '2.021',
	'IO::Uncompress::RawInflate'=> '2.021',
	'IO::Uncompress::Unzip' => '2.021',
	'IO::Zlib'              => '1.10',
	'IPC::Cmd'              => '0.50',
	'IPC::Msg'              => '2.01',
	'IPC::Open2'            => '1.03',
	'IPC::Open3'            => '1.05',
	'IPC::Semaphore'        => '2.01',
	'IPC::SharedMem'        => '2.01',
	'IPC::SysV'             => '2.01',
	'List::Util'            => '1.21',
	'List::Util::PP'        => '1.21',
	'List::Util::XS'        => '1.21',
	'Locale::Constants'     => '2.07',
	'Locale::Country'       => '2.07',
	'Locale::Currency'      => '2.07',
	'Locale::Language'      => '2.07',
	'Locale::Maketext'      => '1.13',
	'Locale::Maketext::Guts'=> '1.13',
	'Locale::Maketext::GutsLoader'=> '1.13',
	'Locale::Maketext::Simple'=> '0.21',
	'Locale::Script'        => '2.07',
	'Log::Message'          => '0.02',
	'Log::Message::Config'  => '0.01',
	'Log::Message::Handlers'=> undef,
	'Log::Message::Item'    => undef,
	'Log::Message::Simple'  => '0.06',
	'MIME::Base64'          => '3.08',
	'MIME::QuotedPrint'     => '3.08',
	'Math::BigFloat'        => '1.60',
	'Math::BigFloat::Trace' => '0.01',
	'Math::BigInt'          => '1.89_01',
	'Math::BigInt::Calc'    => '0.52',
	'Math::BigInt::CalcEmu' => '0.05',
	'Math::BigInt::FastCalc'=> '0.19',
	'Math::BigInt::Trace'   => '0.01',
	'Math::BigRat'          => '0.24',
	'Math::Complex'         => '1.56',
	'Math::Trig'            => '1.2',
	'Memoize'               => '1.01_03',
	'Memoize::AnyDBM_File'  => '0.65',
	'Memoize::Expire'       => '1.00',
	'Memoize::ExpireFile'   => '1.01',
	'Memoize::ExpireTest'   => '0.65',
	'Memoize::NDBM_File'    => '0.65',
	'Memoize::SDBM_File'    => '0.65',
	'Memoize::Storable'     => '0.65',
	'Module::Build'         => '0.35',
	'Module::Build::Base'   => '0.35',
	'Module::Build::Compat' => '0.35',
	'Module::Build::Config' => '0.35',
	'Module::Build::ConfigData'=> undef,
	'Module::Build::Cookbook'=> '0.35',
	'Module::Build::Dumper' => '0.35',
	'Module::Build::ModuleInfo'=> '0.35',
	'Module::Build::Notes'  => '0.35',
	'Module::Build::PPMMaker'=> '0.35',
	'Module::Build::Platform::Amiga'=> '0.35',
	'Module::Build::Platform::Default'=> '0.35',
	'Module::Build::Platform::EBCDIC'=> '0.35',
	'Module::Build::Platform::MPEiX'=> '0.35',
	'Module::Build::Platform::MacOS'=> '0.35',
	'Module::Build::Platform::RiscOS'=> '0.35',
	'Module::Build::Platform::Unix'=> '0.35',
	'Module::Build::Platform::VMS'=> '0.35',
	'Module::Build::Platform::VOS'=> '0.35',
	'Module::Build::Platform::Windows'=> '0.35',
	'Module::Build::Platform::aix'=> '0.35',
	'Module::Build::Platform::cygwin'=> '0.35',
	'Module::Build::Platform::darwin'=> '0.35',
	'Module::Build::Platform::os2'=> '0.35',
	'Module::Build::PodParser'=> '0.35',
	'Module::Build::Version'=> '0.77',
	'Module::Build::YAML'   => '0.50',
	'Module::CoreList'      => '2.19',
	'Module::Load'          => '0.16',
	'Module::Load::Conditional'=> '0.30',
	'Module::Loaded'        => '0.06',
	'Module::Pluggable'     => '3.9',
	'Module::Pluggable::Object'=> '3.9',
	'Moped::Msg'            => '0.01',
	'NDBM_File'             => '1.08',
	'NEXT'                  => '0.64',
	'Net::Cmd'              => '2.29',
	'Net::Config'           => '1.11',
	'Net::Domain'           => '2.20',
	'Net::FTP'              => '2.77',
	'Net::FTP::A'           => '1.18',
	'Net::FTP::E'           => '0.01',
	'Net::FTP::I'           => '1.12',
	'Net::FTP::L'           => '0.01',
	'Net::FTP::dataconn'    => '0.11',
	'Net::NNTP'             => '2.24',
	'Net::Netrc'            => '2.12',
	'Net::POP3'             => '2.29',
	'Net::Ping'             => '2.36',
	'Net::SMTP'             => '2.31',
	'Net::Time'             => '2.10',
	'Net::hostent'          => '1.01',
	'Net::netent'           => '1.00',
	'Net::protoent'         => '1.00',
	'Net::servent'          => '1.01',
	'O'                     => '1.01',
	'ODBM_File'             => '1.07',
	'Object::Accessor'      => '0.34',
	'Opcode'                => '1.13',
	'POSIX'                 => '1.17',
	'Package::Constants'    => '0.02',
	'Params::Check'         => '0.26',
	'Parse::CPAN::Meta'     => '1.39',
	'PerlIO'                => '1.06',
	'PerlIO::encoding'      => '0.11',
	'PerlIO::scalar'        => '0.07',
	'PerlIO::via'           => '0.08',
	'PerlIO::via::QuotedPrint'=> '0.06',
	'Pod::Checker'          => '1.45',
	'Pod::Escapes'          => '1.04',
	'Pod::Find'             => '1.35',
	'Pod::Functions'        => '1.03',
	'Pod::Html'             => '1.09',
	'Pod::InputObjects'     => '1.31',
	'Pod::LaTeX'            => '0.58',
	'Pod::Man'              => '2.22',
	'Pod::ParseLink'        => '1.09',
	'Pod::ParseUtils'       => '1.36',
	'Pod::Parser'           => '1.37',
	'Pod::Perldoc'          => '3.15_01',
	'Pod::Perldoc::BaseTo'  => undef,
	'Pod::Perldoc::GetOptsOO'=> undef,
	'Pod::Perldoc::ToChecker'=> undef,
	'Pod::Perldoc::ToMan'   => undef,
	'Pod::Perldoc::ToNroff' => undef,
	'Pod::Perldoc::ToPod'   => undef,
	'Pod::Perldoc::ToRtf'   => undef,
	'Pod::Perldoc::ToText'  => undef,
	'Pod::Perldoc::ToTk'    => undef,
	'Pod::Perldoc::ToXml'   => undef,
	'Pod::PlainText'        => '2.04',
	'Pod::Plainer'          => '1.01',
	'Pod::Select'           => '1.36',
	'Pod::Simple'           => '3.07',
	'Pod::Simple::BlackBox' => undef,
	'Pod::Simple::Checker'  => '2.02',
	'Pod::Simple::Debug'    => undef,
	'Pod::Simple::DumpAsText'=> '2.02',
	'Pod::Simple::DumpAsXML'=> '2.02',
	'Pod::Simple::HTML'     => '3.03',
	'Pod::Simple::HTMLBatch'=> '3.02',
	'Pod::Simple::HTMLLegacy'=> '5.01',
	'Pod::Simple::LinkSection'=> undef,
	'Pod::Simple::Methody'  => '2.02',
	'Pod::Simple::Progress' => '1.01',
	'Pod::Simple::PullParser'=> '2.02',
	'Pod::Simple::PullParserEndToken'=> undef,
	'Pod::Simple::PullParserStartToken'=> undef,
	'Pod::Simple::PullParserTextToken'=> undef,
	'Pod::Simple::PullParserToken'=> '2.02',
	'Pod::Simple::RTF'      => '2.02',
	'Pod::Simple::Search'   => '3.04',
	'Pod::Simple::SimpleTree'=> '2.02',
	'Pod::Simple::Text'     => '2.02',
	'Pod::Simple::TextContent'=> '2.02',
	'Pod::Simple::TiedOutFH'=> undef,
	'Pod::Simple::Transcode'=> undef,
	'Pod::Simple::TranscodeDumb'=> '2.02',
	'Pod::Simple::TranscodeSmart'=> undef,
	'Pod::Simple::XHTML'    => '3.04',
	'Pod::Simple::XMLOutStream'=> '2.02',
	'Pod::Text'             => '3.13',
	'Pod::Text::Color'      => '2.05',
	'Pod::Text::Overstrike' => '2.03',
	'Pod::Text::Termcap'    => '2.05',
	'Pod::Usage'            => '1.36',
	'SDBM_File'             => '1.06',
	'Safe'                  => '2.19',
	'Scalar::Util'          => '1.21',
	'Scalar::Util::PP'      => '1.21',
	'Search::Dict'          => '1.02',
	'SelectSaver'           => '1.02',
	'SelfLoader'            => '1.17',
	'Shell'                 => '0.72_01',
	'Socket'                => '1.84',
	'Storable'              => '2.20',
	'Switch'                => '2.14_01',
	'Symbol'                => '1.07',
	'Sys::Hostname'         => '1.11',
	'Sys::Syslog'           => '0.27',
	'Sys::Syslog::win32::Win32'=> undef,
	'TAP::Base'             => '3.17',
	'TAP::Formatter::Base'  => '3.17',
	'TAP::Formatter::Color' => '3.17',
	'TAP::Formatter::Console'=> '3.17',
	'TAP::Formatter::Console::ParallelSession'=> '3.17',
	'TAP::Formatter::Console::Session'=> '3.17',
	'TAP::Formatter::File'  => '3.17',
	'TAP::Formatter::File::Session'=> '3.17',
	'TAP::Formatter::Session'=> '3.17',
	'TAP::Harness'          => '3.17',
	'TAP::Object'           => '3.17',
	'TAP::Parser'           => '3.17',
	'TAP::Parser::Aggregator'=> '3.17',
	'TAP::Parser::Grammar'  => '3.17',
	'TAP::Parser::Iterator' => '3.17',
	'TAP::Parser::Iterator::Array'=> '3.17',
	'TAP::Parser::Iterator::Process'=> '3.17',
	'TAP::Parser::Iterator::Stream'=> '3.17',
	'TAP::Parser::IteratorFactory'=> '3.17',
	'TAP::Parser::Multiplexer'=> '3.17',
	'TAP::Parser::Result'   => '3.17',
	'TAP::Parser::Result::Bailout'=> '3.17',
	'TAP::Parser::Result::Comment'=> '3.17',
	'TAP::Parser::Result::Plan'=> '3.17',
	'TAP::Parser::Result::Pragma'=> '3.17',
	'TAP::Parser::Result::Test'=> '3.17',
	'TAP::Parser::Result::Unknown'=> '3.17',
	'TAP::Parser::Result::Version'=> '3.17',
	'TAP::Parser::Result::YAML'=> '3.17',
	'TAP::Parser::ResultFactory'=> '3.17',
	'TAP::Parser::Scheduler'=> '3.17',
	'TAP::Parser::Scheduler::Job'=> '3.17',
	'TAP::Parser::Scheduler::Spinner'=> '3.17',
	'TAP::Parser::Source'   => '3.17',
	'TAP::Parser::Source::Perl'=> '3.17',
	'TAP::Parser::Utils'    => '3.17',
	'TAP::Parser::YAMLish::Reader'=> '3.17',
	'TAP::Parser::YAMLish::Writer'=> '3.17',
	'Term::ANSIColor'       => '2.02',
	'Term::Cap'             => '1.12',
	'Term::Complete'        => '1.402',
	'Term::ReadLine'        => '1.05',
	'Term::UI'              => '0.20',
	'Term::UI::History'     => undef,
	'Test'                  => '1.25_02',
	'Test::Builder'         => '0.92',
	'Test::Builder::Module' => '0.92',
	'Test::Builder::Tester' => '1.18',
	'Test::Builder::Tester::Color'=> '1.18',
	'Test::Harness'         => '3.17',
	'Test::More'            => '0.92',
	'Test::Simple'          => '0.92',
	'Text::Abbrev'          => '1.01',
	'Text::Balanced'        => '2.02',
	'Text::ParseWords'      => '3.27',
	'Text::Soundex'         => '3.03_01',
	'Text::Tabs'            => '2009.0305',
	'Text::Wrap'            => '2009.0305',
	'Thread'                => '3.02',
	'Thread::Queue'         => '2.11',
	'Thread::Semaphore'     => '2.09',
	'Tie::Array'            => '1.03',
	'Tie::File'             => '0.97_02',
	'Tie::Handle'           => '4.2',
	'Tie::Hash'             => '1.03',
	'Tie::Hash::NamedCapture'=> '0.06',
	'Tie::Memoize'          => '1.1',
	'Tie::RefHash'          => '1.38',
	'Tie::Scalar'           => '1.01',
	'Tie::StdHandle'        => '4.2',
	'Tie::SubstrHash'       => '1.00',
	'Time::HiRes'           => '1.9719',
	'Time::Local'           => '1.1901_01',
	'Time::Piece'           => '1.15',
	'Time::Piece::Seconds'  => undef,
	'Time::Seconds'         => undef,
	'Time::gmtime'          => '1.03',
	'Time::localtime'       => '1.02',
	'Time::tm'              => '1.00',
	'UNIVERSAL'             => '1.05',
	'Unicode'               => '5.1.0',
	'Unicode::Collate'      => '0.52_01',
	'Unicode::Normalize'    => '1.03',
	'Unicode::UCD'          => '0.27',
	'User::grent'           => '1.01',
	'User::pwent'           => '1.00',
	'VMS::DCLsym'           => '1.03',
	'VMS::Stdio'            => '2.4',
	'Win32'                 => '0.39',
	'Win32API::File'        => '0.1101',
	'Win32API::File::ExtUtils::Myconst2perl'=> '1',
	'Win32CORE'             => '0.02',
	'XS::APItest'           => '0.15',
	'XS::Typemap'           => '0.03',
	'XSLoader'              => '0.10',
	'XSLoader::XSLoader'    => '0.10',
	'attributes'            => '0.12',
	'autodie'               => '2.06_01',
	'autodie::exception'    => '2.06_01',
	'autodie::exception::system'=> '2.06_01',
	'autodie::hints'        => '2.06_01',
	'autouse'               => '1.06',
	'base'                  => '2.14',
	'bigint'                => '0.23',
	'bignum'                => '0.23',
	'bigrat'                => '0.23',
	'blib'                  => '1.04',
	'bytes'                 => '1.03',
	'charnames'             => '1.07',
	'constant'              => '1.19',
	'deprecate'             => '0.01',
	'diagnostics'           => '1.17',
	'encoding'              => '2.6_01',
	'encoding::warnings'    => '0.11',
	'feature'               => '1.13',
	'fields'                => '2.14',
	'filetest'              => '1.02',
	'if'                    => '0.05',
	'integer'               => '1.00',
	'less'                  => '0.02',
	'lib'                   => '0.62',
	'locale'                => '1.00',
	'mro'                   => '1.01',
	'open'                  => '1.07',
	'ops'                   => '1.02',
	'overload'              => '1.08',
	'overload::numbers'     => undef,
	'overloading'           => '0.01',
	'parent'                => '0.223',
	're'                    => '0.10',
	'sigtrap'               => '1.04',
	'sort'                  => '2.01',
	'strict'                => '1.04',
	'subs'                  => '1.00',
	'threads'               => '1.74',
	'threads::shared'       => '1.31',
	'utf8'                  => '1.07',
	'vars'                  => '1.01',
	'version'               => '0.77',
	'vmsish'                => '1.02',
	'warnings'              => '1.07',
	'warnings::register'    => '1.01',
    },

    5.011001 => {
	'AnyDBM_File'           => '1.00',
	'App::Prove'            => '3.17',
	'App::Prove::State'     => '3.17',
	'App::Prove::State::Result'=> '3.17',
	'App::Prove::State::Result::Test'=> '3.17',
	'Archive::Extract'      => '0.34',
	'Archive::Tar'          => '1.54',
	'Archive::Tar::Constant'=> '0.02',
	'Archive::Tar::File'    => '0.02',
	'Attribute::Handlers'   => '0.87',
	'AutoLoader'            => '5.70',
	'AutoSplit'             => '1.06',
	'B'                     => '1.23',
	'B::Concise'            => '0.77',
	'B::Debug'              => '1.11',
	'B::Deparse'            => '0.92',
	'B::Lint'               => '1.11_01',
	'B::Lint::Debug'        => '0.01',
	'B::Showlex'            => '1.02',
	'B::Terse'              => '1.05',
	'B::Xref'               => '1.02',
	'Benchmark'             => '1.11',
	'CGI'                   => '3.48',
	'CGI::Apache'           => '1.01',
	'CGI::Carp'             => '3.45',
	'CGI::Cookie'           => '1.29',
	'CGI::Fast'             => '1.07',
	'CGI::Pretty'           => '3.46',
	'CGI::Push'             => '1.04',
	'CGI::Switch'           => '1.01',
	'CGI::Util'             => '3.48',
	'CPAN'                  => '1.94_51',
	'CPAN::Author'          => '5.5',
	'CPAN::Bundle'          => '5.5',
	'CPAN::CacheMgr'        => '5.5',
	'CPAN::Complete'        => '5.5',
	'CPAN::Debug'           => '5.5',
	'CPAN::DeferredCode'    => '5.50',
	'CPAN::Distribution'    => '1.94',
	'CPAN::Distroprefs'     => '6',
	'CPAN::Distrostatus'    => '5.5',
	'CPAN::Exception::RecursiveDependency'=> '5.5',
	'CPAN::Exception::blocked_urllist'=> '1.0',
	'CPAN::Exception::yaml_not_installed'=> '5.5',
	'CPAN::FTP'             => '5.5002',
	'CPAN::FTP::netrc'      => '1.00',
	'CPAN::FirstTime'       => '5.53',
	'CPAN::HandleConfig'    => '5.5',
	'CPAN::Index'           => '1.94',
	'CPAN::InfoObj'         => '5.5',
	'CPAN::Kwalify'         => '5.50',
	'CPAN::LWP::UserAgent'  => '1.94',
	'CPAN::Module'          => '5.5',
	'CPAN::Nox'             => '5.50',
	'CPAN::Prompt'          => '5.5',
	'CPAN::Queue'           => '5.5',
	'CPAN::Shell'           => '5.5',
	'CPAN::Tarzip'          => '5.501',
	'CPAN::URL'             => '5.5',
	'CPAN::Version'         => '5.5',
	'CPANPLUS'              => '0.89_03',
	'CPANPLUS::Backend'     => undef,
	'CPANPLUS::Backend::RV' => undef,
	'CPANPLUS::Config'      => undef,
	'CPANPLUS::Configure'   => undef,
	'CPANPLUS::Configure::Setup'=> undef,
	'CPANPLUS::Dist'        => undef,
	'CPANPLUS::Dist::Autobundle'=> undef,
	'CPANPLUS::Dist::Base'  => undef,
	'CPANPLUS::Dist::Build' => '0.40',
	'CPANPLUS::Dist::Build::Constants'=> '0.40',
	'CPANPLUS::Dist::MM'    => undef,
	'CPANPLUS::Dist::Sample'=> undef,
	'CPANPLUS::Error'       => undef,
	'CPANPLUS::Internals'   => '0.89_03',
	'CPANPLUS::Internals::Constants'=> undef,
	'CPANPLUS::Internals::Constants::Report'=> undef,
	'CPANPLUS::Internals::Extract'=> undef,
	'CPANPLUS::Internals::Fetch'=> undef,
	'CPANPLUS::Internals::Report'=> undef,
	'CPANPLUS::Internals::Search'=> undef,
	'CPANPLUS::Internals::Source'=> undef,
	'CPANPLUS::Internals::Source::Memory'=> undef,
	'CPANPLUS::Internals::Source::SQLite'=> undef,
	'CPANPLUS::Internals::Source::SQLite::Tie'=> undef,
	'CPANPLUS::Internals::Utils'=> undef,
	'CPANPLUS::Internals::Utils::Autoflush'=> undef,
	'CPANPLUS::Module'      => undef,
	'CPANPLUS::Module::Author'=> undef,
	'CPANPLUS::Module::Author::Fake'=> undef,
	'CPANPLUS::Module::Checksums'=> undef,
	'CPANPLUS::Module::Fake'=> undef,
	'CPANPLUS::Module::Signature'=> undef,
	'CPANPLUS::Selfupdate'  => undef,
	'CPANPLUS::Shell'       => undef,
	'CPANPLUS::Shell::Classic'=> '0.0562',
	'CPANPLUS::Shell::Default'=> '0.89_03',
	'CPANPLUS::Shell::Default::Plugins::CustomSource'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Remote'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Source'=> undef,
	'Carp'                  => '1.13',
	'Carp::Heavy'           => '1.13',
	'Class::ISA'            => '0.36',
	'Class::Struct'         => '0.63',
	'Compress::Raw::Bzip2'  => '2.021',
	'Compress::Raw::Zlib'   => '2.021',
	'Compress::Zlib'        => '2.021',
	'Config'                => undef,
	'Config::Extensions'    => '0.01',
	'Cwd'                   => '3.3002',
	'DB'                    => '1.02',
	'DBM_Filter'            => '0.02',
	'DBM_Filter::compress'  => '0.02',
	'DBM_Filter::encode'    => '0.02',
	'DBM_Filter::int32'     => '0.02',
	'DBM_Filter::null'      => '0.02',
	'DBM_Filter::utf8'      => '0.02',
	'DB_File'               => '1.820',
	'Data::Dumper'          => '2.125',
	'Devel::DProf'          => '20080331.00',
	'Devel::DProf::dprof::V'=> undef,
	'Devel::InnerPackage'   => '0.3',
	'Devel::PPPort'         => '3.19',
	'Devel::Peek'           => '1.04',
	'Devel::SelfStubber'    => '1.03',
	'Digest'                => '1.16',
	'Digest::MD5'           => '2.39',
	'Digest::SHA'           => '5.47',
	'Digest::base'          => '1.16',
	'Digest::file'          => '1.16',
	'DirHandle'             => '1.03',
	'Dumpvalue'             => '1.13',
	'DynaLoader'            => '1.10',
	'Encode'                => '2.37',
	'Encode::Alias'         => '2.12',
	'Encode::Byte'          => '2.03',
	'Encode::CJKConstants'  => '2.02',
	'Encode::CN'            => '2.02',
	'Encode::CN::HZ'        => '2.05',
	'Encode::Config'        => '2.05',
	'Encode::EBCDIC'        => '2.02',
	'Encode::Encoder'       => '2.01',
	'Encode::Encoding'      => '2.05',
	'Encode::GSM0338'       => '2.01',
	'Encode::Guess'         => '2.03',
	'Encode::JP'            => '2.03',
	'Encode::JP::H2Z'       => '2.02',
	'Encode::JP::JIS7'      => '2.04',
	'Encode::KR'            => '2.02',
	'Encode::KR::2022_KR'   => '2.02',
	'Encode::MIME::Header'  => '2.11',
	'Encode::MIME::Header::ISO_2022_JP'=> '1.03',
	'Encode::MIME::Name'    => '1.01',
	'Encode::Symbol'        => '2.02',
	'Encode::TW'            => '2.02',
	'Encode::Unicode'       => '2.06',
	'Encode::Unicode::UTF7' => '2.04',
	'English'               => '1.04',
	'Env'                   => '1.00',
	'Errno'                 => '1.11',
	'Exporter'              => '5.64',
	'Exporter::Heavy'       => '5.64',
	'ExtUtils::CBuilder'    => '0.260301',
	'ExtUtils::CBuilder::Base'=> '0.260301',
	'ExtUtils::CBuilder::Platform::Unix'=> '0.260301',
	'ExtUtils::CBuilder::Platform::VMS'=> '0.260301',
	'ExtUtils::CBuilder::Platform::Windows'=> '0.260301',
	'ExtUtils::CBuilder::Platform::aix'=> '0.260301',
	'ExtUtils::CBuilder::Platform::cygwin'=> '0.260301',
	'ExtUtils::CBuilder::Platform::darwin'=> '0.260301',
	'ExtUtils::CBuilder::Platform::dec_osf'=> '0.260301',
	'ExtUtils::CBuilder::Platform::os2'=> '0.260301',
	'ExtUtils::Command'     => '1.16',
	'ExtUtils::Command::MM' => '6.55_02',
	'ExtUtils::Constant'    => '0.22',
	'ExtUtils::Constant::Base'=> '0.04',
	'ExtUtils::Constant::ProxySubs'=> '0.06',
	'ExtUtils::Constant::Utils'=> '0.02',
	'ExtUtils::Constant::XS'=> '0.03',
	'ExtUtils::Embed'       => '1.28',
	'ExtUtils::Install'     => '1.55',
	'ExtUtils::Installed'   => '1.999_001',
	'ExtUtils::Liblist'     => '6.55_02',
	'ExtUtils::Liblist::Kid'=> '6.5502',
	'ExtUtils::MM'          => '6.55_02',
	'ExtUtils::MM_AIX'      => '6.55_02',
	'ExtUtils::MM_Any'      => '6.55_02',
	'ExtUtils::MM_BeOS'     => '6.55_02',
	'ExtUtils::MM_Cygwin'   => '6.55_02',
	'ExtUtils::MM_DOS'      => '6.5502',
	'ExtUtils::MM_Darwin'   => '6.55_02',
	'ExtUtils::MM_MacOS'    => '6.5502',
	'ExtUtils::MM_NW5'      => '6.55_02',
	'ExtUtils::MM_OS2'      => '6.55_02',
	'ExtUtils::MM_QNX'      => '6.55_02',
	'ExtUtils::MM_UWIN'     => '6.5502',
	'ExtUtils::MM_Unix'     => '6.55_02',
	'ExtUtils::MM_VMS'      => '6.55_02',
	'ExtUtils::MM_VOS'      => '6.55_02',
	'ExtUtils::MM_Win32'    => '6.55_02',
	'ExtUtils::MM_Win95'    => '6.55_02',
	'ExtUtils::MY'          => '6.5502',
	'ExtUtils::MakeMaker'   => '6.55_02',
	'ExtUtils::MakeMaker::Config'=> '6.55_02',
	'ExtUtils::Manifest'    => '1.57',
	'ExtUtils::Miniperl'    => undef,
	'ExtUtils::Mkbootstrap' => '6.55_02',
	'ExtUtils::Mksymlists'  => '6.55_02',
	'ExtUtils::Packlist'    => '1.44',
	'ExtUtils::ParseXS'     => '2.21',
	'ExtUtils::XSSymSet'    => '1.1',
	'ExtUtils::testlib'     => '6.5502',
	'Fatal'                 => '2.06_01',
	'Fcntl'                 => '1.06',
	'File::Basename'        => '2.78',
	'File::CheckTree'       => '4.4',
	'File::Compare'         => '1.1006',
	'File::Copy'            => '2.16',
	'File::DosGlob'         => '1.01',
	'File::Fetch'           => '0.20',
	'File::Find'            => '1.14',
	'File::Glob'            => '1.07',
	'File::GlobMapper'      => '1.000',
	'File::Path'            => '2.08',
	'File::Spec'            => '3.30',
	'File::Spec::Cygwin'    => '3.30',
	'File::Spec::Epoc'      => '3.30',
	'File::Spec::Functions' => '3.30',
	'File::Spec::Mac'       => '3.30',
	'File::Spec::OS2'       => '3.30',
	'File::Spec::Unix'      => '3.30',
	'File::Spec::VMS'       => '3.30',
	'File::Spec::Win32'     => '3.30',
	'File::Temp'            => '0.22',
	'File::stat'            => '1.02',
	'FileCache'             => '1.08',
	'FileHandle'            => '2.02',
	'VMS::Filespec'         => '1.12',
	'Filter::Simple'        => '0.84',
	'Filter::Util::Call'    => '1.08',
	'FindBin'               => '1.50',
	'GDBM_File'             => '1.09',
	'Getopt::Long'          => '2.38',
	'Getopt::Std'           => '1.06',
	'Hash::Util'            => '0.07',
	'Hash::Util::FieldHash' => '1.04',
	'I18N::Collate'         => '1.01',
	'I18N::LangTags'        => '0.35',
	'I18N::LangTags::Detect'=> '1.03',
	'I18N::LangTags::List'  => '0.35',
	'I18N::Langinfo'        => '0.02',
	'IO'                    => '1.25_02',
	'IO::Compress::Adapter::Bzip2'=> '2.021',
	'IO::Compress::Adapter::Deflate'=> '2.021',
	'IO::Compress::Adapter::Identity'=> '2.021',
	'IO::Compress::Base'    => '2.021',
	'IO::Compress::Base::Common'=> '2.021',
	'IO::Compress::Bzip2'   => '2.021',
	'IO::Compress::Deflate' => '2.021',
	'IO::Compress::Gzip'    => '2.021',
	'IO::Compress::Gzip::Constants'=> '2.021',
	'IO::Compress::RawDeflate'=> '2.021',
	'IO::Compress::Zip'     => '2.021',
	'IO::Compress::Zip::Constants'=> '2.021',
	'IO::Compress::Zlib::Constants'=> '2.021',
	'IO::Compress::Zlib::Extra'=> '2.021',
	'IO::Dir'               => '1.07',
	'IO::File'              => '1.14',
	'IO::Handle'            => '1.28',
	'IO::Pipe'              => '1.13',
	'IO::Poll'              => '0.07',
	'IO::Seekable'          => '1.10',
	'IO::Select'            => '1.17',
	'IO::Socket'            => '1.31',
	'IO::Socket::INET'      => '1.31',
	'IO::Socket::UNIX'      => '1.23',
	'IO::Uncompress::Adapter::Bunzip2'=> '2.021',
	'IO::Uncompress::Adapter::Identity'=> '2.021',
	'IO::Uncompress::Adapter::Inflate'=> '2.021',
	'IO::Uncompress::AnyInflate'=> '2.021',
	'IO::Uncompress::AnyUncompress'=> '2.021',
	'IO::Uncompress::Base'  => '2.021',
	'IO::Uncompress::Bunzip2'=> '2.021',
	'IO::Uncompress::Gunzip'=> '2.021',
	'IO::Uncompress::Inflate'=> '2.021',
	'IO::Uncompress::RawInflate'=> '2.021',
	'IO::Uncompress::Unzip' => '2.021',
	'IO::Zlib'              => '1.10',
	'IPC::Cmd'              => '0.50',
	'IPC::Msg'              => '2.01',
	'IPC::Open2'            => '1.03',
	'IPC::Open3'            => '1.05',
	'IPC::Semaphore'        => '2.01',
	'IPC::SharedMem'        => '2.01',
	'IPC::SysV'             => '2.01',
	'List::Util'            => '1.21',
	'List::Util::PP'        => '1.21',
	'List::Util::XS'        => '1.21',
	'Locale::Constants'     => '2.07',
	'Locale::Country'       => '2.07',
	'Locale::Currency'      => '2.07',
	'Locale::Language'      => '2.07',
	'Locale::Maketext'      => '1.13',
	'Locale::Maketext::Guts'=> '1.13',
	'Locale::Maketext::GutsLoader'=> '1.13',
	'Locale::Maketext::Simple'=> '0.21',
	'Locale::Script'        => '2.07',
	'Log::Message'          => '0.02',
	'Log::Message::Config'  => '0.01',
	'Log::Message::Handlers'=> undef,
	'Log::Message::Item'    => undef,
	'Log::Message::Simple'  => '0.06',
	'MIME::Base64'          => '3.08',
	'MIME::QuotedPrint'     => '3.08',
	'Math::BigFloat'        => '1.60',
	'Math::BigFloat::Trace' => '0.01',
	'Math::BigInt'          => '1.89_01',
	'Math::BigInt::Calc'    => '0.52',
	'Math::BigInt::CalcEmu' => '0.05',
	'Math::BigInt::FastCalc'=> '0.19',
	'Math::BigInt::Trace'   => '0.01',
	'Math::BigRat'          => '0.24',
	'Math::Complex'         => '1.56',
	'Math::Trig'            => '1.2',
	'Memoize'               => '1.01_03',
	'Memoize::AnyDBM_File'  => '0.65',
	'Memoize::Expire'       => '1.00',
	'Memoize::ExpireFile'   => '1.01',
	'Memoize::ExpireTest'   => '0.65',
	'Memoize::NDBM_File'    => '0.65',
	'Memoize::SDBM_File'    => '0.65',
	'Memoize::Storable'     => '0.65',
	'Module::Build'         => '0.35',
	'Module::Build::Base'   => '0.35',
	'Module::Build::Compat' => '0.35',
	'Module::Build::Config' => '0.35',
	'Module::Build::ConfigData'=> undef,
	'Module::Build::Cookbook'=> '0.35',
	'Module::Build::Dumper' => '0.35',
	'Module::Build::ModuleInfo'=> '0.35',
	'Module::Build::Notes'  => '0.35',
	'Module::Build::PPMMaker'=> '0.35',
	'Module::Build::Platform::Amiga'=> '0.35',
	'Module::Build::Platform::Default'=> '0.35',
	'Module::Build::Platform::EBCDIC'=> '0.35',
	'Module::Build::Platform::MPEiX'=> '0.35',
	'Module::Build::Platform::MacOS'=> '0.35',
	'Module::Build::Platform::RiscOS'=> '0.35',
	'Module::Build::Platform::Unix'=> '0.35',
	'Module::Build::Platform::VMS'=> '0.35',
	'Module::Build::Platform::VOS'=> '0.35',
	'Module::Build::Platform::Windows'=> '0.35',
	'Module::Build::Platform::aix'=> '0.35',
	'Module::Build::Platform::cygwin'=> '0.35',
	'Module::Build::Platform::darwin'=> '0.35',
	'Module::Build::Platform::os2'=> '0.35',
	'Module::Build::PodParser'=> '0.35',
	'Module::Build::Version'=> '0.77',
	'Module::Build::YAML'   => '0.50',
	'Module::CoreList'      => '2.21',
	'Module::Load'          => '0.16',
	'Module::Load::Conditional'=> '0.30',
	'Module::Loaded'        => '0.06',
	'Module::Pluggable'     => '3.9',
	'Module::Pluggable::Object'=> '3.9',
	'Moped::Msg'            => '0.01',
	'NDBM_File'             => '1.08',
	'NEXT'                  => '0.64',
	'Net::Cmd'              => '2.29',
	'Net::Config'           => '1.11',
	'Net::Domain'           => '2.20',
	'Net::FTP'              => '2.77',
	'Net::FTP::A'           => '1.18',
	'Net::FTP::E'           => '0.01',
	'Net::FTP::I'           => '1.12',
	'Net::FTP::L'           => '0.01',
	'Net::FTP::dataconn'    => '0.11',
	'Net::NNTP'             => '2.24',
	'Net::Netrc'            => '2.12',
	'Net::POP3'             => '2.29',
	'Net::Ping'             => '2.36',
	'Net::SMTP'             => '2.31',
	'Net::Time'             => '2.10',
	'Net::hostent'          => '1.01',
	'Net::netent'           => '1.00',
	'Net::protoent'         => '1.00',
	'Net::servent'          => '1.01',
	'O'                     => '1.01',
	'ODBM_File'             => '1.07',
	'Object::Accessor'      => '0.36',
	'Opcode'                => '1.15',
	'POSIX'                 => '1.18',
	'Package::Constants'    => '0.02',
	'Params::Check'         => '0.26',
	'Parse::CPAN::Meta'     => '1.40',
	'PerlIO'                => '1.06',
	'PerlIO::encoding'      => '0.11',
	'PerlIO::scalar'        => '0.07',
	'PerlIO::via'           => '0.09',
	'PerlIO::via::QuotedPrint'=> '0.06',
	'Pod::Checker'          => '1.45',
	'Pod::Escapes'          => '1.04',
	'Pod::Find'             => '1.35',
	'Pod::Functions'        => '1.03',
	'Pod::Html'             => '1.09',
	'Pod::InputObjects'     => '1.31',
	'Pod::LaTeX'            => '0.58',
	'Pod::Man'              => '2.22',
	'Pod::ParseLink'        => '1.09',
	'Pod::ParseUtils'       => '1.36',
	'Pod::Parser'           => '1.37',
	'Pod::Perldoc'          => '3.15_01',
	'Pod::Perldoc::BaseTo'  => undef,
	'Pod::Perldoc::GetOptsOO'=> undef,
	'Pod::Perldoc::ToChecker'=> undef,
	'Pod::Perldoc::ToMan'   => undef,
	'Pod::Perldoc::ToNroff' => undef,
	'Pod::Perldoc::ToPod'   => undef,
	'Pod::Perldoc::ToRtf'   => undef,
	'Pod::Perldoc::ToText'  => undef,
	'Pod::Perldoc::ToTk'    => undef,
	'Pod::Perldoc::ToXml'   => undef,
	'Pod::PlainText'        => '2.04',
	'Pod::Plainer'          => '1.01',
	'Pod::Select'           => '1.36',
	'Pod::Simple'           => '3.08',
	'Pod::Simple::BlackBox' => undef,
	'Pod::Simple::Checker'  => '2.02',
	'Pod::Simple::Debug'    => undef,
	'Pod::Simple::DumpAsText'=> '2.02',
	'Pod::Simple::DumpAsXML'=> '2.02',
	'Pod::Simple::HTML'     => '3.03',
	'Pod::Simple::HTMLBatch'=> '3.02',
	'Pod::Simple::HTMLLegacy'=> '5.01',
	'Pod::Simple::LinkSection'=> undef,
	'Pod::Simple::Methody'  => '2.02',
	'Pod::Simple::Progress' => '1.01',
	'Pod::Simple::PullParser'=> '2.02',
	'Pod::Simple::PullParserEndToken'=> undef,
	'Pod::Simple::PullParserStartToken'=> undef,
	'Pod::Simple::PullParserTextToken'=> undef,
	'Pod::Simple::PullParserToken'=> '2.02',
	'Pod::Simple::RTF'      => '2.02',
	'Pod::Simple::Search'   => '3.04',
	'Pod::Simple::SimpleTree'=> '2.02',
	'Pod::Simple::Text'     => '2.02',
	'Pod::Simple::TextContent'=> '2.02',
	'Pod::Simple::TiedOutFH'=> undef,
	'Pod::Simple::Transcode'=> undef,
	'Pod::Simple::TranscodeDumb'=> '2.02',
	'Pod::Simple::TranscodeSmart'=> undef,
	'Pod::Simple::XHTML'    => '3.04',
	'Pod::Simple::XMLOutStream'=> '2.02',
	'Pod::Text'             => '3.13',
	'Pod::Text::Color'      => '2.05',
	'Pod::Text::Overstrike' => '2.03',
	'Pod::Text::Termcap'    => '2.05',
	'Pod::Usage'            => '1.36',
	'SDBM_File'             => '1.06',
	'Safe'                  => '2.19',
	'Scalar::Util'          => '1.21',
	'Scalar::Util::PP'      => '1.21',
	'Search::Dict'          => '1.02',
	'SelectSaver'           => '1.02',
	'SelfLoader'            => '1.17',
	'Shell'                 => '0.72_01',
	'Socket'                => '1.85',
	'Storable'              => '2.22',
	'Switch'                => '2.15',
	'Symbol'                => '1.07',
	'Sys::Hostname'         => '1.11',
	'Sys::Syslog'           => '0.27',
	'Sys::Syslog::win32::Win32'=> undef,
	'TAP::Base'             => '3.17',
	'TAP::Formatter::Base'  => '3.17',
	'TAP::Formatter::Color' => '3.17',
	'TAP::Formatter::Console'=> '3.17',
	'TAP::Formatter::Console::ParallelSession'=> '3.17',
	'TAP::Formatter::Console::Session'=> '3.17',
	'TAP::Formatter::File'  => '3.17',
	'TAP::Formatter::File::Session'=> '3.17',
	'TAP::Formatter::Session'=> '3.17',
	'TAP::Harness'          => '3.17',
	'TAP::Object'           => '3.17',
	'TAP::Parser'           => '3.17',
	'TAP::Parser::Aggregator'=> '3.17',
	'TAP::Parser::Grammar'  => '3.17',
	'TAP::Parser::Iterator' => '3.17',
	'TAP::Parser::Iterator::Array'=> '3.17',
	'TAP::Parser::Iterator::Process'=> '3.17',
	'TAP::Parser::Iterator::Stream'=> '3.17',
	'TAP::Parser::IteratorFactory'=> '3.17',
	'TAP::Parser::Multiplexer'=> '3.17',
	'TAP::Parser::Result'   => '3.17',
	'TAP::Parser::Result::Bailout'=> '3.17',
	'TAP::Parser::Result::Comment'=> '3.17',
	'TAP::Parser::Result::Plan'=> '3.17',
	'TAP::Parser::Result::Pragma'=> '3.17',
	'TAP::Parser::Result::Test'=> '3.17',
	'TAP::Parser::Result::Unknown'=> '3.17',
	'TAP::Parser::Result::Version'=> '3.17',
	'TAP::Parser::Result::YAML'=> '3.17',
	'TAP::Parser::ResultFactory'=> '3.17',
	'TAP::Parser::Scheduler'=> '3.17',
	'TAP::Parser::Scheduler::Job'=> '3.17',
	'TAP::Parser::Scheduler::Spinner'=> '3.17',
	'TAP::Parser::Source'   => '3.17',
	'TAP::Parser::Source::Perl'=> '3.17',
	'TAP::Parser::Utils'    => '3.17',
	'TAP::Parser::YAMLish::Reader'=> '3.17',
	'TAP::Parser::YAMLish::Writer'=> '3.17',
	'Term::ANSIColor'       => '2.02',
	'Term::Cap'             => '1.12',
	'Term::Complete'        => '1.402',
	'Term::ReadLine'        => '1.05',
	'Term::UI'              => '0.20',
	'Term::UI::History'     => undef,
	'Test'                  => '1.25_02',
	'Test::Builder'         => '0.94',
	'Test::Builder::Module' => '0.94',
	'Test::Builder::Tester' => '1.18',
	'Test::Builder::Tester::Color'=> '1.18',
	'Test::Harness'         => '3.17',
	'Test::More'            => '0.94',
	'Test::Simple'          => '0.94',
	'Text::Abbrev'          => '1.01',
	'Text::Balanced'        => '2.02',
	'Text::ParseWords'      => '3.27',
	'Text::Soundex'         => '3.03_01',
	'Text::Tabs'            => '2009.0305',
	'Text::Wrap'            => '2009.0305',
	'Thread'                => '3.02',
	'Thread::Queue'         => '2.11',
	'Thread::Semaphore'     => '2.09',
	'Tie::Array'            => '1.03',
	'Tie::File'             => '0.97_02',
	'Tie::Handle'           => '4.2',
	'Tie::Hash'             => '1.03',
	'Tie::Hash::NamedCapture'=> '0.06',
	'Tie::Memoize'          => '1.1',
	'Tie::RefHash'          => '1.38',
	'Tie::Scalar'           => '1.01',
	'Tie::StdHandle'        => '4.2',
	'Tie::SubstrHash'       => '1.00',
	'Time::HiRes'           => '1.9719',
	'Time::Local'           => '1.1901_01',
	'Time::Piece'           => '1.15',
	'Time::Piece::Seconds'  => undef,
	'Time::Seconds'         => undef,
	'Time::gmtime'          => '1.03',
	'Time::localtime'       => '1.02',
	'Time::tm'              => '1.00',
	'UNIVERSAL'             => '1.05',
	'Unicode'               => '5.1.0',
	'Unicode::Collate'      => '0.52_01',
	'Unicode::Normalize'    => '1.03',
	'Unicode::UCD'          => '0.27',
	'User::grent'           => '1.01',
	'User::pwent'           => '1.00',
	'VMS::DCLsym'           => '1.03',
	'VMS::Stdio'            => '2.4',
	'Win32'                 => '0.39',
	'Win32API::File'        => '0.1101',
	'Win32API::File::ExtUtils::Myconst2perl'=> '1',
	'Win32CORE'             => '0.02',
	'XS::APItest'           => '0.16',
	'XS::Typemap'           => '0.03',
	'XSLoader'              => '0.10',
	'XSLoader::XSLoader'    => '0.10',
	'attributes'            => '0.12',
	'autodie'               => '2.06_01',
	'autodie::exception'    => '2.06_01',
	'autodie::exception::system'=> '2.06_01',
	'autodie::hints'        => '2.06_01',
	'autouse'               => '1.06',
	'base'                  => '2.14',
	'bigint'                => '0.23',
	'bignum'                => '0.23',
	'bigrat'                => '0.23',
	'blib'                  => '1.04',
	'bytes'                 => '1.03',
	'charnames'             => '1.07',
	'constant'              => '1.19',
	'deprecate'             => '0.01',
	'diagnostics'           => '1.17',
	'encoding'              => '2.6_01',
	'encoding::warnings'    => '0.11',
	'feature'               => '1.13',
	'fields'                => '2.14',
	'filetest'              => '1.02',
	'if'                    => '0.05',
	'integer'               => '1.00',
	'less'                  => '0.02',
	'lib'                   => '0.62',
	'locale'                => '1.00',
	'mro'                   => '1.02',
	'open'                  => '1.07',
	'ops'                   => '1.02',
	'overload'              => '1.09',
	'overload::numbers'     => undef,
	'overloading'           => '0.01',
	'parent'                => '0.223',
	're'                    => '0.10',
	'sigtrap'               => '1.04',
	'sort'                  => '2.01',
	'strict'                => '1.04',
	'subs'                  => '1.00',
	'threads'               => '1.74',
	'threads::shared'       => '1.32',
	'utf8'                  => '1.07',
	'vars'                  => '1.01',
	'version'               => '0.77',
	'vmsish'                => '1.02',
	'warnings'              => '1.07',
	'warnings::register'    => '1.01',
    },

    5.011002 => {
	'AnyDBM_File'           => '1.00',
	'App::Prove'            => '3.17',
	'App::Prove::State'     => '3.17',
	'App::Prove::State::Result'=> '3.17',
	'App::Prove::State::Result::Test'=> '3.17',
	'Archive::Extract'      => '0.34',
	'Archive::Tar'          => '1.54',
	'Archive::Tar::Constant'=> '0.02',
	'Archive::Tar::File'    => '0.02',
	'Attribute::Handlers'   => '0.87',
	'AutoLoader'            => '5.70',
	'AutoSplit'             => '1.06',
	'B'                     => '1.23',
	'B::Concise'            => '0.78',
	'B::Debug'              => '1.11',
	'B::Deparse'            => '0.93',
	'B::Lint'               => '1.11_01',
	'B::Lint::Debug'        => '0.01',
	'B::Showlex'            => '1.02',
	'B::Terse'              => '1.05',
	'B::Xref'               => '1.02',
	'Benchmark'             => '1.11',
	'CGI'                   => '3.48',
	'CGI::Apache'           => '1.01',
	'CGI::Carp'             => '3.45',
	'CGI::Cookie'           => '1.29',
	'CGI::Fast'             => '1.07',
	'CGI::Pretty'           => '3.46',
	'CGI::Push'             => '1.04',
	'CGI::Switch'           => '1.01',
	'CGI::Util'             => '3.48',
	'CPAN'                  => '1.94_51',
	'CPAN::Author'          => '5.5',
	'CPAN::Bundle'          => '5.5',
	'CPAN::CacheMgr'        => '5.5',
	'CPAN::Complete'        => '5.5',
	'CPAN::Debug'           => '5.5',
	'CPAN::DeferredCode'    => '5.50',
	'CPAN::Distribution'    => '1.94',
	'CPAN::Distroprefs'     => '6',
	'CPAN::Distrostatus'    => '5.5',
	'CPAN::Exception::RecursiveDependency'=> '5.5',
	'CPAN::Exception::blocked_urllist'=> '1.0',
	'CPAN::Exception::yaml_not_installed'=> '5.5',
	'CPAN::FTP'             => '5.5002',
	'CPAN::FTP::netrc'      => '1.00',
	'CPAN::FirstTime'       => '5.53',
	'CPAN::HandleConfig'    => '5.5',
	'CPAN::Index'           => '1.94',
	'CPAN::InfoObj'         => '5.5',
	'CPAN::Kwalify'         => '5.50',
	'CPAN::LWP::UserAgent'  => '1.94',
	'CPAN::Module'          => '5.5',
	'CPAN::Nox'             => '5.50',
	'CPAN::Prompt'          => '5.5',
	'CPAN::Queue'           => '5.5',
	'CPAN::Shell'           => '5.5',
	'CPAN::Tarzip'          => '5.501',
	'CPAN::URL'             => '5.5',
	'CPAN::Version'         => '5.5',
	'CPANPLUS'              => '0.89_09',
	'CPANPLUS::Backend'     => undef,
	'CPANPLUS::Backend::RV' => undef,
	'CPANPLUS::Config'      => undef,
	'CPANPLUS::Configure'   => undef,
	'CPANPLUS::Configure::Setup'=> undef,
	'CPANPLUS::Dist'        => undef,
	'CPANPLUS::Dist::Autobundle'=> undef,
	'CPANPLUS::Dist::Base'  => undef,
	'CPANPLUS::Dist::Build' => '0.44',
	'CPANPLUS::Dist::Build::Constants'=> '0.44',
	'CPANPLUS::Dist::MM'    => undef,
	'CPANPLUS::Dist::Sample'=> undef,
	'CPANPLUS::Error'       => undef,
	'CPANPLUS::Internals'   => '0.89_09',
	'CPANPLUS::Internals::Constants'=> undef,
	'CPANPLUS::Internals::Constants::Report'=> undef,
	'CPANPLUS::Internals::Extract'=> undef,
	'CPANPLUS::Internals::Fetch'=> undef,
	'CPANPLUS::Internals::Report'=> undef,
	'CPANPLUS::Internals::Search'=> undef,
	'CPANPLUS::Internals::Source'=> undef,
	'CPANPLUS::Internals::Source::Memory'=> undef,
	'CPANPLUS::Internals::Source::SQLite'=> undef,
	'CPANPLUS::Internals::Source::SQLite::Tie'=> undef,
	'CPANPLUS::Internals::Utils'=> undef,
	'CPANPLUS::Internals::Utils::Autoflush'=> undef,
	'CPANPLUS::Module'      => undef,
	'CPANPLUS::Module::Author'=> undef,
	'CPANPLUS::Module::Author::Fake'=> undef,
	'CPANPLUS::Module::Checksums'=> undef,
	'CPANPLUS::Module::Fake'=> undef,
	'CPANPLUS::Module::Signature'=> undef,
	'CPANPLUS::Selfupdate'  => undef,
	'CPANPLUS::Shell'       => undef,
	'CPANPLUS::Shell::Classic'=> '0.0562',
	'CPANPLUS::Shell::Default'=> '0.89_09',
	'CPANPLUS::Shell::Default::Plugins::CustomSource'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Remote'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Source'=> undef,
	'Carp'                  => '1.14',
	'Carp::Heavy'           => '1.14',
	'Class::ISA'            => '0.36',
	'Class::Struct'         => '0.63',
	'Compress::Raw::Bzip2'  => '2.021',
	'Compress::Raw::Zlib'   => '2.021',
	'Compress::Zlib'        => '2.022',
	'Config'                => undef,
	'Config::Extensions'    => '0.01',
	'Cwd'                   => '3.3002',
	'DB'                    => '1.02',
	'DBM_Filter'            => '0.03',
	'DBM_Filter::compress'  => '0.02',
	'DBM_Filter::encode'    => '0.02',
	'DBM_Filter::int32'     => '0.02',
	'DBM_Filter::null'      => '0.02',
	'DBM_Filter::utf8'      => '0.02',
	'DB_File'               => '1.820',
	'Data::Dumper'          => '2.125',
	'Devel::DProf'          => '20080331.00',
	'Devel::DProf::dprof::V'=> undef,
	'Devel::InnerPackage'   => '0.3',
	'Devel::PPPort'         => '3.19',
	'Devel::Peek'           => '1.04',
	'Devel::SelfStubber'    => '1.03',
	'Digest'                => '1.16',
	'Digest::MD5'           => '2.39',
	'Digest::SHA'           => '5.47',
	'Digest::base'          => '1.16',
	'Digest::file'          => '1.16',
	'DirHandle'             => '1.03',
	'Dumpvalue'             => '1.13',
	'DynaLoader'            => '1.10',
	'Encode'                => '2.38',
	'Encode::Alias'         => '2.12',
	'Encode::Byte'          => '2.04',
	'Encode::CJKConstants'  => '2.02',
	'Encode::CN'            => '2.03',
	'Encode::CN::HZ'        => '2.05',
	'Encode::Config'        => '2.05',
	'Encode::EBCDIC'        => '2.02',
	'Encode::Encoder'       => '2.01',
	'Encode::Encoding'      => '2.05',
	'Encode::GSM0338'       => '2.01',
	'Encode::Guess'         => '2.03',
	'Encode::JP'            => '2.04',
	'Encode::JP::H2Z'       => '2.02',
	'Encode::JP::JIS7'      => '2.04',
	'Encode::KR'            => '2.03',
	'Encode::KR::2022_KR'   => '2.02',
	'Encode::MIME::Header'  => '2.11',
	'Encode::MIME::Header::ISO_2022_JP'=> '1.03',
	'Encode::MIME::Name'    => '1.01',
	'Encode::Symbol'        => '2.02',
	'Encode::TW'            => '2.03',
	'Encode::Unicode'       => '2.07',
	'Encode::Unicode::UTF7' => '2.04',
	'English'               => '1.04',
	'Env'                   => '1.01',
	'Errno'                 => '1.11',
	'Exporter'              => '5.64_01',
	'Exporter::Heavy'       => '5.64_01',
	'ExtUtils::CBuilder'    => '0.27',
	'ExtUtils::CBuilder::Base'=> '0.27',
	'ExtUtils::CBuilder::Platform::Unix'=> '0.27',
	'ExtUtils::CBuilder::Platform::VMS'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.27',
	'ExtUtils::CBuilder::Platform::aix'=> '0.27',
	'ExtUtils::CBuilder::Platform::cygwin'=> '0.27',
	'ExtUtils::CBuilder::Platform::darwin'=> '0.27',
	'ExtUtils::CBuilder::Platform::dec_osf'=> '0.27',
	'ExtUtils::CBuilder::Platform::os2'=> '0.27',
	'ExtUtils::Command'     => '1.16',
	'ExtUtils::Command::MM' => '6.55_02',
	'ExtUtils::Constant'    => '0.22',
	'ExtUtils::Constant::Base'=> '0.04',
	'ExtUtils::Constant::ProxySubs'=> '0.06',
	'ExtUtils::Constant::Utils'=> '0.02',
	'ExtUtils::Constant::XS'=> '0.03',
	'ExtUtils::Embed'       => '1.28',
	'ExtUtils::Install'     => '1.55',
	'ExtUtils::Installed'   => '1.999_001',
	'ExtUtils::Liblist'     => '6.55_02',
	'ExtUtils::Liblist::Kid'=> '6.5502',
	'ExtUtils::MM'          => '6.55_02',
	'ExtUtils::MM_AIX'      => '6.55_02',
	'ExtUtils::MM_Any'      => '6.55_02',
	'ExtUtils::MM_BeOS'     => '6.55_02',
	'ExtUtils::MM_Cygwin'   => '6.55_02',
	'ExtUtils::MM_DOS'      => '6.5502',
	'ExtUtils::MM_Darwin'   => '6.55_02',
	'ExtUtils::MM_MacOS'    => '6.5502',
	'ExtUtils::MM_NW5'      => '6.55_02',
	'ExtUtils::MM_OS2'      => '6.55_02',
	'ExtUtils::MM_QNX'      => '6.55_02',
	'ExtUtils::MM_UWIN'     => '6.5502',
	'ExtUtils::MM_Unix'     => '6.55_02',
	'ExtUtils::MM_VMS'      => '6.55_02',
	'ExtUtils::MM_VOS'      => '6.55_02',
	'ExtUtils::MM_Win32'    => '6.55_02',
	'ExtUtils::MM_Win95'    => '6.55_02',
	'ExtUtils::MY'          => '6.5502',
	'ExtUtils::MakeMaker'   => '6.55_02',
	'ExtUtils::MakeMaker::Config'=> '6.55_02',
	'ExtUtils::Manifest'    => '1.57',
	'ExtUtils::Miniperl'    => undef,
	'ExtUtils::Mkbootstrap' => '6.55_02',
	'ExtUtils::Mksymlists'  => '6.55_02',
	'ExtUtils::Packlist'    => '1.44',
	'ExtUtils::ParseXS'     => '2.21',
	'ExtUtils::XSSymSet'    => '1.1',
	'ExtUtils::testlib'     => '6.5502',
	'Fatal'                 => '2.06_01',
	'Fcntl'                 => '1.06',
	'File::Basename'        => '2.78',
	'File::CheckTree'       => '4.4',
	'File::Compare'         => '1.1006',
	'File::Copy'            => '2.16',
	'File::DosGlob'         => '1.01',
	'File::Fetch'           => '0.22',
	'File::Find'            => '1.14',
	'File::Glob'            => '1.07',
	'File::GlobMapper'      => '1.000',
	'File::Path'            => '2.08',
	'File::Spec'            => '3.30',
	'File::Spec::Cygwin'    => '3.30',
	'File::Spec::Epoc'      => '3.30',
	'File::Spec::Functions' => '3.30',
	'File::Spec::Mac'       => '3.30',
	'File::Spec::OS2'       => '3.30',
	'File::Spec::Unix'      => '3.30',
	'File::Spec::VMS'       => '3.30',
	'File::Spec::Win32'     => '3.30',
	'File::Temp'            => '0.22',
	'File::stat'            => '1.02',
	'FileCache'             => '1.08',
	'FileHandle'            => '2.02',
	'VMS::Filespec'         => '1.12',
	'Filter::Simple'        => '0.84',
	'Filter::Util::Call'    => '1.08',
	'FindBin'               => '1.50',
	'GDBM_File'             => '1.09',
	'Getopt::Long'          => '2.38',
	'Getopt::Std'           => '1.06',
	'Hash::Util'            => '0.07',
	'Hash::Util::FieldHash' => '1.04',
	'I18N::Collate'         => '1.01',
	'I18N::LangTags'        => '0.35',
	'I18N::LangTags::Detect'=> '1.04',
	'I18N::LangTags::List'  => '0.35',
	'I18N::Langinfo'        => '0.03',
	'IO'                    => '1.25_02',
	'IO::Compress::Adapter::Bzip2'=> '2.022',
	'IO::Compress::Adapter::Deflate'=> '2.022',
	'IO::Compress::Adapter::Identity'=> '2.022',
	'IO::Compress::Base'    => '2.022',
	'IO::Compress::Base::Common'=> '2.022',
	'IO::Compress::Bzip2'   => '2.022',
	'IO::Compress::Deflate' => '2.022',
	'IO::Compress::Gzip'    => '2.022',
	'IO::Compress::Gzip::Constants'=> '2.022',
	'IO::Compress::RawDeflate'=> '2.022',
	'IO::Compress::Zip'     => '2.022',
	'IO::Compress::Zip::Constants'=> '2.022',
	'IO::Compress::Zlib::Constants'=> '2.022',
	'IO::Compress::Zlib::Extra'=> '2.022',
	'IO::Dir'               => '1.07',
	'IO::File'              => '1.14',
	'IO::Handle'            => '1.28',
	'IO::Pipe'              => '1.13',
	'IO::Poll'              => '0.07',
	'IO::Seekable'          => '1.10',
	'IO::Select'            => '1.17',
	'IO::Socket'            => '1.31',
	'IO::Socket::INET'      => '1.31',
	'IO::Socket::UNIX'      => '1.23',
	'IO::Uncompress::Adapter::Bunzip2'=> '2.022',
	'IO::Uncompress::Adapter::Identity'=> '2.022',
	'IO::Uncompress::Adapter::Inflate'=> '2.022',
	'IO::Uncompress::AnyInflate'=> '2.022',
	'IO::Uncompress::AnyUncompress'=> '2.022',
	'IO::Uncompress::Base'  => '2.022',
	'IO::Uncompress::Bunzip2'=> '2.022',
	'IO::Uncompress::Gunzip'=> '2.022',
	'IO::Uncompress::Inflate'=> '2.022',
	'IO::Uncompress::RawInflate'=> '2.022',
	'IO::Uncompress::Unzip' => '2.022',
	'IO::Zlib'              => '1.10',
	'IPC::Cmd'              => '0.54',
	'IPC::Msg'              => '2.01',
	'IPC::Open2'            => '1.03',
	'IPC::Open3'            => '1.05',
	'IPC::Semaphore'        => '2.01',
	'IPC::SharedMem'        => '2.01',
	'IPC::SysV'             => '2.01',
	'List::Util'            => '1.22',
	'List::Util::PP'        => '1.22',
	'List::Util::XS'        => '1.22',
	'Locale::Constants'     => '2.07',
	'Locale::Country'       => '2.07',
	'Locale::Currency'      => '2.07',
	'Locale::Language'      => '2.07',
	'Locale::Maketext'      => '1.14',
	'Locale::Maketext::Guts'=> '1.13',
	'Locale::Maketext::GutsLoader'=> '1.13',
	'Locale::Maketext::Simple'=> '0.21',
	'Locale::Script'        => '2.07',
	'Log::Message'          => '0.02',
	'Log::Message::Config'  => '0.01',
	'Log::Message::Handlers'=> undef,
	'Log::Message::Item'    => undef,
	'Log::Message::Simple'  => '0.06',
	'MIME::Base64'          => '3.08',
	'MIME::QuotedPrint'     => '3.08',
	'Math::BigFloat'        => '1.60',
	'Math::BigFloat::Trace' => '0.01',
	'Math::BigInt'          => '1.89_01',
	'Math::BigInt::Calc'    => '0.52',
	'Math::BigInt::CalcEmu' => '0.05',
	'Math::BigInt::FastCalc'=> '0.19',
	'Math::BigInt::Trace'   => '0.01',
	'Math::BigRat'          => '0.24',
	'Math::Complex'         => '1.56',
	'Math::Trig'            => '1.2',
	'Memoize'               => '1.01_03',
	'Memoize::AnyDBM_File'  => '0.65',
	'Memoize::Expire'       => '1.00',
	'Memoize::ExpireFile'   => '1.01',
	'Memoize::ExpireTest'   => '0.65',
	'Memoize::NDBM_File'    => '0.65',
	'Memoize::SDBM_File'    => '0.65',
	'Memoize::Storable'     => '0.65',
	'Module::Build'         => '0.35_09',
	'Module::Build::Base'   => '0.35_09',
	'Module::Build::Compat' => '0.35_09',
	'Module::Build::Config' => '0.35_09',
	'Module::Build::ConfigData'=> undef,
	'Module::Build::Cookbook'=> '0.35_09',
	'Module::Build::Dumper' => '0.35_09',
	'Module::Build::ModuleInfo'=> '0.35_09',
	'Module::Build::Notes'  => '0.35_09',
	'Module::Build::PPMMaker'=> '0.35_09',
	'Module::Build::Platform::Amiga'=> '0.35_09',
	'Module::Build::Platform::Default'=> '0.35_09',
	'Module::Build::Platform::EBCDIC'=> '0.35_09',
	'Module::Build::Platform::MPEiX'=> '0.35_09',
	'Module::Build::Platform::MacOS'=> '0.35_09',
	'Module::Build::Platform::RiscOS'=> '0.35_09',
	'Module::Build::Platform::Unix'=> '0.35_09',
	'Module::Build::Platform::VMS'=> '0.35_09',
	'Module::Build::Platform::VOS'=> '0.35_09',
	'Module::Build::Platform::Windows'=> '0.35_09',
	'Module::Build::Platform::aix'=> '0.35_09',
	'Module::Build::Platform::cygwin'=> '0.35_09',
	'Module::Build::Platform::darwin'=> '0.35_09',
	'Module::Build::Platform::os2'=> '0.35_09',
	'Module::Build::PodParser'=> '0.35_09',
	'Module::Build::Version'=> '0.77',
	'Module::Build::YAML'   => '1.40',
	'Module::CoreList'      => '2.23',
	'Module::Load'          => '0.16',
	'Module::Load::Conditional'=> '0.34',
	'Module::Loaded'        => '0.06',
	'Module::Pluggable'     => '3.9',
	'Module::Pluggable::Object'=> '3.9',
	'Moped::Msg'            => '0.01',
	'NDBM_File'             => '1.08',
	'NEXT'                  => '0.64',
	'Net::Cmd'              => '2.29',
	'Net::Config'           => '1.11',
	'Net::Domain'           => '2.20',
	'Net::FTP'              => '2.77',
	'Net::FTP::A'           => '1.18',
	'Net::FTP::E'           => '0.01',
	'Net::FTP::I'           => '1.12',
	'Net::FTP::L'           => '0.01',
	'Net::FTP::dataconn'    => '0.11',
	'Net::NNTP'             => '2.24',
	'Net::Netrc'            => '2.12',
	'Net::POP3'             => '2.29',
	'Net::Ping'             => '2.36',
	'Net::SMTP'             => '2.31',
	'Net::Time'             => '2.10',
	'Net::hostent'          => '1.01',
	'Net::netent'           => '1.00',
	'Net::protoent'         => '1.00',
	'Net::servent'          => '1.01',
	'O'                     => '1.01',
	'ODBM_File'             => '1.07',
	'Object::Accessor'      => '0.36',
	'Opcode'                => '1.15',
	'POSIX'                 => '1.18',
	'Package::Constants'    => '0.02',
	'Params::Check'         => '0.26',
	'Parse::CPAN::Meta'     => '1.40',
	'PerlIO'                => '1.06',
	'PerlIO::encoding'      => '0.11',
	'PerlIO::scalar'        => '0.07',
	'PerlIO::via'           => '0.09',
	'PerlIO::via::QuotedPrint'=> '0.06',
	'Pod::Checker'          => '1.45',
	'Pod::Escapes'          => '1.04',
	'Pod::Find'             => '1.35',
	'Pod::Functions'        => '1.03',
	'Pod::Html'             => '1.09',
	'Pod::InputObjects'     => '1.31',
	'Pod::LaTeX'            => '0.58',
	'Pod::Man'              => '2.22',
	'Pod::ParseLink'        => '1.09',
	'Pod::ParseUtils'       => '1.36',
	'Pod::Parser'           => '1.37',
	'Pod::Perldoc'          => '3.15_01',
	'Pod::Perldoc::BaseTo'  => undef,
	'Pod::Perldoc::GetOptsOO'=> undef,
	'Pod::Perldoc::ToChecker'=> undef,
	'Pod::Perldoc::ToMan'   => undef,
	'Pod::Perldoc::ToNroff' => undef,
	'Pod::Perldoc::ToPod'   => undef,
	'Pod::Perldoc::ToRtf'   => undef,
	'Pod::Perldoc::ToText'  => undef,
	'Pod::Perldoc::ToTk'    => undef,
	'Pod::Perldoc::ToXml'   => undef,
	'Pod::PlainText'        => '2.04',
	'Pod::Plainer'          => '1.01',
	'Pod::Select'           => '1.36',
	'Pod::Simple'           => '3.10',
	'Pod::Simple::BlackBox' => undef,
	'Pod::Simple::Checker'  => '2.02',
	'Pod::Simple::Debug'    => undef,
	'Pod::Simple::DumpAsText'=> '2.02',
	'Pod::Simple::DumpAsXML'=> '2.02',
	'Pod::Simple::HTML'     => '3.03',
	'Pod::Simple::HTMLBatch'=> '3.02',
	'Pod::Simple::HTMLLegacy'=> '5.01',
	'Pod::Simple::LinkSection'=> undef,
	'Pod::Simple::Methody'  => '2.02',
	'Pod::Simple::Progress' => '1.01',
	'Pod::Simple::PullParser'=> '2.02',
	'Pod::Simple::PullParserEndToken'=> undef,
	'Pod::Simple::PullParserStartToken'=> undef,
	'Pod::Simple::PullParserTextToken'=> undef,
	'Pod::Simple::PullParserToken'=> '2.02',
	'Pod::Simple::RTF'      => '2.02',
	'Pod::Simple::Search'   => '3.04',
	'Pod::Simple::SimpleTree'=> '2.02',
	'Pod::Simple::Text'     => '2.02',
	'Pod::Simple::TextContent'=> '2.02',
	'Pod::Simple::TiedOutFH'=> undef,
	'Pod::Simple::Transcode'=> undef,
	'Pod::Simple::TranscodeDumb'=> '2.02',
	'Pod::Simple::TranscodeSmart'=> undef,
	'Pod::Simple::XHTML'    => '3.10',
	'Pod::Simple::XMLOutStream'=> '2.02',
	'Pod::Text'             => '3.13',
	'Pod::Text::Color'      => '2.05',
	'Pod::Text::Overstrike' => '2.03',
	'Pod::Text::Termcap'    => '2.05',
	'Pod::Usage'            => '1.36',
	'SDBM_File'             => '1.06',
	'Safe'                  => '2.19',
	'Scalar::Util'          => '1.22',
	'Scalar::Util::PP'      => '1.22',
	'Search::Dict'          => '1.02',
	'SelectSaver'           => '1.02',
	'SelfLoader'            => '1.17',
	'Shell'                 => '0.72_01',
	'Simple'                => '0.01',
	'Socket'                => '1.85',
	'Storable'              => '2.22',
	'Switch'                => '2.16',
	'Symbol'                => '1.07',
	'Sys::Hostname'         => '1.11',
	'Sys::Syslog'           => '0.27',
	'Sys::Syslog::win32::Win32'=> undef,
	'TAP::Base'             => '3.17',
	'TAP::Formatter::Base'  => '3.17',
	'TAP::Formatter::Color' => '3.17',
	'TAP::Formatter::Console'=> '3.17',
	'TAP::Formatter::Console::ParallelSession'=> '3.17',
	'TAP::Formatter::Console::Session'=> '3.17',
	'TAP::Formatter::File'  => '3.17',
	'TAP::Formatter::File::Session'=> '3.17',
	'TAP::Formatter::Session'=> '3.17',
	'TAP::Harness'          => '3.17',
	'TAP::Object'           => '3.17',
	'TAP::Parser'           => '3.17',
	'TAP::Parser::Aggregator'=> '3.17',
	'TAP::Parser::Grammar'  => '3.17',
	'TAP::Parser::Iterator' => '3.17',
	'TAP::Parser::Iterator::Array'=> '3.17',
	'TAP::Parser::Iterator::Process'=> '3.17',
	'TAP::Parser::Iterator::Stream'=> '3.17',
	'TAP::Parser::IteratorFactory'=> '3.17',
	'TAP::Parser::Multiplexer'=> '3.17',
	'TAP::Parser::Result'   => '3.17',
	'TAP::Parser::Result::Bailout'=> '3.17',
	'TAP::Parser::Result::Comment'=> '3.17',
	'TAP::Parser::Result::Plan'=> '3.17',
	'TAP::Parser::Result::Pragma'=> '3.17',
	'TAP::Parser::Result::Test'=> '3.17',
	'TAP::Parser::Result::Unknown'=> '3.17',
	'TAP::Parser::Result::Version'=> '3.17',
	'TAP::Parser::Result::YAML'=> '3.17',
	'TAP::Parser::ResultFactory'=> '3.17',
	'TAP::Parser::Scheduler'=> '3.17',
	'TAP::Parser::Scheduler::Job'=> '3.17',
	'TAP::Parser::Scheduler::Spinner'=> '3.17',
	'TAP::Parser::Source'   => '3.17',
	'TAP::Parser::Source::Perl'=> '3.17',
	'TAP::Parser::Utils'    => '3.17',
	'TAP::Parser::YAMLish::Reader'=> '3.17',
	'TAP::Parser::YAMLish::Writer'=> '3.17',
	'Term::ANSIColor'       => '2.02',
	'Term::Cap'             => '1.12',
	'Term::Complete'        => '1.402',
	'Term::ReadLine'        => '1.05',
	'Term::UI'              => '0.20',
	'Term::UI::History'     => undef,
	'Test'                  => '1.25_02',
	'Test::Builder'         => '0.94',
	'Test::Builder::Module' => '0.94',
	'Test::Builder::Tester' => '1.18',
	'Test::Builder::Tester::Color'=> '1.18',
	'Test::Harness'         => '3.17',
	'Test::More'            => '0.94',
	'Test::Simple'          => '0.94',
	'Text::Abbrev'          => '1.01',
	'Text::Balanced'        => '2.02',
	'Text::ParseWords'      => '3.27',
	'Text::Soundex'         => '3.03_01',
	'Text::Tabs'            => '2009.0305',
	'Text::Wrap'            => '2009.0305',
	'Thread'                => '3.02',
	'Thread::Queue'         => '2.11',
	'Thread::Semaphore'     => '2.09',
	'Tie::Array'            => '1.03',
	'Tie::File'             => '0.97_02',
	'Tie::Handle'           => '4.2',
	'Tie::Hash'             => '1.03',
	'Tie::Hash::NamedCapture'=> '0.06',
	'Tie::Memoize'          => '1.1',
	'Tie::RefHash'          => '1.38',
	'Tie::Scalar'           => '1.01',
	'Tie::StdHandle'        => '4.2',
	'Tie::SubstrHash'       => '1.00',
	'Time::HiRes'           => '1.9719',
	'Time::Local'           => '1.1901_01',
	'Time::Piece'           => '1.15',
	'Time::Piece::Seconds'  => undef,
	'Time::Seconds'         => undef,
	'Time::gmtime'          => '1.03',
	'Time::localtime'       => '1.02',
	'Time::tm'              => '1.00',
	'UNIVERSAL'             => '1.05',
	'Unicode'               => '5.1.0',
	'Unicode::Collate'      => '0.52_01',
	'Unicode::Normalize'    => '1.03',
	'Unicode::UCD'          => '0.27',
	'User::grent'           => '1.01',
	'User::pwent'           => '1.00',
	'VMS::DCLsym'           => '1.03',
	'VMS::Stdio'            => '2.4',
	'Win32'                 => '0.39',
	'Win32API::File'        => '0.1101',
	'Win32API::File::ExtUtils::Myconst2perl'=> '1',
	'Win32CORE'             => '0.02',
	'XS::APItest'           => '0.17',
	'XS::APItest::KeywordRPN'=> '0.003',
	'XS::Typemap'           => '0.03',
	'XSLoader'              => '0.10',
	'XSLoader::XSLoader'    => '0.10',
	'attributes'            => '0.12',
	'autodie'               => '2.06_01',
	'autodie::exception'    => '2.06_01',
	'autodie::exception::system'=> '2.06_01',
	'autodie::hints'        => '2.06_01',
	'autouse'               => '1.06',
	'base'                  => '2.15',
	'bigint'                => '0.23',
	'bignum'                => '0.23',
	'bigrat'                => '0.23',
	'blib'                  => '1.04',
	'bytes'                 => '1.03',
	'charnames'             => '1.07',
	'constant'              => '1.19',
	'deprecate'             => '0.01',
	'diagnostics'           => '1.18',
	'encoding'              => '2.6_01',
	'encoding::warnings'    => '0.11',
	'feature'               => '1.13',
	'fields'                => '2.15',
	'filetest'              => '1.02',
	'if'                    => '0.05',
	'inc::latest'           => '0.35_09',
	'integer'               => '1.00',
	'legacy'                => '1.00',
	'less'                  => '0.02',
	'lib'                   => '0.62',
	'locale'                => '1.00',
	'mro'                   => '1.02',
	'open'                  => '1.07',
	'ops'                   => '1.02',
	'overload'              => '1.10',
	'overload::numbers'     => undef,
	'overloading'           => '0.01',
	'parent'                => '0.223',
	're'                    => '0.10',
	'sigtrap'               => '1.04',
	'sort'                  => '2.01',
	'strict'                => '1.04',
	'subs'                  => '1.00',
	'threads'               => '1.74',
	'threads::shared'       => '1.32',
	'utf8'                  => '1.07',
	'vars'                  => '1.01',
	'version'               => '0.77',
	'vmsish'                => '1.02',
	'warnings'              => '1.07',
	'warnings::register'    => '1.01',
    },

    5.011003 => {
	'AnyDBM_File'           => '1.00',
	'App::Cpan'             => '1.570001',
	'App::Prove'            => '3.17',
	'App::Prove::State'     => '3.17',
	'App::Prove::State::Result'=> '3.17',
	'App::Prove::State::Result::Test'=> '3.17',
	'Archive::Extract'      => '0.36',
	'Archive::Tar'          => '1.54',
	'Archive::Tar::Constant'=> '0.02',
	'Archive::Tar::File'    => '0.02',
	'Attribute::Handlers'   => '0.87',
	'AutoLoader'            => '5.70',
	'AutoSplit'             => '1.06',
	'B'                     => '1.23',
	'B::Concise'            => '0.78',
	'B::Debug'              => '1.11',
	'B::Deparse'            => '0.93',
	'B::Lint'               => '1.11_01',
	'B::Lint::Debug'        => '0.01',
	'B::Showlex'            => '1.02',
	'B::Terse'              => '1.05',
	'B::Xref'               => '1.02',
	'Benchmark'             => '1.11',
	'CGI'                   => '3.48',
	'CGI::Apache'           => '1.01',
	'CGI::Carp'             => '3.45',
	'CGI::Cookie'           => '1.29',
	'CGI::Fast'             => '1.07',
	'CGI::Pretty'           => '3.46',
	'CGI::Push'             => '1.04',
	'CGI::Switch'           => '1.01',
	'CGI::Util'             => '3.48',
	'CPAN'                  => '1.94_5301',
	'CPAN::Author'          => '5.5',
	'CPAN::Bundle'          => '5.5',
	'CPAN::CacheMgr'        => '5.5',
	'CPAN::Complete'        => '5.5',
	'CPAN::Debug'           => '5.5',
	'CPAN::DeferredCode'    => '5.50',
	'CPAN::Distribution'    => '1.94',
	'CPAN::Distroprefs'     => '6',
	'CPAN::Distrostatus'    => '5.5',
	'CPAN::Exception::RecursiveDependency'=> '5.5',
	'CPAN::Exception::blocked_urllist'=> '1.0',
	'CPAN::Exception::yaml_not_installed'=> '5.5',
	'CPAN::FTP'             => '5.5004',
	'CPAN::FTP::netrc'      => '1.00',
	'CPAN::FirstTime'       => '5.530001',
	'CPAN::HandleConfig'    => '5.5',
	'CPAN::Index'           => '1.94',
	'CPAN::InfoObj'         => '5.5',
	'CPAN::Kwalify'         => '5.50',
	'CPAN::LWP::UserAgent'  => '1.94',
	'CPAN::Mirrors'         => '1.770001',
	'CPAN::Module'          => '5.5',
	'CPAN::Nox'             => '5.50',
	'CPAN::Prompt'          => '5.5',
	'CPAN::Queue'           => '5.5',
	'CPAN::Shell'           => '5.5',
	'CPAN::Tarzip'          => '5.501',
	'CPAN::URL'             => '5.5',
	'CPAN::Version'         => '5.5',
	'CPANPLUS'              => '0.90',
	'CPANPLUS::Backend'     => undef,
	'CPANPLUS::Backend::RV' => undef,
	'CPANPLUS::Config'      => undef,
	'CPANPLUS::Configure'   => undef,
	'CPANPLUS::Configure::Setup'=> undef,
	'CPANPLUS::Dist'        => undef,
	'CPANPLUS::Dist::Autobundle'=> undef,
	'CPANPLUS::Dist::Base'  => undef,
	'CPANPLUS::Dist::Build' => '0.44',
	'CPANPLUS::Dist::Build::Constants'=> '0.44',
	'CPANPLUS::Dist::MM'    => undef,
	'CPANPLUS::Dist::Sample'=> undef,
	'CPANPLUS::Error'       => undef,
	'CPANPLUS::Internals'   => '0.90',
	'CPANPLUS::Internals::Constants'=> undef,
	'CPANPLUS::Internals::Constants::Report'=> undef,
	'CPANPLUS::Internals::Extract'=> undef,
	'CPANPLUS::Internals::Fetch'=> undef,
	'CPANPLUS::Internals::Report'=> undef,
	'CPANPLUS::Internals::Search'=> undef,
	'CPANPLUS::Internals::Source'=> undef,
	'CPANPLUS::Internals::Source::Memory'=> undef,
	'CPANPLUS::Internals::Source::SQLite'=> undef,
	'CPANPLUS::Internals::Source::SQLite::Tie'=> undef,
	'CPANPLUS::Internals::Utils'=> undef,
	'CPANPLUS::Internals::Utils::Autoflush'=> undef,
	'CPANPLUS::Module'      => undef,
	'CPANPLUS::Module::Author'=> undef,
	'CPANPLUS::Module::Author::Fake'=> undef,
	'CPANPLUS::Module::Checksums'=> undef,
	'CPANPLUS::Module::Fake'=> undef,
	'CPANPLUS::Module::Signature'=> undef,
	'CPANPLUS::Selfupdate'  => undef,
	'CPANPLUS::Shell'       => undef,
	'CPANPLUS::Shell::Classic'=> '0.0562',
	'CPANPLUS::Shell::Default'=> '0.90',
	'CPANPLUS::Shell::Default::Plugins::CustomSource'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Remote'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Source'=> undef,
	'Carp'                  => '1.14',
	'Carp::Heavy'           => '1.14',
	'Class::ISA'            => '0.36',
	'Class::Struct'         => '0.63',
	'Compress::Raw::Bzip2'  => '2.021',
	'Compress::Raw::Zlib'   => '2.021',
	'Compress::Zlib'        => '2.022',
	'Config'                => undef,
	'Config::Extensions'    => '0.01',
	'Cwd'                   => '3.31',
	'DB'                    => '1.02',
	'DBM_Filter'            => '0.03',
	'DBM_Filter::compress'  => '0.02',
	'DBM_Filter::encode'    => '0.02',
	'DBM_Filter::int32'     => '0.02',
	'DBM_Filter::null'      => '0.02',
	'DBM_Filter::utf8'      => '0.02',
	'DB_File'               => '1.820',
	'Data::Dumper'          => '2.125',
	'Devel::DProf'          => '20080331.00',
	'Devel::DProf::V'       => undef,
	'Devel::DProf::dprof::V'=> undef,
	'Devel::InnerPackage'   => '0.3',
	'Devel::PPPort'         => '3.19',
	'Devel::Peek'           => '1.04',
	'Devel::SelfStubber'    => '1.03',
	'Digest'                => '1.16',
	'Digest::MD5'           => '2.39',
	'Digest::SHA'           => '5.47',
	'Digest::base'          => '1.16',
	'Digest::file'          => '1.16',
	'DirHandle'             => '1.03',
	'Dumpvalue'             => '1.13',
	'DynaLoader'            => '1.10',
	'Encode'                => '2.39',
	'Encode::Alias'         => '2.12',
	'Encode::Byte'          => '2.04',
	'Encode::CJKConstants'  => '2.02',
	'Encode::CN'            => '2.03',
	'Encode::CN::HZ'        => '2.05',
	'Encode::Config'        => '2.05',
	'Encode::EBCDIC'        => '2.02',
	'Encode::Encoder'       => '2.01',
	'Encode::Encoding'      => '2.05',
	'Encode::GSM0338'       => '2.01',
	'Encode::Guess'         => '2.03',
	'Encode::JP'            => '2.04',
	'Encode::JP::H2Z'       => '2.02',
	'Encode::JP::JIS7'      => '2.04',
	'Encode::KR'            => '2.03',
	'Encode::KR::2022_KR'   => '2.02',
	'Encode::MIME::Header'  => '2.11',
	'Encode::MIME::Header::ISO_2022_JP'=> '1.03',
	'Encode::MIME::Name'    => '1.01',
	'Encode::Symbol'        => '2.02',
	'Encode::TW'            => '2.03',
	'Encode::Unicode'       => '2.07',
	'Encode::Unicode::UTF7' => '2.04',
	'English'               => '1.04',
	'Env'                   => '1.01',
	'Errno'                 => '1.11',
	'Exporter'              => '5.64_01',
	'Exporter::Heavy'       => '5.64_01',
	'ExtUtils::CBuilder'    => '0.27',
	'ExtUtils::CBuilder::Base'=> '0.27',
	'ExtUtils::CBuilder::Platform::Unix'=> '0.27',
	'ExtUtils::CBuilder::Platform::VMS'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.27',
	'ExtUtils::CBuilder::Platform::aix'=> '0.27',
	'ExtUtils::CBuilder::Platform::cygwin'=> '0.27',
	'ExtUtils::CBuilder::Platform::darwin'=> '0.27',
	'ExtUtils::CBuilder::Platform::dec_osf'=> '0.27',
	'ExtUtils::CBuilder::Platform::os2'=> '0.27',
	'ExtUtils::Command'     => '1.16',
	'ExtUtils::Command::MM' => '6.56',
	'ExtUtils::Constant'    => '0.22',
	'ExtUtils::Constant::Base'=> '0.04',
	'ExtUtils::Constant::ProxySubs'=> '0.06',
	'ExtUtils::Constant::Utils'=> '0.02',
	'ExtUtils::Constant::XS'=> '0.03',
	'ExtUtils::Embed'       => '1.28',
	'ExtUtils::Install'     => '1.55',
	'ExtUtils::Installed'   => '1.999_001',
	'ExtUtils::Liblist'     => '6.56',
	'ExtUtils::Liblist::Kid'=> '6.56',
	'ExtUtils::MM'          => '6.56',
	'ExtUtils::MM_AIX'      => '6.56',
	'ExtUtils::MM_Any'      => '6.56',
	'ExtUtils::MM_BeOS'     => '6.56',
	'ExtUtils::MM_Cygwin'   => '6.56',
	'ExtUtils::MM_DOS'      => '6.56',
	'ExtUtils::MM_Darwin'   => '6.56',
	'ExtUtils::MM_MacOS'    => '6.56',
	'ExtUtils::MM_NW5'      => '6.56',
	'ExtUtils::MM_OS2'      => '6.56',
	'ExtUtils::MM_QNX'      => '6.56',
	'ExtUtils::MM_UWIN'     => '6.56',
	'ExtUtils::MM_Unix'     => '6.56',
	'ExtUtils::MM_VMS'      => '6.56',
	'ExtUtils::MM_VOS'      => '6.56',
	'ExtUtils::MM_Win32'    => '6.56',
	'ExtUtils::MM_Win95'    => '6.56',
	'ExtUtils::MY'          => '6.56',
	'ExtUtils::MakeMaker'   => '6.56',
	'ExtUtils::MakeMaker::Config'=> '6.56',
	'ExtUtils::Manifest'    => '1.57',
	'ExtUtils::Miniperl'    => undef,
	'ExtUtils::Mkbootstrap' => '6.56',
	'ExtUtils::Mksymlists'  => '6.56',
	'ExtUtils::Packlist'    => '1.44',
	'ExtUtils::ParseXS'     => '2.21',
	'ExtUtils::XSSymSet'    => '1.1',
	'ExtUtils::testlib'     => '6.56',
	'Fatal'                 => '2.06_01',
	'Fcntl'                 => '1.06',
	'File::Basename'        => '2.78',
	'File::CheckTree'       => '4.4',
	'File::Compare'         => '1.1006',
	'File::Copy'            => '2.16',
	'File::DosGlob'         => '1.01',
	'File::Fetch'           => '0.22',
	'File::Find'            => '1.15',
	'File::Glob'            => '1.07',
	'File::GlobMapper'      => '1.000',
	'File::Path'            => '2.08_01',
	'File::Spec'            => '3.31',
	'File::Spec::Cygwin'    => '3.30',
	'File::Spec::Epoc'      => '3.30',
	'File::Spec::Functions' => '3.30',
	'File::Spec::Mac'       => '3.30',
	'File::Spec::OS2'       => '3.30',
	'File::Spec::Unix'      => '3.30',
	'File::Spec::VMS'       => '3.30',
	'File::Spec::Win32'     => '3.30',
	'File::Temp'            => '0.22',
	'File::stat'            => '1.02',
	'FileCache'             => '1.08',
	'FileHandle'            => '2.02',
	'VMS::Filespec'         => '1.12',
	'Filter::Simple'        => '0.84',
	'Filter::Util::Call'    => '1.08',
	'FindBin'               => '1.50',
	'GDBM_File'             => '1.09',
	'Getopt::Long'          => '2.38',
	'Getopt::Std'           => '1.06',
	'Hash::Util'            => '0.07',
	'Hash::Util::FieldHash' => '1.04',
	'I18N::Collate'         => '1.01',
	'I18N::LangTags'        => '0.35',
	'I18N::LangTags::Detect'=> '1.04',
	'I18N::LangTags::List'  => '0.35',
	'I18N::Langinfo'        => '0.03',
	'IO'                    => '1.25_02',
	'IO::Compress::Adapter::Bzip2'=> '2.022',
	'IO::Compress::Adapter::Deflate'=> '2.022',
	'IO::Compress::Adapter::Identity'=> '2.022',
	'IO::Compress::Base'    => '2.022',
	'IO::Compress::Base::Common'=> '2.022',
	'IO::Compress::Bzip2'   => '2.022',
	'IO::Compress::Deflate' => '2.022',
	'IO::Compress::Gzip'    => '2.022',
	'IO::Compress::Gzip::Constants'=> '2.022',
	'IO::Compress::RawDeflate'=> '2.022',
	'IO::Compress::Zip'     => '2.022',
	'IO::Compress::Zip::Constants'=> '2.022',
	'IO::Compress::Zlib::Constants'=> '2.022',
	'IO::Compress::Zlib::Extra'=> '2.022',
	'IO::Dir'               => '1.07',
	'IO::File'              => '1.14',
	'IO::Handle'            => '1.28',
	'IO::Pipe'              => '1.13',
	'IO::Poll'              => '0.07',
	'IO::Seekable'          => '1.10',
	'IO::Select'            => '1.17',
	'IO::Socket'            => '1.31',
	'IO::Socket::INET'      => '1.31',
	'IO::Socket::UNIX'      => '1.23',
	'IO::Uncompress::Adapter::Bunzip2'=> '2.022',
	'IO::Uncompress::Adapter::Identity'=> '2.022',
	'IO::Uncompress::Adapter::Inflate'=> '2.022',
	'IO::Uncompress::AnyInflate'=> '2.022',
	'IO::Uncompress::AnyUncompress'=> '2.022',
	'IO::Uncompress::Base'  => '2.022',
	'IO::Uncompress::Bunzip2'=> '2.022',
	'IO::Uncompress::Gunzip'=> '2.022',
	'IO::Uncompress::Inflate'=> '2.022',
	'IO::Uncompress::RawInflate'=> '2.022',
	'IO::Uncompress::Unzip' => '2.022',
	'IO::Zlib'              => '1.10',
	'IPC::Cmd'              => '0.54',
	'IPC::Msg'              => '2.01',
	'IPC::Open2'            => '1.03',
	'IPC::Open3'            => '1.05',
	'IPC::Semaphore'        => '2.01',
	'IPC::SharedMem'        => '2.01',
	'IPC::SysV'             => '2.01',
	'List::Util'            => '1.22',
	'List::Util::PP'        => '1.22',
	'List::Util::XS'        => '1.22',
	'Locale::Constants'     => '2.07',
	'Locale::Country'       => '2.07',
	'Locale::Currency'      => '2.07',
	'Locale::Language'      => '2.07',
	'Locale::Maketext'      => '1.14',
	'Locale::Maketext::Guts'=> '1.13',
	'Locale::Maketext::GutsLoader'=> '1.13',
	'Locale::Maketext::Simple'=> '0.21',
	'Locale::Script'        => '2.07',
	'Log::Message'          => '0.02',
	'Log::Message::Config'  => '0.01',
	'Log::Message::Handlers'=> undef,
	'Log::Message::Item'    => undef,
	'Log::Message::Simple'  => '0.06',
	'MIME::Base64'          => '3.08',
	'MIME::QuotedPrint'     => '3.08',
	'Math::BigFloat'        => '1.60',
	'Math::BigFloat::Trace' => '0.01',
	'Math::BigInt'          => '1.89_01',
	'Math::BigInt::Calc'    => '0.52',
	'Math::BigInt::CalcEmu' => '0.05',
	'Math::BigInt::FastCalc'=> '0.19',
	'Math::BigInt::Trace'   => '0.01',
	'Math::BigRat'          => '0.24',
	'Math::Complex'         => '1.56',
	'Math::Trig'            => '1.2',
	'Memoize'               => '1.01_03',
	'Memoize::AnyDBM_File'  => '0.65',
	'Memoize::Expire'       => '1.00',
	'Memoize::ExpireFile'   => '1.01',
	'Memoize::ExpireTest'   => '0.65',
	'Memoize::NDBM_File'    => '0.65',
	'Memoize::SDBM_File'    => '0.65',
	'Memoize::Storable'     => '0.65',
	'Module::Build'         => '0.36',
	'Module::Build::Base'   => '0.36',
	'Module::Build::Compat' => '0.36',
	'Module::Build::Config' => '0.36',
	'Module::Build::ConfigData'=> undef,
	'Module::Build::Cookbook'=> '0.36',
	'Module::Build::Dumper' => '0.36',
	'Module::Build::ModuleInfo'=> '0.36',
	'Module::Build::Notes'  => '0.36',
	'Module::Build::PPMMaker'=> '0.36',
	'Module::Build::Platform::Amiga'=> '0.36',
	'Module::Build::Platform::Default'=> '0.36',
	'Module::Build::Platform::EBCDIC'=> '0.36',
	'Module::Build::Platform::MPEiX'=> '0.36',
	'Module::Build::Platform::MacOS'=> '0.36',
	'Module::Build::Platform::RiscOS'=> '0.36',
	'Module::Build::Platform::Unix'=> '0.36',
	'Module::Build::Platform::VMS'=> '0.36',
	'Module::Build::Platform::VOS'=> '0.36',
	'Module::Build::Platform::Windows'=> '0.36',
	'Module::Build::Platform::aix'=> '0.36',
	'Module::Build::Platform::cygwin'=> '0.36',
	'Module::Build::Platform::darwin'=> '0.36',
	'Module::Build::Platform::os2'=> '0.36',
	'Module::Build::PodParser'=> '0.36',
	'Module::Build::Version'=> '0.77',
	'Module::Build::YAML'   => '1.40',
	'Module::CoreList'      => '2.24',
	'Module::Load'          => '0.16',
	'Module::Load::Conditional'=> '0.34',
	'Module::Loaded'        => '0.06',
	'Module::Pluggable'     => '3.9',
	'Module::Pluggable::Object'=> '3.9',
	'Moped::Msg'            => '0.01',
	'NDBM_File'             => '1.08',
	'NEXT'                  => '0.64',
	'Net::Cmd'              => '2.29',
	'Net::Config'           => '1.11',
	'Net::Domain'           => '2.20',
	'Net::FTP'              => '2.77',
	'Net::FTP::A'           => '1.18',
	'Net::FTP::E'           => '0.01',
	'Net::FTP::I'           => '1.12',
	'Net::FTP::L'           => '0.01',
	'Net::FTP::dataconn'    => '0.11',
	'Net::NNTP'             => '2.24',
	'Net::Netrc'            => '2.12',
	'Net::POP3'             => '2.29',
	'Net::Ping'             => '2.36',
	'Net::SMTP'             => '2.31',
	'Net::Time'             => '2.10',
	'Net::hostent'          => '1.01',
	'Net::netent'           => '1.00',
	'Net::protoent'         => '1.00',
	'Net::servent'          => '1.01',
	'O'                     => '1.01',
	'ODBM_File'             => '1.07',
	'Object::Accessor'      => '0.36',
	'Opcode'                => '1.15',
	'POSIX'                 => '1.19',
	'Package::Constants'    => '0.02',
	'Params::Check'         => '0.26',
	'Parse::CPAN::Meta'     => '1.40',
	'PerlIO'                => '1.06',
	'PerlIO::encoding'      => '0.11',
	'PerlIO::scalar'        => '0.07',
	'PerlIO::via'           => '0.09',
	'PerlIO::via::QuotedPrint'=> '0.06',
	'Pod::Checker'          => '1.45',
	'Pod::Escapes'          => '1.04',
	'Pod::Find'             => '1.35',
	'Pod::Functions'        => '1.03',
	'Pod::Html'             => '1.09',
	'Pod::InputObjects'     => '1.31',
	'Pod::LaTeX'            => '0.58',
	'Pod::Man'              => '2.22',
	'Pod::ParseLink'        => '1.09',
	'Pod::ParseUtils'       => '1.36',
	'Pod::Parser'           => '1.37',
	'Pod::Perldoc'          => '3.15_01',
	'Pod::Perldoc::BaseTo'  => undef,
	'Pod::Perldoc::GetOptsOO'=> undef,
	'Pod::Perldoc::ToChecker'=> undef,
	'Pod::Perldoc::ToMan'   => undef,
	'Pod::Perldoc::ToNroff' => undef,
	'Pod::Perldoc::ToPod'   => undef,
	'Pod::Perldoc::ToRtf'   => undef,
	'Pod::Perldoc::ToText'  => undef,
	'Pod::Perldoc::ToTk'    => undef,
	'Pod::Perldoc::ToXml'   => undef,
	'Pod::PlainText'        => '2.04',
	'Pod::Plainer'          => '1.01',
	'Pod::Select'           => '1.36',
	'Pod::Simple'           => '3.13',
	'Pod::Simple::BlackBox' => '3.13',
	'Pod::Simple::Checker'  => '3.13',
	'Pod::Simple::Debug'    => '3.13',
	'Pod::Simple::DumpAsText'=> '3.13',
	'Pod::Simple::DumpAsXML'=> '3.13',
	'Pod::Simple::HTML'     => '3.13',
	'Pod::Simple::HTMLBatch'=> '3.13',
	'Pod::Simple::HTMLLegacy'=> '5.01',
	'Pod::Simple::LinkSection'=> '3.13',
	'Pod::Simple::Methody'  => '3.13',
	'Pod::Simple::Progress' => '3.13',
	'Pod::Simple::PullParser'=> '3.13',
	'Pod::Simple::PullParserEndToken'=> '3.13',
	'Pod::Simple::PullParserStartToken'=> '3.13',
	'Pod::Simple::PullParserTextToken'=> '3.13',
	'Pod::Simple::PullParserToken'=> '3.13',
	'Pod::Simple::RTF'      => '3.13',
	'Pod::Simple::Search'   => '3.13',
	'Pod::Simple::SimpleTree'=> '3.13',
	'Pod::Simple::Text'     => '3.13',
	'Pod::Simple::TextContent'=> '3.13',
	'Pod::Simple::TiedOutFH'=> '3.13',
	'Pod::Simple::Transcode'=> '3.13',
	'Pod::Simple::TranscodeDumb'=> '3.13',
	'Pod::Simple::TranscodeSmart'=> '3.13',
	'Pod::Simple::XHTML'    => '3.13',
	'Pod::Simple::XMLOutStream'=> '3.13',
	'Pod::Text'             => '3.13',
	'Pod::Text::Color'      => '2.05',
	'Pod::Text::Overstrike' => '2.03',
	'Pod::Text::Termcap'    => '2.05',
	'Pod::Usage'            => '1.36',
	'SDBM_File'             => '1.06',
	'Safe'                  => '2.20',
	'Scalar::Util'          => '1.22',
	'Scalar::Util::PP'      => '1.22',
	'Search::Dict'          => '1.02',
	'SelectSaver'           => '1.02',
	'SelfLoader'            => '1.17',
	'Shell'                 => '0.72_01',
	'Socket'                => '1.85',
	'Storable'              => '2.22',
	'Switch'                => '2.16',
	'Symbol'                => '1.07',
	'Sys::Hostname'         => '1.11',
	'Sys::Syslog'           => '0.27',
	'Sys::Syslog::win32::Win32'=> undef,
	'TAP::Base'             => '3.17',
	'TAP::Formatter::Base'  => '3.17',
	'TAP::Formatter::Color' => '3.17',
	'TAP::Formatter::Console'=> '3.17',
	'TAP::Formatter::Console::ParallelSession'=> '3.17',
	'TAP::Formatter::Console::Session'=> '3.17',
	'TAP::Formatter::File'  => '3.17',
	'TAP::Formatter::File::Session'=> '3.17',
	'TAP::Formatter::Session'=> '3.17',
	'TAP::Harness'          => '3.17',
	'TAP::Object'           => '3.17',
	'TAP::Parser'           => '3.17',
	'TAP::Parser::Aggregator'=> '3.17',
	'TAP::Parser::Grammar'  => '3.17',
	'TAP::Parser::Iterator' => '3.17',
	'TAP::Parser::Iterator::Array'=> '3.17',
	'TAP::Parser::Iterator::Process'=> '3.17',
	'TAP::Parser::Iterator::Stream'=> '3.17',
	'TAP::Parser::IteratorFactory'=> '3.17',
	'TAP::Parser::Multiplexer'=> '3.17',
	'TAP::Parser::Result'   => '3.17',
	'TAP::Parser::Result::Bailout'=> '3.17',
	'TAP::Parser::Result::Comment'=> '3.17',
	'TAP::Parser::Result::Plan'=> '3.17',
	'TAP::Parser::Result::Pragma'=> '3.17',
	'TAP::Parser::Result::Test'=> '3.17',
	'TAP::Parser::Result::Unknown'=> '3.17',
	'TAP::Parser::Result::Version'=> '3.17',
	'TAP::Parser::Result::YAML'=> '3.17',
	'TAP::Parser::ResultFactory'=> '3.17',
	'TAP::Parser::Scheduler'=> '3.17',
	'TAP::Parser::Scheduler::Job'=> '3.17',
	'TAP::Parser::Scheduler::Spinner'=> '3.17',
	'TAP::Parser::Source'   => '3.17',
	'TAP::Parser::Source::Perl'=> '3.17',
	'TAP::Parser::Utils'    => '3.17',
	'TAP::Parser::YAMLish::Reader'=> '3.17',
	'TAP::Parser::YAMLish::Writer'=> '3.17',
	'Term::ANSIColor'       => '2.02',
	'Term::Cap'             => '1.12',
	'Term::Complete'        => '1.402',
	'Term::ReadLine'        => '1.05',
	'Term::UI'              => '0.20',
	'Term::UI::History'     => undef,
	'Test'                  => '1.25_02',
	'Test::Builder'         => '0.94',
	'Test::Builder::Module' => '0.94',
	'Test::Builder::Tester' => '1.18',
	'Test::Builder::Tester::Color'=> '1.18',
	'Test::Harness'         => '3.17',
	'Test::More'            => '0.94',
	'Test::Simple'          => '0.94',
	'Text::Abbrev'          => '1.01',
	'Text::Balanced'        => '2.02',
	'Text::ParseWords'      => '3.27',
	'Text::Soundex'         => '3.03_01',
	'Text::Tabs'            => '2009.0305',
	'Text::Wrap'            => '2009.0305',
	'Thread'                => '3.02',
	'Thread::Queue'         => '2.11',
	'Thread::Semaphore'     => '2.09',
	'Tie::Array'            => '1.03',
	'Tie::File'             => '0.97_02',
	'Tie::Handle'           => '4.2',
	'Tie::Hash'             => '1.03',
	'Tie::Hash::NamedCapture'=> '0.06',
	'Tie::Memoize'          => '1.1',
	'Tie::RefHash'          => '1.38',
	'Tie::Scalar'           => '1.01',
	'Tie::StdHandle'        => '4.2',
	'Tie::SubstrHash'       => '1.00',
	'Time::HiRes'           => '1.9719',
	'Time::Local'           => '1.1901_01',
	'Time::Piece'           => '1.15',
	'Time::Piece::Seconds'  => undef,
	'Time::Seconds'         => undef,
	'Time::gmtime'          => '1.03',
	'Time::localtime'       => '1.02',
	'Time::tm'              => '1.00',
	'UNIVERSAL'             => '1.05',
	'Unicode'               => '5.2.0',
	'Unicode::Collate'      => '0.52_01',
	'Unicode::Normalize'    => '1.03',
	'Unicode::UCD'          => '0.27',
	'User::grent'           => '1.01',
	'User::pwent'           => '1.00',
	'VMS::DCLsym'           => '1.03',
	'VMS::Stdio'            => '2.4',
	'Win32'                 => '0.39',
	'Win32API::File'        => '0.1101',
	'Win32API::File::ExtUtils::Myconst2perl'=> '1',
	'Win32CORE'             => '0.02',
	'XS::APItest'           => '0.17',
	'XS::APItest::KeywordRPN'=> '0.003',
	'XS::Typemap'           => '0.03',
	'XSLoader'              => '0.10',
	'XSLoader::XSLoader'    => '0.10',
	'attributes'            => '0.12',
	'autodie'               => '2.06_01',
	'autodie::exception'    => '2.06_01',
	'autodie::exception::system'=> '2.06_01',
	'autodie::hints'        => '2.06_01',
	'autouse'               => '1.06',
	'base'                  => '2.15',
	'bigint'                => '0.23',
	'bignum'                => '0.23',
	'bigrat'                => '0.23',
	'blib'                  => '1.04',
	'bytes'                 => '1.03',
	'charnames'             => '1.07',
	'constant'              => '1.20',
	'deprecate'             => '0.01',
	'diagnostics'           => '1.19',
	'encoding'              => '2.6_01',
	'encoding::warnings'    => '0.11',
	'feature'               => '1.14',
	'fields'                => '2.15',
	'filetest'              => '1.02',
	'if'                    => '0.05',
	'inc::latest'           => '0.36',
	'integer'               => '1.00',
	'less'                  => '0.02',
	'lib'                   => '0.62',
	'locale'                => '1.00',
	'mro'                   => '1.02',
	'open'                  => '1.07',
	'ops'                   => '1.02',
	'overload'              => '1.10',
	'overload::numbers'     => undef,
	'overloading'           => '0.01',
	'parent'                => '0.223',
	're'                    => '0.10',
	'sigtrap'               => '1.04',
	'sort'                  => '2.01',
	'strict'                => '1.04',
	'subs'                  => '1.00',
	'threads'               => '1.75',
	'threads::shared'       => '1.32',
	'utf8'                  => '1.07',
	'vars'                  => '1.01',
	'version'               => '0.77',
	'vmsish'                => '1.02',
	'warnings'              => '1.08',
	'warnings::register'    => '1.01',
    },

    5.011004 => {
	'AnyDBM_File'           => '1.00',
	'App::Cpan'             => '1.5701',
	'App::Prove'            => '3.17',
	'App::Prove::State'     => '3.17',
	'App::Prove::State::Result'=> '3.17',
	'App::Prove::State::Result::Test'=> '3.17',
	'Archive::Extract'      => '0.38',
	'Archive::Tar'          => '1.54',
	'Archive::Tar::Constant'=> '0.02',
	'Archive::Tar::File'    => '0.02',
	'Attribute::Handlers'   => '0.87',
	'AutoLoader'            => '5.70',
	'AutoSplit'             => '1.06',
	'B'                     => '1.23',
	'B::Concise'            => '0.78',
	'B::Debug'              => '1.11',
	'B::Deparse'            => '0.94',
	'B::Lint'               => '1.11_01',
	'B::Lint::Debug'        => '0.01',
	'B::Showlex'            => '1.02',
	'B::Terse'              => '1.05',
	'B::Xref'               => '1.02',
	'Benchmark'             => '1.11',
	'CGI'                   => '3.48',
	'CGI::Apache'           => '1.01',
	'CGI::Carp'             => '3.45',
	'CGI::Cookie'           => '1.29',
	'CGI::Fast'             => '1.07',
	'CGI::Pretty'           => '3.46',
	'CGI::Push'             => '1.04',
	'CGI::Switch'           => '1.01',
	'CGI::Util'             => '3.48',
	'CPAN'                  => '1.94_54',
	'CPAN::Author'          => '5.5',
	'CPAN::Bundle'          => '5.5',
	'CPAN::CacheMgr'        => '5.5',
	'CPAN::Complete'        => '5.5',
	'CPAN::Debug'           => '5.5',
	'CPAN::DeferredCode'    => '5.50',
	'CPAN::Distribution'    => '1.94',
	'CPAN::Distroprefs'     => '6',
	'CPAN::Distrostatus'    => '5.5',
	'CPAN::Exception::RecursiveDependency'=> '5.5',
	'CPAN::Exception::blocked_urllist'=> '1.0',
	'CPAN::Exception::yaml_not_installed'=> '5.5',
	'CPAN::FTP'             => '5.5004',
	'CPAN::FTP::netrc'      => '1.00',
	'CPAN::FirstTime'       => '5.53',
	'CPAN::HandleConfig'    => '5.5',
	'CPAN::Index'           => '1.94',
	'CPAN::InfoObj'         => '5.5',
	'CPAN::Kwalify'         => '5.50',
	'CPAN::LWP::UserAgent'  => '1.94',
	'CPAN::Mirrors'         => '1.77',
	'CPAN::Module'          => '5.5',
	'CPAN::Nox'             => '5.50',
	'CPAN::Prompt'          => '5.5',
	'CPAN::Queue'           => '5.5',
	'CPAN::Shell'           => '5.5',
	'CPAN::Tarzip'          => '5.501',
	'CPAN::URL'             => '5.5',
	'CPAN::Version'         => '5.5',
	'CPANPLUS'              => '0.90',
	'CPANPLUS::Backend'     => undef,
	'CPANPLUS::Backend::RV' => undef,
	'CPANPLUS::Config'      => undef,
	'CPANPLUS::Configure'   => undef,
	'CPANPLUS::Configure::Setup'=> undef,
	'CPANPLUS::Dist'        => undef,
	'CPANPLUS::Dist::Autobundle'=> undef,
	'CPANPLUS::Dist::Base'  => undef,
	'CPANPLUS::Dist::Build' => '0.44',
	'CPANPLUS::Dist::Build::Constants'=> '0.44',
	'CPANPLUS::Dist::MM'    => undef,
	'CPANPLUS::Dist::Sample'=> undef,
	'CPANPLUS::Error'       => undef,
	'CPANPLUS::Internals'   => '0.90',
	'CPANPLUS::Internals::Constants'=> undef,
	'CPANPLUS::Internals::Constants::Report'=> undef,
	'CPANPLUS::Internals::Extract'=> undef,
	'CPANPLUS::Internals::Fetch'=> undef,
	'CPANPLUS::Internals::Report'=> undef,
	'CPANPLUS::Internals::Search'=> undef,
	'CPANPLUS::Internals::Source'=> undef,
	'CPANPLUS::Internals::Source::Memory'=> undef,
	'CPANPLUS::Internals::Source::SQLite'=> undef,
	'CPANPLUS::Internals::Source::SQLite::Tie'=> undef,
	'CPANPLUS::Internals::Utils'=> undef,
	'CPANPLUS::Internals::Utils::Autoflush'=> undef,
	'CPANPLUS::Module'      => undef,
	'CPANPLUS::Module::Author'=> undef,
	'CPANPLUS::Module::Author::Fake'=> undef,
	'CPANPLUS::Module::Checksums'=> undef,
	'CPANPLUS::Module::Fake'=> undef,
	'CPANPLUS::Module::Signature'=> undef,
	'CPANPLUS::Selfupdate'  => undef,
	'CPANPLUS::Shell'       => undef,
	'CPANPLUS::Shell::Classic'=> '0.0562',
	'CPANPLUS::Shell::Default'=> '0.90',
	'CPANPLUS::Shell::Default::Plugins::CustomSource'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Remote'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Source'=> undef,
	'Carp'                  => '1.15',
	'Carp::Heavy'           => '1.15',
	'Class::ISA'            => '0.36',
	'Class::Struct'         => '0.63',
	'Compress::Raw::Bzip2'  => '2.024',
	'Compress::Raw::Zlib'   => '2.024',
	'Compress::Zlib'        => '2.024',
	'Config'                => undef,
	'Config::Extensions'    => '0.01',
	'Cwd'                   => '3.31',
	'DB'                    => '1.02',
	'DBM_Filter'            => '0.03',
	'DBM_Filter::compress'  => '0.02',
	'DBM_Filter::encode'    => '0.02',
	'DBM_Filter::int32'     => '0.02',
	'DBM_Filter::null'      => '0.02',
	'DBM_Filter::utf8'      => '0.02',
	'DB_File'               => '1.820',
	'Data::Dumper'          => '2.125',
	'Devel::DProf'          => '20080331.00',
	'Devel::DProf::dprof::V'=> undef,
	'Devel::InnerPackage'   => '0.3',
	'Devel::PPPort'         => '3.19',
	'Devel::Peek'           => '1.04',
	'Devel::SelfStubber'    => '1.03',
	'Digest'                => '1.16',
	'Digest::MD5'           => '2.39',
	'Digest::SHA'           => '5.47',
	'Digest::base'          => '1.16',
	'Digest::file'          => '1.16',
	'DirHandle'             => '1.03',
	'Dumpvalue'             => '1.13',
	'DynaLoader'            => '1.10',
	'Encode'                => '2.39',
	'Encode::Alias'         => '2.12',
	'Encode::Byte'          => '2.04',
	'Encode::CJKConstants'  => '2.02',
	'Encode::CN'            => '2.03',
	'Encode::CN::HZ'        => '2.05',
	'Encode::Config'        => '2.05',
	'Encode::EBCDIC'        => '2.02',
	'Encode::Encoder'       => '2.01',
	'Encode::Encoding'      => '2.05',
	'Encode::GSM0338'       => '2.01',
	'Encode::Guess'         => '2.03',
	'Encode::JP'            => '2.04',
	'Encode::JP::H2Z'       => '2.02',
	'Encode::JP::JIS7'      => '2.04',
	'Encode::KR'            => '2.03',
	'Encode::KR::2022_KR'   => '2.02',
	'Encode::MIME::Header'  => '2.11',
	'Encode::MIME::Header::ISO_2022_JP'=> '1.03',
	'Encode::MIME::Name'    => '1.01',
	'Encode::Symbol'        => '2.02',
	'Encode::TW'            => '2.03',
	'Encode::Unicode'       => '2.07',
	'Encode::Unicode::UTF7' => '2.04',
	'English'               => '1.04',
	'Env'                   => '1.01',
	'Errno'                 => '1.11',
	'Exporter'              => '5.64_01',
	'Exporter::Heavy'       => '5.64_01',
	'ExtUtils::CBuilder'    => '0.27',
	'ExtUtils::CBuilder::Base'=> '0.27',
	'ExtUtils::CBuilder::Platform::Unix'=> '0.27',
	'ExtUtils::CBuilder::Platform::VMS'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.27',
	'ExtUtils::CBuilder::Platform::aix'=> '0.27',
	'ExtUtils::CBuilder::Platform::cygwin'=> '0.27',
	'ExtUtils::CBuilder::Platform::darwin'=> '0.27',
	'ExtUtils::CBuilder::Platform::dec_osf'=> '0.27',
	'ExtUtils::CBuilder::Platform::os2'=> '0.27',
	'ExtUtils::Command'     => '1.16',
	'ExtUtils::Command::MM' => '6.56',
	'ExtUtils::Constant'    => '0.22',
	'ExtUtils::Constant::Base'=> '0.04',
	'ExtUtils::Constant::ProxySubs'=> '0.06',
	'ExtUtils::Constant::Utils'=> '0.02',
	'ExtUtils::Constant::XS'=> '0.03',
	'ExtUtils::Embed'       => '1.28',
	'ExtUtils::Install'     => '1.55',
	'ExtUtils::Installed'   => '1.999_001',
	'ExtUtils::Liblist'     => '6.56',
	'ExtUtils::Liblist::Kid'=> '6.56',
	'ExtUtils::MM'          => '6.56',
	'ExtUtils::MM_AIX'      => '6.56',
	'ExtUtils::MM_Any'      => '6.56',
	'ExtUtils::MM_BeOS'     => '6.56',
	'ExtUtils::MM_Cygwin'   => '6.56',
	'ExtUtils::MM_DOS'      => '6.56',
	'ExtUtils::MM_Darwin'   => '6.56',
	'ExtUtils::MM_MacOS'    => '6.56',
	'ExtUtils::MM_NW5'      => '6.56',
	'ExtUtils::MM_OS2'      => '6.56',
	'ExtUtils::MM_QNX'      => '6.56',
	'ExtUtils::MM_UWIN'     => '6.56',
	'ExtUtils::MM_Unix'     => '6.56',
	'ExtUtils::MM_VMS'      => '6.56',
	'ExtUtils::MM_VOS'      => '6.56',
	'ExtUtils::MM_Win32'    => '6.56',
	'ExtUtils::MM_Win95'    => '6.56',
	'ExtUtils::MY'          => '6.56',
	'ExtUtils::MakeMaker'   => '6.56',
	'ExtUtils::MakeMaker::Config'=> '6.56',
	'ExtUtils::Manifest'    => '1.57',
	'ExtUtils::Miniperl'    => undef,
	'ExtUtils::Mkbootstrap' => '6.56',
	'ExtUtils::Mksymlists'  => '6.56',
	'ExtUtils::Packlist'    => '1.44',
	'ExtUtils::ParseXS'     => '2.21',
	'ExtUtils::XSSymSet'    => '1.1',
	'ExtUtils::testlib'     => '6.56',
	'Fatal'                 => '2.06_01',
	'Fcntl'                 => '1.06',
	'File::Basename'        => '2.78',
	'File::CheckTree'       => '4.4',
	'File::Compare'         => '1.1006',
	'File::Copy'            => '2.17',
	'File::DosGlob'         => '1.01',
	'File::Fetch'           => '0.24',
	'File::Find'            => '1.15',
	'File::Glob'            => '1.07',
	'File::GlobMapper'      => '1.000',
	'File::Path'            => '2.08_01',
	'File::Spec'            => '3.31',
	'File::Spec::Cygwin'    => '3.30',
	'File::Spec::Epoc'      => '3.30',
	'File::Spec::Functions' => '3.30',
	'File::Spec::Mac'       => '3.30',
	'File::Spec::OS2'       => '3.30',
	'File::Spec::Unix'      => '3.30',
	'File::Spec::VMS'       => '3.30',
	'File::Spec::Win32'     => '3.30',
	'File::Temp'            => '0.22',
	'File::stat'            => '1.02',
	'FileCache'             => '1.08',
	'FileHandle'            => '2.02',
	'VMS::Filespec'         => '1.12',
	'Filter::Simple'        => '0.84',
	'Filter::Util::Call'    => '1.08',
	'FindBin'               => '1.50',
	'GDBM_File'             => '1.10',
	'Getopt::Long'          => '2.38',
	'Getopt::Std'           => '1.06',
	'Hash::Util'            => '0.07',
	'Hash::Util::FieldHash' => '1.04',
	'I18N::Collate'         => '1.01',
	'I18N::LangTags'        => '0.35',
	'I18N::LangTags::Detect'=> '1.04',
	'I18N::LangTags::List'  => '0.35',
	'I18N::Langinfo'        => '0.03',
	'IO'                    => '1.25_02',
	'IO::Compress::Adapter::Bzip2'=> '2.024',
	'IO::Compress::Adapter::Deflate'=> '2.024',
	'IO::Compress::Adapter::Identity'=> '2.024',
	'IO::Compress::Base'    => '2.024',
	'IO::Compress::Base::Common'=> '2.024',
	'IO::Compress::Bzip2'   => '2.024',
	'IO::Compress::Deflate' => '2.024',
	'IO::Compress::Gzip'    => '2.024',
	'IO::Compress::Gzip::Constants'=> '2.024',
	'IO::Compress::RawDeflate'=> '2.024',
	'IO::Compress::Zip'     => '2.024',
	'IO::Compress::Zip::Constants'=> '2.024',
	'IO::Compress::Zlib::Constants'=> '2.024',
	'IO::Compress::Zlib::Extra'=> '2.024',
	'IO::Dir'               => '1.07',
	'IO::File'              => '1.14',
	'IO::Handle'            => '1.28',
	'IO::Pipe'              => '1.13',
	'IO::Poll'              => '0.07',
	'IO::Seekable'          => '1.10',
	'IO::Select'            => '1.17',
	'IO::Socket'            => '1.31',
	'IO::Socket::INET'      => '1.31',
	'IO::Socket::UNIX'      => '1.23',
	'IO::Uncompress::Adapter::Bunzip2'=> '2.024',
	'IO::Uncompress::Adapter::Identity'=> '2.024',
	'IO::Uncompress::Adapter::Inflate'=> '2.024',
	'IO::Uncompress::AnyInflate'=> '2.024',
	'IO::Uncompress::AnyUncompress'=> '2.024',
	'IO::Uncompress::Base'  => '2.024',
	'IO::Uncompress::Bunzip2'=> '2.024',
	'IO::Uncompress::Gunzip'=> '2.024',
	'IO::Uncompress::Inflate'=> '2.024',
	'IO::Uncompress::RawInflate'=> '2.024',
	'IO::Uncompress::Unzip' => '2.024',
	'IO::Zlib'              => '1.10',
	'IPC::Cmd'              => '0.54',
	'IPC::Msg'              => '2.01',
	'IPC::Open2'            => '1.03',
	'IPC::Open3'            => '1.05',
	'IPC::Semaphore'        => '2.01',
	'IPC::SharedMem'        => '2.01',
	'IPC::SysV'             => '2.01',
	'List::Util'            => '1.22',
	'List::Util::PP'        => '1.22',
	'List::Util::XS'        => '1.22',
	'Locale::Constants'     => '2.07',
	'Locale::Country'       => '2.07',
	'Locale::Currency'      => '2.07',
	'Locale::Language'      => '2.07',
	'Locale::Maketext'      => '1.14',
	'Locale::Maketext::Guts'=> '1.13',
	'Locale::Maketext::GutsLoader'=> '1.13',
	'Locale::Maketext::Simple'=> '0.21',
	'Locale::Script'        => '2.07',
	'Log::Message'          => '0.02',
	'Log::Message::Config'  => '0.01',
	'Log::Message::Handlers'=> undef,
	'Log::Message::Item'    => undef,
	'Log::Message::Simple'  => '0.06',
	'MIME::Base64'          => '3.08',
	'MIME::QuotedPrint'     => '3.08',
	'Math::BigFloat'        => '1.60',
	'Math::BigFloat::Trace' => '0.01',
	'Math::BigInt'          => '1.89_01',
	'Math::BigInt::Calc'    => '0.52',
	'Math::BigInt::CalcEmu' => '0.05',
	'Math::BigInt::FastCalc'=> '0.19',
	'Math::BigInt::Trace'   => '0.01',
	'Math::BigRat'          => '0.24',
	'Math::Complex'         => '1.56',
	'Math::Trig'            => '1.2',
	'Memoize'               => '1.01_03',
	'Memoize::AnyDBM_File'  => '0.65',
	'Memoize::Expire'       => '1.00',
	'Memoize::ExpireFile'   => '1.01',
	'Memoize::ExpireTest'   => '0.65',
	'Memoize::NDBM_File'    => '0.65',
	'Memoize::SDBM_File'    => '0.65',
	'Memoize::Storable'     => '0.65',
	'Module::Build'         => '0.3603',
	'Module::Build::Base'   => '0.3603',
	'Module::Build::Compat' => '0.3603',
	'Module::Build::Config' => '0.3603',
	'Module::Build::ConfigData'=> undef,
	'Module::Build::Cookbook'=> '0.3603',
	'Module::Build::Dumper' => '0.3603',
	'Module::Build::ModuleInfo'=> '0.3603',
	'Module::Build::Notes'  => '0.3603',
	'Module::Build::PPMMaker'=> '0.3603',
	'Module::Build::Platform::Amiga'=> '0.3603',
	'Module::Build::Platform::Default'=> '0.3603',
	'Module::Build::Platform::EBCDIC'=> '0.3603',
	'Module::Build::Platform::MPEiX'=> '0.3603',
	'Module::Build::Platform::MacOS'=> '0.3603',
	'Module::Build::Platform::RiscOS'=> '0.3603',
	'Module::Build::Platform::Unix'=> '0.3603',
	'Module::Build::Platform::VMS'=> '0.3603',
	'Module::Build::Platform::VOS'=> '0.3603',
	'Module::Build::Platform::Windows'=> '0.3603',
	'Module::Build::Platform::aix'=> '0.3603',
	'Module::Build::Platform::cygwin'=> '0.3603',
	'Module::Build::Platform::darwin'=> '0.3603',
	'Module::Build::Platform::os2'=> '0.3603',
	'Module::Build::PodParser'=> '0.3603',
	'Module::Build::Version'=> '0.77',
	'Module::Build::YAML'   => '1.40',
	'Module::CoreList'      => '2.25',
	'Module::Load'          => '0.16',
	'Module::Load::Conditional'=> '0.34',
	'Module::Loaded'        => '0.06',
	'Module::Pluggable'     => '3.9',
	'Module::Pluggable::Object'=> '3.9',
	'Moped::Msg'            => '0.01',
	'NDBM_File'             => '1.08',
	'NEXT'                  => '0.64',
	'Net::Cmd'              => '2.29',
	'Net::Config'           => '1.11',
	'Net::Domain'           => '2.20',
	'Net::FTP'              => '2.77',
	'Net::FTP::A'           => '1.18',
	'Net::FTP::E'           => '0.01',
	'Net::FTP::I'           => '1.12',
	'Net::FTP::L'           => '0.01',
	'Net::FTP::dataconn'    => '0.11',
	'Net::NNTP'             => '2.24',
	'Net::Netrc'            => '2.12',
	'Net::POP3'             => '2.29',
	'Net::Ping'             => '2.36',
	'Net::SMTP'             => '2.31',
	'Net::Time'             => '2.10',
	'Net::hostent'          => '1.01',
	'Net::netent'           => '1.00',
	'Net::protoent'         => '1.00',
	'Net::servent'          => '1.01',
	'O'                     => '1.01',
	'ODBM_File'             => '1.07',
	'Object::Accessor'      => '0.36',
	'Opcode'                => '1.15',
	'POSIX'                 => '1.19',
	'Package::Constants'    => '0.02',
	'Params::Check'         => '0.26',
	'Parse::CPAN::Meta'     => '1.40',
	'PerlIO'                => '1.06',
	'PerlIO::encoding'      => '0.12',
	'PerlIO::scalar'        => '0.07',
	'PerlIO::via'           => '0.09',
	'PerlIO::via::QuotedPrint'=> '0.06',
	'Pod::Checker'          => '1.45',
	'Pod::Escapes'          => '1.04',
	'Pod::Find'             => '1.35',
	'Pod::Functions'        => '1.03',
	'Pod::Html'             => '1.09',
	'Pod::InputObjects'     => '1.31',
	'Pod::LaTeX'            => '0.58',
	'Pod::Man'              => '2.22',
	'Pod::ParseLink'        => '1.09',
	'Pod::ParseUtils'       => '1.36',
	'Pod::Parser'           => '1.37',
	'Pod::Perldoc'          => '3.15_01',
	'Pod::Perldoc::BaseTo'  => undef,
	'Pod::Perldoc::GetOptsOO'=> undef,
	'Pod::Perldoc::ToChecker'=> undef,
	'Pod::Perldoc::ToMan'   => undef,
	'Pod::Perldoc::ToNroff' => undef,
	'Pod::Perldoc::ToPod'   => undef,
	'Pod::Perldoc::ToRtf'   => undef,
	'Pod::Perldoc::ToText'  => undef,
	'Pod::Perldoc::ToTk'    => undef,
	'Pod::Perldoc::ToXml'   => undef,
	'Pod::PlainText'        => '2.04',
	'Pod::Plainer'          => '1.01',
	'Pod::Select'           => '1.36',
	'Pod::Simple'           => '3.13',
	'Pod::Simple::BlackBox' => '3.13',
	'Pod::Simple::Checker'  => '3.13',
	'Pod::Simple::Debug'    => '3.13',
	'Pod::Simple::DumpAsText'=> '3.13',
	'Pod::Simple::DumpAsXML'=> '3.13',
	'Pod::Simple::HTML'     => '3.13',
	'Pod::Simple::HTMLBatch'=> '3.13',
	'Pod::Simple::HTMLLegacy'=> '5.01',
	'Pod::Simple::LinkSection'=> '3.13',
	'Pod::Simple::Methody'  => '3.13',
	'Pod::Simple::Progress' => '3.13',
	'Pod::Simple::PullParser'=> '3.13',
	'Pod::Simple::PullParserEndToken'=> '3.13',
	'Pod::Simple::PullParserStartToken'=> '3.13',
	'Pod::Simple::PullParserTextToken'=> '3.13',
	'Pod::Simple::PullParserToken'=> '3.13',
	'Pod::Simple::RTF'      => '3.13',
	'Pod::Simple::Search'   => '3.13',
	'Pod::Simple::SimpleTree'=> '3.13',
	'Pod::Simple::Text'     => '3.13',
	'Pod::Simple::TextContent'=> '3.13',
	'Pod::Simple::TiedOutFH'=> '3.13',
	'Pod::Simple::Transcode'=> '3.13',
	'Pod::Simple::TranscodeDumb'=> '3.13',
	'Pod::Simple::TranscodeSmart'=> '3.13',
	'Pod::Simple::XHTML'    => '3.13',
	'Pod::Simple::XMLOutStream'=> '3.13',
	'Pod::Text'             => '3.13',
	'Pod::Text::Color'      => '2.05',
	'Pod::Text::Overstrike' => '2.03',
	'Pod::Text::Termcap'    => '2.05',
	'Pod::Usage'            => '1.36',
	'SDBM_File'             => '1.06',
	'Safe'                  => '2.21',
	'Scalar::Util'          => '1.22',
	'Scalar::Util::PP'      => '1.22',
	'Search::Dict'          => '1.02',
	'SelectSaver'           => '1.02',
	'SelfLoader'            => '1.17',
	'Shell'                 => '0.72_01',
	'Socket'                => '1.85',
	'Storable'              => '2.22',
	'Switch'                => '2.16',
	'Symbol'                => '1.07',
	'Sys::Hostname'         => '1.11',
	'Sys::Syslog'           => '0.27',
	'Sys::Syslog::win32::Win32'=> undef,
	'TAP::Base'             => '3.17',
	'TAP::Formatter::Base'  => '3.17',
	'TAP::Formatter::Color' => '3.17',
	'TAP::Formatter::Console'=> '3.17',
	'TAP::Formatter::Console::ParallelSession'=> '3.17',
	'TAP::Formatter::Console::Session'=> '3.17',
	'TAP::Formatter::File'  => '3.17',
	'TAP::Formatter::File::Session'=> '3.17',
	'TAP::Formatter::Session'=> '3.17',
	'TAP::Harness'          => '3.17',
	'TAP::Object'           => '3.17',
	'TAP::Parser'           => '3.17',
	'TAP::Parser::Aggregator'=> '3.17',
	'TAP::Parser::Grammar'  => '3.17',
	'TAP::Parser::Iterator' => '3.17',
	'TAP::Parser::Iterator::Array'=> '3.17',
	'TAP::Parser::Iterator::Process'=> '3.17',
	'TAP::Parser::Iterator::Stream'=> '3.17',
	'TAP::Parser::IteratorFactory'=> '3.17',
	'TAP::Parser::Multiplexer'=> '3.17',
	'TAP::Parser::Result'   => '3.17',
	'TAP::Parser::Result::Bailout'=> '3.17',
	'TAP::Parser::Result::Comment'=> '3.17',
	'TAP::Parser::Result::Plan'=> '3.17',
	'TAP::Parser::Result::Pragma'=> '3.17',
	'TAP::Parser::Result::Test'=> '3.17',
	'TAP::Parser::Result::Unknown'=> '3.17',
	'TAP::Parser::Result::Version'=> '3.17',
	'TAP::Parser::Result::YAML'=> '3.17',
	'TAP::Parser::ResultFactory'=> '3.17',
	'TAP::Parser::Scheduler'=> '3.17',
	'TAP::Parser::Scheduler::Job'=> '3.17',
	'TAP::Parser::Scheduler::Spinner'=> '3.17',
	'TAP::Parser::Source'   => '3.17',
	'TAP::Parser::Source::Perl'=> '3.17',
	'TAP::Parser::Utils'    => '3.17',
	'TAP::Parser::YAMLish::Reader'=> '3.17',
	'TAP::Parser::YAMLish::Writer'=> '3.17',
	'Term::ANSIColor'       => '2.02',
	'Term::Cap'             => '1.12',
	'Term::Complete'        => '1.402',
	'Term::ReadLine'        => '1.05',
	'Term::UI'              => '0.20',
	'Term::UI::History'     => undef,
	'Test'                  => '1.25_02',
	'Test::Builder'         => '0.94',
	'Test::Builder::Module' => '0.94',
	'Test::Builder::Tester' => '1.18',
	'Test::Builder::Tester::Color'=> '1.18',
	'Test::Harness'         => '3.17',
	'Test::More'            => '0.94',
	'Test::Simple'          => '0.94',
	'Text::Abbrev'          => '1.01',
	'Text::Balanced'        => '2.02',
	'Text::ParseWords'      => '3.27',
	'Text::Soundex'         => '3.03_01',
	'Text::Tabs'            => '2009.0305',
	'Text::Wrap'            => '2009.0305',
	'Thread'                => '3.02',
	'Thread::Queue'         => '2.11',
	'Thread::Semaphore'     => '2.09',
	'Tie::Array'            => '1.03',
	'Tie::File'             => '0.97_02',
	'Tie::Handle'           => '4.2',
	'Tie::Hash'             => '1.03',
	'Tie::Hash::NamedCapture'=> '0.06',
	'Tie::Memoize'          => '1.1',
	'Tie::RefHash'          => '1.38',
	'Tie::Scalar'           => '1.01',
	'Tie::StdHandle'        => '4.2',
	'Tie::SubstrHash'       => '1.00',
	'Time::HiRes'           => '1.9719',
	'Time::Local'           => '1.1901_01',
	'Time::Piece'           => '1.15',
	'Time::Piece::Seconds'  => undef,
	'Time::Seconds'         => undef,
	'Time::gmtime'          => '1.03',
	'Time::localtime'       => '1.02',
	'Time::tm'              => '1.00',
	'UNIVERSAL'             => '1.06',
	'Unicode'               => '5.2.0',
	'Unicode::Collate'      => '0.52_01',
	'Unicode::Normalize'    => '1.03',
	'Unicode::UCD'          => '0.27',
	'User::grent'           => '1.01',
	'User::pwent'           => '1.00',
	'VMS::DCLsym'           => '1.03',
	'VMS::Stdio'            => '2.4',
	'Win32'                 => '0.39',
	'Win32API::File'        => '0.1101',
	'Win32API::File::ExtUtils::Myconst2perl'=> '1',
	'Win32CORE'             => '0.02',
	'XS::APItest'           => '0.17',
	'XS::APItest::KeywordRPN'=> '0.003',
	'XS::Typemap'           => '0.03',
	'XSLoader'              => '0.10',
	'XSLoader::XSLoader'    => '0.10',
	'attributes'            => '0.12',
	'autodie'               => '2.06_01',
	'autodie::exception'    => '2.06_01',
	'autodie::exception::system'=> '2.06_01',
	'autodie::hints'        => '2.06_01',
	'autouse'               => '1.06',
	'base'                  => '2.15',
	'bigint'                => '0.23',
	'bignum'                => '0.23',
	'bigrat'                => '0.23',
	'blib'                  => '1.04',
	'bytes'                 => '1.03',
	'charnames'             => '1.07',
	'constant'              => '1.20',
	'deprecate'             => '0.01',
	'diagnostics'           => '1.19',
	'encoding'              => '2.6_01',
	'encoding::warnings'    => '0.11',
	'feature'               => '1.15',
	'fields'                => '2.15',
	'filetest'              => '1.02',
	'if'                    => '0.05',
	'inc::latest'           => '0.3603',
	'integer'               => '1.00',
	'less'                  => '0.03',
	'lib'                   => '0.62',
	'locale'                => '1.00',
	'mro'                   => '1.02',
	'open'                  => '1.07',
	'ops'                   => '1.02',
	'overload'              => '1.10',
	'overload::numbers'     => undef,
	'overloading'           => '0.01',
	'parent'                => '0.223',
	're'                    => '0.11',
	'sigtrap'               => '1.04',
	'sort'                  => '2.01',
	'strict'                => '1.04',
	'subs'                  => '1.00',
	'threads'               => '1.75',
	'threads::shared'       => '1.32',
	'utf8'                  => '1.07',
	'vars'                  => '1.01',
	'version'               => '0.81',
	'vmsish'                => '1.02',
	'warnings'              => '1.09',
	'warnings::register'    => '1.01',
    },

    5.011005 => {
	'AnyDBM_File'           => '1.00',
	'App::Cpan'             => '1.5701',
	'App::Prove'            => '3.17',
	'App::Prove::State'     => '3.17',
	'App::Prove::State::Result'=> '3.17',
	'App::Prove::State::Result::Test'=> '3.17',
	'Archive::Extract'      => '0.38',
	'Archive::Tar'          => '1.54',
	'Archive::Tar::Constant'=> '0.02',
	'Archive::Tar::File'    => '0.02',
	'Attribute::Handlers'   => '0.87',
	'AutoLoader'            => '5.70',
	'AutoSplit'             => '1.06',
	'B'                     => '1.23',
	'B::Concise'            => '0.78',
	'B::Debug'              => '1.12',
	'B::Deparse'            => '0.94',
	'B::Lint'               => '1.11_01',
	'B::Lint::Debug'        => '0.01',
	'B::Showlex'            => '1.02',
	'B::Terse'              => '1.05',
	'B::Xref'               => '1.02',
	'Benchmark'             => '1.11',
	'CGI'                   => '3.48',
	'CGI::Apache'           => '1.01',
	'CGI::Carp'             => '3.45',
	'CGI::Cookie'           => '1.29',
	'CGI::Fast'             => '1.07',
	'CGI::Pretty'           => '3.46',
	'CGI::Push'             => '1.04',
	'CGI::Switch'           => '1.01',
	'CGI::Util'             => '3.48',
	'CPAN'                  => '1.94_56',
	'CPAN::Author'          => '5.5',
	'CPAN::Bundle'          => '5.5',
	'CPAN::CacheMgr'        => '5.5',
	'CPAN::Complete'        => '5.5',
	'CPAN::Debug'           => '5.5001',
	'CPAN::DeferredCode'    => '5.50',
	'CPAN::Distribution'    => '1.9456',
	'CPAN::Distroprefs'     => '6',
	'CPAN::Distrostatus'    => '5.5',
	'CPAN::Exception::RecursiveDependency'=> '5.5',
	'CPAN::Exception::blocked_urllist'=> '1.0',
	'CPAN::Exception::yaml_not_installed'=> '5.5',
	'CPAN::FTP'             => '5.5004',
	'CPAN::FTP::netrc'      => '1.00',
	'CPAN::FirstTime'       => '5.5301',
	'CPAN::HandleConfig'    => '5.5001',
	'CPAN::Index'           => '1.94',
	'CPAN::InfoObj'         => '5.5',
	'CPAN::Kwalify'         => '5.50',
	'CPAN::LWP::UserAgent'  => '1.94',
	'CPAN::Mirrors'         => '1.77',
	'CPAN::Module'          => '5.5',
	'CPAN::Nox'             => '5.50',
	'CPAN::Prompt'          => '5.5',
	'CPAN::Queue'           => '5.5',
	'CPAN::Shell'           => '5.5001',
	'CPAN::Tarzip'          => '5.5011',
	'CPAN::URL'             => '5.5',
	'CPAN::Version'         => '5.5',
	'CPANPLUS'              => '0.90',
	'CPANPLUS::Backend'     => undef,
	'CPANPLUS::Backend::RV' => undef,
	'CPANPLUS::Config'      => undef,
	'CPANPLUS::Configure'   => undef,
	'CPANPLUS::Configure::Setup'=> undef,
	'CPANPLUS::Dist'        => undef,
	'CPANPLUS::Dist::Autobundle'=> undef,
	'CPANPLUS::Dist::Base'  => undef,
	'CPANPLUS::Dist::Build' => '0.46',
	'CPANPLUS::Dist::Build::Constants'=> '0.46',
	'CPANPLUS::Dist::MM'    => undef,
	'CPANPLUS::Dist::Sample'=> undef,
	'CPANPLUS::Error'       => undef,
	'CPANPLUS::Internals'   => '0.90',
	'CPANPLUS::Internals::Constants'=> undef,
	'CPANPLUS::Internals::Constants::Report'=> undef,
	'CPANPLUS::Internals::Extract'=> undef,
	'CPANPLUS::Internals::Fetch'=> undef,
	'CPANPLUS::Internals::Report'=> undef,
	'CPANPLUS::Internals::Search'=> undef,
	'CPANPLUS::Internals::Source'=> undef,
	'CPANPLUS::Internals::Source::Memory'=> undef,
	'CPANPLUS::Internals::Source::SQLite'=> undef,
	'CPANPLUS::Internals::Source::SQLite::Tie'=> undef,
	'CPANPLUS::Internals::Utils'=> undef,
	'CPANPLUS::Internals::Utils::Autoflush'=> undef,
	'CPANPLUS::Module'      => undef,
	'CPANPLUS::Module::Author'=> undef,
	'CPANPLUS::Module::Author::Fake'=> undef,
	'CPANPLUS::Module::Checksums'=> undef,
	'CPANPLUS::Module::Fake'=> undef,
	'CPANPLUS::Module::Signature'=> undef,
	'CPANPLUS::Selfupdate'  => undef,
	'CPANPLUS::Shell'       => undef,
	'CPANPLUS::Shell::Classic'=> '0.0562',
	'CPANPLUS::Shell::Default'=> '0.90',
	'CPANPLUS::Shell::Default::Plugins::CustomSource'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Remote'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Source'=> undef,
	'Carp'                  => '1.15',
	'Carp::Heavy'           => '1.15',
	'Class::ISA'            => '0.36',
	'Class::Struct'         => '0.63',
	'Compress::Raw::Bzip2'  => '2.024',
	'Compress::Raw::Zlib'   => '2.024',
	'Compress::Zlib'        => '2.024',
	'Config'                => undef,
	'Config::Extensions'    => '0.01',
	'Cwd'                   => '3.31',
	'DB'                    => '1.02',
	'DBM_Filter'            => '0.03',
	'DBM_Filter::compress'  => '0.02',
	'DBM_Filter::encode'    => '0.02',
	'DBM_Filter::int32'     => '0.02',
	'DBM_Filter::null'      => '0.02',
	'DBM_Filter::utf8'      => '0.02',
	'DB_File'               => '1.820',
	'Data::Dumper'          => '2.125',
	'Devel::DProf'          => '20080331.00',
	'Devel::DProf::V'       => undef,
	'Devel::DProf::dprof::V'=> undef,
	'Devel::InnerPackage'   => '0.3',
	'Devel::PPPort'         => '3.19',
	'Devel::Peek'           => '1.04',
	'Devel::SelfStubber'    => '1.03',
	'Digest'                => '1.16',
	'Digest::MD5'           => '2.39',
	'Digest::SHA'           => '5.47',
	'Digest::base'          => '1.16',
	'Digest::file'          => '1.16',
	'DirHandle'             => '1.03',
	'Dumpvalue'             => '1.13',
	'DynaLoader'            => '1.10',
	'Encode'                => '2.39',
	'Encode::Alias'         => '2.12',
	'Encode::Byte'          => '2.04',
	'Encode::CJKConstants'  => '2.02',
	'Encode::CN'            => '2.03',
	'Encode::CN::HZ'        => '2.05',
	'Encode::Config'        => '2.05',
	'Encode::EBCDIC'        => '2.02',
	'Encode::Encoder'       => '2.01',
	'Encode::Encoding'      => '2.05',
	'Encode::GSM0338'       => '2.01',
	'Encode::Guess'         => '2.03',
	'Encode::JP'            => '2.04',
	'Encode::JP::H2Z'       => '2.02',
	'Encode::JP::JIS7'      => '2.04',
	'Encode::KR'            => '2.03',
	'Encode::KR::2022_KR'   => '2.02',
	'Encode::MIME::Header'  => '2.11',
	'Encode::MIME::Header::ISO_2022_JP'=> '1.03',
	'Encode::MIME::Name'    => '1.01',
	'Encode::Symbol'        => '2.02',
	'Encode::TW'            => '2.03',
	'Encode::Unicode'       => '2.07',
	'Encode::Unicode::UTF7' => '2.04',
	'English'               => '1.04',
	'Env'                   => '1.01',
	'Errno'                 => '1.11',
	'Exporter'              => '5.64_01',
	'Exporter::Heavy'       => '5.64_01',
	'ExtUtils::CBuilder'    => '0.27',
	'ExtUtils::CBuilder::Base'=> '0.27',
	'ExtUtils::CBuilder::Platform::Unix'=> '0.27',
	'ExtUtils::CBuilder::Platform::VMS'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.27',
	'ExtUtils::CBuilder::Platform::aix'=> '0.27',
	'ExtUtils::CBuilder::Platform::cygwin'=> '0.27',
	'ExtUtils::CBuilder::Platform::darwin'=> '0.27',
	'ExtUtils::CBuilder::Platform::dec_osf'=> '0.27',
	'ExtUtils::CBuilder::Platform::os2'=> '0.27',
	'ExtUtils::Command'     => '1.16',
	'ExtUtils::Command::MM' => '6.56',
	'ExtUtils::Constant'    => '0.22',
	'ExtUtils::Constant::Base'=> '0.04',
	'ExtUtils::Constant::ProxySubs'=> '0.06',
	'ExtUtils::Constant::Utils'=> '0.02',
	'ExtUtils::Constant::XS'=> '0.03',
	'ExtUtils::Embed'       => '1.28',
	'ExtUtils::Install'     => '1.55',
	'ExtUtils::Installed'   => '1.999_001',
	'ExtUtils::Liblist'     => '6.56',
	'ExtUtils::Liblist::Kid'=> '6.56',
	'ExtUtils::MM'          => '6.56',
	'ExtUtils::MM_AIX'      => '6.56',
	'ExtUtils::MM_Any'      => '6.56',
	'ExtUtils::MM_BeOS'     => '6.56',
	'ExtUtils::MM_Cygwin'   => '6.56',
	'ExtUtils::MM_DOS'      => '6.56',
	'ExtUtils::MM_Darwin'   => '6.56',
	'ExtUtils::MM_MacOS'    => '6.56',
	'ExtUtils::MM_NW5'      => '6.56',
	'ExtUtils::MM_OS2'      => '6.56',
	'ExtUtils::MM_QNX'      => '6.56',
	'ExtUtils::MM_UWIN'     => '6.56',
	'ExtUtils::MM_Unix'     => '6.56',
	'ExtUtils::MM_VMS'      => '6.56',
	'ExtUtils::MM_VOS'      => '6.56',
	'ExtUtils::MM_Win32'    => '6.56',
	'ExtUtils::MM_Win95'    => '6.56',
	'ExtUtils::MY'          => '6.56',
	'ExtUtils::MakeMaker'   => '6.56',
	'ExtUtils::MakeMaker::Config'=> '6.56',
	'ExtUtils::Manifest'    => '1.57',
	'ExtUtils::Miniperl'    => undef,
	'ExtUtils::Mkbootstrap' => '6.56',
	'ExtUtils::Mksymlists'  => '6.56',
	'ExtUtils::Packlist'    => '1.44',
	'ExtUtils::ParseXS'     => '2.21',
	'ExtUtils::XSSymSet'    => '1.1',
	'ExtUtils::testlib'     => '6.56',
	'Fatal'                 => '2.06_01',
	'Fcntl'                 => '1.06',
	'File::Basename'        => '2.78',
	'File::CheckTree'       => '4.4',
	'File::Compare'         => '1.1006',
	'File::Copy'            => '2.17',
	'File::DosGlob'         => '1.01',
	'File::Fetch'           => '0.24',
	'File::Find'            => '1.15',
	'File::Glob'            => '1.07',
	'File::GlobMapper'      => '1.000',
	'File::Path'            => '2.08_01',
	'File::Spec'            => '3.31',
	'File::Spec::Cygwin'    => '3.30',
	'File::Spec::Epoc'      => '3.30',
	'File::Spec::Functions' => '3.30',
	'File::Spec::Mac'       => '3.30',
	'File::Spec::OS2'       => '3.30',
	'File::Spec::Unix'      => '3.30',
	'File::Spec::VMS'       => '3.30',
	'File::Spec::Win32'     => '3.30',
	'File::Temp'            => '0.22',
	'File::stat'            => '1.02',
	'FileCache'             => '1.08',
	'FileHandle'            => '2.02',
	'VMS::Filespec'         => '1.12',
	'Filter::Simple'        => '0.84',
	'Filter::Util::Call'    => '1.08',
	'FindBin'               => '1.50',
	'GDBM_File'             => '1.10',
	'Getopt::Long'          => '2.38',
	'Getopt::Std'           => '1.06',
	'Hash::Util'            => '0.07',
	'Hash::Util::FieldHash' => '1.04',
	'I18N::Collate'         => '1.01',
	'I18N::LangTags'        => '0.35',
	'I18N::LangTags::Detect'=> '1.04',
	'I18N::LangTags::List'  => '0.35',
	'I18N::Langinfo'        => '0.03',
	'IO'                    => '1.25_02',
	'IO::Compress::Adapter::Bzip2'=> '2.024',
	'IO::Compress::Adapter::Deflate'=> '2.024',
	'IO::Compress::Adapter::Identity'=> '2.024',
	'IO::Compress::Base'    => '2.024',
	'IO::Compress::Base::Common'=> '2.024',
	'IO::Compress::Bzip2'   => '2.024',
	'IO::Compress::Deflate' => '2.024',
	'IO::Compress::Gzip'    => '2.024',
	'IO::Compress::Gzip::Constants'=> '2.024',
	'IO::Compress::RawDeflate'=> '2.024',
	'IO::Compress::Zip'     => '2.024',
	'IO::Compress::Zip::Constants'=> '2.024',
	'IO::Compress::Zlib::Constants'=> '2.024',
	'IO::Compress::Zlib::Extra'=> '2.024',
	'IO::Dir'               => '1.07',
	'IO::File'              => '1.14',
	'IO::Handle'            => '1.28',
	'IO::Pipe'              => '1.13',
	'IO::Poll'              => '0.07',
	'IO::Seekable'          => '1.10',
	'IO::Select'            => '1.17',
	'IO::Socket'            => '1.31',
	'IO::Socket::INET'      => '1.31',
	'IO::Socket::UNIX'      => '1.23',
	'IO::Uncompress::Adapter::Bunzip2'=> '2.024',
	'IO::Uncompress::Adapter::Identity'=> '2.024',
	'IO::Uncompress::Adapter::Inflate'=> '2.024',
	'IO::Uncompress::AnyInflate'=> '2.024',
	'IO::Uncompress::AnyUncompress'=> '2.024',
	'IO::Uncompress::Base'  => '2.024',
	'IO::Uncompress::Bunzip2'=> '2.024',
	'IO::Uncompress::Gunzip'=> '2.024',
	'IO::Uncompress::Inflate'=> '2.024',
	'IO::Uncompress::RawInflate'=> '2.024',
	'IO::Uncompress::Unzip' => '2.024',
	'IO::Zlib'              => '1.10',
	'IPC::Cmd'              => '0.54',
	'IPC::Msg'              => '2.01',
	'IPC::Open2'            => '1.03',
	'IPC::Open3'            => '1.05',
	'IPC::Semaphore'        => '2.01',
	'IPC::SharedMem'        => '2.01',
	'IPC::SysV'             => '2.01',
	'List::Util'            => '1.22',
	'List::Util::PP'        => '1.22',
	'List::Util::XS'        => '1.22',
	'Locale::Constants'     => '2.07',
	'Locale::Country'       => '2.07',
	'Locale::Currency'      => '2.07',
	'Locale::Language'      => '2.07',
	'Locale::Maketext'      => '1.14',
	'Locale::Maketext::Guts'=> '1.13',
	'Locale::Maketext::GutsLoader'=> '1.13',
	'Locale::Maketext::Simple'=> '0.21',
	'Locale::Script'        => '2.07',
	'Log::Message'          => '0.02',
	'Log::Message::Config'  => '0.01',
	'Log::Message::Handlers'=> undef,
	'Log::Message::Item'    => undef,
	'Log::Message::Simple'  => '0.06',
	'MIME::Base64'          => '3.08',
	'MIME::QuotedPrint'     => '3.08',
	'Math::BigFloat'        => '1.60',
	'Math::BigFloat::Trace' => '0.01',
	'Math::BigInt'          => '1.89_01',
	'Math::BigInt::Calc'    => '0.52',
	'Math::BigInt::CalcEmu' => '0.05',
	'Math::BigInt::FastCalc'=> '0.19',
	'Math::BigInt::Trace'   => '0.01',
	'Math::BigRat'          => '0.24',
	'Math::Complex'         => '1.56',
	'Math::Trig'            => '1.2',
	'Memoize'               => '1.01_03',
	'Memoize::AnyDBM_File'  => '0.65',
	'Memoize::Expire'       => '1.00',
	'Memoize::ExpireFile'   => '1.01',
	'Memoize::ExpireTest'   => '0.65',
	'Memoize::NDBM_File'    => '0.65',
	'Memoize::SDBM_File'    => '0.65',
	'Memoize::Storable'     => '0.65',
	'Module::Build'         => '0.3603',
	'Module::Build::Base'   => '0.3603',
	'Module::Build::Compat' => '0.3603',
	'Module::Build::Config' => '0.3603',
	'Module::Build::ConfigData'=> undef,
	'Module::Build::Cookbook'=> '0.3603',
	'Module::Build::Dumper' => '0.3603',
	'Module::Build::ModuleInfo'=> '0.3603',
	'Module::Build::Notes'  => '0.3603',
	'Module::Build::PPMMaker'=> '0.3603',
	'Module::Build::Platform::Amiga'=> '0.3603',
	'Module::Build::Platform::Default'=> '0.3603',
	'Module::Build::Platform::EBCDIC'=> '0.3603',
	'Module::Build::Platform::MPEiX'=> '0.3603',
	'Module::Build::Platform::MacOS'=> '0.3603',
	'Module::Build::Platform::RiscOS'=> '0.3603',
	'Module::Build::Platform::Unix'=> '0.3603',
	'Module::Build::Platform::VMS'=> '0.3603',
	'Module::Build::Platform::VOS'=> '0.3603',
	'Module::Build::Platform::Windows'=> '0.3603',
	'Module::Build::Platform::aix'=> '0.3603',
	'Module::Build::Platform::cygwin'=> '0.3603',
	'Module::Build::Platform::darwin'=> '0.3603',
	'Module::Build::Platform::os2'=> '0.3603',
	'Module::Build::PodParser'=> '0.3603',
	'Module::Build::Version'=> '0.77',
	'Module::Build::YAML'   => '1.40',
	'Module::CoreList'      => '2.26',
	'Module::Load'          => '0.16',
	'Module::Load::Conditional'=> '0.34',
	'Module::Loaded'        => '0.06',
	'Module::Pluggable'     => '3.9',
	'Module::Pluggable::Object'=> '3.9',
	'Moped::Msg'            => '0.01',
	'NDBM_File'             => '1.08',
	'NEXT'                  => '0.64',
	'Net::Cmd'              => '2.29',
	'Net::Config'           => '1.11',
	'Net::Domain'           => '2.20',
	'Net::FTP'              => '2.77',
	'Net::FTP::A'           => '1.18',
	'Net::FTP::E'           => '0.01',
	'Net::FTP::I'           => '1.12',
	'Net::FTP::L'           => '0.01',
	'Net::FTP::dataconn'    => '0.11',
	'Net::NNTP'             => '2.24',
	'Net::Netrc'            => '2.12',
	'Net::POP3'             => '2.29',
	'Net::Ping'             => '2.36',
	'Net::SMTP'             => '2.31',
	'Net::Time'             => '2.10',
	'Net::hostent'          => '1.01',
	'Net::netent'           => '1.00',
	'Net::protoent'         => '1.00',
	'Net::servent'          => '1.01',
	'O'                     => '1.01',
	'ODBM_File'             => '1.07',
	'Object::Accessor'      => '0.36',
	'Opcode'                => '1.15',
	'POSIX'                 => '1.19',
	'Package::Constants'    => '0.02',
	'Params::Check'         => '0.26',
	'Parse::CPAN::Meta'     => '1.40',
	'PerlIO'                => '1.06',
	'PerlIO::encoding'      => '0.12',
	'PerlIO::scalar'        => '0.07',
	'PerlIO::via'           => '0.09',
	'PerlIO::via::QuotedPrint'=> '0.06',
	'Pod::Checker'          => '1.45',
	'Pod::Escapes'          => '1.04',
	'Pod::Find'             => '1.35',
	'Pod::Functions'        => '1.03',
	'Pod::Html'             => '1.09',
	'Pod::InputObjects'     => '1.31',
	'Pod::LaTeX'            => '0.58',
	'Pod::Man'              => '2.23',
	'Pod::ParseLink'        => '1.10',
	'Pod::ParseUtils'       => '1.36',
	'Pod::Parser'           => '1.37',
	'Pod::Perldoc'          => '3.15_02',
	'Pod::Perldoc::BaseTo'  => undef,
	'Pod::Perldoc::GetOptsOO'=> undef,
	'Pod::Perldoc::ToChecker'=> undef,
	'Pod::Perldoc::ToMan'   => undef,
	'Pod::Perldoc::ToNroff' => undef,
	'Pod::Perldoc::ToPod'   => undef,
	'Pod::Perldoc::ToRtf'   => undef,
	'Pod::Perldoc::ToText'  => undef,
	'Pod::Perldoc::ToTk'    => undef,
	'Pod::Perldoc::ToXml'   => undef,
	'Pod::PlainText'        => '2.04',
	'Pod::Plainer'          => '1.02',
	'Pod::Select'           => '1.36',
	'Pod::Simple'           => '3.13',
	'Pod::Simple::BlackBox' => '3.13',
	'Pod::Simple::Checker'  => '3.13',
	'Pod::Simple::Debug'    => '3.13',
	'Pod::Simple::DumpAsText'=> '3.13',
	'Pod::Simple::DumpAsXML'=> '3.13',
	'Pod::Simple::HTML'     => '3.13',
	'Pod::Simple::HTMLBatch'=> '3.13',
	'Pod::Simple::HTMLLegacy'=> '5.01',
	'Pod::Simple::LinkSection'=> '3.13',
	'Pod::Simple::Methody'  => '3.13',
	'Pod::Simple::Progress' => '3.13',
	'Pod::Simple::PullParser'=> '3.13',
	'Pod::Simple::PullParserEndToken'=> '3.13',
	'Pod::Simple::PullParserStartToken'=> '3.13',
	'Pod::Simple::PullParserTextToken'=> '3.13',
	'Pod::Simple::PullParserToken'=> '3.13',
	'Pod::Simple::RTF'      => '3.13',
	'Pod::Simple::Search'   => '3.13',
	'Pod::Simple::SimpleTree'=> '3.13',
	'Pod::Simple::Text'     => '3.13',
	'Pod::Simple::TextContent'=> '3.13',
	'Pod::Simple::TiedOutFH'=> '3.13',
	'Pod::Simple::Transcode'=> '3.13',
	'Pod::Simple::TranscodeDumb'=> '3.13',
	'Pod::Simple::TranscodeSmart'=> '3.13',
	'Pod::Simple::XHTML'    => '3.13',
	'Pod::Simple::XMLOutStream'=> '3.13',
	'Pod::Text'             => '3.14',
	'Pod::Text::Color'      => '2.06',
	'Pod::Text::Overstrike' => '2.04',
	'Pod::Text::Termcap'    => '2.06',
	'Pod::Usage'            => '1.36',
	'SDBM_File'             => '1.06',
	'Safe'                  => '2.22',
	'Scalar::Util'          => '1.22',
	'Scalar::Util::PP'      => '1.22',
	'Search::Dict'          => '1.02',
	'SelectSaver'           => '1.02',
	'SelfLoader'            => '1.17',
	'Shell'                 => '0.72_01',
	'Socket'                => '1.86',
	'Storable'              => '2.22',
	'Switch'                => '2.16',
	'Symbol'                => '1.07',
	'Sys::Hostname'         => '1.11',
	'Sys::Syslog'           => '0.27',
	'Sys::Syslog::win32::Win32'=> undef,
	'TAP::Base'             => '3.17',
	'TAP::Formatter::Base'  => '3.17',
	'TAP::Formatter::Color' => '3.17',
	'TAP::Formatter::Console'=> '3.17',
	'TAP::Formatter::Console::ParallelSession'=> '3.17',
	'TAP::Formatter::Console::Session'=> '3.17',
	'TAP::Formatter::File'  => '3.17',
	'TAP::Formatter::File::Session'=> '3.17',
	'TAP::Formatter::Session'=> '3.17',
	'TAP::Harness'          => '3.17',
	'TAP::Object'           => '3.17',
	'TAP::Parser'           => '3.17',
	'TAP::Parser::Aggregator'=> '3.17',
	'TAP::Parser::Grammar'  => '3.17',
	'TAP::Parser::Iterator' => '3.17',
	'TAP::Parser::Iterator::Array'=> '3.17',
	'TAP::Parser::Iterator::Process'=> '3.17',
	'TAP::Parser::Iterator::Stream'=> '3.17',
	'TAP::Parser::IteratorFactory'=> '3.17',
	'TAP::Parser::Multiplexer'=> '3.17',
	'TAP::Parser::Result'   => '3.17',
	'TAP::Parser::Result::Bailout'=> '3.17',
	'TAP::Parser::Result::Comment'=> '3.17',
	'TAP::Parser::Result::Plan'=> '3.17',
	'TAP::Parser::Result::Pragma'=> '3.17',
	'TAP::Parser::Result::Test'=> '3.17',
	'TAP::Parser::Result::Unknown'=> '3.17',
	'TAP::Parser::Result::Version'=> '3.17',
	'TAP::Parser::Result::YAML'=> '3.17',
	'TAP::Parser::ResultFactory'=> '3.17',
	'TAP::Parser::Scheduler'=> '3.17',
	'TAP::Parser::Scheduler::Job'=> '3.17',
	'TAP::Parser::Scheduler::Spinner'=> '3.17',
	'TAP::Parser::Source'   => '3.17',
	'TAP::Parser::Source::Perl'=> '3.17',
	'TAP::Parser::Utils'    => '3.17',
	'TAP::Parser::YAMLish::Reader'=> '3.17',
	'TAP::Parser::YAMLish::Writer'=> '3.17',
	'Term::ANSIColor'       => '2.02',
	'Term::Cap'             => '1.12',
	'Term::Complete'        => '1.402',
	'Term::ReadLine'        => '1.05',
	'Term::UI'              => '0.20',
	'Term::UI::History'     => undef,
	'Test'                  => '1.25_02',
	'Test::Builder'         => '0.94',
	'Test::Builder::Module' => '0.94',
	'Test::Builder::Tester' => '1.18',
	'Test::Builder::Tester::Color'=> '1.18',
	'Test::Harness'         => '3.17',
	'Test::More'            => '0.94',
	'Test::Simple'          => '0.94',
	'Text::Abbrev'          => '1.01',
	'Text::Balanced'        => '2.02',
	'Text::ParseWords'      => '3.27',
	'Text::Soundex'         => '3.03_01',
	'Text::Tabs'            => '2009.0305',
	'Text::Wrap'            => '2009.0305',
	'Thread'                => '3.02',
	'Thread::Queue'         => '2.11',
	'Thread::Semaphore'     => '2.09',
	'Tie::Array'            => '1.03',
	'Tie::File'             => '0.97_02',
	'Tie::Handle'           => '4.2',
	'Tie::Hash'             => '1.03',
	'Tie::Hash::NamedCapture'=> '0.06',
	'Tie::Memoize'          => '1.1',
	'Tie::RefHash'          => '1.38',
	'Tie::Scalar'           => '1.01',
	'Tie::StdHandle'        => '4.2',
	'Tie::SubstrHash'       => '1.00',
	'Time::HiRes'           => '1.9719',
	'Time::Local'           => '1.1901_01',
	'Time::Piece'           => '1.15',
	'Time::Piece::Seconds'  => undef,
	'Time::Seconds'         => undef,
	'Time::gmtime'          => '1.03',
	'Time::localtime'       => '1.02',
	'Time::tm'              => '1.00',
	'UNIVERSAL'             => '1.06',
	'Unicode'               => '5.2.0',
	'Unicode::Collate'      => '0.52_01',
	'Unicode::Normalize'    => '1.03',
	'Unicode::UCD'          => '0.27',
	'User::grent'           => '1.01',
	'User::pwent'           => '1.00',
	'VMS::DCLsym'           => '1.03',
	'VMS::Stdio'            => '2.4',
	'Win32'                 => '0.39',
	'Win32API::File'        => '0.1101',
	'Win32API::File::ExtUtils::Myconst2perl'=> '1',
	'Win32CORE'             => '0.02',
	'XS::APItest'           => '0.17',
	'XS::APItest::KeywordRPN'=> '0.003',
	'XS::Typemap'           => '0.03',
	'XSLoader'              => '0.10',
	'XSLoader::XSLoader'    => '0.10',
	'attributes'            => '0.12',
	'autodie'               => '2.06_01',
	'autodie::exception'    => '2.06_01',
	'autodie::exception::system'=> '2.06_01',
	'autodie::hints'        => '2.06_01',
	'autouse'               => '1.06',
	'base'                  => '2.15',
	'bigint'                => '0.23',
	'bignum'                => '0.23',
	'bigrat'                => '0.23',
	'blib'                  => '1.04',
	'bytes'                 => '1.03',
	'charnames'             => '1.07',
	'constant'              => '1.20',
	'deprecate'             => '0.01',
	'diagnostics'           => '1.19',
	'encoding'              => '2.6_01',
	'encoding::warnings'    => '0.11',
	'feature'               => '1.15',
	'fields'                => '2.15',
	'filetest'              => '1.02',
	'if'                    => '0.05',
	'inc::latest'           => '0.3603',
	'integer'               => '1.00',
	'less'                  => '0.03',
	'lib'                   => '0.62',
	'locale'                => '1.00',
	'mro'                   => '1.02',
	'open'                  => '1.07',
	'ops'                   => '1.02',
	'overload'              => '1.10',
	'overload::numbers'     => undef,
	'overloading'           => '0.01',
	'parent'                => '0.223',
	're'                    => '0.11',
	'sigtrap'               => '1.04',
	'sort'                  => '2.01',
	'strict'                => '1.04',
	'subs'                  => '1.00',
	'threads'               => '1.75',
	'threads::shared'       => '1.32',
	'utf8'                  => '1.07',
	'vars'                  => '1.01',
	'version'               => '0.82',
	'vmsish'                => '1.02',
	'warnings'              => '1.09',
	'warnings::register'    => '1.01',
    },

    5.012000 => {
	'AnyDBM_File'           => '1.00',
	'App::Cpan'             => '1.5701',
	'App::Prove'            => '3.17',
	'App::Prove::State'     => '3.17',
	'App::Prove::State::Result'=> '3.17',
	'App::Prove::State::Result::Test'=> '3.17',
	'Archive::Extract'      => '0.38',
	'Archive::Tar'          => '1.54',
	'Archive::Tar::Constant'=> '0.02',
	'Archive::Tar::File'    => '0.02',
	'Attribute::Handlers'   => '0.87',
	'AutoLoader'            => '5.70',
	'AutoSplit'             => '1.06',
	'B'                     => '1.23',
	'B::Concise'            => '0.78',
	'B::Debug'              => '1.12',
	'B::Deparse'            => '0.96',
	'B::Lint'               => '1.11_01',
	'B::Lint::Debug'        => '0.01',
	'B::Showlex'            => '1.02',
	'B::Terse'              => '1.05',
	'B::Xref'               => '1.02',
	'Benchmark'             => '1.11',
	'CGI'                   => '3.48',
	'CGI::Apache'           => '1.01',
	'CGI::Carp'             => '3.45',
	'CGI::Cookie'           => '1.29',
	'CGI::Fast'             => '1.07',
	'CGI::Pretty'           => '3.46',
	'CGI::Push'             => '1.04',
	'CGI::Switch'           => '1.01',
	'CGI::Util'             => '3.48',
	'CPAN'                  => '1.94_56',
	'CPAN::Author'          => '5.5',
	'CPAN::Bundle'          => '5.5',
	'CPAN::CacheMgr'        => '5.5',
	'CPAN::Complete'        => '5.5',
	'CPAN::Debug'           => '5.5001',
	'CPAN::DeferredCode'    => '5.50',
	'CPAN::Distribution'    => '1.9456_01',
	'CPAN::Distroprefs'     => '6',
	'CPAN::Distrostatus'    => '5.5',
	'CPAN::Exception::RecursiveDependency'=> '5.5',
	'CPAN::Exception::blocked_urllist'=> '1.0',
	'CPAN::Exception::yaml_not_installed'=> '5.5',
	'CPAN::FTP'             => '5.5004',
	'CPAN::FTP::netrc'      => '1.00',
	'CPAN::FirstTime'       => '5.5301',
	'CPAN::HandleConfig'    => '5.5001',
	'CPAN::Index'           => '1.94',
	'CPAN::InfoObj'         => '5.5',
	'CPAN::Kwalify'         => '5.50',
	'CPAN::LWP::UserAgent'  => '1.94',
	'CPAN::Mirrors'         => '1.77',
	'CPAN::Module'          => '5.5',
	'CPAN::Nox'             => '5.50',
	'CPAN::Prompt'          => '5.5',
	'CPAN::Queue'           => '5.5',
	'CPAN::Shell'           => '5.5001',
	'CPAN::Tarzip'          => '5.5011',
	'CPAN::URL'             => '5.5',
	'CPAN::Version'         => '5.5',
	'CPANPLUS'              => '0.90',
	'CPANPLUS::Backend'     => undef,
	'CPANPLUS::Backend::RV' => undef,
	'CPANPLUS::Config'      => undef,
	'CPANPLUS::Configure'   => undef,
	'CPANPLUS::Configure::Setup'=> undef,
	'CPANPLUS::Dist'        => undef,
	'CPANPLUS::Dist::Autobundle'=> undef,
	'CPANPLUS::Dist::Base'  => undef,
	'CPANPLUS::Dist::Build' => '0.46',
	'CPANPLUS::Dist::Build::Constants'=> '0.46',
	'CPANPLUS::Dist::MM'    => undef,
	'CPANPLUS::Dist::Sample'=> undef,
	'CPANPLUS::Error'       => undef,
	'CPANPLUS::Internals'   => '0.90',
	'CPANPLUS::Internals::Constants'=> undef,
	'CPANPLUS::Internals::Constants::Report'=> undef,
	'CPANPLUS::Internals::Extract'=> undef,
	'CPANPLUS::Internals::Fetch'=> undef,
	'CPANPLUS::Internals::Report'=> undef,
	'CPANPLUS::Internals::Search'=> undef,
	'CPANPLUS::Internals::Source'=> undef,
	'CPANPLUS::Internals::Source::Memory'=> undef,
	'CPANPLUS::Internals::Source::SQLite'=> undef,
	'CPANPLUS::Internals::Source::SQLite::Tie'=> undef,
	'CPANPLUS::Internals::Utils'=> undef,
	'CPANPLUS::Internals::Utils::Autoflush'=> undef,
	'CPANPLUS::Module'      => undef,
	'CPANPLUS::Module::Author'=> undef,
	'CPANPLUS::Module::Author::Fake'=> undef,
	'CPANPLUS::Module::Checksums'=> undef,
	'CPANPLUS::Module::Fake'=> undef,
	'CPANPLUS::Module::Signature'=> undef,
	'CPANPLUS::Selfupdate'  => undef,
	'CPANPLUS::Shell'       => undef,
	'CPANPLUS::Shell::Classic'=> '0.0562',
	'CPANPLUS::Shell::Default'=> '0.90',
	'CPANPLUS::Shell::Default::Plugins::CustomSource'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Remote'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Source'=> undef,
	'Carp'                  => '1.15',
	'Carp::Heavy'           => '1.15',
	'Class::ISA'            => '0.36',
	'Class::Struct'         => '0.63',
	'Compress::Raw::Bzip2'  => '2.024',
	'Compress::Raw::Zlib'   => '2.024',
	'Compress::Zlib'        => '2.024',
	'Config'                => undef,
	'Config::Extensions'    => '0.01',
	'Cwd'                   => '3.31',
	'DB'                    => '1.02',
	'DBM_Filter'            => '0.03',
	'DBM_Filter::compress'  => '0.02',
	'DBM_Filter::encode'    => '0.02',
	'DBM_Filter::int32'     => '0.02',
	'DBM_Filter::null'      => '0.02',
	'DBM_Filter::utf8'      => '0.02',
	'DB_File'               => '1.820',
	'Data::Dumper'          => '2.125',
	'Devel::DProf'          => '20080331.00',
	'Devel::DProf::dprof::V'=> undef,
	'Devel::InnerPackage'   => '0.3',
	'Devel::PPPort'         => '3.19',
	'Devel::Peek'           => '1.04',
	'Devel::SelfStubber'    => '1.03',
	'Digest'                => '1.16',
	'Digest::MD5'           => '2.39',
	'Digest::SHA'           => '5.47',
	'Digest::base'          => '1.16',
	'Digest::file'          => '1.16',
	'DirHandle'             => '1.03',
	'Dumpvalue'             => '1.13',
	'DynaLoader'            => '1.10',
	'Encode'                => '2.39',
	'Encode::Alias'         => '2.12',
	'Encode::Byte'          => '2.04',
	'Encode::CJKConstants'  => '2.02',
	'Encode::CN'            => '2.03',
	'Encode::CN::HZ'        => '2.05',
	'Encode::Config'        => '2.05',
	'Encode::EBCDIC'        => '2.02',
	'Encode::Encoder'       => '2.01',
	'Encode::Encoding'      => '2.05',
	'Encode::GSM0338'       => '2.01',
	'Encode::Guess'         => '2.03',
	'Encode::JP'            => '2.04',
	'Encode::JP::H2Z'       => '2.02',
	'Encode::JP::JIS7'      => '2.04',
	'Encode::KR'            => '2.03',
	'Encode::KR::2022_KR'   => '2.02',
	'Encode::MIME::Header'  => '2.11',
	'Encode::MIME::Header::ISO_2022_JP'=> '1.03',
	'Encode::MIME::Name'    => '1.01',
	'Encode::Symbol'        => '2.02',
	'Encode::TW'            => '2.03',
	'Encode::Unicode'       => '2.07',
	'Encode::Unicode::UTF7' => '2.04',
	'English'               => '1.04',
	'Env'                   => '1.01',
	'Errno'                 => '1.11',
	'Exporter'              => '5.64_01',
	'Exporter::Heavy'       => '5.64_01',
	'ExtUtils::CBuilder'    => '0.27',
	'ExtUtils::CBuilder::Base'=> '0.27',
	'ExtUtils::CBuilder::Platform::Unix'=> '0.27',
	'ExtUtils::CBuilder::Platform::VMS'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.27',
	'ExtUtils::CBuilder::Platform::aix'=> '0.27',
	'ExtUtils::CBuilder::Platform::cygwin'=> '0.27',
	'ExtUtils::CBuilder::Platform::darwin'=> '0.27',
	'ExtUtils::CBuilder::Platform::dec_osf'=> '0.27',
	'ExtUtils::CBuilder::Platform::os2'=> '0.27',
	'ExtUtils::Command'     => '1.16',
	'ExtUtils::Command::MM' => '6.56',
	'ExtUtils::Constant'    => '0.22',
	'ExtUtils::Constant::Base'=> '0.04',
	'ExtUtils::Constant::ProxySubs'=> '0.06',
	'ExtUtils::Constant::Utils'=> '0.02',
	'ExtUtils::Constant::XS'=> '0.03',
	'ExtUtils::Embed'       => '1.28',
	'ExtUtils::Install'     => '1.55',
	'ExtUtils::Installed'   => '1.999_001',
	'ExtUtils::Liblist'     => '6.56',
	'ExtUtils::Liblist::Kid'=> '6.56',
	'ExtUtils::MM'          => '6.56',
	'ExtUtils::MM_AIX'      => '6.56',
	'ExtUtils::MM_Any'      => '6.56',
	'ExtUtils::MM_BeOS'     => '6.56',
	'ExtUtils::MM_Cygwin'   => '6.56',
	'ExtUtils::MM_DOS'      => '6.56',
	'ExtUtils::MM_Darwin'   => '6.56',
	'ExtUtils::MM_MacOS'    => '6.56',
	'ExtUtils::MM_NW5'      => '6.56',
	'ExtUtils::MM_OS2'      => '6.56',
	'ExtUtils::MM_QNX'      => '6.56',
	'ExtUtils::MM_UWIN'     => '6.56',
	'ExtUtils::MM_Unix'     => '6.56',
	'ExtUtils::MM_VMS'      => '6.56',
	'ExtUtils::MM_VOS'      => '6.56',
	'ExtUtils::MM_Win32'    => '6.56',
	'ExtUtils::MM_Win95'    => '6.56',
	'ExtUtils::MY'          => '6.56',
	'ExtUtils::MakeMaker'   => '6.56',
	'ExtUtils::MakeMaker::Config'=> '6.56',
	'ExtUtils::Manifest'    => '1.57',
	'ExtUtils::Miniperl'    => undef,
	'ExtUtils::Mkbootstrap' => '6.56',
	'ExtUtils::Mksymlists'  => '6.56',
	'ExtUtils::Packlist'    => '1.44',
	'ExtUtils::ParseXS'     => '2.21',
	'ExtUtils::XSSymSet'    => '1.1',
	'ExtUtils::testlib'     => '6.56',
	'Fatal'                 => '2.06_01',
	'Fcntl'                 => '1.06',
	'File::Basename'        => '2.78',
	'File::CheckTree'       => '4.4',
	'File::Compare'         => '1.1006',
	'File::Copy'            => '2.17',
	'File::DosGlob'         => '1.01',
	'File::Fetch'           => '0.24',
	'File::Find'            => '1.15',
	'File::Glob'            => '1.07',
	'File::GlobMapper'      => '1.000',
	'File::Path'            => '2.08_01',
	'File::Spec'            => '3.31',
	'File::Spec::Cygwin'    => '3.30',
	'File::Spec::Epoc'      => '3.30',
	'File::Spec::Functions' => '3.30',
	'File::Spec::Mac'       => '3.30',
	'File::Spec::OS2'       => '3.30',
	'File::Spec::Unix'      => '3.30',
	'File::Spec::VMS'       => '3.30',
	'File::Spec::Win32'     => '3.30',
	'File::Temp'            => '0.22',
	'File::stat'            => '1.02',
	'FileCache'             => '1.08',
	'FileHandle'            => '2.02',
	'VMS::Filespec'         => '1.12',
	'Filter::Simple'        => '0.84',
	'Filter::Util::Call'    => '1.08',
	'FindBin'               => '1.50',
	'GDBM_File'             => '1.10',
	'Getopt::Long'          => '2.38',
	'Getopt::Std'           => '1.06',
	'Hash::Util'            => '0.07',
	'Hash::Util::FieldHash' => '1.04',
	'I18N::Collate'         => '1.01',
	'I18N::LangTags'        => '0.35',
	'I18N::LangTags::Detect'=> '1.04',
	'I18N::LangTags::List'  => '0.35',
	'I18N::Langinfo'        => '0.03',
	'IO'                    => '1.25_02',
	'IO::Compress::Adapter::Bzip2'=> '2.024',
	'IO::Compress::Adapter::Deflate'=> '2.024',
	'IO::Compress::Adapter::Identity'=> '2.024',
	'IO::Compress::Base'    => '2.024',
	'IO::Compress::Base::Common'=> '2.024',
	'IO::Compress::Bzip2'   => '2.024',
	'IO::Compress::Deflate' => '2.024',
	'IO::Compress::Gzip'    => '2.024',
	'IO::Compress::Gzip::Constants'=> '2.024',
	'IO::Compress::RawDeflate'=> '2.024',
	'IO::Compress::Zip'     => '2.024',
	'IO::Compress::Zip::Constants'=> '2.024',
	'IO::Compress::Zlib::Constants'=> '2.024',
	'IO::Compress::Zlib::Extra'=> '2.024',
	'IO::Dir'               => '1.07',
	'IO::File'              => '1.14',
	'IO::Handle'            => '1.28',
	'IO::Pipe'              => '1.13',
	'IO::Poll'              => '0.07',
	'IO::Seekable'          => '1.10',
	'IO::Select'            => '1.17',
	'IO::Socket'            => '1.31',
	'IO::Socket::INET'      => '1.31',
	'IO::Socket::UNIX'      => '1.23',
	'IO::Uncompress::Adapter::Bunzip2'=> '2.024',
	'IO::Uncompress::Adapter::Identity'=> '2.024',
	'IO::Uncompress::Adapter::Inflate'=> '2.024',
	'IO::Uncompress::AnyInflate'=> '2.024',
	'IO::Uncompress::AnyUncompress'=> '2.024',
	'IO::Uncompress::Base'  => '2.024',
	'IO::Uncompress::Bunzip2'=> '2.024',
	'IO::Uncompress::Gunzip'=> '2.024',
	'IO::Uncompress::Inflate'=> '2.024',
	'IO::Uncompress::RawInflate'=> '2.024',
	'IO::Uncompress::Unzip' => '2.024',
	'IO::Zlib'              => '1.10',
	'IPC::Cmd'              => '0.54',
	'IPC::Msg'              => '2.01',
	'IPC::Open2'            => '1.03',
	'IPC::Open3'            => '1.05',
	'IPC::Semaphore'        => '2.01',
	'IPC::SharedMem'        => '2.01',
	'IPC::SysV'             => '2.01',
	'List::Util'            => '1.22',
	'List::Util::PP'        => '1.22',
	'List::Util::XS'        => '1.22',
	'Locale::Constants'     => '2.07',
	'Locale::Country'       => '2.07',
	'Locale::Currency'      => '2.07',
	'Locale::Language'      => '2.07',
	'Locale::Maketext'      => '1.14',
	'Locale::Maketext::Guts'=> '1.13',
	'Locale::Maketext::GutsLoader'=> '1.13',
	'Locale::Maketext::Simple'=> '0.21',
	'Locale::Script'        => '2.07',
	'Log::Message'          => '0.02',
	'Log::Message::Config'  => '0.01',
	'Log::Message::Handlers'=> undef,
	'Log::Message::Item'    => undef,
	'Log::Message::Simple'  => '0.06',
	'MIME::Base64'          => '3.08',
	'MIME::QuotedPrint'     => '3.08',
	'Math::BigFloat'        => '1.60',
	'Math::BigFloat::Trace' => '0.01',
	'Math::BigInt'          => '1.89_01',
	'Math::BigInt::Calc'    => '0.52',
	'Math::BigInt::CalcEmu' => '0.05',
	'Math::BigInt::FastCalc'=> '0.19',
	'Math::BigInt::Trace'   => '0.01',
	'Math::BigRat'          => '0.24',
	'Math::Complex'         => '1.56',
	'Math::Trig'            => '1.2',
	'Memoize'               => '1.01_03',
	'Memoize::AnyDBM_File'  => '0.65',
	'Memoize::Expire'       => '1.00',
	'Memoize::ExpireFile'   => '1.01',
	'Memoize::ExpireTest'   => '0.65',
	'Memoize::NDBM_File'    => '0.65',
	'Memoize::SDBM_File'    => '0.65',
	'Memoize::Storable'     => '0.65',
	'Module::Build'         => '0.3603',
	'Module::Build::Base'   => '0.3603',
	'Module::Build::Compat' => '0.3603',
	'Module::Build::Config' => '0.3603',
	'Module::Build::ConfigData'=> undef,
	'Module::Build::Cookbook'=> '0.3603',
	'Module::Build::Dumper' => '0.3603',
	'Module::Build::ModuleInfo'=> '0.3603',
	'Module::Build::Notes'  => '0.3603',
	'Module::Build::PPMMaker'=> '0.3603',
	'Module::Build::Platform::Amiga'=> '0.3603',
	'Module::Build::Platform::Default'=> '0.3603',
	'Module::Build::Platform::EBCDIC'=> '0.3603',
	'Module::Build::Platform::MPEiX'=> '0.3603',
	'Module::Build::Platform::MacOS'=> '0.3603',
	'Module::Build::Platform::RiscOS'=> '0.3603',
	'Module::Build::Platform::Unix'=> '0.3603',
	'Module::Build::Platform::VMS'=> '0.3603',
	'Module::Build::Platform::VOS'=> '0.3603',
	'Module::Build::Platform::Windows'=> '0.3603',
	'Module::Build::Platform::aix'=> '0.3603',
	'Module::Build::Platform::cygwin'=> '0.3603',
	'Module::Build::Platform::darwin'=> '0.3603',
	'Module::Build::Platform::os2'=> '0.3603',
	'Module::Build::PodParser'=> '0.3603',
	'Module::Build::Version'=> '0.77',
	'Module::Build::YAML'   => '1.40',
	'Module::CoreList'      => '2.29',
	'Module::Load'          => '0.16',
	'Module::Load::Conditional'=> '0.34',
	'Module::Loaded'        => '0.06',
	'Module::Pluggable'     => '3.9',
	'Module::Pluggable::Object'=> '3.9',
	'Moped::Msg'            => '0.01',
	'NDBM_File'             => '1.08',
	'NEXT'                  => '0.64',
	'Net::Cmd'              => '2.29',
	'Net::Config'           => '1.11',
	'Net::Domain'           => '2.20',
	'Net::FTP'              => '2.77',
	'Net::FTP::A'           => '1.18',
	'Net::FTP::E'           => '0.01',
	'Net::FTP::I'           => '1.12',
	'Net::FTP::L'           => '0.01',
	'Net::FTP::dataconn'    => '0.11',
	'Net::NNTP'             => '2.24',
	'Net::Netrc'            => '2.12',
	'Net::POP3'             => '2.29',
	'Net::Ping'             => '2.36',
	'Net::SMTP'             => '2.31',
	'Net::Time'             => '2.10',
	'Net::hostent'          => '1.01',
	'Net::netent'           => '1.00',
	'Net::protoent'         => '1.00',
	'Net::servent'          => '1.01',
	'O'                     => '1.01',
	'ODBM_File'             => '1.07',
	'Object::Accessor'      => '0.36',
	'Opcode'                => '1.15',
	'POSIX'                 => '1.19',
	'Package::Constants'    => '0.02',
	'Params::Check'         => '0.26',
	'Parse::CPAN::Meta'     => '1.40',
	'PerlIO'                => '1.06',
	'PerlIO::encoding'      => '0.12',
	'PerlIO::scalar'        => '0.07',
	'PerlIO::via'           => '0.09',
	'PerlIO::via::QuotedPrint'=> '0.06',
	'Pod::Checker'          => '1.45',
	'Pod::Escapes'          => '1.04',
	'Pod::Find'             => '1.35',
	'Pod::Functions'        => '1.03',
	'Pod::Html'             => '1.09',
	'Pod::InputObjects'     => '1.31',
	'Pod::LaTeX'            => '0.58',
	'Pod::Man'              => '2.23',
	'Pod::ParseLink'        => '1.10',
	'Pod::ParseUtils'       => '1.36',
	'Pod::Parser'           => '1.37',
	'Pod::Perldoc'          => '3.15_02',
	'Pod::Perldoc::BaseTo'  => undef,
	'Pod::Perldoc::GetOptsOO'=> undef,
	'Pod::Perldoc::ToChecker'=> undef,
	'Pod::Perldoc::ToMan'   => undef,
	'Pod::Perldoc::ToNroff' => undef,
	'Pod::Perldoc::ToPod'   => undef,
	'Pod::Perldoc::ToRtf'   => undef,
	'Pod::Perldoc::ToText'  => undef,
	'Pod::Perldoc::ToTk'    => undef,
	'Pod::Perldoc::ToXml'   => undef,
	'Pod::PlainText'        => '2.04',
	'Pod::Plainer'          => '1.02',
	'Pod::Select'           => '1.36',
	'Pod::Simple'           => '3.13',
	'Pod::Simple::BlackBox' => '3.13',
	'Pod::Simple::Checker'  => '3.13',
	'Pod::Simple::Debug'    => '3.13',
	'Pod::Simple::DumpAsText'=> '3.13',
	'Pod::Simple::DumpAsXML'=> '3.13',
	'Pod::Simple::HTML'     => '3.13',
	'Pod::Simple::HTMLBatch'=> '3.13',
	'Pod::Simple::HTMLLegacy'=> '5.01',
	'Pod::Simple::LinkSection'=> '3.13',
	'Pod::Simple::Methody'  => '3.13',
	'Pod::Simple::Progress' => '3.13',
	'Pod::Simple::PullParser'=> '3.13',
	'Pod::Simple::PullParserEndToken'=> '3.13',
	'Pod::Simple::PullParserStartToken'=> '3.13',
	'Pod::Simple::PullParserTextToken'=> '3.13',
	'Pod::Simple::PullParserToken'=> '3.13',
	'Pod::Simple::RTF'      => '3.13',
	'Pod::Simple::Search'   => '3.13',
	'Pod::Simple::SimpleTree'=> '3.13',
	'Pod::Simple::Text'     => '3.13',
	'Pod::Simple::TextContent'=> '3.13',
	'Pod::Simple::TiedOutFH'=> '3.13',
	'Pod::Simple::Transcode'=> '3.13',
	'Pod::Simple::TranscodeDumb'=> '3.13',
	'Pod::Simple::TranscodeSmart'=> '3.13',
	'Pod::Simple::XHTML'    => '3.13',
	'Pod::Simple::XMLOutStream'=> '3.13',
	'Pod::Text'             => '3.14',
	'Pod::Text::Color'      => '2.06',
	'Pod::Text::Overstrike' => '2.04',
	'Pod::Text::Termcap'    => '2.06',
	'Pod::Usage'            => '1.36',
	'SDBM_File'             => '1.06',
	'Safe'                  => '2.25',
	'Scalar::Util'          => '1.22',
	'Scalar::Util::PP'      => '1.22',
	'Search::Dict'          => '1.02',
	'SelectSaver'           => '1.02',
	'SelfLoader'            => '1.17',
	'Shell'                 => '0.72_01',
	'Socket'                => '1.87',
	'Storable'              => '2.22',
	'Switch'                => '2.16',
	'Symbol'                => '1.07',
	'Sys::Hostname'         => '1.11',
	'Sys::Syslog'           => '0.27',
	'Sys::Syslog::win32::Win32'=> undef,
	'TAP::Base'             => '3.17',
	'TAP::Formatter::Base'  => '3.17',
	'TAP::Formatter::Color' => '3.17',
	'TAP::Formatter::Console'=> '3.17',
	'TAP::Formatter::Console::ParallelSession'=> '3.17',
	'TAP::Formatter::Console::Session'=> '3.17',
	'TAP::Formatter::File'  => '3.17',
	'TAP::Formatter::File::Session'=> '3.17',
	'TAP::Formatter::Session'=> '3.17',
	'TAP::Harness'          => '3.17',
	'TAP::Object'           => '3.17',
	'TAP::Parser'           => '3.17',
	'TAP::Parser::Aggregator'=> '3.17',
	'TAP::Parser::Grammar'  => '3.17',
	'TAP::Parser::Iterator' => '3.17',
	'TAP::Parser::Iterator::Array'=> '3.17',
	'TAP::Parser::Iterator::Process'=> '3.17',
	'TAP::Parser::Iterator::Stream'=> '3.17',
	'TAP::Parser::IteratorFactory'=> '3.17',
	'TAP::Parser::Multiplexer'=> '3.17',
	'TAP::Parser::Result'   => '3.17',
	'TAP::Parser::Result::Bailout'=> '3.17',
	'TAP::Parser::Result::Comment'=> '3.17',
	'TAP::Parser::Result::Plan'=> '3.17',
	'TAP::Parser::Result::Pragma'=> '3.17',
	'TAP::Parser::Result::Test'=> '3.17',
	'TAP::Parser::Result::Unknown'=> '3.17',
	'TAP::Parser::Result::Version'=> '3.17',
	'TAP::Parser::Result::YAML'=> '3.17',
	'TAP::Parser::ResultFactory'=> '3.17',
	'TAP::Parser::Scheduler'=> '3.17',
	'TAP::Parser::Scheduler::Job'=> '3.17',
	'TAP::Parser::Scheduler::Spinner'=> '3.17',
	'TAP::Parser::Source'   => '3.17',
	'TAP::Parser::Source::Perl'=> '3.17',
	'TAP::Parser::Utils'    => '3.17',
	'TAP::Parser::YAMLish::Reader'=> '3.17',
	'TAP::Parser::YAMLish::Writer'=> '3.17',
	'Term::ANSIColor'       => '2.02',
	'Term::Cap'             => '1.12',
	'Term::Complete'        => '1.402',
	'Term::ReadLine'        => '1.05',
	'Term::UI'              => '0.20',
	'Term::UI::History'     => undef,
	'Test'                  => '1.25_02',
	'Test::Builder'         => '0.94',
	'Test::Builder::Module' => '0.94',
	'Test::Builder::Tester' => '1.18',
	'Test::Builder::Tester::Color'=> '1.18',
	'Test::Harness'         => '3.17',
	'Test::More'            => '0.94',
	'Test::Simple'          => '0.94',
	'Text::Abbrev'          => '1.01',
	'Text::Balanced'        => '2.02',
	'Text::ParseWords'      => '3.27',
	'Text::Soundex'         => '3.03_01',
	'Text::Tabs'            => '2009.0305',
	'Text::Wrap'            => '2009.0305',
	'Thread'                => '3.02',
	'Thread::Queue'         => '2.11',
	'Thread::Semaphore'     => '2.09',
	'Tie::Array'            => '1.03',
	'Tie::File'             => '0.97_02',
	'Tie::Handle'           => '4.2',
	'Tie::Hash'             => '1.03',
	'Tie::Hash::NamedCapture'=> '0.06',
	'Tie::Memoize'          => '1.1',
	'Tie::RefHash'          => '1.38',
	'Tie::Scalar'           => '1.02',
	'Tie::StdHandle'        => '4.2',
	'Tie::SubstrHash'       => '1.00',
	'Time::HiRes'           => '1.9719',
	'Time::Local'           => '1.1901_01',
	'Time::Piece'           => '1.15_01',
	'Time::Piece::Seconds'  => undef,
	'Time::Seconds'         => undef,
	'Time::gmtime'          => '1.03',
	'Time::localtime'       => '1.02',
	'Time::tm'              => '1.00',
	'UNIVERSAL'             => '1.06',
	'Unicode'               => '5.2.0',
	'Unicode::Collate'      => '0.52_01',
	'Unicode::Normalize'    => '1.03',
	'Unicode::UCD'          => '0.27',
	'User::grent'           => '1.01',
	'User::pwent'           => '1.00',
	'VMS::DCLsym'           => '1.03',
	'VMS::Stdio'            => '2.4',
	'Win32'                 => '0.39',
	'Win32API::File'        => '0.1101',
	'Win32API::File::ExtUtils::Myconst2perl'=> '1',
	'Win32CORE'             => '0.02',
	'XS::APItest'           => '0.17',
	'XS::APItest::KeywordRPN'=> '0.003',
	'XS::Typemap'           => '0.03',
	'XSLoader'              => '0.10',
	'XSLoader::XSLoader'    => '0.10',
	'attributes'            => '0.12',
	'autodie'               => '2.06_01',
	'autodie::exception'    => '2.06_01',
	'autodie::exception::system'=> '2.06_01',
	'autodie::hints'        => '2.06_01',
	'autouse'               => '1.06',
	'base'                  => '2.15',
	'bigint'                => '0.23',
	'bignum'                => '0.23',
	'bigrat'                => '0.23',
	'blib'                  => '1.04',
	'bytes'                 => '1.04',
	'charnames'             => '1.07',
	'constant'              => '1.20',
	'deprecate'             => '0.01',
	'diagnostics'           => '1.19',
	'encoding'              => '2.6_01',
	'encoding::warnings'    => '0.11',
	'feature'               => '1.16',
	'fields'                => '2.15',
	'filetest'              => '1.02',
	'if'                    => '0.05',
	'inc::latest'           => '0.3603',
	'integer'               => '1.00',
	'less'                  => '0.03',
	'lib'                   => '0.62',
	'locale'                => '1.00',
	'mro'                   => '1.02',
	'open'                  => '1.07',
	'ops'                   => '1.02',
	'overload'              => '1.10',
	'overload::numbers'     => undef,
	'overloading'           => '0.01',
	'parent'                => '0.223',
	're'                    => '0.11',
	'sigtrap'               => '1.04',
	'sort'                  => '2.01',
	'strict'                => '1.04',
	'subs'                  => '1.00',
	'threads'               => '1.75',
	'threads::shared'       => '1.32',
	'utf8'                  => '1.08',
	'vars'                  => '1.01',
	'version'               => '0.82',
	'vmsish'                => '1.02',
	'warnings'              => '1.09',
	'warnings::register'    => '1.01',
    },

    5.013000 => {
	'AnyDBM_File'           => '1.00',
	'App::Cpan'             => '1.5701',
	'App::Prove'            => '3.17',
	'App::Prove::State'     => '3.17',
	'App::Prove::State::Result'=> '3.17',
	'App::Prove::State::Result::Test'=> '3.17',
	'Archive::Extract'      => '0.38',
	'Archive::Tar'          => '1.54',
	'Archive::Tar::Constant'=> '0.02',
	'Archive::Tar::File'    => '0.02',
	'Attribute::Handlers'   => '0.87',
	'AutoLoader'            => '5.70',
	'AutoSplit'             => '1.06',
	'B'                     => '1.23',
	'B::Concise'            => '0.78',
	'B::Debug'              => '1.12',
	'B::Deparse'            => '0.96',
	'B::Lint'               => '1.11_01',
	'B::Lint::Debug'        => '0.01',
	'B::Showlex'            => '1.02',
	'B::Terse'              => '1.05',
	'B::Xref'               => '1.02',
	'Benchmark'             => '1.11',
	'CGI'                   => '3.49',
	'CGI::Apache'           => '1.01',
	'CGI::Carp'             => '3.45',
	'CGI::Cookie'           => '1.29',
	'CGI::Fast'             => '1.08',
	'CGI::Pretty'           => '3.46',
	'CGI::Push'             => '1.04',
	'CGI::Switch'           => '1.01',
	'CGI::Util'             => '3.48',
	'CPAN'                  => '1.94_56',
	'CPAN::Author'          => '5.5',
	'CPAN::Bundle'          => '5.5',
	'CPAN::CacheMgr'        => '5.5',
	'CPAN::Complete'        => '5.5',
	'CPAN::Debug'           => '5.5001',
	'CPAN::DeferredCode'    => '5.50',
	'CPAN::Distribution'    => '1.9456_01',
	'CPAN::Distroprefs'     => '6',
	'CPAN::Distrostatus'    => '5.5',
	'CPAN::Exception::RecursiveDependency'=> '5.5',
	'CPAN::Exception::blocked_urllist'=> '1.0',
	'CPAN::Exception::yaml_not_installed'=> '5.5',
	'CPAN::FTP'             => '5.5004',
	'CPAN::FTP::netrc'      => '1.00',
	'CPAN::FirstTime'       => '5.5301',
	'CPAN::HandleConfig'    => '5.5001',
	'CPAN::Index'           => '1.94',
	'CPAN::InfoObj'         => '5.5',
	'CPAN::Kwalify'         => '5.50',
	'CPAN::LWP::UserAgent'  => '1.94',
	'CPAN::Mirrors'         => '1.77',
	'CPAN::Module'          => '5.5',
	'CPAN::Nox'             => '5.50',
	'CPAN::Prompt'          => '5.5',
	'CPAN::Queue'           => '5.5',
	'CPAN::Shell'           => '5.5001',
	'CPAN::Tarzip'          => '5.5011',
	'CPAN::URL'             => '5.5',
	'CPAN::Version'         => '5.5',
	'CPANPLUS'              => '0.90',
	'CPANPLUS::Backend'     => undef,
	'CPANPLUS::Backend::RV' => undef,
	'CPANPLUS::Config'      => undef,
	'CPANPLUS::Configure'   => undef,
	'CPANPLUS::Configure::Setup'=> undef,
	'CPANPLUS::Dist'        => undef,
	'CPANPLUS::Dist::Autobundle'=> undef,
	'CPANPLUS::Dist::Base'  => undef,
	'CPANPLUS::Dist::Build' => '0.46',
	'CPANPLUS::Dist::Build::Constants'=> '0.46',
	'CPANPLUS::Dist::MM'    => undef,
	'CPANPLUS::Dist::Sample'=> undef,
	'CPANPLUS::Error'       => undef,
	'CPANPLUS::Internals'   => '0.90',
	'CPANPLUS::Internals::Constants'=> undef,
	'CPANPLUS::Internals::Constants::Report'=> undef,
	'CPANPLUS::Internals::Extract'=> undef,
	'CPANPLUS::Internals::Fetch'=> undef,
	'CPANPLUS::Internals::Report'=> undef,
	'CPANPLUS::Internals::Search'=> undef,
	'CPANPLUS::Internals::Source'=> undef,
	'CPANPLUS::Internals::Source::Memory'=> undef,
	'CPANPLUS::Internals::Source::SQLite'=> undef,
	'CPANPLUS::Internals::Source::SQLite::Tie'=> undef,
	'CPANPLUS::Internals::Utils'=> undef,
	'CPANPLUS::Internals::Utils::Autoflush'=> undef,
	'CPANPLUS::Module'      => undef,
	'CPANPLUS::Module::Author'=> undef,
	'CPANPLUS::Module::Author::Fake'=> undef,
	'CPANPLUS::Module::Checksums'=> undef,
	'CPANPLUS::Module::Fake'=> undef,
	'CPANPLUS::Module::Signature'=> undef,
	'CPANPLUS::Selfupdate'  => undef,
	'CPANPLUS::Shell'       => undef,
	'CPANPLUS::Shell::Classic'=> '0.0562',
	'CPANPLUS::Shell::Default'=> '0.90',
	'CPANPLUS::Shell::Default::Plugins::CustomSource'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Remote'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Source'=> undef,
	'Carp'                  => '1.15',
	'Carp::Heavy'           => '1.15',
	'Class::ISA'            => '0.36',
	'Class::Struct'         => '0.63',
	'Compress::Raw::Bzip2'  => '2.024',
	'Compress::Raw::Zlib'   => '2.024',
	'Compress::Zlib'        => '2.024',
	'Config'                => undef,
	'Config::Extensions'    => '0.01',
	'Cwd'                   => '3.31',
	'DB'                    => '1.02',
	'DBM_Filter'            => '0.03',
	'DBM_Filter::compress'  => '0.02',
	'DBM_Filter::encode'    => '0.02',
	'DBM_Filter::int32'     => '0.02',
	'DBM_Filter::null'      => '0.02',
	'DBM_Filter::utf8'      => '0.02',
	'DB_File'               => '1.820',
	'Data::Dumper'          => '2.126',
	'Devel::DProf'          => '20080331.00',
	'Devel::DProf::dprof::V'=> undef,
	'Devel::InnerPackage'   => '0.3',
	'Devel::PPPort'         => '3.19',
	'Devel::Peek'           => '1.04',
	'Devel::SelfStubber'    => '1.03',
	'Digest'                => '1.16',
	'Digest::MD5'           => '2.39',
	'Digest::SHA'           => '5.47',
	'Digest::base'          => '1.16',
	'Digest::file'          => '1.16',
	'DirHandle'             => '1.03',
	'Dumpvalue'             => '1.13',
	'DynaLoader'            => '1.10',
	'Encode'                => '2.39',
	'Encode::Alias'         => '2.12',
	'Encode::Byte'          => '2.04',
	'Encode::CJKConstants'  => '2.02',
	'Encode::CN'            => '2.03',
	'Encode::CN::HZ'        => '2.05',
	'Encode::Config'        => '2.05',
	'Encode::EBCDIC'        => '2.02',
	'Encode::Encoder'       => '2.01',
	'Encode::Encoding'      => '2.05',
	'Encode::GSM0338'       => '2.01',
	'Encode::Guess'         => '2.03',
	'Encode::JP'            => '2.04',
	'Encode::JP::H2Z'       => '2.02',
	'Encode::JP::JIS7'      => '2.04',
	'Encode::KR'            => '2.03',
	'Encode::KR::2022_KR'   => '2.02',
	'Encode::MIME::Header'  => '2.11',
	'Encode::MIME::Header::ISO_2022_JP'=> '1.03',
	'Encode::MIME::Name'    => '1.01',
	'Encode::Symbol'        => '2.02',
	'Encode::TW'            => '2.03',
	'Encode::Unicode'       => '2.07',
	'Encode::Unicode::UTF7' => '2.04',
	'English'               => '1.04',
	'Env'                   => '1.01',
	'Errno'                 => '1.11',
	'Exporter'              => '5.64_01',
	'Exporter::Heavy'       => '5.64_01',
	'ExtUtils::CBuilder'    => '0.27',
	'ExtUtils::CBuilder::Base'=> '0.27',
	'ExtUtils::CBuilder::Platform::Unix'=> '0.27',
	'ExtUtils::CBuilder::Platform::VMS'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.27',
	'ExtUtils::CBuilder::Platform::aix'=> '0.27',
	'ExtUtils::CBuilder::Platform::cygwin'=> '0.27',
	'ExtUtils::CBuilder::Platform::darwin'=> '0.27',
	'ExtUtils::CBuilder::Platform::dec_osf'=> '0.27',
	'ExtUtils::CBuilder::Platform::os2'=> '0.27',
	'ExtUtils::Command'     => '1.16',
	'ExtUtils::Command::MM' => '6.56',
	'ExtUtils::Constant'    => '0.22',
	'ExtUtils::Constant::Base'=> '0.04',
	'ExtUtils::Constant::ProxySubs'=> '0.06',
	'ExtUtils::Constant::Utils'=> '0.02',
	'ExtUtils::Constant::XS'=> '0.03',
	'ExtUtils::Embed'       => '1.28',
	'ExtUtils::Install'     => '1.55',
	'ExtUtils::Installed'   => '1.999_001',
	'ExtUtils::Liblist'     => '6.56',
	'ExtUtils::Liblist::Kid'=> '6.56',
	'ExtUtils::MM'          => '6.56',
	'ExtUtils::MM_AIX'      => '6.56',
	'ExtUtils::MM_Any'      => '6.56',
	'ExtUtils::MM_BeOS'     => '6.56',
	'ExtUtils::MM_Cygwin'   => '6.56',
	'ExtUtils::MM_DOS'      => '6.56',
	'ExtUtils::MM_Darwin'   => '6.56',
	'ExtUtils::MM_MacOS'    => '6.56',
	'ExtUtils::MM_NW5'      => '6.56',
	'ExtUtils::MM_OS2'      => '6.56',
	'ExtUtils::MM_QNX'      => '6.56',
	'ExtUtils::MM_UWIN'     => '6.56',
	'ExtUtils::MM_Unix'     => '6.5601',
	'ExtUtils::MM_VMS'      => '6.56',
	'ExtUtils::MM_VOS'      => '6.56',
	'ExtUtils::MM_Win32'    => '6.56',
	'ExtUtils::MM_Win95'    => '6.56',
	'ExtUtils::MY'          => '6.56',
	'ExtUtils::MakeMaker'   => '6.5601',
	'ExtUtils::MakeMaker::Config'=> '6.56',
	'ExtUtils::Manifest'    => '1.57',
	'ExtUtils::Miniperl'    => undef,
	'ExtUtils::Mkbootstrap' => '6.56',
	'ExtUtils::Mksymlists'  => '6.56',
	'ExtUtils::Packlist'    => '1.44',
	'ExtUtils::ParseXS'     => '2.21',
	'ExtUtils::XSSymSet'    => '1.1',
	'ExtUtils::testlib'     => '6.56',
	'Fatal'                 => '2.06_01',
	'Fcntl'                 => '1.06',
	'File::Basename'        => '2.78',
	'File::CheckTree'       => '4.4',
	'File::Compare'         => '1.1006',
	'File::Copy'            => '2.18',
	'File::DosGlob'         => '1.01',
	'File::Fetch'           => '0.24',
	'File::Find'            => '1.15',
	'File::Glob'            => '1.07',
	'File::GlobMapper'      => '1.000',
	'File::Path'            => '2.08_01',
	'File::Spec'            => '3.31',
	'File::Spec::Cygwin'    => '3.30',
	'File::Spec::Epoc'      => '3.30',
	'File::Spec::Functions' => '3.30',
	'File::Spec::Mac'       => '3.30',
	'File::Spec::OS2'       => '3.30',
	'File::Spec::Unix'      => '3.30',
	'File::Spec::VMS'       => '3.30',
	'File::Spec::Win32'     => '3.30',
	'File::Temp'            => '0.22',
	'File::stat'            => '1.02',
	'FileCache'             => '1.08',
	'FileHandle'            => '2.02',
	'VMS::Filespec'         => '1.12',
	'Filter::Simple'        => '0.84',
	'Filter::Util::Call'    => '1.08',
	'FindBin'               => '1.50',
	'GDBM_File'             => '1.10',
	'Getopt::Long'          => '2.38',
	'Getopt::Std'           => '1.06',
	'Hash::Util'            => '0.07',
	'Hash::Util::FieldHash' => '1.04',
	'I18N::Collate'         => '1.01',
	'I18N::LangTags'        => '0.35',
	'I18N::LangTags::Detect'=> '1.04',
	'I18N::LangTags::List'  => '0.35',
	'I18N::Langinfo'        => '0.03',
	'IO'                    => '1.25_02',
	'IO::Compress::Adapter::Bzip2'=> '2.024',
	'IO::Compress::Adapter::Deflate'=> '2.024',
	'IO::Compress::Adapter::Identity'=> '2.024',
	'IO::Compress::Base'    => '2.024',
	'IO::Compress::Base::Common'=> '2.024',
	'IO::Compress::Bzip2'   => '2.024',
	'IO::Compress::Deflate' => '2.024',
	'IO::Compress::Gzip'    => '2.024',
	'IO::Compress::Gzip::Constants'=> '2.024',
	'IO::Compress::RawDeflate'=> '2.024',
	'IO::Compress::Zip'     => '2.024',
	'IO::Compress::Zip::Constants'=> '2.024',
	'IO::Compress::Zlib::Constants'=> '2.024',
	'IO::Compress::Zlib::Extra'=> '2.024',
	'IO::Dir'               => '1.07',
	'IO::File'              => '1.14',
	'IO::Handle'            => '1.28',
	'IO::Pipe'              => '1.13',
	'IO::Poll'              => '0.07',
	'IO::Seekable'          => '1.10',
	'IO::Select'            => '1.17',
	'IO::Socket'            => '1.31',
	'IO::Socket::INET'      => '1.31',
	'IO::Socket::UNIX'      => '1.23',
	'IO::Uncompress::Adapter::Bunzip2'=> '2.024',
	'IO::Uncompress::Adapter::Identity'=> '2.024',
	'IO::Uncompress::Adapter::Inflate'=> '2.024',
	'IO::Uncompress::AnyInflate'=> '2.024',
	'IO::Uncompress::AnyUncompress'=> '2.024',
	'IO::Uncompress::Base'  => '2.024',
	'IO::Uncompress::Bunzip2'=> '2.024',
	'IO::Uncompress::Gunzip'=> '2.024',
	'IO::Uncompress::Inflate'=> '2.024',
	'IO::Uncompress::RawInflate'=> '2.024',
	'IO::Uncompress::Unzip' => '2.024',
	'IO::Zlib'              => '1.10',
	'IPC::Cmd'              => '0.54',
	'IPC::Msg'              => '2.01',
	'IPC::Open2'            => '1.03',
	'IPC::Open3'            => '1.06',
	'IPC::Semaphore'        => '2.01',
	'IPC::SharedMem'        => '2.01',
	'IPC::SysV'             => '2.01',
	'List::Util'            => '1.22',
	'List::Util::PP'        => '1.22',
	'List::Util::XS'        => '1.22',
	'Locale::Constants'     => '2.07',
	'Locale::Country'       => '2.07',
	'Locale::Currency'      => '2.07',
	'Locale::Language'      => '2.07',
	'Locale::Maketext'      => '1.14',
	'Locale::Maketext::Guts'=> '1.13',
	'Locale::Maketext::GutsLoader'=> '1.13',
	'Locale::Maketext::Simple'=> '0.21',
	'Locale::Script'        => '2.07',
	'Log::Message'          => '0.02',
	'Log::Message::Config'  => '0.01',
	'Log::Message::Handlers'=> undef,
	'Log::Message::Item'    => undef,
	'Log::Message::Simple'  => '0.06',
	'MIME::Base64'          => '3.09',
	'MIME::QuotedPrint'     => '3.09',
	'Math::BigFloat'        => '1.60',
	'Math::BigFloat::Trace' => '0.01',
	'Math::BigInt'          => '1.89_01',
	'Math::BigInt::Calc'    => '0.52',
	'Math::BigInt::CalcEmu' => '0.05',
	'Math::BigInt::FastCalc'=> '0.19',
	'Math::BigInt::Trace'   => '0.01',
	'Math::BigRat'          => '0.24',
	'Math::Complex'         => '1.56',
	'Math::Trig'            => '1.2',
	'Memoize'               => '1.01_03',
	'Memoize::AnyDBM_File'  => '0.65',
	'Memoize::Expire'       => '1.00',
	'Memoize::ExpireFile'   => '1.01',
	'Memoize::ExpireTest'   => '0.65',
	'Memoize::NDBM_File'    => '0.65',
	'Memoize::SDBM_File'    => '0.65',
	'Memoize::Storable'     => '0.65',
	'Module::Build'         => '0.3603',
	'Module::Build::Base'   => '0.3603',
	'Module::Build::Compat' => '0.3603',
	'Module::Build::Config' => '0.3603',
	'Module::Build::ConfigData'=> undef,
	'Module::Build::Cookbook'=> '0.3603',
	'Module::Build::Dumper' => '0.3603',
	'Module::Build::ModuleInfo'=> '0.3603',
	'Module::Build::Notes'  => '0.3603',
	'Module::Build::PPMMaker'=> '0.3603',
	'Module::Build::Platform::Amiga'=> '0.3603',
	'Module::Build::Platform::Default'=> '0.3603',
	'Module::Build::Platform::EBCDIC'=> '0.3603',
	'Module::Build::Platform::MPEiX'=> '0.3603',
	'Module::Build::Platform::MacOS'=> '0.3603',
	'Module::Build::Platform::RiscOS'=> '0.3603',
	'Module::Build::Platform::Unix'=> '0.3603',
	'Module::Build::Platform::VMS'=> '0.3603',
	'Module::Build::Platform::VOS'=> '0.3603',
	'Module::Build::Platform::Windows'=> '0.3603',
	'Module::Build::Platform::aix'=> '0.3603',
	'Module::Build::Platform::cygwin'=> '0.3603',
	'Module::Build::Platform::darwin'=> '0.3603',
	'Module::Build::Platform::os2'=> '0.3603',
	'Module::Build::PodParser'=> '0.3603',
	'Module::Build::Version'=> '0.77',
	'Module::Build::YAML'   => '1.40',
	'Module::CoreList'      => '2.31',
	'Module::Load'          => '0.16',
	'Module::Load::Conditional'=> '0.34',
	'Module::Loaded'        => '0.06',
	'Module::Pluggable'     => '3.9',
	'Module::Pluggable::Object'=> '3.9',
	'Moped::Msg'            => '0.01',
	'NDBM_File'             => '1.08',
	'NEXT'                  => '0.64',
	'Net::Cmd'              => '2.29',
	'Net::Config'           => '1.11',
	'Net::Domain'           => '2.20',
	'Net::FTP'              => '2.77',
	'Net::FTP::A'           => '1.18',
	'Net::FTP::E'           => '0.01',
	'Net::FTP::I'           => '1.12',
	'Net::FTP::L'           => '0.01',
	'Net::FTP::dataconn'    => '0.11',
	'Net::NNTP'             => '2.24',
	'Net::Netrc'            => '2.12',
	'Net::POP3'             => '2.29',
	'Net::Ping'             => '2.36',
	'Net::SMTP'             => '2.31',
	'Net::Time'             => '2.10',
	'Net::hostent'          => '1.01',
	'Net::netent'           => '1.00',
	'Net::protoent'         => '1.00',
	'Net::servent'          => '1.01',
	'O'                     => '1.01',
	'ODBM_File'             => '1.07',
	'Object::Accessor'      => '0.36',
	'Opcode'                => '1.15',
	'POSIX'                 => '1.19',
	'Package::Constants'    => '0.02',
	'Params::Check'         => '0.26',
	'Parse::CPAN::Meta'     => '1.40',
	'PerlIO'                => '1.06',
	'PerlIO::encoding'      => '0.12',
	'PerlIO::scalar'        => '0.07',
	'PerlIO::via'           => '0.09',
	'PerlIO::via::QuotedPrint'=> '0.06',
	'Pod::Checker'          => '1.45',
	'Pod::Escapes'          => '1.04',
	'Pod::Find'             => '1.35',
	'Pod::Functions'        => '1.04',
	'Pod::Html'             => '1.09',
	'Pod::InputObjects'     => '1.31',
	'Pod::LaTeX'            => '0.58',
	'Pod::Man'              => '2.23',
	'Pod::ParseLink'        => '1.10',
	'Pod::ParseUtils'       => '1.36',
	'Pod::Parser'           => '1.37',
	'Pod::Perldoc'          => '3.15_02',
	'Pod::Perldoc::BaseTo'  => undef,
	'Pod::Perldoc::GetOptsOO'=> undef,
	'Pod::Perldoc::ToChecker'=> undef,
	'Pod::Perldoc::ToMan'   => undef,
	'Pod::Perldoc::ToNroff' => undef,
	'Pod::Perldoc::ToPod'   => undef,
	'Pod::Perldoc::ToRtf'   => undef,
	'Pod::Perldoc::ToText'  => undef,
	'Pod::Perldoc::ToTk'    => undef,
	'Pod::Perldoc::ToXml'   => undef,
	'Pod::PlainText'        => '2.04',
	'Pod::Plainer'          => '1.02',
	'Pod::Select'           => '1.36',
	'Pod::Simple'           => '3.13',
	'Pod::Simple::BlackBox' => '3.13',
	'Pod::Simple::Checker'  => '3.13',
	'Pod::Simple::Debug'    => '3.13',
	'Pod::Simple::DumpAsText'=> '3.13',
	'Pod::Simple::DumpAsXML'=> '3.13',
	'Pod::Simple::HTML'     => '3.13',
	'Pod::Simple::HTMLBatch'=> '3.13',
	'Pod::Simple::HTMLLegacy'=> '5.01',
	'Pod::Simple::LinkSection'=> '3.13',
	'Pod::Simple::Methody'  => '3.13',
	'Pod::Simple::Progress' => '3.13',
	'Pod::Simple::PullParser'=> '3.13',
	'Pod::Simple::PullParserEndToken'=> '3.13',
	'Pod::Simple::PullParserStartToken'=> '3.13',
	'Pod::Simple::PullParserTextToken'=> '3.13',
	'Pod::Simple::PullParserToken'=> '3.13',
	'Pod::Simple::RTF'      => '3.13',
	'Pod::Simple::Search'   => '3.13',
	'Pod::Simple::SimpleTree'=> '3.13',
	'Pod::Simple::Text'     => '3.13',
	'Pod::Simple::TextContent'=> '3.13',
	'Pod::Simple::TiedOutFH'=> '3.13',
	'Pod::Simple::Transcode'=> '3.13',
	'Pod::Simple::TranscodeDumb'=> '3.13',
	'Pod::Simple::TranscodeSmart'=> '3.13',
	'Pod::Simple::XHTML'    => '3.13',
	'Pod::Simple::XMLOutStream'=> '3.13',
	'Pod::Text'             => '3.14',
	'Pod::Text::Color'      => '2.06',
	'Pod::Text::Overstrike' => '2.04',
	'Pod::Text::Termcap'    => '2.06',
	'Pod::Usage'            => '1.36',
	'SDBM_File'             => '1.06',
	'Safe'                  => '2.25',
	'Scalar::Util'          => '1.22',
	'Scalar::Util::PP'      => '1.22',
	'Search::Dict'          => '1.02',
	'SelectSaver'           => '1.02',
	'SelfLoader'            => '1.17',
	'Shell'                 => '0.72_01',
	'Socket'                => '1.87',
	'Storable'              => '2.22',
	'Switch'                => '2.16',
	'Symbol'                => '1.07',
	'Sys::Hostname'         => '1.11',
	'Sys::Syslog'           => '0.27',
	'Sys::Syslog::win32::Win32'=> undef,
	'TAP::Base'             => '3.17',
	'TAP::Formatter::Base'  => '3.17',
	'TAP::Formatter::Color' => '3.17',
	'TAP::Formatter::Console'=> '3.17',
	'TAP::Formatter::Console::ParallelSession'=> '3.17',
	'TAP::Formatter::Console::Session'=> '3.17',
	'TAP::Formatter::File'  => '3.17',
	'TAP::Formatter::File::Session'=> '3.17',
	'TAP::Formatter::Session'=> '3.17',
	'TAP::Harness'          => '3.17',
	'TAP::Object'           => '3.17',
	'TAP::Parser'           => '3.17',
	'TAP::Parser::Aggregator'=> '3.17',
	'TAP::Parser::Grammar'  => '3.17',
	'TAP::Parser::Iterator' => '3.17',
	'TAP::Parser::Iterator::Array'=> '3.17',
	'TAP::Parser::Iterator::Process'=> '3.17',
	'TAP::Parser::Iterator::Stream'=> '3.17',
	'TAP::Parser::IteratorFactory'=> '3.17',
	'TAP::Parser::Multiplexer'=> '3.17',
	'TAP::Parser::Result'   => '3.17',
	'TAP::Parser::Result::Bailout'=> '3.17',
	'TAP::Parser::Result::Comment'=> '3.17',
	'TAP::Parser::Result::Plan'=> '3.17',
	'TAP::Parser::Result::Pragma'=> '3.17',
	'TAP::Parser::Result::Test'=> '3.17',
	'TAP::Parser::Result::Unknown'=> '3.17',
	'TAP::Parser::Result::Version'=> '3.17',
	'TAP::Parser::Result::YAML'=> '3.17',
	'TAP::Parser::ResultFactory'=> '3.17',
	'TAP::Parser::Scheduler'=> '3.17',
	'TAP::Parser::Scheduler::Job'=> '3.17',
	'TAP::Parser::Scheduler::Spinner'=> '3.17',
	'TAP::Parser::Source'   => '3.17',
	'TAP::Parser::Source::Perl'=> '3.17',
	'TAP::Parser::Utils'    => '3.17',
	'TAP::Parser::YAMLish::Reader'=> '3.17',
	'TAP::Parser::YAMLish::Writer'=> '3.17',
	'Term::ANSIColor'       => '2.02',
	'Term::Cap'             => '1.12',
	'Term::Complete'        => '1.402',
	'Term::ReadLine'        => '1.05',
	'Term::UI'              => '0.20',
	'Term::UI::History'     => undef,
	'Test'                  => '1.25_02',
	'Test::Builder'         => '0.94',
	'Test::Builder::Module' => '0.94',
	'Test::Builder::Tester' => '1.18',
	'Test::Builder::Tester::Color'=> '1.18',
	'Test::Harness'         => '3.17',
	'Test::More'            => '0.94',
	'Test::Simple'          => '0.94',
	'Text::Abbrev'          => '1.01',
	'Text::Balanced'        => '2.02',
	'Text::ParseWords'      => '3.27',
	'Text::Soundex'         => '3.03_01',
	'Text::Tabs'            => '2009.0305',
	'Text::Wrap'            => '2009.0305',
	'Thread'                => '3.02',
	'Thread::Queue'         => '2.11',
	'Thread::Semaphore'     => '2.09',
	'Tie::Array'            => '1.03',
	'Tie::File'             => '0.97_02',
	'Tie::Handle'           => '4.2',
	'Tie::Hash'             => '1.03',
	'Tie::Hash::NamedCapture'=> '0.06',
	'Tie::Memoize'          => '1.1',
	'Tie::RefHash'          => '1.38',
	'Tie::Scalar'           => '1.02',
	'Tie::StdHandle'        => '4.2',
	'Tie::SubstrHash'       => '1.00',
	'Time::HiRes'           => '1.9719',
	'Time::Local'           => '1.1901_01',
	'Time::Piece'           => '1.15_01',
	'Time::Piece::Seconds'  => undef,
	'Time::Seconds'         => undef,
	'Time::gmtime'          => '1.03',
	'Time::localtime'       => '1.02',
	'Time::tm'              => '1.00',
	'UNIVERSAL'             => '1.06',
	'Unicode'               => '5.2.0',
	'Unicode::Collate'      => '0.52_01',
	'Unicode::Normalize'    => '1.03',
	'Unicode::UCD'          => '0.27',
	'User::grent'           => '1.01',
	'User::pwent'           => '1.00',
	'VMS::DCLsym'           => '1.03',
	'VMS::Stdio'            => '2.4',
	'Win32'                 => '0.39',
	'Win32API::File'        => '0.1101',
	'Win32API::File::ExtUtils::Myconst2perl'=> '1',
	'Win32CORE'             => '0.02',
	'XS::APItest'           => '0.18',
	'XS::APItest::KeywordRPN'=> '0.004',
	'XS::Typemap'           => '0.03',
	'XSLoader'              => '0.10',
	'XSLoader::XSLoader'    => '0.10',
	'attributes'            => '0.12',
	'autodie'               => '2.06_01',
	'autodie::exception'    => '2.06_01',
	'autodie::exception::system'=> '2.06_01',
	'autodie::hints'        => '2.06_01',
	'autouse'               => '1.06',
	'base'                  => '2.15',
	'bigint'                => '0.23',
	'bignum'                => '0.23',
	'bigrat'                => '0.23',
	'blib'                  => '1.04',
	'bytes'                 => '1.04',
	'charnames'             => '1.07',
	'constant'              => '1.20',
	'deprecate'             => '0.01',
	'diagnostics'           => '1.19',
	'encoding'              => '2.6_01',
	'encoding::warnings'    => '0.11',
	'feature'               => '1.17',
	'fields'                => '2.15',
	'filetest'              => '1.02',
	'if'                    => '0.05',
	'inc::latest'           => '0.3603',
	'integer'               => '1.00',
	'less'                  => '0.03',
	'lib'                   => '0.62',
	'locale'                => '1.00',
	'mro'                   => '1.02',
	'open'                  => '1.07',
	'ops'                   => '1.02',
	'overload'              => '1.10',
	'overload::numbers'     => undef,
	'overloading'           => '0.01',
	'parent'                => '0.223',
	're'                    => '0.11',
	'sigtrap'               => '1.04',
	'sort'                  => '2.01',
	'strict'                => '1.04',
	'subs'                  => '1.00',
	'threads'               => '1.77_01',
	'threads::shared'       => '1.33',
	'utf8'                  => '1.08',
	'vars'                  => '1.01',
	'version'               => '0.82',
	'vmsish'                => '1.02',
	'warnings'              => '1.09',
	'warnings::register'    => '1.01',
    },

    5.012001 => {
	'AnyDBM_File'           => '1.00',
	'App::Cpan'             => '1.5701',
	'App::Prove'            => '3.17',
	'App::Prove::State'     => '3.17',
	'App::Prove::State::Result'=> '3.17',
	'App::Prove::State::Result::Test'=> '3.17',
	'Archive::Extract'      => '0.38',
	'Archive::Tar'          => '1.54',
	'Archive::Tar::Constant'=> '0.02',
	'Archive::Tar::File'    => '0.02',
	'Attribute::Handlers'   => '0.87',
	'AutoLoader'            => '5.70',
	'AutoSplit'             => '1.06',
	'B'                     => '1.23',
	'B::Concise'            => '0.78',
	'B::Debug'              => '1.12',
	'B::Deparse'            => '0.97',
	'B::Lint'               => '1.11_01',
	'B::Lint::Debug'        => '0.01',
	'B::Showlex'            => '1.02',
	'B::Terse'              => '1.05',
	'B::Xref'               => '1.02',
	'Benchmark'             => '1.11',
	'CGI'                   => '3.49',
	'CGI::Apache'           => '1.01',
	'CGI::Carp'             => '3.45',
	'CGI::Cookie'           => '1.29',
	'CGI::Fast'             => '1.08',
	'CGI::Pretty'           => '3.46',
	'CGI::Push'             => '1.04',
	'CGI::Switch'           => '1.01',
	'CGI::Util'             => '3.48',
	'CPAN'                  => '1.94_56',
	'CPAN::Author'          => '5.5',
	'CPAN::Bundle'          => '5.5',
	'CPAN::CacheMgr'        => '5.5',
	'CPAN::Complete'        => '5.5',
	'CPAN::Debug'           => '5.5001',
	'CPAN::DeferredCode'    => '5.50',
	'CPAN::Distribution'    => '1.9456_01',
	'CPAN::Distroprefs'     => '6',
	'CPAN::Distrostatus'    => '5.5',
	'CPAN::Exception::RecursiveDependency'=> '5.5',
	'CPAN::Exception::blocked_urllist'=> '1.0',
	'CPAN::Exception::yaml_not_installed'=> '5.5',
	'CPAN::FTP'             => '5.5004',
	'CPAN::FTP::netrc'      => '1.00',
	'CPAN::FirstTime'       => '5.5301',
	'CPAN::HandleConfig'    => '5.5001',
	'CPAN::Index'           => '1.94',
	'CPAN::InfoObj'         => '5.5',
	'CPAN::Kwalify'         => '5.50',
	'CPAN::LWP::UserAgent'  => '1.94',
	'CPAN::Mirrors'         => '1.77',
	'CPAN::Module'          => '5.5',
	'CPAN::Nox'             => '5.50',
	'CPAN::Prompt'          => '5.5',
	'CPAN::Queue'           => '5.5',
	'CPAN::Shell'           => '5.5001',
	'CPAN::Tarzip'          => '5.5011',
	'CPAN::URL'             => '5.5',
	'CPAN::Version'         => '5.5',
	'CPANPLUS'              => '0.90',
	'CPANPLUS::Backend'     => undef,
	'CPANPLUS::Backend::RV' => undef,
	'CPANPLUS::Config'      => undef,
	'CPANPLUS::Configure'   => undef,
	'CPANPLUS::Configure::Setup'=> undef,
	'CPANPLUS::Dist'        => undef,
	'CPANPLUS::Dist::Autobundle'=> undef,
	'CPANPLUS::Dist::Base'  => undef,
	'CPANPLUS::Dist::Build' => '0.46',
	'CPANPLUS::Dist::Build::Constants'=> '0.46',
	'CPANPLUS::Dist::MM'    => undef,
	'CPANPLUS::Dist::Sample'=> undef,
	'CPANPLUS::Error'       => undef,
	'CPANPLUS::Internals'   => '0.90',
	'CPANPLUS::Internals::Constants'=> undef,
	'CPANPLUS::Internals::Constants::Report'=> undef,
	'CPANPLUS::Internals::Extract'=> undef,
	'CPANPLUS::Internals::Fetch'=> undef,
	'CPANPLUS::Internals::Report'=> undef,
	'CPANPLUS::Internals::Search'=> undef,
	'CPANPLUS::Internals::Source'=> undef,
	'CPANPLUS::Internals::Source::Memory'=> undef,
	'CPANPLUS::Internals::Source::SQLite'=> undef,
	'CPANPLUS::Internals::Source::SQLite::Tie'=> undef,
	'CPANPLUS::Internals::Utils'=> undef,
	'CPANPLUS::Internals::Utils::Autoflush'=> undef,
	'CPANPLUS::Module'      => undef,
	'CPANPLUS::Module::Author'=> undef,
	'CPANPLUS::Module::Author::Fake'=> undef,
	'CPANPLUS::Module::Checksums'=> undef,
	'CPANPLUS::Module::Fake'=> undef,
	'CPANPLUS::Module::Signature'=> undef,
	'CPANPLUS::Selfupdate'  => undef,
	'CPANPLUS::Shell'       => undef,
	'CPANPLUS::Shell::Classic'=> '0.0562',
	'CPANPLUS::Shell::Default'=> '0.90',
	'CPANPLUS::Shell::Default::Plugins::CustomSource'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Remote'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Source'=> undef,
	'Carp'                  => '1.16',
	'Carp::Heavy'           => '1.16',
	'Class::ISA'            => '0.36',
	'Class::Struct'         => '0.63',
	'Compress::Raw::Bzip2'  => '2.024',
	'Compress::Raw::Zlib'   => '2.024',
	'Compress::Zlib'        => '2.024',
	'Config'                => undef,
	'Config::Extensions'    => '0.01',
	'Cwd'                   => '3.31',
	'DB'                    => '1.02',
	'DBM_Filter'            => '0.03',
	'DBM_Filter::compress'  => '0.02',
	'DBM_Filter::encode'    => '0.02',
	'DBM_Filter::int32'     => '0.02',
	'DBM_Filter::null'      => '0.02',
	'DBM_Filter::utf8'      => '0.02',
	'DB_File'               => '1.820',
	'Data::Dumper'          => '2.125',
	'Devel::DProf'          => '20080331.00',
	'Devel::DProf::dprof::V'=> undef,
	'Devel::InnerPackage'   => '0.3',
	'Devel::PPPort'         => '3.19',
	'Devel::Peek'           => '1.04',
	'Devel::SelfStubber'    => '1.03',
	'Digest'                => '1.16',
	'Digest::MD5'           => '2.39',
	'Digest::SHA'           => '5.47',
	'Digest::base'          => '1.16',
	'Digest::file'          => '1.16',
	'DirHandle'             => '1.03',
	'Dumpvalue'             => '1.13',
	'DynaLoader'            => '1.10',
	'Encode'                => '2.39',
	'Encode::Alias'         => '2.12',
	'Encode::Byte'          => '2.04',
	'Encode::CJKConstants'  => '2.02',
	'Encode::CN'            => '2.03',
	'Encode::CN::HZ'        => '2.05',
	'Encode::Config'        => '2.05',
	'Encode::EBCDIC'        => '2.02',
	'Encode::Encoder'       => '2.01',
	'Encode::Encoding'      => '2.05',
	'Encode::GSM0338'       => '2.01',
	'Encode::Guess'         => '2.03',
	'Encode::JP'            => '2.04',
	'Encode::JP::H2Z'       => '2.02',
	'Encode::JP::JIS7'      => '2.04',
	'Encode::KR'            => '2.03',
	'Encode::KR::2022_KR'   => '2.02',
	'Encode::MIME::Header'  => '2.11',
	'Encode::MIME::Header::ISO_2022_JP'=> '1.03',
	'Encode::MIME::Name'    => '1.01',
	'Encode::Symbol'        => '2.02',
	'Encode::TW'            => '2.03',
	'Encode::Unicode'       => '2.07',
	'Encode::Unicode::UTF7' => '2.04',
	'English'               => '1.04',
	'Env'                   => '1.01',
	'Errno'                 => '1.11',
	'Exporter'              => '5.64_01',
	'Exporter::Heavy'       => '5.64_01',
	'ExtUtils::CBuilder'    => '0.27',
	'ExtUtils::CBuilder::Base'=> '0.27',
	'ExtUtils::CBuilder::Platform::Unix'=> '0.27',
	'ExtUtils::CBuilder::Platform::VMS'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.27',
	'ExtUtils::CBuilder::Platform::aix'=> '0.27',
	'ExtUtils::CBuilder::Platform::cygwin'=> '0.27',
	'ExtUtils::CBuilder::Platform::darwin'=> '0.27',
	'ExtUtils::CBuilder::Platform::dec_osf'=> '0.27',
	'ExtUtils::CBuilder::Platform::os2'=> '0.27',
	'ExtUtils::Command'     => '1.16',
	'ExtUtils::Command::MM' => '6.56',
	'ExtUtils::Constant'    => '0.22',
	'ExtUtils::Constant::Base'=> '0.04',
	'ExtUtils::Constant::ProxySubs'=> '0.06',
	'ExtUtils::Constant::Utils'=> '0.02',
	'ExtUtils::Constant::XS'=> '0.03',
	'ExtUtils::Embed'       => '1.28',
	'ExtUtils::Install'     => '1.55',
	'ExtUtils::Installed'   => '1.999_001',
	'ExtUtils::Liblist'     => '6.56',
	'ExtUtils::Liblist::Kid'=> '6.56',
	'ExtUtils::MM'          => '6.56',
	'ExtUtils::MM_AIX'      => '6.56',
	'ExtUtils::MM_Any'      => '6.56',
	'ExtUtils::MM_BeOS'     => '6.56',
	'ExtUtils::MM_Cygwin'   => '6.56',
	'ExtUtils::MM_DOS'      => '6.56',
	'ExtUtils::MM_Darwin'   => '6.56',
	'ExtUtils::MM_MacOS'    => '6.56',
	'ExtUtils::MM_NW5'      => '6.56',
	'ExtUtils::MM_OS2'      => '6.56',
	'ExtUtils::MM_QNX'      => '6.56',
	'ExtUtils::MM_UWIN'     => '6.56',
	'ExtUtils::MM_Unix'     => '6.56',
	'ExtUtils::MM_VMS'      => '6.56',
	'ExtUtils::MM_VOS'      => '6.56',
	'ExtUtils::MM_Win32'    => '6.56',
	'ExtUtils::MM_Win95'    => '6.56',
	'ExtUtils::MY'          => '6.56',
	'ExtUtils::MakeMaker'   => '6.56',
	'ExtUtils::MakeMaker::Config'=> '6.56',
	'ExtUtils::Manifest'    => '1.57',
	'ExtUtils::Miniperl'    => undef,
	'ExtUtils::Mkbootstrap' => '6.56',
	'ExtUtils::Mksymlists'  => '6.56',
	'ExtUtils::Packlist'    => '1.44',
	'ExtUtils::ParseXS'     => '2.21',
	'ExtUtils::XSSymSet'    => '1.1',
	'ExtUtils::testlib'     => '6.56',
	'Fatal'                 => '2.06_01',
	'Fcntl'                 => '1.06',
	'File::Basename'        => '2.78',
	'File::CheckTree'       => '4.4',
	'File::Compare'         => '1.1006',
	'File::Copy'            => '2.18',
	'File::DosGlob'         => '1.01',
	'File::Fetch'           => '0.24',
	'File::Find'            => '1.15',
	'File::Glob'            => '1.07',
	'File::GlobMapper'      => '1.000',
	'File::Path'            => '2.08_01',
	'File::Spec'            => '3.31',
	'File::Spec::Cygwin'    => '3.30',
	'File::Spec::Epoc'      => '3.30',
	'File::Spec::Functions' => '3.30',
	'File::Spec::Mac'       => '3.30',
	'File::Spec::OS2'       => '3.30',
	'File::Spec::Unix'      => '3.30',
	'File::Spec::VMS'       => '3.30',
	'File::Spec::Win32'     => '3.30',
	'File::Temp'            => '0.22',
	'File::stat'            => '1.02',
	'FileCache'             => '1.08',
	'FileHandle'            => '2.02',
	'VMS::Filespec'         => '1.12',
	'Filter::Simple'        => '0.84',
	'Filter::Util::Call'    => '1.08',
	'FindBin'               => '1.50',
	'GDBM_File'             => '1.10',
	'Getopt::Long'          => '2.38',
	'Getopt::Std'           => '1.06',
	'Hash::Util'            => '0.07',
	'Hash::Util::FieldHash' => '1.04',
	'I18N::Collate'         => '1.01',
	'I18N::LangTags'        => '0.35',
	'I18N::LangTags::Detect'=> '1.04',
	'I18N::LangTags::List'  => '0.35',
	'I18N::Langinfo'        => '0.03',
	'IO'                    => '1.25_02',
	'IO::Compress::Adapter::Bzip2'=> '2.024',
	'IO::Compress::Adapter::Deflate'=> '2.024',
	'IO::Compress::Adapter::Identity'=> '2.024',
	'IO::Compress::Base'    => '2.024',
	'IO::Compress::Base::Common'=> '2.024',
	'IO::Compress::Bzip2'   => '2.024',
	'IO::Compress::Deflate' => '2.024',
	'IO::Compress::Gzip'    => '2.024',
	'IO::Compress::Gzip::Constants'=> '2.024',
	'IO::Compress::RawDeflate'=> '2.024',
	'IO::Compress::Zip'     => '2.024',
	'IO::Compress::Zip::Constants'=> '2.024',
	'IO::Compress::Zlib::Constants'=> '2.024',
	'IO::Compress::Zlib::Extra'=> '2.024',
	'IO::Dir'               => '1.07',
	'IO::File'              => '1.14',
	'IO::Handle'            => '1.28',
	'IO::Pipe'              => '1.13',
	'IO::Poll'              => '0.07',
	'IO::Seekable'          => '1.10',
	'IO::Select'            => '1.17',
	'IO::Socket'            => '1.31',
	'IO::Socket::INET'      => '1.31',
	'IO::Socket::UNIX'      => '1.23',
	'IO::Uncompress::Adapter::Bunzip2'=> '2.024',
	'IO::Uncompress::Adapter::Identity'=> '2.024',
	'IO::Uncompress::Adapter::Inflate'=> '2.024',
	'IO::Uncompress::AnyInflate'=> '2.024',
	'IO::Uncompress::AnyUncompress'=> '2.024',
	'IO::Uncompress::Base'  => '2.024',
	'IO::Uncompress::Bunzip2'=> '2.024',
	'IO::Uncompress::Gunzip'=> '2.024',
	'IO::Uncompress::Inflate'=> '2.024',
	'IO::Uncompress::RawInflate'=> '2.024',
	'IO::Uncompress::Unzip' => '2.024',
	'IO::Zlib'              => '1.10',
	'IPC::Cmd'              => '0.54',
	'IPC::Msg'              => '2.01',
	'IPC::Open2'            => '1.03',
	'IPC::Open3'            => '1.05',
	'IPC::Semaphore'        => '2.01',
	'IPC::SharedMem'        => '2.01',
	'IPC::SysV'             => '2.01',
	'List::Util'            => '1.22',
	'List::Util::PP'        => '1.22',
	'List::Util::XS'        => '1.22',
	'Locale::Constants'     => '2.07',
	'Locale::Country'       => '2.07',
	'Locale::Currency'      => '2.07',
	'Locale::Language'      => '2.07',
	'Locale::Maketext'      => '1.14',
	'Locale::Maketext::Guts'=> '1.13',
	'Locale::Maketext::GutsLoader'=> '1.13',
	'Locale::Maketext::Simple'=> '0.21',
	'Locale::Script'        => '2.07',
	'Log::Message'          => '0.02',
	'Log::Message::Config'  => '0.01',
	'Log::Message::Handlers'=> undef,
	'Log::Message::Item'    => undef,
	'Log::Message::Simple'  => '0.06',
	'MIME::Base64'          => '3.08',
	'MIME::QuotedPrint'     => '3.08',
	'Math::BigFloat'        => '1.60',
	'Math::BigFloat::Trace' => '0.01',
	'Math::BigInt'          => '1.89_01',
	'Math::BigInt::Calc'    => '0.52',
	'Math::BigInt::CalcEmu' => '0.05',
	'Math::BigInt::FastCalc'=> '0.19',
	'Math::BigInt::Trace'   => '0.01',
	'Math::BigRat'          => '0.24',
	'Math::Complex'         => '1.56',
	'Math::Trig'            => '1.2',
	'Memoize'               => '1.01_03',
	'Memoize::AnyDBM_File'  => '0.65',
	'Memoize::Expire'       => '1.00',
	'Memoize::ExpireFile'   => '1.01',
	'Memoize::ExpireTest'   => '0.65',
	'Memoize::NDBM_File'    => '0.65',
	'Memoize::SDBM_File'    => '0.65',
	'Memoize::Storable'     => '0.65',
	'Module::Build'         => '0.3603',
	'Module::Build::Base'   => '0.3603',
	'Module::Build::Compat' => '0.3603',
	'Module::Build::Config' => '0.3603',
	'Module::Build::ConfigData'=> undef,
	'Module::Build::Cookbook'=> '0.3603',
	'Module::Build::Dumper' => '0.3603',
	'Module::Build::ModuleInfo'=> '0.3603',
	'Module::Build::Notes'  => '0.3603',
	'Module::Build::PPMMaker'=> '0.3603',
	'Module::Build::Platform::Amiga'=> '0.3603',
	'Module::Build::Platform::Default'=> '0.3603',
	'Module::Build::Platform::EBCDIC'=> '0.3603',
	'Module::Build::Platform::MPEiX'=> '0.3603',
	'Module::Build::Platform::MacOS'=> '0.3603',
	'Module::Build::Platform::RiscOS'=> '0.3603',
	'Module::Build::Platform::Unix'=> '0.3603',
	'Module::Build::Platform::VMS'=> '0.3603',
	'Module::Build::Platform::VOS'=> '0.3603',
	'Module::Build::Platform::Windows'=> '0.3603',
	'Module::Build::Platform::aix'=> '0.3603',
	'Module::Build::Platform::cygwin'=> '0.3603',
	'Module::Build::Platform::darwin'=> '0.3603',
	'Module::Build::Platform::os2'=> '0.3603',
	'Module::Build::PodParser'=> '0.3603',
	'Module::Build::Version'=> '0.77',
	'Module::Build::YAML'   => '1.40',
	'Module::CoreList'      => '2.32',
	'Module::Load'          => '0.16',
	'Module::Load::Conditional'=> '0.34',
	'Module::Loaded'        => '0.06',
	'Module::Pluggable'     => '3.9',
	'Module::Pluggable::Object'=> '3.9',
	'Moped::Msg'            => '0.01',
	'NDBM_File'             => '1.08',
	'NEXT'                  => '0.64',
	'Net::Cmd'              => '2.29',
	'Net::Config'           => '1.11',
	'Net::Domain'           => '2.20',
	'Net::FTP'              => '2.77',
	'Net::FTP::A'           => '1.18',
	'Net::FTP::E'           => '0.01',
	'Net::FTP::I'           => '1.12',
	'Net::FTP::L'           => '0.01',
	'Net::FTP::dataconn'    => '0.11',
	'Net::NNTP'             => '2.24',
	'Net::Netrc'            => '2.12',
	'Net::POP3'             => '2.29',
	'Net::Ping'             => '2.36',
	'Net::SMTP'             => '2.31',
	'Net::Time'             => '2.10',
	'Net::hostent'          => '1.01',
	'Net::netent'           => '1.00',
	'Net::protoent'         => '1.00',
	'Net::servent'          => '1.01',
	'O'                     => '1.01',
	'ODBM_File'             => '1.07',
	'Object::Accessor'      => '0.36',
	'Opcode'                => '1.15',
	'POSIX'                 => '1.19',
	'Package::Constants'    => '0.02',
	'Params::Check'         => '0.26',
	'Parse::CPAN::Meta'     => '1.40',
	'PerlIO'                => '1.06',
	'PerlIO::encoding'      => '0.12',
	'PerlIO::scalar'        => '0.07',
	'PerlIO::via'           => '0.09',
	'PerlIO::via::QuotedPrint'=> '0.06',
	'Pod::Checker'          => '1.45',
	'Pod::Escapes'          => '1.04',
	'Pod::Find'             => '1.35',
	'Pod::Functions'        => '1.04',
	'Pod::Html'             => '1.09',
	'Pod::InputObjects'     => '1.31',
	'Pod::LaTeX'            => '0.58',
	'Pod::Man'              => '2.23',
	'Pod::ParseLink'        => '1.10',
	'Pod::ParseUtils'       => '1.36',
	'Pod::Parser'           => '1.37',
	'Pod::Perldoc'          => '3.15_02',
	'Pod::Perldoc::BaseTo'  => undef,
	'Pod::Perldoc::GetOptsOO'=> undef,
	'Pod::Perldoc::ToChecker'=> undef,
	'Pod::Perldoc::ToMan'   => undef,
	'Pod::Perldoc::ToNroff' => undef,
	'Pod::Perldoc::ToPod'   => undef,
	'Pod::Perldoc::ToRtf'   => undef,
	'Pod::Perldoc::ToText'  => undef,
	'Pod::Perldoc::ToTk'    => undef,
	'Pod::Perldoc::ToXml'   => undef,
	'Pod::PlainText'        => '2.04',
	'Pod::Plainer'          => '1.02',
	'Pod::Select'           => '1.36',
	'Pod::Simple'           => '3.14',
	'Pod::Simple::BlackBox' => '3.14',
	'Pod::Simple::Checker'  => '3.14',
	'Pod::Simple::Debug'    => '3.14',
	'Pod::Simple::DumpAsText'=> '3.14',
	'Pod::Simple::DumpAsXML'=> '3.14',
	'Pod::Simple::HTML'     => '3.14',
	'Pod::Simple::HTMLBatch'=> '3.14',
	'Pod::Simple::HTMLLegacy'=> '5.01',
	'Pod::Simple::LinkSection'=> '3.14',
	'Pod::Simple::Methody'  => '3.14',
	'Pod::Simple::Progress' => '3.14',
	'Pod::Simple::PullParser'=> '3.14',
	'Pod::Simple::PullParserEndToken'=> '3.14',
	'Pod::Simple::PullParserStartToken'=> '3.14',
	'Pod::Simple::PullParserTextToken'=> '3.14',
	'Pod::Simple::PullParserToken'=> '3.14',
	'Pod::Simple::RTF'      => '3.14',
	'Pod::Simple::Search'   => '3.14',
	'Pod::Simple::SimpleTree'=> '3.14',
	'Pod::Simple::Text'     => '3.14',
	'Pod::Simple::TextContent'=> '3.14',
	'Pod::Simple::TiedOutFH'=> '3.14',
	'Pod::Simple::Transcode'=> '3.14',
	'Pod::Simple::TranscodeDumb'=> '3.14',
	'Pod::Simple::TranscodeSmart'=> '3.14',
	'Pod::Simple::XHTML'    => '3.14',
	'Pod::Simple::XMLOutStream'=> '3.14',
	'Pod::Text'             => '3.14',
	'Pod::Text::Color'      => '2.06',
	'Pod::Text::Overstrike' => '2.04',
	'Pod::Text::Termcap'    => '2.06',
	'Pod::Usage'            => '1.36',
	'SDBM_File'             => '1.06',
	'Safe'                  => '2.27',
	'Scalar::Util'          => '1.22',
	'Scalar::Util::PP'      => '1.22',
	'Search::Dict'          => '1.02',
	'SelectSaver'           => '1.02',
	'SelfLoader'            => '1.17',
	'Shell'                 => '0.72_01',
	'Socket'                => '1.87',
	'Storable'              => '2.22',
	'Switch'                => '2.16',
	'Symbol'                => '1.07',
	'Sys::Hostname'         => '1.11',
	'Sys::Syslog'           => '0.27',
	'Sys::Syslog::win32::Win32'=> undef,
	'TAP::Base'             => '3.17',
	'TAP::Formatter::Base'  => '3.17',
	'TAP::Formatter::Color' => '3.17',
	'TAP::Formatter::Console'=> '3.17',
	'TAP::Formatter::Console::ParallelSession'=> '3.17',
	'TAP::Formatter::Console::Session'=> '3.17',
	'TAP::Formatter::File'  => '3.17',
	'TAP::Formatter::File::Session'=> '3.17',
	'TAP::Formatter::Session'=> '3.17',
	'TAP::Harness'          => '3.17',
	'TAP::Object'           => '3.17',
	'TAP::Parser'           => '3.17',
	'TAP::Parser::Aggregator'=> '3.17',
	'TAP::Parser::Grammar'  => '3.17',
	'TAP::Parser::Iterator' => '3.17',
	'TAP::Parser::Iterator::Array'=> '3.17',
	'TAP::Parser::Iterator::Process'=> '3.17',
	'TAP::Parser::Iterator::Stream'=> '3.17',
	'TAP::Parser::IteratorFactory'=> '3.17',
	'TAP::Parser::Multiplexer'=> '3.17',
	'TAP::Parser::Result'   => '3.17',
	'TAP::Parser::Result::Bailout'=> '3.17',
	'TAP::Parser::Result::Comment'=> '3.17',
	'TAP::Parser::Result::Plan'=> '3.17',
	'TAP::Parser::Result::Pragma'=> '3.17',
	'TAP::Parser::Result::Test'=> '3.17',
	'TAP::Parser::Result::Unknown'=> '3.17',
	'TAP::Parser::Result::Version'=> '3.17',
	'TAP::Parser::Result::YAML'=> '3.17',
	'TAP::Parser::ResultFactory'=> '3.17',
	'TAP::Parser::Scheduler'=> '3.17',
	'TAP::Parser::Scheduler::Job'=> '3.17',
	'TAP::Parser::Scheduler::Spinner'=> '3.17',
	'TAP::Parser::Source'   => '3.17',
	'TAP::Parser::Source::Perl'=> '3.17',
	'TAP::Parser::Utils'    => '3.17',
	'TAP::Parser::YAMLish::Reader'=> '3.17',
	'TAP::Parser::YAMLish::Writer'=> '3.17',
	'Term::ANSIColor'       => '2.02',
	'Term::Cap'             => '1.12',
	'Term::Complete'        => '1.402',
	'Term::ReadLine'        => '1.05',
	'Term::UI'              => '0.20',
	'Term::UI::History'     => undef,
	'Test'                  => '1.25_02',
	'Test::Builder'         => '0.94',
	'Test::Builder::Module' => '0.94',
	'Test::Builder::Tester' => '1.18',
	'Test::Builder::Tester::Color'=> '1.18',
	'Test::Harness'         => '3.17',
	'Test::More'            => '0.94',
	'Test::Simple'          => '0.94',
	'Text::Abbrev'          => '1.01',
	'Text::Balanced'        => '2.02',
	'Text::ParseWords'      => '3.27',
	'Text::Soundex'         => '3.03_01',
	'Text::Tabs'            => '2009.0305',
	'Text::Wrap'            => '2009.0305',
	'Thread'                => '3.02',
	'Thread::Queue'         => '2.11',
	'Thread::Semaphore'     => '2.09',
	'Tie::Array'            => '1.03',
	'Tie::File'             => '0.97_02',
	'Tie::Handle'           => '4.2',
	'Tie::Hash'             => '1.03',
	'Tie::Hash::NamedCapture'=> '0.06',
	'Tie::Memoize'          => '1.1',
	'Tie::RefHash'          => '1.38',
	'Tie::Scalar'           => '1.02',
	'Tie::StdHandle'        => '4.2',
	'Tie::SubstrHash'       => '1.00',
	'Time::HiRes'           => '1.9719',
	'Time::Local'           => '1.1901_01',
	'Time::Piece'           => '1.15_01',
	'Time::Piece::Seconds'  => undef,
	'Time::Seconds'         => undef,
	'Time::gmtime'          => '1.03',
	'Time::localtime'       => '1.02',
	'Time::tm'              => '1.00',
	'UNIVERSAL'             => '1.06',
	'Unicode'               => '5.2.0',
	'Unicode::Collate'      => '0.52_01',
	'Unicode::Normalize'    => '1.03',
	'Unicode::UCD'          => '0.27',
	'User::grent'           => '1.01',
	'User::pwent'           => '1.00',
	'VMS::DCLsym'           => '1.03',
	'VMS::Stdio'            => '2.4',
	'Win32'                 => '0.39',
	'Win32API::File'        => '0.1101',
	'Win32API::File::ExtUtils::Myconst2perl'=> '1',
	'Win32CORE'             => '0.02',
	'XS::APItest'           => '0.17',
	'XS::APItest::KeywordRPN'=> '0.003',
	'XS::Typemap'           => '0.03',
	'XSLoader'              => '0.10',
	'XSLoader::XSLoader'    => '0.10',
	'attributes'            => '0.12',
	'autodie'               => '2.06_01',
	'autodie::exception'    => '2.06_01',
	'autodie::exception::system'=> '2.06_01',
	'autodie::hints'        => '2.06_01',
	'autouse'               => '1.06',
	'base'                  => '2.15',
	'bigint'                => '0.23',
	'bignum'                => '0.23',
	'bigrat'                => '0.23',
	'blib'                  => '1.04',
	'bytes'                 => '1.04',
	'charnames'             => '1.07',
	'constant'              => '1.20',
	'deprecate'             => '0.01',
	'diagnostics'           => '1.19',
	'encoding'              => '2.6_01',
	'encoding::warnings'    => '0.11',
	'feature'               => '1.16',
	'fields'                => '2.15',
	'filetest'              => '1.02',
	'if'                    => '0.05',
	'inc::latest'           => '0.3603',
	'integer'               => '1.00',
	'less'                  => '0.03',
	'lib'                   => '0.62',
	'locale'                => '1.00',
	'mro'                   => '1.02',
	'open'                  => '1.07',
	'ops'                   => '1.02',
	'overload'              => '1.10',
	'overload::numbers'     => undef,
	'overloading'           => '0.01',
	'parent'                => '0.223',
	're'                    => '0.11',
	'sigtrap'               => '1.04',
	'sort'                  => '2.01',
	'strict'                => '1.04',
	'subs'                  => '1.00',
	'threads'               => '1.75',
	'threads::shared'       => '1.32',
	'utf8'                  => '1.08',
	'vars'                  => '1.01',
	'version'               => '0.82',
	'vmsish'                => '1.02',
	'warnings'              => '1.09',
	'warnings::register'    => '1.01',
    },

    5.013001 => {
	'AnyDBM_File'           => '1.00',
	'App::Cpan'             => '1.5701',
	'App::Prove'            => '3.17',
	'App::Prove::State'     => '3.17',
	'App::Prove::State::Result'=> '3.17',
	'App::Prove::State::Result::Test'=> '3.17',
	'Archive::Extract'      => '0.38',
	'Archive::Tar'          => '1.54',
	'Archive::Tar::Constant'=> '0.02',
	'Archive::Tar::File'    => '0.02',
	'Attribute::Handlers'   => '0.87',
	'AutoLoader'            => '5.70',
	'AutoSplit'             => '1.06',
	'B'                     => '1.23',
	'B::Concise'            => '0.78',
	'B::Debug'              => '1.12',
	'B::Deparse'            => '0.97',
	'B::Lint'               => '1.11_01',
	'B::Lint::Debug'        => '0.01',
	'B::Showlex'            => '1.02',
	'B::Terse'              => '1.05',
	'B::Xref'               => '1.02',
	'Benchmark'             => '1.11',
	'CGI'                   => '3.49',
	'CGI::Apache'           => '1.01',
	'CGI::Carp'             => '3.45',
	'CGI::Cookie'           => '1.29',
	'CGI::Fast'             => '1.08',
	'CGI::Pretty'           => '3.46',
	'CGI::Push'             => '1.04',
	'CGI::Switch'           => '1.01',
	'CGI::Util'             => '3.48',
	'CPAN'                  => '1.94_56',
	'CPAN::Author'          => '5.5',
	'CPAN::Bundle'          => '5.5',
	'CPAN::CacheMgr'        => '5.5',
	'CPAN::Complete'        => '5.5',
	'CPAN::Debug'           => '5.5001',
	'CPAN::DeferredCode'    => '5.50',
	'CPAN::Distribution'    => '1.9456_01',
	'CPAN::Distroprefs'     => '6',
	'CPAN::Distrostatus'    => '5.5',
	'CPAN::Exception::RecursiveDependency'=> '5.5',
	'CPAN::Exception::blocked_urllist'=> '1.0',
	'CPAN::Exception::yaml_not_installed'=> '5.5',
	'CPAN::FTP'             => '5.5004',
	'CPAN::FTP::netrc'      => '1.00',
	'CPAN::FirstTime'       => '5.5301',
	'CPAN::HandleConfig'    => '5.5001',
	'CPAN::Index'           => '1.94',
	'CPAN::InfoObj'         => '5.5',
	'CPAN::Kwalify'         => '5.50',
	'CPAN::LWP::UserAgent'  => '1.94',
	'CPAN::Mirrors'         => '1.77',
	'CPAN::Module'          => '5.5',
	'CPAN::Nox'             => '5.50',
	'CPAN::Prompt'          => '5.5',
	'CPAN::Queue'           => '5.5',
	'CPAN::Shell'           => '5.5001',
	'CPAN::Tarzip'          => '5.5011',
	'CPAN::URL'             => '5.5',
	'CPAN::Version'         => '5.5',
	'CPANPLUS'              => '0.90',
	'CPANPLUS::Backend'     => undef,
	'CPANPLUS::Backend::RV' => undef,
	'CPANPLUS::Config'      => undef,
	'CPANPLUS::Configure'   => undef,
	'CPANPLUS::Configure::Setup'=> undef,
	'CPANPLUS::Dist'        => undef,
	'CPANPLUS::Dist::Autobundle'=> undef,
	'CPANPLUS::Dist::Base'  => undef,
	'CPANPLUS::Dist::Build' => '0.46',
	'CPANPLUS::Dist::Build::Constants'=> '0.46',
	'CPANPLUS::Dist::MM'    => undef,
	'CPANPLUS::Dist::Sample'=> undef,
	'CPANPLUS::Error'       => undef,
	'CPANPLUS::Internals'   => '0.90',
	'CPANPLUS::Internals::Constants'=> undef,
	'CPANPLUS::Internals::Constants::Report'=> undef,
	'CPANPLUS::Internals::Extract'=> undef,
	'CPANPLUS::Internals::Fetch'=> undef,
	'CPANPLUS::Internals::Report'=> undef,
	'CPANPLUS::Internals::Search'=> undef,
	'CPANPLUS::Internals::Source'=> undef,
	'CPANPLUS::Internals::Source::Memory'=> undef,
	'CPANPLUS::Internals::Source::SQLite'=> undef,
	'CPANPLUS::Internals::Source::SQLite::Tie'=> undef,
	'CPANPLUS::Internals::Utils'=> undef,
	'CPANPLUS::Internals::Utils::Autoflush'=> undef,
	'CPANPLUS::Module'      => undef,
	'CPANPLUS::Module::Author'=> undef,
	'CPANPLUS::Module::Author::Fake'=> undef,
	'CPANPLUS::Module::Checksums'=> undef,
	'CPANPLUS::Module::Fake'=> undef,
	'CPANPLUS::Module::Signature'=> undef,
	'CPANPLUS::Selfupdate'  => undef,
	'CPANPLUS::Shell'       => undef,
	'CPANPLUS::Shell::Classic'=> '0.0562',
	'CPANPLUS::Shell::Default'=> '0.90',
	'CPANPLUS::Shell::Default::Plugins::CustomSource'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Remote'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Source'=> undef,
	'Carp'                  => '1.16',
	'Carp::Heavy'           => '1.16',
	'Class::Struct'         => '0.63',
	'Compress::Raw::Bzip2'  => '2.024',
	'Compress::Raw::Zlib'   => '2.024',
	'Compress::Zlib'        => '2.024',
	'Config'                => undef,
	'Config::Extensions'    => '0.01',
	'Cwd'                   => '3.31',
	'DB'                    => '1.02',
	'DBM_Filter'            => '0.03',
	'DBM_Filter::compress'  => '0.02',
	'DBM_Filter::encode'    => '0.02',
	'DBM_Filter::int32'     => '0.02',
	'DBM_Filter::null'      => '0.02',
	'DBM_Filter::utf8'      => '0.02',
	'DB_File'               => '1.820',
	'Data::Dumper'          => '2.126',
	'Devel::DProf'          => '20080331.00',
	'Devel::DProf::dprof::V'=> undef,
	'Devel::InnerPackage'   => '0.3',
	'Devel::PPPort'         => '3.19',
	'Devel::Peek'           => '1.04',
	'Devel::SelfStubber'    => '1.03',
	'Digest'                => '1.16',
	'Digest::MD5'           => '2.39',
	'Digest::SHA'           => '5.47',
	'Digest::base'          => '1.16',
	'Digest::file'          => '1.16',
	'DirHandle'             => '1.03',
	'Dumpvalue'             => '1.14',
	'DynaLoader'            => '1.10',
	'Encode'                => '2.39',
	'Encode::Alias'         => '2.12',
	'Encode::Byte'          => '2.04',
	'Encode::CJKConstants'  => '2.02',
	'Encode::CN'            => '2.03',
	'Encode::CN::HZ'        => '2.05',
	'Encode::Config'        => '2.05',
	'Encode::EBCDIC'        => '2.02',
	'Encode::Encoder'       => '2.01',
	'Encode::Encoding'      => '2.05',
	'Encode::GSM0338'       => '2.01',
	'Encode::Guess'         => '2.03',
	'Encode::JP'            => '2.04',
	'Encode::JP::H2Z'       => '2.02',
	'Encode::JP::JIS7'      => '2.04',
	'Encode::KR'            => '2.03',
	'Encode::KR::2022_KR'   => '2.02',
	'Encode::MIME::Header'  => '2.11',
	'Encode::MIME::Header::ISO_2022_JP'=> '1.03',
	'Encode::MIME::Name'    => '1.01',
	'Encode::Symbol'        => '2.02',
	'Encode::TW'            => '2.03',
	'Encode::Unicode'       => '2.07',
	'Encode::Unicode::UTF7' => '2.04',
	'English'               => '1.04',
	'Env'                   => '1.01',
	'Errno'                 => '1.12',
	'Exporter'              => '5.64_01',
	'Exporter::Heavy'       => '5.64_01',
	'ExtUtils::CBuilder'    => '0.27',
	'ExtUtils::CBuilder::Base'=> '0.27',
	'ExtUtils::CBuilder::Platform::Unix'=> '0.27',
	'ExtUtils::CBuilder::Platform::VMS'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.27',
	'ExtUtils::CBuilder::Platform::aix'=> '0.27',
	'ExtUtils::CBuilder::Platform::cygwin'=> '0.27',
	'ExtUtils::CBuilder::Platform::darwin'=> '0.27',
	'ExtUtils::CBuilder::Platform::dec_osf'=> '0.27',
	'ExtUtils::CBuilder::Platform::os2'=> '0.27',
	'ExtUtils::Command'     => '1.16',
	'ExtUtils::Command::MM' => '6.56',
	'ExtUtils::Constant'    => '0.22',
	'ExtUtils::Constant::Base'=> '0.04',
	'ExtUtils::Constant::ProxySubs'=> '0.06',
	'ExtUtils::Constant::Utils'=> '0.02',
	'ExtUtils::Constant::XS'=> '0.03',
	'ExtUtils::Embed'       => '1.28',
	'ExtUtils::Install'     => '1.55',
	'ExtUtils::Installed'   => '1.999_001',
	'ExtUtils::Liblist'     => '6.56',
	'ExtUtils::Liblist::Kid'=> '6.56',
	'ExtUtils::MM'          => '6.56',
	'ExtUtils::MM_AIX'      => '6.56',
	'ExtUtils::MM_Any'      => '6.56',
	'ExtUtils::MM_BeOS'     => '6.56',
	'ExtUtils::MM_Cygwin'   => '6.56',
	'ExtUtils::MM_DOS'      => '6.56',
	'ExtUtils::MM_Darwin'   => '6.56',
	'ExtUtils::MM_MacOS'    => '6.56',
	'ExtUtils::MM_NW5'      => '6.56',
	'ExtUtils::MM_OS2'      => '6.56',
	'ExtUtils::MM_QNX'      => '6.56',
	'ExtUtils::MM_UWIN'     => '6.56',
	'ExtUtils::MM_Unix'     => '6.5601',
	'ExtUtils::MM_VMS'      => '6.56',
	'ExtUtils::MM_VOS'      => '6.56',
	'ExtUtils::MM_Win32'    => '6.56',
	'ExtUtils::MM_Win95'    => '6.56',
	'ExtUtils::MY'          => '6.56',
	'ExtUtils::MakeMaker'   => '6.5601',
	'ExtUtils::MakeMaker::Config'=> '6.56',
	'ExtUtils::Manifest'    => '1.57',
	'ExtUtils::Miniperl'    => undef,
	'ExtUtils::Mkbootstrap' => '6.56',
	'ExtUtils::Mksymlists'  => '6.56',
	'ExtUtils::Packlist'    => '1.44',
	'ExtUtils::ParseXS'     => '2.2205',
	'ExtUtils::XSSymSet'    => '1.1',
	'ExtUtils::testlib'     => '6.56',
	'Fatal'                 => '2.06_01',
	'Fcntl'                 => '1.06',
	'File::Basename'        => '2.78',
	'File::CheckTree'       => '4.4',
	'File::Compare'         => '1.1006',
	'File::Copy'            => '2.18',
	'File::DosGlob'         => '1.01',
	'File::Fetch'           => '0.24',
	'File::Find'            => '1.16',
	'File::Glob'            => '1.07',
	'File::GlobMapper'      => '1.000',
	'File::Path'            => '2.08_01',
	'File::Spec'            => '3.31',
	'File::Spec::Cygwin'    => '3.30',
	'File::Spec::Epoc'      => '3.30',
	'File::Spec::Functions' => '3.30',
	'File::Spec::Mac'       => '3.30',
	'File::Spec::OS2'       => '3.30',
	'File::Spec::Unix'      => '3.30',
	'File::Spec::VMS'       => '3.30',
	'File::Spec::Win32'     => '3.30',
	'File::Temp'            => '0.22',
	'File::stat'            => '1.02',
	'FileCache'             => '1.08',
	'FileHandle'            => '2.02',
	'VMS::Filespec'         => '1.12',
	'Filter::Simple'        => '0.84',
	'Filter::Util::Call'    => '1.08',
	'FindBin'               => '1.50',
	'GDBM_File'             => '1.10',
	'Getopt::Long'          => '2.38',
	'Getopt::Std'           => '1.06',
	'Hash::Util'            => '0.07',
	'Hash::Util::FieldHash' => '1.04',
	'I18N::Collate'         => '1.01',
	'I18N::LangTags'        => '0.35',
	'I18N::LangTags::Detect'=> '1.04',
	'I18N::LangTags::List'  => '0.35',
	'I18N::Langinfo'        => '0.03',
	'IO'                    => '1.25_02',
	'IO::Compress::Adapter::Bzip2'=> '2.024',
	'IO::Compress::Adapter::Deflate'=> '2.024',
	'IO::Compress::Adapter::Identity'=> '2.024',
	'IO::Compress::Base'    => '2.024',
	'IO::Compress::Base::Common'=> '2.024',
	'IO::Compress::Bzip2'   => '2.024',
	'IO::Compress::Deflate' => '2.024',
	'IO::Compress::Gzip'    => '2.024',
	'IO::Compress::Gzip::Constants'=> '2.024',
	'IO::Compress::RawDeflate'=> '2.024',
	'IO::Compress::Zip'     => '2.024',
	'IO::Compress::Zip::Constants'=> '2.024',
	'IO::Compress::Zlib::Constants'=> '2.024',
	'IO::Compress::Zlib::Extra'=> '2.024',
	'IO::Dir'               => '1.07',
	'IO::File'              => '1.14',
	'IO::Handle'            => '1.28',
	'IO::Pipe'              => '1.13',
	'IO::Poll'              => '0.07',
	'IO::Seekable'          => '1.10',
	'IO::Select'            => '1.17',
	'IO::Socket'            => '1.31',
	'IO::Socket::INET'      => '1.31',
	'IO::Socket::UNIX'      => '1.23',
	'IO::Uncompress::Adapter::Bunzip2'=> '2.024',
	'IO::Uncompress::Adapter::Identity'=> '2.024',
	'IO::Uncompress::Adapter::Inflate'=> '2.024',
	'IO::Uncompress::AnyInflate'=> '2.024',
	'IO::Uncompress::AnyUncompress'=> '2.024',
	'IO::Uncompress::Base'  => '2.024',
	'IO::Uncompress::Bunzip2'=> '2.024',
	'IO::Uncompress::Gunzip'=> '2.024',
	'IO::Uncompress::Inflate'=> '2.024',
	'IO::Uncompress::RawInflate'=> '2.024',
	'IO::Uncompress::Unzip' => '2.024',
	'IO::Zlib'              => '1.10',
	'IPC::Cmd'              => '0.58',
	'IPC::Msg'              => '2.01',
	'IPC::Open2'            => '1.03',
	'IPC::Open3'            => '1.06',
	'IPC::Semaphore'        => '2.01',
	'IPC::SharedMem'        => '2.01',
	'IPC::SysV'             => '2.01',
	'List::Util'            => '1.23',
	'List::Util::PP'        => '1.23',
	'List::Util::XS'        => '1.23',
	'Locale::Codes'         => '3.12',
	'Locale::Codes::Country'=> '3.12',
	'Locale::Codes::Currency'=> '3.12',
	'Locale::Codes::Language'=> '3.12',
	'Locale::Codes::Script' => '3.12',
	'Locale::Constants'     => '3.12',
	'Locale::Country'       => '3.12',
	'Locale::Currency'      => '3.12',
	'Locale::Language'      => '3.12',
	'Locale::Maketext'      => '1.14',
	'Locale::Maketext::Guts'=> '1.13',
	'Locale::Maketext::GutsLoader'=> '1.13',
	'Locale::Maketext::Simple'=> '0.21',
	'Locale::Script'        => '3.12',
	'Log::Message'          => '0.02',
	'Log::Message::Config'  => '0.01',
	'Log::Message::Handlers'=> undef,
	'Log::Message::Item'    => undef,
	'Log::Message::Simple'  => '0.06',
	'MIME::Base64'          => '3.09',
	'MIME::QuotedPrint'     => '3.09',
	'Math::BigFloat'        => '1.60',
	'Math::BigFloat::Trace' => '0.01',
	'Math::BigInt'          => '1.89_01',
	'Math::BigInt::Calc'    => '0.52',
	'Math::BigInt::CalcEmu' => '0.05',
	'Math::BigInt::FastCalc'=> '0.19',
	'Math::BigInt::Trace'   => '0.01',
	'Math::BigRat'          => '0.24',
	'Math::Complex'         => '1.56',
	'Math::Trig'            => '1.2',
	'Memoize'               => '1.01_03',
	'Memoize::AnyDBM_File'  => '0.65',
	'Memoize::Expire'       => '1.00',
	'Memoize::ExpireFile'   => '1.01',
	'Memoize::ExpireTest'   => '0.65',
	'Memoize::NDBM_File'    => '0.65',
	'Memoize::SDBM_File'    => '0.65',
	'Memoize::Storable'     => '0.65',
	'Module::Build'         => '0.3603',
	'Module::Build::Base'   => '0.3603',
	'Module::Build::Compat' => '0.3603',
	'Module::Build::Config' => '0.3603',
	'Module::Build::ConfigData'=> undef,
	'Module::Build::Cookbook'=> '0.3603',
	'Module::Build::Dumper' => '0.3603',
	'Module::Build::ModuleInfo'=> '0.3603',
	'Module::Build::Notes'  => '0.3603',
	'Module::Build::PPMMaker'=> '0.3603',
	'Module::Build::Platform::Amiga'=> '0.3603',
	'Module::Build::Platform::Default'=> '0.3603',
	'Module::Build::Platform::EBCDIC'=> '0.3603',
	'Module::Build::Platform::MPEiX'=> '0.3603',
	'Module::Build::Platform::MacOS'=> '0.3603',
	'Module::Build::Platform::RiscOS'=> '0.3603',
	'Module::Build::Platform::Unix'=> '0.3603',
	'Module::Build::Platform::VMS'=> '0.3603',
	'Module::Build::Platform::VOS'=> '0.3603',
	'Module::Build::Platform::Windows'=> '0.3603',
	'Module::Build::Platform::aix'=> '0.3603',
	'Module::Build::Platform::cygwin'=> '0.360301',
	'Module::Build::Platform::darwin'=> '0.3603',
	'Module::Build::Platform::os2'=> '0.3603',
	'Module::Build::PodParser'=> '0.3603',
	'Module::Build::Version'=> '0.77',
	'Module::Build::YAML'   => '1.40',
	'Module::CoreList'      => '2.34',
	'Module::Load'          => '0.16',
	'Module::Load::Conditional'=> '0.38',
	'Module::Loaded'        => '0.06',
	'Module::Pluggable'     => '3.9',
	'Module::Pluggable::Object'=> '3.9',
	'Moped::Msg'            => '0.01',
	'NDBM_File'             => '1.08',
	'NEXT'                  => '0.64',
	'Net::Cmd'              => '2.29',
	'Net::Config'           => '1.11',
	'Net::Domain'           => '2.20',
	'Net::FTP'              => '2.77',
	'Net::FTP::A'           => '1.18',
	'Net::FTP::E'           => '0.01',
	'Net::FTP::I'           => '1.12',
	'Net::FTP::L'           => '0.01',
	'Net::FTP::dataconn'    => '0.11',
	'Net::NNTP'             => '2.24',
	'Net::Netrc'            => '2.12',
	'Net::POP3'             => '2.29',
	'Net::Ping'             => '2.36',
	'Net::SMTP'             => '2.31',
	'Net::Time'             => '2.10',
	'Net::hostent'          => '1.01',
	'Net::netent'           => '1.00',
	'Net::protoent'         => '1.00',
	'Net::servent'          => '1.01',
	'O'                     => '1.01',
	'ODBM_File'             => '1.07',
	'Object::Accessor'      => '0.36',
	'Opcode'                => '1.15',
	'POSIX'                 => '1.19',
	'Package::Constants'    => '0.02',
	'Params::Check'         => '0.26',
	'Parse::CPAN::Meta'     => '1.40',
	'PerlIO'                => '1.06',
	'PerlIO::encoding'      => '0.12',
	'PerlIO::scalar'        => '0.08',
	'PerlIO::via'           => '0.09',
	'PerlIO::via::QuotedPrint'=> '0.06',
	'Pod::Checker'          => '1.45',
	'Pod::Escapes'          => '1.04',
	'Pod::Find'             => '1.35',
	'Pod::Functions'        => '1.04',
	'Pod::Html'             => '1.09',
	'Pod::InputObjects'     => '1.31',
	'Pod::LaTeX'            => '0.58',
	'Pod::Man'              => '2.23',
	'Pod::ParseLink'        => '1.10',
	'Pod::ParseUtils'       => '1.36',
	'Pod::Parser'           => '1.37',
	'Pod::Perldoc'          => '3.15_02',
	'Pod::Perldoc::BaseTo'  => undef,
	'Pod::Perldoc::GetOptsOO'=> undef,
	'Pod::Perldoc::ToChecker'=> undef,
	'Pod::Perldoc::ToMan'   => undef,
	'Pod::Perldoc::ToNroff' => undef,
	'Pod::Perldoc::ToPod'   => undef,
	'Pod::Perldoc::ToRtf'   => undef,
	'Pod::Perldoc::ToText'  => undef,
	'Pod::Perldoc::ToTk'    => undef,
	'Pod::Perldoc::ToXml'   => undef,
	'Pod::PlainText'        => '2.04',
	'Pod::Select'           => '1.36',
	'Pod::Simple'           => '3.14',
	'Pod::Simple::BlackBox' => '3.14',
	'Pod::Simple::Checker'  => '3.14',
	'Pod::Simple::Debug'    => '3.14',
	'Pod::Simple::DumpAsText'=> '3.14',
	'Pod::Simple::DumpAsXML'=> '3.14',
	'Pod::Simple::HTML'     => '3.14',
	'Pod::Simple::HTMLBatch'=> '3.14',
	'Pod::Simple::HTMLLegacy'=> '5.01',
	'Pod::Simple::LinkSection'=> '3.14',
	'Pod::Simple::Methody'  => '3.14',
	'Pod::Simple::Progress' => '3.14',
	'Pod::Simple::PullParser'=> '3.14',
	'Pod::Simple::PullParserEndToken'=> '3.14',
	'Pod::Simple::PullParserStartToken'=> '3.14',
	'Pod::Simple::PullParserTextToken'=> '3.14',
	'Pod::Simple::PullParserToken'=> '3.14',
	'Pod::Simple::RTF'      => '3.14',
	'Pod::Simple::Search'   => '3.14',
	'Pod::Simple::SimpleTree'=> '3.14',
	'Pod::Simple::Text'     => '3.14',
	'Pod::Simple::TextContent'=> '3.14',
	'Pod::Simple::TiedOutFH'=> '3.14',
	'Pod::Simple::Transcode'=> '3.14',
	'Pod::Simple::TranscodeDumb'=> '3.14',
	'Pod::Simple::TranscodeSmart'=> '3.14',
	'Pod::Simple::XHTML'    => '3.14',
	'Pod::Simple::XMLOutStream'=> '3.14',
	'Pod::Text'             => '3.14',
	'Pod::Text::Color'      => '2.06',
	'Pod::Text::Overstrike' => '2.04',
	'Pod::Text::Termcap'    => '2.06',
	'Pod::Usage'            => '1.36',
	'SDBM_File'             => '1.06',
	'Safe'                  => '2.27',
	'Scalar::Util'          => '1.23',
	'Scalar::Util::PP'      => '1.23',
	'Search::Dict'          => '1.02',
	'SelectSaver'           => '1.02',
	'SelfLoader'            => '1.17',
	'Shell'                 => '0.72_01',
	'Socket'                => '1.88',
	'Storable'              => '2.22',
	'Symbol'                => '1.07',
	'Sys::Hostname'         => '1.11',
	'Sys::Syslog'           => '0.27',
	'Sys::Syslog::win32::Win32'=> undef,
	'TAP::Base'             => '3.17',
	'TAP::Formatter::Base'  => '3.17',
	'TAP::Formatter::Color' => '3.17',
	'TAP::Formatter::Console'=> '3.17',
	'TAP::Formatter::Console::ParallelSession'=> '3.17',
	'TAP::Formatter::Console::Session'=> '3.17',
	'TAP::Formatter::File'  => '3.17',
	'TAP::Formatter::File::Session'=> '3.17',
	'TAP::Formatter::Session'=> '3.17',
	'TAP::Harness'          => '3.17',
	'TAP::Object'           => '3.17',
	'TAP::Parser'           => '3.17',
	'TAP::Parser::Aggregator'=> '3.17',
	'TAP::Parser::Grammar'  => '3.17',
	'TAP::Parser::Iterator' => '3.17',
	'TAP::Parser::Iterator::Array'=> '3.17',
	'TAP::Parser::Iterator::Process'=> '3.17',
	'TAP::Parser::Iterator::Stream'=> '3.17',
	'TAP::Parser::IteratorFactory'=> '3.17',
	'TAP::Parser::Multiplexer'=> '3.17',
	'TAP::Parser::Result'   => '3.17',
	'TAP::Parser::Result::Bailout'=> '3.17',
	'TAP::Parser::Result::Comment'=> '3.17',
	'TAP::Parser::Result::Plan'=> '3.17',
	'TAP::Parser::Result::Pragma'=> '3.17',
	'TAP::Parser::Result::Test'=> '3.17',
	'TAP::Parser::Result::Unknown'=> '3.17',
	'TAP::Parser::Result::Version'=> '3.17',
	'TAP::Parser::Result::YAML'=> '3.17',
	'TAP::Parser::ResultFactory'=> '3.17',
	'TAP::Parser::Scheduler'=> '3.17',
	'TAP::Parser::Scheduler::Job'=> '3.17',
	'TAP::Parser::Scheduler::Spinner'=> '3.17',
	'TAP::Parser::Source'   => '3.17',
	'TAP::Parser::Source::Perl'=> '3.17',
	'TAP::Parser::Utils'    => '3.17',
	'TAP::Parser::YAMLish::Reader'=> '3.17',
	'TAP::Parser::YAMLish::Writer'=> '3.17',
	'Term::ANSIColor'       => '2.02',
	'Term::Cap'             => '1.12',
	'Term::Complete'        => '1.402',
	'Term::ReadLine'        => '1.06',
	'Term::UI'              => '0.20',
	'Term::UI::History'     => undef,
	'Test'                  => '1.25_02',
	'Test::Builder'         => '0.94',
	'Test::Builder::Module' => '0.94',
	'Test::Builder::Tester' => '1.18',
	'Test::Builder::Tester::Color'=> '1.18',
	'Test::Harness'         => '3.17',
	'Test::More'            => '0.94',
	'Test::Simple'          => '0.94',
	'Text::Abbrev'          => '1.01',
	'Text::Balanced'        => '2.02',
	'Text::ParseWords'      => '3.27',
	'Text::Soundex'         => '3.03_01',
	'Text::Tabs'            => '2009.0305',
	'Text::Wrap'            => '2009.0305',
	'Thread'                => '3.02',
	'Thread::Queue'         => '2.11',
	'Thread::Semaphore'     => '2.09',
	'Tie::Array'            => '1.03',
	'Tie::File'             => '0.97_02',
	'Tie::Handle'           => '4.2',
	'Tie::Hash'             => '1.03',
	'Tie::Hash::NamedCapture'=> '0.06',
	'Tie::Memoize'          => '1.1',
	'Tie::RefHash'          => '1.38',
	'Tie::Scalar'           => '1.02',
	'Tie::StdHandle'        => '4.2',
	'Tie::SubstrHash'       => '1.00',
	'Time::HiRes'           => '1.9719',
	'Time::Local'           => '1.1901_01',
	'Time::Piece'           => '1.15_01',
	'Time::Piece::Seconds'  => undef,
	'Time::Seconds'         => undef,
	'Time::gmtime'          => '1.03',
	'Time::localtime'       => '1.02',
	'Time::tm'              => '1.00',
	'UNIVERSAL'             => '1.06',
	'Unicode'               => '5.2.0',
	'Unicode::Collate'      => '0.52_01',
	'Unicode::Normalize'    => '1.03',
	'Unicode::UCD'          => '0.28',
	'User::grent'           => '1.01',
	'User::pwent'           => '1.00',
	'VMS::DCLsym'           => '1.03',
	'VMS::Stdio'            => '2.4',
	'Win32'                 => '0.39',
	'Win32API::File'        => '0.1101',
	'Win32API::File::ExtUtils::Myconst2perl'=> '1',
	'Win32CORE'             => '0.02',
	'XS::APItest'           => '0.19',
	'XS::APItest::KeywordRPN'=> '0.004',
	'XS::Typemap'           => '0.03',
	'XSLoader'              => '0.10',
	'XSLoader::XSLoader'    => '0.10',
	'attributes'            => '0.12',
	'autodie'               => '2.06_01',
	'autodie::exception'    => '2.06_01',
	'autodie::exception::system'=> '2.06_01',
	'autodie::hints'        => '2.06_01',
	'autouse'               => '1.06',
	'base'                  => '2.15',
	'bigint'                => '0.23',
	'bignum'                => '0.23',
	'bigrat'                => '0.23',
	'blib'                  => '1.04',
	'bytes'                 => '1.04',
	'charnames'             => '1.08',
	'constant'              => '1.20',
	'deprecate'             => '0.01',
	'diagnostics'           => '1.19',
	'encoding'              => '2.6_01',
	'encoding::warnings'    => '0.11',
	'feature'               => '1.17',
	'fields'                => '2.15',
	'filetest'              => '1.02',
	'if'                    => '0.05',
	'inc::latest'           => '0.3603',
	'integer'               => '1.00',
	'less'                  => '0.03',
	'lib'                   => '0.62',
	'locale'                => '1.00',
	'mro'                   => '1.02',
	'open'                  => '1.07',
	'ops'                   => '1.02',
	'overload'              => '1.10',
	'overload::numbers'     => undef,
	'overloading'           => '0.01',
	'parent'                => '0.223',
	're'                    => '0.11',
	'sigtrap'               => '1.04',
	'sort'                  => '2.01',
	'strict'                => '1.04',
	'subs'                  => '1.00',
	'threads'               => '1.77_01',
	'threads::shared'       => '1.33',
	'utf8'                  => '1.08',
	'vars'                  => '1.01',
	'version'               => '0.82',
	'vmsish'                => '1.02',
	'warnings'              => '1.09',
	'warnings::register'    => '1.01',
    },
    5.013002 => {
	'AnyDBM_File'           => '1.00',
	'App::Cpan'             => '1.5701',
	'App::Prove'            => '3.17',
	'App::Prove::State'     => '3.17',
	'App::Prove::State::Result'=> '3.17',
	'App::Prove::State::Result::Test'=> '3.17',
	'Archive::Extract'      => '0.38',
	'Archive::Tar'          => '1.54',
	'Archive::Tar::Constant'=> '0.02',
	'Archive::Tar::File'    => '0.02',
	'Attribute::Handlers'   => '0.87',
	'AutoLoader'            => '5.70',
	'AutoSplit'             => '1.06',
	'B'                     => '1.23',
	'B::Concise'            => '0.79',
	'B::Debug'              => '1.12',
	'B::Deparse'            => '0.98',
	'B::Lint'               => '1.11_01',
	'B::Lint::Debug'        => '0.01',
	'B::Showlex'            => '1.02',
	'B::Terse'              => '1.05',
	'B::Xref'               => '1.02',
	'Benchmark'             => '1.11',
	'CGI'                   => '3.49',
	'CGI::Apache'           => '1.01',
	'CGI::Carp'             => '3.45',
	'CGI::Cookie'           => '1.29',
	'CGI::Fast'             => '1.08',
	'CGI::Pretty'           => '3.46',
	'CGI::Push'             => '1.04',
	'CGI::Switch'           => '1.01',
	'CGI::Util'             => '3.48',
	'CPAN'                  => '1.94_57',
	'CPAN::Author'          => '5.5',
	'CPAN::Bundle'          => '5.5',
	'CPAN::CacheMgr'        => '5.5',
	'CPAN::Complete'        => '5.5',
	'CPAN::Debug'           => '5.5001',
	'CPAN::DeferredCode'    => '5.50',
	'CPAN::Distribution'    => '1.9600',
	'CPAN::Distroprefs'     => '6',
	'CPAN::Distrostatus'    => '5.5',
	'CPAN::Exception::RecursiveDependency'=> '5.5',
	'CPAN::Exception::blocked_urllist'=> '1.0',
	'CPAN::Exception::yaml_not_installed'=> '5.5',
	'CPAN::FTP'             => '5.5004',
	'CPAN::FTP::netrc'      => '1.00',
	'CPAN::FirstTime'       => '5.5301',
	'CPAN::HandleConfig'    => '5.5001',
	'CPAN::Index'           => '1.94',
	'CPAN::InfoObj'         => '5.5',
	'CPAN::Kwalify'         => '5.50',
	'CPAN::LWP::UserAgent'  => '1.94',
	'CPAN::Mirrors'         => '1.77',
	'CPAN::Module'          => '5.5',
	'CPAN::Nox'             => '5.50',
	'CPAN::Prompt'          => '5.5',
	'CPAN::Queue'           => '5.5',
	'CPAN::Shell'           => '5.5001',
	'CPAN::Tarzip'          => '5.5011',
	'CPAN::URL'             => '5.5',
	'CPAN::Version'         => '5.5',
	'CPANPLUS'              => '0.90',
	'CPANPLUS::Backend'     => undef,
	'CPANPLUS::Backend::RV' => undef,
	'CPANPLUS::Config'      => undef,
	'CPANPLUS::Configure'   => undef,
	'CPANPLUS::Configure::Setup'=> undef,
	'CPANPLUS::Dist'        => undef,
	'CPANPLUS::Dist::Autobundle'=> undef,
	'CPANPLUS::Dist::Base'  => undef,
	'CPANPLUS::Dist::Build' => '0.46',
	'CPANPLUS::Dist::Build::Constants'=> '0.46',
	'CPANPLUS::Dist::MM'    => undef,
	'CPANPLUS::Dist::Sample'=> undef,
	'CPANPLUS::Error'       => undef,
	'CPANPLUS::Internals'   => '0.90',
	'CPANPLUS::Internals::Constants'=> undef,
	'CPANPLUS::Internals::Constants::Report'=> undef,
	'CPANPLUS::Internals::Extract'=> undef,
	'CPANPLUS::Internals::Fetch'=> undef,
	'CPANPLUS::Internals::Report'=> undef,
	'CPANPLUS::Internals::Search'=> undef,
	'CPANPLUS::Internals::Source'=> undef,
	'CPANPLUS::Internals::Source::Memory'=> undef,
	'CPANPLUS::Internals::Source::SQLite'=> undef,
	'CPANPLUS::Internals::Source::SQLite::Tie'=> undef,
	'CPANPLUS::Internals::Utils'=> undef,
	'CPANPLUS::Internals::Utils::Autoflush'=> undef,
	'CPANPLUS::Module'      => undef,
	'CPANPLUS::Module::Author'=> undef,
	'CPANPLUS::Module::Author::Fake'=> undef,
	'CPANPLUS::Module::Checksums'=> undef,
	'CPANPLUS::Module::Fake'=> undef,
	'CPANPLUS::Module::Signature'=> undef,
	'CPANPLUS::Selfupdate'  => undef,
	'CPANPLUS::Shell'       => undef,
	'CPANPLUS::Shell::Classic'=> '0.0562',
	'CPANPLUS::Shell::Default'=> '0.90',
	'CPANPLUS::Shell::Default::Plugins::CustomSource'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Remote'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Source'=> undef,
	'Carp'                  => '1.16',
	'Carp::Heavy'           => '1.16',
	'Class::Struct'         => '0.63',
	'Compress::Raw::Bzip2'  => '2.024',
	'Compress::Raw::Zlib'   => '2.024',
	'Compress::Zlib'        => '2.024',
	'Config'                => undef,
	'Config::Extensions'    => '0.01',
	'Cwd'                   => '3.31',
	'DB'                    => '1.02',
	'DBM_Filter'            => '0.03',
	'DBM_Filter::compress'  => '0.02',
	'DBM_Filter::encode'    => '0.02',
	'DBM_Filter::int32'     => '0.02',
	'DBM_Filter::null'      => '0.02',
	'DBM_Filter::utf8'      => '0.02',
	'DB_File'               => '1.820',
	'Data::Dumper'          => '2.126',
	'Devel::DProf'          => '20080331.00',
	'Devel::DProf::dprof::V'=> undef,
	'Devel::InnerPackage'   => '0.3',
	'Devel::PPPort'         => '3.19',
	'Devel::Peek'           => '1.04',
	'Devel::SelfStubber'    => '1.03',
	'Digest'                => '1.16',
	'Digest::MD5'           => '2.39',
	'Digest::SHA'           => '5.47',
	'Digest::base'          => '1.16',
	'Digest::file'          => '1.16',
	'DirHandle'             => '1.03',
	'Dumpvalue'             => '1.14',
	'DynaLoader'            => '1.10',
	'Encode'                => '2.39',
	'Encode::Alias'         => '2.12',
	'Encode::Byte'          => '2.04',
	'Encode::CJKConstants'  => '2.02',
	'Encode::CN'            => '2.03',
	'Encode::CN::HZ'        => '2.05',
	'Encode::Config'        => '2.05',
	'Encode::EBCDIC'        => '2.02',
	'Encode::Encoder'       => '2.01',
	'Encode::Encoding'      => '2.05',
	'Encode::GSM0338'       => '2.01',
	'Encode::Guess'         => '2.03',
	'Encode::JP'            => '2.04',
	'Encode::JP::H2Z'       => '2.02',
	'Encode::JP::JIS7'      => '2.04',
	'Encode::KR'            => '2.03',
	'Encode::KR::2022_KR'   => '2.02',
	'Encode::MIME::Header'  => '2.11',
	'Encode::MIME::Header::ISO_2022_JP'=> '1.03',
	'Encode::MIME::Name'    => '1.01',
	'Encode::Symbol'        => '2.02',
	'Encode::TW'            => '2.03',
	'Encode::Unicode'       => '2.07',
	'Encode::Unicode::UTF7' => '2.04',
	'English'               => '1.04',
	'Env'                   => '1.01',
	'Errno'                 => '1.12',
	'Exporter'              => '5.64_02',
	'Exporter::Heavy'       => '5.64_02',
	'ExtUtils::CBuilder'    => '0.27',
	'ExtUtils::CBuilder::Base'=> '0.27',
	'ExtUtils::CBuilder::Platform::Unix'=> '0.27',
	'ExtUtils::CBuilder::Platform::VMS'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.27',
	'ExtUtils::CBuilder::Platform::aix'=> '0.27',
	'ExtUtils::CBuilder::Platform::cygwin'=> '0.27',
	'ExtUtils::CBuilder::Platform::darwin'=> '0.27',
	'ExtUtils::CBuilder::Platform::dec_osf'=> '0.27',
	'ExtUtils::CBuilder::Platform::os2'=> '0.27',
	'ExtUtils::Command'     => '1.16',
	'ExtUtils::Command::MM' => '6.56',
	'ExtUtils::Constant'    => '0.22',
	'ExtUtils::Constant::Base'=> '0.04',
	'ExtUtils::Constant::ProxySubs'=> '0.06',
	'ExtUtils::Constant::Utils'=> '0.02',
	'ExtUtils::Constant::XS'=> '0.03',
	'ExtUtils::Embed'       => '1.28',
	'ExtUtils::Install'     => '1.55',
	'ExtUtils::Installed'   => '1.999_001',
	'ExtUtils::Liblist'     => '6.56',
	'ExtUtils::Liblist::Kid'=> '6.56',
	'ExtUtils::MM'          => '6.56',
	'ExtUtils::MM_AIX'      => '6.56',
	'ExtUtils::MM_Any'      => '6.56',
	'ExtUtils::MM_BeOS'     => '6.56',
	'ExtUtils::MM_Cygwin'   => '6.56',
	'ExtUtils::MM_DOS'      => '6.56',
	'ExtUtils::MM_Darwin'   => '6.56',
	'ExtUtils::MM_MacOS'    => '6.56',
	'ExtUtils::MM_NW5'      => '6.56',
	'ExtUtils::MM_OS2'      => '6.56',
	'ExtUtils::MM_QNX'      => '6.56',
	'ExtUtils::MM_UWIN'     => '6.56',
	'ExtUtils::MM_Unix'     => '6.5601',
	'ExtUtils::MM_VMS'      => '6.56',
	'ExtUtils::MM_VOS'      => '6.56',
	'ExtUtils::MM_Win32'    => '6.56',
	'ExtUtils::MM_Win95'    => '6.56',
	'ExtUtils::MY'          => '6.56',
	'ExtUtils::MakeMaker'   => '6.5601',
	'ExtUtils::MakeMaker::Config'=> '6.56',
	'ExtUtils::Manifest'    => '1.57',
	'ExtUtils::Miniperl'    => undef,
	'ExtUtils::Mkbootstrap' => '6.56',
	'ExtUtils::Mksymlists'  => '6.56',
	'ExtUtils::Packlist'    => '1.44',
	'ExtUtils::ParseXS'     => '2.2205',
	'ExtUtils::XSSymSet'    => '1.1',
	'ExtUtils::testlib'     => '6.56',
	'Fatal'                 => '2.06_01',
	'Fcntl'                 => '1.06',
	'File::Basename'        => '2.78',
	'File::CheckTree'       => '4.4',
	'File::Compare'         => '1.1006',
	'File::Copy'            => '2.19',
	'File::DosGlob'         => '1.01',
	'File::Fetch'           => '0.24',
	'File::Find'            => '1.16',
	'File::Glob'            => '1.07',
	'File::GlobMapper'      => '1.000',
	'File::Path'            => '2.08_01',
	'File::Spec'            => '3.31',
	'File::Spec::Cygwin'    => '3.30',
	'File::Spec::Epoc'      => '3.30',
	'File::Spec::Functions' => '3.30',
	'File::Spec::Mac'       => '3.30',
	'File::Spec::OS2'       => '3.30',
	'File::Spec::Unix'      => '3.30',
	'File::Spec::VMS'       => '3.30',
	'File::Spec::Win32'     => '3.30',
	'File::Temp'            => '0.22',
	'File::stat'            => '1.02',
	'FileCache'             => '1.08',
	'FileHandle'            => '2.02',
	'VMS::Filespec'         => '1.12',
	'Filter::Simple'        => '0.84',
	'Filter::Util::Call'    => '1.08',
	'FindBin'               => '1.50',
	'GDBM_File'             => '1.10',
	'Getopt::Long'          => '2.38',
	'Getopt::Std'           => '1.06',
	'Hash::Util'            => '0.08',
	'Hash::Util::FieldHash' => '1.04',
	'I18N::Collate'         => '1.01',
	'I18N::LangTags'        => '0.35',
	'I18N::LangTags::Detect'=> '1.04',
	'I18N::LangTags::List'  => '0.35',
	'I18N::Langinfo'        => '0.03',
	'IO'                    => '1.25_02',
	'IO::Compress::Adapter::Bzip2'=> '2.024',
	'IO::Compress::Adapter::Deflate'=> '2.024',
	'IO::Compress::Adapter::Identity'=> '2.024',
	'IO::Compress::Base'    => '2.024',
	'IO::Compress::Base::Common'=> '2.024',
	'IO::Compress::Bzip2'   => '2.024',
	'IO::Compress::Deflate' => '2.024',
	'IO::Compress::Gzip'    => '2.024',
	'IO::Compress::Gzip::Constants'=> '2.024',
	'IO::Compress::RawDeflate'=> '2.024',
	'IO::Compress::Zip'     => '2.024',
	'IO::Compress::Zip::Constants'=> '2.024',
	'IO::Compress::Zlib::Constants'=> '2.024',
	'IO::Compress::Zlib::Extra'=> '2.024',
	'IO::Dir'               => '1.07',
	'IO::File'              => '1.14',
	'IO::Handle'            => '1.28',
	'IO::Pipe'              => '1.13',
	'IO::Poll'              => '0.07',
	'IO::Seekable'          => '1.10',
	'IO::Select'            => '1.17',
	'IO::Socket'            => '1.32',
	'IO::Socket::INET'      => '1.31',
	'IO::Socket::UNIX'      => '1.23',
	'IO::Uncompress::Adapter::Bunzip2'=> '2.024',
	'IO::Uncompress::Adapter::Identity'=> '2.024',
	'IO::Uncompress::Adapter::Inflate'=> '2.024',
	'IO::Uncompress::AnyInflate'=> '2.024',
	'IO::Uncompress::AnyUncompress'=> '2.024',
	'IO::Uncompress::Base'  => '2.024',
	'IO::Uncompress::Bunzip2'=> '2.024',
	'IO::Uncompress::Gunzip'=> '2.024',
	'IO::Uncompress::Inflate'=> '2.024',
	'IO::Uncompress::RawInflate'=> '2.024',
	'IO::Uncompress::Unzip' => '2.024',
	'IO::Zlib'              => '1.10',
	'IPC::Cmd'              => '0.58',
	'IPC::Msg'              => '2.01',
	'IPC::Open2'            => '1.03',
	'IPC::Open3'            => '1.06',
	'IPC::Semaphore'        => '2.01',
	'IPC::SharedMem'        => '2.01',
	'IPC::SysV'             => '2.01',
	'List::Util'            => '1.23',
	'List::Util::PP'        => '1.23',
	'List::Util::XS'        => '1.23',
	'Locale::Codes'         => '3.13',
	'Locale::Codes::Country'=> '3.13',
	'Locale::Codes::Currency'=> '3.13',
	'Locale::Codes::Language'=> '3.13',
	'Locale::Codes::Script' => '3.13',
	'Locale::Constants'     => '3.13',
	'Locale::Country'       => '3.13',
	'Locale::Currency'      => '3.13',
	'Locale::Language'      => '3.13',
	'Locale::Maketext'      => '1.14',
	'Locale::Maketext::Guts'=> '1.13',
	'Locale::Maketext::GutsLoader'=> '1.13',
	'Locale::Maketext::Simple'=> '0.21',
	'Locale::Script'        => '3.13',
	'Log::Message'          => '0.02',
	'Log::Message::Config'  => '0.01',
	'Log::Message::Handlers'=> undef,
	'Log::Message::Item'    => undef,
	'Log::Message::Simple'  => '0.06',
	'MIME::Base64'          => '3.09',
	'MIME::QuotedPrint'     => '3.09',
	'Math::BigFloat'        => '1.60',
	'Math::BigFloat::Trace' => '0.01',
	'Math::BigInt'          => '1.89_01',
	'Math::BigInt::Calc'    => '0.52',
	'Math::BigInt::CalcEmu' => '0.05',
	'Math::BigInt::FastCalc'=> '0.19',
	'Math::BigInt::Trace'   => '0.01',
	'Math::BigRat'          => '0.24',
	'Math::Complex'         => '1.56',
	'Math::Trig'            => '1.2',
	'Memoize'               => '1.01_03',
	'Memoize::AnyDBM_File'  => '0.65',
	'Memoize::Expire'       => '1.00',
	'Memoize::ExpireFile'   => '1.01',
	'Memoize::ExpireTest'   => '0.65',
	'Memoize::NDBM_File'    => '0.65',
	'Memoize::SDBM_File'    => '0.65',
	'Memoize::Storable'     => '0.65',
	'Module::Build'         => '0.3603',
	'Module::Build::Base'   => '0.3603',
	'Module::Build::Compat' => '0.3603',
	'Module::Build::Config' => '0.3603',
	'Module::Build::ConfigData'=> undef,
	'Module::Build::Cookbook'=> '0.3603',
	'Module::Build::Dumper' => '0.3603',
	'Module::Build::ModuleInfo'=> '0.3603',
	'Module::Build::Notes'  => '0.3603',
	'Module::Build::PPMMaker'=> '0.3603',
	'Module::Build::Platform::Amiga'=> '0.3603',
	'Module::Build::Platform::Default'=> '0.3603',
	'Module::Build::Platform::EBCDIC'=> '0.3603',
	'Module::Build::Platform::MPEiX'=> '0.3603',
	'Module::Build::Platform::MacOS'=> '0.3603',
	'Module::Build::Platform::RiscOS'=> '0.3603',
	'Module::Build::Platform::Unix'=> '0.3603',
	'Module::Build::Platform::VMS'=> '0.3603',
	'Module::Build::Platform::VOS'=> '0.3603',
	'Module::Build::Platform::Windows'=> '0.3603',
	'Module::Build::Platform::aix'=> '0.3603',
	'Module::Build::Platform::cygwin'=> '0.360301',
	'Module::Build::Platform::darwin'=> '0.3603',
	'Module::Build::Platform::os2'=> '0.3603',
	'Module::Build::PodParser'=> '0.3603',
	'Module::Build::Version'=> '0.77',
	'Module::Build::YAML'   => '1.40',
	'Module::CoreList'      => '2.34',
	'Module::Load'          => '0.16',
	'Module::Load::Conditional'=> '0.38',
	'Module::Loaded'        => '0.06',
	'Module::Pluggable'     => '3.9',
	'Module::Pluggable::Object'=> '3.9',
	'Moped::Msg'            => '0.01',
	'NDBM_File'             => '1.08',
	'NEXT'                  => '0.64',
	'Net::Cmd'              => '2.29',
	'Net::Config'           => '1.11',
	'Net::Domain'           => '2.20',
	'Net::FTP'              => '2.77',
	'Net::FTP::A'           => '1.18',
	'Net::FTP::E'           => '0.01',
	'Net::FTP::I'           => '1.12',
	'Net::FTP::L'           => '0.01',
	'Net::FTP::dataconn'    => '0.11',
	'Net::NNTP'             => '2.24',
	'Net::Netrc'            => '2.12',
	'Net::POP3'             => '2.29',
	'Net::Ping'             => '2.36',
	'Net::SMTP'             => '2.31',
	'Net::Time'             => '2.10',
	'Net::hostent'          => '1.01',
	'Net::netent'           => '1.00',
	'Net::protoent'         => '1.00',
	'Net::servent'          => '1.01',
	'O'                     => '1.01',
	'ODBM_File'             => '1.07',
	'Object::Accessor'      => '0.36',
	'Opcode'                => '1.15',
	'POSIX'                 => '1.19',
	'Package::Constants'    => '0.02',
	'Params::Check'         => '0.26',
	'Parse::CPAN::Meta'     => '1.40',
	'PerlIO'                => '1.06',
	'PerlIO::encoding'      => '0.12',
	'PerlIO::scalar'        => '0.08',
	'PerlIO::via'           => '0.09',
	'PerlIO::via::QuotedPrint'=> '0.06',
	'Pod::Checker'          => '1.45',
	'Pod::Escapes'          => '1.04',
	'Pod::Find'             => '1.35',
	'Pod::Functions'        => '1.04',
	'Pod::Html'             => '1.09',
	'Pod::InputObjects'     => '1.31',
	'Pod::LaTeX'            => '0.58',
	'Pod::Man'              => '2.23',
	'Pod::ParseLink'        => '1.10',
	'Pod::ParseUtils'       => '1.36',
	'Pod::Parser'           => '1.37',
	'Pod::Perldoc'          => '3.15_02',
	'Pod::Perldoc::BaseTo'  => undef,
	'Pod::Perldoc::GetOptsOO'=> undef,
	'Pod::Perldoc::ToChecker'=> undef,
	'Pod::Perldoc::ToMan'   => undef,
	'Pod::Perldoc::ToNroff' => undef,
	'Pod::Perldoc::ToPod'   => undef,
	'Pod::Perldoc::ToRtf'   => undef,
	'Pod::Perldoc::ToText'  => undef,
	'Pod::Perldoc::ToTk'    => undef,
	'Pod::Perldoc::ToXml'   => undef,
	'Pod::PlainText'        => '2.04',
	'Pod::Select'           => '1.36',
	'Pod::Simple'           => '3.14',
	'Pod::Simple::BlackBox' => '3.14',
	'Pod::Simple::Checker'  => '3.14',
	'Pod::Simple::Debug'    => '3.14',
	'Pod::Simple::DumpAsText'=> '3.14',
	'Pod::Simple::DumpAsXML'=> '3.14',
	'Pod::Simple::HTML'     => '3.14',
	'Pod::Simple::HTMLBatch'=> '3.14',
	'Pod::Simple::HTMLLegacy'=> '5.01',
	'Pod::Simple::LinkSection'=> '3.14',
	'Pod::Simple::Methody'  => '3.14',
	'Pod::Simple::Progress' => '3.14',
	'Pod::Simple::PullParser'=> '3.14',
	'Pod::Simple::PullParserEndToken'=> '3.14',
	'Pod::Simple::PullParserStartToken'=> '3.14',
	'Pod::Simple::PullParserTextToken'=> '3.14',
	'Pod::Simple::PullParserToken'=> '3.14',
	'Pod::Simple::RTF'      => '3.14',
	'Pod::Simple::Search'   => '3.14',
	'Pod::Simple::SimpleTree'=> '3.14',
	'Pod::Simple::Text'     => '3.14',
	'Pod::Simple::TextContent'=> '3.14',
	'Pod::Simple::TiedOutFH'=> '3.14',
	'Pod::Simple::Transcode'=> '3.14',
	'Pod::Simple::TranscodeDumb'=> '3.14',
	'Pod::Simple::TranscodeSmart'=> '3.14',
	'Pod::Simple::XHTML'    => '3.14',
	'Pod::Simple::XMLOutStream'=> '3.14',
	'Pod::Text'             => '3.14',
	'Pod::Text::Color'      => '2.06',
	'Pod::Text::Overstrike' => '2.04',
	'Pod::Text::Termcap'    => '2.06',
	'Pod::Usage'            => '1.36',
	'SDBM_File'             => '1.06',
	'Safe'                  => '2.27',
	'Scalar::Util'          => '1.23',
	'Scalar::Util::PP'      => '1.23',
	'Search::Dict'          => '1.03',
	'SelectSaver'           => '1.02',
	'SelfLoader'            => '1.17',
	'Shell'                 => '0.72_01',
	'Socket'                => '1.89',
	'Storable'              => '2.22',
	'Symbol'                => '1.07',
	'Sys::Hostname'         => '1.11',
	'Sys::Syslog'           => '0.27',
	'Sys::Syslog::win32::Win32'=> undef,
	'TAP::Base'             => '3.17',
	'TAP::Formatter::Base'  => '3.17',
	'TAP::Formatter::Color' => '3.17',
	'TAP::Formatter::Console'=> '3.17',
	'TAP::Formatter::Console::ParallelSession'=> '3.17',
	'TAP::Formatter::Console::Session'=> '3.17',
	'TAP::Formatter::File'  => '3.17',
	'TAP::Formatter::File::Session'=> '3.17',
	'TAP::Formatter::Session'=> '3.17',
	'TAP::Harness'          => '3.17',
	'TAP::Object'           => '3.17',
	'TAP::Parser'           => '3.17',
	'TAP::Parser::Aggregator'=> '3.17',
	'TAP::Parser::Grammar'  => '3.17',
	'TAP::Parser::Iterator' => '3.17',
	'TAP::Parser::Iterator::Array'=> '3.17',
	'TAP::Parser::Iterator::Process'=> '3.17',
	'TAP::Parser::Iterator::Stream'=> '3.17',
	'TAP::Parser::IteratorFactory'=> '3.17',
	'TAP::Parser::Multiplexer'=> '3.17',
	'TAP::Parser::Result'   => '3.17',
	'TAP::Parser::Result::Bailout'=> '3.17',
	'TAP::Parser::Result::Comment'=> '3.17',
	'TAP::Parser::Result::Plan'=> '3.17',
	'TAP::Parser::Result::Pragma'=> '3.17',
	'TAP::Parser::Result::Test'=> '3.17',
	'TAP::Parser::Result::Unknown'=> '3.17',
	'TAP::Parser::Result::Version'=> '3.17',
	'TAP::Parser::Result::YAML'=> '3.17',
	'TAP::Parser::ResultFactory'=> '3.17',
	'TAP::Parser::Scheduler'=> '3.17',
	'TAP::Parser::Scheduler::Job'=> '3.17',
	'TAP::Parser::Scheduler::Spinner'=> '3.17',
	'TAP::Parser::Source'   => '3.17',
	'TAP::Parser::Source::Perl'=> '3.17',
	'TAP::Parser::Utils'    => '3.17',
	'TAP::Parser::YAMLish::Reader'=> '3.17',
	'TAP::Parser::YAMLish::Writer'=> '3.17',
	'Term::ANSIColor'       => '2.02',
	'Term::Cap'             => '1.12',
	'Term::Complete'        => '1.402',
	'Term::ReadLine'        => '1.06',
	'Term::UI'              => '0.20',
	'Term::UI::History'     => undef,
	'Test'                  => '1.25_02',
	'Test::Builder'         => '0.94',
	'Test::Builder::Module' => '0.94',
	'Test::Builder::Tester' => '1.18',
	'Test::Builder::Tester::Color'=> '1.18',
	'Test::Harness'         => '3.17',
	'Test::More'            => '0.94',
	'Test::Simple'          => '0.94',
	'Text::Abbrev'          => '1.01',
	'Text::Balanced'        => '2.02',
	'Text::ParseWords'      => '3.27',
	'Text::Soundex'         => '3.03_01',
	'Text::Tabs'            => '2009.0305',
	'Text::Wrap'            => '2009.0305',
	'Thread'                => '3.02',
	'Thread::Queue'         => '2.11',
	'Thread::Semaphore'     => '2.11',
	'Tie::Array'            => '1.03',
	'Tie::File'             => '0.97_02',
	'Tie::Handle'           => '4.2',
	'Tie::Hash'             => '1.03',
	'Tie::Hash::NamedCapture'=> '0.06',
	'Tie::Memoize'          => '1.1',
	'Tie::RefHash'          => '1.38',
	'Tie::Scalar'           => '1.02',
	'Tie::StdHandle'        => '4.2',
	'Tie::SubstrHash'       => '1.00',
	'Time::HiRes'           => '1.9719',
	'Time::Local'           => '1.1901_01',
	'Time::Piece'           => '1.15_01',
	'Time::Piece::Seconds'  => undef,
	'Time::Seconds'         => undef,
	'Time::gmtime'          => '1.03',
	'Time::localtime'       => '1.02',
	'Time::tm'              => '1.00',
	'UNIVERSAL'             => '1.07',
	'Unicode'               => '5.2.0',
	'Unicode::Collate'      => '0.52_01',
	'Unicode::Normalize'    => '1.03',
	'Unicode::UCD'          => '0.28',
	'User::grent'           => '1.01',
	'User::pwent'           => '1.00',
	'VMS::DCLsym'           => '1.04',
	'VMS::Stdio'            => '2.4',
	'Win32'                 => '0.39',
	'Win32API::File'        => '0.1101',
	'Win32API::File::ExtUtils::Myconst2perl'=> '1',
	'Win32CORE'             => '0.02',
	'XS::APItest'           => '0.19',
	'XS::APItest::KeywordRPN'=> '0.004',
	'XS::Typemap'           => '0.03',
	'XSLoader'              => '0.10',
	'XSLoader::XSLoader'    => '0.10',
	'attributes'            => '0.12',
	'autodie'               => '2.06_01',
	'autodie::exception'    => '2.06_01',
	'autodie::exception::system'=> '2.06_01',
	'autodie::hints'        => '2.06_01',
	'autouse'               => '1.06',
	'base'                  => '2.15',
	'bigint'                => '0.23',
	'bignum'                => '0.23',
	'bigrat'                => '0.23',
	'blib'                  => '1.04',
	'bytes'                 => '1.04',
	'charnames'             => '1.08',
	'constant'              => '1.20',
	'deprecate'             => '0.01',
	'diagnostics'           => '1.19',
	'encoding'              => '2.6_01',
	'encoding::warnings'    => '0.11',
	'feature'               => '1.17',
	'fields'                => '2.15',
	'filetest'              => '1.02',
	'if'                    => '0.05',
	'inc::latest'           => '0.3603',
	'integer'               => '1.00',
	'less'                  => '0.03',
	'lib'                   => '0.62',
	'locale'                => '1.00',
	'mro'                   => '1.03',
	'open'                  => '1.07',
	'ops'                   => '1.02',
	'overload'              => '1.10',
	'overload::numbers'     => undef,
	'overloading'           => '0.01',
	'parent'                => '0.223',
	're'                    => '0.11',
	'sigtrap'               => '1.04',
	'sort'                  => '2.01',
	'strict'                => '1.04',
	'subs'                  => '1.00',
	'threads'               => '1.77_02',
	'threads::shared'       => '1.33_01',
	'utf8'                  => '1.08',
	'vars'                  => '1.01',
	'version'               => '0.82',
	'vmsish'                => '1.02',
	'warnings'              => '1.09',
	'warnings::register'    => '1.01',
    },
    5.013003 => {
	'AnyDBM_File'           => '1.00',
	'App::Cpan'             => '1.5701',
	'App::Prove'            => '3.21',
	'App::Prove::State'     => '3.21',
	'App::Prove::State::Result'=> '3.21',
	'App::Prove::State::Result::Test'=> '3.21',
	'Archive::Extract'      => '0.42',
	'Archive::Tar'          => '1.64',
	'Archive::Tar::Constant'=> '1.64',
	'Archive::Tar::File'    => '1.64',
	'Attribute::Handlers'   => '0.88',
	'AutoLoader'            => '5.70',
	'AutoSplit'             => '1.06',
	'B'                     => '1.23',
	'B::Concise'            => '0.79',
	'B::Debug'              => '1.12',
	'B::Deparse'            => '0.98',
	'B::Lint'               => '1.11_01',
	'B::Lint::Debug'        => '0.01',
	'B::Showlex'            => '1.02',
	'B::Terse'              => '1.05',
	'B::Xref'               => '1.02',
	'Benchmark'             => '1.11',
	'CGI'                   => '3.49',
	'CGI::Apache'           => '1.01',
	'CGI::Carp'             => '3.45',
	'CGI::Cookie'           => '1.29',
	'CGI::Fast'             => '1.08',
	'CGI::Pretty'           => '3.46',
	'CGI::Push'             => '1.04',
	'CGI::Switch'           => '1.01',
	'CGI::Util'             => '3.48',
	'CPAN'                  => '1.94_57',
	'CPAN::Author'          => '5.5',
	'CPAN::Bundle'          => '5.5',
	'CPAN::CacheMgr'        => '5.5',
	'CPAN::Complete'        => '5.5',
	'CPAN::Debug'           => '5.5001',
	'CPAN::DeferredCode'    => '5.50',
	'CPAN::Distribution'    => '1.9600',
	'CPAN::Distroprefs'     => '6',
	'CPAN::Distrostatus'    => '5.5',
	'CPAN::Exception::RecursiveDependency'=> '5.5',
	'CPAN::Exception::blocked_urllist'=> '1.0',
	'CPAN::Exception::yaml_not_installed'=> '5.5',
	'CPAN::FTP'             => '5.5004',
	'CPAN::FTP::netrc'      => '1.00',
	'CPAN::FirstTime'       => '5.5301',
	'CPAN::HandleConfig'    => '5.5001',
	'CPAN::Index'           => '1.94',
	'CPAN::InfoObj'         => '5.5',
	'CPAN::Kwalify'         => '5.50',
	'CPAN::LWP::UserAgent'  => '1.94',
	'CPAN::Mirrors'         => '1.77',
	'CPAN::Module'          => '5.5',
	'CPAN::Nox'             => '5.50',
	'CPAN::Prompt'          => '5.5',
	'CPAN::Queue'           => '5.5',
	'CPAN::Shell'           => '5.5001',
	'CPAN::Tarzip'          => '5.5011',
	'CPAN::URL'             => '5.5',
	'CPAN::Version'         => '5.5',
	'CPANPLUS'              => '0.9007',
	'CPANPLUS::Backend'     => undef,
	'CPANPLUS::Backend::RV' => undef,
	'CPANPLUS::Config'      => undef,
	'CPANPLUS::Configure'   => undef,
	'CPANPLUS::Configure::Setup'=> undef,
	'CPANPLUS::Dist'        => undef,
	'CPANPLUS::Dist::Autobundle'=> undef,
	'CPANPLUS::Dist::Base'  => undef,
	'CPANPLUS::Dist::Build' => '0.46',
	'CPANPLUS::Dist::Build::Constants'=> '0.46',
	'CPANPLUS::Dist::MM'    => undef,
	'CPANPLUS::Dist::Sample'=> undef,
	'CPANPLUS::Error'       => undef,
	'CPANPLUS::Internals'   => '0.9007',
	'CPANPLUS::Internals::Constants'=> undef,
	'CPANPLUS::Internals::Constants::Report'=> undef,
	'CPANPLUS::Internals::Extract'=> undef,
	'CPANPLUS::Internals::Fetch'=> undef,
	'CPANPLUS::Internals::Report'=> undef,
	'CPANPLUS::Internals::Search'=> undef,
	'CPANPLUS::Internals::Source'=> undef,
	'CPANPLUS::Internals::Source::Memory'=> undef,
	'CPANPLUS::Internals::Source::SQLite'=> undef,
	'CPANPLUS::Internals::Source::SQLite::Tie'=> undef,
	'CPANPLUS::Internals::Utils'=> undef,
	'CPANPLUS::Internals::Utils::Autoflush'=> undef,
	'CPANPLUS::Module'      => undef,
	'CPANPLUS::Module::Author'=> undef,
	'CPANPLUS::Module::Author::Fake'=> undef,
	'CPANPLUS::Module::Checksums'=> undef,
	'CPANPLUS::Module::Fake'=> undef,
	'CPANPLUS::Module::Signature'=> undef,
	'CPANPLUS::Selfupdate'  => undef,
	'CPANPLUS::Shell'       => undef,
	'CPANPLUS::Shell::Classic'=> '0.0562',
	'CPANPLUS::Shell::Default'=> '0.9007',
	'CPANPLUS::Shell::Default::Plugins::CustomSource'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Remote'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Source'=> undef,
	'Carp'                  => '1.16',
	'Carp::Heavy'           => '1.16',
	'Class::Struct'         => '0.63',
	'Compress::Raw::Bzip2'  => '2.027',
	'Compress::Raw::Zlib'   => '2.027_01',
	'Compress::Zlib'        => '2.027',
	'Config'                => undef,
	'Config::Extensions'    => '0.01',
	'Cwd'                   => '3.31',
	'DB'                    => '1.03',
	'DBM_Filter'            => '0.03',
	'DBM_Filter::compress'  => '0.02',
	'DBM_Filter::encode'    => '0.02',
	'DBM_Filter::int32'     => '0.02',
	'DBM_Filter::null'      => '0.02',
	'DBM_Filter::utf8'      => '0.02',
	'DB_File'               => '1.820',
	'Data::Dumper'          => '2.126',
	'Devel::DProf'          => '20080331.00',
	'Devel::DProf::dprof::V'=> undef,
	'Devel::InnerPackage'   => '0.3',
	'Devel::PPPort'         => '3.19',
	'Devel::Peek'           => '1.04',
	'Devel::SelfStubber'    => '1.03',
	'Digest'                => '1.16',
	'Digest::MD5'           => '2.40',
	'Digest::SHA'           => '5.48',
	'Digest::base'          => '1.16',
	'Digest::file'          => '1.16',
	'DirHandle'             => '1.03',
	'Dumpvalue'             => '1.14',
	'DynaLoader'            => '1.10',
	'Encode'                => '2.39',
	'Encode::Alias'         => '2.12',
	'Encode::Byte'          => '2.04',
	'Encode::CJKConstants'  => '2.02',
	'Encode::CN'            => '2.03',
	'Encode::CN::HZ'        => '2.05',
	'Encode::Config'        => '2.05',
	'Encode::EBCDIC'        => '2.02',
	'Encode::Encoder'       => '2.01',
	'Encode::Encoding'      => '2.05',
	'Encode::GSM0338'       => '2.01',
	'Encode::Guess'         => '2.03',
	'Encode::JP'            => '2.04',
	'Encode::JP::H2Z'       => '2.02',
	'Encode::JP::JIS7'      => '2.04',
	'Encode::KR'            => '2.03',
	'Encode::KR::2022_KR'   => '2.02',
	'Encode::MIME::Header'  => '2.11',
	'Encode::MIME::Header::ISO_2022_JP'=> '1.03',
	'Encode::MIME::Name'    => '1.01',
	'Encode::Symbol'        => '2.02',
	'Encode::TW'            => '2.03',
	'Encode::Unicode'       => '2.07',
	'Encode::Unicode::UTF7' => '2.04',
	'English'               => '1.04',
	'Env'                   => '1.01',
	'Errno'                 => '1.12',
	'Exporter'              => '5.64_03',
	'Exporter::Heavy'       => '5.64_03',
	'ExtUtils::CBuilder'    => '0.2703',
	'ExtUtils::CBuilder::Base'=> '0.2703_01',
	'ExtUtils::CBuilder::Platform::Unix'=> '0.2703',
	'ExtUtils::CBuilder::Platform::VMS'=> '0.2703',
	'ExtUtils::CBuilder::Platform::Windows'=> '0.2703',
	'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.2703',
	'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.2703',
	'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.2703',
	'ExtUtils::CBuilder::Platform::aix'=> '0.2703',
	'ExtUtils::CBuilder::Platform::cygwin'=> '0.2703',
	'ExtUtils::CBuilder::Platform::darwin'=> '0.2703',
	'ExtUtils::CBuilder::Platform::dec_osf'=> '0.2703',
	'ExtUtils::CBuilder::Platform::os2'=> '0.2703',
	'ExtUtils::Command'     => '1.16',
	'ExtUtils::Command::MM' => '6.56',
	'ExtUtils::Constant'    => '0.22',
	'ExtUtils::Constant::Base'=> '0.04',
	'ExtUtils::Constant::ProxySubs'=> '0.06',
	'ExtUtils::Constant::Utils'=> '0.02',
	'ExtUtils::Constant::XS'=> '0.03',
	'ExtUtils::Embed'       => '1.28',
	'ExtUtils::Install'     => '1.55',
	'ExtUtils::Installed'   => '1.999_001',
	'ExtUtils::Liblist'     => '6.56',
	'ExtUtils::Liblist::Kid'=> '6.56',
	'ExtUtils::MM'          => '6.56',
	'ExtUtils::MM_AIX'      => '6.56',
	'ExtUtils::MM_Any'      => '6.56',
	'ExtUtils::MM_BeOS'     => '6.56',
	'ExtUtils::MM_Cygwin'   => '6.56',
	'ExtUtils::MM_DOS'      => '6.56',
	'ExtUtils::MM_Darwin'   => '6.56',
	'ExtUtils::MM_MacOS'    => '6.56',
	'ExtUtils::MM_NW5'      => '6.56',
	'ExtUtils::MM_OS2'      => '6.56',
	'ExtUtils::MM_QNX'      => '6.56',
	'ExtUtils::MM_UWIN'     => '6.56',
	'ExtUtils::MM_Unix'     => '6.5601',
	'ExtUtils::MM_VMS'      => '6.56',
	'ExtUtils::MM_VOS'      => '6.56',
	'ExtUtils::MM_Win32'    => '6.56',
	'ExtUtils::MM_Win95'    => '6.56',
	'ExtUtils::MY'          => '6.56',
	'ExtUtils::MakeMaker'   => '6.5601',
	'ExtUtils::MakeMaker::Config'=> '6.56',
	'ExtUtils::Manifest'    => '1.58',
	'ExtUtils::Mkbootstrap' => '6.56',
	'ExtUtils::Mksymlists'  => '6.56',
	'ExtUtils::Packlist'    => '1.44',
	'ExtUtils::ParseXS'     => '2.2206',
	'ExtUtils::XSSymSet'    => '1.1',
	'ExtUtils::testlib'     => '6.56',
	'Fatal'                 => '2.10',
	'Fcntl'                 => '1.06',
	'File::Basename'        => '2.79',
	'File::CheckTree'       => '4.4',
	'File::Compare'         => '1.1006',
	'File::Copy'            => '2.20',
	'File::DosGlob'         => '1.02',
	'File::Fetch'           => '0.24',
	'File::Find'            => '1.17',
	'File::Glob'            => '1.08',
	'File::GlobMapper'      => '1.000',
	'File::Path'            => '2.08_01',
	'File::Spec'            => '3.31',
	'File::Spec::Cygwin'    => '3.30',
	'File::Spec::Epoc'      => '3.30',
	'File::Spec::Functions' => '3.30',
	'File::Spec::Mac'       => '3.30',
	'File::Spec::OS2'       => '3.30',
	'File::Spec::Unix'      => '3.30',
	'File::Spec::VMS'       => '3.30',
	'File::Spec::Win32'     => '3.30',
	'File::Temp'            => '0.22',
	'File::stat'            => '1.03',
	'FileCache'             => '1.08',
	'FileHandle'            => '2.02',
	'VMS::Filespec'         => '1.12',
	'Filter::Simple'        => '0.84',
	'Filter::Util::Call'    => '1.08',
	'FindBin'               => '1.50',
	'GDBM_File'             => '1.10',
	'Getopt::Long'          => '2.38',
	'Getopt::Std'           => '1.06',
	'Hash::Util'            => '0.08',
	'Hash::Util::FieldHash' => '1.04',
	'I18N::Collate'         => '1.01',
	'I18N::LangTags'        => '0.35_01',
	'I18N::LangTags::Detect'=> '1.04',
	'I18N::LangTags::List'  => '0.35_01',
	'I18N::Langinfo'        => '0.03',
	'IO'                    => '1.25_02',
	'IO::Compress::Adapter::Bzip2'=> '2.027',
	'IO::Compress::Adapter::Deflate'=> '2.027',
	'IO::Compress::Adapter::Identity'=> '2.027',
	'IO::Compress::Base'    => '2.027',
	'IO::Compress::Base::Common'=> '2.027',
	'IO::Compress::Bzip2'   => '2.027',
	'IO::Compress::Deflate' => '2.027',
	'IO::Compress::Gzip'    => '2.027',
	'IO::Compress::Gzip::Constants'=> '2.027',
	'IO::Compress::RawDeflate'=> '2.027',
	'IO::Compress::Zip'     => '2.027',
	'IO::Compress::Zip::Constants'=> '2.027',
	'IO::Compress::Zlib::Constants'=> '2.027',
	'IO::Compress::Zlib::Extra'=> '2.027',
	'IO::Dir'               => '1.07',
	'IO::File'              => '1.14',
	'IO::Handle'            => '1.28',
	'IO::Pipe'              => '1.13',
	'IO::Poll'              => '0.07',
	'IO::Seekable'          => '1.10',
	'IO::Select'            => '1.17',
	'IO::Socket'            => '1.32',
	'IO::Socket::INET'      => '1.31',
	'IO::Socket::UNIX'      => '1.23',
	'IO::Uncompress::Adapter::Bunzip2'=> '2.027',
	'IO::Uncompress::Adapter::Identity'=> '2.027',
	'IO::Uncompress::Adapter::Inflate'=> '2.027',
	'IO::Uncompress::AnyInflate'=> '2.027',
	'IO::Uncompress::AnyUncompress'=> '2.027',
	'IO::Uncompress::Base'  => '2.027',
	'IO::Uncompress::Bunzip2'=> '2.027',
	'IO::Uncompress::Gunzip'=> '2.027',
	'IO::Uncompress::Inflate'=> '2.027',
	'IO::Uncompress::RawInflate'=> '2.027',
	'IO::Uncompress::Unzip' => '2.027',
	'IO::Zlib'              => '1.10',
	'IPC::Cmd'              => '0.60',
	'IPC::Msg'              => '2.03',
	'IPC::Open2'            => '1.03',
	'IPC::Open3'            => '1.06',
	'IPC::Semaphore'        => '2.03',
	'IPC::SharedMem'        => '2.03',
	'IPC::SysV'             => '2.03',
	'List::Util'            => '1.23',
	'List::Util::PP'        => '1.23',
	'List::Util::XS'        => '1.23',
	'Locale::Codes'         => '3.13',
	'Locale::Codes::Country'=> '3.13',
	'Locale::Codes::Currency'=> '3.13',
	'Locale::Codes::Language'=> '3.13',
	'Locale::Codes::Script' => '3.13',
	'Locale::Constants'     => '3.13',
	'Locale::Country'       => '3.13',
	'Locale::Currency'      => '3.13',
	'Locale::Language'      => '3.13',
	'Locale::Maketext'      => '1.15',
	'Locale::Maketext::Guts'=> undef,
	'Locale::Maketext::GutsLoader'=> undef,
	'Locale::Maketext::Simple'=> '0.21',
	'Locale::Script'        => '3.13',
	'Log::Message'          => '0.02',
	'Log::Message::Config'  => '0.01',
	'Log::Message::Handlers'=> undef,
	'Log::Message::Item'    => undef,
	'Log::Message::Simple'  => '0.06',
	'MIME::Base64'          => '3.09',
	'MIME::QuotedPrint'     => '3.09',
	'Math::BigFloat'        => '1.60',
	'Math::BigFloat::Trace' => '0.01',
	'Math::BigInt'          => '1.89_01',
	'Math::BigInt::Calc'    => '0.52',
	'Math::BigInt::CalcEmu' => '0.05',
	'Math::BigInt::FastCalc'=> '0.19',
	'Math::BigInt::Trace'   => '0.01',
	'Math::BigRat'          => '0.24',
	'Math::Complex'         => '1.56',
	'Math::Trig'            => '1.2',
	'Memoize'               => '1.01_03',
	'Memoize::AnyDBM_File'  => '0.65',
	'Memoize::Expire'       => '1.00',
	'Memoize::ExpireFile'   => '1.01',
	'Memoize::ExpireTest'   => '0.65',
	'Memoize::NDBM_File'    => '0.65',
	'Memoize::SDBM_File'    => '0.65',
	'Memoize::Storable'     => '0.65',
	'Module::Build'         => '0.3607',
	'Module::Build::Base'   => '0.3607',
	'Module::Build::Compat' => '0.3607',
	'Module::Build::Config' => '0.3607',
	'Module::Build::ConfigData'=> undef,
	'Module::Build::Cookbook'=> '0.3607',
	'Module::Build::Dumper' => '0.3607',
	'Module::Build::ModuleInfo'=> '0.3607',
	'Module::Build::Notes'  => '0.3607',
	'Module::Build::PPMMaker'=> '0.3607',
	'Module::Build::Platform::Amiga'=> '0.3607',
	'Module::Build::Platform::Default'=> '0.3607',
	'Module::Build::Platform::EBCDIC'=> '0.3607',
	'Module::Build::Platform::MPEiX'=> '0.3607',
	'Module::Build::Platform::MacOS'=> '0.3607',
	'Module::Build::Platform::RiscOS'=> '0.3607',
	'Module::Build::Platform::Unix'=> '0.3607',
	'Module::Build::Platform::VMS'=> '0.3607',
	'Module::Build::Platform::VOS'=> '0.3607',
	'Module::Build::Platform::Windows'=> '0.3607',
	'Module::Build::Platform::aix'=> '0.3607',
	'Module::Build::Platform::cygwin'=> '0.3607',
	'Module::Build::Platform::darwin'=> '0.3607',
	'Module::Build::Platform::os2'=> '0.3607',
	'Module::Build::PodParser'=> '0.3607',
	'Module::Build::Version'=> '0.77',
	'Module::Build::YAML'   => '1.40',
	'Module::CoreList'      => '2.36',
	'Module::Load'          => '0.18',
	'Module::Load::Conditional'=> '0.38',
	'Module::Loaded'        => '0.06',
	'Module::Pluggable'     => '3.9',
	'Module::Pluggable::Object'=> '3.9',
	'Moped::Msg'            => '0.01',
	'NDBM_File'             => '1.08',
	'NEXT'                  => '0.64',
	'Net::Cmd'              => '2.29',
	'Net::Config'           => '1.11',
	'Net::Domain'           => '2.20',
	'Net::FTP'              => '2.77',
	'Net::FTP::A'           => '1.18',
	'Net::FTP::E'           => '0.01',
	'Net::FTP::I'           => '1.12',
	'Net::FTP::L'           => '0.01',
	'Net::FTP::dataconn'    => '0.11',
	'Net::NNTP'             => '2.24',
	'Net::Netrc'            => '2.12',
	'Net::POP3'             => '2.29',
	'Net::Ping'             => '2.36',
	'Net::SMTP'             => '2.31',
	'Net::Time'             => '2.10',
	'Net::hostent'          => '1.01',
	'Net::netent'           => '1.00',
	'Net::protoent'         => '1.00',
	'Net::servent'          => '1.01',
	'O'                     => '1.01',
	'ODBM_File'             => '1.07',
	'Object::Accessor'      => '0.36',
	'Opcode'                => '1.15',
	'POSIX'                 => '1.19',
	'Package::Constants'    => '0.02',
	'Params::Check'         => '0.26',
	'Parse::CPAN::Meta'     => '1.40',
	'PerlIO'                => '1.06',
	'PerlIO::encoding'      => '0.12',
	'PerlIO::scalar'        => '0.08',
	'PerlIO::via'           => '0.09',
	'PerlIO::via::QuotedPrint'=> '0.06',
	'Pod::Checker'          => '1.45',
	'Pod::Escapes'          => '1.04',
	'Pod::Find'             => '1.35',
	'Pod::Functions'        => '1.04',
	'Pod::Html'             => '1.09',
	'Pod::InputObjects'     => '1.31',
	'Pod::LaTeX'            => '0.58',
	'Pod::Man'              => '2.23',
	'Pod::ParseLink'        => '1.10',
	'Pod::ParseUtils'       => '1.36',
	'Pod::Parser'           => '1.37',
	'Pod::Perldoc'          => '3.15_02',
	'Pod::Perldoc::BaseTo'  => undef,
	'Pod::Perldoc::GetOptsOO'=> undef,
	'Pod::Perldoc::ToChecker'=> undef,
	'Pod::Perldoc::ToMan'   => undef,
	'Pod::Perldoc::ToNroff' => undef,
	'Pod::Perldoc::ToPod'   => undef,
	'Pod::Perldoc::ToRtf'   => undef,
	'Pod::Perldoc::ToText'  => undef,
	'Pod::Perldoc::ToTk'    => undef,
	'Pod::Perldoc::ToXml'   => undef,
	'Pod::PlainText'        => '2.04',
	'Pod::Select'           => '1.36',
	'Pod::Simple'           => '3.14',
	'Pod::Simple::BlackBox' => '3.14',
	'Pod::Simple::Checker'  => '3.14',
	'Pod::Simple::Debug'    => '3.14',
	'Pod::Simple::DumpAsText'=> '3.14',
	'Pod::Simple::DumpAsXML'=> '3.14',
	'Pod::Simple::HTML'     => '3.14',
	'Pod::Simple::HTMLBatch'=> '3.14',
	'Pod::Simple::HTMLLegacy'=> '5.01',
	'Pod::Simple::LinkSection'=> '3.14',
	'Pod::Simple::Methody'  => '3.14',
	'Pod::Simple::Progress' => '3.14',
	'Pod::Simple::PullParser'=> '3.14',
	'Pod::Simple::PullParserEndToken'=> '3.14',
	'Pod::Simple::PullParserStartToken'=> '3.14',
	'Pod::Simple::PullParserTextToken'=> '3.14',
	'Pod::Simple::PullParserToken'=> '3.14',
	'Pod::Simple::RTF'      => '3.14',
	'Pod::Simple::Search'   => '3.14',
	'Pod::Simple::SimpleTree'=> '3.14',
	'Pod::Simple::Text'     => '3.14',
	'Pod::Simple::TextContent'=> '3.14',
	'Pod::Simple::TiedOutFH'=> '3.14',
	'Pod::Simple::Transcode'=> '3.14',
	'Pod::Simple::TranscodeDumb'=> '3.14',
	'Pod::Simple::TranscodeSmart'=> '3.14',
	'Pod::Simple::XHTML'    => '3.14',
	'Pod::Simple::XMLOutStream'=> '3.14',
	'Pod::Text'             => '3.14',
	'Pod::Text::Color'      => '2.06',
	'Pod::Text::Overstrike' => '2.04',
	'Pod::Text::Termcap'    => '2.06',
	'Pod::Usage'            => '1.36',
	'SDBM_File'             => '1.06',
	'Safe'                  => '2.27',
	'Scalar::Util'          => '1.23',
	'Scalar::Util::PP'      => '1.23',
	'Search::Dict'          => '1.03',
	'SelectSaver'           => '1.02',
	'SelfLoader'            => '1.17',
	'Shell'                 => '0.72_01',
	'Socket'                => '1.89',
	'Storable'              => '2.22',
	'Symbol'                => '1.07',
	'Sys::Hostname'         => '1.11',
	'Sys::Syslog'           => '0.27',
	'Sys::Syslog::win32::Win32'=> undef,
	'TAP::Base'             => '3.21',
	'TAP::Formatter::Base'  => '3.21',
	'TAP::Formatter::Color' => '3.21',
	'TAP::Formatter::Console'=> '3.21',
	'TAP::Formatter::Console::ParallelSession'=> '3.21',
	'TAP::Formatter::Console::Session'=> '3.21',
	'TAP::Formatter::File'  => '3.21',
	'TAP::Formatter::File::Session'=> '3.21',
	'TAP::Formatter::Session'=> '3.21',
	'TAP::Harness'          => '3.21',
	'TAP::Object'           => '3.21',
	'TAP::Parser'           => '3.21',
	'TAP::Parser::Aggregator'=> '3.21',
	'TAP::Parser::Grammar'  => '3.21',
	'TAP::Parser::Iterator' => '3.21',
	'TAP::Parser::Iterator::Array'=> '3.21',
	'TAP::Parser::Iterator::Process'=> '3.21',
	'TAP::Parser::Iterator::Stream'=> '3.21',
	'TAP::Parser::IteratorFactory'=> '3.21',
	'TAP::Parser::Multiplexer'=> '3.21',
	'TAP::Parser::Result'   => '3.21',
	'TAP::Parser::Result::Bailout'=> '3.21',
	'TAP::Parser::Result::Comment'=> '3.21',
	'TAP::Parser::Result::Plan'=> '3.21',
	'TAP::Parser::Result::Pragma'=> '3.21',
	'TAP::Parser::Result::Test'=> '3.21',
	'TAP::Parser::Result::Unknown'=> '3.21',
	'TAP::Parser::Result::Version'=> '3.21',
	'TAP::Parser::Result::YAML'=> '3.21',
	'TAP::Parser::ResultFactory'=> '3.21',
	'TAP::Parser::Scheduler'=> '3.21',
	'TAP::Parser::Scheduler::Job'=> '3.21',
	'TAP::Parser::Scheduler::Spinner'=> '3.21',
	'TAP::Parser::Source'   => '3.21',
	'TAP::Parser::SourceHandler'=> '3.21',
	'TAP::Parser::SourceHandler::Executable'=> '3.21',
	'TAP::Parser::SourceHandler::File'=> '3.21',
	'TAP::Parser::SourceHandler::Handle'=> '3.21',
	'TAP::Parser::SourceHandler::Perl'=> '3.21',
	'TAP::Parser::SourceHandler::RawTAP'=> '3.21',
	'TAP::Parser::SourceHandler::pgTAP'=> '3.21',
	'TAP::Parser::Utils'    => '3.21',
	'TAP::Parser::YAMLish::Reader'=> '3.21',
	'TAP::Parser::YAMLish::Writer'=> '3.21',
	'Term::ANSIColor'       => '3.00',
	'Term::Cap'             => '1.12',
	'Term::Complete'        => '1.402',
	'Term::ReadLine'        => '1.07',
	'Term::UI'              => '0.20',
	'Term::UI::History'     => undef,
	'Test'                  => '1.25_02',
	'Test::Builder'         => '0.94',
	'Test::Builder::Module' => '0.94',
	'Test::Builder::Tester' => '1.18',
	'Test::Builder::Tester::Color'=> '1.18',
	'Test::Harness'         => '3.21',
	'Test::More'            => '0.94',
	'Test::Simple'          => '0.94',
	'Text::Abbrev'          => '1.01',
	'Text::Balanced'        => '2.02',
	'Text::ParseWords'      => '3.27',
	'Text::Soundex'         => '3.03_01',
	'Text::Tabs'            => '2009.0305',
	'Text::Wrap'            => '2009.0305',
	'Thread'                => '3.02',
	'Thread::Queue'         => '2.11',
	'Thread::Semaphore'     => '2.11',
	'Tie::Array'            => '1.04',
	'Tie::File'             => '0.97_02',
	'Tie::Handle'           => '4.2',
	'Tie::Hash'             => '1.03',
	'Tie::Hash::NamedCapture'=> '0.06',
	'Tie::Memoize'          => '1.1',
	'Tie::RefHash'          => '1.38',
	'Tie::Scalar'           => '1.02',
	'Tie::StdHandle'        => '4.2',
	'Tie::SubstrHash'       => '1.00',
	'Time::HiRes'           => '1.9721',
	'Time::Local'           => '1.1901_01',
	'Time::Piece'           => '1.20_01',
	'Time::Piece::Seconds'  => undef,
	'Time::Seconds'         => undef,
	'Time::gmtime'          => '1.03',
	'Time::localtime'       => '1.02',
	'Time::tm'              => '1.00',
	'UNIVERSAL'             => '1.07',
	'Unicode'               => '5.2.0',
	'Unicode::Collate'      => '0.53',
	'Unicode::Normalize'    => '1.06',
	'Unicode::UCD'          => '0.29',
	'User::grent'           => '1.01',
	'User::pwent'           => '1.00',
	'VMS::DCLsym'           => '1.04',
	'VMS::Stdio'            => '2.4',
	'Win32'                 => '0.39',
	'Win32API::File'        => '0.1101',
	'Win32API::File::ExtUtils::Myconst2perl'=> '1',
	'Win32CORE'             => '0.02',
	'XS::APItest'           => '0.19',
	'XS::APItest::KeywordRPN'=> '0.004',
	'XS::Typemap'           => '0.03',
	'XSLoader'              => '0.10',
	'XSLoader::XSLoader'    => '0.10',
	'attributes'            => '0.12',
	'autodie'               => '2.10',
	'autodie::exception'    => '2.10',
	'autodie::exception::system'=> '2.10',
	'autodie::hints'        => '2.10',
	'autouse'               => '1.06',
	'base'                  => '2.15',
	'bigint'                => '0.23',
	'bignum'                => '0.23',
	'bigrat'                => '0.23',
	'blib'                  => '1.05',
	'bytes'                 => '1.04',
	'charnames'             => '1.11',
	'constant'              => '1.20',
	'deprecate'             => '0.01',
	'diagnostics'           => '1.20',
	'encoding'              => '2.6_01',
	'encoding::warnings'    => '0.11',
	'feature'               => '1.17',
	'fields'                => '2.15',
	'filetest'              => '1.02',
	'if'                    => '0.05',
	'inc::latest'           => '0.3607',
	'integer'               => '1.00',
	'less'                  => '0.03',
	'lib'                   => '0.63',
	'locale'                => '1.00',
	'mro'                   => '1.03',
	'open'                  => '1.07',
	'ops'                   => '1.02',
	'overload'              => '1.10',
	'overload::numbers'     => undef,
	'overloading'           => '0.01',
	'parent'                => '0.223',
	're'                    => '0.12',
	'sigtrap'               => '1.04',
	'sort'                  => '2.01',
	'strict'                => '1.04',
	'subs'                  => '1.00',
	'threads'               => '1.77_03',
	'threads::shared'       => '1.33_02',
	'utf8'                  => '1.08',
	'vars'                  => '1.02',
	'version'               => '0.82',
	'vmsish'                => '1.02',
	'warnings'              => '1.10',
	'warnings::register'    => '1.01',
    },
    5.013004 => {
	'AnyDBM_File'           => '1.00',
	'App::Cpan'             => '1.5701',
	'App::Prove'            => '3.22',
	'App::Prove::State'     => '3.22',
	'App::Prove::State::Result'=> '3.22',
	'App::Prove::State::Result::Test'=> '3.22',
	'Archive::Extract'      => '0.42',
	'Archive::Tar'          => '1.68',
	'Archive::Tar::Constant'=> '1.68',
	'Archive::Tar::File'    => '1.68',
	'Attribute::Handlers'   => '0.88',
	'AutoLoader'            => '5.70',
	'AutoSplit'             => '1.06',
	'B'                     => '1.23',
	'B::Concise'            => '0.79',
	'B::Debug'              => '1.12',
	'B::Deparse'            => '0.98',
	'B::Lint'               => '1.12',
	'B::Lint::Debug'        => '1.12',
	'B::Showlex'            => '1.02',
	'B::Terse'              => '1.05',
	'B::Xref'               => '1.02',
	'Benchmark'             => '1.11',
	'CGI'                   => '3.49',
	'CGI::Apache'           => '1.01',
	'CGI::Carp'             => '3.45',
	'CGI::Cookie'           => '1.29',
	'CGI::Fast'             => '1.08',
	'CGI::Pretty'           => '3.46',
	'CGI::Push'             => '1.04',
	'CGI::Switch'           => '1.01',
	'CGI::Util'             => '3.48',
	'CPAN'                  => '1.94_57',
	'CPAN::Author'          => '5.5',
	'CPAN::Bundle'          => '5.5',
	'CPAN::CacheMgr'        => '5.5',
	'CPAN::Complete'        => '5.5',
	'CPAN::Debug'           => '5.5001',
	'CPAN::DeferredCode'    => '5.50',
	'CPAN::Distribution'    => '1.9600',
	'CPAN::Distroprefs'     => '6',
	'CPAN::Distrostatus'    => '5.5',
	'CPAN::Exception::RecursiveDependency'=> '5.5',
	'CPAN::Exception::blocked_urllist'=> '1.0',
	'CPAN::Exception::yaml_not_installed'=> '5.5',
	'CPAN::FTP'             => '5.5004',
	'CPAN::FTP::netrc'      => '1.00',
	'CPAN::FirstTime'       => '5.5301',
	'CPAN::HandleConfig'    => '5.5001',
	'CPAN::Index'           => '1.94',
	'CPAN::InfoObj'         => '5.5',
	'CPAN::Kwalify'         => '5.50',
	'CPAN::LWP::UserAgent'  => '1.94',
	'CPAN::Mirrors'         => '1.77',
	'CPAN::Module'          => '5.5',
	'CPAN::Nox'             => '5.50',
	'CPAN::Prompt'          => '5.5',
	'CPAN::Queue'           => '5.5',
	'CPAN::Shell'           => '5.5001',
	'CPAN::Tarzip'          => '5.5011',
	'CPAN::URL'             => '5.5',
	'CPAN::Version'         => '5.5',
	'CPANPLUS'              => '0.9007',
	'CPANPLUS::Backend'     => undef,
	'CPANPLUS::Backend::RV' => undef,
	'CPANPLUS::Config'      => undef,
	'CPANPLUS::Configure'   => undef,
	'CPANPLUS::Configure::Setup'=> undef,
	'CPANPLUS::Dist'        => undef,
	'CPANPLUS::Dist::Autobundle'=> undef,
	'CPANPLUS::Dist::Base'  => undef,
	'CPANPLUS::Dist::Build' => '0.46',
	'CPANPLUS::Dist::Build::Constants'=> '0.46',
	'CPANPLUS::Dist::MM'    => undef,
	'CPANPLUS::Dist::Sample'=> undef,
	'CPANPLUS::Error'       => undef,
	'CPANPLUS::Internals'   => '0.9007',
	'CPANPLUS::Internals::Constants'=> undef,
	'CPANPLUS::Internals::Constants::Report'=> undef,
	'CPANPLUS::Internals::Extract'=> undef,
	'CPANPLUS::Internals::Fetch'=> undef,
	'CPANPLUS::Internals::Report'=> undef,
	'CPANPLUS::Internals::Search'=> undef,
	'CPANPLUS::Internals::Source'=> undef,
	'CPANPLUS::Internals::Source::Memory'=> undef,
	'CPANPLUS::Internals::Source::SQLite'=> undef,
	'CPANPLUS::Internals::Source::SQLite::Tie'=> undef,
	'CPANPLUS::Internals::Utils'=> undef,
	'CPANPLUS::Internals::Utils::Autoflush'=> undef,
	'CPANPLUS::Module'      => undef,
	'CPANPLUS::Module::Author'=> undef,
	'CPANPLUS::Module::Author::Fake'=> undef,
	'CPANPLUS::Module::Checksums'=> undef,
	'CPANPLUS::Module::Fake'=> undef,
	'CPANPLUS::Module::Signature'=> undef,
	'CPANPLUS::Selfupdate'  => undef,
	'CPANPLUS::Shell'       => undef,
	'CPANPLUS::Shell::Classic'=> '0.0562',
	'CPANPLUS::Shell::Default'=> '0.9007',
	'CPANPLUS::Shell::Default::Plugins::CustomSource'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Remote'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Source'=> undef,
	'Carp'                  => '1.18',
	'Carp::Heavy'           => '1.18',
	'Class::Struct'         => '0.63',
	'Compress::Raw::Bzip2'  => '2.030',
	'Compress::Raw::Zlib'   => '2.030',
	'Compress::Zlib'        => '2.030',
	'Config'                => undef,
	'Config::Extensions'    => '0.01',
	'Cwd'                   => '3.31',
	'DB'                    => '1.03',
	'DBM_Filter'            => '0.03',
	'DBM_Filter::compress'  => '0.02',
	'DBM_Filter::encode'    => '0.02',
	'DBM_Filter::int32'     => '0.02',
	'DBM_Filter::null'      => '0.02',
	'DBM_Filter::utf8'      => '0.02',
	'DB_File'               => '1.820',
	'Data::Dumper'          => '2.126',
	'Devel::DProf'          => '20080331.00',
	'Devel::DProf::dprof::V'=> undef,
	'Devel::InnerPackage'   => '0.3',
	'Devel::PPPort'         => '3.19',
	'Devel::Peek'           => '1.04',
	'Devel::SelfStubber'    => '1.03',
	'Digest'                => '1.16',
	'Digest::MD5'           => '2.40',
	'Digest::SHA'           => '5.48',
	'Digest::base'          => '1.16',
	'Digest::file'          => '1.16',
	'DirHandle'             => '1.03',
	'Dumpvalue'             => '1.14',
	'DynaLoader'            => '1.10',
	'Encode'                => '2.39',
	'Encode::Alias'         => '2.12',
	'Encode::Byte'          => '2.04',
	'Encode::CJKConstants'  => '2.02',
	'Encode::CN'            => '2.03',
	'Encode::CN::HZ'        => '2.05',
	'Encode::Config'        => '2.05',
	'Encode::EBCDIC'        => '2.02',
	'Encode::Encoder'       => '2.01',
	'Encode::Encoding'      => '2.05',
	'Encode::GSM0338'       => '2.01',
	'Encode::Guess'         => '2.03',
	'Encode::JP'            => '2.04',
	'Encode::JP::H2Z'       => '2.02',
	'Encode::JP::JIS7'      => '2.04',
	'Encode::KR'            => '2.03',
	'Encode::KR::2022_KR'   => '2.02',
	'Encode::MIME::Header'  => '2.11',
	'Encode::MIME::Header::ISO_2022_JP'=> '1.03',
	'Encode::MIME::Name'    => '1.01',
	'Encode::Symbol'        => '2.02',
	'Encode::TW'            => '2.03',
	'Encode::Unicode'       => '2.07',
	'Encode::Unicode::UTF7' => '2.04',
	'English'               => '1.04',
	'Env'                   => '1.01',
	'Errno'                 => '1.12',
	'Exporter'              => '5.64_03',
	'Exporter::Heavy'       => '5.64_03',
	'ExtUtils::CBuilder'    => '0.2703',
	'ExtUtils::CBuilder::Base'=> '0.2703_01',
	'ExtUtils::CBuilder::Platform::Unix'=> '0.2703',
	'ExtUtils::CBuilder::Platform::VMS'=> '0.2703',
	'ExtUtils::CBuilder::Platform::Windows'=> '0.2703',
	'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.2703',
	'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.2703',
	'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.2703',
	'ExtUtils::CBuilder::Platform::aix'=> '0.2703',
	'ExtUtils::CBuilder::Platform::cygwin'=> '0.2703',
	'ExtUtils::CBuilder::Platform::darwin'=> '0.2703',
	'ExtUtils::CBuilder::Platform::dec_osf'=> '0.2703',
	'ExtUtils::CBuilder::Platform::os2'=> '0.2703',
	'ExtUtils::Command'     => '1.16',
	'ExtUtils::Command::MM' => '6.56',
	'ExtUtils::Constant'    => '0.22',
	'ExtUtils::Constant::Base'=> '0.04',
	'ExtUtils::Constant::ProxySubs'=> '0.06',
	'ExtUtils::Constant::Utils'=> '0.02',
	'ExtUtils::Constant::XS'=> '0.03',
	'ExtUtils::Embed'       => '1.28',
	'ExtUtils::Install'     => '1.55',
	'ExtUtils::Installed'   => '1.999_001',
	'ExtUtils::Liblist'     => '6.56',
	'ExtUtils::Liblist::Kid'=> '6.56',
	'ExtUtils::MM'          => '6.56',
	'ExtUtils::MM_AIX'      => '6.56',
	'ExtUtils::MM_Any'      => '6.56',
	'ExtUtils::MM_BeOS'     => '6.56',
	'ExtUtils::MM_Cygwin'   => '6.56',
	'ExtUtils::MM_DOS'      => '6.56',
	'ExtUtils::MM_Darwin'   => '6.56',
	'ExtUtils::MM_MacOS'    => '6.56',
	'ExtUtils::MM_NW5'      => '6.56',
	'ExtUtils::MM_OS2'      => '6.56',
	'ExtUtils::MM_QNX'      => '6.56',
	'ExtUtils::MM_UWIN'     => '6.56',
	'ExtUtils::MM_Unix'     => '6.5601',
	'ExtUtils::MM_VMS'      => '6.56',
	'ExtUtils::MM_VOS'      => '6.56',
	'ExtUtils::MM_Win32'    => '6.56',
	'ExtUtils::MM_Win95'    => '6.56',
	'ExtUtils::MY'          => '6.56',
	'ExtUtils::MakeMaker'   => '6.5601',
	'ExtUtils::MakeMaker::Config'=> '6.56',
	'ExtUtils::Manifest'    => '1.58',
	'ExtUtils::Miniperl'    => undef,
	'ExtUtils::Mkbootstrap' => '6.56',
	'ExtUtils::Mksymlists'  => '6.56',
	'ExtUtils::Packlist'    => '1.44',
	'ExtUtils::ParseXS'     => '2.2207',
	'ExtUtils::XSSymSet'    => '1.1',
	'ExtUtils::testlib'     => '6.56',
	'Fatal'                 => '2.10',
	'Fcntl'                 => '1.06',
	'File::Basename'        => '2.79',
	'File::CheckTree'       => '4.4',
	'File::Compare'         => '1.1006',
	'File::Copy'            => '2.20',
	'File::DosGlob'         => '1.02',
	'File::Fetch'           => '0.24',
	'File::Find'            => '1.17',
	'File::Glob'            => '1.08',
	'File::GlobMapper'      => '1.000',
	'File::Path'            => '2.08_01',
	'File::Spec'            => '3.31_01',
	'File::Spec::Cygwin'    => '3.30',
	'File::Spec::Epoc'      => '3.30',
	'File::Spec::Functions' => '3.30',
	'File::Spec::Mac'       => '3.30',
	'File::Spec::OS2'       => '3.30',
	'File::Spec::Unix'      => '3.30',
	'File::Spec::VMS'       => '3.30',
	'File::Spec::Win32'     => '3.30',
	'File::Temp'            => '0.22',
	'File::stat'            => '1.03',
	'FileCache'             => '1.08',
	'FileHandle'            => '2.02',
	'Filter::Simple'        => '0.84',
	'Filter::Util::Call'    => '1.08',
	'FindBin'               => '1.50',
	'GDBM_File'             => '1.10',
	'Getopt::Long'          => '2.38',
	'Getopt::Std'           => '1.06',
	'Hash::Util'            => '0.08',
	'Hash::Util::FieldHash' => '1.04',
	'I18N::Collate'         => '1.01',
	'I18N::LangTags'        => '0.35_01',
	'I18N::LangTags::Detect'=> '1.04',
	'I18N::LangTags::List'  => '0.35_01',
	'I18N::Langinfo'        => '0.04',
	'IO'                    => '1.25_02',
	'IO::Compress::Adapter::Bzip2'=> '2.030',
	'IO::Compress::Adapter::Deflate'=> '2.030',
	'IO::Compress::Adapter::Identity'=> '2.030',
	'IO::Compress::Base'    => '2.030',
	'IO::Compress::Base::Common'=> '2.030',
	'IO::Compress::Bzip2'   => '2.030',
	'IO::Compress::Deflate' => '2.030',
	'IO::Compress::Gzip'    => '2.030',
	'IO::Compress::Gzip::Constants'=> '2.030',
	'IO::Compress::RawDeflate'=> '2.030',
	'IO::Compress::Zip'     => '2.030',
	'IO::Compress::Zip::Constants'=> '2.030',
	'IO::Compress::Zlib::Constants'=> '2.030',
	'IO::Compress::Zlib::Extra'=> '2.030',
	'IO::Dir'               => '1.07',
	'IO::File'              => '1.14',
	'IO::Handle'            => '1.28',
	'IO::Pipe'              => '1.13',
	'IO::Poll'              => '0.07',
	'IO::Seekable'          => '1.10',
	'IO::Select'            => '1.17',
	'IO::Socket'            => '1.32',
	'IO::Socket::INET'      => '1.31',
	'IO::Socket::UNIX'      => '1.23',
	'IO::Uncompress::Adapter::Bunzip2'=> '2.030',
	'IO::Uncompress::Adapter::Identity'=> '2.030',
	'IO::Uncompress::Adapter::Inflate'=> '2.030',
	'IO::Uncompress::AnyInflate'=> '2.030',
	'IO::Uncompress::AnyUncompress'=> '2.030',
	'IO::Uncompress::Base'  => '2.030',
	'IO::Uncompress::Bunzip2'=> '2.030',
	'IO::Uncompress::Gunzip'=> '2.030',
	'IO::Uncompress::Inflate'=> '2.030',
	'IO::Uncompress::RawInflate'=> '2.030',
	'IO::Uncompress::Unzip' => '2.030',
	'IO::Zlib'              => '1.10',
	'IPC::Cmd'              => '0.60',
	'IPC::Msg'              => '2.03',
	'IPC::Open2'            => '1.03',
	'IPC::Open3'            => '1.06',
	'IPC::Semaphore'        => '2.03',
	'IPC::SharedMem'        => '2.03',
	'IPC::SysV'             => '2.03',
	'List::Util'            => '1.23',
	'List::Util::PP'        => '1.23',
	'List::Util::XS'        => '1.23',
	'Locale::Codes'         => '3.13',
	'Locale::Codes::Country'=> '3.13',
	'Locale::Codes::Currency'=> '3.13',
	'Locale::Codes::Language'=> '3.13',
	'Locale::Codes::Script' => '3.13',
	'Locale::Constants'     => '3.13',
	'Locale::Country'       => '3.13',
	'Locale::Currency'      => '3.13',
	'Locale::Language'      => '3.13',
	'Locale::Maketext'      => '1.15',
	'Locale::Maketext::Guts'=> undef,
	'Locale::Maketext::GutsLoader'=> undef,
	'Locale::Maketext::Simple'=> '0.21',
	'Locale::Script'        => '3.13',
	'Log::Message'          => '0.02',
	'Log::Message::Config'  => '0.01',
	'Log::Message::Handlers'=> undef,
	'Log::Message::Item'    => undef,
	'Log::Message::Simple'  => '0.06',
	'MIME::Base64'          => '3.09',
	'MIME::QuotedPrint'     => '3.09',
	'Math::BigFloat'        => '1.60',
	'Math::BigFloat::Trace' => '0.01',
	'Math::BigInt'          => '1.89_01',
	'Math::BigInt::Calc'    => '0.52',
	'Math::BigInt::CalcEmu' => '0.05',
	'Math::BigInt::FastCalc'=> '0.19',
	'Math::BigInt::Trace'   => '0.01',
	'Math::BigRat'          => '0.24',
	'Math::Complex'         => '1.56',
	'Math::Trig'            => '1.2',
	'Memoize'               => '1.01_03',
	'Memoize::AnyDBM_File'  => '0.65',
	'Memoize::Expire'       => '1.00',
	'Memoize::ExpireFile'   => '1.01',
	'Memoize::ExpireTest'   => '0.65',
	'Memoize::NDBM_File'    => '0.65',
	'Memoize::SDBM_File'    => '0.65',
	'Memoize::Storable'     => '0.65',
	'Module::Build'         => '0.3607',
	'Module::Build::Base'   => '0.3607',
	'Module::Build::Compat' => '0.3607',
	'Module::Build::Config' => '0.3607',
	'Module::Build::ConfigData'=> undef,
	'Module::Build::Cookbook'=> '0.3607',
	'Module::Build::Dumper' => '0.3607',
	'Module::Build::ModuleInfo'=> '0.3607',
	'Module::Build::Notes'  => '0.3607',
	'Module::Build::PPMMaker'=> '0.3607',
	'Module::Build::Platform::Amiga'=> '0.3607',
	'Module::Build::Platform::Default'=> '0.3607',
	'Module::Build::Platform::EBCDIC'=> '0.3607',
	'Module::Build::Platform::MPEiX'=> '0.3607',
	'Module::Build::Platform::MacOS'=> '0.3607',
	'Module::Build::Platform::RiscOS'=> '0.3607',
	'Module::Build::Platform::Unix'=> '0.3607',
	'Module::Build::Platform::VMS'=> '0.3607',
	'Module::Build::Platform::VOS'=> '0.3607',
	'Module::Build::Platform::Windows'=> '0.3607',
	'Module::Build::Platform::aix'=> '0.3607',
	'Module::Build::Platform::cygwin'=> '0.3607',
	'Module::Build::Platform::darwin'=> '0.3607',
	'Module::Build::Platform::os2'=> '0.3607',
	'Module::Build::PodParser'=> '0.3607',
	'Module::Build::Version'=> '0.77',
	'Module::Build::YAML'   => '1.40',
	'Module::CoreList'      => '2.37',
	'Module::Load'          => '0.18',
	'Module::Load::Conditional'=> '0.38',
	'Module::Loaded'        => '0.06',
	'Module::Pluggable'     => '3.9',
	'Module::Pluggable::Object'=> '3.9',
	'Moped::Msg'            => '0.01',
	'NDBM_File'             => '1.08',
	'NEXT'                  => '0.64',
	'Net::Cmd'              => '2.29',
	'Net::Config'           => '1.11',
	'Net::Domain'           => '2.20',
	'Net::FTP'              => '2.77',
	'Net::FTP::A'           => '1.18',
	'Net::FTP::E'           => '0.01',
	'Net::FTP::I'           => '1.12',
	'Net::FTP::L'           => '0.01',
	'Net::FTP::dataconn'    => '0.11',
	'Net::NNTP'             => '2.24',
	'Net::Netrc'            => '2.12',
	'Net::POP3'             => '2.29',
	'Net::Ping'             => '2.36',
	'Net::SMTP'             => '2.31',
	'Net::Time'             => '2.10',
	'Net::hostent'          => '1.01',
	'Net::netent'           => '1.00',
	'Net::protoent'         => '1.00',
	'Net::servent'          => '1.01',
	'O'                     => '1.01',
	'ODBM_File'             => '1.07',
	'Object::Accessor'      => '0.36',
	'Opcode'                => '1.15',
	'POSIX'                 => '1.19',
	'Package::Constants'    => '0.02',
	'Params::Check'         => '0.26',
	'Parse::CPAN::Meta'     => '1.40',
	'PerlIO'                => '1.06',
	'PerlIO::encoding'      => '0.12',
	'PerlIO::scalar'        => '0.08',
	'PerlIO::via'           => '0.09',
	'PerlIO::via::QuotedPrint'=> '0.06',
	'Pod::Checker'          => '1.45',
	'Pod::Escapes'          => '1.04',
	'Pod::Find'             => '1.35',
	'Pod::Functions'        => '1.04',
	'Pod::Html'             => '1.09',
	'Pod::InputObjects'     => '1.31',
	'Pod::LaTeX'            => '0.58',
	'Pod::Man'              => '2.23',
	'Pod::ParseLink'        => '1.10',
	'Pod::ParseUtils'       => '1.36',
	'Pod::Parser'           => '1.37',
	'Pod::Perldoc'          => '3.15_02',
	'Pod::Perldoc::BaseTo'  => undef,
	'Pod::Perldoc::GetOptsOO'=> undef,
	'Pod::Perldoc::ToChecker'=> undef,
	'Pod::Perldoc::ToMan'   => undef,
	'Pod::Perldoc::ToNroff' => undef,
	'Pod::Perldoc::ToPod'   => undef,
	'Pod::Perldoc::ToRtf'   => undef,
	'Pod::Perldoc::ToText'  => undef,
	'Pod::Perldoc::ToTk'    => undef,
	'Pod::Perldoc::ToXml'   => undef,
	'Pod::PlainText'        => '2.04',
	'Pod::Select'           => '1.36',
	'Pod::Simple'           => '3.14',
	'Pod::Simple::BlackBox' => '3.14',
	'Pod::Simple::Checker'  => '3.14',
	'Pod::Simple::Debug'    => '3.14',
	'Pod::Simple::DumpAsText'=> '3.14',
	'Pod::Simple::DumpAsXML'=> '3.14',
	'Pod::Simple::HTML'     => '3.14',
	'Pod::Simple::HTMLBatch'=> '3.14',
	'Pod::Simple::HTMLLegacy'=> '5.01',
	'Pod::Simple::LinkSection'=> '3.14',
	'Pod::Simple::Methody'  => '3.14',
	'Pod::Simple::Progress' => '3.14',
	'Pod::Simple::PullParser'=> '3.14',
	'Pod::Simple::PullParserEndToken'=> '3.14',
	'Pod::Simple::PullParserStartToken'=> '3.14',
	'Pod::Simple::PullParserTextToken'=> '3.14',
	'Pod::Simple::PullParserToken'=> '3.14',
	'Pod::Simple::RTF'      => '3.14',
	'Pod::Simple::Search'   => '3.14',
	'Pod::Simple::SimpleTree'=> '3.14',
	'Pod::Simple::Text'     => '3.14',
	'Pod::Simple::TextContent'=> '3.14',
	'Pod::Simple::TiedOutFH'=> '3.14',
	'Pod::Simple::Transcode'=> '3.14',
	'Pod::Simple::TranscodeDumb'=> '3.14',
	'Pod::Simple::TranscodeSmart'=> '3.14',
	'Pod::Simple::XHTML'    => '3.14',
	'Pod::Simple::XMLOutStream'=> '3.14',
	'Pod::Text'             => '3.14',
	'Pod::Text::Color'      => '2.06',
	'Pod::Text::Overstrike' => '2.04',
	'Pod::Text::Termcap'    => '2.06',
	'Pod::Usage'            => '1.36',
	'SDBM_File'             => '1.06',
	'Safe'                  => '2.27',
	'Scalar::Util'          => '1.23',
	'Scalar::Util::PP'      => '1.23',
	'Search::Dict'          => '1.03',
	'SelectSaver'           => '1.02',
	'SelfLoader'            => '1.17',
	'Shell'                 => '0.72_01',
	'Socket'                => '1.89',
	'Storable'              => '2.22',
	'Symbol'                => '1.07',
	'Sys::Hostname'         => '1.11',
	'Sys::Syslog'           => '0.27',
	'Sys::Syslog::win32::Win32'=> undef,
	'TAP::Base'             => '3.22',
	'TAP::Formatter::Base'  => '3.22',
	'TAP::Formatter::Color' => '3.22',
	'TAP::Formatter::Console'=> '3.22',
	'TAP::Formatter::Console::ParallelSession'=> '3.22',
	'TAP::Formatter::Console::Session'=> '3.22',
	'TAP::Formatter::File'  => '3.22',
	'TAP::Formatter::File::Session'=> '3.22',
	'TAP::Formatter::Session'=> '3.22',
	'TAP::Harness'          => '3.22',
	'TAP::Object'           => '3.22',
	'TAP::Parser'           => '3.22',
	'TAP::Parser::Aggregator'=> '3.22',
	'TAP::Parser::Grammar'  => '3.22',
	'TAP::Parser::Iterator' => '3.22',
	'TAP::Parser::Iterator::Array'=> '3.22',
	'TAP::Parser::Iterator::Process'=> '3.22',
	'TAP::Parser::Iterator::Stream'=> '3.22',
	'TAP::Parser::IteratorFactory'=> '3.22',
	'TAP::Parser::Multiplexer'=> '3.22',
	'TAP::Parser::Result'   => '3.22',
	'TAP::Parser::Result::Bailout'=> '3.22',
	'TAP::Parser::Result::Comment'=> '3.22',
	'TAP::Parser::Result::Plan'=> '3.22',
	'TAP::Parser::Result::Pragma'=> '3.22',
	'TAP::Parser::Result::Test'=> '3.22',
	'TAP::Parser::Result::Unknown'=> '3.22',
	'TAP::Parser::Result::Version'=> '3.22',
	'TAP::Parser::Result::YAML'=> '3.22',
	'TAP::Parser::ResultFactory'=> '3.22',
	'TAP::Parser::Scheduler'=> '3.22',
	'TAP::Parser::Scheduler::Job'=> '3.22',
	'TAP::Parser::Scheduler::Spinner'=> '3.22',
	'TAP::Parser::Source'   => '3.22',
	'TAP::Parser::SourceHandler'=> '3.22',
	'TAP::Parser::SourceHandler::Executable'=> '3.22',
	'TAP::Parser::SourceHandler::File'=> '3.22',
	'TAP::Parser::SourceHandler::Handle'=> '3.22',
	'TAP::Parser::SourceHandler::Perl'=> '3.22',
	'TAP::Parser::SourceHandler::RawTAP'=> '3.22',
	'TAP::Parser::Utils'    => '3.22',
	'TAP::Parser::YAMLish::Reader'=> '3.22',
	'TAP::Parser::YAMLish::Writer'=> '3.22',
	'Term::ANSIColor'       => '3.00',
	'Term::Cap'             => '1.12',
	'Term::Complete'        => '1.402',
	'Term::ReadLine'        => '1.07',
	'Term::UI'              => '0.20',
	'Term::UI::History'     => undef,
	'Test'                  => '1.25_02',
	'Test::Builder'         => '0.96',
	'Test::Builder::Module' => '0.96',
	'Test::Builder::Tester' => '1.20',
	'Test::Builder::Tester::Color'=> '1.20',
	'Test::Harness'         => '3.22',
	'Test::More'            => '0.96',
	'Test::Simple'          => '0.96',
	'Text::Abbrev'          => '1.01',
	'Text::Balanced'        => '2.02',
	'Text::ParseWords'      => '3.27',
	'Text::Soundex'         => '3.03_01',
	'Text::Tabs'            => '2009.0305',
	'Text::Wrap'            => '2009.0305',
	'Thread'                => '3.02',
	'Thread::Queue'         => '2.11',
	'Thread::Semaphore'     => '2.11',
	'Tie::Array'            => '1.04',
	'Tie::File'             => '0.97_02',
	'Tie::Handle'           => '4.2',
	'Tie::Hash'             => '1.03',
	'Tie::Hash::NamedCapture'=> '0.06',
	'Tie::Memoize'          => '1.1',
	'Tie::RefHash'          => '1.38',
	'Tie::Scalar'           => '1.02',
	'Tie::StdHandle'        => '4.2',
	'Tie::SubstrHash'       => '1.00',
	'Time::HiRes'           => '1.9721',
	'Time::Local'           => '1.1901_01',
	'Time::Piece'           => '1.20_01',
	'Time::Piece::Seconds'  => undef,
	'Time::Seconds'         => undef,
	'Time::gmtime'          => '1.03',
	'Time::localtime'       => '1.02',
	'Time::tm'              => '1.00',
	'UNIVERSAL'             => '1.07',
	'Unicode'               => '5.2.0',
	'Unicode::Collate'      => '0.56',
	'Unicode::Collate::Locale'=> '0.56',
	'Unicode::Normalize'    => '1.06',
	'Unicode::UCD'          => '0.29',
	'User::grent'           => '1.01',
	'User::pwent'           => '1.00',
	'VMS::DCLsym'           => '1.04',
	'VMS::Filespec'         => '1.12',
	'VMS::Stdio'            => '2.4',
	'Win32'                 => '0.39',
	'Win32API::File'        => '0.1101',
	'Win32API::File::ExtUtils::Myconst2perl'=> '1',
	'Win32CORE'             => '0.02',
	'XS::APItest'           => '0.20',
	'XS::APItest::KeywordRPN'=> '0.004',
	'XS::Typemap'           => '0.03',
	'XSLoader'              => '0.10',
	'XSLoader::XSLoader'    => '0.10',
	'attributes'            => '0.12',
	'autodie'               => '2.10',
	'autodie::exception'    => '2.10',
	'autodie::exception::system'=> '2.10',
	'autodie::hints'        => '2.10',
	'autouse'               => '1.06',
	'base'                  => '2.15',
	'bigint'                => '0.23',
	'bignum'                => '0.23',
	'bigrat'                => '0.23',
	'blib'                  => '1.05',
	'bytes'                 => '1.04',
	'charnames'             => '1.15',
	'constant'              => '1.20',
	'deprecate'             => '0.01',
	'diagnostics'           => '1.20',
	'encoding'              => '2.6_01',
	'encoding::warnings'    => '0.11',
	'feature'               => '1.18',
	'fields'                => '2.15',
	'filetest'              => '1.02',
	'if'                    => '0.05',
	'inc::latest'           => '0.3607',
	'integer'               => '1.00',
	'less'                  => '0.03',
	'lib'                   => '0.63',
	'locale'                => '1.00',
	'mro'                   => '1.03',
	'open'                  => '1.07',
	'ops'                   => '1.02',
	'overload'              => '1.10',
	'overload::numbers'     => undef,
	'overloading'           => '0.01',
	'parent'                => '0.223',
	're'                    => '0.12',
	'sigtrap'               => '1.04',
	'sort'                  => '2.01',
	'strict'                => '1.04',
	'subs'                  => '1.00',
	'threads'               => '1.77_03',
	'threads::shared'       => '1.33_02',
	'utf8'                  => '1.08',
	'vars'                  => '1.02',
	'version'               => '0.82',
	'vmsish'                => '1.02',
	'warnings'              => '1.10',
	'warnings::register'    => '1.01',
    },
    5.012002 => {
	'AnyDBM_File'           => '1.00',
	'App::Cpan'             => '1.5701',
	'App::Prove'            => '3.17',
	'App::Prove::State'     => '3.17',
	'App::Prove::State::Result'=> '3.17',
	'App::Prove::State::Result::Test'=> '3.17',
	'Archive::Extract'      => '0.38',
	'Archive::Tar'          => '1.54',
	'Archive::Tar::Constant'=> '0.02',
	'Archive::Tar::File'    => '0.02',
	'Attribute::Handlers'   => '0.87',
	'AutoLoader'            => '5.70',
	'AutoSplit'             => '1.06',
	'B'                     => '1.23',
	'B::Concise'            => '0.78',
	'B::Debug'              => '1.12',
	'B::Deparse'            => '0.97',
	'B::Lint'               => '1.11_01',
	'B::Lint::Debug'        => '0.01',
	'B::Showlex'            => '1.02',
	'B::Terse'              => '1.05',
	'B::Xref'               => '1.02',
	'Benchmark'             => '1.11',
	'CGI'                   => '3.49',
	'CGI::Apache'           => '1.01',
	'CGI::Carp'             => '3.45',
	'CGI::Cookie'           => '1.29',
	'CGI::Fast'             => '1.08',
	'CGI::Pretty'           => '3.46',
	'CGI::Push'             => '1.04',
	'CGI::Switch'           => '1.01',
	'CGI::Util'             => '3.48',
	'CPAN'                  => '1.94_56',
	'CPAN::Author'          => '5.5',
	'CPAN::Bundle'          => '5.5',
	'CPAN::CacheMgr'        => '5.5',
	'CPAN::Complete'        => '5.5',
	'CPAN::Debug'           => '5.5001',
	'CPAN::DeferredCode'    => '5.50',
	'CPAN::Distribution'    => '1.9456_01',
	'CPAN::Distroprefs'     => '6',
	'CPAN::Distrostatus'    => '5.5',
	'CPAN::Exception::RecursiveDependency'=> '5.5',
	'CPAN::Exception::blocked_urllist'=> '1.0',
	'CPAN::Exception::yaml_not_installed'=> '5.5',
	'CPAN::FTP'             => '5.5004',
	'CPAN::FTP::netrc'      => '1.00',
	'CPAN::FirstTime'       => '5.5301',
	'CPAN::HandleConfig'    => '5.5001',
	'CPAN::Index'           => '1.94',
	'CPAN::InfoObj'         => '5.5',
	'CPAN::Kwalify'         => '5.50',
	'CPAN::LWP::UserAgent'  => '1.94',
	'CPAN::Mirrors'         => '1.77',
	'CPAN::Module'          => '5.5',
	'CPAN::Nox'             => '5.50',
	'CPAN::Prompt'          => '5.5',
	'CPAN::Queue'           => '5.5',
	'CPAN::Shell'           => '5.5001',
	'CPAN::Tarzip'          => '5.5011',
	'CPAN::URL'             => '5.5',
	'CPAN::Version'         => '5.5',
	'CPANPLUS'              => '0.90',
	'CPANPLUS::Backend'     => undef,
	'CPANPLUS::Backend::RV' => undef,
	'CPANPLUS::Config'      => undef,
	'CPANPLUS::Configure'   => undef,
	'CPANPLUS::Configure::Setup'=> undef,
	'CPANPLUS::Dist'        => undef,
	'CPANPLUS::Dist::Autobundle'=> undef,
	'CPANPLUS::Dist::Base'  => undef,
	'CPANPLUS::Dist::Build' => '0.46',
	'CPANPLUS::Dist::Build::Constants'=> '0.46',
	'CPANPLUS::Dist::MM'    => undef,
	'CPANPLUS::Dist::Sample'=> undef,
	'CPANPLUS::Error'       => undef,
	'CPANPLUS::Internals'   => '0.90',
	'CPANPLUS::Internals::Constants'=> undef,
	'CPANPLUS::Internals::Constants::Report'=> undef,
	'CPANPLUS::Internals::Extract'=> undef,
	'CPANPLUS::Internals::Fetch'=> undef,
	'CPANPLUS::Internals::Report'=> undef,
	'CPANPLUS::Internals::Search'=> undef,
	'CPANPLUS::Internals::Source'=> undef,
	'CPANPLUS::Internals::Source::Memory'=> undef,
	'CPANPLUS::Internals::Source::SQLite'=> undef,
	'CPANPLUS::Internals::Source::SQLite::Tie'=> undef,
	'CPANPLUS::Internals::Utils'=> undef,
	'CPANPLUS::Internals::Utils::Autoflush'=> undef,
	'CPANPLUS::Module'      => undef,
	'CPANPLUS::Module::Author'=> undef,
	'CPANPLUS::Module::Author::Fake'=> undef,
	'CPANPLUS::Module::Checksums'=> undef,
	'CPANPLUS::Module::Fake'=> undef,
	'CPANPLUS::Module::Signature'=> undef,
	'CPANPLUS::Selfupdate'  => undef,
	'CPANPLUS::Shell'       => undef,
	'CPANPLUS::Shell::Classic'=> '0.0562',
	'CPANPLUS::Shell::Default'=> '0.90',
	'CPANPLUS::Shell::Default::Plugins::CustomSource'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Remote'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Source'=> undef,
	'Carp'                  => '1.17',
	'Carp::Heavy'           => '1.17',
	'Class::ISA'            => '0.36',
	'Class::Struct'         => '0.63',
	'Compress::Raw::Bzip2'  => '2.024',
	'Compress::Raw::Zlib'   => '2.024',
	'Compress::Zlib'        => '2.024',
	'Config'                => undef,
	'Config::Extensions'    => '0.01',
	'Cwd'                   => '3.31',
	'DB'                    => '1.02',
	'DBM_Filter'            => '0.03',
	'DBM_Filter::compress'  => '0.02',
	'DBM_Filter::encode'    => '0.02',
	'DBM_Filter::int32'     => '0.02',
	'DBM_Filter::null'      => '0.02',
	'DBM_Filter::utf8'      => '0.02',
	'DB_File'               => '1.820',
	'Data::Dumper'          => '2.125',
	'Devel::DProf'          => '20080331.00',
	'Devel::DProf::dprof::V'=> undef,
	'Devel::InnerPackage'   => '0.3',
	'Devel::PPPort'         => '3.19',
	'Devel::Peek'           => '1.04',
	'Devel::SelfStubber'    => '1.03',
	'Digest'                => '1.16',
	'Digest::MD5'           => '2.39',
	'Digest::SHA'           => '5.47',
	'Digest::base'          => '1.16',
	'Digest::file'          => '1.16',
	'DirHandle'             => '1.03',
	'Dumpvalue'             => '1.13',
	'DynaLoader'            => '1.10',
	'Encode'                => '2.39',
	'Encode::Alias'         => '2.12',
	'Encode::Byte'          => '2.04',
	'Encode::CJKConstants'  => '2.02',
	'Encode::CN'            => '2.03',
	'Encode::CN::HZ'        => '2.05',
	'Encode::Config'        => '2.05',
	'Encode::EBCDIC'        => '2.02',
	'Encode::Encoder'       => '2.01',
	'Encode::Encoding'      => '2.05',
	'Encode::GSM0338'       => '2.01',
	'Encode::Guess'         => '2.03',
	'Encode::JP'            => '2.04',
	'Encode::JP::H2Z'       => '2.02',
	'Encode::JP::JIS7'      => '2.04',
	'Encode::KR'            => '2.03',
	'Encode::KR::2022_KR'   => '2.02',
	'Encode::MIME::Header'  => '2.11',
	'Encode::MIME::Header::ISO_2022_JP'=> '1.03',
	'Encode::MIME::Name'    => '1.01',
	'Encode::Symbol'        => '2.02',
	'Encode::TW'            => '2.03',
	'Encode::Unicode'       => '2.07',
	'Encode::Unicode::UTF7' => '2.04',
	'English'               => '1.04',
	'Env'                   => '1.01',
	'Errno'                 => '1.11',
	'Exporter'              => '5.64_01',
	'Exporter::Heavy'       => '5.64_01',
	'ExtUtils::CBuilder'    => '0.27',
	'ExtUtils::CBuilder::Base'=> '0.27',
	'ExtUtils::CBuilder::Platform::Unix'=> '0.27',
	'ExtUtils::CBuilder::Platform::VMS'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.27',
	'ExtUtils::CBuilder::Platform::aix'=> '0.27',
	'ExtUtils::CBuilder::Platform::cygwin'=> '0.27',
	'ExtUtils::CBuilder::Platform::darwin'=> '0.27',
	'ExtUtils::CBuilder::Platform::dec_osf'=> '0.27',
	'ExtUtils::CBuilder::Platform::os2'=> '0.27',
	'ExtUtils::Command'     => '1.16',
	'ExtUtils::Command::MM' => '6.56',
	'ExtUtils::Constant'    => '0.22',
	'ExtUtils::Constant::Base'=> '0.04',
	'ExtUtils::Constant::ProxySubs'=> '0.06',
	'ExtUtils::Constant::Utils'=> '0.02',
	'ExtUtils::Constant::XS'=> '0.03',
	'ExtUtils::Embed'       => '1.28',
	'ExtUtils::Install'     => '1.55',
	'ExtUtils::Installed'   => '1.999_001',
	'ExtUtils::Liblist'     => '6.56',
	'ExtUtils::Liblist::Kid'=> '6.56',
	'ExtUtils::MM'          => '6.56',
	'ExtUtils::MM_AIX'      => '6.56',
	'ExtUtils::MM_Any'      => '6.56',
	'ExtUtils::MM_BeOS'     => '6.56',
	'ExtUtils::MM_Cygwin'   => '6.56',
	'ExtUtils::MM_DOS'      => '6.56',
	'ExtUtils::MM_Darwin'   => '6.56',
	'ExtUtils::MM_MacOS'    => '6.56',
	'ExtUtils::MM_NW5'      => '6.56',
	'ExtUtils::MM_OS2'      => '6.56',
	'ExtUtils::MM_QNX'      => '6.56',
	'ExtUtils::MM_UWIN'     => '6.56',
	'ExtUtils::MM_Unix'     => '6.56',
	'ExtUtils::MM_VMS'      => '6.56',
	'ExtUtils::MM_VOS'      => '6.56',
	'ExtUtils::MM_Win32'    => '6.56',
	'ExtUtils::MM_Win95'    => '6.56',
	'ExtUtils::MY'          => '6.56',
	'ExtUtils::MakeMaker'   => '6.56',
	'ExtUtils::MakeMaker::Config'=> '6.56',
	'ExtUtils::Manifest'    => '1.57',
	'ExtUtils::Miniperl'    => undef,
	'ExtUtils::Mkbootstrap' => '6.56',
	'ExtUtils::Mksymlists'  => '6.56',
	'ExtUtils::Packlist'    => '1.44',
	'ExtUtils::ParseXS'     => '2.21',
	'ExtUtils::XSSymSet'    => '1.1',
	'ExtUtils::testlib'     => '6.56',
	'Fatal'                 => '2.06_01',
	'Fcntl'                 => '1.06',
	'File::Basename'        => '2.78',
	'File::CheckTree'       => '4.4',
	'File::Compare'         => '1.1006',
	'File::Copy'            => '2.18',
	'File::DosGlob'         => '1.01',
	'File::Fetch'           => '0.24',
	'File::Find'            => '1.15',
	'File::Glob'            => '1.07',
	'File::GlobMapper'      => '1.000',
	'File::Path'            => '2.08_01',
	'File::Spec'            => '3.31_01',
	'File::Spec::Cygwin'    => '3.30',
	'File::Spec::Epoc'      => '3.30',
	'File::Spec::Functions' => '3.30',
	'File::Spec::Mac'       => '3.30',
	'File::Spec::OS2'       => '3.30',
	'File::Spec::Unix'      => '3.30',
	'File::Spec::VMS'       => '3.30',
	'File::Spec::Win32'     => '3.30',
	'File::Temp'            => '0.22',
	'File::stat'            => '1.02',
	'FileCache'             => '1.08',
	'FileHandle'            => '2.02',
	'Filter::Simple'        => '0.84',
	'Filter::Util::Call'    => '1.08',
	'FindBin'               => '1.50',
	'GDBM_File'             => '1.10',
	'Getopt::Long'          => '2.38',
	'Getopt::Std'           => '1.06',
	'Hash::Util'            => '0.07',
	'Hash::Util::FieldHash' => '1.04',
	'I18N::Collate'         => '1.01',
	'I18N::LangTags'        => '0.35',
	'I18N::LangTags::Detect'=> '1.04',
	'I18N::LangTags::List'  => '0.35',
	'I18N::Langinfo'        => '0.03',
	'IO'                    => '1.25_02',
	'IO::Compress::Adapter::Bzip2'=> '2.024',
	'IO::Compress::Adapter::Deflate'=> '2.024',
	'IO::Compress::Adapter::Identity'=> '2.024',
	'IO::Compress::Base'    => '2.024',
	'IO::Compress::Base::Common'=> '2.024',
	'IO::Compress::Bzip2'   => '2.024',
	'IO::Compress::Deflate' => '2.024',
	'IO::Compress::Gzip'    => '2.024',
	'IO::Compress::Gzip::Constants'=> '2.024',
	'IO::Compress::RawDeflate'=> '2.024',
	'IO::Compress::Zip'     => '2.024',
	'IO::Compress::Zip::Constants'=> '2.024',
	'IO::Compress::Zlib::Constants'=> '2.024',
	'IO::Compress::Zlib::Extra'=> '2.024',
	'IO::Dir'               => '1.07',
	'IO::File'              => '1.14',
	'IO::Handle'            => '1.28',
	'IO::Pipe'              => '1.13',
	'IO::Poll'              => '0.07',
	'IO::Seekable'          => '1.10',
	'IO::Select'            => '1.17',
	'IO::Socket'            => '1.31',
	'IO::Socket::INET'      => '1.31',
	'IO::Socket::UNIX'      => '1.23',
	'IO::Uncompress::Adapter::Bunzip2'=> '2.024',
	'IO::Uncompress::Adapter::Identity'=> '2.024',
	'IO::Uncompress::Adapter::Inflate'=> '2.024',
	'IO::Uncompress::AnyInflate'=> '2.024',
	'IO::Uncompress::AnyUncompress'=> '2.024',
	'IO::Uncompress::Base'  => '2.024',
	'IO::Uncompress::Bunzip2'=> '2.024',
	'IO::Uncompress::Gunzip'=> '2.024',
	'IO::Uncompress::Inflate'=> '2.024',
	'IO::Uncompress::RawInflate'=> '2.024',
	'IO::Uncompress::Unzip' => '2.024',
	'IO::Zlib'              => '1.10',
	'IPC::Cmd'              => '0.54',
	'IPC::Msg'              => '2.01',
	'IPC::Open2'            => '1.03',
	'IPC::Open3'            => '1.05',
	'IPC::Semaphore'        => '2.01',
	'IPC::SharedMem'        => '2.01',
	'IPC::SysV'             => '2.01',
	'List::Util'            => '1.22',
	'List::Util::PP'        => '1.22',
	'List::Util::XS'        => '1.22',
	'Locale::Constants'     => '2.07',
	'Locale::Country'       => '2.07',
	'Locale::Currency'      => '2.07',
	'Locale::Language'      => '2.07',
	'Locale::Maketext'      => '1.14',
	'Locale::Maketext::Guts'=> '1.13',
	'Locale::Maketext::GutsLoader'=> '1.13',
	'Locale::Maketext::Simple'=> '0.21',
	'Locale::Script'        => '2.07',
	'Log::Message'          => '0.02',
	'Log::Message::Config'  => '0.01',
	'Log::Message::Handlers'=> undef,
	'Log::Message::Item'    => undef,
	'Log::Message::Simple'  => '0.06',
	'MIME::Base64'          => '3.08',
	'MIME::QuotedPrint'     => '3.08',
	'Math::BigFloat'        => '1.60',
	'Math::BigFloat::Trace' => '0.01',
	'Math::BigInt'          => '1.89_01',
	'Math::BigInt::Calc'    => '0.52',
	'Math::BigInt::CalcEmu' => '0.05',
	'Math::BigInt::FastCalc'=> '0.19',
	'Math::BigInt::Trace'   => '0.01',
	'Math::BigRat'          => '0.24',
	'Math::Complex'         => '1.56',
	'Math::Trig'            => '1.2',
	'Memoize'               => '1.01_03',
	'Memoize::AnyDBM_File'  => '0.65',
	'Memoize::Expire'       => '1.00',
	'Memoize::ExpireFile'   => '1.01',
	'Memoize::ExpireTest'   => '0.65',
	'Memoize::NDBM_File'    => '0.65',
	'Memoize::SDBM_File'    => '0.65',
	'Memoize::Storable'     => '0.65',
	'Module::Build'         => '0.3603',
	'Module::Build::Base'   => '0.3603',
	'Module::Build::Compat' => '0.3603',
	'Module::Build::Config' => '0.3603',
	'Module::Build::ConfigData'=> undef,
	'Module::Build::Cookbook'=> '0.3603',
	'Module::Build::Dumper' => '0.3603',
	'Module::Build::ModuleInfo'=> '0.3603',
	'Module::Build::Notes'  => '0.3603',
	'Module::Build::PPMMaker'=> '0.3603',
	'Module::Build::Platform::Amiga'=> '0.3603',
	'Module::Build::Platform::Default'=> '0.3603',
	'Module::Build::Platform::EBCDIC'=> '0.3603',
	'Module::Build::Platform::MPEiX'=> '0.3603',
	'Module::Build::Platform::MacOS'=> '0.3603',
	'Module::Build::Platform::RiscOS'=> '0.3603',
	'Module::Build::Platform::Unix'=> '0.3603',
	'Module::Build::Platform::VMS'=> '0.3603',
	'Module::Build::Platform::VOS'=> '0.3603',
	'Module::Build::Platform::Windows'=> '0.3603',
	'Module::Build::Platform::aix'=> '0.3603',
	'Module::Build::Platform::cygwin'=> '0.3603',
	'Module::Build::Platform::darwin'=> '0.3603',
	'Module::Build::Platform::os2'=> '0.3603',
	'Module::Build::PodParser'=> '0.3603',
	'Module::Build::Version'=> '0.77',
	'Module::Build::YAML'   => '1.40',
	'Module::CoreList'      => '2.38',
	'Module::Load'          => '0.16',
	'Module::Load::Conditional'=> '0.38',
	'Module::Loaded'        => '0.06',
	'Module::Pluggable'     => '3.9',
	'Module::Pluggable::Object'=> '3.9',
	'Moped::Msg'            => '0.01',
	'NDBM_File'             => '1.08',
	'NEXT'                  => '0.64',
	'Net::Cmd'              => '2.29',
	'Net::Config'           => '1.11',
	'Net::Domain'           => '2.20',
	'Net::FTP'              => '2.77',
	'Net::FTP::A'           => '1.18',
	'Net::FTP::E'           => '0.01',
	'Net::FTP::I'           => '1.12',
	'Net::FTP::L'           => '0.01',
	'Net::FTP::dataconn'    => '0.11',
	'Net::NNTP'             => '2.24',
	'Net::Netrc'            => '2.12',
	'Net::POP3'             => '2.29',
	'Net::Ping'             => '2.36',
	'Net::SMTP'             => '2.31',
	'Net::Time'             => '2.10',
	'Net::hostent'          => '1.01',
	'Net::netent'           => '1.00',
	'Net::protoent'         => '1.00',
	'Net::servent'          => '1.01',
	'O'                     => '1.01',
	'ODBM_File'             => '1.07',
	'Object::Accessor'      => '0.36',
	'Opcode'                => '1.15',
	'POSIX'                 => '1.19',
	'Package::Constants'    => '0.02',
	'Params::Check'         => '0.26',
	'Parse::CPAN::Meta'     => '1.40',
	'PerlIO'                => '1.06',
	'PerlIO::encoding'      => '0.12',
	'PerlIO::scalar'        => '0.08',
	'PerlIO::via'           => '0.09',
	'PerlIO::via::QuotedPrint'=> '0.06',
	'Pod::Checker'          => '1.45',
	'Pod::Escapes'          => '1.04',
	'Pod::Find'             => '1.35',
	'Pod::Functions'        => '1.04',
	'Pod::Html'             => '1.09',
	'Pod::InputObjects'     => '1.31',
	'Pod::LaTeX'            => '0.58',
	'Pod::Man'              => '2.23',
	'Pod::ParseLink'        => '1.10',
	'Pod::ParseUtils'       => '1.36',
	'Pod::Parser'           => '1.37',
	'Pod::Perldoc'          => '3.15_02',
	'Pod::Perldoc::BaseTo'  => undef,
	'Pod::Perldoc::GetOptsOO'=> undef,
	'Pod::Perldoc::ToChecker'=> undef,
	'Pod::Perldoc::ToMan'   => undef,
	'Pod::Perldoc::ToNroff' => undef,
	'Pod::Perldoc::ToPod'   => undef,
	'Pod::Perldoc::ToRtf'   => undef,
	'Pod::Perldoc::ToText'  => undef,
	'Pod::Perldoc::ToTk'    => undef,
	'Pod::Perldoc::ToXml'   => undef,
	'Pod::PlainText'        => '2.04',
	'Pod::Plainer'          => '1.02',
	'Pod::Select'           => '1.36',
	'Pod::Simple'           => '3.14',
	'Pod::Simple::BlackBox' => '3.14',
	'Pod::Simple::Checker'  => '3.14',
	'Pod::Simple::Debug'    => '3.14',
	'Pod::Simple::DumpAsText'=> '3.14',
	'Pod::Simple::DumpAsXML'=> '3.14',
	'Pod::Simple::HTML'     => '3.14',
	'Pod::Simple::HTMLBatch'=> '3.14',
	'Pod::Simple::HTMLLegacy'=> '5.01',
	'Pod::Simple::LinkSection'=> '3.14',
	'Pod::Simple::Methody'  => '3.14',
	'Pod::Simple::Progress' => '3.14',
	'Pod::Simple::PullParser'=> '3.14',
	'Pod::Simple::PullParserEndToken'=> '3.14',
	'Pod::Simple::PullParserStartToken'=> '3.14',
	'Pod::Simple::PullParserTextToken'=> '3.14',
	'Pod::Simple::PullParserToken'=> '3.14',
	'Pod::Simple::RTF'      => '3.14',
	'Pod::Simple::Search'   => '3.14',
	'Pod::Simple::SimpleTree'=> '3.14',
	'Pod::Simple::Text'     => '3.14',
	'Pod::Simple::TextContent'=> '3.14',
	'Pod::Simple::TiedOutFH'=> '3.14',
	'Pod::Simple::Transcode'=> '3.14',
	'Pod::Simple::TranscodeDumb'=> '3.14',
	'Pod::Simple::TranscodeSmart'=> '3.14',
	'Pod::Simple::XHTML'    => '3.14',
	'Pod::Simple::XMLOutStream'=> '3.14',
	'Pod::Text'             => '3.14',
	'Pod::Text::Color'      => '2.06',
	'Pod::Text::Overstrike' => '2.04',
	'Pod::Text::Termcap'    => '2.06',
	'Pod::Usage'            => '1.36',
	'SDBM_File'             => '1.06',
	'Safe'                  => '2.27',
	'Scalar::Util'          => '1.22',
	'Scalar::Util::PP'      => '1.22',
	'Search::Dict'          => '1.02',
	'SelectSaver'           => '1.02',
	'SelfLoader'            => '1.17',
	'Shell'                 => '0.72_01',
	'Socket'                => '1.87',
	'Storable'              => '2.22',
	'Switch'                => '2.16',
	'Symbol'                => '1.07',
	'Sys::Hostname'         => '1.11',
	'Sys::Syslog'           => '0.27',
	'Sys::Syslog::win32::Win32'=> undef,
	'TAP::Base'             => '3.17',
	'TAP::Formatter::Base'  => '3.17',
	'TAP::Formatter::Color' => '3.17',
	'TAP::Formatter::Console'=> '3.17',
	'TAP::Formatter::Console::ParallelSession'=> '3.17',
	'TAP::Formatter::Console::Session'=> '3.17',
	'TAP::Formatter::File'  => '3.17',
	'TAP::Formatter::File::Session'=> '3.17',
	'TAP::Formatter::Session'=> '3.17',
	'TAP::Harness'          => '3.17',
	'TAP::Object'           => '3.17',
	'TAP::Parser'           => '3.17',
	'TAP::Parser::Aggregator'=> '3.17',
	'TAP::Parser::Grammar'  => '3.17',
	'TAP::Parser::Iterator' => '3.17',
	'TAP::Parser::Iterator::Array'=> '3.17',
	'TAP::Parser::Iterator::Process'=> '3.17',
	'TAP::Parser::Iterator::Stream'=> '3.17',
	'TAP::Parser::IteratorFactory'=> '3.17',
	'TAP::Parser::Multiplexer'=> '3.17',
	'TAP::Parser::Result'   => '3.17',
	'TAP::Parser::Result::Bailout'=> '3.17',
	'TAP::Parser::Result::Comment'=> '3.17',
	'TAP::Parser::Result::Plan'=> '3.17',
	'TAP::Parser::Result::Pragma'=> '3.17',
	'TAP::Parser::Result::Test'=> '3.17',
	'TAP::Parser::Result::Unknown'=> '3.17',
	'TAP::Parser::Result::Version'=> '3.17',
	'TAP::Parser::Result::YAML'=> '3.17',
	'TAP::Parser::ResultFactory'=> '3.17',
	'TAP::Parser::Scheduler'=> '3.17',
	'TAP::Parser::Scheduler::Job'=> '3.17',
	'TAP::Parser::Scheduler::Spinner'=> '3.17',
	'TAP::Parser::Source'   => '3.17',
	'TAP::Parser::Source::Perl'=> '3.17',
	'TAP::Parser::Utils'    => '3.17',
	'TAP::Parser::YAMLish::Reader'=> '3.17',
	'TAP::Parser::YAMLish::Writer'=> '3.17',
	'Term::ANSIColor'       => '2.02',
	'Term::Cap'             => '1.12',
	'Term::Complete'        => '1.402',
	'Term::ReadLine'        => '1.05',
	'Term::UI'              => '0.20',
	'Term::UI::History'     => undef,
	'Test'                  => '1.25_02',
	'Test::Builder'         => '0.94',
	'Test::Builder::Module' => '0.94',
	'Test::Builder::Tester' => '1.18',
	'Test::Builder::Tester::Color'=> '1.18',
	'Test::Harness'         => '3.17',
	'Test::More'            => '0.94',
	'Test::Simple'          => '0.94',
	'Text::Abbrev'          => '1.01',
	'Text::Balanced'        => '2.02',
	'Text::ParseWords'      => '3.27',
	'Text::Soundex'         => '3.03_01',
	'Text::Tabs'            => '2009.0305',
	'Text::Wrap'            => '2009.0305',
	'Thread'                => '3.02',
	'Thread::Queue'         => '2.11',
	'Thread::Semaphore'     => '2.09',
	'Tie::Array'            => '1.03',
	'Tie::File'             => '0.97_02',
	'Tie::Handle'           => '4.2',
	'Tie::Hash'             => '1.03',
	'Tie::Hash::NamedCapture'=> '0.06',
	'Tie::Memoize'          => '1.1',
	'Tie::RefHash'          => '1.38',
	'Tie::Scalar'           => '1.02',
	'Tie::StdHandle'        => '4.2',
	'Tie::SubstrHash'       => '1.00',
	'Time::HiRes'           => '1.9719',
	'Time::Local'           => '1.1901_01',
	'Time::Piece'           => '1.15_01',
	'Time::Piece::Seconds'  => undef,
	'Time::Seconds'         => undef,
	'Time::gmtime'          => '1.03',
	'Time::localtime'       => '1.02',
	'Time::tm'              => '1.00',
	'UNIVERSAL'             => '1.06',
	'Unicode'               => '5.2.0',
	'Unicode::Collate'      => '0.52_01',
	'Unicode::Normalize'    => '1.03',
	'Unicode::UCD'          => '0.27',
	'User::grent'           => '1.01',
	'User::pwent'           => '1.00',
	'VMS::DCLsym'           => '1.03',
	'VMS::Filespec'         => '1.12',
	'VMS::Stdio'            => '2.4',
	'Win32'                 => '0.39',
	'Win32API::File'        => '0.1101',
	'Win32API::File::ExtUtils::Myconst2perl'=> '1',
	'Win32CORE'             => '0.02',
	'XS::APItest'           => '0.17',
	'XS::APItest::KeywordRPN'=> '0.003',
	'XS::Typemap'           => '0.03',
	'XSLoader'              => '0.10',
	'XSLoader::XSLoader'    => '0.10',
	'attributes'            => '0.12',
	'autodie'               => '2.06_01',
	'autodie::exception'    => '2.06_01',
	'autodie::exception::system'=> '2.06_01',
	'autodie::hints'        => '2.06_01',
	'autouse'               => '1.06',
	'base'                  => '2.15',
	'bigint'                => '0.23',
	'bignum'                => '0.23',
	'bigrat'                => '0.23',
	'blib'                  => '1.04',
	'bytes'                 => '1.04',
	'charnames'             => '1.07',
	'constant'              => '1.20',
	'deprecate'             => '0.01',
	'diagnostics'           => '1.19',
	'encoding'              => '2.6_01',
	'encoding::warnings'    => '0.11',
	'feature'               => '1.16',
	'fields'                => '2.15',
	'filetest'              => '1.02',
	'if'                    => '0.05',
	'inc::latest'           => '0.3603',
	'integer'               => '1.00',
	'less'                  => '0.03',
	'lib'                   => '0.62',
	'locale'                => '1.00',
	'mro'                   => '1.02',
	'open'                  => '1.07',
	'ops'                   => '1.02',
	'overload'              => '1.10',
	'overload::numbers'     => undef,
	'overloading'           => '0.01',
	'parent'                => '0.223',
	're'                    => '0.11',
	'sigtrap'               => '1.04',
	'sort'                  => '2.01',
	'strict'                => '1.04',
	'subs'                  => '1.00',
	'threads'               => '1.75',
	'threads::shared'       => '1.32',
	'utf8'                  => '1.08',
	'vars'                  => '1.01',
	'version'               => '0.82',
	'vmsish'                => '1.02',
	'warnings'              => '1.09',
	'warnings::register'    => '1.01',
    },
    5.013005 => {
	'AnyDBM_File'           => '1.00',
	'App::Cpan'             => '1.5701',
	'App::Prove'            => '3.22',
	'App::Prove::State'     => '3.22',
	'App::Prove::State::Result'=> '3.22',
	'App::Prove::State::Result::Test'=> '3.22',
	'Archive::Extract'      => '0.42',
	'Archive::Tar'          => '1.68',
	'Archive::Tar::Constant'=> '1.68',
	'Archive::Tar::File'    => '1.68',
	'Attribute::Handlers'   => '0.88',
	'AutoLoader'            => '5.70',
	'AutoSplit'             => '1.06',
	'B'                     => '1.23',
	'B::Concise'            => '0.79',
	'B::Debug'              => '1.16',
	'B::Deparse'            => '0.98',
	'B::Lint'               => '1.12',
	'B::Lint::Debug'        => '1.12',
	'B::Showlex'            => '1.02',
	'B::Terse'              => '1.05',
	'B::Xref'               => '1.02',
	'Benchmark'             => '1.11',
	'CGI'                   => '3.49',
	'CGI::Apache'           => '1.01',
	'CGI::Carp'             => '3.45',
	'CGI::Cookie'           => '1.29',
	'CGI::Fast'             => '1.08',
	'CGI::Pretty'           => '3.46',
	'CGI::Push'             => '1.04',
	'CGI::Switch'           => '1.01',
	'CGI::Util'             => '3.48',
	'CPAN'                  => '1.94_57',
	'CPAN::Author'          => '5.5',
	'CPAN::Bundle'          => '5.5',
	'CPAN::CacheMgr'        => '5.5',
	'CPAN::Complete'        => '5.5',
	'CPAN::Debug'           => '5.5001',
	'CPAN::DeferredCode'    => '5.50',
	'CPAN::Distribution'    => '1.9600',
	'CPAN::Distroprefs'     => '6',
	'CPAN::Distrostatus'    => '5.5',
	'CPAN::Exception::RecursiveDependency'=> '5.5',
	'CPAN::Exception::blocked_urllist'=> '1.0',
	'CPAN::Exception::yaml_not_installed'=> '5.5',
	'CPAN::FTP'             => '5.5004',
	'CPAN::FTP::netrc'      => '1.00',
	'CPAN::FirstTime'       => '5.5301',
	'CPAN::HandleConfig'    => '5.5001',
	'CPAN::Index'           => '1.94',
	'CPAN::InfoObj'         => '5.5',
	'CPAN::Kwalify'         => '5.50',
	'CPAN::LWP::UserAgent'  => '1.94',
	'CPAN::Mirrors'         => '1.77',
	'CPAN::Module'          => '5.5',
	'CPAN::Nox'             => '5.50',
	'CPAN::Prompt'          => '5.5',
	'CPAN::Queue'           => '5.5',
	'CPAN::Shell'           => '5.5001',
	'CPAN::Tarzip'          => '5.5011',
	'CPAN::URL'             => '5.5',
	'CPAN::Version'         => '5.5',
	'CPANPLUS'              => '0.9007',
	'CPANPLUS::Backend'     => undef,
	'CPANPLUS::Backend::RV' => undef,
	'CPANPLUS::Config'      => undef,
	'CPANPLUS::Configure'   => undef,
	'CPANPLUS::Configure::Setup'=> undef,
	'CPANPLUS::Dist'        => undef,
	'CPANPLUS::Dist::Autobundle'=> undef,
	'CPANPLUS::Dist::Base'  => undef,
	'CPANPLUS::Dist::Build' => '0.48',
	'CPANPLUS::Dist::Build::Constants'=> '0.48',
	'CPANPLUS::Dist::MM'    => undef,
	'CPANPLUS::Dist::Sample'=> undef,
	'CPANPLUS::Error'       => undef,
	'CPANPLUS::Internals'   => '0.9007',
	'CPANPLUS::Internals::Constants'=> undef,
	'CPANPLUS::Internals::Constants::Report'=> undef,
	'CPANPLUS::Internals::Extract'=> undef,
	'CPANPLUS::Internals::Fetch'=> undef,
	'CPANPLUS::Internals::Report'=> undef,
	'CPANPLUS::Internals::Search'=> undef,
	'CPANPLUS::Internals::Source'=> undef,
	'CPANPLUS::Internals::Source::Memory'=> undef,
	'CPANPLUS::Internals::Source::SQLite'=> undef,
	'CPANPLUS::Internals::Source::SQLite::Tie'=> undef,
	'CPANPLUS::Internals::Utils'=> undef,
	'CPANPLUS::Internals::Utils::Autoflush'=> undef,
	'CPANPLUS::Module'      => undef,
	'CPANPLUS::Module::Author'=> undef,
	'CPANPLUS::Module::Author::Fake'=> undef,
	'CPANPLUS::Module::Checksums'=> undef,
	'CPANPLUS::Module::Fake'=> undef,
	'CPANPLUS::Module::Signature'=> undef,
	'CPANPLUS::Selfupdate'  => undef,
	'CPANPLUS::Shell'       => undef,
	'CPANPLUS::Shell::Classic'=> '0.0562',
	'CPANPLUS::Shell::Default'=> '0.9007',
	'CPANPLUS::Shell::Default::Plugins::CustomSource'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Remote'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Source'=> undef,
	'Carp'                  => '1.18',
	'Carp::Heavy'           => '1.18',
	'Class::Struct'         => '0.63',
	'Compress::Raw::Bzip2'  => '2.030',
	'Compress::Raw::Zlib'   => '2.030',
	'Compress::Zlib'        => '2.030',
	'Config'                => undef,
	'Config::Extensions'    => '0.01',
	'Cwd'                   => '3.31',
	'DB'                    => '1.03',
	'DBM_Filter'            => '0.03',
	'DBM_Filter::compress'  => '0.02',
	'DBM_Filter::encode'    => '0.02',
	'DBM_Filter::int32'     => '0.02',
	'DBM_Filter::null'      => '0.02',
	'DBM_Filter::utf8'      => '0.02',
	'DB_File'               => '1.820',
	'Data::Dumper'          => '2.128',
	'Devel::DProf'          => '20080331.00',
	'Devel::DProf::V'       => undef,
	'Devel::DProf::dprof::V'=> undef,
	'Devel::InnerPackage'   => '0.3',
	'Devel::PPPort'         => '3.19',
	'Devel::Peek'           => '1.04',
	'Devel::SelfStubber'    => '1.03',
	'Digest'                => '1.16',
	'Digest::MD5'           => '2.40',
	'Digest::SHA'           => '5.48',
	'Digest::base'          => '1.16',
	'Digest::file'          => '1.16',
	'DirHandle'             => '1.03',
	'Dumpvalue'             => '1.14',
	'DynaLoader'            => '1.10',
	'Encode'                => '2.40',
	'Encode::Alias'         => '2.12',
	'Encode::Byte'          => '2.04',
	'Encode::CJKConstants'  => '2.02',
	'Encode::CN'            => '2.03',
	'Encode::CN::HZ'        => '2.05',
	'Encode::Config'        => '2.05',
	'Encode::EBCDIC'        => '2.02',
	'Encode::Encoder'       => '2.01',
	'Encode::Encoding'      => '2.05',
	'Encode::GSM0338'       => '2.01',
	'Encode::Guess'         => '2.04',
	'Encode::JP'            => '2.04',
	'Encode::JP::H2Z'       => '2.02',
	'Encode::JP::JIS7'      => '2.04',
	'Encode::KR'            => '2.03',
	'Encode::KR::2022_KR'   => '2.02',
	'Encode::MIME::Header'  => '2.12',
	'Encode::MIME::Header::ISO_2022_JP'=> '1.03',
	'Encode::MIME::Name'    => '1.01',
	'Encode::Symbol'        => '2.02',
	'Encode::TW'            => '2.03',
	'Encode::Unicode'       => '2.07',
	'Encode::Unicode::UTF7' => '2.05',
	'English'               => '1.04',
	'Env'                   => '1.01',
	'Errno'                 => '1.13',
	'Exporter'              => '5.64_03',
	'Exporter::Heavy'       => '5.64_03',
	'ExtUtils::CBuilder'    => '0.2703',
	'ExtUtils::CBuilder::Base'=> '0.2703_01',
	'ExtUtils::CBuilder::Platform::Unix'=> '0.2703',
	'ExtUtils::CBuilder::Platform::VMS'=> '0.2703',
	'ExtUtils::CBuilder::Platform::Windows'=> '0.2703',
	'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.2703',
	'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.2703',
	'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.2703',
	'ExtUtils::CBuilder::Platform::aix'=> '0.2703',
	'ExtUtils::CBuilder::Platform::cygwin'=> '0.2703',
	'ExtUtils::CBuilder::Platform::darwin'=> '0.2703',
	'ExtUtils::CBuilder::Platform::dec_osf'=> '0.2703',
	'ExtUtils::CBuilder::Platform::os2'=> '0.2703',
	'ExtUtils::Command'     => '1.16',
	'ExtUtils::Command::MM' => '6.57_05',
	'ExtUtils::Constant'    => '0.22',
	'ExtUtils::Constant::Base'=> '0.04',
	'ExtUtils::Constant::ProxySubs'=> '0.06',
	'ExtUtils::Constant::Utils'=> '0.02',
	'ExtUtils::Constant::XS'=> '0.03',
	'ExtUtils::Embed'       => '1.28',
	'ExtUtils::Install'     => '1.55',
	'ExtUtils::Installed'   => '1.999_001',
	'ExtUtils::Liblist'     => '6.57_05',
	'ExtUtils::Liblist::Kid'=> '6.5705',
	'ExtUtils::MM'          => '6.57_05',
	'ExtUtils::MM_AIX'      => '6.57_05',
	'ExtUtils::MM_Any'      => '6.57_05',
	'ExtUtils::MM_BeOS'     => '6.57_05',
	'ExtUtils::MM_Cygwin'   => '6.57_05',
	'ExtUtils::MM_DOS'      => '6.5705',
	'ExtUtils::MM_Darwin'   => '6.57_05',
	'ExtUtils::MM_MacOS'    => '6.5705',
	'ExtUtils::MM_NW5'      => '6.57_05',
	'ExtUtils::MM_OS2'      => '6.57_05',
	'ExtUtils::MM_QNX'      => '6.57_05',
	'ExtUtils::MM_UWIN'     => '6.5705',
	'ExtUtils::MM_Unix'     => '6.57_05',
	'ExtUtils::MM_VMS'      => '6.57_05',
	'ExtUtils::MM_VOS'      => '6.57_05',
	'ExtUtils::MM_Win32'    => '6.57_05',
	'ExtUtils::MM_Win95'    => '6.57_05',
	'ExtUtils::MY'          => '6.5705',
	'ExtUtils::MakeMaker'   => '6.57_05',
	'ExtUtils::MakeMaker::Config'=> '6.57_05',
	'ExtUtils::MakeMaker::YAML'=> '1.44',
	'ExtUtils::Manifest'    => '1.58',
	'ExtUtils::Miniperl'    => undef,
	'ExtUtils::Mkbootstrap' => '6.57_05',
	'ExtUtils::Mksymlists'  => '6.57_05',
	'ExtUtils::Packlist'    => '1.44',
	'ExtUtils::ParseXS'     => '2.2207',
	'ExtUtils::XSSymSet'    => '1.1',
	'ExtUtils::testlib'     => '6.5705',
	'Fatal'                 => '2.10',
	'Fcntl'                 => '1.06',
	'File::Basename'        => '2.79',
	'File::CheckTree'       => '4.4',
	'File::Compare'         => '1.1006',
	'File::Copy'            => '2.20',
	'File::DosGlob'         => '1.02',
	'File::Fetch'           => '0.24',
	'File::Find'            => '1.17',
	'File::Glob'            => '1.08',
	'File::GlobMapper'      => '1.000',
	'File::Path'            => '2.08_01',
	'File::Spec'            => '3.31_01',
	'File::Spec::Cygwin'    => '3.30',
	'File::Spec::Epoc'      => '3.30',
	'File::Spec::Functions' => '3.30',
	'File::Spec::Mac'       => '3.30',
	'File::Spec::OS2'       => '3.30',
	'File::Spec::Unix'      => '3.30',
	'File::Spec::VMS'       => '3.30',
	'File::Spec::Win32'     => '3.30',
	'File::Temp'            => '0.22',
	'File::stat'            => '1.03',
	'FileCache'             => '1.08',
	'FileHandle'            => '2.02',
	'Filter::Simple'        => '0.85',
	'Filter::Util::Call'    => '1.08',
	'FindBin'               => '1.50',
	'GDBM_File'             => '1.10',
	'Getopt::Long'          => '2.38',
	'Getopt::Std'           => '1.06',
	'Hash::Util'            => '0.09',
	'Hash::Util::FieldHash' => '1.04',
	'I18N::Collate'         => '1.01',
	'I18N::LangTags'        => '0.35_01',
	'I18N::LangTags::Detect'=> '1.04',
	'I18N::LangTags::List'  => '0.35_01',
	'I18N::Langinfo'        => '0.04',
	'IO'                    => '1.25_02',
	'IO::Compress::Adapter::Bzip2'=> '2.030',
	'IO::Compress::Adapter::Deflate'=> '2.030',
	'IO::Compress::Adapter::Identity'=> '2.030',
	'IO::Compress::Base'    => '2.030',
	'IO::Compress::Base::Common'=> '2.030',
	'IO::Compress::Bzip2'   => '2.030',
	'IO::Compress::Deflate' => '2.030',
	'IO::Compress::Gzip'    => '2.030',
	'IO::Compress::Gzip::Constants'=> '2.030',
	'IO::Compress::RawDeflate'=> '2.030',
	'IO::Compress::Zip'     => '2.030',
	'IO::Compress::Zip::Constants'=> '2.030',
	'IO::Compress::Zlib::Constants'=> '2.030',
	'IO::Compress::Zlib::Extra'=> '2.030',
	'IO::Dir'               => '1.07',
	'IO::File'              => '1.14',
	'IO::Handle'            => '1.28',
	'IO::Pipe'              => '1.13',
	'IO::Poll'              => '0.07',
	'IO::Seekable'          => '1.10',
	'IO::Select'            => '1.17',
	'IO::Socket'            => '1.32',
	'IO::Socket::INET'      => '1.31',
	'IO::Socket::UNIX'      => '1.23',
	'IO::Uncompress::Adapter::Bunzip2'=> '2.030',
	'IO::Uncompress::Adapter::Identity'=> '2.030',
	'IO::Uncompress::Adapter::Inflate'=> '2.030',
	'IO::Uncompress::AnyInflate'=> '2.030',
	'IO::Uncompress::AnyUncompress'=> '2.030',
	'IO::Uncompress::Base'  => '2.030',
	'IO::Uncompress::Bunzip2'=> '2.030',
	'IO::Uncompress::Gunzip'=> '2.030',
	'IO::Uncompress::Inflate'=> '2.030',
	'IO::Uncompress::RawInflate'=> '2.030',
	'IO::Uncompress::Unzip' => '2.030',
	'IO::Zlib'              => '1.10',
	'IPC::Cmd'              => '0.60',
	'IPC::Msg'              => '2.03',
	'IPC::Open2'            => '1.03',
	'IPC::Open3'            => '1.06',
	'IPC::Semaphore'        => '2.03',
	'IPC::SharedMem'        => '2.03',
	'IPC::SysV'             => '2.03',
	'List::Util'            => '1.23',
	'List::Util::PP'        => '1.23',
	'List::Util::XS'        => '1.23',
	'Locale::Codes'         => '3.13',
	'Locale::Codes::Country'=> '3.13',
	'Locale::Codes::Currency'=> '3.13',
	'Locale::Codes::Language'=> '3.13',
	'Locale::Codes::Script' => '3.13',
	'Locale::Constants'     => '3.13',
	'Locale::Country'       => '3.13',
	'Locale::Currency'      => '3.13',
	'Locale::Language'      => '3.13',
	'Locale::Maketext'      => '1.15',
	'Locale::Maketext::Guts'=> undef,
	'Locale::Maketext::GutsLoader'=> undef,
	'Locale::Maketext::Simple'=> '0.21',
	'Locale::Script'        => '3.13',
	'Log::Message'          => '0.02',
	'Log::Message::Config'  => '0.01',
	'Log::Message::Handlers'=> undef,
	'Log::Message::Item'    => undef,
	'Log::Message::Simple'  => '0.06',
	'MIME::Base64'          => '3.09',
	'MIME::QuotedPrint'     => '3.09',
	'Math::BigFloat'        => '1.62',
	'Math::BigFloat::Trace' => '0.01',
	'Math::BigInt'          => '1.95',
	'Math::BigInt::Calc'    => '0.54',
	'Math::BigInt::CalcEmu' => '0.06',
	'Math::BigInt::FastCalc'=> '0.22',
	'Math::BigInt::Trace'   => '0.01',
	'Math::BigRat'          => '0.26',
	'Math::Complex'         => '1.56',
	'Math::Trig'            => '1.2',
	'Memoize'               => '1.01_03',
	'Memoize::AnyDBM_File'  => '0.65',
	'Memoize::Expire'       => '1.00',
	'Memoize::ExpireFile'   => '1.01',
	'Memoize::ExpireTest'   => '0.65',
	'Memoize::NDBM_File'    => '0.65',
	'Memoize::SDBM_File'    => '0.65',
	'Memoize::Storable'     => '0.65',
	'Module::Build'         => '0.3607',
	'Module::Build::Base'   => '0.3607',
	'Module::Build::Compat' => '0.3607',
	'Module::Build::Config' => '0.3607',
	'Module::Build::ConfigData'=> undef,
	'Module::Build::Cookbook'=> '0.3607',
	'Module::Build::Dumper' => '0.3607',
	'Module::Build::ModuleInfo'=> '0.3607',
	'Module::Build::Notes'  => '0.3607',
	'Module::Build::PPMMaker'=> '0.3607',
	'Module::Build::Platform::Amiga'=> '0.3607',
	'Module::Build::Platform::Default'=> '0.3607',
	'Module::Build::Platform::EBCDIC'=> '0.3607',
	'Module::Build::Platform::MPEiX'=> '0.3607',
	'Module::Build::Platform::MacOS'=> '0.3607',
	'Module::Build::Platform::RiscOS'=> '0.3607',
	'Module::Build::Platform::Unix'=> '0.3607',
	'Module::Build::Platform::VMS'=> '0.3607',
	'Module::Build::Platform::VOS'=> '0.3607',
	'Module::Build::Platform::Windows'=> '0.3607',
	'Module::Build::Platform::aix'=> '0.3607',
	'Module::Build::Platform::cygwin'=> '0.3607',
	'Module::Build::Platform::darwin'=> '0.3607',
	'Module::Build::Platform::os2'=> '0.3607',
	'Module::Build::PodParser'=> '0.3607',
	'Module::Build::Version'=> '0.77',
	'Module::Build::YAML'   => '1.40',
	'Module::CoreList'      => '2.39',
	'Module::Load'          => '0.18',
	'Module::Load::Conditional'=> '0.38',
	'Module::Loaded'        => '0.06',
	'Module::Pluggable'     => '3.9',
	'Module::Pluggable::Object'=> '3.9',
	'Moped::Msg'            => '0.01',
	'NDBM_File'             => '1.08',
	'NEXT'                  => '0.64',
	'Net::Cmd'              => '2.29',
	'Net::Config'           => '1.11',
	'Net::Domain'           => '2.20',
	'Net::FTP'              => '2.77',
	'Net::FTP::A'           => '1.18',
	'Net::FTP::E'           => '0.01',
	'Net::FTP::I'           => '1.12',
	'Net::FTP::L'           => '0.01',
	'Net::FTP::dataconn'    => '0.11',
	'Net::NNTP'             => '2.24',
	'Net::Netrc'            => '2.12',
	'Net::POP3'             => '2.29',
	'Net::Ping'             => '2.36',
	'Net::SMTP'             => '2.31',
	'Net::Time'             => '2.10',
	'Net::hostent'          => '1.01',
	'Net::netent'           => '1.00',
	'Net::protoent'         => '1.00',
	'Net::servent'          => '1.01',
	'O'                     => '1.01',
	'ODBM_File'             => '1.07',
	'Object::Accessor'      => '0.36',
	'Opcode'                => '1.15',
	'POSIX'                 => '1.20',
	'Package::Constants'    => '0.02',
	'Params::Check'         => '0.26',
	'Parse::CPAN::Meta'     => '1.40',
	'PerlIO'                => '1.06',
	'PerlIO::encoding'      => '0.12',
	'PerlIO::scalar'        => '0.09',
	'PerlIO::via'           => '0.09',
	'PerlIO::via::QuotedPrint'=> '0.06',
	'Pod::Checker'          => '1.45',
	'Pod::Escapes'          => '1.04',
	'Pod::Find'             => '1.35',
	'Pod::Functions'        => '1.04',
	'Pod::Html'             => '1.09',
	'Pod::InputObjects'     => '1.31',
	'Pod::LaTeX'            => '0.58',
	'Pod::Man'              => '2.23',
	'Pod::ParseLink'        => '1.10',
	'Pod::ParseUtils'       => '1.36',
	'Pod::Parser'           => '1.37',
	'Pod::Perldoc'          => '3.15_02',
	'Pod::Perldoc::BaseTo'  => undef,
	'Pod::Perldoc::GetOptsOO'=> undef,
	'Pod::Perldoc::ToChecker'=> undef,
	'Pod::Perldoc::ToMan'   => undef,
	'Pod::Perldoc::ToNroff' => undef,
	'Pod::Perldoc::ToPod'   => undef,
	'Pod::Perldoc::ToRtf'   => undef,
	'Pod::Perldoc::ToText'  => undef,
	'Pod::Perldoc::ToTk'    => undef,
	'Pod::Perldoc::ToXml'   => undef,
	'Pod::PlainText'        => '2.04',
	'Pod::Select'           => '1.36',
	'Pod::Simple'           => '3.14',
	'Pod::Simple::BlackBox' => '3.14',
	'Pod::Simple::Checker'  => '3.14',
	'Pod::Simple::Debug'    => '3.14',
	'Pod::Simple::DumpAsText'=> '3.14',
	'Pod::Simple::DumpAsXML'=> '3.14',
	'Pod::Simple::HTML'     => '3.14',
	'Pod::Simple::HTMLBatch'=> '3.14',
	'Pod::Simple::HTMLLegacy'=> '5.01',
	'Pod::Simple::LinkSection'=> '3.14',
	'Pod::Simple::Methody'  => '3.14',
	'Pod::Simple::Progress' => '3.14',
	'Pod::Simple::PullParser'=> '3.14',
	'Pod::Simple::PullParserEndToken'=> '3.14',
	'Pod::Simple::PullParserStartToken'=> '3.14',
	'Pod::Simple::PullParserTextToken'=> '3.14',
	'Pod::Simple::PullParserToken'=> '3.14',
	'Pod::Simple::RTF'      => '3.14',
	'Pod::Simple::Search'   => '3.14',
	'Pod::Simple::SimpleTree'=> '3.14',
	'Pod::Simple::Text'     => '3.14',
	'Pod::Simple::TextContent'=> '3.14',
	'Pod::Simple::TiedOutFH'=> '3.14',
	'Pod::Simple::Transcode'=> '3.14',
	'Pod::Simple::TranscodeDumb'=> '3.14',
	'Pod::Simple::TranscodeSmart'=> '3.14',
	'Pod::Simple::XHTML'    => '3.14',
	'Pod::Simple::XMLOutStream'=> '3.14',
	'Pod::Text'             => '3.14',
	'Pod::Text::Color'      => '2.06',
	'Pod::Text::Overstrike' => '2.04',
	'Pod::Text::Termcap'    => '2.06',
	'Pod::Usage'            => '1.36',
	'SDBM_File'             => '1.06',
	'Safe'                  => '2.28',
	'Scalar::Util'          => '1.23',
	'Scalar::Util::PP'      => '1.23',
	'Search::Dict'          => '1.03',
	'SelectSaver'           => '1.02',
	'SelfLoader'            => '1.17',
	'Shell'                 => '0.72_01',
	'Socket'                => '1.89',
	'Storable'              => '2.22',
	'Symbol'                => '1.07',
	'Sys::Hostname'         => '1.11',
	'Sys::Syslog'           => '0.27',
	'Sys::Syslog::win32::Win32'=> undef,
	'TAP::Base'             => '3.22',
	'TAP::Formatter::Base'  => '3.22',
	'TAP::Formatter::Color' => '3.22',
	'TAP::Formatter::Console'=> '3.22',
	'TAP::Formatter::Console::ParallelSession'=> '3.22',
	'TAP::Formatter::Console::Session'=> '3.22',
	'TAP::Formatter::File'  => '3.22',
	'TAP::Formatter::File::Session'=> '3.22',
	'TAP::Formatter::Session'=> '3.22',
	'TAP::Harness'          => '3.22',
	'TAP::Object'           => '3.22',
	'TAP::Parser'           => '3.22',
	'TAP::Parser::Aggregator'=> '3.22',
	'TAP::Parser::Grammar'  => '3.22',
	'TAP::Parser::Iterator' => '3.22',
	'TAP::Parser::Iterator::Array'=> '3.22',
	'TAP::Parser::Iterator::Process'=> '3.22',
	'TAP::Parser::Iterator::Stream'=> '3.22',
	'TAP::Parser::IteratorFactory'=> '3.22',
	'TAP::Parser::Multiplexer'=> '3.22',
	'TAP::Parser::Result'   => '3.22',
	'TAP::Parser::Result::Bailout'=> '3.22',
	'TAP::Parser::Result::Comment'=> '3.22',
	'TAP::Parser::Result::Plan'=> '3.22',
	'TAP::Parser::Result::Pragma'=> '3.22',
	'TAP::Parser::Result::Test'=> '3.22',
	'TAP::Parser::Result::Unknown'=> '3.22',
	'TAP::Parser::Result::Version'=> '3.22',
	'TAP::Parser::Result::YAML'=> '3.22',
	'TAP::Parser::ResultFactory'=> '3.22',
	'TAP::Parser::Scheduler'=> '3.22',
	'TAP::Parser::Scheduler::Job'=> '3.22',
	'TAP::Parser::Scheduler::Spinner'=> '3.22',
	'TAP::Parser::Source'   => '3.22',
	'TAP::Parser::SourceHandler'=> '3.22',
	'TAP::Parser::SourceHandler::Executable'=> '3.22',
	'TAP::Parser::SourceHandler::File'=> '3.22',
	'TAP::Parser::SourceHandler::Handle'=> '3.22',
	'TAP::Parser::SourceHandler::Perl'=> '3.22',
	'TAP::Parser::SourceHandler::RawTAP'=> '3.22',
	'TAP::Parser::Utils'    => '3.22',
	'TAP::Parser::YAMLish::Reader'=> '3.22',
	'TAP::Parser::YAMLish::Writer'=> '3.22',
	'Term::ANSIColor'       => '3.00',
	'Term::Cap'             => '1.12',
	'Term::Complete'        => '1.402',
	'Term::ReadLine'        => '1.07',
	'Term::UI'              => '0.20',
	'Term::UI::History'     => undef,
	'Test'                  => '1.25_02',
	'Test::Builder'         => '0.97_01',
	'Test::Builder::Module' => '0.97_01',
	'Test::Builder::Tester' => '1.21_01',
	'Test::Builder::Tester::Color'=> '1.21_01',
	'Test::Harness'         => '3.22',
	'Test::More'            => '0.97_01',
	'Test::Simple'          => '0.97_01',
	'Text::Abbrev'          => '1.01',
	'Text::Balanced'        => '2.02',
	'Text::ParseWords'      => '3.27',
	'Text::Soundex'         => '3.03_01',
	'Text::Tabs'            => '2009.0305',
	'Text::Wrap'            => '2009.0305',
	'Thread'                => '3.02',
	'Thread::Queue'         => '2.11',
	'Thread::Semaphore'     => '2.11',
	'Tie::Array'            => '1.04',
	'Tie::File'             => '0.97_02',
	'Tie::Handle'           => '4.2',
	'Tie::Hash'             => '1.04',
	'Tie::Hash::NamedCapture'=> '0.06',
	'Tie::Memoize'          => '1.1',
	'Tie::RefHash'          => '1.38',
	'Tie::Scalar'           => '1.02',
	'Tie::StdHandle'        => '4.2',
	'Tie::SubstrHash'       => '1.00',
	'Time::HiRes'           => '1.9721',
	'Time::Local'           => '1.1901_01',
	'Time::Piece'           => '1.20_01',
	'Time::Piece::Seconds'  => undef,
	'Time::Seconds'         => undef,
	'Time::gmtime'          => '1.03',
	'Time::localtime'       => '1.02',
	'Time::tm'              => '1.00',
	'UNIVERSAL'             => '1.07',
	'Unicode'               => '5.2.0',
	'Unicode::Collate'      => '0.59',
	'Unicode::Collate::Locale'=> '0.59',
	'Unicode::Normalize'    => '1.06',
	'Unicode::UCD'          => '0.29',
	'User::grent'           => '1.01',
	'User::pwent'           => '1.00',
	'VMS::DCLsym'           => '1.04',
	'VMS::Filespec'         => '1.12',
	'VMS::Stdio'            => '2.4',
	'Win32'                 => '0.39',
	'Win32API::File'        => '0.1101',
	'Win32API::File::ExtUtils::Myconst2perl'=> '1',
	'Win32CORE'             => '0.02',
	'XS::APItest'           => '0.21',
	'XS::APItest::KeywordRPN'=> '0.005',
	'XS::Typemap'           => '0.03',
	'XSLoader'              => '0.11',
	'XSLoader::XSLoader'    => '0.11',
	'attributes'            => '0.12',
	'autodie'               => '2.10',
	'autodie::exception'    => '2.10',
	'autodie::exception::system'=> '2.10',
	'autodie::hints'        => '2.10',
	'autouse'               => '1.06',
	'base'                  => '2.15',
	'bigint'                => '0.25',
	'bignum'                => '0.25',
	'bigrat'                => '0.25',
	'blib'                  => '1.06',
	'bytes'                 => '1.04',
	'charnames'             => '1.15',
	'constant'              => '1.20',
	'deprecate'             => '0.01',
	'diagnostics'           => '1.20',
	'encoding'              => '2.6_01',
	'encoding::warnings'    => '0.11',
	'feature'               => '1.18',
	'fields'                => '2.15',
	'filetest'              => '1.02',
	'if'                    => '0.05',
	'inc::latest'           => '0.3607',
	'integer'               => '1.00',
	'less'                  => '0.03',
	'lib'                   => '0.63',
	'locale'                => '1.00',
	'mro'                   => '1.03',
	'open'                  => '1.08',
	'ops'                   => '1.02',
	'overload'              => '1.10',
	'overload::numbers'     => undef,
	'overloading'           => '0.01',
	'parent'                => '0.223',
	're'                    => '0.12',
	'sigtrap'               => '1.04',
	'sort'                  => '2.01',
	'strict'                => '1.04',
	'subs'                  => '1.00',
	'threads'               => '1.77_03',
	'threads::shared'       => '1.33_03',
	'utf8'                  => '1.08',
	'vars'                  => '1.02',
	'version'               => '0.82',
	'vmsish'                => '1.02',
	'warnings'              => '1.11',
	'warnings::register'    => '1.02',
    },
    5.013006 => {
	'AnyDBM_File'           => '1.00',
	'App::Cpan'             => '1.5701',
	'App::Prove'            => '3.22',
	'App::Prove::State'     => '3.22',
	'App::Prove::State::Result'=> '3.22',
	'App::Prove::State::Result::Test'=> '3.22',
	'Archive::Extract'      => '0.44',
	'Archive::Tar'          => '1.68',
	'Archive::Tar::Constant'=> '1.68',
	'Archive::Tar::File'    => '1.68',
	'Attribute::Handlers'   => '0.88',
	'AutoLoader'            => '5.70',
	'AutoSplit'             => '1.06',
	'B'                     => '1.24',
	'B::Concise'            => '0.79',
	'B::Debug'              => '1.16',
	'B::Deparse'            => '0.99',
	'B::Lint'               => '1.12',
	'B::Lint::Debug'        => '1.12',
	'B::Showlex'            => '1.02',
	'B::Terse'              => '1.05',
	'B::Xref'               => '1.02',
	'Benchmark'             => '1.11',
	'CGI'                   => '3.49',
	'CGI::Apache'           => '1.01',
	'CGI::Carp'             => '3.45',
	'CGI::Cookie'           => '1.29',
	'CGI::Fast'             => '1.08',
	'CGI::Pretty'           => '3.46',
	'CGI::Push'             => '1.04',
	'CGI::Switch'           => '1.01',
	'CGI::Util'             => '3.48',
	'CPAN'                  => '1.94_61',
	'CPAN::Author'          => '5.5',
	'CPAN::Bundle'          => '5.5',
	'CPAN::CacheMgr'        => '5.5',
	'CPAN::Complete'        => '5.5',
	'CPAN::Debug'           => '5.5001',
	'CPAN::DeferredCode'    => '5.50',
	'CPAN::Distribution'    => '1.9600',
	'CPAN::Distroprefs'     => '6',
	'CPAN::Distrostatus'    => '5.5',
	'CPAN::Exception::RecursiveDependency'=> '5.5',
	'CPAN::Exception::blocked_urllist'=> '1.0',
	'CPAN::Exception::yaml_not_installed'=> '5.5',
	'CPAN::FTP'             => '5.5005',
	'CPAN::FTP::netrc'      => '1.00',
	'CPAN::FirstTime'       => '5.5301',
	'CPAN::HandleConfig'    => '5.5001',
	'CPAN::Index'           => '1.94',
	'CPAN::InfoObj'         => '5.5',
	'CPAN::Kwalify'         => '5.50',
	'CPAN::LWP::UserAgent'  => '1.94',
	'CPAN::Mirrors'         => '1.77',
	'CPAN::Module'          => '5.5',
	'CPAN::Nox'             => '5.50',
	'CPAN::Prompt'          => '5.5',
	'CPAN::Queue'           => '5.5001',
	'CPAN::Shell'           => '5.5001',
	'CPAN::Tarzip'          => '5.5011',
	'CPAN::URL'             => '5.5',
	'CPAN::Version'         => '5.5001',
	'CPANPLUS'              => '0.9007',
	'CPANPLUS::Backend'     => undef,
	'CPANPLUS::Backend::RV' => undef,
	'CPANPLUS::Config'      => undef,
	'CPANPLUS::Configure'   => undef,
	'CPANPLUS::Configure::Setup'=> undef,
	'CPANPLUS::Dist'        => undef,
	'CPANPLUS::Dist::Autobundle'=> undef,
	'CPANPLUS::Dist::Base'  => undef,
	'CPANPLUS::Dist::Build' => '0.48',
	'CPANPLUS::Dist::Build::Constants'=> '0.48',
	'CPANPLUS::Dist::MM'    => undef,
	'CPANPLUS::Dist::Sample'=> undef,
	'CPANPLUS::Error'       => undef,
	'CPANPLUS::Internals'   => '0.9007',
	'CPANPLUS::Internals::Constants'=> undef,
	'CPANPLUS::Internals::Constants::Report'=> undef,
	'CPANPLUS::Internals::Extract'=> undef,
	'CPANPLUS::Internals::Fetch'=> undef,
	'CPANPLUS::Internals::Report'=> undef,
	'CPANPLUS::Internals::Search'=> undef,
	'CPANPLUS::Internals::Source'=> undef,
	'CPANPLUS::Internals::Source::Memory'=> undef,
	'CPANPLUS::Internals::Source::SQLite'=> undef,
	'CPANPLUS::Internals::Source::SQLite::Tie'=> undef,
	'CPANPLUS::Internals::Utils'=> undef,
	'CPANPLUS::Internals::Utils::Autoflush'=> undef,
	'CPANPLUS::Module'      => undef,
	'CPANPLUS::Module::Author'=> undef,
	'CPANPLUS::Module::Author::Fake'=> undef,
	'CPANPLUS::Module::Checksums'=> undef,
	'CPANPLUS::Module::Fake'=> undef,
	'CPANPLUS::Module::Signature'=> undef,
	'CPANPLUS::Selfupdate'  => undef,
	'CPANPLUS::Shell'       => undef,
	'CPANPLUS::Shell::Classic'=> '0.0562',
	'CPANPLUS::Shell::Default'=> '0.9007',
	'CPANPLUS::Shell::Default::Plugins::CustomSource'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Remote'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Source'=> undef,
	'Carp'                  => '1.19',
	'Carp::Heavy'           => '1.19',
	'Class::Struct'         => '0.63',
	'Compress::Raw::Bzip2'  => '2.031',
	'Compress::Raw::Zlib'   => '2.030',
	'Compress::Zlib'        => '2.030',
	'Config'                => undef,
	'Config::Extensions'    => '0.01',
	'Cwd'                   => '3.34',
	'DB'                    => '1.03',
	'DBM_Filter'            => '0.03',
	'DBM_Filter::compress'  => '0.02',
	'DBM_Filter::encode'    => '0.02',
	'DBM_Filter::int32'     => '0.02',
	'DBM_Filter::null'      => '0.02',
	'DBM_Filter::utf8'      => '0.02',
	'DB_File'               => '1.820',
	'Data::Dumper'          => '2.129',
	'Devel::DProf'          => '20080331.00',
	'Devel::DProf::V'       => undef,
	'Devel::DProf::dprof::V'=> undef,
	'Devel::InnerPackage'   => '0.3',
	'Devel::PPPort'         => '3.19',
	'Devel::Peek'           => '1.05',
	'Devel::SelfStubber'    => '1.03',
	'Digest'                => '1.16',
	'Digest::MD5'           => '2.51',
	'Digest::SHA'           => '5.48',
	'Digest::base'          => '1.16',
	'Digest::file'          => '1.16',
	'DirHandle'             => '1.03',
	'Dumpvalue'             => '1.14',
	'DynaLoader'            => '1.10',
	'Encode'                => '2.40',
	'Encode::Alias'         => '2.12',
	'Encode::Byte'          => '2.04',
	'Encode::CJKConstants'  => '2.02',
	'Encode::CN'            => '2.03',
	'Encode::CN::HZ'        => '2.05',
	'Encode::Config'        => '2.05',
	'Encode::EBCDIC'        => '2.02',
	'Encode::Encoder'       => '2.01',
	'Encode::Encoding'      => '2.05',
	'Encode::GSM0338'       => '2.01',
	'Encode::Guess'         => '2.04',
	'Encode::JP'            => '2.04',
	'Encode::JP::H2Z'       => '2.02',
	'Encode::JP::JIS7'      => '2.04',
	'Encode::KR'            => '2.03',
	'Encode::KR::2022_KR'   => '2.02',
	'Encode::MIME::Header'  => '2.12',
	'Encode::MIME::Header::ISO_2022_JP'=> '1.03',
	'Encode::MIME::Name'    => '1.01',
	'Encode::Symbol'        => '2.02',
	'Encode::TW'            => '2.03',
	'Encode::Unicode'       => '2.07',
	'Encode::Unicode::UTF7' => '2.05',
	'English'               => '1.04',
	'Env'                   => '1.01',
	'Errno'                 => '1.13',
	'Exporter'              => '5.64_03',
	'Exporter::Heavy'       => '5.64_03',
	'ExtUtils::CBuilder'    => '0.2703',
	'ExtUtils::CBuilder::Base'=> '0.2703_01',
	'ExtUtils::CBuilder::Platform::Unix'=> '0.2703',
	'ExtUtils::CBuilder::Platform::VMS'=> '0.2703',
	'ExtUtils::CBuilder::Platform::Windows'=> '0.2703',
	'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.2703',
	'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.2703',
	'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.2703',
	'ExtUtils::CBuilder::Platform::aix'=> '0.2703',
	'ExtUtils::CBuilder::Platform::cygwin'=> '0.2703',
	'ExtUtils::CBuilder::Platform::darwin'=> '0.2703',
	'ExtUtils::CBuilder::Platform::dec_osf'=> '0.2703',
	'ExtUtils::CBuilder::Platform::os2'=> '0.2703',
	'ExtUtils::Command'     => '1.16',
	'ExtUtils::Command::MM' => '6.57_05',
	'ExtUtils::Constant'    => '0.22',
	'ExtUtils::Constant::Base'=> '0.05',
	'ExtUtils::Constant::ProxySubs'=> '0.07',
	'ExtUtils::Constant::Utils'=> '0.02',
	'ExtUtils::Constant::XS'=> '0.03',
	'ExtUtils::Embed'       => '1.29',
	'ExtUtils::Install'     => '1.55',
	'ExtUtils::Installed'   => '1.999_001',
	'ExtUtils::Liblist'     => '6.57_05',
	'ExtUtils::Liblist::Kid'=> '6.5705',
	'ExtUtils::MM'          => '6.57_05',
	'ExtUtils::MM_AIX'      => '6.57_05',
	'ExtUtils::MM_Any'      => '6.57_05',
	'ExtUtils::MM_BeOS'     => '6.57_05',
	'ExtUtils::MM_Cygwin'   => '6.57_05',
	'ExtUtils::MM_DOS'      => '6.5705',
	'ExtUtils::MM_Darwin'   => '6.57_05',
	'ExtUtils::MM_MacOS'    => '6.5705',
	'ExtUtils::MM_NW5'      => '6.57_05',
	'ExtUtils::MM_OS2'      => '6.57_05',
	'ExtUtils::MM_QNX'      => '6.57_05',
	'ExtUtils::MM_UWIN'     => '6.5705',
	'ExtUtils::MM_Unix'     => '6.57_05',
	'ExtUtils::MM_VMS'      => '6.57_05',
	'ExtUtils::MM_VOS'      => '6.57_05',
	'ExtUtils::MM_Win32'    => '6.57_05',
	'ExtUtils::MM_Win95'    => '6.57_05',
	'ExtUtils::MY'          => '6.5705',
	'ExtUtils::MakeMaker'   => '6.57_05',
	'ExtUtils::MakeMaker::Config'=> '6.57_05',
	'ExtUtils::MakeMaker::YAML'=> '1.44',
	'ExtUtils::Manifest'    => '1.58',
	'ExtUtils::Miniperl'    => undef,
	'ExtUtils::Mkbootstrap' => '6.57_05',
	'ExtUtils::Mksymlists'  => '6.57_05',
	'ExtUtils::Packlist'    => '1.44',
	'ExtUtils::ParseXS'     => '2.2207',
	'ExtUtils::XSSymSet'    => '1.2',
	'ExtUtils::testlib'     => '6.5705',
	'Fatal'                 => '2.10',
	'Fcntl'                 => '1.09',
	'File::Basename'        => '2.79',
	'File::CheckTree'       => '4.4',
	'File::Compare'         => '1.1006',
	'File::Copy'            => '2.20',
	'File::DosGlob'         => '1.03',
	'File::Fetch'           => '0.24',
	'File::Find'            => '1.18',
	'File::Glob'            => '1.09',
	'File::GlobMapper'      => '1.000',
	'File::Path'            => '2.08_01',
	'File::Spec'            => '3.33',
	'File::Spec::Cygwin'    => '3.33',
	'File::Spec::Epoc'      => '3.33',
	'File::Spec::Functions' => '3.33',
	'File::Spec::Mac'       => '3.33',
	'File::Spec::OS2'       => '3.33',
	'File::Spec::Unix'      => '3.33',
	'File::Spec::VMS'       => '3.33',
	'File::Spec::Win32'     => '3.33',
	'File::Temp'            => '0.22',
	'File::stat'            => '1.03',
	'FileCache'             => '1.08',
	'FileHandle'            => '2.02',
	'Filter::Simple'        => '0.85',
	'Filter::Util::Call'    => '1.08',
	'FindBin'               => '1.50',
	'GDBM_File'             => '1.11',
	'Getopt::Long'          => '2.38',
	'Getopt::Std'           => '1.06',
	'Hash::Util'            => '0.09',
	'Hash::Util::FieldHash' => '1.05',
	'I18N::Collate'         => '1.01',
	'I18N::LangTags'        => '0.35_01',
	'I18N::LangTags::Detect'=> '1.04',
	'I18N::LangTags::List'  => '0.35_01',
	'I18N::Langinfo'        => '0.06',
	'IO'                    => '1.25_02',
	'IO::Compress::Adapter::Bzip2'=> '2.030',
	'IO::Compress::Adapter::Deflate'=> '2.030',
	'IO::Compress::Adapter::Identity'=> '2.030',
	'IO::Compress::Base'    => '2.030',
	'IO::Compress::Base::Common'=> '2.030',
	'IO::Compress::Bzip2'   => '2.030',
	'IO::Compress::Deflate' => '2.030',
	'IO::Compress::Gzip'    => '2.030',
	'IO::Compress::Gzip::Constants'=> '2.030',
	'IO::Compress::RawDeflate'=> '2.030',
	'IO::Compress::Zip'     => '2.030',
	'IO::Compress::Zip::Constants'=> '2.030',
	'IO::Compress::Zlib::Constants'=> '2.030',
	'IO::Compress::Zlib::Extra'=> '2.030',
	'IO::Dir'               => '1.07',
	'IO::File'              => '1.14',
	'IO::Handle'            => '1.28',
	'IO::Pipe'              => '1.13',
	'IO::Poll'              => '0.07',
	'IO::Seekable'          => '1.10',
	'IO::Select'            => '1.17',
	'IO::Socket'            => '1.32',
	'IO::Socket::INET'      => '1.31',
	'IO::Socket::UNIX'      => '1.23',
	'IO::Uncompress::Adapter::Bunzip2'=> '2.030',
	'IO::Uncompress::Adapter::Identity'=> '2.030',
	'IO::Uncompress::Adapter::Inflate'=> '2.030',
	'IO::Uncompress::AnyInflate'=> '2.030',
	'IO::Uncompress::AnyUncompress'=> '2.030',
	'IO::Uncompress::Base'  => '2.030',
	'IO::Uncompress::Bunzip2'=> '2.030',
	'IO::Uncompress::Gunzip'=> '2.030',
	'IO::Uncompress::Inflate'=> '2.030',
	'IO::Uncompress::RawInflate'=> '2.030',
	'IO::Uncompress::Unzip' => '2.030',
	'IO::Zlib'              => '1.10',
	'IPC::Cmd'              => '0.64',
	'IPC::Msg'              => '2.03',
	'IPC::Open2'            => '1.03',
	'IPC::Open3'            => '1.07',
	'IPC::Semaphore'        => '2.03',
	'IPC::SharedMem'        => '2.03',
	'IPC::SysV'             => '2.03',
	'List::Util'            => '1.23',
	'List::Util::PP'        => '1.23',
	'List::Util::XS'        => '1.23',
	'Locale::Codes'         => '3.14',
	'Locale::Codes::Country'=> '3.14',
	'Locale::Codes::Currency'=> '3.14',
	'Locale::Codes::Language'=> '3.14',
	'Locale::Codes::Script' => '3.14',
	'Locale::Constants'     => '3.14',
	'Locale::Country'       => '3.14',
	'Locale::Currency'      => '3.14',
	'Locale::Language'      => '3.14',
	'Locale::Maketext'      => '1.16',
	'Locale::Maketext::Guts'=> undef,
	'Locale::Maketext::GutsLoader'=> undef,
	'Locale::Maketext::Simple'=> '0.21',
	'Locale::Script'        => '3.14',
	'Log::Message'          => '0.02',
	'Log::Message::Config'  => '0.01',
	'Log::Message::Handlers'=> undef,
	'Log::Message::Item'    => undef,
	'Log::Message::Simple'  => '0.06',
	'MIME::Base64'          => '3.09',
	'MIME::QuotedPrint'     => '3.09',
	'Math::BigFloat'        => '1.63',
	'Math::BigFloat::Trace' => '0.01',
	'Math::BigInt'          => '1.97',
	'Math::BigInt::Calc'    => '0.55',
	'Math::BigInt::CalcEmu' => '0.07',
	'Math::BigInt::FastCalc'=> '0.22',
	'Math::BigInt::Trace'   => '0.01',
	'Math::BigRat'          => '0.26',
	'Math::Complex'         => '1.56',
	'Math::Trig'            => '1.2',
	'Memoize'               => '1.01_03',
	'Memoize::AnyDBM_File'  => '0.65',
	'Memoize::Expire'       => '1.00',
	'Memoize::ExpireFile'   => '1.01',
	'Memoize::ExpireTest'   => '0.65',
	'Memoize::NDBM_File'    => '0.65',
	'Memoize::SDBM_File'    => '0.65',
	'Memoize::Storable'     => '0.65',
	'Module::Build'         => '0.3607',
	'Module::Build::Base'   => '0.3607',
	'Module::Build::Compat' => '0.3607',
	'Module::Build::Config' => '0.3607',
	'Module::Build::ConfigData'=> undef,
	'Module::Build::Cookbook'=> '0.3607',
	'Module::Build::Dumper' => '0.3607',
	'Module::Build::ModuleInfo'=> '0.3607',
	'Module::Build::Notes'  => '0.3607',
	'Module::Build::PPMMaker'=> '0.3607',
	'Module::Build::Platform::Amiga'=> '0.3607',
	'Module::Build::Platform::Default'=> '0.3607',
	'Module::Build::Platform::EBCDIC'=> '0.3607',
	'Module::Build::Platform::MPEiX'=> '0.3607',
	'Module::Build::Platform::MacOS'=> '0.3607',
	'Module::Build::Platform::RiscOS'=> '0.3607',
	'Module::Build::Platform::Unix'=> '0.3607',
	'Module::Build::Platform::VMS'=> '0.3607',
	'Module::Build::Platform::VOS'=> '0.3607',
	'Module::Build::Platform::Windows'=> '0.3607',
	'Module::Build::Platform::aix'=> '0.3607',
	'Module::Build::Platform::cygwin'=> '0.3607',
	'Module::Build::Platform::darwin'=> '0.3607',
	'Module::Build::Platform::os2'=> '0.3607',
	'Module::Build::PodParser'=> '0.3607',
	'Module::Build::Version'=> '0.77',
	'Module::Build::YAML'   => '1.40',
	'Module::CoreList'      => '2.40',
	'Module::Load'          => '0.18',
	'Module::Load::Conditional'=> '0.38',
	'Module::Loaded'        => '0.06',
	'Module::Pluggable'     => '3.9',
	'Module::Pluggable::Object'=> '3.9',
	'Moped::Msg'            => '0.01',
	'NDBM_File'             => '1.09',
	'NEXT'                  => '0.65',
	'Net::Cmd'              => '2.29',
	'Net::Config'           => '1.11',
	'Net::Domain'           => '2.20',
	'Net::FTP'              => '2.77',
	'Net::FTP::A'           => '1.18',
	'Net::FTP::E'           => '0.01',
	'Net::FTP::I'           => '1.12',
	'Net::FTP::L'           => '0.01',
	'Net::FTP::dataconn'    => '0.11',
	'Net::NNTP'             => '2.24',
	'Net::Netrc'            => '2.12',
	'Net::POP3'             => '2.29',
	'Net::Ping'             => '2.36',
	'Net::SMTP'             => '2.31',
	'Net::Time'             => '2.10',
	'Net::hostent'          => '1.01',
	'Net::netent'           => '1.00',
	'Net::protoent'         => '1.00',
	'Net::servent'          => '1.01',
	'O'                     => '1.01',
	'ODBM_File'             => '1.08',
	'Object::Accessor'      => '0.36',
	'Opcode'                => '1.16',
	'POSIX'                 => '1.21',
	'Package::Constants'    => '0.02',
	'Params::Check'         => '0.26',
	'Parse::CPAN::Meta'     => '1.40',
	'PerlIO'                => '1.06',
	'PerlIO::encoding'      => '0.13',
	'PerlIO::scalar'        => '0.10',
	'PerlIO::via'           => '0.10',
	'PerlIO::via::QuotedPrint'=> '0.06',
	'Pod::Checker'          => '1.45',
	'Pod::Escapes'          => '1.04',
	'Pod::Find'             => '1.35',
	'Pod::Functions'        => '1.04',
	'Pod::Html'             => '1.09',
	'Pod::InputObjects'     => '1.31',
	'Pod::LaTeX'            => '0.58',
	'Pod::Man'              => '2.25',
	'Pod::ParseLink'        => '1.10',
	'Pod::ParseUtils'       => '1.36',
	'Pod::Parser'           => '1.37',
	'Pod::Perldoc'          => '3.15_02',
	'Pod::Perldoc::BaseTo'  => undef,
	'Pod::Perldoc::GetOptsOO'=> undef,
	'Pod::Perldoc::ToChecker'=> undef,
	'Pod::Perldoc::ToMan'   => undef,
	'Pod::Perldoc::ToNroff' => undef,
	'Pod::Perldoc::ToPod'   => undef,
	'Pod::Perldoc::ToRtf'   => undef,
	'Pod::Perldoc::ToText'  => undef,
	'Pod::Perldoc::ToTk'    => undef,
	'Pod::Perldoc::ToXml'   => undef,
	'Pod::PlainText'        => '2.04',
	'Pod::Select'           => '1.36',
	'Pod::Simple'           => '3.14',
	'Pod::Simple::BlackBox' => '3.14',
	'Pod::Simple::Checker'  => '3.14',
	'Pod::Simple::Debug'    => '3.14',
	'Pod::Simple::DumpAsText'=> '3.14',
	'Pod::Simple::DumpAsXML'=> '3.14',
	'Pod::Simple::HTML'     => '3.14',
	'Pod::Simple::HTMLBatch'=> '3.14',
	'Pod::Simple::HTMLLegacy'=> '5.01',
	'Pod::Simple::LinkSection'=> '3.14',
	'Pod::Simple::Methody'  => '3.14',
	'Pod::Simple::Progress' => '3.14',
	'Pod::Simple::PullParser'=> '3.14',
	'Pod::Simple::PullParserEndToken'=> '3.14',
	'Pod::Simple::PullParserStartToken'=> '3.14',
	'Pod::Simple::PullParserTextToken'=> '3.14',
	'Pod::Simple::PullParserToken'=> '3.14',
	'Pod::Simple::RTF'      => '3.14',
	'Pod::Simple::Search'   => '3.14',
	'Pod::Simple::SimpleTree'=> '3.14',
	'Pod::Simple::Text'     => '3.14',
	'Pod::Simple::TextContent'=> '3.14',
	'Pod::Simple::TiedOutFH'=> '3.14',
	'Pod::Simple::Transcode'=> '3.14',
	'Pod::Simple::TranscodeDumb'=> '3.14',
	'Pod::Simple::TranscodeSmart'=> '3.14',
	'Pod::Simple::XHTML'    => '3.14',
	'Pod::Simple::XMLOutStream'=> '3.14',
	'Pod::Text'             => '3.15',
	'Pod::Text::Color'      => '2.06',
	'Pod::Text::Overstrike' => '2.04',
	'Pod::Text::Termcap'    => '2.06',
	'Pod::Usage'            => '1.36',
	'SDBM_File'             => '1.07',
	'Safe'                  => '2.28',
	'Scalar::Util'          => '1.23',
	'Scalar::Util::PP'      => '1.23',
	'Search::Dict'          => '1.03',
	'SelectSaver'           => '1.02',
	'SelfLoader'            => '1.17',
	'Shell'                 => '0.72_01',
	'Socket'                => '1.90',
	'Storable'              => '2.22',
	'Symbol'                => '1.07',
	'Sys::Hostname'         => '1.13',
	'Sys::Syslog'           => '0.27',
	'Sys::Syslog::win32::Win32'=> undef,
	'TAP::Base'             => '3.22',
	'TAP::Formatter::Base'  => '3.22',
	'TAP::Formatter::Color' => '3.22',
	'TAP::Formatter::Console'=> '3.22',
	'TAP::Formatter::Console::ParallelSession'=> '3.22',
	'TAP::Formatter::Console::Session'=> '3.22',
	'TAP::Formatter::File'  => '3.22',
	'TAP::Formatter::File::Session'=> '3.22',
	'TAP::Formatter::Session'=> '3.22',
	'TAP::Harness'          => '3.22',
	'TAP::Object'           => '3.22',
	'TAP::Parser'           => '3.22',
	'TAP::Parser::Aggregator'=> '3.22',
	'TAP::Parser::Grammar'  => '3.22',
	'TAP::Parser::Iterator' => '3.22',
	'TAP::Parser::Iterator::Array'=> '3.22',
	'TAP::Parser::Iterator::Process'=> '3.22',
	'TAP::Parser::Iterator::Stream'=> '3.22',
	'TAP::Parser::IteratorFactory'=> '3.22',
	'TAP::Parser::Multiplexer'=> '3.22',
	'TAP::Parser::Result'   => '3.22',
	'TAP::Parser::Result::Bailout'=> '3.22',
	'TAP::Parser::Result::Comment'=> '3.22',
	'TAP::Parser::Result::Plan'=> '3.22',
	'TAP::Parser::Result::Pragma'=> '3.22',
	'TAP::Parser::Result::Test'=> '3.22',
	'TAP::Parser::Result::Unknown'=> '3.22',
	'TAP::Parser::Result::Version'=> '3.22',
	'TAP::Parser::Result::YAML'=> '3.22',
	'TAP::Parser::ResultFactory'=> '3.22',
	'TAP::Parser::Scheduler'=> '3.22',
	'TAP::Parser::Scheduler::Job'=> '3.22',
	'TAP::Parser::Scheduler::Spinner'=> '3.22',
	'TAP::Parser::Source'   => '3.22',
	'TAP::Parser::SourceHandler'=> '3.22',
	'TAP::Parser::SourceHandler::Executable'=> '3.22',
	'TAP::Parser::SourceHandler::File'=> '3.22',
	'TAP::Parser::SourceHandler::Handle'=> '3.22',
	'TAP::Parser::SourceHandler::Perl'=> '3.22',
	'TAP::Parser::SourceHandler::RawTAP'=> '3.22',
	'TAP::Parser::Utils'    => '3.22',
	'TAP::Parser::YAMLish::Reader'=> '3.22',
	'TAP::Parser::YAMLish::Writer'=> '3.22',
	'Term::ANSIColor'       => '3.00',
	'Term::Cap'             => '1.12',
	'Term::Complete'        => '1.402',
	'Term::ReadLine'        => '1.07',
	'Term::UI'              => '0.20',
	'Term::UI::History'     => undef,
	'Test'                  => '1.25_02',
	'Test::Builder'         => '0.97_01',
	'Test::Builder::Module' => '0.97_01',
	'Test::Builder::Tester' => '1.21_01',
	'Test::Builder::Tester::Color'=> '1.21_01',
	'Test::Harness'         => '3.22',
	'Test::More'            => '0.97_01',
	'Test::Simple'          => '0.97_01',
	'Text::Abbrev'          => '1.01',
	'Text::Balanced'        => '2.02',
	'Text::ParseWords'      => '3.27',
	'Text::Soundex'         => '3.03_01',
	'Text::Tabs'            => '2009.0305',
	'Text::Wrap'            => '2009.0305',
	'Thread'                => '3.02',
	'Thread::Queue'         => '2.11',
	'Thread::Semaphore'     => '2.11',
	'Tie::Array'            => '1.04',
	'Tie::File'             => '0.97_02',
	'Tie::Handle'           => '4.2',
	'Tie::Hash'             => '1.04',
	'Tie::Hash::NamedCapture'=> '0.07',
	'Tie::Memoize'          => '1.1',
	'Tie::RefHash'          => '1.38',
	'Tie::Scalar'           => '1.02',
	'Tie::StdHandle'        => '4.2',
	'Tie::SubstrHash'       => '1.00',
	'Time::HiRes'           => '1.9721',
	'Time::Local'           => '1.1901_01',
	'Time::Piece'           => '1.20_01',
	'Time::Piece::Seconds'  => undef,
	'Time::Seconds'         => undef,
	'Time::gmtime'          => '1.03',
	'Time::localtime'       => '1.02',
	'Time::tm'              => '1.00',
	'UNIVERSAL'             => '1.07',
	'Unicode'               => '5.2.0',
	'Unicode::Collate'      => '0.63',
	'Unicode::Collate::Locale'=> '0.63',
	'Unicode::Normalize'    => '1.07',
	'Unicode::UCD'          => '0.29',
	'User::grent'           => '1.01',
	'User::pwent'           => '1.00',
	'VMS::DCLsym'           => '1.04',
	'VMS::Filespec'         => '1.12',
	'VMS::Stdio'            => '2.4',
	'Win32'                 => '0.39',
	'Win32API::File'        => '0.1101',
	'Win32API::File::ExtUtils::Myconst2perl'=> '1',
	'Win32CORE'             => '0.02',
	'XS::APItest'           => '0.23',
	'XS::Typemap'           => '0.03',
	'XSLoader'              => '0.13',
	'XSLoader::XSLoader'    => '0.13',
	'attributes'            => '0.13',
	'autodie'               => '2.10',
	'autodie::exception'    => '2.10',
	'autodie::exception::system'=> '2.10',
	'autodie::hints'        => '2.10',
	'autouse'               => '1.06',
	'base'                  => '2.15',
	'bigint'                => '0.25',
	'bignum'                => '0.25',
	'bigrat'                => '0.25',
	'blib'                  => '1.06',
	'bytes'                 => '1.04',
	'charnames'             => '1.16',
	'constant'              => '1.20',
	'deprecate'             => '0.01',
	'diagnostics'           => '1.20',
	'encoding'              => '2.6_01',
	'encoding::warnings'    => '0.11',
	'feature'               => '1.18',
	'fields'                => '2.15',
	'filetest'              => '1.02',
	'if'                    => '0.06',
	'inc::latest'           => '0.3607',
	'integer'               => '1.00',
	'less'                  => '0.03',
	'lib'                   => '0.63',
	'locale'                => '1.00',
	'mro'                   => '1.04',
	'open'                  => '1.08',
	'ops'                   => '1.02',
	'overload'              => '1.11',
	'overload::numbers'     => undef,
	'overloading'           => '0.01',
	'parent'                => '0.223',
	're'                    => '0.13',
	'sigtrap'               => '1.05',
	'sort'                  => '2.01',
	'strict'                => '1.04',
	'subs'                  => '1.00',
	'threads'               => '1.81_01',
	'threads::shared'       => '1.34',
	'utf8'                  => '1.08',
	'vars'                  => '1.02',
	'version'               => '0.82',
	'vmsish'                => '1.02',
	'warnings'              => '1.11',
	'warnings::register'    => '1.02',
    },
    5.013007 => {
	'AnyDBM_File'           => '1.00',
	'App::Cpan'             => '1.5701',
	'App::Prove'            => '3.22',
	'App::Prove::State'     => '3.22',
	'App::Prove::State::Result'=> '3.22',
	'App::Prove::State::Result::Test'=> '3.22',
	'Archive::Extract'      => '0.46',
	'Archive::Tar'          => '1.72',
	'Archive::Tar::Constant'=> '1.72',
	'Archive::Tar::File'    => '1.72',
	'Attribute::Handlers'   => '0.88',
	'AutoLoader'            => '5.71',
	'AutoSplit'             => '1.06',
	'B'                     => '1.26',
	'B::Concise'            => '0.81',
	'B::Debug'              => '1.16',
	'B::Deparse'            => '1.01',
	'B::Lint'               => '1.12',
	'B::Lint::Debug'        => '1.12',
	'B::Showlex'            => '1.02',
	'B::Terse'              => '1.05',
	'B::Xref'               => '1.02',
	'Benchmark'             => '1.11',
	'CGI'                   => '3.50',
	'CGI::Apache'           => '1.01',
	'CGI::Carp'             => '3.45',
	'CGI::Cookie'           => '1.29',
	'CGI::Fast'             => '1.08',
	'CGI::Pretty'           => '3.46',
	'CGI::Push'             => '1.04',
	'CGI::Switch'           => '1.01',
	'CGI::Util'             => '3.48',
	'CPAN'                  => '1.94_62',
	'CPAN::Author'          => '5.5',
	'CPAN::Bundle'          => '5.5',
	'CPAN::CacheMgr'        => '5.5',
	'CPAN::Complete'        => '5.5',
	'CPAN::Debug'           => '5.5001',
	'CPAN::DeferredCode'    => '5.50',
	'CPAN::Distribution'    => '1.9600',
	'CPAN::Distroprefs'     => '6',
	'CPAN::Distrostatus'    => '5.5',
	'CPAN::Exception::RecursiveDependency'=> '5.5',
	'CPAN::Exception::blocked_urllist'=> '1.0',
	'CPAN::Exception::yaml_not_installed'=> '5.5',
	'CPAN::FTP'             => '5.5005',
	'CPAN::FTP::netrc'      => '1.00',
	'CPAN::FirstTime'       => '5.5301',
	'CPAN::HandleConfig'    => '5.5001',
	'CPAN::Index'           => '1.94',
	'CPAN::InfoObj'         => '5.5',
	'CPAN::Kwalify'         => '5.50',
	'CPAN::LWP::UserAgent'  => '1.94',
	'CPAN::Mirrors'         => '1.77',
	'CPAN::Module'          => '5.5',
	'CPAN::Nox'             => '5.50',
	'CPAN::Prompt'          => '5.5',
	'CPAN::Queue'           => '5.5001',
	'CPAN::Shell'           => '5.5001',
	'CPAN::Tarzip'          => '5.5011',
	'CPAN::URL'             => '5.5',
	'CPAN::Version'         => '5.5001',
	'CPANPLUS'              => '0.9010',
	'CPANPLUS::Backend'     => undef,
	'CPANPLUS::Backend::RV' => undef,
	'CPANPLUS::Config'      => undef,
	'CPANPLUS::Configure'   => undef,
	'CPANPLUS::Configure::Setup'=> undef,
	'CPANPLUS::Dist'        => undef,
	'CPANPLUS::Dist::Autobundle'=> undef,
	'CPANPLUS::Dist::Base'  => undef,
	'CPANPLUS::Dist::Build' => '0.50',
	'CPANPLUS::Dist::Build::Constants'=> '0.50',
	'CPANPLUS::Dist::MM'    => undef,
	'CPANPLUS::Dist::Sample'=> undef,
	'CPANPLUS::Error'       => undef,
	'CPANPLUS::Internals'   => '0.9010',
	'CPANPLUS::Internals::Constants'=> undef,
	'CPANPLUS::Internals::Constants::Report'=> undef,
	'CPANPLUS::Internals::Extract'=> undef,
	'CPANPLUS::Internals::Fetch'=> undef,
	'CPANPLUS::Internals::Report'=> undef,
	'CPANPLUS::Internals::Search'=> undef,
	'CPANPLUS::Internals::Source'=> undef,
	'CPANPLUS::Internals::Source::Memory'=> undef,
	'CPANPLUS::Internals::Source::SQLite'=> undef,
	'CPANPLUS::Internals::Source::SQLite::Tie'=> undef,
	'CPANPLUS::Internals::Utils'=> undef,
	'CPANPLUS::Internals::Utils::Autoflush'=> undef,
	'CPANPLUS::Module'      => undef,
	'CPANPLUS::Module::Author'=> undef,
	'CPANPLUS::Module::Author::Fake'=> undef,
	'CPANPLUS::Module::Checksums'=> undef,
	'CPANPLUS::Module::Fake'=> undef,
	'CPANPLUS::Module::Signature'=> undef,
	'CPANPLUS::Selfupdate'  => undef,
	'CPANPLUS::Shell'       => undef,
	'CPANPLUS::Shell::Classic'=> '0.0562',
	'CPANPLUS::Shell::Default'=> '0.9010',
	'CPANPLUS::Shell::Default::Plugins::CustomSource'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Remote'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Source'=> undef,
	'Carp'                  => '1.19',
	'Carp::Heavy'           => '1.19',
	'Class::Struct'         => '0.63',
	'Compress::Raw::Bzip2'  => '2.031',
	'Compress::Raw::Zlib'   => '2.030',
	'Compress::Zlib'        => '2.030',
	'Config'                => undef,
	'Config::Extensions'    => '0.01',
	'Cwd'                   => '3.34',
	'DB'                    => '1.03',
	'DBM_Filter'            => '0.03',
	'DBM_Filter::compress'  => '0.02',
	'DBM_Filter::encode'    => '0.02',
	'DBM_Filter::int32'     => '0.02',
	'DBM_Filter::null'      => '0.02',
	'DBM_Filter::utf8'      => '0.02',
	'DB_File'               => '1.820',
	'Data::Dumper'          => '2.130_01',
	'Devel::DProf'          => '20080331.00',
	'Devel::DProf::dprof::V'=> undef,
	'Devel::InnerPackage'   => '0.3',
	'Devel::PPPort'         => '3.19',
	'Devel::Peek'           => '1.05',
	'Devel::SelfStubber'    => '1.03',
	'Digest'                => '1.16',
	'Digest::MD5'           => '2.51',
	'Digest::SHA'           => '5.48',
	'Digest::base'          => '1.16',
	'Digest::file'          => '1.16',
	'DirHandle'             => '1.03',
	'Dumpvalue'             => '1.14',
	'DynaLoader'            => '1.11',
	'Encode'                => '2.40',
	'Encode::Alias'         => '2.12',
	'Encode::Byte'          => '2.04',
	'Encode::CJKConstants'  => '2.02',
	'Encode::CN'            => '2.03',
	'Encode::CN::HZ'        => '2.05',
	'Encode::Config'        => '2.05',
	'Encode::EBCDIC'        => '2.02',
	'Encode::Encoder'       => '2.01',
	'Encode::Encoding'      => '2.05',
	'Encode::GSM0338'       => '2.01',
	'Encode::Guess'         => '2.04',
	'Encode::JP'            => '2.04',
	'Encode::JP::H2Z'       => '2.02',
	'Encode::JP::JIS7'      => '2.04',
	'Encode::KR'            => '2.03',
	'Encode::KR::2022_KR'   => '2.02',
	'Encode::MIME::Header'  => '2.12',
	'Encode::MIME::Header::ISO_2022_JP'=> '1.03',
	'Encode::MIME::Name'    => '1.01',
	'Encode::Symbol'        => '2.02',
	'Encode::TW'            => '2.03',
	'Encode::Unicode'       => '2.07',
	'Encode::Unicode::UTF7' => '2.05',
	'English'               => '1.04',
	'Env'                   => '1.01',
	'Errno'                 => '1.13',
	'Exporter'              => '5.64_03',
	'Exporter::Heavy'       => '5.64_03',
	'ExtUtils::CBuilder'    => '0.2703',
	'ExtUtils::CBuilder::Base'=> '0.2703_01',
	'ExtUtils::CBuilder::Platform::Unix'=> '0.2703',
	'ExtUtils::CBuilder::Platform::VMS'=> '0.2703',
	'ExtUtils::CBuilder::Platform::Windows'=> '0.2703',
	'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.2703',
	'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.2703',
	'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.2703',
	'ExtUtils::CBuilder::Platform::aix'=> '0.2703',
	'ExtUtils::CBuilder::Platform::cygwin'=> '0.2703',
	'ExtUtils::CBuilder::Platform::darwin'=> '0.2703',
	'ExtUtils::CBuilder::Platform::dec_osf'=> '0.2703',
	'ExtUtils::CBuilder::Platform::os2'=> '0.2703',
	'ExtUtils::Command'     => '1.16',
	'ExtUtils::Command::MM' => '6.57_05',
	'ExtUtils::Constant'    => '0.23',
	'ExtUtils::Constant::Base'=> '0.05',
	'ExtUtils::Constant::ProxySubs'=> '0.08',
	'ExtUtils::Constant::Utils'=> '0.02',
	'ExtUtils::Constant::XS'=> '0.03',
	'ExtUtils::Embed'       => '1.29',
	'ExtUtils::Install'     => '1.55',
	'ExtUtils::Installed'   => '1.999_001',
	'ExtUtils::Liblist'     => '6.57_05',
	'ExtUtils::Liblist::Kid'=> '6.5705',
	'ExtUtils::MM'          => '6.57_05',
	'ExtUtils::MM_AIX'      => '6.57_05',
	'ExtUtils::MM_Any'      => '6.57_05',
	'ExtUtils::MM_BeOS'     => '6.57_05',
	'ExtUtils::MM_Cygwin'   => '6.57_05',
	'ExtUtils::MM_DOS'      => '6.5705',
	'ExtUtils::MM_Darwin'   => '6.57_05',
	'ExtUtils::MM_MacOS'    => '6.5705',
	'ExtUtils::MM_NW5'      => '6.57_05',
	'ExtUtils::MM_OS2'      => '6.57_05',
	'ExtUtils::MM_QNX'      => '6.57_05',
	'ExtUtils::MM_UWIN'     => '6.5705',
	'ExtUtils::MM_Unix'     => '6.57_05',
	'ExtUtils::MM_VMS'      => '6.57_05',
	'ExtUtils::MM_VOS'      => '6.57_05',
	'ExtUtils::MM_Win32'    => '6.57_05',
	'ExtUtils::MM_Win95'    => '6.57_05',
	'ExtUtils::MY'          => '6.5705',
	'ExtUtils::MakeMaker'   => '6.57_05',
	'ExtUtils::MakeMaker::Config'=> '6.57_05',
	'ExtUtils::MakeMaker::YAML'=> '1.44',
	'ExtUtils::Manifest'    => '1.58',
	'ExtUtils::Miniperl'    => undef,
	'ExtUtils::Mkbootstrap' => '6.57_05',
	'ExtUtils::Mksymlists'  => '6.57_05',
	'ExtUtils::Packlist'    => '1.44',
	'ExtUtils::ParseXS'     => '2.2207',
	'ExtUtils::XSSymSet'    => '1.2',
	'ExtUtils::testlib'     => '6.5705',
	'Fatal'                 => '2.10',
	'Fcntl'                 => '1.10',
	'File::Basename'        => '2.79',
	'File::CheckTree'       => '4.4',
	'File::Compare'         => '1.1006',
	'File::Copy'            => '2.20',
	'File::DosGlob'         => '1.03',
	'File::Fetch'           => '0.28',
	'File::Find'            => '1.18',
	'File::Glob'            => '1.10',
	'File::GlobMapper'      => '1.000',
	'File::Path'            => '2.08_01',
	'File::Spec'            => '3.33',
	'File::Spec::Cygwin'    => '3.33',
	'File::Spec::Epoc'      => '3.33',
	'File::Spec::Functions' => '3.33',
	'File::Spec::Mac'       => '3.33',
	'File::Spec::OS2'       => '3.33',
	'File::Spec::Unix'      => '3.33',
	'File::Spec::VMS'       => '3.33',
	'File::Spec::Win32'     => '3.33',
	'File::Temp'            => '0.22',
	'File::stat'            => '1.04',
	'FileCache'             => '1.08',
	'FileHandle'            => '2.02',
	'Filter::Simple'        => '0.85',
	'Filter::Util::Call'    => '1.08',
	'FindBin'               => '1.50',
	'GDBM_File'             => '1.12',
	'Getopt::Long'          => '2.38',
	'Getopt::Std'           => '1.06',
	'Hash::Util'            => '0.10',
	'Hash::Util::FieldHash' => '1.06',
	'I18N::Collate'         => '1.01',
	'I18N::LangTags'        => '0.35_01',
	'I18N::LangTags::Detect'=> '1.04',
	'I18N::LangTags::List'  => '0.35_01',
	'I18N::Langinfo'        => '0.07',
	'IO'                    => '1.25_02',
	'IO::Compress::Adapter::Bzip2'=> '2.030',
	'IO::Compress::Adapter::Deflate'=> '2.030',
	'IO::Compress::Adapter::Identity'=> '2.030',
	'IO::Compress::Base'    => '2.030',
	'IO::Compress::Base::Common'=> '2.030',
	'IO::Compress::Bzip2'   => '2.030',
	'IO::Compress::Deflate' => '2.030',
	'IO::Compress::Gzip'    => '2.030',
	'IO::Compress::Gzip::Constants'=> '2.030',
	'IO::Compress::RawDeflate'=> '2.030',
	'IO::Compress::Zip'     => '2.030',
	'IO::Compress::Zip::Constants'=> '2.030',
	'IO::Compress::Zlib::Constants'=> '2.030',
	'IO::Compress::Zlib::Extra'=> '2.030',
	'IO::Dir'               => '1.07',
	'IO::File'              => '1.14',
	'IO::Handle'            => '1.28',
	'IO::Pipe'              => '1.13',
	'IO::Poll'              => '0.07',
	'IO::Seekable'          => '1.10',
	'IO::Select'            => '1.17',
	'IO::Socket'            => '1.32',
	'IO::Socket::INET'      => '1.31',
	'IO::Socket::UNIX'      => '1.23',
	'IO::Uncompress::Adapter::Bunzip2'=> '2.030',
	'IO::Uncompress::Adapter::Identity'=> '2.030',
	'IO::Uncompress::Adapter::Inflate'=> '2.030',
	'IO::Uncompress::AnyInflate'=> '2.030',
	'IO::Uncompress::AnyUncompress'=> '2.030',
	'IO::Uncompress::Base'  => '2.030',
	'IO::Uncompress::Bunzip2'=> '2.030',
	'IO::Uncompress::Gunzip'=> '2.030',
	'IO::Uncompress::Inflate'=> '2.030',
	'IO::Uncompress::RawInflate'=> '2.030',
	'IO::Uncompress::Unzip' => '2.030',
	'IO::Zlib'              => '1.10',
	'IPC::Cmd'              => '0.64',
	'IPC::Msg'              => '2.03',
	'IPC::Open2'            => '1.03',
	'IPC::Open3'            => '1.07',
	'IPC::Semaphore'        => '2.03',
	'IPC::SharedMem'        => '2.03',
	'IPC::SysV'             => '2.03',
	'List::Util'            => '1.23',
	'List::Util::PP'        => '1.23',
	'List::Util::XS'        => '1.23',
	'Locale::Codes'         => '3.14',
	'Locale::Codes::Country'=> '3.14',
	'Locale::Codes::Currency'=> '3.14',
	'Locale::Codes::Language'=> '3.14',
	'Locale::Codes::Script' => '3.14',
	'Locale::Constants'     => '3.14',
	'Locale::Country'       => '3.14',
	'Locale::Currency'      => '3.14',
	'Locale::Language'      => '3.14',
	'Locale::Maketext'      => '1.17',
	'Locale::Maketext::Guts'=> '1.17',
	'Locale::Maketext::GutsLoader'=> '1.17',
	'Locale::Maketext::Simple'=> '0.21',
	'Locale::Script'        => '3.14',
	'Log::Message'          => '0.02',
	'Log::Message::Config'  => '0.01',
	'Log::Message::Handlers'=> undef,
	'Log::Message::Item'    => undef,
	'Log::Message::Simple'  => '0.06',
	'MIME::Base64'          => '3.10',
	'MIME::QuotedPrint'     => '3.10',
	'Math::BigFloat'        => '1.99_01',
	'Math::BigFloat::Trace' => '0.01',
	'Math::BigInt'          => '1.99_01',
	'Math::BigInt::Calc'    => '1.99_01',
	'Math::BigInt::CalcEmu' => '1.99_01',
	'Math::BigInt::FastCalc'=> '0.24_01',
	'Math::BigInt::Trace'   => '0.01',
	'Math::BigRat'          => '0.26_01',
	'Math::Complex'         => '1.56',
	'Math::Trig'            => '1.2',
	'Memoize'               => '1.01_03',
	'Memoize::AnyDBM_File'  => '0.65',
	'Memoize::Expire'       => '1.00',
	'Memoize::ExpireFile'   => '1.01',
	'Memoize::ExpireTest'   => '0.65',
	'Memoize::NDBM_File'    => '0.65',
	'Memoize::SDBM_File'    => '0.65',
	'Memoize::Storable'     => '0.65',
	'Module::Build'         => '0.3607',
	'Module::Build::Base'   => '0.3607',
	'Module::Build::Compat' => '0.3607',
	'Module::Build::Config' => '0.3607',
	'Module::Build::ConfigData'=> undef,
	'Module::Build::Cookbook'=> '0.3607',
	'Module::Build::Dumper' => '0.3607',
	'Module::Build::ModuleInfo'=> '0.3607',
	'Module::Build::Notes'  => '0.3607',
	'Module::Build::PPMMaker'=> '0.3607',
	'Module::Build::Platform::Amiga'=> '0.3607',
	'Module::Build::Platform::Default'=> '0.3607',
	'Module::Build::Platform::EBCDIC'=> '0.3607',
	'Module::Build::Platform::MPEiX'=> '0.3607',
	'Module::Build::Platform::MacOS'=> '0.3607',
	'Module::Build::Platform::RiscOS'=> '0.3607',
	'Module::Build::Platform::Unix'=> '0.3607',
	'Module::Build::Platform::VMS'=> '0.3607',
	'Module::Build::Platform::VOS'=> '0.3607',
	'Module::Build::Platform::Windows'=> '0.3607',
	'Module::Build::Platform::aix'=> '0.3607',
	'Module::Build::Platform::cygwin'=> '0.3607',
	'Module::Build::Platform::darwin'=> '0.3607',
	'Module::Build::Platform::os2'=> '0.3607',
	'Module::Build::PodParser'=> '0.3607',
	'Module::Build::Version'=> '0.77',
	'Module::Build::YAML'   => '1.40',
	'Module::CoreList'      => '2.41',
	'Module::Load'          => '0.18',
	'Module::Load::Conditional'=> '0.38',
	'Module::Loaded'        => '0.06',
	'Module::Pluggable'     => '3.9',
	'Module::Pluggable::Object'=> '3.9',
	'Moped::Msg'            => '0.01',
	'NDBM_File'             => '1.10',
	'NEXT'                  => '0.65',
	'Net::Cmd'              => '2.29',
	'Net::Config'           => '1.11',
	'Net::Domain'           => '2.20',
	'Net::FTP'              => '2.77',
	'Net::FTP::A'           => '1.18',
	'Net::FTP::E'           => '0.01',
	'Net::FTP::I'           => '1.12',
	'Net::FTP::L'           => '0.01',
	'Net::FTP::dataconn'    => '0.11',
	'Net::NNTP'             => '2.24',
	'Net::Netrc'            => '2.12',
	'Net::POP3'             => '2.29',
	'Net::Ping'             => '2.36',
	'Net::SMTP'             => '2.31',
	'Net::Time'             => '2.10',
	'Net::hostent'          => '1.01',
	'Net::netent'           => '1.00',
	'Net::protoent'         => '1.00',
	'Net::servent'          => '1.01',
	'O'                     => '1.01',
	'ODBM_File'             => '1.09',
	'Object::Accessor'      => '0.36',
	'Opcode'                => '1.17',
	'POSIX'                 => '1.22',
	'Package::Constants'    => '0.02',
	'Params::Check'         => '0.26',
	'Parse::CPAN::Meta'     => '1.40',
	'PerlIO'                => '1.06',
	'PerlIO::encoding'      => '0.13',
	'PerlIO::scalar'        => '0.10',
	'PerlIO::via'           => '0.10',
	'PerlIO::via::QuotedPrint'=> '0.06',
	'Pod::Checker'          => '1.45',
	'Pod::Escapes'          => '1.04',
	'Pod::Find'             => '1.35',
	'Pod::Functions'        => '1.04',
	'Pod::Html'             => '1.09',
	'Pod::InputObjects'     => '1.31',
	'Pod::LaTeX'            => '0.58',
	'Pod::Man'              => '2.25',
	'Pod::ParseLink'        => '1.10',
	'Pod::ParseUtils'       => '1.36',
	'Pod::Parser'           => '1.37',
	'Pod::Perldoc'          => '3.15_02',
	'Pod::Perldoc::BaseTo'  => undef,
	'Pod::Perldoc::GetOptsOO'=> undef,
	'Pod::Perldoc::ToChecker'=> undef,
	'Pod::Perldoc::ToMan'   => undef,
	'Pod::Perldoc::ToNroff' => undef,
	'Pod::Perldoc::ToPod'   => undef,
	'Pod::Perldoc::ToRtf'   => undef,
	'Pod::Perldoc::ToText'  => undef,
	'Pod::Perldoc::ToTk'    => undef,
	'Pod::Perldoc::ToXml'   => undef,
	'Pod::PlainText'        => '2.04',
	'Pod::Select'           => '1.36',
	'Pod::Simple'           => '3.15',
	'Pod::Simple::BlackBox' => '3.15',
	'Pod::Simple::Checker'  => '3.15',
	'Pod::Simple::Debug'    => '3.15',
	'Pod::Simple::DumpAsText'=> '3.15',
	'Pod::Simple::DumpAsXML'=> '3.15',
	'Pod::Simple::HTML'     => '3.15',
	'Pod::Simple::HTMLBatch'=> '3.15',
	'Pod::Simple::HTMLLegacy'=> '5.01',
	'Pod::Simple::LinkSection'=> '3.15',
	'Pod::Simple::Methody'  => '3.15',
	'Pod::Simple::Progress' => '3.15',
	'Pod::Simple::PullParser'=> '3.15',
	'Pod::Simple::PullParserEndToken'=> '3.15',
	'Pod::Simple::PullParserStartToken'=> '3.15',
	'Pod::Simple::PullParserTextToken'=> '3.15',
	'Pod::Simple::PullParserToken'=> '3.15',
	'Pod::Simple::RTF'      => '3.15',
	'Pod::Simple::Search'   => '3.15',
	'Pod::Simple::SimpleTree'=> '3.15',
	'Pod::Simple::Text'     => '3.15',
	'Pod::Simple::TextContent'=> '3.15',
	'Pod::Simple::TiedOutFH'=> '3.15',
	'Pod::Simple::Transcode'=> '3.15',
	'Pod::Simple::TranscodeDumb'=> '3.15',
	'Pod::Simple::TranscodeSmart'=> '3.15',
	'Pod::Simple::XHTML'    => '3.15',
	'Pod::Simple::XMLOutStream'=> '3.15',
	'Pod::Text'             => '3.15',
	'Pod::Text::Color'      => '2.06',
	'Pod::Text::Overstrike' => '2.04',
	'Pod::Text::Termcap'    => '2.06',
	'Pod::Usage'            => '1.36',
	'SDBM_File'             => '1.08',
	'Safe'                  => '2.29',
	'Scalar::Util'          => '1.23',
	'Scalar::Util::PP'      => '1.23',
	'Search::Dict'          => '1.03',
	'SelectSaver'           => '1.02',
	'SelfLoader'            => '1.18',
	'Shell'                 => '0.72_01',
	'Socket'                => '1.91',
	'Storable'              => '2.24',
	'Symbol'                => '1.07',
	'Sys::Hostname'         => '1.14',
	'Sys::Syslog'           => '0.27',
	'Sys::Syslog::win32::Win32'=> undef,
	'TAP::Base'             => '3.22',
	'TAP::Formatter::Base'  => '3.22',
	'TAP::Formatter::Color' => '3.22',
	'TAP::Formatter::Console'=> '3.22',
	'TAP::Formatter::Console::ParallelSession'=> '3.22',
	'TAP::Formatter::Console::Session'=> '3.22',
	'TAP::Formatter::File'  => '3.22',
	'TAP::Formatter::File::Session'=> '3.22',
	'TAP::Formatter::Session'=> '3.22',
	'TAP::Harness'          => '3.22',
	'TAP::Object'           => '3.22',
	'TAP::Parser'           => '3.22',
	'TAP::Parser::Aggregator'=> '3.22',
	'TAP::Parser::Grammar'  => '3.22',
	'TAP::Parser::Iterator' => '3.22',
	'TAP::Parser::Iterator::Array'=> '3.22',
	'TAP::Parser::Iterator::Process'=> '3.22',
	'TAP::Parser::Iterator::Stream'=> '3.22',
	'TAP::Parser::IteratorFactory'=> '3.22',
	'TAP::Parser::Multiplexer'=> '3.22',
	'TAP::Parser::Result'   => '3.22',
	'TAP::Parser::Result::Bailout'=> '3.22',
	'TAP::Parser::Result::Comment'=> '3.22',
	'TAP::Parser::Result::Plan'=> '3.22',
	'TAP::Parser::Result::Pragma'=> '3.22',
	'TAP::Parser::Result::Test'=> '3.22',
	'TAP::Parser::Result::Unknown'=> '3.22',
	'TAP::Parser::Result::Version'=> '3.22',
	'TAP::Parser::Result::YAML'=> '3.22',
	'TAP::Parser::ResultFactory'=> '3.22',
	'TAP::Parser::Scheduler'=> '3.22',
	'TAP::Parser::Scheduler::Job'=> '3.22',
	'TAP::Parser::Scheduler::Spinner'=> '3.22',
	'TAP::Parser::Source'   => '3.22',
	'TAP::Parser::SourceHandler'=> '3.22',
	'TAP::Parser::SourceHandler::Executable'=> '3.22',
	'TAP::Parser::SourceHandler::File'=> '3.22',
	'TAP::Parser::SourceHandler::Handle'=> '3.22',
	'TAP::Parser::SourceHandler::Perl'=> '3.22',
	'TAP::Parser::SourceHandler::RawTAP'=> '3.22',
	'TAP::Parser::Utils'    => '3.22',
	'TAP::Parser::YAMLish::Reader'=> '3.22',
	'TAP::Parser::YAMLish::Writer'=> '3.22',
	'Term::ANSIColor'       => '3.00',
	'Term::Cap'             => '1.12',
	'Term::Complete'        => '1.402',
	'Term::ReadLine'        => '1.07',
	'Term::UI'              => '0.20',
	'Term::UI::History'     => undef,
	'Test'                  => '1.25_02',
	'Test::Builder'         => '0.97_01',
	'Test::Builder::Module' => '0.97_01',
	'Test::Builder::Tester' => '1.21_01',
	'Test::Builder::Tester::Color'=> '1.21_01',
	'Test::Harness'         => '3.22',
	'Test::More'            => '0.97_01',
	'Test::Simple'          => '0.97_01',
	'Text::Abbrev'          => '1.01',
	'Text::Balanced'        => '2.02',
	'Text::ParseWords'      => '3.27',
	'Text::Soundex'         => '3.03_01',
	'Text::Tabs'            => '2009.0305',
	'Text::Wrap'            => '2009.0305',
	'Thread'                => '3.02',
	'Thread::Queue'         => '2.11',
	'Thread::Semaphore'     => '2.11',
	'Tie::Array'            => '1.04',
	'Tie::File'             => '0.97_02',
	'Tie::Handle'           => '4.2',
	'Tie::Hash'             => '1.04',
	'Tie::Hash::NamedCapture'=> '0.07',
	'Tie::Memoize'          => '1.1',
	'Tie::RefHash'          => '1.38',
	'Tie::Scalar'           => '1.02',
	'Tie::StdHandle'        => '4.2',
	'Tie::SubstrHash'       => '1.00',
	'Time::HiRes'           => '1.9721',
	'Time::Local'           => '1.1901_01',
	'Time::Piece'           => '1.20_01',
	'Time::Piece::Seconds'  => undef,
	'Time::Seconds'         => undef,
	'Time::gmtime'          => '1.03',
	'Time::localtime'       => '1.02',
	'Time::tm'              => '1.00',
	'UNIVERSAL'             => '1.07',
	'Unicode'               => '6.0.0',
	'Unicode::Collate'      => '0.67',
	'Unicode::Collate::CJK::Big5'=> '0.65',
	'Unicode::Collate::CJK::GB2312'=> '0.65',
	'Unicode::Collate::CJK::JISX0208'=> '0.64',
	'Unicode::Collate::CJK::Korean'=> '0.66',
	'Unicode::Collate::CJK::Pinyin'=> '0.65',
	'Unicode::Collate::CJK::Stroke'=> '0.65',
	'Unicode::Collate::Locale'=> '0.67',
	'Unicode::Normalize'    => '1.07',
	'Unicode::UCD'          => '0.29',
	'User::grent'           => '1.01',
	'User::pwent'           => '1.00',
	'VMS::DCLsym'           => '1.04',
	'VMS::Filespec'         => '1.12',
	'VMS::Stdio'            => '2.4',
	'Win32'                 => '0.39',
	'Win32API::File'        => '0.1101',
	'Win32API::File::ExtUtils::Myconst2perl'=> '1',
	'Win32CORE'             => '0.02',
	'XS::APItest'           => '0.26',
	'XS::Typemap'           => '0.04',
	'XSLoader'              => '0.13',
	'XSLoader::XSLoader'    => '0.13',
	'attributes'            => '0.13',
	'autodie'               => '2.10',
	'autodie::exception'    => '2.10',
	'autodie::exception::system'=> '2.10',
	'autodie::hints'        => '2.10',
	'autouse'               => '1.06',
	'base'                  => '2.15',
	'bigint'                => '0.25',
	'bignum'                => '0.25',
	'bigrat'                => '0.25',
	'blib'                  => '1.06',
	'bytes'                 => '1.04',
	'charnames'             => '1.17',
	'constant'              => '1.20',
	'deprecate'             => '0.01',
	'diagnostics'           => '1.20',
	'encoding'              => '2.6_01',
	'encoding::warnings'    => '0.11',
	'feature'               => '1.18',
	'fields'                => '2.15',
	'filetest'              => '1.02',
	'if'                    => '0.06',
	'inc::latest'           => '0.3607',
	'integer'               => '1.00',
	'less'                  => '0.03',
	'lib'                   => '0.63',
	'locale'                => '1.00',
	'mro'                   => '1.05',
	'open'                  => '1.08',
	'ops'                   => '1.02',
	'overload'              => '1.11',
	'overload::numbers'     => undef,
	'overloading'           => '0.01',
	'parent'                => '0.224',
	're'                    => '0.14',
	'sigtrap'               => '1.05',
	'sort'                  => '2.01',
	'strict'                => '1.04',
	'subs'                  => '1.00',
	'threads'               => '1.81_02',
	'threads::shared'       => '1.34',
	'utf8'                  => '1.08',
	'vars'                  => '1.02',
	'version'               => '0.82',
	'vmsish'                => '1.02',
	'warnings'              => '1.11',
	'warnings::register'    => '1.02',
    },
    5.013008 => {
	'AnyDBM_File'           => '1.00',
	'App::Cpan'             => '1.5701',
	'App::Prove'            => '3.22',
	'App::Prove::State'     => '3.22',
	'App::Prove::State::Result'=> '3.22',
	'App::Prove::State::Result::Test'=> '3.22',
	'Archive::Extract'      => '0.46',
	'Archive::Tar'          => '1.74',
	'Archive::Tar::Constant'=> '1.74',
	'Archive::Tar::File'    => '1.74',
	'Attribute::Handlers'   => '0.88',
	'AutoLoader'            => '5.71',
	'AutoSplit'             => '1.06',
	'B'                     => '1.27',
	'B::Concise'            => '0.82',
	'B::Debug'              => '1.16',
	'B::Deparse'            => '1.02',
	'B::Lint'               => '1.12',
	'B::Lint::Debug'        => '1.12',
	'B::Showlex'            => '1.02',
	'B::Terse'              => '1.05',
	'B::Xref'               => '1.02',
	'Benchmark'             => '1.11',
	'CGI'                   => '3.50',
	'CGI::Apache'           => '1.01',
	'CGI::Carp'             => '3.45',
	'CGI::Cookie'           => '1.29',
	'CGI::Fast'             => '1.08',
	'CGI::Pretty'           => '3.46',
	'CGI::Push'             => '1.04',
	'CGI::Switch'           => '1.01',
	'CGI::Util'             => '3.48',
	'CPAN'                  => '1.94_62',
	'CPAN::Author'          => '5.5',
	'CPAN::Bundle'          => '5.5',
	'CPAN::CacheMgr'        => '5.5',
	'CPAN::Complete'        => '5.5',
	'CPAN::Debug'           => '5.5001',
	'CPAN::DeferredCode'    => '5.50',
	'CPAN::Distribution'    => '1.9600',
	'CPAN::Distroprefs'     => '6',
	'CPAN::Distrostatus'    => '5.5',
	'CPAN::Exception::RecursiveDependency'=> '5.5',
	'CPAN::Exception::blocked_urllist'=> '1.0',
	'CPAN::Exception::yaml_not_installed'=> '5.5',
	'CPAN::FTP'             => '5.5005',
	'CPAN::FTP::netrc'      => '1.00',
	'CPAN::FirstTime'       => '5.5301',
	'CPAN::HandleConfig'    => '5.5001',
	'CPAN::Index'           => '1.94',
	'CPAN::InfoObj'         => '5.5',
	'CPAN::Kwalify'         => '5.50',
	'CPAN::LWP::UserAgent'  => '1.94',
	'CPAN::Mirrors'         => '1.77',
	'CPAN::Module'          => '5.5',
	'CPAN::Nox'             => '5.50',
	'CPAN::Prompt'          => '5.5',
	'CPAN::Queue'           => '5.5001',
	'CPAN::Shell'           => '5.5001',
	'CPAN::Tarzip'          => '5.5011',
	'CPAN::URL'             => '5.5',
	'CPAN::Version'         => '5.5001',
	'CPANPLUS'              => '0.9010',
	'CPANPLUS::Backend'     => undef,
	'CPANPLUS::Backend::RV' => undef,
	'CPANPLUS::Config'      => undef,
	'CPANPLUS::Configure'   => undef,
	'CPANPLUS::Configure::Setup'=> undef,
	'CPANPLUS::Dist'        => undef,
	'CPANPLUS::Dist::Autobundle'=> undef,
	'CPANPLUS::Dist::Base'  => undef,
	'CPANPLUS::Dist::Build' => '0.50',
	'CPANPLUS::Dist::Build::Constants'=> '0.50',
	'CPANPLUS::Dist::MM'    => undef,
	'CPANPLUS::Dist::Sample'=> undef,
	'CPANPLUS::Error'       => undef,
	'CPANPLUS::Internals'   => '0.9010',
	'CPANPLUS::Internals::Constants'=> undef,
	'CPANPLUS::Internals::Constants::Report'=> undef,
	'CPANPLUS::Internals::Extract'=> undef,
	'CPANPLUS::Internals::Fetch'=> undef,
	'CPANPLUS::Internals::Report'=> undef,
	'CPANPLUS::Internals::Search'=> undef,
	'CPANPLUS::Internals::Source'=> undef,
	'CPANPLUS::Internals::Source::Memory'=> undef,
	'CPANPLUS::Internals::Source::SQLite'=> undef,
	'CPANPLUS::Internals::Source::SQLite::Tie'=> undef,
	'CPANPLUS::Internals::Utils'=> undef,
	'CPANPLUS::Internals::Utils::Autoflush'=> undef,
	'CPANPLUS::Module'      => undef,
	'CPANPLUS::Module::Author'=> undef,
	'CPANPLUS::Module::Author::Fake'=> undef,
	'CPANPLUS::Module::Checksums'=> undef,
	'CPANPLUS::Module::Fake'=> undef,
	'CPANPLUS::Module::Signature'=> undef,
	'CPANPLUS::Selfupdate'  => undef,
	'CPANPLUS::Shell'       => undef,
	'CPANPLUS::Shell::Classic'=> '0.0562',
	'CPANPLUS::Shell::Default'=> '0.9010',
	'CPANPLUS::Shell::Default::Plugins::CustomSource'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Remote'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Source'=> undef,
	'Carp'                  => '1.19',
	'Carp::Heavy'           => '1.17',
	'Class::Struct'         => '0.63',
	'Compress::Raw::Bzip2'  => '2.031',
	'Compress::Raw::Zlib'   => '2.030',
	'Compress::Zlib'        => '2.030',
	'Config'                => undef,
	'Config::Extensions'    => '0.01',
	'Cwd'                   => '3.35',
	'DB'                    => '1.03',
	'DBM_Filter'            => '0.03',
	'DBM_Filter::compress'  => '0.02',
	'DBM_Filter::encode'    => '0.02',
	'DBM_Filter::int32'     => '0.02',
	'DBM_Filter::null'      => '0.02',
	'DBM_Filter::utf8'      => '0.02',
	'DB_File'               => '1.820',
	'Data::Dumper'          => '2.130_02',
	'Devel::DProf'          => '20080331.00',
	'Devel::DProf::dprof::V'=> undef,
	'Devel::InnerPackage'   => '0.3',
	'Devel::PPPort'         => '3.19',
	'Devel::Peek'           => '1.06',
	'Devel::SelfStubber'    => '1.05',
	'Digest'                => '1.16',
	'Digest::MD5'           => '2.51',
	'Digest::SHA'           => '5.50',
	'Digest::base'          => '1.16',
	'Digest::file'          => '1.16',
	'DirHandle'             => '1.03',
	'Dumpvalue'             => '1.15',
	'DynaLoader'            => '1.12',
	'Encode'                => '2.40',
	'Encode::Alias'         => '2.12',
	'Encode::Byte'          => '2.04',
	'Encode::CJKConstants'  => '2.02',
	'Encode::CN'            => '2.03',
	'Encode::CN::HZ'        => '2.05',
	'Encode::Config'        => '2.05',
	'Encode::EBCDIC'        => '2.02',
	'Encode::Encoder'       => '2.01',
	'Encode::Encoding'      => '2.05',
	'Encode::GSM0338'       => '2.01',
	'Encode::Guess'         => '2.04',
	'Encode::JP'            => '2.04',
	'Encode::JP::H2Z'       => '2.02',
	'Encode::JP::JIS7'      => '2.04',
	'Encode::KR'            => '2.03',
	'Encode::KR::2022_KR'   => '2.02',
	'Encode::MIME::Header'  => '2.12',
	'Encode::MIME::Header::ISO_2022_JP'=> '1.03',
	'Encode::MIME::Name'    => '1.01',
	'Encode::Symbol'        => '2.02',
	'Encode::TW'            => '2.03',
	'Encode::Unicode'       => '2.07',
	'Encode::Unicode::UTF7' => '2.05',
	'English'               => '1.04',
	'Env'                   => '1.02',
	'Errno'                 => '1.13',
	'Exporter'              => '5.64_03',
	'Exporter::Heavy'       => '5.64_01',
	'ExtUtils::CBuilder'    => '0.280201',
	'ExtUtils::CBuilder::Base'=> '0.280201',
	'ExtUtils::CBuilder::Platform::Unix'=> '0.280201',
	'ExtUtils::CBuilder::Platform::VMS'=> '0.280201',
	'ExtUtils::CBuilder::Platform::Windows'=> '0.280201',
	'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280201',
	'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.280201',
	'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.280201',
	'ExtUtils::CBuilder::Platform::aix'=> '0.280201',
	'ExtUtils::CBuilder::Platform::cygwin'=> '0.280201',
	'ExtUtils::CBuilder::Platform::darwin'=> '0.280201',
	'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280201',
	'ExtUtils::CBuilder::Platform::os2'=> '0.280201',
	'ExtUtils::Command'     => '1.16',
	'ExtUtils::Command::MM' => '6.57_05',
	'ExtUtils::Constant'    => '0.23',
	'ExtUtils::Constant::Base'=> '0.05',
	'ExtUtils::Constant::ProxySubs'=> '0.08',
	'ExtUtils::Constant::Utils'=> '0.03',
	'ExtUtils::Constant::XS'=> '0.03',
	'ExtUtils::Embed'       => '1.30',
	'ExtUtils::Install'     => '1.55',
	'ExtUtils::Installed'   => '1.999_001',
	'ExtUtils::Liblist'     => '6.57_05',
	'ExtUtils::Liblist::Kid'=> '6.5705',
	'ExtUtils::MM'          => '6.57_05',
	'ExtUtils::MM_AIX'      => '6.57_05',
	'ExtUtils::MM_Any'      => '6.57_05',
	'ExtUtils::MM_BeOS'     => '6.57_05',
	'ExtUtils::MM_Cygwin'   => '6.57_05',
	'ExtUtils::MM_DOS'      => '6.5705',
	'ExtUtils::MM_Darwin'   => '6.57_05',
	'ExtUtils::MM_MacOS'    => '6.5705',
	'ExtUtils::MM_NW5'      => '6.57_05',
	'ExtUtils::MM_OS2'      => '6.57_05',
	'ExtUtils::MM_QNX'      => '6.57_05',
	'ExtUtils::MM_UWIN'     => '6.5705',
	'ExtUtils::MM_Unix'     => '6.57_05',
	'ExtUtils::MM_VMS'      => '6.57_05',
	'ExtUtils::MM_VOS'      => '6.57_05',
	'ExtUtils::MM_Win32'    => '6.57_05',
	'ExtUtils::MM_Win95'    => '6.57_05',
	'ExtUtils::MY'          => '6.5705',
	'ExtUtils::MakeMaker'   => '6.57_05',
	'ExtUtils::MakeMaker::Config'=> '6.57_05',
	'ExtUtils::MakeMaker::YAML'=> '1.44',
	'ExtUtils::Manifest'    => '1.58',
	'ExtUtils::Miniperl'    => undef,
	'ExtUtils::Mkbootstrap' => '6.57_05',
	'ExtUtils::Mksymlists'  => '6.57_05',
	'ExtUtils::Packlist'    => '1.44',
	'ExtUtils::ParseXS'     => '2.2208',
	'ExtUtils::XSSymSet'    => '1.2',
	'ExtUtils::testlib'     => '6.5705',
	'Fatal'                 => '2.1001',
	'Fcntl'                 => '1.11',
	'File::Basename'        => '2.79',
	'File::CheckTree'       => '4.41',
	'File::Compare'         => '1.1006',
	'File::Copy'            => '2.20',
	'File::DosGlob'         => '1.03',
	'File::Fetch'           => '0.28',
	'File::Find'            => '1.18',
	'File::Glob'            => '1.11',
	'File::GlobMapper'      => '1.000',
	'File::Path'            => '2.08_01',
	'File::Spec'            => '3.33',
	'File::Spec::Cygwin'    => '3.33',
	'File::Spec::Epoc'      => '3.33',
	'File::Spec::Functions' => '3.33',
	'File::Spec::Mac'       => '3.33',
	'File::Spec::OS2'       => '3.33',
	'File::Spec::Unix'      => '3.33',
	'File::Spec::VMS'       => '3.33',
	'File::Spec::Win32'     => '3.33',
	'File::Temp'            => '0.22',
	'File::stat'            => '1.04',
	'FileCache'             => '1.08',
	'FileHandle'            => '2.02',
	'Filter::Simple'        => '0.85',
	'Filter::Util::Call'    => '1.08',
	'FindBin'               => '1.50',
	'GDBM_File'             => '1.13',
	'Getopt::Long'          => '2.38',
	'Getopt::Std'           => '1.06',
	'Hash::Util'            => '0.10',
	'Hash::Util::FieldHash' => '1.07',
	'I18N::Collate'         => '1.02',
	'I18N::LangTags'        => '0.35_01',
	'I18N::LangTags::Detect'=> '1.04',
	'I18N::LangTags::List'  => '0.35_01',
	'I18N::Langinfo'        => '0.07',
	'IO'                    => '1.25_03',
	'IO::Compress::Adapter::Bzip2'=> '2.030',
	'IO::Compress::Adapter::Deflate'=> '2.030',
	'IO::Compress::Adapter::Identity'=> '2.030',
	'IO::Compress::Base'    => '2.030',
	'IO::Compress::Base::Common'=> '2.030',
	'IO::Compress::Bzip2'   => '2.030',
	'IO::Compress::Deflate' => '2.030',
	'IO::Compress::Gzip'    => '2.030',
	'IO::Compress::Gzip::Constants'=> '2.030',
	'IO::Compress::RawDeflate'=> '2.030',
	'IO::Compress::Zip'     => '2.030',
	'IO::Compress::Zip::Constants'=> '2.030',
	'IO::Compress::Zlib::Constants'=> '2.030',
	'IO::Compress::Zlib::Extra'=> '2.030',
	'IO::Dir'               => '1.07',
	'IO::File'              => '1.14',
	'IO::Handle'            => '1.28',
	'IO::Pipe'              => '1.13',
	'IO::Poll'              => '0.07',
	'IO::Seekable'          => '1.10',
	'IO::Select'            => '1.17',
	'IO::Socket'            => '1.32',
	'IO::Socket::INET'      => '1.31',
	'IO::Socket::UNIX'      => '1.23',
	'IO::Uncompress::Adapter::Bunzip2'=> '2.030',
	'IO::Uncompress::Adapter::Identity'=> '2.030',
	'IO::Uncompress::Adapter::Inflate'=> '2.030',
	'IO::Uncompress::AnyInflate'=> '2.030',
	'IO::Uncompress::AnyUncompress'=> '2.030',
	'IO::Uncompress::Base'  => '2.030',
	'IO::Uncompress::Bunzip2'=> '2.030',
	'IO::Uncompress::Gunzip'=> '2.030',
	'IO::Uncompress::Inflate'=> '2.030',
	'IO::Uncompress::RawInflate'=> '2.030',
	'IO::Uncompress::Unzip' => '2.030',
	'IO::Zlib'              => '1.10',
	'IPC::Cmd'              => '0.66',
	'IPC::Msg'              => '2.03',
	'IPC::Open2'            => '1.03',
	'IPC::Open3'            => '1.08',
	'IPC::Semaphore'        => '2.03',
	'IPC::SharedMem'        => '2.03',
	'IPC::SysV'             => '2.03',
	'List::Util'            => '1.23',
	'List::Util::PP'        => '1.23',
	'List::Util::XS'        => '1.23',
	'Locale::Codes'         => '3.15',
	'Locale::Codes::Country'=> '3.15',
	'Locale::Codes::Currency'=> '3.15',
	'Locale::Codes::Language'=> '3.15',
	'Locale::Codes::Script' => '3.15',
	'Locale::Constants'     => '3.15',
	'Locale::Country'       => '3.15',
	'Locale::Currency'      => '3.15',
	'Locale::Language'      => '3.15',
	'Locale::Maketext'      => '1.17',
	'Locale::Maketext::Guts'=> '1.17',
	'Locale::Maketext::GutsLoader'=> '1.17',
	'Locale::Maketext::Simple'=> '0.21',
	'Locale::Script'        => '3.15',
	'Log::Message'          => '0.02',
	'Log::Message::Config'  => '0.01',
	'Log::Message::Handlers'=> undef,
	'Log::Message::Item'    => undef,
	'Log::Message::Simple'  => '0.06',
	'MIME::Base64'          => '3.13',
	'MIME::QuotedPrint'     => '3.13',
	'Math::BigFloat'        => '1.99_02',
	'Math::BigFloat::Trace' => '0.01',
	'Math::BigInt'          => '1.99_02',
	'Math::BigInt::Calc'    => '1.99_02',
	'Math::BigInt::CalcEmu' => '1.99_02',
	'Math::BigInt::FastCalc'=> '0.24_01',
	'Math::BigInt::Trace'   => '0.01',
	'Math::BigRat'          => '0.26_01',
	'Math::Complex'         => '1.56',
	'Math::Trig'            => '1.2',
	'Memoize'               => '1.02',
	'Memoize::AnyDBM_File'  => '1.02',
	'Memoize::Expire'       => '1.02',
	'Memoize::ExpireFile'   => '1.02',
	'Memoize::ExpireTest'   => '1.02',
	'Memoize::NDBM_File'    => '1.02',
	'Memoize::SDBM_File'    => '1.02',
	'Memoize::Storable'     => '1.02',
	'Module::Build'         => '0.3607',
	'Module::Build::Base'   => '0.3607',
	'Module::Build::Compat' => '0.3607',
	'Module::Build::Config' => '0.3607',
	'Module::Build::ConfigData'=> undef,
	'Module::Build::Cookbook'=> '0.3607',
	'Module::Build::Dumper' => '0.3607',
	'Module::Build::ModuleInfo'=> '0.3607',
	'Module::Build::Notes'  => '0.3607',
	'Module::Build::PPMMaker'=> '0.3607',
	'Module::Build::Platform::Amiga'=> '0.3607',
	'Module::Build::Platform::Default'=> '0.3607',
	'Module::Build::Platform::EBCDIC'=> '0.3607',
	'Module::Build::Platform::MPEiX'=> '0.3607',
	'Module::Build::Platform::MacOS'=> '0.3607',
	'Module::Build::Platform::RiscOS'=> '0.3607',
	'Module::Build::Platform::Unix'=> '0.3607',
	'Module::Build::Platform::VMS'=> '0.3607',
	'Module::Build::Platform::VOS'=> '0.3607',
	'Module::Build::Platform::Windows'=> '0.3607',
	'Module::Build::Platform::aix'=> '0.3607',
	'Module::Build::Platform::cygwin'=> '0.3607',
	'Module::Build::Platform::darwin'=> '0.3607',
	'Module::Build::Platform::os2'=> '0.3607',
	'Module::Build::PodParser'=> '0.3607',
	'Module::Build::Version'=> '0.77',
	'Module::Build::YAML'   => '1.40',
	'Module::CoreList'      => '2.43',
	'Module::Load'          => '0.18',
	'Module::Load::Conditional'=> '0.38',
	'Module::Loaded'        => '0.06',
	'Module::Pluggable'     => '3.9',
	'Module::Pluggable::Object'=> '3.9',
	'Moped::Msg'            => '0.01',
	'NDBM_File'             => '1.11',
	'NEXT'                  => '0.65',
	'Net::Cmd'              => '2.29',
	'Net::Config'           => '1.11',
	'Net::Domain'           => '2.20',
	'Net::FTP'              => '2.77',
	'Net::FTP::A'           => '1.18',
	'Net::FTP::E'           => '0.01',
	'Net::FTP::I'           => '1.12',
	'Net::FTP::L'           => '0.01',
	'Net::FTP::dataconn'    => '0.11',
	'Net::NNTP'             => '2.24',
	'Net::Netrc'            => '2.12',
	'Net::POP3'             => '2.29',
	'Net::Ping'             => '2.37',
	'Net::SMTP'             => '2.31',
	'Net::Time'             => '2.10',
	'Net::hostent'          => '1.01',
	'Net::netent'           => '1.00',
	'Net::protoent'         => '1.00',
	'Net::servent'          => '1.01',
	'O'                     => '1.01',
	'ODBM_File'             => '1.10',
	'Object::Accessor'      => '0.36',
	'Opcode'                => '1.18',
	'POSIX'                 => '1.23',
	'Package::Constants'    => '0.02',
	'Params::Check'         => '0.26',
	'Parse::CPAN::Meta'     => '1.40',
	'PerlIO'                => '1.06',
	'PerlIO::encoding'      => '0.14',
	'PerlIO::scalar'        => '0.11',
	'PerlIO::via'           => '0.11',
	'PerlIO::via::QuotedPrint'=> '0.06',
	'Pod::Checker'          => '1.45',
	'Pod::Escapes'          => '1.04',
	'Pod::Find'             => '1.35',
	'Pod::Functions'        => '1.04',
	'Pod::Html'             => '1.09',
	'Pod::InputObjects'     => '1.31',
	'Pod::LaTeX'            => '0.58',
	'Pod::Man'              => '2.25',
	'Pod::ParseLink'        => '1.10',
	'Pod::ParseUtils'       => '1.36',
	'Pod::Parser'           => '1.37',
	'Pod::Perldoc'          => '3.15_02',
	'Pod::Perldoc::BaseTo'  => undef,
	'Pod::Perldoc::GetOptsOO'=> undef,
	'Pod::Perldoc::ToChecker'=> undef,
	'Pod::Perldoc::ToMan'   => undef,
	'Pod::Perldoc::ToNroff' => undef,
	'Pod::Perldoc::ToPod'   => undef,
	'Pod::Perldoc::ToRtf'   => undef,
	'Pod::Perldoc::ToText'  => undef,
	'Pod::Perldoc::ToTk'    => undef,
	'Pod::Perldoc::ToXml'   => undef,
	'Pod::PlainText'        => '2.04',
	'Pod::Select'           => '1.36',
	'Pod::Simple'           => '3.15',
	'Pod::Simple::BlackBox' => '3.15',
	'Pod::Simple::Checker'  => '3.15',
	'Pod::Simple::Debug'    => '3.15',
	'Pod::Simple::DumpAsText'=> '3.15',
	'Pod::Simple::DumpAsXML'=> '3.15',
	'Pod::Simple::HTML'     => '3.15',
	'Pod::Simple::HTMLBatch'=> '3.15',
	'Pod::Simple::HTMLLegacy'=> '5.01',
	'Pod::Simple::LinkSection'=> '3.15',
	'Pod::Simple::Methody'  => '3.15',
	'Pod::Simple::Progress' => '3.15',
	'Pod::Simple::PullParser'=> '3.15',
	'Pod::Simple::PullParserEndToken'=> '3.15',
	'Pod::Simple::PullParserStartToken'=> '3.15',
	'Pod::Simple::PullParserTextToken'=> '3.15',
	'Pod::Simple::PullParserToken'=> '3.15',
	'Pod::Simple::RTF'      => '3.15',
	'Pod::Simple::Search'   => '3.15',
	'Pod::Simple::SimpleTree'=> '3.15',
	'Pod::Simple::Text'     => '3.15',
	'Pod::Simple::TextContent'=> '3.15',
	'Pod::Simple::TiedOutFH'=> '3.15',
	'Pod::Simple::Transcode'=> '3.15',
	'Pod::Simple::TranscodeDumb'=> '3.15',
	'Pod::Simple::TranscodeSmart'=> '3.15',
	'Pod::Simple::XHTML'    => '3.15',
	'Pod::Simple::XMLOutStream'=> '3.15',
	'Pod::Text'             => '3.15',
	'Pod::Text::Color'      => '2.06',
	'Pod::Text::Overstrike' => '2.04',
	'Pod::Text::Termcap'    => '2.06',
	'Pod::Usage'            => '1.36',
	'SDBM_File'             => '1.09',
	'Safe'                  => '2.29',
	'Scalar::Util'          => '1.23',
	'Scalar::Util::PP'      => '1.23',
	'Search::Dict'          => '1.03',
	'SelectSaver'           => '1.02',
	'SelfLoader'            => '1.18',
	'Shell'                 => '0.72_01',
	'Socket'                => '1.92',
	'Storable'              => '2.25',
	'Symbol'                => '1.07',
	'Sys::Hostname'         => '1.14',
	'Sys::Syslog'           => '0.27',
	'Sys::Syslog::win32::Win32'=> undef,
	'TAP::Base'             => '3.22',
	'TAP::Formatter::Base'  => '3.22',
	'TAP::Formatter::Color' => '3.22',
	'TAP::Formatter::Console'=> '3.22',
	'TAP::Formatter::Console::ParallelSession'=> '3.22',
	'TAP::Formatter::Console::Session'=> '3.22',
	'TAP::Formatter::File'  => '3.22',
	'TAP::Formatter::File::Session'=> '3.22',
	'TAP::Formatter::Session'=> '3.22',
	'TAP::Harness'          => '3.22',
	'TAP::Object'           => '3.22',
	'TAP::Parser'           => '3.22',
	'TAP::Parser::Aggregator'=> '3.22',
	'TAP::Parser::Grammar'  => '3.22',
	'TAP::Parser::Iterator' => '3.22',
	'TAP::Parser::Iterator::Array'=> '3.22',
	'TAP::Parser::Iterator::Process'=> '3.22',
	'TAP::Parser::Iterator::Stream'=> '3.22',
	'TAP::Parser::IteratorFactory'=> '3.22',
	'TAP::Parser::Multiplexer'=> '3.22',
	'TAP::Parser::Result'   => '3.22',
	'TAP::Parser::Result::Bailout'=> '3.22',
	'TAP::Parser::Result::Comment'=> '3.22',
	'TAP::Parser::Result::Plan'=> '3.22',
	'TAP::Parser::Result::Pragma'=> '3.22',
	'TAP::Parser::Result::Test'=> '3.22',
	'TAP::Parser::Result::Unknown'=> '3.22',
	'TAP::Parser::Result::Version'=> '3.22',
	'TAP::Parser::Result::YAML'=> '3.22',
	'TAP::Parser::ResultFactory'=> '3.22',
	'TAP::Parser::Scheduler'=> '3.22',
	'TAP::Parser::Scheduler::Job'=> '3.22',
	'TAP::Parser::Scheduler::Spinner'=> '3.22',
	'TAP::Parser::Source'   => '3.22',
	'TAP::Parser::SourceHandler'=> '3.22',
	'TAP::Parser::SourceHandler::Executable'=> '3.22',
	'TAP::Parser::SourceHandler::File'=> '3.22',
	'TAP::Parser::SourceHandler::Handle'=> '3.22',
	'TAP::Parser::SourceHandler::Perl'=> '3.22',
	'TAP::Parser::SourceHandler::RawTAP'=> '3.22',
	'TAP::Parser::Utils'    => '3.22',
	'TAP::Parser::YAMLish::Reader'=> '3.22',
	'TAP::Parser::YAMLish::Writer'=> '3.22',
	'Term::ANSIColor'       => '3.00',
	'Term::Cap'             => '1.12',
	'Term::Complete'        => '1.402',
	'Term::ReadLine'        => '1.07',
	'Term::UI'              => '0.20',
	'Term::UI::History'     => undef,
	'Test'                  => '1.25_02',
	'Test::Builder'         => '0.97_01',
	'Test::Builder::Module' => '0.97_01',
	'Test::Builder::Tester' => '1.21_01',
	'Test::Builder::Tester::Color'=> '1.21_01',
	'Test::Harness'         => '3.22',
	'Test::More'            => '0.97_01',
	'Test::Simple'          => '0.97_01',
	'Text::Abbrev'          => '1.01',
	'Text::Balanced'        => '2.02',
	'Text::ParseWords'      => '3.27',
	'Text::Soundex'         => '3.03_01',
	'Text::Tabs'            => '2009.0305',
	'Text::Wrap'            => '2009.0305',
	'Thread'                => '3.02',
	'Thread::Queue'         => '2.11',
	'Thread::Semaphore'     => '2.11',
	'Tie::Array'            => '1.04',
	'Tie::File'             => '0.97_02',
	'Tie::Handle'           => '4.2',
	'Tie::Hash'             => '1.04',
	'Tie::Hash::NamedCapture'=> '0.07',
	'Tie::Memoize'          => '1.1',
	'Tie::RefHash'          => '1.38',
	'Tie::Scalar'           => '1.02',
	'Tie::StdHandle'        => '4.2',
	'Tie::SubstrHash'       => '1.00',
	'Time::HiRes'           => '1.9721_01',
	'Time::Local'           => '1.1901_01',
	'Time::Piece'           => '1.20_01',
	'Time::Piece::Seconds'  => undef,
	'Time::Seconds'         => undef,
	'Time::gmtime'          => '1.03',
	'Time::localtime'       => '1.02',
	'Time::tm'              => '1.00',
	'UNIVERSAL'             => '1.07',
	'Unicode'               => '6.0.0',
	'Unicode::Collate'      => '0.6801',
	'Unicode::Collate::CJK::Big5'=> '0.65',
	'Unicode::Collate::CJK::GB2312'=> '0.65',
	'Unicode::Collate::CJK::JISX0208'=> '0.64',
	'Unicode::Collate::CJK::Korean'=> '0.66',
	'Unicode::Collate::CJK::Pinyin'=> '0.65',
	'Unicode::Collate::CJK::Stroke'=> '0.65',
	'Unicode::Collate::Locale'=> '0.68',
	'Unicode::Normalize'    => '1.08',
	'Unicode::UCD'          => '0.30',
	'User::grent'           => '1.01',
	'User::pwent'           => '1.00',
	'VMS::DCLsym'           => '1.04',
	'VMS::Filespec'         => '1.12',
	'VMS::Stdio'            => '2.4',
	'Win32'                 => '0.41',
	'Win32API::File'        => '0.1101',
	'Win32API::File::ExtUtils::Myconst2perl'=> '1',
	'Win32CORE'             => '0.02',
	'XS::APItest'           => '0.27',
	'XS::Typemap'           => '0.04',
	'XSLoader'              => '0.13',
	'XSLoader::XSLoader'    => '0.13',
	'attributes'            => '0.13',
	'autodie'               => '2.1001',
	'autodie::exception'    => '2.1001',
	'autodie::exception::system'=> '2.1001',
	'autodie::hints'        => '2.1001',
	'autouse'               => '1.06',
	'base'                  => '2.15',
	'bigint'                => '0.25',
	'bignum'                => '0.25',
	'bigrat'                => '0.25',
	'blib'                  => '1.06',
	'bytes'                 => '1.04',
	'charnames'             => '1.17',
	'constant'              => '1.20',
	'deprecate'             => '0.01',
	'diagnostics'           => '1.20',
	'encoding'              => '2.6_01',
	'encoding::warnings'    => '0.11',
	'feature'               => '1.19',
	'fields'                => '2.15',
	'filetest'              => '1.02',
	'if'                    => '0.0601',
	'inc::latest'           => '0.3607',
	'integer'               => '1.00',
	'less'                  => '0.03',
	'lib'                   => '0.63',
	'locale'                => '1.00',
	'mro'                   => '1.06',
	'open'                  => '1.08',
	'ops'                   => '1.02',
	'overload'              => '1.12',
	'overload::numbers'     => undef,
	'overloading'           => '0.01',
	'parent'                => '0.224',
	're'                    => '0.15',
	'sigtrap'               => '1.05',
	'sort'                  => '2.01',
	'strict'                => '1.04',
	'subs'                  => '1.00',
	'threads'               => '1.81_03',
	'threads::shared'       => '1.35',
	'utf8'                  => '1.08',
	'vars'                  => '1.02',
	'version'               => '0.86',
	'vmsish'                => '1.02',
	'warnings'              => '1.11',
	'warnings::register'    => '1.02',
    },

    5.012003 => {
	'AnyDBM_File'           => '1.00',
	'App::Cpan'             => '1.5701',
	'App::Prove'            => '3.17',
	'App::Prove::State'     => '3.17',
	'App::Prove::State::Result'=> '3.17',
	'App::Prove::State::Result::Test'=> '3.17',
	'Archive::Extract'      => '0.38',
	'Archive::Tar'          => '1.54',
	'Archive::Tar::Constant'=> '0.02',
	'Archive::Tar::File'    => '0.02',
	'Attribute::Handlers'   => '0.87',
	'AutoLoader'            => '5.70',
	'AutoSplit'             => '1.06',
	'B'                     => '1.23',
	'B::Concise'            => '0.78',
	'B::Debug'              => '1.12',
	'B::Deparse'            => '0.9701',
	'B::Lint'               => '1.11_01',
	'B::Lint::Debug'        => '0.01',
	'B::Showlex'            => '1.02',
	'B::Terse'              => '1.05',
	'B::Xref'               => '1.02',
	'Benchmark'             => '1.11',
	'CGI'                   => '3.49',
	'CGI::Apache'           => '1.01',
	'CGI::Carp'             => '3.45',
	'CGI::Cookie'           => '1.29',
	'CGI::Fast'             => '1.08',
	'CGI::Pretty'           => '3.46',
	'CGI::Push'             => '1.04',
	'CGI::Switch'           => '1.01',
	'CGI::Util'             => '3.48',
	'CPAN'                  => '1.94_56',
	'CPAN::Author'          => '5.5',
	'CPAN::Bundle'          => '5.5',
	'CPAN::CacheMgr'        => '5.5',
	'CPAN::Complete'        => '5.5',
	'CPAN::Debug'           => '5.5001',
	'CPAN::DeferredCode'    => '5.50',
	'CPAN::Distribution'    => '1.9456_01',
	'CPAN::Distroprefs'     => '6',
	'CPAN::Distrostatus'    => '5.5',
	'CPAN::Exception::RecursiveDependency'=> '5.5',
	'CPAN::Exception::blocked_urllist'=> '1.0',
	'CPAN::Exception::yaml_not_installed'=> '5.5',
	'CPAN::FTP'             => '5.5004',
	'CPAN::FTP::netrc'      => '1.00',
	'CPAN::FirstTime'       => '5.5301',
	'CPAN::HandleConfig'    => '5.5001',
	'CPAN::Index'           => '1.94',
	'CPAN::InfoObj'         => '5.5',
	'CPAN::Kwalify'         => '5.50',
	'CPAN::LWP::UserAgent'  => '1.94',
	'CPAN::Mirrors'         => '1.77',
	'CPAN::Module'          => '5.5',
	'CPAN::Nox'             => '5.50',
	'CPAN::Prompt'          => '5.5',
	'CPAN::Queue'           => '5.5',
	'CPAN::Shell'           => '5.5001',
	'CPAN::Tarzip'          => '5.5011',
	'CPAN::URL'             => '5.5',
	'CPAN::Version'         => '5.5',
	'CPANPLUS'              => '0.90',
	'CPANPLUS::Backend'     => undef,
	'CPANPLUS::Backend::RV' => undef,
	'CPANPLUS::Config'      => undef,
	'CPANPLUS::Configure'   => undef,
	'CPANPLUS::Configure::Setup'=> undef,
	'CPANPLUS::Dist'        => undef,
	'CPANPLUS::Dist::Autobundle'=> undef,
	'CPANPLUS::Dist::Base'  => undef,
	'CPANPLUS::Dist::Build' => '0.46',
	'CPANPLUS::Dist::Build::Constants'=> '0.46',
	'CPANPLUS::Dist::MM'    => undef,
	'CPANPLUS::Dist::Sample'=> undef,
	'CPANPLUS::Error'       => undef,
	'CPANPLUS::Internals'   => '0.90',
	'CPANPLUS::Internals::Constants'=> undef,
	'CPANPLUS::Internals::Constants::Report'=> undef,
	'CPANPLUS::Internals::Extract'=> undef,
	'CPANPLUS::Internals::Fetch'=> undef,
	'CPANPLUS::Internals::Report'=> undef,
	'CPANPLUS::Internals::Search'=> undef,
	'CPANPLUS::Internals::Source'=> undef,
	'CPANPLUS::Internals::Source::Memory'=> undef,
	'CPANPLUS::Internals::Source::SQLite'=> undef,
	'CPANPLUS::Internals::Source::SQLite::Tie'=> undef,
	'CPANPLUS::Internals::Utils'=> undef,
	'CPANPLUS::Internals::Utils::Autoflush'=> undef,
	'CPANPLUS::Module'      => undef,
	'CPANPLUS::Module::Author'=> undef,
	'CPANPLUS::Module::Author::Fake'=> undef,
	'CPANPLUS::Module::Checksums'=> undef,
	'CPANPLUS::Module::Fake'=> undef,
	'CPANPLUS::Module::Signature'=> undef,
	'CPANPLUS::Selfupdate'  => undef,
	'CPANPLUS::Shell'       => undef,
	'CPANPLUS::Shell::Classic'=> '0.0562',
	'CPANPLUS::Shell::Default'=> '0.90',
	'CPANPLUS::Shell::Default::Plugins::CustomSource'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Remote'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Source'=> undef,
	'Carp'                  => '1.17',
	'Carp::Heavy'           => '1.17',
	'Class::ISA'            => '0.36',
	'Class::Struct'         => '0.63',
	'Compress::Raw::Bzip2'  => '2.024',
	'Compress::Raw::Zlib'   => '2.024',
	'Compress::Zlib'        => '2.024',
	'Config'                => undef,
	'Config::Extensions'    => '0.01',
	'Cwd'                   => '3.31',
	'DB'                    => '1.02',
	'DBM_Filter'            => '0.03',
	'DBM_Filter::compress'  => '0.02',
	'DBM_Filter::encode'    => '0.02',
	'DBM_Filter::int32'     => '0.02',
	'DBM_Filter::null'      => '0.02',
	'DBM_Filter::utf8'      => '0.02',
	'DB_File'               => '1.820',
	'Data::Dumper'          => '2.125',
	'Devel::DProf'          => '20080331.00',
	'Devel::DProf::V'       => undef,
	'Devel::DProf::dprof::V'=> undef,
	'Devel::InnerPackage'   => '0.3',
	'Devel::PPPort'         => '3.19',
	'Devel::Peek'           => '1.04',
	'Devel::SelfStubber'    => '1.03',
	'Digest'                => '1.16',
	'Digest::MD5'           => '2.39',
	'Digest::SHA'           => '5.47',
	'Digest::base'          => '1.16',
	'Digest::file'          => '1.16',
	'DirHandle'             => '1.03',
	'Dumpvalue'             => '1.13',
	'DynaLoader'            => '1.10',
	'Encode'                => '2.39',
	'Encode::Alias'         => '2.12',
	'Encode::Byte'          => '2.04',
	'Encode::CJKConstants'  => '2.02',
	'Encode::CN'            => '2.03',
	'Encode::CN::HZ'        => '2.05',
	'Encode::Config'        => '2.05',
	'Encode::EBCDIC'        => '2.02',
	'Encode::Encoder'       => '2.01',
	'Encode::Encoding'      => '2.05',
	'Encode::GSM0338'       => '2.01',
	'Encode::Guess'         => '2.03',
	'Encode::JP'            => '2.04',
	'Encode::JP::H2Z'       => '2.02',
	'Encode::JP::JIS7'      => '2.04',
	'Encode::KR'            => '2.03',
	'Encode::KR::2022_KR'   => '2.02',
	'Encode::MIME::Header'  => '2.11',
	'Encode::MIME::Header::ISO_2022_JP'=> '1.03',
	'Encode::MIME::Name'    => '1.01',
	'Encode::Symbol'        => '2.02',
	'Encode::TW'            => '2.03',
	'Encode::Unicode'       => '2.07',
	'Encode::Unicode::UTF7' => '2.04',
	'English'               => '1.04',
	'Env'                   => '1.01',
	'Errno'                 => '1.11',
	'Exporter'              => '5.64_01',
	'Exporter::Heavy'       => '5.64_01',
	'ExtUtils::CBuilder'    => '0.27',
	'ExtUtils::CBuilder::Base'=> '0.27',
	'ExtUtils::CBuilder::Platform::Unix'=> '0.27',
	'ExtUtils::CBuilder::Platform::VMS'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.27',
	'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.27',
	'ExtUtils::CBuilder::Platform::aix'=> '0.27',
	'ExtUtils::CBuilder::Platform::cygwin'=> '0.27',
	'ExtUtils::CBuilder::Platform::darwin'=> '0.27',
	'ExtUtils::CBuilder::Platform::dec_osf'=> '0.27',
	'ExtUtils::CBuilder::Platform::os2'=> '0.27',
	'ExtUtils::Command'     => '1.16',
	'ExtUtils::Command::MM' => '6.56',
	'ExtUtils::Constant'    => '0.22',
	'ExtUtils::Constant::Base'=> '0.04',
	'ExtUtils::Constant::ProxySubs'=> '0.06',
	'ExtUtils::Constant::Utils'=> '0.02',
	'ExtUtils::Constant::XS'=> '0.03',
	'ExtUtils::Embed'       => '1.28',
	'ExtUtils::Install'     => '1.55',
	'ExtUtils::Installed'   => '1.999_001',
	'ExtUtils::Liblist'     => '6.56',
	'ExtUtils::Liblist::Kid'=> '6.56',
	'ExtUtils::MM'          => '6.56',
	'ExtUtils::MM_AIX'      => '6.56',
	'ExtUtils::MM_Any'      => '6.56',
	'ExtUtils::MM_BeOS'     => '6.56',
	'ExtUtils::MM_Cygwin'   => '6.56',
	'ExtUtils::MM_DOS'      => '6.56',
	'ExtUtils::MM_Darwin'   => '6.56',
	'ExtUtils::MM_MacOS'    => '6.56',
	'ExtUtils::MM_NW5'      => '6.56',
	'ExtUtils::MM_OS2'      => '6.56',
	'ExtUtils::MM_QNX'      => '6.56',
	'ExtUtils::MM_UWIN'     => '6.56',
	'ExtUtils::MM_Unix'     => '6.56',
	'ExtUtils::MM_VMS'      => '6.56',
	'ExtUtils::MM_VOS'      => '6.56',
	'ExtUtils::MM_Win32'    => '6.56',
	'ExtUtils::MM_Win95'    => '6.56',
	'ExtUtils::MY'          => '6.56',
	'ExtUtils::MakeMaker'   => '6.56',
	'ExtUtils::MakeMaker::Config'=> '6.56',
	'ExtUtils::Manifest'    => '1.57',
	'ExtUtils::Miniperl'    => undef,
	'ExtUtils::Mkbootstrap' => '6.56',
	'ExtUtils::Mksymlists'  => '6.56',
	'ExtUtils::Packlist'    => '1.44',
	'ExtUtils::ParseXS'     => '2.21',
	'ExtUtils::XSSymSet'    => '1.1',
	'ExtUtils::testlib'     => '6.56',
	'Fatal'                 => '2.06_01',
	'Fcntl'                 => '1.06',
	'File::Basename'        => '2.78',
	'File::CheckTree'       => '4.4',
	'File::Compare'         => '1.1006',
	'File::Copy'            => '2.18',
	'File::DosGlob'         => '1.01',
	'File::Fetch'           => '0.24',
	'File::Find'            => '1.15',
	'File::Glob'            => '1.07',
	'File::GlobMapper'      => '1.000',
	'File::Path'            => '2.08_01',
	'File::Spec'            => '3.31_01',
	'File::Spec::Cygwin'    => '3.30',
	'File::Spec::Epoc'      => '3.30',
	'File::Spec::Functions' => '3.30',
	'File::Spec::Mac'       => '3.30',
	'File::Spec::OS2'       => '3.30',
	'File::Spec::Unix'      => '3.30',
	'File::Spec::VMS'       => '3.30',
	'File::Spec::Win32'     => '3.30',
	'File::Temp'            => '0.22',
	'File::stat'            => '1.02',
	'FileCache'             => '1.08',
	'FileHandle'            => '2.02',
	'Filter::Simple'        => '0.84',
	'Filter::Util::Call'    => '1.08',
	'FindBin'               => '1.50',
	'GDBM_File'             => '1.10',
	'Getopt::Long'          => '2.38',
	'Getopt::Std'           => '1.06',
	'Hash::Util'            => '0.07',
	'Hash::Util::FieldHash' => '1.04',
	'I18N::Collate'         => '1.01',
	'I18N::LangTags'        => '0.35',
	'I18N::LangTags::Detect'=> '1.04',
	'I18N::LangTags::List'  => '0.35',
	'I18N::Langinfo'        => '0.03',
	'IO'                    => '1.25_02',
	'IO::Compress::Adapter::Bzip2'=> '2.024',
	'IO::Compress::Adapter::Deflate'=> '2.024',
	'IO::Compress::Adapter::Identity'=> '2.024',
	'IO::Compress::Base'    => '2.024',
	'IO::Compress::Base::Common'=> '2.024',
	'IO::Compress::Bzip2'   => '2.024',
	'IO::Compress::Deflate' => '2.024',
	'IO::Compress::Gzip'    => '2.024',
	'IO::Compress::Gzip::Constants'=> '2.024',
	'IO::Compress::RawDeflate'=> '2.024',
	'IO::Compress::Zip'     => '2.024',
	'IO::Compress::Zip::Constants'=> '2.024',
	'IO::Compress::Zlib::Constants'=> '2.024',
	'IO::Compress::Zlib::Extra'=> '2.024',
	'IO::Dir'               => '1.07',
	'IO::File'              => '1.14',
	'IO::Handle'            => '1.28',
	'IO::Pipe'              => '1.13',
	'IO::Poll'              => '0.07',
	'IO::Seekable'          => '1.10',
	'IO::Select'            => '1.17',
	'IO::Socket'            => '1.31',
	'IO::Socket::INET'      => '1.31',
	'IO::Socket::UNIX'      => '1.23',
	'IO::Uncompress::Adapter::Bunzip2'=> '2.024',
	'IO::Uncompress::Adapter::Identity'=> '2.024',
	'IO::Uncompress::Adapter::Inflate'=> '2.024',
	'IO::Uncompress::AnyInflate'=> '2.024',
	'IO::Uncompress::AnyUncompress'=> '2.024',
	'IO::Uncompress::Base'  => '2.024',
	'IO::Uncompress::Bunzip2'=> '2.024',
	'IO::Uncompress::Gunzip'=> '2.024',
	'IO::Uncompress::Inflate'=> '2.024',
	'IO::Uncompress::RawInflate'=> '2.024',
	'IO::Uncompress::Unzip' => '2.024',
	'IO::Zlib'              => '1.10',
	'IPC::Cmd'              => '0.54',
	'IPC::Msg'              => '2.01',
	'IPC::Open2'            => '1.03',
	'IPC::Open3'            => '1.05',
	'IPC::Semaphore'        => '2.01',
	'IPC::SharedMem'        => '2.01',
	'IPC::SysV'             => '2.01',
	'List::Util'            => '1.22',
	'List::Util::PP'        => '1.22',
	'List::Util::XS'        => '1.22',
	'Locale::Constants'     => '2.07',
	'Locale::Country'       => '2.07',
	'Locale::Currency'      => '2.07',
	'Locale::Language'      => '2.07',
	'Locale::Maketext'      => '1.14',
	'Locale::Maketext::Guts'=> '1.13',
	'Locale::Maketext::GutsLoader'=> '1.13',
	'Locale::Maketext::Simple'=> '0.21',
	'Locale::Script'        => '2.07',
	'Log::Message'          => '0.02',
	'Log::Message::Config'  => '0.01',
	'Log::Message::Handlers'=> undef,
	'Log::Message::Item'    => undef,
	'Log::Message::Simple'  => '0.06',
	'MIME::Base64'          => '3.08',
	'MIME::QuotedPrint'     => '3.08',
	'Math::BigFloat'        => '1.60',
	'Math::BigFloat::Trace' => '0.01',
	'Math::BigInt'          => '1.89_01',
	'Math::BigInt::Calc'    => '0.52',
	'Math::BigInt::CalcEmu' => '0.05',
	'Math::BigInt::FastCalc'=> '0.19',
	'Math::BigInt::Trace'   => '0.01',
	'Math::BigRat'          => '0.24',
	'Math::Complex'         => '1.56',
	'Math::Trig'            => '1.2',
	'Memoize'               => '1.01_03',
	'Memoize::AnyDBM_File'  => '0.65',
	'Memoize::Expire'       => '1.00',
	'Memoize::ExpireFile'   => '1.01',
	'Memoize::ExpireTest'   => '0.65',
	'Memoize::NDBM_File'    => '0.65',
	'Memoize::SDBM_File'    => '0.65',
	'Memoize::Storable'     => '0.65',
	'Module::Build'         => '0.3603',
	'Module::Build::Base'   => '0.3603',
	'Module::Build::Compat' => '0.3603',
	'Module::Build::Config' => '0.3603',
	'Module::Build::ConfigData'=> undef,
	'Module::Build::Cookbook'=> '0.3603',
	'Module::Build::Dumper' => '0.3603',
	'Module::Build::ModuleInfo'=> '0.3603',
	'Module::Build::Notes'  => '0.3603',
	'Module::Build::PPMMaker'=> '0.3603',
	'Module::Build::Platform::Amiga'=> '0.3603',
	'Module::Build::Platform::Default'=> '0.3603',
	'Module::Build::Platform::EBCDIC'=> '0.3603',
	'Module::Build::Platform::MPEiX'=> '0.3603',
	'Module::Build::Platform::MacOS'=> '0.3603',
	'Module::Build::Platform::RiscOS'=> '0.3603',
	'Module::Build::Platform::Unix'=> '0.3603',
	'Module::Build::Platform::VMS'=> '0.3603',
	'Module::Build::Platform::VOS'=> '0.3603',
	'Module::Build::Platform::Windows'=> '0.3603',
	'Module::Build::Platform::aix'=> '0.3603',
	'Module::Build::Platform::cygwin'=> '0.360301',
	'Module::Build::Platform::darwin'=> '0.3603',
	'Module::Build::Platform::os2'=> '0.3603',
	'Module::Build::PodParser'=> '0.3603',
	'Module::Build::Version'=> '0.77',
	'Module::Build::YAML'   => '1.40',
	'Module::CoreList'      => '2.43',
	'Module::Load'          => '0.16',
	'Module::Load::Conditional'=> '0.38',
	'Module::Loaded'        => '0.06',
	'Module::Pluggable'     => '3.9',
	'Module::Pluggable::Object'=> '3.9',
	'Moped::Msg'            => '0.01',
	'NDBM_File'             => '1.08',
	'NEXT'                  => '0.64',
	'Net::Cmd'              => '2.29',
	'Net::Config'           => '1.11',
	'Net::Domain'           => '2.20',
	'Net::FTP'              => '2.77',
	'Net::FTP::A'           => '1.18',
	'Net::FTP::E'           => '0.01',
	'Net::FTP::I'           => '1.12',
	'Net::FTP::L'           => '0.01',
	'Net::FTP::dataconn'    => '0.11',
	'Net::NNTP'             => '2.24',
	'Net::Netrc'            => '2.12',
	'Net::POP3'             => '2.29',
	'Net::Ping'             => '2.36',
	'Net::SMTP'             => '2.31',
	'Net::Time'             => '2.10',
	'Net::hostent'          => '1.01',
	'Net::netent'           => '1.00',
	'Net::protoent'         => '1.00',
	'Net::servent'          => '1.01',
	'O'                     => '1.01',
	'ODBM_File'             => '1.07',
	'Object::Accessor'      => '0.36',
	'Opcode'                => '1.15',
	'POSIX'                 => '1.19',
	'Package::Constants'    => '0.02',
	'Params::Check'         => '0.26',
	'Parse::CPAN::Meta'     => '1.40',
	'PerlIO'                => '1.06',
	'PerlIO::encoding'      => '0.12',
	'PerlIO::scalar'        => '0.08',
	'PerlIO::via'           => '0.09',
	'PerlIO::via::QuotedPrint'=> '0.06',
	'Pod::Checker'          => '1.45',
	'Pod::Escapes'          => '1.04',
	'Pod::Find'             => '1.35',
	'Pod::Functions'        => '1.04',
	'Pod::Html'             => '1.09',
	'Pod::InputObjects'     => '1.31',
	'Pod::LaTeX'            => '0.58',
	'Pod::Man'              => '2.23',
	'Pod::ParseLink'        => '1.10',
	'Pod::ParseUtils'       => '1.36',
	'Pod::Parser'           => '1.37',
	'Pod::Perldoc'          => '3.15_02',
	'Pod::Perldoc::BaseTo'  => undef,
	'Pod::Perldoc::GetOptsOO'=> undef,
	'Pod::Perldoc::ToChecker'=> undef,
	'Pod::Perldoc::ToMan'   => undef,
	'Pod::Perldoc::ToNroff' => undef,
	'Pod::Perldoc::ToPod'   => undef,
	'Pod::Perldoc::ToRtf'   => undef,
	'Pod::Perldoc::ToText'  => undef,
	'Pod::Perldoc::ToTk'    => undef,
	'Pod::Perldoc::ToXml'   => undef,
	'Pod::PlainText'        => '2.04',
	'Pod::Plainer'          => '1.02',
	'Pod::Select'           => '1.36',
	'Pod::Simple'           => '3.14',
	'Pod::Simple::BlackBox' => '3.14',
	'Pod::Simple::Checker'  => '3.14',
	'Pod::Simple::Debug'    => '3.14',
	'Pod::Simple::DumpAsText'=> '3.14',
	'Pod::Simple::DumpAsXML'=> '3.14',
	'Pod::Simple::HTML'     => '3.14',
	'Pod::Simple::HTMLBatch'=> '3.14',
	'Pod::Simple::HTMLLegacy'=> '5.01',
	'Pod::Simple::LinkSection'=> '3.14',
	'Pod::Simple::Methody'  => '3.14',
	'Pod::Simple::Progress' => '3.14',
	'Pod::Simple::PullParser'=> '3.14',
	'Pod::Simple::PullParserEndToken'=> '3.14',
	'Pod::Simple::PullParserStartToken'=> '3.14',
	'Pod::Simple::PullParserTextToken'=> '3.14',
	'Pod::Simple::PullParserToken'=> '3.14',
	'Pod::Simple::RTF'      => '3.14',
	'Pod::Simple::Search'   => '3.14',
	'Pod::Simple::SimpleTree'=> '3.14',
	'Pod::Simple::Text'     => '3.14',
	'Pod::Simple::TextContent'=> '3.14',
	'Pod::Simple::TiedOutFH'=> '3.14',
	'Pod::Simple::Transcode'=> '3.14',
	'Pod::Simple::TranscodeDumb'=> '3.14',
	'Pod::Simple::TranscodeSmart'=> '3.14',
	'Pod::Simple::XHTML'    => '3.14',
	'Pod::Simple::XMLOutStream'=> '3.14',
	'Pod::Text'             => '3.14',
	'Pod::Text::Color'      => '2.06',
	'Pod::Text::Overstrike' => '2.04',
	'Pod::Text::Termcap'    => '2.06',
	'Pod::Usage'            => '1.36',
	'SDBM_File'             => '1.06',
	'Safe'                  => '2.27',
	'Scalar::Util'          => '1.22',
	'Scalar::Util::PP'      => '1.22',
	'Search::Dict'          => '1.02',
	'SelectSaver'           => '1.02',
	'SelfLoader'            => '1.17',
	'Shell'                 => '0.72_01',
	'Socket'                => '1.87_01',
	'Storable'              => '2.22',
	'Switch'                => '2.16',
	'Symbol'                => '1.07',
	'Sys::Hostname'         => '1.11',
	'Sys::Syslog'           => '0.27',
	'Sys::Syslog::win32::Win32'=> undef,
	'TAP::Base'             => '3.17',
	'TAP::Formatter::Base'  => '3.17',
	'TAP::Formatter::Color' => '3.17',
	'TAP::Formatter::Console'=> '3.17',
	'TAP::Formatter::Console::ParallelSession'=> '3.17',
	'TAP::Formatter::Console::Session'=> '3.17',
	'TAP::Formatter::File'  => '3.17',
	'TAP::Formatter::File::Session'=> '3.17',
	'TAP::Formatter::Session'=> '3.17',
	'TAP::Harness'          => '3.17',
	'TAP::Object'           => '3.17',
	'TAP::Parser'           => '3.17',
	'TAP::Parser::Aggregator'=> '3.17',
	'TAP::Parser::Grammar'  => '3.17',
	'TAP::Parser::Iterator' => '3.17',
	'TAP::Parser::Iterator::Array'=> '3.17',
	'TAP::Parser::Iterator::Process'=> '3.17',
	'TAP::Parser::Iterator::Stream'=> '3.17',
	'TAP::Parser::IteratorFactory'=> '3.17',
	'TAP::Parser::Multiplexer'=> '3.17',
	'TAP::Parser::Result'   => '3.17',
	'TAP::Parser::Result::Bailout'=> '3.17',
	'TAP::Parser::Result::Comment'=> '3.17',
	'TAP::Parser::Result::Plan'=> '3.17',
	'TAP::Parser::Result::Pragma'=> '3.17',
	'TAP::Parser::Result::Test'=> '3.17',
	'TAP::Parser::Result::Unknown'=> '3.17',
	'TAP::Parser::Result::Version'=> '3.17',
	'TAP::Parser::Result::YAML'=> '3.17',
	'TAP::Parser::ResultFactory'=> '3.17',
	'TAP::Parser::Scheduler'=> '3.17',
	'TAP::Parser::Scheduler::Job'=> '3.17',
	'TAP::Parser::Scheduler::Spinner'=> '3.17',
	'TAP::Parser::Source'   => '3.17',
	'TAP::Parser::Source::Perl'=> '3.17',
	'TAP::Parser::Utils'    => '3.17',
	'TAP::Parser::YAMLish::Reader'=> '3.17',
	'TAP::Parser::YAMLish::Writer'=> '3.17',
	'Term::ANSIColor'       => '2.02',
	'Term::Cap'             => '1.12',
	'Term::Complete'        => '1.402',
	'Term::ReadLine'        => '1.05',
	'Term::UI'              => '0.20',
	'Term::UI::History'     => undef,
	'Test'                  => '1.25_02',
	'Test::Builder'         => '0.94',
	'Test::Builder::Module' => '0.94',
	'Test::Builder::Tester' => '1.18',
	'Test::Builder::Tester::Color'=> '1.18',
	'Test::Harness'         => '3.17',
	'Test::More'            => '0.94',
	'Test::Simple'          => '0.94',
	'Text::Abbrev'          => '1.01',
	'Text::Balanced'        => '2.02',
	'Text::ParseWords'      => '3.27',
	'Text::Soundex'         => '3.03_01',
	'Text::Tabs'            => '2009.0305',
	'Text::Wrap'            => '2009.0305',
	'Thread'                => '3.02',
	'Thread::Queue'         => '2.11',
	'Thread::Semaphore'     => '2.09',
	'Tie::Array'            => '1.03',
	'Tie::File'             => '0.97_02',
	'Tie::Handle'           => '4.2',
	'Tie::Hash'             => '1.03',
	'Tie::Hash::NamedCapture'=> '0.06',
	'Tie::Memoize'          => '1.1',
	'Tie::RefHash'          => '1.38',
	'Tie::Scalar'           => '1.02',
	'Tie::StdHandle'        => '4.2',
	'Tie::SubstrHash'       => '1.00',
	'Time::HiRes'           => '1.9719',
	'Time::Local'           => '1.1901_01',
	'Time::Piece'           => '1.15_01',
	'Time::Piece::Seconds'  => undef,
	'Time::Seconds'         => undef,
	'Time::gmtime'          => '1.03',
	'Time::localtime'       => '1.02',
	'Time::tm'              => '1.00',
	'UNIVERSAL'             => '1.06',
	'Unicode'               => '5.2.0',
	'Unicode::Collate'      => '0.52_01',
	'Unicode::Normalize'    => '1.03',
	'Unicode::UCD'          => '0.27',
	'User::grent'           => '1.01',
	'User::pwent'           => '1.00',
	'VMS::DCLsym'           => '1.03',
	'VMS::Stdio'            => '2.4',
	'Win32'                 => '0.39',
	'Win32API::File'        => '0.1101',
	'Win32API::File::ExtUtils::Myconst2perl'=> '1',
	'Win32CORE'             => '0.02',
	'XS::APItest'           => '0.17',
	'XS::APItest::KeywordRPN'=> '0.003',
	'XS::Typemap'           => '0.03',
	'XSLoader'              => '0.10',
	'XSLoader::XSLoader'    => '0.10',
	'attributes'            => '0.12',
	'autodie'               => '2.06_01',
	'autodie::exception'    => '2.06_01',
	'autodie::exception::system'=> '2.06_01',
	'autodie::hints'        => '2.06_01',
	'autouse'               => '1.06',
	'base'                  => '2.15',
	'bigint'                => '0.23',
	'bignum'                => '0.23',
	'bigrat'                => '0.23',
	'blib'                  => '1.04',
	'bytes'                 => '1.04',
	'charnames'             => '1.07',
	'constant'              => '1.20',
	'deprecate'             => '0.01',
	'diagnostics'           => '1.19',
	'encoding'              => '2.6_01',
	'encoding::warnings'    => '0.11',
	'feature'               => '1.16',
	'fields'                => '2.15',
	'filetest'              => '1.02',
	'if'                    => '0.05',
	'inc::latest'           => '0.3603',
	'integer'               => '1.00',
	'less'                  => '0.03',
	'lib'                   => '0.62',
	'locale'                => '1.00',
	'mro'                   => '1.02',
	'open'                  => '1.07',
	'ops'                   => '1.02',
	'overload'              => '1.10',
	'overload::numbers'     => undef,
	'overloading'           => '0.01',
	'parent'                => '0.223',
	're'                    => '0.11',
	'sigtrap'               => '1.04',
	'sort'                  => '2.01',
	'strict'                => '1.04',
	'subs'                  => '1.00',
	'threads'               => '1.75',
	'threads::shared'       => '1.32',
	'utf8'                  => '1.08',
	'vars'                  => '1.01',
	'version'               => '0.82',
	'vmsish'                => '1.02',
	'warnings'              => '1.09',
	'warnings::register'    => '1.01',
    },
    5.013009 => {
	'AnyDBM_File'           => '1.00',
	'App::Cpan'             => '1.5701',
	'App::Prove'            => '3.22',
	'App::Prove::State'     => '3.22',
	'App::Prove::State::Result'=> '3.22',
	'App::Prove::State::Result::Test'=> '3.22',
	'Archive::Extract'      => '0.48',
	'Archive::Tar'          => '1.76',
	'Archive::Tar::Constant'=> '1.76',
	'Archive::Tar::File'    => '1.76',
	'Attribute::Handlers'   => '0.88',
	'AutoLoader'            => '5.71',
	'AutoSplit'             => '1.06',
	'B'                     => '1.27',
	'B::Concise'            => '0.83',
	'B::Debug'              => '1.16',
	'B::Deparse'            => '1.03',
	'B::Lint'               => '1.13',
	'B::Lint::Debug'        => '1.12',
	'B::Showlex'            => '1.02',
	'B::Terse'              => '1.05',
	'B::Xref'               => '1.02',
	'Benchmark'             => '1.12',
	'CGI'                   => '3.51',
	'CGI::Apache'           => '1.01',
	'CGI::Carp'             => '3.51',
	'CGI::Cookie'           => '1.30',
	'CGI::Fast'             => '1.08',
	'CGI::Pretty'           => '3.46',
	'CGI::Push'             => '1.05',
	'CGI::Switch'           => '1.01',
	'CGI::Util'             => '3.51',
	'CPAN'                  => '1.94_63',
	'CPAN::Author'          => '5.5',
	'CPAN::Bundle'          => '5.5',
	'CPAN::CacheMgr'        => '5.5',
	'CPAN::Complete'        => '5.5',
	'CPAN::Debug'           => '5.5001',
	'CPAN::DeferredCode'    => '5.50',
	'CPAN::Distribution'    => '1.9600',
	'CPAN::Distroprefs'     => '6',
	'CPAN::Distrostatus'    => '5.5',
	'CPAN::Exception::RecursiveDependency'=> '5.5',
	'CPAN::Exception::blocked_urllist'=> '1.0',
	'CPAN::Exception::yaml_not_installed'=> '5.5',
	'CPAN::FTP'             => '5.5005',
	'CPAN::FTP::netrc'      => '1.00',
	'CPAN::FirstTime'       => '5.5301',
	'CPAN::HTTP::Client'    => '1.94',
	'CPAN::HTTP::Credentials'=> '1.94',
	'CPAN::HandleConfig'    => '5.5001',
	'CPAN::Index'           => '1.94',
	'CPAN::InfoObj'         => '5.5',
	'CPAN::Kwalify'         => '5.50',
	'CPAN::LWP::UserAgent'  => '1.94',
	'CPAN::Meta::YAML'      => '0.003',
	'CPAN::Mirrors'         => '1.77',
	'CPAN::Module'          => '5.5',
	'CPAN::Nox'             => '5.50',
	'CPAN::Prompt'          => '5.5',
	'CPAN::Queue'           => '5.5001',
	'CPAN::Shell'           => '5.5001',
	'CPAN::Tarzip'          => '5.5011',
	'CPAN::URL'             => '5.5',
	'CPAN::Version'         => '5.5001',
	'CPANPLUS'              => '0.9011',
	'CPANPLUS::Backend'     => undef,
	'CPANPLUS::Backend::RV' => undef,
	'CPANPLUS::Config'      => undef,
	'CPANPLUS::Configure'   => undef,
	'CPANPLUS::Configure::Setup'=> undef,
	'CPANPLUS::Dist'        => undef,
	'CPANPLUS::Dist::Autobundle'=> undef,
	'CPANPLUS::Dist::Base'  => undef,
	'CPANPLUS::Dist::Build' => '0.52',
	'CPANPLUS::Dist::Build::Constants'=> '0.52',
	'CPANPLUS::Dist::MM'    => undef,
	'CPANPLUS::Dist::Sample'=> undef,
	'CPANPLUS::Error'       => undef,
	'CPANPLUS::Internals'   => '0.9011',
	'CPANPLUS::Internals::Constants'=> undef,
	'CPANPLUS::Internals::Constants::Report'=> undef,
	'CPANPLUS::Internals::Extract'=> undef,
	'CPANPLUS::Internals::Fetch'=> undef,
	'CPANPLUS::Internals::Report'=> undef,
	'CPANPLUS::Internals::Search'=> undef,
	'CPANPLUS::Internals::Source'=> undef,
	'CPANPLUS::Internals::Source::Memory'=> undef,
	'CPANPLUS::Internals::Source::SQLite'=> undef,
	'CPANPLUS::Internals::Source::SQLite::Tie'=> undef,
	'CPANPLUS::Internals::Utils'=> undef,
	'CPANPLUS::Internals::Utils::Autoflush'=> undef,
	'CPANPLUS::Module'      => undef,
	'CPANPLUS::Module::Author'=> undef,
	'CPANPLUS::Module::Author::Fake'=> undef,
	'CPANPLUS::Module::Checksums'=> undef,
	'CPANPLUS::Module::Fake'=> undef,
	'CPANPLUS::Module::Signature'=> undef,
	'CPANPLUS::Selfupdate'  => undef,
	'CPANPLUS::Shell'       => undef,
	'CPANPLUS::Shell::Classic'=> '0.0562',
	'CPANPLUS::Shell::Default'=> '0.9011',
	'CPANPLUS::Shell::Default::Plugins::CustomSource'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Remote'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Source'=> undef,
	'Carp'                  => '1.19',
	'Carp::Heavy'           => '1.19',
	'Class::Struct'         => '0.63',
	'Compress::Raw::Bzip2'  => '2.033',
	'Compress::Raw::Zlib'   => '2.033',
	'Compress::Zlib'        => '2.033',
	'Config'                => undef,
	'Config::Extensions'    => '0.01',
	'Cwd'                   => '3.36',
	'DB'                    => '1.03',
	'DBM_Filter'            => '0.04',
	'DBM_Filter::compress'  => '0.02',
	'DBM_Filter::encode'    => '0.02',
	'DBM_Filter::int32'     => '0.02',
	'DBM_Filter::null'      => '0.02',
	'DBM_Filter::utf8'      => '0.02',
	'DB_File'               => '1.821',
	'Data::Dumper'          => '2.130_02',
	'Devel::DProf'          => '20080331.00',
	'Devel::DProf::dprof::V'=> undef,
	'Devel::InnerPackage'   => '0.3',
	'Devel::PPPort'         => '3.19',
	'Devel::Peek'           => '1.07',
	'Devel::SelfStubber'    => '1.05',
	'Digest'                => '1.16',
	'Digest::MD5'           => '2.51',
	'Digest::SHA'           => '5.50',
	'Digest::base'          => '1.16',
	'Digest::file'          => '1.16',
	'DirHandle'             => '1.04',
	'Dumpvalue'             => '1.16',
	'DynaLoader'            => '1.12',
	'Encode'                => '2.42',
	'Encode::Alias'         => '2.13',
	'Encode::Byte'          => '2.04',
	'Encode::CJKConstants'  => '2.02',
	'Encode::CN'            => '2.03',
	'Encode::CN::HZ'        => '2.05',
	'Encode::Config'        => '2.05',
	'Encode::EBCDIC'        => '2.02',
	'Encode::Encoder'       => '2.01',
	'Encode::Encoding'      => '2.05',
	'Encode::GSM0338'       => '2.01',
	'Encode::Guess'         => '2.04',
	'Encode::JP'            => '2.04',
	'Encode::JP::H2Z'       => '2.02',
	'Encode::JP::JIS7'      => '2.04',
	'Encode::KR'            => '2.03',
	'Encode::KR::2022_KR'   => '2.02',
	'Encode::MIME::Header'  => '2.13',
	'Encode::MIME::Header::ISO_2022_JP'=> '1.03',
	'Encode::MIME::Name'    => '1.01',
	'Encode::Symbol'        => '2.02',
	'Encode::TW'            => '2.03',
	'Encode::Unicode'       => '2.07',
	'Encode::Unicode::UTF7' => '2.05',
	'English'               => '1.04',
	'Env'                   => '1.02',
	'Errno'                 => '1.13',
	'Exporter'              => '5.64_03',
	'Exporter::Heavy'       => '5.64_03',
	'ExtUtils::CBuilder'    => '0.280201',
	'ExtUtils::CBuilder::Base'=> '0.280201',
	'ExtUtils::CBuilder::Platform::Unix'=> '0.280201',
	'ExtUtils::CBuilder::Platform::VMS'=> '0.280201',
	'ExtUtils::CBuilder::Platform::Windows'=> '0.280201',
	'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280201',
	'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.280201',
	'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.280201',
	'ExtUtils::CBuilder::Platform::aix'=> '0.280201',
	'ExtUtils::CBuilder::Platform::cygwin'=> '0.280201',
	'ExtUtils::CBuilder::Platform::darwin'=> '0.280201',
	'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280201',
	'ExtUtils::CBuilder::Platform::os2'=> '0.280201',
	'ExtUtils::Command'     => '1.16',
	'ExtUtils::Command::MM' => '6.57_05',
	'ExtUtils::Constant'    => '0.23',
	'ExtUtils::Constant::Base'=> '0.05',
	'ExtUtils::Constant::ProxySubs'=> '0.08',
	'ExtUtils::Constant::Utils'=> '0.03',
	'ExtUtils::Constant::XS'=> '0.03',
	'ExtUtils::Embed'       => '1.30',
	'ExtUtils::Install'     => '1.56',
	'ExtUtils::Installed'   => '1.999_001',
	'ExtUtils::Liblist'     => '6.57_05',
	'ExtUtils::Liblist::Kid'=> '6.5705',
	'ExtUtils::MM'          => '6.57_05',
	'ExtUtils::MM_AIX'      => '6.57_05',
	'ExtUtils::MM_Any'      => '6.57_05',
	'ExtUtils::MM_BeOS'     => '6.57_05',
	'ExtUtils::MM_Cygwin'   => '6.57_05',
	'ExtUtils::MM_DOS'      => '6.5705',
	'ExtUtils::MM_Darwin'   => '6.57_05',
	'ExtUtils::MM_MacOS'    => '6.5705',
	'ExtUtils::MM_NW5'      => '6.57_05',
	'ExtUtils::MM_OS2'      => '6.57_05',
	'ExtUtils::MM_QNX'      => '6.57_05',
	'ExtUtils::MM_UWIN'     => '6.5705',
	'ExtUtils::MM_Unix'     => '6.57_05',
	'ExtUtils::MM_VMS'      => '6.57_05',
	'ExtUtils::MM_VOS'      => '6.57_05',
	'ExtUtils::MM_Win32'    => '6.57_05',
	'ExtUtils::MM_Win95'    => '6.57_05',
	'ExtUtils::MY'          => '6.5705',
	'ExtUtils::MakeMaker'   => '6.57_05',
	'ExtUtils::MakeMaker::Config'=> '6.57_05',
	'ExtUtils::MakeMaker::YAML'=> '1.44',
	'ExtUtils::Manifest'    => '1.58',
	'ExtUtils::Miniperl'    => undef,
	'ExtUtils::Mkbootstrap' => '6.57_05',
	'ExtUtils::Mksymlists'  => '6.57_05',
	'ExtUtils::Packlist'    => '1.44',
	'ExtUtils::ParseXS'     => '2.2209',
	'ExtUtils::XSSymSet'    => '1.2',
	'ExtUtils::testlib'     => '6.5705',
	'Fatal'                 => '2.1001',
	'Fcntl'                 => '1.11',
	'File::Basename'        => '2.80',
	'File::CheckTree'       => '4.41',
	'File::Compare'         => '1.1006',
	'File::Copy'            => '2.21',
	'File::DosGlob'         => '1.04',
	'File::Fetch'           => '0.32',
	'File::Find'            => '1.19',
	'File::Glob'            => '1.11',
	'File::GlobMapper'      => '1.000',
	'File::Path'            => '2.08_01',
	'File::Spec'            => '3.33',
	'File::Spec::Cygwin'    => '3.33',
	'File::Spec::Epoc'      => '3.33',
	'File::Spec::Functions' => '3.33',
	'File::Spec::Mac'       => '3.34',
	'File::Spec::OS2'       => '3.33',
	'File::Spec::Unix'      => '3.33',
	'File::Spec::VMS'       => '3.34',
	'File::Spec::Win32'     => '3.33',
	'File::Temp'            => '0.22',
	'File::stat'            => '1.05',
	'FileCache'             => '1.08',
	'FileHandle'            => '2.02',
	'Filter::Simple'        => '0.85',
	'Filter::Util::Call'    => '1.08',
	'FindBin'               => '1.50',
	'GDBM_File'             => '1.13',
	'Getopt::Long'          => '2.38',
	'Getopt::Std'           => '1.06',
	'HTTP::Tiny'            => '0.009',
	'Hash::Util'            => '0.10',
	'Hash::Util::FieldHash' => '1.08',
	'I18N::Collate'         => '1.02',
	'I18N::LangTags'        => '0.35_01',
	'I18N::LangTags::Detect'=> '1.04',
	'I18N::LangTags::List'  => '0.35_01',
	'I18N::Langinfo'        => '0.07',
	'IO'                    => '1.25_03',
	'IO::Compress::Adapter::Bzip2'=> '2.033',
	'IO::Compress::Adapter::Deflate'=> '2.033',
	'IO::Compress::Adapter::Identity'=> '2.033',
	'IO::Compress::Base'    => '2.033',
	'IO::Compress::Base::Common'=> '2.033',
	'IO::Compress::Bzip2'   => '2.033',
	'IO::Compress::Deflate' => '2.033',
	'IO::Compress::Gzip'    => '2.033',
	'IO::Compress::Gzip::Constants'=> '2.033',
	'IO::Compress::RawDeflate'=> '2.033',
	'IO::Compress::Zip'     => '2.033',
	'IO::Compress::Zip::Constants'=> '2.033',
	'IO::Compress::Zlib::Constants'=> '2.033',
	'IO::Compress::Zlib::Extra'=> '2.033',
	'IO::Dir'               => '1.07',
	'IO::File'              => '1.14',
	'IO::Handle'            => '1.29',
	'IO::Pipe'              => '1.13',
	'IO::Poll'              => '0.07',
	'IO::Seekable'          => '1.10',
	'IO::Select'            => '1.17',
	'IO::Socket'            => '1.32',
	'IO::Socket::INET'      => '1.31',
	'IO::Socket::UNIX'      => '1.23',
	'IO::Uncompress::Adapter::Bunzip2'=> '2.033',
	'IO::Uncompress::Adapter::Identity'=> '2.033',
	'IO::Uncompress::Adapter::Inflate'=> '2.033',
	'IO::Uncompress::AnyInflate'=> '2.033',
	'IO::Uncompress::AnyUncompress'=> '2.033',
	'IO::Uncompress::Base'  => '2.033',
	'IO::Uncompress::Bunzip2'=> '2.033',
	'IO::Uncompress::Gunzip'=> '2.033',
	'IO::Uncompress::Inflate'=> '2.033',
	'IO::Uncompress::RawInflate'=> '2.033',
	'IO::Uncompress::Unzip' => '2.033',
	'IO::Zlib'              => '1.10',
	'IPC::Cmd'              => '0.68',
	'IPC::Msg'              => '2.03',
	'IPC::Open2'            => '1.03',
	'IPC::Open3'            => '1.09',
	'IPC::Semaphore'        => '2.03',
	'IPC::SharedMem'        => '2.03',
	'IPC::SysV'             => '2.03',
	'JSON::PP'              => '2.27103',
	'JSON::PP::Boolean'     => undef,
	'List::Util'            => '1.23',
	'List::Util::PP'        => '1.23',
	'List::Util::XS'        => '1.23',
	'Locale::Codes'         => '3.15',
	'Locale::Codes::Country'=> '3.15',
	'Locale::Codes::Currency'=> '3.15',
	'Locale::Codes::Language'=> '3.15',
	'Locale::Codes::Script' => '3.15',
	'Locale::Constants'     => '3.15',
	'Locale::Country'       => '3.15',
	'Locale::Currency'      => '3.15',
	'Locale::Language'      => '3.15',
	'Locale::Maketext'      => '1.18',
	'Locale::Maketext::Guts'=> '1.17',
	'Locale::Maketext::GutsLoader'=> '1.17',
	'Locale::Maketext::Simple'=> '0.21',
	'Locale::Script'        => '3.15',
	'Log::Message'          => '0.04',
	'Log::Message::Config'  => '0.04',
	'Log::Message::Handlers'=> '0.04',
	'Log::Message::Item'    => '0.04',
	'Log::Message::Simple'  => '0.08',
	'MIME::Base64'          => '3.13',
	'MIME::QuotedPrint'     => '3.13',
	'Math::BigFloat'        => '1.99_03',
	'Math::BigFloat::Trace' => '0.01',
	'Math::BigInt'          => '1.99_03',
	'Math::BigInt::Calc'    => '1.99_03',
	'Math::BigInt::CalcEmu' => '1.99_02',
	'Math::BigInt::FastCalc'=> '0.24_02',
	'Math::BigInt::Trace'   => '0.01',
	'Math::BigRat'          => '0.26_02',
	'Math::Complex'         => '1.56',
	'Math::Trig'            => '1.2',
	'Memoize'               => '1.02',
	'Memoize::AnyDBM_File'  => '1.02',
	'Memoize::Expire'       => '1.02',
	'Memoize::ExpireFile'   => '1.02',
	'Memoize::ExpireTest'   => '1.02',
	'Memoize::NDBM_File'    => '1.02',
	'Memoize::SDBM_File'    => '1.02',
	'Memoize::Storable'     => '1.02',
	'Module::Build'         => '0.3607',
	'Module::Build::Base'   => '0.3607',
	'Module::Build::Compat' => '0.3607',
	'Module::Build::Config' => '0.3607',
	'Module::Build::ConfigData'=> undef,
	'Module::Build::Cookbook'=> '0.3607',
	'Module::Build::Dumper' => '0.3607',
	'Module::Build::ModuleInfo'=> '0.3607',
	'Module::Build::Notes'  => '0.3607',
	'Module::Build::PPMMaker'=> '0.3607',
	'Module::Build::Platform::Amiga'=> '0.3607',
	'Module::Build::Platform::Default'=> '0.3607',
	'Module::Build::Platform::EBCDIC'=> '0.3607',
	'Module::Build::Platform::MPEiX'=> '0.3607',
	'Module::Build::Platform::MacOS'=> '0.3607',
	'Module::Build::Platform::RiscOS'=> '0.3607',
	'Module::Build::Platform::Unix'=> '0.3607',
	'Module::Build::Platform::VMS'=> '0.3607',
	'Module::Build::Platform::VOS'=> '0.3607',
	'Module::Build::Platform::Windows'=> '0.3607',
	'Module::Build::Platform::aix'=> '0.3607',
	'Module::Build::Platform::cygwin'=> '0.3607',
	'Module::Build::Platform::darwin'=> '0.3607',
	'Module::Build::Platform::os2'=> '0.3607',
	'Module::Build::PodParser'=> '0.3607',
	'Module::Build::Version'=> '0.77',
	'Module::Build::YAML'   => '1.40',
	'Module::CoreList'      => '2.42_01',
	'Module::Load'          => '0.18',
	'Module::Load::Conditional'=> '0.40',
	'Module::Loaded'        => '0.06',
	'Module::Metadata'      => '1.000003',
	'Module::Pluggable'     => '3.9',
	'Module::Pluggable::Object'=> '3.9',
	'Moped::Msg'            => '0.01',
	'NDBM_File'             => '1.11',
	'NEXT'                  => '0.65',
	'Net::Cmd'              => '2.29',
	'Net::Config'           => '1.11',
	'Net::Domain'           => '2.20',
	'Net::FTP'              => '2.77',
	'Net::FTP::A'           => '1.18',
	'Net::FTP::E'           => '0.01',
	'Net::FTP::I'           => '1.12',
	'Net::FTP::L'           => '0.01',
	'Net::FTP::dataconn'    => '0.11',
	'Net::NNTP'             => '2.24',
	'Net::Netrc'            => '2.12',
	'Net::POP3'             => '2.29',
	'Net::Ping'             => '2.38',
	'Net::SMTP'             => '2.31',
	'Net::Time'             => '2.10',
	'Net::hostent'          => '1.01',
	'Net::netent'           => '1.00',
	'Net::protoent'         => '1.00',
	'Net::servent'          => '1.01',
	'O'                     => '1.01',
	'ODBM_File'             => '1.10',
	'Object::Accessor'      => '0.38',
	'Opcode'                => '1.18',
	'POSIX'                 => '1.24',
	'Package::Constants'    => '0.02',
	'Params::Check'         => '0.28',
	'Parse::CPAN::Meta'     => '1.40',
	'Perl::OSType'          => '1.002',
	'PerlIO'                => '1.06',
	'PerlIO::encoding'      => '0.14',
	'PerlIO::scalar'        => '0.11',
	'PerlIO::via'           => '0.11',
	'PerlIO::via::QuotedPrint'=> '0.06',
	'Pod::Checker'          => '1.45',
	'Pod::Escapes'          => '1.04',
	'Pod::Find'             => '1.35',
	'Pod::Functions'        => '1.04',
	'Pod::Html'             => '1.09',
	'Pod::InputObjects'     => '1.31',
	'Pod::LaTeX'            => '0.59',
	'Pod::Man'              => '2.25',
	'Pod::ParseLink'        => '1.10',
	'Pod::ParseUtils'       => '1.36',
	'Pod::Parser'           => '1.37',
	'Pod::Perldoc'          => '3.15_03',
	'Pod::Perldoc::BaseTo'  => undef,
	'Pod::Perldoc::GetOptsOO'=> undef,
	'Pod::Perldoc::ToChecker'=> undef,
	'Pod::Perldoc::ToMan'   => undef,
	'Pod::Perldoc::ToNroff' => undef,
	'Pod::Perldoc::ToPod'   => undef,
	'Pod::Perldoc::ToRtf'   => undef,
	'Pod::Perldoc::ToText'  => undef,
	'Pod::Perldoc::ToTk'    => undef,
	'Pod::Perldoc::ToXml'   => undef,
	'Pod::PlainText'        => '2.04',
	'Pod::Select'           => '1.36',
	'Pod::Simple'           => '3.15',
	'Pod::Simple::BlackBox' => '3.15',
	'Pod::Simple::Checker'  => '3.15',
	'Pod::Simple::Debug'    => '3.15',
	'Pod::Simple::DumpAsText'=> '3.15',
	'Pod::Simple::DumpAsXML'=> '3.15',
	'Pod::Simple::HTML'     => '3.15',
	'Pod::Simple::HTMLBatch'=> '3.15',
	'Pod::Simple::HTMLLegacy'=> '5.01',
	'Pod::Simple::LinkSection'=> '3.15',
	'Pod::Simple::Methody'  => '3.15',
	'Pod::Simple::Progress' => '3.15',
	'Pod::Simple::PullParser'=> '3.15',
	'Pod::Simple::PullParserEndToken'=> '3.15',
	'Pod::Simple::PullParserStartToken'=> '3.15',
	'Pod::Simple::PullParserTextToken'=> '3.15',
	'Pod::Simple::PullParserToken'=> '3.15',
	'Pod::Simple::RTF'      => '3.15',
	'Pod::Simple::Search'   => '3.15',
	'Pod::Simple::SimpleTree'=> '3.15',
	'Pod::Simple::Text'     => '3.15',
	'Pod::Simple::TextContent'=> '3.15',
	'Pod::Simple::TiedOutFH'=> '3.15',
	'Pod::Simple::Transcode'=> '3.15',
	'Pod::Simple::TranscodeDumb'=> '3.15',
	'Pod::Simple::TranscodeSmart'=> '3.15',
	'Pod::Simple::XHTML'    => '3.15',
	'Pod::Simple::XMLOutStream'=> '3.15',
	'Pod::Text'             => '3.15',
	'Pod::Text::Color'      => '2.06',
	'Pod::Text::Overstrike' => '2.04',
	'Pod::Text::Termcap'    => '2.06',
	'Pod::Usage'            => '1.36',
	'SDBM_File'             => '1.09',
	'Safe'                  => '2.29',
	'Scalar::Util'          => '1.23',
	'Scalar::Util::PP'      => '1.23',
	'Search::Dict'          => '1.03',
	'SelectSaver'           => '1.02',
	'SelfLoader'            => '1.18',
	'Shell'                 => '0.72_01',
	'Socket'                => '1.93',
	'Storable'              => '2.26',
	'Symbol'                => '1.07',
	'Sys::Hostname'         => '1.15',
	'Sys::Syslog'           => '0.27',
	'Sys::Syslog::win32::Win32'=> undef,
	'TAP::Base'             => '3.22',
	'TAP::Formatter::Base'  => '3.22',
	'TAP::Formatter::Color' => '3.22',
	'TAP::Formatter::Console'=> '3.22',
	'TAP::Formatter::Console::ParallelSession'=> '3.22',
	'TAP::Formatter::Console::Session'=> '3.22',
	'TAP::Formatter::File'  => '3.22',
	'TAP::Formatter::File::Session'=> '3.22',
	'TAP::Formatter::Session'=> '3.22',
	'TAP::Harness'          => '3.22',
	'TAP::Object'           => '3.22',
	'TAP::Parser'           => '3.22',
	'TAP::Parser::Aggregator'=> '3.22',
	'TAP::Parser::Grammar'  => '3.22',
	'TAP::Parser::Iterator' => '3.22',
	'TAP::Parser::Iterator::Array'=> '3.22',
	'TAP::Parser::Iterator::Process'=> '3.22',
	'TAP::Parser::Iterator::Stream'=> '3.22',
	'TAP::Parser::IteratorFactory'=> '3.22',
	'TAP::Parser::Multiplexer'=> '3.22',
	'TAP::Parser::Result'   => '3.22',
	'TAP::Parser::Result::Bailout'=> '3.22',
	'TAP::Parser::Result::Comment'=> '3.22',
	'TAP::Parser::Result::Plan'=> '3.22',
	'TAP::Parser::Result::Pragma'=> '3.22',
	'TAP::Parser::Result::Test'=> '3.22',
	'TAP::Parser::Result::Unknown'=> '3.22',
	'TAP::Parser::Result::Version'=> '3.22',
	'TAP::Parser::Result::YAML'=> '3.22',
	'TAP::Parser::ResultFactory'=> '3.22',
	'TAP::Parser::Scheduler'=> '3.22',
	'TAP::Parser::Scheduler::Job'=> '3.22',
	'TAP::Parser::Scheduler::Spinner'=> '3.22',
	'TAP::Parser::Source'   => '3.22',
	'TAP::Parser::SourceHandler'=> '3.22',
	'TAP::Parser::SourceHandler::Executable'=> '3.22',
	'TAP::Parser::SourceHandler::File'=> '3.22',
	'TAP::Parser::SourceHandler::Handle'=> '3.22',
	'TAP::Parser::SourceHandler::Perl'=> '3.22',
	'TAP::Parser::SourceHandler::RawTAP'=> '3.22',
	'TAP::Parser::Utils'    => '3.22',
	'TAP::Parser::YAMLish::Reader'=> '3.22',
	'TAP::Parser::YAMLish::Writer'=> '3.22',
	'Term::ANSIColor'       => '3.00',
	'Term::Cap'             => '1.12',
	'Term::Complete'        => '1.402',
	'Term::ReadLine'        => '1.07',
	'Term::UI'              => '0.24',
	'Term::UI::History'     => undef,
	'Test'                  => '1.25_02',
	'Test::Builder'         => '0.97_01',
	'Test::Builder::Module' => '0.97_01',
	'Test::Builder::Tester' => '1.21_01',
	'Test::Builder::Tester::Color'=> '1.21_01',
	'Test::Harness'         => '3.22',
	'Test::More'            => '0.97_01',
	'Test::Simple'          => '0.97_01',
	'Text::Abbrev'          => '1.01',
	'Text::Balanced'        => '2.02',
	'Text::ParseWords'      => '3.27',
	'Text::Soundex'         => '3.03_01',
	'Text::Tabs'            => '2009.0305',
	'Text::Wrap'            => '2009.0305',
	'Thread'                => '3.02',
	'Thread::Queue'         => '2.12',
	'Thread::Semaphore'     => '2.12',
	'Tie::Array'            => '1.04',
	'Tie::File'             => '0.97_02',
	'Tie::Handle'           => '4.2',
	'Tie::Hash'             => '1.04',
	'Tie::Hash::NamedCapture'=> '0.07',
	'Tie::Memoize'          => '1.1',
	'Tie::RefHash'          => '1.38',
	'Tie::Scalar'           => '1.02',
	'Tie::StdHandle'        => '4.2',
	'Tie::SubstrHash'       => '1.00',
	'Time::HiRes'           => '1.9721_01',
	'Time::Local'           => '1.2000',
	'Time::Piece'           => '1.20_01',
	'Time::Piece::Seconds'  => undef,
	'Time::Seconds'         => undef,
	'Time::gmtime'          => '1.03',
	'Time::localtime'       => '1.02',
	'Time::tm'              => '1.00',
	'UNIVERSAL'             => '1.08',
	'Unicode'               => '6.0.0',
	'Unicode::Collate'      => '0.6801',
	'Unicode::Collate::CJK::Big5'=> '0.65',
	'Unicode::Collate::CJK::GB2312'=> '0.65',
	'Unicode::Collate::CJK::JISX0208'=> '0.64',
	'Unicode::Collate::CJK::Korean'=> '0.66',
	'Unicode::Collate::CJK::Pinyin'=> '0.65',
	'Unicode::Collate::CJK::Stroke'=> '0.65',
	'Unicode::Collate::Locale'=> '0.68',
	'Unicode::Normalize'    => '1.10',
	'Unicode::UCD'          => '0.30',
	'User::grent'           => '1.01',
	'User::pwent'           => '1.00',
	'VMS::DCLsym'           => '1.04',
	'VMS::Filespec'         => '1.12',
	'VMS::Stdio'            => '2.4',
	'Win32'                 => '0.44',
	'Win32API::File'        => '0.1101',
	'Win32API::File::ExtUtils::Myconst2perl'=> '1',
	'Win32CORE'             => '0.02',
	'XS::APItest'           => '0.27',
	'XS::Typemap'           => '0.04',
	'XSLoader'              => '0.13',
	'XSLoader::XSLoader'    => '0.13',
	'attributes'            => '0.13',
	'autodie'               => '2.1001',
	'autodie::exception'    => '2.1001',
	'autodie::exception::system'=> '2.1001',
	'autodie::hints'        => '2.1001',
	'autouse'               => '1.06',
	'base'                  => '2.15',
	'bigint'                => '0.26',
	'bignum'                => '0.26',
	'bigrat'                => '0.26',
	'blib'                  => '1.06',
	'bytes'                 => '1.04',
	'charnames'             => '1.18',
	'constant'              => '1.20',
	'deprecate'             => '0.01',
	'diagnostics'           => '1.21',
	'encoding'              => '2.6_01',
	'encoding::warnings'    => '0.11',
	'feature'               => '1.19',
	'fields'                => '2.15',
	'filetest'              => '1.02',
	'if'                    => '0.0601',
	'inc::latest'           => '0.3607',
	'integer'               => '1.00',
	'less'                  => '0.03',
	'lib'                   => '0.63',
	'locale'                => '1.00',
	'mro'                   => '1.06',
	'open'                  => '1.08',
	'ops'                   => '1.02',
	'overload'              => '1.12',
	'overload::numbers'     => undef,
	'overloading'           => '0.01',
	'parent'                => '0.224',
	're'                    => '0.16',
	'sigtrap'               => '1.05',
	'sort'                  => '2.01',
	'strict'                => '1.04',
	'subs'                  => '1.00',
	'threads'               => '1.83',
	'threads::shared'       => '1.36',
	'utf8'                  => '1.08',
	'vars'                  => '1.02',
	'version'               => '0.88',
	'vmsish'                => '1.02',
	'warnings'              => '1.11',
	'warnings::register'    => '1.02',
    },
    5.013010 => {
	'AnyDBM_File'           => '1.00',
	'App::Cpan'             => '1.5701',
	'App::Prove'            => '3.22',
	'App::Prove::State'     => '3.22',
	'App::Prove::State::Result'=> '3.22',
	'App::Prove::State::Result::Test'=> '3.22',
	'Archive::Extract'      => '0.48',
	'Archive::Tar'          => '1.76',
	'Archive::Tar::Constant'=> '1.76',
	'Archive::Tar::File'    => '1.76',
	'Attribute::Handlers'   => '0.89',
	'AutoLoader'            => '5.71',
	'AutoSplit'             => '1.06',
	'B'                     => '1.28',
	'B::Concise'            => '0.83',
	'B::Debug'              => '1.16',
	'B::Deparse'            => '1.03',
	'B::Lint'               => '1.13',
	'B::Lint::Debug'        => '1.12',
	'B::Showlex'            => '1.03',
	'B::Terse'              => '1.05',
	'B::Xref'               => '1.02',
	'Benchmark'             => '1.12',
	'CGI'                   => '3.52',
	'CGI::Apache'           => '1.01',
	'CGI::Carp'             => '3.51',
	'CGI::Cookie'           => '1.30',
	'CGI::Fast'             => '1.08',
	'CGI::Pretty'           => '3.46',
	'CGI::Push'             => '1.05',
	'CGI::Switch'           => '1.01',
	'CGI::Util'             => '3.51',
	'CPAN'                  => '1.94_65',
	'CPAN::Author'          => '5.5',
	'CPAN::Bundle'          => '5.5',
	'CPAN::CacheMgr'        => '5.5',
	'CPAN::Complete'        => '5.5',
	'CPAN::Debug'           => '5.5001',
	'CPAN::DeferredCode'    => '5.50',
	'CPAN::Distribution'    => '1.9601',
	'CPAN::Distroprefs'     => '6',
	'CPAN::Distrostatus'    => '5.5',
	'CPAN::Exception::RecursiveDependency'=> '5.5',
	'CPAN::Exception::blocked_urllist'=> '1.0',
	'CPAN::Exception::yaml_not_installed'=> '5.5',
	'CPAN::FTP'             => '5.5005',
	'CPAN::FTP::netrc'      => '1.01',
	'CPAN::FirstTime'       => '5.5303',
	'CPAN::HTTP::Client'    => '1.94',
	'CPAN::HTTP::Credentials'=> '1.94',
	'CPAN::HandleConfig'    => '5.5003',
	'CPAN::Index'           => '1.94',
	'CPAN::InfoObj'         => '5.5',
	'CPAN::Kwalify'         => '5.50',
	'CPAN::LWP::UserAgent'  => '1.94',
	'CPAN::Meta'            => '2.110440',
	'CPAN::Meta::Converter' => '2.110440',
	'CPAN::Meta::Feature'   => '2.110440',
	'CPAN::Meta::History'   => '2.110440',
	'CPAN::Meta::Prereqs'   => '2.110440',
	'CPAN::Meta::Spec'      => '2.110440',
	'CPAN::Meta::Validator' => '2.110440',
	'CPAN::Meta::YAML'      => '0.003',
	'CPAN::Mirrors'         => '1.77',
	'CPAN::Module'          => '5.5',
	'CPAN::Nox'             => '5.50',
	'CPAN::Prompt'          => '5.5',
	'CPAN::Queue'           => '5.5001',
	'CPAN::Shell'           => '5.5002',
	'CPAN::Tarzip'          => '5.5011',
	'CPAN::URL'             => '5.5',
	'CPAN::Version'         => '5.5001',
	'CPANPLUS'              => '0.9101',
	'CPANPLUS::Backend'     => undef,
	'CPANPLUS::Backend::RV' => undef,
	'CPANPLUS::Config'      => undef,
	'CPANPLUS::Configure'   => undef,
	'CPANPLUS::Configure::Setup'=> undef,
	'CPANPLUS::Dist'        => undef,
	'CPANPLUS::Dist::Autobundle'=> undef,
	'CPANPLUS::Dist::Base'  => undef,
	'CPANPLUS::Dist::Build' => '0.52',
	'CPANPLUS::Dist::Build::Constants'=> '0.52',
	'CPANPLUS::Dist::MM'    => undef,
	'CPANPLUS::Dist::Sample'=> undef,
	'CPANPLUS::Error'       => undef,
	'CPANPLUS::Internals'   => '0.9101',
	'CPANPLUS::Internals::Constants'=> undef,
	'CPANPLUS::Internals::Constants::Report'=> undef,
	'CPANPLUS::Internals::Extract'=> undef,
	'CPANPLUS::Internals::Fetch'=> undef,
	'CPANPLUS::Internals::Report'=> undef,
	'CPANPLUS::Internals::Search'=> undef,
	'CPANPLUS::Internals::Source'=> undef,
	'CPANPLUS::Internals::Source::Memory'=> undef,
	'CPANPLUS::Internals::Source::SQLite'=> undef,
	'CPANPLUS::Internals::Source::SQLite::Tie'=> undef,
	'CPANPLUS::Internals::Utils'=> undef,
	'CPANPLUS::Internals::Utils::Autoflush'=> undef,
	'CPANPLUS::Module'      => undef,
	'CPANPLUS::Module::Author'=> undef,
	'CPANPLUS::Module::Author::Fake'=> undef,
	'CPANPLUS::Module::Checksums'=> undef,
	'CPANPLUS::Module::Fake'=> undef,
	'CPANPLUS::Module::Signature'=> undef,
	'CPANPLUS::Selfupdate'  => undef,
	'CPANPLUS::Shell'       => undef,
	'CPANPLUS::Shell::Classic'=> '0.0562',
	'CPANPLUS::Shell::Default'=> '0.9101',
	'CPANPLUS::Shell::Default::Plugins::CustomSource'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Remote'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Source'=> undef,
	'Carp'                  => '1.20',
	'Carp::Heavy'           => '1.20',
	'Class::Struct'         => '0.63',
	'Compress::Raw::Bzip2'  => '2.033',
	'Compress::Raw::Zlib'   => '2.033',
	'Compress::Zlib'        => '2.033',
	'Config'                => undef,
	'Config::Extensions'    => '0.01',
	'Cwd'                   => '3.37',
	'DB'                    => '1.03',
	'DBM_Filter'            => '0.04',
	'DBM_Filter::compress'  => '0.02',
	'DBM_Filter::encode'    => '0.02',
	'DBM_Filter::int32'     => '0.02',
	'DBM_Filter::null'      => '0.02',
	'DBM_Filter::utf8'      => '0.02',
	'DB_File'               => '1.821',
	'Data::Dumper'          => '2.130_02',
	'Devel::DProf'          => '20110217.00',
	'Devel::DProf::dprof::V'=> undef,
	'Devel::InnerPackage'   => '0.3',
	'Devel::PPPort'         => '3.19',
	'Devel::Peek'           => '1.07',
	'Devel::SelfStubber'    => '1.05',
	'Digest'                => '1.16',
	'Digest::MD5'           => '2.51',
	'Digest::SHA'           => '5.50',
	'Digest::base'          => '1.16',
	'Digest::file'          => '1.16',
	'DirHandle'             => '1.04',
	'Dumpvalue'             => '1.16',
	'DynaLoader'            => '1.13',
	'Encode'                => '2.42',
	'Encode::Alias'         => '2.13',
	'Encode::Byte'          => '2.04',
	'Encode::CJKConstants'  => '2.02',
	'Encode::CN'            => '2.03',
	'Encode::CN::HZ'        => '2.05',
	'Encode::Config'        => '2.05',
	'Encode::EBCDIC'        => '2.02',
	'Encode::Encoder'       => '2.01',
	'Encode::Encoding'      => '2.05',
	'Encode::GSM0338'       => '2.01',
	'Encode::Guess'         => '2.04',
	'Encode::JP'            => '2.04',
	'Encode::JP::H2Z'       => '2.02',
	'Encode::JP::JIS7'      => '2.04',
	'Encode::KR'            => '2.03',
	'Encode::KR::2022_KR'   => '2.02',
	'Encode::MIME::Header'  => '2.13',
	'Encode::MIME::Header::ISO_2022_JP'=> '1.03',
	'Encode::MIME::Name'    => '1.01',
	'Encode::Symbol'        => '2.02',
	'Encode::TW'            => '2.03',
	'Encode::Unicode'       => '2.07',
	'Encode::Unicode::UTF7' => '2.05',
	'English'               => '1.04',
	'Env'                   => '1.02',
	'Errno'                 => '1.13',
	'Exporter'              => '5.64_03',
	'Exporter::Heavy'       => '5.64_03',
	'ExtUtils::CBuilder'    => '0.280202',
	'ExtUtils::CBuilder::Base'=> '0.280202',
	'ExtUtils::CBuilder::Platform::Unix'=> '0.280202',
	'ExtUtils::CBuilder::Platform::VMS'=> '0.280202',
	'ExtUtils::CBuilder::Platform::Windows'=> '0.280202',
	'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280202',
	'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.280202',
	'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.280202',
	'ExtUtils::CBuilder::Platform::aix'=> '0.280202',
	'ExtUtils::CBuilder::Platform::cygwin'=> '0.280202',
	'ExtUtils::CBuilder::Platform::darwin'=> '0.280202',
	'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280202',
	'ExtUtils::CBuilder::Platform::os2'=> '0.280202',
	'ExtUtils::Command'     => '1.16',
	'ExtUtils::Command::MM' => '6.57_05',
	'ExtUtils::Constant'    => '0.23',
	'ExtUtils::Constant::Base'=> '0.05',
	'ExtUtils::Constant::ProxySubs'=> '0.08',
	'ExtUtils::Constant::Utils'=> '0.03',
	'ExtUtils::Constant::XS'=> '0.03',
	'ExtUtils::Embed'       => '1.30',
	'ExtUtils::Install'     => '1.56',
	'ExtUtils::Installed'   => '1.999_001',
	'ExtUtils::Liblist'     => '6.57_05',
	'ExtUtils::Liblist::Kid'=> '6.5705',
	'ExtUtils::MM'          => '6.57_05',
	'ExtUtils::MM_AIX'      => '6.57_05',
	'ExtUtils::MM_Any'      => '6.57_05',
	'ExtUtils::MM_BeOS'     => '6.57_05',
	'ExtUtils::MM_Cygwin'   => '6.57_05',
	'ExtUtils::MM_DOS'      => '6.5705',
	'ExtUtils::MM_Darwin'   => '6.57_05',
	'ExtUtils::MM_MacOS'    => '6.5705',
	'ExtUtils::MM_NW5'      => '6.57_05',
	'ExtUtils::MM_OS2'      => '6.57_05',
	'ExtUtils::MM_QNX'      => '6.57_05',
	'ExtUtils::MM_UWIN'     => '6.5705',
	'ExtUtils::MM_Unix'     => '6.57_05',
	'ExtUtils::MM_VMS'      => '6.57_05',
	'ExtUtils::MM_VOS'      => '6.57_05',
	'ExtUtils::MM_Win32'    => '6.57_05',
	'ExtUtils::MM_Win95'    => '6.57_05',
	'ExtUtils::MY'          => '6.5705',
	'ExtUtils::MakeMaker'   => '6.57_05',
	'ExtUtils::MakeMaker::Config'=> '6.57_05',
	'ExtUtils::MakeMaker::YAML'=> '1.44',
	'ExtUtils::Manifest'    => '1.58',
	'ExtUtils::Miniperl'    => undef,
	'ExtUtils::Mkbootstrap' => '6.57_05',
	'ExtUtils::Mksymlists'  => '6.57_05',
	'ExtUtils::Packlist'    => '1.44',
	'ExtUtils::ParseXS'     => '2.2209',
	'ExtUtils::XSSymSet'    => '1.2',
	'ExtUtils::testlib'     => '6.5705',
	'Fatal'                 => '2.1001',
	'Fcntl'                 => '1.11',
	'File::Basename'        => '2.80',
	'File::CheckTree'       => '4.41',
	'File::Compare'         => '1.1006',
	'File::Copy'            => '2.22',
	'File::DosGlob'         => '1.04',
	'File::Fetch'           => '0.32',
	'File::Find'            => '1.19',
	'File::Glob'            => '1.11',
	'File::GlobMapper'      => '1.000',
	'File::Path'            => '2.08_01',
	'File::Spec'            => '3.33',
	'File::Spec::Cygwin'    => '3.33',
	'File::Spec::Epoc'      => '3.33',
	'File::Spec::Functions' => '3.33',
	'File::Spec::Mac'       => '3.34',
	'File::Spec::OS2'       => '3.33',
	'File::Spec::Unix'      => '3.33',
	'File::Spec::VMS'       => '3.34',
	'File::Spec::Win32'     => '3.33',
	'File::Temp'            => '0.22',
	'File::stat'            => '1.05',
	'FileCache'             => '1.08',
	'FileHandle'            => '2.02',
	'Filter::Simple'        => '0.86',
	'Filter::Util::Call'    => '1.08',
	'FindBin'               => '1.50',
	'GDBM_File'             => '1.13',
	'Getopt::Long'          => '2.38',
	'Getopt::Std'           => '1.06',
	'HTTP::Tiny'            => '0.010',
	'Hash::Util'            => '0.10',
	'Hash::Util::FieldHash' => '1.08',
	'I18N::Collate'         => '1.02',
	'I18N::LangTags'        => '0.35_01',
	'I18N::LangTags::Detect'=> '1.05',
	'I18N::LangTags::List'  => '0.35_01',
	'I18N::Langinfo'        => '0.07',
	'IO'                    => '1.25_03',
	'IO::Compress::Adapter::Bzip2'=> '2.033',
	'IO::Compress::Adapter::Deflate'=> '2.033',
	'IO::Compress::Adapter::Identity'=> '2.033',
	'IO::Compress::Base'    => '2.033',
	'IO::Compress::Base::Common'=> '2.033',
	'IO::Compress::Bzip2'   => '2.033',
	'IO::Compress::Deflate' => '2.033',
	'IO::Compress::Gzip'    => '2.033',
	'IO::Compress::Gzip::Constants'=> '2.033',
	'IO::Compress::RawDeflate'=> '2.033',
	'IO::Compress::Zip'     => '2.033',
	'IO::Compress::Zip::Constants'=> '2.033',
	'IO::Compress::Zlib::Constants'=> '2.033',
	'IO::Compress::Zlib::Extra'=> '2.033',
	'IO::Dir'               => '1.07',
	'IO::File'              => '1.14',
	'IO::Handle'            => '1.29',
	'IO::Pipe'              => '1.13',
	'IO::Poll'              => '0.07',
	'IO::Seekable'          => '1.10',
	'IO::Select'            => '1.18',
	'IO::Socket'            => '1.32',
	'IO::Socket::INET'      => '1.31',
	'IO::Socket::UNIX'      => '1.23',
	'IO::Uncompress::Adapter::Bunzip2'=> '2.033',
	'IO::Uncompress::Adapter::Identity'=> '2.033',
	'IO::Uncompress::Adapter::Inflate'=> '2.033',
	'IO::Uncompress::AnyInflate'=> '2.033',
	'IO::Uncompress::AnyUncompress'=> '2.033',
	'IO::Uncompress::Base'  => '2.033',
	'IO::Uncompress::Bunzip2'=> '2.033',
	'IO::Uncompress::Gunzip'=> '2.033',
	'IO::Uncompress::Inflate'=> '2.033',
	'IO::Uncompress::RawInflate'=> '2.033',
	'IO::Uncompress::Unzip' => '2.033',
	'IO::Zlib'              => '1.10',
	'IPC::Cmd'              => '0.70',
	'IPC::Msg'              => '2.03',
	'IPC::Open2'            => '1.03',
	'IPC::Open3'            => '1.09',
	'IPC::Semaphore'        => '2.03',
	'IPC::SharedMem'        => '2.03',
	'IPC::SysV'             => '2.03',
	'JSON::PP'              => '2.27103',
	'JSON::PP::Boolean'     => undef,
	'List::Util'            => '1.23',
	'List::Util::PP'        => '1.23',
	'List::Util::XS'        => '1.23',
	'Locale::Codes'         => '3.15',
	'Locale::Codes::Country'=> '3.15',
	'Locale::Codes::Currency'=> '3.15',
	'Locale::Codes::Language'=> '3.15',
	'Locale::Codes::Script' => '3.15',
	'Locale::Constants'     => '3.15',
	'Locale::Country'       => '3.15',
	'Locale::Currency'      => '3.15',
	'Locale::Language'      => '3.15',
	'Locale::Maketext'      => '1.19',
	'Locale::Maketext::Guts'=> '1.17',
	'Locale::Maketext::GutsLoader'=> '1.17',
	'Locale::Maketext::Simple'=> '0.21',
	'Locale::Script'        => '3.15',
	'Log::Message'          => '0.04',
	'Log::Message::Config'  => '0.04',
	'Log::Message::Handlers'=> '0.04',
	'Log::Message::Item'    => '0.04',
	'Log::Message::Simple'  => '0.08',
	'MIME::Base64'          => '3.13',
	'MIME::QuotedPrint'     => '3.13',
	'Math::BigFloat'        => '1.992',
	'Math::BigFloat::Trace' => '0.01',
	'Math::BigInt'          => '1.992',
	'Math::BigInt::Calc'    => '1.992',
	'Math::BigInt::CalcEmu' => '1.992',
	'Math::BigInt::FastCalc'=> '0.24_02',
	'Math::BigInt::Trace'   => '0.01',
	'Math::BigRat'          => '0.26_02',
	'Math::Complex'         => '1.56',
	'Math::Trig'            => '1.2',
	'Memoize'               => '1.02',
	'Memoize::AnyDBM_File'  => '1.02',
	'Memoize::Expire'       => '1.02',
	'Memoize::ExpireFile'   => '1.02',
	'Memoize::ExpireTest'   => '1.02',
	'Memoize::NDBM_File'    => '1.02',
	'Memoize::SDBM_File'    => '1.02',
	'Memoize::Storable'     => '1.02',
	'Module::Build'         => '0.37_05',
	'Module::Build::Base'   => '0.37_05',
	'Module::Build::Compat' => '0.37_05',
	'Module::Build::Config' => '0.37_05',
	'Module::Build::ConfigData'=> undef,
	'Module::Build::Cookbook'=> '0.37_05',
	'Module::Build::Dumper' => '0.37_05',
	'Module::Build::ModuleInfo'=> '0.37_05',
	'Module::Build::Notes'  => '0.37_05',
	'Module::Build::PPMMaker'=> '0.37_05',
	'Module::Build::Platform::Amiga'=> '0.37_05',
	'Module::Build::Platform::Default'=> '0.37_05',
	'Module::Build::Platform::EBCDIC'=> '0.37_05',
	'Module::Build::Platform::MPEiX'=> '0.37_05',
	'Module::Build::Platform::MacOS'=> '0.37_05',
	'Module::Build::Platform::RiscOS'=> '0.37_05',
	'Module::Build::Platform::Unix'=> '0.37_05',
	'Module::Build::Platform::VMS'=> '0.37_05',
	'Module::Build::Platform::VOS'=> '0.37_05',
	'Module::Build::Platform::Windows'=> '0.37_05',
	'Module::Build::Platform::aix'=> '0.37_05',
	'Module::Build::Platform::cygwin'=> '0.37_05',
	'Module::Build::Platform::darwin'=> '0.37_05',
	'Module::Build::Platform::os2'=> '0.37_05',
	'Module::Build::PodParser'=> '0.37_05',
	'Module::Build::Version'=> '0.87',
	'Module::Build::YAML'   => '1.41',
	'Module::CoreList'      => '2.45',
	'Module::Load'          => '0.18',
	'Module::Load::Conditional'=> '0.44',
	'Module::Loaded'        => '0.06',
	'Module::Metadata'      => '1.000004',
	'Module::Pluggable'     => '3.9',
	'Module::Pluggable::Object'=> '3.9',
	'Moped::Msg'            => '0.01',
	'NDBM_File'             => '1.11',
	'NEXT'                  => '0.65',
	'Net::Cmd'              => '2.29',
	'Net::Config'           => '1.11',
	'Net::Domain'           => '2.20',
	'Net::FTP'              => '2.77',
	'Net::FTP::A'           => '1.18',
	'Net::FTP::E'           => '0.01',
	'Net::FTP::I'           => '1.12',
	'Net::FTP::L'           => '0.01',
	'Net::FTP::dataconn'    => '0.11',
	'Net::NNTP'             => '2.24',
	'Net::Netrc'            => '2.12',
	'Net::POP3'             => '2.29',
	'Net::Ping'             => '2.38',
	'Net::SMTP'             => '2.31',
	'Net::Time'             => '2.10',
	'Net::hostent'          => '1.01',
	'Net::netent'           => '1.00',
	'Net::protoent'         => '1.00',
	'Net::servent'          => '1.01',
	'O'                     => '1.01',
	'ODBM_File'             => '1.10',
	'Object::Accessor'      => '0.38',
	'Opcode'                => '1.18',
	'POSIX'                 => '1.24',
	'Package::Constants'    => '0.02',
	'Params::Check'         => '0.28',
	'Parse::CPAN::Meta'     => '1.4401',
	'Perl::OSType'          => '1.002',
	'PerlIO'                => '1.06',
	'PerlIO::encoding'      => '0.14',
	'PerlIO::scalar'        => '0.11',
	'PerlIO::via'           => '0.11',
	'PerlIO::via::QuotedPrint'=> '0.06',
	'Pod::Checker'          => '1.45',
	'Pod::Escapes'          => '1.04',
	'Pod::Find'             => '1.35',
	'Pod::Functions'        => '1.04',
	'Pod::Html'             => '1.1',
	'Pod::InputObjects'     => '1.31',
	'Pod::LaTeX'            => '0.59',
	'Pod::Man'              => '2.25',
	'Pod::ParseLink'        => '1.10',
	'Pod::ParseUtils'       => '1.36',
	'Pod::Parser'           => '1.37',
	'Pod::Perldoc'          => '3.15_03',
	'Pod::Perldoc::BaseTo'  => undef,
	'Pod::Perldoc::GetOptsOO'=> undef,
	'Pod::Perldoc::ToChecker'=> undef,
	'Pod::Perldoc::ToMan'   => undef,
	'Pod::Perldoc::ToNroff' => undef,
	'Pod::Perldoc::ToPod'   => undef,
	'Pod::Perldoc::ToRtf'   => undef,
	'Pod::Perldoc::ToText'  => undef,
	'Pod::Perldoc::ToTk'    => undef,
	'Pod::Perldoc::ToXml'   => undef,
	'Pod::PlainText'        => '2.04',
	'Pod::Select'           => '1.36',
	'Pod::Simple'           => '3.15',
	'Pod::Simple::BlackBox' => '3.15',
	'Pod::Simple::Checker'  => '3.15',
	'Pod::Simple::Debug'    => '3.15',
	'Pod::Simple::DumpAsText'=> '3.15',
	'Pod::Simple::DumpAsXML'=> '3.15',
	'Pod::Simple::HTML'     => '3.15',
	'Pod::Simple::HTMLBatch'=> '3.15',
	'Pod::Simple::HTMLLegacy'=> '5.01',
	'Pod::Simple::LinkSection'=> '3.15',
	'Pod::Simple::Methody'  => '3.15',
	'Pod::Simple::Progress' => '3.15',
	'Pod::Simple::PullParser'=> '3.15',
	'Pod::Simple::PullParserEndToken'=> '3.15',
	'Pod::Simple::PullParserStartToken'=> '3.15',
	'Pod::Simple::PullParserTextToken'=> '3.15',
	'Pod::Simple::PullParserToken'=> '3.15',
	'Pod::Simple::RTF'      => '3.15',
	'Pod::Simple::Search'   => '3.15',
	'Pod::Simple::SimpleTree'=> '3.15',
	'Pod::Simple::Text'     => '3.15',
	'Pod::Simple::TextContent'=> '3.15',
	'Pod::Simple::TiedOutFH'=> '3.15',
	'Pod::Simple::Transcode'=> '3.15',
	'Pod::Simple::TranscodeDumb'=> '3.15',
	'Pod::Simple::TranscodeSmart'=> '3.15',
	'Pod::Simple::XHTML'    => '3.15',
	'Pod::Simple::XMLOutStream'=> '3.15',
	'Pod::Text'             => '3.15',
	'Pod::Text::Color'      => '2.06',
	'Pod::Text::Overstrike' => '2.04',
	'Pod::Text::Termcap'    => '2.06',
	'Pod::Usage'            => '1.36',
	'SDBM_File'             => '1.09',
	'Safe'                  => '2.29',
	'Scalar::Util'          => '1.23',
	'Scalar::Util::PP'      => '1.23',
	'Search::Dict'          => '1.03',
	'SelectSaver'           => '1.02',
	'SelfLoader'            => '1.18',
	'Shell'                 => '0.72_01',
	'Socket'                => '1.94',
	'Storable'              => '2.26',
	'Symbol'                => '1.07',
	'Sys::Hostname'         => '1.15',
	'Sys::Syslog'           => '0.27',
	'Sys::Syslog::win32::Win32'=> undef,
	'TAP::Base'             => '3.22',
	'TAP::Formatter::Base'  => '3.22',
	'TAP::Formatter::Color' => '3.22',
	'TAP::Formatter::Console'=> '3.22',
	'TAP::Formatter::Console::ParallelSession'=> '3.22',
	'TAP::Formatter::Console::Session'=> '3.22',
	'TAP::Formatter::File'  => '3.22',
	'TAP::Formatter::File::Session'=> '3.22',
	'TAP::Formatter::Session'=> '3.22',
	'TAP::Harness'          => '3.22',
	'TAP::Object'           => '3.22',
	'TAP::Parser'           => '3.22',
	'TAP::Parser::Aggregator'=> '3.22',
	'TAP::Parser::Grammar'  => '3.22',
	'TAP::Parser::Iterator' => '3.22',
	'TAP::Parser::Iterator::Array'=> '3.22',
	'TAP::Parser::Iterator::Process'=> '3.22',
	'TAP::Parser::Iterator::Stream'=> '3.22',
	'TAP::Parser::IteratorFactory'=> '3.22',
	'TAP::Parser::Multiplexer'=> '3.22',
	'TAP::Parser::Result'   => '3.22',
	'TAP::Parser::Result::Bailout'=> '3.22',
	'TAP::Parser::Result::Comment'=> '3.22',
	'TAP::Parser::Result::Plan'=> '3.22',
	'TAP::Parser::Result::Pragma'=> '3.22',
	'TAP::Parser::Result::Test'=> '3.22',
	'TAP::Parser::Result::Unknown'=> '3.22',
	'TAP::Parser::Result::Version'=> '3.22',
	'TAP::Parser::Result::YAML'=> '3.22',
	'TAP::Parser::ResultFactory'=> '3.22',
	'TAP::Parser::Scheduler'=> '3.22',
	'TAP::Parser::Scheduler::Job'=> '3.22',
	'TAP::Parser::Scheduler::Spinner'=> '3.22',
	'TAP::Parser::Source'   => '3.22',
	'TAP::Parser::SourceHandler'=> '3.22',
	'TAP::Parser::SourceHandler::Executable'=> '3.22',
	'TAP::Parser::SourceHandler::File'=> '3.22',
	'TAP::Parser::SourceHandler::Handle'=> '3.22',
	'TAP::Parser::SourceHandler::Perl'=> '3.22',
	'TAP::Parser::SourceHandler::RawTAP'=> '3.22',
	'TAP::Parser::Utils'    => '3.22',
	'TAP::Parser::YAMLish::Reader'=> '3.22',
	'TAP::Parser::YAMLish::Writer'=> '3.22',
	'Term::ANSIColor'       => '3.00',
	'Term::Cap'             => '1.12',
	'Term::Complete'        => '1.402',
	'Term::ReadLine'        => '1.07',
	'Term::UI'              => '0.26',
	'Term::UI::History'     => undef,
	'Test'                  => '1.25_02',
	'Test::Builder'         => '0.97_01',
	'Test::Builder::Module' => '0.97_01',
	'Test::Builder::Tester' => '1.21_01',
	'Test::Builder::Tester::Color'=> '1.21_01',
	'Test::Harness'         => '3.22',
	'Test::More'            => '0.97_01',
	'Test::Simple'          => '0.97_01',
	'Text::Abbrev'          => '1.01',
	'Text::Balanced'        => '2.02',
	'Text::ParseWords'      => '3.27',
	'Text::Soundex'         => '3.03_01',
	'Text::Tabs'            => '2009.0305',
	'Text::Wrap'            => '2009.0305',
	'Thread'                => '3.02',
	'Thread::Queue'         => '2.12',
	'Thread::Semaphore'     => '2.12',
	'Tie::Array'            => '1.04',
	'Tie::File'             => '0.97_02',
	'Tie::Handle'           => '4.2',
	'Tie::Hash'             => '1.04',
	'Tie::Hash::NamedCapture'=> '0.07',
	'Tie::Memoize'          => '1.1',
	'Tie::RefHash'          => '1.38',
	'Tie::Scalar'           => '1.02',
	'Tie::StdHandle'        => '4.2',
	'Tie::SubstrHash'       => '1.00',
	'Time::HiRes'           => '1.9721_01',
	'Time::Local'           => '1.2000',
	'Time::Piece'           => '1.20_01',
	'Time::Piece::Seconds'  => undef,
	'Time::Seconds'         => undef,
	'Time::gmtime'          => '1.03',
	'Time::localtime'       => '1.02',
	'Time::tm'              => '1.00',
	'UNIVERSAL'             => '1.08',
	'Unicode'               => '6.0.0',
	'Unicode::Collate'      => '0.72',
	'Unicode::Collate::CJK::Big5'=> '0.65',
	'Unicode::Collate::CJK::GB2312'=> '0.65',
	'Unicode::Collate::CJK::JISX0208'=> '0.64',
	'Unicode::Collate::CJK::Korean'=> '0.66',
	'Unicode::Collate::CJK::Pinyin'=> '0.65',
	'Unicode::Collate::CJK::Stroke'=> '0.65',
	'Unicode::Collate::Locale'=> '0.71',
	'Unicode::Normalize'    => '1.10',
	'Unicode::UCD'          => '0.31',
	'User::grent'           => '1.01',
	'User::pwent'           => '1.00',
	'VMS::DCLsym'           => '1.05',
	'VMS::Filespec'         => '1.12',
	'VMS::Stdio'            => '2.4',
	'Version::Requirements' => '0.101020',
	'Win32'                 => '0.44',
	'Win32API::File'        => '0.1101',
	'Win32API::File::ExtUtils::Myconst2perl'=> '1',
	'Win32CORE'             => '0.02',
	'XS::APItest'           => '0.27',
	'XS::Typemap'           => '0.04',
	'XSLoader'              => '0.13',
	'XSLoader::XSLoader'    => '0.13',
	'attributes'            => '0.13',
	'autodie'               => '2.1001',
	'autodie::exception'    => '2.1001',
	'autodie::exception::system'=> '2.1001',
	'autodie::hints'        => '2.1001',
	'autouse'               => '1.06',
	'base'                  => '2.15',
	'bigint'                => '0.26',
	'bignum'                => '0.26',
	'bigrat'                => '0.27',
	'blib'                  => '1.06',
	'bytes'                 => '1.04',
	'charnames'             => '1.18',
	'constant'              => '1.20',
	'deprecate'             => '0.02',
	'diagnostics'           => '1.22',
	'encoding'              => '2.6_01',
	'encoding::warnings'    => '0.11',
	'feature'               => '1.19',
	'fields'                => '2.15',
	'filetest'              => '1.02',
	'if'                    => '0.0601',
	'inc::latest'           => '0.37_05',
	'integer'               => '1.00',
	'less'                  => '0.03',
	'lib'                   => '0.63',
	'locale'                => '1.00',
	'mro'                   => '1.06',
	'open'                  => '1.08',
	'ops'                   => '1.02',
	'overload'              => '1.13',
	'overload::numbers'     => undef,
	'overloading'           => '0.01',
	'parent'                => '0.224',
	're'                    => '0.17',
	'sigtrap'               => '1.05',
	'sort'                  => '2.01',
	'strict'                => '1.04',
	'subs'                  => '1.00',
	'threads'               => '1.83',
	'threads::shared'       => '1.36',
	'utf8'                  => '1.09',
	'vars'                  => '1.02',
	'version'               => '0.88',
	'vmsish'                => '1.02',
	'warnings'              => '1.12',
	'warnings::register'    => '1.02',
    },
    5.013011 => {
	'AnyDBM_File'           => '1.00',
	'App::Cpan'             => '1.5701',
	'App::Prove'            => '3.23',
	'App::Prove::State'     => '3.23',
	'App::Prove::State::Result'=> '3.23',
	'App::Prove::State::Result::Test'=> '3.23',
	'Archive::Extract'      => '0.48',
	'Archive::Tar'          => '1.76',
	'Archive::Tar::Constant'=> '1.76',
	'Archive::Tar::File'    => '1.76',
	'Attribute::Handlers'   => '0.89',
	'AutoLoader'            => '5.71',
	'AutoSplit'             => '1.06',
	'B'                     => '1.29',
	'B::Concise'            => '0.83',
	'B::Debug'              => '1.16',
	'B::Deparse'            => '1.03',
	'B::Lint'               => '1.13',
	'B::Lint::Debug'        => '1.12',
	'B::Showlex'            => '1.03',
	'B::Terse'              => '1.05',
	'B::Xref'               => '1.02',
	'Benchmark'             => '1.12',
	'CGI'                   => '3.52',
	'CGI::Apache'           => '1.01',
	'CGI::Carp'             => '3.51',
	'CGI::Cookie'           => '1.30',
	'CGI::Fast'             => '1.08',
	'CGI::Pretty'           => '3.46',
	'CGI::Push'             => '1.05',
	'CGI::Switch'           => '1.01',
	'CGI::Util'             => '3.51',
	'CPAN'                  => '1.9600',
	'CPAN::Author'          => '5.5001',
	'CPAN::Bundle'          => '5.5',
	'CPAN::CacheMgr'        => '5.5001',
	'CPAN::Complete'        => '5.5',
	'CPAN::Debug'           => '5.5001',
	'CPAN::DeferredCode'    => '5.50',
	'CPAN::Distribution'    => '1.9602',
	'CPAN::Distroprefs'     => '6',
	'CPAN::Distrostatus'    => '5.5',
	'CPAN::Exception::RecursiveDependency'=> '5.5',
	'CPAN::Exception::blocked_urllist'=> '1.001',
	'CPAN::Exception::yaml_not_installed'=> '5.5',
	'CPAN::FTP'             => '5.5005',
	'CPAN::FTP::netrc'      => '1.01',
	'CPAN::FirstTime'       => '5.5303',
	'CPAN::HTTP::Client'    => '1.9600',
	'CPAN::HTTP::Credentials'=> '1.9600',
	'CPAN::HandleConfig'    => '5.5003',
	'CPAN::Index'           => '1.9600',
	'CPAN::InfoObj'         => '5.5',
	'CPAN::Kwalify'         => '5.50',
	'CPAN::LWP::UserAgent'  => '1.9600',
	'CPAN::Meta'            => '2.110440',
	'CPAN::Meta::Converter' => '2.110440',
	'CPAN::Meta::Feature'   => '2.110440',
	'CPAN::Meta::History'   => '2.110440',
	'CPAN::Meta::Prereqs'   => '2.110440',
	'CPAN::Meta::Spec'      => '2.110440',
	'CPAN::Meta::Validator' => '2.110440',
	'CPAN::Meta::YAML'      => '0.003',
	'CPAN::Mirrors'         => '1.9600',
	'CPAN::Module'          => '5.5001',
	'CPAN::Nox'             => '5.50',
	'CPAN::Prompt'          => '5.5',
	'CPAN::Queue'           => '5.5001',
	'CPAN::Shell'           => '5.5002',
	'CPAN::Tarzip'          => '5.5011',
	'CPAN::URL'             => '5.5',
	'CPAN::Version'         => '5.5001',
	'CPANPLUS'              => '0.9103',
	'CPANPLUS::Backend'     => undef,
	'CPANPLUS::Backend::RV' => undef,
	'CPANPLUS::Config'      => undef,
	'CPANPLUS::Configure'   => undef,
	'CPANPLUS::Configure::Setup'=> undef,
	'CPANPLUS::Dist'        => undef,
	'CPANPLUS::Dist::Autobundle'=> undef,
	'CPANPLUS::Dist::Base'  => undef,
	'CPANPLUS::Dist::Build' => '0.54',
	'CPANPLUS::Dist::Build::Constants'=> '0.54',
	'CPANPLUS::Dist::MM'    => undef,
	'CPANPLUS::Dist::Sample'=> undef,
	'CPANPLUS::Error'       => undef,
	'CPANPLUS::Internals'   => '0.9103',
	'CPANPLUS::Internals::Constants'=> undef,
	'CPANPLUS::Internals::Constants::Report'=> undef,
	'CPANPLUS::Internals::Extract'=> undef,
	'CPANPLUS::Internals::Fetch'=> undef,
	'CPANPLUS::Internals::Report'=> undef,
	'CPANPLUS::Internals::Search'=> undef,
	'CPANPLUS::Internals::Source'=> undef,
	'CPANPLUS::Internals::Source::Memory'=> undef,
	'CPANPLUS::Internals::Source::SQLite'=> undef,
	'CPANPLUS::Internals::Source::SQLite::Tie'=> undef,
	'CPANPLUS::Internals::Utils'=> undef,
	'CPANPLUS::Internals::Utils::Autoflush'=> undef,
	'CPANPLUS::Module'      => undef,
	'CPANPLUS::Module::Author'=> undef,
	'CPANPLUS::Module::Author::Fake'=> undef,
	'CPANPLUS::Module::Checksums'=> undef,
	'CPANPLUS::Module::Fake'=> undef,
	'CPANPLUS::Module::Signature'=> undef,
	'CPANPLUS::Selfupdate'  => undef,
	'CPANPLUS::Shell'       => undef,
	'CPANPLUS::Shell::Classic'=> '0.0562',
	'CPANPLUS::Shell::Default'=> '0.9103',
	'CPANPLUS::Shell::Default::Plugins::CustomSource'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Remote'=> undef,
	'CPANPLUS::Shell::Default::Plugins::Source'=> undef,
	'Carp'                  => '1.20',
	'Carp::Heavy'           => '1.20',
	'Class::Struct'         => '0.63',
	'Compress::Raw::Bzip2'  => '2.033',
	'Compress::Raw::Zlib'   => '2.033',
	'Compress::Zlib'        => '2.033',
	'Config'                => undef,
	'Config::Extensions'    => '0.01',
	'Cwd'                   => '3.36',
	'DB'                    => '1.03',
	'DBM_Filter'            => '0.04',
	'DBM_Filter::compress'  => '0.02',
	'DBM_Filter::encode'    => '0.02',
	'DBM_Filter::int32'     => '0.02',
	'DBM_Filter::null'      => '0.02',
	'DBM_Filter::utf8'      => '0.02',
	'DB_File'               => '1.821',
	'Data::Dumper'          => '2.130_02',
	'Devel::DProf'          => '20110228.00',
	'Devel::DProf::dprof::V'=> undef,
	'Devel::InnerPackage'   => '0.3',
	'Devel::PPPort'         => '3.19',
	'Devel::Peek'           => '1.07',
	'Devel::SelfStubber'    => '1.05',
	'Digest'                => '1.16',
	'Digest::MD5'           => '2.51',
	'Digest::SHA'           => '5.61',
	'Digest::base'          => '1.16',
	'Digest::file'          => '1.16',
	'DirHandle'             => '1.04',
	'Dumpvalue'             => '1.16',
	'DynaLoader'            => '1.13',
	'Encode'                => '2.42',
	'Encode::Alias'         => '2.13',
	'Encode::Byte'          => '2.04',
	'Encode::CJKConstants'  => '2.02',
	'Encode::CN'            => '2.03',
	'Encode::CN::HZ'        => '2.05',
	'Encode::Config'        => '2.05',
	'Encode::EBCDIC'        => '2.02',
	'Encode::Encoder'       => '2.01',
	'Encode::Encoding'      => '2.05',
	'Encode::GSM0338'       => '2.01',
	'Encode::Guess'         => '2.04',
	'Encode::JP'            => '2.04',
	'Encode::JP::H2Z'       => '2.02',
	'Encode::JP::JIS7'      => '2.04',
	'Encode::KR'            => '2.03',
	'Encode::KR::2022_KR'   => '2.02',
	'Encode::MIME::Header'  => '2.13',
	'Encode::MIME::Header::ISO_2022_JP'=> '1.03',
	'Encode::MIME::Name'    => '1.01',
	'Encode::Symbol'        => '2.02',
	'Encode::TW'            => '2.03',
	'Encode::Unicode'       => '2.07',
	'Encode::Unicode::UTF7' => '2.05',
	'English'               => '1.04',
	'Env'                   => '1.02',
	'Errno'                 => '1.13',
	'Exporter'              => '5.64_03',
	'Exporter::Heavy'       => '5.64_03',
	'ExtUtils::CBuilder'    => '0.280202',
	'ExtUtils::CBuilder::Base'=> '0.280202',
	'ExtUtils::CBuilder::Platform::Unix'=> '0.280202',
	'ExtUtils::CBuilder::Platform::VMS'=> '0.280202',
	'ExtUtils::CBuilder::Platform::Windows'=> '0.280202',
	'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280202',
	'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.280202',
	'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.280202',
	'ExtUtils::CBuilder::Platform::aix'=> '0.280202',
	'ExtUtils::CBuilder::Platform::cygwin'=> '0.280202',
	'ExtUtils::CBuilder::Platform::darwin'=> '0.280202',
	'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280202',
	'ExtUtils::CBuilder::Platform::os2'=> '0.280202',
	'ExtUtils::Command'     => '1.17',
	'ExtUtils::Command::MM' => '6.57_05',
	'ExtUtils::Constant'    => '0.23',
	'ExtUtils::Constant::Base'=> '0.05',
	'ExtUtils::Constant::ProxySubs'=> '0.08',
	'ExtUtils::Constant::Utils'=> '0.03',
	'ExtUtils::Constant::XS'=> '0.03',
	'ExtUtils::Embed'       => '1.30',
	'ExtUtils::Install'     => '1.56',
	'ExtUtils::Installed'   => '1.999_001',
	'ExtUtils::Liblist'     => '6.57_05',
	'ExtUtils::Liblist::Kid'=> '6.5705',
	'ExtUtils::MM'          => '6.57_05',
	'ExtUtils::MM_AIX'      => '6.57_05',
	'ExtUtils::MM_Any'      => '6.57_05',
	'ExtUtils::MM_BeOS'     => '6.57_05',
	'ExtUtils::MM_Cygwin'   => '6.57_05',
	'ExtUtils::MM_DOS'      => '6.5705',
	'ExtUtils::MM_Darwin'   => '6.57_05',
	'ExtUtils::MM_MacOS'    => '6.5705',
	'ExtUtils::MM_NW5'      => '6.57_05',
	'ExtUtils::MM_OS2'      => '6.57_05',
	'ExtUtils::MM_QNX'      => '6.57_05',
	'ExtUtils::MM_UWIN'     => '6.5705',
	'ExtUtils::MM_Unix'     => '6.57_05',
	'ExtUtils::MM_VMS'      => '6.57_05',
	'ExtUtils::MM_VOS'      => '6.57_05',
	'ExtUtils::MM_Win32'    => '6.57_05',
	'ExtUtils::MM_Win95'    => '6.57_05',
	'ExtUtils::MY'          => '6.5705',
	'ExtUtils::MakeMaker'   => '6.57_05',
	'ExtUtils::MakeMaker::Config'=> '6.57_05',
	'ExtUtils::MakeMaker::YAML'=> '1.44',
	'ExtUtils::Manifest'    => '1.58',
	'ExtUtils::Miniperl'    => undef,
	'ExtUtils::Mkbootstrap' => '6.57_05',
	'ExtUtils::Mksymlists'  => '6.57_05',
	'ExtUtils::Packlist'    => '1.44',
	'ExtUtils::ParseXS'     => '2.2209',
	'ExtUtils::XSSymSet'    => '1.2',
	'ExtUtils::testlib'     => '6.5705',
	'Fatal'                 => '2.1001',
	'Fcntl'                 => '1.11',
	'File::Basename'        => '2.81',
	'File::CheckTree'       => '4.41',
	'File::Compare'         => '1.1006',
	'File::Copy'            => '2.21',
	'File::DosGlob'         => '1.04',
	'File::Fetch'           => '0.32',
	'File::Find'            => '1.19',
	'File::Glob'            => '1.12',
	'File::GlobMapper'      => '1.000',
	'File::Path'            => '2.08_01',
	'File::Spec'            => '3.33',
	'File::Spec::Cygwin'    => '3.33',
	'File::Spec::Epoc'      => '3.33',
	'File::Spec::Functions' => '3.33',
	'File::Spec::Mac'       => '3.34',
	'File::Spec::OS2'       => '3.33',
	'File::Spec::Unix'      => '3.33',
	'File::Spec::VMS'       => '3.34',
	'File::Spec::Win32'     => '3.33',
	'File::Temp'            => '0.22',
	'File::stat'            => '1.05',
	'FileCache'             => '1.08',
	'FileHandle'            => '2.02',
	'Filter::Simple'        => '0.86',
	'Filter::Util::Call'    => '1.08',
	'FindBin'               => '1.50',
	'GDBM_File'             => '1.14',
	'Getopt::Long'          => '2.38',
	'Getopt::Std'           => '1.06',
	'HTTP::Tiny'            => '0.011',
	'Hash::Util'            => '0.11',
	'Hash::Util::FieldHash' => '1.09',
	'I18N::Collate'         => '1.02',
	'I18N::LangTags'        => '0.35_01',
	'I18N::LangTags::Detect'=> '1.05',
	'I18N::LangTags::List'  => '0.35_01',
	'I18N::Langinfo'        => '0.08',
	'IO'                    => '1.25_04',
	'IO::Compress::Adapter::Bzip2'=> '2.033',
	'IO::Compress::Adapter::Deflate'=> '2.033',
	'IO::Compress::Adapter::Identity'=> '2.033',
	'IO::Compress::Base'    => '2.033',
	'IO::Compress::Base::Common'=> '2.033',
	'IO::Compress::Bzip2'   => '2.033',
	'IO::Compress::Deflate' => '2.033',
	'IO::Compress::Gzip'    => '2.033',
	'IO::Compress::Gzip::Constants'=> '2.033',
	'IO::Compress::RawDeflate'=> '2.033',
	'IO::Compress::Zip'     => '2.033',
	'IO::Compress::Zip::Constants'=> '2.033',
	'IO::Compress::Zlib::Constants'=> '2.033',
	'IO::Compress::Zlib::Extra'=> '2.033',
	'IO::Dir'               => '1.08',
	'IO::File'              => '1.15',
	'IO::Handle'            => '1.30',
	'IO::Pipe'              => '1.14',
	'IO::Poll'              => '0.08',
	'IO::Seekable'          => '1.10',
	'IO::Select'            => '1.20',
	'IO::Socket'            => '1.32',
	'IO::Socket::INET'      => '1.31',
	'IO::Socket::UNIX'      => '1.23',
	'IO::Uncompress::Adapter::Bunzip2'=> '2.033',
	'IO::Uncompress::Adapter::Identity'=> '2.033',
	'IO::Uncompress::Adapter::Inflate'=> '2.033',
	'IO::Uncompress::AnyInflate'=> '2.033',
	'IO::Uncompress::AnyUncompress'=> '2.033',
	'IO::Uncompress::Base'  => '2.033',
	'IO::Uncompress::Bunzip2'=> '2.033',
	'IO::Uncompress::Gunzip'=> '2.033',
	'IO::Uncompress::Inflate'=> '2.033',
	'IO::Uncompress::RawInflate'=> '2.033',
	'IO::Uncompress::Unzip' => '2.033',
	'IO::Zlib'              => '1.10',
	'IPC::Cmd'              => '0.70',
	'IPC::Msg'              => '2.03',
	'IPC::Open2'            => '1.03',
	'IPC::Open3'            => '1.09',
	'IPC::Semaphore'        => '2.03',
	'IPC::SharedMem'        => '2.03',
	'IPC::SysV'             => '2.03',
	'JSON::PP'              => '2.27105',
	'JSON::PP::Boolean'     => undef,
	'List::Util'            => '1.23',
	'List::Util::PP'        => '1.23',
	'List::Util::XS'        => '1.23',
	'Locale::Codes'         => '3.16',
	'Locale::Codes::Country'=> '3.16',
	'Locale::Codes::Currency'=> '3.16',
	'Locale::Codes::Language'=> '3.16',
	'Locale::Codes::Script' => '3.16',
	'Locale::Constants'     => '3.16',
	'Locale::Country'       => '3.16',
	'Locale::Currency'      => '3.16',
	'Locale::Language'      => '3.16',
	'Locale::Maketext'      => '1.19',
	'Locale::Maketext::Guts'=> '1.17',
	'Locale::Maketext::GutsLoader'=> '1.17',
	'Locale::Maketext::Simple'=> '0.21',
	'Locale::Script'        => '3.16',
	'Log::Message'          => '0.04',
	'Log::Message::Config'  => '0.04',
	'Log::Message::Handlers'=> '0.04',
	'Log::Message::Item'    => '0.04',
	'Log::Message::Simple'  => '0.08',
	'MIME::Base64'          => '3.13',
	'MIME::QuotedPrint'     => '3.13',
	'Math::BigFloat'        => '1.993',
	'Math::BigFloat::Trace' => '0.01',
	'Math::BigInt'          => '1.994',
	'Math::BigInt::Calc'    => '1.993',
	'Math::BigInt::CalcEmu' => '1.993',
	'Math::BigInt::FastCalc'=> '0.28',
	'Math::BigInt::Trace'   => '0.01',
	'Math::BigRat'          => '0.26_02',
	'Math::Complex'         => '1.56',
	'Math::Trig'            => '1.2',
	'Memoize'               => '1.02',
	'Memoize::AnyDBM_File'  => '1.02',
	'Memoize::Expire'       => '1.02',
	'Memoize::ExpireFile'   => '1.02',
	'Memoize::ExpireTest'   => '1.02',
	'Memoize::NDBM_File'    => '1.02',
	'Memoize::SDBM_File'    => '1.02',
	'Memoize::Storable'     => '1.02',
	'Module::Build'         => '0.3800',
	'Module::Build::Base'   => '0.3800',
	'Module::Build::Compat' => '0.3800',
	'Module::Build::Config' => '0.3800',
	'Module::Build::ConfigData'=> undef,
	'Module::Build::Cookbook'=> '0.3800',
	'Module::Build::Dumper' => '0.3800',
	'Module::Build::ModuleInfo'=> '0.3800',
	'Module::Build::Notes'  => '0.3800',
	'Module::Build::PPMMaker'=> '0.3800',
	'Module::Build::Platform::Amiga'=> '0.3800',
	'Module::Build::Platform::Default'=> '0.3800',
	'Module::Build::Platform::EBCDIC'=> '0.3800',
	'Module::Build::Platform::MPEiX'=> '0.3800',
	'Module::Build::Platform::MacOS'=> '0.3800',
	'Module::Build::Platform::RiscOS'=> '0.3800',
	'Module::Build::Platform::Unix'=> '0.3800',
	'Module::Build::Platform::VMS'=> '0.3800',
	'Module::Build::Platform::VOS'=> '0.3800',
	'Module::Build::Platform::Windows'=> '0.3800',
	'Module::Build::Platform::aix'=> '0.3800',
	'Module::Build::Platform::cygwin'=> '0.3800',
	'Module::Build::Platform::darwin'=> '0.3800',
	'Module::Build::Platform::os2'=> '0.3800',
	'Module::Build::PodParser'=> '0.3800',
	'Module::Build::Version'=> '0.87',
	'Module::Build::YAML'   => '1.41',
	'Module::CoreList'      => '2.46',
	'Module::Load'          => '0.18',
	'Module::Load::Conditional'=> '0.44',
	'Module::Loaded'        => '0.06',
	'Module::Metadata'      => '1.000004',
	'Module::Pluggable'     => '3.9',
	'Module::Pluggable::Object'=> '3.9',
	'Moped::Msg'            => '0.01',
	'NDBM_File'             => '1.12',
	'NEXT'                  => '0.65',
	'Net::Cmd'              => '2.29',
	'Net::Config'           => '1.11',
	'Net::Domain'           => '2.20',
	'Net::FTP'              => '2.77',
	'Net::FTP::A'           => '1.18',
	'Net::FTP::E'           => '0.01',
	'Net::FTP::I'           => '1.12',
	'Net::FTP::L'           => '0.01',
	'Net::FTP::dataconn'    => '0.11',
	'Net::NNTP'             => '2.24',
	'Net::Netrc'            => '2.12',
	'Net::POP3'             => '2.29',
	'Net::Ping'             => '2.38',
	'Net::SMTP'             => '2.31',
	'Net::Time'             => '2.10',
	'Net::hostent'          => '1.01',
	'Net::netent'           => '1.00',
	'Net::protoent'         => '1.00',
	'Net::servent'          => '1.01',
	'O'                     => '1.01',
	'ODBM_File'             => '1.10',
	'Object::Accessor'      => '0.38',
	'Opcode'                => '1.18',
	'POSIX'                 => '1.24',
	'Package::Constants'    => '0.02',
	'Params::Check'         => '0.28',
	'Parse::CPAN::Meta'     => '1.4401',
	'Perl::OSType'          => '1.002',
	'PerlIO'                => '1.06',
	'PerlIO::encoding'      => '0.14',
	'PerlIO::scalar'        => '0.11',
	'PerlIO::via'           => '0.11',
	'PerlIO::via::QuotedPrint'=> '0.06',
	'Pod::Checker'          => '1.45',
	'Pod::Escapes'          => '1.04',
	'Pod::Find'             => '1.35',
	'Pod::Functions'        => '1.04',
	'Pod::Html'             => '1.1',
	'Pod::InputObjects'     => '1.31',
	'Pod::LaTeX'            => '0.59',
	'Pod::Man'              => '2.25',
	'Pod::ParseLink'        => '1.10',
	'Pod::ParseUtils'       => '1.36',
	'Pod::Parser'           => '1.37',
	'Pod::Perldoc'          => '3.15_03',
	'Pod::Perldoc::BaseTo'  => undef,
	'Pod::Perldoc::GetOptsOO'=> undef,
	'Pod::Perldoc::ToChecker'=> undef,
	'Pod::Perldoc::ToMan'   => undef,
	'Pod::Perldoc::ToNroff' => undef,
	'Pod::Perldoc::ToPod'   => undef,
	'Pod::Perldoc::ToRtf'   => undef,
	'Pod::Perldoc::ToText'  => undef,
	'Pod::Perldoc::ToTk'    => undef,
	'Pod::Perldoc::ToXml'   => undef,
	'Pod::PlainText'        => '2.04',
	'Pod::Select'           => '1.36',
	'Pod::Simple'           => '3.16',
	'Pod::Simple::BlackBox' => '3.16',
	'Pod::Simple::Checker'  => '3.16',
	'Pod::Simple::Debug'    => '3.16',
	'Pod::Simple::DumpAsText'=> '3.16',
	'Pod::Simple::DumpAsXML'=> '3.16',
	'Pod::Simple::HTML'     => '3.16',
	'Pod::Simple::HTMLBatch'=> '3.16',
	'Pod::Simple::HTMLLegacy'=> '5.01',
	'Pod::Simple::LinkSection'=> '3.16',
	'Pod::Simple::Methody'  => '3.16',
	'Pod::Simple::Progress' => '3.16',
	'Pod::Simple::PullParser'=> '3.16',
	'Pod::Simple::PullParserEndToken'=> '3.16',
	'Pod::Simple::PullParserStartToken'=> '3.16',
	'Pod::Simple::PullParserTextToken'=> '3.16',
	