package Jenkins::i18n::Stats;

use 5.014004;
use strict;
use warnings;
use Hash::Util qw(lock_keys);
use Carp qw(confess);

our $VERSION = '0.04';

=pod

=head1 NAME

Jenkins::i18n::Stats - class to provide translations processing statistics

=head1 SYNOPSIS

  use Jenkins::i18n::Stats;

=head1 DESCRIPTION

C<Jenkins::i18n::Stats>

=head2 EXPORT

None by default.

=head1 ATTRIBUTES

All attributes are counters.

=over

=item *

files: all the found translation files.

=item *

keys: all the keys loaded from the properties files.

=item *

missing: all keys that are missing after comparing a language to the original
in English,

=item *

unused: all keys that are available in the a language but not in the original
English.

=item *

empty: all keys in the language that are available but doesn't actually have a
translated value.

=item *

same: all keys that have the same values as the original in English. Not
necessarilly an error.

=item *

no_jenkins: all keys that are not related to Jenkins, but coming from Hudson.

=back

=head1 METHODS

=head2 new

Creates a new instance.

=cut

sub new {
    my $class = shift;
    my $self  = {
        files      => 0,
        keys       => 0,
        missing    => 0,
        unused     => 0,
        empty      => 0,
        same       => 0,
        no_jenkins => 0
    };

    bless $self, $class;
    lock_keys( %{$self} );
    return $self;
}

=head2 inc

Increments a counter.

=cut

sub inc {
    my ( $self, $item ) = @_;
    confess "item is a required parameter" unless ($item);
    confess "there is no such counter '$item'"
        unless ( exists( $self->{$item} ) );
    $self->{$item}++;
    return 1;
}

=head2 summary

Prints to C<STDOUT> a summary of all statistics in text format.

=cut

sub summary {
    my $self = shift;

    my $done
        = $self->{keys}
        - $self->{missing}
        - $self->{unused}
        - $self->{empty}
        - $self->{same}
        - $self->{no_jenkins};

    my ( $pdone, $pmissing, $punused, $pempty, $psame, $pnojenkins );

    unless ( $self->{keys} == 0 ) {
        $pdone    = $done / $self->{keys} * 100;
        $pmissing = $self->{missing} / $self->{keys} * 100;
        $punused  = $self->{unused} / $self->{keys} * 100;

        $pempty     = $self->{empty} / $self->{keys} * 100;
        $psame      = $self->{same} / $self->{keys} * 100;
        $pnojenkins = $self->{no_jenkins} / $self->{keys} * 100;
        format STDOUT_TOP =

         Translation Status

    Status         Total      %
    -----------------------------
.

        format STDOUT =
    @<<<<<<<<<<    @<<<<    @<<<<
    'Done', $done, $pdone
    @<<<<<<<<<<    @<<<<    @<<<<
    'Missing', $self->{missing}, $pmissing
    @<<<<<<<<<<    @<<<<    @<<<<
    'Orphan', $self->{unused}, $punused
    @<<<<<<<<<<    @<<<<    @<<<<
    'Empty', $self->{empty}, $pempty
    @<<<<<<<<<<    @<<<<    @<<<<
    'Same', $self->{same}, $psame
    @<<<<<<<<<<    @<<<<    @<<<<
    'No Jenkins', $self->{no_jenkins}, $pnojenkins

.

        write;
        print 'Total of files: ', $self->{files}, "\n";
    }
    else {
        warn "Not a single key was processed\n";
    }
}

1;
__END__

=head1 SEE ALSO

=over

=item *

L<Config::Properties>

=back

=head1 AUTHOR

Alceu Rodrigues de Freitas Junior, E<lt>arfreitas@cpan.orgE<gt>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2022 of Alceu Rodrigues de Freitas Junior,
E<lt>arfreitas@cpan.orgE<gt>

This file is part of Jenkins Translation Tool project.

Jenkins Translation Tool is free software: you can redistribute it and/or
modify it under the terms of the GNU General Public License as published by the
Free Software Foundation, either version 3 of the License, or (at your option)
any later version.

Jenkins Translation Tool is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public License along with
Jenkins Translation Tool. If not, see (http://www.gnu.org/licenses/).

The original `translation-tool.pl` script was licensed through the MIT License,
copyright (c) 2004-, Kohsuke Kawaguchi, Sun Microsystems, Inc., and a number of
other of contributors. Translations files generated by the Jenkins Translation
Tool CLI are distributed with the same MIT License.

=cut
