// $Id: f_cisco.cc 1.12.1.8 Mon, 02 Feb 1998 18:26:49 -0800 cengiz $
// 
//  Copyright (c) 1994 by the University of Southern California
//  and/or the International Business Machines Corporation.
//  All rights reserved.
//
//  Permission to use, copy, modify, and distribute this software and
//  its documentation in source and binary forms for lawful
//  non-commercial purposes and without fee is hereby granted, provided
//  that the above copyright notice appear in all copies and that both
//  the copyright notice and this permission notice appear in supporting
//  documentation, and that any documentation, advertising materials,
//  and other materials related to such distribution and use acknowledge
//  that the software was developed by the University of Southern
//  California, Information Sciences Institute and/or the International
//  Business Machines Corporation.  The name of the USC or IBM may not
//  be used to endorse or promote products derived from this software
//  without specific prior written permission.
//
//  NEITHER THE UNIVERSITY OF SOUTHERN CALIFORNIA NOR INTERNATIONAL
//  BUSINESS MACHINES CORPORATION MAKES ANY REPRESENTATIONS ABOUT
//  THE SUITABILITY OF THIS SOFTWARE FOR ANY PURPOSE.  THIS SOFTWARE IS
//  PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES,
//  INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
//  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, TITLE, AND 
//  NON-INFRINGEMENT.
//
//  IN NO EVENT SHALL USC, IBM, OR ANY OTHER CONTRIBUTOR BE LIABLE FOR ANY
//  SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES, WHETHER IN CONTRACT,
//  TORT, OR OTHER FORM OF ACTION, ARISING OUT OF OR IN CONNECTION WITH,
//  THE USE OR PERFORMANCE OF THIS SOFTWARE.
//
//  Questions concerning this software should be directed to 
//  info-ra@isi.edu.
//
//  Author(s): Cengiz Alaettinoglu <cengiz@isi.edu>

#include "config.hh"
#include <cstring>
#include <iostream.h>
#include <iomanip.h>
#include <cctype>
#include "RtConfig.hh"
#include "Node.h"

#define DBG_CISCO 7

#define EXPORT 0
#define IMPORT 1

static int different_pref_count = 0;
static int prfx_alist_no = 0;
static int route_map_is_used = 0;
int cisco_aspath_access_list_no = 1;
int cisco_access_list_no = 1;
int cisco_max_preference = 1000;
int cisco_route_map_no = 1;
char cisco_map_name[80] = "foo";   
int cisco_map_inc = 1;
int cisco_map_first_no = 1;
static char last_cisco_map_name[80] = "";

void cisco_export(Pix asno, Pix addr, Pix peer_asno, Pix peer_addr);
void cisco_import(Pix asno, Pix addr, Pix peer_asno, Pix peer_addr);
void cisco_packet_filter(Pix asno, Pix addr, Pix peer_asno, Pix peer_addr, 
			 char *name);
void cisco_outbound_packet_filter(Pix asno, Pix addr, 
				  Pix peer_asno, Pix peer_addr, 
				  char *name);
void cisco_networks(Pix asno);
void cisco_default(Pix asno, Pix asno);

void cisco_process_line(int command, Pix p1, Pix p2, Pix p3, Pix p4, char *p5) {
   switch (command) {
   case IMPORT_COMMAND:
      cisco_import(p1, p2, p3, p4);
      break;
   case EXPORT_COMMAND:
      cisco_export(p1, p2, p3, p4);
      break;
   case NETWORKS_COMMAND:
      cisco_networks(p1);
      break;
   case DEFAULT_COMMAND:
      cisco_default(p1, p2);
      break;
   case PKT_FILTER_COMMAND:
      cisco_packet_filter(p1, p2, p3, p4, p5);
      break;
   case OUTBOUND_PKT_FILTER_COMMAND:
      cisco_outbound_packet_filter(p1, p2, p3, p4, p5);
      break;
  default:
      cerr << "Error: Unknown RtConfig command." << endl;
   }
}

void cisco_networks(Pix asno) {
   static char buffer[128];
   static char buffer2[128];
   _SetOfPix &nets = AS_map.expand(asno);

   for (Pix p = nets.first(); p; nets.next(p)) {
      Prefask &address = Prefask_map(nets(p));
      cout << "network " << int2quad(buffer, address.get_prefix())
           << " mask " << int2quad(buffer2, address.get_mask()) 
           << "\n";
   }

   cout << "!\n";
}

void cisco_default(Pix asno, Pix peer_asno) {
   ASPolicy *p;
   Filter_Action *fap;
   AutNum *autnum;
   char buffer[64];

   // define as
   if (!(autnum = AS_map.define(asno))) {
      cerr << "Error: no object for " << AS_map(asno) << endl;
      return;
    }

   // find peer as
   if (!(p = autnum->find_peer(peer_asno))) {
      cerr << "Error: " << AS_map(asno) 
         << " has no policy for " << AS_map(peer_asno) << endl;
      return;
   }

   if (!(fap = p->dflt.head())) {
      cerr << "Error: " << AS_map(asno) 
         << " has no default policy for " << AS_map(peer_asno) << endl;
      return;
   }
   
   for (; fap; fap = p->dflt.next(fap->falist)) {
      NormalExpression *ne = new NormalExpression(*fap->expand());

      if (ne->universal()) {
	 cout << "ip default-network 0.0.0.0n";
      } else {
	 NormalTerm *nt = ne->first();

	 RadixSet::SortedPrefixIterator itr(&(ne->first()->prfx_set.members));
	 u_int addr;
	 u_int leng;
	 char buffer[64];

	 for (bool ok = itr.first(addr, leng);
	      ok;
	      ok = itr.next(addr, leng)) {
	    cout << "ip default-network " 
		 << int2quad(buffer, addr)
	       // << " mask " 
	       // << int2quad(buffer, masks[leng]) 
		 << "\n";
	 }
      }
   }

   cout << "!\n";
}

class access_list_cache_node {
public:
   access_list_cache_node() : l(this) {}
   access_list_cache_node(access_list_cache_node &b) : 
      nets(b.nets), allow_flag(b.allow_flag), no(b.no), l(this) {}
   access_list_cache_node(SetOfPrefix& _nets, int _allow_flag, int _no) : 
      nets(_nets), allow_flag(_allow_flag), no(_no), l(this) {}

public:
   SetOfPrefix nets;
   int      allow_flag;
   int      no;

   ListNode l;
};

ListHead<access_list_cache_node> access_list_cache;

int cisco_print_packet_filter(SetOfPrefix &set) {
   RadixSet::SortedPrefixIterator itr(&set.members);
   u_int addr;
   u_int leng;
   u_int64_t rngs;
   char buffer[64];
   bool allow_flag = ! set.negated();

   cout << "no access-list " << cisco_access_list_no << "\n";

   for (bool ok = itr.first(addr, leng); ok; ok = itr.next(addr, leng)) {
      if (!addr && !leng) // skip 0.0.0.0/0
	 continue;

      cout << "access-list " << cisco_access_list_no;
      if (allow_flag)
	 cout << " permit ip ";
      else 
	 cout << " deny ip ";

      cout << int2quad(buffer, addr) << " ";
      cout << int2quad(buffer, ~masks[leng]) << " any \n";
   }

   if (allow_flag)
      cout << "access-list " << cisco_access_list_no << " deny ip any any\n";
   else
      cout << "access-list " << cisco_access_list_no << " permit ip any any\n";

   cout << "!\n";

   return cisco_access_list_no++;
}

void cisco_packet_filter(Pix asno, Pix addr, Pix peer_asno, Pix peer_addr, 
			 char *name) {
   ASPolicy *p;
   Filter_Action *fap;
   Filter_Action *ifap;
   InterASPolicy *ip;
   int interas_match;
   AutNum *autnum;

   // define as
   if (!(autnum = AS_map.define(asno))) {
      cerr << "Error: no object for " << AS_map(asno) << endl;
      return;
    }
   
   // find peer as
   if (!(p = autnum->find_peer(peer_asno))) {
      cerr << "Error: " << AS_map(asno) 
         << " has no policy for " << AS_map(peer_asno) << endl;
      return;
   }

   if (!(fap = p->in.head())) {
      cerr << "Error: " << AS_map(asno) 
         << " has no import policy for " << AS_map(peer_asno) << endl;
      return;
   }
   
   // find peering
   ip = p->find_peering(addr, peer_addr);
   
   // for each as-in policy
   NormalExpression *fap_exp, *ifap_exp;
   SetOfPrefix set;

   for (; fap; fap = p->in.next(fap->falist)) {
      // calculate imports for this interface
      if (ip) { // is there an interas-in policy
         // for each interas-in policy
         for (ifap = ip->in.head(); 
              ifap; 
              ifap = ip->in.next(ifap->falist)) {
            fap_exp = new NormalExpression(*fap->expand());
            ifap_exp = new NormalExpression(*ifap->expand());
	    
	    fap_exp->do_and(*ifap_exp);
	    for (NormalTerm *nt = fap_exp->first(); nt; nt = fap_exp->next())
	       set |= nt->prfx_set;

            delete fap_exp;
            delete ifap_exp;
         }
      }
      // left over policies
      fap_exp = new NormalExpression(*fap->expand());
      ifap_exp = new NormalExpression(*p->get_remaining_in());
      
      fap_exp->do_and(*ifap_exp);
      for (NormalTerm *nt = fap_exp->first(); nt; nt = fap_exp->next())
	 set |= nt->prfx_set;

      delete fap_exp;
      delete ifap_exp;
   }

   int aclid = cisco_print_packet_filter(set);

   static char neighbor[128];
   int2quad(neighbor, Prefask_map(peer_addr).get_prefix());

   cout << "interface " << name 
	<< "\n ip access-group " << aclid << " in\n";
}


void cisco_outbound_packet_filter(Pix asno, Pix addr, 
				  Pix peer_asno, Pix peer_addr, 
				  char *name) {
   ASPolicy *p;
   Filter_Action *fap;
   Filter_Action *ifap;
   InterASPolicy *ip;
   int interas_match;
   AutNum *autnum;

   // define as
   if (!(autnum = AS_map.define(asno))) {
      cerr << "Error: no object for " << AS_map(asno) << endl;
      return;
    }
   
   // find peer as
   if (!(p = autnum->find_peer(peer_asno))) {
      cerr << "Error: " << AS_map(asno) 
         << " has no policy for " << AS_map(peer_asno) << endl;
      return;
   }

   if (!(fap = p->out.head())) {
      cerr << "Error: " << AS_map(asno) 
         << " has no import policy for " << AS_map(peer_asno) << endl;
      return;
   }
   
   // find peering
   ip = p->find_peering(addr, peer_addr);
   
   // for each as-out policy
   NormalExpression *fap_exp, *ifap_exp;
   SetOfPrefix set;

   for (; fap; fap = p->out.next(fap->falist)) {
      // calculate imports for this interface
      if (ip) { // is there an interas-out policy
         // for each interas-out policy
         for (ifap = ip->out.head(); 
              ifap; 
              ifap = ip->out.next(ifap->falist)) {
            fap_exp = new NormalExpression(*fap->expand());
            ifap_exp = new NormalExpression(*ifap->expand());
	    
	    fap_exp->do_and(*ifap_exp);
	    for (NormalTerm *nt = fap_exp->first(); nt; nt = fap_exp->next())
	       set |= nt->prfx_set;

            delete fap_exp;
            delete ifap_exp;
         }
      }
      // left over policies
      fap_exp = new NormalExpression(*fap->expand());
      ifap_exp = new NormalExpression(*p->get_remaining_out());
      
      fap_exp->do_and(*ifap_exp);
      for (NormalTerm *nt = fap_exp->first(); nt; nt = fap_exp->next())
	 set |= nt->prfx_set;

      delete fap_exp;
      delete ifap_exp;
   }

   int aclid = cisco_print_packet_filter(set);

   static char neighbor[128];
   int2quad(neighbor, Prefask_map(peer_addr).get_prefix());

   cout << "interface " << name 
	<< "\n ip access-group " << aclid << " out\n";
}


int cisco_print_net_list(SetOfPrefix& nets, 
			 int& access_list_id,
			 int allow_flag) {
// return the access list number if something is printed
   access_list_cache_node *cnode;

   if (nets.universal())
      return 0;

   if (opt_cisco_access_list_cache)
      for (cnode = access_list_cache.head(); 
           cnode;
           cnode = access_list_cache.next(cnode->l))
         if (cnode->nets == nets && cnode->allow_flag == allow_flag)
	    return cnode->no;

   char *martians[] = { 
      "host 0.0.0.0 any",
      "127.0.0.0 0.255.255.255 255.0.0.0 0.255.255.255",
      "10.0.0.0 0.255.255.255 255.0.0.0 0.255.255.255",
      "172.16.0.0 0.15.255.255 255.240.0.0 0.15.255.255",
      "192.168.0.0 0.0.255.255 255.255.0.0 0.0.255.255",
      "192.0.2.0 0.0.0.255 255.255.255.0 0.0.0.255",
      "128.0.0.0 0.0.255.255 255.255.0.0 0.0.255.255",
      "191.255.0.0 0.0.255.255 255.255.0.0 0.0.255.255",
      "192.0.0.0 0.0.0.255 255.255.255.0 0.0.0.255",
      "223.255.255.0 0.0.0.255 255.255.255.0 0.0.0.255",
      "224.0.0.0 31.255.255.255 224.0.0.0 31.255.255.255",
      "any 255.255.255.128 0.0.0.127",
      NULL
   };

   cout << "no access-list " << access_list_id << "\n";
   if (opt_cisco_supress_martian)
      for (int i = 0; martians[i]; ++i)
         cout << "access-list " << access_list_id 
              << " deny   ip " << martians[i] << "\n";

   if (opt_cisco_compress_acls) {
      RadixSet::SortedPrefixRangeIterator itr(&nets.members);
      u_int addr;
      u_int leng;
      u_int start;
      u_int end;
      char buffer[64];

      for (bool ok = itr.first(addr, leng, start, end);
	   ok;
	   ok = itr.next(addr, leng, start, end)) {
	 cout << "access-list " << access_list_id;
	 if (allow_flag)
	    cout << " permit ip ";
	 else 
	    cout << " deny ip ";

	 /* need to look at WeeSan's code */
	 cout << int2quad(buffer, addr) << "   ";
	 cout << int2quad(buffer, ones(leng + 1, end)) << "   ";
	 cout << int2quad(buffer, masks[start]) << "   ";
	 cout << int2quad(buffer, ones(start + 1, end))
	      << "\n";
      }
   } else {
      RadixSet::SortedPrefixIterator itr(&nets.members);
      u_int addr;
      u_int leng;
      char buffer[64];

      for (bool ok = itr.first(addr, leng);
	   ok;
	   ok = itr.next(addr, leng)) {
	 cout << "access-list " << access_list_id;
	 if (allow_flag)
	    cout << " permit ip ";
	 else 
	    cout << " deny ip ";

	 cout << int2quad(buffer, addr) << "   0.0.0.0   ";
	 cout << int2quad(buffer, masks[leng]) << "   0.0.0.0\n";
      }
   }

   if (allow_flag)
      cout << "access-list " << access_list_id 
           << " deny ip 0.0.0.0 255.255.255.255 0.0.0.0 255.255.255.255\n";
   else
      cout << "access-list " << access_list_id 
           << " permit ip 0.0.0.0 255.255.255.255 0.0.0.0 255.255.255.255\n";

   cout << "!\n";

   if (opt_cisco_access_list_cache) {
      access_list_cache_node *new_cnode = 
         new access_list_cache_node(nets, allow_flag, access_list_id);
      access_list_cache.append(new_cnode->l);
   }
   return access_list_id++;
}

void cisco_print_re_asno(ostream& out, const re_asno_t &no) {
   re_asno_t::range *pi, *qi;
   int first = 1;
   int put_par = 0;

   if (no.universal())
      out << "(_[0-9]+)";
   else {
      out << "_";
      pi = no.ranges.head();
      put_par = no.ranges.size() != 1 || pi->high != pi->low;
      if (put_par)
         out << "("; 

      for (; pi; pi = no.ranges.next(pi->ranges)) {

         if (pi->high - pi->low < 10) 
            for (int i = pi->low; i <= pi->high; ++i) {
               if (!first) 
                  out << "|";
               else
                  first = 0;
               out << i;
            } 
         else {
            out << "(error)";
            cerr << "Error: cannot yet convert this regexp with [,] to cisco format." << endl;
         }
      }

      if (put_par)
         out << ")";
   }
}

int cisco_print_re_(ostream& os, const regexp& r) {
   int flag = 0;
   switch (r.regexp_type) {
   case REGEXP_BOL : 
      os << "^";
      break;
   case REGEXP_EOL : 
      os << "$";
      flag = 1;
      break;
   case REGEXP_SYMBOL :
      cisco_print_re_asno(os, ((regexp_symbol &) r).asnumbers);
      break;
   case REGEXP_STAR :
      os << "(";
      flag = cisco_print_re_(os, *((regexp_star &) r).left);
      os << ")*";
      break;
   case REGEXP_QUESTION :
      os << "(";
      flag = cisco_print_re_(os, *((regexp_question &) r).left);
      os << ")?";
      break;
   case REGEXP_PLUS :
      os << "(";
      flag = cisco_print_re_(os, *((regexp_plus &) r).left);
      os << ")+";
      break;
   case REGEXP_CAT :
      cisco_print_re_(os, *((regexp_cat &) r).left);
      flag = cisco_print_re_(os, *((regexp_cat &) r).right);
      break;
   case REGEXP_OR :
      os << "(";
      flag = cisco_print_re_(os, *((regexp_or &) r).left);
      os << "|";
      flag = cisco_print_re_(os, *((regexp_or &) r).right);
      os  << ")";
      break;
   default:
      os << "REGEXP_UNKNOWN";
   }
   return flag;
}

void cisco_print_re(ostream &s, regexp &r) {
   if (!cisco_print_re_(s, r)) // true if the expression was missing $
      s << "_";
}

static void do_print_map(int sno, int eno, int import_flag, int pref, int med, int dpa) {
   
   route_map_is_used = 1;

   cout << "!\n";

   if (strcmp(cisco_map_name, last_cisco_map_name)) {
      strcpy(last_cisco_map_name, cisco_map_name);
      cout << "no route-map " << cisco_map_name << "\n";
      cisco_route_map_no = cisco_map_first_no;
   }
  
   cout << "route-map " << cisco_map_name
        << " permit " << cisco_route_map_no << "\n";
   cisco_route_map_no += cisco_map_inc;

   if (prfx_alist_no && (!import_flag || opt_force_match_ip_inbound))
      cout << " match ip address " << prfx_alist_no << "\n";

   for (int i = sno; i <= eno; i++)
      cout << " match as-path " << i << "\n";

   if (import_flag) {
      if (pref >= 0)
         cout << " set local-preference " << pref << "\n";
   } else {
      if (med >= 0)
         cout << " set metric " << med << "\n";
      else
         if (med == -1) // metric-out=IGP
            cout << " set metric-type internal\n";
      if (dpa >= 0)
         cout << " set dpa " << dpa << "\n";
   }
}

int cisco_print_as_path(regexp_nf& path, int& access_list_id, int import_flag, 
			int pref, int med, int dpa) {
// return the access list number if something is printed
   int last_no_access_list_no = -1;

   if (path.is_universal())
      return 0;

   regexp_nf::RegexpConjunct *rc;
   regexp_nf::RegexpConjunct::ReInt *ri;
   regexp_nf::RegexpConjunct::ReInt *positive_ri = NULL;
   int sno = 0; 

   // we will print as many access lists as necessary, often more than one
   // to minimize the number of access lists generated, we will do 3 passes
   // pass 1: RegexpConjuncts w/ one positive as path and no negative
   // pass 2: RegexpConjuncts w/ one or zero positive as path and negatives
   // pass 3: rest

   int pass1_found = 0;
   for (rc = path.rclist.head(); rc; rc = path.rclist.next(rc->rclist)) {
      ri = rc->regexs.head();
      if (rc->regexs.size() == 1 && !ri->negated) {
	 pass1_found = 1;
         if (last_no_access_list_no != access_list_id) {
            last_no_access_list_no = access_list_id;
            cout << "!\nno ip as-path access-list  " << access_list_id << "\n"; 
         }
         cout << "ip as-path access-list " << access_list_id 
              << " permit ";
         cisco_print_re(cout, *ri->re);
         cout << "\n";
	 rc->mark = 1;
      } else
	 rc->mark = 0;
   }

   // pass 2
   int pass2_found = 0;
   int need_no_ip = ! pass1_found;
   for (rc = path.rclist.head(); rc; rc = path.rclist.next(rc->rclist)) {
      if (rc->mark) // done in pass 1, skip
	 continue;

      int found = 0;
      for (ri = rc->regexs.head(); ri; ri = rc->regexs.next(ri->regexs)) {
	 if (!ri->negated)
	    found++;
	 if (found > 1) 
	    break;
      }
      if (found <= 1) { // really found 
	 pass2_found = 1;
	 rc->mark = 1;
	 if (need_no_ip)
	    cout << "!\nno ip as-path access-list  " << access_list_id << "\n"; 
	 for (ri = rc->regexs.head(); ri; ri = rc->regexs.next(ri->regexs)) {
	    if (!ri->negated)
	       positive_ri = ri;
	    else {
	       cout << "ip as-path access-list " << access_list_id 
		    << " deny ";
	       cisco_print_re(cout, *ri->re);
	       cout << "\n";
	    }
	 }
	 if (positive_ri) {
	    cout << "ip as-path access-list " << access_list_id 
		 << " permit ";
	    cisco_print_re(cout, *positive_ri->re);
	    cout << "\n";
	 } else {
	    cout << "ip as-path access-list " << access_list_id 
		 << " permit .*\n";
	 }
	 do_print_map(access_list_id, access_list_id, 
		      import_flag, pref, med, dpa);
	 access_list_id++;
	 need_no_ip = 1;
     }
   }

   if (pass1_found && !pass2_found) {
      do_print_map(access_list_id, access_list_id, import_flag, pref, med, dpa);
      access_list_id++;
   }

   // pass 3
   for (rc = path.rclist.head(); rc; rc = path.rclist.next(rc->rclist)) {
      if (rc->mark) // done in pass 1 or 2, skip
	 continue;

      // print negative ones all in one access list
      sno = access_list_id;
      cout << "!\nno ip as-path access-list  " << access_list_id << "\n"; 
      for (ri = rc->regexs.head(); ri; ri = rc->regexs.next(ri->regexs)) {
	 if (ri->negated) {
	    cout << "ip as-path access-list " << access_list_id 
		 << " deny ";
	    cisco_print_re(cout, *ri->re);
	    cout << "\n";
	 }
      }
      // print positives ones each in its own access list, except first
      int first = 1;
      for (ri = rc->regexs.head(); ri; ri = rc->regexs.next(ri->regexs)) {
	 if (!ri->negated) {
	    if (first)
	       first = 0;
	    else {
	       access_list_id++;
	       cout << "!\nno ip as-path access-list  " << access_list_id << "\n"; 
 	    }
	    cout << "ip as-path access-list " << access_list_id 
		 << " permit ";
	    cisco_print_re(cout, *ri->re);
	    cout << "\n";
	 }
      }

      if (first) { // no positive one found
	 cout << "ip as-path access-list " << access_list_id 
	      << " permit .*\n";
      }

      do_print_map(sno, access_list_id, import_flag, pref, med, dpa);
      access_list_id++;
   }

   return access_list_id;
}
   

int get_pref(ActionNode *action1, ActionNode *action2,
             intintVHMap &pref_map1, intintVHMap &pref_map2) {
// returns non-negative integer preference
   PrefNode *p;

   int as_in_pref = 0;
   if (action1 && (p = (PrefNode *) action1->FindFirst(T_PrefNode)))
      as_in_pref = pref_map1[p->pref()];

   int interas_in_pref = 0;
   if (action2 && (p = (PrefNode *) action2->FindFirst(T_PrefNode)))
      interas_in_pref = p->pref();
   if (interas_in_pref == -1) // i.e. (pref=MED)
      interas_in_pref = 0;
   else
      interas_in_pref = pref_map2[interas_in_pref];
   
   return 100 + as_in_pref + interas_in_pref;
}

int get_cisco_pref(ActionNode *action1, ActionNode *action2,
                   intintVHMap &pref_map1, intintVHMap &pref_map2) {
   // cisco gives higher preferences to higher numbers
   // this is opposite of the language
   int result = cisco_max_preference + 100 
      - get_pref(action1, action2, pref_map1, pref_map2);
   
   if (result < 0) {
      result = 0;
      cerr << "Warning: negative preference calculated, using 0 instead." 
           << endl;
   }

   return result;
}

int get_med(ActionNode *action1, ActionNode *action2) {
   // returns either non-negative integer metric
   // or -1 to mean (metric-out=IGP)
   // or -2 to mean no metric-out specified
   MEDNode *p;

   int interas_in_med = -2; 
   if (action2 && (p = (MEDNode *) action2->FindFirst(T_MEDNode)))
      interas_in_med = p->med();

   return interas_in_med;
}

int get_dpa(ActionNode *action1, ActionNode *action2) {
   // returns either non-negative integer dpa
   // or -2 to mean no dpa specified
   DPANode *p;

   int interas_in_dpa = -2;
   if (action2 && (p = (DPANode *) action2->FindFirst(T_DPANode)))
      interas_in_dpa = p->dpa();

   return interas_in_dpa;
}

static int do_print(NormalExpression *ne1,
                    NormalExpression *ne2,
                    ActionNode *as, 
                    ActionNode *interas,
                    intintVHMap &pref_map1,
                    intintVHMap &pref_map2, 
                    int import_flag,
                    int& warning_alist_no) {
   int last = 0;

   Debug(Channel(DBG_CISCO) << "# ne1: " << *ne1 << "\n");
   Debug(Channel(DBG_CISCO) << "# ne2: " << *ne2 << "\n");
   ne1->do_and(*ne2);
   Debug(Channel(DBG_CISCO) << "# ne1 and ne2:" << *ne1 << "\n");


   if (ne1->empty()) 
      return last;

   if (ne1->universal()) 
      last = 1;

   for (NormalTerm *nt = ne1->first(); nt; nt = ne1->next()) {

      int pref = get_cisco_pref(as, interas, pref_map1, pref_map2);
      int med = get_med(as, interas);
      int dpa = get_dpa(as, interas);

      int allow_flag = 1;
      if (nt->prfx_set.negated()) {
	 allow_flag = 0;
	 ~(nt->prfx_set);
      }

      prfx_alist_no = 
	 cisco_print_net_list(nt->prfx_set, cisco_access_list_no, allow_flag);

      if (!nt->prfx_set.universal() 
	  && !opt_force_match_ip_inbound && different_pref_count)
	 warning_alist_no = prfx_alist_no;
      else
	 warning_alist_no = -1;


      if (! cisco_print_as_path(nt->as_path, cisco_aspath_access_list_no, 
				import_flag, pref, med, dpa))
	 do_print_map(0, -1, import_flag, pref, med, dpa);

   }

   Debug(Channel(DBG_CISCO) << "# " << *ne1 << "\n");

   return last;
}

void cisco_export (Pix asno, Pix addr, Pix peer_asno, Pix peer_addr) {
   ASPolicy *p;
   Filter_Action *fap;
   Filter_Action *ifap;
   InterASPolicy *ip;
   int interas_match;
   AutNum *autnum;

   // define as
   if (!(autnum = AS_map.define(asno))) {
      cerr << "Error: no object for " << AS_map(asno) << endl;
      return;
    }

   // find peer as
   if (!(p = autnum->find_peer(peer_asno))) {
      cerr << "Error: " << AS_map(asno) 
         << " has no policy for " << AS_map(peer_asno) << endl;
      return;
   }

   if (!(fap = p->out.head())) {
      cerr << "Error: " << AS_map(asno) 
         << " has no export policy for " << AS_map(peer_asno) << endl;
      return;
   }
   
   // find peering
   ip = p->find_peering(addr, peer_addr);
   
   // for each as-out policy
   NormalExpression *fap_exp, *ifap_exp;
   NormalExpression *remaining_exp;
   different_pref_count = 0;
   int last = 0;
   int new_warning_alist_no; // not needed for export
   
   for (; fap && !last; fap = p->out.next(fap->falist)) {
      Debug(Channel(DBG_CISCO) << "# fap: " << *fap->filter << "\n");

      // calculate exports for this interface
      if (ip) { // is there an interas-out policy
         // for each interas-out policy
         for (ifap = ip->out.head(); 
              ifap && !last; 
              ifap = ip->out.next(ifap->falist)) {
            fap_exp = new NormalExpression(*fap->expand());
            ifap_exp = new NormalExpression(*ifap->expand());

            last = do_print(fap_exp, ifap_exp, 
                            fap->action, ifap->action, 
                            autnum->pref_map, p->ipref_map,
                            EXPORT, new_warning_alist_no);

            delete fap_exp;
            delete ifap_exp;
         }
      }

      if (!last) {
         // left over policies
         fap_exp = new NormalExpression(*fap->expand());
         remaining_exp = new NormalExpression(*p->get_remaining_out());

         last = do_print(fap_exp, remaining_exp, fap->action, NULL, 
                         autnum->pref_map, p->ipref_map,
                         EXPORT, new_warning_alist_no);

         delete fap_exp;
         delete remaining_exp;
      }
   }

   static char neighbor[128];
   int2quad(neighbor, Prefask_map(peer_addr).get_prefix());

   if (route_map_is_used)
      cout << "!\nrouter bgp " << AS_map(asno) + 2 << "\n"
	   << "neighbor " << neighbor
	   << " route-map " << cisco_map_name << " out\n";
   else {
      SetOfPrefix s;
      prfx_alist_no = cisco_print_net_list(s, cisco_access_list_no, 1);
      cout << "!\nrouter bgp " << AS_map(asno) + 2 << "\n"
	   << "neighbor " << neighbor
	   << " distribute-list " << prfx_alist_no << " out\n";
   }

   prfx_alist_no = 0;
   route_map_is_used = 0;
}

void cisco_import(Pix asno, Pix addr, Pix peer_asno, Pix peer_addr) {
   ASPolicy *p;
   Filter_Action *fap;
   Filter_Action *ifap;
   InterASPolicy *ip;
   int interas_match;
   AutNum *autnum;

   // define as
   if (!(autnum = AS_map.define(asno))) {
      cerr << "Error: no object for " << AS_map(asno) << endl;
      return;
    }
   
   // find peer as
   if (!(p = autnum->find_peer(peer_asno))) {
      cerr << "Error: " << AS_map(asno) 
         << " has no policy for " << AS_map(peer_asno) << endl;
      return;
   }

   if (!(fap = p->in.head())) {
      cerr << "Error: " << AS_map(asno) 
         << " has no import policy for " << AS_map(peer_asno) << endl;
      return;
   }
   
   // find peering
   ip = p->find_peering(addr, peer_addr);
   
   // for each as-in policy
   NormalExpression *fap_exp, *ifap_exp;
   NormalExpression *remaining_exp;
   different_pref_count = (p->in.size() > 1) || (ip && ip->in.size() > 1);
   int last = 0;
   int warning_alist_no = -1;
   int new_warning_alist_no = -1;
   int issue_warning = 0;

   for (; fap && !last; fap = p->in.next(fap->falist)) {
      // calculate imports for this interface
      if (ip) { // is there an interas-in policy
         // for each interas-in policy
         for (ifap = ip->in.head(); 
              ifap && !last; 
              ifap = ip->in.next(ifap->falist)) {
            fap_exp = new NormalExpression(*fap->expand());
            ifap_exp = new NormalExpression(*ifap->expand());

            last = do_print(fap_exp, ifap_exp, 
                            fap->action, ifap->action, 
                            autnum->pref_map, p->ipref_map,
                            IMPORT, new_warning_alist_no);

            if (warning_alist_no == -1)
               warning_alist_no = new_warning_alist_no;
            else
               if (warning_alist_no != new_warning_alist_no)
                  issue_warning = 1;

            delete fap_exp;
            delete ifap_exp;
         }
      }

      if (!last) {
         // left over policies
         fap_exp = new NormalExpression(*fap->expand());
         remaining_exp = new NormalExpression(*p->get_remaining_in());

         last = do_print(fap_exp, remaining_exp, fap->action, NULL, 
                         autnum->pref_map, p->ipref_map,
                         IMPORT, new_warning_alist_no);

         if (warning_alist_no == -1)
            warning_alist_no = new_warning_alist_no;
         else
            if (warning_alist_no != new_warning_alist_no)
               issue_warning = 1;

         delete fap_exp;
         delete remaining_exp;
      }
   }

   if (issue_warning)
      cerr << 
"Warning: Attempt to assign different local-preference to different routes 
Warning: based on the destination address prefix may not work due to a
Warning: Cisco limitation (upto IOS v 11).
Warning: The limitation is that 'match ip address' does not work for inbound
Warning: route maps (Cisco will support this in the future).
Warning: Use the -force_match_ip_inbound option if it is already supported.
Warning: I will generate an approximate config file anyway." 
           << endl;

   static char neighbor[128];
   int2quad(neighbor, Prefask_map(peer_addr).get_prefix());

   if (! (route_map_is_used 
	  || (prfx_alist_no && ! opt_force_match_ip_inbound))) {
      SetOfPrefix s;
      prfx_alist_no = cisco_print_net_list(s, cisco_access_list_no, 1);
      cout << "!\nrouter bgp " << AS_map(asno) + 2 << "\n"
	   << "neighbor " << neighbor
	   << " distribute-list " << prfx_alist_no << " in\n";
   } else {
      cout << "!\nrouter bgp " << AS_map(asno) + 2 << "\n";
      if (route_map_is_used)
	 cout << "neighbor " << neighbor
	      << " route-map " << cisco_map_name << " in\n";
      
      if (prfx_alist_no && ! opt_force_match_ip_inbound)
	 cout << "neighbor " << neighbor
	      << " distribute-list " << prfx_alist_no << " in\n";
   }

   prfx_alist_no = 0;
   route_map_is_used = 0;
}
