// $Id: Node.h 1.4 Fri, 18 Jul 1997 16:00:26 -0700 wlee $
// 
//  Copyright (c) 1994 by the University of Southern California
//  and/or the International Business Machines Corporation.
//  All rights reserved.
//
//  Permission to use, copy, modify, and distribute this software and
//  its documentation in source and binary forms for lawful
//  non-commercial purposes and without fee is hereby granted, provided
//  that the above copyright notice appear in all copies and that both
//  the copyright notice and this permission notice appear in supporting
//  documentation, and that any documentation, advertising materials,
//  and other materials related to such distribution and use acknowledge
//  that the software was developed by the University of Southern
//  California, Information Sciences Institute and/or the International
//  Business Machines Corporation.  The name of the USC or IBM may not
//  be used to endorse or promote products derived from this software
//  without specific prior written permission.
//
//  NEITHER THE UNIVERSITY OF SOUTHERN CALIFORNIA NOR INTERNATIONAL
//  BUSINESS MACHINES CORPORATION MAKES ANY REPRESENTATIONS ABOUT
//  THE SUITABILITY OF THIS SOFTWARE FOR ANY PURPOSE.  THIS SOFTWARE IS
//  PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES,
//  INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
//  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, TITLE, AND 
//  NON-INFRINGEMENT.
//
//  IN NO EVENT SHALL USC, IBM, OR ANY OTHER CONTRIBUTOR BE LIABLE FOR ANY
//  SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES, WHETHER IN CONTRACT,
//  TORT, OR OTHER FORM OF ACTION, ARISING OUT OF OR IN CONNECTION WITH,
//  THE USE OR PERFORMANCE OF THIS SOFTWARE.
//
//  Questions concerning this software should be directed to 
//  info-ra@isi.edu.
//
//  Author(s): Cengiz Alaettinoglu (cengiz@isi.edu)

/*
 *[C] The Regents of the University of Michigan and Merit Network, Inc.1993 
 *All Rights Reserved 
 *  
 *  Permission to use, copy, modify, and distribute this software and its 
 *  documentation for any purpose and without fee is hereby granted, provided 
 *  that the above copyright notice and this permission notice appear in all 
 *  copies of the software and derivative works or modified versions thereof, 
 *  and that both the copyright notice and this permission and disclaimer 
 *  notice appear in supporting documentation. 
 *   
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER 
 *   EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION WARRANTIES OF 
 *   MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE REGENTS OF THE 
 *   UNIVERSITY OF MICHIGAN AND MERIT NETWORK, INC. DO NOT WARRANT THAT THE 
 *   FUNCTIONS CONTAINED IN THE SOFTWARE WILL MEET LICENSEE'S REQUIREMENTS OR 
 *   THAT OPERATION WILL BE UNINTERRUPTED OR ERROR FREE. The Regents of the 
 *   University of Michigan and Merit Network, Inc. shall not be liable for any 
 *   special, indirect, incidental or consequential damages with respect to any 
 *   claim by Licensee or any third party arising from use of the software. 
 */

#ifndef NODE_H
#define NODE_H

#include "config.hh"
#include <cstdarg>

#include <iostream.h>
#include <Pix.h>
#include "_SetOfPix.hh"
//#include "SetOfRoute.hh"
#include "debug.hh"

enum Node_t { T_ANDNode,
	      T_ORNode,
	      T_NotNode,
	      T_AssignNode,
	      T_ANYNode,
	      T_ASNode,
	      T_ASMacroNode,
	      T_CommNode,
	      T_NetListNode,
	      T_ASPathNode,
	      T_RegisterNode,
	      T_PrefNode,
	      T_MEDNode,
	      T_DPANode,
	      T_NoopNode,
	      T_ComposeNode
};

#ifndef TRUE
#define TRUE 1
#define FALSE 0
#endif

class Route;
class NormalExpression; 

class Node {
  public:
   Node_t type;
  protected:
   virtual void print(ostream &os) = 0;
  public:
   virtual operator== (Node &b) = 0;
   virtual void InOrderPrint() = 0;
   virtual void PostOrderPrint() { InOrderPrint(); }
   virtual ~Node() {};
   friend ostream &operator<<(ostream &os, Node *no) {
      no->print(os);
      return os;
   }
   friend ostream &operator<<(ostream &os, Node &no) {
      return os << &no;
   }
};

class FilterNode : public Node {
  protected:
   virtual void print(ostream &os) = 0;

  public:
   virtual NormalExpression *Evaluate(int expand) = 0;
   virtual int match(Route &r, char dontcare) = 0;
   virtual int match_exact_or_more_specific(Route &r, char dontcare) {
      return match(r, dontcare);
   }
   virtual int match_exact_or_more_specific(Route &r) {
      return match_exact_or_more_specific(r, 1) == TRUE ? TRUE : FALSE;
   }
   virtual int match(Route &r) {
      return match(r, 1) == TRUE ? TRUE : FALSE;
   }
   friend ostream &operator<<(ostream &os, FilterNode *fn) {
      fn->print(os);
      return os;
   }
   friend ostream &operator<<(ostream &os, FilterNode &fn) {
      return os << &fn;
   }
};

class ActionNode : public Node {
  protected:
   virtual void print(ostream &os) = 0;

  public:
   virtual Execute(Route &r) = 0;
   virtual ActionNode *FindFirst(Node_t _type) = 0;
   int operator< (ActionNode &b);
   friend ostream &operator<<(ostream &os, ActionNode *an) {
      an->print(os);
      return os;
   }
   friend ostream &operator<<(ostream &os, ActionNode &an) {
      return os << &an;
   }
};


////////////////////////////// ANDNode //////////////////////////////

class ANDNode : public FilterNode {
   friend FilterNode *rsd_convert_bol(FilterNode *n, int asno);
  private:
   FilterNode *left, *right;         // and of left child and right child
  protected:
   virtual void print(ostream &os);
  public:
   ANDNode(FilterNode *l, FilterNode *r) {
      left = l;
      right = r;
      type = T_ANDNode;
   }

   virtual operator== (Node& b) {
      return b.type == T_ANDNode 
	 && left  == ((ANDNode&) b).left
	 && right == ((ANDNode&) b).right;
   }

   virtual void InOrderPrint();
   virtual void PostOrderPrint();
   virtual NormalExpression *Evaluate(int expand);
   virtual int match(Route &r, char dontcare);
   virtual ~ANDNode();
};

////////////////////////////// ORNode //////////////////////////////

class ORNode : public FilterNode {
   friend FilterNode *rsd_convert_bol(FilterNode *n, int asno);
  private:
   FilterNode *left, *right;         // or of left child and right child
  protected:
   virtual void print(ostream &os);
  public:
   ORNode(FilterNode *l, FilterNode *r) {
      left = l;
      right = r;
      type = T_ORNode;
   }

   virtual operator== (Node& b) {
      return b.type == T_ORNode 
	 && left  == ((ORNode&) b).left
	 && right == ((ORNode&) b).right;
   }

   virtual void InOrderPrint();
   virtual void PostOrderPrint();
   virtual NormalExpression *Evaluate(int expand);
   virtual int match(Route &r, char dontcare);
   virtual ~ORNode();
};

////////////////////////////// NotNode //////////////////////////////

class NotNode : public FilterNode {
   friend FilterNode *rsd_convert_bol(FilterNode *n, int asno);
  private:
   FilterNode *left;         // not of left child
  protected:
   virtual void print(ostream &os);
  public:
   NotNode(FilterNode *l) {
      left = l;
      type = T_NotNode;
   }

   virtual operator== (Node& b) {
      return b.type == T_NotNode 
	 && left  == ((NotNode&) b).left;
   }

   virtual void InOrderPrint();
   virtual void PostOrderPrint();
   virtual NormalExpression *Evaluate(int expand);
   virtual int match(Route &r, char dontcare);
   virtual ~NotNode();
};

////////////////////////////// AssignNode //////////////////////////////

class AssignNode : public FilterNode {
private:
   Pix val;
   FilterNode *left;         

protected:
   virtual void print(ostream &os);

public:
   AssignNode(Pix pi, FilterNode *n) {
      val = pi;
      left = n;
      type = T_AssignNode;
   }

   virtual operator== (Node& b) {
      return b.type == T_AssignNode 
	 && val == ((AssignNode&) b).val
	 && left  == ((AssignNode&) b).left;
   }

   virtual void InOrderPrint();
   virtual void PostOrderPrint();
   virtual NormalExpression *Evaluate(int expand);
   virtual int match(Route &r, char dontcare);
   virtual ~AssignNode();
};


////////////////////////////// ANYNode //////////////////////////////

class ANYNode : public FilterNode {
protected:
   virtual void print(ostream &os);

public:
   ANYNode() {
      type = T_ANYNode;
   }

   virtual operator== (Node& b) {
      return b.type == T_ANYNode;
   }

   virtual void InOrderPrint();
   virtual NormalExpression *Evaluate(int expand);
   virtual int match(Route &r, char dontcare);
};

////////////////////////////// ASNode //////////////////////////////

class ASNode : public FilterNode {
  private:
   Pix val;

  protected:
   virtual void print(ostream &os);

  public:
   ASNode(Pix pi) {
      val  = pi;
      type = T_ASNode;
   }

   virtual operator== (Node& b) {
      return b.type == T_ASNode 
	 && val == ((ASNode&) b).val;
   }

   virtual void InOrderPrint();
   virtual NormalExpression *Evaluate(int expand);
   virtual int match(Route &r, char dontcare);
};

////////////////////////////// ASMacroNode //////////////////////////////

class ASMacroNode : public FilterNode {
 private:
   Pix val;

 protected:
   virtual void print(ostream &os);

 public:
   ASMacroNode(Pix p) {
      val  = p;
      type = T_ASMacroNode;
  }

   virtual operator== (Node& b) {
      return b.type == T_ASMacroNode 
	 && val == ((ASMacroNode&) b).val;
   }

   virtual void InOrderPrint();
   virtual NormalExpression *Evaluate(int expand);
   virtual int match(Route &r, char dontcare);
};

////////////////////////////// CommNode //////////////////////////////

class CommNode : public FilterNode {
 private:
   Pix val;
   
 protected:
   virtual void print(ostream &os);

 public:
   CommNode(Pix p) {
      val  = p;
      type = T_CommNode;
   }

    virtual operator== (Node& b) {
      return b.type == T_CommNode 
	 && val == ((CommNode&) b).val;
   }

   virtual void InOrderPrint();
   virtual NormalExpression *Evaluate(int expand);
   virtual int match(Route &r, char dontcare);
};

////////////////////////////// NetListNode //////////////////////////////

class radix_node;

class NetListNode : public FilterNode {
 private:
//   SetOfRoute nets;
   _SetOfPix nets;
   radix_node *cidr_root;
 protected:
   virtual void print(ostream &os);
 public:
   NetListNode() {
      type = T_NetListNode;
      cidr_root = NULL;
   }

   virtual operator== (Node& b) {
      return b.type == T_NetListNode 
	 && nets == ((NetListNode&) b).nets;
   }

   virtual void InOrderPrint();
   virtual NormalExpression *Evaluate(int expand);
   virtual int match(Route &r, char dontcare);
   virtual int match_exact_or_more_specific(Route &r, char dontcare);
   add(Pix pi) { nets.add(pi); }
};

////////////////////////////// ASPathNode //////////////////////////////

class regexp;

class ASPathNode : public FilterNode {
   friend FilterNode *rsd_convert_bol(FilterNode *n, int asno);
 private:
   regexp *re;

 protected:
   virtual void print(ostream &os);

 public:
   ASPathNode(regexp *r) {
      re   = r;
      type = T_ASPathNode;
   }

   virtual operator== (Node& b) {
      ASSERT(0);
      return 0;
   }

   virtual void InOrderPrint();
   virtual NormalExpression *Evaluate(int expand);
   virtual int match(Route &r, char dontcare);
   virtual ~ASPathNode();
};

////////////////////////////// RegisterNode //////////////////////////////

class RegisterNode : public FilterNode {
 private:
   Pix val;

 protected:
   virtual void print(ostream &os);

  public:
   RegisterNode(Pix pi) {
      val  = pi;
      type = T_RegisterNode;
   }

    virtual operator== (Node& b) {
      return b.type == T_RegisterNode 
	 && val == ((RegisterNode&) b).val;
   }

   virtual void InOrderPrint();
   virtual NormalExpression *Evaluate(int expand);
   virtual int match(Route &r, char dontcare);
};

////////////////////////////// PrefNode //////////////////////////////

class PrefNode : public ActionNode {
private:
   int val; // a non-neg intefer of -1 to mean use MED
 protected:
   virtual void print(ostream &os);
public:
   PrefNode(int _val) : val(_val) {
      type = T_PrefNode;
   }

   virtual operator== (Node& b) {
      return b.type == T_PrefNode 
	 && val == ((PrefNode&) b).val;
   }

   virtual void InOrderPrint();
   virtual Execute(Route &r) {
   }
   virtual ActionNode *FindFirst(Node_t _type) {
      if (type == _type)
	 return this;
      return NULL;
   }
   int pref() { 
      return val; 
   }
};

////////////////////////////// MEDNode //////////////////////////////

class MEDNode : public ActionNode {
private:
   int val; // a non-neg intefer of -1 to mean use MED

 protected:
   virtual void print(ostream &os);

public:
   MEDNode(int _val) : val(_val) {
      type = T_MEDNode;
   }

   virtual operator== (Node& b) {
      return b.type == T_MEDNode 
	 && val == ((MEDNode&) b).val;
   }

   virtual void InOrderPrint();
   virtual Execute(Route &r) {
   }
   virtual ActionNode *FindFirst(Node_t _type) {
      if (type == _type)
	 return this;
      return NULL;
   }
   int med() { 
      return val; 
   }
};

////////////////////////////// DPANode //////////////////////////////

class DPANode : public ActionNode {
private:
   int val; // a non-neg intefer

 protected:
   virtual void print(ostream &os);

public:
   DPANode(int _val) : val(_val) {
      type = T_DPANode;
   }

   virtual operator== (Node& b) {
      return b.type == T_DPANode 
	 && val == ((DPANode&) b).val;
   }

   virtual void InOrderPrint();
   virtual Execute(Route &r) {
   }
   virtual ActionNode *FindFirst(Node_t _type) {
      if (type == _type)
	 return this;
      return NULL;
   }
   int dpa() { 
      return val; 
   }
};

////////////////////////////// NoopNode //////////////////////////////

class NoopNode : public ActionNode {
 protected:
   virtual void print(ostream &os);
public:
   NoopNode() {
      type = T_NoopNode;
   }

   virtual operator== (Node& b) {
      return b.type == T_NoopNode;
   }

   virtual void InOrderPrint();
   virtual Execute(Route &r) {}
   virtual ActionNode *FindFirst(Node_t _type) {
      if (type == _type)
	 return this;
      return NULL;
   }
};


////////////////////////////// ComposeNode //////////////////////////////

class ComposeNode : public ActionNode {
private:
   ActionNode *left, *right;         // left followed by right
 protected:
   virtual void print(ostream &os);
public:
   ComposeNode(ActionNode *l, ActionNode *r) {
      left = l;
      right = r;
      type = T_ComposeNode;
   }
   virtual ~ComposeNode();

   virtual operator== (Node& b) {
      return b.type == T_ComposeNode
	 && left  == ((ComposeNode&) b).left
	 && right == ((ComposeNode&) b).right;
   }

   virtual void InOrderPrint();
   virtual Execute(Route &r) {}
   virtual ActionNode *FindFirst(Node_t _type) {
      ActionNode *r;
      if (type == _type)
	 r = this;
      else {
	 r = left->FindFirst(_type);
	 if (!r)
	    r = right->FindFirst(_type);
      }
      return r;
   }
};


#endif // NODE_H
