//  $Id: aoe.cc 1.8 Fri, 31 Oct 1997 18:34:26 -0800 wlee $
// 
//  Copyright (c) 1994 by the University of Southern California
//  and/or the International Business Machines Corporation.
//  All rights reserved.
//
//  Permission to use, copy, modify, and distribute this software and
//  its documentation in source and binary forms for lawful
//  non-commercial purposes and without fee is hereby granted, provided
//  that the above copyright notice appear in all copies and that both
//  the copyright notice and this permission notice appear in supporting
//  documentation, and that any documentation, advertising materials,
//  and other materials related to such distribution and use acknowledge
//  that the software was developed by the University of Southern
//  California, Information Sciences Institute and/or the International
//  Business Machines Corporation.  The name of the USC or IBM may not
//  be used to endorse or promote products derived from this software
//  without specific prior written permission.
//
//  NEITHER THE UNIVERSITY OF SOUTHERN CALIFORNIA NOR INTERNATIONAL
//  BUSINESS MACHINES CORPORATION MAKES ANY REPRESENTATIONS ABOUT
//  THE SUITABILITY OF THIS SOFTWARE FOR ANY PURPOSE.  THIS SOFTWARE IS
//  PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES,
//  INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
//  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, TITLE, AND 
//  NON-INFRINGEMENT.
//
//  IN NO EVENT SHALL USC, IBM, OR ANY OTHER CONTRIBUTOR BE LIABLE FOR ANY
//  SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES, WHETHER IN CONTRACT,
//  TORT, OR OTHER FORM OF ACTION, ARISING OUT OF OR IN CONNECTION WITH,
//  THE USE OR PERFORMANCE OF THIS SOFTWARE.
//
//  Questions concerning this software should be directed to 
//  info-ra@isi.edu.
//
//  Author(s): WeeSan Lee (wlee@isi.edu)


#if !defined(IDENT_OFF)
#if __GNUC__ > 2 || (__GNUC__ == 2 && __GNUC_MINOR__ >= 7) 
static char ident[] = 
   "@(#)$Id: aoe.cc 1.8 Fri, 31 Oct 1997 18:34:26 -0800 wlee $";
#endif
#endif


#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <cctype>
#include <strstream.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "aoe.hh"
#include "ASMap.h"      // To include AS_map global variable
#include "Argv.hh"
#include "version.hh"
#include "trace.hh"
#include "aoe.xbm"
#include "rusage.hh"
#include "net.hh"       // for class ipAddr
#include "TclVar.hh"

// Note:
// 1. AS_Map is a global declared class variable.


//*******************************
// Some TK variables defined here
char *tkASPeerList                 = ".body.aspeer.listbox.list";
char *tkPolicyText                 = ".body.workingpolicy.text";
char *tkDefLabel                   = ".body.templatearea.deflabel";
char *tkDefText                    = ".body.templatearea.def.text";
char *tkNeighborLabel              = ".body.templatearea.neighborlabel";
char *tkNeighborText               = ".body.templatearea.neighbor.text";
char *tkExportButton               = ".body.templatebutton.export";
char *tkImportButton               = ".body.templatebutton.import";
char *tkCategoryButton             = ".body.templatebutton.category.menu";
char *tkLogoButton		   = ".body.templatebutton.logo";
char *tkCategoryVar                = "categoryVar";
char *tkCommitButton               = ".body.workingpolicybutton.commit";
char *tkPolicyEditButton           = ".body.workingpolicybutton.edit";
char *tkPolicyShowButton           = ".body.workingpolicybutton.show";
char *tkStatusLine                 = ".statusline.content";


//***********************************
// Some global variables defined here
Rusage ru;
int opt_rusage   = 0;
char *opt_my_as  = NULL;
char *opt_rcfile = NULL;
char *display    = NULL;


//**********************
// Define the usage here
void usage(void)  
{
   fprintf(stderr, "\nUsage: aoe  [-help] [-T all | whois_query | whois_response | input]"
	           "\n            [-D debug_level] [-version]"
	           "\n            [-h server_ip] [-p server_port] [-s db_order]"
	           "\n            [-rusage] [-ignore_errors] [-report_errors]"
	           "\n            [-display  display]"
	           "\n            [-as as_no]"
	           "\n\n\a\a"); 
}


//**************************
// Some initialization stuff 
int start_tracing(char *dst, char *key, char *nextArg) {
   if (nextArg)
      trace.enable(nextArg);
   return 1; // return 1 to signify nextArg is used by us
}

int start_debugging(char *dst, char *key, char *nextArg) {
   Debug(if (nextArg) dbg.enable(atoi(argv[i])));
   return 1; // return 1 to signify nextArg is used by us
}

void init_and_set_options (int argc, char **argv, char **envp) {
   ArgvInfo argTable[] = {
      // RAToolSet common arguments
      // key, type, src, dst, help
      {"-T", ARGV_FUNC, (char *) &start_tracing,      (char *) NULL,
       "Start tracing the next argument"},
      {"-D", ARGV_FUNC, (char *) &start_debugging,    (char *) NULL,
       "Start debugging the next argument"},
      {"-version", ARGV_FUNC, (char *) &version,      (char *) NULL,
       "Show version"},
      {"-h", ARGV_FUNC, (char *) &Whois::ArgvHost,    (char *) NULL,
       "Host name of the RAWhoisd server"},
      {"-p", ARGV_FUNC, (char *) &Whois::ArgvPort,    (char *) NULL,
       "Port number of the RAWhoisd server"},
      {"-s", ARGV_FUNC, (char *) &Whois::ArgvSources, (char *) NULL,
       "Order of databases"},
      {"-rusage", ARGV_CONSTANT, (char *) 1,          (char *) &opt_rusage,
       "On termination print resource usage"},
      {"-ignore_errors", ARGV_FUNC, (char *)&Whois::IgnoreErrors, (char *)NULL,
       "Ignore IRR error and warning messages"},
      {"-report_errors", ARGV_FUNC, (char *)&Whois::ReportErrors, (char *)NULL,
       "Print IRR error and warning messages"},

      // aoe specific arguments
      {"-as",  ARGV_STRING,    (char *) NULL,    (char *) &opt_my_as,
       "AS number of the aut-num object to use"},
      {"-rcfile", ARGV_STRING, (char *)NULL, (char *)&opt_rcfile,
       "aoe resource file (default: ~/.aoerc"},
      {"-display", ARGV_STRING, (char *)NULL, (char *)&display,
       "X display"},

      // End of argument
      {(char *) NULL, ARGV_END, (char *) NULL, (char *) NULL, (char *) NULL}
   };

   for (char **p = envp; *p != NULL; p++) {
      if (strncmp(*p, "IRR_HOST=", 9) == 0)  {
         whois.SetDefaultHost(*p + 9);
         continue;
      }
      if (strncmp(*p, "IRR_PORT=", 9) == 0)  {
         whois.SetDefaultPort(atoi(*p + 9));
         continue;
      }
      if (strncmp(*p, "IRR_SOURCES=", 12) == 0)  {
         whois.SetDefaultSources(*p + 12);
         continue;
      }
   }

   if (ParseArgv(&argc, argv, argTable, 0) != ARGV_OK) {
      usage();
      exit(1);
   }

   switch (argc) {
   case 2: 
      opt_my_as = argv[1];
      break;
   case 1:
      if (!opt_my_as) {
	 ipAddr myip;
	 myip.setToMyHost();
	 opt_my_as = AS_map(AS_map.add_entry(myip));
      }
      break;
   default:
      usage();
      exit(1);
   }
}


//**************************
// Some misc. functions here
static char *strupr(char *c) 
{
   if (!c) return NULL;
   for (char *p = c; *p; p++)
      if (isascii(*p) && islower(*p))
         *p = toupper(*p);
   return c;
}

TclText &operator<<(TclText &tt, AutNum &au)
{
   strstream ss; 
   //   tt.enable();
   tt.clear();
   ss << au << '\0';       // Always null-terminated!!!
   tt.insert(ss.str());
   tt.removeLastLine();    // Usually a blank line at the end
   tt.addLineTag("tag", "0", "14");
   tt.configureTag("tag", "-foreground DeepPink");
   tt.addLineTag("content", "14", "end");
   tt.configureTag("content", "-foreground blue");
   ss.rdbuf()->freeze(0);  // Users have to unfreeze the buffer explicitly
   //   tt.disable();
   return tt;
}

TclText &operator<<(TclText &t1, TclText &t2)
{
   const char *p = t2.getAll();
   if (*p == '\n') return t1;
   //   t1.enable();
   //   t1.insert("\n");
   t1.insert(p);
   //   t1.removeLastLine();
   //   t1.disable();
   return t1;
}

TclList &operator<<(TclList &tl, ListHead<ASPeer> &lh)
{
   char pzcBuffer[64];
   for (ASPeer *pcASPeer = lh.head();
	pcASPeer;
	pcASPeer = lh.next(pcASPeer->cNode))
      {
      switch (pcASPeer->getType())
	 {
	 case dASPeerFromIRR:
	    sprintf(pzcBuffer, "{%-8s (IRR)}", 
		    AS_map(pcASPeer->getNoInPix()));
	    break;
	 case dASPeerFromBGP:
	    sprintf(pzcBuffer, "{%-8s (BGP)}", 
		    AS_map(pcASPeer->getNoInPix()));
	    break;
	 case dASPeerFromBoth:
	    sprintf(pzcBuffer, "{%-8s (IRR & BGP)}", 
		    AS_map(pcASPeer->getNoInPix()));
	    break;
	 case dASPeerNew:
	    sprintf(pzcBuffer, "{%-8s (NEW)}",
		    AS_map(pcASPeer->getNoInPix()));
	    break;
	 default:
	    sprintf(pzcBuffer, "{Shoudn't happen!}");
	    break;
	 }
      tl.insert(pzcBuffer);
      }
   return tl;
}

ListHead<ASPeer> &operator<<(ListHead<ASPeer> &lh, BgpASPath &ap)
{
   ASPeer *pcASPeer;
   FirstAndLastAS *pcFirstAndLastAS;
   for (pcFirstAndLastAS = ap.cFirstAndLastASes.head(); 
	pcFirstAndLastAS; 
	pcFirstAndLastAS = ap.cFirstAndLastASes.next(pcFirstAndLastAS->cNode))
      {
      for (pcASPeer = lh.head(); pcASPeer; pcASPeer = lh.next(pcASPeer->cNode))
	 if (pcASPeer->getNoInPix() == pcFirstAndLastAS->pvFirst) 
	    break;
      if (pcASPeer) 
	 // Found!
	 pcASPeer->setType(pcASPeer->getType() | dASPeerFromBGP);
      else
	 {
	 // Not found!
	 pcASPeer = new ASPeer(pcFirstAndLastAS->pvFirst, dASPeerFromBGP);
	 lh.append(pcASPeer->cNode);
	 }
      }
   return lh;
}



//***************************
// Command classes start here

int ListPeer::updateWindowTitle(TclApplication *pcTcl, int iCategory, 
				char *pzcASNo)
{
   char *title, *dummy = "Shouldn't happen!";
   char pzcBuffer[128];
   AoeApplication *pcApp = (AoeApplication *)pcTcl;
   switch (iCategory)
      {
      case 0: title  = "from IRR"; break;
      case 1: title  = "from BGP Dump"; break;
      case 2: title  = "from Peer's aut-num"; break;
      case 3: title  = dummy; break;  // Shouldn't happen!
      default: 
	{
	sprintf(pzcBuffer, "PolicyTemplate(%d,Description)", iCategory - 4);
	title = pcApp->getVar(pzcBuffer);
	break;
	}
      }
   sprintf(pzcBuffer, "%s -- %s", pzcASNo, title);
   // Update label on top of def text widget
   TCL(!pcApp->pcDefLabel->attach(pzcBuffer));
   // Update title of main window
   TCL(!pcApp->evalf("wm title . {%s (%s)}", pcApp->getAppName(), pzcBuffer));
   return TCL_OK;
}

int ListPeer::displayIRRPolicy(int iImport, int iExport, Pix pvASPeer,
			       TclText *pcText, AutNum *pcAutNum)
{
   // Clear the text widget first no matter if the peer exist or not!
  //   pcText->enable();
   pcText->clear();

   ASPolicy *p = pcAutNum->find_peer(pvASPeer);
   if (p) 
      {
      strstream ss;
      if (iImport) p->printASIn(ss);
      if (iExport) p->printASOut(ss);
      ss << '\0';  // This is a must otherwise garbage will appear
      // Fill the text widget
      pcText->insert(ss.str());
      pcText->removeLastLine();
      ss.rdbuf()->freeze(0);  // This is also a must
      }

   //   pcText->disable();

   return 1;
}

int ListPeer::generatePolicyFromBGP(int iImport, int iExport,
				    Pix pvAS, Pix pvASPeer, TclText *pcText, 
				    BgpASPath *pcBgpASPath, int iASPeerType)
{
   strstream ss;

   // Clear the text widget first no matter if the peer exist or not!
   //   pcText->enable();
   pcText->clear();   

   if (iASPeerType & dASPeerFromBGP)
      {
      if (iImport)
	 {
	 // as-in
	    ss << "as-in:        from "
	       << AS_map(pvASPeer)
	       << " 10 accept ";
	    FirstAndLastAS *pcFirstAndLastAS = pcBgpASPath->find(pvASPeer);
	    for (Pix pix = pcFirstAndLastAS->pvLast.first(); 
		 pix; 
		 pcFirstAndLastAS->pvLast.next(pix))
	       ss << AS_map(pcFirstAndLastAS->pvLast(pix)) 
		  << " ";
	    ss << endl;
	 }
      if (iExport)
	 // as-out
	 // Temp hack ???
	 ss << "as-out:       to "
	    << AS_map(pvASPeer)
	    << " announce ANY"
	    << endl;
      }

   // Fill the text widget
   ss << '\0';
   pcText->insert(ss.str());
   pcText->removeLastLine();
   //   pcText->disable();
   ss.rdbuf()->freeze(0);  // This is also a must
   return 1;
}

int ListPeer::generatePolicyFromPeer(int iImport, int iExport,
				     Pix pvAS, Pix pvASPeer, 
				     TclText *pcText, int iASPeerType)
{
   strstream ss;

   // Clear the text widget first no matter if the peer exist or not!
   //   pcText->enable();
   pcText->clear();   

   //   AutNum *pcAutNumPeer = AS_map.define(pvASPeer);
   AutNum *pcAutNumPeer = (AutNum *)AS_map.contents(pvASPeer).definition;
   ASPolicy *pcPolicyPeer = pcAutNumPeer->find_peer(pvAS);

   if (pcPolicyPeer && (iASPeerType & dASPeerFromIRR))
      {
      if (iImport) 
	{
        // Get the as-out of peer
	for (Filter_Action *fap = pcPolicyPeer->out.head(); 
	     fap; 
	     fap = pcPolicyPeer->out.next(fap->falist))
	  ss << "as-in:        from "
	     << AS_map(pvASPeer)
	     << " 10 accept "
	     << fap->filter 
	     << endl;
	}
      if (iExport)
	{
        // Get the as-in of peer
	for (Filter_Action *fap = pcPolicyPeer->in.head(); 
	     fap; 
	     fap = pcPolicyPeer->in.next(fap->falist))
	  ss << "as-out:       to "
	     << AS_map(pvASPeer)
	     << " announce "
	     << fap->filter 
	     << endl;
	}
      }

   // Fill the text widget
   ss << '\0';
   pcText->insert(ss.str());
   pcText->removeLastLine();
   //   pcText->disable();
   ss.rdbuf()->freeze(0);  // This is also a must
   return 1;
}

int ListPeer::generateTemplatePolicy(int iCategory, int iImport, int iExport,
				     Pix pvAS, Pix pvASPeer, TclText *pcText)
{
   TclApplication *pcApp = (TclApplication *)getExtraArgument();
   strstream ss;
   char pzcBuffer[128];
   // Clear the text widget first no matter if the peer exist or not!
   //   pcText->enable();
   pcText->clear();   

   if (iImport)
      {
      // as-in
      sprintf(pzcBuffer, "PolicyTemplate(%d,as-in)", iCategory - 4);
      ss << "as-in:        "
	 << substitute(pcApp->getVar(pzcBuffer), pvAS, pvASPeer)
	 << endl;
      }
   if (iExport)
      {
      // as-out
      sprintf(pzcBuffer, "PolicyTemplate(%d,as-out)", iCategory - 4);
      ss << "as-out:       "
	 << substitute(pcApp->getVar(pzcBuffer), pvAS, pvASPeer)
	 << endl;
      }
   // Fill the text widget
   ss << '\0';
   pcText->insert(ss.str());
   pcText->removeLastLine();
   //   pcText->disable();
   ss.rdbuf()->freeze(0);  // This is also a must
   return 1;
}

char *ListPeer::substitute(char *pzcString, Pix pvAS, Pix pvASPeer)
{
   static char pzcBuffer[dBufferLength + 1 + 32];
   char *pzcMyAS = "$MyAS", *pzcPeerAS = "$PeerAS";
   char *p1, *p2;
   int iMyASLen = strlen(pzcMyAS), iPeerASLen = strlen(pzcPeerAS);
   int iTotal = 0;
   p1 = p2 = pzcString;
   pzcBuffer[0] = 0;
   while (p2 = strchr(p2, '$'))
      {      
      int iLen = p2 - p1;
      strncpy(pzcBuffer + iTotal, p1, iLen);
      iTotal += iLen;
      if (strncmp(p2, pzcMyAS, iMyASLen) == 0)
	 {
	 p2 += iMyASLen;
	 char *p = AS_map(pvAS);
	 iLen = strlen(p);
	 strncpy(pzcBuffer + iTotal, p, iLen);
	 iTotal += iLen;
	 }
      else
	if (strncmp(p2, pzcPeerAS, iPeerASLen) == 0)
	   {
	   p2 += iPeerASLen;
	   char *p = AS_map(pvASPeer);
	   iLen = strlen(p);
	   strncpy(pzcBuffer + iTotal, p, iLen);
	   iTotal += iLen;
	   }
        else
	   strncpy(pzcBuffer + iTotal++, p2++, 1); 
      p1 = p2;
      }
   pzcBuffer[iTotal] = 0;
   // Copy the rest
   if (*p1) strcat(pzcBuffer, p1);
   return pzcBuffer;
}

int ListPeer::command(int argc, char *argv[])
{
   if (argc != 2) return TCL_ERROR;
   AoeApplication *pcApp = (AoeApplication *)getExtraArgument();
/*
   printf("\nAS = %s, Name = %s, argc = %d, argv[0] = %s, argv[1] = %s", 
	  pcApp->getASNo(), getName(), argc, argv[0], argv[1]);
*/
   // Check if the list is empty of not,
   // If empty, it is NOT an error but do nothing
   if (!argv[1][0]) 
      {
      if (pcApp->cASPeers.head())
	 {	
	 if (!pcApp->eval("showWarning {Please point the peer in the listbox "
			  "on the left hand side you are going to show!}")) 
	    return TCL_ERROR;
	 }
      else
	 {
	 if (!pcApp->eval("showWarning {Peer list is empty now!}")) 
	    return TCL_ERROR;
	 }
      return TCL_OK;
      }

   int iImport      = pcApp->pcImportButton->getOption();
   int iExport      = pcApp->pcExportButton->getOption();
   int iCategory    = pcApp->pcCategoryButton->getOption();
   int iASPeerIndex = atoi(argv[1]);
   int iASPeerType  = pcApp->getASPeerType(iASPeerIndex);
   Pix pvAS         = pcApp->getASNoInPix();
   Pix pvASPeer     = pcApp->getASPeerNoInPix(iASPeerIndex);

   // Setup $PeerAS properly
   if (!pcApp->evalf("set PeerAS %s", AS_map(pvASPeer))) return TCL_ERROR;

   // Make the window title more meaningful
   updateWindowTitle(pcApp, iCategory, pcApp->getASNoInString());
   if (!pcApp->pcNeighborLabel->attach(AS_map(pvASPeer))) return TCL_ERROR;
   pcApp->pcPolicyShowButton->labelBothString(AS_map(pvASPeer));
   if (!pcApp->eval("showPolicyText")) return TCL_ERROR;
   

   // Try to define peer first.  It's fine to do it everytime 
   // since it will be cached in the memory and won't be duplicated
   AutNum *pcAutNumPeer = AS_map.define(pvASPeer);
   // Display peer of working AS
   displayIRRPolicy(iExport, iImport, pcApp->getASNoInPix(), 
			  pcApp->pcNeighborText, pcAutNumPeer);

   switch (iCategory)
      {
      case 0: // from IRR Database
	 {
	 // Display working AS
	 displayIRRPolicy(iImport, iExport, pvASPeer, 
			  pcApp->pcDefText, pcApp->pcAutNum);
/*
	 // Try to define peer first.  It's fine to do it everytime 
	 // since it will be cached in the memory and won't be duplicated
	 AutNum *pcAutNumPeer = AS_map.define(pvASPeer);

	 // Display peer of working AS
	 displayIRRPolicy(iExport, iImport, pcApp->getASNoInPix(), 
			  pcApp->pcNeighborText, pcAutNumPeer);
*/
	 break;
	 }
      case 1: // from BGP Dump (AS-Path)
//	 pcApp->pcNeighborText->clear();
	 generatePolicyFromBGP(iImport, iExport, 
			       pvAS, pvASPeer, pcApp->pcDefText, 
			       &pcApp->cBgpASPath, iASPeerType);
	 break;
      case 2: // copy from peer (valid only for AS from IRR)
	 generatePolicyFromPeer(iImport, iExport, 
				pvAS, pvASPeer,	
				pcApp->pcDefText, iASPeerType);
	 break;
      case 3: // Shouldn't happen here since it's just a separator!
	 break;
      default:
	 {
//	 pcApp->pcNeighborText->clear();
	 generateTemplatePolicy(iCategory, iImport, iExport, 
				pvAS, pvASPeer,	pcApp->pcDefText);
	 break;
	 }
      }
   return TCL_OK;
}


int LoadBgpDump::command(int argc, char *argv[])
{
   if (argc != 2) return TCL_ERROR;
   AoeApplication *pcApp = (AoeApplication *)getExtraArgument();
   // Call lexer 
   extern int loadBgpDump(BgpASPath &);  // Declared in bgp_dump.l
   extern FILE *bgpin;
   FILE *fpIn = fopen(argv[1], "r");
   if (!fpIn) return TCL_ERROR;
   bgpin = fpIn;
   loadBgpDump(pcApp->cBgpASPath);
   fclose(fpIn);
   // Duplicate first to last if last is empty
   pcApp->cBgpASPath.verify();
   //   pcApp->cBgpASPath.print();
   // Try to fill the result of BGP dump into AS Peer list
   if (pcApp->pcASPeerList->size()) pcApp->pcASPeerList->clear();
   (*pcApp->pcASPeerList << (pcApp->cASPeers << pcApp->cBgpASPath));
   // Assume the first peer is selected at the first place
   if (pcApp->pcASPeerList->size()) pcApp->pcASPeerList->select(0, 0);
   // Do the bootstrap action
   if (!pcApp->bootStrap()) return TCL_ERROR;
   return TCL_OK;
}

int UpdatePolicy::command(int argc, char *argv[])
{
   if (argc != 3) return TCL_ERROR;
   char *pzcOption = argv[1];
   int iTarget = atoi(argv[2]);
   int iResult = TCL_ERROR;
   if (strcmp(pzcOption, "append") == 0)
      iResult = append();
   else
      if (strcmp(pzcOption, "replace") == 0)
	 iResult = replace(iTarget);
   return iResult;
}

int UpdatePolicy::append(void)
{
   AoeApplication *pcApp = (AoeApplication *)getExtraArgument();
   pcApp->pcPolicyText->insert("\n");
   *pcApp->pcPolicyText << *pcApp->pcDefText;

   // Reparse it
   TCL(!pcApp->reparseWorkingAutNumArea());

   return TCL_OK;
}

int UpdatePolicy::replace(int iTarget)
{
   AoeApplication *pcApp = (AoeApplication *)getExtraArgument();
   ASPeer *pcASPeer;
   int i;
   for (i = 0, pcASPeer = pcApp->cASPeers.head(); 
	i < iTarget && pcASPeer; 
	i++, pcASPeer = pcApp->cASPeers.next(pcASPeer->cNode)) ;
   if (pcASPeer)
     {
     Pix pvASPeer = pcASPeer->getNoInPix();   
     pcApp->pcAutNum->remove_peer(pvASPeer);
     *pcApp->pcPolicyText << *pcApp->pcAutNum; // Redisplay
     }
   return append();
}

/*
int DeletePolicyText::command(int argc, char *argv[])
{
   if (argc != 1) return TCL_ERROR;
   AoeApplication *pcApp = (AoeApplication *)getExtraArgument();
   if (!pcApp->pcPolicyText->removeSelectedText()) return TCL_ERROR;
   return TCL_OK;
}
*/

int EditPolicyText::command(int argc, char *argv[])
{
   if (argc != 1) return TCL_ERROR;
   AoeApplication *pcApp = (AoeApplication *)getExtraArgument();
   if (pcApp->pcPolicyEditButton->getOption())
      {
      if (!pcApp->pcPolicyText->enable()) return TCL_ERROR;
      if (!pcApp->pcPolicyText->focus()) return TCL_ERROR;
      }
   else 
      {
      if (!pcApp->pcPolicyText->disable()) return TCL_ERROR;
      if (!pcApp->pcPolicyText->unfocus()) return TCL_ERROR;
      if (!pcApp->reparseWorkingAutNumArea()) return TCL_ERROR;
      }
   return TCL_OK;
}

int ShowPolicyText::command(int argc, char *argv[])
{
   if (argc != 1) return TCL_ERROR;
   AoeApplication *pcApp = (AoeApplication *)getExtraArgument();
   if (pcApp->pcPolicyShowButton->getOption())
      {
      if (pcApp->cASPeers.head())
	 {
	 int iActiveItem = pcApp->pcASPeerList->getCurrentItem();
	 if (iActiveItem >= 0)
	    {
	    Pix pvASPeer = pcApp->getASPeerNoInPix(iActiveItem);
	    pcApp->pcPolicyText->clear();
	    *pcApp->pcPolicyText << *(AS_map.define(pvASPeer));
	    // Disable improper buttons
	    //	    pcApp->pcPolicyDeleteButton->disable();
	    pcApp->pcPolicyEditButton->disable();
	    pcApp->pcCommitButton->disable();
	    }
	 else
	   {
	   pcApp->pcPolicyShowButton->toggle();
	   if (!pcApp->eval("showWarning {Please point the peer " 
			    "you are going to display!}")) 
	      return TCL_ERROR;
	   }
	 }
      else
	 {
	 pcApp->pcPolicyShowButton->toggle();
	 if (!pcApp->eval("showWarning {No peer to be displayed!}"))
	    return TCL_ERROR;
	 }
      }
   else
      {
      pcApp->pcPolicyText->clear();
      *pcApp->pcPolicyText << *pcApp->pcAutNum;
      // Enable some proper buttons
      //      pcApp->pcPolicyDeleteButton->enable();
      pcApp->pcPolicyEditButton->enable();
      pcApp->pcCommitButton->enable();
      }
   return TCL_OK;
}

int FileOpen::command(int argc, char *argv[])
{
   if (argc != 2) return TCL_ERROR;
   AoeApplication *pcApp = (AoeApplication *)getExtraArgument();
   FILE *fp;
   struct stat sFileStat;
   // Delete original one
   if (pcApp->pcAutNum == AS_map.contents(pcApp->getASNoInPix()).definition)
      delete pcApp->pcAutNum;
   // Get size of the file
   if (stat(argv[1], &sFileStat)) return TCL_ERROR;
   // Open the file for reading
   if ((fp = fopen(argv[1], "r")) == NULL) return TCL_ERROR;
   // Simulate WhoisParser::ParseObject()
   extern AutNum *irr_parser_as;
   extern int irr_lexer_input_size; 
   extern FILE *irrin;
   extern int irrparse();
   irr_parser_as = pcApp->pcAutNum = new AutNum;
   AS_map.contents(pcApp->getASNoInPix()).definition = (void *)irr_parser_as;
   irr_lexer_input_size = sFileStat.st_size; 
   irrin = fp;
   irrparse();

   AS_map.handle_extended_and_original_attr(pcApp->pcAutNum);
   
   fclose(fp);

   // Refill policy text widget, if the show button is inappropriate, toggle it
   if (pcApp->pcPolicyShowButton->getOption()) 
      pcApp->pcPolicyShowButton->toggle();
   if (!pcApp->eval("showPolicyText")) return TCL_ERROR;
   return TCL_OK;
}

int FileSave::command(int argc, char *argv[])
{
   if (argc != 2) return TCL_ERROR;
   AoeApplication *pcApp = (AoeApplication *)getExtraArgument();
   FILE *fp;
   if ((fp = fopen(argv[1], "w")) == NULL) return TCL_ERROR;
   fputs(pcApp->pcPolicyText->getAll(), fp);
   fclose(fp);
   return TCL_OK;
}

int FileRevert::command(int argc, char *argv[])
{
   if (argc != 1) return TCL_ERROR;
   AoeApplication *pcApp = (AoeApplication *)getExtraArgument();
   // Delete original one
   if (pcApp->pcAutNum == AS_map.contents(pcApp->getASNoInPix()).definition)
      {
      delete pcApp->pcAutNum;
      pcApp->pcAutNum = NULL;
      AS_map.contents(pcApp->getASNoInPix()).definition = NULL;
      }
   // Clear the peer list first
   pcApp->cASPeers.clear();       // Clear the list from memory
   pcApp->pcASPeerList->clear();  // Clear the list from Tk
   // Then ask the parser requery the AutNum object and redisplay the peer list
   pcApp->getASInfoFromServer();
   pcApp->bootStrap();
   return TCL_OK;
}

int FilePrint::command(int argc, char *argv[])
{   
   if (argc != 2) return TCL_ERROR;
   char *pzcCommand = argv[1];
   AoeApplication *pcApp = (AoeApplication *)getExtraArgument();
   const char *pzcText;
   int iLen;
   FILE *fp;
   if ((fp = popen(pzcCommand, "w")) == NULL)
      {
      TCL(!pcApp->evalf("showError {Open pipe error with command: %s}", 
			pzcCommand));
      return TCL_ERROR;
      } 
   pzcText = pcApp->pcPolicyText->getAll();
   iLen = strlen(pzcText);
   if (fwrite(pzcText, 1, iLen, fp) != iLen) 
      {
      TCL(!pcApp->evalf("showError {Printing error!}"));
      return TCL_ERROR; 
      }   
   pclose(fp);
   TCL(!pcApp->evalf("showInfo {Printing is done!}"));
   return TCL_OK;
}


int AddPeer::command(int argc, char *argv[])
{
   if (argc != 2) return TCL_ERROR;
   AoeApplication *pcApp = (AoeApplication *)getExtraArgument();
   // Allocate a new peer
   ASPeer *pcASPeer = new ASPeer(AS_map.add_entry(argv[1]), dASPeerNew);
   if (!pcASPeer) return TCL_ERROR;
   pcApp->cASPeers.append(pcASPeer->cNode);
   // Clear the aspeer list
   if (!pcApp->pcASPeerList->clear()) TCL_ERROR;
   // Refill the AS Peer list
   *pcApp->pcASPeerList << pcApp->cASPeers;   
   return TCL_OK;
}

int DeletePeer::command(int argc, char *argv[])
{
   if (argc != 2) return TCL_ERROR;
   AoeApplication *pcApp = (AoeApplication *)getExtraArgument();
   // Check if the list is empty of not,
   if (!argv[1][0]) 
      {
      if (pcApp->cASPeers.head())
	 {	
	 if (!pcApp->eval("showWarning {Please point the peer " 
			  "you are going to delete!}")) 
	 return TCL_ERROR;
	 }
      else
	 {
	 if (!pcApp->eval("showWarning {Nothing to be deleted!}")) 
	    return TCL_ERROR;
	 }
      return TCL_OK;
      }
   Pix pvASPeer = NULL;
   ASPeer *pcASPeer;
   int i, iTarget = atoi(argv[1]);
   for (i = 0, pcASPeer = pcApp->cASPeers.head(); 
	i < iTarget && pcASPeer; 
	i++, pcASPeer = pcApp->cASPeers.next(pcASPeer->cNode)) ;
   if (pcASPeer)
     {
     pvASPeer = pcASPeer->getNoInPix();
     pcASPeer->cNode.un_link();
     delete pcASPeer;
     // Clear the aspeer list
     if (!pcApp->pcASPeerList->clear()) TCL_ERROR;
     // Refill the AS Peer list
     *pcApp->pcASPeerList << pcApp->cASPeers;      
     // Clear the corresponding text widgets
     if (!pcApp->pcDefText->clear()) return TCL_ERROR;
     if (!pcApp->pcNeighborText->clear()) return TCL_ERROR;
     if (!pcApp->pcNeighborLabel->attach("Empty")) return TCL_ERROR;
     pcApp->pcPolicyShowButton->labelBothString("Empty");
     }
   // else should happen here

   if (pvASPeer)
      if (pcApp->pcAutNum->remove_peer(pvASPeer))
	 // Redisplay the contain of AutNum in policy text widget
	 *pcApp->pcPolicyText << *pcApp->pcAutNum;

   // Take care of show policy button
   if (pcApp->pcPolicyShowButton->getOption())
     {
     pcApp->pcPolicyShowButton->toggle();
     if (!pcApp->eval("showPolicyText")) return TCL_ERROR;
     }

   return TCL_OK;
}


int CreatePolicyTemplateMenu::command(int argc, char *argv[])
{
   if (argc != 1) return TCL_ERROR;
   AoeApplication *pcApp = (AoeApplication *)getExtraArgument();
   // Delete menu first
   for (int i = pcApp->pcCategoryButton->getSize(); i > 4; i--)
      pcApp->pcCategoryButton->del(i);
   // Create menu
   char *pzcTotal = pcApp->getVar("PolicyTemplate(Total)");
   if (pzcTotal)
      {
      for (int i = 0; i < atoi(pzcTotal); i++)
	 {
	 char pzcBuffer[256];
	 sprintf(pzcBuffer, "PolicyTemplate(%d,Description)", i);
	 if (!pcApp->pcCategoryButton->add(pcApp->getVar(pzcBuffer), 
	      "listPeer [.body.aspeer.listbox.list curselection]"))
	 return TCL_ERROR;
	 }
      }
   return TCL_OK;
}


//******************************
// Application class starts here

AoeApplication::AoeApplication(char *pzcAppName, 
			       char *pzcDisplay,
			       char *pzcRCFile,
			       char *pzcASNo) :
   TclApplication(pzcAppName, pzcDisplay, pzcRCFile),
   pzcASNo(pzcASNo),
   pcASPeerList(NULL),
   pcPolicyText(NULL),
   pcDefText(NULL),
   pcNeighborText(NULL),
   pcImportButton(NULL),
   pcExportButton(NULL),
   pcCategoryButton(NULL),
   pcDefLabel(NULL),
   pcNeighborLabel(NULL),
//   pcPolicyDeleteButton(NULL),
   pcCommitButton(NULL),
   pcPolicyEditButton(NULL),
   pcPolicyShowButton(NULL),
   pcStatusLine(NULL)
{
}

AoeApplication::~AoeApplication(void)
{
   // Save settings into resource file before program ends
   evalf("aoe:saveResource %s", getResource());
   // Free allocate memory
   if (!cASPeers.is_empty()) cASPeers.clear();
   if (pcASPeerList) delete pcASPeerList;
   if (pcPolicyText) delete pcPolicyText;
   if (pcDefText) delete pcDefText;
   if (pcNeighborText) delete pcNeighborText;
   if (pcImportButton) delete pcImportButton;
   if (pcExportButton) delete pcExportButton;
   if (pcCategoryButton) delete pcCategoryButton;
   if (pcDefLabel) delete pcDefLabel;
   if (pcNeighborLabel) delete pcNeighborLabel;
//   if (pcPolicyDeleteButton) delete pcPolicyDeleteButton;
   if (pcCommitButton) delete pcCommitButton;
   if (pcPolicyEditButton) delete pcPolicyEditButton;
   if (pcPolicyShowButton) delete pcPolicyShowButton;
   if (pcStatusLine) delete pcStatusLine;
}

int AoeApplication::getASInfoFromServer(void)
{
   whois.QueryKillResponse("!uF=0");

   // Create an AS entry in the AS map
   pvASNo = AS_map.add_entry(pzcASNo);
   pcAutNum = AS_map.define(pvASNo);

   if (!pcAutNum->peers.is_empty())
      {
      // Get all peers' pix
      ASPolicy *p;
      int i;
      for (p = pcAutNum->peers.head(), i = 0;
	   p; 
	   p = pcAutNum->peers.next(p->peers), i++)
	 {
	 ASPeer *pcASPeer = new ASPeer(p->peeras, dASPeerFromIRR);
	 cASPeers.append(pcASPeer->cNode);
	 }
      }

   // Fill out the AS Peer list
   *pcASPeerList << cASPeers;

   // Assume the first peer is selected at the first place
   if (pcASPeerList->size()) pcASPeerList->select(0, 0);

   // Fill out the policy text widget for users reference
   *pcPolicyText << *pcAutNum;

   // Insert peers into the peer list
   //   for (Pix pix = cPeers.first(); pix; cPeers.next(pix))
   //      evalf("%s insert end \"%s\"", tkASPeerList, AS_map(cPeers(pix)));

   return 1;
}

Pix AoeApplication::getASPeerNoInPix(int i)
{
   ASPeer *pcASPeer;
   int iIndex;
   for (pcASPeer = cASPeers.head(), iIndex = 0;
	pcASPeer;
	pcASPeer = cASPeers.next(pcASPeer->cNode), iIndex++)
      if (iIndex == i) break;
   if (pcASPeer) return pcASPeer->getNoInPix();
   // Shouldn't come to here
   return NULL;
}

int AoeApplication::getASPeerType(int i)
{
   ASPeer *pcASPeer;
   int iIndex;
   for (pcASPeer = cASPeers.head(), iIndex = 0;
	pcASPeer;
	pcASPeer = cASPeers.next(pcASPeer->cNode), iIndex++)
      if (iIndex == i) break;
   if (pcASPeer) return pcASPeer->getType();
   // Shouldn't come to here
   return dASPeerUnknown;   
}

int AoeApplication::bootStrap(void)
{
   // Create a menu for policy template
   if (!evalf("createPolicyTemplateMenu")) return False;
   // Setup $MyAS properly
   if (!evalf("set MyAS %s", getASNoInString())) return False;
   // Select the fist one
   if (!evalf("listPeer [%s curselection]", tkASPeerList)) return False;
   return True;
}

int AoeApplication::reparseWorkingAutNumArea(void)
{
   FILE *fp;
   struct stat sFileStat;
   char *pzcTmpFile = getVar("tmpFile");
   // Write policy to a file
   if ((fp = fopen(pzcTmpFile, "w")) == NULL)
      return 0;
   fputs(pcPolicyText->getAll(), fp);
   fclose(fp);
   // Get size of the file
   if (stat(pzcTmpFile, &sFileStat)) 
      return 0;
   // Parse policy from the file
   if ((fp = fopen(pzcTmpFile, "r")) == NULL)
      return 0;
   
   // For safety check!
   if (pcAutNum == AS_map.contents(getASNoInPix()).definition)
      delete pcAutNum;

   // Simulate WhoisParser::ParseObject()
   extern AutNum *irr_parser_as;
   extern int irr_lexer_input_size; 
   extern FILE *irrin;
   extern int irrparse();
   irr_parser_as = pcAutNum = new AutNum;
   AS_map.contents(getASNoInPix()).definition = (void *)irr_parser_as;
   irr_lexer_input_size = sFileStat.st_size; 
   irrin = fp;
   irrparse();

   AS_map.handle_extended_and_original_attr(pcAutNum);
   fclose(fp);
   unlink(pzcTmpFile);

   // Refill policy text widget
   *pcPolicyText << *pcAutNum;

   return 1;
}

int AoeApplication::init(void)
{
   // Need to call init() of base class first, since it does all the
   // Tcl_Init() & Tk_Init() stuff
   if (!TclApplication::init()) return 0;

   // eval() is only working after Tcl_Init()

   // Check if the resource file (~/.aoerc) exist or not
   if (!evalf("file exists %s", getResource())) return 0;
   if (atoi(getResult()) == 0)
      {
      // If not, use the default resource
      extern char *aoerc_tk;
      if (!eval(aoerc_tk)) return 0;
      }

   // Load the file select module from fileselect.tk
   extern char *fileselect_tk;
   if (!eval(fileselect_tk)) return 0;

   // Load the utils module from utils.tk
   extern char *utils_tk;
   if (!eval(utils_tk)) return 0;

   // Load the layout converted from layout.tk
   // Should be the last one to be loaded
   extern char *layout_tk;
   if (!eval(layout_tk)) return 0;

   // Initiate bitmap
   Tk_DefineBitmap(getInterp(), Tk_GetUid("aoe_logo"), 
		   aoe_bits, aoe_width, aoe_height);
   if (!evalf("%s configure -bitmap aoe_logo", tkLogoButton)) return 0;

   // Display something first
   if (!eval("update")) return 0;

   // Create Tcl commands here
   if (!createCommand(new ListPeer("listPeer"))) return 0;
   if (!createCommand(new LoadBgpDump("loadBgpDump"))) return 0;
   if (!createCommand(new UpdatePolicy("updatePolicy"))) return 0;
//   if (!createCommand(new DeletePolicyText("deletePolicyText"))) return 0;
   if (!createCommand(new EditPolicyText("editPolicyText"))) return 0;
   if (!createCommand(new ShowPolicyText("showPolicyText"))) return 0;
   if (!createCommand(new FileOpen("fileOpen"))) return 0;
   if (!createCommand(new FileSave("fileSave"))) return 0;
   if (!createCommand(new FileRevert("fileRevert"))) return 0;
   if (!createCommand(new FilePrint("filePrint"))) return 0;
   if (!createCommand(new AddPeer("addPeer"))) return 0;
   if (!createCommand(new DeletePeer("deletePeer"))) return 0;
   if (!createCommand(
	   new CreatePolicyTemplateMenu("createPolicyTemplateMenu")))
      return 0;

   // Create Widgets (operations) here
   pcASPeerList = new TclList(tkASPeerList);
   if (!insert(pcASPeerList)) return 0;

   pcPolicyText = new TclText(tkPolicyText);
   if (!insert(pcPolicyText)) return 0;

   pcDefText = new TclText(tkDefText);
   if (!insert(pcDefText)) return 0;

   pcNeighborText = new TclText(tkNeighborText);
   if (!insert(pcNeighborText)) return 0;

   pcImportButton = new TclCheckButton(tkImportButton, cbCheck);
   if (!insert(pcImportButton)) return 0;

   pcExportButton = new TclCheckButton(tkExportButton, cbCheck);
   if (!insert(pcExportButton)) return 0;

   pcCategoryButton = new TclRadioButton(tkCategoryButton, tkCategoryVar, 0);
   if (!insert(pcCategoryButton)) return 0;

   pcDefLabel = new TclLabel(tkDefLabel);
   if (!insert(pcCategoryButton)) return 0;
   if (!pcDefLabel->setColor("red")) return 0;

   pcNeighborLabel = new TclLabel(tkNeighborLabel);
   if (!insert(pcNeighborLabel)) return 0;
/*
   pcPolicyDeleteButton = new TclButton(tkPolicyDeleteButton);
   if (!insert(pcPolicyDeleteButton)) return 0;
*/
   pcCommitButton = new TclButton(tkCommitButton);
   if (!insert(pcCommitButton)) return 0;

   pcPolicyEditButton = new TclToggleButton(tkPolicyEditButton,
					    "Edit", "Edit done!",
					    cbUnCheck);
   if (!insert(pcPolicyEditButton)) return 0;

   pcPolicyShowButton = new TclToggleButton(tkPolicyShowButton,
//					    getASNoInString(), "Peer AS", 
					    "Peer AS", "Peer AS", 
					    cbUnCheck);
   if (!insert(pcPolicyShowButton)) return 0;

   pcStatusLine = new TclLabel(tkStatusLine);
   if (!insert(pcStatusLine)) return 0;

   getASInfoFromServer();

   // Bootstrap something here
   if (!pcDefLabel->attach(pzcASNo)) return 0;
   if (!pcStatusLine->detach()) return 0;
   if (!bootStrap()) return 0;

   return 1;
}

// The reason I overide run() is to let main() looks better  :-)
int AoeApplication::run(void)
{
   if (!TclApplication::run())
      {
      fprintf(stderr, "%s", getResult());
      exit(1);
      }    
}

// Main program
int main(int argc, char *argv[], char *envp[])
{
   init_and_set_options(argc, argv, envp);
   if (!opt_rcfile) opt_rcfile = "~/.aoerc";
   AoeApplication cAoeApplication("aoe", display, opt_rcfile, 
				  strupr(opt_my_as));
   cAoeApplication.run();
   if (opt_rusage) clog << ru;
   return 0;
}

