/*
 * Copyright (C) 2010  Internet Systems Consortium, Inc. ("ISC")
 *
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND ISC DISCLAIMS ALL WARRANTIES WITH
 * REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS.  IN NO EVENT SHALL ISC BE LIABLE FOR ANY SPECIAL, DIRECT,
 * INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

/* $Id: simple.c 1207 2011-06-29 17:13:22Z pselkirk $ */

/*
 * PCP (port-control-protocol) client library simple test client
 *
 * Francis_Dupont@isc.org, July 2010
 *
 * ./simple <server> <priv> <[!]pub> <proto> <lifetime>
 */

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "pcp.h"

int
main(int argc, char *argv[])
{
	pcp_t pcp;
	pcp_request_t request;
	pcp_option_t **options;
	pcp_response_t response;
	int ret, fd, tries;
	fd_set set;
	struct sockaddr_in server;
	struct timeval tv;
	struct in_addr in;

#define USAGE	\
	"usage: %s <server> <priv> <[!]pub> <proto> <lifetime>\n"
	if (argc != 6) {
		fprintf(stderr, USAGE, argv[0]);
		exit(1);
	}
	memset(&server, 0, sizeof(server));
	server.sin_family = AF_INET;
	inet_aton(argv[1], &server.sin_addr);
	ret = pcp_init((struct sockaddr *) &server, NULL, &pcp);
	if (ret < PCP_OK) {
		fprintf(stderr, "pcp_init: %s\n", pcp_strerror(ret));
		exit(1);
	}
	ret = pcp_getsocket(&pcp, &fd);
	if (ret < PCP_OK) {
		fprintf(stderr, "pcp_getsocket: %s\n", pcp_strerror(ret));
		exit(1);
	}
	memset(&request, 0, sizeof(request));
	request.opcode = PCP_OPCODE_MAP4;
	request.lifetime = atoi(argv[5]);
	if (strcmp(argv[4], "all") == 0)
		request.protocol = 0;
	else if (strcmp(argv[4], "udp") == 0)
		request.protocol = IPPROTO_UDP;
	else if (strcmp(argv[4], "tcp") == 0)
		request.protocol = IPPROTO_TCP;
	else {
		fprintf(stderr, "unknown protocol '%s'\n", argv[4]);
		exit(1);
	}
	request.intport = atoi(argv[2]);
	options = NULL;
	if (*argv[3] == '!') {
		ret = pcp_honor_external_port(&options);
		if (ret != PCP_OK) {
			fprintf(stderr,
				"pcp_honor_external_port: %s\n",
				pcp_strerror(ret));
			exit(1);
		}
		request.extport = atoi(argv[3] + 1);
	} else
		request.extport = atoi(argv[3]);
	ret = pcp_makerequest(&pcp, &request, options);
	if (ret < PCP_OK) {
		fprintf(stderr, "pcp_makerequest: %s\n", pcp_strerror(ret));
		exit(1);
	}
	ret = pcp_sendrequest(&pcp);
	if (ret < PCP_OK) {
		fprintf(stderr, "pcp_sendrequest: %s\n", pcp_strerror(ret));
		exit(1);
	}
	for (;;) {
		ret = pcp_gettries(&pcp, &tries);
		if (ret < PCP_OK) {
			fprintf(stderr,
				"pcp_gettries: %s\n",
				pcp_strerror(ret));
			exit(1);
		}
		if (tries > PCP_MAX_TRIES) {
			fprintf(stderr, "too many retries: giving up\n");
			exit(1);
		}
		FD_ZERO(&set);
		FD_SET(fd, &set);
		ret = pcp_gettimeout(&pcp, &tv, 1);
		if (ret < PCP_OK) {
			fprintf(stderr,
				"pcp_gettimeout: %s\n",
				pcp_strerror(ret));
			exit(1);
		}
		ret = select(fd + 1, &set, NULL, NULL, &tv);
		if (ret < 0) {
			fprintf(stderr, "select: %s\n", strerror(errno));
			continue;
		}
		if (FD_ISSET(fd, &set)) {
			memset(&response, 0, sizeof(response));
			ret = pcp_recvresponse(&pcp, &response);
			if (ret < PCP_OK) {
				fprintf(stderr,
					"pcp_recvresponse: %s\n",
					pcp_strerror(ret));
				if (ret == PCP_ERR_RECVBAD) {
					fprintf(stderr,
						"at line %d\n",
						pcp_debug_line);
					abort();
				}
				exit(1);
			}
			if (ret == PCP_TRY_AGAIN)
				fprintf(stderr, "try again\n");
			else
				break;
		}
		ret = pcp_sendrequest(&pcp);
		if (ret < PCP_OK) {
			fprintf(stderr,
				"pcp_sendrequest: %s\n",
				pcp_strerror(ret));
			exit(1);
		}
	}
	printf("epoch: %d\n", (int) response.epoch);
	printf("internal address: %s\n", inet_ntoa(in));
	in.s_addr = response.assigned.extaddr4;
	printf("external address: %s\n", inet_ntoa(in));
	printf("lifetime: %d\n", (int) response.assigned.lifetime);
	printf("internal port: %d\n", (int) response.assigned.intport);
	printf("external port: %d\n", (int) response.assigned.extport);
	ret = pcp_close(&pcp);
	if (ret < PCP_OK) {
		fprintf(stderr, "pcp_close: %s\n", pcp_strerror(ret));
		exit(1);
	}
	exit(0);
}
