/* $Id: upnpsoap.c,v 1.66 2011/01/01 20:17:44 nanard Exp $ */
/* MiniUPnP project
 * http://miniupnp.free.fr/ or http://miniupnp.tuxfamily.org/
 * (c) 2006-2010 Thomas Bernard 
 * This software is subject to the conditions detailed
 * in the LICENCE file provided within the distribution */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <unistd.h>
#include <syslog.h>
#include <sys/types.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <netdb.h>

#include "config.h"
#include "upnpglobalvars.h"
#include "upnphttp.h"
#include "upnpsoap.h"
#include "upnpreplyparse.h"
#include "upnpredirect.h"
#include "getifstats.h"

static void
BuildSendAndCloseSoapResp(struct upnphttp * h,
                          const char * body, int bodylen)
{
	static const char beforebody[] =
		"<?xml version=\"1.0\"?>\r\n"
		"<s:Envelope xmlns:s=\"http://schemas.xmlsoap.org/soap/envelope/\" "
		"s:encodingStyle=\"http://schemas.xmlsoap.org/soap/encoding/\">"
		"<s:Body>";

	static const char afterbody[] =
		"</s:Body>"
		"</s:Envelope>\r\n";

	BuildHeader_upnphttp(h, 200, "OK",  sizeof(beforebody) - 1
		+ sizeof(afterbody) - 1 + bodylen );

	memcpy(h->res_buf + h->res_buflen, beforebody, sizeof(beforebody) - 1);
	h->res_buflen += sizeof(beforebody) - 1;

	memcpy(h->res_buf + h->res_buflen, body, bodylen);
	h->res_buflen += bodylen;

	memcpy(h->res_buf + h->res_buflen, afterbody, sizeof(afterbody) - 1);
	h->res_buflen += sizeof(afterbody) - 1;

	SendResp_upnphttp(h);
	CloseSocket_upnphttp(h);
}

static void
GetConnectionTypeInfo(struct upnphttp * h, const char * action, int v2)
{
	static const char resp[] =
		"<u:GetConnectionTypeInfoResponse "
		"xmlns:u=\"urn:schemas-upnp-org:service:WANIPConnection:%d\">"
		"<NewConnectionType>IP_Routed</NewConnectionType>"
		"<NewPossibleConnectionTypes>IP_Routed</NewPossibleConnectionTypes>"
		"</u:GetConnectionTypeInfoResponse>";
	char body[512];
	int bodylen;

	action = action;
	bodylen = snprintf(body, sizeof(body), resp, v2 ? 2 : 1);
	BuildSendAndCloseSoapResp(h, body, bodylen);
}

static void
GetTotalBytesSent(struct upnphttp * h, const char * action, int v2)
{
	int r;

	static const char resp[] =
		"<u:%sResponse "
		"xmlns:u=\"%s\">"
		"<NewTotalBytesSent>%lu</NewTotalBytesSent>"
		"</u:%sResponse>";

	char body[512];
	int bodylen;
	struct ifdata data;

	v2 = v2;
	r = getifstats(ext_if_name, &data);
	bodylen = snprintf(body, sizeof(body), resp,
	         action, "urn:schemas-upnp-org:service:WANCommonInterfaceConfig:1",
             r<0?0:data.obytes, action);
	BuildSendAndCloseSoapResp(h, body, bodylen);
}

static void
GetTotalBytesReceived(struct upnphttp * h, const char * action, int v2)
{
	int r;

	static const char resp[] =
		"<u:%sResponse "
		"xmlns:u=\"%s\">"
		"<NewTotalBytesReceived>%lu</NewTotalBytesReceived>"
		"</u:%sResponse>";

	char body[512];
	int bodylen;
	struct ifdata data;

	v2 = v2;
	r = getifstats(ext_if_name, &data);
	bodylen = snprintf(body, sizeof(body), resp,
	         action, "urn:schemas-upnp-org:service:WANCommonInterfaceConfig:1",
	         r<0?0:data.ibytes, action);
	BuildSendAndCloseSoapResp(h, body, bodylen);
}

static void
GetTotalPacketsSent(struct upnphttp * h, const char * action, int v2)
{
	int r;

	static const char resp[] =
		"<u:%sResponse "
		"xmlns:u=\"%s\">"
		"<NewTotalPacketsSent>%lu</NewTotalPacketsSent>"
		"</u:%sResponse>";

	char body[512];
	int bodylen;
	struct ifdata data;

	v2 = v2;
	r = getifstats(ext_if_name, &data);
	bodylen = snprintf(body, sizeof(body), resp,
	         action, "urn:schemas-upnp-org:service:WANCommonInterfaceConfig:1",
	         r<0?0:data.opackets, action);
	BuildSendAndCloseSoapResp(h, body, bodylen);
}

static void
GetTotalPacketsReceived(struct upnphttp * h, const char * action, int v2)
{
	int r;

	static const char resp[] =
		"<u:%sResponse "
		"xmlns:u=\"%s\">"
		"<NewTotalPacketsReceived>%lu</NewTotalPacketsReceived>"
		"</u:%sResponse>";

	char body[512];
	int bodylen;
	struct ifdata data;

	v2 = v2;
	r = getifstats(ext_if_name, &data);
	bodylen = snprintf(body, sizeof(body), resp,
	         action, "urn:schemas-upnp-org:service:WANCommonInterfaceConfig:1",
	         r<0?0:data.ipackets, action);
	BuildSendAndCloseSoapResp(h, body, bodylen);
}

static void
GetCommonLinkProperties(struct upnphttp * h, const char * action, int v2)
{
	/* WANAccessType : set depending on the hardware :
	 * DSL, POTS (plain old Telephone service), Cable, Ethernet */
	static const char resp[] =
		"<u:%sResponse "
		"xmlns:u=\"%s\">"
		/*"<NewWANAccessType>DSL</NewWANAccessType>"*/
		"<NewWANAccessType>Cable</NewWANAccessType>"
		"<NewLayer1UpstreamMaxBitRate>%lu</NewLayer1UpstreamMaxBitRate>"
		"<NewLayer1DownstreamMaxBitRate>%lu</NewLayer1DownstreamMaxBitRate>"
		"<NewPhysicalLinkStatus>%s</NewPhysicalLinkStatus>"
		"</u:%sResponse>";

	char body[2048];
	int bodylen;
	struct ifdata data;
	const char * status = "Up";	/* Up, Down (Required),
	                             * Initializing, Unavailable (Optional) */

	v2 = v2;
	if((downstream_bitrate == 0) || (upstream_bitrate == 0))
	{
		if(getifstats(ext_if_name, &data) >= 0)
		{
			if(downstream_bitrate == 0) downstream_bitrate = data.baudrate;
			if(upstream_bitrate == 0) upstream_bitrate = data.baudrate;
		}
	}
	if (external_addr == 0)
		status = "Down";
	bodylen = snprintf(body, sizeof(body), resp,
	    action, "urn:schemas-upnp-org:service:WANCommonInterfaceConfig:1",
		upstream_bitrate, downstream_bitrate,
	    status, action);
	BuildSendAndCloseSoapResp(h, body, bodylen);
}

static void
GetStatusInfo(struct upnphttp * h, const char * action, int v2)
{
	static const char resp[] =
		"<u:%sResponse "
		"xmlns:u=\"%s%d\">"
		"<NewConnectionStatus>%s</NewConnectionStatus>"
		"<NewLastConnectionError>ERROR_NONE</NewLastConnectionError>"
		"<NewUptime>%ld</NewUptime>"
		"</u:%sResponse>";

	char body[512];
	int bodylen;
	time_t uptime;
	const char * status = "Connected";
	/* ConnectionStatus possible values :
	 * Unconfigured, Connecting, Connected, PendingDisconnect,
	 * Disconnecting, Disconnected */

	if (external_addr == 0)
		status = "Disconnected";
	uptime = (time(NULL) - startup_time);
	bodylen = snprintf(body, sizeof(body), resp,
		"urn:schemas-upnp-org:service:WANIPConnection:",
		action, v2 ? 2 : 1,
		status, (long)uptime, action);	
	BuildSendAndCloseSoapResp(h, body, bodylen);
}

static void
GetNATRSIPStatus(struct upnphttp * h, const char * action, int v2)
{
	static const char resp[] =
		"<u:GetNATRSIPStatusResponse "
		"xmlns:u=\"urn:schemas-upnp-org:service:WANIPConnection:%d\">"
		"<NewRSIPAvailable>0</NewRSIPAvailable>"
		"<NewNATEnabled>1</NewNATEnabled>"
		"</u:GetNATRSIPStatusResponse>";
	char body[512];
	int bodylen;
	/* 2.2.9. RSIPAvailable
	 * This variable indicates if Realm-specific IP (RSIP) is available
	 * as a feature on the InternetGatewayDevice. RSIP is being defined
	 * in the NAT working group in the IETF to allow host-NATing using
	 * a standard set of message exchanges. It also allows end-to-end
	 * applications that otherwise break if NAT is introduced
	 * (e.g. IPsec-based VPNs).
	 * A gateway that does not support RSIP should set this variable to 0. */
	action = action;
	bodylen = snprintf(body, sizeof(body), resp, v2 ? 2 : 1);
	BuildSendAndCloseSoapResp(h, body, bodylen);
}

static void
GetExternalIPAddress(struct upnphttp *h, const char *action, int v2)
{
	static const char resp[] =
		"<u:GetExternalIPAddressResponse "
		"xmlns:u=\"urn:schemas-upnp-org:service:WANIPConnection:%d\">"
		"<NewExternalIPAddress>%s</NewExternalIPAddress>"
		"</u:GetExternalIPAddressResponse>";
	char body[512];
	int bodylen;
	struct in_addr in;

	action = action;
	if (external_addr != 0) {
		in.s_addr = external_addr;
		bodylen = snprintf(body, sizeof(body), resp,
				   v2 ? 2 : 1, inet_ntoa(in));
	} else
		bodylen = snprintf(body, sizeof(body), resp,
				   v2 ? 2 : 1, "");
	BuildSendAndCloseSoapResp(h, body, bodylen);
}

void
AddPortMappingHandler(struct upnphttp *h, int v2)
{
	static const char resp[] =
		"<u:AddPortMappingResponse "
		"xmlns:u=\"urn:schemas-upnp-org:service:WANIPConnection:%d\"/>";
	char body[512];
	int bodylen;

	bodylen = snprintf(body, sizeof(body), resp, v2 ? 2 : 1);
	BuildSendAndCloseSoapResp(h, body, bodylen);
}

void
AddAnyPortMappingHandler(struct upnphttp *h, unsigned short eport)
{
	static const char resp[] =
		"<u:AddAnyPortMappingResponse "
		"xmlns:u=\"urn:schemas-upnp-org:service:WANIPConnection:2\">"
		"<NewReservedPort>%hu</NewReservedPort>"
		"</u:AddAnyPortMappingResponse>";
	char body[512];
	int bodylen;

	bodylen = snprintf(body, sizeof(body), resp, eport);
	BuildSendAndCloseSoapResp(h, body, bodylen);
}

/* possible error codes for AddPortMapping :
 * 402 - Invalid Args
 * 501 - Action Failed
 * 606 - Action not authorized (v2)
 * 715 - Wildcard not permited in InternalClient
 * 716 - Wildcard not permited in ExternalPort
 * 718 - ConflictInMappingEntry
 * 724 - SamePortValuesRequired
 * 725 - OnlyPermanentLeasesSupported
 *	 The NAT implementation only supports permanent lease
 *	 times on port mappings
 * 726 - RemoteHostOnlySupportsWildcard
 *	 RemoteHost must be a wildcard and cannot be a
 *	 specific IP address or DNS name
 * 727 - ExternalPortOnlySupportsWildcard
 *	 ExternalPort must be a wildcard and cannot be a
 *	 specific port value 
 * 728 - NoPortMapsAvailable
 *       There are not enough free prots available to complete
 *       the mapping (added in IGD v2)
 * 729 - ConflictWithOtherMechanisms
 *       Attempted port mapping is not allowed due to
 *       conflict with other mechanisms (added in IGD v2)
 * 732 - WildCardNotPermittedInIntPort
 *       InternalPort cannot be wild-carded (added in IGD v2) */

static void
AddPortMapping(struct upnphttp *h, const char *action, int v2)
{
	struct NameValueParserData data;
	char *int_ip, *int_port, *ext_port, *protocol, *desc;
	char *r_host, *leaseduration;
	unsigned short iport, eport;
	unsigned int lifetime;
	struct hostent *hp;
	char **ptr;
	struct in_addr result_ip;

	if (external_addr == 0) {
		SoapError(h, 501, "ActionFailed");
		return;
	}

	ParseNameValue(h->req_buf + h->req_contentoff,
		       h->req_contentlen, &data);
	r_host = GetValueFromNameValueList(&data, "NewRemoteHost");
	if (r_host != NULL) {
		ClearNameValueList(&data);
		SoapError(h, 726, "RemoteHostOnlySupportsWildcard");
		return;
	}

	int_ip = GetValueFromNameValueList(&data, "NewInternalClient");
	if (int_ip == NULL) {
		ClearNameValueList(&data);
		SoapError(h, 715, "WildCardNotPermittedInSrcIP");
		return;
	}

	/* if ip not valid assume hostname and convert */
	if (inet_pton(AF_INET, int_ip, &result_ip) <= 0) {
		hp = gethostbyname(int_ip);
		if (hp && (hp->h_addrtype == AF_INET))
			for (ptr = hp->h_addr_list;
			     (ptr != NULL) && (*ptr != NULL);
			     ptr++) {
				int_ip = inet_ntoa(*((struct in_addr *) *ptr));
				result_ip = *((struct in_addr *) *ptr);
				/* TODO : deal with more than
				   one ip per hostname */
				break;
			}
		else {
			syslog(LOG_ERR,
			       "Failed to convert hostname '%s' to ip address",
			       int_ip); 
			ClearNameValueList(&data);
			SoapError(h, 402, "Invalid Args");
			return;
		}				
	}

	/* check if NewInternalAddress is the client address */
	if (GETFLAG(SECUREMODEMASK))
		if (h->clientaddr.s_addr != result_ip.s_addr) {
			syslog(LOG_INFO,
			       "Client %s tried to redirect port to %s",
			       inet_ntoa(h->clientaddr), int_ip);
			ClearNameValueList(&data);
			if (v2)
				SoapError(h, 606, "Action not authorized");
			else
				SoapError(h, 718, "ConflictInMappingEntry");
			return;
		}

	int_port = GetValueFromNameValueList(&data, "NewInternalPort");
	ext_port = GetValueFromNameValueList(&data, "NewExternalPort");
	protocol = GetValueFromNameValueList(&data, "NewProtocol");
	desc = GetValueFromNameValueList(&data, "NewPortMappingDescription");
	leaseduration = GetValueFromNameValueList(&data, "NewLeaseDuration");

	if ((int_port == NULL) || (ext_port == NULL) || (protocol == NULL)) {
		ClearNameValueList(&data);
		SoapError(h, 402, "Invalid Args");
		return;
	}
	if ((strcmp(protocol, "TCP") != 0) &&
	    (strcmp(protocol, "UDP") != 0)) {
	      ClearNameValueList(&data);
	      if (v2)
		      SoapError(h, 601, "Argument Value Out of Range");
	      else
		      SoapError(h, 402, "Invalid Args");
	      return;
	}

	eport = (unsigned short) atoi(ext_port);
	iport = (unsigned short) atoi(int_port);

	if (eport == 0) {
		ClearNameValueList(&data);
		SoapError(h, 716, "WildCardNotPermittedInExtPort");
		return;
	}

	if (iport == 0) {
		ClearNameValueList(&data);
		if (v2)
			SoapError(h, 732, "WildCardNotPermittedInIntPort");
		else
			SoapError(h, 402, "Invalid Args");
		return;
	}

	if (desc == NULL)
		desc = "miniupnpd";

	if (leaseduration != NULL)
		lifetime = (unsigned int) atoi(leaseduration);
	else
		lifetime = 0U;

	syslog(LOG_INFO,
	       "%s: ext port %hu to %s:%hu protocol %s life %u for: %s",
	       action, eport, int_ip, iport, protocol, lifetime, desc);

	upnp_dynamic_redirect(h, &eport, int_ip, iport,
			      protocol, lifetime, desc, v2);

	ClearNameValueList(&data);
}

/* AddAnyPortMapping was added in WANIPConnection v2 */
static void
AddAnyPortMapping(struct upnphttp *h, const char *action, int v2)
{
	struct NameValueParserData data;
	const char *int_ip, *int_port, *ext_port, *protocol, *desc;
	const char *r_host, *leaseduration;
	unsigned short iport, eport;
	unsigned int lifetime;
	struct hostent *hp;
	char **ptr;
	struct in_addr result_ip;

	action = action;
	v2 = v2;
	if (external_addr == 0) {
		SoapError(h, 501, "ActionFailed");
		return;
	}

	ParseNameValue(h->req_buf + h->req_contentoff,
		       h->req_contentlen, &data);
	r_host = GetValueFromNameValueList(&data, "NewRemoteHost");
	r_host = r_host;
	ext_port = GetValueFromNameValueList(&data, "NewExternalPort");
	protocol = GetValueFromNameValueList(&data, "NewProtocol");
	int_port = GetValueFromNameValueList(&data, "NewInternalPort");
	int_ip = GetValueFromNameValueList(&data, "NewInternalClient");
	/* NewEnabled */
	desc = GetValueFromNameValueList(&data, "NewPortMappingDescription");
	leaseduration = GetValueFromNameValueList(&data, "NewLeaseDuration");

	if ((int_port == NULL) || (ext_port == NULL) ||
	    (protocol == NULL) || (leaseduration == NULL)) {
		ClearNameValueList(&data);
		SoapError(h, 402, "Invalid Args");
		return;
	}
	if ((strcmp(protocol, "TCP") != 0) &&
	    (strcmp(protocol, "UDP") != 0)) {
	      ClearNameValueList(&data);
	      if (v2)
		      SoapError(h, 601, "Argument Value Out of Range");
	      else
		      SoapError(h, 402, "Invalid Args");
	      return;
	}

	lifetime = (unsigned int) atoi(leaseduration);
	if (lifetime == 0)
		lifetime = 604800U;

	eport = (unsigned short) atoi(ext_port);
	iport = (unsigned short) atoi(int_port);

	if (iport == 0) {
		ClearNameValueList(&data);
		SoapError(h, 732, "WildCardNotPermittedInIntPort");
		return;
	}

	if (int_ip == NULL) {
		ClearNameValueList(&data);
		SoapError(h, 715, "WildCardNotPermittedInSrcIP");
		return;
	}

	/* if ip not valid assume hostname and convert */
	if (inet_pton(AF_INET, int_ip, &result_ip) <= 0) {
		hp = gethostbyname(int_ip);
		if(hp && hp->h_addrtype == AF_INET) { 
			for(ptr = hp->h_addr_list;
			    (ptr != NULL) && (*ptr != NULL);
			    ptr++) {
				int_ip = inet_ntoa(*((struct in_addr *) *ptr));
				result_ip = *((struct in_addr *) *ptr);
				/* TODO : deal with more than
				 * one ip per hostname */
				break;
			}
		} 
		else {
			syslog(LOG_ERR,
			       "Failed to convert hostname '%s' to ip address",
			       int_ip); 
			ClearNameValueList(&data);
			SoapError(h, 402, "Invalid Args");
			return;
		}				
	}

	/* check if NewInternalAddress is the client address */
	if(GETFLAG(SECUREMODEMASK))
		if(h->clientaddr.s_addr != result_ip.s_addr) {
			syslog(LOG_INFO,
			       "Client %s tried to redirect port to %s",
			       inet_ntoa(h->clientaddr), int_ip);
			ClearNameValueList(&data);
			SoapError(h, 606, "Action not authorized");
			return;
		}

	upnp_dynamic_redirect(h, &eport, int_ip, iport,
			      protocol, lifetime, desc, -1);

	ClearNameValueList(&data);
}

static unsigned int
get_lifetime(struct upnp_redirect *r)
{
	int ret;

	if (r->lifetime == 0U)
		return 0U;
	ret = r->expire.when.tv_sec - now.tv_sec;
	if (ret <= 0)
		return 1U;
	return (unsigned int) ret;
}

static void
GetSpecificPortMappingEntry(struct upnphttp *h, const char *action, int v2)
{
	struct upnp_redirect *r;
	static const char resp[] =
		"<u:%sResponse "
		"xmlns:u=\"%s%d\">"
		"<NewInternalPort>%u</NewInternalPort>"
		"<NewInternalClient>%s</NewInternalClient>"
		"<NewEnabled>1</NewEnabled>"
		"<NewPortMappingDescription>%s</NewPortMappingDescription>"
		"<NewLeaseDuration>%u</NewLeaseDuration>"
		"</u:%sResponse>";
	char body[1024];
	int bodylen;
	struct NameValueParserData data;
	const char *r_host, *ext_port, *protocol;
	unsigned short eport;

	if (external_addr == 0) {
		SoapError(h, 501, "ActionFailed");
		return;
	}

	ParseNameValue(h->req_buf + h->req_contentoff,
		       h->req_contentlen, &data);
	r_host = GetValueFromNameValueList(&data, "NewRemoteHost");
	ext_port = GetValueFromNameValueList(&data, "NewExternalPort");
	protocol = GetValueFromNameValueList(&data, "NewProtocol");

	if ((ext_port == NULL) || (protocol == NULL)) {
		ClearNameValueList(&data);
		SoapError(h, 402, "Invalid Args");
		return;
	}
	if ((r_host != NULL) ||
	    ((strcmp(protocol, "TCP") != 0) &&
	     (strcmp(protocol, "UDP") != 0))) {
		ClearNameValueList(&data);
		SoapError(h, 714, "NoSuchEntryInArray");
		return;
	}

	eport = (unsigned short) atoi(ext_port);

	r = upnp_get_redirect(eport, protocol);

	if (r == NULL)
		SoapError(h, 714, "NoSuchEntryInArray");
	else {
		syslog(LOG_INFO,
		       "%s: rhost='%s' %s %s found => %s:%u desc='%s'",
		       action, r_host == NULL ? "" : r_host,
		       ext_port, protocol,
		       r->iaddr, r->iport, r->desc);
		bodylen = snprintf(body, sizeof(body), resp, action,
				   "urn:schemas-upnp-org:"
				   "service:WANIPConnection:",
				   v2 ? 2 : 1,
				   r->iport, r->iaddr, r->desc,
				   get_lifetime(r), action);
		BuildSendAndCloseSoapResp(h, body, bodylen);
	}

	ClearNameValueList(&data);
}

void
DelPortMappingHandler(struct upnphttp *h, int v2)
{
	static const char resp[] =
		"<u:DeletePortMappingResponse "
		"xmlns:u=\"urn:schemas-upnp-org:service:WANIPConnection:%d\">"
		"</u:DeletePortMappingResponse>";
	char body[512];
	int bodylen;

	bodylen = snprintf(body, sizeof(body), resp, v2 ? 2 : 1);
	BuildSendAndCloseSoapResp(h, body, bodylen);
}

static void
DeletePortMapping(struct upnphttp *h, const char *action, int v2)
{
	struct upnp_redirect *r;
	struct NameValueParserData data;
	const char *r_host, *ext_port, *protocol;
	unsigned short eport;
	struct in_addr iaddr;

	if (external_addr == 0) {
		SoapError(h, 501, "ActionFailed");
		return;
	}

	ParseNameValue(h->req_buf + h->req_contentoff,
		       h->req_contentlen, &data);
	r_host = GetValueFromNameValueList(&data, "NewRemoteHost");
	ext_port = GetValueFromNameValueList(&data, "NewExternalPort");
	protocol = GetValueFromNameValueList(&data, "NewProtocol");

	if ((ext_port == NULL) || (protocol == NULL)) {
		ClearNameValueList(&data);
		SoapError(h, 402, "Invalid Args");
		return;
	} else if (r_host != NULL) {
		ClearNameValueList(&data);
		SoapError(h, 714, "NoSuchEntryInArray");
		return;
	} else if ((strcmp(protocol, "TCP") != 0) &&
		   (strcmp(protocol, "UDP") != 0)) {
	      ClearNameValueList(&data);
	      if (v2)
		      SoapError(h, 601, "Argument Value Out of Range");
	      else
		      SoapError(h, 714, "NoSuchEntryInArray");
	      return;
	}

	eport = (unsigned short) atoi(ext_port);

	syslog(LOG_INFO,
	       "%s: external port: %hu, protocol: %s", 
	       action, eport, protocol);

	r = upnp_get_redirect(eport, protocol);

	if (r == NULL) {
		ClearNameValueList(&data);
		SoapError(h, 714, "NoSuchEntryInArray");
	}

	if (GETFLAG(SECUREMODEMASK)) {
		if (inet_pton(AF_INET, r->iaddr, &iaddr) <= 0) {
			syslog(LOG_ERR, "%s: inet_pton", action);
			ClearNameValueList(&data);
			SoapError(h, 501, "ActionFailed");
		}
		if (h->clientaddr.s_addr != iaddr.s_addr) {
			syslog(LOG_INFO,
			       "Client %s tried to delete port to %s",
			       inet_ntoa(h->clientaddr), r->iaddr);
			ClearNameValueList(&data);
			if (v2)
				SoapError(h, 606, "Action not authorized");
			else
				SoapError(h, 501, "ActionFailed");
			return;
		}
	}

	if (upnp_delete_redirect(eport, protocol, v2) < 0)
		SoapError(h, 501, "ActionFailed");

	ClearNameValueList(&data);
}

/* DeletePortMappingRange was added in IGD spec v2 */
static void
DeletePortMappingRange(struct upnphttp * h, const char * action, int v2)
{
	struct upnp_redirect *r;
	static const char resp[] =
		"<u:DeletePortMappingRangeResponse "
		"xmlns:u=\"urn:schemas-upnp-org:service:WANIPConnection:2\">"
		"</u:DeletePortMappingRangeResponse>";
	struct NameValueParserData data;
	const char * protocol, * start_port, * end_port, * manage;
	unsigned short startport, endport;
	unsigned int i, found;
	struct in_addr iaddr;

	/* possible errors :
	   606 - Action not authorized
	   730 - PortMappingNotFound
	   733 - InconsistentParameters
	 */
	v2 = v2;
	if (external_addr == 0) {
		SoapError(h, 501, "ActionFailed");
		return;
	}

	ParseNameValue(h->req_buf + h->req_contentoff,
		       h->req_contentlen, &data);
	start_port = GetValueFromNameValueList(&data, "NewStartPort");
	end_port = GetValueFromNameValueList(&data, "NewEndPort");
	protocol = GetValueFromNameValueList(&data, "NewProtocol");
	manage = GetValueFromNameValueList(&data, "NewManage");

	if ((start_port == NULL) || (end_port == NULL) ||
	    (protocol == NULL) || (manage == NULL)) {
		ClearNameValueList(&data);
		SoapError(h, 402, "Invalid Args");
		return;
	} else if ((strcmp(protocol, "TCP") != 0) &&
		   (strcmp(protocol, "UDP") != 0)) {
	      ClearNameValueList(&data);
	      SoapError(h, 601, "Argument Value Out of Range");
	      return;
	}

	startport = (unsigned short)atoi(start_port);
	endport = (unsigned short)atoi(end_port);
	if (endport > startport) {
		ClearNameValueList(&data);
		SoapError(h, 733, "InconsistentParameters");
	}

	syslog(LOG_INFO,
	       "%s: start port: %hu, end port: %hu, protocol: %s, manage: %d",
	       action, startport, endport, protocol, atoi(manage));

	found = 0;
	for (i = startport; i <= endport; i++) {
		r = upnp_get_redirect((unsigned short) i, protocol);

		if (r == NULL)
			continue;
		if (inet_pton(AF_INET, r->iaddr, &iaddr) <= 0) {
			syslog(LOG_ERR, "%s: inet_pton", action);
			continue;
		}
		if (h->clientaddr.s_addr != iaddr.s_addr)
			if (GETFLAG(SECUREMODEMASK) ||
			    (atoi(manage) == 0))
				continue;
		if (upnp_delete_redirect((unsigned short) i,
					 protocol, -1) < 0)
			continue;
		found++;
	}
	if (found == 0) {
		ClearNameValueList(&data);
		SoapError(h, 730, "PortMappingNotFound");
		return;
	}

	BuildSendAndCloseSoapResp(h, resp, sizeof(resp) - 1);
	ClearNameValueList(&data);
}

static void
GetGenericPortMappingEntry(struct upnphttp *h, const char *action, int v2)
{
	struct upnp_redirect *r;
	static const char resp[] =
		"<u:%sResponse "
		"xmlns:u=\"%s%d\">"
		"<NewRemoteHost></NewRemoteHost>"
		"<NewExternalPort>%hu</NewExternalPort>"
		"<NewProtocol>%s</NewProtocol>"
		"<NewInternalPort>%u</NewInternalPort>"
		"<NewInternalClient>%s</NewInternalClient>"
		"<NewEnabled>1</NewEnabled>"
		"<NewPortMappingDescription>%s</NewPortMappingDescription>"
		"<NewLeaseDuration>%u</NewLeaseDuration>"
		"</u:%sResponse>";
	int index = 0;
	const char *m_index;
	struct NameValueParserData data;

	if (external_addr == 0) {
		SoapError(h, 501, "ActionFailed");
		return;
	}

	ParseNameValue(h->req_buf + h->req_contentoff,
		       h->req_contentlen, &data);
	m_index = GetValueFromNameValueList(&data, "NewPortMappingIndex");

	if (m_index == NULL) {
		ClearNameValueList(&data);
		SoapError(h, 402, "Invalid Args");
		return;
	}	

	index = (int) atoi(m_index);

	syslog(LOG_INFO, "%s: index=%d", action, index);

	r = upnp_get_redirect_by_index(index);

	if (r == NULL)
		SoapError(h, 713, "SpecifiedArrayIndexInvalid");
	else {
		int bodylen;
		char body[2048];

		bodylen = snprintf(body, sizeof(body), resp, action,
				   "urn:schemas-upnp-org:"
				   "service:WANIPConnection:",
				   v2 ? 2 : 1, r->eport,
				   r->proto == IPPROTO_UDP ? "UDP" : "TCP",
				   r->iport, r->iaddr, r->desc,
				   get_lifetime(r), action);
		BuildSendAndCloseSoapResp(h, body, bodylen);
	}

	ClearNameValueList(&data);
}

/* strcat_str()
 * concatenate the string and use realloc to increase the
 * memory buffer if needed. */
/* from upnpdescgen.c */
static char *
strcat_str(char * str, int * len, int * tmplen, const char * s2)
{
	int s2len;
	s2len = (int)strlen(s2);
	if(*tmplen <= (*len + s2len))
	{
		if(s2len < 256)
			*tmplen += 256;
		else
			*tmplen += s2len + 1;
		str = (char *)realloc(str, *tmplen);
	}
	/*strcpy(str + *len, s2); */
	memcpy(str + *len, s2, s2len + 1);
	*len += s2len;
	return str;
}

/*
build the PortMappingList xml document :

<p:PortMappingList xmlns:p="urn:schemas-upnp-org:gw:WANIPConnection"
xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
xsi:schemaLocation="urn:schemas-upnp-org:gw:WANIPConnection
http://www.upnp.org/schemas/gw/WANIPConnection-v2.xsd">
<p:PortMappingEntry>
<p:NewRemoteHost>202.233.2.1</p:NewRemoteHost>
<p:NewExternalPort>2345</p:NewExternalPort>
<p:NewProtocol>TCP</p:NewProtocol>
<p:NewInternalPort>2345</p:NewInternalPort>
<p:NewInternalClient>192.168.1.137</p:NewInternalClient>
<p:NewEnabled>1</p:NewEnabled>
<p:NewDescription>dooom</p:NewDescription>
<p:NewLeaseTime>345</p:NewLeaseTime>
</p:PortMappingEntry>
</p:PortMappingList>
*/

static char *
genEntry(struct upnp_redirect *r, char * str, int * len, int * tmplen)
{
	char tmp[2048];

	(void) snprintf(tmp, sizeof(tmp),
			"<p:PortMappingEntry>"
			"<p:NewRemoteHost></p:NewRemoteHost>"
			"<p:NewExternalPort>%hu</p:NewExternalPort>"
			"<p:NewProtocol>%s</p:NewProtocol>"
			"<p:NewInternalPort>%u</p:NewInternalPort>"
			"<p:NewInternalClient>%s</p:NewInternalClient>"
			"<p:NewEnabled>1</p:NewEnabled>"
			"<p:NewDescription>%s</p:NewDescription>"
			"<p:NewLeaseTime>%u</p:NewLeaseTime>"
			"</p:PortMappingEntry>",
			r->eport,
			r->proto == IPPROTO_UDP ? "UDP" : "TCP",
			r->iport, r->iaddr, r->desc,
			get_lifetime(r));
	return strcat_str(str, len, tmplen, tmp);
}

/* GetListOfPortMappings was added in the IGD v2 specification */
static void
GetListOfPortMappings(struct upnphttp * h, const char * action, int v2)
{
	struct upnp_redirect *r;
	char * str;
	int len, tmplen;
	struct NameValueParserData data;
	const char * protocol, * start_port, * end_port, * manage, * number;
	unsigned short startport, endport;
	unsigned int limit, i, found;
	struct in_addr iaddr;

	v2 = v2;
	if (external_addr == 0) {
		SoapError(h, 501, "ActionFailed");
		return;
	}

	ParseNameValue(h->req_buf + h->req_contentoff,
		       h->req_contentlen, &data);
	start_port = GetValueFromNameValueList(&data, "NewStartPort");
	end_port = GetValueFromNameValueList(&data, "NewEndPort");
	protocol = GetValueFromNameValueList(&data, "NewProtocol");
	manage = GetValueFromNameValueList(&data, "NewManage");
	number = GetValueFromNameValueList(&data, "NewNumberOfPorts");

	if ((start_port == NULL) || (end_port == NULL) ||
	    (protocol == NULL) || (manage == NULL) || (number == NULL)) {
		ClearNameValueList(&data);
		SoapError(h, 402, "Invalid Args");
		return;
	} else if ((strcmp(protocol, "TCP") != 0) &&
		   (strcmp(protocol, "UDP") != 0)) {
	      ClearNameValueList(&data);
	      SoapError(h, 601, "Argument Value Out of Range");
	      return;
	}

	startport = (unsigned short)atoi(start_port);
	endport = (unsigned short)atoi(end_port);
	if (endport > startport) {
		ClearNameValueList(&data);
		SoapError(h, 733, "InconsistentParameters");
	}

	syslog(LOG_INFO,
	       "%s: start port: %hu, end port: %hu, protocol: %s, manage: %d",
	       action, startport, endport, protocol, atoi(manage));

	len = 0;
	tmplen = 2048;
	str = (char *) malloc(tmplen);
	if (str == NULL) {
		ClearNameValueList(&data);
		SoapError(h, 501, "ActionFailed");
		return;
	}
	str = strcat_str(str, &len, &tmplen,
		"<u:GetListOfPortMappingsResponse "
		"xmlns:u=\"urn:schemas-upnp-org:service:WANIPConnection:2\">"
		"<NewPortListing>");
	limit = (unsigned int)atoi(number);
	found = 0;
	for (i = startport; i <= endport; i++) {
		r = upnp_get_redirect((unsigned short) i, protocol);

		if (r == NULL)
			continue;
		if (inet_pton(AF_INET, r->iaddr, &iaddr) <= 0) {
			syslog(LOG_ERR, "%s: inet_pton", action);
			continue;
		}
		if (h->clientaddr.s_addr != iaddr.s_addr)
			if (GETFLAG(SECUREMODEMASK) ||
			    (atoi(manage) == 0))
				continue;
		str = genEntry(r, str, &len, &tmplen);
		found++;
		if (found == limit)
			break;
	}
	if (found == 0) {
		ClearNameValueList(&data);
		SoapError(h, 730, "PortMappingNotFound");
		return;
	}

	str = strcat_str(str, &len, &tmplen,
		"</NewPortListing></u:GetListOfPortMappingsResponse>");
	ClearNameValueList(&data);
	BuildSendAndCloseSoapResp(h, str, len);
}

#ifdef ENABLE_L3F_SERVICE
static void
SetDefaultConnectionService(struct upnphttp * h, const char * action, int v2)
{
	static const char resp[] =
		"<u:SetDefaultConnectionServiceResponse "
		"xmlns:u=\"urn:schemas-upnp-org:service:Layer3Forwarding:1\">"
		"</u:SetDefaultConnectionServiceResponse>";
	struct NameValueParserData data;
	char * p;

	v2 = v2;
	ParseNameValue(h->req_buf + h->req_contentoff,
		       h->req_contentlen, &data);
	p = GetValueFromNameValueList(&data, "NewDefaultConnectionService");
	if(p) {
		syslog(LOG_INFO, "%s(%s) : Ignored", action, p);
	}
	ClearNameValueList(&data);
	BuildSendAndCloseSoapResp(h, resp, sizeof(resp)-1);
}

static void
GetDefaultConnectionService(struct upnphttp * h, const char * action, int v2)
{
	static const char resp[] =
		"<u:%sResponse "
		"xmlns:u=\"urn:schemas-upnp-org:service:Layer3Forwarding:1\">"
		"<NewDefaultConnectionService>%s:WANConnectionDevice:2,"
		"urn:upnp-org:serviceId:WANIPConn1</NewDefaultConnectionService>"
		"</u:%sResponse>";
	/* example from UPnP_IGD_Layer3Forwarding 1.0.pdf :
	 * uuid:44f5824f-c57d-418c-a131-f22b34e14111:WANConnectionDevice:1,
	 * urn:upnp-org:serviceId:WANPPPConn1 */
	char body[1024];
	int bodylen;

	v2 = v2;
	bodylen = snprintf(body, sizeof(body), resp,
	                   action, uuidvalue, action);
	BuildSendAndCloseSoapResp(h, body, bodylen);
}
#endif

/* Added for compliance with WANIPConnection v2 */
static void
SetConnectionType(struct upnphttp * h, const char * action, int v2)
{
	const char * connection_type;
	struct NameValueParserData data;

	action = action;
	ParseNameValue(h->req_buf + h->req_contentoff,
		       h->req_contentlen, &data);
	connection_type = GetValueFromNameValueList(&data, "NewConnectionType");
	/* Unconfigured, IP_Routed, IP_Bridged */
	connection_type = connection_type;
	ClearNameValueList(&data);
	if (v2)
		SoapError(h, 731, "ReadOnly");
	else
		SoapError(h, 501, "ActionFailed");
}

/* Added for compliance with WANIPConnection v2 */
static void
RequestConnection(struct upnphttp * h, const char * action, int v2)
{
	action = action;
	if (v2)
		SoapError(h, 606, "Action not authorized");
	else
		SoapError(h, 501, "ActionFailed");
}

/* Added for compliance with WANIPConnection v2 */
static void
ForceTermination(struct upnphttp * h, const char * action, int v2)
{
	action = action;
	if (v2)
		SoapError(h, 606, "Action not authorized");
	else
		SoapError(h, 501, "ActionFailed");
}

/*
If a control point calls QueryStateVariable on a state variable that is not
buffered in memory within (or otherwise available from) the service,
the service must return a SOAP fault with an errorCode of 404 Invalid Var.

QueryStateVariable remains useful as a limited test tool but may not be
part of some future versions of UPnP.
*/
static void
QueryStateVariable(struct upnphttp * h, const char * action, int v2)
{
	static const char resp[] =
        "<u:%sResponse "
        "xmlns:u=\"%s\">"
		"<return>%s</return>"
        "</u:%sResponse>";

	char body[512];
	int bodylen;
	struct NameValueParserData data;
	const char * var_name;

	v2 = v2;
	ParseNameValue(h->req_buf + h->req_contentoff,
		       h->req_contentlen, &data);
	/*var_name = GetValueFromNameValueList(&data, "QueryStateVariable"); */
	/*var_name = GetValueFromNameValueListIgnoreNS(&data, "varName");*/
	var_name = GetValueFromNameValueList(&data, "varName");

	/*syslog(LOG_INFO, "QueryStateVariable(%.40s)", var_name); */

	if(!var_name)
	{
		SoapError(h, 402, "Invalid Args");
	}
	else if(strcmp(var_name, "ConnectionStatus") == 0)
	{	
		const char * status = "Connected";

		if (external_addr == 0)
			status = "Disconnected";
		bodylen = snprintf(body, sizeof(body), resp,
                           action, "urn:schemas-upnp-org:control-1-0",
		                   status, action);
		BuildSendAndCloseSoapResp(h, body, bodylen);
	}
	else if(strcmp(var_name, "PortMappingNumberOfEntries") == 0)
	{
		char strn[10];
		snprintf(strn, sizeof(strn), "%i",
		         upnp_get_redirect_number());
		bodylen = snprintf(body, sizeof(body), resp,
                           action, "urn:schemas-upnp-org:control-1-0",
		                   strn, action);
		BuildSendAndCloseSoapResp(h, body, bodylen);
	}
	else if(strcmp(var_name, "SystemUpdateID") == 0)
	{
		char strn[10];
		snprintf(strn, sizeof(strn), "%lu", SystemUpdateID);
		bodylen = snprintf(body, sizeof(body), resp,
                           action, "urn:schemas-upnp-org:control-1-0",
		                   strn, action);
		BuildSendAndCloseSoapResp(h, body, bodylen);
	}
	else
	{
		syslog(LOG_NOTICE, "%s: Unknown: %s", action, var_name?var_name:"");
		SoapError(h, 404, "Invalid Var");
	}

	ClearNameValueList(&data);	
}

/* Windows XP as client send the following requests :
 * GetConnectionTypeInfo
 * GetNATRSIPStatus
 * ? GetTotalBytesSent - WANCommonInterfaceConfig
 * ? GetTotalBytesReceived - idem
 * ? GetTotalPacketsSent - idem
 * ? GetTotalPacketsReceived - idem
 * GetCommonLinkProperties - idem
 * GetStatusInfo - WANIPConnection
 * GetExternalIPAddress
 * QueryStateVariable / ConnectionStatus!
 */
static const struct 
{
	const char * methodName; 
	void (*methodImpl)(struct upnphttp *, const char *, int);
}
soapMethods[] =
{
	{ "GetConnectionTypeInfo", GetConnectionTypeInfo },
	{ "GetNATRSIPStatus", GetNATRSIPStatus},
	{ "GetExternalIPAddress", GetExternalIPAddress},
	{ "AddPortMapping", AddPortMapping},
	{ "DeletePortMapping", DeletePortMapping},
	{ "GetGenericPortMappingEntry", GetGenericPortMappingEntry},
	{ "GetSpecificPortMappingEntry", GetSpecificPortMappingEntry},
	{ "QueryStateVariable", QueryStateVariable},
	{ "GetTotalBytesSent", GetTotalBytesSent},
	{ "GetTotalBytesReceived", GetTotalBytesReceived},
	{ "GetTotalPacketsSent", GetTotalPacketsSent},
	{ "GetTotalPacketsReceived", GetTotalPacketsReceived},
	{ "GetCommonLinkProperties", GetCommonLinkProperties},
	{ "GetStatusInfo", GetStatusInfo},
/* Required in WANIPConnection:2 */
	{ "SetConnectionType", SetConnectionType},
	{ "RequestConnection", RequestConnection},
	{ "ForceTermination", ForceTermination},
	{ "AddAnyPortMapping", AddAnyPortMapping},
	{ "DeletePortMappingRange", DeletePortMappingRange},
	{ "GetListOfPortMappings", GetListOfPortMappings},
#ifdef ENABLE_L3F_SERVICE
	{ "SetDefaultConnectionService", SetDefaultConnectionService},
	{ "GetDefaultConnectionService", GetDefaultConnectionService},
#endif
	{ 0, 0 }
};

void
ExecuteSoapAction(struct upnphttp * h, const char * action, int n)
{
	char * p;
	char * p2;
	int i, v2, len, methodlen;

	i = 0;
	v2 = 0;
	p = strchr(action, '#');

	if(p)
	{
		if ((p > action) && (p[-1] == '2'))
			v2 = 1;
		p++;
		p2 = strchr(p, '"');
		if(p2)
			methodlen = p2 - p;
		else
			methodlen = n - (p - action);
		/*syslog(LOG_DEBUG, "SoapMethod: %.*s", methodlen, p);*/
		while(soapMethods[i].methodName)
		{
			len = strlen(soapMethods[i].methodName);
			if(strncmp(p, soapMethods[i].methodName, len) == 0)
			{
				soapMethods[i].methodImpl(h, soapMethods[i].methodName, v2);
				return;
			}
			i++;
		}

		syslog(LOG_NOTICE, "SoapMethod: Unknown: %.*s", methodlen, p);
	}

	SoapError(h, 401, "Invalid Action");
}

/* Standard Errors:
 *
 * errorCode errorDescription Description
 * --------	---------------- -----------
 * 401 		Invalid Action 	No action by that name at this service.
 * 402 		Invalid Args 	Could be any of the following: not enough in args,
 * 							too many in args, no in arg by that name, 
 * 							one or more in args are of the wrong data type.
 * 403 		Out of Sync 	Out of synchronization.
 * 501 		Action Failed 	May be returned in current state of service
 * 							prevents invoking that action.
 * 600-699 	TBD 			Common action errors. Defined by UPnP Forum
 * 							Technical Committee.
 * 700-799 	TBD 			Action-specific errors for standard actions.
 * 							Defined by UPnP Forum working committee.
 * 800-899 	TBD 			Action-specific errors for non-standard actions. 
 * 							Defined by UPnP vendor.
*/
void
SoapError(struct upnphttp * h, int errCode, const char * errDesc)
{
	static const char resp[] = 
		"<s:Envelope "
		"xmlns:s=\"http://schemas.xmlsoap.org/soap/envelope/\" "
		"s:encodingStyle=\"http://schemas.xmlsoap.org/soap/encoding/\">"
		"<s:Body>"
		"<s:Fault>"
		"<faultcode>s:Client</faultcode>"
		"<faultstring>UPnPError</faultstring>"
		"<detail>"
		"<UPnPError xmlns=\"urn:schemas-upnp-org:control-1-0\">"
		"<errorCode>%d</errorCode>"
		"<errorDescription>%s</errorDescription>"
		"</UPnPError>"
		"</detail>"
		"</s:Fault>"
		"</s:Body>"
		"</s:Envelope>";

	char body[2048];
	int bodylen;

	syslog(LOG_INFO, "Returning UPnPError %d: %s", errCode, errDesc);
	bodylen = snprintf(body, sizeof(body), resp, errCode, errDesc);
	BuildResp2_upnphttp(h, 500, "Internal Server Error", body, bodylen);
	SendResp_upnphttp(h);
	CloseSocket_upnphttp(h);
}

