/*
 * Copyright (C) 2010-2011  Internet Systems Consortium, Inc. ("ISC")
 *
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND ISC DISCLAIMS ALL WARRANTIES WITH
 * REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS.  IN NO EVENT SHALL ISC BE LIABLE FOR ANY SPECIAL, DIRECT,
 * INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

/* $Id: simple.c 1253 2011-07-04 14:54:05Z fdupont $ */

/*
 * PCP (port-control-protocol) client library simple test client
 *
 * Francis_Dupont@isc.org, July 2010
 *
 * ./simple <server> <address> <priv> <[!]pub> <proto> <lifetime>
 */

#ifdef WIN32
#include <ws2tcpip.h>
#else
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>
#endif
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "pcp.h"

int
main(int argc, char *argv[])
{
	pcp_t pcp;
	pcp_request_t request;
	pcp_option_t **options, *opt;
	pcp_response_t response;
	int ret, fd, tries;
	fd_set set;
	struct sockaddr_in server;
	struct timeval tv;
	struct in_addr in;
	unsigned int optidx;
#ifdef WIN32
	WSADATA wsaData;
#endif

#define USAGE	\
	"usage: %s <server> <address> <priv> <[!]pub> <proto> <lifetime>\n"
	if (argc != 7) {
		fprintf(stderr, USAGE, argv[0]);
		exit(1);
	}
#ifdef WIN32
	ret = WSAStartup(MAKEWORD(2,2), &wsaData);
	if (ret != 0) {
		fprintf(stderr, "WSAStartup() failed.\n");
		exit(1);
	}
#endif
	memset(&server, 0, sizeof(server));
	server.sin_family = AF_INET;
	server.sin_addr.s_addr = inet_addr(argv[1]);
	ret = pcp_init((struct sockaddr *) &server, NULL, &pcp);
	if (ret < PCP_OK) {
		fprintf(stderr, "pcp_init: %s\n", pcp_strerror(ret));
		exit(1);
	}
	ret = pcp_getsocket(&pcp, &fd);
	if (ret < PCP_OK) {
		fprintf(stderr, "pcp_getsocket: %s\n", pcp_strerror(ret));
		exit(1);
	}
	memset(&request, 0, sizeof(request));
	request.opcode = PCP_OPCODE_MAP4;
	request.intaddr4 = inet_addr(argv[2]);
	request.lifetime = atoi(argv[6]);
	if (strcmp(argv[5], "all") == 0)
		request.protocol = 0;
	else if (strcmp(argv[5], "udp") == 0)
		request.protocol = IPPROTO_UDP;
	else if (strcmp(argv[5], "tcp") == 0)
		request.protocol = IPPROTO_TCP;
	else {
		fprintf(stderr, "unknown protocol '%s'\n", argv[5]);
		exit(1);
	}
	request.intport = atoi(argv[3]);
	options = NULL;
	if (*argv[4] == '!') {
		ret = pcp_prefer_failure(&options);
		if (ret != PCP_OK) {
			fprintf(stderr,
				"pcp_prefer_failure: %s\n",
				pcp_strerror(ret));
			exit(1);
		}
		request.extport = atoi(argv[4] + 1);
	} else
		request.extport = atoi(argv[4]);
	ret = pcp_makerequest(&pcp, &request, options);
	pcp_freeoptions(options);
	if (ret < PCP_OK) {
		fprintf(stderr, "pcp_makerequest: %s\n", pcp_strerror(ret));
		exit(1);
	}
	ret = pcp_sendrequest(&pcp);
	if (ret < PCP_OK) {
		fprintf(stderr, "pcp_sendrequest: %s\n", pcp_strerror(ret));
		exit(1);
	}
	for (;;) {
		ret = pcp_gettries(&pcp, &tries);
		if (ret < PCP_OK) {
			fprintf(stderr,
				"pcp_gettries: %s\n",
				pcp_strerror(ret));
			exit(1);
		}
		if (tries > PCP_MAX_TRIES) {
			fprintf(stderr, "too many retries: giving up\n");
			exit(1);
		}
		FD_ZERO(&set);
		FD_SET(fd, &set);
		ret = pcp_gettimeout(&pcp, &tv, 1);
		if (ret < PCP_OK) {
			fprintf(stderr,
				"pcp_gettimeout: %s\n",
				pcp_strerror(ret));
			exit(1);
		}
		ret = select(fd + 1, &set, NULL, NULL, &tv);
		if (ret < 0) {
			fprintf(stderr, "select: %s\n", strerror(errno));
			continue;
		}
		if (FD_ISSET(fd, &set)) {
			memset(&response, 0, sizeof(response));
			ret = pcp_recvresponse(&pcp, &response);
			if (ret < PCP_OK) {
				fprintf(stderr,
					"pcp_recvresponse: %s\n",
					pcp_strerror(ret));
				if (ret == PCP_ERR_RECVBAD) {
					fprintf(stderr,
						"at line %d\n",
						pcp_debug_line);
					abort();
				}
				exit(1);
			}
			if (ret == PCP_TRY_AGAIN)
				fprintf(stderr, "try again\n");
			else
				break;
		}
		ret = pcp_sendrequest(&pcp);
		if (ret < PCP_OK) {
			fprintf(stderr,
				"pcp_sendrequest: %s\n",
				pcp_strerror(ret));
			exit(1);
		}
	}
	printf("epoch: %d\n", (int) response.epoch);
	in.s_addr = response.assigned.intaddr4;
	printf("internal address: %s\n", inet_ntoa(in));
	printf("internal port: %d\n", (int) response.assigned.intport);
	in.s_addr = response.assigned.extaddr4;
	printf("external address: %s\n", inet_ntoa(in));
	printf("external port: %d\n", (int) response.assigned.extport);
	printf("lifetime: %d\n", (int) response.assigned.lifetime);
	for (optidx = 0; optidx < response.optcnt; optidx++) {
		opt = &response.options[optidx];
		switch (opt->code) {
		case PCP_OPTION_UNPROCESSED:
			if (opt->length == 0)
				break;
			printf("option: unprocessed\n");
			break;
		case PCP_OPTION_THIRD_PARTY:
			if (opt->length != 4) {
				printf("option: third party?\n");
				break;
			}
			memcpy(&in, opt->data, 4);
			printf("option: third party: %s\n", inet_ntoa(in));
			break;
		case PCP_OPTION_PREF_FAIL:
			printf("option: prefer failure\n");
			break;
		case PCP_OPTION_FILTER:
			printf("option: filter\n");
			break;
		default:
			printf("option: %u\n", opt->code);
			break;
		}
	}
	ret = pcp_close(&pcp);
	if (ret < PCP_OK) {
		fprintf(stderr, "pcp_close: %s\n", pcp_strerror(ret));
		exit(1);
	}
#ifdef WIN32
	(void) WSACleanup();
#endif
	exit(0);
}
