/*
 * Copyright (C) Internet Systems Consortium, Inc. ("ISC")
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * See the COPYRIGHT file distributed with this work for additional
 * information regarding copyright ownership.
 */


/* RFC1183 */

#ifndef RDATA_GENERIC_RP_17_C
#define RDATA_GENERIC_RP_17_C

#define RRTYPE_RP_ATTRIBUTES (0)

static inline isc_result_t
fromtext_rp(ARGS_FROMTEXT) {
	isc_token_t token;
	dns_name_t name;
	isc_buffer_t buffer;
	int i;
	bool ok;

	REQUIRE(type == dns_rdatatype_rp);

	UNUSED(type);
	UNUSED(rdclass);
	UNUSED(callbacks);

	if (origin == NULL)
		origin = dns_rootname;

	for (i = 0; i < 2; i++) {
		RETERR(isc_lex_getmastertoken(lexer, &token,
					      isc_tokentype_string,
					      false));
		dns_name_init(&name, NULL);
		buffer_fromregion(&buffer, &token.value.as_region);
		RETTOK(dns_name_fromtext(&name, &buffer, origin,
					 options, target));
		ok = true;
		if ((options & DNS_RDATA_CHECKNAMES) != 0 && i == 0)
			ok = dns_name_ismailbox(&name);
		if (!ok && (options & DNS_RDATA_CHECKNAMESFAIL) != 0)
			RETTOK(DNS_R_BADNAME);
		if (!ok && callbacks != NULL)
			warn_badname(&name, lexer, callbacks);
	}
	return (ISC_R_SUCCESS);
}

static inline isc_result_t
totext_rp(ARGS_TOTEXT) {
	isc_region_t region;
	dns_name_t rmail;
	dns_name_t email;
	dns_name_t prefix;
	bool sub;

	REQUIRE(rdata->type == dns_rdatatype_rp);
	REQUIRE(rdata->length != 0);

	dns_name_init(&rmail, NULL);
	dns_name_init(&email, NULL);
	dns_name_init(&prefix, NULL);

	dns_rdata_toregion(rdata, &region);

	dns_name_fromregion(&rmail, &region);
	isc_region_consume(&region, rmail.length);

	dns_name_fromregion(&email, &region);
	isc_region_consume(&region, email.length);

	sub = name_prefix(&rmail, tctx->origin, &prefix);
	RETERR(dns_name_totext(&prefix, sub, target));

	RETERR(str_totext(" ", target));

	sub = name_prefix(&email, tctx->origin, &prefix);
	return (dns_name_totext(&prefix, sub, target));
}

static inline isc_result_t
fromwire_rp(ARGS_FROMWIRE) {
	dns_name_t rmail;
	dns_name_t email;

	REQUIRE(type == dns_rdatatype_rp);

	UNUSED(type);
	UNUSED(rdclass);

	dns_decompress_setmethods(dctx, DNS_COMPRESS_NONE);

	dns_name_init(&rmail, NULL);
	dns_name_init(&email, NULL);

	RETERR(dns_name_fromwire(&rmail, source, dctx, options, target));
	return (dns_name_fromwire(&email, source, dctx, options, target));
}

static inline isc_result_t
towire_rp(ARGS_TOWIRE) {
	isc_region_t region;
	dns_name_t rmail;
	dns_name_t email;
	dns_offsets_t roffsets;
	dns_offsets_t eoffsets;

	REQUIRE(rdata->type == dns_rdatatype_rp);
	REQUIRE(rdata->length != 0);

	dns_compress_setmethods(cctx, DNS_COMPRESS_NONE);
	dns_name_init(&rmail, roffsets);
	dns_name_init(&email, eoffsets);

	dns_rdata_toregion(rdata, &region);

	dns_name_fromregion(&rmail, &region);
	isc_region_consume(&region, rmail.length);

	RETERR(dns_name_towire(&rmail, cctx, target));

	dns_name_fromregion(&rmail, &region);
	isc_region_consume(&region, rmail.length);

	return (dns_name_towire(&rmail, cctx, target));
}

static inline int
compare_rp(ARGS_COMPARE) {
	isc_region_t region1;
	isc_region_t region2;
	dns_name_t name1;
	dns_name_t name2;
	int order;

	REQUIRE(rdata1->type == rdata2->type);
	REQUIRE(rdata1->rdclass == rdata2->rdclass);
	REQUIRE(rdata1->type == dns_rdatatype_rp);
	REQUIRE(rdata1->length != 0);
	REQUIRE(rdata2->length != 0);

	dns_name_init(&name1, NULL);
	dns_name_init(&name2, NULL);

	dns_rdata_toregion(rdata1, &region1);
	dns_rdata_toregion(rdata2, &region2);

	dns_name_fromregion(&name1, &region1);
	dns_name_fromregion(&name2, &region2);

	order = dns_name_rdatacompare(&name1, &name2);
	if (order != 0)
		return (order);

	isc_region_consume(&region1, name_length(&name1));
	isc_region_consume(&region2, name_length(&name2));

	dns_name_init(&name1, NULL);
	dns_name_init(&name2, NULL);

	dns_name_fromregion(&name1, &region1);
	dns_name_fromregion(&name2, &region2);

	return (dns_name_rdatacompare(&name1, &name2));
}

static inline isc_result_t
fromstruct_rp(ARGS_FROMSTRUCT) {
	dns_rdata_rp_t *rp;
	isc_region_t region;

	REQUIRE(type == dns_rdatatype_rp);
	REQUIRE(((dns_rdata_rp_t *)source) != NULL);
	REQUIRE(((dns_rdata_rp_t *)source)->common.rdtype == type);
	REQUIRE(((dns_rdata_rp_t *)source)->common.rdclass == rdclass);

	rp = source;

	UNUSED(type);
	UNUSED(rdclass);

	dns_name_toregion(&rp->mail, &region);
	RETERR(isc_buffer_copyregion(target, &region));
	dns_name_toregion(&rp->text, &region);
	return (isc_buffer_copyregion(target, &region));
}

static inline isc_result_t
tostruct_rp(ARGS_TOSTRUCT) {
	isc_result_t result;
	isc_region_t region;
	dns_rdata_rp_t *rp;
	dns_name_t name;

	REQUIRE(((dns_rdata_rp_t *)target) != NULL);
	REQUIRE(rdata->type == dns_rdatatype_rp);
	REQUIRE(rdata->length != 0);

	rp = target;

	rp->common.rdclass = rdata->rdclass;
	rp->common.rdtype = rdata->type;
	ISC_LINK_INIT(&rp->common, link);

	dns_name_init(&name, NULL);
	dns_rdata_toregion(rdata, &region);
	dns_name_fromregion(&name, &region);
	dns_name_init(&rp->mail, NULL);
	RETERR(name_duporclone(&name, mctx, &rp->mail));
	isc_region_consume(&region, name_length(&name));
	dns_name_fromregion(&name, &region);
	dns_name_init(&rp->text, NULL);
	result = name_duporclone(&name, mctx, &rp->text);
	if (result != ISC_R_SUCCESS)
		goto cleanup;

	rp->mctx = mctx;
	return (ISC_R_SUCCESS);

 cleanup:
	if (mctx != NULL)
		dns_name_free(&rp->mail, mctx);
	return (ISC_R_NOMEMORY);
}

static inline void
freestruct_rp(ARGS_FREESTRUCT) {
	dns_rdata_rp_t *rp;

	REQUIRE(((dns_rdata_rp_t *)source) != NULL);
	REQUIRE(((dns_rdata_rp_t *)source)->common.rdtype == dns_rdatatype_rp);

	rp = source;

	if (rp->mctx == NULL)
		return;

	dns_name_free(&rp->mail, rp->mctx);
	dns_name_free(&rp->text, rp->mctx);
	rp->mctx = NULL;
}

static inline isc_result_t
additionaldata_rp(ARGS_ADDLDATA) {
	REQUIRE(rdata->type == dns_rdatatype_rp);

	UNUSED(rdata);
	UNUSED(add);
	UNUSED(arg);

	return (ISC_R_SUCCESS);
}

static inline isc_result_t
digest_rp(ARGS_DIGEST) {
	isc_region_t r;
	dns_name_t name;

	REQUIRE(rdata->type == dns_rdatatype_rp);

	dns_rdata_toregion(rdata, &r);
	dns_name_init(&name, NULL);

	dns_name_fromregion(&name, &r);
	RETERR(dns_name_digest(&name, digest, arg));
	isc_region_consume(&r, name_length(&name));

	dns_name_init(&name, NULL);
	dns_name_fromregion(&name, &r);

	return (dns_name_digest(&name, digest, arg));
}

static inline bool
checkowner_rp(ARGS_CHECKOWNER) {

	REQUIRE(type == dns_rdatatype_rp);

	UNUSED(name);
	UNUSED(type);
	UNUSED(rdclass);
	UNUSED(wildcard);

	return (true);
}

static inline bool
checknames_rp(ARGS_CHECKNAMES) {
	isc_region_t region;
	dns_name_t name;

	REQUIRE(rdata->type == dns_rdatatype_rp);

	UNUSED(owner);

	dns_rdata_toregion(rdata, &region);
	dns_name_init(&name, NULL);
	dns_name_fromregion(&name, &region);
	if (!dns_name_ismailbox(&name)) {
		if (bad != NULL)
				dns_name_clone(&name, bad);
		return (false);
	}
	return (true);
}

static inline int
casecompare_rp(ARGS_COMPARE) {
	return (compare_rp(rdata1, rdata2));
}
#endif	/* RDATA_GENERIC_RP_17_C */
