/*
 * Copyright (C) 2010 Andrew Tridgell
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the
 * above copyright notice and this permission notice appear in all
 * copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR
 * DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
 * STICHTING NLNET BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE
 * USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/*
  this provides a very simple example of an external loadable DLZ
  driver, with update support
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>
#include <stdint.h>
#include <stdarg.h>

#include "dlz_minimal.h"


#define CHECK(x) \
	do { \
		result = (x); \
		if (result != ISC_R_SUCCESS) \
			goto failure; \
	} while (0)

/* For this simple example, use fixed sized strings */
struct record {
	char name[100];
	char type[10];
	char data[200];
	uint32_t ttl;
};

#define MAX_RECORDS 100

struct dlz_example_data {
	char *zone_name;

	/* an example driver doesn't need good memory management :-) */
	struct record current[MAX_RECORDS];
	struct record adds[MAX_RECORDS];
	struct record deletes[MAX_RECORDS];

	bool transaction_started;

	/* helper functions from the dlz_dlopen driver */
	void (*log)(int level, const char *fmt, ...);
	isc_result_t (*putrr)(dns_sdlzlookup_t *handle, const char *type,
			      dns_ttl_t ttl, const char *data);
	isc_result_t (*putnamedrr)(dns_sdlzlookup_t *handle, const char *name,
				   const char *type, dns_ttl_t ttl, const char *data);
	isc_result_t (*writeable_zone)(dns_view_t *view, const char *zone_name);
};

static bool single_valued(const char *type)
{
	const char *single[] = { "soa", "cname", NULL };
	int i;
	for (i=0; single[i]; i++) {
		if (strcasecmp(single[i], type) == 0) {
			return true;
		}
	}
	return false;
}

/*
  add a record to a list
 */
static isc_result_t add_name(struct dlz_example_data *state,
			     struct record *list, const char *name, const char *type,
			     uint32_t ttl, const char *data)
{
	int i;
	bool single = single_valued(type);
	int first_empty = -1;

	for (i=0; i<MAX_RECORDS; i++) {
		if (first_empty == -1 && strlen(list[i].name) == 0) {
			first_empty = i;
		}
		if (strcasecmp(list[i].name, name) != 0)
			continue;
		if (strcasecmp(list[i].type, type) != 0)
			continue;
		if (!single && strcasecmp(list[i].data, data) != 0)
			continue;
		break;
	}
	if (i == MAX_RECORDS && first_empty != -1) {
		i = first_empty;
	}
	if (i == MAX_RECORDS) {
		if (state->log != NULL)
			state->log(ISC_LOG_ERROR,
				   "dlz_example: out of record space");
		return (ISC_R_FAILURE);
	}

	if (strlen(name) >= sizeof(list[i].name) ||
	    strlen(type) >= sizeof(list[i].type) ||
	    strlen(data) >= sizeof(list[i].data))
		return (ISC_R_NOSPACE);

	strncpy(list[i].name, name, sizeof(list[i].name));
	strncpy(list[i].type, type, sizeof(list[i].type));
	strncpy(list[i].data, data, sizeof(list[i].data));
	list[i].ttl = ttl;

	return (ISC_R_SUCCESS);
}

/*
  delete a record from a list
 */
static isc_result_t del_name(struct dlz_example_data *state,
			     struct record *list, const char *name, const char *type,
			     uint32_t ttl, const char *data)
{
	int i;
	for (i=0; i<MAX_RECORDS; i++) {
		if (strcasecmp(name, list[i].name) == 0 &&
		    strcasecmp(type, list[i].type) == 0 &&
		    strcasecmp(data, list[i].data) == 0 &&
		    ttl == list[i].ttl) {
			break;
		}
	}
	if (i == MAX_RECORDS) {
		return ISC_R_NOTFOUND;
	}
	memset(&list[i], 0, sizeof(struct record));
	return ISC_R_SUCCESS;
}



/*
  return the version of the API
 */
int dlz_version(unsigned int *flags)
{
	return DLZ_DLOPEN_VERSION;
}

/*
   remember a helper function from the bind9 dlz_dlopen driver
 */
static void b9_add_helper(struct dlz_example_data *state, const char *helper_name, void *ptr)
{
	if (strcmp(helper_name, "log") == 0) {
		state->log = ptr;
	}
	if (strcmp(helper_name, "putrr") == 0) {
		state->putrr = ptr;
	}
	if (strcmp(helper_name, "putnamedrr") == 0) {
		state->putnamedrr = ptr;
	}
	if (strcmp(helper_name, "writeable_zone") == 0) {
		state->writeable_zone = ptr;
	}
}

/*
  called to initialise the driver
 */
isc_result_t dlz_create(const char *dlzname, unsigned int argc, char *argv[],
			void **dbdata, ...)
{
	struct dlz_example_data *state;
	const char *helper_name;
	va_list ap;
	char soa_data[200];
	const char *extra;
	isc_result_t result;
	int n;

	state = calloc(1, sizeof(struct dlz_example_data));
	if (state == NULL) {
		return ISC_R_NOMEMORY;
	}

	/* fill in the helper functions */
	va_start(ap, dbdata);
	while ((helper_name = va_arg(ap, const char *)) != NULL) {
		b9_add_helper(state, helper_name, va_arg(ap, void *));
	}
	va_end(ap);

	if (argc < 2) {
		if (state->log != NULL)
			state->log(ISC_LOG_ERROR,
				   "dlz_example: please specify a zone name");
		dlz_destroy(state);
		return (ISC_R_FAILURE);
	}

	state->zone_name = strdup(argv[1]);
	if (state->zone_name == NULL) {
		free(state);
		return (ISC_R_NOMEMORY);
	}

	if (strcmp(state->zone_name, ".") == 0)
		extra = ".root";
	else
		extra = ".";

	n = sprintf(soa_data, "%s hostmaster%s%s 123 900 600 86400 3600",
		    state->zone_name, extra, state->zone_name);

	if (n < 0)
		CHECK(ISC_R_FAILURE);
	if ((unsigned)n >= sizeof(soa_data))
		CHECK(ISC_R_NOSPACE);

	add_name(state, &state->current[0], state->zone_name, "soa", 3600, soa_data);
	add_name(state, &state->current[0], state->zone_name, "ns", 3600, state->zone_name);
	add_name(state, &state->current[0], state->zone_name, "a", 1800, "10.53.0.1");

	if (state->log != NULL)
		state->log(ISC_LOG_INFO, "dlz_example: started for zone %s",
			   state->zone_name);

	*dbdata = state;
	return (ISC_R_SUCCESS);

 failure:
	free(state);
	return (result);

}

/*
  shutdown the backend
 */
void dlz_destroy(void *dbdata)
{
	struct dlz_example_data *state = (struct dlz_example_data *)dbdata;

	if (state->log != NULL)
		state->log(ISC_LOG_INFO,
			   "dlz_example: shutting down zone %s",
			   state->zone_name);
	free(state->zone_name);
	free(state);
}


/*
  see if we handle a given zone
 */
isc_result_t dlz_findzonedb(void *dbdata, const char *name)
{
	struct dlz_example_data *state = (struct dlz_example_data *)dbdata;
	if (strcasecmp(state->zone_name, name) == 0) {
		return ISC_R_SUCCESS;
	}
	return ISC_R_NOTFOUND;
}



/*
  lookup one record
 */
isc_result_t dlz_lookup(const char *zone, const char *name,
			void *dbdata, dns_sdlzlookup_t *lookup)
{
	struct dlz_example_data *state = (struct dlz_example_data *)dbdata;
	isc_boolean_t found = ISC_FALSE;
	isc_sockaddr_t *src;
	char full_name[256];
	int i;

	UNUSED(zone);

	if (state->putrr == NULL)
		return (ISC_R_NOTIMPLEMENTED);

	if (strcmp(name, "@") == 0) {
		strncpy(full_name, state->zone_name, 255);
		full_name[255] = '\0';
	} else
		snprintf(full_name, 255, "%s.%s", name, state->zone_name);

	for (i=0; i<MAX_RECORDS; i++) {
		if (strcasecmp(state->current[i].name, full_name) == 0) {
			isc_result_t result;
			found = true;
			result = state->putrr(lookup, state->current[i].type,
					      state->current[i].ttl,
					      state->current[i].data);
			if (result != ISC_R_SUCCESS) {
				return result;
			}
		}
	}
	if (!found) {
		return ISC_R_NOTFOUND;
	}
	return ISC_R_SUCCESS;
}


/*
  see if a zone transfer is allowed
 */
isc_result_t dlz_allowzonexfr(void *dbdata, const char *name, const char *client)
{
	/* just say yes for all our zones */
	return dlz_findzonedb(dbdata, name);
}

/*
  perform a zone transfer
 */
isc_result_t dlz_allnodes(const char *zone, void *dbdata,
			  dns_sdlzallnodes_t *allnodes)
{
	struct dlz_example_data *state = (struct dlz_example_data *)dbdata;
	int i;

	UNUSED(zone);

	if (state->putnamedrr == NULL)
		return (ISC_R_NOTIMPLEMENTED);

	for (i = 0; i < MAX_RECORDS; i++) {
		isc_result_t result;
		if (strlen(state->current[i].name) == 0) {
			continue;
		}
		result = state->putnamedrr(allnodes, state->current[i].name, state->current[i].type,
					   state->current[i].ttl, state->current[i].data);
		if (result != ISC_R_SUCCESS) {
			return result;
		}
	}

	return ISC_R_SUCCESS;
}


/*
  start a transaction
 */
isc_result_t dlz_newversion(const char *zone, void *dbdata, void **versionp)
{
	struct dlz_example_data *state = (struct dlz_example_data *)dbdata;

	if (state->transaction_started) {
		if (state->log != NULL)
			state->log(ISC_LOG_INFO,
				   "dlz_example: transaction already "
				   "started for zone %s", zone);
		return (ISC_R_FAILURE);
	}

	state->transaction_started = true;

	*versionp = (void *) &state->transaction_started;

	return ISC_R_SUCCESS;
}

/*
  end a transaction
 */
void dlz_closeversion(const char *zone, isc_boolean_t commit, void *dbdata, void **versionp)
{
	struct dlz_example_data *state = (struct dlz_example_data *)dbdata;

	if (!state->transaction_started) {
		if (state->log != NULL)
			state->log(ISC_LOG_INFO, "dlz_example: transaction not "
				   "started for zone %s", zone);
		*versionp = NULL;
		return;
	}

	state->transaction_started = false;

	*versionp = NULL;

	if (commit) {
		int i;
		if (state->log != NULL)
			state->log(ISC_LOG_INFO, "dlz_example: committing "
				   "transaction on zone %s", zone);
		for (i = 0; i < MAX_RECORDS; i++) {
			if (strlen(state->deletes[i].name) > 0U) {
				(void)del_name(state, &state->current[0],
					       state->deletes[i].name,
					       state->deletes[i].type,
					       state->deletes[i].ttl,
					       state->deletes[i].data);
			}
		}
		for (i = 0; i < MAX_RECORDS; i++) {
			if (strlen(state->adds[i].name) > 0U) {
				(void)add_name(state, &state->current[0],
					       state->adds[i].name,
					       state->adds[i].type,
					       state->adds[i].ttl,
					       state->adds[i].data);
			}
		}
	} else {
		if (state->log != NULL)
			state->log(ISC_LOG_INFO, "dlz_example: cancelling "
				   "transaction on zone %s", zone);
	}
	memset(state->adds, 0, sizeof(state->adds));
	memset(state->deletes, 0, sizeof(state->deletes));
}


/*
  configure a writeable zone
 */
isc_result_t dlz_configure(dns_view_t *view, void *dbdata)
{
	struct dlz_example_data *state = (struct dlz_example_data *)dbdata;
	isc_result_t result;

	if (state->log != NULL)
		state->log(ISC_LOG_INFO, "dlz_example: starting configure");

	if (state->writeable_zone == NULL) {
		if (state->log != NULL)
			state->log(ISC_LOG_INFO, "dlz_example: no "
				   "writeable_zone method available");
		return (ISC_R_FAILURE);
	}

	result = state->writeable_zone(view, state->zone_name);
	if (result != ISC_R_SUCCESS) {
		if (state->log != NULL)
			state->log(ISC_LOG_ERROR, "dlz_example: failed to "
				   "configure zone %s", state->zone_name);
		return (result);
	}

	if (state->log != NULL)
		state->log(ISC_LOG_INFO, "dlz_example: configured writeable "
			   "zone %s", state->zone_name);
	return (ISC_R_SUCCESS);
}

/*
  authorize a zone update
 */
isc_boolean_t dlz_ssumatch(const char *signer, const char *name, const char *tcpaddr,
			   const char *type, const char *key, uint32_t keydatalen, uint8_t *keydata,
			   void *dbdata)
{
	struct dlz_example_data *state = (struct dlz_example_data *)dbdata;
	if (strncmp(name, "deny.", 5) == 0) {
		if (state->log != NULL)
			state->log(ISC_LOG_INFO, "dlz_example: denying update "
				   "of name=%s by %s", name, signer);
		return (ISC_FALSE);
	}
	if (state->log != NULL)
		state->log(ISC_LOG_INFO, "dlz_example: allowing update of "
			   "name=%s by %s", name, signer);
	return (ISC_TRUE);
}


static isc_result_t modrdataset(struct dlz_example_data *state, const char *name, const char *rdatastr,
				struct record *list)
{
	char *full_name, *dclass, *type, *data, *ttlstr, *buf;
	isc_result_t result;
	char *saveptr = NULL;

	buf = strdup(rdatastr);
	if (buf == NULL)
		return (ISC_R_FAILURE);

	/*
	 * The format is:
	 * FULLNAME\tTTL\tDCLASS\tTYPE\tDATA
	 *
	 * The DATA field is space separated, and is in the data format
	 * for the type used by dig
	 */

	full_name = STRTOK_R(buf, "\t", &saveptr);
	if (full_name == NULL)
		goto error;

	ttlstr = STRTOK_R(NULL, "\t", &saveptr);
	if (ttlstr == NULL)
		goto error;

	dclass = STRTOK_R(NULL, "\t", &saveptr);
	if (dclass == NULL)
		goto error;

	type = STRTOK_R(NULL, "\t", &saveptr);
	if (type == NULL)
		goto error;

	data = STRTOK_R(NULL, "\t", &saveptr);
	if (data == NULL)
		goto error;

	result = add_name(state, list, name, type, strtoul(ttlstr, NULL, 10), data);
	free(buf);
	return (result);

 error:
	free(buf);
	return (ISC_R_FAILURE);
}


isc_result_t dlz_addrdataset(const char *name, const char *rdatastr, void *dbdata, void *version)
{
	struct dlz_example_data *state = (struct dlz_example_data *)dbdata;

	if (version != (void *) &state->transaction_started) {
		return ISC_R_FAILURE;
	}

	if (state->log != NULL)
		state->log(ISC_LOG_INFO, "dlz_example: adding rdataset %s '%s'",
			   name, rdatastr);

	return modrdataset(state, name, rdatastr, &state->adds[0]);
}

isc_result_t dlz_subrdataset(const char *name, const char *rdatastr, void *dbdata, void *version)
{
	struct dlz_example_data *state = (struct dlz_example_data *)dbdata;

	if (version != (void *) &state->transaction_started)
		return (ISC_R_FAILURE);

	if (state->log != NULL)
		state->log(ISC_LOG_INFO, "dlz_example: subtracting rdataset "
			   "%s '%s'", name, rdatastr);

	return modrdataset(state, name, rdatastr, &state->deletes[0]);
}


isc_result_t dlz_delrdataset(const char *name, const char *type, void *dbdata, void *version)
{
	struct dlz_example_data *state = (struct dlz_example_data *)dbdata;

	if (version != (void *) &state->transaction_started)
		return (ISC_R_FAILURE);

	if (state->log != NULL)
		state->log(ISC_LOG_INFO, "dlz_example: deleting rdataset %s "
			   "of type %s", name, type);

	return ISC_R_SUCCESS;
}
