#include "port_before.h"

#include <unistd.h>
#include <stdio.h>
#include <syslog.h>
#include <net/if.h>
#include <errno.h>

#include <sys/socket.h>
#include <io.h>
#include <lmaccess.h>
#include <lmserver.h>
#include <lmapibuf.h>
#include <lmwksta.h>
#include <fcntl.h>

#include "port_after.h"

/*
 * sockets are file descriptors in UNIX.  This is not so in NT
 * We must keep track of what is a socket and what is an FD to
 * make everything flow right.
 */
fddata fdset[FOPEN_MAX];
extern sockdata sockset[REAL_FD_SETSIZE];

/* Interface data */
#define MAX_INTERFACES	8

struct ifdata *IFData[MAX_INTERFACES];
static int numInterfaces = 0;

/*
 * Initialize for FD tracking, ioctl()
 */
void unistdInit()
{
	int i;
	for(i=0; i < FOPEN_MAX; i++)
	{
		fdset[i].fd = 0;
		fdset[i].flags = 0;
	}
	
	numInterfaces = GetInterfaces(IFData);
}

/*
 * Clean up - free memory allocated by GetInterfaces()
 */
void unistdCleanup()
{
	int i;

	for(i = 0; i < MAX_INTERFACES; i++)
		if(IFData[i])
			free(IFData[i]);
}


/*
 * Add a fd to the list
 */
void fdin(int fd)
{
	int i = 0;
	while(fdset[i].fd > 0)
		i++;

	if(i < REAL_FD_SETSIZE)
		fdset[i].fd = fd;
}

/*
 * Remove a fd from the list
 */
void fdout(int fd)
{
	int i = 0;
	while(fdset[i].fd != fd && i < FOPEN_MAX)
		i++;

	if(i < FOPEN_MAX)
	 	fdset[i].fd = 0;
}

int setfdflags(int fd, int flags)
{
	int i = 0;
	while(fdset[i].fd != fd && i < FOPEN_MAX)
		i++;

	if(i < FOPEN_MAX)
	{
 		fdset[i].flags |= flags;
		return(0);
	}
	return(-1);
}

int getfdflags(int fd)
{
	int i = 0;
	while(fdset[i].fd != fd && i < FOPEN_MAX)
		i++;

	if(i < FOPEN_MAX)
	 	return(fdset[i].flags);

	return(-1);
}


/*
 * ioctl() is the root of all socket evil in UNIX.
 */
int ioctl(int fd, int request, ...)
{
	struct ifconf *ifc;
	struct ifreq *ifr;
	char *buf;
	int i, rc = 0;
	va_list ap;

	va_start(ap, request);
	
	switch(request)
	{
		case SIOCGIFCONF:
			ifc = va_arg(ap, struct ifconf *);
			ifc->ifc_len = 0;
			buf = ifc->ifc_buf;
			
			for(i = 0; i < numInterfaces; i++)
			{
				ifr = (struct ifreq *)buf;
				memset(ifr->ifr_oname, 0, IFNAMSIZ);

				strcpy(ifr->ifr_name,IFData[i]->if_name);
				memcpy(&(ifr->ifr_addr), &(IFData[i]->if_addr), sizeof(struct sockaddr)); 
				
				/* now increment pointer */
				buf += sizeof(struct ifreq);
				ifc->ifc_len += sizeof(struct ifreq);
			}
			break;
		case SIOCGIFADDR:
			ifr = va_arg(ap, struct ifreq *);
			memset(ifr->ifr_oname, 0, IFNAMSIZ);
			for(i = 0; i < numInterfaces; i++)
			{
				if(!strcmp(ifr->ifr_name, IFData[i]->if_name))
				{
					memcpy(&(ifr->ifr_addr), &(IFData[i]->if_addr), sizeof(struct sockaddr)); 
					break;
				}
			}
			break;
		case SIOCGIFFLAGS:
			ifr = va_arg(ap, struct ifreq *);
			memset(ifr->ifr_oname, 0, IFNAMSIZ);
			for(i = 0; i < numInterfaces; i++)
			{
				if(!strcmp(ifr->ifr_name, IFData[i]->if_name))
				{
					ifr->ifr_flags  = IFData[i]->if_flags;
					break;
				}
			}
			break;
		case SIOCGIFDSTADDR:
			ifr = va_arg(ap, struct ifreq *);
			memset(ifr->ifr_oname, 0, IFNAMSIZ);
			for(i = 0; i < numInterfaces; i++)
			{
				if(!strcmp(ifr->ifr_name, IFData[i]->if_name))
				{
					memcpy(&(ifr->ifr_dstaddr), &(IFData[i]->if_dstaddr), sizeof(struct sockaddr)); 
					break;
				}
			}
			break;
		default:
			break;
	}	
	
	va_end(ap);
	return rc;
}


/*
 * _SC_OPEN_MAX should return FOPEN_MAX, but BIND thinks that file
 * descriptors can be sockets or files.  A socket number can be over 1000,
 * so we need to return a high number so evHighestFD() doesn't crap out on us.
 */ 
long sysconf(int which)
{
	long rc = -1;

	switch(which)
	{
		case _SC_OPEN_MAX:
			rc = 16384;
			break;
		default:
			break;
	}
	return rc;
}

int gethostid()
{
	/* Build a hostid out of the hostname */

	NET_API_STATUS status;
	LPBYTE buf;
	LPTSTR hostname;
	int id = 0;
	int	i = 0,
		nulCount = 0;
	
	status = NetServerGetInfo(NULL, 100, &buf);
	hostname = ((LPSERVER_INFO_100)buf)->sv100_name;
	/* hostname is Unicode, so it is terminated double-NULL */
	while(nulCount < 2)
	{
		if(hostname[i] == 0)
			nulCount++;
		else
		{
			nulCount = 0;
			id += hostname[i];
		}
		i++;
	}
	NetApiBufferFree(buf);
	return(id);
}

int getppid()
{
	/****
	***** If we ever need this for anything but seeding the HMAC key in prandom, 
	***** something clever needs to happen here.
	****/
	
	/* No NT equivalent.  We'll use the our own process ID */
	return _getpid();
}

uid_t getuid()
{
	/* Build a uid out of the username string */
	
	char username[64];
	int id = 0;
	int	i = 0,
		size = 64;
	
	GetUserName(username, &size);
	for(i = 0; i < size; i++)
		id += username[i];
	return(id);
}

gid_t getgid()
{
	/* Groups are a pain on NT.  Return a dummy */
	return 42; /* d. adams */
}

/*
 * open(), close(), read(), write(), fsync()
 * sockets are file descriptors in UNIX.  This is not so in NT
 * We keep track of what is a socket and what is an FD to
 * make everything flow right.
 */
int open(const char *fn, int flags, ...)
{
	va_list args;
	int pmode;
	int fd;

	/* Extract the cmd parameter */
	va_start(args, flags);
	pmode = va_arg(args, int);
	fd = _open(fn, flags, pmode);
	fdin(fd);
	return fd;
}

int close(int fd)
{
	if(S_ISSOCK(fd))
		return closesocket(fd);
	else
	{
		fdout(fd);
		return _close(fd);
	}
}

int read(int fd, char *buf, int len)
{
	if(S_ISSOCK(fd))
		return recv(fd, buf, len, 0);
	else
		return _read(fd, buf, len);
}

int write(int fd, char *buf, int len)
{
	if(S_ISSOCK(fd))
		return send(fd, buf, len, 0);
	else
		return _write(fd, buf, len);
}

int fsync(int fd)
{
	if(S_ISSOCK(fd))
	{
		/* No way to flush a socket */
		return 0;
	}
	else
	{
		/* Sync memory contents with disk */
		return _commit(fd);
	}
}


int chown(const char *path, uid_t owner, gid_t group)
{
	/* Do nothing, return all OK. */
	return 0;
}

/*
 * Passthrough to NT _pipe()
 */
int pipe(int filedes[2])
{
	return _pipe(filedes, 65535, _O_TEXT);
}


/*
 * Holds flags, sets block/nonblock for sockets
 * not much else
 */
int fcntl(int fd, int cmd, ...)
{
	va_list args;
	u_long arg;
	int i, rc = -1;
	va_start(args, cmd);
	

	switch(cmd)
	{
		/* Set status flags */
		case F_SETFL:
			/* Extract the flags */
			arg = va_arg(args, u_long);
			i = arg & PORT_NONBLOCK;
			ioctlsocket(fd, FIONBIO, &i);
			rc = setsockflags(fd, i ? PORT_NONBLOCK : 0);
			break;
		/* Get status flags */
		case F_GETFL:
			rc = getsockflags(fd);
			break;
		/* Set FD flags */
		case F_SETFD:
			arg = va_arg(args, u_long);
			rc = setfdflags(fd, arg);
			break;
		/* Get FD flags */
		case F_GETFD:
			rc = getfdflags(fd);
			break;
		default:
			break;
	}
	va_end(args);	
	return rc;
}


/*
 * Emulate UNIX mkstemp, which returns an open FD to the new file
 */
int mkstemp(char *tmpname)
{
	_mktemp(tmpname);
	return(_open(tmpname, _O_WRONLY | _O_CREAT, _S_IREAD | _S_IWRITE));
}
