#!/usr/bin/env perl
#
# Copyright (C) 2002, 2003  Internet Software Consortium.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM
# DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
# INTERNET SOFTWARE CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
# FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: epp-from-file.pl,v 1.19 2003/02/13 18:41:01 lidl Exp $

use lib '@prefix@';

use strict;
use warnings;

use Data::Dumper;
use Event qw(loop unloop);
use Time::HiRes qw(gettimeofday);
use IO::Socket::INET;
use Getopt::Long;

use ISC::Net::LengthPrefix;
use ISC::Date ':all';

my $exit_code;

my $host = "127.0.0.1";
my $port = 0;
my $ssl = 0;
my $logfile;
my $infile;
my $show_vars;
my $ret = GetOptions("host=s" => \$host,
		     "port=s" => \$port,
		     "ssl" => \$ssl,
		     "log=s" => \$logfile,
		     "file=s" => \$infile,
		     "showvars!" => \$show_vars);

if ($ssl) {
    $port = 648 unless ($port);
} else {
    $port = 5544 unless ($port);
}

if ($logfile) {
    open(LOGFD, ">$logfile") or die "Cannot open $logfile for writing";
} else {
    *LOGFD = *STDOUT;
}

if ($infile) {
    open(INFILE, "<$infile") or die "Cannot open $infile for reading";
} else {
    *INFILE = *STDIN;
}

my $srs;
if ($ssl) {
    require IO::Socket::SSL;
    $srs = new IO::Socket::SSL(PeerAddr => $host,
			       PeerPort => $port,
			       SSL_verify_mode => 0,
			       Proto => "tcp",
			       Type => SOCK_STREAM);
} else {
    $srs = new IO::Socket::INET(PeerAddr => $host,
				PeerPort => $port,
				Proto => "tcp",
				Type => SOCK_STREAM);
}
die "no socket opened: $!" if (!$srs);

my $lp = new ISC::Net::LengthPrefix(cb => \&_msg_cb,
				    error_cb => \&_msg_err,
				    socket => $srs,
				    lenbytes => 2);

my $seed = time;
my $prefix = $$;

sub random_domain {
    return 'D' . $prefix . '-' . $seed++;
}

sub random_contact {
    return 'C' . $prefix . '-' . $seed++;
}

sub random_host {
    return 'H' . $prefix . '-' . $seed++;
}

sub random_xid {
    return $$ . '-' . $seed++;
}

sub then {
    my ($add) = @_;

    my $date = date_ansi(date_add(0, $add * 12));
    my ($d, undef) = split(/ /, $date);

    return ($d);
}

my $vars = {};
sub _subst {
    my ($var, $partial) = @_;

    my $subd = 1;
    my $oldvar = $var;
    while ($subd) {
	$subd = 0;
	foreach my $v (keys %$vars) {
	    my $l = $vars->{$v};
	    $var =~ s/$v/$l/g;
	}
	if (!$partial) {
	    if ($var =~ /\{xid\}/) {
		my $x = random_xid();
		$var =~ s/\{xid\}/$x/g;
	    }
	}
	if ($var ne $oldvar) {
	    $oldvar = $var;
	    $subd = 1;
	}
    }

    return $var;
}

sub _msg_err {
    my ($lp, $msg) = @_;

    print STDERR "lp error: $msg\n";
    unloop;
}

sub LOG {
    my ($source, $lines) = @_;

    if (!$source) {
	print LOGFD "\n";
	return;
    }

    if (!ref($lines)) {
	print LOGFD "$source (" . length($lines) . "):$lines\n";
    }
}

my $xstart;

my @files = (*INFILE);

sub _msg_cb {
    my ($lp, $lines) = @_;

    LOG('S', $lines);

    if ($xstart) {
	my $now = gettimeofday;
	my $t = sprintf("%5.3f ms", ($now - $xstart) * 1000);

	LOG('T', $t);
	LOG();
    }

    $xstart = gettimeofday;

    my $msg = '';

    while (scalar(@files)) {
	my $sendit = 0;
	while (<INFILE>) {
	    chomp;
	    next if (/^\#/);
	    next if (/^$/);

	    if (/^\.VAR (.*)$/) {
		my ($var, $val) = split(/[\ \t=]+/, $1, 2);

		my $subd = 1;
		while ($subd) {
		    $subd = 0;
		    if ($val =~ /\{domain\}/) {
			my $d = random_domain();
			$val =~ s/\{domain\}/$d/g;
			$subd = 1;
		    }
		    if ($val =~ /\{host\}/) {
			my $d = random_host();
			$val =~ s/\{host\}/$d/g;
			$subd = 1;
		    }
		    if ($val =~ /\{contact\}/) {
			my $d = random_contact();
			$val =~ s/\{contact\}/$d/g;
			$subd = 1;
		    }
		    if ($val =~ /\{now\}/) {
			my ($d, undef) = split(/ /, date_ansi());
			$val =~ s/\{now\}/$d/g;
			$subd = 1;
		    }
		    if ($val =~ /\{now\+([0-9]+)\}/) {
			my $d = then($1);
			$val =~ s/\{now\+$1\}/$d/g;
			$subd = 1;
		    }
		    my $nval = _subst($val, 1);
		    if ($val ne $nval) {
			$val = $nval;
			$subd = 1;
		    }

		    $vars->{$var} = $val;
		}
		print "$var == '$val'\n"
		    if ($show_vars);
		next;
	    }

	    if (/^\.SEND$/) {
		$sendit = 1;
		last;
	    }

	    if (/^\.SLEEP (.+)$/) {
		sleep ($1);
		last;
	    }

	    if (/^\.INCLUDE (.+)$/) {
		my $fh;
		open($fh, "<$1") or die "Cannot open $1: $!";
		push(@files, $fh);
		*INFILE = $fh;
		last;
	    }

	    if (/^\.#/) {
		next;
	    }

	    die "Unknown . directive: $_" if (/^\./);

	    $msg .= _subst($_) . "\n";
	}
	if ($sendit) {
	    last;
	}
	if (eof INFILE) {
	    pop @files;
	    last if (!$files[0]);
	    *INFILE = $files[0];
	}
    }

    if ($msg) {
	LOG('C', $msg);
	$lp->send($msg);
	return;
    }

    unloop();
}

$Event::DIED = sub {
    Event::verbose_exception_handler(@_);
    Event::unloop_all();
};

Event::loop();

if ($exit_code) {
    exit $exit_code;
}
